
% imdisplay.sl: Render a stack of still images or display an animation. {{{
%
%		The image specifications passed in may be names of files
%		(of various types), GDK pixbufs, S-Lang arrays (2D/greyscale,
%		3D/RGB, or 3D/RGBA or arbitrary 3D volumes), even List_Type[N]
%		& Array_Type[N] for animations.  If any input still image
%		contains an alpha channel then the rendered result will as
%		well.  Input image files may be of any type readable by a
%		GdkPixbuf loader module (including FITS) or GIF module,
%		while output may be saved to any writable GdkPixbuf format
%		(again, including FITS) or GIF.
%
%		Hint: don't rely upon XV (which is otherwise great!) to
%		judge whether transparency has been correctly preserved
%		in output files.  Most "standard" installations of XV do
%		not really support layering, and my observation,
%		corroborated by others on the WWW, is that it will assign
%		colors to transparent pixels in an inconsistent manner
%		(sometimes white, sometimes black, ...).  Instead, try
%		using a more actively developed tool, and with better RGBA
%		support, such as the GIMP or ImageMagick.
%		
%		This file is part of SLgtk, the S-Lang bindings to Gtk.
%		Copyright (C) 2003-2010 Massachusetts Institute of Technology
%		Copyright (C) 2002 Michael S. Noble (mnoble@space.mit.edu)
% }}}

require("gtk");

private variable Context = struct {	% {{{
   wscale, hscale, scale_type,		% resizing along X/Y
   flip, flop,				% mirror along Y/X?
   fill_rule,				% how to grow image when win enlarged
   controller,				% toplevel control window
   size_label,				% size of image (e.g. beneath mouse)
   transparency,			% does any image have alpha channel?
   layout,

   autosave,
   file_name,
   file_type,

   panes,
   npanes,
   largest_w,
   largest_h,

   revert_button,
   save_button,
   animate,
   animation,
   frame_delay				% global/common value for all frames
};

private variable EMPTY = "";
private variable version = 500;
private variable version_string = "0.5.0";
define _imdisplay_version_string() { return version_string; }
define _imdisplay_version() { return version; }
private define load();					% forward declaration
private define anim_iterate();			% forward declaration

% }}}

private variable Animation = struct { 	% {{{
   frames,
   frame_num,
   num_frames,
   play_button,
   pause_button,
   prev_button,
   next_button,
   playing				% signal id of frame iterator callback
}; % }}}

% Customization options handling {{{

private variable pixbuf_flip = NULL; % {{{
#ifexists gdk_pixbuf_flip
   pixbuf_flip = &gdk_pixbuf_flip;
#endif
if (andelse {pixbuf_flip == NULL}
	    {is_defined( current_namespace + "->gdk_pixbuf_flip")}) {
   eval("&gdk_pixbuf_flip", current_namespace);
   pixbuf_flip = ();
} % }}}

private variable DEFAULT_FRAME_DELAY = 500;			% 0.5 seconds

private variable
	SCALE_PERCENT = 0x1,
	SCALE_PIXELS  = 0x2;

private variable
	FILL_NONE  = 0,
	FILL_TILE  = 1,
	FILL_SCALE = 2,
	FILL_OPTION_STRINGS = ["none", "tile", "scale" ];

private variable
	PANE_SINGLE = 0,
	PANE_VERT   = 2,		% corresponds to chain gravity = South
	PANE_HORIZ  = 3;		% corresponds to chain gravity = East

private define default_context()
{
   variable ctx = @Context;
   ctx.fill_rule = FILL_SCALE;
   ctx.layout = PANE_SINGLE;
   ctx.transparency = 0;
   ctx.largest_w = 0;
   ctx.largest_h = 0;
   ctx.autosave = 0;
   ctx.file_name = "imdisplay_tmp";
   ctx.animate = 0;
   ctx.frame_delay = DEFAULT_FRAME_DELAY;
   return ctx;
}

private variable Default_Context = NULL;

private define eprint() % {{{
{
  variable args = __pop_args(_NARGS);
  () = fprintf(stderr,__push_args(args));
  () = fflush(stderr);
} % }}}

private define parse_tuple(tuple, delimiter)  % {{{
{
   variable toks = strtok(tuple, delimiter);
   if (length(toks) == 1)
       toks = [toks, toks];

   return atof(toks[0]), atof(toks[1]);
} % }}}

private define parse_options(ctx, values)  % {{{
{
   if (values == NULL or values == "")
      return;

   variable prev_traceback = _traceback;
   _traceback = -1;

   values = strtok(values, ",");
   foreach(values) {

	variable option = (), value = EMPTY;
	option = array_map(String_Type, &strtrim, strtok(option, "="));

	if (length(option) > 1)
	   value = option[1];

	option[0] = strlow(option[0]);

	switch( option[0] )
	{ case "fill" :

		if (value == EMPTY)
		   continue;

		value = where(FILL_OPTION_STRINGS == strlow(value));
		if (length(value) == 0)
		   continue;

		ctx.fill_rule = value[0];
	}
	{ case "save" or case "autosave":

		if (value == EMPTY)
		   continue;

		ctx.autosave = 1;

		value = strtok(value, "/");	% Look for /gif, /png, etc
		ctx.file_name = value[0];

		if (length(value) > 1)
		   ctx.file_type = value[1];
		else {
		   value = strtok(ctx.file_name,".");
		   if (length(value) > 1)
			ctx.file_type = value[-1];
		   else
			ctx.file_type = "png";
		}

		if (not _image_format_is_writable(ctx.file_type))
		   verror("Unsupported image output format: %S", ctx.file_type);
	}
	{ case "size" or case "scale":

		if (value == EMPTY)
		   continue;

		(ctx.wscale, ctx.hscale) = parse_tuple(value, "x");

		if (ctx.wscale <= 0 or ctx.hscale <= 0) {
		   eprint("Illegal size option: %s\n", value);
		   continue;
		}

		if (is_substr(value, "%")) {
		   ctx.scale_type = SCALE_PERCENT;
		   ctx.wscale /= 100.0;
		   ctx.hscale /= 100.0;
		}
		else
		   ctx.scale_type = SCALE_PIXELS;
	}
	{ case "flip" or case "flop":

		if (pixbuf_flip == NULL) {
		   eprint("Gtk 2.6 or newer is needed for flip/flop\n");
		   continue;
		}
		set_struct_field(ctx, option[0], TRUE);
	}
	{ case "panes" :

		if (value == EMPTY)
		   continue;

		value = strlow(value);
		switch (value)
		{ case "one" or case "single"      : ctx.layout = PANE_SINGLE;}
		{ case "horiz" or case "horizontal": ctx.layout = PANE_HORIZ; }
		{ case "vert" or case "vertical"   : ctx.layout = PANE_VERT;  }
	}
	{ case "horiz" or case "horizontal": ctx.layout = PANE_HORIZ; }
	{ case "vert" or case "vertical"   : ctx.layout = PANE_VERT;  }
	{ case "anim" or case "animate"    :

		ctx.animate = 1;
		if (value != EMPTY) {
		   value = atol(value);
		   if (value > 0)
			ctx.frame_delay = value;
		}
	}
   }

   _traceback = prev_traceback;
} % }}}

private define scale(pane) % {{{
{
   variable ctx = pane.ctx;

   if (ctx.scale_type == NULL)
      return;

   variable w = pane.w, h = pane.h, pb = pane.pixbuf;  % width, height, pixbuf

   switch(ctx.scale_type)
   { case SCALE_PERCENT: w = int(w*ctx.wscale); h = int(h*ctx.hscale); }
   { case SCALE_PIXELS : w = int(ctx.wscale);   h = int(ctx.hscale);   }

   pb = gdk_pixbuf_scale_simple(pb, w, h, GDK_INTERP_BILINEAR);
   pane.w = w;
   pane.h = h;
   pane.pixbuf = pb;
} % }}}

private define flipflop(pane) % {{{
{
   if (pane.ctx.flip != NULL)
	pane.pixbuf  = (@pixbuf_flip)(pane.pixbuf, FALSE);

   if (pane.ctx.flop != NULL)
	pane.pixbuf  = (@pixbuf_flip)(pane.pixbuf, TRUE);

} % }}}

% }}}

% Pane object & methods {{{

private variable Pane = struct {  % {{{
   w, h, resize_w, resize_h,		% image dimensions
   viewport_w, viewport_h,		% starting window dimensions
   win,					% toplevel image display window
   image, drawable, pixmap, gc,		% image widget & related
   pixbuf, resize_pixbuf,		% RGB(A) pixel buffers
   transparency,			% does image have alpha channel?
   title,
   delay,				% how long should it be shown?
   ctx					% parent context
};  % }}}

private define pane_align(master_pane, delta) % {{{
{
   % Ensure proper alignment of all panes, regardless of WM placement strategy
   if (master_pane == NULL || master_pane.win == NULL) return;

   variable x, y;
   gtk_window_get_position(master_pane.win, &x, &y);
   gtk_window_move(master_pane.win, x + delta, y + delta);
} % }}}

private define pane_new(ctx, pixbuf, title_info, delay) % {{{
{
   variable p = @Pane;

   p.ctx = ctx;
   p.pixbuf = pixbuf;
   p.resize_pixbuf = pixbuf;
   p.title = sprintf("%S", title_info);

   p.w = gdk_pixbuf_get_width(pixbuf),
   p.h = gdk_pixbuf_get_height(pixbuf);
   p.transparency = gdk_pixbuf_get_has_alpha(pixbuf);
   p.delay = delay;

   if (p.w > p.ctx.largest_w)
	p.ctx.largest_w = p.w;

   if (p.h > p.ctx.largest_h)
	p.ctx.largest_h = p.h;

   ctx.npanes++;
   return p;
} % }}}

private define pane_remove(pane, realign) % {{{
{
   variable i, ctx = pane.ctx, panes = ctx.panes;

   _for i (0, ctx.npanes-1, 1)
	if ( __eqs(panes[i].win, pane.win)) {
	   list_delete(panes, i);
	   ctx.npanes--;
	   break;
	}

   if (ctx.npanes && realign)
	pane_align(panes[0], 0);

} % }}}

private define pane_destroy(pane) % {{{
{
   pane_remove(pane, 1);
   if (pane.ctx.npanes == 0)
	gtk_widget_destroy(pane.ctx.controller);
} % }}}

private define pane_set_tiling(pane) % {{{
{
   if (pane.pixmap == NULL) {
	pane.pixmap = gdk_pixmap_new(pane.drawable, pane.w, pane.h, -1);
	gdk_draw_rectangle(pane.pixmap, pane.gc, 1, 0, 0, pane.w, pane.h);
	gdk_draw_pixbuf(pane.pixmap, pane.gc, pane.pixbuf, 0,0,0,0,
				pane.w, pane.h, GDK_RGB_DITHER_NORMAL, 0, 0);
   }
   % Gdk automatically tiles any applied background pixmap
   gdk_window_set_back_pixmap(pane.drawable, pane.pixmap, 0);
} % }}}

private define pane_image_sizeof(pane) % {{{
{
   if (pane.ctx.fill_rule == FILL_NONE) {
   	variable w = pane.w;
	variable h = pane.h;
   }
   else
	gdk_drawable_get_size(pane.drawable, &w, &h);

   variable size = sprintf("%S x %S", w, h);

   if (pane.ctx.size_label != NULL)
	gtk_label_set_text(pane.ctx.size_label, size);

   return w, h;
} % }}}

private define expose_cb(event, pane) % {{{
{
   variable x, y, w, h, pb, iw, ih;

   (iw, ih) = pane_image_sizeof(pane);

   switch(pane.ctx.fill_rule)
   { case FILL_TILE : return FALSE; }
   { case FILL_NONE : x = 0; y = 0; w = pane.w; h = pane.h; pb = pane.pixbuf; }
   { case FILL_SCALE:

	variable a = event.area;
	if (iw <= pane.w and ih <= pane.h)
	   pb = pane.pixbuf;
	else {
	   if (pane.resize_w != iw or pane.resize_h != ih) {

		pane.resize_pixbuf = gdk_pixbuf_scale_simple (pane.pixbuf,
						iw, ih, GDK_INTERP_BILINEAR);
		pane.resize_w = iw; pane.resize_h = ih;
	   }
	   pb = pane.resize_pixbuf;
	}
	x = a.x; y = a.y; w = a.width; h = a.height;
   }

   gdk_draw_pixbuf (pane.drawable, pane.gc, pb, x, y, x, y, w, h,
						GDK_RGB_DITHER_NORMAL, 0, 0);

   gtk_window_get_position(pane.win, &x, &y);
   if (y < 22)
	gtk_window_move(pane.win, x, 22);

   return FALSE;
} % }}}

private define realize_cb(pane)  % {{{
{
   pane.drawable = gtk_widget_get_window(pane.image);
   pane.gc = gdk_gc_new(pane.drawable);
   gtk_widget_modify_bg(pane.image, GTK_STATE_NORMAL, @gdk_white);
   gdk_gc_set_foreground(pane.gc, @gdk_white);
   if (pane.ctx.fill_rule == FILL_TILE)
	pane_set_tiling(pane);
   () = g_signal_connect_swapped(pane.image, "expose_event", &expose_cb, pane);
} % }}}

private define show_size_cb(widget, event, pane)  % {{{
{
  if (pane.image != NULL)
	( , ) = pane_image_sizeof(pane);

  % This callback is used for both signals and events,
  % but needs to return a disposition only for events
  if (typeof(event) == Struct_Type)
	return FALSE;
} % }}}

private define pane_set_viewport(pane) % {{{
{
   pane.viewport_w = min ([ pane.w, 7 * gdk_screen_width()  / 10]);
   pane.viewport_h = min ([ pane.h, 7 * gdk_screen_height() / 10]);

   % Put unusually large images within a scrolling window
   if (pane.viewport_w < pane.w or pane.viewport_h < pane.h) {

	% Prevent images with very small dimensions along 1 axis
	% (e.g. 1x4000) from being scaled to scroll window size
	pane.ctx.fill_rule = FILL_NONE;

	gtk_window_resize(pane.win, pane.viewport_w, pane.viewport_h);
	variable sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (sw, 0);
	gtk_scrolled_window_set_policy(sw, GTK_POLICY_AUTOMATIC,
		 					GTK_POLICY_AUTOMATIC);
	gtk_container_add(pane.win, sw);
	gtk_scrolled_window_add_with_viewport(sw, pane.image);
   }
   else
	gtk_container_add(pane.win, pane.image);
} % }}}

private define pane_make_drawable(pane)  % {{{
{
   scale(pane);
   flipflop(pane);

   pane.image = gtk_drawing_area_new();

   gtk_widget_add_events(pane.image, GDK_ENTER_NOTIFY_MASK);
   () = g_signal_connect_swapped(pane.image, "realize", &realize_cb, pane);
   () = g_signal_connect(pane.image,"enter-notify-event", &show_size_cb, pane);

   variable ctx = pane.ctx;
   if (ctx.animate)
	gtk_widget_set_size_request(pane.image, ctx.largest_w, ctx.largest_h);
   else
	gtk_widget_set_size_request(pane.image, pane.w, pane.h);

}  % }}}

private define pane_make_window(pane) % {{{
{
   pane.win = gtk_window_new (GTK_WINDOW_TOPLEVEL);

   gtk_window_set_title(pane.win, pane.title);
   () = g_signal_connect_swapped(pane.win, "destroy", &pane_destroy, pane);
   () = g_signal_connect(pane.win, "focus-in-event", &show_size_cb, pane);
   () = g_signal_connect(pane.win, "grab-focus", &show_size_cb, 0, pane);

   pane_make_drawable(pane);
   pane_set_viewport(pane);
   gtk_widget_show_all(pane.win);
}  % }}}

% }}}

private define load(image, ctx, delay)	% {{{
{
   variable pb = NULL;
   variable info = sprintf("%S", image);

   try switch( typeof(image))
   { case String_Type:

	if (stat_file(image) == NULL)
	   return parse_options(ctx, image);

	info = path_basename(image);
	pb = anim_read(image ; delay=&delay);	% first try to read as animation

	if (pb != NULL) {
	    if (ctx.frame_delay != DEFAULT_FRAME_DELAY)
		delay = ctx.frame_delay;

	    return load(pb, ctx, delay);
	}

	pb = gdk_pixbuf_new_from_file(image);	% then try as still image
   }
   { case List_Type:

	% convenience: automatically treat List_Type[N] as animation
   	load( list_to_array(image), ctx, delay);
	return;
   }
   { case Array_Type:

	variable dims, ndims; (dims, ndims,) = array_info(image);

	if (ndims == 1) {

	   if (_typeof(image) == Array_Type) {

		% convenience: automaticaly treat Array_Type[N] as animation
		ctx.animate = 0;
		array_map(Void_Type, &load, image, ctx, delay);
		ctx.animate = 1;
		return;
	   }

	   % convenience: treat 1D images as NxN if possible & fall thru to 2D
	   variable size = sqrt(dims[0]);
	   if (int(size) != size)
		return eprint("This 1D array cannot be treated as NxN image\n");
	   dims = [ int(size), int(size) ];
	   ndims = 2;
	}

	if (ndims == 2) {
	   variable rgbbuf = UChar_Type[dims[0], dims[1], 3];
	   image = norm_array(image);
	   rgbbuf[*,*,0] = image;
	   rgbbuf[*,*,1] = image;
	   rgbbuf[*,*,2] = image;
	   image = rgbbuf;
	   info = sprintf("%S x %S", dims[1], dims[0]);
	}
	else if (ndims == 3) {

	   % convenience: treat [N,M,1] as [N,M]
	   if (dims[2] == 1)
		return load(image[*,*,0], ctx, delay);

	   % convenience: animate [N,M,Z>4] because it cannot be RGBA
	   if (dims[2] > 4)
		ctx.animate = 1;
	
	   % convenience: animate 3D as series of 2D slices if delay was given
	   if (ctx.animate) {
		variable i = dims[0];		% how many slices

		if (length(delay) < i)
		   delay = Integer_Type[i] + delay;
		
		_for i (0, i-1, 1)
		   load(image[i, *, *], ctx, delay[i]);
		return;
	   }
	   image = norm_array(image);
	}
	else 
	   return eprint("%SD arrays not supported\n", length(dims));

	pb = gdk_pixbuf_new_from_data(image);
	image = sprintf("%S x %S", dims[1], dims[0]);
   }
   { case GdkPixbuf_Type:  pb = image; }

   catch AnyError: { pb = NULL; }
   if (pb == NULL)
	return eprint("Could not create 2D image from: %S\n", image);

   list_append( ctx.panes, pane_new(ctx, pb, info, delay) );

} % }}}

private define composite(ctx)  % {{{
{
   variable title, pane, transparency = 0;
   variable cpb = NULL, cw, ch;  	% composite pixbuf, width, height

   % Composite: render all frames to a single pane & discard panes 2 thru N
   foreach pane (ctx.panes) {

	variable pb = pane.pixbuf;
	variable w = gdk_pixbuf_get_width(pb), h = gdk_pixbuf_get_height(pb);

	if (cpb == NULL) {
	   cpb = pb; cw = w; ch = h;
	   transparency = pane.transparency;
	   title = pane.title;
	   continue;
	}

	if (pane.transparency and not(transparency)) {
	   cpb = gdk_pixbuf_add_alpha(cpb, FALSE, 0, 0, 0);
	   transparency = 1;
	}

	if (w != cw or h != ch) {

	   variable w2 = max([cw, w]), h2 = max ([ch, h]);
	   variable bigger = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE,8,w2,h2);
	   gdk_pixbuf_fill(bigger, 0xFFFFFF00);	% 100% transparent, white

	   % Copy existing image into new pixbuf, then composite new image
	   % Pixels not covered by these two operations remain transparent
	   gdk_pixbuf_copy_area(cpb, 0, 0, cw, ch, bigger, 0, 0);
   	   gdk_pixbuf_composite(pb, bigger, 0, 0, w, h, 0, 0, 1, 1,
						GDK_INTERP_BILINEAR, 255);
	   cpb = bigger;
	   cw = w2; ch = h2;
	   transparency = 1;
   	}
	else
	   gdk_pixbuf_composite(pb, cpb, 0, 0, w, h, 0, 0, 1, 1,
						GDK_INTERP_BILINEAR, 255);
   }

   ctx.transparency = transparency;

   if (cpb != NULL)
	ctx.panes = { pane_new(ctx, cpb, title, 0) };

   ctx.npanes = 1;
} % }}}

% Still images {{{

private define name_iter_next(this) % {{{
{
   this.index++;
   (this.name, ) = strreplace(this.pattern, "%d", string(this.index), 1);
   return this.name;
} % }}}

private define name_iter_new(name, ctx) % {{{
{
  variable ni = struct { pattern, index, next, name};
  if (ctx.npanes > 1) {
	name = strtok(name, ".");
	ni.pattern = name[0] + "-%d";
	if (length(name) > 1)
	   ni.pattern += "." + name[1];
  }
  else
	ni.pattern = name;
  ni.next = &name_iter_next;
  ni.index = 0;
  return ni;
} % }}}

private define still_save_file(file, type, pane) % {{{
{
   if (pane.drawable == NULL)
	variable pb = pane.pixbuf;
   else {
	variable w, h;
	gdk_drawable_get_size(pane.drawable, &w, &h);
	pb = gdk_pixbuf_get_from_drawable(NULL,pane.drawable,NULL,0,0,0,0,w,h);
   }

   if (pane.transparency)
	pb = gdk_pixbuf_add_alpha(pb, FALSE, 0, 0, 0);

   _pixbuf_save (pb, file, type);

   return FALSE;
} % }}}

private define still_save_cb(ignore, ctx) % {{{
{
   variable name, type, pane;
   (name, type) = _image_save_dialog(["png", "jpeg", "fits"]);

   if (name == NULL)
      return;

   name = name_iter_new(name, ctx);
   foreach pane (ctx.panes) {
	% Save the drawn window, rather than pixmap, in order to get the
	% tiling for free.  So, raise it to ensure that it's fully exposed
	gtk_window_present(pane.win);
	() = gtk_timeout_add(750, &still_save_file, name.next(), type, pane);
   }
} % }}}

private define set_fill_rule_cb(fill_button, ctx, fill_rule) % {{{
{
   if (not gtk_toggle_button_get_active (fill_button)) return;
   if (ctx.fill_rule == fill_rule) return;

   ctx.fill_rule = fill_rule;

   foreach (ctx.panes) {

	variable p = ();
	if (p.drawable == NULL)
	   continue;

	switch (fill_rule)
	{ case FILL_TILE:  pane_set_tiling(p); }
	{ gdk_window_set_background(p.drawable, gdk_white); }

	gtk_widget_queue_draw(p.image);
   }
} % }}}

private define revert_cb(revert_button, default_radio_button, ctx) % {{{
{  
   variable p;
   foreach p (ctx.panes) {
	if (p.win != NULL)
	   gtk_window_resize(p.win, p.viewport_w, p.viewport_h);
	if (default_radio_button != NULL)
	   gtk_toggle_button_set_active(default_radio_button, TRUE);
   }
} % }}}

private define make_fill_button(fb, kind, parent, ctx) % {{{
{
   variable label = FILL_OPTION_STRINGS[kind];
   label = sprintf("%c%s", label[0]-32, label[[1:]]);	% Upcase first letter

   if (fb == NULL)
	fb = gtk_radio_button_new_with_label(NULL, label);
   else
	fb = gtk_radio_button_new_with_label_from_widget(fb, label);

   if (ctx.fill_rule == kind) 
	() = g_signal_connect(ctx.revert_button,"clicked", &revert_cb, fb, ctx);

   gtk_toggle_button_set_active(fb, ctx.fill_rule == kind);
   gtk_box_pack_end(parent, fb, FALSE, FALSE, 0);

   () = g_signal_connect(fb, "clicked", &set_fill_rule_cb, ctx, kind);
   gtk_widget_unset_flags (fb, GTK_CAN_FOCUS);

   return fb;
} % }}}

private define still_draw_controls(hbox, ctx)  % {{{
{
   variable frame = gtk_frame_new("Fill Rule");
   gtk_container_add(hbox,frame);

   variable fillbox = gtk_hbox_new(TRUE, 0);
   gtk_container_add(frame, fillbox);

   variable button = make_fill_button(NULL, FILL_NONE, fillbox, ctx);
   button = make_fill_button(button, FILL_SCALE, fillbox, ctx);
   button = make_fill_button(button, FILL_TILE,  fillbox, ctx);

   frame = gtk_frame_new("Transparency");
   gtk_container_add(hbox, frame);
   if (ctx.transparency)
	gtk_container_add(frame, gtk_label_new("Yes"));
   else
	gtk_container_add(frame, gtk_label_new("No"));
}  % }}}

private define still_display(ctx) % {{{
{
   if (ctx.layout == PANE_SINGLE)
	composite(ctx);

   variable pane, ni = name_iter_new(ctx.file_name, ctx);

   foreach pane (ctx.panes) {

	() = ni.next();
	if (ctx.autosave)
	   () = still_save_file(ni.name, ctx.file_type, pane);
	else {

	   pane_make_window(pane);
	   if (ni.index > 1)
	      _gtk_window_chain(ctx.panes[ni.index-2].win, pane.win,ctx.layout);
	}
   }

}  % }}}

% }}}

% Animated images {{{

private define anim_make_button(kind)  % {{{
{
   variable button = gtk_button_new();
   variable image = gtk_image_new_from_stock(kind, GTK_ICON_SIZE_BUTTON);
   gtk_button_set_image(button, image);
   gtk_widget_unset_flags(button, GTK_CAN_FOCUS);

   return button;
}  % }}}

private define anim_play_timer(anim)  %{{{
{
   anim_iterate(anim, 1);
   return TRUE;
}  % }}}

private define anim_iterate(anim, direction) % {{{
{
   anim.frame_num = anim.frame_num + direction;

   if (anim.frame_num < 0)			% iterate to next frame,
      anim.frame_num += anim.num_frames;	% either forward or back
   else if (anim.frame_num == anim.num_frames)
      anim.frame_num = 0;

   variable main = anim.frames[0];		% render to main pane/window
   variable this = anim.frames[anim.frame_num];

   % Ideally one animates frames of consistent size, but inconsistently-
   % sized images may be animated, too, by doing a minimal intersection.
   gdk_draw_pixbuf(main.drawable, main.gc, this.pixbuf, 0,0,0,0,
			min([this.w, main.w]), min([this.h, main.h]),
			GDK_RGB_DITHER_NORMAL, 0, 0);

   gtk_window_set_title(main.win, sprintf("Frame %d", anim.frame_num + 1) );

   % Each frame may have a custom delay value, so multiple timeouts needed
   if (anim.playing) {
	gtk_timeout_remove(anim.playing);
	anim.playing = gtk_timeout_add(anim.frames[anim.frame_num].delay,
						&anim_play_timer, anim);
   }

}  % }}}

private define anim_pause(anim) % {{{
{
   if (not anim.playing)
   	return;

   gtk_timeout_remove(anim.playing);
   anim.playing = 0;
   gtk_widget_show(anim.play_button);
   gtk_widget_set_sensitive(anim.prev_button, TRUE);
   gtk_widget_set_sensitive(anim.next_button, TRUE);
   gtk_widget_hide(anim.pause_button);
}  % }}}

private define anim_play(anim) % {{{
{
   gtk_widget_show(anim.pause_button);
   gtk_widget_hide(anim.play_button);
   gtk_widget_set_sensitive(anim.prev_button, FALSE);
   gtk_widget_set_sensitive(anim.next_button, FALSE);
   anim.playing = gtk_timeout_add(anim.frames[0].delay, &anim_play_timer, anim);
}  % }}}

private define anim_draw_controls(hbox, anim)  % {{{
{
   gtk_container_add(hbox, anim.prev_button);
   gtk_container_add(hbox, anim.play_button);
   gtk_container_add(hbox, anim.pause_button);
   gtk_container_add(hbox, anim.next_button);

   g_signal_connect_swapped(anim.pause_button,"clicked", &anim_pause, anim);
   g_signal_connect_swapped(anim.play_button, "clicked", &anim_play,  anim);
   g_signal_connect_swapped(anim.next_button,"clicked", &anim_iterate,anim, 1);
   g_signal_connect_swapped(anim.prev_button,"clicked", &anim_iterate,anim, -1);
   _pop_n(4);
}  % }}}

private define anim_save_file(name, type, frames) % {{{
{
   variable frame, images = {}, delays = {};
   foreach frame (frames) {
	list_append(images, gdk_pixbuf_get_pixels(frame.pixbuf));
	list_append(delays, frame.delay);
   }

   anim_write(name, images; type=type, delay=delays);
} % }}}

private define anim_save_cb(ignore, ctx) % {{{
{
   variable name;
   variable anim = ctx.animation;
   variable was_playing = anim.playing;

   anim_pause(anim);

   (name, ) = _image_save_dialog(["gif"] ; kind="Animation");

   if (was_playing)
	anim_play(anim);

   if (name == NULL)
	return;

   anim_save_file(name, "gif", anim.frames);

}  % }}}

private define anim_display(ctx) % {{{
{
   variable pane = ctx.panes[0];
	
   if (not ctx.autosave)
	pane_make_window(pane);		% Only first frame gets toplevel window

   foreach pane (ctx.panes) {
	scale(pane);			% Subsequent frames rendered to toplevel
	flipflop(pane);			% window, scaled/transformed accordingly
   }

   if (ctx.autosave)
	return anim_save_file(ctx.file_name, "gif", ctx.panes);

   variable anim = @Animation;
   anim.frames = ctx.panes;
   anim.num_frames = ctx.npanes;
   anim.frame_num = 0;

   anim.prev_button = anim_make_button(GTK_STOCK_MEDIA_PREVIOUS);
   anim.play_button = anim_make_button(GTK_STOCK_MEDIA_PLAY);
   anim.pause_button = anim_make_button(GTK_STOCK_MEDIA_PAUSE);
   anim.next_button = anim_make_button(GTK_STOCK_MEDIA_NEXT);

   ctx.fill_rule = FILL_NONE;
   ctx.animation = anim;

   anim_play(anim);
}  % }}}

% }}}

% Control window {{{

private define help_cb(button) % {{{
{
   _info_window("Imdisplay Help", _get_slgtk_doc_string("imdisplay"));
} % }}}

private define controller_destroy(ctx) % {{{
{
   if (ctx.animate)
	anim_pause(ctx.animation);

   if (ctx.npanes) {
	variable p;
	foreach p (@ctx.panes)	% dup panes list, b/c destroy deletes them
	   if (p.win != NULL)
		gtk_widget_destroy(p.win);
   }
   gtk_main_quit();
} % }}}

private define controller_new(ctx) % {{{
{
   variable win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(win, sprintf("  imdisplay %s  ", version_string));
   gtk_window_set_resizable(win, FALSE);
   gtk_container_set_border_width(win, 2);
   () = g_signal_connect_swapped(win, "destroy", &controller_destroy, ctx);

   variable vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_add(win, vbox);

   variable hbox = gtk_hbox_new(FALSE, 5);
   gtk_box_pack_end(vbox, hbox, FALSE, FALSE, 0);

   variable button = gtk_button_new_with_label(" Help ");
   gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
   gtk_box_pack_start(hbox, button, FALSE, FALSE, 0);
   () = g_signal_connect(button, "clicked", &help_cb);

   ctx.revert_button = gtk_button_new_with_label("Revert");
   gtk_widget_unset_flags (ctx.revert_button, GTK_CAN_FOCUS);
   gtk_box_pack_start(hbox,ctx.revert_button, FALSE, FALSE, 0);
   () = g_signal_connect(ctx.revert_button,"clicked", &revert_cb, NULL, ctx);

   variable pane = ctx.panes[0];
   ctx.size_label = gtk_label_new(sprintf("%S x %S", pane.w, pane.h));
   gtk_box_pack_start(hbox, ctx.size_label, TRUE, FALSE, 0);

   ctx.save_button = gtk_button_new_with_label(" Save ");
   gtk_widget_unset_flags (ctx.save_button, GTK_CAN_FOCUS);
   gtk_box_pack_start(hbox,ctx.save_button,FALSE,FALSE,0);

   button = gtk_button_new_with_label(" Done ");
   gtk_box_pack_end(hbox, button, FALSE, FALSE, 0);
   gtk_widget_grab_focus(button);
   () = g_signal_connect_swapped(button, "clicked", &gtk_widget_destroy, win);

   hbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_end(vbox, hbox, FALSE, FALSE, 0);

   variable layout = ctx.layout;

   if (ctx.animate) {
	anim_draw_controls(hbox, ctx.animation);
	() = g_signal_connect(ctx.save_button, "clicked", &anim_save_cb, ctx);
   }
   else {

	still_draw_controls(hbox, ctx);

	if (ctx.npanes > 1) {

	   pane = ctx.panes[ctx.npanes/2];	% Select middle-ish pane 

	   if (ctx.layout == PANE_HORIZ)
		layout = PANE_VERT;
	   else
		layout = PANE_HORIZ;
	}

	() = g_signal_connect(ctx.save_button, "clicked", &still_save_cb, ctx);
   }

   ctx.controller = win;
   gtk_widget_show_all(win);

   if (ctx.animate)
	gtk_widget_hide(ctx.animation.play_button);

   % Chain controller to pane
   switch(layout)
   { case PANE_SINGLE: _gtk_window_chain(pane.win, win, 2);}
   { _gtk_window_chain(pane.win, win, layout); }

   pane_align(ctx.panes[0], 1);

} % }}}

% }}}

private define commence(ctx) % {{{
{
   % See if window placement is broken in window manager; done explicitly
   % here to avoid signal delivery delays with CPU-gobbling animations
   if (not ctx.autosave)
	__wm_placement_test();

   if (ctx.animate)
	anim_display(ctx);
   else 
   	still_display(ctx);

   if (ctx.autosave)
	return;

   controller_new(ctx);

   if (gtk_main_level() < 1)
	gtk_main();
} % }}}

define imdisplay() % {{{
{
   if (_NARGS == 0) {
	vmessage("imdisplay( FileName_or_ImageArray_or_Option [, ...])\n"+
	      "For detailed information, call as imdisplay(\"help\").");
	return;
   }

   if (Default_Context == NULL)
	variable ctx = default_context();
   else
	ctx = @Default_Context;

   ctx.panes = {};
   ctx.npanes = 0;

   variable n, q = __qualifiers();
   if (q != NULL)
	foreach n (get_struct_field_names(q))
	   parse_options(ctx, sprintf("%S=%S", n, get_struct_field(q, n)));

   foreach (__pop_args(_NARGS)) {
      	variable value = ().value;
	if (__is_same(value, "help")) {
   	   variable docs =  _get_slgtk_doc_string("imdisplay") + "\n" +
			    " Version: " + version_string;
	   vmessage("%S", docs);
	   return;
	}
	load(value, ctx, ctx.frame_delay);
   }

   if (ctx.npanes == 0)
	verror("No drawable images were found!\n");

   commence(ctx);

} % }}}

define imdisplay_defaults() % {{{
{
   if (_NARGS == 0 && __qualifiers == NULL) {
	Default_Context = NULL;
	return;
   }

   Default_Context = default_context();

   variable n, q = __qualifiers();
   if (q != NULL)
	foreach n (get_struct_field_names(q))
	   parse_options(Default_Context, sprintf("%S=%S",
	   					n, get_struct_field(q, n)));

   variable setting, settings = __pop_args(_NARGS);
   foreach setting ( settings )
	parse_options(Default_Context, setting.value);

} % }}}

provide("imdisplay");
