
/************************************************************************** 
 * rgba_to_indexed(): converts an RGB(A) image to an 8-bit image indexed with
 * a corresponding colormap.  The quantization is done with routines excerpted
 * from pbmplus (by Jef Poskanzer) via XV (by John Bradley) and adapted to:
 * make it self-contained; allow for transparency (i.e. both 24- AND 32-bit
 * may be quanitzed); remove fatal aborts (the code is used as a plugin 
 * module and by every means possible SHOULD NOT abort its parent app); allow
 * the colormap size to be adjusted; add fold markers; modernize function &
 * param declarations; and more cleanly integrate with the SLang interpreter
 * and Gif module (e.g. by replacing malloc with SLmalloc).  See below and the
 * bundled xv_copyright.h for pbmplus and XV authorship/copyright information.
 *
 * This software was developed at the MIT Kavli Institute for Astrophysics,
 * funded by the NASA AISRP grant NNG06GE58G.
 * 
 * The Massachusetts Institute of Technology makes no representations about
 * the suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 * 
 * THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
 * INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/* Declarations & Macros {{{ */

#define malloc 	SLmalloc	/* informs SLang interpreter of malloc errs */

/* ppmquant.c - quantize the colors in a pixmap down to a specified number
**
** Copyright (C) 1989, 1991 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

typedef uint8_t byte;
typedef struct { byte r, g, b; } Color;

#define PPM_GETR(p) ((p).r)
#define PPM_GETG(p) ((p).g)
#define PPM_GETB(p) ((p).b)

#define PPM_ASSIGN(p,red,grn,blu) \
  { (p).r = (red); (p).g = (grn); (p).b = (blu); }

#define PPM_EQUAL(p,q) ( (p).r == (q).r && (p).g == (q).g && (p).b == (q).b )

/* Color scaling macro -- to make writing ppmtowhatever easier. */

#define PPM_DEPTH(newp,p,oldmaxval,newmaxval) \
    PPM_ASSIGN( (newp), \
	        ((int) PPM_GETR(p)) * ((int)newmaxval) / ((int)oldmaxval), \
	        ((int) PPM_GETG(p)) * ((int)newmaxval) / ((int)oldmaxval), \
	        ((int) PPM_GETB(p)) * ((int)newmaxval) / ((int)oldmaxval) )

/* Luminance macro. */

/*
 * #define PPM_LUMIN(p) \
 *   ( 0.299 * PPM_GETR(p) + 0.587 * PPM_GETG(p) + 0.114 * PPM_GETB(p) )
 */

/* Luminance macro, using only integer ops.  Returns an int (*256)  JHB */
#define PPM_LUMIN(p) \
  ( 77 * PPM_GETR(p) + 150 * PPM_GETG(p) + 29 * PPM_GETB(p) )

/* Color histogram stuff. */

typedef struct chist_item* chist_vec;
struct chist_item { Color color; int value; };

typedef struct chist_list_item* chist_list;
struct chist_list_item { struct chist_item ch; chist_list next; };

typedef chist_list* chash_table;

typedef struct box* box_vector;
struct box { int index; int colors; int sum; };

#define MAXCOLORS 32767
#define HASH_SIZE 6553

#define ppm_hashpixel(p) ((((int) PPM_GETR(p) * 33023 +    \
			    (int) PPM_GETG(p) * 30013 +    \
			    (int) PPM_GETB(p) * 27011) & 0x7fffffff)   \
			  % HASH_SIZE)
/* }}} */

static int redcompare(const void *p1, const void *p2)  /* {{{ */
{
  return (int) PPM_GETR( ((chist_vec)p1)->color ) -
         (int) PPM_GETR( ((chist_vec)p2)->color );
}  /* }}} */

static int greencompare(const void *p1, const void *p2)  /* {{{ */
{
  return (int) PPM_GETG( ((chist_vec)p1)->color ) -
         (int) PPM_GETG( ((chist_vec)p2)->color );
}  /* }}} */

static int bluecompare(const void *p1, const void *p2)  /* {{{ */
{
  return (int) PPM_GETB( ((chist_vec)p1)->color ) -
         (int) PPM_GETB( ((chist_vec)p2)->color );
}  /* }}} */

static int sumcompare(const void *p1, const void *p2)  /* {{{ */
{
  return ((box_vector) p2)->sum - ((box_vector) p1)->sum;
}  /* }}} */

static chash_table ppm_allocchash(void)  /* {{{ */
{
  chash_table cht;
  int i;

  if ( !(cht = (chash_table) malloc( HASH_SIZE * sizeof(chist_list))) )
	return NULL;

  for (i=0; i<HASH_SIZE; i++ )
    cht[i] = (chist_list) 0;

  return cht;
} /* }}} */ 

static void ppm_freechash(chash_table cht)  /* {{{ */
{
   int i;
   chist_list chl, chlnext;

   for (i=0; i<HASH_SIZE; i++)
	for (chl = cht[i];  chl != (chist_list) 0; chl = chlnext) {
	    chlnext = chl->next;
	    free(chl);
    }

   free(cht);
} /* }}} */

static chist_vec mediancut(chist_vec chv, int colors, int sum,    /* {{{ */
					int newcolors, int maxval)
{
  /* Here is the fun part, the median-cut colormap generator.  This is based
     on Paul Heckbert's paper "Color Image Quantization for Frame Buffer
     Display", SIGGRAPH '82 Proceedings, page 297.  */

  chist_vec colormap;
  box_vector bv;
  register int bi, i;
  int boxes;

  bv = (box_vector) malloc(sizeof(struct box) * newcolors);
  colormap = (chist_vec) malloc(sizeof(struct chist_item) * newcolors );

  if (!bv || !colormap)
	return NULL;

  for (i=0; i<newcolors; i++)
    PPM_ASSIGN(colormap[i].color, 0, 0, 0);

  /* Set up the initial box */

  bv[0].index = 0;
  bv[0].colors = colors;
  bv[0].sum = sum;
  boxes = 1;

  /* Main loop: split boxes until we have enough */

  while ( boxes < newcolors ) {
    register int indx, clrs;
    int sm;
    register int minr, maxr, ming, maxg, minb, maxb, v;
    int halfsum, lowersum;

    /* Find the first splittable box */
    for (bi=0; bi<boxes && bv[bi].colors<2; bi++) ;
    if (bi == boxes) break;	/* ran out of colors! */

    indx = bv[bi].index;
    clrs = bv[bi].colors;
    sm = bv[bi].sum;

    /* Go through the box finding the minimum and maximum of each
       component - the boundaries of the box.  */
    minr = maxr = PPM_GETR( chv[indx].color );
    ming = maxg = PPM_GETG( chv[indx].color );
    minb = maxb = PPM_GETB( chv[indx].color );

    for (i=1; i<clrs; i++) {
      v = PPM_GETR( chv[indx + i].color );
      if (v < minr) minr = v;
      if (v > maxr) maxr = v;

      v = PPM_GETG( chv[indx + i].color );
      if (v < ming) ming = v;
      if (v > maxg) maxg = v;

      v = PPM_GETB( chv[indx + i].color );
      if (v < minb) minb = v;
      if (v > maxb) maxb = v;
    }

    /*
     ** Find the largest dimension, and sort by that component.  I have
     ** included two methods for determining the "largest" dimension;
     ** first by simply comparing the range in RGB space, and second
     ** by transforming into luminosities before the comparison.  You
     ** can switch which method is used by switching the commenting on
     ** the LARGE_ defines at the beginning of this source file.
     */
    {
      /* LARGE_LUM version */

      Color p;
      int rl, gl, bl;

      PPM_ASSIGN(p, maxr - minr, 0, 0);
      rl = PPM_LUMIN(p);

      PPM_ASSIGN(p, 0, maxg - ming, 0);
      gl = PPM_LUMIN(p);

      PPM_ASSIGN(p, 0, 0, maxb - minb);
      bl = PPM_LUMIN(p);

      if (rl >= gl && rl >= bl)
	qsort((char*) &(chv[indx]), (size_t) clrs, sizeof(struct chist_item),
	      redcompare );
      else if (gl >= bl)
	qsort((char*) &(chv[indx]), (size_t) clrs, sizeof(struct chist_item),
	      greencompare );
      else
	qsort((char*) &(chv[indx]), (size_t) clrs, sizeof(struct chist_item),
	      bluecompare );
    }

    /*
     ** Now find the median based on the counts, so that about half the
     ** pixels (not colors, pixels) are in each subdivision.
     */
    lowersum = chv[indx].value;
    halfsum = sm / 2;
    for (i=1; i<clrs-1; i++) {
      if (lowersum >= halfsum) break;
      lowersum += chv[indx + i].value;
    }

    /*
     ** Split the box, and sort to bring the biggest boxes to the top.
     */
    bv[bi].colors = i;
    bv[bi].sum = lowersum;
    bv[boxes].index = indx + i;
    bv[boxes].colors = clrs - i;
    bv[boxes].sum = sm - lowersum;
    ++boxes;
    qsort((char*) bv, (size_t) boxes, sizeof(struct box), sumcompare);
  }  /* while (boxes ... */

  /*
   ** Ok, we've got enough boxes.  Now choose a representative color for
   ** each box.  There are a number of possible ways to make this choice.
   ** One would be to choose the center of the box; this ignores any structure
   ** within the boxes.  Another method would be to average all the colors in
   ** the box - this is the method specified in Heckbert's paper.  A third
   ** method is to average all the pixels in the box.  You can switch which
   ** method is used by switching the commenting on the REP_ defines at
   ** the beginning of this source file.
   */

  for (bi=0; bi<boxes; bi++) {
    /* REP_AVERAGE_PIXELS version */
    register int indx = bv[bi].index;
    register int clrs = bv[bi].colors;
    register long r = 0, g = 0, b = 0, sum = 0;

    for (i=0; i<clrs; i++) {
      r += PPM_GETR( chv[indx + i].color ) * chv[indx + i].value;
      g += PPM_GETG( chv[indx + i].color ) * chv[indx + i].value;
      b += PPM_GETB( chv[indx + i].color ) * chv[indx + i].value;
      sum += chv[indx + i].value;
    }

    r = r / sum;  if (r>maxval) r = maxval;	/* avoid math errors */
    g = g / sum;  if (g>maxval) g = maxval;
    b = b / sum;  if (b>maxval) b = maxval;

    PPM_ASSIGN( colormap[bi].color, r, g, b );
  }

  free(bv);
  return colormap;
} /* }}} */

static chash_table ppm_computechash(Color** pixels, int cols,  /* {{{ */
			int rows, int maxcolors, int* colorsP)
{
  chash_table cht;
  register Color* pP;
  chist_list chl;
  int col, row, hash;

  if ( !(cht = ppm_allocchash()))
	return NULL;

  *colorsP = 0;

  /* Go through the entire image, building a hash table of colors. */
  for (row=0; row<rows; row++)
    for (col=0, pP=pixels[row];  col<cols;  col++, pP++) {

      hash = ppm_hashpixel(*pP);

      for (chl = cht[hash]; chl != (chist_list) 0; chl = chl->next)
	if (PPM_EQUAL(chl->ch.color, *pP)) break;

      if (chl == (chist_list) 0) {

	if ( (*colorsP)++ > maxcolors ||
	     !(chl = (chist_list) malloc(sizeof(struct chist_list_item)))) {
	   ppm_freechash(cht);
	   return NULL;
	}

	chl->ch.color = *pP;
	chl->ch.value = 1;
	chl->next = cht[hash];
	cht[hash] = chl;
      }
      else
	++(chl->ch.value);
    }

  return cht;
} /* }}} */

static chist_vec ppm_chashtochist(chash_table cht, int maxcolors) /* {{{ */
{
  chist_vec chv;
  chist_list chl;
  int i, j;

  /* Now collate the hash table into a simple chist array. */
  if ( !(chv = (chist_vec) malloc( maxcolors * sizeof(struct chist_item) )))
	return NULL;

  /* (Leave room for expansion by caller.) */

  /* Loop through the hash table. */
  j = 0;
  for (i=0; i<HASH_SIZE; i++)
    for (chl = cht[i];  chl != (chist_list) 0;  chl = chl->next) {
      /* Add the new entry. */
      chv[j] = chl->ch;
      ++j;
    }

  return chv;
} /* }}} */

static chist_vec ppm_computechist(Color** pixels, int cols, 	/*{{{*/
			int rows, int maxcolors, int* colorsP)
{
  chash_table cht;
  chist_vec chv;

  if ( !(cht = ppm_computechash(pixels, cols, rows, maxcolors, colorsP)))
	return NULL;

  chv = ppm_chashtochist(cht, maxcolors);
  ppm_freechash(cht);
  return chv;
} /* }}} */

static int ppm_quant(byte *rgba, byte* indexed, int cols, int rows, /*{{{*/
	int *transparent_index, byte *r,  byte *g, byte *b, int *colmap_size)
{
  register Color    *pP, *transparent_pixel;
  register int      col, limitcol;
  register int      index;
  register Color**  pixels;
  register int	    i, row, rtnval;
  register int	    nchannels, has_transparency;
  byte              maxval, newmaxval;
  int               numcolors;
  chash_table       cht = NULL;
  chist_vec         chv, colormap = NULL;

  transparent_pixel = NULL;
  row = rtnval = index = 0;
  maxval = 255;
  has_transparency = *transparent_index;

  nchannels = 3;
  if (has_transparency)
	nchannels++;
  else
	*transparent_index = -1;		/* per GIF spec */

  /* reformat RGB 24-bit or RGBA 32-bit image into 2D array of pixel structs */

  if (!(pixels = (Color**) malloc(rows * sizeof(Color*))))
	goto finished;

  for (row=0; row<rows; row++) {

    if ( !(pixels[row] = (Color*) malloc(cols * sizeof(Color))))
	goto finished;

    for (col=0, pP=pixels[row]; col<cols; col++, pP++) {

	pP->r = rgba[0];
	pP->g = rgba[1];
	pP->b = rgba[2];

        /* An arbitrary RGBA image can have one of 256 levels of transparency
         assigned to each pixel, but with GIF only 1 color may be designated
	 as transparent, and it must be either fully opaque (255) or fully
	 transparent (0).  We assign a transparent color to the output image
	 by selecting from the first RGBA color with an opacity of zero */
	if (has_transparency && rgba[3] == 0 && !transparent_pixel)
	   transparent_pixel = pP;

      rgba += nchannels;
    }
  }

  /*  attempt to make a histogram of the colors, unclustered.
   *  If at first we don't succeed, lower maxval to increase color
   *  coherence and try again.  This will eventually terminate, with
   *  maxval at worst 15, since 32^3 is approximately MAXCOLORS.  */

  for ( ; ; ) {

    chv = ppm_computechist(pixels, cols, rows, MAXCOLORS, &numcolors);
    if (chv != (chist_vec) 0) break;

    newmaxval = maxval / 2;

    for (row=0; row<rows; ++row)
      for (col=0, pP=pixels[row]; col<cols; ++col, ++pP)
	PPM_DEPTH( *pP, *pP, maxval, newmaxval );

    maxval = newmaxval;
  }

  /* Step 3: apply median-cut to histogram, making the new colormap    */
  if (numcolors > *colmap_size)
	numcolors = *colmap_size;	/* what else can we do? */

  colormap = mediancut(chv, numcolors, rows * cols, maxval, *colmap_size);
  if (colormap == NULL)
	goto finished;

  free(chv);
  *colmap_size = numcolors;

  /* Step 4: map the colors in the image to their closest match
     in the new colormap, and write 'em out */

  if ( !(cht = ppm_allocchash()) )
	goto finished;

  for (row = 0;  row < rows;  ++row) {

    col = 0;  limitcol = cols;  pP = pixels[row];

    do {
      int hash;
      chist_list chl;

      /* Check hash table to see if we have already matched this color. */

      hash = ppm_hashpixel(*pP);
      for (chl = cht[hash];  chl;  chl = chl->next)
	if (PPM_EQUAL(chl->ch.color, *pP)) {index = chl->ch.value; break;}

      if (!chl /*index = -1*/) {/* No; search colormap for closest match. */
	register int r1, g1, b1, r2, g2, b2;
	register long dist, newdist;

	r1 = PPM_GETR( *pP );
	g1 = PPM_GETG( *pP );
	b1 = PPM_GETB( *pP );
	dist = 2000000000;

	for (i=0; i < numcolors; i++) {
	  r2 = PPM_GETR( colormap[i].color );
	  g2 = PPM_GETG( colormap[i].color );
	  b2 = PPM_GETB( colormap[i].color );

	  newdist = ( r1 - r2 ) * ( r1 - r2 ) +
	            ( g1 - g2 ) * ( g1 - g2 ) +
	            ( b1 - b2 ) * ( b1 - b2 );

	  if (newdist<dist) { index = i;  dist = newdist; }
	}

	hash = ppm_hashpixel(*pP);
	if ( !(chl = (chist_list) malloc(sizeof(struct chist_list_item))))
	   goto finished;

	chl->ch.color = *pP;
	chl->ch.value = index;
	chl->next = cht[hash];
	cht[hash] = chl;
      }

      *indexed++ = index;

      ++col;
      ++pP;
    }
    while (col != limitcol);
  }

  /* rescale the colormap and load it back into the r, g, b arrays */
  for (i=0; i < numcolors; i++) {
    PPM_DEPTH(colormap[i].color, colormap[i].color, maxval, 255);
    r[i] = PPM_GETR( colormap[i].color );
    g[i] = PPM_GETG( colormap[i].color );
    b[i] = PPM_GETB( colormap[i].color );

    if (transparent_pixel &&
    		transparent_pixel->r == r[i] &&
		transparent_pixel->g == g[i] &&
		transparent_pixel->b == b[i]) {

	if (rgba[i*nchannels+3] == 0) {
		*transparent_index = i;
		/*transparent_pixel = NULL;*/
	}
    }
  }

  rtnval = 1;

  finished:

  /* free the pixels array */
  if (pixels) {
	for (i=0; i<rows; i++)
	    free(pixels[i]);
	free(pixels);
  }

  /* free cht and colormap */
  free(colormap);
  ppm_freechash(cht);

  return rtnval;
} /* }}} */

static byte* rgba_to_indexed(byte *rgba, uint16_t w, uint16_t h,   /* {{{ */
				int *transparent, Gif_Colormap **cmap)
{
   byte red[256];
   byte green[256];
   byte blue[256];
   int ncolor = 256;
   byte *indexed = 0;
   register Gif_Colormap *cm;

   ncolor = 256;
   if (!rgba || !(indexed = (byte *) malloc((size_t) (w * h))))
	return NULL;

  if (!ppm_quant(rgba, indexed, w, h, transparent, red, green, blue, &ncolor)
			|| !(cm = Gif_NewFullColormap(ncolor, ncolor))) {
	free(indexed);
	return NULL;
   }	

   while (ncolor--) {
	cm->col[ncolor].red = red[ncolor];
	cm->col[ncolor].green = green[ncolor];
	cm->col[ncolor].blue = blue[ncolor];
   }

   *cmap = cm;
   return indexed;
}  /* }}} */
