
/************************************************************************** 
 * gifxtns.c:  extensions to gifsicle to support S-Lang gif module.
 *
 * This software was developed at the MIT Kavli Institute for Astrophysics,
 * funded by the NASA AISRP grant NNG06GE58G.
 * 
 * The Massachusetts Institute of Technology makes no representations about
 * the suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 * 
 * THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
 * INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE. 
 **************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <lcdfgif/gif.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "gifxtns.h"

Gif_RGBA* Gif_NewRGBAFull(uint8_t *pixels, uint16_t w, uint16_t h, /*{{{ */
						uint16_t nchannels)
{
   Gif_RGBA *gfr = Gif_New(Gif_RGBA);
   if (!gfr) return 0;
   gfr->pixels = pixels;
   gfr->width = w;
   gfr->height = h;
   gfr->nchannels = nchannels;
   gfr->refcount = 0;
   return gfr;
}  /* }}} */

Gif_RGBA* Gif_NewRGBA(void)  /* {{{ */
{
   return Gif_NewRGBAFull(0, 0, 0, 0);
}  /* }}} */

void Gif_DeleteRGBA(Gif_RGBA* gfr) /* {{{ */
{
   if (!gfr) return;
   if (--gfr->refcount > 0) return;

   Gif_DeleteColormap(gfr->cmap);
   Gif_DeleteArray(gfr->pixels);
   Gif_Delete(gfr);

}  /* }}} */

static int is_greyscale(Gif_Colormap *cmap) /*{{{*/
{
   register int i;
   register Gif_Color *col;

   for (i=0; i < cmap->ncol; i++) {
	col = cmap->col + i;
	if (col->red != col->green ||
	    col->red != col->blue  ||
	    col->green != col->blue)
		return 0;
   }	
   return 1;
} /*}}}*/

Gif_RGBA* Gif_ImageToRGBA(Gif_Image *gfi, Gif_Colormap *global_cmap) /*{{{*/
{
   Gif_RGBA *gfr;
   Gif_Colormap *cmap;
   register Gif_Color *color;
   register uint8_t *rgb;
   register uint8_t *gif;
   register Gif_Color *colors;
   register unsigned long pixel;
   register unsigned long npixels;
   register short transparent = -1;

   if (!gfi || !(gfr = Gif_NewRGBAFull(0, gfi->width, gfi->height, 0)))
   	return 0;

   if (!(cmap = gfi->local) && !(cmap = global_cmap))
	return 0;

   if ( (transparent = gfi->transparent) < 0) {
	if (is_greyscale(cmap))
	   gfr->nchannels = 1;
	else
	   gfr->nchannels = 3;
   }
   else
	gfr->nchannels = 4;

   npixels  = gfr->width * gfr->height;

   if (!(gfr->pixels = Gif_NewArray(uint8_t, npixels * gfr->nchannels))) {
	Gif_Delete(gfr);
	return 0;
   }

   pixel = 0;
   rgb = gfr->pixels;
   gif = gfi->image_data;
   colors = cmap->col;

   if (gfr->nchannels == 1) {

	do {
	  color = colors + gif[pixel];
	  *rgb++ = color->red;		   /* greyscale implies r == g == b */
	} while (++pixel < npixels);
	 
   }
   else if (gfr->nchannels == 3) {

	do {
	  color = colors + gif[pixel];
	  rgb[0] = color->red;
	  rgb[1] = color->green;
	  rgb[2] = color->blue;
	  rgb += 3;
	} while (++pixel < npixels);
   }
   else do {

	  color = colors + gif[pixel];
	  rgb[0] = color->red;
	  rgb[1] = color->green;
	  rgb[2] = color->blue;

	  /* Unlike other formats, say PNG, the alpha channel in a GIF does
	     not indicate the % of transparency, but only on/off; that is,
	     GIF pixels can ONLY be fully opaque OR fully transparent. */

	  if (gif[pixel] == transparent)
		rgb[3] = 0;		   /* transparent ==> opacity = 0 */
	  else
		rgb[3] = 255;		   /* otherwise fully opaque */

	  rgb += 4;

   } while (++pixel < npixels);

   cmap->refcount++;		/* keep colormap to preserve transparency idx */
   gfr->cmap = cmap;
   gfr->refcount++;
   return gfr;
}  /* }}} */

#include "quantize.c"

static Gif_Colormap* get_grey_colormap(void) /*{{{*/
{
   register int i;
   static Gif_Colormap *cmap;
   register Gif_Color *color;

   if (cmap)
	return cmap;

   if (!(cmap = Gif_NewFullColormap(256, 256)))
	return 0;

   for (i=0; i<256; i++) {
	color = cmap->col + i;
	color->red = color->blue = color->green = i;
	color->pixel = color->haspixel = 0;
   }

   cmap->refcount = 1;

   return cmap;
} /*}}}*/

Gif_Image* Gif_ImageFromRGBABuf(uint8_t *rgba, uint16_t w, uint16_t h,  /*{{{*/
							int nchannels)
{
   Gif_Image *gfi = 0;
   Gif_Colormap *cmap = 0;
   register uint8_t *pixels = 0;

   if (!rgba || !(gfi = Gif_NewImage()))
	return 0;

   gfi->width = w;
   gfi->height = h;

   if (nchannels == 1) {

	register unsigned long pixel = 0;
	register unsigned long npixels = w * h;

	if ( !(pixels = Gif_NewArray(uint8_t, w * h * nchannels)))
	   goto fail;

	if ( !(cmap = get_grey_colormap()))
	   goto fail;

	cmap->refcount++;

      	do {
	   pixels[pixel] = rgba[pixel];
	} while (++pixel < npixels);

   }
   else {
	int transparent = (nchannels == 4);
	if (!(pixels = rgba_to_indexed(rgba, w, h, &transparent, &cmap)))
	   goto fail;
	gfi->transparent = transparent;
   }

   if (! Gif_SetUncompressedImage(gfi, pixels, free, 0))
	goto fail;

   gfi->local = cmap;	/* FIXME: local cmap for now, but merge to global */
   return gfi;		/* refcount should be incremented by caller */

   fail:
   Gif_DeleteImage(gfi);
   Gif_Delete(pixels);
   return 0;
}  /* }}} */

Gif_Image* Gif_ImageFromRGBA(Gif_RGBA *gfr) /*{{{*/
{
   if (!gfr)
	return 0;

   return Gif_ImageFromRGBABuf(gfr->pixels, gfr->width, gfr->height,
   							gfr->nchannels);
}  /* }}} */

Gif_RGBA * Gif_GetRGBA(Gif_Stream *gfs, int which)  /* {{{ */
{
   Gif_Image *gfi;
   Gif_Colormap *cmap;

   if (!gfs || which < 0 || which >= gfs->nimages)
	return 0;

   gfi = gfs->images[which];
   if (gfi->local)
	cmap = gfi->local;
   else
	cmap = gfs->global;

   return Gif_ImageToRGBA(gfi, cmap);
} /*}}}*/

Gif_Colormap*  Gif_GetGlobalColormap(Gif_Stream *gfs) /*{{{*/
{
   if (!gfs)
	return 0;

   if (gfs->global)
	gfs->global->refcount++;
   return gfs->global;
} /*}}}*/

#include "reflect.c"

void Gif_ImageSetDelay(Gif_Image *gfi, uint16_t delay)  /*{{{*/
{
   if (gfi)
	gfi->delay = delay;
}  /* }}} */

uint16_t Gif_ImageGetDelay(Gif_Image *gfi)  /*{{{*/
{
   if (gfi)
	return gfi->delay;
    return 0;
}  /* }}} */

void Gif_ImageRef(Gif_Image *gfi) /*{{{*/
{
   if (gfi) gfi->refcount++;
} /*}}}*/

#ifdef __cplusplus
}
#endif
