
%  SLgif: A GIF image I/O module for S-Lang {{{
%
%  Copyright (C) 2010 Massachusetts Institute of Technology
%  Michael S. Noble <mnoble@space.mit.edu>
%
%  This software was developed at the MIT Kavli Institute for Astrophysics,
%  funded by the NASA AISRP grant NNG06GE58G.
% 
%  Permission to use, copy, modify, distribute, and sell this software
%  and its documentation for any purpose is hereby granted without fee,
%  provided that the above copyright notice appear in all copies and
%  that both that copyright notice and this permission notice appear in
%  the supporting documentation, and that the name of the Massachusetts
%  Institute of Technology not be used in advertising or publicity
%  pertaining to distribution of the software without specific, written
%  prior permission.  The Massachusetts Institute of Technology makes
%  no representations about the suitability of this software for any
%  purpose.  It is provided "as is" without express or implied warranty.
%  
%  THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
%  WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
%  MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
%  INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
%  CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
%  OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
%  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
%  WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
% }}}

import("gif");

private variable verbose = 0;

define gif_set_verbose() % {{{
{
   if (_NARGS != 1)
	usage("gif_set_verbose(int_verbosity_value);");

   verbose = int(());
} % }}}

define gif_get_verbose() % {{{
{
   return verbose;
} % }}}

private define emsg() % {{{
{
   variable args = __pop_list(_NARGS);

   if (not verbose) return;

   variable msg = sprintf(__push_list(args));
   () = fprintf(stderr, msg + "\n");
   () = fflush(stderr);
} % }}}

#ifndef list_concat
private define list_concat(a, b) % {{{
{
   if (length(a))
	a = @a;
   else
	a = {};		% work around slang 2.1 bug for empty lists

   foreach b (b)
	list_append(a, b);

   return a;
} % }}}
#endif

private define make_frame_list(framenums, nframes); % {{{
private define make_frame_list(framenums, nframes)
{
   variable fr, frames = {};

   if (any(framenums < 0) || nframes == 0)
	return NULL;

   foreach fr (framenums) {
	if (fr == 0)
	   frames = list_concat(frames, make_frame_list([1:nframes], nframes));
	else
	   list_append(frames, fr - 1);
   }

   return frames;
} % }}}

private define get_frames(gif_stream, frames, _delays) % {{{
{
   variable i = Gif_ImageCount(gif_stream);

   frames = make_frame_list(frames, i);
   if (frames == NULL)
	return NULL;

   variable result = {}, delays = {};
   variable cmap = Gif_GetGlobalColormap(gif_stream);

   foreach i (frames) {
	variable gif_image = Gif_GetImage(gif_stream, i);
	list_append(result, Gif_ImageToRGBA(gif_image, cmap));
	list_append(delays, Gif_ImageGetDelay(gif_image));
   }

   frames = list_to_array(result);

   % GIF frame delays are specified in centisec, but we return millisec 
   % for cleaner integration with other use cases (e.g. Gtk, Xt/X11, Qt)
   delays = list_to_array(delays) * 10;
   delays [ where (delays >= USHORT_MAX) ] = USHORT_MAX;
   delays = typecast(delays, UShort_Type);

   if (length(frames) == 1) {
	frames = frames[0];
	delays = delays[0];
   }
   else if (all(delays == delays[0]))
	delays = delays[0];

   if (typeof(_delays) == Ref_Type)
	   @_delays = delays;

   return frames;

} % }}}

private define file_to_fp(file, mode)  % {{{
{
   switch(typeof(file))
   { case File_Type   : return file; }
   { case FD_Type     : return fdopen(file, mode + "b"); }
   { case String_Type : return fopen(file, mode + "b");  }
   { usage("Unsupported file type: $file\n"$); }
}  % }}}

define gif_read() % {{{
{
   variable file, frames;
   variable delays = qualifier("delay", NULL);
   switch (_NARGS)
   { case 1 : file = (); frames = 0; }
   { case 2 : (file, frames) = ();   }
   { usage("image(s) = gif_read(file [,framenum=Int_or_IntArray]"+
						" [; delays=reference] )"); }
   file = file_to_fp(file, "r");

   variable gfs = Gif_ReadFile(file);

   if (gfs == NULL)
	return NULL, emsg("Unable to open/read $file as GIF"$);

   frames = get_frames(gfs, frames, delays);
   if (frames == NULL)
	return NULL, emsg("Frame nums must be >= 1, or 0 for all frames");

   return frames;
} % }}}

private define has_alpha(rgba_image) % {{{
{
   % Unlike other formats, say PNG, the alpha channel in a GIF image
   % does not indicate the % of transparency, but rather only on/off
   % That is, GIF pixels can ONLY be fully opaque OR fully transparent.

   variable dims = array_shape(rgba_image);
   if (length(dims) == 3 && dims[0] == 4)
	return 1;
   return 0;
} % }}}

define gif_has_alpha() % {{{
{
   variable image, result;
   switch(_NARGS)
   { case 1 : image = (); }
   { usage("gif_has_alpha(gif_rgba_image)"); }

   if (_typeof(image) == Array_Type)
       result = array_map(UChar_Type, &has_alpha, [image]);
   else
       result = has_alpha(image);

   return result;
} % }}}

define gif_normalize() % {{{
{
   % Transform input image into one where each pixel has a value
   % in the range (0 < intensity < nshades), and nshades <= 256.

   switch (_NARGS)
   { case 1:  variable in = ();  variable nshades = 256; }
   { case 2:  (in , nshades) = (); }
   { usage("normed_image = gif_normalize(input_image[, num_shades=256])"); }

   if ((_typeof (in) == UChar_Type) and (nshades == 256))
     return in;

   nshades = min([nshades, 256]) - 1;

   variable g0, g1;
   variable is_bad = isnan(in) or isinf(in);
   variable any_is_bad = any(is_bad);
   if (any_is_bad) {
	variable good = in[where(is_bad == 0)];
	g0 = min (good);
	g1 = max (good);
   }
   else {
	g0 = min(in);
	g1 = max(in);
   }

   if (g0 != g1) {
	variable factor = nshades / double(g1-g0);
	variable out = typecast ((in - g0)*factor, UChar_Type);
	out[where (out > nshades)] = nshades;
   }
   else
     out = typecast (in*0 + 127, UChar_Type);

   if (any_is_bad)
     out[where(is_bad)] = 0;

   return out;
}

% }}}

private define im2gif(gfs, image, flipflop, delay) % {{{
{
   image = gif_normalize(image, 256);
   if (flipflop)
	gif_reflect(image, flipflop);

   if (length(array_shape(image)) > 3)
	error("Image arrays must be less than 4D");

   variable gfi = Gif_ImageFromRGBABuf(image);

   delay /= 10;			% convert millisec to centisec for GIF format

   if (delay < 0)
	delay = 0;
   else if (delay > USHORT_MAX)
	delay = USHORT_MAX;

   delay = typecast(delay, UShort_Type);
   Gif_ImageSetDelay(gfi, delay);

   if (not Gif_AddImage(gfs, gfi))
	error("Could not create/add gif image");

} % }}}

define gif_write() % {{{
{
   % Write greyscale, RGB, or RGBA images to a GIF file

   variable file, image;
   variable err = qualifier("err", struct {error=0, descr="no error"});
   variable delay = qualifier("delay", 500);
   variable flip = qualifier_exists("flip");	% mirror around Y = height/2
   variable flop = qualifier_exists("flop");	% mirror around X = width/2

   switch(_NARGS)
   { case 2: (file, image) = (); }
   { usage("gif_write(file, image(s) [ ; delay=Int_or_IntArray, flip, flop]"); }

   file = file_to_fp(file, "w");
   if (file == NULL)
	error("Could not open file: errno = $errno"$);

   variable gfs = Gif_NewStream();
 
   % These input combinations will result in animations being written to file:
   % 	List_Type[N]
   %	Array_Type[N]
   % 	3D images with an explicit delay= qualifier

   if (typeof(image) == List_Type)
	image = list_to_array(image);
   else if (typeof(image) == Array_Type) {
	variable i = array_shape(image);
	if (length(i) == 3 && qualifier_exists("delay")) {
	   i = i[0];
	   variable unrolled = Array_Type[i];
	   _for i (0, i-1, 1)
		unrolled[i] = image[i, *, *];
	   image = unrolled;
	}
   }

   if (typeof(delay) == List_Type)
	delay = list_to_array(delay);
   else
	delay = [ delay ];

   if (_typeof(image) == Array_Type)
	array_map(Void_Type, &im2gif, gfs, image, flop|(flip << 1), delay);
   else
	im2gif(gfs, image, flop | (flip << 1), delay[0]);

   if (1 != Gif_WriteFile(gfs, file))
	error("Could not write GIF image to file : $file"$);

   () = fflush(file);

} % }}}

% Convenience functions?
% 
%  gif_width()
%  gif_height()
%  gif_get_red()
%  gif_get_blue()
%  gif_get_green()
%  gif_get_alpha()

$1 = path_concat (path_dirname (__FILE__), "../help/giffuns.hlp");
if (NULL != stat_file ($1))
  add_doc_file ($1);

provide("gif");
