
require("gtkplot");

% Front matter (structure and variable definitions) {{{

static variable Scale_Funcs = [ &gtk_plot_set_xscale, &gtk_plot_set_yscale];
static variable Range_Funcs = [ &_gtk_plot_set_xrange, &_gtk_plot_set_yrange];

public variable RGBLimits = struct { r0, r1, g0, g1, b0, b1 };
static variable Slider = struct {
	values,
	lneighbor,
	rneighbor,
	widget,
	parent,
	expose_handler,
	grid,
	nbins,
	inverted};

static variable Region = struct {
   	color,
	lslider,
	rslider,
	sliderbox,
	canvas,
	widget,
	sd};

static variable SelectorData = struct { 
	x_non_zeroes,
	angstroms,
	grids,
	histograms,
	scale,				% x/y log/lin
	rgblims,
	single_image,
	plots,
	plot_tabs,
	current_plot,
	plots_area,
	apply_cb,
	apply_cb_args,
	sliders,
	slider_vbox,
	dirty};
% }}}

% Miscellaneous {{{
static define make_radiob()
{
   variable box, widget, label, callback, callback_args;
   callback_args = __pop_args (_NARGS - 4);
   (box, widget, label, callback) = ();
   
   variable rb = gtk_radio_button_new_with_label_from_widget (widget, label);
   () = g_signal_connect (rb, "clicked", callback, __push_args(callback_args));
   gtk_box_pack_start (box, rb, 1, 1, 0);

   return rb;
}

static define make_aligned_button()
{
   variable vbox, text, cb, cbargs = __pop_args(_NARGS - 3);
   (vbox, text, cb) = ();

   variable align = gtk_alignment_new(0.5, 0.5, 0, 0);
   gtk_box_pack_start(vbox,align,FALSE,FALSE,5);
   variable button = gtk_button_new_with_label(text);
   gtk_container_add(align,button);
   if (cb != NULL)
      () = g_signal_connect_swapped(button,"clicked", cb, __push_args(cbargs) );
   else
      gtk_widget_set_sensitive(button,FALSE);
}

#ifnexists hist_bsearch
static define hist_bsearch (x, xs)
{
   variable n0 = 0;
   variable n = length (xs);
   variable n1 = n;
   
   while (n1 > n0 + 1)
     {
	variable n2 = (n1 + n0)/2;
	if (xs[n2] >= x)
	  {
	     n1 = n2;
	     continue;
	  }
	n0 = n2;
     }
   return n0;
}
#endif

static define index_of(X, value)
{
   variable i = hist_bsearch(value, X);
   if (X[i] != value) i++;
   return i;
}

static define data_to_canvas(pd,dx,dy)
{
   % Map values in (input) data space to positions in canvas (drawing) space
   variable px, py, cx, cy;
   gtk_plot_get_pixel(pd.plot, dx, dy, &px, &py);
   gtk_plot_canvas_get_position(pd.canvas, int(px), int(py), &cx, &cy);
   return (cx,cy);
}
% }}}

% Regions and sliders (region controllers) {{{
static define region_draw(r)
{
   if (r.widget != NULL) {
	() = gtk_plot_canvas_remove_child(r.canvas,r.widget);
	r.widget = NULL;
   }

   variable span = [ int ( gtk_range_get_value (r.lslider.widget)) :
			int( gtk_range_get_value(r.rslider.widget)) ];
   if (length(span) == 1) return;

   variable pd = r.sd.current_plot;
   variable yaxis = pd.dsets[0].axes[1], x = pd.dsets[0].axes[0].data;
   variable rx = [ x[ span[0] ],  x[span], x[ span[-1] ] ];
   variable ry = [ yaxis.min,  yaxis.data[span], yaxis.min ];

   % Would like to use array_map, but can't
   _for(0,length(rx)-1,1) {
	variable i = ();
	(rx[i], ry[i]) = data_to_canvas(pd,rx[i],ry[i]);
   }

   r.canvas = pd.canvas;
   r.widget = _gtk_plot_canvas_put_polygon(r.canvas, rx, ry,
	 		GTK_PLOT_LINE_SOLID, 0, gdk_black,  r.color, TRUE);
}

static define update_regions(sd)
{
   foreach (sd.current_plot.user_data)
	region_draw(());
   _gtk_plot_redraw(sd.current_plot);
}

static define update_slider(slider, sd)
{
   if (sd.single_image) {
	variable value = int(gtk_range_get_value(slider.widget));

	variable neighbor = sd.sliders[slider.lneighbor];
	if (neighbor != NULL) {
	  if (value <= int(gtk_range_get_value(neighbor.widget)))
		gtk_range_set_value(neighbor.widget,value);
	}

	neighbor = sd.sliders[slider.rneighbor];
	if (neighbor != NULL) {
	   if (value >= int(gtk_range_get_value(neighbor.widget)))
		gtk_range_set_value(neighbor.widget,value);
	}
   }

   update_regions(sd);
}

static define format_slider_value(widget, index, sd)
{
   variable X = sd.current_plot.dsets[0].axes[0].data;
   return sprintf("%3.2f", X[ int(index) ]);
}

static define make_sd_dirty(widget, sd) { sd.dirty = 1; }

static define set_all_slider_values(sd,r0,r1,g0,g1,b0,b1)
{
   gtk_range_set_value(sd.sliders[1].widget, r0);
   gtk_range_set_value(sd.sliders[2].widget, r1);
   gtk_range_set_value(sd.sliders[3].widget, g0);
   gtk_range_set_value(sd.sliders[4].widget, g1);
   gtk_range_set_value(sd.sliders[5].widget, b0);
   gtk_range_set_value(sd.sliders[6].widget, b1);
   update_regions(sd);
}

static define slider_expose(event, slider, sd) {
   % This handler ensures plots exist before format_value is first called
   () = g_signal_connect(slider.widget,"format_value", &format_slider_value,sd);
   () = g_signal_connect(slider.widget,"value_changed", &make_sd_dirty, sd);
   g_signal_handler_disconnect(slider.widget, slider.expose_handler);
   return FALSE;
}

static define make_slider(sd, grid, parent, which, value, lneighbor, rneighbor)
{
   variable s	= @Slider;
   s.parent	= parent;
   s.inverted	= FALSE;
   s.grid	= grid;
   s.nbins	= length(grid);

   if (sd.single_image) {
	s.lneighbor = lneighbor;
	s.rneighbor = rneighbor;
   }

   value = index_of(grid, value);
   variable adj = gtk_adjustment_new(value, 0, s.nbins - 1, 1, 1, 0.0);
   s.widget = gtk_hscale_new(adj);
   s.expose_handler = g_signal_connect_swapped(s.widget,"expose_event",
	 					&slider_expose, s, sd);
   gtk_container_add(parent, s.widget);
   sd.sliders[which] = s;
   return s;
}

static define make_region(sd, grid, color, left, right, lval, rval)
{ 
   if (sd.sliders == NULL) {
	sd.slider_vbox = gtk_vbox_new(FALSE,10);
	sd.sliders = Struct_Type[8];
	sd.sliders[0] = NULL;
	sd.sliders[7] = NULL;
   }

   variable box = gtk_hbox_new(FALSE,10);
   gtk_container_add(sd.slider_vbox, box);

   variable r = @Region;
   r.color	= color;
   r.sd         = sd;
   r.lslider	= make_slider(sd, grid, box, left, lval, left-1, left+1);
   r.rslider	= make_slider(sd, grid, box, right, rval, right-1, right+1);
   r.sliderbox	= box;

   gtk_widget_modify_bg(r.lslider.widget,GTK_STATE_NORMAL,color);
   gtk_widget_modify_bg(r.rslider.widget,GTK_STATE_NORMAL,color);
   () = g_signal_connect_swapped(r.lslider.widget,"value-changed",
					&update_slider, r.lslider, sd);
   () = g_signal_connect_swapped(r.rslider.widget,"value-changed",
  					&update_slider, r.rslider, sd);

   return r;
}

static define slider_to_rgb(slider)
{
   return slider.grid [ int( gtk_range_get_value(slider.widget) ) ];
}
static define set_rgb_limits(rgblims, sd)
{
   rgblims.r0 = slider_to_rgb(sd.sliders[1]);
   rgblims.r1 = slider_to_rgb(sd.sliders[2]);
   rgblims.g0 = slider_to_rgb(sd.sliders[3]);
   rgblims.g1 = slider_to_rgb(sd.sliders[4]);
   rgblims.b0 = slider_to_rgb(sd.sliders[5]);
   rgblims.b1 = slider_to_rgb(sd.sliders[6]);
}

static define equalize(sd)
{
   % Compute total area of spectrum covered by color regions,
   % then assign each color region an equal slice. 

   variable rmin = int(gtk_range_get_value(sd.sliders[1].widget));
   variable bmax = int(gtk_range_get_value(sd.sliders[6].widget));
   if ( bmax - rmin <= 3) {
	rmin = 0;
	bmax = sd.sliders[1].nbins - 1;
   }

   variable counts = sd.histograms[0][ [ rmin : bmax ] ];
   variable total_color = cumsum(counts), slice = total_color [-1] / 3.0;

   variable rmax = rmin + hist_bsearch(slice, total_color);
   variable gmax = rmin + hist_bsearch(2*slice, total_color);

   set_all_slider_values(sd,rmin,rmax,rmax,gmax,gmax,bmax);
}

static define revert(sd)
{
   variable dirty = sd.dirty;
   variable origlims = sd.rgblims, newlims = @RGBLimits;
   set_rgb_limits(newlims, sd);

  % Avoid superfluous redraws by explicitly checking limits
  if (orelse {newlims.r0 != origlims.r0} {newlims.r1 != origlims.r1}
  	     {newlims.g0 != origlims.g0} {newlims.g1 != origlims.g1}
  	     {newlims.b0 != origlims.b0} {newlims.b1 != origlims.b1}) {

	variable s = sd.sliders, r = s[1].grid, g = s[3].grid, b= s[5].grid;

	set_all_slider_values(sd,
		index_of(r, origlims.r0), index_of(r, origlims.r1),
		index_of(g, origlims.g0), index_of(g, origlims.g1),
		index_of(b, origlims.b0), index_of(b, origlims.b1));
   }

   if (dirty) sd.dirty = 0;
}

static define cancel(win, sd) { sd.dirty = 0; gtk_widget_destroy(win); }

static define apply(win, sd, dismiss)
{
   if (dismiss) {
	if (sd.dirty)
	   set_rgb_limits(sd.rgblims, sd);
	gtk_widget_destroy(win);
	gtk_main_quit();
	return;
   }

   if (andelse {sd.apply_cb != NULL} {sd.dirty} ) {
	variable new = @RGBLimits;
	set_rgb_limits(new, sd);
	(@sd.apply_cb)(new, __push_args(sd.apply_cb_args) );
   }
   sd.dirty = 0;
}
% }}}

% Plots {{{

static define set_scale(which_axis, what_scale, sd)
{
   variable func, plotd = sd.current_plot;
   variable axis = plotd.dsets[0].axes[which_axis];

   if (what_scale == GTK_PLOT_SCALE_LOG10) {
	if (axis.min <= 0.0) {
	   () = printf("Log scaling disabled: some values are non-positive\n");
	   return FALSE;
	}
	gtk_plot_axis_set_ticks(plotd.plot, which_axis , 1, 8);
   }
   else {
	func = Range_Funcs[which_axis];
	@func(plotd, axis.min, axis.max);	% this will reset ticks, too
   }

   sd.scale[which_axis] = what_scale;
   func = Scale_Funcs[which_axis];		% this use 2 LOC due to S-Lang
   (@func)(plotd.plot, what_scale);		% bug (where _NARGS remains 0)
   update_regions(sd);

   return TRUE;
}

static define set_scale_cb(radiob, which_axis, what_scale, default_radiob, sd)
{
   if (0 == gtk_toggle_button_get_active (radiob)) return;

   !if (set_scale(which_axis,what_scale,sd))
	gtk_toggle_button_set_active (default_radiob,TRUE);
}
static define make_axis_scaler(vbox, which_axis, sd)
{
   variable frame = gtk_frame_new( char(88 + which_axis) + " Axis Scale");
   gtk_box_pack_start(vbox,frame,FALSE,FALSE,5);

   variable rvbox = gtk_vbox_new(FALSE,5);
   gtk_container_add(frame,rvbox);

   variable linrb = make_radiob(rvbox, NULL, "Linear", &set_scale_cb,
   			which_axis, GTK_PLOT_SCALE_LINEAR,  NULL, sd);

   () = make_radiob(rvbox, linrb,"Log10", &set_scale_cb, which_axis,
			GTK_PLOT_SCALE_LOG10,  linrb, sd);
}

static define plot_grab_focus(canvas,dummy,notebook,sd)
{
   variable page = where(sd.plots == sd.current_plot)[0];
   gtk_notebook_set_current_page(notebook, page);
}

static define plot_switch(notebook, notebook_page_dummy, which, sd)
{
   if (sd.current_plot == sd.plots[which]) return;

   foreach (sd.current_plot.user_data) {
	variable region = ();
	gtk_widget_hide(region.sliderbox);
   }

   sd.current_plot = sd.plots[which];
   foreach (sd.current_plot.user_data) {
	region = ();
	gtk_widget_show(region.sliderbox);
   }

   () = set_scale(0, sd.scale[0], sd);
   () = set_scale(1, sd.scale[1], sd);

   !if (sd.single_image) return;

   % Invert the sliders to properly reflect Energy / Wavelength ranges
   _for (1,6,1) {
	variable i = (), slider = sd.sliders[i];
	gtk_box_reorder_child (slider.parent, slider.widget,
					(i + slider.inverted) mod 2);
	slider.inverted = not(slider.inverted);
	gtk_range_set_inverted(slider.widget,slider.inverted);
	gtk_widget_queue_draw(slider.widget);
   }
}

static define plot_expose(event,sd) { update_regions(sd); return FALSE; }

static define make_plot(x, y, xlabel, sd, colored_regions)
{
   variable plotd = _gtk_plot(x, y);

   !if (length(sd.plots))
	() = g_signal_connect_after(plotd.canvas, "size_allocate",
					&plot_grab_focus, sd.plot_tabs, sd);
   sd.plots = [ sd.plots, plotd ];

   plotd.user_data = colored_regions;

   gtk_plot_canvas_unset_flags(plotd.canvas, GTK_PLOT_CANVAS_CAN_SELECT);
   gtk_plot_data_set_connector(plotd.dset.widget, GTK_PLOT_CONNECT_STRAIGHT);

   gtk_plot_axis_set_title(plotd.plot,GTK_PLOT_AXIS_LEFT," Counts");
   gtk_plot_axis_set_title(plotd.plot,GTK_PLOT_AXIS_BOTTOM, xlabel);
   () = g_signal_connect_swapped(plotd.plot,"expose_event",&plot_expose,sd);

   variable tab_label = gtk_label_new(strtok(xlabel, "(")[0]);
   gtk_notebook_append_page(sd.plot_tabs,plotd.canvas, tab_label);
   if (_gtk_version >= 20400) pop;

}

static define make_plots(sd)
{
   variable plotd, x = sd.grids, y = sd.histograms, rgblims = sd.rgblims;
   sd.plots = Struct_Type[0];
   sd.plot_tabs = gtk_notebook_new();
   gtk_notebook_set_tab_pos(sd.plot_tabs,GTK_POS_TOP);
   gtk_notebook_set_scrollable(sd.plot_tabs, TRUE);

   variable R = make_region(sd, x[0], gdk_red,   1, 2, rgblims.r0, rgblims.r1);
   variable G = make_region(sd, x[1], gdk_green, 3, 4, rgblims.g0, rgblims.g1);
   variable B = make_region(sd, x[2], gdk_blue,  5, 6, rgblims.b0, rgblims.b1);

   if (sd.single_image) {
	make_plot(x[0], y[0], "Energy (keV)", sd, [R, G, B]);
	sd.current_plot = sd.plots[0];

	sd.x_non_zeroes = where(x[0] != 0.0);
	if (length(sd.x_non_zeroes) == 0)
	   () = fprintf(stderr, "Cannot plot Wavelength, Energy contains "+
				"only zeroes\n");
	else {
	   sd.angstroms = 12.3984185734 / x[0][ sd.x_non_zeroes ];  % keV to A
	   make_plot(sd.angstroms, y[0][sd.x_non_zeroes], "Wavelength (A)",
							sd, [R, G, B]);
	}
   }
   else {
	make_plot(x[0], y[0], "Red Pixel Intensity", sd, [R] );
	make_plot(x[1], y[1], "Green Pixel Intensity", sd, [G] );
	make_plot(x[2], y[2], "Blue Pixel Intensity", sd, [B] );
	sd.current_plot = sd.plots[0];
   }
   () = g_signal_connect(sd.plot_tabs,"switch_page", &plot_switch, sd);
}
% }}}

public define rgbselect() % {{{
{
   variable sd = @SelectorData, x, y, dims, button, vbox, frame;

   switch(_NARGS) 
        { case 0 or case 1 or case 2:
		usage("Integer_Type = rgbselect(x, y, rgb_limits"+
			"[,apply_callback_func [, apply_callback_args] );");
	}
	{ case 3: (x, y, sd.rgblims) = (); }
	{
	   	sd.apply_cb_args = __pop_args(_NARGS - 4);
		(x, y, sd.rgblims, sd.apply_cb) = ();
	}

   if ( andelse {sd.apply_cb != NULL} {not(_is_callable(sd.apply_cb))} )
	error("Specified Apply callback is not a callable function");

   sd.scale = [ GTK_PLOT_SCALE_LINEAR, GTK_PLOT_SCALE_LINEAR ];

   sd.grids = Array_Type[3];
   sd.histograms = Array_Type[3];
   if (_typeof(x) != Array_Type) {
	sd.single_image = 1;
	sd.grids[*] = typecast(x, Double_Type);
	sd.histograms[*] = typecast(y, Double_Type);
   }
   else {
	sd.single_image = 0;
	sd.grids = array_map(Array_Type, &typecast, x, Double_Type);
	sd.histograms = array_map(Array_Type, &typecast, y, Double_Type);
   }

   sd.dirty = 0;
   make_plots(sd);

   variable win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   () = g_signal_connect(win,"destroy",&gtk_main_quit);
   gtk_window_set_title(win,"rgbselect 0.93");

   variable main_vbox = gtk_vbox_new(FALSE,5);
   gtk_container_add(win,main_vbox);

   sd.plots_area = gtk_hbox_new(FALSE,5);
   gtk_box_pack_start(main_vbox,sd.plots_area,TRUE,TRUE,0);
   gtk_box_pack_start(sd.plots_area,sd.plot_tabs,TRUE,TRUE,1);

   variable button_vbox = gtk_vbox_new(FALSE,20);
   gtk_box_pack_end(sd.plots_area,button_vbox,FALSE,FALSE,5);

   make_axis_scaler(button_vbox, GTK_PLOT_AXIS_X, sd);
   make_axis_scaler(button_vbox, GTK_PLOT_AXIS_Y, sd);

   frame = gtk_frame_new("Color Selection");
   gtk_box_pack_start(main_vbox,frame,FALSE,FALSE,0);
   gtk_container_add(frame,sd.slider_vbox);

   variable hbox = gtk_hbox_new(FALSE,5);
   gtk_box_pack_end(main_vbox,hbox,FALSE,FALSE,5);
   button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
   () = g_signal_connect_swapped(button,"clicked", &apply, win, sd, 0);
   gtk_box_pack_start(hbox,button,FALSE,FALSE,5);

   button = gtk_button_new_from_stock(GTK_STOCK_OK);
   () = g_signal_connect_swapped(button,"clicked", &apply, win, sd, 1);
   gtk_box_pack_start(hbox,button,FALSE,FALSE,5);

   button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
   () = g_signal_connect_swapped(button,"clicked",&cancel, win, sd);
   gtk_box_pack_end(hbox,button,FALSE,FALSE,5);

   frame = gtk_frame_new("Color Ratios");
   gtk_box_pack_start(button_vbox,frame,FALSE,FALSE,0);
   vbox = gtk_vbox_new(FALSE,5);
   gtk_container_add(frame,vbox);

   if (sd.single_image) make_aligned_button(vbox,"Equalize", &equalize, sd);
   make_aligned_button(vbox," Revert ",&revert, sd);

   gtk_widget_show_all(win);
   !if (sd.single_image) {
	gtk_widget_hide(sd.sliders[3].parent);
	gtk_widget_hide(sd.sliders[5].parent);
   }
   gtk_main();
   return sd.dirty;
} % }}}

provide("rgbselect");
