/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A widget to select what to show.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgshow.h"
#include "skgservices.h"

#include "klocale.h"
#include "kmenu.h"
#include "kicon.h"

SKGShow::SKGShow(QWidget *parent)
    : QToolButton(parent)
{
    setPopupMode(QToolButton::InstantPopup);
    setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
    setAutoRaise(true);
    setText(i18n("Show"));
    setIcon(KIcon("show-menu"));

    m_menu = new KMenu(this);
    setMenu(m_menu);

    //Time to emit stateChanged
    m_timer.setSingleShot(true);
    connect(&m_timer, SIGNAL(timeout()), this, SIGNAL(stateChanged()));

    hide();
}

SKGShow::~SKGShow()
{
    m_menu = NULL;
}

QString SKGShow::getState()
{
    QStringList itemsChecked;
    if(m_menu) {
        QList<QAction*> actions = m_menu->actions();
        int nb = actions.count();
        for(int i = 0; i < nb; ++i) {
            QAction* act = actions.at(i);
            if(act && act->isChecked()) itemsChecked.push_back(act->data().toString());
        }
    }
    return SKGServices::stringsToCsv(itemsChecked);
}

QString SKGShow::getWhereClause()
{
    QString wc;
    if(m_menu) {
        QList<QAction*> actions = m_menu->actions();
        int nb = actions.count();
        bool noCheck = true;
        for(int i = 0; i < nb; ++i) {
            QAction* act = actions.at(i);
            if(act && act->isChecked()) {
                if(!wc.isEmpty()) wc += " OR ";
                wc += '(' % m_whereclause[act] % ')';
                noCheck = false;

                if(m_whereclause[act].isEmpty()) {
                    wc = "";
                    break;
                }
            };
        }
        if(nb && noCheck) wc = "1=0";
    }
    return wc;
}

void SKGShow::setDefaultState(const QString& iState)
{
    m_defaultState = iState;
    setState(m_defaultState);
}

void SKGShow::setState(const QString& iState)
{
    if(m_menu) {
        QStringList itemsChecked = SKGServices::splitCSVLine(iState.isEmpty() ? m_defaultState : iState);

        QList<QAction*> actions = m_menu->actions();
        int nb = actions.count();
        for(int i = 0; i < nb; ++i) {
            QAction* act = actions.at(i);
            if(act) act->setChecked(itemsChecked.contains(act->data().toString()));
        }

        emit stateChanged();
    }
}

void SKGShow::addItem(const QString& iIdentifier, const QString& iText, const QString& iIcon,
                      const QString& iWhereClose,
                      const QString& iListIdToCheckWhenChecked ,
                      const QString& iListIdToUncheckWhenChecked ,
                      const QString& iListIdToCheckWhenUnchecked ,
                      const QString& iListIdToUncheckWhenUnchecked
                     )
{
    if(m_menu) {
        QAction* act = m_menu->addAction(iText);
        if(act) {
            act->setIcon(KIcon(iIcon));
            act->setData(iIdentifier);
            act->setCheckable(true);

            m_check_to_check[act] = iListIdToCheckWhenChecked;
            m_check_to_uncheck[act] = iListIdToUncheckWhenChecked;
            m_uncheck_to_check[act] = iListIdToCheckWhenUnchecked;
            m_uncheck_to_uncheck[act] = iListIdToUncheckWhenUnchecked;

            m_whereclause[act] = iWhereClose;

            connect(act, SIGNAL(triggered(bool)), this, SLOT(itemChanged()));
        }

        show();
    }
}

void SKGShow::addSeparator()
{
    if(m_menu) m_menu->addSeparator();
}

void SKGShow::itemChanged()
{
    QAction* act = qobject_cast<QAction*>(sender());
    if(act) {
        //Apply rules
        if(act->isChecked()) {
            {
                //Check items
                QStringList items = SKGServices::splitCSVLine(m_check_to_check[act]);
                int nb = items.count();
                for(int i = 0; i < nb; ++i) {
                    QAction* act2 = getAction(items.at(i));
                    if(act2) act2->setChecked(true);
                }
            }

            {
                //Uncheck items
                QStringList items = SKGServices::splitCSVLine(m_check_to_uncheck[act]);
                int nb = items.count();
                for(int i = 0; i < nb; ++i) {
                    QAction* act2 = getAction(items.at(i));
                    if(act2) act2->setChecked(false);
                }
            }
        } else {
            {
                //Check items
                QStringList items = SKGServices::splitCSVLine(m_uncheck_to_check[act]);
                int nb = items.count();
                for(int i = 0; i < nb; ++i) {
                    QAction* act2 = getAction(items.at(i));
                    if(act2) act2->setChecked(true);
                }
            }

            {
                //Uncheck items
                QStringList items = SKGServices::splitCSVLine(m_uncheck_to_uncheck[act]);
                int nb = items.count();
                for(int i = 0; i < nb; ++i) {
                    QAction* act2 = getAction(items.at(i));
                    if(act2) act2->setChecked(false);
                }
            }
        }

        //Emit event
        m_timer.start(300);
    }
}

QAction* SKGShow::getAction(const QString& iIdentifier)
{
    QAction* output = NULL;
    if(m_menu) {
        QList<QAction*> actions = m_menu->actions();
        int nb = actions.count();
        for(int i = 0; !output && i < nb; ++i) {
            QAction* act = actions.at(i);
            if(act && act->data().toString() == iIdentifier) output = act;
        }
    }
    return output;
}

#include "skgshow.moc"

