/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file implements classes SKGNodeObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgnodeobject.h"
#include "skgtraces.h"
#include "skgdefine.h"

#include <klocale.h>
#include <kicon.h>

SKGNodeObject::SKGNodeObject(SKGDocument* iDocument, int iID): SKGNamedObject(iDocument, "v_node", iID)
{
}

SKGNodeObject::~SKGNodeObject()
{
}

SKGNodeObject::SKGNodeObject(const SKGNodeObject& iObject)
        : SKGNamedObject(iObject)
{
}
SKGNodeObject::SKGNodeObject(const SKGObjectBase& iObject)
        : SKGNamedObject(iObject.getDocument(), "v_node", iObject.getID())
{
}

const SKGNodeObject& SKGNodeObject::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

SKGError SKGNodeObject::setName(const QString& iName)
{
    SKGError err;
    if (iName.contains(OBJECTSEPARATOR)) err=SKGError(ERR_FAIL, i18nc("Error message", "Invalid name '%1' because of the name cannot contain '%2'",iName, QString(OBJECTSEPARATOR)));
    else err=SKGNamedObject::setName(iName);
    return err;
}

QString SKGNodeObject::getWhereclauseId() const
{
    //Could we use the id
    QString output = SKGObjectBase::getWhereclauseId();
    if ( output.isEmpty() ) {
        if ( ! ( getAttribute ( "t_name" ).isEmpty() ) ) {
            output = "t_name='"+SKGServices::stringToSqlString(getAttribute("t_name"))+'\'';
        }
        QString r_node_id=getAttribute ( "r_node_id" );
        if ( !output.isEmpty() ) output+=" AND ";
        if ( r_node_id.isEmpty() ) output += "(r_node_id=0 OR r_node_id IS NULL OR r_node_id='')";
        else output += "r_node_id="+r_node_id;
    }
    return output;
}

QString SKGNodeObject::getFullName() const
{
    return getAttribute("t_fullname");
}

SKGError SKGNodeObject::setData(const QString& iData)
{
    return setAttribute("t_data", iData);
}

QString SKGNodeObject::getData() const
{
    return getAttribute("t_data");
}

bool SKGNodeObject::isFolder() const
{
    return (SKGServices::splitCSVLine(getData()).count()<=2);
}

QIcon SKGNodeObject::getIcon() const
{
    QStringList data=SKGServices::splitCSVLine(getData());
    QIcon icon=KIcon("folder");
    if (data.count()>2)  icon=(QIcon) KIcon(data.at(2));

    if (isAutoStart()) {
        //Modify icon
        QStringList overlay;
        overlay.push_back("user-online");
        if (data.count()>2)  icon=(QIcon) KIcon(data.at(2), NULL, overlay);
        else icon=(QIcon) KIcon("folder", NULL, overlay);
    }
    return icon;
}

SKGError SKGNodeObject::setAutoStart(bool iAutoStart)
{
    return setAttribute("t_autostart", iAutoStart ? "Y" : "N");
}

bool SKGNodeObject::isAutoStart() const
{
    return (getAttribute("t_autostart")=="Y");
}

SKGError SKGNodeObject::setOrder(double iOrder)
{
    SKGError err;
    double order=iOrder;
    if (order==-1) {
        order=1;
        SKGStringListList result;
        err=SKGServices::executeSelectSqliteOrder(getDocument(), "SELECT max(f_sortorder) from node", result);
        if (err.isSucceeded() && result.count()==2) order=SKGServices::stringToDouble(result.at(1).at(0))+1;
    }
    if (err.isSucceeded()) err=setAttribute("f_sortorder", SKGServices::doubleToString(order));
    return err;
}

double SKGNodeObject::getOrder() const
{
    return SKGServices::stringToDouble(getAttribute("f_sortorder"));
}

SKGError SKGNodeObject::createPathNode(SKGDocument* iDocument,
                                       const QString& iFullPath,
                                       SKGNodeObject& oNode,
                                       bool iRenameIfAlreadyExist)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGNodeObject::createPathNode", err);
    SKGTRACEL(10) << "Input parameter [iFullPath]=" << iFullPath << endl;
    //Check if node is already existing
    if (!iRenameIfAlreadyExist) SKGObjectBase::getObject(iDocument, "v_node", "t_fullname='"+SKGServices::stringToSqlString(iFullPath)+'\'', oNode);
    if (oNode.getID()==0) {
        //No, we have to create it
        //Search node separator
        int posSeparator=iFullPath.lastIndexOf(OBJECTSEPARATOR);
        if (posSeparator==-1) {
            oNode = SKGNodeObject(iDocument);
            err=oNode.setName(iFullPath);

            //Check if already existing
            if (err.isSucceeded() && iRenameIfAlreadyExist) {
                int index=1;
                while ( err.isSucceeded() && oNode.exist() ) {
                    index++;
                    err = oNode.setName(iFullPath+" ("+SKGServices::intToString(index)+')');
                }
            }

            if (err.isSucceeded()) err=oNode.setOrder(-1);
            if (err.isSucceeded()) err=oNode.save();
        } else {
            //Get first and second parts of the branch
            QString first = iFullPath.mid( 0,posSeparator );
            QString second = iFullPath.mid( posSeparator+QString(OBJECTSEPARATOR).length(), iFullPath.length()-posSeparator-QString(OBJECTSEPARATOR).length());

            //Get first node
            SKGNodeObject FirstNode;
            err=SKGNodeObject::createPathNode(iDocument, first, FirstNode);

            if (err.isSucceeded()) {
                //Get second node
                err=FirstNode.addNode(oNode);

                //Add second under first
                if (err.isSucceeded()) err = oNode.setName(second);

                //Check if already existing
                if (err.isSucceeded() && iRenameIfAlreadyExist) {
                    int index=2;
                    while ( err.isSucceeded() && oNode.exist() ) {
                        err = oNode.setName(second+" ("+SKGServices::intToString(index)+')');
                        ++index;
                    }
                }

                //save
                if (err.isSucceeded()) err=oNode.setOrder(-1);
                if (err.isSucceeded()) err=oNode.save();
            }
        }
    }

    return err;
}

SKGError SKGNodeObject::addNode(SKGNodeObject& oNode)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGNodeObject::addNode", err);
    if (getID() == 0) err=SKGError(ERR_FAIL, i18nc("Error message", "%1 failed because linked object is not yet saved in the database.",QString("SKGNodeObject::addNode")));
    else {
        oNode = SKGNodeObject(getDocument());
        err = oNode.setAttribute("r_node_id", SKGServices::intToString(getID()));
    }
    return err;
}

SKGError SKGNodeObject::removeParentNode()
{
    return setAttribute("r_node_id", "");
}

SKGError SKGNodeObject::setParentNode(const SKGNodeObject& iNode)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGNodeObject::setParentNode", err);
    if (iNode.getID() == 0) err=SKGError(ERR_FAIL, i18nc("Error message", "%1 failed because linked object is not yet saved in the database.",QString("SKGNodeObject::setParentNode")));
    else {
        //Check if it's a loop
        SKGNodeObject current=iNode;
        do {
            if (current==*this) {
                err=SKGError(ERR_FAIL, i18nc("Error message", "You cannot create a loop."));
            } else {
                SKGNodeObject parent;
                current.getParentNode(parent);
                current=parent;
            }
        } while (err.isSucceeded() && current.getID()!=0);

        if (err.isSucceeded()) err = setAttribute("r_node_id", SKGServices::intToString(iNode.getID()));
    }
    return err;
}

SKGError SKGNodeObject::getParentNode(SKGNodeObject& oNode) const
{
    SKGError err;
    QString parent_id=getAttribute("r_node_id");
    if (!parent_id.isEmpty()) err=getObject(getDocument(), "v_node", "id="+parent_id , oNode);
    return err;
}

SKGError SKGNodeObject::getNodes(SKGListSKGObjectBase& oNodeList) const
{
    return getObjects(getDocument(), "v_node",
                      "r_node_id="+SKGServices::intToString(getID())+" ORDER BY f_sortorder, t_name",
                      oNodeList);
}

#include "skgnodeobject.moc"
