/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A date editor with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdateedit.h"

#include <KCalendarSystem>
#include <KDebug>
#include <KGlobal>
#include <KGlobalSettings>
#include <KLocale>
#include <KCompletion>

#include <QApplication>
#include <QEvent>
#include <QKeyEvent>
#include <QLineEdit>
#include <QMouseEvent>

#include "skgdatepickerpopup.h"
#include "skgservices.h"

SKGDateEdit::SKGDateEdit ( QWidget *parent, const char *name )
        : SKGComboBox ( parent ), mReadOnly ( false ), mDiscardNextMousePress ( false ), mMode ( CURRENT )
{
    setObjectName ( name );

    // need at least one entry for popup to work
    setMaxCount ( 1 );
    setEditable ( true );

    mDate = QDate::currentDate();
    QString today = KGlobal::locale()->formatDate ( mDate, KLocale::ShortDate );

    addItem ( today );
    setCurrentIndex ( 0 );
    //setSizeAdjustPolicy( AdjustToContents );

    connect ( lineEdit(), SIGNAL ( returnPressed() ), this, SLOT ( lineEnterPressed() ) );

    mPopup = new SKGDatePickerPopup ( SKGDatePickerPopup::DatePicker, QDate::currentDate(), this );
    mPopup->hide();
    mPopup->installEventFilter ( this );

    connect ( mPopup, SIGNAL ( dateChanged ( const QDate& ) ), this, SLOT ( dateSelected ( const QDate& ) ) );

    // handle keyword entry
    // Create the keyword list. This will be used to match against when the user
    // enters information.
    mKeywordMap.insert ( i18nc ( "the day after today", "tomorrow" ), 1 );
    mKeywordMap.insert ( i18nc ( "this day", "today" ), 0 );
    mKeywordMap.insert ( i18nc ( "the day before today", "yesterday" ), -1 );

    QString dayName;
    for ( int i = 1; i <= 7; ++i )
    {
        dayName = KGlobal::locale()->calendar()->weekDayName ( i ).toLower();
        mKeywordMap.insert ( dayName, i + 100 );
    }
    KCompletion *comp = completionObject();
    if ( comp )
    {
        comp->setIgnoreCase ( true );
        comp->clear ();
        comp->insertItems ( mKeywordMap.keys() );
    }

    lineEdit()->installEventFilter ( this );
    font_color = lineEdit()->palette().color ( QPalette::Text );
    mTextChanged = false;
}

SKGDateEdit::~SKGDateEdit()
{
    mPopup=NULL;
}

SKGDateEdit::Mode SKGDateEdit::mode() const
{
    return mMode;
}

void SKGDateEdit::setMode ( Mode iMode )
{
    mMode=iMode;
}

void SKGDateEdit::setDate ( const QDate &date )
{
    assignDate ( date );
    updateView();
}

QDate SKGDateEdit::date()
{
    if ( !mDate.isValid () ) mDate=QDate::currentDate();
    return mDate;
}

void SKGDateEdit::setReadOnly ( bool readOnly )
{
    mReadOnly = readOnly;
    lineEdit()->setReadOnly ( readOnly );
}

bool SKGDateEdit::isReadOnly() const
{
    return mReadOnly;
}

void SKGDateEdit::showPopup()
{
    if ( mReadOnly || !mPopup ) return;

    //Position popup
    QRect desk = KGlobalSettings::desktopGeometry ( this );

    QPoint popupPoint = mapToGlobal ( QPoint ( 0, 0 ) );

    int dateFrameHeight = mPopup->sizeHint().height();
    if ( popupPoint.y() + height() + dateFrameHeight > desk.bottom() )
    {
        popupPoint.setY ( popupPoint.y() - dateFrameHeight );
    }
    else
    {
        popupPoint.setY ( popupPoint.y() + height() );
    }

    int dateFrameWidth = mPopup->sizeHint().width();
    if ( popupPoint.x() + dateFrameWidth > desk.right() )
    {
        popupPoint.setX ( desk.right() - dateFrameWidth );
    }

    if ( popupPoint.x() < desk.left() )
    {
        popupPoint.setX ( desk.left() );
    }

    if ( popupPoint.y() < desk.top() )
    {
        popupPoint.setY ( desk.top() );
    }

    if ( mDate.isValid() )
    {
        mPopup->setDate ( mDate );
    }
    else
    {
        mPopup->setDate ( QDate::currentDate() );
    }

    mPopup->popup ( popupPoint );
}

void SKGDateEdit::dateSelected ( const QDate &date )
{
    assignDate ( date );
    updateView();
    Q_EMIT dateChanged ( date );
    Q_EMIT dateEntered ( date );

    if ( date.isValid() )
    {
        if (mPopup) mPopup->hide();
        hidePopup ();
    }
}

void SKGDateEdit::lineEnterPressed()
{
    bool replaced = false;

    QDate date = parseDate ( &replaced );
    assignDate ( date );
    if ( replaced ) updateView();

    Q_EMIT dateChanged ( date );
    Q_EMIT dateEntered ( date );
}

QDate SKGDateEdit::parseDate ( bool *replaced )
{
    QString text = currentText();
    QDate result;
    QPalette field_palette = lineEdit()->palette();
    field_palette.setColor ( QPalette::Text,font_color );

    if ( replaced ) ( *replaced ) = false;

    if ( text.isEmpty() )
    {
        result = QDate();
    }
    else if ( mKeywordMap.contains ( text.toLower() ) )
    {
        QDate today = QDate::currentDate();
        int i = mKeywordMap[ text.toLower() ];
        if ( i >= 100 )
        {
            /* A day name has been entered. Convert to offset from today.
             * This uses some math tricks to figure out the offset in days
             * to the next date the given day of the week occurs. There
             * are two cases, that the new day is >= the current day, which means
             * the new day has not occurred yet or that the new day < the current day,
             * which means the new day is already passed (so we need to find the
             * day in the next week).
             */
            i -= 100;
            int currentDay = today.dayOfWeek();
            if ( i >= currentDay )
            {
                i -= currentDay;
            }
            else
            {
                i += 7 - currentDay;
            }
        }

        result = today.addDays ( i );
        if ( replaced ) ( *replaced ) = true;
    }
    else
    {
        bool ok;
        result = KGlobal::locale()->readDate ( text, &ok );
        if ( !ok )
        {
            //It's not a complete date or not a date
            QStringList items=text.split ( '/' );
            int size=items.count();
            if ( size>0 && size<3 )
            {
                if ( size>=1 )
                {
                    result=QDate::currentDate().addDays ( -QDate::currentDate().day() ).addDays ( items.at ( 0 ).toInt ( &ok ) );
                    if ( mMode==SKGDateEdit::PREVIOUS )
                    {
                        if ( result>QDate::currentDate() ) result=result.addMonths ( -1 );
                    }
                    else if ( mMode==SKGDateEdit::NEXT )
                    {
                        if ( result<QDate::currentDate() ) result=result.addMonths ( 1 );

                    }
                }

                if ( ok && size==2 )
                {
                    result=result.addMonths ( -result.month() ).addMonths ( items.at ( 1 ).toInt ( &ok ) );
                    if ( mMode==SKGDateEdit::PREVIOUS )
                    {
                        if ( result>QDate::currentDate() ) result=result.addYears ( -1 );
                    }
                    else if ( mMode==SKGDateEdit::NEXT )
                    {
                        if ( result<QDate::currentDate() ) result=result.addYears ( -1 );
                    }
                }

                if ( !ok ) field_palette.setColor ( QPalette::Text,Qt::red );
                else if ( replaced ) ( *replaced ) = true;
            }
            else
            {
                field_palette.setColor ( QPalette::Text,Qt::red );
            }
        }
        else
        {
            if ( 2000+result.year() <QDate::currentDate().year() )  result=result.addYears ( 2000 );
            else if ( result.year() <100 )  result=result.addYears ( 1900 );
            else if ( result.year() <1000 )  result=result.addYears ( 1000 );
            if ( replaced ) ( *replaced ) = true;
        }
    }

    //Set color
    lineEdit()->setPalette ( field_palette );

    return result;
}

void SKGDateEdit::focusOutEvent ( QFocusEvent *event )
{
    lineEnterPressed();
    SKGComboBox::focusOutEvent ( event );
}

void SKGDateEdit::keyPressEvent ( QKeyEvent* event )
{
    if ( event )
    {
        if ( event->count() ==1 && event->key() ==Qt::Key_Plus )
        {
            QDate date = parseDate();
            assignDate ( QApplication::keyboardModifiers() &Qt::ControlModifier ? this->date().addMonths ( 1 ) :this->date().addDays ( 1 ) );
            updateView();

            event->accept();
            Q_EMIT dateChanged ( date );
            Q_EMIT dateEntered ( date );
        }
        else if ( event->count() ==1 && event->key() ==Qt::Key_Minus )
        {
            QDate date = parseDate();
            assignDate ( QApplication::keyboardModifiers() &Qt::ControlModifier ? this->date().addMonths ( -1 ) :this->date().addDays ( -1 ) );
            updateView();

            event->accept();
            Q_EMIT dateChanged ( date );
            Q_EMIT dateEntered ( date );
        }
        else if ( event->count() ==1 && event->key() ==Qt::Key_Equal )
        {
            QDate date = parseDate();
            assignDate ( QDate::currentDate() );
            updateView();

            event->accept();
            Q_EMIT dateChanged ( date );
            Q_EMIT dateEntered ( date );
        }
        else SKGComboBox::keyPressEvent ( event );
    }
}

bool SKGDateEdit::eventFilter ( QObject *object, QEvent *event )
{
    if ( event )
    {
        if ( object == lineEdit() )
        {
            // We only process the focus out event if the text has changed
            // since we got focus
            if ( ( event->type() == QEvent::FocusOut ) && mTextChanged )
            {
                lineEnterPressed();
                mTextChanged = false;
            }
            else if ( event->type() == QEvent::KeyPress )
            {
                QKeyEvent *keyEvent = ( QKeyEvent * ) event;
                if ( keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter )
                {
                    lineEnterPressed();
                    return true;
                }
            }
        }
        else
        {
            // It's a date picker event
            switch ( event->type() )
            {
            case QEvent::MouseButtonDblClick:
            case QEvent::MouseButtonPress:
            {
                QMouseEvent *mouseEvent = ( QMouseEvent* ) event;
                if ( mPopup && !mPopup->rect().contains ( mouseEvent->pos() ) )
                {
                    QPoint globalPos = mPopup->mapToGlobal ( mouseEvent->pos() );
                    if ( QApplication::widgetAt ( globalPos ) == this )
                    {
                        // The date picker is being closed by a click on the
                        // SKGDateEdit widget. Avoid popping it up again immediately.
                        mDiscardNextMousePress = true;
                    }
                }

                break;
            }
            default:
                break;
            }
        }
    }
    return false;
}

bool SKGDateEdit::assignDate ( const QDate &date )
{
    mDate = date;
    mTextChanged = false;
    return true;
}

void SKGDateEdit::updateView()
{
    QString dateString;
    if ( mDate.isValid() )
    {
        dateString = KGlobal::locale()->formatDate ( mDate, KLocale::ShortDate );
    }

    // We do not want to generate a signal here,
    // since we explicitly setting the date
    bool blocked = signalsBlocked();
    blockSignals ( true );
    removeItem ( 0 );
    insertItem ( 0, dateString );
    blockSignals ( blocked );
}

#include "skgdateedit.moc"
