/* $Id: ext2_fileops.c,v 1.7 2003/05/30 00:42:37 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h> /* To make ext2lib happy.  We don't actually use anything from it */
#include "getfile.h"
#include "libc.h"
#include "partition.h"
#include "cfe_api.h"
#include "ext2_fs.h"  /* From linux kernel */
#include "ext2fs.h"   /* From e2fsprogs    */


typedef struct {
//	int32_t offset;
	ext2_filsys fs;
        ext2_file_t file;
} ext2_ops_data_t;


/*
 * I/O Manager Glue
 */
static errcode_t cfe_manager_open(const char *dev, int flags, io_channel *channel);
static errcode_t cfe_manager_close(io_channel channel);
static errcode_t cfe_manager_set_blksize(io_channel channel, int blksize);
static errcode_t cfe_manager_read_blk(io_channel channel, unsigned long block, int count, void *buf);
static errcode_t cfe_manager_write_blk(io_channel channel, unsigned long block, int count, const void *buf);
static errcode_t cfe_manager_flush(io_channel channel);

static struct struct_io_manager struct_cfe_manager = {
	EXT2_ET_MAGIC_IO_MANAGER,
	"CFE I/O Manager",
	cfe_manager_open,
	cfe_manager_close,
	cfe_manager_set_blksize,
	cfe_manager_read_blk,
	cfe_manager_write_blk,
	cfe_manager_flush
};

io_manager cfe_io_manager = &struct_cfe_manager;

/* This is probably a big no-no, but I'm 99% sure we don't need reentrancy for 
   the intended use, and we've not good way of allocating memory from the firmware...*/

#define CFE_CHANNEL_NAME_SIZE 64
static char cfe_channel_name[CFE_CHANNEL_NAME_SIZE];

typedef struct cfe_private_data_struct {
	int handle;
	unsigned long linear_offset;
} cfe_channel_private_t;

static cfe_channel_private_t cfe_channel_private = {
    handle: 0,
    linear_offset: 0
};

static struct struct_io_channel cfe_channel = {
	magic:        EXT2_ET_MAGIC_IO_CHANNEL,
	manager:      &struct_cfe_manager,
	name:         cfe_channel_name,
	block_size:   1024,
	read_error:   NULL,
	write_error:  NULL,
	refcount:     0,
	private_data: (void *)&cfe_channel_private,
	app_data:     NULL
};



static unsigned long long my_strtoull(char *ptr) 
{
	unsigned long long retval = 0;
	unsigned int newdigit;
	while(*ptr) {
		if (*ptr >= 'a' &&  *ptr <= 'f') {
			newdigit = *ptr - ('a' - 10);
		} else if (*ptr >= 'A' && *ptr <= 'F') {
			newdigit = *ptr - ('A' - 10);
		} else if (*ptr >= '0' && *ptr <= '9') {
			newdigit = *ptr - '0';
		} else if (*ptr == ' ' 
			   || *ptr == '\t'
			   || *ptr == '\n') {
			/* Just skip it */
		} else {
			return -1UL;
		}
		retval <<= 4;
		retval |= newdigit;
		ptr++;
	}
	return retval;
}

static errcode_t cfe_manager_open(const char *name, int flags, io_channel *channel)
{
    char *tmp;
    cfe_channel_private_t *priv;
    (*channel) = &cfe_channel;

    if ((*channel)->refcount) {
	return -1;
	}

    priv = (cfe_channel_private_t *)((*channel)->private_data);
    for (tmp = (char *) name; *tmp; tmp++) {
	if (*tmp == '@') {
	    *tmp = '\0';
	    tmp++;
	    priv->linear_offset = my_strtoull(tmp);
	    if (priv->linear_offset == -1UL) {
		return -1;
		}
	    break;
	    }
	}
	
    lib_strncpy(cfe_channel_name, (char *) name, CFE_CHANNEL_NAME_SIZE-1);
    cfe_channel_name[CFE_CHANNEL_NAME_SIZE-1] = '\0';
    priv->handle = cfe_open(cfe_channel_name);

    (*channel)->refcount++;
	
    if (priv->handle < 0) {
	return -1;
	}
    return 0;
}


static errcode_t cfe_manager_close(io_channel channel)
{
	cfe_channel_private_t *priv;

	/* Should only have exactly 1 open when close is called */
	if (channel->refcount != 1 ) {
		return -1;
	}
	priv = (cfe_channel_private_t *)channel->private_data;
	cfe_close(priv->handle);
	channel->refcount--;
	return 0;
}


static errcode_t cfe_manager_set_blksize(io_channel channel, int blksize)
{
	channel->block_size = blksize;
	return 0;
}


static errcode_t cfe_manager_read_blk(io_channel channel, unsigned long block, int count, void *buf)
{
	int retval;
	long size;
	cfe_channel_private_t *priv;
	priv = (cfe_channel_private_t *)channel->private_data;
	size = (count & 1<<((sizeof(long)*8)-1))?((~count) + 1):(count * channel->block_size);

	retval = cfe_readblk(priv->handle, priv->linear_offset + (block * channel->block_size), 
			     (unsigned char *)buf, size);

	return (retval < 0)?-1:0;
}


static errcode_t cfe_manager_write_blk(io_channel channel, unsigned long block, int count, const void *buf)
{
	int retval; 
	long size;
	cfe_channel_private_t *priv;
	size = (count & 1<<((sizeof(long)*8)-1))?((~count) + 1):(count * channel->block_size);
	priv = (cfe_channel_private_t *)channel->private_data;

	retval = -1;
	return retval;
#if 0
	/* XXX we don't support writing. */
	retval = cfe_writeblk(priv->handle, priv->linear_offset + (block * channel->block_size), 
			      (unsigned char *)buf, size);
	return (retval < 0)?-1:0;
#endif
}


static errcode_t cfe_manager_flush(io_channel channel)
{
	return 0;
}



/* Print an ascii version of val in buf.  return the number of characters written */
static  int hexprint(char *buf, unsigned long long val)
{
	int i; 
	int tmp;
	int digits = sizeof(val)<<1;
	for (i = 0; i < digits; i++) {
		tmp = (val >> ((digits - (i + 1))<<2)) & 0xf;
		if (tmp > 9) {
			*buf = 'a' + (tmp-10);
		} else {
		        *buf = '0' + tmp;
		}
		buf++;
	}
	*buf = 0;
	return digits;
}

/*
 * Expects a string in this format:
 * 
 *  ext2:devicename:partition#:filename\0
 * 
 */
static void *ext2_op_open(char *loc)
{
    char delims[] = ": \t";
    ext2_ops_data_t *ret;
    char *dev;
    char *part;
    char buf[128];
    char *tok;
    int cfe_handle;
    int partition;
    part_entry_t part_info;
    ino_t ino;
    char *name;

    if (lib_strncmp(loc, "ext2:", 5)) {
	return 0;
	}
    loc += 5;
    /* OK, we think it's for us.  Next field should be a cfe device name */

    dev = lib_strtok(loc, delims);
    part = lib_strtok(0, delims);
    name = lib_strtok(0, delims);

    if (!dev || !part || !name) {
	lib_printf("Invalid file specification in config file: %s\n",loc);
	lib_printf("The correct format is:  ext2:devicename:partition#:filename\n");
	return 0;
	}

    if (lib_strlen(dev) > 64) {
	lib_die("Absurdly long device string\n");
	}
	
    /* Then a partition number */
    partition = (part[0] == '*') ? -1 :  lib_atoi(part);
    ret = lib_malloc(sizeof(ext2_ops_data_t));
    if (!ret) {
	lib_die("Out of memory");
	}
    cfe_handle = cfe_open(dev);
    if (cfe_handle < 0) {
	lib_printf("CFE open of %s failed: %d\n", dev,cfe_handle);
	goto fail;
	}
    if (part_get_info(cfe_handle, partition, &part_info)) {
	lib_printf("Failed to get partition information for partition %i\n", partition);
	cfe_close(cfe_handle);
	goto fail;
	}
    cfe_close(cfe_handle);
    lib_strcpy(buf, dev);
    for (tok = buf; *tok; tok++) {}
    *tok = '@';
    tok++;
    hexprint(tok, le_to_cpu32(part_info.start_sector) * 512);

    if (ext2fs_open(buf, 0, 0, 0, cfe_io_manager, &(ret->fs))) {
	lib_printf("Failed to open ext2 filesystem on partition %i of device %s\n", partition, dev);
	goto fail;
	}


    /* ex2fs_lookup expects the name *not* to start with a slash. */

    if (*name == '/') name++;

    ino = 0;
    if (ext2fs_lookup(ret->fs,EXT2_ROOT_INO,name,lib_strlen(name),0,&ino)) {
	lib_printf("Could not find inode for file %s\n",name);
	ext2fs_close(ret->fs);
	goto fail;
	}

    if (ext2fs_file_open(ret->fs,ino,0,&(ret->file))) {
	ext2fs_close(ret->fs);
	goto fail;
	}

    return ret;
	
fail:
    lib_free(ret);
    return 0;
}	

static int32_t ext2_op_seek(void *private, int32_t ofs, file_handler_whence_t whence)
{
    ext2_ops_data_t *data = (ext2_ops_data_t *)private;
    ext2_off_t newoff = -1;

    switch (whence) {
	case GF_SEEK_SET:
	    ext2fs_file_lseek(data->file,(ext2_off_t) ofs,EXT2_SEEK_SET,&newoff);
	    break;
	case GF_SEEK_CUR:
	    ext2fs_file_lseek(data->file,(ext2_off_t) ofs,EXT2_SEEK_CUR,&newoff);
	    break;
	default:
	    lib_die("Unknown whence value\n");
	}

    return (int32_t) newoff;
}

static int32_t ext2_op_read(void *private, void *buf, int32_t amount)
{
	ext2_ops_data_t *data = (ext2_ops_data_t *)private;
	unsigned int retval = 0;

	if (ext2fs_file_read(data->file,buf,(unsigned int) amount,&retval)) {
	    return -1;
	    }

	return (int32_t) retval;
}

static void ext2_op_close(void *private)
{
    ext2_ops_data_t *data = (ext2_ops_data_t *)private;

    ext2fs_file_close(data->file);
    ext2fs_close(data->fs);

    lib_free(data);
}


file_ops_t ext2_ops = { 
	ext2_op_open,
	ext2_op_seek,
	ext2_op_read,
	ext2_op_close
};


