/*
 *  Part of the shrinkta program, a dvd copy tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <bonobo.h>
#include <gnome.h>

#include <dvd.h>

#include "preferences.h"
#include "nautilus-burn.h"
#include "preferences-dialog.h"
#include "support.h"

struct _PreferencesDialog {
	GtkDialog dialog;
	
	Preferences *prefs;
	
	PreferencesDialogTab tab;
	
	GtkWidget *notebook;
	GtkWidget *drive_selector;
	GtkWidget *filechooserbutton_image_dir;
	GtkWidget *filechooserbutton_temp_dir;
	GtkWidget *combobox_preferred_track;
	GtkWidget *combobox_burn;
	GtkWidget *radiobutton_dvd;
	GtkWidget *radiobutton_dldvd;
	GtkWidget *checkbutton_region_free;
	GtkWidget *checkbutton_include_dts_audio;
	GtkWidget *combobox_audio_language;
	GtkWidget *checkbutton_audio_fallback_undefined;
	GtkWidget *spinbutton_worst_factor;
	GtkWidget *checkbutton_drop_audio_2;
	GtkWidget *checkbutton_make_2disks;
};

enum {
	LAST_SIGNAL
};

static GtkDialogClass *preferences_dialog_parent_class = NULL;
static gint	  preferences_dialog_signals[LAST_SIGNAL];

static void	  preferences_dialog_dispose				(GObject		*object);
static void	  preferences_dialog_init				(PreferencesDialog	*pd);
static void	  preferences_dialog_class_init				(PreferencesDialogClass	*Klass);
static void	  notebook_switch_page					(GtkNotebook		*notebook,
									 GtkNotebookPage	*page,
									 guint			 page_num,
									 PreferencesDialog	*pd);
static void	  drive_selector_device_changed				(GtkWidget		*selection,
                             						 const char		*device_path,
									 PreferencesDialog	*pd);
static void	  filechooserbutton_image_dir_current_folder_changed	(GtkFileChooser	*filechooser,
									 PreferencesDialog	*pd);
static void	  filechooserbutton_temp_dir_current_folder_changed	(GtkFileChooser	*filechooser,
									 PreferencesDialog	*pd);
static void	  combobox_burn_changed					(GtkComboBox		*combobox,
									 PreferencesDialog	*pd);
static void	  radiobutton_dvd_toggled				(GtkToggleButton	*togglebutton,
									 PreferencesDialog	*pd);
static void	  combobox_preferred_track_changed			(GtkComboBox		*combobox,
									 PreferencesDialog	*pd);
static void	  checkbutton_include_dts_audio_toggled			(GtkToggleButton	*togglebutton,
									 PreferencesDialog	*pd);
static void	  combobox_audio_language_changed			(GtkComboBox		*combobox,
									 PreferencesDialog	*pd);
static void	  checkbutton_audio_fallback_undefined_toggled		(GtkToggleButton	*togglebutton,
									 PreferencesDialog	*pd);
static void	  spinbutton_worst_factor_value_changed			(GtkSpinButton	*spinbutton,
									 PreferencesDialog	*pd);
static void	  checkbutton_drop_audio_2_toggled			(GtkToggleButton	*togglebutton,
									 PreferencesDialog	*pd);
static void
preferences_dialog_dispose	(GObject		*object)

{
	PreferencesDialog *prefs_dialog;
	
	/* why is this function called twice on dispose ?? */
	
	prefs_dialog = PREFERENCES_DIALOG (object);
	if (prefs_dialog->prefs != NULL) {
		g_object_unref (G_OBJECT (prefs_dialog->prefs));
		prefs_dialog->prefs = NULL;
	}
	G_OBJECT_CLASS (preferences_dialog_parent_class)->dispose (object);
}

static void
preferences_dialog_class_init	(PreferencesDialogClass		*Klass)
{
	GObjectClass *object_class = (GObjectClass *) Klass;
	
	preferences_dialog_parent_class = g_type_class_peek_parent (Klass);
	object_class->dispose = preferences_dialog_dispose;
}

static void
preferences_dialog_init		(PreferencesDialog		*pd)
{
	GtkWidget *dialog_prefs;
	GtkWidget *dialog_vbox;
	GtkWidget *vbox_general;
	GtkWidget *label_god;
	GtkWidget *hbox_god;
	GtkWidget *label_opt_read;
	GtkWidget *label_gpt;
	GtkWidget *hbox_image_dir;
	GtkWidget *label_image_dir;
	GtkWidget *hbox_temp_dir;
	GtkWidget *label_temp_dir;
	GtkWidget *label_burners;
	GtkWidget *hbox8;
	GtkWidget *label_burn;
	GtkWidget *label_general;
	GtkWidget *vbox_audio;
	GtkWidget *label_as;
	GtkWidget *hbox_audio_preferred;
	GtkWidget *label_preferred_audio;
	GtkWidget *hbox_audio_track_2;
	GtkWidget *label_al;
	GtkWidget *hbox_audio_lang;
	GtkWidget *label_audio_language;
	GtkWidget *hbox7;
	GtkWidget *label_audio;
	GtkWidget *vbox_video;
	GtkWidget *label_vq;
	GtkWidget *hbox_worst_quality;
	GtkWidget *label_worst_quality;
	GtkObject *spinbutton_worst_factor_adj;
	GtkWidget *hbox_drop_audio_2;
	GtkWidget *hbox_make_2disks;
	GtkWidget *label_video;
	GtkWidget *vbox_media;
	GtkWidget *label_mt;
	GtkWidget *label_ms;
	GtkWidget *hbox_dvd;
	GtkWidget *hbox_rf;
	GtkWidget *label_media;
	GSList    *media_group = NULL;
	GtkWidget *hbox_dldvd;
	GtkWidget *dialog_action_area;
	GtkWidget *closebutton;
	GtkWidget *helpbutton;
	GtkSizeGroup *size_group;
	DvdLanguage lang;
	
	dialog_prefs = GTK_WIDGET (pd);
	gtk_window_set_title (GTK_WINDOW (dialog_prefs), _("Dvd Movie Backup - Preferences"));
	gtk_window_set_type_hint (GTK_WINDOW (dialog_prefs), GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog_prefs), FALSE);
	
	dialog_vbox = GTK_DIALOG (dialog_prefs)->vbox;
	gtk_widget_show (dialog_vbox);
	gtk_box_set_spacing (GTK_BOX (dialog_vbox), 0);
	
	pd->notebook = gtk_notebook_new ();
	gtk_widget_show (pd->notebook);
	gtk_box_pack_start (GTK_BOX (dialog_vbox), pd->notebook, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (pd->notebook), 6);
	
	vbox_general = gtk_vbox_new (FALSE, 6);
	gtk_widget_show (vbox_general);
	gtk_container_add (GTK_CONTAINER (pd->notebook), vbox_general);
	gtk_container_set_border_width (GTK_CONTAINER (vbox_general), 12);
	
	label_god = gtk_label_new (_("<b>Optical Device</b>"));
	gtk_widget_show (label_god);
	gtk_box_pack_start (GTK_BOX (vbox_general), label_god, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_god), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_god), 0, 0.5);
	
	hbox_god = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_god);
	gtk_box_pack_start (GTK_BOX (vbox_general), hbox_god, FALSE, TRUE, 0);
	
	size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	label_opt_read = gtk_label_new (_("DVD read drive"));
	gtk_widget_show (label_opt_read);
	gtk_box_pack_start (GTK_BOX (hbox_god), label_opt_read, FALSE, FALSE, 0);
	gtk_misc_set_padding (GTK_MISC (label_opt_read), 10, 0);
	gtk_size_group_add_widget (size_group, label_opt_read);
	
	pd->drive_selector = nautilus_burn_drive_selection_new ();
	gtk_widget_show (pd->drive_selector);
	gtk_box_pack_start (GTK_BOX (hbox_god), pd->drive_selector, FALSE, FALSE, 0);
	 
	label_gpt = gtk_label_new (_("<b>Disk Paths</b>"));
	gtk_widget_show (label_gpt);
	gtk_box_pack_start (GTK_BOX (vbox_general), label_gpt, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_gpt), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_gpt), 0, 0.5);
	
	hbox_image_dir = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_image_dir);
	gtk_box_pack_start (GTK_BOX (vbox_general), hbox_image_dir, FALSE, TRUE, 0);
	
	label_image_dir = gtk_label_new (_("Image folder"));
	gtk_widget_show (label_image_dir);
	gtk_box_pack_start (GTK_BOX (hbox_image_dir), label_image_dir, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label_image_dir), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (label_image_dir), 10, 0);
	gtk_size_group_add_widget (size_group, label_image_dir);
	
	pd->filechooserbutton_image_dir = gtk_file_chooser_button_new (_("Dvd Movie Backup - Select An Image Folder"), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
	gtk_widget_show (pd->filechooserbutton_image_dir);
	gtk_box_pack_start (GTK_BOX (hbox_image_dir), pd->filechooserbutton_image_dir, FALSE, FALSE, 0);
	
	hbox_temp_dir = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_temp_dir);
	gtk_box_pack_start (GTK_BOX (vbox_general), hbox_temp_dir, FALSE, TRUE, 0);
	
	label_temp_dir = gtk_label_new (_("Temp folder"));
	gtk_widget_show (label_temp_dir);
	gtk_box_pack_start (GTK_BOX (hbox_temp_dir), label_temp_dir, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label_temp_dir), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (label_temp_dir), 10, 0);
	gtk_size_group_add_widget (size_group, label_temp_dir);
	
	pd->filechooserbutton_temp_dir = gtk_file_chooser_button_new (_("Dvd Movie Backup - Select An Image Folder"), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
	gtk_widget_show (pd->filechooserbutton_temp_dir);
	gtk_box_pack_start (GTK_BOX (hbox_temp_dir), pd->filechooserbutton_temp_dir, FALSE, FALSE, 0);
	
	label_burners = gtk_label_new (_("<b>Burner</b>"));
	gtk_widget_show (label_burners);
	gtk_box_pack_start (GTK_BOX (vbox_general), label_burners, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_burners), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_burners), 0, 0.5);

	hbox8 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox8);
	gtk_box_pack_start (GTK_BOX (vbox_general), hbox8, TRUE, TRUE, 0);

	label_burn = gtk_label_new (_("Burner program"));
	gtk_widget_show (label_burn);
	gtk_box_pack_start (GTK_BOX (hbox8), label_burn, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label_burn), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (label_burn), 10, 0);
	gtk_size_group_add_widget (size_group, label_burn);
	g_object_unref (G_OBJECT (size_group));
	
	pd->combobox_burn = gtk_combo_box_new_text ();
	gtk_widget_show (pd->combobox_burn);
	gtk_box_pack_start (GTK_BOX (hbox8), pd->combobox_burn, FALSE, FALSE, 0);
  
	label_general = gtk_label_new (_("General"));
	gtk_widget_show (label_general);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (pd->notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (pd->notebook), 0), label_general);
	
	/* media */
	vbox_media = gtk_vbox_new (FALSE, 6);
	gtk_widget_show (vbox_media);
	gtk_container_add (GTK_CONTAINER (pd->notebook), vbox_media);
	gtk_container_set_border_width (GTK_CONTAINER (vbox_media), 12);
	
	label_mt = gtk_label_new (_("<b>Media Type</b>"));
	gtk_widget_show (label_mt);
	gtk_box_pack_start (GTK_BOX (vbox_media), label_mt, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_mt), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_mt), 0, 0.5);
	
	hbox_dvd = gtk_hbox_new (FALSE, 10);
	gtk_widget_show (hbox_dvd);
	gtk_box_pack_start (GTK_BOX (vbox_media), hbox_dvd, FALSE, TRUE, 0);
	
	pd->radiobutton_dvd = gtk_radio_button_new_with_mnemonic (NULL, _("DVD, 4.7GB Single Layer"));
	gtk_widget_show (pd->radiobutton_dvd);
	gtk_box_pack_start (GTK_BOX (hbox_dvd), pd->radiobutton_dvd, FALSE, FALSE, 10);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (pd->radiobutton_dvd), media_group);
	media_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (pd->radiobutton_dvd));
	
	hbox_dldvd = gtk_hbox_new (FALSE, 10);
	gtk_widget_show (hbox_dldvd);
	gtk_box_pack_start (GTK_BOX (vbox_media), hbox_dldvd, FALSE, TRUE, 0);
	
	pd->radiobutton_dldvd = gtk_radio_button_new_with_mnemonic (NULL, _("DVD, 8.5GB Duel Layer"));
	gtk_widget_show (pd->radiobutton_dldvd);
	gtk_box_pack_start (GTK_BOX (hbox_dldvd), pd->radiobutton_dldvd, FALSE, FALSE, 10);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (pd->radiobutton_dldvd), media_group);
	media_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (pd->radiobutton_dldvd));
	
	label_ms = gtk_label_new (_("<b>Media Settings</b>"));
	gtk_widget_show (label_ms);
	gtk_box_pack_start (GTK_BOX (vbox_media), label_ms, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_ms), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_ms), 0, 0.5);
	
	hbox_rf = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_rf);
	gtk_box_pack_start (GTK_BOX (vbox_media), hbox_rf, FALSE, FALSE, 0);
	
	pd->checkbutton_region_free = gtk_check_button_new_with_mnemonic (_("Make all backup disks region free"));
	gtk_widget_show (pd->checkbutton_region_free);
	gtk_box_pack_start (GTK_BOX (hbox_rf), pd->checkbutton_region_free, FALSE, FALSE, 10);
	
	label_media = gtk_label_new (_("Media"));
	gtk_widget_show (label_media);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (pd->notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (pd->notebook), 1), label_media);
	
	/* audio */
	vbox_audio = gtk_vbox_new (FALSE, 6);
	gtk_widget_show (vbox_audio);
	gtk_container_add (GTK_CONTAINER (pd->notebook), vbox_audio);
	gtk_container_set_border_width (GTK_CONTAINER (vbox_audio), 12);
	
	label_as = gtk_label_new (_("<b>Audio Selection</b>"));
	gtk_widget_show (label_as);
	gtk_box_pack_start (GTK_BOX (vbox_audio), label_as, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_as), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_as), 0, 0.5);
	
	hbox_audio_preferred = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_audio_preferred);
	gtk_box_pack_start (GTK_BOX (vbox_audio), hbox_audio_preferred, FALSE, TRUE, 0);
	
	size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	label_preferred_audio = gtk_label_new (_("Preferred track type"));
	gtk_widget_show (label_preferred_audio);
	gtk_box_pack_start (GTK_BOX (hbox_audio_preferred), label_preferred_audio, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label_preferred_audio), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (label_preferred_audio), 10, 0);
	gtk_size_group_add_widget (size_group, label_preferred_audio);
	
	pd->combobox_preferred_track = gtk_combo_box_new_text ();
	gtk_widget_show (pd->combobox_preferred_track);
	gtk_box_pack_start (GTK_BOX (hbox_audio_preferred), pd->combobox_preferred_track, FALSE, FALSE, 0);
	gtk_combo_box_append_text (GTK_COMBO_BOX (pd->combobox_preferred_track), _("AC3"));
	gtk_combo_box_append_text (GTK_COMBO_BOX (pd->combobox_preferred_track), _("LPCM"));
	
	hbox_audio_track_2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_audio_track_2);
	gtk_box_pack_start (GTK_BOX (vbox_audio), hbox_audio_track_2, FALSE, FALSE, 0);
	
	pd->checkbutton_include_dts_audio = gtk_check_button_new_with_mnemonic (_("Include 6ch DTS audio track if available"));
	gtk_widget_show (pd->checkbutton_include_dts_audio);
	gtk_box_pack_start (GTK_BOX (hbox_audio_track_2), pd->checkbutton_include_dts_audio, FALSE, FALSE, 10);
	
	label_al = gtk_label_new (_("<b>Audio Language</b>"));
	gtk_widget_show (label_al);
	gtk_box_pack_start (GTK_BOX (vbox_audio), label_al, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_al), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_al), 0, 0.5);
	
	hbox_audio_lang = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_audio_lang);
	gtk_box_pack_start (GTK_BOX (vbox_audio), hbox_audio_lang, FALSE, TRUE, 0);
	
	label_audio_language = gtk_label_new (_("Preferred language"));
	gtk_widget_show (label_audio_language);
	gtk_box_pack_start (GTK_BOX (hbox_audio_lang), label_audio_language, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label_audio_language), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (label_audio_language), 10, 0);
	gtk_size_group_add_widget (size_group, label_audio_language);
	g_object_unref (G_OBJECT (size_group));
	
	pd->combobox_audio_language = gtk_combo_box_new_text ();
	gtk_widget_show (pd->combobox_audio_language);
	gtk_box_pack_start (GTK_BOX (hbox_audio_lang), pd->combobox_audio_language, FALSE, FALSE, 0);
	
	hbox7 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox7);
	gtk_box_pack_start (GTK_BOX (vbox_audio), hbox7, TRUE, TRUE, 0);
	
	pd->checkbutton_audio_fallback_undefined = gtk_check_button_new_with_mnemonic (_("Use tracks with \"undefined\" language if needed"));
	gtk_widget_show (pd->checkbutton_audio_fallback_undefined);
	gtk_box_pack_start (GTK_BOX (hbox7), pd->checkbutton_audio_fallback_undefined, FALSE, FALSE, 10);
	
	label_audio = gtk_label_new (_("Audio"));
	gtk_widget_show (label_audio);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (pd->notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (pd->notebook), 2), label_audio);
	
	/* video */
	vbox_video = gtk_vbox_new (FALSE, 6);
	gtk_widget_show (vbox_video);
	gtk_container_add (GTK_CONTAINER (pd->notebook), vbox_video);
	gtk_container_set_border_width (GTK_CONTAINER (vbox_video), 12);
	
	label_vq = gtk_label_new (_("<b>Video Quality</b>"));
	gtk_widget_show (label_vq);
	gtk_box_pack_start (GTK_BOX (vbox_video), label_vq, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label_vq), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label_vq), 0, 0.5);
	
	/* not in use yet */
	/*
  	hbox_make_2disks = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_make_2disks);
	gtk_box_pack_start (GTK_BOX (vbox_video), hbox_make_2disks, FALSE, TRUE, 0);
	
	pd->checkbutton_make_2disks = gtk_check_button_new_with_mnemonic (_("Create two DVD disks rather than shrinking video"));
	gtk_widget_show (pd->checkbutton_make_2disks);
	gtk_box_pack_start (GTK_BOX (hbox_make_2disks), pd->checkbutton_make_2disks, FALSE, FALSE, 10);
	
	
	gtk_widget_set_sensitive (pd->checkbutton_make_2disks, FALSE);
	*/
	
	hbox_worst_quality = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_worst_quality);
	gtk_box_pack_start (GTK_BOX (vbox_video), hbox_worst_quality, FALSE, TRUE, 0);
	
	label_worst_quality = gtk_label_new (_("Warn if shrink factor exceeds"));
	gtk_widget_show (label_worst_quality);
	gtk_box_pack_start (GTK_BOX (hbox_worst_quality), label_worst_quality, FALSE, FALSE, 10);
	
	spinbutton_worst_factor_adj = gtk_adjustment_new (1.4, 1.10000002384, 2, 0.10000000149, 10, 10);
	pd->spinbutton_worst_factor = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton_worst_factor_adj), 1, 1);
	gtk_widget_show (pd->spinbutton_worst_factor);
	gtk_box_pack_start (GTK_BOX (hbox_worst_quality), pd->spinbutton_worst_factor, FALSE, FALSE, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (pd->spinbutton_worst_factor), TRUE);
	
	hbox_drop_audio_2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_drop_audio_2);
	gtk_box_pack_start (GTK_BOX (vbox_video), hbox_drop_audio_2, FALSE, TRUE, 0);
	
	pd->checkbutton_drop_audio_2 = gtk_check_button_new_with_mnemonic (_("Drop second audio track to maintain video quality"));
	gtk_widget_show (pd->checkbutton_drop_audio_2);
	gtk_box_pack_start (GTK_BOX (hbox_drop_audio_2), pd->checkbutton_drop_audio_2, FALSE, FALSE, 10);
	
	label_video = gtk_label_new (_("Video"));
	gtk_widget_show (label_video);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (pd->notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (pd->notebook), 3), label_video);
	
	/* buttons */
	dialog_action_area = GTK_DIALOG (dialog_prefs)->action_area;
	gtk_widget_show (dialog_action_area);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_END);
	
	helpbutton = gtk_button_new_from_stock ("gtk-help");
	gtk_widget_show (helpbutton);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog_prefs), helpbutton, GTK_RESPONSE_HELP);
	
	closebutton = gtk_button_new_from_stock ("gtk-close");
	gtk_widget_show (closebutton);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog_prefs), closebutton, GTK_RESPONSE_CLOSE);
	GTK_WIDGET_SET_FLAGS (closebutton, GTK_CAN_DEFAULT);

	lang = DVD_LANG_UNDEFINED + 1;
	while (lang < DVD_LANG_UNKNOWN) {
		gtk_combo_box_append_text (GTK_COMBO_BOX (pd->combobox_audio_language), dvd_language_get_name (lang));
		lang++;
	}
	pd->tab = PREFERENCES_DIALOG_TAB_GENERAL;
}

GType
preferences_dialog_get_type	(void)
{
	static GType pd_type = 0;

	if (pd_type == 0) {
		static const GTypeInfo pd_info = {
			sizeof (PreferencesDialogClass),
			NULL,
			NULL,
			(GClassInitFunc) preferences_dialog_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (PreferencesDialog),
			0,    /* n_preallocs    */
			(GInstanceInitFunc) preferences_dialog_init,
		};
		pd_type = g_type_register_static (GTK_TYPE_DIALOG,
						 "PreferencesDialog",
						  &pd_info, 0);
	}
	return pd_type;
}

static void
notebook_switch_page				(GtkNotebook		*notebook,
						 GtkNotebookPage	*page,
						 guint			 page_num,
						 PreferencesDialog	*pd)
{
	pd->tab = page_num;
	g_message ("notebook_switch_page %d", page_num);
}

static void
drive_selector_device_changed			(GtkWidget		*selection,
                             			 const char		*device_path,
						 PreferencesDialog	*pd)
{
	preferences_set_drive_path (pd->prefs, device_path);
}

static void
filechooserbutton_image_dir_current_folder_changed (GtkFileChooser	*filechooser,
						    PreferencesDialog	*pd)
{
	gchar *new_dir;
	
	new_dir = gtk_file_chooser_get_filename (filechooser);
	if (new_dir != NULL) {
		preferences_set_image_folder (pd->prefs, new_dir);
		g_free (new_dir);
	}
}

static void
filechooserbutton_temp_dir_current_folder_changed  (GtkFileChooser	*filechooser,
						    PreferencesDialog	*pd)
{
	gchar *new_dir;
	
	new_dir = gtk_file_chooser_get_filename (filechooser);
	if (new_dir != NULL) {
		preferences_set_temp_folder (pd->prefs, new_dir);
		g_free (new_dir);
	}
}

static void
combobox_burn_changed				(GtkComboBox		*combobox,
						 PreferencesDialog	*pd)
{
	gint active;
	
	active = gtk_combo_box_get_active (combobox);
	if (active == -1) return;
	if ((active == 0) &&
	    (pd->prefs->have_nautilus_cd_burner == TRUE)) {
		preferences_set_burn_command (pd->prefs, BURNER_COMMAND_NAUTILUS_CD_BURNER);
	} else {
		preferences_set_burn_command (pd->prefs, BURNER_COMMAND_BRASERO);
	}
}

static void
radiobutton_dvd_toggled				(GtkToggleButton	*togglebutton,
						 PreferencesDialog	*pd)
{
	if (gtk_toggle_button_get_active (togglebutton) == TRUE) {
		preferences_set_media (pd->prefs, DVD_MEDIA_SL);
	} else {
		preferences_set_media (pd->prefs, DVD_MEDIA_DL);
	}
}

static void
checkbutton_region_free_toggled			(GtkToggleButton	*togglebutton,
						 PreferencesDialog	*pd)
{
	preferences_set_region_free (pd->prefs,
				       gtk_toggle_button_get_active (togglebutton));
}

static void
combobox_preferred_track_changed			(GtkComboBox		*combobox,
						 PreferencesDialog	*pd)
{
	gint active;
	
	active = gtk_combo_box_get_active (GTK_COMBO_BOX (combobox));
	if (active == -1) return;
	
	switch (active) {
	case 0:
		preferences_set_audio_format (pd->prefs, DVD_AUDIO_FORMAT_AC3);
		break;
	case 1:
		preferences_set_audio_format (pd->prefs, DVD_AUDIO_FORMAT_LPCM);
		break;
	default:
		g_assert_not_reached ();
	}
}

static void
checkbutton_include_dts_audio_toggled		(GtkToggleButton	*togglebutton,
						 PreferencesDialog	*pd)
{
	preferences_set_include_dts (pd->prefs,
				     gtk_toggle_button_get_active (togglebutton));
}

static void
combobox_audio_language_changed			(GtkComboBox		*combobox,
						 PreferencesDialog	*pd)
{
	gint active;
	
	active = gtk_combo_box_get_active (combobox);
	if (active == -1) return;
	preferences_set_language  (pd->prefs, active + 1);
}

static void
checkbutton_audio_fallback_undefined_toggled	(GtkToggleButton	*togglebutton,
						 PreferencesDialog	*pd)
{
	preferences_set_use_undefined (pd->prefs,
				       gtk_toggle_button_get_active (togglebutton));
}

static void
spinbutton_worst_factor_value_changed		(GtkSpinButton	*spinbutton,
						 PreferencesDialog	*pd)
{
	preferences_set_maximum_shrink (pd->prefs,
					(gfloat) gtk_spin_button_get_value (spinbutton));
}

static void
checkbutton_drop_audio_2_toggled		(GtkToggleButton	*togglebutton,
						 PreferencesDialog	*pd)
{
	preferences_set_drop_dts (pd->prefs,
				  gtk_toggle_button_get_active (togglebutton));
}


GtkWidget
*preferences_dialog_new		(Preferences	*prefs,
				 GtkWindow	*parent)
{
	GtkWidget *w;
	PreferencesDialog *pd;
	gchar *drive_path;
	gchar *img_folder;
	gchar *tmp_folder;
	gchar *burn_command;
	DvdMedia media;
	gboolean region_free;
	DvdAudioFormat audio_format;
	gboolean include_dts;
	DvdLanguage language;
	gboolean allow_undefined;
	gfloat max_shrink;
	gboolean drop_dts;
	
	g_assert (prefs != NULL);
	
	w = gtk_type_new (preferences_dialog_get_type ());
	pd = PREFERENCES_DIALOG (w);
	
	pd->prefs	= g_object_ref (G_OBJECT (prefs));
	drive_path	= preferences_get_drive_path (pd->prefs);
	burn_command	= preferences_get_burn_command (pd->prefs);
	img_folder	= preferences_get_image_folder (pd->prefs);
	tmp_folder	= preferences_get_temp_folder (pd->prefs);
	media		= preferences_get_media (pd->prefs);
	region_free	= preferences_get_region_free (pd->prefs);
	audio_format	= preferences_get_audio_format (pd->prefs);
	include_dts	= preferences_get_include_dts (pd->prefs);
	language	= preferences_get_language (pd->prefs);
	allow_undefined	= preferences_get_use_undefined (pd->prefs);
	max_shrink	= preferences_get_maximum_shrink (pd->prefs);
	drop_dts	= preferences_get_drop_dts (pd->prefs);
	
	
	if (parent != NULL) {
		gtk_window_set_transient_for (GTK_WINDOW (pd), parent);
		gtk_window_set_position (GTK_WINDOW (pd), GTK_WIN_POS_CENTER_ON_PARENT);
	}
	if (drive_path != NULL) {
		nautilus_burn_drive_selection_set_device (NAUTILUS_BURN_DRIVE_SELECTION (pd->drive_selector), drive_path);
	} else {
		drive_path = g_strdup (nautilus_burn_drive_selection_get_default_device (NAUTILUS_BURN_DRIVE_SELECTION (pd->drive_selector)));
		if (drive_path == NULL) {
			drive_path = g_strdup (nautilus_burn_drive_selection_get_device (NAUTILUS_BURN_DRIVE_SELECTION (pd->drive_selector)));
		}
		if (drive_path == NULL) {
			g_error ("no optical drives configured");
		} else {
			preferences_set_drive_path (pd->prefs, drive_path);
		}
	}
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (pd->filechooserbutton_image_dir),
					     img_folder);
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (pd->filechooserbutton_temp_dir),
					     tmp_folder);
	if (prefs->have_nautilus_cd_burner == TRUE) {
		gtk_combo_box_append_text (GTK_COMBO_BOX (pd->combobox_burn), _("Nautilus CD Burner"));
	}
	
	if (prefs->have_brasero == TRUE) {
		gtk_combo_box_append_text (GTK_COMBO_BOX (pd->combobox_burn), _("Brasero"));
	}
	if (burn_command != NULL) {
		if (strcmp (burn_command, BURNER_COMMAND_BRASERO) == 0) {
			if (prefs->have_nautilus_cd_burner == TRUE) {
				gtk_combo_box_set_active (GTK_COMBO_BOX (pd->combobox_burn), 1);
			} else {
				gtk_combo_box_set_active (GTK_COMBO_BOX (pd->combobox_burn), 0);
			}
		} else {
			gtk_combo_box_set_active (GTK_COMBO_BOX (pd->combobox_burn), 0);
		}
	}
	if (media == DVD_MEDIA_SL) {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pd->radiobutton_dvd), TRUE);
	} else {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pd->radiobutton_dldvd), TRUE);
	}
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pd->checkbutton_region_free), region_free);
	if (audio_format == DVD_AUDIO_FORMAT_AC3) {
		gtk_combo_box_set_active (GTK_COMBO_BOX (pd->combobox_preferred_track), 0);
	} else {
		gtk_combo_box_set_active (GTK_COMBO_BOX (pd->combobox_preferred_track), 1);
	}
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pd->checkbutton_include_dts_audio), include_dts);
	gtk_combo_box_set_active (GTK_COMBO_BOX (pd->combobox_audio_language), language - 1);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pd->checkbutton_audio_fallback_undefined), allow_undefined);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (pd->spinbutton_worst_factor), (gdouble) max_shrink);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pd->checkbutton_drop_audio_2), drop_dts);
	g_free (drive_path);
	g_free (img_folder);
	g_free (tmp_folder);
	g_free (burn_command);
	g_signal_connect ((gpointer) pd->notebook, "switch_page",
			  G_CALLBACK (notebook_switch_page),
			  pd);
	g_signal_connect ((gpointer) pd->drive_selector, "device_changed",
			  G_CALLBACK (drive_selector_device_changed),
			  pd);
	g_signal_connect ((gpointer) pd->filechooserbutton_image_dir, "current_folder_changed",
			  G_CALLBACK (filechooserbutton_image_dir_current_folder_changed),
			  pd);
	g_signal_connect ((gpointer) pd->filechooserbutton_temp_dir, "current_folder_changed",
			  G_CALLBACK (filechooserbutton_temp_dir_current_folder_changed),
			  pd);
	g_signal_connect ((gpointer) pd->combobox_burn, "changed",
			  G_CALLBACK (combobox_burn_changed),
			  pd);
	g_signal_connect ((gpointer) pd->radiobutton_dvd, "toggled",
			  G_CALLBACK (radiobutton_dvd_toggled),
			  pd);
	g_signal_connect ((gpointer) pd->checkbutton_region_free, "toggled",
			  G_CALLBACK (checkbutton_region_free_toggled),
			  pd);
	g_signal_connect ((gpointer) pd->combobox_preferred_track, "changed",
			  G_CALLBACK (combobox_preferred_track_changed),
			  pd);
	g_signal_connect ((gpointer) pd->checkbutton_include_dts_audio, "toggled",
			  G_CALLBACK (checkbutton_include_dts_audio_toggled),
			  pd);
	g_signal_connect ((gpointer) pd->combobox_audio_language, "changed",
			  G_CALLBACK (combobox_audio_language_changed),
			  pd);
	g_signal_connect ((gpointer) pd->checkbutton_audio_fallback_undefined, "toggled",
			  G_CALLBACK (checkbutton_audio_fallback_undefined_toggled),
			  pd);
	g_signal_connect ((gpointer) pd->spinbutton_worst_factor, "value_changed",
			  G_CALLBACK (spinbutton_worst_factor_value_changed),
			  pd);
	g_signal_connect ((gpointer) pd->checkbutton_drop_audio_2, "toggled",
			  G_CALLBACK (checkbutton_drop_audio_2_toggled),
			  pd);
	return w;
}

PreferencesDialogTab
preferences_dialog_get_tab	(PreferencesDialog	*preferences_dialog)
{
	return preferences_dialog->tab;
}

