/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvd.h>
#include "pes.h" /* only for getbits () */


static DvdDecoderClass *dvd_decoder_lpcm_parent_class = NULL;

static void     dvd_decoder_lpcm_class_init	(DvdDecoderLpcmClass	*class);
static void     dvd_decoder_lpcm_instance_init	(GTypeInstance	*instance,
						 gpointer	 g_class);
static void     dvd_decoder_lpcm_dispose	(GObject	*object);

static void	dvd_decoder_lpcm_read_startcode	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	dvd_decoder_lpcm_read_header	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	dvd_decoder_lpcm_read_frame	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	calculate_bytes_per_frame	(DvdDecoderLpcm	 *lpcm_decoder);

static void
dvd_decoder_lpcm_class_init	(DvdDecoderLpcmClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_decoder_lpcm_parent_class = g_type_class_ref (DVD_DECODER_TYPE);
	
	object_class->dispose = dvd_decoder_lpcm_dispose;
}

static void
dvd_decoder_lpcm_instance_init	(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdDecoder *decoder;
	DvdDecoderLpcm *lpcm_decoder;
	
	decoder = DVD_DECODER (instance);
	lpcm_decoder = DVD_DECODER_LPCM (instance);
	decoder->stream_type = DVD_STREAM_LPCM_AUDIO;
	
	/* these functions called by parent DvdDecoder */
	decoder->read_startcode = dvd_decoder_lpcm_read_startcode;
	decoder->read_header    = dvd_decoder_lpcm_read_header;
	decoder->read_frame     = dvd_decoder_lpcm_read_frame;
	
	/* sane defaults */
	lpcm_decoder->quantization = DVD_AUDIO_QUANT_16BIT;
	lpcm_decoder->samplerate = DVD_AUDIO_SAMP_48000;
	lpcm_decoder->clocks_per_frame = 150;
	lpcm_decoder->channels = 2;
	calculate_bytes_per_frame (lpcm_decoder);
	g_assert (lpcm_decoder->bytes_per_frame == 320);
}

static void
dvd_decoder_lpcm_dispose	(GObject	*object)
{
	DvdDecoderLpcm *decoder;
	
	decoder = DVD_DECODER_LPCM (object);
	
	G_OBJECT_CLASS (dvd_decoder_lpcm_parent_class)->dispose (object);
}

/**
 * dvd_decoder_lpcm_get_type
 * @return The GType for the DvdDecoderLpcm class.
 */
GType
dvd_decoder_lpcm_get_type	(void)
{
	static GType decoder_lpcm_type = 0;

	if (decoder_lpcm_type == 0) {
		GTypeInfo decoder_lpcm_info = {
			sizeof (DvdDecoderLpcmClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_decoder_lpcm_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdDecoderLpcm),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_decoder_lpcm_instance_init
	    	};
		decoder_lpcm_type = g_type_register_static (DVD_DECODER_TYPE,
						  	    "DvdDecoderLpcm",
						 	    &decoder_lpcm_info, 0);
	}
	return decoder_lpcm_type;
}


static void
dvd_decoder_lpcm_read_startcode	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	dvd_decoder_set_start_pts (decoder, pts);
	dvd_decoder_set_current_pts (decoder, pts);
	dvd_decoder_set_state (decoder, DVD_DECODER_STATE_FRAME);
}

static void
dvd_decoder_lpcm_read_header	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	/* lpcm has no headers! */
}

static void
dvd_decoder_lpcm_read_frame	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	DvdDecoderLpcm *lpcm_decoder;
	guint16 length;
	guint16 data_length;
	
	lpcm_decoder = DVD_DECODER_LPCM (decoder);
	
	data_length = end - *start;
	length = lpcm_decoder->bytes_per_frame - decoder->buffer_pos;
	if (length > data_length) {
		dvd_decoder_write_frame_bytes (decoder, *start, data_length);
		*start = end;
	} else {
		dvd_decoder_write_frame_bytes (decoder, *start, length);
		dvd_decoder_output_frame (decoder);
		*start = (*start) + length;
		/* no change of decoder state */
		/* add next frames PTS here */
		dvd_decoder_add_to_current_pts (decoder, lpcm_decoder->clocks_per_frame);
	}
}

static void
calculate_bytes_per_frame		(DvdDecoderLpcm	*lpcm_decoder)
{
	guint samples;
	guint quant;
	
	switch (lpcm_decoder->quantization) {
	case DVD_AUDIO_QUANT_16BIT:
		quant = 16;
		break;
	case DVD_AUDIO_QUANT_20BIT:
		quant = 20;
		break;
	case DVD_AUDIO_QUANT_24BIT:
		quant = 24;
		break;
	case DVD_AUDIO_QUANT_DRC:
		/* does not apply to LPCM */
		g_assert_not_reached ();
	default:
		g_assert_not_reached ();
	}
	switch (lpcm_decoder->samplerate) {
	case DVD_AUDIO_SAMP_48000:
		samples = 48000;
		break;
	case DVD_AUDIO_SAMP_96000:
		samples = 96000;
		break;
	case DVD_AUDIO_SAMP_44100:
		samples = 44100;
		break;
	case DVD_AUDIO_SAMP_32000:
		samples = 32000;
		break;
	default:
		g_assert_not_reached ();
	}
	lpcm_decoder->bytes_per_frame = samples * quant * lpcm_decoder->channels / (90000 / lpcm_decoder->clocks_per_frame) / 8;
	dvd_decoder_set_frame_clocks (DVD_DECODER (lpcm_decoder), lpcm_decoder->clocks_per_frame * 300);
}

void
dvd_decoder_lpcm_set_quantization		(DvdDecoderLpcm	*lpcm_decoder,
						 DvdAudioQuant	 quantization)
{
	lpcm_decoder->quantization = quantization;
	calculate_bytes_per_frame (lpcm_decoder);
}

void
dvd_decoder_lpcm_set_samplerate			(DvdDecoderLpcm	*lpcm_decoder,
						 DvdAudioSamp	 samplerate)
{
	lpcm_decoder->samplerate = samplerate;
	calculate_bytes_per_frame (lpcm_decoder);
}

void
dvd_decoder_lpcm_set_clocks_per_frame		(DvdDecoderLpcm	*lpcm_decoder,
						 guint32	 clocks_per_frame)
{
	lpcm_decoder->clocks_per_frame = clocks_per_frame;
	calculate_bytes_per_frame (lpcm_decoder);
}

void
dvd_decoder_lpcm_set_channels			(DvdDecoderLpcm	*lpcm_decoder,
						 guint8		 channels)
{
	lpcm_decoder->channels = channels;
	calculate_bytes_per_frame (lpcm_decoder);
}
