//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_car,vel_car,jacob]=CL__oe_kep2carEllip(kep,mu)

// Authors
// CNES - DCT/SB
//
// Bibliography
// MSLIB 90
//

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

[lhs,rhs]=argn();

if rhs==1 then mu=%CL_mu; end
if lhs==3
  compute_jacob=%t
else
  compute_jacob=%f
end

N = size(kep,2);

a    = kep(1,:)
ecc  = kep(2,:)
ai   = kep(3,:)
apom = kep(4,:)
agom = kep(5,:)
am   = kep(6,:)

ae=CL_kp_M2E(ecc,am) // calcul anomalie eccentrique E par equation de kepler

// 1- calcul vecteur position r = a (ld0*p + mu0*q)
//    et vecteur vitesse rv = sqrt(mu/a) (nu0*p + ki0*q)
//******************************************************
// calcul de variables intermediaires

cai = cos(ai)       //
sai = sin(ai)       //
cgo = cos(agom)     //
sgo = sin(agom)     //    cosinus et sinus des angles i,gomega,pomega,E
cpo = cos(apom)     //
spo = sin(apom)     //
cae = cos(ae)       //
sae = sin(ae)       //

cgocpo = cgo .* cpo
sgocpo = sgo .* cpo
sgospo = sgo .* spo
cgospo = cgo .* spo

v1 = sqrt(1-(ecc.*ecc))//     calcul des variables sqrt(1-e**2)
v2 = 1 - (ecc.*cae)    //     et 1-e*cosE

p = zeros(3,size(kep,2));
q = p;
p(1,:) = cgocpo - (cai .* sgospo)   //
p(2,:) = sgocpo + (cai .* cgospo)   //
p(3,:) = sai .* spo                 // calcul des composantes des vecteurs p et q en fonction de
q(1,:) = - cgospo - (cai .* sgocpo) // angles i,pomega,gomega
q(2,:) = - sgospo + (cai .* cgocpo) //
q(3,:) = sai .* cpo                 //

rld0 = cae - ecc    // calcul de ld0 = cosE - e et de
rmu0 = sae .* v1     // mu0 = sinE * sqrt(1-e**2)

sqmusa = sqrt(mu./a)    // calcul de sqmusa = sqrt(mu/a)
rnu0 = -sae ./ v2       //        de nu0 = -sinE/(1-e*cosE)
rki0 = (cae .* v1) ./ v2 //     et de ki0 = cosE*sqrt(1-e**2) / 1-e*cosE

rld0_2 = rld0.*.ones(3,1)     //
rmu0_2 = rmu0.*.ones(3,1)     //
rnu0_2 = rnu0.*.ones(3,1)     // auxiliar variables
rki0_2 = rki0.*.ones(3,1)     //
a_2 = a.*.ones(3,1)           //
sqmusa_2 = sqmusa.*.ones(3,1) //
pos_car = a_2 .* ((rld0_2 .* p) + (rmu0_2 .* q))       // calcul des composantes des vecteurs position
vel_car = sqmusa_2 .* ((rnu0_2 .* p) + (rki0_2 .* q))  // et vitesse

// 2- calcul du jacobien de la transformation
// ********************************************
if compute_jacob
  xdpai = zeros(3,size(kep,2));
  xdqai = xdpai;
  xdpgo = xdpai;
  xdpai(1,:) = sai  .* sgospo  //
  xdpai(2,:) = -sai .* cgospo  //
  xdpai(3,:) = cai  .* spo     // derivees des composantes de p et q par rapport a i
  xdqai(1,:) = sai  .* sgocpo  //
  xdqai(2,:) = -sai .* cgocpo  //
  xdqai(3,:) = cai  .* cpo     //

  xdpgo(1,:) = - sgocpo - (cai .* cgospo) //
  xdpgo(2,:) = cgocpo - (cai .* sgospo)   //
  xdpgo(3,:) = 0                // derivees des composantes de p et q par rapport a gomega
  xdqgo(1,:) = sgospo - (cai .* cgocpo)   //
  xdqgo(2,:) = - cgospo - (cai .* sgocpo) //
  xdqgo(3,:) = 0                //

  xdldex = -1     //
  xdmuex = sae .* (-ecc./v1) //  calcul des derivees de ld0 et mu0 par rapport a e et E
  xdldae = -sae            //
  xdmuae = cae .* v1        //

  xdaeam = 1 ./ v2  // calcul derivees de E par rapport a M et e
  xdaeex = sae .* xdaeam     // dE/dam = 1 / 1-ecc*cosE et dE/dex = sinE / 1-ecc*cosE

  // calcul des derivees de nu0 et ki0 par rapport a e et E
  // avec les variables : xdaeam = 1/v2  et xdaeex = sae/v2
  xdnuex = -cae .* xdaeam .* xdaeex                        //     calcul de xdnuex = (-cae*sae) / (v2**2)
  xdkiex = xdaeam .* cae .* ((-ecc./v1) + cae.*v1.*xdaeam)    //     calcul de xdkiex=((-ecc*cae)/(v1*v2)) + (((cae**2)*v1)/(v2**2))
  xdnuae = xdaeam .* (-cae + (ecc.*sae.*xdaeex))            //     calcul de xdnuae = (-cae/v2) + ((ecc*(sae**2)) / (v2**2))
  xdkiae = v1 .* xdaeex .* (-1 - (ecc.*cae.*xdaeam))//     calcul de xdkiae = v1 * ((-sae/v2) - ((ecc*cae*sae) / (v2**2)))

  //===========================================================
  // calcul des derivees partielles des vecteurs position et
  // vitesse par rapport a a,e,i,pomega,gomega,M
  //===========================================================
  xdmuex_2 = xdmuex.*.ones(3,1)
  xdldae_2 = xdldae.*.ones(3,1)
  xdmuae_2 = xdmuae.*.ones(3,1)
  xdaeex_2 = xdaeex.*.ones(3,1)
  xdnuex_2 = xdnuex.*.ones(3,1)
  xdkiex_2 = xdkiex.*.ones(3,1)
  xdnuae_2 = xdnuae.*.ones(3,1)
  xdkiae_2 = xdkiae.*.ones(3,1)
  xdaeam_2 = xdaeam.*.ones(3,1)
  //jnmk = jacob(n:m,k)
  j131 = (rld0_2 .* p) + (rmu0_2 .* q)      // ---par rapport a a
  j461 = (-0.5*sqmusa_2./a_2) .*((rnu0_2 .* p) + (rki0_2 .* q))

  j132 = a_2 .* ( ( (xdldex*p) + (xdmuex_2.*q) ) + ( (xdldae_2.*p) + (xdmuae_2.*q) ) .* xdaeex_2 )     // ---par rapport a e
  j462 = sqmusa_2 .* (((xdnuex_2.*p) + (xdkiex_2.*q)) +((xdnuae_2.*p) + (xdkiae_2.*q)) .* xdaeex_2)
  //( = derivee/e + derivee/E * derivee E/e)

  j133 = a_2 .* ((rld0_2 .* xdpai) + (rmu0_2 .* xdqai))      // ---par rapport a i
  j463 = sqmusa_2 .* ((rnu0_2 .* xdpai) + (rki0_2 .* xdqai))

  j134 = a_2 .* ((rld0_2 .* q) - (rmu0_2 .* p))             // ---par rapport a pomega (q = derivee de p /pomega)
  j464 = sqmusa_2 .* ((rnu0_2 .* q) - (rki0_2 .* p))

  j135 = a_2 .* ((rld0_2 .* xdpgo) + (rmu0_2 .* xdqgo))     // ---par rapport a gomega
  j465 = sqmusa_2 .* ((rnu0_2 .* xdpgo) + (rki0_2 .* xdqgo))

  j136 = a_2 .* ((xdldae_2 .* p) + (xdmuae_2 .* q)) .* xdaeam_2 // ---par rapport a M ( = derivee/E * derivee de E/M)
  j466 = sqmusa_2 .* ((xdnuae_2 .* p) + (xdkiae_2 .* q)) .*xdaeam_2

  // Jacobienne
  jacob=hypermat([6,6,N],[j131;j461;j132;j462;j133;j463;j134;j464;j135;j465;j136;j466] );
  if (N==1) then jacob=jacob(:,:,1); end
end

endfunction

