//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_cir] = CL_ex_meanEckHech(osc_cir,er,mu,zonals)
// Eckstein Hechler orbit propagation analytical model (mean elements)
//
// Calling Sequence
// [mean_cir] = CL_ex_meanEckHech(osc_cir [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// Computes the mean orbital elements <emphasis role="bold">mean_cir</emphasis> from the osculating elements <emphasis role="bold">osc_cir</emphasis>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="cir_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>Warnings :
// <para> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</para>
// <para> - This function nominally works for eccentricities smaller than 5.e-3</para>
// <para> - This function works with a lesser accuracy for eccentricities between 5.e-3 and 0.1</para>
// <para> - This function does not work for eccentricities greater than 0.1</para></listitem>
// </itemizedlist>
//
// Parameters
// osc_cir: Osculating circular-adapted elements [sma;ex;ey;inc;raan;pom+anm] (6xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonals coefficients J1 to Jn (troncated to J6) to be used (default is %CL_j1jn(1:6)) (Nz x 1)
// mean_cir: Mean circular-adapted elements [sma;ex;ey;inc;raan;pom+anm] (6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume E (me_eck_hech)
//
// See also
// CL_ex_lyddane
// CL_ex_eckHech
//
// Examples
// osc_cir = [42166712;2.9e-3;1.2e-3;CL_deg2rad([97.2;125;5])];
// mean_cir = CL_ex_meanEckHech(osc_cir)
// // check 
// [mean_cir2,osc_cir2] = CL_ex_eckHech(0,mean_cir,0) 


//************************************************************************
//
// But:  Modele analytique d'extrapolation d'orbite de ECKSTEIN-HECHLER.
// ===
//
// Note d'utilisation:  L'excentricite  doit etre superieure >= 0 et < 0.1.
// ==================   Le domaine d'utilisation du modele est .. [ 0E+0 , 5E-3 [.
//                      Le domaine de precision degradee est .... [ 5E-3 , 1E-1 [.
//                      Le domaine d'erreur est ................. [ 1E-1 , +Oo  [.
//                      L'inclinaison  doit etre > 0. et < pi et non proches des inclinaisons critiques
//                      ( pm_i_critique_non_retro, pm_i_critique_retro ) a pm_eps_i_critique pres.
//
// Historique:
// ==========
//   + Version 2.0 (SP 341 ed01 rev00): creation a partir de la routine MEEHJ6 de la MSLIB f77
//   + Version 3.1 (DE globale 439 ed01 rev00) : ajout des champs %biblio et %message pour le code retour
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 6.0 (DE globale 618 ed01 rev00): ajout de tests sur mu et les coefficients CN0
//************************************************************************

// Declarations:
if(~exists('%CL_criticalInc1')) then global %CL_criticalInc1; end;
if(~exists('%CL_criticalInc2')) then global %CL_criticalInc2; end;
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_j1jn')) then global %CL_j1jn; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:6); end


// initialisation de la valeur epsilon 
eps100 = 100 * %eps

// controle des donnees
// demi grand axe
if or( osc_cir(1,:) <= 0 )
  CL__error("Error: semi major axis <= 0");
end

// eccentricite
ecc = sqrt(osc_cir(2,:).^2+(osc_cir(3,:).^2));
if or( ecc > 0.1 )
  CL__error("Error: eccentricity higher than 0.1");
end
if or( ecc > 5.e-3)
  warning("*** CelestLab (Warning): lesser precision for eccentricities between 5.e-3 and 0.1");
end

// inclinaison
if or( osc_cir(4,:) < 0 )
  CL__error("Negative inclination");
end
if or( osc_cir(4,:) > %pi )
  CL__error("Inclination > pi");
end

// inclinaison proche de l'inclinaison critique.
eps_i_critique = 4.e-5; // obtenu experimentalement : au dela ca plante
if (abs(osc_cir(4,:) - %CL_criticalInc1) < eps_i_critique | ...
     abs(osc_cir(4,:) - %CL_criticalInc2) < eps_i_critique) then
   CL__error("inclination close (less than "+string(eps_i_critique)+" rad) to the critical value");
end


// interface de eckstein-hechler (paramtres adapts circ) 
// NB: er,mu,zonals : are those from the upper level 
function [osc_cir] = Calcul_osc(moy_cir)
  t0 = 0; 
  t1 = 0; 
  [moy1, osc1] = CL_ex_eckHech(t0, moy_cir, t1, er,mu,zonals);
  osc_cir = osc1; 
endfunction

// ajustement des paramtres moyens (mod 2*%pi + traitement inclinaison si > pi) 
function [moy_cir1] = Ajuste(moy_cir)
  moy_cir1 = moy_cir;
  moy_cir1(4,:) = CL_rMod(moy_cir1(4,:), 0, 2*%pi); // inclination
  I = find(moy_cir1(4,:) > %pi); 
  if (I ~= [])
    moy_cir1(4,I) = 2*%pi - moy_cir1(4,I); // inc
    moy_cir1(5,I) = moy_cir1(5,I) + %pi;   // gom 
  end
  moy_cir1(5:6,:) = CL_rMod(moy_cir1(5:6,:), 0, 2*%pi);
endfunction


// initialisation des sorties : moyens = osculateurs
mean_cir = osc_cir;
mean_cir(5:6,:) = CL_rMod(mean_cir(5:6,:), 0, 2*%pi);
osc = zeros(osc_cir);
ecart = zeros(osc_cir); 

// ecarts max admis
ecart_admi = eps100 * (1 + abs(mean_cir));


// debut d'iteration
I = 1 : size(osc_cir,2);  // I : indices non OK 
itermax = 50; // nombre maximum d'iterations admises
iter = 0; // current number of iterations

while (I ~= [] & iter <= itermax)  

  // calcul des osculateurs correspondant aux moyens (indices courants)
  osc(:,I) = Calcul_osc(mean_cir(:,I));

  // calcul des ecarts entre parametres osculateurs calcules et ceux de depart
  // Les ecarts i, gom, psom sont ramenes sur [-pi,pi]
  ecart(:,I)  = osc_cir(:,I) - osc(:,I);
  ecart(4:6,I) = CL_rMod(ecart(4:6,I), -%pi, %pi); 

  // mise  jour des moyens (indices non convergs)
  mean_cir(:,I) = Ajuste(mean_cir(:,I) + ecart(:,I)); 

  //  indices non encore OK
  I = find( max(abs(ecart) - ecart_admi, 'r') > 0 );

  iter = iter + 1

end

if (I ~= [])
  CL__error("No convergence in computation of mean elements")
end 

mean_cir(5:6,:) = CL_rMod(mean_cir(5:6,:), osc_cir(5:6,:)-%pi, osc_cir(5:6,:)+%pi);

endfunction
