//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [M] = CL_kp_E2M(ecc,E)
// Eccentric anomaly to mean anomaly 
//
// Calling Sequence
// M = CL_kp_E2M(ecc,E)
//
// Description
// <itemizedlist><listitem>
// Computes mean anomaly from eccentric anomaly and eccentricity for elliptical 
// and hyperbolic orbits:
// <para> - Elliptical orbits ( ecc < 1 ) : </para>
// <para> <emphasis role="bold">M = E - ecc * sin(E)</emphasis> </para>
// <para> - Hyperbolic orbits ( ecc > 1 ) : </para>
// <para> <emphasis role="bold">M = ecc * sinh(E) - E</emphasis> </para>
// </listitem>
// </itemizedlist>
//
// Parameters
// E: Eccentric anomaly [rad] (PxN or Px1)
// ecc: Eccentricity (PxN or Px1)
// M: Mean anomaly [rad] (PxN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Orbital Mechanics for Engineering Students, H D Curtis, Chapter 3, equation 3.11
// 2 Mecanique spatiale, CNES - Cepadues 1995, Tome I, section 4.6.6, page 181
//
// See also
// CL_kp_M2Ecir
// CL_kp_M2E
// CL_kp_E2v
// CL_kp_v2E
//
// Examples
// // Consistency test: 
// E = [1.7281 5.7281];
// ecc = [0.37255 1.55];
// M = CL_kp_E2M(ecc,E);
// E2 = CL_kp_M2E(ecc,M)

// Declarations:
if(~exists('%CL_epsilon')) then global %CL_epsilon; end;

// Code:

Ne = size(ecc,2);
NE = size(E,2);
N = max(Ne,NE);

if ~( NE==1 | Ne==1 | NE==Ne )
  CL__error('bad size of input arguments');
end

if Ne == 1; ecc = ecc * ones(1,N); end
if NE == 1; E = E * ones(1,N); end

if find(ecc < 0)~=[] 
  CL__error('Negative eccentricity');
end

if find(ecc >= 1-%CL_epsilon.parab & ecc <= 1+%CL_epsilon.parab) ~= [] 
  CL__error('Parabola case not handled');
end

M = zeros(E);

I = find(ecc < 1); // ellipse
M(I) = E(I) - ecc(I).*sin(E(I));

I = find(ecc > 1);  // hyperbola
M(I) = ecc(I).*sinh(E(I)) - E(I);

endfunction


