//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [dv, ecc, tanoe] = CL_ip_insertionDv(vinf, rph, sma, tanoh, mu)
// Delta-V needed for an insertion around a planet
//
// Calling Sequence
// [dv, ecc, tanoe] = CL_ip_insertionDv(vinf, rph, sma [, tanoh, mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the delta-v needed for the transfer from an hyperbolic orbit to an elliptical orbit.
// <para>The insertion maneuver is tangential, i.e. the delta-V is parallel to the velocity vector on the hyperbolic orbit. </para>
// <para>The initial orbit is defined by its velocity at infinity (<emphasis role="bold">vinf</emphasis>) and periapsis radius (<emphasis role="bold">rph</emphasis>).</para>
// <para>The final orbit is defined by its semi major axis(<emphasis role="bold">sma</emphasis>).</para>
// <para>The true anomaly of the maneuver on the initial orbit (<emphasis role="bold">tanoh</emphasis>) can optionally be specified. By default tanoh=0 (meaning 'at the periapsis').</para>
// <para>The planet is defined by its gravitational constant mu (default is %CL_mu)</para></listitem>
// </itemizedlist>
//
// Parameters
// vinf: Velocity on hyperbolic orbit at infinity [m/s]. (1xN)
// rph: Periapsis radius of hyperbolic orbit [m]. (1xN)
// sma: Semi-major axis of target (elliptical) orbit [m]. (1xN)
// tanoh: (optional) True anomaly of the maneuvre (on the hyperbolic orbit) [rad]. Default value is 0. (1xN)  
// mu: (optional) gravitational constant [m3/s2]. Default value is %CL_mu.
// dv: Norm of the delta-v. (1xN)
// ecc: Eccentricity of the final (elliptical) orbit. (1xN)
// tanoe: True anomaly on the elliptical orbit at the time of the maneuver [rad]. (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Insertion around Earth:
// vinf = [5 , 6] * 1.e3;
// rph = [%CL_eqRad+250.e3 , %CL_eqRad+500.e3];
// sma = [%CL_eqRad+350.e3 , %CL_eqRad+700.e3];
// [dv, ecc, tanoe] = CL_ip_insertionDv(vinf,rph, sma)
//
// // Insertion around Mars:
// global %CL_planetC;
// vinf = [5 , 6] * 1.e3;
// eqrad_mars = %CL_planetC.mars.eqRad;
// rph = [eqrad_mars+250.e3 , eqrad_mars+500.e3];
// sma = [eqrad_mars+350.e3 , eqrad_mars+700.e3];
// tanoh = [0, 0.1]; // radians
// mu_mars = %CL_planetC.mars.mu;
// [dv,ecc,tanoe] = CL_ip_insertionDv(vinf,rph,sma,tanoh,mu_mars)
//

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:
[arg_out ,arg_in] = argn();
if ~exists('tanoh','local') then tanoh = 0; end
if ~exists('mu','local') then mu = %CL_mu; end

  Nvinf = size(vinf,2);
  Nrph = size(rph,2);
  Nsma = size(sma,2);
  Ntanoh = size(tanoh,2);
  Nmax = max(Nvinf,Nrph,Nsma,Ntanoh);
  coherence = ((Nvinf==1) | (Nvinf==Nmax)) & ...
			  ((Nrph==1) | (Nrph==Nmax)) & ...
			  ((Nsma==1) | (Nsma==Nmax)) & ...
			  ((Ntanoh==1) | (Ntanoh==Nmax))
			  	  
  if coherence
    if Nvinf==1 then vinf=vinf*ones(1,Nmax); end
	if Nrph==1 then rph=rph*ones(1,Nmax); end
    if Nsma==1 then sma=sma*ones(1,Nmax); end
	if Ntanoh==1 then tanoh=tanoh*ones(1,Nmax); end
  else
    CL__error('bad size of input arguments');
  end

  ii = find(sma <= 0);
  if(ii ~= [])
	CL__error('sma must be strictly positive');
  end
  ii = find(vinf <= 0);
  if(ii ~= [])
	CL__error('vinf must be strictly positive');
  end
  ii = find(rph <= 0);
  if(ii ~= [])
	CL__error('rph must be strictly positive');
  end
  ii = find(tanoh <= -%pi | tanoh >= %pi);
  if(ii ~= [])
	CL__error('tanoh must be in ]-pi, pi[');
  end
  
  
  // Vitesse initiale (orbite hyperbolique) :
  sma_hyp = mu ./ vinf.^2;
  ecc_hyp = 1 + rph ./ sma_hyp;
  p_hyp = sma_hyp.*(ecc_hyp.^2-1);
  r_hyp = %nan*ones(1,Nmax);
  ii = find(1 + ecc_hyp .* cos(tanoh) > 0);
  if(ii ~= [])
	r_hyp(ii) = p_hyp(ii) ./ (1 + ecc_hyp(ii) .* cos(tanoh(ii)));
  end
  v_hyp = sqrt(mu * (2 ./ r_hyp + 1 ./ sma_hyp));
  
  // Vitesse finale (orbite elliptique) :
  r_ell = r_hyp;
  ii = find(sma > r_ell/2);
  v_ell = %nan*ones(1,Nmax);
  if(ii ~= [])
    v_ell(ii) = sqrt(mu * (2 ./ r_ell(ii) - 1 ./ sma(ii)));
  end
  dv = v_hyp - v_ell;
  
  if(arg_out >= 2)
    // on utilise le fait que la pente est identique, avant et aprés la maneuvre :
    // C_ell ./ (r_ell .* v_ell) = C_hyp ./ (r_hyp .* v_hyp)
	C_hyp = sqrt(mu * p_hyp);
	C_ell = C_hyp .* v_ell ./ v_hyp;
	ecc = sqrt( 1 - C_ell.^2 ./ (mu*sma));
  end
  if(arg_out == 3)
    p_ell = sma .* (1 - ecc.^2);
	sin_gamma = ecc_hyp .* sqrt(mu ./ p_hyp) .* sin(tanoh) ./ v_hyp;
	
	sin_tanoe = sin_gamma ./ ecc ./ sqrt(mu ./ p_ell) .* v_ell;
	cos_tanoe = p_ell ./ (r_ell .* ecc) - 1 ./ ecc;
	
	tanoe = atan(sin_tanoe,cos_tanoe);
  
  end  
  
  
endfunction
