//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [M] = CL_fr_topoNMat(orig)
// Terrestrial frame to topocentric North frame transformation matrix
//
// Calling Sequence
// [M] = CL_fr_topoNMat(orig)
//
// Description
// <itemizedlist><listitem>
// Computes the frame transformation matrix <emphasis role="bold">M</emphasis> from the terrestrial reference frame to the local topocentric North frame.
// <para> By convention, multiplying <emphasis role="bold">M</emphasis> by coordinates relative to the terrestrial frame yields coordinates relative to the topocentric North frame. </para>
// </listitem>
// <listitem>
// Notes: 
// <para> - <emphasis role="bold">The origin of the topocentric
// North frame is not the center of the planet</emphasis>, but
// the location defined by <emphasis role="bold">orig</emphasis>. </para>
// <para> - This function can be used for another planet than the Earth. </para>
// </listitem>
// <listitem>
// See <link linkend="Local Frames">Local Frames</link> for more details on the definition of local frames. 
// </listitem>
// </itemizedlist>
//
// Parameters
// orig: [lon;lat;alt] Topocentric frame origin in elliptical (geodetic) coordinates [rad;rad;m] (3xN)
// M : Terrestrial to topocentric North frame transformation matrix (3x3xN)
//
// Bibliography
// CNES - MSLIB FORTRAN 90, Volume T (mt_def_topo_N)
//
// See also
// CL_fr_topoN2ter
// CL_fr_ter2topoN
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Conversion from pos_ter to pos_topoN : 
// orig = [CL_deg2rad(40) ; CL_deg2rad(10) ; 0];
// pos_orig = CL_co_ell2car(orig);
// pos_ter = [ 1000.e3 ; 6578.e3 ; 2000.e3 ];
// M = CL_fr_topoNMat(orig);
// pos_topoN = M * (pos_ter - pos_orig)
// // same obtained by: 
// pos_topoN = CL_fr_ter2topoN(orig,pos_ter)
//
// // Conversion from pos_topoN to pos_ter :
// orig = [CL_deg2rad(40) ; CL_deg2rad(10) ; 0];
// pos_orig = CL_co_ell2car(orig);
// pos_topoN = [ 0. ; 0. ; 1000.e3 ];  
// M = CL_fr_topoNMat(orig);
// pos_ter = M' * pos_topoN + pos_orig
// // same obtained by: 
// pos_ter = CL_fr_topoN2ter(orig,pos_topoN)

// Declarations:


// Code:

// check number of output arguments 
[lhs,rhs] = argn();

if (lhs > 1)
  CL__error("Invalid number of output arguments");
end

if (rhs < 1)
  CL__error("Invalid number of input arguments");
end

N = size(orig,2);


// NB: the vectors u,v,w are the COLUMNS of the transformation
// matrix from terrestrial frame to topocentric North frame 

lon = orig(1,:);
lat = orig(2,:);

u = [ -sin(lat).*cos(lon); sin(lon); cos(lat).*cos(lon) ]; 
v = [ -sin(lat).*sin(lon); -cos(lon); cos(lat).*sin(lon) ]; 
w = [ cos(lat); zeros(lon);  sin(lat)] ;

M = [];

if (N > 0) 
  M = (hypermat([3,3,N], [u; v; w])); 
  if (N == 1); M = M(:,:,1); end
end


endfunction
