//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [cjd] = CL_dat_cal2cjd(varargin)
// Calendar date to modified (1950.0) Julian day
//
// Calling Sequence
// [cjd] = CL_dat_cal2cjd(year,month,day [,hour,minute,second])
// [cjd] = CL_dat_cal2cjd(cal) // cal = [year;month;day [;hour;minute;second]]
//
// Description
// <itemizedlist><listitem>
// This function converts a calendar date (year,month,day,hour,minute,second) into a modified (1950.0) Julian day (julian day from 1950.0).
// </listitem>
// <listitem>
// See <link linkend="Dates and Time">Dates and Time</link> for more details.
// </listitem>
// </itemizedlist>
//
// Parameters
// year: (integer) Year. (1xN)
// month: (integer in [1,12]) Month. (1xN). 
// day: (integer in [1,31]) Day. (1xN) 
// hour: (optional) Hours. Default is 0. (1xN or 1x1)
// minute: (optional) Minutes. Default is 0. (1xN or 1x1)
// second: (optional) Seconds. Default is 0. (1xN or 1x1)
// cjd: Modified (1950.0) Julian day (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_dat_cjd2cal
// CL_dat_cal2jd
//
// Examples
// // Example 1
// cjd = CL_dat_cal2cjd(2000,1,1)
// cjd = CL_dat_cal2cjd(2010,2,20,12,3,45)
// cjd = CL_dat_cal2cjd([2000;1;1])
// cjd = CL_dat_cal2cjd([2010;2;20;12;3;45])
//
// // Example 2
// year=[2000,2010]
// month=[1,2]
// day=[1,20]
// hour=[0,12]
// minute=[0,3]
// second=[0,45]
// cjd = CL_dat_cal2cjd(year,month,day,hour,minute,second)
// cjd = CL_dat_cal2cjd([year;month;day;hour;minute;second])


// Declarations:


// Code:

[lhs, rhs] = argn();

if ~(rhs == 1 | (rhs >= 3 & rhs <= 6))
  CL__error("Wrong number of input arguments");
end
if ~(lhs == 1)
  CL__error("Wrong number of output arguments");
end

hour = 0; 
minute = 0; 
second = 0; 

if (rhs == 1) 
  Nin = size(varargin(1),1); 
  if (Nin < 3 | Nin > 6) 
     CL__error("Wrong size of input argument");
  end
  year = varargin(1)(1,:);
  month = varargin(1)(2,:);
  day = varargin(1)(3,:);
  if (Nin >= 4); hour = varargin(1)(4,:); end
  if (Nin >= 5); minute = varargin(1)(5,:); end
  if (Nin >= 6); second = varargin(1)(6,:); end
else
  year = varargin(1);
  month = varargin(2);
  day = varargin(3);
  if (rhs >= 4); hour = varargin(4); end
  if (rhs >= 5); minute = varargin(5); end
  if (rhs >= 6); second = varargin(6); end
end


// algorithm

jd = CL_dat_cal2jd(year,month,day); // julian day  

// day part: rounded to improve accuracy (possibly);  
// jd - 2433282.5 is a whole number anyway. 
j50 = round(jd - 2433282.5);  
cjd = j50 + (hour*3600 + minute*60 + second)/86400.; 

endfunction
