//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [kep_t2]=CL_ex_kepler(t1,kep_t1,t2,mu)
// Keplerian orbit propagation
//
// Calling Sequence
// [kep_t2]=CL_ex_kepler(t1,kep_t1,t2 [,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the orbital elements <emphasis role="bold">kep_t2</emphasis>
// at time <emphasis role="bold">t2</emphasis> given the orbital elements 
// <emphasis role="bold">kep_t1</emphasis> at time <emphasis role="bold">t1</emphasis>, 
// using a Keplerian model. 
// <para> As the model is Keplerian, only the mean anomaly varies. </para>
// <para> There can either be several sets of orbital elements and one final time or 
// one set of orbital elements and multiple final times. </para>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="kep_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem> Notes:  
// <para> - This function works for any type of orbit. </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// t1: Initial time [days] (1xP)
// kep_t1: Keplerian elements at time t1 [sma;ecc;inc;pom;raan;anm] (6xP). P can be > 1 if there is only one final time (i.e. if N==1). 
// t2: Final time [days] (1xN). N can be > 1 if there is only one set of orbital elements (i.e. if P==1). 
// mu : (optional) Gravitational constant. [m^3/s^2]. (default value is %CL_mu)
// kep_t2: Keplerian elements [sma;ecc;inc;pom;raan;anm] at time t2. (6 x max(P,N))
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// Mecanique spatiale, CNES - Cepadues 1995, Tome I
//
// See also
// CL_ex_secularJ2
// CL_ex_lyddane
// CL_ex_eckHech
// CL_oe_kep2car
// CL_kp_M2v
//
// Examples
// // Example 1 : 
// t1 = 21482.125 // initial time (days, origin = 1950.0)
// kep_t1 = [ 24464560; 0.7311; 0.122138; 3.10686; 1.00681; 0.048363 ]
// T = CL_kp_params('per', kep_t1(1,:)) // orbital period in seconds
// t2 = t1 + (0:1/4:1)*T/86400 // final times (days, origin = 1950.0)
// kep_t2 = CL_ex_kepler(t1, kep_t1, t2)
//
// // Example 2 : 
// t1 = 0.  // initial time (arbitrary origin)
// kep_t1 = [ 7.e6, 0.,  %pi/2,  0., %pi, 0.; ..
//            8.e6, 0.1, %pi/2,  0., %pi, 0.  ]'
// t2 = 1. // final time : 1 day later
// kep_t2 = CL_ex_kepler(t1, kep_t1, t2)
//

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

Nkep1 = size(kep_t1,2);
Nt2 = size(t2,2);
N = max(Nkep1,Nt2); // size of result

if ~( Nkep1 == 1 | Nt2 == 1 | Nkep1 == Nt2 )
   CL__error('Wrong size of input arguments'); 
end

if ~exists('mu','local')
   mu = %CL_mu; 
end

if ( find( kep_t1(1,:) <= 0) ~= [] ) 
   CL__error('Invalid orbit parameters'); 
end


sma = kep_t1(1,:)                    // semi major axis
n = CL_kp_params('mm', sma, mu)      // orbit's mean motion 

kep_t2 = [  kep_t1(1,:) .* ones(1,N); 
            kep_t1(2,:) .* ones(1,N); 
            kep_t1(3,:) .* ones(1,N); 
            kep_t1(4,:) .* ones(1,N); 
            kep_t1(5,:) .* ones(1,N); 
            kep_t1(6,:) + n .* (t2 - t1)* 86400 ];  

endfunction
