//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function q = CL_rot_defQuat(varargin)
// Quaternion definition from real and imaginary components
//
// Calling Sequence
// [q] = CL_rot_defQuat(r,i1,i2,i3)
// [q] = CL_rot_defQuat(r,i)
// [q] = CL_rot_defQuat(ri)
// [q] = CL_rot_defQuat(ri,order)
//
// Description
// <itemizedlist><listitem>
// Returns a quaternion defined by its real or imaginary components. 
// <para><inlinemediaobject><imageobject><imagedata fileref="ualph2q.gif"/></imageobject></inlinemediaobject></para>
// <para> The quaternions components may be defined in several ways: </para>
// <para> - real part (r) and imaginary part (i1,i2,i3) = 4 arguments.  </para>
// <para> - real part (r) and imaginary part (i) = 2 arguments.  </para>
// <para> - real and imaginary part (ri) = 1 argument.  </para>
// <para> - real and imaginary part (ri) + order = 2 arguments. The order helps determine the 
// position of the real part in the 4-component vector. </para>
// <para> => order = 'ri': the input vector corresponds to [r,i1,i2,i3] (real part first). </para>
// <para> => order = 'ir': the input vector corresponds to [i1,i2,i3,r] (real part last). </para>
// </listitem>
// <listitem>
// <emphasis role="bold">Examples</emphasis> : 
// <para> All the following calls return the same quaternion: </para>
// <para> q = CL_rot_defQuat(1, 2, 3, 4) </para>
// <para> q = CL_rot_defQuat(1, [2; 3; 4]) </para>
// <para> q = CL_rot_defQuat([1; 2; 3; 4]) </para>
// <para> q = CL_rot_defQuat([1; 2; 3; 4], "ri" ) // (real part first) </para>
// <para> q = CL_rot_defQuat([2; 3; 4; 1], "ir" ) // (real part last) </para>
// </listitem>
// <listitem>
// Notes: 
// <para> - The quaternion as defined is NOT normalized. </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// r, i1, i2, i3 : Real part (1xN) and Imaginary part (1xN)
// r, i : Real part (1xN) and Imaginary part (3xN)
// ri : Real and imaginary part (4xN)
// ri, order : Real and imaginary part (4xN) and order ('ri' or 'ir' : see above).
// q : Quaternion (dim N)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_matrix2quat
// CL_rot_quat2matrix
// CL_rot_rotVect
//
// Examples
// // 1000 random quaternions :
// q = CL_rot_defQuat(rand(4,1000));
// // Normalization :
// q = q ./ norm(q);
// // Size : 
// size(q)
// // real and imaginary parts : 
// q.r
// q.i
//
// // A few other examples of overloaded operations :
// q = CL_rot_defQuat([1 2 3 4;5 6 7 8]')  
// q = q ./ norm(q);
// alpha1 = 2;
// alpha2 = [2 3];
// 
// // Both quaternions are multiplied by 2 :
// q1 = q * alpha1
//
// // First quaternion * 2 and second quaternion * 3 :
// q2 = q .* alpha2
//
// // Both quaternions are divided by 2 :
// q3 = q / alpha1
//
// // First quaternion / 2 and second quaternion / 3 :
// q4 = q ./ alpha2
//
// // inverse of q multiplied by 2 : 
// // inverse of q is conj(q)/(q*conj(q)) (= conj(q) if norm(q) == 1)
// q5 = alpha1 / q


// Declarations:


// Code:

[lhs, rhs] = argn(0);

li=[];
co=[];

select rhs
  case 4
    // ----------------------------------------------
    // 4 arguments : r,i1, i2, i3
    // ----------------------------------------------
    r = varargin(1);
    i1 = varargin(2);
    i2 = varargin(3);
    i3 = varargin(4);

    [li(1), co(1)] = size(r);
    [li(2), co(2)] = size(i1);
    [li(3), co(3)] = size(i2);
    [li(4), co(4)] = size(i3);
 
    min_li = min(li);
    min_co = min(co);
    if ( max(li) <> min_li | min_li <> 1 | max(co) <> min_co | min_co < 1) 
       CL__error("Invalid argument sizes"); 
    end

    q = mlist(['CLquat','r','i'],r,[i1;i2;i3]);


  case 2
    if (typeof(varargin(2)) == 'string') 
      // ----------------------------------------------
      // 2 arguments : ri, order
      // ----------------------------------------------
      ri = varargin(1);
      order = varargin(2);

      [li(1), co(1)] = size(ri);
      [li(2), co(2)] = size(order);

      if ( li(1) <> 4 | co(1) < 1 | li(2) <> 1 | co(2) <> 1) 
         CL__error("Invalid argument sizes"); 
      end

      if (order == "ir") then
        q = mlist(['CLquat','r','i'],ri(4,:),ri(1:3,:));
      elseif (order == "ri")
        q = mlist(['CLquat','r','i'],ri(1,:),ri(2:4,:));
      else
        CL__error("Invalid argument value (''ri'' or ''ir'' expected)"); 
      end

    else
      // ----------------------------------------------
      // 2 arguments : r, i
      // ----------------------------------------------
      r = varargin(1);
      i = varargin(2);

      [li(1), co(1)] = size(r);
      [li(2), co(2)] = size(i);

      if ( li(1) <> 1 | li(2) <> 3 | co(1) <> co(2) | co(1) < 1) 
         CL__error("Invalid argument sizes"); 
      end

      q = mlist(['CLquat','r','i'],r(1,:),i(1:3,:));

    end


  case 1
    // ----------------------------------------------
    // 1 argument : ri 
    // ----------------------------------------------
    ri = varargin(1);

    [li(1), co(1)] = size(ri);
    if ( li(1) <> 4 | co(1) < 1) 
       CL__error("Invalid argument sizes"); 
    end
   
    q = mlist(['CLquat','r','i'],ri(1,:),ri(2:4,:));

  else

    CL__error("Invalid number of input arguments"); 

end


endfunction
