//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [angle] = CL_vectAngle(u,v)
// Angle between vectors
//
// Calling Sequence
// [angle] = CL_vectAngle(u,v)
//
// Description
// <itemizedlist><listitem>
// This function computes the angle between two vectors.
// <para> The angle is between 0 and %pi (included). It is 0 if at least one of the 
// 2 vectors is a null vector.  </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// u: Vector or matrix of vectors (3xN)
// v: Vector or matrix of vectors (3xN)
// angle: Angle between u and v (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_cross
// CL_dot
// CL_colNorm
//
// Examples
// u = [1,1,0;0,0,0;0,0,0];
// v = [0,-2,0;2,0,0;0,0,0]; 
// CL_vectAngle(u,v) * 180/%pi;
 
// Declarations:


// Code:

if (size(u,1) <> size(v,1) | size(u,2) <> size(v,2))
  CL__error("Wrong size of input arguments"); 
end

N = size(u,2); // nb of columns

angle = zeros(1,N); 

[u1,norm_u] = CL_unitVector(u);
[v1,norm_v] = CL_unitVector(v);

K = find( norm_u > 0 & norm_v > 0); // angle is 0 if at least one vector is 0

angle(K) = real(acos(CL_dot(u1(:,K),v1(:,K)))); // real() in case numerical errors appear ! 
 
endfunction
