//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [c] = CL_dot(u,v)
// Dot product of column vectors
//
// Calling Sequence
// [c] = CL_dot(u,v)
//
// Description
// <itemizedlist><listitem>
// Computes the dot product of vectors (or matrices considered as set of column vectors).
// <para> The arguments may be matrices or hypermatrices of dimension 3. </para>
// <para> The second argument can be omitted: CL_dot(u) = CL_dot(u,u) (self dot product). </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// u: Matrix (considered as a set of column vectors) (PxNxK)
// v: (optional) Matrix (considered as a set of column vectors). By default: same as u. (PxNxK)
// c: Matrix (dot products of column vectors) (1xNxK)
//
// See also
// CL_cross
// CL_norm
//
// Authors
// CNES - DCT/SB
//
// Examples
// u = [1,2;3,4];
// v = [-2,1;1,1];
// CL_dot(u,v)
//
// u = rand(2,3,4);
// v = rand(2,3,4);
// CL_dot(u,v)

// Declarations:


// Code:

rhs = argn(2); // number of input arguments

if (rhs < 1 | rhs > 2)
  CL__error('Wrong number of input arguments');
end

if (rhs == 1) // only one input => self product
  c = CL_dot(u,u); // recursive call to simplify the code

else // general case 

  u_type = typeof(u);
  v_type = typeof(v);

  if (u_type ~= v_type) 
     CL__error('Arguments must be of the same type');
  end

  if (u_type ~= 'constant' & u_type ~= 'hypermat' ) 
     CL__error('Invalid argument types (constant or hypermat expected)');
  end

  if (u_type == 'constant') // matrix
     c = sum(u.*v,'r');

  else // hypermatrix
     c = ones(1, u.dims(2), u.dims(3));

     for k = 1:u.dims(3)
        c(:,:,k) = sum(u(:,:,k) .* v(:,:,k), 'r');
     end

  end

end

endfunction


