//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [dv,anv] = CL_man_inclination(ai,ei,inci,pomi,incf,posman,mu)
// Delta V required to change the inclination
//
// Calling Sequence
// [dv,anv] = CL_man_inclination(ai,ei,inci,pomi,incf [,posman,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the velocity increment needed to change the inclination.
// <para> The maneuver consists in making the orbit rotate around the line of nodes, thus changing the velocity direction at one of the nodes. </para>
// <para>The output argument <emphasis role="bold">dv</emphasis> is the velocity increment vector expressed in spherical coordinates in the QSW local frame: [lambda; phi; |dv|], where lambda is the in-plane angle (+%pi: towards planet and +%pi/2: ~along velocity), phi is the out-of-plane angle, positive towards the angular momentum vector (the angular momentum vector is perpendicular to the orbit plane and oriented according to right hand convention), dv is the norm of the velocity increment. </para>
// <para><emphasis role="bold">anv</emphasis> is the true anomaly at the maneuver position.</para>
// <para>The optional flag <emphasis role="bold">posman</emphasis> can be used to define the position of the maneuver (1->ascending node, -1->descending node, 0->minimal |DV|)</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// ai : Semi major axis [m] (1xN)
// ei: Eccentricity (1xN)
// inci : Initial inclination [rad] (1xN)
// pomi: Argument of periapsis [rad] (1xN)
// incf : Final inclination [rad] (1xN)
// posman: (optional) Flag specifying the position of maneuver (1->ascending node, -1->descending node, 0->minimal |dv|). Default is ascending node) (1xN)
// mu : (optional) Gravitational constant. [m^3/s^2] (default value is %CL_mu)
// dv : velocity increment vector in spherical coordinates in the QSW frame [lambda;phi;|dv|] [rad,rad,m/s] (3xN)
// anv: true anomaly at maneuver position [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_man_biElliptic
// CL_man_hohmann
// CL_man_sma
// CL_man_hohmannG
// CL_man_apsidesLine
//
// Examples
// ai = 7200.e3;
// ei = 0.1;
// inci = CL_deg2rad(65);
// pomi = CL_deg2rad(110);
// incf = CL_deg2rad(70);
// [dv,anv] = CL_man_inclination(ai,ei,inci,pomi,incf,0)
// // Check results :
// anm = CL_kp_v2M(ei,anv);
// kep = [ai ; ei ; inci ; pomi ; 0 ; anm];
// kep1 = CL_man_applyDv(kep,dv)

// Declarations:
global %CL_mu;

// Code:

if ~exists('posman','local') then posman=1; end

Nai = size(ai,2);
Nei = size(ei,2);
Ninci = size(inci,2);
Npomi = size(pomi,2);
Nincf = size(incf,2);
Nposman = size(posman,2);
N = max(Nai,Ninci,Nei,Npomi,Nposman,Nincf);
coherence = (Nai==N | Nai==1) & (Nei==N | Nei==1) &..
            (Ninci==N | Ninci==1) & (Npomi==N | Npomi==1) &..
            (Nposman==N | Nposman==1) & (Nincf==N | Nincf==1);
if ~coherence then CL__error('bad input argument size'); end
if N~=1
  if Nai==1 then ai=ai*ones(1,N); end
  if Ninci==1 then inci=inci*ones(1,N); end
  if Nei==1 then ei=ei*ones(1,N); end
  if Npomi==1 then pomi=pomi*ones(1,N); end
  if Nposman==1 then posman=posman*ones(1,N); end
  if Nincf==1 then incf=incf*ones(1,N); end
end

anv = zeros(1,N);
dv = zeros(3,N);
posm = zeros(1,N);
anvdv1 = zeros(1,N);
anvdv2 = zeros(1,N);

if ~exists('mu','local') then mu=%CL_mu; end

i_an = find(posman==1)  //maneuver at ascending node
  posm(i_an) = 1;
  anv(i_an) = pmodulo(-pomi(i_an),2*%pi);
i_dn =find(posman==-1) //maneuver at descending node
  posm(i_dn) = -1;
  anv(i_dn) = pmodulo(%pi-pomi(i_dn),2*%pi);
i_op = find(posman==0)  //maneuver at node nearest to apoapsis (optimal point)
  anvdv1(i_op) = modulo(-pomi(i_op),2*%pi)
  anvdv2(i_op) = modulo(%pi-pomi(i_op),2*%pi)
  i_op1 = find(abs(anvdv1)>abs(anvdv2) & posman==0);
    posm(i_op1) = 1;
    anv(i_op1) = anvdv1(i_op1);
  i_op2 = find(~(abs(anvdv1)>abs(anvdv2)) & posman==0);
    posm(i_op2) = -1;
    anv(i_op2) = anvdv2(i_op2);
  i_op3 = find(anv<0 & posman==0)
    anv(i_op3) = anv(i_op3) + 2*%pi;
if find(posman~=1 & posman~=-1 & posman~=0)~=[] then CL__error('bad value for posman argument'); end

//radius, velocity and cosinus of slope at maneuver execution point
rayon = ai.*(1-ei.*ei)./(1+ei.*cos(anv))
v1 = sqrt(mu.*(2.0./rayon-1.0./ai))
cosp = sqrt(mu.*ai.*(1-ei.*ei))./(rayon.*v1)

//inclination correction (angle between 0 and 2pi)
dinc = modulo(incf-inci,2*%pi)

//impulsion in spheric coordinates
deltav = abs(2.0.*v1.*sin(dinc.*0.5).*cosp)
dv(3,:) = deltav
dv(1,:) = 3.0 .* %pi/2

j_an = find(posm==1 & dinc>0) //ascending node
  dv(2,j_an) = %pi/2 - dinc(j_an)./2
j_an2 = find(posm==1 & ~(dinc>0))
  dv(2,j_an2) = -%pi/2 - dinc(j_an2)./2
j_dn = find(posm~=1 & dinc>0) //inversion at descending node
  dv(2,j_dn) = -%pi/2 + dinc(j_dn)./2
j_dn2 = find(posm~=1 & ~(dinc>0))
  dv(2,j_dn2) = %pi/2 + dinc(j_dn2)./2

endfunction
