//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_t2,osc_t2] = CL_ex_lyddaneMan(t1,mean_t1,t2,tman,dvman,er,mu,zonals)
// Lyddane orbit propagation analytical model with maneuvers
//
// Calling Sequence
// [mean_t2[,osc_t2]] = CL_ex_lyddaneMan(t1,mean_t1,t2 [,tman,dvman,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// This function computes the mean and osculating orbital elements <emphasis role="bold">mean_t2</emphasis> 
// and <emphasis role="bold">osc_t2</emphasis> at time <emphasis role="bold">t2</emphasis> given the mean elements 
// <emphasis role="bold">mean_t1</emphasis> at time <emphasis role="bold">t1</emphasis>. 
// <para> Maneuvers (i.e. velocity increment defined by <emphasis role="bold">tman,dvman</emphasis>)
// are taken into account. </para>
// <para><emphasis role="bold">tman</emphasis> defines the time of the maneuver.</para>
// <para><emphasis role="bold">dvman</emphasis> = [delta-Vx;delta-Vy;delta-Vz] defines the delta-V in the QSW local frame at the time of maneuver.</para>
// <para>If one maneuver occurs at one of the final times, the output orbital elements mean_t2 and osc_t2 include the effect of the maneuver.</para>
// <para>The final times as well as the maneuver time should be sorted (increasing order).</para>
// <para>Zonals coefficients up to J5 are taken into account.</para>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="kep_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>Warnings :
// <para> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</para>
// <para> - This function does not work for eccentricities higher than 0.9</para></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// t1: Initial time [days] (1x1)
// mean_t1: Mean Keplerian elements at time t1 [sma;ecc;inc;pom;raan;anm] (6x1)
// t2: Final time [days] (1xN)
// tman: (optional) Maneuver times [days] (1xP)
// dvman: (optional) Delta-V in QSW local frame [dvx;dvy;dvz] [m/s] (3xP)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonal coefficients J1 to Jn (troncated to J5) to be used (default is %CL_j1jn(1:5)) (1xM)
// mean_t2: Mean Keplerian elements at t2 [sma;ecc;inc;pom;raan;anm] (6 x N)
// osc_t2: Osculating Keplerian elements at t2 [sma;ecc;inc;pom;raan;anm] (6 x N)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_ex_lyddane
// CL_ex_meanLyddane
// CL_fr_qswMat
//
// Examples
// t1 = 21915;
// mean_t1 = [42166712 ; 7.9e-3 ; CL_deg2rad([97.2; 89; 125; 0])];
// t2 = [21916 21918 21920];
// tman = [21915.5 21916.5];
// dvman = [ [ 1 ; 1 ; 1] , [ 1 ; 0 ; 1]];
// [mean_t2,osc_t2] = CL_ex_lyddaneMan(t1,mean_t1,t2,tman,dvman,er=6378138)
//

// Declarations:
global %CL_eqRad %CL_j1jn %CL_mu;

// Code:
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:5); end

// No maneuvers :
if ~exists('tman','local') then 
	[mean_t2,osc_t2] = CL_ex_lyddane(t1,mean_t1,t2,er,mu,zonals);
	return;
else
	if(isempty(tman))
		[mean_t2,osc_t2] = CL_ex_lyddane(t1,mean_t1,t2,er,mu,zonals);
		return;
	end
end
if ~exists('dvman','local') then 
	[mean_t2,osc_t2] = CL_ex_lyddane(t1,mean_t1,t2,er,mu,zonals);
	return;
else
	if(isempty(dvman))
		[mean_t2,osc_t2] = CL_ex_lyddane(t1,mean_t1,t2,er,mu,zonals);
		return;
	end
end


Nt1 = size(t1,2)        // should be 1
Nmean_t1 = size(mean_t1,2)  // should be 1
if (Nt1 ~= 1) then
 CL__error('t1 should be (1x1)');
end
if (Nmean_t1 ~= 1) then
 CL__error('mean_t1 should be (6x1)');
end

Nman = size(tman,2)      //number of maneuvers
Ndvman = size(dvman,2)
Nt2 = size(t2,2)        //extrapolation dates

if (Nman ~= Ndvman ) then
 CL__error('tman et dvman should be (1xP) and (3xP)');
end

mean_t2 = zeros(6,Nt2);   //extrapolations at t2
osc_t2 = zeros(6,Nt2);


// Extrapolation jusqu'a la premiere maneuvre :
ii = find(t2 < tman(1));

if(ii ~= [])
	[mean_t2(:,ii),osc_t2(:,ii)] = CL_ex_lyddane(t1,mean_t1,t2(ii),er,mu,zonals);
end
// Premiere maneuvre :
[mean_tman,osc_tman] = CL_ex_lyddane(t1,mean_t1,tman(1),er,mu,zonals);
[osc_tman_car_pos,osc_tman_car_vel] = CL_oe_kep2car(osc_tman,mu);
dvman_car = CL_fr_qsw2inertial(osc_tman_car_pos,osc_tman_car_vel,dvman(:,1));
osc_tman_car_vel = osc_tman_car_vel + dvman_car;
osc_tman = CL_oe_car2kep(osc_tman_car_pos,osc_tman_car_vel,mu);
mean_tman = CL_ex_meanLyddane(osc_tman,er,mu,zonals);


// Extrapolation entre la kieme-1 et la kieme maneuvre :
for kman = 2 : Nman
    
	ii = find(t2 >= tman(kman-1) & t2 < tman(kman));
	
	if(ii ~= [])
		[mean_t2(:,ii),osc_t2(:,ii)] = CL_ex_lyddane(tman(kman-1),mean_tman,t2(ii),er,mu,zonals);
	end
	
	// kieme maneuvre :
	[mean_tman,osc_tman] = CL_ex_lyddane(tman(kman-1),mean_tman,tman(kman),er,mu,zonals);
	[osc_tman_car_pos,osc_tman_car_vel] = CL_oe_kep2car(osc_tman,mu);
	dvman_car = CL_fr_qsw2inertial(osc_tman_car_pos,osc_tman_car_vel,dvman(:,kman-1));
	osc_tman_car_vel = osc_tman_car_vel + dvman_car;
	osc_tman = CL_oe_car2kep(osc_tman_car_pos,osc_tman_car_vel,mu);
	mean_tman = CL_ex_meanLyddane(osc_tman,er,mu,zonals);
end

// Extrapolation jusqu'a la derniere date :
ii = find(t2 > tman($));
if(ii ~= [])
	[mean_t2(:,ii),osc_t2(:,ii)] = CL_ex_lyddane(tman($),mean_tman,t2(ii),er,mu,zonals);
end

endfunction
