//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(t1,mean_kep_t1,t2,er,mu,zonals)
// Lyddane orbit propagation analytical model
//
// Calling Sequence
// [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(t1,mean_kep_t1,t2 [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// This function computes the mean orbital elements <emphasis role="bold">mean_kep_t2</emphasis> 
// and the osculating elements <emphasis role="bold">osc_kep_t2</emphasis>
// at time <emphasis role="bold">t2</emphasis> given the mean elements <emphasis role="bold">mean_kep_t1</emphasis> at time <emphasis role="bold">t1</emphasis>.
// <para>Zonals coefficients up to J5 are taken into account</para>
// <para> There can either be several sets of orbital elements and one final time or one set of orbital elements 
// and multiple final times. </para>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="kep_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>Warnings :
// <para> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</para>
// <para> - This function does not work for eccentricities higher than 0.9</para></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// t1: Initial time [days] (1xP)
// mean_kep_t1: Mean Keplerian elements at time t1 [sma;ecc;inc;pom;raan;anm] (6xP). P can be > 1 if there is only one final time (i.e. if N==1). 
// t2: Final time [days] (1xN). N can be > 1 if there is only one set of orbital elements (i.e. if P==1). 
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonals coefficients J1 to Jn (troncated to J5) to be used (default is %CL_j1jn(1:5)) (1 x N)
// mean_kep_t2: Mean Keplerian elements at t2 [sma;ecc;inc;pom;raan;anm] (6 x max(P,N))
// osc_kep_t2: Osculating Keplerian elements at t2 [sma;ecc;inc;pom;raan;anm] (6 x max(P,N))
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume E (me_lyddane)
//
// See also
// CL_ex_meanLyddane
// CL_ex_lyddaneMan
//
// Examples
// t1 = 12584
// mean_kep_t1 = [42166712,7.9e-3,CL_deg2rad([97.2, 89, 125, 0])]'
// t2 = 12587:0.2:12588;
// [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(t1,mean_kep_t1,t2,er=6378138)
//

//
// Note d'utilisation: les unites de distance sont en metres OBLIGATOIREMENT
// ==================  (pour mu, er, et a)
//
//                     l'eccentricite doit appartenir a [ 0. , 0.9 [
//                     le domaine d'erreur est          < 0. et >= 0.9
//
//                     l'inclinaison  doit appartenir a [ 0. , pi ]
//                     et non proches des inclinaisons critiques definies par
//                     (pm_i_critique_non_retro, pm_i_critique_retro) a pm_eps_i_critique pres.
//************************************************************************

//initialisations

// Declarations:
global %CL_criticalInc1 %CL_criticalInc2 %CL_eqRad %CL_j1jn %CL_mu;

// Code:

lhs = argn(1);
if(lhs == 1)
  compute_osc_kep_t2 = %f;
elseif(lhs == 2)
  compute_osc_kep_t2 = %t;
elseif(lhs > 2)
  CL__error('too many outputs');
end

if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:5); end

if(length(zonals) < 5)
  warning('zonals must be a vector of size 5 (or more), completing with zeros...');
  zonals = [zonals 0 0 0 0 0];
end

// initialisation de la valeur epsilon * 100
eps100 = 100 * %eps
eccel = 0.05

//controle des donnees
//demi grand axe
if or( mean_kep_t1(1,:) <= eps100 )
  CL__error("semi major axis < or = 0");
end
//eccentricite
if or( mean_kep_t1(2,:) < 0 )
  CL__error("negative eccentricity");
end
//if (mean_kep_t1%e > e_borne_sup_util) then
//   code_retour%valeur = pm_err_e_grand_brouwer // eccentricite trop forte > e_borne_sup_util
//   go to 6000
//end if

// inclinaison
if or( mean_kep_t1(3,:) < 0 )
  CL__error("negative inclination");
end
if or( mean_kep_t1(3,:) > %pi )
  CL__error("inclination > pi");
end

// inclinaison proche de l'inclinaison critique.
eps_i_critique = 4e-5; // obtenu experimentalement : au dela ca plante
if (abs(mean_kep_t1(3,:) - %CL_criticalInc1) < eps_i_critique | ...
     abs(mean_kep_t1(3,:) - %CL_criticalInc2) < eps_i_critique) then
   CL__error("inclination close (less than "+string(eps_i_critique)+" rad) to the critical value");
end


// Passage aux unites utilisees par l'algorithme : km
// --------------------------------------------------
rte = er ./ 1000
mean_kep_t1(1,:) = mean_kep_t1(1,:) ./ 1000
xmu = mu .* 1.e-9

// DEBUT DU MODELE
// ===============

ak2=zonals(2).*(rte.^2)./2
ak3=-zonals(3).*(rte.^3)
ak4=-zonals(4).*(rte.^4).*(3/8)
ak5=-zonals(5).*(rte.^5)

// constantes liees a 1/2 grand-axe, eccentricite et inclinaison
// ============================================================

g2=ak2./(mean_kep_t1(1,:).^2)
g3=ak3./(mean_kep_t1(1,:).^3)
g4=ak4./(mean_kep_t1(1,:).^4)
g5=ak5./(mean_kep_t1(1,:).^5)

e2=mean_kep_t1(2,:).^2
tt1=sqrt(1-e2)
tt2=tt1.^2
t3=tt2.*tt1
t4=t3.*tt1
t5=t4.*tt1
t6=t5.*tt1

h2=g2./tt2.^2
h3=g3./t3.^2
h4=g4./t4.^2
h5=g5./t5.^2

an=86400.*sqrt(xmu./mean_kep_t1(1,:))./mean_kep_t1(1,:)

tt=t2
tb=t1
ant=(tt-tb).*an

c1=cos(mean_kep_t1(3,:))
c2=c1.^2
c3=c2.*c1
c4=c3.*c1
c5=c4.*c1
c6=c5.*c1
dc=1 - 5.*c2

s1=sin(mean_kep_t1(3,:))
s2=s1.^2

es1=mean_kep_t1(2,:).*s1
ec1=mean_kep_t1(2,:).*c1
cc1=c1.*(1-c1)

r1=4+3.*e2
r2=4+9.*e2
r4=e2.*c6./dc.^2


// calcul des termes seculaires en J2,J2**2,J4
// ===========================================
h21=-dc
h22=-35+24.*tt1+25.*tt2
h22=h22+(90-192.*tt1-126.*tt2).*c2
h22=h22+(385+360.*tt1+45.*tt2).*c4
h41=21-9.*tt2
h41=h41+(-270+126.*tt2).*c2
h41=h41+(385-189.*tt2).*c4
po1= 1.5.*h2.*( h21 + h2.*h22./16 )
po1= mean_kep_t1(4,:) + ant.*( po1 + (5/16).*h4.*h41 ) // determination argument du perigee po1

h21=-c1
h22=(-5+12.*tt1+9.*tt2).*c1+(-35-36.*tt1-5.*tt2).*c3
h41=(5-3.*tt2).*c1.*(3-7.*c2)
go1= 3.*h2.*(h21 + h2.*h22./8 )
go1= mean_kep_t1(5,:) + ant.*( go1 + 5.*h4.*h41./4 )    // determination ascension droite du noeud go1


h21=-1+3.*c2
h22=-15+16.*tt1+25.*tt2
h22=h22+(30-96.*tt1-90.*tt2).*c2
h22=h22+(105+144.*tt1+25.*tt2).*c4
h41=(3-30.*c2+35.*c4)
am1= 1.5.*h2.*( tt1.*h21 + h2.*tt1.*h22./16 )
am1= am1 + 15.*h4.*tt1.*e2.*h41./16
am1= mean_kep_t1(6,:) + ant.*( 1 + am1 )
am1=CL_rMod(am1,2*%pi)                                           // determination anomalie meanenne am1

// FIN DU MODELE POUR LES meanENS
//==============================

// Sortie des elements meanens a la date t2
// =======================================
mean_kep_t2 = zeros(6,max(size(t2,2),size(mean_kep_t1,2)))
mean_kep_t2(1,:) = mean_kep_t1(1,:) .* 1000   // passage en metres
mean_kep_t2(2,:) = mean_kep_t1(2,:)
mean_kep_t2(3,:) = mean_kep_t1(3,:)
// seuls pom, gom et M sont modifies
mean_kep_t2(4,:) = po1
mean_kep_t2(5,:) = go1
mean_kep_t2(6,:) = am1


// MODELE POUR LES OSCULATEURS
//============================
if compute_osc_kep_t2

  cam1=cos(am1)
  sam1=sin(am1)
  cdai=cos(0.5.*mean_kep_t1(3,:))
  sdai=sin(0.5.*mean_kep_t1(3,:))
  cgom1=cos(go1)
  sgom1=sin(go1)

  //  calcul des termes a longues periodes en J2,J3,J4,J5
  //  ===================================================
  c1p=cos(po1)
  c2p=cos(2.*po1)
  c3p=cos(3.*po1)

  s1p=sin(po1)
  s2p=sin(2.*po1)
  s3p=sin(3.*po1)

  cc=2.*c2./dc
  xp1=(1- 5.*cc).*s1
  xp2=(1- cc).*s1
  xp4=(1- 2.*cc).*s1

  p1=xp1.*s1
  p2=xp2.*s1
  p4=xp4.*s1
  p3=3.*p2 - 2

  q0=16.*(c2./dc) + 40.*(c4./dc.^2)
  q1=03+q0
  q2=05+2.*q0
  q3=11+5.*q0

  a=0.5.*xp1.*h2 - 5.*xp2.*h4/(3.*h2)
  b=(5/16).*r1.*p3./tt2
  b=( tt2.*h3 + b.*h5 )./h2
  c=(35/32).*h5.*p4./(3.*h2)

  ac=a.*c2p
  bs=b.*s1p
  cs=c.*s3p

  dai2=mean_kep_t1(2,:).*( -ac + mean_kep_t1(2,:).*cs )
  dex2=0.25.*( -tt2.*dai2 + bs ).*s1
  dai2=0.25.*( dai2 - bs./tt2 ).*ec1

  edm2= - h3 - (5/16).*r2.*h5.*p3
  edm2= edm2.*c1p./h2 + mean_kep_t1(2,:).*a.*s2p
  edm2= edm2 + e2.*c.*c3p
  edm2= 0.25.*edm2.*t3.*s1

  a= -0.5.*q3.*h2 + 5.*q1.*h4./(3.*h2)
  a= es1.*a.*s2p
  b= (5/8).*r1.*( 0.5.*p3 + 3.*s2.*q1 )
  b= ( h3 + b.*h5 ).*c1p./h2
  c= (35/16).*( 0.5.*p4 + s2.*q2 ).*h5./(9.*h2)
  c= e2.*c.*c3p

  sidg2= 0.25.*ec1.*(a+b-c)

  a= c1./(1+c1)
  b= tt1./(1+tt1)
  c= tt1 + 1.0./(1+tt1)

  cp=c+2.5
  som1= -cp.*p1 - c1.*q3 - 11.0.*c2
  som1= (som1 + 2).*e2 + 200.0.*r4
  som2= cp.*p2 + c1.*q1 + 3.0.*c2
  som2= (som2 -2).*e2 - 40.0.*r4
  som3= (a+c).*es1
  som4= (a + tt1.*b).*r1
  som4= som4 + e2.*(9 + 6.0.*c) +20
  som4= som4.*p3 + 6.0.*r1.*cc1.*q1
  som4= es1.*som4
  som5= -0.5.*p4.*(a + 3.0.*c + 2)
  som5= (som5 - cc1.*q2).*es1.*e2

  pgm2= 0.125.*( h2.*som1 + 10.0.*h4.*som2./(3.0.*h2) ).*s2p
  pgm2= pgm2 + ( 0.25.*h3.*som3 + (5/64).*h5.*som4 ).*c1p./h2
  pgm2= pgm2 + (35/64).*h5.*som5.*c3p./(9.0.*h2)
  pgm2= pgm2 + (po1+go1+am1)

  // calcul des termes a courtes periodes en J2
  // ==========================================
  am2=am1
  ecc2=mean_kep_t1(2,:)
  po2=po1

  x = zeros(cam1);
  y = zeros(cam1);
  ede2 = zeros(cam1);
  gom2 = zeros(cam1);
  aux = zeros(cam1);
  auxi = zeros(cam1);
  mkt12 = mean_kep_t1(2,:)

  ii = find(mean_kep_t1(2,:) > eccel)
    ede2(ii) = mkt12(ii) + dex2(ii)
    x(ii) = ede2(ii).*cam1(ii) - edm2(ii).*sam1(ii)
    y(ii) = ede2(ii).*sam1(ii) + edm2(ii).*cam1(ii)

    am2(ii) = CL__sc2angle(x(ii),y(ii))

    ecc2(ii) = sqrt(x(ii).^2 + y(ii).^2)
    aux(ii) = sdai(ii) + 0.5.*dai2(ii).*cdai(ii)
    auxi(ii) = 0.5*sidg2(ii)./cdai(ii)
    x(ii)=aux(ii).*cgom1(ii) - auxi(ii).*sgom1(ii)
    y(ii)=aux(ii).*sgom1(ii) + auxi(ii).*cgom1(ii)

    gom2(ii) = CL__sc2angle(x(ii),y(ii))

    po2(ii) = pgm2(ii) - gom2(ii) - am2(ii)

  //end ii

  e = CL_kp_M2E(ecc2,am2)

  x = cos(e) - ecc2
  y = sqrt(1 - ecc2.*ecc2).*sin(e)
  v = CL__sc2angle(x,y)

  sv=sin(v)
  cv=cos(v)
  cv2=cv.*cv
  xc=( 3.*(1+mean_kep_t1(2,:).*cv) + e2.*cv2 ).*cv
  xa=xc+mean_kep_t1(2,:).*c
  xb=xc+mean_kep_t1(2,:)

  pc2=2.*po2
  pc21=pc2+v
  pc22=pc2+2.*v
  pc23=pc2+3.*v

  c2p1v=cos(pc21)
  s2p1v=sin(pc21)
  c2p2v=cos(pc22)
  s2p2v=sin(pc22)
  c2p3v=cos(pc23)
  s2p3v=sin(pc23)

  da=(-1+3.*c2).*xa
  db=3.*s2.*c2p2v

  dax3= mean_kep_t1(2,:).*da + (1 + mean_kep_t1(2,:).*xc).*db
  dax3= mean_kep_t1(1,:).*dax3.*g2./t6

  de1= (da + db.*xb)./t6
  de2= c2p3v + 3.*c2p1v

  dex3= 0.5.*tt2.*(de1.*g2 - de2.*s2.*h2)

  dai3= (mean_kep_t1(2,:).*de2 + 3.*c2p2v)./2
  dai3= c1.*s1.*dai3.*h2

  a1=1+mean_kep_t1(2,:).*cv
  a2=a1.*(1+a1)./tt2
  d1 = 2.*(3.*c2-1).*(1+a2).*sv
  d1 = d1+s2.*( 3.*(1-a2).*s2p1v + (3.*a2+1).*s2p3v )

  d21=6.*(v + mean_kep_t1(2,:).*sv - am2)
  d22=3.*s2p2v + mean_kep_t1(2,:).*(3.*s2p1v + s2p3v)

  a1=5.*c2 - 2.*c1 -1
  d2=mean_kep_t1(2,:).*b.*tt1.*d1

  dpgm3= d2 + a1.*d21 + (2-a1).*d22
  dpgm3= 0.25.*dpgm3.*h2

  edm3= - 0.25 .* t3 .* d1 .* h2
  sidg3= 0.5.*c1.*s1.*(d22-d21).*h2

  ax3= mean_kep_t1(1,:) + dax3
  dex= dex2 + dex3
  dai= dai2 + dai3
  edm= edm2 + edm3
  sidg= sidg2 + sidg3
  pgm= pgm2 + dpgm3
  // qlm=modulo(pgm,2*%pi)
  // qclm=cos(qlm)
  // qslm=sin(qlm)

  ede= mean_kep_t1(2,:) + dex
  qex= ede.*cam1 - edm.*sam1
  qey= ede.*sam1 + edm.*cam1

  ex3= sqrt( ede.*ede + edm.*edm )

  am3 = CL__sc2angle(qex,qey)

  sidi=sdai + 0.5.*dai.*cdai
  sidg=0.5.*sidg./cdai
  qix= sidi.*cgom1 - sidg.*sgom1
  qiy= sidi.*sgom1 + sidg.*cgom1
  si3= sqrt( sidi.*sidi + sidg.*sidg )

  ai3= 2 .* asin(si3)

  go3 = CL__sc2angle(qix,qiy)

  po3= pgm-go3-am3

  am3=pmodulo(am3,2*%pi)
  po3=pmodulo(po3,2*%pi)
  go3=pmodulo(go3,2*%pi)

  // FIN DU MODELE POUR LES OSCULATEURS
  //===================================

  osc_kep_t2(1,:)=ax3 .* 1000 // passage en metres
  osc_kep_t2(2,:)=ex3
  osc_kep_t2(3,:)=ai3
  osc_kep_t2(4,:)=po3
  osc_kep_t2(5,:)=go3
  osc_kep_t2(6,:)=am3

end //end of compute_osc_kep_t2


endfunction


