//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [q] = CL_rot_axAng2quat(u, alpha)
// Rotation axis and angle to quaternion
//
// Calling Sequence
// [q] = CL_rot_axAng2quat(u, alpha)
//
// Description
// <itemizedlist><listitem>
// Computes the quaternion <emphasis role="bold">q</emphasis> that corresponds to the 
// rotation of angle <emphasis role="bold">alpha</emphasis> around the rotation axis 
// <emphasis role="bold">u</emphasis>.
// </listitem>
// <listitem>
// Notes: 
// <para> - The output quaternion has a norm equal to 1.  </para>
// <para> - For compatibility reasons, the argument order: (alpha, u) is also 
// possible but should be avoided.  </para>
// <para> - See <link linkend="Conventions">Conventions</link> or <link linkend="CL_rot_defQuat">CL_rot_defQuat</link> for more details on quaternions. </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// u : Rotation axis [ux;uy;uz]. ||u|| should be > 0 (3xN)
// alpha : Rotation angle [rad] (1xN)
// q : Quaternion (with norm equal to 1) (dim N)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_defQuat
//
// Examples
// // 10 degree rotation around Z axis :
// alpha = CL_deg2rad(10);
// q1 = CL_rot_axAng2quat([0;0;1],alpha)
//
// // same : 
// q2 = CL_rot_eul2quat(0,0,alpha)
//
// // same :
// M = CL_rot_angles2matrix(3,alpha);
// q3 = CL_rot_matrix2quat(M)
//

// Declarations:


// Code:
size_u = size(u); 
size_alpha = size(alpha); 

// Code for compatibility (inversion of arguments) : 
if (size_alpha(1) == 3 & size_u(1) == 1) 
  tmp = alpha; 
  alpha = u; 
  u = tmp;
  size_u = size(u); 
  size_alpha = size(alpha);
end

if (size_alpha(1) <> 1 | size_u(1) <> 3 | size_alpha(2) <> size_u(2) )
  CL__error("Invalid size for rotation angle or axis"); 
end

[axis, axis_norm] = CL_unitVector(u);

if (find(axis_norm == 0) ~= [])
  CL__error("Axis should have norm > 0"); 
end

ri = zeros(4,size(u,2));
ri(1,:) = cos(alpha./2);
ri(2:4,:)= ( ones(size(u,1),1)*sin(alpha/2) ) .* axis;

q = CL_rot_defQuat(ri);

endfunction
