//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [kep,jacob]=CL_oe_car2kep(pos_car,vel_car,mu)
// Cartesian to keplerian orbital elements
//
// Calling Sequence
// [kep[,jacob]] = CL_oe_car2kep(pos_car,vel_car[,mu])
//
// Description
// <itemizedlist><listitem>
// Converts position and velocity to classical keplerian orbital elements for 
// elliptic (non circular and non equatorial), hyperbolic or parabolic orbit.
// <para>The transformation jacobian is optionally computed.</para>
// <para>See <link linkend="OrbitalElements">Orbital elements</link> for more details</para>
// </listitem>
// <listitem>Warning : transformation is not exact for eccentricites and/or inclinations close to 0</listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// pos_car: position [X;Y;Z] [m] (3xN)
// vel_car: velocity [Vx;Vy;Vz] [m/s] (3xN)
// mu : (optional) gravitational constant. [m^3/s^2] (default value is %CL_mu)
// kep: classical keplerian orbital elements [sma;ecc;inc;pom;raan;anm] [m,rad] (6xN)
// jacob: (optional) transformation jacobian d(sma,ecc,inc,pom,raan,anm)/d(X,Y,Z,Vx,Vy,Vz) (6x6xN)
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume V (mv_car_kep)
//
// See also
// CL_oe_kep2car
// CL_oe_car2cir
// CL_oe_car2cirEqua
// CL_co_car2sph
// CL_co_car2ell
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Example 1
// kep=[7000000,0.001,0.9,%pi/3,%pi/4,0]'
// [pos_car,vel_car,jacob1]=CL_oe_kep2car(kep)
// [kep2,jacob2]=CL_oe_car2kep(pos_car,vel_car)
//
// // Example 2
// pos = [-26655470,29881667,-113657]';
// vel = [-1125,-1122,195]';
// [kep,jacob1] = CL_oe_car2kep(pos,vel);
// [pos2,vel2,jacob2] = CL_oe_kep2car(kep);

// Declarations:
global %CL_epsilon %CL_mu;

// Code:

jacob = [];

[lhs,rhs] = argn();
if ~exists('mu','local') then mu=%CL_mu; end
if lhs==2
  compute_jacob=%t
  jacob = zeros(6,6,size(pos_car,2))
else
  compute_jacob=%f
end

kep = zeros(6,size(pos_car,2))

eps100 = 100*%eps

//DATA TESTS
//position-velocity vectors
norm_pos = CL_colNorm(pos_car)
norm_vel = CL_colNorm(vel_car)
kinetic_moment = CL_cross(pos_car,vel_car)
norm_kinetic = CL_colNorm(kinetic_moment)

i_pos = find(norm_pos<eps100)
i_vel = find(norm_vel<eps100)
i_colinear = find(norm_kinetic<eps100)

if i_pos~=[] then CL__error('null position vector'); end
if i_vel~=[] then CL__error('null velocity vector'); end
if i_colinear~=[] then CL__error('zero kinetic moment norm: position and velocity colinear'); end

//INTERMEDIATE VARIABLES
pos_vel = CL_dot(pos_car,vel_car) //scalar product position*velocity
parameter = (norm_kinetic.^2)./mu //parameter p = (rxv)^2/mu = c^2/mu; p>0 because mu>0
norm_vel_sq = norm_vel.^2
one_over_a = 2.0./norm_pos - norm_vel_sq./mu  //1/a

coefk = 0.5.*norm_vel_sq - mu./norm_pos //eccentricity
e2 = 1 + 2.*parameter.*coefk./mu        //valid for any orbit type: k = v2/2-|mu|/r

eccentricity = sqrt(e2) //e^2 = 1 + 2.c^2.k/mu^2

//***********************************************
//calcul transformation cartesian --> keplerian
//following orbit type
//***********************************************
//elliptic orbit
i_ellip = find(eccentricity <= 1-%CL_epsilon.parab)
if i_ellip~=[]

  i_ellip_a = find( (eccentricity <= 1-%CL_epsilon.parab) & (one_over_a<eps100) )
  if i_ellip_a~=[] then CL__error('a negatif or infinite'); end

  a = 1 ./ one_over_a
  i_ellip_1a = find( (eccentricity <= 1-%CL_epsilon.parab) & (a<=eps100) )
  if i_ellip_1a~=[] then CL__error('a near 0'); end

  if compute_jacob
    [kepi,jac] = CL__oe_car2kepEllip(pos_car(:,i_ellip),vel_car(:,i_ellip),eccentricity(:,i_ellip),norm_pos(:,i_ellip),...
                                   norm_vel(:,i_ellip),one_over_a(:,i_ellip),kinetic_moment(:,i_ellip),pos_vel(:,i_ellip),mu)
    jacob(:,:,i_ellip) = jac
  else
    [kepi] = CL__oe_car2kepEllip(pos_car(:,i_ellip),vel_car(:,i_ellip),eccentricity(:,i_ellip),norm_pos(:,i_ellip),...
                                   norm_vel(:,i_ellip),one_over_a(:,i_ellip),kinetic_moment(:,i_ellip),pos_vel(:,i_ellip),mu)
  end
  kep(:,i_ellip) = kepi

end

//hyperbolic orbit
i_hyper = find(eccentricity>=1+%CL_epsilon.parab)
if i_hyper~=[]

  one_over_a = abs(one_over_a)  //we use absolute value of 1/a in the hyperbolic case

  i_hyper_1a = find( (eccentricity>=1+%CL_epsilon.parab)&(one_over_a<eps100) )
  if i_hyper_1a~=[] then CL__error('infinit a'); end

  a = 1 ./ one_over_a
  i_hyper_a = find( (eccentricity>=1+%CL_epsilon.parab)&(a<=eps100) )
  if i_hyper_a~=[] then CL__error('null a'); end

  if compute_jacob
    [kepi,jac] = CL__oe_car2kepHyperb(pos_car(:,i_hyper),vel_car(:,i_hyper),parameter(:,i_hyper),eccentricity(:,i_hyper),...
                                   norm_pos(:,i_hyper),norm_vel(:,i_hyper),kinetic_moment(:,i_hyper),...
                                   norm_kinetic(:,i_hyper),one_over_a(:,i_hyper),pos_vel(:,i_hyper),mu)
    jacob(:,:,i_hyper) = jac
  else
    [kepi] = CL__oe_car2kepHyperb(pos_car(:,i_hyper),vel_car(:,i_hyper),parameter(:,i_hyper),eccentricity(:,i_hyper),...
                                   norm_pos(:,i_hyper),norm_vel(:,i_hyper),kinetic_moment(:,i_hyper),...
                                   norm_kinetic(:,i_hyper),one_over_a(:,i_hyper),pos_vel(:,i_hyper),mu)
  end
  kep(:,i_hyper) = kepi

end

//parabolic orbit
i_para = find( (eccentricity>1-%CL_epsilon.parab)&(eccentricity<1+%CL_epsilon.parab) )
if i_para~=[]

  if compute_jacob
    [kepi,jac] = CL__oe_car2kepParab(pos_car(:,i_para),vel_car(:,i_para),parameter(:,i_para),eccentricity(:,i_para),...
                                   norm_pos(:,i_para),kinetic_moment(:,i_para),norm_kinetic(:,i_para),pos_vel(:,i_para),mu)
    jacob(:,:,i_para) = jac
  else
    [kepi] = CL__oe_car2kepParab(pos_car(:,i_para),vel_car(:,i_para),parameter(:,i_para),eccentricity(:,i_para),...
                                   norm_pos(:,i_para),kinetic_moment(:,i_para),norm_kinetic(:,i_para),pos_vel(:,i_para),mu)
  end
  kep(:,i_para) = kepi

end

//if (compute_jacob & size(jacob,3)==1) then jacob=jacob(:,:,1); end

endfunction
