//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_G50,vel_G50,jacob] = CL_fr_ter2G50(cjd,pos_ter,vel_ter,ut1_utc)
// Terrestrial frame to Gamma50 (Veis) vector transformation
//
// Calling Sequence
// [pos_G50,[vel_G50,jacob]] = CL_fr_ter2G50(cjd,pos_ter,[vel_ter,ut1_utc])
//
// Description
// <itemizedlist><listitem>
// Converts position and (optionally) velocity vectors relative to the terrestrial ("Earth fixed") reference frame into position and (optionally) velocity vectors relative to Gamma50 (Veis). 
// <para>The jacobian of the transformation is optionally computed.</para>
// </listitem>
// <listitem>
// Notes: 
// <para> Transformation from Gamma50 (Veis) to terrestrial frame consists in a single rotation of the Veis sideral time (see CL_mod_sidTimeG50) around the Z axis.</para>
// </listitem>
// <listitem>
// See <link linkend="ReferenceFrames">Reference Frames</link> for more details on the definition of reference frames. 
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// cjd: modified julian day from 1950.0 (UTC) (1xN)
// pos_ter: position vector relative to terrestrial frame [m] (3xN)
// vel_ter: (optionnal) velocity vector relative to terrestrial frame [m/s] (3xN)
// ut1_utc : (optional) ut1-utc [seconds] (default is 0) (1xN)
// pos_G50: position vector relative to Gamma50 (Veis) [m] (3xN)
// vel_G50: (optional) velocity vector relative to Gamma50 (Veis) [m/s] (3xN)
// jacob: (optional) jacobian of the transformation [d(x,y,z,vx,vy,vz)_G50/d(x,y,z,vx,vy,vz)_ter] (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume R (mr_TerVrai_veis)
//
// See also
// CL_fr_ter2J2000Mat
// CL_fr_G502terMat
// CL_fr_G502ter
// CL_mod_sidTimeG50
//
// Examples
// // Conversion of position ter to G50
// pos_ter = [[3500.e3;2500.e3;5800.e3] , [4500.e3;2100.e3;6800.e3]];
// cjd = [21010 , 21011];
// pos_G50=CL_fr_ter2G50(cjd,pos_ter);
//
// // Conversion of position and velocity : ter to G50  + jacobian
// pos_ter = [[3500.e3;2500.e3;5800.e3] , [4500.e3;2100.e3;6800.e3]];
// vel_ter = [[1.e3;3.e3;7.e3] , [2.e3;3.e3;6.e3]];
// cjd = [21010 , 21011];
// [pos_G50,vel_G50,jacob]=CL_fr_ter2G50(cjd,pos_ter,vel_ter);


// Declarations:


// Code:

if ~exists('vel_ter','local') then vel_present = %f; end
if ~exists('ut1_utc','local') then ut1_utc=zeros(cjd); end

vel_present = %t;
compute_vel = %f;
compute_jacob = %f;
vel_G50 = [];
jacob = [];

[lhs,rhs] = argn()
if (rhs<2 | rhs > 4) then CL__error('check number of input arguments'); end


if(lhs >=2)
  compute_vel = %t;
end
if(lhs ==3)
  compute_jacob = %t;
end

if(~vel_present & compute_vel)
  CL__error("Input velocity missing");
end



// pos_G50 computation :
cjd_ut1 = cjd + ut1_utc/86400.;
[tsid,tsidt] = CL_mod_sidTimeG50(cjd_ut1);

costsid = cos(tsid);
sintsid = sin(tsid);

pos_G50 = [ costsid .* pos_ter(1,:) - sintsid .* pos_ter(2,:) ;
           +sintsid .* pos_ter(1,:) + costsid .* pos_ter(2,:) ;
          pos_ter(3,:) ];

// vel_G50 computation :
if compute_vel
N = size(pos_G50,2)


vel_G50_iner = [ costsid .* vel_ter(1,:) - sintsid .* vel_ter(2,:) ;
                 +sintsid .* vel_ter(1,:) + costsid .* vel_ter(2,:) ;
               vel_ter(3,:) ];

   // calcul du vecteur rotation
   vect_omega = [ zeros(1,N) ;
                  zeros(1,N) ;
          -tsidt.*ones(1,N)];


   // vitesse d'entrainement : omega vectoriel position
   vel_entrainement = CL_cross(vect_omega,pos_G50);

   // vitesse par rapport au repere G50restre :
   vel_G50 = vel_G50_iner - vel_entrainement;
end


// jacob computation :
if compute_jacob

    jacob = zeros(6,6,N);

  jacob(1,1,:) = costsid;
  jacob(1,2,:) = -sintsid;
  jacob(2,1,:) = sintsid;
  jacob(2,2,:) = costsid;
  jacob(3,3,:) = ones(1,N);

    jacob(4,4,:) = costsid;
  jacob(4,5,:) = -sintsid;
  jacob(5,4,:) = sintsid;
  jacob(5,5,:) = costsid;
  jacob(6,6,:) = ones(1,N);

    // dans les lignes 4 a 6, colonnes 1 a 3 on trouve le resultat de :
  // matrice associee au vecteur rotation * matrice de passage
    //  0    om3  -om2   *  ct  st   0  (et seul om3 est non nul dans notre cas)
    // -om3   0   om1      -st  ct   0
  //  om2  -om1  0        0    0   1

  jacob(4,1,:) = vect_omega(3,:).*sintsid;
    jacob(4,2,:) = vect_omega(3,:).*costsid;
  jacob(5,1,:) = -vect_omega(3,:).*costsid;
  jacob(5,2,:) = vect_omega(3,:).*sintsid;


end

endfunction
