//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_ter,vel_ter,jacob] = CL_fr_J20002ter(cjd,pos_J2000,vel_J2000,ut1_utc,xp,yp,dPsi,dEps,conv)
// EME2000 (J2000) to terrestrial frame vector transformation
//
// Calling Sequence
// [pos_ter,[vel_ter,jacob]] = CL_fr_J20002ter(cjd,pos_J2000,[vel_J2000,ut1_utc,xp,yp,dPsi,dEps,conv])
//
// Description
// <itemizedlist><listitem>
// Converts position and (optionally) velocity vectors from EME2000 to the terrestrial ("Earth fixed") reference frame. 
// <para>The jacobian of the transformation is optionally computed.</para>
// </listitem>
// <listitem>
// See <link linkend="ReferenceFrames">Reference Frames</link> for more details on the definition of reference frames. 
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// cjd: modified julian day from 1950.0 (UTC) (1xN)
// pos_J2000: position vector relative to EME2000 [m] (3xN)
// vel_J2000: (optional) velocity vector relative to EME2000 [m/s] (3xN)
// ut1_utc : (optional) ut1-utc [seconds] (default is 0) (1xN)
// xp : (optional) x polar coordinate [radians] (default is 0) (1xN)
// yp : (optional) y polar coordinate [radians] (default is 0) (1xN)
// dPsi : (optional) Nutation corrections [radians] (default is 0) (1xN)
// dEps : (optional) Nutation corrections [radians] (default is 0) (1xN)
// conv : (optional) Convention IERS. Only iers 1996 (Lieske/Wahr) is implemented (default is "iers_1996")
// pos_ter: position vector relative to terrestrial frame [m] (3xN)
// vel_ter: (optional) velocity vector relative to terrestrial frame [m/s] (3xN)
// jacob: (optional) jacobian of the transformation (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 IERS Conventions (1996), Dennis D. McCarthy
// 2 Explanatory Supplement to the Astronomical Almanac, Seidelman (1992)
//
// See also
// CL_fr_ter2J2000Mat
// CL_fr_J20002ter
//
// Examples
// // Conversion J2000 to terrestrial
// pos_J2000=[[3500.e3;2500.e3;5800.e3] , [4500.e3;2100.e3;6800.e3]];
// cjd = [21010 , 21011];
// pos_ter = CL_fr_J20002ter(cjd,pos_J2000);
// M=CL_fr_ter2J2000Mat(cjd);
// pos_ter_2 = M'*pos_J2000;
//
// // Conversion J2000 to terrestrial
// pos_J2000 = [1000.e3;7078.e3;700.e3];
// vel_J2000 = [4.e3;4.2e3;3.e3];
// cjd = 20010;
// [pos_ter,vel_ter,jacob] = CL_fr_J20002ter(cjd,pos_J2000,vel_J2000)


// Declarations:


// Code:

[lhs,rhs]=argn();

vel_present = %t;

if ~exists('vel_J2000','local') then vel_present = %f; end
if ~exists('ut1_utc','local') then ut1_utc = zeros(cjd); end
if ~exists('xp','local') then xp = zeros(cjd); end
if ~exists('yp','local') then yp = zeros(cjd); end
if ~exists('dPsi','local') then dPsi = zeros(cjd); end
if ~exists('dEps','local') then dEps = zeros(cjd); end
if ~exists('conv','local') then conv = "iers_1996"; end

if conv~="iers_1996" then
      CL__error("only model iers_1996 is implemented, sorry...");
end

//verif dimension des args en entree
sz=size(cjd);
if (sz(1)~=1) then CL__error("check dim of input arguments (column vect.)"); end
sz=size(pos_J2000);
if (sz(1)~=3) then CL__error("check dim of input arguments (column vect.)"); end
sz=size(ut1_utc);
if (sz(1)~=1) then CL__error("check dim of input arguments (column vect.)"); end
sz=size(xp);
if (sz(1)~=1) then CL__error("check dim of input arguments (column vect.)"); end
sz=size(yp);
if (sz(1)~=1) then CL__error("check dim of input arguments (column vect.)"); end
sz=size(dPsi);
if (sz(1)~=1) then CL__error("check dim of input arguments (column vect.)"); end
sz=size(dEps);
if (sz(1)~=1) then CL__error("check dim of input arguments (column vect.)"); end


compute_vel = %f;
compute_jacob = %f;
vel_ter = [];
jacob = [];
flag = "n";

if (lhs>=2)
  compute_vel = %t;
  flag = "p";
  if(lhs ==3)
    compute_jacob = %t;
  end
end

if(~vel_present & compute_vel)
  CL__error("Input velocity missing");
end

W=CL_mod_polarMotionMatrix(xp,yp);

//Conversion TUC --> TAI
[cjdtai]=CL_dat_utc2tai(cjd);
jj_tai = floor(cjdtai);
sec_tai = (cjdtai-jj_tai)*86400.0;

cjdut1 = cjd + ut1_utc/86400.0;

[TempsSideralMoyen,TSMP] = CL_mod_sidTime(cjdut1);

[eps0,eps0p] = CL_mod_meanObliquity(jj_tai,sec_tai,flag);

[NUT,F,NUTP,FP] = CL_mod_nutationAngles(jj_tai,sec_tai,flag);

[equi,equip] = CL_mod_equinoxesEquation(NUT(1,:),eps0,F(5,:),NUTP(1,:),eps0p,FP(5,:));
TempsSideralVrai = TempsSideralMoyen+equi;
R = CL_rot_angles2matrix(3,-TempsSideralVrai);

NUT=NUT+[dPsi;dEps];
[N,OM_nut] = CL_mod_nutationMatrix([eps0; NUT;],[eps0p ; NUTP],[],flag);

[K,KP,KPP]=CL_mod_precessionAngles(jj_tai,sec_tai,flag);
[PREC,OM_prec] = CL_mod_precessionMatrix(K,KP,KPP,flag);



a_1_1 = matrix((PREC(1, 1,:).*N(1, 1,:)+PREC(1, 2,:).*N(2, 1,:)+PREC(1, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 1,:)+R(1, 2,:).*W(2, 1,:)+R(1, 3,:).*W(3, 1,:))+(PREC(1, 1,:).*N(1, 2,:)+PREC(1, 2,:).*N(2, 2,:)+PREC(1, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 1,:)+R(2, 2,:).*W(2, 1,:)+R(2, 3,:).*W(3, 1,:))+(PREC(1, 1,:).*N(1, 3,:)+PREC(1, 2,:).*N(2, 3,:)+PREC(1, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 1,:)+R(3, 2,:).*W(2, 1,:)+R(3, 3,:).*W(3, 1,:)),1,-1);
a_1_2 = matrix((PREC(1, 1,:).*N(1, 1,:)+PREC(1, 2,:).*N(2, 1,:)+PREC(1, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 2,:)+R(1, 2,:).*W(2, 2,:)+R(1, 3,:).*W(3, 2,:))+(PREC(1, 1,:).*N(1, 2,:)+PREC(1, 2,:).*N(2, 2,:)+PREC(1, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 2,:)+R(2, 2,:).*W(2, 2,:)+R(2, 3,:).*W(3, 2,:))+(PREC(1, 1,:).*N(1, 3,:)+PREC(1, 2,:).*N(2, 3,:)+PREC(1, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 2,:)+R(3, 2,:).*W(2, 2,:)+R(3, 3,:).*W(3, 2,:)),1,-1);
a_1_3 = matrix((PREC(1, 1,:).*N(1, 1,:)+PREC(1, 2,:).*N(2, 1,:)+PREC(1, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 3,:)+R(1, 2,:).*W(2, 3,:)+R(1, 3,:).*W(3, 3,:))+(PREC(1, 1,:).*N(1, 2,:)+PREC(1, 2,:).*N(2, 2,:)+PREC(1, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 3,:)+R(2, 2,:).*W(2, 3,:)+R(2, 3,:).*W(3, 3,:))+(PREC(1, 1,:).*N(1, 3,:)+PREC(1, 2,:).*N(2, 3,:)+PREC(1, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 3,:)+R(3, 2,:).*W(2, 3,:)+R(3, 3,:).*W(3, 3,:)),1,-1);
a_2_1 = matrix((PREC(2, 1,:).*N(1, 1,:)+PREC(2, 2,:).*N(2, 1,:)+PREC(2, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 1,:)+R(1, 2,:).*W(2, 1,:)+R(1, 3,:).*W(3, 1,:))+(PREC(2, 1,:).*N(1, 2,:)+PREC(2, 2,:).*N(2, 2,:)+PREC(2, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 1,:)+R(2, 2,:).*W(2, 1,:)+R(2, 3,:).*W(3, 1,:))+(PREC(2, 1,:).*N(1, 3,:)+PREC(2, 2,:).*N(2, 3,:)+PREC(2, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 1,:)+R(3, 2,:).*W(2, 1,:)+R(3, 3,:).*W(3, 1,:)),1,-1);
a_2_2 = matrix((PREC(2, 1,:).*N(1, 1,:)+PREC(2, 2,:).*N(2, 1,:)+PREC(2, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 2,:)+R(1, 2,:).*W(2, 2,:)+R(1, 3,:).*W(3, 2,:))+(PREC(2, 1,:).*N(1, 2,:)+PREC(2, 2,:).*N(2, 2,:)+PREC(2, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 2,:)+R(2, 2,:).*W(2, 2,:)+R(2, 3,:).*W(3, 2,:))+(PREC(2, 1,:).*N(1, 3,:)+PREC(2, 2,:).*N(2, 3,:)+PREC(2, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 2,:)+R(3, 2,:).*W(2, 2,:)+R(3, 3,:).*W(3, 2,:)),1,-1);
a_2_3 = matrix((PREC(2, 1,:).*N(1, 1,:)+PREC(2, 2,:).*N(2, 1,:)+PREC(2, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 3,:)+R(1, 2,:).*W(2, 3,:)+R(1, 3,:).*W(3, 3,:))+(PREC(2, 1,:).*N(1, 2,:)+PREC(2, 2,:).*N(2, 2,:)+PREC(2, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 3,:)+R(2, 2,:).*W(2, 3,:)+R(2, 3,:).*W(3, 3,:))+(PREC(2, 1,:).*N(1, 3,:)+PREC(2, 2,:).*N(2, 3,:)+PREC(2, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 3,:)+R(3, 2,:).*W(2, 3,:)+R(3, 3,:).*W(3, 3,:)),1,-1);
a_3_1 = matrix((PREC(3, 1,:).*N(1, 1,:)+PREC(3, 2,:).*N(2, 1,:)+PREC(3, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 1,:)+R(1, 2,:).*W(2, 1,:)+R(1, 3,:).*W(3, 1,:))+(PREC(3, 1,:).*N(1, 2,:)+PREC(3, 2,:).*N(2, 2,:)+PREC(3, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 1,:)+R(2, 2,:).*W(2, 1,:)+R(2, 3,:).*W(3, 1,:))+(PREC(3, 1,:).*N(1, 3,:)+PREC(3, 2,:).*N(2, 3,:)+PREC(3, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 1,:)+R(3, 2,:).*W(2, 1,:)+R(3, 3,:).*W(3, 1,:)),1,-1);
a_3_2 = matrix((PREC(3, 1,:).*N(1, 1,:)+PREC(3, 2,:).*N(2, 1,:)+PREC(3, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 2,:)+R(1, 2,:).*W(2, 2,:)+R(1, 3,:).*W(3, 2,:))+(PREC(3, 1,:).*N(1, 2,:)+PREC(3, 2,:).*N(2, 2,:)+PREC(3, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 2,:)+R(2, 2,:).*W(2, 2,:)+R(2, 3,:).*W(3, 2,:))+(PREC(3, 1,:).*N(1, 3,:)+PREC(3, 2,:).*N(2, 3,:)+PREC(3, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 2,:)+R(3, 2,:).*W(2, 2,:)+R(3, 3,:).*W(3, 2,:)),1,-1);
a_3_3 = matrix((PREC(3, 1,:).*N(1, 1,:)+PREC(3, 2,:).*N(2, 1,:)+PREC(3, 3,:).*N(3, 1,:)).*(R(1, 1,:).*W(1, 3,:)+R(1, 2,:).*W(2, 3,:)+R(1, 3,:).*W(3, 3,:))+(PREC(3, 1,:).*N(1, 2,:)+PREC(3, 2,:).*N(2, 2,:)+PREC(3, 3,:).*N(3, 2,:)).*(R(2, 1,:).*W(1, 3,:)+R(2, 2,:).*W(2, 3,:)+R(2, 3,:).*W(3, 3,:))+(PREC(3, 1,:).*N(1, 3,:)+PREC(3, 2,:).*N(2, 3,:)+PREC(3, 3,:).*N(3, 3,:)).*(R(3, 1,:).*W(1, 3,:)+R(3, 2,:).*W(2, 3,:)+R(3, 3,:).*W(3, 3,:)),1,-1);

pos_ter = [ a_1_1.*pos_J2000(1,:) + a_2_1.*pos_J2000(2,:) + a_3_1.*pos_J2000(3,:)
               a_1_2.*pos_J2000(1,:) + a_2_2.*pos_J2000(2,:) + a_3_2.*pos_J2000(3,:)
         a_1_3.*pos_J2000(1,:) + a_2_3.*pos_J2000(2,:) + a_3_3.*pos_J2000(3,:)];

if compute_vel
    Npos = size(pos_ter,2)
  // vitesse d'entrainement :
  // vecteur rotation de la precession : OM_prec
  pos_tmp = PREC'*pos_J2000;
  OM_prec = -OM_prec;
  vel_entrainement = CL_cross(OM_prec,pos_tmp);
  vel_tmp_iner = PREC'*vel_J2000;
  vel_tmp = vel_tmp_iner - vel_entrainement;
  // jacobienne
  if compute_jacob
    jacob1 = zeros(6,6,Npos);
    jacob1(1:3,1:3,:) = PREC';
    jacob1(4:6,4:6,:) = PREC';
    mat_vit_rot = zeros(3,3,Npos);
    mat_vit_rot(1,2,:) = OM_prec(3,:);
    mat_vit_rot(1,3,:) = -OM_prec(2,:);
    mat_vit_rot(2,1,:) = -OM_prec(3,:);
    mat_vit_rot(2,3,:) = OM_prec(1,:);
    mat_vit_rot(3,1,:) = OM_prec(2,:);
    mat_vit_rot(3,2,:) = -OM_prec(1,:);
    jacob1(4:6,1:3,:) = mat_vit_rot*(PREC');
  end
  // vecteur rotation de la nutation : OM_nut
  pos_tmp = N'*pos_tmp;
  OM_nut = -OM_nut;
  vel_entrainement = CL_cross(OM_nut,pos_tmp);
  vel_tmp_iner = N'*vel_tmp;
  vel_tmp = vel_tmp_iner - vel_entrainement;
  // jacobienne
  if compute_jacob
    jacob2 = zeros(6,6,Npos);
    jacob2(1:3,1:3,:) = N';
    jacob2(4:6,4:6,:) = N';
    mat_vit_rot = zeros(3,3,Npos);
    mat_vit_rot(1,2,:) = OM_nut(3,:);
    mat_vit_rot(1,3,:) = -OM_nut(2,:);
    mat_vit_rot(2,1,:) = -OM_nut(3,:);
    mat_vit_rot(2,3,:) = OM_nut(1,:);
    mat_vit_rot(3,1,:) = OM_nut(2,:);
    mat_vit_rot(3,2,:) = -OM_nut(1,:);
    jacob2(4:6,1:3,:) = mat_vit_rot*(N');
  end
  // vecteur rotation du temps sideral : rotation autour de Z, de equip+TSMP
  OM_tsid =  W*[ zeros(1,Npos) ; zeros(1,Npos) ; (equip+TSMP).*ones(1,Npos)];
  pos_tmp = (R*W)'*pos_tmp;
  vel_entrainement = CL_cross(OM_tsid,pos_tmp);
  vel_tmp_iner = (R*W)'*vel_tmp;
  vel_tmp = vel_tmp_iner - vel_entrainement;
  // jacobienne
  if compute_jacob
    jacob3 = zeros(6,6,Npos);
    jacob3(1:3,1:3,:) = (R*W)';
    jacob3(4:6,4:6,:) = (R*W)';
    mat_vit_rot = zeros(3,3,Npos);
    mat_vit_rot(1,2,:) = OM_tsid(3,:);
    mat_vit_rot(1,3,:) = -OM_tsid(2,:);
    mat_vit_rot(2,1,:) = -OM_tsid(3,:);
    mat_vit_rot(2,3,:) = OM_tsid(1,:);
    mat_vit_rot(3,1,:) = OM_tsid(2,:);
    mat_vit_rot(3,2,:) = -OM_tsid(1,:);
    jacob3(4:6,1:3,:) = mat_vit_rot*((R*W)');
    jacob = jacob3*jacob2*jacob1; // jacobienne finale
  end
  // vecteur rotation du mouvement polaire : aucun (pas de modelisation de la vitesse, on a juste les angles xp et yp)

  // vitesse par rapport au repere J2000 :
   vel_ter = vel_tmp;
end

  // dans les lignes 4 a 6, colonnes 1 a 3 on trouve le resultat de :
  // matrice associee au vecteur rotation * matrice de passage
    //  0    om3  -om2   *  a11  a12  a13
    // -om3   0   om1       a21  a22  a23
  //  om2  -om1  0        a31  a32  a33


endfunction




