//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

// -----------------------------------------------------------
//> Repeating ground tracks:  
//> Altitudes of circular Sun-synchonous orbits (i.e. semi major axis minus 
//> equatorial radius) such that the ground tracks repeat exactly after a 
//> certain number of planet revolutions (Q). 
//
// Auteur: A. Lamy
// -----------------------------------------------------------

// Declarations:
global %CL_eqRad;

// Code:

hmin = 500.e3;
hmax = 700.e3;
Qmin = 1;
Qmax = 20;

desc_param = list(..
   CL_defParam("Altitude - min", hmin, units=['m', 'km'], id='hmin', valid='hmin>=0' ),..
   CL_defParam("Altitude - max", hmax, units=['m', 'km'], id='hmax', valid='hmax>hmin' ),..
   CL_defParam("Q (number of planet revolutions / node) - min", Qmin, id='Qmin', accv = 1:100),..
   CL_defParam("Q (number of planet revolutions / node) - max", Qmax, id='Qmax', accv = 1:100, valid='Qmax >= Qmin')..
);

[hmin,hmax,Qmin,Qmax] = CL_inputParam(desc_param);


smaMin = %CL_eqRad + hmin;
smaMax = %CL_eqRad + hmax;
ecc = 0.0;
sso = 1; // sun-synchronous

result = CL_op_searchRepeatOrbits(smaMin,smaMax,Qmin,Qmax,ecc,sso);

// -----------------------------------------------------------
// plot
// -----------------------------------------------------------
f=scf();
f.visible = "off";
f.immediate_drawing="off";

plot((result(:,1)-%CL_eqRad)/1000,result(:,9),'d');  

a = gca();
a.data_bounds=[hmin/1000,Qmin-1;hmax/1000,Qmax+1];
a.tight_limits = "on";
CL_g_stdaxes(a);

a.grid_position = "background";
a.x_label.text = 'Altitude [km]';
a.y_label.text = 'Duration of repeat cycle [days]';
a.title.text = 'Sun-synchronous phased orbits for ecc = ' + string(ecc);

h = CL_g_select(a, "Polyline");
h.mark_size=6;
h.mark_background=18;
h.mark_foreground=2;

f.immediate_drawing="on";
f.visible = "on";
