# bitmaps.tcl:
# Chess piece bitmaps used by Scid.

# The piece images used here were generated from freeware chess
# fonts available in TrueType format at the En Passant website
# "Chess Fonts" page, http://www.enpassant.dk/chess/fonteng.htm

# The authors of the original TrueType fonts are:
#   Alpha: Eric Bentzen.
#   Leipzig, Merida: Armando H. Marroquin.

set png_image_support 1
if { [catch { package require img::png } ] } {
  set png_image_support 0
  ::splash::add "Img package not found (tkimg). Some piece sets are disabled."
}

set boardStyles {}

##########
# Size 20 is only used in Material search window, not for boards.
# It has two special extra images, wm20 and bm20, which contain a
# bishop and knight, for indicating a minor piece.
#

image create photo wm20 -data {
  R0lGODlhFAAUAMIAAH9/f7+/vz8/PwAAAP///////////////yH5BAEKAAcALAAAAAAUABQA
  AANweLp8cCG02cSwNB8RSACctgBAR3iDqJDd5wlidBLCHGb1CQzzx+yPDYAWyJ1gixpSwOLM
  CClFESSRup7RImF42zxP0Vpg0EE6SGjSCqxTKYxHN4RJ6sYETHxVNa3vM2gDQyBoGkNbhIdq
  FHRBZyAaCQA7}

image create photo bm20 -data {
  R0lGODlhFAAUAMIAAH9/f7+/vwAAAD8/P////////////////yH5BAEKAAcALAAAAAAUABQA
  AANneLp8cCG02YQYgupj+5CbEgyYAAQCuJHlNYZo1wHDo7VyOjSAebQxS69R25UCvVlmMXIp
  TrmhSGgB4S5LzoVQegK+YJtWwLWEFjnzGVL7ftYQMoGQggerZ3CrLealoxomXxJIX1kNCQA7
}

image create photo p20 -data {
  R0lGODlh8AAUAMIAAH9/fz8/P7+/vwAAAP///////////////yH5BAEKAAcALAAAAADwABQA
  AAP+eLrc/jDKSau9OOvNu/8VAIBkJAhl2ohqe5xuCgTBGL/oQaMwJew30KwWhAkGA0Vv8gvk
  LALRUxJ9BTSAk40qFXWzkKZWCmQit5Uo2GdDW4ZuiFQ9koZ33mxZAjhjBidJFDNIRBgBhRQD
  Q4t9NH0NP3o1BEgEYjNTDix/SIITfQOIcROIooOFpouekV6PlQMEQ2qaK6QSsZUholGit5GA
  BJeAuMIixgDCnwrJAbKLsMPNsiY0VxeeyxGhnoZm2cTb4OMrP88C0X3NVWF+2CLaELnCUTRm
  6CfDtQuUwv7G1xb8iHUkmSV1lZy0GpErSZR9DbJVUOULCUQl3VRdPDL+rtsKRM8MxuqDjlcr
  FBIflkomK+CdLP8I2Ivg5NIOmxIe5RnygOSzhDKlLGqg01LCGjMhUHQpj1AhmfEYmHIy8JSJ
  jlZXAHIUDWRBojWcFnK1zZk/bw9oBLt09lcuMcpA7eS0CU8WVyIeMTBHD9ARdMjkjPt14BhF
  QEkddDuhSm7MqIUrrgJ0ZhSDvJIr+7o14x9dB3z9zTtCE3A+nHxiUpNXsFKgXj+mHPN3pKa/
  z5cH48LqJJwDVWoT8enYDis4W9k4cjzjliWk0p5ZBn5QcKdvOardXqqXN1nJVjFpxMTNgJw4
  4zypj3V6kRlxecYl7g0+mZtewcLQ/vYMjTb+U6lh5fXfJtmVNcpOj/xnGzL/kHaeO/AZ1xtN
  AaY3nHk9dZOHKnH0th43M1D4T2KXzebEbKKVFcoMhDEz1y8cvUjIPo3AU2MmNI0zYGEU2eiJ
  a3JUqF9PFT6nnnd5GHMdRrScQMeSC3Q23oCdxXaEapAdMI+Sisy1I0YyQslMgOi48iU34AzY
  yxlQJTfUA1hRoJMXYmJkHESOLIXIl1v+A5mAMgE2IkS9qLUGdDH9gIt0fprAaHQRxHeHeIfV
  eEc2CuV0Z6TrNVYcVrdEodp0ZY36WVVsPrPYb/HxmVFykfrYyJfLddTeCx15MZ8ovJlEVHx1
  zoNillrWICgh2zxpeluLfbZVnllK9pefNiiaSopPWLrVD0BZoqnbboOhBexxEQF7bXxuGfdg
  rlTEJxt9YDh1n0Dj7rOmjhtVmmmG6E2ArlRpapGmYsDa6+2qlwYcxAWHyrHwwxAX1h47EVds
  8cUYZ6xxBwkAADs=
}

image create photo e20 -height 20 -width 20
set x 0
foreach p {wp wn wb wr wq wk bp bn bb br bq bk} {
  image create photo ${p}20 -width 20 -height 20
  ${p}20 copy p20 -from $x 0 [expr $x + 19] 19
  incr x 20
}

####################
# Alpha:

lappend boardStyles Alpha

set pieceImageData(Alpha,25) {
  R0lGODlhLAEZAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAAAsARkA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq4QAHSBELD0QFcuLN+rzUsC3yZAmP0WA6EiqQEM
  gq+jI9gRWKVL5YGZcUI7SW038IQNjBYnUz14haOSM3crzzyt4kvYA4gJdGg4e2xuSXARd0KJ
  dlZUQ2GBGGcABACOGH2TegMEf0kETAKgeQ1yinUafzpWkXqNeE0xAYcTd56NNrgUi0uwqWCc
  BKNjwcJFrXGzlbMWn7igjcMUAcwKfcgVqhu2vhd9r6zYENwCtIjg6N0SMXCW4hPaGp/Ccziz
  lsXHFtTGwtQV8/oJrBehHJpZ8cbJYIevnDt2AMGRaiELWiMn+9K9Q5L+Dt1EBqwW+Nm10GEQ
  k9TKASxGME45gf3+1SJCzxiZbJRgCrNkjg4cUcz6dFqgpgG/neyWKfO3cYtEV3iSvJrVlJvV
  bB1f9TS04OIBS1esETy6rFwbajn1TePUJw0Znf14ShBqTJTNng6ohdkr8wERVU4q2XjbRgFQ
  BnRjUkp8d1qjptb8ZDWI1xq4Tlor6N2bpO+UsGCpSI7SCDERP6d5nmZVpHLkxhMypR1o813A
  aPQ+IiHDGZXfM1mE/bHLhUygNjCFLo2rO7QV15ZTpksE/BxuzOoQ8eZ8E0JIp6Vf+RiJxAsu
  5ebxYIxAlukuF3BpQgMi0G7dlTpbEn0yY+n+k2AYxVBYV/GdAVdCRmlEQRCSgQNWGwiCdBk6
  EXU0kRf9ZTXYFwSiU4SB4PwVgRp6bQZdUjAtZ0kcs9knTS35GKMfA2RY0Y+LeLSCHEz/6bTe
  AyQ+xhUQWdUYxFDWRfMUSRZO0OBkO5SSXlRBkNOGblsshmUWBgkZY4REyZcidEvABUpsVJlJ
  lTkBxeViTFjasiUd04XGCZiWTUYmeNPFBmVmpUA4IXYXdUefcBfIoNIBy93o5CycXWPBm446
  OZuP2DRaF6ZYlTbpc1k0CKGTMnAX5YKTOfZnpi5c9pdWbaEJ20oKtFfpOtt1BplhZuKJRIw8
  kmKJTpR2QqZzi9L+yuh0ZTzK3a4HQMkklMICJZ5Ujxk7TVqxenOkj74G18+MDxSL6IKCSUYG
  Jb7ShS0uwVRD6lRdaNhsMtzNGW2qqLY5SlgUjSKjjJzIO6KYrXl76bjhRtvbvQs+LBVW6xrj
  hCx+VswcRBQgC2oak00Mz8MNdyUxxAUJNhVyG2VyRoidHYuwP3syujDDu/x3Ermm5fosdDsK
  xi2WL9HzbSUle+yVZiGj/MC7UdlTJckGP63uy6wQLRkntsQwzcy1ZWOm0ePwVeMZPhd0s5uk
  5tQj0kTbx1NdDT+JDmRQ1lg2JDDvVdCfAEOw2pF3EEE0doOLEq6BY1ftQGBjy2VUpCua3sOd
  d2PfB8TaSE+T+UbkdASd0hIlK1JH+97tHeDhmiXxik6e3NlMsvs2js8ka2HKXirhk+/juD/8
  ju8nm+7A7hIb+jTguoXeJAPO9/48Yqy3O7XExpcSPF8fsSH76LW/0YD3e0Dfmznkfz9i+AO2
  wL5r6T8M/vvjPxxopI+zPySQ9K8ffs1YCKAAB0jAAhrwgAhMoAIXGIEEAAA7
}

set pieceImageData(Alpha,30) {
  R0lGODlhaAEeAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAABoAR4A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s64aACAxD/N44aYNznf9ATmDnEdAEwRwtaRmG
  BFAk87WcSgKCwJNAkFoPgIC2ISZmZmLf14Edf6Jea9i9KHfQAfWHVg1hu35QdBcDcAqGB3AD
  FwJzDGVxGYVZeh58iyJYUIFZG5ObiVGHUZgVUXSakRiTeWYbaQNiPFlchaVCYoMWAAJcUr2A
  ib63E71hjwDBHX0fqZ0fvHBGzYIb0aCIiozVdaKWxByxXFxOHWXjBHOuFbwDtY0awEaFw/TK
  7FBE0eur4J7S8MwBzMVvwrUoBSMo+jQP0a58DPbt8aeBBjpmGcSkc4f+DkvCB7ySjQuZAdil
  Wnx8lczDIJYqCaxsTMIUBgrFCAxtCqQkLdZHBwfh/GwAsKi3bbcmrRIkUycYTTchWByH8WEA
  ApNEjvN4ISQ6LiQZfR077iXOIXEasaTAq1UhtABCYqnhcqiCnNM0XLtKwFmvcviMBrQguOhS
  UKPyGqQ0I0tIuUbQDGbLl5zdR43Ifn3GNjNYrWEreNQMljNQiHfdrRXXR2JEz1s9t/0aerG0
  y90K2zwW+JTfy34F61rg+sAnTMeJo1aejEYyj5Cu9kg3eYLGvsMlzCaNrp0prVg5gs0OIWjR
  hJm95KF+qLkbY5HWj5UvbmysC87Il9ctrW/+1AUAHQGHfqdxVx1RRzmTGGcOKbBeKs9dIt0p
  WBHIXGmXFRIXd+J1YSEv6MARIm7zXHKJWXXQ8N5WKpYFoIpEaQZiF+CF+NBthGQR3IBRVNiZ
  iGUJhUGJJh5hHTehkJITgEgegtVtMzYSlDsotjREd2tRwEdlZHFZiJYdxiIfSkMaqOVcTpYG
  mxRvEWNPly1qphhORv23gC3mCaVklXf1FFdOdjLA35l5DWocJQ3QU5R/4iw65wNzFflWBbF0
  OFYYFx2YqKVTsVgmaXwq0FYuHXGZC1cNzEjWPqB+tOMplPJ3FSm1PYCXgNpsU1hgYsiqiSsg
  nrcPXr1oaitHVcH+pCNVFh1R46MFdvcsiWaKdkSHG1JlRJYxggqMnDmSstSr+RzXxX/m9Yqj
  roI1wV+AEPSC16x/BXjPBFMlG0FckAk4Zoh2XqeZLDkaaJcxNdIWF4r/bsblV/dZwFCgxC28
  53oTe3ekbha2xN/B7zZIBpUAhddFyRF/V9oZOroh8FjQPiBlkQtL8vBX3F5hKWn6ggEbxDtj
  yC5iN3LGR4C2mDKPSD0QHatuFJNb1E2PIY0rQrXu+y2NFXmhIaiBRlPkchIHzexD3I0I080o
  DUyxz+sudadgjYXdCM3G0gl10e8mJM6U7uQUIX5hdrzfzxDHLHPCvFRkNpmEMU3zytb+VdbD
  TM5tZbiDA92h25MU7yoJf28vdBFDFGjUDrLMZkYwPjDjdiGHijuwtY2SPB4eIXnYZKJNOUOQ
  hu9jb7s53ADJLip/oBMGrs2Sbvl2feGdRE6gAkufEjmbe0Vb3hG8rNmXT9eZkaw56p60aFeO
  BJ4jXRWmvNRQPClxYW87qP7mRwvGRxPqEkR+XteZhAmNd2l7R9lsUS6wxI9tOAPfndpHHdiE
  gWI+iQ0txvO2PMWtfPIjH5hu1bPThEyCdwkgX1KRBokFkGm+oZj3ZITCFCXwSQUpkkbqB4tL
  FMOAqwrVodpXE6ahSWmm6pAm+CY//LzLWTgxUXDyYKJinNC4aVKpgSautMUeRO4gjcGaEGdI
  mqzF64bU8cfqLhGhv42tILdLmxCpR5rgxQuI3RnjFUH2xJh50HdNjNcV50g/RJmikFiglO5O
  5zw0xqxhFRpDGuyTHVXdUIhxfB4FMkmWOV4xdCFrA6SEwzmjVHKQmHRkIxPoyROC8ooyM9/c
  jCIVVKJrj1r6ZC5dSZhBxnJqHqOlrWxpm5D9hFj+W4Myl8nMZjrzmdCMpjSnSc1qWvOa2Mym
  Ni+QAAA7
}

set pieceImageData(Alpha,35) {
  R0lGODlhpAEjAMIAAD8/P7+/vwAAAH9/f////////////////yH5BAEKAAcALAAAAACkASMA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLIPDbN/4Wed8fwKmgGAY8BkDg6JRAywJ
  icseUhm1CKkiwYAwEFRz3d2Xci0Nz+NbOD0RApojLdfLVgCgjCf8cx7WIXpmfX8HdwJYB4Eh
  fXQkAXcAiCJdkYeOZ5IZfTuMCl1oGQMAXHsHcgBiHqICqEGQmTR9sBxPlh6bCp0Hn36hZ6W4
  NJCpIXcDx6yOdwEEBMYjvMQaWqKcqHeeqF0aonMNp9IcayWGv8qMWtBn4RjBuryNF/Df65Pp
  l8zNZcWszf5Psy4sI+DmVrNqYeYcQ9UsnhVMDWoFxDBuRC1ZIsoxWgb+QmOyDn20MVo4KIPE
  iBBDVJy0xR8XdhkGtnT5jFY/mramHdw0Z13DDZ+IBQV6DAukPMdgUoBXr+MvdNU6XGQ0MQK6
  q+iADhG6lWgSBkcXIDkGcuZOp8yMuWwW1uTNtRynmV1Ldxs3ZN+S+mI1JFIhLcke9e06DatT
  Ij37OCv1ECseTY4da+21oKm8wayKGAtsyJDSCMfWkpWamWBSus5yPhyFOnVVB29b41yKSM69
  XQK8IY0wUB+k3AT5Rvr3keLVz21CJkU3yi+GqVhf04t8lfb02/N2Q+hN8DcXN6X1Fb9whbj0
  B2plu3yiifVOs9hMvskn+5Fzq4RzMbNraov+EAa8oDccXHLs5B5xjCWHznkOeERdAP+1M1Ji
  oDznoGP3QWCZKSVxWCFulIE1XDVpUWPiKEgwVJAvW6z0XHjqueSiBN3IWKA/t2FwIXNLodiI
  EDgW0UWQlaVFTHmoveGPkqjtc1kwGkBHnW3ILcCUHFdW2SBwso0HQXaJUAViSh7mGGaJmCxG
  QG5KXuQjg0gGx6AdN6oHoUshTlAnIzZqldSff5r5QDf8DXkQEAw1JAY4DdTYGpZcJqmlAlLO
  +deUfZWjWwXorJUVN9QAGihyYDKF6EiVbbhLcwuu2WJu0G3RigVd0CfepHYkEWlroS3pUARA
  yjiMp5Y6ymuV5Sj+0auikY5WC2N11tXcXKL9CppjuP61xZTIDLGpgpswg8SCoHJbQbKeQAUV
  pebktSZzpDgj61XfSjCMqG1VUFSMDVVbpaE2UivoUtSKNmln/QXcJ8IPREsXYDHyN8GO7dKK
  qbeazjjocRSqqi+mB/dVJnWXJphwdK7CilW9242EjDZfWRybpP5acIi/a0UYSsEy4sqKIbau
  mXOm1uYaY18Rm9wAxTCfdyVW256hkM1QdZynzNT5fLFh20VdNSSp0StKKO4N3KMbpnn2yYE4
  asnkP0nApbS9PPsz9mWp0RX0mqOYjVvEDnuaLVPZZiMlTxpNfS5zq2y0AcUhGbc1ql/+5va0
  5bZx7PeXM41m0hzjBnrzo1rG2ZqT7dR9VuoE8du6xA3vrbBssH8M5XMUitodnxo/cPhVlo4c
  GWSTfxh72BNiDm/tFNzIfAVP1Grl6KRfNlY52ghReC6qszyBsa5v/nd9uz5ceHUbXETF9WKL
  D+BWc3W7vfCSSV78Z0NKua3VN7uPErEdqIj2joarzgCKYRsIXJ9eJLvTFW1pbPOUet4QJeAF
  8Da/81bvWmaaPzHDSxLClFuK90ARBadTXpPFYpjANgqCQA+Ay1b5cGQQfpVQQ/H7k2jmJpaZ
  vSQMy/KVpS7Ew8ZQR3EXIBlILlaYX4iqcSIjQz/0cJoWkeT+ZnEhD/XEE7w8wIh2/ptel7po
  ivBxYxRQZJy8VkMK5WiEIVnUYnQ+kEHFbHBjT0QGGbcGqkTRriZSjGB3ZBdHC7yNJkXUkyBR
  c0cHAPGJYWxYHgF1w43Bj5HRkCM1aMeL4DGNDxdD4scmGUnYmNEXB3laryKJtm4dZ15DXOQK
  xTFDXpVShFEyxiSXkyFO7Y1aOrNYXRxWygVADoRPwtS7tKZE+YRql5lpj7o2UslGcexC2ULb
  BFWTxO4FrIKhQ92LZEmXQiqSPtH4R3wuc6B0+iqbWwve5FJTuErZhIQMOkQ5OveLYOqrEj7Z
  CSZm9RBvugaVrnuPBMZCSVax6on+VVEgGOUIn9h8gkFBWc+ABIqrOpLLfpgK3UIhCTVRRZSE
  VwOW8hBzkW1h9FVO1IgVWZfQ54GGnMOEiRtMGhxTiAtf3DRlQvFkPddprx04nU0SSbiXi1VC
  p1Oin5gkgNKUVu6Yg/EZVp+Ct6G6iiIGxdMDt4iTImhTbyUMllfXJB0/8stzvhyqP2mEUlxd
  DkPrhM2DwtRM31U1qF0bKlyb18AmVTNdJLRrVTNx18fUESaNzRrVihfCySVzcopFKWP3yq7I
  QPavWkJpZflohV1OEk6mPa3vVOsAhpoUWKllbRti+1ro0RZQ0nHtbVF7Ww+uNo+Z0K1vAdFb
  3Nq2uDEgIwNyk0uI5jr3udCNrnSnS93qWve62M2udrfL3e5WIAEAOw==
}

set pieceImageData(Alpha,40) {
  R0lGODlh4AEoAMIAAL+/vwAAAH9/fz8/P////////////////yH5BAEKAAcALAAAAADgASgA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987/eCQEDwKxqPkuAQyWxu
  AAFVcDBYOnvCawYqFQqJ2l02bPmmqgRCYEBmAAQAyDt+GrcnZlTVy74f5nJwKHZ+D1VUKUFp
  Vm1QQnRuXpAiXl6FhkJ9J0p5jZIOjgGTIZWEJgB7A6Onj1CrI6kjpWClUQecphhDBEEOu70k
  AkrCKqiZryShj2eZspW0pQq4thvRDdYjwl9gJ1ADwlXIIYcAi2vdqeIZa9q03wJV0kPw1Bq/
  9Qr3dfispeofe0ppSpaO0rN81qZ1wLaAobNia9KkCWdCSTmJvJZli0j+oAo3D7vU0BKQBpii
  jB08nmugcuDDFMoq/dswrRLFjbNIHTxQS16llHwcBHRpUAUVjIs+juOIdOJKEOEwegMRclbJ
  Ulc97FnljWhRZrUYiQgY9huipWWphl3bE6goUM32qQiJEdhYUQRQDWja8SkHRxelrpkpgS7f
  w3arwSNKJXEGXKoUKKNjzItSe2xJkAUQU0hHrxg6CyQcga1pSwvjMgjYATJlmZLJXl5HEqnj
  Dsb2hgqM8RBupnxvKj5MHOVCOG8XuLpdwZEwbUO8cQ437LnGaplDVB7sVODn36evYw6/FuQn
  5T81OLcefbCxN9ugk34Ar+m3EBaLI+20bq/+/ngceKSfbX5FAI9SJ52lwFFqjHJgBN7Y5wVG
  fEg4XwM18dfaLM+RZVlHs1VAHmobeDhigRA8yABkq8G24n1yANddAHmVUyFSwm0BXI5PHDIg
  UoBpEBWFNAoGWgVT8HaYMSGy9I1fJ7kiGV4itbjYAxHyBUVtVykpEY8WiJYcbiaSl9GFC4Zl
  jkAbnqghBEOt6M+UO6WpWiT+KeHflmrQGMRFe9QGpgVTONUkodz9iCOKE7jSlGcEohnJiaTh
  lRiD3x2AqYI8kcSFUP4dNmSWoh45QU2HIummVYxKIOaac/5lGGJj4pGenUFpymZDt67mn4fl
  UIecbt5lqmOofUn+uoCAivIiGD8ToEFkJfuZSgE7z2WbLYAiBlsPXa7wmVVDxD4w66O/Fkkc
  cxSIqSyuJ3JCErcWmPjorhxgq6229NpKIk+1cBYWAw7lM28t/nEkbVnGhSZjsvki5yVfwuxX
  67VcllQdgSBlTOsFoURmMIVUqNvwdqucVJwi4hYnVnOmvftFX+FBd8+FYVXMS4bmjQgyH5CU
  VaZyAaVc5Fp7sXyPmlVeMGRTU9lDksfEfUNgqtdQ3SfF0F5wblPsGmjZLSZnNOsS8qZY9mHz
  rM3Xy9GGZy0EmWQYFnwTBoE1wWvBWrAuJz42tt1rEZE2fUf3LdJJbCXl9Ihz+/L0ymv+T6GL
  1pByfNyAYYtd99qZSzREQE2qXFzJWj+6N7ymRX6Nm3uRlZHrCDGdXb6BC/55vKSL7fczvAhI
  uOPNaTtPO9miKROQXNLjpRIYSGvx27TDpejFIqqSCdhUb9WvL6nXNbN+nTfwRra1aKv8icRS
  C/3jtqclK6XjrdpWaQf3DVifDJfvu+iro5u3ALCd0bCtazEKX6Q2dL0ALuBPIFJUfaQkAUy5
  zG32qZ6ccrIQb5nGU7Uwm8NW9S6yxUsDhKNfBdt3t60RLlgafKDJ4BaavgwLDPm5YAYmt6RW
  IaqB1UBWs/pCAQjqJ3Q9fBeqPODBuklDZxkSYfRg54EUpg/+O/brldqaaJWK2Y1YMsuY/xp1
  CL+Qiji5oIBHbGYZbcQQQxhUHU3iWJwtVUB6aKTjIt5Yu7+FJk7KAQfSpDjFb3jpDb6popsc
  KMMsvgkTM8JKxQapBj46SiIUVOSYHDEgR6jnGcY7DwN/REMRKfBjMBMiuv6DvTCtpYSNhISY
  gPc+h+VJW06BpQlHxEheOdKV6QphbRgGsRKBjo+gkEwOKYfAJNCxlBgwnct6maYh7vFnqpTI
  YsI3qHohTQSUKVP6CBnNmoHAihwUUs5Cia92RWQ773DWzoaSS1lR7U8jcI4E01jEZ1ITjoqC
  EQrVxUZQArCQaUBk6+DgFGTOspX+8yPP1Ph5quPtS2/nXCQH5DUxTB7Om3lSEtXe0FCgsA2Z
  vqDSjywntdOhdGQrReYaUvEcAJyPHjNtJpxyii3enC8ViQxQ65zBy7688WG9eSlP3nFRfsmU
  SzdtjDACM0aWMOWnNpnql3yoRpUCiauP6yitevlQF6XGTamxGvn2MEe1SuV5HtGK3NBS1Ec2
  Sl8X7abXTqlNnRYGDugErF8R9ySBmIge/9SVWPMCVnfy9WrqsU5hETtV8DS1qeLZK9So8LzB
  muue4xOfXMnz0rKO066uYk/6Kjs/pl5WkBD112gKqEWv5ewqAvEsKHiopcGY1JoYtSdg8KlI
  VQyRSZj+MW5Cl5dQtg7UMJMRHzRdJU6klbC6g0StO+/2AXR+UGrw7A3QqgpQyIj3Q9ONkXI7
  qVc8PBayjdpXKsyWP9ey9lrZ/NH3rvWIZ/n0XxfQk0g9VhmyOlJm9ssI1m7aIbntq1KOBLAp
  abY9CkeEvHLKi4cKatz0miu/Ls3XYmml00BdlLHaKAdmG9spa/a1pc3CsAxH/NVeRpiPWSxs
  3Grm3chFWML4hZ1u5YRdhHnYfF5V1KcE52LGNtOIve0DbzGZ3ks2ObaIc/GRs+ZiGZvWNHPM
  oic9dxopS/SuP8ayA6RJShu/V7TlfO1rtywnOV8WtfGRr18OcWed6s3Od3Zv4BoBLVl72BfQ
  +3VmhHvp3e/Sba7LIk9F0yxoixKaxdc4tJ0TTeYsMjrNRyozV4o86R8H8MtsWd8vy7no5964
  NK0b9WlK3WpXRljVWbyErnfN6177+tfADrawh03sYhv72MhOtrKXzexmO/vZM0gAADs=
}

set pieceImageData(Alpha,45) {
  R0lGODlhHAItAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAAAcAi0A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoJAzKA6PyGSoOFA6
  nylA0xUQWAPQpxSQ/Uhf1WtXuR13qq6BYEBQT81CN3yDbqmt1vf8J99fwnopagSEa34MUkYQ
  TFwoRVUBiocSgHZ4eZMKiYEMjIIDkJKZDmphlmxVnHNXAlgOkFYpl3iji5eWl6pwl64NYbEo
  s8AsAL8BjS3FrW0CyCp3hiXCU9CYB9XDRAKEag9rhbogj62iKspXzivneOmOeOEd0wrY1MIf
  2N65jqDk8CMAV/g1w8WmUKt2JYyZ6Jdq3poi0UqNu7eNmbeC0dz5IwH+UBhCE9jwHFuhUNqs
  evqu2fOAz8HJZxnVHSREQNlHER0D0CR0biOHjgCYDRyxjFk9mhG/EajTAWCkVg9SVbm5JKY5
  YeyuYk1pouMsqhrkqUwZ0guvqJfAUvQZYurOmlZJhAn6licktRlK0a3ZUIQVbsKQnjQIwqsq
  aHhZxkXxS+yJxlifjiQR8h3RrZixFrZMKitMtiCKvmVaohTPgnXnjoBUdynUEH9rMmFCc3YR
  utnilULbzQvESA0QI+J3229mdyIr65y6OvPr0M6jC4At5hUt3xKDe9ZEfEBiCkrf9k5YtCPq
  0Va+U2rWGnAvxe3j71yc9+HQBQAhgkYrNlT+38r7ReDcY7wAJsxy93kA2VbvdVCZdJwptp0m
  XNGB1RT+JeWYB6LtRFoIORHSWHymfOCWfE6Jw5587eWnYiENVgHOT3fRpMwxokk1k10HfTAg
  R7PoJNiBroGwIGYNbvAghPRhUE2MFWJw415T5rhGjTxO5mBFbzVp4Yos7pSTgmCethdPz/2U
  R5jipWdBGEnK6BoyTsFoXZoulWkjHl2m11pHFG0V4ARHShcUnhkcGYmiZjGJVWJw3rmdV9Ux
  EKmAeqL5F0CH+lkXoB44RSVzIF7BZmt3qNdJh4CdJ6KXFdx4qmtaTuDGbgyUl9GJ4ymQSh/W
  nflWnR5mumeSUmL+pmoElDpqqhrLOnShUBtusI6jtUrwZAOOLehLlAucaAyaCCJI17hwIeuk
  n+aJMNesrVWaQWxtukqYkUxCgkFRveZHE1Oi9XqAUh9SaG+9tYXXHrQSvtSWs1jZuQFmbCz5
  8MF14WrBShRyrFmuszzgr2sFGkXANm68WjLDah5jI6lNmQpvG21Op0HKO9FTr4q29TwbrL4I
  SRq0ezZBNJp6FIXoAXKySC+98gHNrLKNQtxMWifbXN80NTsM36lShxvyAgCO5fXAZ1mncpCv
  IiV0ZDNay+qr0ZL9jcIsFlOXNetmPFiXg3Y2a9gdyztIsQMLiZQejSE095/fHM1iwRb+NOuR
  hDotSWQuRt08bc4XqujofpS+9yDaGzbujU6WvwNXxewt+aqFiqe29AVrFCOsfEXYPijUoGOc
  m5Jchkk4d780ArxXwr7mVSQf4R3fMsDHJ/Cb0qm78TKFXlgN7L9TTO3HDUMYoBTJKwCxK897
  B0HkmcF4spCZwa4BID4zIe+8t83Kqd9aw13xgtee4XlucIFDnSGkx5o2mS2AFzkVtKS3MAhW
  oHsHiofVxgc7C4JHfJoTHZMSqA+rPXAC8APh1TooKItc4FrZixbWwkSsmnludyb71PEg0DTj
  JTA/sLDX/9oUigSh5VTMA5v2RIbByMgQYp1y3R0818IlJdD+Us76IS+sprr1GIRBX/ziViSm
  m4QZxzS1mQ0A0Pc4oWhgZMPyTmtgpsHBeRB7cDHWwYAyPwpIzmnVa9EO8zQ26GjuK+NL4RuR
  JJBHQcdZZ7Aag/xYEQwGhTWtE8nrvra4trwKFQzq3fTuSChjCXKJuKvd5EhJAX/pDV4uY5kE
  arhKVZLIiFs72xnWRrGghKSDX3IWKrcHSTVJ8nITcGX9TKY5X+Kyb/O5YgNk1ReHuAyQcrPl
  KP2CMVSxkgI9nBkZS4lDv3VTTLe7QNlABBldwHBNU1wkxIxTTA40EULD9FXs4gcQMXItRRxS
  JeWsBacEdYRNoFrkMRrJC+/QkUP+53TgliLKu2eKrI0IC9NDw7IVnHSROxBR4TenRg7rqYY6
  9STeMSc0y6J0rxD+LNlGk+UqWYoOShR1ozaSM5vGSJNbOQWMj7QJtpGuiqgEXOVPM2lRlb7n
  ndOKJ/EUZ5t/DRJTKZ3YSg0YATQWChWJ1KRrfhrOcS4BGXdAqg0jWc50kYCCFaSIODuJgRNZ
  D64Jy2cpJ0kZtI5Qp/ZsBZLSqdWsUjGUGAyQuKbhS9kQiRl6nZo2B7qZNUmQq9vDJk7w6re6
  GSxnIYnEeWyKvTMlgmJyPFZlMYMCplYRsxVg0j+s5tkDlC61f8LGssQlypxlDBTkwhcOAWqS
  tlbQqAL+iuj1JjorYHEgd9gQLUOhi1zreIcePTuJFAj7B+dEdl/OimJAVzmCQwrKR9LbxO74
  EVgcMkG6kPuuLbrp3IcZ97jMrehP7Sbd/PX0qiB7R1u3e4vABuS+bNzfTp2z328564v5vWuD
  VXJfdP40NgmODGpgW0o2EDgz0KPVZtQ6p/JitKLVlVR05AsBqK64tjjLWAWlGUjZlJPDlWNS
  bbUDxayl2MHZY+cAjZet+jzoyABWHyMxqCISf9I4cwXszfS3hnfkgQmVDWrCmkrJ3U1JhzS+
  ZrFkJibKqjOL9Myijx1EZe+VQ24r9WwpDhSUd1ZzYgFYEJUOZOYLnhg9LO7+jJbXSiY28Elj
  JtrRzG6009o1Cz1JRp2r0LXlH6s4XyJU83XIxFd8bZW7Xd1G6frEjpPRmIWeyqMULY3FQUNW
  rlGWqq26x7pS03o/f9Yoq5lGZquKWWWBllZ6cVvpK94TSYFaH8ksMN2VAkeAn961jCxXZ8gc
  KtgqQXVAFgS+9s6VuHiOspRbXCjRzo5fzVZwcsUtVK1WeFg4DrWrW2XgY2JbgRALaSuPHQtM
  E/PTz82zMPusrSrnG8C5viVExQ0UcqKIOaJCEahty1k2kXZj83Yh8cSNcIBr8JjtWg+EuODa
  IMUq2m4KS3/92zOCy5vlLEd4xtv93JkbBB5w/FSrqvCTh7ZeXNDs7lx9OE7CIdvxuR5HurO9
  VPK08HzklET5oA4nThdpleO7jjaHUF7wF0/z2Bsx7wg56mylr9Tsx9y61kPtdUSAPerRDl/Z
  C3t2JcH87tLEu95nefebbMK/avm73gE/5cHnjwiGP7zdE98zljBe8S3uO7Mkn8zH529ZlrcN
  4jP/5lp4/vOgD73oR0/60pv+9KhPvepXz/rWu/71sI+97GfP+gQAADs=
}

set pieceImageData(Alpha,50) {
  R0lGODlhWAIyAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAABYAjIA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWwe
  ATNAQCAIQJ3YLOsak1Kt2rB4FBjMqGjB2Eldg8rntNq9bNM5AzssPyAQ+HdJeoEYeXN7cgJm
  hEaDjBQAaIsvfH6AjwqGARJTii0DoFSgk5gQkaIxhmmkjJqcqCyjoqClEaenXbNUXCWjvgsA
  vqBcwrQheWVVEFMDzCyJjrXAcrlyvCTFk8HCxMUiqsoPnbAq0IcvXmibNJ0Af54wp7sp0Avg
  5PUhVZYC1weRfQhMiRVKkbF448LNSEjOhbx+9BLZSzQpH4hE/h5GMyH+yyCrLZJU+Vtxyoqf
  kiNRgPs4YlazQ8hAOVNQpaMIRfw+Vnr3YmMLjWhSpgCqTijHVRGpZaK4wOIHpg3unVvh81k/
  P/ymqkgXEGu6dSumWPECtsQUrHkmBrx0AKefqhmYGYKgqyxVrS7MwS2RTq9CFQz/kgg8UU5F
  iSEY0k1jtxzeWFex5vz5RbJleUY/nOr6DmIvAWgpZl2FFR6IvilVgcn7eIVUpCD91pyHgmjQ
  o7JzN/SAGoLqzCT2oqBiuXTrEYbcFbfc7kTy4ps/hzY8WlJplngk+daOfNQDQyx/mcgNWXcA
  gZ4/y8be4bVuqMe4O1DFfoP4qLszeR/PWXL+2q27+CHWcgLSRkYVykEnFjagEehgcaaBkAcu
  D+wSYQd9yXdAQmVJtdpNspFkDQG2nRcJcHG915gH7r0HW3xoLJbfBhk2xCEDHqJ4gVvF/YdC
  OwUGSGBzIgz4IHorZtffkT1e6AFxbKlFgHAUaOLLbUYuKM+VM3JAXgoMJRhYZSce6GKS9rm4
  XksvFhbjB1Z2BFGWymwpZ30ZELcclTR+odw2wzxI5AdGMrlgfCYxuZwXeDawzWJvaaWnT8NQ
  AKRlmuyDlViiQPcFiH7VlkgZo4zZ6RQ6TmCqX2hm0KKaTkrwaIVKMYDYfAMAd6l/X+wDgDuc
  8ujVp/ElOGyjG3T+sqSiJBLLoqaYLjvTG0Iye5KzFSR0DUCTSfnOtm06UKinekoWFIFkwWgO
  shqs+t5brVrQYkHQsGsprNDE24C2jsJ3gFTgaigutNsMS9wtUwq5zZ+HXiTsXHz5ae1yyvJW
  rbkNXmagZl9k4/FL6VUQiRXI4NfVIOVGuaEiXqR0ypEIG2zsniF3EGrE+Iqmcga2TapXqhPY
  +bEwt12gkV2v2ervymm4XG5CAd0itZDgnBcp0BGAl5W9Fig7sYNvcsAMgeViKpjFOWM9YYH4
  JXjJTs2yoilAy0T2IDKSjQ3z2WnWa1bO+TZ73L16RWqOvkaXzeTGFUiFH1SvyW1Y3e7+EGU1
  3qCgF9CY1yIe9MW3iIDS1wI2WTMGindGmmV8WuDS0Da56pYjittR+1R67vwPtEei2jmzA3H8
  s+iASyLH1TbXa/qt+mSsaOtuhk2TOdM71dbkDowc5KgF/qmpqVYHj+HF156e5nkBWD2x8+Vz
  jaOw1XVqtvsSpP4g9AvcqPT8cLPNgP7zgZ+DtIeemTlId4mTDda2czB8JQwNBaLfUpRnNr+B
  ClbJYkwD9EIv5jENW+/rQ8+uQkDfjdAdCGwcgtA1rfbghHeKygPNPBfAZUXqgLHykvoUJT7Z
  FUUB3GIdBM12hQyxp3+9Aw0BY5hDFepGgo8TIaz6cLx30PD+cYV74LrYhK/s/HB3aQMi1LKG
  k1eFT4nf80sfUqgqGFKMb4VgGemmJC38eYtmOIRihWxIIB/16TZI7MySAPGQzAQyj4fsYxPb
  eCY4FQ+NpIGYBl7VoDVdUE1omwclOQhGxtVQdeaonEFIuJ53SNBdh3vWuHjIvukoCWytDI0e
  HSDAuw3uXpo4Cx73BLKSTSCRZOvExNgYAVTm61mAWyN1JOnDLFqyeV30AIeMuS4AklGZrIKX
  nlCpTPployal8sh9JskYZjFKkYu0hRuNA8sFboePyyHm56wwjj7uEGMI8mT2aumgTtwziQv8
  Zr284UV8WU1nXfqcXwwXym8kM5P+j4QGcIKYm60JbqHf0gdn/CihfRgpGBxaCQ7TWUy7WWtk
  yOGnIme5MhSu8yRDKiMcaRnLkcITQrd0nfUcib4zbc864BHbI6/4y4dKM6LHrEDuSokwSgbE
  jt95GEkvkKsC9fQLDMnckRKqUNLp85XWkuc8T2JAJlU1YToCpunKulJ1Ye8YyhlT+lYRV+UF
  9Y+AcycD8cXSEeZVXrOTzQ21aI5uORJCLN2nRwUTs8VNdTv/FNRjAXvT+VGLrcOcKa0wi9OT
  QjWqFhwBUFrVIpdONoCj8sWYEvu+nCWWmo3cUYP8KqCD+kWUOVVqKz9rC2tOr7JbOyqp1Jgu
  BnGWl6f+eenEUBXHyJKtpm9k7fWoR4IMNUZOWbyrUA1yN1cwyLXJRWrRAHs5Nf4Bm+uKYAiC
  yCu9ymq8ClilLU8rDjmQSib2PYpXWWu/r9lRb44FnmYxcEITYGZfLgruUbsCqLNaETd8haZ4
  2+WWVa3RtIFDUpFM2jmVdGhXZuWqyOSHTveW1KtEfeccZamBxt4PupIJXWJY5WEcwXap9FXs
  q5Jjghvnt6PipV/MTLXG+KmjfCaeRk15iwF/HreCiVXppnKbJxg/d71u5JIvisPcFpOPRPi1
  b6A0Zj686qUFPvZZjmn6nhM8EstitmGYm/bHunLXPz3SBJI3LOcBOyxRmV3+MwPYi8Mk08qr
  VJ6AnoLxnrPyNmYgU1HUvppB2aS4yg58YF/BObQuB6d4oCrDMMcxPg73iEkyTq5zuwcm5R6J
  mWC1pXT3599ES4APjNGqfyKtCEEDsVchYes2GENpL+vG0CXNmeE6WlmxVhl2AnXoZTo4i5n5
  GozGmnOcY1xsAn+Z22Xm6Yo74zBoF2Q8oFaXg2FGH+GJ+msgQ/avXSRvLGZael4S59CYDKla
  O3SNsBW1s9U5yinyGM5PTjXxTB1Wfm9yaWi7r7nDbGhg6rpJiZUhur79lln7VaI4A9yykakm
  j3ds4uEMN1XX41QgH+6nEPfik9vHIOD2k98bBI3+1nAOWpubjqXIMKA8AO2VFub7n9Y1YCcc
  Ct6UFm/kl6zVd5EK9CoEJlE1svq1pzdaY2WdQoTi+Mb9LNtxB+lA2FTmpV/BcM+SnYyVKTpj
  jDWOKFsH3OQDx4yLt3ZaFc8d+D6qXvpeN/GCUHZSLEqMtWVKh8Vvh1mNFLVWnUTCT8Pn6DQa
  JW27zWeW3exTJrkBtUbmrU+X6GhZVl+oFVHLz5vvD9S84WOu6NkzuRKBWQv3Bh4BnGS94PZ1
  qbjHbfQ0YT6eIn79ujY3yrVQEmhSzqwqV1x82aH+pNVfeZBd/nftvrfkD6ezbG3/rI+HcuvI
  SLM6sq9T0Ceso+73vinwxD7ldQB4gN2+vPv54d7+Msvh0fdq10Z+joRUCdMo5qdBH/QeOpKA
  atJ/+oZy56ZKElgM7KdUlCdg8Od+hxdAx0UWRSQxskZZ+9d4fXJ8RPQsHDhLBEhyQ9UwcKci
  IVhygGV7gkZopDMhKgh6Azh73DdhtaciA+MiNWh7UBR+ucGCPpg8hveDSEUBatIYPlaEs3eE
  hqeETciEQOiC21c/T+Qy4UeFV5hvWUiGW7gBLehCRhiE6wGGbeZEa4iGS2iGTygNdniHeJiH
  eriHfNiHfviHgBiIgjiIhFiIhniIiJiIiriIjNiIjviIkBiJPJAAADs=
}

set pieceImageData(Alpha,55) {
  R0lGODlhlAI3AMIAAH9/fwAAAL+/vz8/P////////////////yH5BAEKAAcALAAAAACUAjcA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CoVACoCqTYrPZDtW6/YNfgahMEzmdyWFsFrEvjmxkdUL+l7bvIPLid22d6WGiCIHx+dIGF
  UISLHHMBNgBnBJVpjhAAdhFVNY2YGZCSiZegDZoVnTSfphYDaG40c5UEpa2QdRG4myuTpLGt
  E69/NbhovI67unTIKb6JwDfROQKvY4o01gDWLG3ewF3fat95JH++EefYLc900zXvcsOsMvNn
  fb3k4Prj+iWk6RLBaAdLGrQczwQQeBXvxTx8vUhFUkCQmAKJEENMAlD+aeODjR0DNDzRBRaq
  Ue4QShzZ4iE7iQsqilyAkUTFeDdflDzXrAWudj1ZVPtDa04cGUPvBRUxaQCgiQeatmFIU2QV
  qiNAEvDoQCtXGPRmGGNWzF6iozGSBkDrDCbFlVUPZv31kW6MsDLu0SIwNCOMYWP2bkXjt5ve
  YSw1iqSFTetgYJRCJs7QlMorZHw0Xc67bpXEe9k+n40hc/IHmTHhXrTLlBRm15yh0mgquBJW
  nbBq00roc5jCSkZVOD63bXFIp1UiP4ZDp/ACe85Z4B0o2iLu6oRtqVBLKnrrX1cxPpU7AnoE
  87ER/RY8B4ZR3ewBr3hf2zeK4b92S9RvOsP+vJHPeLfCdC9g19l8BhLWF4IGLtWBTAmK1h8G
  /3FCmGeyzaBcbQemMAdHtoEIn3z3EQWfLw5u4Bg/RenDnwlpdLhaeyRQQZ4CxmxC0EkwGngd
  MxCuNZiAp0WYiwlBGnmjCAJB0OQINqbEQI6nQJPiBhsKJiOSa4EIiYi6kchcHfAVBVoJjpWp
  5ov/BLDQTF0JueUGZpF1gFHFHTnWhT1i50JFvwU5AF9rnVAndkSqqGSCE16w5Ft0kHBoKXja
  d6doiWLpJnxz7gEYcImsKaYIvq0JaqE2CWmqqbe1NiiNDaTxZqPLmAicfHrZFghg6w2mXQgJ
  CmVWYIxJSGimFC7+euZcylZHKwUVPcCaB1/ulVQfub4ZCa978WbOprp1CkJCq3L6qwcolkvo
  kXtkq65gll5gI48MaNVqVKp+NeVGDtJXGy6D7mVPrwKf60Gwwq30DXYKbfZds1dasGezx4TC
  b11L5uTAvP26a21zhDbMzIgGP6gqvHCS9Om7tT3TLpkcZnlqxBVAQnC5NoeSCC8MnRorzE1t
  vLME/uoGSbiylhnvuNg9K++kiwYsrsTVFfsZzRRMDLHOdjJw6M+jNUBlrQFvtY0mp26KhiVJ
  W1YFLUuDYM16C5a41s0s65qyB9aYCy7KyCbbpT6E5xk4A7Ie8nPBwNBWrAOXrStB32v+zkFw
  uqKiSiqmbVG833JyV832Z4e7ck/hhI/qaNiIq4avW14DdJ6q3MFs+ZuDYk4Qsdx8W/YwKCQF
  Zt7sESivzFaXWTK6njt9gHIHbqj4AZE/vjjbteKcZdKmLq8p7CXU7nlu6HCBndVXczm8qfpi
  AP5q3T33mQPvTxnZxL/NUgvbv036m/cYKJq2sAYtpRCvI2ECIAV6JpjStExzc0EO6r5xr9VZ
  LzVl09UV+ICyaUAvQ7Fbn9KMgzt1ta8DWlMgZcYHC2jw7wOTEiGESpcKEpbrhBZ4X5A2KCGw
  RQoCDIRa/xYjsoBpLWA41MDt/qXC1XGECgf8m372pindODD+ZupjWRIL6BTrvGV4h2Agm1Ij
  kgpW5V36qwXMyjW1rCWIgBJwR5Kqdha2OS8u+Xng/NDUPA7sqEqko14P6/WoM/pKNApJY+IY
  hr0irZGJVOSb2h6pPb9RC3k7sQremniBNK1qiwtknSDX9weZTU+QzYmjFFm1GMepq40SmKOU
  QoAYvSlpUKQICQgDiKm3dSRPEoFjLPvIgWHFTkLVyYgxA6JLTOmKI46DmolgCUTuwSc4inki
  3liVwb0Yr4Cr9CbyinXHTNjwk5HUwEPIgLxniFARasnUOMvEQWuuipoPkGZNFPMHfV5NjXTQ
  JfPON7pBMqtZ5VTAOuFHMRy5hAL+kZFlIuVEphROc5cX8Oc+i3Q24plBhL66oy/UtDblsas1
  IC0TKKm2FmKMs6RYJEiK5mnSNL4So6ZTFg0x5rlX5TKkJiMoQBHJR4ol9FIpYeHrKgbRTckS
  RJdRlT9BhM8FWBRRwiSjPU1l0wZ6UQM0hSnSekS8quazSyCz4jmn2E8IqjJv7sybWZFKMbYo
  anwKqYguy2kg9O3xW56jJQsNhKyIJqiZVDVQYr+lJZwOtFQN9MYvIavWdGZ0rck7kVuBFc7u
  OdZ9dRCe8lSqFFgVMKWa3eqaVpoB0ZzAFxp1zbpCBdQO6NOv8WtTYDk7WNdaQCuMGl0zRcda
  93XzFa/+rWji/Ems1Jazq+qyHFnl+tlOSpWS2guYGSEAXc92lqQnBYFv+ThRIwVmYnu95GCz
  GpDdMq23rnMjEgmrK+0q9oUo9epRN+bQu31pKse4aUIpx7Letams1V1d/g7IF5+BFrVI+65m
  d2qhz+xXbP0FT3ickj/RpFeS46PwBJTKJPjKTsG29PBjCkq646RqeMXdwNGkaz/MOle8J9uN
  ZFGW4CpSdy4Qftd2syfkHN8zvI6Mb42CGUtpfvhgCltY/Urs3vea+Ktu3NRVn0gmieK3xGNV
  2T3ORWB0WpaXg2tantj71h+/jMExI2CZ2WdjkonYSdUJnll4MR6C+uKoX9r+ZouY2qeGvuzK
  SA6lEZtmieEGs75L7uaXTXAtyKmWtGdG81kADBuS0HSEbA4hnPWz0yWiM8hTDHWNiarnZclP
  SU9Wb6/CMWtOcs1zqlYookttIos2eqiP9lWudS3DzZJ3SiszIZb92KXVDplpqH7ghfElaeI9
  m2iXlrb2Eg1iziXMDuJDlN4u/GeodTHTSTbqQeHrvCWmEIHA3tnMlvzdudqWMHBdtqJQHeN9
  E6/fFRhnOLyxTXujUtDe0EYb6OnqEFy1x+VB6OjIvahpx2l8FiedFdgTnilf1naE2fhurNVx
  kBtbktWGtBj+IGg1/Tm/dLZ4laKt32+tJzmKFVH+0EBMN2ASFm2nunM1CesQo07cVRMMj8wJ
  iXHAsjy67RA6ZAWAcHp2+OQopCmNV0DZfKP7si03k9CHSfNufR0DG/H54Ci4aYAPPUpyLNwv
  sEnvN7akeUcXLL+xbmV1R/BmA28D4K9dga7reCrfMFrDOQvh8qnA8FrUN1gl3Ni2SDDpCT+7
  dZsTZMsUUp1WgrooR7AoFsTWwnnH8U3bUvYxZnPQiAI66GCYNLUn00uLD13KAbedbJcLePys
  uJhDPOznNRuuwBdsc/8l+38ZWFJR6z3GR/cwJS39TmJ8V2XMMQZZrsS0tG+pxs2de2rNWU1z
  Q4GtGPzN1TWL0qfDvD7+4gZll9dZjUDGdJ3dnrXTN6f4rTN9AEVlygJ/8Sd/nMZt32MssrRu
  K8FWFkZvu0cyCph1lMdGBsddkVFStuYfHpM3qtMBBlQ8IxMfGch01dYdCaR5LIVrBhhi1Bc+
  DJMC/pcgYzcjdXJc3QF+zINztPUx4MGCoWRkv1d+JtN6lkRvXCYr9CdY2OVRRgha63dIU9gO
  pFeCBfOBRnGCr8ZCN8h0LBSDUGJ3JlCDEXKDhuUOXqWG+FdiQ+RCaYNvkuOEYacmdMdZowY3
  kvcA4VZHs3JdWKUpdRhdJwgJ1dYXeDMPucaGzLdNVuiEvfWFDDV9BfE0FRc1cIRoP3RJS3j+
  Fol0KCKziE41fqjXhubzhGjUgW2Why+XU7fnM7U3VZOXh5XnSIN4TXv4YKPmeOplYsNmYrVV
  eO+HUJOniRBnIbZnXvyHZ97nLFyYfQy2c4ZAhAcEW781hbiYC6ZmfxUYJxNobd14WrsohBby
  jTdEjmd1ZV/4cGd4h8OkLDwEj2hnjKpYJZeHgJm3X1KBj6mDjvRzgav3erQoCgGHijwGEclm
  h9TkSazoj61Di1zIkP/mkAF4ZdU3WAOQfCOmU12oJPNIjwn1aRgokON4VMTBj/rAhdSGknKn
  QB8yQRUChveYkuGIQSyJOoSHbVZxk08RavvIk0qHY4h2kRjpksqnMg3N2DXgZIzPdZI8WY/2
  CJQAZpKaSJTstpQ2iBn+NyFJeZTdBl9jZ4xCaZHBh2iT0ZV2lWERwpX0SJGoZGJhWZWnIZX+
  4HB0WThZg4CpIH8RE3h3KXjo8pfeMC6CWQ6BWZgX5peCmZfyt5eY15eISXCHWZiEiZjBcJmY
  mZmauZmc2Zme+ZmgGZqiOZqkWZqmeZqomZqquZqs2Zqu+ZqwGZuyOZtBkAAAOw==
}

set pieceImageData(Alpha,60) {
  R0lGODlh0AI8AMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAADQAjwA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtigIBq3bLhWG74HAP0BsIzoKBeM2WkXlmtLpNr58AgncugEZn7V18An+AInh6OIJ9
  hIVWgoyNHmdzOWcABASCkVqPmyCTO319nlWdpBxxeTpnmJhnpw9YAYgQALI1tgFxA1i0sBOp
  vjSiaL8NssINuZAwubu9xhWCZszDeZh4AtEKqaoR2aAziqLV2wzj5TLEr9vdyQrgaTTji2PQ
  QHGz7Dj50y/Lsr4sCBjwHMF7JczwEpQsGxZq4tYNsjcrSLd9Ny7KcwH+sODAg4xAVjShsSGx
  dCro+dmhEuUMcANcbUxU74VKjBIZSHTJgRUmPuWAtsL47+EkhDTJjZFI6QY6LzkXRFWw88S6
  oOteGk2D1IbGcDmCtVL07kW3piz4LNSFc5CstmshmvCZaWKsQUO11TC14+tMfhLRlHVxFmpW
  qYepXrV6EsJNG3zDWmpV9+88P5R/yhmcAhwAzy6EatarmHLbAGPtljiDGoBcB2ovadrLGk5g
  yxFvb6YRrw/nK1MPBK86t/FdYpBr62CYmYBDXLsuNceUiifJSZTztRBdOSBdAqzf4tVsXUNv
  tA0u/k4R2alu1W78ihqgPYZ8+Ox3ij+5X6n+ifMSqJccfjawNl1lvGF2YCsw/ROHdJT5s4Ja
  qa1j2lWtvFZCPCiNs15+BNbwHlH/jLiZQg6+h54JN5moW3nm1eSYb+6F+NJ3zZHYQjYDSGfL
  gtRN8mEHnh0IGogVEnPhSXnBqAFruOk0iI6GPMSLMPEIY2VX/5l443wtYhcbC73pNqQGYbpI
  HGNgPSBKZ1tiSSNsuix0Jpo4ZqZciWkwiB2QwajAI5BBenOCIIQmqqeNICBqSS2s7DmCa2se
  oFFTLfYI4ntOhlBmTM7dFlM2d2JQ5nulWpCmmsaxKIpJMk7qFyOXnsNUqtLkeWGnH6SS5DWA
  trnhn4k+iEI2iib+Cx6jHwilFgTU1MUrMHJ4F840hZJBaXhbRRnCqiuZRQyohTIV6oqGsCrY
  seqaOG2ufaB70bsRpLKWInNgS4Cx27rVLboh6JoXn31ic1CxwqZbcLLGMgahskZSGUK0y/qC
  bF0AbxBdc4roW2jHzW2bsSQjEhxvZqf6du4d7a7LZsuBpVCSA5+SsHFmHRO7L5QGorwLCpNN
  lw1hgkEsdLiT6szwo9ctbHR2CVfQbzJBP5veeEM/ANNvDy4YD7kZbuZ11J+U3MJ9rAZJL6Qw
  M51Q20xtMDUE66D3FdtpcB300XJQB6o7Y3vb7N5+Mquw009H6LIISleI2nQoHlp04g3+bkCM
  xd+5fcDFMmH+JrXABivAgY2HTLYH4K5tQcowt6Z52YE9nubIHYCrJu10v0qzRIik7PkooEMI
  ctijdycTubYoGKrgHuSjOPMiiJU46afjGTrOPZtuOJF+iOQ9FotnQOHKx3wXOTdOS4q+dBru
  fj2QjceRLKlNx70C2u3+DT0Gq8r+IrtH+R5IArUBlaAnUwyYlQM0ghJSYWMdEMKOA+X3wPn4
  6HWMCxo4VgCT6WFjOooYQfY4NkLskeZlLUPdeJYFmxUu7w0OTA0z6AKUACEOSDGs4MMIdT4S
  3IaDcLvNaET4otH8MAUlVNRsOPCY8vEOHrFzwGLqVTDWXfD+GvR5YQUhGCTcmSp+GIze5Dz4
  uBypj4kCkwj1VCeBAAqQIHFQ4eOyJqUydhF9YENQA6rGqBwqioLHgxgdDxcYXIEuiFMix/LY
  eIA0Ua9Sq3FhspZYwMQgxn73kWKr3Me+23RydJ9Z1hxFVUESDApyYSzbqOz4NF6AsHoXoJgZ
  lfRKL1queE+TmNTCc0bCvdCPy2uhW1LZSIHBbzyIMtoZuTciQ0ZgEbYrJD0eB0sL3OeRdUMi
  3DxQphl6knUkUsk7kgmuJo0mL7eh5vZWV7oKOdNNxZNlLne4xYBJcixs4QorK6TN6elyl7pb
  AOegRp8bVsxWAZ3RPhM1PnlOcp3+ADXROzVZsGiOq5iLKJQ9YzcLg61Km237QJl6p5uCehKh
  c1JouWKnGf35r4jSmpgv+WZLC/hEF2TE5SzfaczRLPRC/UzcP6UGPjkMRFdZQiUhusGl8v2U
  UJYYqLJqCAL8XdSHRwHn/+Y1Gp5yyohHRGG7QJCLeB0VbkuF5mBEY6K8fIYubY1pr9wowJ8R
  0Tn0VBYe8gg1iLbxfSZ8qh4Z488TzpVG3MnMLg5UQwDlSrALosb4jEbV5rWsphRAJDvG5VPY
  pROsMQPaNsWI2CBmwbEUYOuIHifZisa1snKzamA0Zc+uJc4Mee3sB5L4PMFSkgTJzKWhVLkL
  3mrxlWn+mEQcVXVPJSa3uQhbW7/yN1HhIFJbFuqOZ1laGd2gwKKFJGK8NKuRC4gmmg+0rbvU
  hjqQcEs8IsmgygSpECBJyLJ87W2wGHmcwvIXozsbY3OyuKCTLZNthe3pMQ0pknSCRL5tm8NX
  RoPZB+Dvpen8rmlHAN62wehiWp1cH0BbyM4Vx4wqEJJtf5JPriSvb/YthkhnyqBeBK66wpxe
  +3aLF3DgMLcyucZvMwtYRY34aYMUryWL0yPZzkcBLTIiWVXUrfBKDq1Y1WxYbQosq4LqESud
  bSlPvKgUUwINYd4JkJMk0/wqq4clSCzEYNsouhBYx+Q6MBXdTCgwt9KvGNj+8lzOfDsYBkbK
  VSXvploHXC1HEQP1Xd57uhpkM7E3zjfccZei2r2ALHaqMp4xdBE2XExDdkF0LlsZ15wo4Q2V
  k0/zsyCJWbsMA/Eq8V2AfL4cam5qFsfC9HAkHY2cDMTQquokcd106Kp9phrTnI4Smufca1+/
  ryMdpQz98nNqxgIaXh4s8LfxGGvjHrPC8IKkq676jf4hesrjEslXgB1sRjea2LGK5cIu7Lc0
  c/aOxaHnkDdkLx2t+KGvtmZ48LeQag6u26/8L2JYTdmEu4/PEdPp0gyr5ODcobx0cjKGR/CM
  Sdp10S2TuGLwbXFl/CnExcNCpUscYJZp3K1kIiD+A05J7ZZTQD5FzeYKzN1nn1cg0uEmaK2L
  fEyjbXtYZsp5+KDIKtZW+7AC3yePWKDZ+hEb3VDW4KQzoU5PMvs6boZzZ3BDlnIbHXTZZux9
  UzzqonO8zjcPN61/TmNxz3p/qFARR3Dj5Iwu6+027FGaZqF2saYQ2vg2pG21+hN08q6CYCdy
  bpPMwQTOt+KIj4CcUT3uW1I844yBbEDsnmim17ioyCDU1kX7nmZ4PuWVhryLVJ67IDoemg0W
  tMbEHi9ZjCVCVQbW7NG+3xlUJ6ehzx3ErTb0vCsr+lRkZdA5ij02FknbLS5p3OuJAtny/uge
  zn2Wb9eCrnvdxfMLf+n+Xa7BKzHWa7q44t091fezi8v1CKdy/Ycy2Mdlerd/gxNzhXcUMsFI
  RTJdxScLftEjkkZvAqUmFnhIuHd4Alh3z9N+iHRlrkRG9SFqAPgTC9FqU0dyBqVYgMeCJ/hH
  elZJ1kc9GbhHNWhkCGht8zJ+4EcPy8V/deODQiN/KwiDJpJ5w4dWQSZxwaWDN1hHb+Q9Bag1
  eZJ81bJPVUh112MLDDd+R9I0FLd8E0J01zeDGqAQUygeShhRcJOBxXd6NdYNplQdchgy+BKF
  C/iCeMeEHGgz7/VGDmdtBwiHxeOFt5NtWwgPD2KESagPeVN+A6htg8hjGAcxSANvLROFMOf+
  IhYoJD+GQ0F4V6JIgcekAntYiSRjWt31X534RACUg4lyhDzmiHJwH7ywiJckKp9hVTIziQTI
  h2gki5iIhqZCEEdRLf2ADBoGjDiUifYkWJ/mbTtIg0B2FjcmiTDThqmlWajRMZOCjC8SEDfY
  Yas1bGYnacKXgPLxMKzTMKYkYPNDi9GYdEtyfpszGSP2dL/Ygn8mjESWNyT0KiGTCpC3UJeC
  XPMnAa8YdbTHiuBocyd1PywHkBPAGhB4T5fyYpgWEzdBT6eCGpqmQgJJOfAYj8RYjAvZbn9y
  ZPkQhcpjj91FRAfnHOoRjFeXgNOGk6a4KNDYh9clkUEEKj9JVtH+NBK/52hE9Eli01cwxnkP
  Z3lNmZG49GzMdIkyWGrf4oHKZIyw4WUthVv+Fm8d4IxOt4iGN5A/NQ7o2JMc81T9wo07F3nr
  9mt/KD57aFKJWJYVmZM9EXM7oWzjoYvW1RpAFyYUSH3whpUbR28OlXRzh35U1iTBZYtPdksy
  OR1byHNGwmqjWFt36DWfaW10CYhaFhMnp3Dtsn3sd0t9WY0XQColhTZQSTKW6SK5CJvm5Y+3
  tZLwFJoM5ZfQYoZKZym8aUIV9oSZ2YBbyZgyKHEK1pUcVpEd6GiQqJu6dlnokz/W85qMpBZr
  +EYj+ZfhOYVoyZU9F0nLuUXrAUykxzT+36eCWjkByrmcXsl3EBec/1WfhQhh+IaOWrYt7Qkz
  hraJ4vOa90kt+Vl0+xmdxVhdDiqdWLWeTahwp4aImgMTREhCnTJZFGqVGRChybKIHmqP4xmi
  fQmg+FYeqThcDZlv3YigjCSiOjidzrk0/PWaKvp1XMYq8jJW/IOgvolRKwocLJejKbp+LOdF
  3GlhQLqbMgqU1mmkRbpRLLejxPYbiVgWGOolkCakbNSigmelPEqmxIaljqaluImUWnObxaZv
  UXpY+CaX1rWk5nCneJqnerqnfNqnfvqngBqogjqohFqohnqoiJqoirqojNqojvqokBqpkjqp
  lFqplnqpmHoDBAkAADs=
}

set pieceImageData(Alpha,65) {
  R0lGODlhDANBAMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAAAMA0EA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWq/YrHbL7Xq/4PAvICgHxOi0ekM2r99wGEAQHJTvg7g6wNen5nV3ZXl+YnxnhSZ2
  AGNmfG6JX3eRJYuNgpCUXJOaIXYChD1mBKRtnZtlpyCfoTyYgqpanLEbc3eMogIBpASmtAy2
  uhVtArg2AId3h8a/FMHFPq+zzcGIE8TMNMiPjgHZzRC2rD52A7zl4AfEgxOfyjfrmNbp4Zit
  Ou6C97Hx+w35wmzEEzSvxzZvRbD54lHtloyBmQ5IS7UAYsATtna1+bb+IGOvMhxfWLzo42DI
  HhBLSjuZwuK8iXQqTixI4pnDeq9YshhJMwfEnjoAuANAip1BTDpRuGQAk8FSpbp4kQGqLmop
  kjIOKuOTdMbPIEIn+rths6uJpwqaypRGFcRXCG+1JetmFgZAfZdA8foYEUe8ti27BbDzEtSh
  T04NzwXsYapUrIl3XWXsYho5mGPhwXy346/XmUwnJmYb+BXVuDcs82BFlFRYUAwFtd7rbkDd
  P6wGG4XRZu/U0JL5Rt7bl0RvUoQjkDFXFDIN1Tvu2ou9Wd9tFK+nx0ArUfRa06XlKSedA7pP
  kHt52bqewl3w9ASyU/6j7DFI3mV8kxx1Nab+Ase8FEeCNBMQqIN5flV3Xw7SSePeMdWx1xho
  C6j1H4UoGBiBhjggmJpe8CFHEYSDzBaia3hIuEqJ6b2W2Vn52WeajPIQ59wI2d34HSgqnuCh
  DQr+2IJNCg7Qhm0zNKhdCyMFGeR8HOQI2DpIHjgidSa2uKA29Z0Y4lAxsOLlOEPG2N+MZ9YY
  IDQqHAmbBOUIaMJBEhDT1jJyOMkml3riwVePHCj5yos19WnollDh0Q5BQx5SJ2dwcYWfAF42
  B+UJn8xWTqUiIqpCMFnCB2oLZsZ3yKneqIfqMjbuxKOnwICUHG6DfgPQPrrZA+gFTYpXw127
  EGlPfLBid+ieKhz+26erskVg06Ua5FiiA7c6kKt1TJYaopwszLHIOV1WCuYKoHKKYrElaGvu
  ugRcmWFU6EDwiXAYCVrlAbmReYC0uq7QK6MzCDtbr+as162yyL6LsIIsDBrBXdjZa0y+u/GL
  bZvqpsetCu6BK5u54xqLHrujJmsVu+sulMIoGz3gkco44vHetYyI2WkewRjpW8gn/AupXStp
  HCGxCZcgbJ+7UrBwkA3n5DKHMQ8yM5g2EzBOzlPfJHKltmSVoqqohrqtkB2UizLRRQ848tle
  BuPBNhLYsqY/8zbHkqQXDHVizoOkx8ote2s9p55eO7iqoDF2LfLSSUuwNMNl9+Gsgw/+SHc3
  QnkDHiLfIEr299paoisCYWO++ULHbAe+W7qg+8aH6oR+YGfqX/6cQT5AHfdnAx5ZChdeFYgD
  sp/w1cbpa43LG2TslTw+rNWms77ZXJhh7Dx5G+D++0q8Y9gAlRYI36LfxG8Kfd97ZYk8fcVs
  LvrgLNJ+Is+jt24jpbB/ypr88bHWwSe54oj5mnOPuomoJ+I4kjPQN7z2iYqBnMrU4jaTvOBd
  7xXnquACEIe/dlWHeWXjoKHupYG7CFAs//BeRxw2AfGliTmdIkq5XHg+/sSwPfHrhb5YgDr+
  uW91IaAhfO4grvd5YiurSuKqFFU2My3nATaklwJ615/Kncz+iAZcVxbndjYgkgBxIDziBdXk
  xQ8cDRNritAK3KPENlIvjE9jYQM2EysLMcByi+rg0TQVI/MxcI9zM5kD0eaCnInNhx5L2/8y
  dj9OlbEEPlMQtKxlJsXN8WS+u1AHq+iAtW1sPWcTYqZCaUQPVKeQY3RQfzQowjRuBo4c0F3q
  NjYM71wSQ02CIvbi2JozOlB4cyDKKNEWNKJpsCPDVM+0OBYuH6bqgbTMQBTvh8EQtcx6bsxm
  NFvoRHdVZZOc7M577NbJKypSAQNE2UaUiUl2bTNaYDwmtVJ5LSISbZKO24yX1Gg9/r1zclD7
  poX4GRqnPQxEHtTnKntxz3NhZk3++OQmjzYnwUTZhqGIfF3tvNkBWQpNGvvUkQg82kWRWkBq
  baAJST14hpUWJ6W5KlA73YnQLbILZkHUkzx3xI2F9dSeCR1Br0IKk4jyynlGrRB4vledljaV
  kq+QqWT+BUMGGpCDU+Xo2yC40Z1+s1OInOmZRpDOj1azdkmtJThLmtZ9ce+OCNUPHaYponsI
  TKaH5JTu6JoyrTYRaYky09IEqjmyWQBxRFUhjB7X1rvC9ZRF6p5BN+RAJ7Vmr1bxGXNw6oGq
  FTF6ahNRXvs6zvu1NS1i/ai5OGucuLJtVmZcDsBQW9rmADU9KqOSLpLCyHWxrLfm8mv2jgXL
  C5TPp4T+5Q8ARaAnV57Seks7LTJAMdvuOG80g9it0jrYXEvxxbkK9SBZ+XoiBZoAuGwDru06
  QN5G6lW4IQAQ7X4TtRTRtrzNlOsUAVIX9FaqN+0Nrn9WpKziViCVNUNTSpnrJPDasWdILdTF
  LrhfpBiXu07KKqX4E0kMB1GErzymrGAoP9KVDrSLJPGJbrtiFDevtjd18YSQstI0VsoMRLqU
  /UgGkh2vy5JvAzFmRHxBQrClPwa2YpAcTDlmPi7J46HxBb2x1JNW1knMWY9guwvkKF1Qg8+j
  nU1xa1gK1Ni9NzYpCAL83gEz2DZntWZqbXsLxGCAzWlm8dlYiwFfGgrMFE7+LofLfI0G03l6
  Kwi02lIJuQtrOEitMR4xmxpORbSTvonOrmvTxxXXac5LxBABFVu8VjKfk8Ew7qubc9o+Se9z
  tMQoWClTqOKz9TB1sLW0Yn2kaYThYiZjZbWCmDxZhT0OfoyGyXXM52eCnMlzOj3gyi4NX1S7
  FaHTZQt/V0voPCKSgE1LNbdXfcTNslncfIF2cc887jkf758aeG6mz9BsseCiQY8e7xihfOHr
  6jrZcjxqB4Vs40O/8lynhUDGqh0CNiGvhtPTaLujVuvXzvoDeAY1w6M1MqGM9scXNeYGuIoy
  s+F64xyQd7JwYbEQb7CowRb29bxa0GMjG+DNirf+Veo9MvR1+LIo14BL1SxUwPkPmZ+OcdDN
  7OMfw5u9g6Td0jNX8W/XNZZNd6R/VZfwB/Sq60x3EDPotMLpxTy+UybV9VAg5IUleZ18keRk
  OiX3e+Kw4rl2sl4qmsKoK53cnQXdQVZlInFUptQlTdTH5ZcRqFddwIhvoN53jamJJKWewI57
  80zTaddQD3hN8/dicf700MAQxFdpDZY9ZrLa8vnf6/3m4//b7QU6Qk89pflowkr0kbubf3bG
  +u9JPd+Lp7zRPKwyMDBf91BL2DDqpF9gGWtsnM84OK0sCrH1YR+agxKauuddPuIY+TYDfqsL
  C/9aeA/23VtdaGlNJsr+TExKGYsa0i/IEfkN5Upkp9f4spN2EEZ68bc2HKR6aSIW7MRvFDBm
  4CYHlsBUGSdnU2dBjpBX2yA4/rJ1adZ+Zbd4bHNNppR1coZu0HR+9RNZEFg0HXZWsAA/tsF8
  MrM+LtCCT4URpHdqVvZLdWduQ0NIradxKNgSEkhy6lR7VmaC9jEp7DdtSrhnFegAQqRX5fCE
  thV+hPMQRags/cdrxzIpEVZ9YpFEIiRs7MR5zyQZWlErDqV77MZSN5APICiEQwh1Vvh6i9WE
  iWJNMohSY5NwU+h5ImQkh8R303eDv/I4XXheygKG1HeIUgOCyCAdgAhBhJFqmLgctAGAHPf+
  eyIoA/uTUUhoe/+nfjJhhfjlgesniCP0TB5UQTTUh8sTHDTIPjqViIpIZzioGNnEB4a4gdej
  ipqEY/yTbZwYN0YIarZ2jLezdY+UfJsmPy8ISRwIUaczfO8ljKhlJGxxKr1wKtoWhQClC8DS
  eS3yeVKjgZXXJwy4SM6ziK1Vfn94jcFoMlWHKrrgjRtlNMl4GIbjiunzi7omiaXnFskodc84
  gqgoRU1jJL3oRsHnhQRxhynBj/awkKVgQtBoKNoYZe+INu2oS7O3j4eHkaYWhFLRYRLHkPfX
  cTZIjGgTfoE4P8z4VyP5f73XZ4xTJmNEc+4xh5tjJ7uoM3q1WgL+eXN/licX9GxIeSimOGrF
  p4OBpxFtd5EfoYqcMz2SpI7UGI2f1ZF9B5S+VZB5hDAhKVG8+JBkyJVoZ4LBxG3CyGYmh191
  aJDI5QJVWR1MuXlmeTAopZZrWZNhVyQQh2gSeXB0p3LreJMbdZYSZZJ5Bpb4ohezEy/06HeM
  x0RdiYGu9kNx6ZX9oysntkbO45jIOEZ7OXlNdpmMZpqnqCQhN2n2kHcjYAZD1iJKgofoB5oN
  JJUBKI8ImZPCRxSz03g8yZtixpYd9TWmpiYBSV1Y6Wwn+DHwsQ5exXO3CF0zF3dg6TOSOUU5
  KJi/EyzAQoHWAYfwk4BGSInkhGyQKSP+ujeBJSWcGOCL6CNpuQGW+fV+zld0gmCeRugrXbmf
  z0aXsQcC2AlY0zZGwXKgbiGLgyI5Cxqe1BgcR5Y+IDUZA0gj0hl3zkmWp4mZtFMyvCaWNGV/
  hbZ6lmImQpZW8pk6uilNzCkq5XBI2eGaw+huyABr1dWSyUZzyWZbb4cwL+mgO0ihakOe9haa
  YvcROFoVPjiGggJtHniQ0iiOKfp+5DN12EmU59MdMAShFnZ8Wtpi0mOCaDqKefSe1ZmQx2d9
  A5hK5uAOnYgw1qUsPRKeaqpw+biVTRWjJdSnwfhE8Wii5mKcXVmmMqJzuzV/9zGTQTmKb/ht
  IKpLZdplqKb+qIAqcDinikUKaf3Zb3h6bXbKBnpaqeWUbIfJaM3DmCf3pPJVpkKZOSK6N/oS
  ik6HolGmqENEn48CnP9XpbVajFiqZDgHqzj3GpeSl6DHrJpZn6fqq7tapqiqS4b6bnV5Z9gI
  o8U6nrwqpNr6hDfad42qaqYKrD5UrS/HpnLmmnLDq+KZqqQXp8lmXkd6KAW0MOeqp11Hf/yp
  q534rfF6S/M6ejnIqU7pdUQqdNHadQcbj526aAU7oA+rVsfCEZ/qqxl7KA47sSOVg2Cnp/RK
  gBZQpFNyLJPkrMT1sSTrnxFbm0hKsS07mH1yshwbqKd6lhv7JBALcGAHmEC7KhhxEbREiyfB
  E7R1MXgPyR5F27QSKjtOC5g1EbVLaxxU+5BDe7VVe7RAm7RIK3RaK7XxFbZuNLVkm0SmSA9q
  u7Zs27Zu+7ZwG7dyO7d0W7d2e7d4m7d6u7d827d++7eAG7iCO7iEW7iGe7iIm7iKu7hQkAAA
  Ow==
}

set pieceImageData(Alpha,70) {
  R0lGODlhSANGAMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAABIA0YA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWq/YrHbL7Xq/4LB4TC4rAuiAec1uq9Jqt3zuiw8BgrweQO/7aXZCeHp5fH+HiCkD
  AkSDhHmJbQF5gZEli42PepZrkwKVnCGeoD2jBASjoWSpqiOsQZ6apK1br7QejgNDeqenm7cR
  sxLCObbAG7m7mr/HDsQQzzfGzRiPhkC8vczUAIt5uhXeAgPXOLHL0dQM1kLL283deuAU4uQ7
  57LqFt6e8z+eAE4NShfJkTx6mu65o6Qv3CN/ddwRTGTwG8JHChdOtAH+AE65IrlQFWqnCWIM
  fIQCLWS0AKWejRwWkSMwcFghmjJ1uHwJ0mOSintIPrSx89O6hQyKwtwgLqXNkjpXLo0BdGSR
  etrGDWlq8qTGo+6Sfl1BqdekrgsWBTALKUdHNN4GpPnoo6oAukCaDhXCVdpYBSvFSmThDq0C
  ve9qvA0Qdy7flYaj7ulFQNzUFhXt1RgFh+GCb3C8CYYzzUTZU54knKbZlkfpH4gJRS62kueP
  zHhfKAW7TDA6wmEjIC3leetLyrFmK8ZKmTWh3Jtfxros6mZWlZ+Qtz7giXJNsuN6qbU5QPx2
  ydRhxN5bd7076IAGx5cPePiZvynoP9gZ0ej+nW/NmXdXf9YFKNJBbsUVEE7fwKfCagQUdwCE
  qbWUh3YDrpBcUBFUlR4Kr/FglyYOzlBUSQjiEM9CmsmAX2AWvseCS/B52N+HLWRjYIRa7bBi
  djtiaJVixzUnToknQKiRkJpg6IJe0eCjnIkSYlNbigle+QgaHBKpJZKmaSkmjCxASYGU/1RZ
  F4QBAqRDLgsG6d2RNRy5YzIxKCkRk484+aRsF1mUpn9BjLgMmCwYClllQ8rg3jJTljDmpL39
  iSUEfQ2Ko4YXysnjpiHIg9xanjIqaAxxebpipCNQSNpc3r2ahp85fiKaBIPIdV4KHW2J14/j
  /IpPAIiKcGI+OUj+dGx4uc7waGF5UirtrvlhhGtCmA37K1fC+uqVAKWqKcOqlKVaaj0whFSq
  qS0CR+q68LLFUq3Eihtjr/PyClk5dvHrXrucjgkqLrEtqCizOb1wMIvFfjDttC8gCw22Khja
  br8MAPtQwx5056m9uokqkDXroouZyPDSSW+8LB/oAkCmqsYsyCFwlUZTcairss1wAdrCsk4R
  9V65V5KacKIPP+dC0pS+XCkEvyki281B68whz4z5PGOnctK8wo/ljdxkyV3qC2DL4iwdXssp
  U3uCx985Y53XHRj0rpC6eBPngXkX2VxyHGsAtG0zaDyZd186F3gGC2upNtNiOi1xAy7+DdyA
  3QYmd4DeGPbNJmryLI7BQHK6SeXhbAdokOj7nM12XLUGm3qQP36QNcANeCMg1GvzGFk8co2O
  soEah020bHsHqLLZYrLecW1yUX1l2IM4HyjkrHoA+Zi2ywTferxD1aFMG6l7p15hD5D+xnIu
  L7XxBp6aLnOzZ37pJa6rjkbyx2ffgWX1+9v9MoA+4QDJOaCAGejylTv2UMBv7Rve8XpXusSY
  4FkDTBQGH2a0HqEAg9EDYQs2mDT/OaCA4ZvcfYLzABRWAIKLMZLIzEU/1KBBgG4jQa52pDcZ
  nIN/AcxKo3SYvzZ97njWq0BygmgqQnGgH1lz4lHudqujwK/+QnIbRxTTwbm2UbB/8erH15rn
  w+0tSnEoWBipgJbEM5kxah2oXAop9hn7UA6OEugig1ICRh6tzVwLHJ4YA/YJg8WijRcAGxNp
  l0EQoM5+BySeBU9gOKbhTgOrOZpg9hY37hQIgS3sHRY7VMRzfVJs8CNbhlKwLERaoJJvPJz8
  SCCmNUYOM+IIYOgcCan9SMU3eEwLC0nZO7vAj3PVc84xh3aKHq4AkOz6luwWGST3GeuIGLob
  JC03ATNW0WEHHGWMlLedUoprNeIMJbha5ogA6YhlszzBo0xou1jWxpardEUtA3nPiH0RbTnM
  gB1jZMd+OmCg6uyFxRSaDfX5gmv+TTxUVuiZSH4YMVi4hGAQO3InwrUKos1ZSZDiWa3oyeqk
  pPmm9sJJLd015zsKpAxJ0RnQA5CObaaQlzbhZTpKgtCV47OnPFCCT25OqDZdg4w/d8qydHIA
  oZsz6LG6AtWMrWZ6DM0OAAKS04i6Q15AXYAjgOicfIKoQTSh5g13RIgS3DR+CwmSU1GATZ7W
  9AKcKc0Rb1VKl2UxDV6LKUDX+dBUxquTRBxTWLMIGhLesypCdOuVkkom4LzxmiqsT200W1mC
  Bi0Y1gGa8VxaWAHdU2xGtekjVVe2t3UqmUGUi1xbW8+7jDSuHZVitZgaxrteADEmeSviPtFV
  WjEAuG7+JCzbbirYltENGdJaLDBNmzTVjs2PrlwWZfHzoFiOALmX0xKXrqROrYGWsIpF7QIN
  NqbrXBCtquLHGzrVV3ZGEoe6fZ5yZ3tftuY3Sf1tqm9/e6hN+pcQuQ1EVaZE2tSdrb7wei4B
  OTjfszVOqS4RouUGByQOp/YB9iRBwYC5PQX38iJCxGoz1+Y6x7pXxBA2EOzOSljeApSsD/Vg
  dfa7o+Le9sMHDXBvGSgCF3oMknUtjgsdwmMB30W4bCOpBxx7YtcO1Yx8yPACKTqxfR5IvI8z
  ozzZw2EwH8aBKM5x4g601dNcGL06BoFlUHrSRtYWJ4scxMcm+USQ9riuWQX+slWFbFeziqAk
  fHinOxUdUkhUJQNHnh0lIu3g/8Zke1yeQJnFBI6CBZIEHuanQccYy8Ui+qghfjQGjjwp402a
  a1R+MQjefKWw9iPG64LyRYmsXxvrtFQhArWfo2xpXFDCMoD2MQ6T01MLUPp148B1uIotvFKb
  YNNf4iwfvyxsL3v4Z/ZMrYeE+okCr5prlGKLRcsqJlnb7qTkS6kWT3rtktSvH8Dm81P/eVFf
  r5jXVmaihDWgu4r4e9h8qleclXjKlp3D36oauKbhDRmU1jvcK5TFp2npbe6BO5YfprI9TajA
  MpNqdaI+LSjnu9NJCPqOxy7lYtSyxijye9eW0+P+nfYX8YXTWODUrhuAfoRjG05bds4UKKFz
  /ZwAzrXb+iGkzR8GjhHhM+gYIGEIOS25N36Y1tbG5AHTreYUa0le7rpoyDRjLnwV5iGFxnoi
  pe1F6WYR6EDu4rMlDaRmMQXh8UKwLn0uqc06je1UN4Shrm45sJPx416XGrmhd2nCUnlB637z
  u6TM8ZYPuMhinczgeD5kuRPYthvl/MXxbrZ3dYOaM/n34nQNT7rz1+517OwzQy+tFilqQdPB
  n1AzDWmMp3HytZn9VQW2wKuffb0B8+/LyxuQxoAY0Pw1/QVWa9/P75j1kqdmUjPNaCdDnJGE
  hzp3R1gYgmBQXtq3ALn+YYDtX0oe+eLbN2HfnBWzJ3+iZYJwElczouJ3J2R7OGcCYzUqdDZc
  htZdTeZc6RcCtCd+zaYBOzQ7yhYvSXd/6/dM0HImJAR/RrVgpBcrU0dbEPhGxDcBjpc0rJOB
  XzZZDyU24iUguFcfvPV0rPQ04QV4+TaAziYdkzJe8cdL5+cpPBgqQAh0AJd12MdfSXhRLfhC
  jiNNbfUALwhnQshw8AUvPzJ9DTB/K4h/+tY6qOdVS4IKzndGfLM1QnaBGeVRvLd0SkiH34Vp
  LmKH67KEINBc4odY0NVwfRiFkuR9cfR48/NZg/YwKRZyCCggVah0IVZhZtiFP5iGG1RzKQf+
  KVn1hDAWgYVFAyuihYKngXiYh6Dhaz2jervFRH6IhOInQ6BYbaJYQXzIWkP0QVxXOCqlAFuI
  bqlYZGXRiiVxQ4MULZVYhvgnAu1EXXuCCsCnYmWVg/VhWMfDER8xdUUXhEc4d7mIdjNgiHdY
  iwR4i4M3iVF1c7g4hZIYZuSlGNsocsvwiMwDMeMoVAEoMIA1JuoITe5Beq5nhPb2jnBYdIJY
  J8gTRMH3cyyTkGnHkIi4AYAoEP2YEif4UpiIKTH2FvOUkeQ0gepXNDtAj1+lZkBWf8MIHsvY
  g+3XjTSRgowoZ/kzZz0mV0tkkKSGjbSoA5YBk4W4i/qEjnAlhj/+6I5/Zo6zBiEdQWVysTex
  OGFf5HbTQiy0qI4uKC0+yYIpZnd5RWeksZFCp4+klhJASTwo8V4UBHHnR0WuqIowaYAKSYh8
  N5FQSJSLhpVDGI5yApGdF5MsEhoawVV2eRHlQZVbglKDGU2W0mo4YJLRaBAq0GARJpJWZnwQ
  aJW04xF9GQtjxo5mcTOkYUqWiVk8GZLW+BRpuEhZSFd8KY7014SB941SiQ5suRufeYzuaIxo
  ZomUkprn5XX81F2nWU2F2WtgmZzjNX2UaUNFw1RimUdF9BZFqJk6WZa194AnQ5f105oXZ1LK
  2YpG6VmRR0igoYFmwotDBYuZsp0PA5z+DhCM6DCcNAae4SlfsYmX6xKdXqhQKmmd/BkBR4KY
  Vdlmb+mM3NeXKmhZZ5mdSql/kJMuyJeDUKRW+KmeuaY+QHmgCAo58MkbwjmD1bI9Eiqblamd
  Q0mgHOSX56gl5ABLVVaWDfqJH+pJoNmdHCo45bmP8xaedGYZFRaUG0qbaOiNfKhJFfNGNaqS
  SPVlekmeSWOUOOOjdDaTJRBqaviBqlg0fMIi7MedPVeaH6WfxCamTBg2gveLc7iaTAQn9Ylj
  wBJ7xHOhAMaOC4ZjacOj3gQDyBdIT2pA8UgVkNlq0gUQ/8JJhxqgBqRwnqg67oEvX2Mn9eOm
  SPOaDmopCyL+eHLpnnIaiNaEWX21E0CEM2XIW8CVlGc4a4PKMO4pVPSpp0pVope4oHfGbhoF
  jXNDpFbYhtxnTAuUpF84qZ+6erMoU6laN9ITNg/3b/YpKxx3o7F1rDoqG4fYdCG5kktJMnMa
  rA5Iq9k6oXBIbmtxDrwiPZyWnM86q9JafPvXqBNkrZ/SergKX7BkPNVzfOQYdzkYiRJ4nDDq
  rjIhE/PKquCElKkTleyabDH2cC7JrW3ib8DydbO6r8iHpusan6uqPpZkPepqLVfaKSfCq9um
  qL60Tv9KXNCTqboqnWyaZzm6bwbLgRfbiC76LgARUzK5Es6Tr/d2nCzbqXcSl3T++rHTRE0r
  AomX+KcgKlTVN7N1CINM6q1W2LE+K6CFAEI/RbJZdLKUQg5ay3vFGUAdGCYxG3dL4Qh0BkR+
  p3P+qZhCuZczOjssum9hmzpqSox1OztI2qETy4xM25C2WEJn5qGJSLVfi2rkdrctin9/unfi
  d7i8V6z4tRQJiosZgrbd96AlK7nSJ1mWGpSvyLkIe27qCmTyiVXYOkfvCYxi1meGu7LxybOF
  FrqSO7qnx7nuRrS4a4Oau0Iz+kM0O6Nze16722jc5LjgR7a4a7toOKspOau6gohR2yinexnT
  W5W6y7mQe1RlG4R1SqXg65k0Fr7gqzSnR6XIE7sYGZ7PLzsa5Bu+X7uK71ulZlptzTq/8oai
  HWC4DduxxZKxdCGfFHW90sJN8ou/stK+r3TACJy/YcW/NEa1G8E0xBC1lWe4WHm6ijiUzhsm
  Hdu/pVsNUaqa+Ehwr1u/FKDB2Ua0HVx4HxzBVCvCXUsQORuolPi6oQi9F9fC6TqrIKyuMtxq
  NGySEArBxKiuT0rAZtYQTNzETvzEUBzFUjzFVFzFVnzFWJzFWrzFXNzFXvzFYBzGYjzGZFzG
  ZnzGaJzGarzGbNzGbvzGcBzHcjzHT5AAADs=
}

set pieceImageData(Alpha,80) {
  R0lGODlhwANQAPcAAAQCBISGhExKTMzKzPwCBCwqLKyqrGxqbOzq7BwWHJyanFxaXNza3Dw6
  PLy6vHR2dPz+/ADmpQASegAAAAAMAgIAAAAAAAAAAABAFgPmAAASAAAAAIhm0OXjABLRAAB3
  AHlF2ggEA4L/AHz/AADgUwADAQG8AAABAFYMFgAAAAAAAAAAAJAA0OQAABIAAAAAADGg2pe5
  A5NPAHwAALABU+UAARIAAAAAABigFu65AJFPAHwAAHCyvQU7AJJFAHwAAP9F2v8EA///AP//
  AG3gUwUDAZK8AHwBABUMMQoAAIIAAHwAAAAAxwAAABcAAAAAAFCglAO55wBPEgAAADjwJ1zm
  xx0S0QAAd7CgMGk4pRdFegAAAAC0MQDmAAASxwAAAH6gjAA4AwBFH8AAAQCeMADGjAAARQAA
  AP+cAP/mAP8SAP8AAP8Jd/+HDP/RAf93sAAAWAAD6wABEgABAAABIQAENwAARQAAAABh0QAE
  tBcA0QAAd8A5g+U2RhIwSAAgAGJ4hgkgA4I4/XwwAbAgAGl4ABcgAAAyANs0BwUgAIJCAHxQ
  AABQAL8AAE8AAAAAALAAAGkAAAEAFwAAAGqgAAC5AABPAAAAAPy8AOQNABIAAAAAAADYAADm
  AAASAAAAAPiDAPcqABKCAAB8ABgAKO4AAJEAuXwAAHAAAAUAAJIAAHwAAP8AAP8AAP8AAP8A
  AG0pHgW3AJKTAHx8AEosHvYAAIC5AHwAAAC+UAA+7ReCEgB8AAD//wD//wD//wD//7AAAGkA
  ABcAAAAAAAC8BAHm6AASEgAAAAC+HgA+qwCCSwB8AFf45Pb354ASEnwAAOgYe+buwBKRTgB8
  ALAAGGm36BeTEgB8AEH/vjv/Pjz/gnf/fBssorMA6NG5EncAAPC+4OU+/xKC/wB8fywAhADo
  6AASEgAAAADdLAA/AACCuQB8AOMWvpQ/PtGCgnd8fEkBLGYAANMAuXcAAOAxvAMADbwAAAEA
  AAxYAADnAAASAAAAAERczmHDcHPTRyB3ACH5BAEAAAQALAAAAADAA1AABwj/AAkIHEiwoMGD
  CBMqXMiwocOHECNKnEixosWLGDNq3Mixo8ePIEOKHEmypMmTKFOqXMmypcuXMGPKnEmzps2b
  OHPq3Mmzp8+fQIMKHUq0qNGjSJMqXcq0qdOnUKNKnUq1qtWrWLNq3cq1q9evYMOKHUu2rNmz
  aNOqXcs2KYK3CNrKnUtXJNy4dfPq3QtxwIG/Bxzg5Uu4MF8GDw4EWGDAsOPHJgcEMPoAgOXL
  CwZD3swZI4PJRQNcvixAc+fTqBs6SNCAKILKoy8fSI02QIIEs2l7HFCgQFHYsQEs0G02QIEE
  w4lvZLAAQILGQhk0sHxgwIAHCQAUcKB87AHLArpr/2RuGXpQBgKCW07AXTxY2K3dX/xOfSh2
  7QwgQCAvXL7BARQxAGBQovXnX0UFGkhgduoBEN6BA7UnEQIDAmWbgxBOpACDADRgWk/pdagf
  AhDQlwCEAyhwQAMBmKdaAAIsEIBgOjGgQAABPDCddorhWGGGCm14WQEMCEVfgwCgqICOLULk
  AIwCzLiTjTgGsGMBPUoJpEIDZHebZaABdV8B+unXnILuKbDjZS5yWcBoBYR5kwEcNvjAlgox
  8OZocvp0YYPxyWdAiJcp4NAAazp3Z050ImnZokW9RYCkSyFqmYyWFfDhTgOk95x1DkzXgIQw
  ISCgdaiiyoBmFKbqapEEnf/qKqqbirSacw00sCepCOmJawMMwjrnng0m0KdQdzUlXXAJCOsT
  eg1uNxMDss66KkGtzopqQdXOWmtIXarX5kEIJHrZjzUZYG5wx/5k4wGJHaCAs5EqcBl7EJxp
  aFBCxmYsTY0i6eKZ0RLkALHqtStSc8aSeKsA6BaEAH0K6GfAm5ridLBzxSrcEwMGJBaAYhEP
  hYABSHrMaJ3rqazSxikTdKR6Jw5kKZKQmjTzaA2UTBBw/tI7U6iO/osswbF5aBRv9zqg3wCW
  NSA0TzCvt+9MFyNZwNUEIB2cbxGue6/LHr0pgAFlisaeQuUKR+J+Z04d02cP1H0lvA9oKRTQ
  PMv//SzCX/usE9N81lQ1s1zz7S9BNzdIdkeA3zsutmJbJvhLNtb9QIgFyFg3rz0ZV3TO7grA
  YQBv77cnbn7fZO9oRg+dXQHxwvvmczI7J8Bfuy8wu8G3A/bXdLGbZPYAaWfHtUEMUJeffq+3
  PhN9DxbVb7GkP+u7o+sJIH26/qJO0+H+Jl4045WDmVL6KiPg6OUzwQZxaKM9MIAD9/8JQG4+
  IWDlkAZIHQT8xzP+De4ADFrAmZADP5Qo4E1Se8uqmrO13MVpVW9Rl3YMNh1NYXBiznkcR2bn
  NDN16HLk8dCIQvS9mMAmOUTRn6MO8C2bIEBN3IsNlnryGpbproEnyVqx/7iWIJqhD0kJyN5I
  Ipcp0GHLa6MBIkwKND+T7YkxAjzZkaQIEwdsD0zIK1OZbpioBSyPJic7E+0EhMMOzWtuFERb
  mb4jrYFUpgByfJqoYpWeBohxWbhDyZkakDcE7q+GBHhd70KkNJ7IzyhMbJATbzKAL+ZQOwk8
  I8CgCCcFINIk/KGZi4r4NWwRSlwp2ZlsEIky9UgtdODhYk3OtIDnjUiM9HnAJ19ysh0hp2Ji
  DOZ+HkCwht3kAVfMIwQMsMDMxISOwCTRd5BTpf8lAJj6IVqcqnQAjN1Sm7vkyLKSNkk+MmsA
  4XzJI4miyhmmkyW9vCTsYrQn8dlEdDmsZUxIGf+bKFUpfWCq0n2K9c6MQEuHk+MWJ2nkyFgO
  pXmW248wSYQABslSJX4Zki6FydEBdnOV4ysQbsJ4y0ruKTAwgaYYT1k+MRItWmK0VKBQ8lKQ
  MoSUcfrJC10jz9GUc26G7OmQcETLgo5EMkJdAENb0k6hOnU9K3EAS/XJkNeN7SdUbKFNXrfR
  jg6wQPa0IcGUOkYBenVQsrloSG6lHWxylK0A8KRL6KhMBN7mrnfFoxjRite7iqhMGgSbSl7H
  oAQ0sG0c219LBMRYdC6EP6NaiKkGwFitjqSVQtUlTl5TLHkeoERvWoBjYRKup4owJIp76lNr
  phLMruenAzEXDFUyWcb/IpIB9BlVDSnUWMuGBIQAQJ5ZgwlR0eIEOAKI6QIe4NUxDjBRmkwJ
  W6U2wOZ6sVBzbatL81bNKs3rjwbobjXdKlOWfIdFO4ItARwQtcVsEKNtJM0bDeI/llWHXAZQ
  pd5WwlmnVvGe0bqRD9WTXF8B4LQgodCA5Ylgj6RWtUJlbUoUl1CDMWslKTJXAxRQshsCzrj0
  DVlwHrBU2tKHpF51n3B8a5IegmevDCpNc8eYqAZzBLjaGa5XF/gAFm9EpTMOcpC1ad62snfF
  bIPNjDqYktIW6wD0amojBeK/SEatwiR5sKO8Z0MZDultTt7yfuhjY47w86k9ZomWIXxJCaNE
  /3H/RQg/3UwSAyPpvj+LFr0e6KgGYNkuJw6yiqmKNWJlZo51QnFzEYCwMmeEYNQV8govM2WU
  AFnSmA4m0WZ6ko31DAJ7wrKKDZsv50T3I6Yr4F8QhkcqH6kA6CQUkQSyIQ517i+WZI1aMdJf
  oXJ5fJXznhg5mbCvPqrIbIaTjzOy5mSf72ViO/WkGiTtjliyQ4Bh9b5wDOtQwprWCMMN7ziU
  gDqiBLj50fEYL7VskCCGWMJWnXqEG+SDdsioF0EApG2JaX1TeiXTDICQ1S3prAkWJVlL7jKz
  Q2iJFWijanN0RDSYKQ6v6uJPKuxsWrm151W5PtdTKjpNJSCpUjqA5/9u6pZ37W58jtishOMe
  vtqWsXM326mcNsnNna2eg3eaiXHGllVjo0SP1PSa1jGVqRzAZ+fM5lZIH1HTN86hAxiAWhcf
  ADNPju+K3BBMBN/PkTUbE4q70ZY3VIDa1x4Afs8Yrs48iVU/nekxEqzaqK542OsezBuGiM4U
  SVFi1CskhTM6hAgJF6kh8HCGXEde6j2IEI1VQo6afQE7WoBzIQAb3DCIRepO0Z5ol5Khe7br
  HkErklhkAAfgj+lW7ieJOuWgdlPE9Dw/sEpUnnskjWdJVq/qghXGgAUXnXGJUUDkDVK1GZ2V
  UJiPmsUgLps9gb6jNvKlAUsCtRN2lKKhnpv//gRQeb5/n/McOn7q9wSxFDOWoyTCbaFQPxHV
  40aOe9+8GM367kxJ3GCpxiM/xTDMBT0M9yEEtD+AhTFcoiOUJlcPATMs4lyp8zaf0U/lNyI7
  Q359V11vo3q6dxLF51R4VxKfEXtDkiu5smCO4zCPQn9OgoLrUQDAwmAwOHEs1Xtao34RGIBY
  8lMqV2lUhnv38i2fsSZb806EEyf7hyp/ZHqjsnD7syp8w4FlcioCdF3HZmmW4Ule9Tpx5xIg
  2ACfJVEd+Db5J0z2lgA32BAVFUvNpQACECVut3+EwnIVgR6fp0zmp2mQtnwLUVpfgh8J0UeV
  hwCecixEs3hwYxkJ/4GI68Ehf2ZOCXA2t3R+JVJYfOiB8OZxKQY3wfJmvmZ7G7F1Orh6TgMt
  zWI8ntV6yCRPgKgR/naKqzeJDWFn9yI4xIZ4BRFz6lErsxgbJSgQ/GGFYrc5c8hhYgQc17Qf
  fPJcpEFSCOBFAkBI+HOFBIOHE0E4uvQWV3gyr8VLRUQkfYhp1/N/D1EgsFZd8Ec9dXhLkHhI
  KvGKXVhZ1EJZ94h1WIeP+khZbIUcpDgQ/laJNxIiNHQQ0BKFyeMgpgEbZfg0Z1IrF0JICrAA
  ECRFuZWGuHQZAsdRpaVCGnlQPjcSuFg0sfgRX0eLjiJHRxZWIUGEjpNNMkgdbRgRMlRuvf+R
  kzq5kzyZk3UCkBpBbGFYEDkIJwYBk5chN9fDM8sHH6lTSV+jTAPFXG+oHWizhLYUSpSGdmpU
  kw4RMMYSRqK3HtqYEUdXgOWIacDxbSLRfXE1Y/1VSzOGWcPYESankh1ii4fiPPrhK0JIawzH
  b5BYQWGjHYoGG0V3Jnl0ISV4Mx1Zd+7IUfKjkXsVfkE0k2WJEXeJlw1yaBBlWCQpVKgDAeTj
  KDwIElBJHWn5NA+4HFozNQHTMUfkTgehco8zTmhZks9YJvCRH+FSgAUCkhAQZsJYJoRTlxpx
  JA/JeJihTvcCZaspabN3OiIhf1MYZPRxaDNGRZlJESy4WjsSkNz/QpN9GTcHISRoWSavc3Av
  JExjhxAXmTZd2BD0sZyZZmB+FEyERZnOdWR/KRJn1k8vEaCcSWklpGSAJprQM5PgcZIbUU/R
  aWDdKZDvI3nf6Rx5xj0HaRDpc5oEID9XiIK5uSP5aTpOY1W25IvqgU3d90om+FHXlEVZM1K0
  ZXZ+Fp0TpXzflyBDaZdJ6VXIdBzk1hsKB39eopdlozu4tqRMugDC06QHEH0A4JUE0H00pIGW
  MTWG9FkCRB6BtGkoRiKtVBoHkR3tZ4Bx5XgQtInmiDHlR3sK2IfxiKQXgRgXeikTioMFuh4r
  EhyfFpfimRBuaZpYV4HUcmTcg5wcISqK/9aH9BGoFIokJbNz+9I4q7cpVkY2vrIdZQJQHGmc
  DIM2TDdBqimTl1RC8aioAbIjxhhMqWluoPR3PcafBUdWHYWoawMSVERwu6gdzXUf6FgR2aFb
  F0dyxnpxyIpBIwenkEpl6REnrXchG1oQa8pR6ykQtMRRBzU12IEbDmAAF1NuhHcptLp/IFpd
  wVmuY/Q60xoS4yRPrYYSWtkhRZlsfpZaHQlRAECnEYGUo+FWE3VJHvoRt9MA1TiHBnuwCpuw
  DIuw1WhRN1avPSoQngpVHypPJeNlmTKJjSKXy9RTaBmct7SeH8ig0AkBYEilB3EzZ4p9jMSv
  FwGngsF4MqKuHf9FTBn4R4QysBLBH/jXgQCVANZlNs0KEccToY/aEYqnKwlELmaqaPF3Jvhz
  OwDbl/SxKX+iK3tSghfymH0Ip2Gkimyaaa3UcB9hbz0Vr5GBNPaTInfaU6jTTpqnHwyyfcvB
  e4UyY5fUMyvBoE6VpwJBnJdCLgx6dQw6NUvJkQ1xIQV4eD31PAdVQl6aR73aUsMJHkUrMXII
  h3D5sidBe0LLeVFjsxxVjY06RkcysRkBuu8IWPDSp9qxXIrRXMuitihhpk5js1GbpR6RuCG0
  Zwx3uigbNanmPVk0QLABgUPIRGS3EMmbliRSGbihH/ehneUIUe3au/WaQ0kYq/fyPIL/y2bI
  U5RlqGJIthvbOxr+pIz7EV5L0mYwmxF+K1SAKxCxWT/0cr+OUgABmEPKC27qkb0IAax96VQP
  qUj6cSEyRppvexmN6zwJ1nlt16aPkrmJ5yn60U2ciqOk6SDCa2xR46ANAbpqSa6SJirxixFH
  u5qzqLK0hrAYOBiMe1Y/6bXBVLaDcR3+sgACwLM7UjGki7LDSiLEY8N1F3+YG8E8l2YkkZqm
  JiDpi3MUIjYPeSUizBBoK7Bi2sAVtxIVq1r1+8LV2E8oR2ucSWg6PE+78xAh8sARJpYYkx+e
  gpZRzC4j4lAegWPl640URVEeSCJ/orpKi8HZ9MHmlx7p2YEQ//CvH0HCcOlReMqOzfWsV3wR
  JBSh5gkukWNPz6tjvVZ7XlVccQGO5SMR+hKdwGUAMpWzfAdRbNkRn5xsZEqSU/XFPXWVYjO3
  w6s+HjGvMteRZndJtnsS89tTYRwr60J2BOpsv6a/IQgRZHbHTrVRwDUvlJxNXPwo0gyrsngk
  xoI3dQPOiRHOmqMY8ObCvYjBiDK2fXgraENw5gsAPNsXhCxoL9S6rgpBlVwRG9Oy10sxI4Go
  o1FL5lu1t6Svbxl6l8EdRTnPA8GYksx3LijPopPIX0uTebxzPfWf4lS5zkZdMLnAf2K3BlXH
  sdGRpbm/KWwR4ctzqhoRg4oZAJLNTv/VGJxE0jeVHR0Zz/KEL5dL0aFVJiZ90tJsthjR0s62
  0iMcIqn2HBzsn/jMmw7cyCFixPAnP1GtniSkEgVi0OZnpehMEA9HMHGCuYtGPYYMXJtDafch
  wIurPDgaLjX4lhGK0Ryhor2n1Fg81BAmRw+mV8xZH73M12Biqmmr1xpyxhZcEERoLHtKTOoh
  yAshJDutWvh3Xpc8nDQtz0VdtK/Bw1GqQDwspXOoQDFyAKnWOTEi2qjtpGFdpTno1Kt5Hz69
  nVP9EYqUf9bpsTOWS6+dEPfBzv3mjG5DkreDP39n1rRrGaxsruvBkTOawvfRVWmplTcanWWb
  ucR02r0zPBf/2gBOCtqi7aTHTIy2rFpXaqlwIkcjDRKudUkoPb+SHRIam3vznRGOu6c99dIE
  IK3phreR/TxDp53LHC0ldGS/thzFqnQZVG4BxOAMQDSepHQLfi2oSSi6sh5sCjLgKocP0HrZ
  xHQ87CWoatt4rLTnImhySIeCVk8rAdFZLWRgTRIFkhnWWcFBZpHp1lyDejbAdd8JMYKRtppF
  ZL3l2F/73CsYtOQZhIJtBxcLPlpnC2nnzb1vs4uAPXdJnnhOxRgZl1kr4a/JZtS9q989ReZu
  uMI8LU/AhNDlu9k7XSCIDRGVoZA6iyYoYSkj5QD0wYR9qSICQIPHETUUCSy2JtyX/7iWc848
  2Zl/k5WG8lPeC9HVHIy9i80tDCJw2VceMo5Oi6YfLYM8brnPxOPVdedaFn3Ew3k7v20QA4Uk
  1dNknsIYGeVstFNCvCfbfxJIsAzgsNMb2cy3XL3ZlwTkvFbMnKnUiVgmyF6+oydHMS3McOwc
  lw6fVrl/pBki8OQ7AUS9mIG8Hh2J4G1GEd1cCRJZH4GohizRmc4SF/LO0TnjNM5uFtMcucvB
  FbVh5UmuFlF4Mf7pCA3vaVkgbh0SXwxPCOSFuwxhPzgiBU53CTLMGxHtKunQG1HgbIbmHYHx
  Zp7gGkK0FvNUwnZQZ8rxJ/02KXsSbePPEEntLCEZxNh0mv9X6/Q6hzY/xlbjehId8YtOX9m5
  7v0W6e4O1yxM3AXfEZsK768B9Jm29NVVWkneNs24mga2wWmpQaCp8iYt6Y+1KCBjZfzbsAq7
  OySFVNwj0gwi8bKo0RAm7INF7PnU6lSG7LQY9cuewU61NRYTvANE2DmmR5aDEl1b7h18YHKf
  J2uieY7bw06EWaN5iZgW8VuuEAlUhxUIj4TfX6uoZl1I+HzXfcbObKXKwUE2pjd4JguclvfM
  wrSUEvX9ci6RxQUUeUvnWQupHZOfECh4Gzw5iEjE9c6r2CZh8pzJ0dCcKWgY7gNdJtlx4E4l
  27sc+kGpzR3YShYfsc+4npNkVan/b35rmfuJF2OeqCKaU/7l//jIyyYukbdxnSnVzi0dhOik
  D7b4HSK6LKeRvpqEBfwGofwuqRIAIQDAQIIEEzggkFDhwoUGBCQoGFHiAQgVAxAsoIDhRo4d
  Ow4oIBFAgwcBFAyomLKigQABFogEoNHjTJo1FUKEmVNnwQQDbP6kyaDBTqJFcxoAOhPBSwAC
  UjoIaRTAgpQQDViUOjBASgUDDyYF2xGiAJQqUzJgiiDsWo8ILhokiLDjAa8IzN7Fa/btSLls
  bQbA2UABYJxEIR5wSPCAWr+NEw4km1fyZLoA+jq2OXTqZM6d9RJkjFkBTgWeJz8YuICB6bxQ
  U2MOG7Uo/wPYft0WLoi0I2AAuLNSrFi4gO7aHEcTFOAAgl3PdhkcKLy4+Nqs1QEQn95RoHXu
  IkM7Pt4zpYGswy0eQIk2q9OKDjQLyF6za9Oyd9Vbjh/WwXbk3xUiqCwB4FhTibkHcBouv4WW
  gou7whrwT8GZUluNQMkYQA0/CRk6roEKLQSRLgGLY2C7BD4k0C4EMhQARQtLHAjCDY3zTSfs
  ZvTIAab6m6u7ggYsKMEZdxzQQgMQjBBHhmr0UaIbN1RAtiatO6A4BnsrC4HjisqoIgXQg+C+
  ogRgDgKmCkhSwisDKNMsBMibijYlZ2LAgB0vW6iy3toEEauBCsAzPzF7E0CAAP8cMMAABxBd
  VFEDGhBAygZ8mtOj1Orrs8DKAlXzpQaUM41P1lxLIE22BjizIlE7AxAylFblDIHtAK1UoR3H
  lLPWjd7Uk1MCMpwSSIyejO+5P5UbwIAvC2W22UOvurIASnVlaMqdiFWwVWurO3G6+9DkyqgE
  roKgpdUG4G+nFlU9E9sNcVJOVLsGwClXahliQIEFFtjIAX0N2ipTlYClFUcGohLAgAEwzWth
  B6Abadp7FaJQYDcz9FXBt6iyOFYIWIwPTgAeEHhjWDvb64F7pSxK4okTyveABfzzF9hgUwpS
  phlhBKCABxoogOWIEgha5j9dnpjJba+rdb7qgIaUuwb/4nNAsTKdhikBSMkNIMufhY5ItZT2
  4lfXqpty0SxjrzOVWgT8U6AB4UjueLm9Ct75JWnvKpO5NgcYSsaXKY6z7pQ2rfXInk8OkwEG
  BlAgcskdcDxts15KQOXstAWAzVgdRzTyAGY1KXLKGWB8JZySo3YA6xRoW9e3v3tUae4GFM7d
  6Tj3UfDBCVgaJt1DBjtISA9A/gEHBlg+AOSRB9r2garcHNgHyuQNowIWQF7hAQz0m4Hlnw8a
  N7HbQzhjBXkmd7KzPfx9pr2ITthwP/lS0r2R+kRgqAR0Htyl7LeigSBtQ+jqDahU8jh/LUAA
  BxBaAhy4gEMNIG2A65n6GqMn/8+5SXwOcMkCNKOTAgiAe4r6Hl4csDp7zUlW1pFW/DyCtaUJ
  61j50wzRErBDHvbQhz3USvwQIL0paTA7PItI0EhiAMvhBQEgFGEBavSV/FgPe0RrwAEWJZkA
  NHE5yopUAtaluhgZcEYZKsBqUseUAMiQI/Prmv1SAizp4Eh/BcgUXYQ0ODiNsW4E1FClRBSw
  5QzAJWADokTEqLwyuQY+EtIT3VQFQkTusDdesaRIHsDERiKnhXPaS3Wu50YOBc8gpVlOYcik
  JAzGRHKvhGUsE/O/+J3NlAUBII4AickFOOBkKlqOZAwpxYJMTUJ0VIkBfqmiOAJTMsp8SmA+
  aUectP+PMyvsTS5lOD8FyhEBBdEcjt7SgDxOL34ZOp/hzkaWWl0kAVuBHCIJ1ZKSQPCSIjkU
  BB41lRlFEgIOsBlPphKAkpSkJQQlqFdEghh9BiZ2CgKJj7QZPxou7VPt+dGcjoPHPqHznLeM
  CDsrJTLlxcoAD0DpAxTgxYrEM0ZnVIxnHlZQkqS0JDKVZuuYAiHWaKaObjRAVA4ixwKBc056
  RCWI9GfMwW2KqBC4VK1syaQHLOB7s/Pg21A6xT8NDzZ6qpEAUIq6t/GtQAiIGcQECpmHCupW
  1pkZKQkQUZBOajk2ox6O0JmpjX60KQ84QFVTugCb2pSwhQ3sYSsj0jlhEwD/q4qc3O45tN6Y
  Cy8C2eMxY2oWEIanKIeJnIscCz9qAYul9gmJBEn5wgQ+NSXfJEg4D+g/lSogUbbFbaKUtVtl
  Re4BqfUqTHsGtQIQ17jFRe5xjZuathYHgRFpQNfIGioELKx2SQwuZgIqRoVN10JvcxyqWNai
  iQWUO3GVIWvrCiq8stJ/C8gtb3lr291G7qTA/d1blDk7/vbXv/+d3VimKSGelWmYxJQK0Uro
  y5ToLbuNcQtGvqcjKRJRJ1jkpFpJq6sDaQVEiuOnDLdzUdeqyqg4WgBOFNzD8rG4wisOzOBc
  A9JT1gpVXhHM8laVugUiKsVdnVOHr7MwyfiNoAzD/wsDo2LViWVvSuj93VtNadddAiCvGxIy
  jH+4ZRj/aXB07abhIPJOGzPFggdowAh9VFwBlMbBSrpNjIBmmJKoOWslRI4ZhftOvuLkpxNz
  3UAk+VQV4dKOIH3w7h7AnzRDKs0CeHSkHT1pSE/a0Qhi05yeS+QCqRClSFZVj0OCtxnR1U2t
  GTHsnGiWDE1USU62FpQBbWEf8fljJ96QY4Pn6hm5ZYRdW1iwhb08YRfb2MxTcRfndJ80FyW6
  AbAzTEoIkYwsm9bIgeYA1GoU2TY2Q5mzWJSmZ8QNbScyJTaxVxLNlodxj3vIQ9AB0pVF/iDP
  3cjjtYS0jZFnMcpR/gb4v/8F7u/RKUbP3tpOfWD1pvestDNCGcm6/eI0a6owAFLKHDSFuZ2y
  STWUNZT4dNJFY6rskpYzepjM7C3kothb5QfI9/oCWsJGW9rSla45ziMdES0qidljApUB7LkT
  YmZ2Q4qLmk5IwjBoSw9SEOm4ppNdty2x7V4MiEsw0Q3bgcRcQtSOV27CpJmDN5bRp7XYQ/7k
  dcw8d2Rhfq2dfJMwWA0TyEpy2riaeCWJpNNN+oOMruhKY9jdi+s0nt5dC9LGlUmF3CgfOeKT
  yPgN3SdfkfdKm/3WHj1JZHsMMrqEyPNOABWvc60x/acwyFRdZs/WdQNx5gZ89OmBmqiH7/aG
  Znf/MK3YxZanf5NXfDk7ajk2c7Yf4MUJwvbGILHromqlRDx0anoF6fGY+bhdVSjZIBFSbW99
  JJyVL3kBzV70kidIydvrwtmZXiTj4m/r3I/+mJQ6LRXZNkbg3lKwpbEib1YSOKEbt0uiFFog
  89qsfWI93QOWBtA4deKPVaoUbUE+b8K1A0Ko3xI+CLgxgrAr3uuZA8jADCwolGousCgM70O3
  9igM82u7kfufvDAvv0uJ6rM+OOu86fGioPqTDsKLikoj8aM/T+KwIQSUW1MMn1MAmrIOASEo
  lHrClHrCE0yK+RvCsnOM+yiTHNwsldC1IPkQvWE+xwAkQgIxkUiOisAQ/9vhmOXQIyz8qvTj
  MRCJMLaqlEDzvzkUGNzDjNrhMh5SpLKIqGm7DlWJtgsjmkyrDYhwwBVMMqZwwSxMF2k5mbWB
  P2HqvNDLj+jrmS6KlaqqwC9Ss3FREgSUPN8xmCEUtOXIKMe4rjEbs964NkKstU6kQpqwQvqD
  Q7/Qwn/CvFNqj1/0ClQCwFITiARQQ1iDiU06xLApC9Q4OSgZswd8KvHRDMrDkUBbDEc0xNhy
  DJYziu3Jkso4qcN4raFLsKhzjJA4N24MEzcciFsEixl7J7TTknyKlbiJketrjP3oQdRpDj2s
  k1kZQ784Q1UcmUrBOlUkJwioN8dQRomKyO6QQP/MaEaEBIBIbAy6yhcrFJCfMYyZ0Rt+9AsE
  KpWLo7VMIooSYgB3Ksi1uBL2cMTKQMUDYi53REL68IvscaBm8UmY0y0LGpsYeZtjlCQG0q1F
  g7RmgRQTksM+hDq0u70MUcT8mA9KLLHnujLRY8RQ7Jh9S8ghwUiM0EjYODz6Uw09+ZSdvLaV
  RCsp8xFZY7e2RLyqDJkGAakdCgl1VBB6RL/CyL3siKhG5EYMIsX8GQgmwkmCIUmFSBnxcZjI
  ZJ4dGz/GA7EHsJzqOrZg25KtXIsOU0GchBPV0Cwrc8TjCD8sqyxH9L2QWMD44ESMDLm1+L0h
  7CIEmc2qsxbG+8In0y7/uqQxxtqccSKsgzLO4zwog0JO5EQpe3rN/HgLsVLO5aTO6mTOBRi1
  GTmOIjnNbIKzDMFJL6mxsGAAagvNh/sSq3HM1VE1IyE7iEzM8CyqqZDHn2AQ8VhBGKlISOqZ
  iisxfywA7QTObSGz1htLaQkQbAQKBMhFo4Ay3Swi8BhLmCDN/KBA1vSUsmwM1DAPbrxPsWyK
  3RNR8BpRtCpRsjJREYXGwCwWMyPRF03RGEXRGT3R2bHBplgL3piZFyUrFO0vHTEIz3Qy5aFR
  tHIcGH2bt/izsOANNgGwJ4XSKDXRs1nS4uBBcoLRIpXRGuVRCBgkHBGRleLSLc3SMd2nAN0Q
  /y5ESM90qwkdmah4zpqYyCaUSwiNy/pcTzcNGzytCdfAo8pBncox0kA9UkE11EIt1AjLHL2q
  rEGtLkf9HkGN1COdVNSpVNDxHz7NjD8BrOfx1E8F1VAV1eTJIZ9jincb1VRV1VAtJg21FU5d
  1edZNJ4ovI3QTZKIVZmZ1cVpjHHq1FwF1lGtqunR1JnAnAcK1mQNrCXDkWPMImUN1kWjthkR
  xiH0OTXFyCAEC7iEIbtUCDvlDjRpjIpy0wqND2hEVmjNVQ0MsU6JEXUN1h+jIgW5SPpLAFPV
  U4OQuAHtO/OrMh/R1HpFyBiKD/XCyJ6YkQa9JWrNVwDAwYb1TrBQ2P+cIK+ZMMXq2EXtgFiv
  yNik4Lux7Fj94FdrYdHamNiFNZhqVUU0TYqRDcG3sYkVUdmdcNWN2Ni4qKJ8Tc38cNkmmRGB
  RT8449aDjY1aOwDUiVloa5KQrZabZZr4+DiM3M/smLEJLVnYeJzN1Nqt5VpjW7auBduw1dqw
  ENtjQ9oFzdqyFba2U9u2Bdti7Yi0ddu5Dbadodu7tdu7xVuf09u+Zdq7PVv71My51VS/9Vu4
  5Qi5NVyx1T3FXVyuRVy5ktzJpdzKtdzLxdzM1dzN5dzO9dzPBd3QFd3RJd3SNd3TRd3UVd3V
  Zd3Wdd3Xhd3Yld3Zpd3atd3bxd3c1d3d5d0m3vXd3wXe4BXe4SXe4jXe40Xe5FXe5WXe5nXe
  54Xe6JXe6a3cgAAAOw==
}

####################
# Leipzig:

lappend boardStyles Leipzig

set pieceImageData(Leipzig,25) {
  R0lGODlhLAEZAMIAAH9/f7+/vz8/P////wAAAP///////////yH5BAEKAAcALAAAAAAsARkA
  AAP+eLrc/jDKSau9OOvNu/8gBYQkFgBAoKDjcaZZW3LybD+verBKHt+ZgAAg0AErrGEtQhwq
  hoKVwHlBRY+VAGFrxEoABGIYuW1tCasyRkC4er+AAaoD8xB9IitLrp9fBANuGkSENGdLGHV0
  BCeMZG0oh2JjF2eLahsofBsBQoIaQkldTHEDpqeofhFaOqw9jlUsiBZgmEGeHkJtbKMQtWfA
  wLMMri6wxbRQwxWWmUqUQYCmRZ++SrLYVEyBC2gr3BMyKUXiGc2DZ9omwLh5fNgpmxLn3gfn
  TAuE2ss4cYoXQkzFYcPPQcA9Q3oZLAMvEqwIStK0iAhB15lGYFIAK5L+5SIodqHWleHycFWw
  k1wmEJT4BJpBNhd/aUyn0EGRAUKioTKlhSOzAHKCypsQquEdCBSpZHxyhccCLQxrAYLJwqPK
  dEFOkvNJAdiAdAV7JNx5gmsDTEopTUkjCGopXVDaAJWTUgKbuNXmvSBbV8KJQP4AqQoH9GbA
  E9tagFnhh5I/BrVQhZIjjactpMHyRoAE8yIyk5Aohwn7BAXcF5rtOfa21J7iLZDDnK5MTXbr
  ioFOAarpQNROnLDDDQyMk/QBaaWEpm4iRcXjQsfVlfkN5Xctv1+98m6QEWqwTvWYbBki2uW8
  Mnd/MVm7w9HS1lNqbMnODvjG7Crn4kzNvXL+qlO3+VKKPwIZp8yBgz3QhgtE7CDHcTAsdlYY
  PM01nkBzXWdXZrSgFIZnBf1CAH3mQYRhCgNOsGBP7Y3AS3sOzMeFdyDJGF5vyWFIwRS/mcKI
  hTc+MKBQCW6mnEBfEebPOIGUhRNi3FH4VULdNSFQibGdtF1LHmoBxlRdnUGicSN9aGY4FHJx
  F1TAIQIGVEKANyMgI/ETj2QF8dijk1cWNCJdpZhlV5zdUUMUioywMaWXJ/TiJV2AZUTEVwOV
  1BtKeWo13jjzFTlhmX2pNE5gZRFVZmdUhfrKfF81UoYWPqrKXY/8xKkfclwAuOAX+vHEH3eF
  AXVYWEOACgl6mPH+8c5o5Gj0q4iH+LULWKx2NhCaWv1xx5ph/KoaWFYEg9lojMhEqVTC7eRp
  PpP62q1sqARnVyRWrIHkvd7u0SMfqU3hCUo8svcFpgPTSBCVh4AzDzZBnvdusfKS0itZDWLW
  RGfj3bVrfqfk5Fccwrrh3ymyAssClBYgqExYd+5b3Dzk7StGyhxKO5LIJ/mz5SSQ2CtHoymm
  W5wsAPITmsxsVHLmjtsucNNvlz0wxYi7YDBiKX/G0TB3VeXsFATlQsuQFkrfA1GiOKM0FFIo
  NmK1mW5FbI2HYOnlUDCmbY2jbluy5fTEJGPJwGhVrVEUsx5Lywt5d7RhaZSTTLWp4Jf+skNG
  2mqv6/TQCldA5ZKlmro46I5v9+VF9n1IWrt8Z/B0j1H3R3EW16hrxXZxToGqv4/HJlO4qj/y
  net0Zz1wzrQzK8yXuFPDs+6f9VYbqMqIMPI0xu0AOCorGfSUPop0QorLBapIODxyK3gTNtPo
  nSXBNBcvEA7fp2eE+AKSv7aCRwMIU5jXkMX/iHK91nVFf/upgQBlACYpyEJ6RsmG0fTnPgWM
  KIBJSJKpite3b3HwQfkQRjfkJSI3FQ9vC3PVIdymInLgTRc72p6vYojA+YklPpwix2ms4C+u
  1VBz9ngdnrzVP+4J7GzF8xbGTigDi2ipCB4SlCYQCEQzYYp0iEvcCBEjGImucBEeg2uBl+hF
  OFhICDIYNIo6hEQ349SCYZQb3Akr6ME5ykeMeBsejKI0x9jxEX4iaCMSvsgyQp6MGDhil/ca
  8AJDdrAH/wBFJPPwRUo6EpFcUyQjDWJI0wBkkiYA5RtGScpSmvKUqCxlAgAAOw==
}

set pieceImageData(Leipzig,30) {
  R0lGODlhaAEeAMIAAH9/fwAAAL+/vz8/P////////////////yH5BAEKAAcALAAAAABoAR4A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5sawJBALiaEAzHEAgYHC+xnyI4q8V4tF6smLTY
  cDqkxRcABq2yjW3XdA4IhIG0KwGYDwFC9WJWA8FrNOGs+c5DArN5zNkeyRQ+aEJTV0OGg1l1
  S3h6AHwbbXNMHgI4JD9RPQBqQWmdQZOReiEwZooeVKiVlyOZXGyesrOrsbUcADo3lH12Yh9i
  ua0flo/CkGWcYMvMzHQTlr9DrdHIgQPYA7xantYXwTcilkt/bNqO6I7aFVtS4QrtkeDbG54g
  N2KMlTtgljY959KlW0fBDhNCBkdhe+bB3odZ9DD4OaJjUzMCAgRcZBj+QdfBOB4jybARMcOn
  OL0wqvFREho/ScEuTNylR1cQbwwcWVmgU8I4IgdgYMuiStoFhx1mzrnlhIiPYjJpSYU1geVO
  BVZ92iSqY2FQT0YrwGzpUmOzrjgh8JvUr6JMOxeZMYX3TNOhBSTpPmowM0jXrlLT5vRE1ucs
  MWHCWvAEh5xEp+mAQsjDxO4gvIoo8/FT80YMwAFvNj1n+bHZi/HMLZMkCsPfz7qGDnOQJg+Q
  yq1g8NQ426/ZccqE/o7dVJbgB339YqQqljEo3a6bZTwrQTKajzzj+Oh9o59GkmBgfDfblR0+
  M0OPN9C2JW7qCohH5rHDPNDv4PclfLF9Fyv+dqwZDbMFXMtoE95pBZZj2FalWRDFUrLsV59+
  SzjzmXpY+cWSTS2F1N9lH27HQHsTURbeKTe9Z5hcGGKWGILMbHFUYyjCkcEj/eCXo0/C3MaD
  TnmARFB2MsQlQy5xZeWScUakoQpYyx31WXA2tUiFNktQ0dJ8rfxhlYwKeDVYGrQYREsnheWA
  oCUZ2ABjgcyUV8EnJ8qlwZELCXOKeZc8goNOMeUwoQ/OmMWSd6vpM8FWn7U5lQzK5bKYc7Jo
  oExGexjKZ4Z/MqJDEQ1+RWZ3yfGjC5rXXDRkIGsZqdJyEz6AWJ3LoCSWHrT2NMEOM+yp05EH
  ZGQrPEeEF4OxRXb+YqiCETxpHauPrlRsrA5s1RgihaCorSlzEvWDVYyAydexRxiz1R41MquW
  e8NCgE1cy1CmTBhzNcDau+HNFgiubE3CEQToySvQHueoVWQxudT4aS4aCULBow4+StF2rVVF
  qV83Hpnlxmk+xdLHS2wR0Wc7aFZiMcLqMie87TqAJLxgYGlhy7StNu+qDiacjZ76dkQvJzrA
  8YVQYVhDdMLzdcLWQqg+PFVztFCRB73HutGtVGnae2BGU98xqcRMPiB1kOhNacojz0ZAZzM0
  M1DyvGog+zKyx+EoXc9V3XfgjnPmApkeGrYdRZBsHduGgU98TUu3Py3h0RwTAV23VHj+l+HU
  xoruSphAlXbkuBqUZRk3lu8UBHe+KxOYsLYwZq4fg4qJpTfkfF9zepI4V/sFPpfDRmbs7lLe
  90yhAU5ExbvOGmSLGU6Jeb32EggzVBTKcG5RsannZozUrjceRtRIXyv0OWGKY9a0oUfaOW3b
  ezu8/9I2EsyTEGIeLcdx2CixUWOUNYpZwoV3MhUeNpzOEc2IXwNCRj8UOcoR3eMLlqLhIpbZ
  rzqr00blkiGJDpLCQf6rSUBwBLwFckJgedrDUhwVNgq0I3H8K5ONVoYRs7QPYBrL4SnQZ71H
  TfBhJ1THCBt2w3FBEENPwAZm3jS+IuYgNvhgXlD45UEFPoD+MNgIEOkuOJnDBGgWzOsLEiFF
  DYnNEIhsEY+ldAgyHhIhi9FQxf1kAcBumIZ7N2LC9mCGrWbdzorNGkqZcqc2P81BDMUAnVse
  xpveHesfj1ncJjBjRqtVhX1H4k8PmBgvHhqSYYncwSJ35acL/QSRTiTWm9ykBfFdpI8AM5DN
  NugA8NBvKcfh3VSGorgRGo+LhpmKFIMFtpNA7TCRjBCj1KUWQVJulNVx5iBTuUfURBA5rlQV
  MIPFC2EsZAx7ScYtbdacgDijbP9bSzragr5S2TGZlWyZJrESm3ACqCrvS1LHGCgdJVmMJuf0
  EHzyibq3jDNBElSHFBATviGKrZj8dETjLSGJxiNFZoURkxgtxwVRjdiqcTdY6P6CxSBkOAts
  hTGcs/awTflhKkKPENclx1nCWmYTZgIqFwk1iMhiwJFgt2DNQQE5DZo6cRecnI8TT4q/4kAU
  UiOinJ7iuMxPuYyg+gQiUzc2KXfuoKXrwWpBC8JJmPHiCeo7TTviladQTfGg5ISGRzgXUi8g
  NUZ+q+l6ILrRMD0VKcQiDR3JAZaymdSXA6tJYUBKOe0t04uKA1tkn2qvMQTJlByyLD22OpWs
  hY6O9pRJHbPUor9OFqKVHdGVvhgy1T70r+TDy2gzWYPZ7gEQuM2tbnfL29769rfADa5wh0tc
  DyQAADs=
}

set pieceImageData(Leipzig,35) {
  R0lGODlhpAEjAMIAAH9/fz8/PwAAAP///7+/v////////////yH5BAEKAAcALAAAAACkASMA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s674wC8yxSADEcefaPfOHmXDhw312tdAweTQi
  e0KgkKYoAjlPJicQAHC1na7gIACMNWYvYCHoBs5B8bpDcF/Bl7egjd/sx38bZntzZH9wg2Ue
  BH93fRNmA5ICjiCFJ2JkZoICA2ozZZ5lM10Dih0znpdKh6ciiXuVH6slgYFof12gh0J6rpyE
  JbC/lp2TtCCUKWkHb8gUkcMAorymzxZ1XrKolAOM18B/4BzKKMzO4a3qreMTjLHCXG1dtQTe
  cCGUdeUlXANcADlJGkiw4EBiEr7hY1alTIBZanTVOkRCHzwTvvQ8xDX+paNHUO2I3FLAUMee
  jdx2IJwlriKXfQG29QgliVEdmRAiFfkYxdqFVFSCuAIaMsIPSkFftRxxMhdODO8O/eP305jB
  q8eqBiM5dCkHMwGmFs0zzcsIRp48yYsJwqY3HPfqzFxHt9HPaWy5FkpFrwIsI17Ccllzg1eY
  XGfZ1cm7iO7iuXUda+VXMlHRvzm0UaJX2GsGtJOeVnBb094glOSmpVW9WHSDN13sYb0B250R
  rgt/bdXxLGorMbx8tXLNwDdVOupi3kPtRxw1SscrCI+MOOFtobkv7b4BwXguadPtZigl6cuR
  blj1EH8dVlLZgWo2vOQZxXyEbIz3MEDHBk7+Nnz9nSTEIfb4NiBF4/3WVl3tvbHeftVYthIF
  g8l2FW3MORAVEPotwJ8CHZoUIgPOweecZQQlkmBGE5KlioUEqeeHaV78k4paqZEyClgDQlPW
  Xq50gshQNDaQCIyitDcVkpZhEF6LFuixy3BxNdcGXLQRkmGUPtA4zQ5yQeJVSSNV5hl2BfZE
  nlpv/UDNWIUpt9h1uJiyB1ZTReeOMVZNUh6AeaQIX4zQCCaFfUKy0VccNzVQYkGgmYLknyPa
  lhydGYDlnmNVQvXopm0AagE74rQCjXhCoTTSbkc6eqdxusDnBZUjjdanpMjZ8xKeeWbK552x
  irLFQDfAZ6GoDoD+ghIpPCSqg0QKOOPIH3hGpJpBKl7gnZ7YnPTcOgVym9N3aWqELAXDNWKq
  j3zg1myH7xQirQPe0qXkk0JWOgFN8G1p66SCEnQSBpEMeq0qX+kioJTxWQAWG1Q4C4qHE1JL
  kIVHEltQreiu4y8F26bbyD8fPyDNt1BKcPBbTFal6im3DELxM4TY2Q2+bdxzJ8cS+GPQotqi
  h+etgxAX6WoxjuWoKjWhl8q5D8wDoilTn+HsPhFY/KeSWsMmsL4R4Kx0Q9QJqVFoQUfIDjkK
  58zwGxgMbIjVHd7yTtY7h4LXdP7glTfUjgLMyGdCz4bX1yXTK+jBKWed1kGCAv5ao5v+kWKj
  Zs+GaXIskF9ps1phWTXIWNQRXnZM+eqduKsnrt0cLDaXAXbY8IhjmYrvfDyIKcrR1Mqm3ji4
  m5hDG114etnEOHtO16ZhYgebtec8yYP/JJgac6qFlBs2SlCzndp/Phj41EwmWbf2EniUt1Sb
  78a6fuAQLzUEFEawOBqxKOCZrPs+he2gANXyknUrgZHuSwU5GGyUBCo4FXAgQBtPL3Z0JVJk
  wBiwOc2sHJKz9tHOGNoQgp3C8oMAjtB86yBdqH4TFQQKpzwOjMzY9vMiVXzpB6vrX9ngBwFf
  4FAXEgKLD6pjq+I5rBNruseBPJei4VFIcJLrWcAeV57xZIT+PhlZXc1i0hEB1YiE5UtbXVxT
  tOlsMBRZOlEOKVYXD5CKEDyUznfgcjEdyc17jQDFEJtiDdTxbF8A24TDBFOQwRwlNpD6433k
  wUIPKKlY7rHQGhegBl4lMIIP2EcaCTFBcl3kAqU74hnniANdmPEenmJgp8hBoEYIJ1OqtGT3
  3CGlFe7ojLBBVbeQZJOgDYpk+wHYJAZosklFUWVte98oMElLWRayKAISJrE2+CARgUuM87hj
  M+qimmNSEpeCREVNakJH+BDOmUkb1ZWkSU4ZacAZJppktK4HTErG8mviitrCFAlKpsXFPdPw
  JhtSxBOBgZKB9FHeFjyWILe1Czf+dOnmesoAo3CybReC4STcLqgK6jztoAT9iEJZGUdbteEx
  RLgnPuGEgys5A1McZVzkNMDBQwDHphaFhDK/w7eGWQlBnkLiQ61pL4AGLaMkzGfHzhg7mV3Q
  Y8MAaqFGCcff+NSKhZQnJbXpM17xzygV/N8MSUQ9UXzJHwIF0QpFcZCArjBuo3BK8GxHzJ4x
  FBdDxZlUEDZIgKomKTSVn/MCU7+cVqApnSiYNUZYVxLZsoN5qupCrzLUMKh0Y06UIlbSmtJb
  0uekcQuLTaQEG7dsdFQx0etJDRslBoVAtRYDbM+edNrPGJI+ypqoaBOrpVDcTKBovKJwwiUf
  5RgEpZb+RedXe+hZUmhVL+g0ax522BR54rScNpTqdCPz3NdQF4RjwVl3b4fbJrWWHf/LRWX3
  9V09vPOy5elutNiZwMx+8w6FScMqfOCXB3qVMWIaheB2obTp7aQnSnznDuW7ze8alXn4vV0D
  +Huq6E4CwCoTULWkdD9OXmVKYzMDfXtJU/r+DCH6ndUlikUnHoEErN91HYX8exX7yHE5H3FQ
  PVEYmWrO7btFMpI6VqzLqF5DgzvcYFG8Zsc5TkWeXttkNJ98wZVhhbUTSGJ0ARtl2dzWKjvy
  Wf16ZTIrO1O2ELgTOmvbsaNR9m5w/a6CY6ylVeALiJv5RpPVkRMglrcjzNTSZ2SL2tj+TCmi
  Tj1idBvHAMVamK/fFM4NU+GWnsBCF5W40aPRnElODrhoQT3exXR5qhgXmM4yJklUf+cWj8I0
  Dn/OcVEUUjYRp3LBSjWSn8sb6B7uOtakoMVQUxyYZvg5r+f6BLB7cepaf4WUTZ7hqmXoJFQb
  xlGoiepDwoMaNjfa2vaFsUef7exMLfsHBDu3RxyBjI58cwlGksWBz+1j+ryCJ6hQ93rUve5k
  lRne7k6WvPkdBUHgWwkHf4TCF87whjv84RCPuMQnTvGKW/ziE0gAADs=
}

set pieceImageData(Leipzig,40) {
  R0lGODlh4AEoAMIAAH9/f7+/vwAAAD8/P////////////////yH5BAEKAAcALAAAAADgASgA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM8pENC4FgC8vey9mwLIE4YCxlzJp2xO
  iMwD1DgdIZ1WgVaQxHYG2tvWA9gKAIvyFq1Qa9mggXzgBQXM3boGzD2MO25naWZwgXAffHR6
  HQIDBARgi4wEPY48gAKPgm0Aj5dtmQSbHwGOopIcZlqoG2tuhxmGg2uzb0dmrLGhmnkkUSty
  XKJlvRZlmgOVnZTJPGCasBylmiqBtifWfTXFJnwBomGYW80AfFrk5qMed38o2dEid4+8KpEt
  at/XHMfghJ6qodRd2NFgjqIf3DSUMaWvBDs89QS4wAcO3oVsADPS0gH+K1GDKyKIhRFIotEj
  eyb6NBqQMASYb2BsHNzAb57NmzhJUkgmp8EqBjxnkuEi547FEABRqGzUMk5Gobp6SJ1KlYfO
  CR4Z/FyQFcRCJDFrEFIakx3LE3dMnWTZc98uSlWlfoMWa5nQrQd4nAyZScswsRtTAjxr4qEZ
  SI0w4Vy8+GqEQD4l1pLs1W9FsXQLZ2KYqCmGtASAfJOnTaHG06qOQph2SitlKQzrgkkm5Gup
  TTtm/3IbmIRhv199a3T00q1V1GaaOV6NyzUDw7IbMSFWFHc56Z4nnPmneZmm0HxUfzbJLJmn
  Zl/OUUVCtSsG9+ySuN/5dIf06zKffkjqMOP+6LC3EOIGS8thhdxpUFUAXzhcnfNefe/cNthQ
  NhWYAWiMuZGdBM940sM8yby2hzCM3ZRWBWYxUMYhahwCoAKzFYNceGqkk5E4Y2Ux3GgMemCj
  JmugtIE5cUk1DooODsJiQy8eECMEAsJ1zhs1UillOxeeAUVpIJBX4kMbPnDHXHDpBaKIewQz
  Y1sUCEOYFHhJ5ZoR5YSGppP+PGJfGHquQWYgCVLw2xZhaoecXVzqABCQa9z5XomLBeoAREPE
  2VCPkDmQSKR8AqmWTfNZYOORLlnyIWPeKNbPcaH0QIZaxPTpXTIWtHoIpq6C8pqZaBpG5jy3
  VfgpsJQ+qJ9LVA7+eo6sODbjYXgWSlCOVIiZNO20tfaGqT5qqNjcc1v0qR4kn+o1II89GjOt
  TdfaoWW4kKY65DJvfUdJtBGwxO6+wErqk1X/HiQnjGU81wxUr5SoL3glvhKdKuiR0ilGhNpZ
  aINv4CRLeod1nFy2ePmRmK5w8AFukt5aO5gcoKqJR7XiiUlinxc/wHIpNkBqZ7oXbAfXwWb+
  BYh3kFTIbsxDWDVTiFw5U2k5QJ0BjxuQXjtzYw1ZoGyiHHRG8WH31pycmc64MpTHfqWmtdJR
  U5ZVig1+0kCL/fgnrkb34qsV0Sc5+t6wkd5UYytEU0tszXsfdzCVBWcwWxq5Si3ZwHj+Sko1
  pGCX2BvINyJyIBfJrdXKonUT+sEyc4mmpycjCnxNcy02aHnG4+yMYN/AZb0T4KJLw7LOouAk
  LwZBl1m43+qKotbviMGVZVvl3KGA5NPjPESTc8MrLpmZx1eh7gYi2IGEB5beyZimqcLot/vc
  TCiY5CJ9siJuTP/HRuwcRXe4SGz9EhL9gNl4fmUT9CmKdxrrxFsGp4ujFa9xQyvaPIqXCbn1
  zHqkEQM1jtGHUkgvAg9hF09ytobQGMU88zDdQA5Us609RRR7CgWbLkK6zonDMKwCnd4WQAsw
  iWwVGoJTyD4SrmUdIx+hy0cn+NCPMNWkMfIDFwKFZ4N6qXD+hSJEITJOp8A/eUceFkyevipI
  h4A4SYEMC6OmJmaelcAMHJZYGANpeChFfY5KfSmhAVeovtKNTBzVktyywDcBDfWjjGMwh+3E
  k45lEIM9ArKPIz9GPL4JLoqVmqKJQMe38Ozhi0U4nMRadJ37vIFrgspEWgJCHiY2EYZPyJjy
  TNWJFhFhlrgr1Nc2l0rUGIY4SAgdJDAJo88h7gc5g0vqzLSDi/3yjqpoCQ4h6QxDSK5/gdCl
  JUFFTH5YslwlFByWtKbJGbZvgiQyCoiIWT2YjUo9y/pg+E4SvA+R8kMsOR/K6Iiabp5jVFXU
  3qC4k6UZeUU5S1xWeGilAWieRkH+eHCE5AIxDA7+0jSYI+Z2tCirUr4BcPXbA5kASLM4/Kpw
  BfTXYwSUzAL2gJKxtFYRysGTf8qhEjDhWQV2ScgIKHRlYMvPy1rjOOQcE2NFTCqpdMG4qnlU
  pZUiFFuqaE2jeLBYKySgKDEgtWEZJQgx2eZD2Ek+VjGUi6Ay2gTZmZerAe888ovPKl1hpHFw
  IT4bcOEVsxVIlspkcanJzHgsaZQQWKNu5qOJJjOqtXHYiCoQMwfiXoKTEznOO7OaiTww11MJ
  +OwmyMuA1GpUttGG1rNvbUyYxrHNynr0mO9c6gXSEaJ96tVnZ+VqDgEWEk/oKZShYV0qUpuT
  1Y4LeLn+2euITvrHAZLLTkJZmOb22bNEJOK0yfMS856hxlpplSqNEW29PGQ40GKXQ8PJKx7g
  hlSNOBKqDxjtVM5LAZraLph8umkqaplDgNiTvvZbDHhzAmAo2bB1dxUI86Y7REFJ1BQv6a5C
  lng0o0m4TTbwKIIyXOBnJHdGJISvofgzJD5lLbZ5guAKaxkTZrDVAfgQEB4661mHqqKoodHw
  cEIssSmg0hhT6sKCGVzgvMTWnPvo4lqJ5bwSE6LF96qtbJHk3vNo5KhCBMgxF1Kcts2IoMTz
  GUktezq/EGlAl3FykHy2EGuI+AcIAstpjqpOcR61CFJMLftq1VJ2FdlmZ1D+TyJKuZBUMMVc
  bN5OEUwm3m8ASs56FJJ650yGX9y2j0S9YCVGK08fJbOZv13mmyOTT31O0kMSaqiWzqUnIuF3
  HV66SZdvoVUiN7TWJ7mzlXR2HHYyBTE6lo5J2KFqKa+pwXz0pY6gmUJVK1scgM1IaX2tr5iM
  KkSOILZoqx1ZiMl5fLEW3o+zhGvG7HmnjNv0i0HR2qrtMMs2Zqc5aNOwAYHbmAFyaLOZ6t6h
  kKNI7Xn3oOLi3xHZmLorXCxisBwr4uYNeaJ5AJd1EnEHyxfgkH1TtpxVb0Y3VI5FOtO978jw
  g+8C4niGcY4+kvIndMjhyCiUnyAVQ0NLp2rmYIT+wvs2lHLHq8H2WVw0zJi9xWlc5SZfObp9
  Xtl3t5VH/5YKZQs7Ooee7eD0EtM7JjXOtnpbWuoJ9mCmHWZhlJalO2NnjXL8jpx5EqPEdXql
  mK6zXv3az2xYl4vZbaJZw7jdb70woOlewOaK6q4KZwmZy+7QF1+6n5mOar+NfFP9ev3Kj4k6
  xtuDybKQvNBF3cyBJGp4PjtcxbqFeU40taJBg6eE7juHo+d9BssBHniCd0D+6L4nZ/4TeGoS
  m8ltnvSuG/t2LOPpRx9ze15jMi0o7rbvA/05fMDd4eNeqepXr3LaUJ5opZjDSfWbXNUUT/W5
  130jq5IcxJXVkmatmfKlMc/v4je4zTcINk1vOnYp4Afs9idw0TcY7jeAX5c+WIeAARhfCGMj
  igBQawSA9sdWH7ZjWOZ/0ed9vJGA9Wd/DAhjDlhMalMQyDN/dcQRGbgbeSV2rHVMULB5csER
  MLh5YlJI0ZB7CfGCMzgVF+h/KlgYLWcHO8iDMjiEcVGD0nKD4pGDRlgVPVhxKgCFuTCFVFiF
  VniFWJiFWriFXNiFXviFYJgAADs=
}

set pieceImageData(Leipzig,45) {
  R0lGODlhHAItAMIAAH9/fwAAAL+/vz8/P////////////////yH5BAEKAAcALAAAAAAcAi0A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987/9AGiBAFARJgkEAoBgO
  QkMikRGVMhdS4nWUDDyPo6oRHOpendBsgKreHtql7pf8cQKUbvrmPhQM+SB2SmwAhUsMXneH
  JAECBANzentwkhxiVXmTWWxZbpQjWZGVGEsEpkOjewAES3cEhYGlqAtDpqyeq38lAbZrLX6F
  wZkoalOpGZiYsZu0n2+du8wswMLDJ6Wnvi13MwJEA6ZSYx5/r8W84cW5i6CO3i5RwcYs5y/c
  Mt5e6Y3L5/7+1jqogVevBTZzK8ahieFNCYF8ATGUoybMVrVCjnRZgsX+IB8/KgDGhfGScZDB
  Ygmbifol5aGWNP9iOkPGcYFHkQcwmuijZMC7FpDcQVqha4BGF0rcFfGGc9IqW1CjSs0WUQKk
  O6KOKrhqtEQto+C0rVASiughslVLkD3XVIOdi3Dh4uGwNiu7rVa8Fgk5L+FAs5B4oZ1WBCqk
  VyvdHrQY96nFu6R6NdD6RjKJWuXEqvDYF0WUtS/9qnvkJZbjqagfp4Xwt1me1iLimdOc4o+h
  tpcFZ4SGouHUw5AVyxwe3AI6Vg2kJLecgWJI1+CMinWOW5M4s+pCn+CsJnpxiYmExbw4dwPs
  Z5rPW6CukqlJBexjH/8uwrcjxKS1c9G3z9b+VdoagKZFY1VIkZgFXL3WGVcHWgVQTgNG0YQ/
  DSIzUxz+ZENfB9wZVYVPnW0gIHHddbCWgprVFeCDf1jhS4FlgeCNY7Nsxx9U93VBEglJ9cKK
  YUyB0GNqUzV0QTAiRYGTch3pB0x1B4wYipQ6UihjQTZSeJ8TUK7YyT6NJCXkP/JYaQGSnCzZ
  GUQd6RQBaB4aSOWJihg4ZlSlmZDPfamJId9pvBz3kIdXNuJcXI4UceYS7ymg3xs1TjhdKx85
  IKCHMIJjCpzBXPqBlBVyOJymXXR55GisZDFjoSERaYsfilZQYHKQMdnMpNc9cKJUdQllBa92
  dmCHFeXx2Iirpnj+FCqC6MhzSyGnDMWqIRFaQa2pWDzV1QJkLZdenk08VRMn30ilI43f8Jks
  aKtBACNvxhJ3X08hjPhUG+B64MdUwfyGraPwRtmZrQBHUuAwYjD21J5QFfYKtBpuKAFnZcZa
  n6FoEaksCNi0koshyUbqgaBMvXoscxUYFU6SFtfCDpu0iDtMKKkN6dJpeErTnEz/rlcumKjS
  +8GXUmECoHlq5KcPSheAxnKlysCn38EPGLhpG+bg6YiA9xIsURKa+udTu8aBUwWyHpH9QMnP
  AtrL0ciEHQ6eeJL9R6MQ1uTyFtK9dlvV4hCp8ik4RzXrHlWG4gcjRDwWk6aU7fxr0Qb+wg1e
  0phfGMHht24RtcC15kqrLBQKyoqUpId4Adsm95zysU8iO1uliikcaL/piGD6yS7h2ZwikTCl
  zd5YBMltIm1x5mpDiiCrs+RsxesxcdrSbqGBGBF+ouWkJF1Xqs9T4IfQUs/zuegCG5X8rzKF
  /W7SVXDgG7CuO6huzcDma6FUgfLP/akVC6Ah1IaIfN1uQreY0E/wwr0sOM9qrgqfz2JSvzeR
  aAlquI/+Loc18FUuNhZRWAhPwYFvFHARUYvf8RqoBQ8qZTgvbJwKV5SaDYqIdxoznUsspgEc
  /kiHEruAoW7UE/8M0RImdFQhvuCKbVHKJo9ygAMNcyOlOYb+YQ3zWtPaV68LBippNxMIqs5B
  wAi0YoewC5whkAiuYEVNGjAzYwsnNzuAOI4XMwTP/XAUxInZzFXQ8gkVbWgcnPWPj2V8QC1+
  1DC6saKCB/DGI66gPn6ojBtMuUcSHjmBtfDxKnd0DCQckkVCTuB96tAX6uyYiMnhrXsdJCNM
  vtiSzFlvgotYixFUtLFIok9Xc5zjKhLVnYeQZnb5WZau5DYVU4rPYbJTGQ5vQhdmHvIxkHSA
  JBnpo0ZyUkR8MUITn0C6KIWFCZIUmSJb8pgdmo19MuRd2m44KlF5kVKwYqfxvES06BWKlo2z
  ZTZ1CSETDuREvtSiA4zGC2KGrA3+Gmxoqj5HisIZ7n8QwKLs3APEKC4qZ7vDqHEEMD/ShC0p
  C6xdJhuKwWFiMFG3SSAFJheFlb2zcUnATAtnV0ISye+ejTsMu8KhzHV2h1rqEYgrboLGranz
  VOI4GL6sFTA5TjFV4+POLfLxNglWYDH8EqmlfIijqOTjj1dTXSGX18dFCTAZkSNFDNNaJk0F
  bp8S+Ey0pPmKfOKxX1jpWlul6FN6/iMsqtpXh7IhRuqFgV+tOoVFRgZUf2ALRj+yDUBvlxkP
  DjY5FoXKU5/p0qLJ7UN7nF02XSE4sX6Vj3xiXUp76KFVUmiJYlXjEh9WMd6uIkwxbOxwNnCT
  9/n1IO/+S9YtF0Ucxjnwe0zrKUAuEhMhqqqSIMMcrLKaVAQxk39FnQxL99iTTEEzlALZkjCU
  kk1FWtORvUikpCIESKQO9qxzE8woMbFf9+yjvbYN1vWImR2akbSOjQuHfNHzjTr9khzF8Kxn
  PyveAaXWXEglG8U22UoY3S1QiWtvH1KzqtqxVrQea5XHUNNL60xPwGXYHR99Jyyyyo4qGICI
  0hpM3WKSpb0FlUkia4q69bbNjoipsUWNtB/vWS26k7hxDdVGsaP2uBOqGpoOlRveBsi2dQ04
  TAQVKsQlu9atGU5zXOUq5WZqgGKhxXCBgAywx/10qjwE3XgYq1QHx2knFlH+L40mO7ILb/TM
  CR3QjYGR5Q9oVLnLLZtylSKKbY5ZrYUkY6TTW9IiXo2kiAZtmy+KNEOjDdM55plwjwXjRO/Z
  JZSVy6ZLCC2mMnWNI6ORfdsAMYT00MaLRnWOy4tbn+irw6TBCeuIVNUjgUWQYAPEfubmzW7O
  mgKDw+3jIHaPDJDyYQH2UC+67MdzGNuepfixl0nkGAo3Q4SEQ0ItHShQD/SksqO8IWPC3Wtn
  pjow3xDTsbUgcCiaupTuDhdDSULuU6JDxt68tsOLjODKJbIhFEdlYOgMRcs6eil55s6rZSqc
  3YREDrkJKL1XLvG87hmVHn0mwIsBFgoV/FNb9nf+BaqkbGA3My9Iw5QwusUIUJ+UiEeEsBda
  aOmVzWYw9Nw1ujrxmRGssuHaJGiTLhi2hEOqT6GWAKMGRRKyC1I6dfgG03e3m6V7nSyXii3X
  kjim1DJZRmM9OEgTvknBLdjL9rqygdq701q3UEn4yU9Ph2kvtuxQBOGuz1jvGV/iYhAgXs/6
  lpEEXsJjw/Cpi1Zae7qlUDCeZo+vO4l1fkO9733IYMXRtnAee0Cm64Y1b1+6yOdtMlUDdV2s
  pwkCnCEFu5hMwa/s7VeUe5h/BuW43/Xj8qn6HGLdKt+NJtCFE54MpyH7Ny5WBoQsE3QvQe8O
  5jjxWz2myjKGjciPhdD+D04R8cMSe3EZkfkrS2e0spj13jVqd8Q9wpBRDIIbBXgmxEYgjVFe
  SHNGyIJUlMUnDGgyYccA6xcjyQdUj3FKwWCAg+cuH6iAtXdjdvCAgwNIa0FZWkB/nEMXPmcu
  mYcI4Bd+ZCYHxdYUKpMEmldbOzIByvcPb1ZFUnZzqVY9KVYNP9MH9lZZf4cFQfh5R5M4y4eB
  7EeFPyh21iJ9ZFQxiIZfvHUpuKNullcY1MBrrUJNP1WDqGGEAcKGi/ZKAlM4C4EXYkMIlHNu
  rLFlAihs6xRnRLJmFOAEiEF/KmN/qaZ8/ReFhfFUthUJeIBBBjN9WsiIXuVyLEUixCRfsrH+
  fje1YN/WZjM4YgJoWgu1dLiVhnCSESS1ig8RWE7gTAlWilk0hCUIdi23NlpwiyimJCYShH8H
  c/h2ihnIamg0KrLIh6M2WhOjcheEVXfGhdOVZxVVinJYZrRYNNYAd8UWMgvzY4QzL+EBdzOl
  jKIYasqCO4ajbbmYUUSDi9R4f5XFgpYYc9zoYBDVE8dlenWyLBHCgAx4jRMgchcEZASpiaIi
  dADZKe0oXgq5kPk3DA0hEjgoHk+AB0OXhZGEiLQCkQzoh5tzT094K/jHkM0GPYoIf/VYHBPZ
  EahzkVrikhKzkmQmKyIpH170hMLoWMdniQhzIOMDLwWiPhAwe3tKSJOBwIUjuBNKOS5KB4wq
  WY8/mVH46DmmhxtGCThIWQdNOZIi+D5LeQxiOZZkWZZmeZZomZZquZZs2ZZu+ZZwGZdyOZd0
  WZcYkAAAOw==
}

set pieceImageData(Leipzig,50) {
  R0lGODlhWAIyAMIAAH9/fz8/PwAAAL+/v////////////////yH5BAEKAAcALAAAAABYAjIA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHDYAxgBRNbAuFgi
  SUfjgOGMTpvRJ8p5TaaiXpUTqxVlu4fqEa0um7hhVEBAFwTib7s9Xb+P6nRlAICBDIOAKQMC
  AHNoeCFzfY8kinR3lXt/gIKEi4aEiYCOkx2NBASKfqQfg6cCB4wEgySLrpwAp4Ocua8otbJu
  qxuRgKrCHId0sIiadZyEz3Up0Mcev7w0wTKKjK+xc7S4BHZRc+KDAeUC2L7nxi+dyjHxvTLa
  MZjezCKE6UfEfdTtM9FvBj0Z12TVm3cPRqQndUZZE0fvHD12JwIEQJX+ziC5bgtdHDTY8MVD
  BRFp0VvJMgXAktMAgmN46tTMFl0OSXwxYI5GPvL+nMsi5VSbIxg7wGkQ6V0aKSrsbBx3E148
  nk3o7HTR0xLQkCBYir36YSmDpg7MnrBTqasMc7xgQtljpNAMn7wEDOAWrqbfv4ABz/LATVeD
  tocXDdoaFkDPAEvAsgBotwW6ZZVjUKb2xx/Rz6C7ld2UWG9prdPs2AwKoxMMdI2SyaWk1W8q
  VKb5iQusZkDgpBt+MWpwyI1wp0LHrRs8D9SLQ5tnj8i3iUAkxhkS/t5ekzmHgp88MQC/lo4r
  1vD8VWXRSuN5ZzhrA3b/k9/Y++g1HCee+cD+/g5nNCCVERqF1IYvzrmQTDEnKcGSbz51FhqB
  noG2nn7F8CfeAuRtEOB4emlUyWFZnKBLXFyptpctC2I3ml557SZibiBshl8nyGGgUV1uFEec
  PxtmsCBpKMGIWz1DwjcXZw7Gg0tXNJ5AXTx7oQbJjWLleMFLGvZIpAZJViaVJTeFGeSVfmWy
  Qlt/OZYXWwj6lgtSNYUoXQWWGMGdYCBdmZ+PAva3pVhIHGSmJCNspiVt9EBmlE8ufgfNm3oZ
  Zh+WI31AjAOAgnjmoCwVetWhXonAjV+LrSnfdlBGaUZCUdi2iGQcsLknYIiBqqYChzjQYFZn
  EsOYjTjmCRKxHRL+Rk+kymYpp04mJGkRIMpNt8kauEYhio6ILtArU/1V4oawErwkpyxNOXNs
  IOdG1m2zRsmZXwmr7knMosGtgyqdNUFGa62BYAntBYr4pkiXAtaFcFYG/0tZOjW5a4lf9+6G
  brIbTDlQefeJ0xW+GE6q0CYXKnvurRG7OgF1C3MoTXhlsAzBvb9BOdVqbPGWoaYsgQygkU7Y
  +y4rvzihHp1d3VmBLr6BNC3EkSm9TJ0xl8qhwp7yR/UD1J2MaiBfGzlfgqX0/MW1yN4c4avX
  ymm0M596oGcVMc4pTkcYDFn1rv697HJIenNNrVGMpCOnKSlPRaGbR87LbSBZDG2qarL+3Xrv
  q935s9cR+o7DRKIWW3dyT236/IlRYOmRsHepGKjvwQ8sGPpfp3YndmBclp2llHWkKdbhfP88
  uGBwSz3BZiODjaOQZBcZ0kCYnOZ45dbhGBlg62g8FTHGNwFh53VutJcJpgwvtNUezJG5FFV0
  bqlubQrW5r8R+GTl1XijhDWv6aSuR44AuRXp+uU1XH1JAysilviY9TgZ3Qd4pjseIX6TDHOo
  BBqyU07zKgAQRyjJeQtJhgN25quIVK9Rt2tdRZLXvU8UMFV5mAoXUIYzlWEoYgTKoZ5QBLq/
  3K4VFIugIbrxDs6NZ3/+OQK4DBeBYqAMcqq51QaZtxIGYiD+HwS6EfCsCIFOFAUVWTxgjTqR
  vFosD0xE1NBCBvJBBfjEMU3snQZX4qgarqSMyrrZXxpRgtugbIA+RJ+k+LTDr7WwiZ5RV10+
  IhqloK8VudGFpSBZhutMoDqWw+SepkiwsXCRAhoLVT/qRD8LrLCMASnBbhg3QzB2p1nGSMYV
  ZMerNlpSAtTKx/ZwpCg74rED9ZKVECmADhoqJDCyIRohZ+edECwGUmBkF0fYEjdQqaI9fgCi
  eNyzIRFOgHoRQxylLIaXQAoKA/cZ5jcxtREv8lB4IjOTKg03TYiNzFF7yd8wrFbBWiqJe97y
  WwRkuSCKjUWP1EomMm5HvEMu0Zj+wECF1zYGJvDZbXaOA+aTLGrPOjVNnS5UU6VMo00kcAN2
  fODLOsFm0FUV1HYsJeU5LUAqMepOYMpp0XmI5k5UCjQEfZijXqhzQpAG1GrbomU+gFXN8Jhx
  YouRwrIgc6qEtjE7evxNRglWTmNCrKNvcmjtusMvUtKLIm0Knb6aqZ8qXcGMSCjp1AZTMLZ2
  UY7hs86J8mGTHXnOnKW85I1A0CqBnciJChGrkyhF0Q9QC4MTbCw6U9K3QtByQQpYKp6Up5jE
  wlSTcOnfOjal0VvZ9YoTg+g43Eo7yWUndIWDbVM5ALXbuSdlH9WU+ExTi1nItbevKJh1TEeM
  lAFXfXP+xA1y9WDc4E0WPy/ClHlaMaTwObanYXLoA5pyQq3osqg8SwkoaAkKXVoAeXlBl1BH
  65uYntFD2kHmbCnQVa9CBrmAVaZtvtgvGyrLaWFMJDn8ewEgcuMXerAIcnVhpPkydVr6Wg4u
  ElqX93z2k0fFj0ORtYlOtPd7mDxtgTGFYQksARV2cxvSZFHiDAesGMXqsE0hMCWJVuKYZHyq
  RNfb4hPt6X0VlSFvQvexYNZQu/E1oFFNvIiPsstdUIPhRHCmnkDsyFjcVcjjXqeXCEeUwrGA
  pEyXjBkNwxNZE64O8EaJGyTfiMw03kis0GVPK0AmLNIVy5an286Rrtdzh9D+K3XIPNbtSFkD
  yMXvvnCoSeIJEkxZxR2BSxG6Oj6qdDUSLY6IQqzCoZalo51w0JwBDD1JQXGUBZgWdSfni1Q3
  FwUioyvgXNNtlQB5GXzvTTE4oUZ92hmKSCwknThSUQ+pxSiJ9NiskT2GNu5B8v1pcJr2mTav
  Bbby+xoIZMc4AS5yphHAhN36F2te+qsnG70fMAe7bqrS0Z1ui4ft8Aw5bV31UteqDlnkZqwC
  /lihnaSW4Qbc6dj0OdVl83fimK1XSVsBXfx1tLrha6xvOxcSaj0ZEDNHWGerViGfrESr+2Eh
  D58Q2QvgMAnzdd+VPFySh3rWpJeWZOXKgYy5xhj+Aj2u2oLtvA8PKznJbD0aZZMS2Ype0Woc
  IVwp3ls/s0NdKASsmEV6pmTBibpqRbwyDELU4jMPuCfT58UOSi9UW3NslRkZWKV4jHHYcmUu
  4Pfx+alaXV8/trUsHc6wr9Njv+AjU4yOvadjFSTq8jt8P/ak4jjZX9rVX93tTtucTb6ejHoQ
  T4MqyxJ2LO3MZuS3UT5QqkazzuYwGLqBCr4qKCruqykb3z/+sT7maXO35Jljoij4s1iUOyv3
  UKzdBB199iXYQfRhbrcNvn80qqwOrh+swrjpQioEzmnR/BgnholRqFyTFx/xfikW+buqhhBV
  Evij4Qv2G/GobBZzfjz+PPM17Hte1+HNXu9TzvPAcHJLjgEpH1Ml7eAKaeVDwLFQ7Qd0Lgdk
  zAMkFXEoWVR+4DJ/iVIbucd/7jdvZOco3AUlCGIekPVY0kZxPfN9h4QOC9gx3UCB/Nd4xBd9
  54V+JaRwWiVZF6AHLtd2w7BRe5RVMDIczEdy0FRPiGd4M+hyYeNrtxYqFxhr3ocpq8SDcdQu
  7bJVZKcaHURUaDGEToQObhNaXpSFSuhTOpdHv7F/ZVE4UWiDN1iCnQRW+yWDzANgiMc5A1Z+
  0BE1sxaGa9Vkv0JxDRNPJlc9LphZY2cqbHh/G5iAqAVGnwEhh+gtT0I48bdD1OZMlsCHVOH+
  h3EBJQ7leHqzhE8GbhkTRTdIel03e0+Eg0ujdXvUR280IZ8BKbqHOruVJyaEOGuzUF+FJVTl
  gNH1O6qSZzfDdcSUTvZHBZBCi5wGJ2QHeOk2YLWxGKtBZtY4TSOIfqjjgsXEHUkzdZP3fxzU
  WUMWfvvUf3sics0yK1KYgRXlJL02hmxjZluQZ3iFjBKUTiTwffhBaHDzjhNHRZrmfoHYgTTU
  i7RBeJmEhEvDYcYXFgyZkFQ4M8+XBQq0feaIMsdygezEO/hoHu8kfD0ziR8xeVmUf+pSQE6A
  f+lYfQLhkr4IUWrYcbD4RzNGBVYwUIrRVL2RNyrojLRoDil4ezT+tAQHCV96JBAVYlAUWGvL
  YnshKZKJtTI7CQG0RGNXSVM+dnnCdmZuaBsAtXk1J3F0yGS/53SqiIhhiTJjGTuJd06xoFdw
  CTlJaX5TuW8Y0pY1g4Wb9SgVh5GbuFMdF5Jr+RVTaRENYSbakJVqxCSCVW/UR0eRA4ebBTTy
  V3Hx5pcUEFSQeIQP111XwpettYyF5pXhmH3OBoZUADRRkjT+52CSKZS0yJl16ZUWJDx91ZaP
  cR6R5490NDl5aULCiJjzxwZSUSBsoHJyMZyZQkW4AJWk1ikmqI0H1V6mWAF+9HE1uXMT2XOf
  wldzdkLHsmNNAR1kpSdEd1deWXiq9p3+FDSQn+YvEBUi8vlrhlmYzkkm4WmQUAViBUmMuJSW
  l2ebqplmWOIuyCZuwDkxCLdPpGlAv6mOdacyJ1ZupPaZbsImvcFYzAVHA9qegXRT1scd3zah
  GWpaObln+Iii+7memYULDZpuTYMlLCZYIuo7stegDIii0iljkUeVFVqRXQSfquUwmeF481Mg
  REad0PGXOUqYtWIjFUKla1lYZWU3xaCK+0lvL2qZgWgm5GYOnRYePvOlrghKPLqlprKmfcCl
  zkmWaFqFdXlGnTYKI9KZaGqg4XaEJXmYWACcTIRvUymnX0qnnMKEd6qalzmnayioIEoJfkqZ
  V0qbzriGlppVqYvJGG3ABh8yUK+oqUIpqaHBExMiJaIKGpiaqpeKlZz6qU+xlVgZqqxaqtNx
  qlyBq9Wwq7zaq776q8AarMI6rMRarMZ6rMiarMq6rMzarM76rAuQAAA7
}

set pieceImageData(Leipzig,55) {
  R0lGODlhlAI3AMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAACUAjcA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBp7gADgyOwE
  nlAHdCqdBl7LpiyZ1Xor1msjLGaEsV8XQMAWDLpp1KC9XLMHp3bbod83+gItc2xwcSZ2d4WG
  JIMCdW14JoB8k399gnSLJ3ZQhJqSbwNXAQJPgZKlAXN8ok+rf61sLKQABGuKnx6Ieri5HXqj
  enmXlsQMlSqkvL4ic7UEtm7MIo4Eqgel0KWo0KR8AdDWp8fgBLIrT+HbNmRPNo16kdMf8djA
  qMLF+cfGKcptZWi0CyiDTTht42oo6TFA1DZSqhKOMCguTLZuqaZcPJf+bIA6ggUBcZwhcqTA
  XjUaBfNTApBFl1aQ+esDMkZJiTYFHDSHMyTKGf8WZOImTmS5oi4RsljjytGNO+EG2biJw9OO
  oAqGtrzJtevSPj9buFH5igZFdT2XdrmzA8AciAcaoaCYhMwzW+2emezg1o2if4r6LmThJtuc
  siRLzgCwVpqOvmyuyB3WtbJMXYf/al0gOCw1R2/tsHMTdV0MUngg+sWB+tkgbfJInN1Juzbt
  vRuYRlTEknMqxClKrWmYJK2Lf/dkoD7w7w3rrrEnZhxIPaZxDLucFnOQXSyhaLhhwHOsvOFh
  nqsXnz84PFpNejpty5cfPoOqj5Rw3tf2Xjb+OIji3DBedC8oM156WxwYGQGD9PdLfPNFeFB9
  GCBnmlB7WejgZ9lgVUNkplyXjE7OIGUVFgvS1h5wIUSmoGXAUHhBOvzlVwV+INRVUykNQeRN
  FYOpIJJAJQEWQ3Z9vMGgjBtQVF11aIWgoY1j0BSCjg8kooRUQHoGAmlKbWFQEhhBhuAKdjyj
  o5qkXdgijHCKaAEp0NxCpRnxMaVLSc5l9R0inL3oJQdDilkSRmeqgKRItUS2VZyWSQnWnQtM
  yRefa71BR1lmTnoCgBhteMh37MWHHIHDeBTqRdEUV8IurTxxl4pTwDMoBY0QiIgiuXrwIiR+
  DtmpSKiCYKGjMCz+Skc4BnrXB1p0KPMqpJXdOkGvDezqALYc/MqWPcJ6S55sEIapRorz2arC
  ehiZc1SrbuZIqoS1palBRCApky9A256oQHF9SiAuHTcpC0ixHlCFInRqDmKtBgpCCxWyIxg8
  UMH3viUFxZXy24DD2baGa5IaEVzSLrUmWcJbOy3nAlT0woqCHS3rxCo0DomqQZrWPDkFeA/H
  lcq4wW7sLqVZxfKTginTVBrJTR3MyE0I41OZqkVVvadIEoMYr7zC0cueMkHDQ2BvCvwDEtrg
  +tsvJPWazWw8sz797ZUgWhf0BuzGzC0JoM7t7k5uXWPCsZDq/E2dOG3GnM0EeZg2hIb+P4Dk
  u3aXi97Nc3sKNsYsDAxWnYd9eihCYBEtAo1it6y4Pns5rnaVJlmI8C517zQI1ohirvsyn3O1
  N3ZgQsG7fH9/1rlBz+LMIjW888x4NaURepR+HD9+NAMNVjl3BOPlHo4dx/etosrBg54CIrEi
  Hk9pWjdJE+vGA8Sk9Z2UkyvO08kJwWWPc9PsuJe9Y4EPbogCCOMwR4sEoqseHxhOZsZ3GGcI
  qRrAEFvyvoQ11h2ETt34UQlA+EEG+u5rFCge9kSogDlsTyga+4NrEDa6CLlwQserjYXk1RCq
  WTA46Noc1eD3qDbIpznPM5ZRNIS41/FjHx0ziUqKsTYIZmn+WUti1OA+qBP3qQo5xmqh5mCT
  tpl4ZBety6LqEnYUVUChZxGhoP/uNUYP9uw2TvQN8Dr2QnvwBE96YoDMIGAhsVEvGoZMjhPE
  yKcyogAwmeMK1konHT0cMUamg8n8jJIbz02OY/0YoB7vtjEjqpFYnHNfBlEmKd9VRFGkemMa
  cbc6VtXFhW6QpblIcAvdle94dspNiM6mitgwLyG5xEmDHIQcV9bmMPhKpMc+oMrsTQsSUYPR
  JLWjxPm1MZtswxsgAJAdcjKKA3UhZbCMCUUoMtIdEUAO6rpyF4M1T5GXyiFahkcB6flNh+68
  lO/oxLldTq1psdqdRuJHgWaRgxb+QvnPOHQjke5NoJli01BG8YlOy/BzAva8ybHKIRpxcm2e
  AGzG+JRQJyioiaU462Z09miMXXzMmqUc0+iGFRliTc+UHJWf+Ka3iXnRi4S/w2mTBgo52rgM
  FAZSQohSkYQGJTEDwIKhOeTRoXG4EDGSiwDKzqek5c3GFsRx6h4fFKlRUQscWnzlm04qRCu2
  hJwtbQWz3KimNcyVnXczBjzwdD80YrRrXOEiUB0HsYLWjKEVMupRSUq4gBIqhwRlah7nZCrN
  9XJC5UAhBpxBHjqJsKsRfeEt1njFIILnrM376RgH+wHRqbObb+WJ3AzKVkAgdpqAU2CSTtlF
  4HYrqwH+5IgxFMk+C1gSjXd0o0hFwSBwGNayWB1jbe53AYelkT/iuI1xj6vWLR4khiM01Qmd
  GtrNcscRgaSTngAElwau4xbBHBnM5qYpWaE0hLS44YSQ29u2NoNadFgRL0IVghf9VqnUBMix
  6jpV92YrOc1MrqMyfABaVkAl6FHVvJrYOf4ZpDkRdmbNLNwvfdIrG2BSLHcvesh2FXTG9uls
  ZfFYIw62qsOGCUQD9UQauPTyqu81ZTdu2Ih5WleBEJnVWjdQTU42GMFGBNBweUvl09U1nJKK
  THY8Mh6ksPiJnujOlKzUHQuwj2xosW6jfMtSZdgCPMj56OOGWtkzL8B8rXP+YeDGN2Xr1cy1
  T50WOFgGv/MWeY4N7aw5n9wXdWiqFrRgSo+di0X0SLRwm/uPKTCtZDBGOE5+ZuTVjLLV15aa
  mwlbohdnxiNSL2ushYM0jXlRzgGVMztOpGsWw3uwwaFrUUpsnWhndNbW+SjGOwkq37Rrmx+e
  wLbEEoGm1ai//e4ugV+EdaQfyLI1RBfKcHWegKd4ajiluspwKzaNDkTSZXeXWnoW2HfPq20s
  ezSyYLmPnU3rNBjDuDhWirDYAhnrZzjTjp1y8SnxpuJSQfaypTqkP5c0Eaxh0zIROQsLOX0W
  R4CwixJdUHG1URfZhtndX4LEr+ZsSUcnhcs7g9T+xZfaFFXtL8SmqKS/FYYBl+D3xMUFhpZX
  bkAlVnxufm7PkqrdQbNxwTZ5brDEHZ3veNbRhHisJZ+q86ud/0uBd4yPyVMO4/DSibpq7Dq8
  E34pqKjSRHTpBF0hCgJsizu4mlS6lXEbD12yN+ToI96zqIt0vMMVYAzyyDjxtvXS6BlAJTqf
  S8aJdWnnHNPGm0KjzJ5j9rou7C/X63fTOd4K7KpVPbLMW0Rh7td2HchxujxUQkqTGisL6iZt
  X/1uC/hNCt4lYlf9vntm0Qz8urgOkSTLTO5J+PCZqCk2h/g4QQVsKL/zrZ+2hIfP2orVOPK+
  rOztsfH0NCIZpEZcFHX+DvQGbQndwE4Qc1es4Cdw+hZHfedYjEN6zsco5fRl6yeAyzdyincH
  IzUQ9FZ9EQRtz/R3OQdMXNcvyoZdSyUf+ZUCbzF8CeWAnUCAEaCA+8aADeiA16dDiOc26SM8
  MTgWEoht16NrE9A/efOBcrBXL+VfeHEXJvg/led+OHhhhdeC7BU+JjBBKkJJMSeEqtIXgYGC
  0caBBRgTWkgY0dc+4xdyETE1y2cb5WcBDTKGh5ZqDyA6Qwg7FFFovJdB/NNvlaGGOXhneKFL
  olcnfjUCArZSKSN8uVOGH+ZaY6gvciBmK0V8HeVz1WAvF1aEAwZmdAQMCBd+IwBCSCUOdbP+
  EXaoPa7jM6j3IAPlM0x1hGYIHea3YI1Qhaj2PZLCZ4WDDsKVQWUmSk5XVQhGewxHKAVliqcI
  giK1MgsGiYJEgRJCd77yZG1EGn04E0t2G7QBd/YWWf/hd9jkHhH2RWVHNd32iW8zeAe2IPZ3
  jIkDfAFYZwqki7Socs8SeNXIbEOHfB6AGku2i99Ia60ATWSTXi4Cg1KXRsr4Cy+YJDyYXtFY
  KjzWR+3We4mAVvYjjp1Uhy31b0Ulg4CnBIwVhzcnV/BRVm5EOjBYfHD1PuOEiw0JDLtTC5AR
  ffRYdxQ5dxYINtuHY26mLpG4fAPZZXnzEvF4KbXAaBBZVuLgEXz+F3xYlEp5AgmYKI8m52ty
  ND/Nd01e8So4aY65BYD593RuBEuqYGt+AZHk6BDrxz5JyUBLeQdNyVlq2ZEOxJQQVlsKmGiA
  UxMB+V07CTFEeUkjMnSfqC+MgxZKAhnq4BpxyVlltSiioCDbBI4EdDWPZJf+9i4/KTBvoEqx
  MCJoWJnjVm+CyZKvwXIm9m4pcjlmVZqHmRuA9jsBswKrKU2pmULtFw2PxILfFTVq2Br8s5je
  uB6cgFsFqU1geAjY1prr4pegFXNwUps9CSN6x5leV1b8KFIRMYWxSWPSWSSYApGOST70AjKu
  aYURkngdsEwuCIXkspfLBxlz5W89ZJP+EKAsLuWCTUSVuRc680iZbahqkGk1ReKTRBdrQ/ee
  qJhk88hudKiEhDaTB6ag87FDOVJlQbIVaCheL+dN7VAk45hLKUhgJoVvx3mgA3YlErp+luCg
  EVJSKZlLshicdvgi1dFI5DKbiiWMKOqB/RBP/CcBufR+OgJPFaIvwBijqdBudiQhl2iHcNZS
  U6V0MIVnRYRgwjiPs3E/Z8CjWNhhOxqkzbaZRVqPcuhsnleJD7lw8ACOr2lDhHhcN/qgWHiJ
  BPoetad9hBQPkHArVFqVDyKJ6bKmnFZv2BQGtoIR+3lTA3ptefpA13GJeYl7jPVJB4OnX0h+
  Yweg8BFjP1r+ePNpYoUaLNTzo7nyMz/VqZPDpxVookLTpikqLRDghIoFEg0RFbpiiH0BWT05
  pES6NTQ6HyNZgNkwHApKhUVxezJJT8SYqKRSHwMDq4WBnmenihWArIoBk29JkeUIk2cUJ0rS
  ip+hqoRGqmLkrcmYFj/nQahRMiyJZ/XDO/5FWzlIbRX6R3VnqtXWq4pnlPRaXVl0e4kKrkIj
  rV5DrsKpd42irv7HFdE6HbhKHfCJlXjnnPv6OSUJKUjRdXe5fNY2gfm6ga2KL+NBUru1HlZ3
  jx/nCp7RpfEqr+UJEG0Kpxv6hy9mV1uTp7fHkUPXsdhYDQDCU/v3lSkErxXasI/+ybNX41AB
  SINvtTu1tKud96Lnl7KWhqW5tK5D9XNPOLXZdAEoG69CS1iBOp8/WpDgqFESYikXmp8bCrAe
  WgztU3ZqZJTeSH5aq7ZZCgZ5OrZ3Kx0LSwZdawl7a4qi1Y9IqLDNskwl83e/mbB/m6tni2rM
  CRPP2agFRqUcQrcQJrhY2Uji4oqKY7mUmGNY5phRhGCOCbCVa7n9cZRSwDSjW3hi9ZOe27fc
  0aTU2QKs17NzIa2nS7epy6DBaRqIIwqv27mxy0u0+5+2e7xLFJmLe6XG0rw+Y7GyQkgTClJz
  Ar3VMQtAmgNRcBzY233P+70DIb3VWynlK1bXK75koL0poluP2zsP8Bu/8ju/9Fu/9nu/+Ju/
  +ru//Nu//vu/ABzAAjzABJwCCQAAOw==
}

set pieceImageData(Leipzig,60) {
  R0lGODlh0AI8AMIAAL+/vz8/PwAAAP///39/f////////////yH5BAEKAAcALAAAAADQAjwA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodBoEBASCAID6ImS3CkKAAEZhz1gHGq1eC2JeAYHrsmK19JYXHx6XT25vDYGCDIFwWHN5
  KldrAYspV2ICW1eNfyWHg5qGbjCSe5AojWiPoiaklAepmCScC4Rta5+lpyUAXwMDdq22HnK6
  cgfAA8JmArrFhbDIyWmbzssrXsm4vb4buIHX2Bprc98prwqxm7MucWjc3RdXAMm7WTwA6zhZ
  8eAEwYqAzcpt8J51itbFXzF+9ghJq5Gq1g56QsKFO3aukydzbNC5QXj+Q+FCGljgBftYw0s9
  G9rkHUin6lg0QgEVEtRYrWUOjzpw7jBZ5Ywplic7eBxKlOQJlliCutCZUOS/nSqBnFGUUimH
  kCNhvgw0c6mcUDvOvIsnsKNCHmClJlJQVUXRt+NWOOKR1OdTHO7g4TJVAwC/qT/saJFjJYA7
  twadKl4ssiwIO18cQObLYHKMO8j2OmZIiPIMv8yM9YCciLTVDXBTnyExuZVlyY08s0i6d0Aj
  HdrG5r4RClccjjji6G6k67BLxsiRb/41Js6D3w+yEBAjG7HtO3dvpFxz+qi87cB5F+2OIdH0
  8+jTq18/fbnQigygy7oDQ2zx1Toaii5pOAv+snR8oGTJWMlYEk91ImCV3IKNGaUBNbpY44B8
  8fkjoVdkjbSDfuEhgpkj5JlAmhvvxCZOYgwu6N4GSNm0AIUvkvjCVFk5iEhs+9FAjSQ1RkaD
  NgEsVhtP/Zynn2rnrfigQTmutFYDEO5zSwR1GTaYUSF60GKHMLR4R2xZhrBdIEFe56IrKLKn
  JoFdgbDlhE9WONEI3GBHRkoRhHmVhUq2oM0uZSqTSoAvAJkMejUF2ed7qsE1gkm6eBEdfhWO
  RaQHI/r0h11mLjOJG4RSBN9nCgXZmgxjKvQOpZk06qgILU66WYt6snUkoSFdok18t9ZKwZ/R
  +NqBNvo0ps+IXB7+E6gyCuJzKWuEqLkeIcIuYIklsn50bVQdeMkdOea190yqG6lA7pkwnHtG
  NY0k248j0zU0gHAnqOuqOiQMmu0D+n7gLb7D+DfVuES5+wGPh3KbLnZO0VOTpi4YGJA+iV1R
  DIKPMZziYpdwcN5zYu0rGWAN+EWGBUeSGXAsKYMaSWc/EqVFwj6ycGSxSCmqMJ33epTlx3Am
  JXLJJFc2nVItz4VT0g6J2OxI1WJgaIqpGJwxJcYW49QYV6QgHHLTIUcsB6wEHV4WXZvtQNkT
  MO3IVEl++BbGmPqMKlwz39ek04HgvJFzKLCEaHLooWE1vwDLeTarMTbJtgQNFQsPgD7+xS3w
  GCJlSvd7ktMsg8Qbs7Q5CAhPrHXmcth4sHmNijs6BJBeF/RHX9EeJyxlXgh732x+SiPNlU9+
  swlEfY7kw697kLLfbmSHCtYbN7yzBbFCyTi4yxUdWsgRiM6mSILrpQ5jYx7eTiKg0lfQshtv
  l/ye1SRJ8eC2+cZIuIJb/pXr3fojqfVCq0ztMJEOAjIpWaAiHymcoikhxWV1Q3lfCP7lkUAJ
  BjF9i1TfJMGI703uPArs36icFMAFbMeAJQxDuag0vgjRb2rwGNChQPiw6ylvbjPCGktS5L7A
  3UODNFrDoTgYCfZdh4FGNEz/EiYrjnihPRwhxcgm1j1HLIj+OA3qnGJSZT4KuE19LWggEIdC
  oI6ZoEXKqdzeRnCkEpWCLF+aHvUuYg4nGq4BUiTanGZHjEzlwjdOgZAfAyU4WMmhdAkbgxxL
  YCDwtK9pJHjiDBcTpYN00RtanFcmwwY+1VXxHYWZVNpiBMVB3EYNZTIO4lqYHIs1xoOKyZkh
  fUemtNzvaXAp4yK11DzGfGONCeIdibwlQp8MTYXaW5ntErc2NGxlIwcJJDJaljtIDksRT+tK
  1CCQm3lx6pHGLEHs5tUcMmhyEgEZg1s2yU5p3pCZ4NKdk0oZo72MDJ4Y+SGD9GkbI/pyhNnA
  5lBMuILtxDA1upTgHIX4z4FdMgP+0UqVX8pBtm9h5A+FbNzOxrQOhn4RRV8sFkA7gMiDPnRP
  kqNh9LzJPTYaxGSS3F/nboeKJKKIn8VR6HPiIBtqcAtC+7GY48xDgW+ELo7QY1BD8lUwr8CN
  lm8hUApDwKGjuVBce3weiTp1D3LpdEI8nR1lMqoAUgCnXQZjKN7E95Ye8QxFMdkmBEDHQ1hy
  lZFwjWZeaXoL6RjpDuiJTWDRVVGFmeR/KgwGX3xjPwaYqAJG5WEvU4SUtxJFrqtUjSInuyum
  RmuMfTOX5PxCoBeaM1LVeqxGPdU08OARjBMQ4piq2TdvjUVezFqUBbpJvqlCC6cMqiRb+cpL
  WAJjr8D+nGDt8Gc4y9HzatxirGiwIpDDuigdh4tsJzM52YTBsrJiQmgK7DUUiiXwPiet4mdZ
  Gi0VhC2lKj3naG2T3ntG947YoxRQ5JRci4QEKcMdT0zWlVtPVoCuHNtl3YCr1EwWKCOzVMxx
  F6Nb5tyHgTdNpYEhmzrKHLcQ1AXxf5bhmwqTMBHSRPGAVfzgpKSYuBv4KGx/ey/zbpUskVQI
  8ygqqlW9scCZgTAIIFkROsJnOxfYYfjm5cakUWcXO24Gfj8ALBURFoIO3uexUORaEQCSgRSL
  pYmvgrN2gs9vjwryX0YsUA3N8y7WOEiSDYdhdEqVzrugh3SAW7XwqgazMqb+1p9El7D62hea
  7H0gmlj3jS2xJM1Fm8hFwHuALlcAKLnVRxx2AVo8+1jTziQrL7MsTUOv0q4LAqRw9TbmosJ1
  wtL8annKHEgzIzYE9zCOK1VCjP/t2hSAVCWHNRaNrJQpfECS8k2FzEsk4bVn/70HUu68aET/
  S9YXaO7ftm1DRgWwXxfpF0fLMxVDveNrzyQwEI9lprviOq9bbHXbXLzShClWzMm8oQf3LOH+
  DlnDGEbifZh6ny1A6DC9pkRt5LzwVu+wxc2whA7pTB1i+Ge0U/7AvTwLbX3k5bzXMbVj1/sv
  kcfWPr1sHpJdiq9pK5M24UhVBryEcWDsCJo2p3f+ovMNv43JW72oTnWukxpgbI+MnZyc3M9n
  vlyZNherz1VusSSRl73k5UB52cMeNPhQVjrTm5pMdHHmB8eUMvsXroJ0VPVjKqyRK1GWbRQL
  xDCvGSZdvoeib46/0ZD+OKLvb4IoNEUq7RF/A5AEBjUdhwzvfnuZxeSDJbGI3aBuVzR6RDQX
  c5+aP3HJlRid0g3n7QNHQl4Zct/0w8TD/nApa/DLuTV6Pl8F3aJ43Ip57tHvnAdBZxeU1KZN
  6eehPR4NcCiaxLFuoz8cJGq0SPYD8fmGJ7D1eDAGc3qJHHL6HGEGdXYFhtlahrcGfWiYjPhf
  8XE2Wkh3YOAC7K0Hez/+i2EpfHo7NYYUCwV13zl1md3kyKQaLdBGXNV2LlNtReF3tCc1CWQx
  m/ZEjvZhFKMzFkVVRCc2CnY+WwZv0qFJg7I/jCFqpONPYgaAsWVmH+ROe1db8RUphdNe68cw
  1gBUE4Us70eD71cjJrgKjVJ+POhi+3d88iUTXMdxb7GDLLQREkVMK/h0lAVVMPYrJBJn0bZz
  lAAp0aZnWeUvoNdK/mYBIWZXt6IfqEZp4UUP8VIKIJSBCYKCk9RJfQVYpBZcIsiAKmYYvkFB
  GyFUO+aDq3WEweR2mhV+SqcVbvZ4+KMe4TQbwsQdTPhWVlVvmfMp9TVb9JVsCBV+fReHQRf+
  Ib7VcxdIMzAFBhA4h532bIlohuBnUwInEkqUCaYoiaPEAVXhRzK1HjcDR5hFXtsQiFroM+dB
  in5FFDGBV3P4fTazDUAGR9aEayQoiUy0YFNRVWrCdi73PM/4YH5YUmyyG2oQiyt2eus3h89S
  UH6Vhuf4KeU0CZiVTdB4ULUHWJ0YS1CIWWWVGtsYLTIyHwJWhDyTHvgjTwVFSKeFdy5URu3Y
  eNC4dAIEL/NYa2T4MtmILWm2LK5EFr1wdSmyhXUDU+shZxoBdfrzgrWDhE7yjr5kcjz1kN7X
  LrckM01oH48zcqnxfzRWXl2gRfRjkDN0Rl0YP9JSkOKjknfAkqn+5pLj1S4pFiqMN3U/RC/N
  ZJQ9soMROBdLoWHdqHMHMnDi9JN2pyaU9IWuBm9BSWqWJyZuw5Qaxx3MmJGtUxNUSRhCQh05
  5GKDwmpddXZu0mt6qGN+Y3LqVmvS0lDJCDfN6Gf9oWGEhWD7dJhCQQabVmgTNSO05oq2Nn1z
  VCx+EWjy6I9caEFQKDMFGYUsBzN8c4ctRZPH4wx+GDDC04WzKBeG6YjcxpDqhX4r9Jnxw5nS
  cWcm2QZsZ5sSAB4XhDjZ2FBniUmkgVVYI46uUJkxdJmAoICdQQ9M0x+ISJ1kYp0RZEYYdBb1
  QiOq9VquMlOYeXJdtZKh51RNp3+1eZ7+wxZB3FkU2ek2ofl3n8hUFpRQKKA5c8VgjamXHlOT
  V3lhDdKKsRmIbMlbniiIisYipbCd2dcZPIeKpdKfOCJHvLl7vCdCQWcHv5ktPvZ1Kiega7lV
  DIqRzGiiS1Iw8WN74OeO0dCaJRNWxvmOi7c6fhCCgDN3TUeSTsc6JpkSWhCZytCNF1kapFAt
  VYGXRRkhbHdb+VmaETReNtpMPWOeLnVVH+lN6NB8X/lBSedxQ6opRqqVdhANZLCkiPgFW/Ji
  ldMWLxOKbGWPv1JSK5Wjv2CUzvkY1BiUoLJNv1ETL6EFpMGaLQaAg7qipdAymkmapCNeqLKh
  eKZBI4A2Cdj+p1SGp9LyPXsgqCyWg8xiqAt0b3qjqDpHLlG2e2MzClrpQGz4W1LZb/a3JMcJ
  l0n5pO94KsrVgWYVQQACKJA6Zz4RQRWUcRaKQwuDfp1TjgPqe/VCqRVULTx1e9KRNFyDHZCB
  qrnQl50BZcNaWCwJJIXyqpKYmxC0ji6IjjS6CXmDklAqnKqJmzN2oo1Ghw8aqa7SrnlCr80q
  ll6Ej9/5dGWpPyyaAWk5WFV6qfQ6UvoqleTKAnsxq2GJrP6CrSdjHfCKYXFHsNJCLTcJjlvE
  picgrbFKZQ27LE8USRhrp8xAsRgIn9zEO6nmJdt0LuvIHiljj5dQbxE7XgC6kBb+CzvpEbBa
  SAHq4TG/UbBMKy7TEYjYV2++Qzz8ZDIp0xxZKbMBS3z82pANq3SHk7TzdrArUbQsYqQbq6YA
  WAoUOyJUW27RM1FkG4Mwi5FeE7TQaGncdLV1gmZ7W52C17DgCUF1y1atiUU7l7NQJTn1Go89
  47Leupre5C62WDP8iCV8G7jpCKTRgotzu7VJ5ZFVgx4EQqQc967pIVgv5JhoJ7IV2594u6tI
  +QDEskmNq5HTo5RnNqVt0LS+Gzfv5rqEI6+dUE1RSzhXd0oISHywK7ikt7IzC5NYephfZBXO
  O7ho95Rv2SnVBkqa5b3Jaasbm3lvVbgLAr2HZldLa7X+/KR1kDkJWaa3EiCjaduh3kC/K6W8
  79actyp+cNSVDVum10sj6KtHR/i+BhI2lnA06AijHPa7v0u84nkvBqi12cKoUZVbrWK+QPQ8
  HDy8HRJHtHRafjR1xxaoLqSGmGpg+LuxxGtWKAkgHAd7R2W6y4t+WjXAHBkw2jk3OdVpc6O/
  86uQ8PrCIXYvZHG46xK5ONe4w9LC0geq/RvDyWUvoKlSsfGVuqQq5Fa/FGbB89ocT9gvRuiV
  YGOVHYuboKrD6HpoqaE3ivln5AbBvku8P0iv7WqyB5hmwnvGO4inXvx6qGe5cnt3DbpZ7FSU
  lZFHYEjEKGnHlabHlssagBfKiVzad5jFxHDxx2y8j/9JyBRUrehTKn9AGuTWyZBcufhop6qM
  N4B2va2Jyg5bafnZyqaArdfgjfHZybegyXEqsb4sj67qvLEsy2epyyWDrfdIytFrvcbcy/ua
  saIVzXaqwzfMxrFVD+fHtkbTaNqsqfnFy/3ZwPAizTHAvrVkzqIiuNdszfOrzcwJyt48AQI5
  xLJspfeJtT9CzmqozuzwzwAd0AI90ARd0AZ90Aid0Aq90Azd0A790BAd0RI90RRd0RZ90Rid
  0T+QAAA7
}

set pieceImageData(Leipzig,65) {
  R0lGODlhDANBAMIAAL+/v39/fz8/PwAAAP///////////////yH5BAEKAAcALAAAAAAMA0EA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWpuAgBZwvW0Z2YBLS/42ymhHOE0jd21r7rsdkCvCY3TdoXc31mU1fnMxAgOHhwKE
  MACIA4oHAY52KY6IYg2WjnyaAzONk4svho6QoiygiQqSiJQonZgMnYecmp+WrqcnpGSquiqS
  AASGCo9ZjyyJWoaxC8rLtGeP0J4yjwQEjaa/u53b3CWsiMWlyZrN5J21ljOa3+AikgLY9Mzw
  JpLYksUB2NEqh/oRkITuQEB6rDgJTBijkcCB1e6NEHdOIgmK1WytgOX+YFbEBRj/vUi1yaII
  ZvT0IQPyLgjBAMQMztNmbuE0NAdVDtCDEuLHMTsR/tzhUaQOUt6CtASCMZ0vgO70eOSpLkbI
  oTqKYr2RU+jWHCuLIMVEMZeJnFcveZ1l8yvAedhi+rhUZlwPrUHCEhm7KtRGrYADt9M4t2jh
  hzqZBjQijuylFmiLPkwb2W0KjGbBJqpnlwfeH6wKKl4c6bG5wKhnzcAM5NCwVgSM5iCY8rVo
  GgJiicvMw5AAAMzk+YYctLbx48hTMiyBdOluCHwHO3wNygfl2zNyg2y115KWcqdTi+9MojmE
  5w+iW7tmSBgpH6Amy7bB/REvl5uxpYo7gPf+iK7JBZjcciMEJw8E5DFg4E6DZQPbfDeQxA5R
  h3DhHX6B+SfCM3106CEa7zFnyVIJLoAUaTEgAhdJoHFEFCntxWZabxUa55s/loUAoIA8Kgdh
  B9Op9EAqQxYHUYMyDoBjixVRmIiLNHrE34//GdnjlT6VIOF8RKpRVYqI0FNSD1dh1+CNPl2i
  oQupCHPcjTRBlVtd46lYRogTGbnPOgXRJuR6tgVFpSBQagbYHjhsec6US1WJWG2A9OHmcQSG
  cNUDja0zJgytZPHaoG3oyaAOoDjopjwXwgFbcje2B1Wd42lppFwMZPrHrDliUCY6TwqjjVHf
  NQmUsIlKlqaZLSj+6g2O/b0Ka2qyElbrjAsoehEsvPYHyjx4rlImp/15RWpOk9KGEbKvlutI
  Ssw4pIKypXh44ixrarCgaLY2cK8I8M5oJzURUcYdcV9GqJV8KK53DsDXxPfus6nVm8G8+FJr
  Yqog9ItiQPH+I3CFLbBSW8I0bHsccPQAV4rE5dUoFES1GSLjCqmQgaVxvYDcTYl9keyMiidF
  bJBhGoMHzFTFAjapysbIMG+YOqn4Kc0QK43CiedZ/PNTHzztkYVEp9aoo7VVKsN+V0qI7rXh
  +gjzyN+t7QEvNw/IdQZMj3orhPmCMbC+WkM33scfu/OXRwaj9lvKgV8mZdSthLaC1zf+fQjT
  LGNPkLdoXfJZpM6A+yz4iFoMFCzHkjmyRRhPZ961a8bdDWbbaZ/oepVwuZ0PpErqnQIpp/ZB
  lx76yY5BQJeLFrhr84mT3k7JU0A59AxPwrg3dBYqYlG3VzJe7p92r+Pjaa4r+vZ11J3SMXpp
  QOzWym+63XwUSz+i3d6lTDFyeZ+fcUCRYh/Ltjepm9lugBwARQFV8raYmW1nVQMakHD1AAlm
  ondYkR8/OKM5bywQclYq3+KM07+/kaBozcrOs5aWCARuoF9rUd0DTZAb9VFKfJ+b0AXb55QM
  8qxgOcxPcvK2wFSADznzciEFUAi6kbgsG1tIGy5WII8YKgn+I/rbU01SxooRZoMX6sqVBHb3
  p9D5cGZ746GfjiSB1gkIKUVUS4BaZwImppARC5teKa53RxSEpAxcPB3qXNA2QEZtMlF00Acu
  5QCkVFCDRtQUzxS0R4SM5SGkOI4QT1c6MRmtSjLMmQZZAKcT3UxCSrxAV2ACyDXITEy0ctwH
  Xwk3EvpOA2vU4rR8xryh9M0Z8pmAO3qkwLI9cY7S2pB4nLYqEHoEfMBzFtSSgwtQWYotn5Ja
  WuSWtWT27JZDK9EvezgoXBSPXv54FAPLtzIQMqcXR+QPGXAIAla4R3V1kxBUlKPOlyFEl79T
  EgnjGZsPxoabCNrJMSp2oAuyURb+OwHnARY0tpBgKUhCMejJ3tc1UbJlGTAQxxHFA01rdqAT
  AvKOSV8HiySta5sf4GhpZKfDrUl0XhW1xDmfWarYKUmPucNYPX9m0Oqs4gX5aY4Nbec9H71U
  p5YU4yIFWjaC0i4lEsXAlnLhkGBMqx9e3c53uDqiCZzrZsFgXD+pKdRF2hRxRyWkHKckNqzy
  EJTrCtA5pOqBbbo0TDD1wFa9ZELWzK+JdyjrGPGZTaX5NDACaWsI0PSmu6qAsmuwIV2zCgLY
  6WeeV6QeYmYoq1le1UHxbEQq+dDFCvbDd+byncxiqQBeIGqxjL3SVep2LhoGhp4goFj1FHc9
  hB6vKFb+LAVfJ9hSnLLThH1tbUeo1duL8cy2a6JIZN3xNXZhDqV/Ne4Sj8m7Pk6OvMTUqD5P
  4FljNtCYy+1AcKB4SdPZCQBhsOxJafrTj8S2GdeQjSMtgJm6gRdL1R2BHVdrL4jlZlmJYS9y
  nVnTy5Yrffp4yT8fApwSWPCtu6RWhSc6SUzhMyRuolxuKCeQLTHLfy/cEc4QezX09oiWM/5w
  lZCTDzLCV66X45B3APbg+JpVxzIjiFghohtBfWOcEcDiyRJp19wGkseSFex4GHwBPR7qr80c
  zlkm/NwRv4KENuMnhhkXDiSr7rAoosiTG2ditZyVM+IZLVSzLN9r8EjMKdj+lkZ5FMKR+YUE
  7dUdVWPG2Wttd4GUPadKxCvMR+iFNh4LipLJwWS/RdMCatNkMzcbQpFSaooF+h7NvPxl8ilw
  zOQTGKVxaR+BJtFBlaz1rCPwyaZ8c2Nvrtahx6tNqHoKzNzTh5v2POypLjqljf6AyTR70Lh8
  UM5j7mePn+3JXeuKqoVW7YzF5O3nvcYUtHEViX21Em10OrHz0G9C5xrDI97Zk6N2Z57qVG4J
  2BG58Qk1f4BLAa0kN9hUfOr9nvZaOnOAqb9OIU4Ti3CS0FOnEsIRWO+JTU9V8a8vNnIEetKj
  hvrWz5qdzhqrvNILrPxlPr4elwkM7n5yu4HRpnX+P9Tt7oZ6Nhox4vk1Nu1yK0+ayX6Wsoxi
  9PLG9puSdSJ4BQqHGp/gOq+4xgeZBTbzqStcpy1FeDxQLSFIVCWJFHd4nWt0CbhMM+RYZ1Zc
  xFG+p//slC3vshA1uzsce1ftGuipe2MeYRakNTbxPLx3i2d3iPJHDrAD2c890dNouBuNWq3k
  Z1FCEmFImSIqqSHLzRvTZ0EwgmDV9KLkGa1ZHLzEbQ7TllJvTrEr0y+aABs7tpR22ENAWYiB
  mqe8tnN2o9wnaOdXoglNemcT1MAzOb5Tfb8B9ILep1IPfFDis75R2xMiXQdcipsFO3lcnjo9
  ocXl4zSxuDcGR8PYLtT+FFqcsfBR3tpfYezF9jg/u7HweLUwQZYUyTINcCFneJYTK9Z8LBUK
  7qAoJUR5htNgq6cTOxda56ByAHIV2VdnBpZzmSd9S9Uf34ccqJYnazVENBZQ3tCCBNhmLQZV
  2YQwKCeDRBeCqxRRB3hFbpdXAlUcT4JJzeZssNJ4B/BvQ9cJnrcirudJ4QdndfKE0sM/J4NE
  J0B1EQR4QWRntHeBghQ5qhcuNeQ1UigyNwNjFJgyc9RPW6VXtjc+dQNoxMFhMDIZMBJGr0BV
  JJE9zfEvZeFJfTZN+fBK2xZy/pBODvJao7d//NYyKuJYT+MmkjgVQmGE5CQ2nOIrhnQslVj+
  Oo2Qh1koGA+HUruzGQShKEyWJImIbCIXAYJ3JSu4XzahV49SQl6ERJ80doNmSwx4haf1Gpqk
  Tu7iWx6VhXPCfvn3RNoSLoUIcrHxNs1YPlJ4hBAjhXukMXC3aOyDTaGndbAyjbi1KM4VWNtz
  MB/yW8yFC9ORSZimCal4EMEAgWOGd63ogailgqKFPX4HX9RXfc93apY4Afv4RZpkUCBIa9jT
  SQICWtrTftMkAPGWegt1Ls0oKDGoY7dXJyckNY2FUl7DRRw3TIAIa3kGAyyGbGV2kBmwK7s4
  Zd8SiPgEkVanDWXySmzHLRMoIik4Yx1YC7imXpgTEreYY9B1Taz+81GIZxU3BxMFGYwnYEQK
  SW32VZQryZGqxQxR9xJUZR6nh4n7JlBY2DH2RmbU4Y3UUzbU0JMIuSikyBZPiTwteUrBwmUW
  pVphuTDLqFjeKEUq6XVKIm64+BtkYAc3EZfsFJAkNjzZE4uB9osx0n2p1YulJ5XQFgJtUmZz
  YjnYcyyO04hfmZIrM5g/MyeB4SOISXL8w5goAHwdOQ9pAY6cRplsZZmkExWWU5N8FjT/6F1q
  uTc8MkOKR0x6uZE7+W6FAD30RRdTOQ2eZ3K+JZvU1JvUuDApN4BDWI6K03p75U0RV5pCIZ23
  Agi+AUh9WSCSmGb2pZDo2WE1Zpjqg4z+WkadNpQFZEg1IshokskBAAKYn2IW54cl3Pk/YNUh
  6TQYA5iZQ3agpFVPhQadw3hNAASdn6WF0rZlJHliJ+g3/GZXb4kaiFlwaBaVprNmYuI4KRgp
  ejBoN0iEJEqZ2/ihezOUOwWbJSh4n7Y37jl69RhlsPCCL+An5BVpuDZpftSgOLOJAbKiHpBv
  PIak+Aie/iY0ZsmRJ+KfzxJ8O5o16Klm+OeLEGlr5lR2/vClH2qGUHSXw8MZjceklOKkqdml
  yqeYZcGCczlwI/eLPRKgMbWAovdFBpiltEZuM3ZtNbea7VWMhxKV6gahaImmWyojMJpGRQGb
  tVVJkTQ6Gzr+kh72bI8KUCHzdWrBcC71ofHBanmmmlXJqV+YqI+Cho74TOelFtvye6h5hrmZ
  QGFYQ0wTWZFqYoLqXuNWoHm4qj16KEY1dgpVOAt4MLs6CloBpc9zEJeqBqbKjay3kTkBVUHW
  q5ryXPHGlhRafae6TPtWB2gqGc36qSSYoqgaXHZiPNO5m9D2hlNlnWwBp9eybOoUnOv0oBca
  L/YXNcsKRO76W0tTrdAajpMaq0VmWUhYgcaJrF6UBQUEI0ope+IIrq66X/o4VyJlr6M0N2Lj
  K9WKr/9jpD5BqdUCcYSVo6JGryLLIcLTj8SJa4ghaE7lefm5pI84TcsnI/hmaTT+e1wVArKl
  xhZ540SoobJ3wLKSeqWmqZPE5KlUA5E7aJUnIhBWu7McK7STJ2pBizWb+iQeBCkqFrJ1tHeV
  xbX1pAW/Z2O2CrMnJaNYxrS1ULLcVZ4XYET6QEtzhTaIN355R2wl1VJp4UU3+jDiYbfechsP
  21KV6GHmShm5gV9sO3bQqbegxpEwkTt0c06YNEIDpmBytI0qUkBCeZGXy1zyOiWMi1sua4K3
  Gqhoxa3flJl9sGF9oB77xkLhojZV+pN4ZoSNgV94m3sDcR/2aaGEEop6RrxFOCya1iF+OLOa
  e2S0U0zAm1fFNCXQm70qZrrmBLr22bp49rrRGruya2b+HWB+U2uy7Xu6Dsq7J8ucEUSfVJmO
  3OMhVqO434gbobiBg6uwXnk0AeyWpKsMjzsJAyiFULnAdAFHimu+f4e+lETBN6anHKBdHcKV
  f0G3Usk02nnA7UqBQ1a7OWmoVWPB1hXAegaoIZad4SFat6mZaIur+9vBWgGOx0tSEzyCq+uu
  GFxyGty+hGO75FCcfDfAlSaAgxlIi3nDPBuh1Ja0/rvCJknC0sfEZnXESIWyUrmg+hkVCOY1
  oOjENkyw2DrE+Im+Qpq57DskKUoB/7VEc7zBXVTDenwnyINoalLFFAGKqEusoYSfV3y/LtDD
  3qmpdowG2TW7R3jHGxAsSmz+Q6FhhPmzxFLctdNgyfMiJ5Qph97DxnzZj7LWRuCWU0jjPlr8
  GRBKyqyysakqhGgMYTbrPQGcsI/UylZiTad8p7koDaucAfGiwzm8u5tcdC0ExUSGBt0Xrsq8
  d1SBu4oqt86mvuXVq28soX46KPIgottopfJhpVT8T/ZwXHuczlIBw+HEzcwCa5zRp79JWUOb
  qgeszbzcTN68tFu4gkxkXPkMV+PjMudaHCwytkl3vKKbzOJayb/JxRWwze68JMCMsqFhB+xj
  aLfFPik4pxbwsxONeZ0Ft/NZzx8trWBsNzoL0RLAy5cV0LBxrNHqoRhtT41gszXNvBWAPOrc
  0yz+DVGuAcE1Mrp4dYHP4poMnarcLMsIGdKyy6MMbFt/h7ziFoHHFyyV67XcBNIT/dMt7NAC
  UjMnR4I3B4t1AM3RTMJlCtOq0U33aypXF3XhytXu7NWVGoYQ04Vj3XAQw5nZxs1iLItOLWqu
  WAe/ArlQ9AwzmmFg9HpMt9Vl3dXsnJjIY8nY1pWwPHES5tKbzdZxTLEQvNismCGmo0qRXdeT
  zQuK3DHwiwEwstoqYrFpe9o2VMJKPdg+1UF04S30ls0k3RiYYLoIRNdLPdkzRTq3SXwGrDqG
  Kb7XS2y8DI4FnYW63ceRUDgAEB9sZ4534NGm7dkmTcdaHKnTPcyx19PsNhyQ34ze6B1tsr0O
  1BxkilCHINI4733S7K3Odi3MqCe9CWm0aG3Praze4D279x3DsGqqt3HgBVfg+43gV+rfeb3c
  A/6v4I0s8AqFJ9gvt/U5xOzgxg0GAP5MHc4mIx6UjFveh2Lhno3h+Md1wjYV/iHTOw3igXbi
  2MPC3a3QJX6F+X2b0v3j7K0h2e1yeoApRw5qCCTkPW3iH6LjmmM5Os7kT44PVK7ORM4yfYDk
  bFABRa4rV67HTu4hUO5vUm4SaJ7mar7mbN7mbv7mcB7ncj7ndF7ndn7neJ7ner7nfN7nfv7n
  gB7ogk7nCQAAOw==
}

set pieceImageData(Leipzig,70) {
  R0lGODlhSANGAMIAAH9/f7+/vz8/PwAAAP///////////////yH5BAEKAAcALAAAAABIA0YA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWq/YbBQA0HpB3HD4IS4HyGVu7vz1qdvwTPrdmHPZ9fk6vhYM/gMCeHwyAYACC4Z/
  gi+AjgMPj45dDpJ/NwCOg4QwAX6Hm5wuioGJoI2WkZYDlA2rmJqiMaSPobIpigB+lIt+iC6v
  lautDME0hgJcfra3J7SxzSzPvI6/LcbFw6qSNs8DzNEkiwEEBJ6X4SqZAOXorOV+qOztkMLz
  5euq5QSZ9TR/+wiUSpdiFTqCJ/o5UsANWCphlogtwCbDkjWEJPoFpCf+EWOIdfQOGJqXSd6+
  g9nu8WOlDx/Kiipf/rAjRojCiB5L3LzkDVVDV9ogPrJBcSZNOj92bVzZMWnTIn90sTwQtZ9J
  jmmiBsyULOuAfVZrvDvpL4hBQEI+BR2SjMkjagsdSppjsGuZojDw9jgrk4fWjWGH/AFnM+6B
  m4RL/BV4VuVOSWD7vljm8qnfRWLUCuFb1uy3JTcXIPbJuTRnG2oBWd5xOohSwFMFD1YSWoHC
  1SYW83XMN3JnE0gbPLvoIPi/QN/KIZOsI/VcqLOT1D6sOq/p62dXGDdVTcL2iqxICVj5m4fG
  jVF/xOY68MinAJ66suJqXeXS+/jzBzaRmnj+ok/+LZBaYiwA5Ic5gTBWmEG4/dOKQgGm9Yh8
  h1iH3YVDodAfBOe058CANwACX4KkyCaIcp8AIeIBy4xTRD/j+fYVAb5cld+NOLpU3keBKBNb
  AwBCwJWPBKoAEI2LhLSZQSrO1l+RODw2YYUO2XXUlUelmNBzDwRJhiRQoqCJcoYB4U10PiDH
  SoKpNYhakhvlAs9n14yV45337SdOjEoKBUGCJ0XokI7zMKfDmXSmqdpaSfWGJJorLIbnpDqK
  +ZCf23Qnlmo6GrqGXjmwORiCb/lACjn3nTOeIlXO4ZxpVvq4Iwh2KpjpnzHNWiepgZDjKQ6I
  elaam7OUhqpmddr+h2MYk+o5gl5lZvPTDIeYMyJaCwaUnqkAjUQWl5+OeqMvSBL7AYboYpsb
  nwJFGO1E7P6qQQBcvNXRYIC2+Fu9kwAQZkaM7oAoIPyoZa4018VYXYE9SpluVVLp6oFFf6oL
  1LQi0LuTcUkqcu2+G/s7A7n71NiDIjGiOid5p9xwqoy1UiYvCAM/vMq/GLy2S8WS6fwjza9O
  iEe1JavbYV04gxC0h4dyto94SZtQs0W8tjC1zbVoCO7FO756MAVHr8IIQ+FpEljYFkU9ca3O
  ukzwfQLwuZOgLry31Xy1IlluC85ZSWlmE+IS+AO0RDC4CA7jRBVftl339blM8nA121T+1rC0
  tm89+vgGS2N9uNSfM1A4z4sAbBpckSe+9QqfLLVtuMndibbaYMAZWTv2IYNM3bZTmp+XHWj8
  HYRCTgKBWrYI/2/nYl+NNmd0k8D85ihMPk5AyNOQOOa1oMy753V9IPxTxH+58IcrOqB8zmLL
  CgpnaFNIMcJ5k0c7CS//HXqVS2WCe//fmFkI7HYjMeDIex7whi2QVRxW7GwbTRmOBZhXmmTA
  R3Wwot9Z7jcC61VDZeKhlkG4Vw3GSEx6/eLC38KgqeCBqUsWY8BNnnI+0e3veHNR2dMcJoZh
  pOpV0RvgkVzHNBoQ0HfD4aAcxHW3/y1FPkq8wCdYiC4qnpD+ApRBUIAUYot15KOBr8vGsRIV
  AURZcCtBI6K9AkIvg0RRAhQs4sjSpRIAvdECqiNhvyDFutj5LlUCjAAQzUdG0RgPjD8jW9Ym
  oEAcOUeHj+odIBeptR45J2JB1A4T/xhJPhoJULBxYpzWRL0N0EJ+e5xLxDypAVAiiZBN8WIi
  WVTDicQkiDcb1yQAyKnfgUqIFYTFw9h1KtJYQo+OYEwp5zWjZSlrKYZQIl5uQ8hZGixTlTNc
  LfoHydtAU0RxgiSvYmi60xEFUON7pi+hsQKPwWxGkDnJF3kHSZQ9cZMIWqYt70a4d1GnXh0Z
  XSVyNQEpifObktLRiQ6oOOBgR5/+kMMMs64DwvS1MyLMAosV8TXLFMQNcAvtD5kaBtAQ4EWg
  MjykDcmpyAxFYGNP8xqqFIEfcT2PNyrFn728YtEYnGcnf3wkC85TmYwCDo0dFdMzo/nNg8aD
  Vs1sV8UemBJbpVRLleDTSCbwqoP+sH5cmdQgQXehOy5Rkhjc5kgLuaV43mhC86xTRFjGqcSZ
  VBLuquXi3jUdrukVSI94WmPoUdMZMU9lNxRfMlbCLuyxRBdmtYA9TShJPAlVHa70n36i6sUX
  gOQ+UdUWfuLKgSFJhYZfKeKRZMLR4lEoMXj1nSvh0dgcXaqDi01rFyBbiDEJtqy/ZatOjomj
  uSTVoSP+XAVdS/WBx6C2dH79jUvr8EIJZG6czRsiegLoNHr4M2P49M2bYldCTnYykx74bKdW
  G1htfccZ9UNQTUcrXA10FYavBEqfFKC7WSqQAnPxXViLFlSM6bQ0491kHC1RUTmWM5nFfctx
  SzAwdgBxuez0wH3RR06M0SKC1bXuLjvJGWV5cCw78ehsN2IyGRxReOZdaxgdqhLhHahdBsQc
  RHNW23IBsMcO5sAZGJjSkmZjv7Q0MmB7dQEw/VG5AjbwCBYc5D5WY6MYavCOX+oo70ZkwiVY
  Gjsa6TD0SpaWfZlhdJc8YwXYsckjFpsb2VgXKKe4rV4d6ZbZ50ck5nknkS3+Mn1FScQ9VyAX
  5KCXWhZbsEPMo431TWA2G8eOl7BXv6RFc6SxGVo8zTXK031wiV2MtWTkck6bbm6XKRuwdrIr
  o+boh780WsdAh5C60cIYcxmQPTjXNbbm6Jz7JlS09t6ZwpW9Z6rDHN5JaVfZ0N2SOjUbX1IZ
  mgJDujK/UtimNgNTteOhKk2Vachw7+u72oRwqlTIyxEDZtZ5kjLNsgwDKg9WbJgzswYwiMzV
  acfEsKa1ONvCn0nv9XXeeFCoqcNSEf+6vcXGTmPjeWxxrBg/WGXBETlZbRkvu7QXXwmh7/bx
  Z82oH5AE0CjHEsgLwIhpAKFP4+AZm1G9ZDlgji7+fsTTbnDS2eewWfhdMTQZ8Hn5uqQK9MxH
  +C031mcsj/R4jHTRttpN+nDOnUioP5yBzMH059jpZnudg9w8r/za/Ww2pVDs1BYiu9MuGbl4
  88Kn+D477kg2OZIuYnM6ffaLNGWVKcbTcoa/TY0JfQy0L74x5GII7eoDH3tNDfG9V2/VrTEm
  hmmeOaEPPdoHj07WRTKtZ+y7hIAmFTzIwbxVjflIQGU12hO6rMJnIH8x/mx8AW2pg1JbnQ0v
  yD12/0zNrgQFmWD9et7xxb9PRSnzRIYWmeluzF2LXoiF+B1IdPflBtpmF5X8eFDOajt5zFJM
  P/pt5RpzBkup0uhWtUr+G2/4hV2SOwYHW3uNndCzPM1+jUYwzxBZ4zYp53d5aqc/5KBe2/VX
  QlR88LRU3mYkdfdMjNdMTwU6MbIMlEYqHcgUZLOAUyF9SBI1SKccjOESBERxvtV98iZpD6N0
  /CV5WnEexDUnkKdpN1h+uyYX7QcZZwJ/wSc+1fUYS7cwXPJfrZRKMyKCQ+SEATZ8o4IqqkF2
  wwV3EZaDuBdjKcgYXmWFDtVjMxRvJad3WoRQA0deV8QB4jIbn7UzWbQ7A7Yz4yZ4p6du46Rn
  LDdicdM6eahHaJdWjYGANEgPUOhuVWd1UKZ+lOSDqwJsXdVJBfFCWTdIWXcmIEc13LMyDjP+
  Iq/3bAxCYwUGeVsYY0pBMmTYUxa3bmFwUEOIfL0CICq0bfPBDxByDjn4T3rmh7mgXVHRhj0i
  Lpnma0x0E2RRMEdXKL4ydXhYcSMgiKFIVmW1NImWL5rIEegHfj61KjMFUI7hRTPlK7n4fkHD
  Qxbxfh3AdIWiFex4TCChEPjAiLaHbR23WZC3cTdidr7QIWbXgwCjTvk4gSogHqvEhFfWa/DV
  TM8gI2MGhVIIdKyER+6GDK/RfsnYDk60DDeGagKZXjZzj4+3eceCKttTNDkIjdHYCQCpUTki
  MoRIg/DTAQ6zEk5oYe+gXO9Yg8nRajyykhiXc7e3GD4ZX89jduX+h4AFtobMljvXE1P4JHOW
  AnaUlzCrpEPzqHWHJxCVxowiJ3tdyXLjp1Wep4hVJI1REUfKR2xBd2rk8ZLoMkdhgDwciU4p
  VGVQBZOZx4bHJH2yxhQ8+BU5mRwHEixt9WRAiQFElUU1lTtecyeXVXBGSUQyOAG712dk0naY
  NZV8YSVUJhX6Vk33EDdd6T+GsF7sQJrv4C120iaw+DCBuEtT84gBBhYUlCsyaG8hZiFRuJu/
  RJbDUAY7dBebKX+Zc2M6KYSBBRICmCDoWJiydZhwNg+oCDcSFTRTWXs5pVM9tlmfiTjVhoVY
  OIzeeY7whicaI2ZK41vJ5y3JtwsFmQz+7sl8gKmQbneFdFRwcEJB8KmJ2LdeJ0hVb4ch3ckj
  +IaTKel4jlaeSESLDshngeUtrYMMPCQz8ukrePVJPhl0J6CTWOifkyBT2Jl/3qloZ2FBV0mE
  wDdf8TaZe3VGXPhuRKZhRNMuOIcu7wFK9ZkbD7Oh+UlHihlrqwaVjsdu0LRKx8AgJMacTjeJ
  RPqiccIvItCYH4WbeBWYOcpsHQpadqmXr9ePvdKKhhRwIfqKEQUxwqmKdEedyjJgLDagGNB9
  TrpdHVQtBZmgWNI8keSmDnoheupw5gY96zNz+1kaP4ehOXKAIjRXZCY2kWKZcSqn4CVhVYol
  5phYB7adyqb+dNPpS4ICpwE5liFAggW0pQPZIyRpMCQBIY+GcpqUoVy4O3OKUTGmMTxplmF3
  Ap0TNxdKOtgBqZfnL5kRK1JVA55Aa/cAaeXZiiXDorr4qPHWp5Enq+ZFqw2lSSHnWGV4VqNU
  R8zxGs+Znf+ID3MgTzbwcv0CRAw6Ra3qrPF2barKrvhwTS6Akmg6nsD2glSaleTWmuZUA02q
  UeKkMXmSizPXkjSRIwBKoD0CrwEInenZKzsHPDwqlrFTTF+CqREGrlOmJgi2KTqScpIkJ1aF
  lAxrkljqqrkHehoncZ8UTwNYPHTkWLk4RWZwpudEdXNZsWpJdSd6aM82bLDyrx3+yTm0x64w
  4lOOhlRDO2/cR5dB5k4c5497smiPtmgceBzypKXxRhYM0z9AWxdiOrLjObB1Crb/io0wAI0E
  S3pqtWhl9JF3k4t3Ry8AxKwO94P7F2CgKgKYka8lCqVXSLZUZhdaurbVVKtYSjQIlHaRmR/r
  N6fSGR6RsVAOC7lZ6zoQGBkMs4D02hjriFypCj3DQhJLi6CIi0K95LYcgjU4dXldYZqJdlpH
  mkxSsnnIqZTpiJe9uQF9aYuSx26yVizB+Bbv0QI3IQgSKxwJeCeP653Lpau7FDdiMXwWKIYY
  iLt66Tgeyjg6mjCneiG9ZSzGyymqm3ZwO3eOJzdRQZL+Isp+8wmEc7W3nzcs25sd3eu9nduz
  G5BsZAGtF/A8goKPhim/2Qu+WHuGK5eGVIi9RJtLIkty1Rg5AboIc7ZyFgEiMZAwozBWMOQ5
  9qG/PBOZ8cHAJtcxkOhkqbGyi0h7RGW/E7yZOvQ8S1p0jvpz/vu/8trBjZux8edCVZGPEWO4
  OOQvqLpRZyoI4mlxfbsY3jq5vkHEPZy7ENuw1cdDEUdqGrzB7eu3n8a1Ghic40omQtwl3Gis
  3whr8EEjhnuQQglWdiJrYZONSJw20GSd/hLFWBqZR7sDAhy1BHx6O1wyYyw6XFyiLNpL7cAn
  BtMpihxVMVrC2PqbJYqtMZD+rze8sb/rqwCjmYZcuUBTj/ozNugXE6HZS96EdxyJfjpknTN1
  OcFVb/yrbJesWNc6wBkWpVlKX51AtYG8boumdCijMjT7GJfELxRLwhXAxh48Gt8joH0Akx+c
  rRO0oy8QNoBzIRslypcqzBxaDV5nOyUysdxcF62nMqzKN7FMnaRaqrVsyw3aAVPUuBFDw1P8
  qEczy7w2Cb9bvoqxmWYQa2e6u5icLvhsSrobzWhXyPSsu5Yqo4sCPqtU0Fh5lhCdwhqEsiQn
  0f+bzpzkNW1lpz2ksZO4UAzbIdnI0MhcAe+H0VsxeuFXasTK0Aj90Y5jIbFyB5SSTu5z0pOa
  Jdz+S9MoLcR9LDsqOwrLa14uLTUMCld2aw8li76Jq20KWqRfu84c8MtxWj4q7DkarX8MPXGe
  LFlLfWVNDRQY66QZh7pv0cuQ1tAP29MFVNVW7dC9jFBd/bY1HKf39796oH8bmHM2NgbBY8yU
  WtiuUhX4ubAvGtEqgI9yrW0rd9f5TIOSbb4MvV0Ho9NuknCHFpeCLWT8wtJ/JGsJ/UGvuiEq
  ZtqcFD9DxdGOWbrjWdc5LbWEozouWTwzNWGqo80uh9J5GapAF6f8iIBaJWyJQ9LFi1l4Ocj5
  i8061tmNQRhJfbhCw7s7ZQcldhR/vNEzKq7XfNMsdsuxGtw6TVLKGlz+ERmpZy1W6X1gso1E
  Iv0fDRPD1FQxmMNl6wujTIaYvv3TuAzKqw3b9sXEpmqenvnE7Ky7TX1iMUgWuCHD1RpdzAGN
  vE2P/S3B/41OzS25kZSDbRKvAkq67/ywzrq4gpPXJduRUJMfwSwcTYyo/DVFLM7PFv7UhdXP
  AD7aZJq9f/3eCPzIao2Xlc22Fw6Mq8HgvK0qNPKnmwDh4dPbRd68a6MVDM5g43TSOcunR3fi
  oo2dzOra7JqwK5VM47PKdaEcY4RvYDxruVlQnpriKQ2zOW6AIy6T1fLmCPsVOWyrePnlUa7P
  SVXlZ26LWR6DUG7jWmqf8ogh3nWSdOrBHZ7+AkTlrAJu3bnMsElMdaeFrk6pVlDrDSQNwKZF
  cCqN5/AKwmw25wVE44k71IdKwUUtasut3H9O27bRFYXstOw7TMpgLpOO6Hn3sB+DNdci2QaT
  65jRvne13klpqMAOGw0yHFZE0jtXfV9lprbuHabOrqie6hIlwlg211c9m/rj1nfp2y1b6/GN
  fxIFjTK2wNnNoI9D2oZd73QR54KE0kOO7E/OpJje7WD07HULYC7i4pgRsO8RzEX6ZgZfeL8O
  7ABPyDBZ4dWDrjnm3V9L8TiO7vyq7vV6MRXu5L0wGCijn6FwNHfo8fheRvz+QVpM2cy64Q/V
  8Spv4XQjiGwwOavXoa78rfIrLwFt7ZplLToyrxpNXfT9aroeb/PaHpNEPoj57rBIT83VM/W3
  Xc1Wf/RF7ug+/ytCWkZlNggQ/l75/DVTny7zWrbtM8gFpfbkibQXzvVdT4+ZFPQuL98YVQFi
  7uY+n/aDS/Zp6/aTOgv2XvhXjz+Gn/h9zUimdAdA7/jzMi+KP/lsjwZMAPhYT/lYIoOarfmG
  fWiNf/jCAfm3J/meX/iYUPnPqPo50fqu//qwH/uyP/u0X/u2f/u4n/u6v/u83/u+//vAH/zC
  P/zEX/zGf/y8nwAAOw==
}

set pieceImageData(Leipzig,80) {
  R0lGODlhwANQAPcAAAQCBIyKjExKTMzKzPwCBCwmLKyqrGxqbOzq7BwaHJyanFxaXNza3DQ2
  NLy6vHx6fPz+/ADmpQASegAAAAAMAgIAAAAAAAAAAADjFgOUAADRAAB3AIjV0OWyABLRAAB3
  AHmM2ggDA4IfAHwBAAAgUwAAAQEAAAAAAFaGFgADAAD9AAABAJAS0OQAABIAAAACADEA2pcA
  A5MAAHwAALCeU+UCARIAAAAAABgBKO4AAJEAAHwAAHCMvQUDAJIfAHwBAP8w2v+MA/9FAP8A
  AG2GUwUDAZL9AHwBABUMKgoAAIIAAHwAAAAAuwAAABcAAAAAAFCglAO55wBPEgAAADjwJ1zm
  xx0S0QAAd7CgMGk4pRdFegAAAAC0KgDmAAASuwAAAH6gjAA4AwBFH8AAAQCeMADGjAAARQAA
  AP+MAP/mAP8SAP8AAP8Ql//gCf/RAf93XACMWAAD6wAfEgABAAAgIQAANwAARQAAAACG0QAD
  tBf90QABd8ASg+UARhIASAACAGIBhgkAA4IA/XwAAbDUAGnmABcSAAAAANu+BwXfAILRAHx3
  AACMAL8DAE8fAAABALAgAGkAAAEAFwAAAGqGAAADAAD9AAABAPy8AOQNABIAAAAAAADYAADm
  AAASAAAAAPiDAPcqABKCAAB8ABgAEO4AAJEAuXwAAHAAAAUAAJIAAHwAAP8AAP8AAP8AAP8A
  AG0pIAW3AJKTAHx8AEoUIPYAAIC5AHwAAAC+UAA+7ReCEgB8AAD//wD//wD//wD//7AAAGkA
  ABcAAAAAAAC8BAHm6AASEgAAAAC+HgA+qwCCSwB8AFf45Pb354ASEnwAAOgYe+buwBKRTgB8
  ALAAGGm36BeTEgB8AEH/vjv/Pjz/gnf/fBsUpLMA6NG5EncAAPC+3uU+/xKC/wB8fywAhADo
  6AASEgAAAADdFAA/AACCuQB8AOMWvpQ/PtGCgnd8fEkBFGYAANMAuXcAAOAxvAMADbwAAAEA
  AAxYAADnAAASAAAAAERczmHDcHPTRyB3ACH5BAEAAAQALAAAAADAA1AABwj/AAkIHEiwoMGD
  CBMqXMiwocOHECNKnEixosWLGDNq3Mixo8ePIEOKHEmypMmTKFOqXMmypcuXMGPKnEmzps2b
  OHPq3Mmzp8+fQIMKHUq0qNGjSJMqXcq0qdOnUKNKnUq1qtWrWLNq3cq1q9evYMNeZcBArNmJ
  CMgOYIBgQFoEB9OulasWAVyDdtcy0Lv2rN+Haf8KXpiWLNnCdwsWNszYLkLGjQdLJih3gQAH
  bCe/bDtgLsHOnROrDCBggenTlw8qKH269YEBcVm3XnDAZtsADtxqdllZgAHdu1uChv25c2aW
  pGdbNnDQgOzZBxzEVX7a9gAFuUUHP4ngQQIA4MEf/yi7HSWCAOEBSBeoIP14lg3Sp1dw8ID8
  9AkCHGRwP3xNBgKkt4B25Y103nfuEVegSeiFR59ABghInkoB9qefQQ/05+B+GgJQEwILpCfA
  hAuKBKKGI5ZIUoYAPBBiA7ANAF4DAnx3oUrx9fdgQSFqmABzBsnYXwI1ZSjAAwGEKACBKnZk
  X38FkNhkR+fJ14B0/Mn3AEsV3ncjQSwOuSNlHdbUYHoNMDllRk9e5pYDFQqw5kcAAhCAXfFt
  KeMBdmW4AJcAFGDAW9+NOZB9BTxgF1sOFJCAoQJl6RYCcIJHU4QFIABBWiFCOidGYSKI5qcd
  MdCjewRkOepKp8r35UBh3v/3KIcaXqrhlqRqNEABAIwIwa+b2vdjrrrGZ8CvAW6JXgC/DpCA
  nCvFdyWwvAJZEKLHAjtAA8MalKWmEAwQIpEzZcgssPb9OZQDARjgrgEKuBvvu0Lt2iF41hJr
  UZ3y/RmhfNBSqOGrAsUq36x4lTlTe/0FHNR170YssZo5oVeAAsBuCkGjCA/FwAMKJtUdAFcq
  kMCwmBrgQIYNAFqAA9S2OIDKDtQ8QICP/qppozLX7PN14DHQbIU0ZQizxkYSdUACBTh68tNM
  Ny1UAKIOXBQDAYSMFAIMh5cfAQ7cVwBLrSbos81l4xfd2bn925+tUCodddNz051pUAFOmzEE
  dTr/HFSESyoFFwO8cgsAn6nGVwDOnpYk7dEQ8Lp4A5RTXmOg2TareOWc5yg03zjT1N4BwCKQ
  LFE5Btr06ukJlTYAVfsNVNhpMkW4e2UhkDp4BJvU5cGVF1B51bI2IDznvNY6k9vy1TbU7x1S
  jFPekGtcactEhd2i9EEh8CR46xFw5teANpD5vQ5mzDP6n/NrntYFOduiu/almFBb04fnG9oI
  kgvU6/IZG1GEhLilxGpHXUsArliFvgY6UGEmCUxCtnUf+yFEgjYZlwB214AFJI97NsnQxfYW
  rvg0zjqJ4Vp4epe9AAlve5yCHcngd5LHAWuDnevcZTImrhwiL02/qpT//0bSmeUcxyAGqNrL
  ErKX5YBQYAIIULZ2VYAeTa2BJ7QJcCAUngUi5WbuscuJZkRD3z3wjGg0yV4O4JsjFsQAyQOP
  oJg4ADYOKoOYs1eghFahJ16KVwIgYYY6tpM93YUB8cFZGYGiwkCtZQE0cpGdkgiALI7EhdUj
  oSY3ycmMuY8kDiAeCwngHPAkYJGhFJAfTxIgZn2HLN9ZC+yEsr4OjfI/AFjAhCp0svAVBQH/
  EtXiJJkeL6YEgGhMZutIIr9iJqRSsMtX/GInJZj0KAG5cdQAIHCqVc5kXK/51ce+g72eCMlO
  pAyU9i7mzZrAETwLUF+OEvArFt1SJDbspD736f/J0I2kb0gK0RLjMk9fEgSgkgzcTJLFt3ke
  S3tD+V7cfKKqLaUyPAW4U1EcsDsN7c6YrITdAh5AUpIGoKQnPSlKS2pSJK30pUg6QPJIYiqx
  SXMgujPlTSOVNgvG5FSn1Fmr2hmTmsKObnJcpBZJBh5JxjN1CbiSULwjHnABy6gYG5lCU0I9
  fnpVn2D00EjaM6BNMSBdj5kPQsjKFk11iiYhIt3IeiW0f3lMoukZaE9U1SK8ytF8UyOerNhy
  qtqpRKCD2pQYF8tYMZausZBVbFhJwjwRXTA9LKzsCuFazCDGEQBE/amG1OUTGQngZicrwDYr
  +6yd3oRfl9EUuMA1zgT/xNN0uVyJCx8gMXkdwIfA5dwGBdAuiQUgPkMESYYwhq5eIeSclURI
  hjIHAbQutEUb699DZxlRFAFFVYI9mAAsORN+oY+efDuV7Eoi0Ezy0wHusio/x0iSrgXwsl1c
  q0c5OyO27OV3oX0JGB8FLwMkUqk1CRs9hXQuBAi2iuQVcI5O28lGDqg9p1RJ8uZWN2UebHWO
  cpqlRmIunUEgQ87zlloPsiwTJy0knEnSAWizgJodJFm3C8BqfgRRg9TsADMO8gAcEOBSdalq
  MPoJdEergORV8Z4wMW8Dj9TQ8CD4I4p7wI9NE2QadznIdGPNl2lMY5Vdh2gj0Sw807pZ1ewX
  /8ZDVoCXF6AyAvUIRtw0HwXBQ6DcuCjIa6vmSigZT2DZs7QVEtoD+ASs+JjmYCDDSVgpvCl2
  mdQAn4MAJQelpNGE18PJTHFI2uOrEkaXVuBpXHsa0L74wNgAkOwoyTqIHYLkTQAvEyeux3VQ
  BTyAcvexnAIEjRIDdwikOJGyho5lgCSZMtI3GTAaH7Cx5JXzJE0OFLBB3cDjOSq3abZaXJzZ
  nNGCBJi/hZ62F4AbgiCWmzPKJp8HwgBfbxtNHczaS1hEbdmaDNyFrJCiFJuxEMHsAZfT37A/
  VCFWa0zKudZYk9MkJGKL5HVMe1ppNm6ZjkfR45aBWnjvRkRAqiyRxP+GbuMQmUt3BWi9Ganj
  p62kUQJ06bTuWtkySaluFLmWQfeCeU3CKiv3XBWSM1o4TcbYH8tE8bOphkDXruwRv3L7jAko
  8kP2LKIy8pWFRkWTxS0icwdm9C4CBTKaSDNvUiITYHdcCW71NjSSab0ke24AJ9kIuQBA3aeb
  6VqmQcfjqA5e6oe7XQFP8k6Mslte1PWqA+alAJlWDcoOWexaP9stFmNWTUuWalwIEODGn/E9
  Pb8P6c90xpqnhOlQujtNUy+2YQMLTghiNdyglDGDOVKPizdJLa8+7Y2IESFwrFrjGtlUNfGV
  ZD93zEQ4ei/B6pIAbw+PXahW9GPLfiJPKrT/i2GH+ZawfI/zbVba4v6S9ZEuY7j9lfG2Cf9k
  RciwKFmf3r8qX32eKsINsTJAtgAKoCast3KtInpBEkcrFwBAdgDs5xCsh0aC0lEncx+l5GGk
  hRK+dx/IVl60N1rVY3oFGBPU5yMZY3UXJkdjBxJ65CMX+B0yCDv9Q4PV94ESIYAuMkqsx0Le
  k1cGRW9xNEqVdwC81U72dTAjJWe8Iio/EoLhAWvixVs18mlbpUaF414DRnUrcTskc3he9W9i
  BxPhR0Kmg15NQ38Zk0RX0nYpcVGRp0ndEQAnBYYklEqEZBHak1c0hFtNdUEhgnAzhBB7GHw4
  BT1R8hCDRHywcyQ1/5NtjAgeQgcS54c+QSgTlehAorJDOnMmibhv1Td+/fEb4dGCLvhZDcBS
  siWHe6MpBmCEi6YluiIrZdQlG0gZEqWA8RMe13cQZaNXDDGBkmh7e8MAUngrJiVrOiWHxlZB
  3/cQS4NdmsQwk+gST+JwX1WM0GM+zzh90qKGGgNG8qdaGpMxxyWFP1dyTQUvS7M6Gdc0efWO
  dGNb8oIgOBgROfUAamEfokYQZ3KFOBWIJpNk3nIqhigQL5IbtwOAOYUivLUXZLFj/ZEmszVX
  AWSEEMkA7BKCXKgRvueE+tONHNGBWEdtOqNHDQCALqiMGNV7Ptcj6SgSt6OPjtV/XmUXZv/1
  Tv1YEX7YL0zCfb3CJD8YHrpIb98zIPVhIZmnbgO3SYsCddgkW3ZBSZ9nk6VzUelxicplSu3i
  dFHUADqWPAf5EpRUatlYjBNIPgx0OJu0AOhFOVYJIql4OCIJEawXVRuEIqlXRYbDi8/IH1Fy
  e9ylGGdCkAVRUw9gMqp1EFjZiwXBLZDDAExDdalXY2xBQnshhqbEXDy0OxdjGK1Yb8SDICRH
  EsJ4L0hJE1jJbWYJNLBzjyNhdfhhYrKZS1kyliXRHnaoSZP3VUGpEQgQPSxGmn2Yi5f4go5p
  axOplTzigTaZG3vjAN+jQJr0PdEhX2mhSVSJUaaYEdv5bWIjKiX/OBPJky0aOWQzA478lJaw
  eW5ZeHsuYhrfoSmAxGWM9ivLkkt1+RAsR1zYsRcGQFW9omNkwY7hcQDD1hkBqiQdyRCGpDG4
  hXz9g2w/iCRf6HlRiBDGIzTgYkIMsYf9op7a+T0Rx0OnUmOseHuyOZ4isZoO5HoxsTvPQiMX
  c00+QpScCQFn0p4f0ZABhEPiJ3U0cm+SmCXJaR6pSDM144AgI1sO6CgCEE58A2S8NWQ1A0nM
  KRHBqSFMQpW2FRexYpgEYV8s6m634qDEA0TNImOUQxtNyjd+F5StuDHw9DkDsGhs1EEYaY6i
  tJ+HSTxNk0P3kaVvWKdcE2uUI6Xph3iu/6ISPPN++GlKsLN/blk1HFpCIZKaK9E07nViTONe
  OVVWxfgs3cgfgXRVUhMXM/iJn+FoHNV5AyFtoIUQjgI5DgarCGF1pWaV5chN5ERdmsIvAfA5
  vGpVUNdf/8R5DxSTKAFdvgEaMFObYtJ79uiocSQqw1ozg3eenRFAx0VXLbE5lXOgmpKE+smo
  szau+IcRz+c1QhlHrEpvMgopsEdsPZeHSSkfatiMFSSlcbVJFRcu39owzGUXZSOmHoFaSJdL
  8DV58IUdLFJFi0OoauQ5mgZ1x8KrTnliXpIS/8Ys4BIh+oie4qSQYGRVddQ0uAlKdNmrHXqq
  pYOfWWdi5dhKf//5HYOyFuiBg7xiGej0RlEXIDuZRAXwJMTGj22FHpp6QbKWALvplEn0I5jJ
  LeeSfprpNQ0qEWdlWywpNh5UlCxhXiUaqWhUKHzKO36aEFcbhfwUbEqbtg56L6r1gl5DbST5
  h8AZXjs5EB/lYxWERJ+llQKqSgwBPTA7fA0DM/+KneHCZ9B0oyapo4MKY0kCMxjWqXzzHdQm
  LoqyGSzicAaDZ2dpaMTDrByBYTnaHvoImmalFiFyqSdCh935EfzIq6XRqxpDNZyEHoDHk9TI
  Rnh7ELwyZAA3EAqGLC1iEOmiYMT2Tg0AvPh6EE2nsZyEYv3nd5GbjT1Hsb5LZ1I3c0b/Zyr6
  CBOsB6nN8mngW0nmuCGMF16Yuzej6Sez6xF3m2ogKiIc2yHVGBHmejgJsTuys2T+ey340bxQ
  CaMIcTDnQrcdkh8BEqQ8lGrHqiGfI4ymixHacydmOKVBMxPrQ2T521+juzehO78YMZDgCJTh
  8SvHuq2HA22OKqdmWEWbZDK8ai4cASKC5Vr2oQAyIkAE8S/Ia0w5RYfJexAMjKv4JR85+lVg
  dFuawrsjLLmTy0xDoiFZKxJ0K6q/kn3oQ10ME69jFV4Zi7sEt6Uy9Jr6yRIuKiJtEV52ey/l
  9xDjksanprz44bf9ohit0rneIqMAcMHEw1zS6iNNXE/KVFZE/4cvJIFb2NRJt6N7MCGGmcM1
  JUW9XpWJ22MSqwaOA5Aki/Yd8pdLJGWE8kVAcEsYuGaHIuRedQLBwAI7Jpx5RLadggY4q3HH
  AqFgmnI6YBIocrbJeAxPsLFKtDjFp9krU2x64cG9GAGFATTLIOGiCYCgRsyIYFxVJrGdGlIA
  Lom+SivNG0F7oBtsmnK32PQRmIGVt/xZryLAtziBYxkmNUZ6D1E1OQMBLAmeUHJ49du127TI
  gWwgDdKUu6vNLtF4mYSTmHxVT3uxTMzJUaWeOAkgb5lrOBmzt5nKCxFKVXQ0MxONsAOBQfRr
  vFiAzfIAvKKSaPE9AEkAq6aRg7nL4P8xxM1JbXFFIC49y2jckqN7zlASh17FzOBjIP2LH14j
  HuLMEeuTvg30vJ/GmV7I0hYBiQ2cgg60tCmxK8O6FuJiGvTHNQ2XNTrDRnSWkalIYkRZTWu7
  i3tspoRbEGEiyRCBzxjDzaZEZNL5aU3TLiYdXroUQ4NqaulxwRVxUdtEvZKZagndhEKdjYQz
  swc9hvU1Q5tUU+P4tP+yspQFT8OFXDTiQdqGa/BEI+QEpGjbopuoHSzXGTibGBiGvKcVqyuN
  ALxSTcyztwwhNhU2eXbIdRiFufHFSYirHkSEzw80xx/Br5qY1M1j292shkKymJQ1mgEkiuhz
  pMVm0A/3sIn/ra2bknPbdHjslsrN1Cta05OpBtvBFj4MDD/Ms78HMcjompWlg9f4McG8mGk+
  amXhAj2GPREslx/6BC48A4zFJjmPfZONq6YbrGxUTREmI7VzGn+RQ442OUYIzB158jG8+Btq
  ESvRYRgBeqCf7EgyCcBa0yk5FTBnxWf6jDkDkTeUMtMC8YLRSxi8zUmQNLHwl17BJqKw1jSU
  RkJEbdwhoclYF+AcceTd10BOGzkdwifgQlXKXRFru+O/Mp3eF66C4oD0B2vGIzzZ6xyrQ1w6
  owDYUasm8juzDddEeRdDiR/Wkjbecq0RTgAHgzFrC7PN5WHkaGJru02Im+eZx280/6tPVKXV
  jfxvIzTFnqS+4XKZRm4lSw0RlJRVmBki8hdUFd4jWWx8QbMpvGuTDXKfLrkkv1LUMMYwCuBo
  JBIfpIMeGQY2L4fIA+VgdsKoogFG2cbZTHvdmyRRRV53dszFzfJZqL6GUEfdPZrMD/TSRt21
  gM7CpjSaOfqNoKTfXoPd9+LMICFHmptd9/F+Wa4oHJMjTI6PlfXmAgE9ta5sMk56xyoaYWeb
  EwE9zGLBv9JsMMMfqcjt7gFJzPLJ53KCzUzFwR0SPION84VbF2MeEXvII6xCjJYkFB0uUHfl
  aCHrG2zhTbOb53fpGBGc2BghTdyhgdl/YfN+OQXuFAHdlP/Ss4lhH2my2BdSJ8wVSukM0ydz
  LC9yUK0k5evORXllh1u6OOOeaS+uJaWjaIGSl498h1Dnx+pM7emrliVhkTeaTN58O1/ujMBS
  0I1M7YHikg1E8hiBZNWFxcAd55VV9BAB3W4MK86ZKmnTLXhdmveuy5geO+SeHkJjL0vSHYQu
  8NJI1ixSVx21TXgF7BYhQgtO3InEHU9C4ZBeOsnSy+ohh3gl7R3xwL/tluMoov1uShytEMGp
  6pq265q0aKbf8mOP5DCGYtX2s6TUSxybIpVyLoq5HomUFgGyI3XCLGGD4FsnSpsUdbQOaxpM
  p80jVAQ4/HwDPtOIsSIB7Wd0bQb/Ur/cNuvi0bh7rvngOhLZh16aUsgd3BIAE8LyAV/htSRx
  b5oeaLwdlSJ9P++ydiN9P0cCznkAgQBCAwAAEhRoAAGCggIGAyiEIFBBQYoVARSICIEBAgYC
  CiqA4GCBgAQUByDwWFEBAZYtXb6EGfMlggMAEkLEmVNnRwACBsgEGjQoTYoOdB5FutPjgIUH
  jR5lYLGBA6FVrT64+DQnSgACCxRgABUrgAAIrJ5F2xKBwbAQDJA9ujZB25wOAByASJBqWr4s
  7SZg+nevA6chC1Id4PFhyIMr18JlQDAAy8hwuU7u27JhxcUQB5wsaZRnygcC7VrES5SiAIGP
  jQrEmbIi/4PMZ1UbtJi7YkmKYGvzRTB2t27ixRMYgPCg5MPTUnEGuGjgN1rounljhCi8eOnp
  3QnwLiggOXGBBHM3QPC2onTvQgnnHmxeJQEGC3IXYA+eImauFh9Ml6+gB5QraAGFjDJgs5tC
  4ihA8BIA6YG21AMAOQbCGoCiBRbQD4Cf2ovJAYIWTAop+/ADUaiaCkKuRBd1+kshjxaAbafN
  NEwRqAQBeAAq+yiaCiqPEtgrx7QeYyo4AAyEsaDOcMKKNYUK+tDImZyUyKCfCCvgtRkRyJBI
  hXiaDKsCmFKgJMc8wsswAIrMLE2LcqrusIjGijIsCgsKEoEG/iypzdMCzUm2gv8OMMtKlgYI
  kKIOC3oUUvYUdUm73CItzjkISuqRp0tbdKshASh1aUfjoMyURVLP8s9Ni6S09FC39Jt0VeKo
  cuDG1cCM9L86OSPAT91q7Yuh4kDC6cebanJAWN3OJEzKFbvCKcPiECU1VwhfRAqr41YN9sfj
  uOVWoAE4HQ8AZLeK6EcgEyUVAfva3EoBATYUIAC6FMqyIHjBBaqmBgbI9VAFFEAAgYN5S+AB
  BdA0oIGSEtDXIwFoA7gmGhXydsCSmFRvQIOS7WkiHrNMYAEFCPIpIuguns5ZkyDy1E43FS4A
  pOb4DExdrkBSkiJ6GbPoXytFTDVpg1Za9TalLQrA3fv/mLLWqIJxq2jomr4Fd1riGvhK1+KM
  BhimA3IeIIDXDFjggEAPaEvhke5aQF+NFDAgqgYwBpdAiyj2G7VY+TQ5NwEi1g3m3/rTjcmy
  IBAXuZJYK9y5ly+UzTSjrCWOb0W1XZdciAxIgGtSTR6X3I1cRHqBAQi8uETOC+oSYNLPjCth
  3ZHaeoGyhToX0gZkS+DP1dwtgKTVJOaN2GzVjEjJ0vmECLqJgQ6rYEIVqnNypiAIHrPpAtCv
  M6IKCDT7kQl7aE+bRKNWYi95nDcjCLxeElxGn84UQnDd519PGHA1qVBNAA8QDf4qgh2BWM95
  VjJVAIkzqt8B5XU1ygnBYOOA/8WAqVndMgDZPocpTGHNOsYhTlm6I6fGQYB8aNoMSEaEgJ1V
  JCFRalCBwJQ88BWHgtlCn1ZEpxDS+S9eQ+qRQr5Xoo6EECc1QpxNlLXEpOTGd+DiCZOGeCCC
  zKWCMpGeyhzAAAMoYFoFOMDDyPgA+ajsYQNQwAMaQqOyIaAkoHKhow4gkgdwCFL3EkADNnSj
  A6qNhXd5IpW8w7iLtGYhPBJMSAiiMB490oYngU55cBeVuRAmiRFplOc+16hLJc2IlGKkTUpo
  EZCQTjcPSViLgkacuUDkPVdcVX0kWBxRfrElK3sjBw9wAARCxIwIWcADvhecBwwzAJ9xwL16
  qSgEiP9tl9e0CJxqIy/i4EVjIWkIxfKIHgBSD4c8GddEDFTOgkwzR0j75BbN9MAU3bItIgod
  u6ISzzE54GyvKs2L7Pg3ERrJW3HJiCNz0jFfyqQxRxEYHpX4JRiVxJ1WmhfBDka+3ljTUSX8
  CngchjcGTMSL3qlchWwJF/tYyD4agcsAdLU3CHhEIejrmXgYAqqU4pJU+wvgo075OV11KYLG
  CYvUgCREB9iLlpLiF1ESUNAc/QqbCThAQ2MSzkAGaAFhSWlPEpbSriqybAokTkgRcpC2HWAB
  YENIXJ920u6wkzXvwZxBkAOdhNRwNeO562YuZB+QoNUmAEMadrYIueisym3/4iQawkqkzs9w
  hGCBo10DDkCwfbGLmzbUpqLOlbohcuWHWm1JhhBlv4wwIAHieSL3qGU/2BD2d3IKpHWKJwDe
  9lZtDgCuGXvLW+ZZJyWsaY8ugdQWmaLnXG166XPB5LWpJAYAFzpIWKADkq/EraihPRopTai0
  oVqJo5DanEctQiMCVqRlGnndAQK5yie9hUgAAyo2bULVCkalOANjgDXFObj9/M6VlzrI8A7n
  AAM02ABMfbCD/clb9GFqb8k17ENqAhKT9ohLJf2akvBSHfEE7yTirR24ommTxbarsaTyHtGk
  lBQKpUy+jnrVtJCDQZ1YlSxl4yu/LjSACxXZyAlr/0p5UVuT1bLrXqzNCPmS8jID66d0Xzkg
  SRebngDYq8IdQhiIZJq1sBBFAQEmknWdWoAHKM+9fkymU3iSx3VVTsmiNdRV6QkidzUsInn+
  2ltLqVIa2i0kgK5IEk3DqbKF9SJhgzT6/iY+1LKEwOi1a7omaOCiJu9gTmyxLcsYNhOqMLni
  RY9rycmbAX5MvAZZQEMktDWEQechjv5x2bz1JHIZK6urasjePsvjuiAaUlHTDQKJvRUfnzZe
  9uGgsW2IQIL8p9KlWtKya6rFnKQpKVihNLjM89pnhvpFcRR0QYwUYEe1SG8iOoh+IbWZgj3F
  v5BiWqPlHR5w3ahHqZRgw/80hsGmqrciNdKls1E5LWIKsS4BkM2vr+2XTDm3V5pOdgXt04Au
  m5tcCmumv3JUmdyARABodNu+w3NOUt6ZUq5MwGJJA64YY0WiPeaTAwZ0gGcGiwATaUAzH6dt
  nP/td7MzSDPdOkxi5hlbE2fJWnCH8y4pZUlIIUmVAOaRf3v84+lRt5HADlU2pjWZzRzQ2Xc+
  IJGlFSTspsieFXVg/UqcUjd6yKt10wADBICYyvS5nNAI8VUWjWbV4W+K/PtVbgWPjlBnSaZa
  Fiv0zPJS4KWUfWLnddFZK/GZASDFIlUAtbO9j2z/+6Xdqpsw/+6zQ3vRWBRHqiHtmIrsciUd
  dff/EoIgDMlDpDukKmiyvHUWImCy7psg3xLS5UsrBvDb4VrEYIavK20NyXeji+dwzk/2sIoq
  XLyBRMwi664150e/QIqs89V3dDWYj5dhTWjhi/LZUdaUWOJos3uXSFtpBA+7snkLuIGjBjMj
  lSmjMpKhClk+v8gZBiAYAXArAbA3iOOTYoqICUQgIhORcPsfnHIRokMopCAK9Jg7C0um8jO/
  84seFly/qJG2qWioq2kT41so7wEYAsmn5FgAITKWC5MJM7HBcnkUucsRdRodgvkMAxgAUNuU
  2YO8QUEI3qKddHutk6tC8yCur/iIz6un0nmNEEwofhlDEdQI+7A2RQmc/5R5sAvpvoQZIJHQ
  j6/SqvySIBnsmxLao9C7qOp4EByDFEAsiONzk+xbFZMqLivDjZQpnvqbwSbTiLYgo2ViMEl8
  sDHBoAAwtbIpGC0iOjICrlBksFEkQTIsGO5QwwdZsI0ww3JJGOD6Jw3Rui+iEGXSCYQ7L0Ok
  lOAZGPuxnrgJgHDaM1dywsWyJrsjlSxqJpKQNIMoHc0SmRSDPGE5AOhbjWf6jDiqiLoZIwbw
  Ow3pu4uYxVXZH15rxVAbMw9MEQLBKge4ve57Ig7ijcdrKHZKGl1UFLgznLC4N3wDiveQNzxy
  PC/sjs96NLjSLLBJsM1AkQb8E7w5MyJqs+KpG/+cMAD5ApsH6KAySgxpxC+PIJGIcMcAQDv5
  irRIg0ZnYsLbO400NK95HCN4JMG04Q3k0ioECMbwUIAPOj4GExcJ+R3Y+R48KbMVORygqIyN
  OcejeBR1tBLXugiLIL2RiBSnvEkqcRmxKjpIzA6thIDD+CLNO4rPGKCDMcuzRMtPGzIMQhn4
  8w7ZWzYiK8u0pMuDcUcIXDYSI0hFyUn+wSpHTJHzWqAkoS58pAz/e5rVUj/J2MvF+Rh3vEsa
  6sbP+Iw245EGJIDMqqTgQQ2BCD4BAafiUZUKEpgxaiozMh79+hMB+DQicyBSgZ24vJDTrEu6
  JDIiOwqTgcJ6lA80EkX/OeoNwySVH6FAEkuSsaBDmWCAmpg6c3sUn8qWnnCrLsLAULmIfMEK
  epw4Tvqet0AW2ICNiygzmumhNqkm5SPNq4segoG4kTC4CRqJPWoW2AC6Q9ye9Yw2qpQgQJLP
  33MLAWnM8HqanPElpRInhQGP3XQJp9GvgOKXmrBJcvQX+9EolVmd0RGr5RsoILkf8jjP8xi7
  uPuisUie9/TLxBG0BFiVHfFED7KXeym8v+FP4PLP03i6ShPME7pMXzKW3WAKBoBLoOAmsOC8
  SAEydZGtAgiAvOEIFv0eiRlH1LKLJvPOuFgSGxzA6kHP3ynN5ZS/3QgppXGrMTKTFS0JJb2Q
  /2FSGjTKv6TZkDGKok1ErUPKlBvltKzpl/C4KAZtUEdSrig1EqDjrQSyIX2RG4LwyGurIfEg
  MBqKFNZwHyM0qKtqxoBb0XDSFy+NUWd8mg3JG6Q5GUU905CbGIPIlxhqqOSTlPOKUJgY0iHc
  IiPFIpKwJBsiJbxQDuH0pbXYmP/ktTFhi6OYUn7xCLe0ki+lpSSKogAqAHBBTOJwHSKKtWuC
  OoBUGl2lFOXKtj0xQTBC1gBy0AxsJ3C5DdIikAXIG44BluXTD0b1UFQL0dEMyl2CLFANKnBR
  Kv6hwKOSoAAVO6hawd05jW79Ipz0qFZ9CQIh0u6T1VUZlIDbUahzjf876Qlh/YijeJnw3FJ9
  M45mGTP8wCAITIkQIrm0mlWplDweE4hLuwgHMdayKdVAlFms8Y1d7VH9IFiYyFGVC1eYui4V
  249WpCHjQcZrQw1XsaGV1Q28qCFJzRF7NIjW8IgHrBEaSgk4/FIVbZpnfRWe5AjMMiWnfTbQ
  dJGxAEJfKqkAQcWYIJBehMeGhc3jWckhu00OVAA/ep/lY5OhNYiccbBRgxQEwpsGUx4IoSTo
  zJgFssJ1wQq3LRQWIyLz4LkAadau6Y1M9bLEoSKaiBSKITKDrNxry9ek8QmtSlsb6sOZ/UPV
  FUTwaLLWGNdDtIklaqq/dTC88Qz50VvPXdL/iBgA+QIkxtuINgMkCvzRB2BCQfLX8dkNKky6
  0QFBnagfLiqQldmNHNwNkuxcuMlN9XqtHjWIa/ssCYEyMiRWPUWtJhxFquKo93pb3SBXkkAg
  CCyXCMTDawPS61A52qGdPUrPAhmjz6iJzqgOQbK3A8g/LSIMAAMf2QhdUlmRAqgRe72PHvxd
  vYOLrryIiRtdpRkY1IrDBuvDTZWgnvUvwIwZzftdCiudFnbh5DkgonhZgMlfboPHYOzBChIO
  CoRAnXuTjHAlA1Yi+YIefgGbz60ZrV0V4VCbhIGjRyG9xzkQBIZisgQn4QPhaYHVnIDKngBU
  qAOxu3hHr4PbEYq5/9hypDSulqC7trRxYM3SxC4bEJLUyDjWRI1kOwWI4za7kaCbYe8gziXi
  poH5DEPRrLsdTAEWR5phyMR1iIcxgGednlRBo2wsXBSmpm+VvC+eOATd39zgSv/649rYH/HT
  V1nDXwu8iDfKxjreY9Nz5TsekDMr5P4d5elYEcdlrJapmYtwGKnpE+UQEyPG4iXmEwxiWQSM
  ReugIu3B5HVbEcMVnR5VUKgbgB/hp/fNDZeEIL26UG4ZIOh44C9KG/lwvi0uraZyMza75cxQ
  FochI9noLR21Id66jjNrKvMYZ0XBH7B5FpQcJLXiQqkwFLoqWE1OlQ/GTPro4P01YQAlFf+o
  HVAQVmVH6SoMTpo/aRR2tlybcJgIXA1EXlNtxLey1GfsLZ40+oyGHo5UwapsDADzeOYUuWba
  +dVz08KZpmHSMF+Z9KiifafjQl53PImA7WE59GjUEpegY0vaGsMydGq2XJnrUDFd4dr91BVs
  bY9Le0YJ/LQE3CDbvduTm+RXMV2EzmidxiKW5lnw7KEr/Sk3uyY0WrLdaOEFGq681uveUkhI
  ceHeWF60oGBjc+HCLuwJChCrDEypwOhdyq3emEHeCJIWq+n9Wj6eQCOZlN6lJVfgdK9h2pC2
  cpDk1Koh4UrN7jbzABhcwzHDdm2GyRTExbO/IT3cDTVv7MveENv/HOHMT5bgy77qugvZAlFr
  vhBmEnZG0m6olEgZukzXLUoPiDzLshO5Fe2f5Omy2jTLsylrwykbbXXpCkOf8T4IEm7IsNQh
  rzPbwB45i0Fnr6PTQzk6TcRgcmtnRQFJ7kNtmhkS2+lc7EReM9Q5jVxmqPkda20AySpFjSDE
  EhFMbv4dcWndCWfdY5bY4NYvCcaJxWPvtEibip4gZZs4JNpvpLiaDreK9grOMXYRMupgoBat
  EsqyzXIROk65blruQ1nKEsFJPilumnbvEs8JBYLxoeAgPUZytbGKhIFQjmDOEPLCpuqyZIqj
  +w6RLgunNMJjkuTyLffyLgfzL99yvzMP/0388RQHD9KbT7f+OI1wYgROtANvHh7T48c6bMOu
  G0PzjM1Q7Mz75Ps48/boj9WdPwov9JkdT5+Fm77goC5z9CVdcsujHShPcUenckj3jrSxXvwQ
  8p34ETN/nt34Kv/csvQQGwjfxUfxCVL/upJ6FFSPl8LRb+gGH95YW1/akc3rdCUSmyJ31Wiq
  1OY9M6HgJnv7ipe9Zu9FozMPDit77WeH9mj/Q06OGeF4mB0vl3gODxQ/C7M9PrYxOOc1Linu
  UMsevj9fWm6vjWVFd+IIVw63DZGosEV0Rr5zRD5N1Jioj3mH7ZQOdGFejb7Dm+w+S3AHm4OP
  K4TX7iUVjtAgFf+rKkYFvF0Exu4ENMCQDZCzXRWrcp2q/VuIHB6I/Nt9CePlqiCA9D143BGX
  w6I5YvESby9fT62YDqTkcZgy0kQ/Kh65sy8yVIyg8KfNSKYExBs2OnjA6ws+bXf+sfIU34wZ
  K1/OS6ymT4urcdCruZcHw5tR3IhR3Mky6iPe6IwdyfefQu5ropibbGx0Hxf1OxTAxEksdOkE
  B/pI8fVrXnvNonagiJWTPMm58ntI65Dd5osxkxT1G2vyvuv7K2xB2ixzkY/70p+Zegroq/kX
  ZpgvuzIJNDSpMZ34vVjUNttAz4wsem/UFib5FoqOuB76rRcL9NyYsK6hiZg7wxkBicj/rQAu
  LQTKIxkLnrPj4Bf+4Sf+4rfjsdj72ojmHdtv9UsJ2TaSaVFWktijb1adUcsZiZhH12Pr/T0T
  tMXwTz4cftGh1e/+v0H6l6hpJeWUIVFHhVn7aUv+hF16bCL8tNAOKbGXs9+71bJWhQMIAgIH
  Eixo8KDABwAWLgwAwcGBBAwnUqw4MUEDAxAGSGQoACHIkCINLCwA4STKlCpXskSJoMHCByJn
  0iSIQMDCAS138uy5EsHCAyIH4FyYQIBOBAMYMBiAACWDAwsFBDCgwKqBAAUANGBw0isEhQAe
  YL2qYMHCBQpcLm17csFWhjJrCkQgFQBYn3r3plzogC5gAmIB/yzIy5fvg44NAjMmEIBhAQcQ
  pB5Q+fRySsxPUzoAYJIow8WNGZO0aPo0aoZCRzdGgDY17NgWCwRw0JkwgpAMXsueSFWBWQUB
  YCbQeEAABAZFD1gFDvxu74mrAYsVIGDBggPZt2vvvh07du3gxXvvXv77QgOsRSqYaACBg7jR
  T1e+GXp9zfYTn0KfX7GATrxxhV9NdqV32GHxASBAbgTW9BgAlSE4IUtPKRdhSPYlsGFMAzxw
  nXUHOMDAZg/AZJoAGiEgnFcOCCAfRQUs4JIDHy4Q4gD6bSiRAAwAhoBYkmEGgWYnFblZkUQq
  OeSRfjl40G4ejUjhTkoZ0BGAT+rGW/8CoElI5VMc4VWUZwNomeFg/s3X1ZkZlqYmnBa9VhhI
  rgGwY5wUFUfkAwtopFxHeRolUQNmUieRTlQqipJE6rUpUFQMPXWioBN59VoCfz1aUKQlDXBl
  paoZaJSmmxLk4p2L+gRTAY6aStAArKqqqoELhDTAAQcAh1N2pjUQAFgIDJdAAcQW6xCR+j2w
  2ZUFFOtsAcsa2WlFAkjVAFnauUrTqELOSmFOryoYmgMImEvhZUzBxVACCrxa0Lie8ZgomHfJ
  l0AA78JKZqioFeCuvgKB1m+lB/iIUFS6nmVRoLLtuWJQJc7XcGjZ+hmYWO9dtvHGRGrWMcgf
  e7ykuRKV+mj/rAsp4ACl/a5MXL4BE5ByWgtQnGcDERkVc8BizfhtS4kRJrNjd64FNLqU1eQA
  xQI8EMBwUuql30KF+gQRu1A/3XKEDQYGZE5IIwgUACejDKNHD5Bl5E9sBfA0v0YBTPQAaBtN
  Jc0XzR0w1QSj9pHMffsN5wJe39q02lHL5hACb+KWXHRvP6AA14WP5rPkmav9dOKbc/5556B/
  Lvnmbzf6LgN2D64y0ZByvXpDrT/WrtgrGbCVrQGn3EDtFEq1d0iBHuAUAQ0iwIDQR9XWEuXp
  iRntZirlaHNQTnl9fACBGvr1XfT27lO47zb+ulF37ng++jfvfHDrxRvwuvJQK7A2/0oeCgf1
  AXbjy37gsJ8GuL7y5r/58EwkwiOe8RigFdg04EYqqxtXjhObBRCveMVToHzMRpc0DTA229rU
  +PyXgAfwj2ghhN0IS/iugRWgAS58IQxjKMMZyjAukQkYy+5Ewx3ysIc8lIgKDSIshijAcDYZ
  mA5r2BFHKYhYMwxUq4w4EPh0ZHiMAZtnfKjFLcrQSQFjgAJ01q8DAEuKrQOjGNmFPhgW63wV
  ISOJ2jcAAXWwJB/clADrGJvt1YlqATBj8eYIm44ADIKxiSJClNIROjGGg3o8zR03xQADpLFS
  CwhABds3kEkKrV+XzKTMmPZIdgFvU/EaZWoAaZOiFFEknP+MG0UasDcDwNI3AlAAHxECKjZx
  D5Wp0eCr4GOADwlAfahpgNMMYIAgarIuDhimdYx5GmQyZ5nNJMApH1nKR+XQl6lhpk3u8kdX
  DpOOMSogqv53y1wepInspIkjvcmQSJpKmJQr5ppypkxwatKeyJSmr6y1z2sShYsGpWEBBADM
  Ns3xoA41aAFUeSq1AGaSWluACxcwOQ3m6AEHyOjTAoDLd+rSTxJNJOUeqtIdZqmZA/jU22Kq
  NheSkXNvM4ADQHnNW8EUajJ1oU21ZoCXnlRfUcHIC1vYAKW6UKlMXWpT2RjVGCaUnmc6qlRr
  OMOnJnWrXmUjPwfigEvWhSacfBvVRhNKRg2CsU9LdVpIX0oXSraSMSmdale1qleqfpWNCw2g
  RW060wIcQLAiJepOabKUrAjWRITt3E1zWlTxDcA2lr0sZjOr2c1iVqcgrCxnQyva0XKWLpOt
  k23CWrzUrue0UCItbGO7WdcS7ZmJfdWnbisSpci2t7MNGG99K1zYnpa2QmTtbpE7GuMShAHD
  fe5m36Nbg+R2ugy1qnWzq93tcre73v0ueMMr3vGSt7zmPS9606ve9bK3ve59L3zjK9/50re+
  9r0vfvOr3/3yt7/+/S+A7xsQADs=
}

################################################################################
#  Merida
################################################################################
lappend boardStyles Merida

set pieceImageData(Merida,25) {
  R0lGODlhLAEZAMIAAL+/v39/fz8/PwAAAP///////////////yH5BAEKAAcALAAAAAAsARkA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqmJAAGTCq8heS1etqwT8zt+P1kIwECx0mcHHBqMB
  N8xLTsYL+JARoYJo3D0pyk5AMLZeikaAoNshsytls1sxKEvU4x3BXH9TWnVfcAOEZhaEaUUf
  hGEWAYV0hJF1d4iTl1KPWEkEnY0VAwRqoWSbGAIEfhM8BIwBe4x7hgujAGqtW3kHpLZFgrSa
  vxOMkmeIrqbGnxSPxM6uDgCEvcVck9TCB4DJh524Fy5rr1Vr2Q+aGKze6949DaE9jzSN0las
  yxD4GM7phcjlGtCBe0Zw1iR7A+bRotRMnwOHjl6RkbKGHYBOeFigigH+rlSVjzwmNqjlY0Ej
  d9I4TrDGoaHBSkUaxhAl0oK0aeBesWPnjsFNKoZOmvlJgWXAhLBeQpipc2cqpUFQdVL5h4zT
  TmMEIdElkJI2qFoOzAzL7ZwzqA64EHE2s+aEm4zMzfAIkpxWKl26DgVraKyhsg3GsHMbIVSR
  q57MVVy35oK4uo0jcNVR5aucl/V41AnlSZPmbA2JoWVQJxVBpAmLElQVYfFg1kfypkZXL1e0
  OpoftWrlWd4wpxCB/VjXFCNsWlJ3T5VLB/gqhkI2HnCB0OAt4qXlDRZ2OviMhKE3I6L5llir
  uGfqPnpeXawidAKRi2LVO7uLVNydr7y4kzP+Y2F+BTLfGfplsd4rShi2yx67mDKTRRvxt44a
  EnSnGkHnFYKfBGslt9Zo75xWnhLN7CIJJDgxEMNa09wSA1yBHCeWhMatUpw3LeykSz7r+PcN
  KAW2hsceLtIXw0s+EqdOfxyu9txqoaGy4wPOZFgMKCIOE9M0rhSy1kNWTlMFjIT8GARSGKWW
  RXLY0ehNcA/ugSOIpAXJYzlquEFGnmY255Q0br4JkzPCkEkMi7H02QCiodFpUpYVHlMEIx0y
  AiaiviT0DCre6dbZfsX1Bhxa121HIJPDuIDGlqv24sCS7CTZDojhgYgpMs8gSGp3zJlIEBjd
  /RqYqlUadqgsa97+mFVhpYoUp5KVgKRmN6hKEAdiISmFJka53YejjKNs+kum0Lh3FrJZlIbV
  tFgKay0ykJX7Ti8e7UZOKBQy+ydEUtmiUqk80SphWDjw+taRH+E2JlWLcqFeSGhoaeFSa6yh
  pqH2rUJGH47KtxPDZyLWoyAV5dDtcJEtFei3EOBx5Hcrw5JqwhQhBnJae/45hoxs2rxSsIqc
  udlQp43jncIhSQF0NhsHO2VaQFvCocipGBiZGjHLLBlddrQbrI0Pg2Rt2GJHCrSBxWCs4XqS
  PUwtYp0+C+FoWSGWUdtU9/SbyGy3PAalgL1DddywzkpYYKU4le27UXfstNZ+s7jnIVH6a3wv
  IuTQ7ZHkmY+dt6Ny67iMyS5gndCLH31x8sO4tU06SDlYa0vYszMONOjBSjl66qZjTbrqvbGu
  N+Jjzrn41sVzezwEhevoqKyiMyAYVj8EU4U3hyNNdt9Qp069ZjJi/vCXkkXtlcRFQz6d+bk+
  VAjZli41zsN7hD+/euq/yjVkz2//d8MxCYkRHvO3lJHGf1XgV86cZ8CHNG9WwQlP0bwGJSIs
  ykm3ClqI2Mcv84WPfWAjW0QQOBKfYCEKMzgHAiszNojRBQ6aydXwVLi9EfqvhMBwwiaAt0Ja
  3euFzIjhWWa4giIa8YhITKISl8jEJjrxiVCM4gYSAAA7
}

set pieceImageData(Merida,30) {
  R0lGODlhaAEeAMIAAH9/fwAAAD8/P////7+/v////////////yH5BAEKAAcALAAAAABoAR4A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s62LAK2MEABAjgc9RzP+R2i23CwECAZ8GqRQI
  SARkEYZMKqoBxZFZiQZ2Atvz4J1mbKOtMsNVOKHSzdZ6wGqr62D8IECOyx1oRgIDA2EbX28A
  TmYeAYVZGjYDgo8DkZOCDjULjwRjlpECBJcLQxaLYY2SWHkWVU9bqxx2rG21c64KnFd+vUhu
  VaazD6mnjoWQcgCUfQNRqiBfUcuUWJB4lLpRRzieY2pupFncX6hOuktYkWewWNEfte3r9K0P
  Xuawd3R9wPjEDYyFsKRMwydDSAqFWUSHAyBJzJJJnKitGDM0D8kUAZT+KZ1FEfQ24JvzhaHH
  Lnvm1aunKxeZlDoWcMRzQdOHRQpPRnBGipS2QtA6OHkjaaGNo0iPHtoUZlQnBkSvyEzitIsN
  gDXtaehnLduzAGM29AsLw2jSs0sb/Hu6IGqdqSknCMGKKpTOYqN8Ukzm9oIThWQtQDuL1ty9
  RQwaArsD9ZjchVU9kLzrgCRLwGLfGfyyd2KUWTXIKmZ3pDHdXZBP67kBtBsGS846Y77wKWKh
  RaqnEpLdB6thPmHBtg1LDYKNJEPC3I4RGizlXStz10lyaeUoQ4Ef04ssmAnh478h/Bbeqy27
  4haR41AlhAzDtBQ4e84u4VEf27IbypWvA2j+eAv9WEdXH7vgdgBSZKRFngPoCOEEdc2gc0Mq
  r1j3GiGWrbObfhLgA1aA/1Xw116MUEDgS+aE40Un9AE3YSrINXWfUi2qtRtF3HVIiHISHTRR
  aRXI9xNQNfZA0EQcGncIgkwKsoguPPYIlkJC8lUkdCuFCEEUzdDj4UXsTLAOeL60812SxVgR
  zpqLHQElM0LA6dwnnB1FSl8WdWaTBNftNQlFaKpFkDHJaCnBEZ0FGlA3k2RxVQyPXIVcZUdK
  qUOVEhlqSpaKNgDimB7aRQE+7DEUVwUZajWBLHMcyAUeXriC6KeX1hoqIZpCl+g5vOGXDJCH
  2kbocu0kStlVCxn+wuUlB4nR3qKyIRWtTqGuU1eWNYxpDQWT6TCZSiyh6sccHo7Uj6zM1BPn
  DfVUNMGIEuF5T6VS5qcTTr/iB5+4xgKI3UWxORehspX5iiS9Eu35wKf6jMrpF9tBcu+Ylj23
  aZbPuePHOxqH2GjECTEcKWXwYgegXj0+yNuVdchWSrF7dZrYV30wjA5nYZpyI4kGZ0rydgBy
  KjJsLD+cs7gYV2h0SJXt2C4j9ex4F52Z5ngPhikrh/BtRzfd2b78xpxxGFLY+Ykq3nVdB8o/
  9uxfFzbTFet26V4GrJFJg0tPxksz7QBnDCMVMXpGsv3VvUcl85kWO/+odkBJtCKzcVv+T37F
  Qob3KIban2kuhlEoL74qpxOzi8UOqVJ3t5E73ycHcmdZPt2435m62CaAmzUXUn+J3kPjE4HN
  wFVA+Q6hn7m2hZ/CQWfJMmrP1G4nwVt+SHth1tOVuqppPthPERhbc5qEYljsKfCzOewyoMRA
  k+/n0lKJ1bIu5wq4gW6g73OQgIqlf/rv8o70amYit1EEMQ5bGkCqkK0iDA1Clrve7TAwozM9
  z0Xbgx3LHhg1EWVuPh1iYFhK1q/RhYuCsKvdgoJFkf60jVsGfKGJ+lYkLxAwGOG7xF3+lDDz
  YXCAz1OD9MJhwiESUUzrKxSqqkLCEtYHcw56nHjM8p0T7Qf+RJDhmPYe9I5PoSNIfZOizsDw
  sIhMLnJjEuPfqEgYK4YwiZDQ3tYAtcMY9rAscNQhqo7wIOaJyY7bCtIHpQS3QSpOeJALo/ky
  yASJ7dF6fqxPGF/ByMihpJJugmEeI/mAOdKxYEgJzVxUcZYe8NGIM1rVizbnHDtxUgsXGaK7
  DhVG2f2CdI5cVChxowpSJsWUrRzgpGhJvL0IYXJuYo0xj1PHTVoMX3DkkCqWc5UyrKs1LUqh
  EY9TxPglpQ18ul4by0RLRWbFaGbMAweNls1Grq9VLKRmUvJVwPfN8zYs4+E7nwlIruWhNtMT
  Hzdao5RZeHJXTzQkkdQ4JTi6kXJ35rxWGa3xT05dClviqaUwnkhDhq7THWDsGxVqiS7egWUH
  dGJjBDS60SDYLnbjkttYqjiukC5tpBEdXlJqhtKalTKjLJWL7ZznmC19tKZASKpSTWEnD6wr
  lEuNalSh2oGnNlWqWM2qVrfK1a569atgDatYx2qBBAAAOw==
}

set pieceImageData(Merida,35) {
  R0lGODlhpAEjAMIAAH9/f7+/vz8/PwAAAP///////////////yH5BAEKAAcALAAAAACkASMA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM/0A9R4rn/3rgGAAAggEDKMKGCvE1Au
  nb2mEllpIgOD7DUoGqSAVA4xHC4pP1LgAqpIcy/WBVYrf4O8J8EAMBDw9H4KemUiRIYefEB4
  B31EeImJF4ADRgIABERtfXshiyV6WYEdfH0Lg0lZnKOpi6yPrEsUoJSCqYFzqh+eIwICBL+9
  iEHBY7Qme5i7GXy/rZd8CswEuQ5SRpSYPZaYgUAEWG1gGa6frqIbpKXphHesiO4K5Afp1A3W
  8anattHw9xjyI7D9ArcK2KBke2KB6EUgWMFpsBCmaqaQQR898QI0jJL+ygiRb3guKqswsh1A
  dKxm5avoYda5H65iyqxnMRW+LBzpzIMHb9wJgSB1fQM1MCFOXqXe/VrKtOlSmvwoHpjzsiod
  aVAnlNQVkytOV4qybgDEY6bZozZgTQ3VwCoteiy15vFl0AOWIHifLiXFDsNdY2IGOB3MVCwl
  LEgsMYi0ZglixMvEdlCZlAMukKz26rTb8R2Uz58NdxYUi3E0x6z6QjBNYlsmJnsEO73E9BRs
  jZADn50Zdy21LIt/l0YryVJlpGxh25JJuyHgDbg2w9RI2OndCHNiAV8jPLjkB734vCyLLTuH
  bd+oV3euWgIf9dfPI1v/SzwFgvie5CJlsbf+HCofARNLE6vE5J8EoBhnFm22wWTgZLvFNF41
  nhBnGn8LfDcVgHRtdMSBEfSi3kAOjQMMUOthpMFBTGExoQUuguZEg1otkRwj+IHCnQTRNTJP
  c2o4kloGlKW0AU6YmUWXikQqqZyMSkgnQS432lTLItBgZ84NYFASJGXt2SPbYM9ZIBhR9D31
  YogdYlLbmvcVGVOYGRoTW4bTZJjlHCHOVx8gQALKlH0w7kZnA33UFyFtx1VAGT2N+oXeYO9d
  oJOVjGyXKR58QqBHc9w0gpgjkyYDZwPeEKYGBszsMWJ9oGp26E5NjdnMrA6kSimI/QWS0A1Y
  TCOEIXj0UhJig83+oearRR0K6YOs/socpBpVasGzEA2p24L/JJWPb29ZqeMDyKaBWxYNQYTb
  FCA5G+ugvFrELLyERRoikJDsdap77+4VL2m0uOqFN2p46cVdL/ppHTLSqLovP2f9e9OCkNJl
  r6ewPEPPw6vtJjEoQriTzg3uUAUBMpT5clcyAQSQMjfX9gsrBgyth+KgZUbgpyK6ZomOzNmw
  qggjl3ipcbDQ/HryvEsFe3NT+E3wLFg0R+icK4oeCpbGJ0UWcbTApTZydJvGdZhMKguGzUxB
  VfF0u3C8vVTN9car8DRN+fwD0KvGvBHDiSRD8B6+9I1ompa0WW+hZ83ao0wtMycVSTH+ZduT
  g9xeMDIsYHFO3OFTJ5r41ZXnaQGLmnEcD9DOsc7kBKW6OajqufItMTPZhYWYFrpPYzbTmrGe
  jCROWhqhnIk6d7qBEhaY+bURery05Vgn8vhEUd9nq7CsKt60eLHnnXNa9GkYga6zSczIicxL
  eKunwiO73muMu4Jr6CkhL5u1UkevvnfPM1P0zMKObVxPYE2YSbVUB4imkMVMkRNfEQ4QvhaZ
  j4JFSFaJflYdw1HOOCoDwzdE+CkMkUtuellPbhxVvA9ObYL6O1e8VJI4IzmPN+MY4Ey0dC4D
  4W1q56KTi+blsglSQG0jEpGeyvcv/P0PVbYbB7vSpITXOQD+dSN0wq/QBxKOHTAldNJCj4y4
  Kcn5TnPJm9sFz2cJGSmmWyuBUlgwBR5sGMIJzYBVlM71MBfRx4pLo41GNtiw6lzMUxkkEe1O
  GKH7jUmOSnjKgcQDKdCoxBHx+mJXjhibTp4Df/M5JHjaWMVFUohv4zsfCgkTH1W2yQ1bdOAF
  6ZaifQWoGEdYZTO28oDtNYMz3oNaKjEWRxnZcAJDTFPTaAQ7HdoyH53SRMR2CcdNWgYncmQm
  7PAyI1tAoQl9FMkAjeUsXxoyTI2gBBnlVz5eospY+XviWoBnwVkVUpkX5CIVm6hDKfUSJw3S
  JJIgorlgvlEDMYKkI+yJNi1CiwL+L3Qo1U4nvNlVgGeiYGc7LYVHLW5rgBITTGiwiUd3BieL
  WiRlJNeov5k8LCwZ9R81IeiGIGSvoMoU0D+CkFO99Y+nymSNVnL6FFbJQZeFMWkbElTFGy2j
  ov6Co0LTIVWFjoZy/VTqGo4qU4JWQX+mjIZKIXlQklAHkkX5x1nl2CzoEfV9K4LqU7R61EQM
  86JyDZql6OkUugbrrXcFYD9xZQqQzjRudiUsqrIa0qz61bGB5c5bzxiZyWoKiiTlGWJKGKWE
  SI0IY81mGy9qNCd2KV4fgcLWIllW90D2f6AEy2Exu0XrGUy1nuVXBR3mwY7pc1f5TEhQhapK
  y/5vt4b+VMZd1vqpQSYqgW0cCP8cUNupRom0obDq5xCUMSgds3+OPa4OGaVcyCpQZwMNKh2L
  Cy8nvGqW2/vMe71oTpsp9m5pumyGWqZPjKTRX5Gb0mQLQzm+1kqrfTDwMrUq0PPm0LBeRVSE
  ymiWKZlXvydTkP8AmeHYwoLDJ7PudTXQCBGbMJc1NEclJLTQKZlYtditbjYkOkOSErGzqoPs
  kZwZlhOmGIxXSkmL0XvhGPezxo5V3Yt7i9cl/4tAtdubk5l8vs5uicqfje1ohfbinzkZDizB
  cpWnfDsrC1nMY97Nln3A5ja7+c1wjrOc50znOtv5znjOs573zOc++/nPD0gBAAA7
}

set pieceImageData(Merida,40) {
  R0lGODlh4AEoAMIAAL+/v39/fz8/PwAAAP///////////////yH5BAEKAAcALAAAAADgASgA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987zOAwG+ICQwGQhYAQJwE
  m1CJEdlaKgVHATM0TSoAgy0LOxCIpsfBAk1VpI/eC1lsDNQX4HLJfs2KP11AYS5kZlxvagps
  XohxFnNrb155hiN8KmACdlh/HXV6bkedJ0ZLbYBIAVhrmqqMracWdaWKAgQEl2B2sR5BApqj
  I3lvwRpolWnFJIuHb5FpjJIZbFvMB8O8Hb7AKb+3t7+HBACcRgThK3a3l6jrqa7rrXW32Q11
  W0j0SeYEbfxGX+5sQJToBJk3lTywCXOMhTVUkjYhtLPQkYN7oaApknb+YGHAehUIpsj3DWCI
  MODg0Mti8UwYXeJwNXInSZ8ELKoSeQMjphAelOjKqEo4rSUJgkdEICWoCcVDhUujNrrp7MDB
  QQp8fkGYEdQGdihIuhOhaVyab60KKeugFQS/b3DjygXpJgA9JsOIqmpwEO+AsRzAnlhKVlKj
  nGXWcmgLUapUo10H5bVHtO81RJ4gk7WFliiHPOcCgCnJWSbdxVlcyl29+nQ+MAxUdUoKpOkC
  lCY3oOvGNETeldBKwinBGCrF48iT79KMmdXsgl9sR2bruUSmlIof4VwdJC6W6qhNgZ/m2LHm
  yZEc0U7P4KDmCNiyayMon8JBTUhFo/UK4ur++CJ/sSagTfClkZBAt0HXUSzubRAfCkGUsZJo
  7QAwoFxG/GcBOVrckgkIlCgnImKKBQHWFIJgFUocFNXXAE/YnRBhIy5CsJBUHcqkYQXk9PZB
  HiOOKIoTFMW23mUqHjDcGnbU+JOF5yQpQib6jZOTkw2QNM+FHvKnwXXeSambhFxGuWMkRqqY
  4SRL1kWBiUXWApd0mzQ5X1RYMmCgaY91eWaBePZXno8V5Lagmm0Cqed71yDHipWhMdlKnpeV
  FialSlpIRpn7UQqmXJy4JVSQOTFqZCV1oBqnVQjiJEGPqailTkmiFbJLYl+Wh6lK/QxqV5Se
  6iqqpay5ekGGz6D+2iYZXhgLAawSJabOEbjUCo1aGngj4G4aMJQFp98ceYEucM3aJaY/yaNc
  qBjkUU0YBaUSL0xIQpYJlN/AGiBwVebLrhzl/fkAmb0OCqW4GxJ0KzEgirUaLe0OeWi8aSTY
  hqLPJhaXvufYYiC+Hv5rgcMYKnjsX/C09mvJpmZJrLkpnbTywyaHhI45bbyUCM5J/FLzAtpy
  x4mfIHsncGRRBYYEh0vx+a1MLeup8I0/ZzBo1RKkNnESVaHRM8LtcQbnrz0GIZ6FY0PJbUhc
  Yp31zKyhFLfbDsC4zlDm8gQCyXEZegG8Qpizl5J3sdpP4H9VXcqF2gbNGr0jD8rB1U7+i7JS
  BtikssTCEntw9UDDNaRkMlZphEYEi1/Fa+MBqi4h5CHB3TfdDzj++L40R73gnCzNfNrIspdL
  uwPy7t5PhPqMdriSu0SABZc82T3g71lKnu2gHjPlju4LgZy57hB83m3F1CyCjRBV1f5X5ntG
  D9RSFlIf9oBr8yg3/cTONfxGGMou/wR8E55u7PILfqQFHJvQx/MGNzDcbSuBbdtf6RxztK0Y
  jH3UutzJppafvVkvA/dZhAglcZ8IUItqKjkgCvMBtgLlL0YRC16YuCSyCbyFVnMB38BkCJhp
  2CITAVrT5rbTj6EJpj0OvN39pnc0DCLCSU5kShR/VUMp5If+ajok3gcBZCBE+MeLHFFf5RBh
  Fp1FxS5eelMG4dI5ObxQeLwSEILUWDSayWxAfpNF67KABD6WQRTwIhNdbji9JMYNfFEUk32u
  RsHtjYuREjThFjcIyaVAhh99OqFUCASgmc0xYRYK3hQ8xkM/YeB5F0rjQEqZx0KhZBjUuAoa
  0EYXcrXtjbPDEgp1mEiEFKxpZjma65ZSwfBNUo+VtGSB7DLFXkUJfjKpEZBKpkjU/QWV/rrY
  EucGwm3mS5WrxGMkLZYMD63MRF1qjvO82T/ogRNQUqnRKH/JkV4eLJLuQY6BlHLMyCVTJFSh
  JVJsURY8BfOUAdxP1KblPf4Qkpv+5DnjCBLaQ6vJA1zVyskii5acAiJnY++EwDA7iMwycuUZ
  Z9QgF/WXxfCpq6PjjMxx6qjP9K2Tli0qlycpgpuQOmBoF4KdFGxBtoP+pJTC0eEvvFc/tlyt
  mMa7G6l8Z6rvYLGmjfhONxnZsgxRYp8/MY8jQfjSoYwAqLeD6pYwOtYJfEcucPJo8LT6yDre
  Tj52G83asHmhT6oxbugC2jtI5VOXDjMtA10FMmHGJYkgMpl49aXWBCtWqGEuniGwKqnoajWy
  TbVJFbWhUEZqyVJZFKOKXWStRKMgtPb1f6wQkXScalfuVNOKjO2rX63I1nDp8J/Dc08e+OKr
  tsriohn+nS1qKqlL4LbJlc4F30Pb1rKDREqwtc2d1SKiTqWVNUhVpIA30wbXSE43ozKE7Ub+
  Wd2pULY8v9pt1moLG6hk17ZY6h0roxVR5LKGIswCIVIfVkHOWiVHnJKvjVLRP/VeZLCfVbAJ
  8cNIn3WLc/CFGMD+WWD+kFZhxqVAqpgiYREbMpW/HXDJLtxbDY/svrf7gGvLVOIXIYbE6Drv
  9H7rXzlKZHw5FRCcWpg158oYkvFVry+ISSFPADfFyWVyk1oJQLsAWMqioXKBekuPXrDztS0d
  bEwfoGMBOZh5LR7nlwdU30ca+U5cDfHfXKHliCbzt6XFsE39OTU9VwxAXF7+3mJwKU6jzAgO
  x5EQtXCSaGiUyLER5umkQBktHDEU0EM+I2jVu0JfORXJcv4INBKdvShhdWlEyklvR1UfKmH0
  SgAqC7hMe7JAn/k2MP5vPQoIJZ4KxTShtJVnm5qeSIvof2z4LnJkidCJBMmLXHxyB+4c6g8D
  9z9sWLVGXEnf7Kp3xvnydhNP3DYnB3q3ZTbLLmB8mnSX6X9EfDWxjdmvMs0oYm/+CrUt+2Bf
  dTqMZHaue1V7Y/Ps4miItvaoUYSBURk7tcaAcKSdRdzDSMQQmtWzhmZh7OTAG9G5NtFkQ8Jg
  Tok8kgJ3ciURaxSFYzxgohX4thfpXITLfKvJhOpteqP7rElcuckfkQcdFPNvro6r4AED+rFU
  91LXcaPZOVc5zx/whIA49g8REvoaiC7zmb8J6RRU+nFzLvYomJ0IY5spCdIuorO73e1st5Mw
  Pvv2utv97njPu973zve++/3vgA+84AdP+MIb3u4JAAA7
}

set pieceImageData(Merida,45) {
  R0lGODlhHAItAMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAAAcAi0A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoAcgLJICSKOSqGx2
  kAEnCHoKCAZMBiAbo5YAUGQWHM6Gxc8A9zAQuAcNcsq6bnlLVqxjS7uPyFBjZ2aDaWttb3FJ
  KHQwAG6QdR9WblELjwKSK4hXeJCVC5SQlgefoBucZpkEj6SicFUCpC2YkZ6nCpiaKpywI6K4
  wLILpsOotgrCrpC+nrMrj1isbbscsm16pQNtXautR7IEBIsHVuIExqUB4lbPFnQE3MkC5+lX
  rOkiSNjkKrWf1TZ8yvap2yd3HpSFMkWqGEIKwhYeJDbxSAB+D0u0OSf+rhOJcLrMRYNhrmPG
  De3YDYCyseNKJOHG5YNAKVvJdrnCNSoXMx2ikxQ4JWIhlFmJYmAGkmQIbiBLUy+FzXxQk4nC
  A/8yzavIBtcxpSxinrNydN20SuwOBgQR7aLWXzGl1mOYFmi7PPMAxBPAoBeDe9+6wgQKcWqK
  YpCOIl6sBpqptRikLp7slSrYiAv8UhylQCjhCThdiE3rKa0bvfhWIsrGiDM4sxxjy+YYOsI2
  0p2RXXor0aM6cfKeGEYx+daoYqrfQBZeeQrl51whFM2sO+eaV5sTJxx+omW9ZiHaqLniHZ3Z
  pL5N1Fp+IeXs97FrSx+Aex54nu7E/zW7bRL+dxNFDTVCLx3RxUpNKaxnHHSUATXdQvfJl9ss
  UPn3mQjRnENNU/S9NxoAiLB3QYYbwhXVGSimqJ8EbaVXCl8NSIhJA+KVyIFmLGR1WlOngYgc
  amfx9sdjr8FnZF0T/AOeGw7IyGRfAHmAI1H0uERCNCAeKRslIlKQIXBC+ndFimSG8ZOX2DhQ
  mXYSOYBNlzTKAtN9KBRFZwc6UiaNaXCyiFifNF0BpJYc+UiYj/et+WSbcbJ2DBQudvfGXJHe
  WOWBhJ6jiwjYxJYmhmhlyicGLrKpgKkvwkidc0AGlkKARoXHjGQ/irOprIvduUGeDFZHQamL
  nhpsqnGyqul/IIj+NxteIMQUoqipyTrosbpqgKiob2awIzE7YUUPb9Go6q24Efjo1DyxGePW
  QIBSp2cIo4DJIGojJftctRmYWyaKymkrpG4K5hTsjGhCZYlIq6hrZ7tdTVtopdqu8xO0YF4I
  gS7w3YprrxCDlt41y5iX2UririjBT2ZWUtJ3NZ0hoAawItdsXByvspfFD+QpGcONEvppBswK
  m44opHACJWEoPyUnPSmNOUoYL2PQypFkDSFLLbDFZ+S2xwwKD7Wv7TsYB5Q0BNNCIp969lYO
  3rNsJMeGKtvPUnP8ASZjTSZVh+iEiUGA6+wNDqHIRlB2dr01xObhJ7u9ZSSPbDGKw3v+4Rso
  4Tg3Sc8jrLy38tuWBzqte8dmLsHn8BUeaHDhGLPSOLCYU7U2/+VhpKEaOv6eje1xbDoDtm+j
  d98GBg00YoEjD1evv/O0aEWaAbPZSbYfqLKtytL3Jjtlo8b7O5du3bzahKIuG9d/hw8GRhx1
  rIH5s6me87eWhNPfb1Xfln8mfv8VPnze0V3qyFUBWimvA74jninm0qeiiOFAwggd+JjHAawJ
  zSOmyI12UiEBN8jrMWD6jmmgUqD3/c9D4yuPkVT4OAk2YDR2og0BhYM5BMKECCDRSg69lTwe
  PsJPWcuYTmxGtf6dzncdiNmfFHiQtLjQf5+IXwb1gURr2IL+SBnMyriGFYchMgYfydOLEvXS
  LQtMzUhlrBvlZHPGrc3wV0GUSRxn55waWmMdnegJTOqxR5l0xY+GGw0a6ZcpCXmJY4DiFWLo
  Qyu+cbECX6TV8hg0PqUwRBkREcoRkzcZMhLSgEM0HTWE+EQtCBCAJ5yNyS4gSDk+Di52FMg6
  IkezlcmCZpEDZKDimDr4eU5+UIJOKXeDSCYusGIaiORiJgmdSgLEIQxByhR3KRPKmKRKz0ne
  7+Qmw2FSh1DmKp/7IHDKcajyjShJ5fnGdxF5qWE82CAPNTzptvtdrJWo5JuW0mgBYSaxZldo
  kBOTuch1FYOZ2WTOZOx0LyBWc3j+nQLlKvhpAWV1Uzhr1NRq1Hk+dAYyow8bn+ZiaUJbveGk
  UEHps9IWgXL+kpe78+gElOjNrXBMiZMq4fFqJtIYVTGdAE1oBzuEU0a2Tk9gqqBF8+YoUtEH
  DLPRF08y1ZxfcTRug9MSMDUnjapwgpOnWQ2SjghSGcKUjVvVAk0BxVPnWbNApgMoj5qp0KAy
  xmPes2bTFmrS3/FjWXRjJR71OSphFTKt4zLLGQpLRZLudCDYcohVy1oXcD6yAjiNlb8Stpgs
  MChvpvtHiqKE0AbV1a6SncBpPlgMV/IVHzKdHxF/2VQ4mo9ubSxi8zJb0yM6NmJQoBgE3VIB
  jBUKRRj+KUNsNDaimtVWtfZbpCmbOVaYebCjvQ2k2MaGQKc88D2AeFCSVhuPQMRHkGS4TTW7
  ZNwjEUywUO2cTA4x20HGtqWvo01wSJBZsFhqUIDYF4DHScx9oHZMIbKWXf01noD6qr80uy85
  T/QUSREuu50S7kWLm1PPbQ+VRmwcZcHGSn1mwlU5Iax9K1hfk/LMTXLarvGSmZygJicDq/ku
  OA1MYOnY9YmtZRwxqVtNFlPmxXEacb2sQVgZe0rC1MECTG4qhsCCb8QhbS5R6WHPFEOLEjfq
  0GLjkV0YYzlaZHuDjm/HY9P5UlSItWlQLfYPvURNkQ4xjaViDLUeXwDCMkv+s0Gda8jCHNg1
  NNbwfoMCkMiZUsVFhPILoTOgMcn4Kb3F52HTeWa0Nu/Qku5FgodMST3LEj50/IB4Lg21TF9S
  bOKNjEO268B0ahiMUhtMer705ct6LAxjMXBPs7PgOw7XgMpNtWCPDasTeTLNh8YZGTS7ReaZ
  +njcG3OcH6DpIiIwvxTzEZPPertF/xnSmeryiPrB615LejMU5vG7W0puVM/bXd5VUaAJijx+
  LfMrB/amHEht7SLzm9LwireMzf3nVAZYDfGr6XWXOwgpGvvWyt6VS/fp6yQNelZIJp9wMw6+
  er903J0+EMMhCep778Guiu04iz6u5pCnCrXCkbf+MLlLUJ3fa8oX1zDJcXzVSFuNZG6xucgp
  NvRfoVucQR+5y1vu8jjN+doaR/r3Znbg7uabTAGy4tf1Pc26mdze8Dr7L0P8cZcZVWJ8juBn
  AGFpVus8c3z2HaQ4nV4GvURyoaU6vGDO2CY5sNWuXDWkzuUlSKn9vIjuJ7i/vHKPTV5Ubgn8
  41/pn1ubBiFtOc+UQwWJEcoJNQYOyF3sXqatSmXhTHnsQfYFOAW33DmEryboqX5kj8VL6pEv
  zObP6+Zuc3ybw5ehvTyve3KenRobV77HmJ8unP115Fa26tXCvS6B3F7VqI35cACKZ8QYmvdb
  nefof66GJbPyxOvXkxjn3kuq5OdtZswXsk/XFRXu7QgTSJBtOVY4byZcW6VmwrU+Mvdylycq
  Cuh9oMZ1V9d8ltErRFB+0REo6HeAvJcv6AdwqFVmNwdqGbENKQMVYxAgJ6JuGoh+mSN3tMcQ
  cAIMCkcmsUd0hyaCLmgbK/gjl6CCZhI6EjVnsgdzM0h1v/NwrNcYk7CEZNInYGA4PKOETlgG
  FbR4euMHaQBoljYEVRgIU/CFhWAtGcGEeCKGKZJIWMgYWkg2NLdQ/SAFcjiHdFiHdniHeJiH
  eriHfNiHfviHgBiIgjiIhFiIhniIiJiIiriIipgAADs=
}

set pieceImageData(Merida,50) {
  R0lGODlhWAIyAMIAAD8/P39/f7+/vwAAAP///////////////yH5BAEKAAcALAAAAABYAjIA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWze
  AAGnNCMADAYAgS0QnVag3vCkes1uuzQyVpu6DtiMKk5tRrnd6MD9mt8PPmpoCnx6fwyFdSpX
  NnRwJ26OCnI3jW17fZcLhXeAVgOCB34NiJGPhjOeAJ6lI1hlca82hIt2UFyfmlhcVn1cAVYf
  ZXwLAp8EBHpwiLSKpzKed6widwCwWDeidpkKm3i5e8F3aMV7yuArzDHFAMfHVtIgxqsH6+s1
  xQHHyXb5x7jcxpD9O6CnnR4MARwlc3dKTz+HC7AcwxcHVIdusV6Q8wP+L1w0etTucdQm7ltJ
  gOcsJDyUjaAfNC8risB4DYaVdu1qmrgy8YqAhbwstnCob2CJgAR07Qqox9avAQadSdhYDWBU
  BkqBReyXFA20jhZ+blpp00/GoxwxcplBU2gIP06huYnbcsxck26wpgy17cBXEGLxgN3JDicB
  rSYK9iyTU9xgElmlQuZq9qHZqxTW3aralAA5az71Cjj8L+i7aZJdmM17guZqwTDqon1N22yF
  jbFogn6j96Rc3m9Tt5BoGHEJfEX5tAt6kgXuxxyQGp5OHfOEhQIXyC2VTlLIQYW7oqZBGwVu
  PpW7uTXx3FLt96yvt9zuoDvIu4P6hqNx07D+zuMSNcVLVO1UwkIqinBV3YI4HUQBdg5yA1F9
  qTkkCHHiBcffalWxR40/fhSYCnQfIOgefO9VoJsmEzZgn0trRaSfBy+2sA5Op3nokwCjIdMg
  VDipQSIH79hTiy9IJqmkksZFoFlTo/znHXAM8NKAaVR2YCAM50FywnnJlRPkPCrI1eFOCjLI
  YIRT7XGmS2/eV4qVVZYD2IioMGUUCbxIZ9hP1CGSwjs9xYkaiinehh9LkhVi0TBXegnISDH8
  tucIv7kmzo9S8glXLWqGSlpYi37jgKMUWvTXpHZq5Illa5RAVI+i/nmWCOTQ6lmsJaxS63Tk
  GOpkMZGgyigoxhL+88aQEmD4mascMvsgopcUeGsIXc4lLQWvLumtktdORWyUeyaLEijEbvuA
  szWekIqCT3VKI5Ce9ffrrlkC5hOwocnK2L2kXVqBlE1JBcDBUrkCSpNjJKnQG0EaJZYv6j5g
  6aetaJsUilksVvED2bZKgqbUrtfsm5tE2p03dWIwMRcPj8bjroK8TBYInVVHhskZBPQqwB6L
  YEXH1JEhrAfBfrskmRr0C9A+W7GJj8TtNqDGS1oohhMuV7v5MUrv8YwBJGGi2GOuQnOsY3ii
  Mt1zlps4womykE4Z1sWfZA1VruJ1Tc3XN6qZYzz5bKTrvUOHMPSfhyduAt60HT1wOgX+GfUJ
  m5VfWDXdUBQYryu6roNHQga5rUHIq30tY3iQc7icvBpcfAvGvfIkargXxOdS3XxRznJ+wlld
  Blc/QUPojq4Mz1W8H2PHIEX65nPHgoAyyPsGOW9N7+uqr3u50rMHQ6dL2QE02kDGDFRwZvtW
  NxanfkacL0IlY2s7iBy6RpnA9F+2MaVoqdXmKPC7buAFJgXUHQT45j48HIYd79OZ05q2PesF
  DwP9iV9xKjidQoSDOvfTnikEeEEVgc4q0FPMPxbyD9GJrX0M8oT22CZByU2AWgP8Ak9IRrbV
  RKWEBPTh/2RzKER9IG5gW+FJ1DMl/olmImOhlQxJ9y9kQIP+dLuyYbM4WJ0c2oWGFgzV9Vwm
  ncBcTldQgwwXQQhE+RhJhVXJYByBVJo3OPFptRqQQMBYnfG9jVr6ohb+qgUisTkJLlzQx+w+
  MhsjemATnPmOmRQgly5cUT79uBjR9Jgzv/3Lj2Nb43S8KAF73S5NgSLlqSD2ww9Zy5BvI+ER
  obIPxTiIOLfsB9fSxzMYipGOouSXFiGAwzZOoHWXQd0PLxIiNjYnY4B85EcwAjzWZAI3vYQY
  D3tkO9uhzhhG0oDW1HRHEx5uTawkZzkjME7tdRGWoZQl0tL3B8VQRE8u4WYUdjg/FwWzg8vS
  4IJUySgUwRMCPKyMQFazv4MWlJD+r3Ho5KKpJXCopXd5uajIiAkkZI4mGcj5ZkAIqiw+bvBj
  gRNVSgUHu2l1cY3rpKA8aVS4y9HrHXrKnk3zwSaOAsymPzWmA0S60dPhkB3fZF1L5SPEiO4E
  h/EQB0261BZq3hCYkfPHLQb5GhB1YHELclwHfPk85YjKg9hL59b4CD01znSsmXRFVG6SSIXl
  BKw99R7AHBXUr6ptXtTKAjKXmdamMvSpFB2rm+DC2MXuxacBow1PlZPQkQrVYmTF1zAlQJSz
  OpCEEgUJKmPYzw/e7rKQNZxP5iIMn3SObKM6plpDxcBahXMDyFyqCYsJn4ZugKipQ6wjAcvb
  RJUSYsD+xdf53iMz3U4LlWnkwEp1RgqBsjG0fpkt9Zy7gRBaEGc3jZwmq+jczv7Ku6ECpThp
  A7hiZqG3jcFubmmnxsRyoLLF1WJncwsiripUINilx88MIyhmjsaUOCqDFpBTK7QaVbv8Kq1p
  fwne75ENEbBaw0YS8hRDZjaMv3qWdHO7WYTOJaFmAC69HEyFYnaPQvYtbHElO7keDXZj+HrN
  2UjKudFacYIYpCWGRMQrGP2KxS02hrd+AuSMvfW+cDlnWWPCPgi/NMRNvoCKs6xD4rwmEfhd
  MY+buIYxMVK4KKrwjJ1aZRuzF8fbDBpuy6wmo3VXyBGOBKFsy+XbkLgZT/7+bYCkbFtfDI6p
  h7NZVZGUZ78e9c4Qw5uezSbn38qQjSW+M/h8MebdKAWLz+N0UdnZPgDYQsro9cwuYKUusOYx
  plHyzDmBPN1fBhg8oFY1w0iwZUl5YM8N2jQqb1tlW6yZEKYLMm8lWggeBdfTko1vOLzVaUX9
  szGqGxqh72VnLVcRxCyV8BeBdmhz4igf5ebLtl96a+fZ6tZt+p6w+0wqW8zXLFAgA7PO44t7
  idrXDy4us6aKRH8aVNrMrA28D2nlRs/TpMhYUljFDTLGiOXehCBdsnN37ZdScMfOVtWQQfvr
  YHL3wet2eMKVBzPqGvqZQU65Sk8egWOjFqMSeZX+AzCOlDEqO3JfWvPXkrFIHMIM1ju3+XeI
  1PA6I31Km3lg0iFuwQCvA2+mpjjKgYY2GlH91cz0sb+fnvRjOxQ37+0Uz3tOdlhQPd2sErpH
  lL5ypd965ADbtV0gMa46iV2dAVbLv31O3DVP2HPfmvXNsTJaRfcboHM3fJKLuna2B/7abb+q
  zCW4+N48vsGLBOznzxp6AwMtYgE2kXaaDnjFip3Jna853Ute9JJBaaz2XvZTpGl3R7+p8pZ3
  /edOvHu3nt6rJT/+Wuel/MYkv/lJyfRQJ735gY7Zb7n/m/GPX+3dZbzlf0oSzP/IckZHPEm/
  KbnSpW+1SS+7MZ33ZLT+6nvsWQZJ2InMyd2353hviRD30BdbM2FdJBR7kvAUqaALL+ZPzdd9
  7sZ1mbd61VdDhVd/IxBm//U/gICAwpBvpmBzs5R1vJV1tzY0wIcFYtVdf9dgBlgfX3daLQgS
  7MATcBccDdiCqdZ/qMZ+WPGC6WWASqdAkWd08KcvMwgVNRhVIFiB+Fd7PCgjUoV/RDQ2K+hZ
  F8h6MDgV+oMkEkEODCF+WKMSQUhlPZMFtddbHBZ7RLEUiHJq0UV+SxgCGJiBQlgR6cFpSLUv
  QTF4sLFbbnCDvxNKVYhOamhWx9dscBWAeaVYAaiBEDA0+WdGn5UKZXNGipSCpDaG4zctLzH+
  b2SYAZriFEtzWNExhtBUXPsTPCfoXn6Yaue1ifLhgyyFXdmjfOqVO4OoJosYHbmoTsHzgAsi
  IFhYKCbUiP7BbMMzgQ2ic5BmC1z3OYoVhKf4fviTiYFVecMkeE0IhoEohoXDh/mTfy3yNt94
  hvj2eEg2NrL4XXJIgK94QWvgMH+yCkaTaA6jd6tkjFsjUXygjMCSjhbwYQAjYhQkje24Zqm4
  RdTCI2snhprYjRNlipAmkXO2jdS2cAAhihbJhVWzZWeCcdBhIRupNG2HdhSzJnwYWldzaoX2
  OYmQAas4I3U3gv8Wb/DxkYgSkg8JkVUWhCoZk1EGZb13kJp4HUOA9XI3QwxNQTGncgFzWD+/
  ZY43mZRRCZRUCYo+OYAPaZRWg5SlwGROUQrw9JSIgl2SeFRXSQXDt2wLKAZuuQP9hyRteSrg
  QnxI8pZ4mZdxyZQpsCTUoCR5GZiCOZiEWZiGeZiImZiKuZiM2ZiO+ZiQGZmSOZmUWZmWeZmY
  mZktkAAAOw==
}

set pieceImageData(Merida,55) {
  R0lGODlhlAI3AMIAAH9/fwAAAD8/P////7+/v////////////yH5BAEKAAcALAAAAACUAjcA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9DoAyCtjgiCgIDAo1o13q94g9Vyd2EdYP0KBNINeI6wBpxX9Xw8z17Q+XIda3d+boaECn8x
  AG9dfS1ugQqSNn92LYBwmQyWeiKDDgSGbogHijCMlIujjSyjclkCPKKjpSasbg24AQyMuCG0
  brIMpAO0DVmHL6lqrKojrw6xs6y2Jbu6vwu+rMCjwwvYDMmky603yXXCrm/Cd9zgOMXHK5Fr
  WboCdfjbWuq8IOxlSaNlgEF+k/z5MvfMBblv7Axt6bdOhzgVFxVk5Db+KgQugtoSdnPB7AYj
  AQZTZrE2IsvBcrQGDLQIICUjdjYBhqtp8GY/ngNyYajT601KoRoDqNR5wE3ONERHcDTUMMXU
  WisewnRmMWSKjE29XmVaIarIkUnRhuW67VGIsVVRrExpUFS8E4wI9CSV16A9li4KvsS592NO
  w37JSgim5edRprEATAundG+Yh4Ax/Il8iYZWiCrG0vnYuQbYE7tSqz4dgTG4seNYxdt1GSuw
  ewozY6xMN6jiEqlSMj4sMa6JgWsa1+MNl/irxxcEKrf7Eg65u8HLnDmZ/LeHkjZWe5cqPrU+
  z7Ijll9vKHp6Ux+RSYyDa7vXt+dM86aL9Cv+ypeR8PfGQ8ZdUQ1hvSWooIL9LWZUYgto9cBM
  DrwX1lMkgKefaupRlZpk0cTAWAC6BcTeieM1MCJSEkqTH2XzpRXiJy/O4FJvdrGjl0QPJmYT
  OSVmSBUk+y1oZIINRhAchNskA4GGsZ1HGYbkFSiXaneh9k1iuPyoDCpDdojiahbA1mSSCUkS
  GTHxVYnDSb1RmEI6A9hhjJ2P4ZhMlinQ4kuQHjh15KC9oQlBMSdVSOKEaNITW02OvpWKlSaM
  FUlWzxmTmp4VtTAioB3Yk8mopC5kAS5ZluPib5Eq8FmGk+IQy1J8CpnMoAL815svtZZAi16/
  LjdmeRd8qiif7a3+GgpbIWjVq0O7PIvfN2WoBlRh0oqw4qK7EeotkxUYmw2yaDqpIrMgOFuJ
  L4+VZkJMWHyrYDCgapCMXnVd+pUwpfaLm6EQ0JErub8lm81vkulTrwSxADvZDKN9tHCxw1JF
  F70qfBYmCu7Iu2C1miVM8AMGs4lwru5+0LAxD8NQbY/XTUzBQHTh6bGmyp2w8sUtHxfgzYdl
  +yS3JmuixYvcLEvpBNTRtZIML5sn8wRagXgiULQIzYHGoF1ZcbQc5KUo0hufpbSvgi01NQdZ
  X+sl0dDwFgzQ+eZ8xawLNjynMHTvKUiSMfYTlHUlJ1VgxK+806NlflyV8gfbprb2oRb+43yi
  rlm/y7VskzP8tYVgAN7pWYSLfrjjikNKYp3nIE5V52zia2SrHvE0N911L90a30fe2ye//ubh
  +98vRgK3S3aLYipFxm1+6ZJPuQ62CGPeUozlKGKur7Zjwh4w7/IOH/b2MhJyHScdFVVjBM43
  EpwhrNthaeAfQH8klAGpDj/uABq4+sU2q5v3HhC1y3XuS0lRHptkwpS89CcYF6ATjqbiNgrK
  LmjTuto10uabYQHFXN6IFgN3McBQ0OxbJzmgqmQEkpJxBH0AW5Xb7uQM1qHEggnilUeKdKQY
  hq4m17nfBRc0ukD9r2aLq1s9ZmgknwSkZU7pj1GceKH+xCL+RVEaFEcKVTkj9WwDFRPSEecX
  Ca7Jroih+hDrUlMPfQQPRNQbnVrmmD5XFY59HFTQFkWxoy7mzW5kSOL9sJiBudxoUHnU4wr/
  piskMlFOqGHigqgYqFwRzSliO0BwSqI83xRCH4TUpKAIdS/hgA9X66PYsEhwr8h1qYOyOUzn
  5lcTWu6mejtc4YFEGaIRFeKOUxil1SIhu1KiJGvC4YjNyPdDeeGPbYlEpCS5qDXPDRE3bsQR
  IK8xzRyGsgIryZmglJOdVtyIRSTyYQLDlza8eYt2Q6mY7pY1LOxV4zHVxKNsBnGnYdLPerhM
  V+DeQ0E71vEhFbge13TlTrwVUCL+SuxAJ1EINw/MxZmCVGRFNTDKwtRQQBjp5q6+SQFGxC8p
  gxHlUYYhGKHAKZXqu9k8otlDmFLAlavxXvtos0YRFmZyuzASa+IW0PqFSBtTYelIlFlSQZHx
  fzOtDE4fxMwM0JRBJJ2A/bx1UUJV1aozxOYavBlSjNKojyg96ShzsSShjLMhWyUUdbpK0Xmu
  BUU0GpZenCcgu961OHbazFDjWNTvjCQ+2LiKjGyaJi6tJl8omctUncpYqh3xSDkCQdPktVlS
  qnMCGY3fJLPKUZEikUa1bMxaZdFOXhwyFzR7Jn1MmyDkNbKuFh1TPnc3rGOK54OflcBqUsuh
  fRWWePP+QdVfYfuLqjWVJ+v5j20BJB7GSfR6s1tkGoc4qF9htLLC5aFliEhasJrVGzwJJ3Pa
  AZRUpI0UgsmkBOJKqJNcFrfXRRHsvkZd8xzGrsPt6WBzidcQ1oI2CK5PfZ7r2JzacIrssW5+
  p/mnZoV2tNj1Fhox0FEAchdckTyvR/5jF0wSIBggAuU8fHNRSg7ttt/a380gGUj2wG6nWFqP
  gHZ7sGj58z4bPG4av/E5YQA5mNKNLpcajKXqaPZncdIuNGHcRB7RdinlLcqVqcSxLbs4jVg+
  mphl46Ex6+qzMQGaOz0WXKatB3aTzfF6gJtlisjzlgUmcJHFEy5Bxbkys/r+M77arNUOJ8av
  Efrfh4mzBfpWeaOqXPS8pEyCC5M1jmeUiLNGSGQZw5NRW34K0BKVwQ+JAMeokp61ZJnfipVw
  bHk20Z6JBc4HxVkvm3zzXnjsAG4w0ddPxNnHNnZCioJXSRnOLqQr7eU6B9MeEK1THWSX4vQC
  z299vm99LT3pZceTPYh+xVPPger1HntoehVWrI0669Vozbv25GmuxYOvYElUgo/mtWOKfbEi
  Gtqr50a3G0klvq80mzyIuRlcIphsLXJbm9vMwEPFo2/S9Rd0y31su9gNZSQi0LjrNmy7rcVw
  Rf9WtE+t94ZVSeU/ervQNfl3wUXZ8ppW3A84noj+Kw5Oo2NK2pmDoLFltc0Hd+eBu5nzwJ9J
  OD5BbQ4R4x7FjjVbygRl7dWHGri/EAICx/HzW5ZokwWqjoWve1OSf/ifDic8Y0rfdIpIj3jH
  Y0xo4WazZnBsQbm7JtAPd4JUfo/40OW3dyzZAdut/trE5hY5goH7Mc7Ox9+DHnkM8NvhN5eP
  PviXwy/O7JSDvDLit/ZwBXk+obc1aWblI16AV76xo0X057+W+U9Ku/DfGOvTQueMOgC9OzPq
  AO4x7p4a/tOgV4N8qUmOqZr/sfaJ+Dfnd58BXtkE+NH1PYD8uua2L42uRqE+RX7+6B0iEvqe
  c74XBW8v2rwR+8cvVvf+p28Ggc76+7HEGH3G1N66U6486KdP75cHK9d+tfQvl+N7q6cBEzdr
  OpdfnOcbE+MO2qFA56JtxvZkfMAZ6oB1iaZ+vcN+mkFMERgxa+NodCNbIzhyE/gc6UBP/HcY
  T3Z3uwIiHoh8RQZ9S6dfIjdyZUN65CdXIshbFsN10YeBGagyGoQXojKABBhwESB9dON/FQKC
  7GRhI8drrgSDMVgYfedVsodkQTg7QgdGvTdtk/SEdTRkVHF0afgPaxgqVghwgpAclgQH8IY7
  FWZEbjREgvV66NZusndIajhcaBgUtSdMhehj2udJsjZydchUMCRPrFZJdZEJS4ECJhWBg2P+
  YVtQgqfHYWYAilQ4W5x4UkpHY3moh191Ab4WQP30g6jlhJkAhYE4cuE2ZjlYiurjg35VBnhY
  ZP1ni0WRc8TYVKd4jKbIiSroHqFWfkp4in4Rhhc4hg6njJOQc9RIH8koe7zSiGD3L4j2irCI
  WXCIaMM3XCewgzxVGH71Y+axjb14f/jBE/uQfWOFioZljwhodPnYjKfSelMIiC5ijdcYhmVg
  SQp5g4LDidjoGMz4kAdAV/yzgBzgg3F4ans2jAgZCx5Zf6Hhi8tHj0qIkYkokEDDi8g2h9sm
  kUcoWm5HHt1oYdwVPFY3hAklkKQiVK+HkcDEcZ/DkdyjFO/zcvX+SIuA4JJV44ZWV3R8d5Fn
  aHYAtIg9iZIyRZALJI2+8Azw2ITGVBBtA4f7JIo+eWQBmXZfMwgqCR83BH/skU3iZ4Y+CVDC
  +F9axTXCUwx8E5ZuyS9kKWOcZ5Y5yZKDVHnIFJFGSQFSmJJYGX2EeZC8dXdomSn04lFtiC/Y
  FCRlOWAOwoj9YkY96Jm1yHxs6INoc392+T2b6W6ayQrJqBYrKI0SKHKy2YruUZuNKUrPWFMw
  1Vl6JDt+gl10gEqtKZtIUmfwUI4oBGzCJ24pOEyZt5nAIYiQRwns+Aqf05rihpRU2XTX14+e
  WUzYeCs9cY/+aDOhGJC4WQKXF5i2qZv+0paP3SYLVadHvgeQWWmc0LEB5HmK6amYpZeEpQmJ
  rESdXKIKUadcw2ccqwmbDGdJDvifoKUF6SgbIDQ+3FmL8gg8GaqGz3Ce8JBoXkkmztihpVKK
  1yGfnIV9FRcMCsM/CAiUghhkezZwsthjbymiIRdeDYqim9miDSqjJElUZfkMyrMd4AmSOIcq
  UTEaoNKjPxku86N1YikxkAMXwaMxj1N9Prmhq2mkb4Ck8PiAifAZA5cI8pWTUCpRCTpnk6Nq
  GzlLJsoHlTKno9I5J8Y0r2an/nIbffkhoFCnw9eHUsGnbvEJhsoHeJoZeQoriboJmlWlhril
  XdeVABioY5A5qZrqBKSyqZ76qVDQqaA6qqRaqqZ6qqiaqqq6qqzaqq76qrAaq7I6q7Raq7Z6
  q7iaq7q6q7yKAQkAADs=
}

set pieceImageData(Merida,60) {
  R0lGODlh0AI8AMIAAH9/fz8/PwAAAL+/v////////////////yH5BAEKAAcALAAAAADQAjwA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWpcDAGBw7Xq/quwWfA0EuM8BmmwKCN6CADDO1pjXTXW9Daf//D4AbnEAMXBnEGZC
  A314LH19DZBwkpMCJAODhQ0Ab1qeDp1xji9vQIJvAZulqaQKikGMcK4plpcMtpWTmJqhk6sL
  oogzpj6DfXIvoqkOg8k+smZvtCe5uJa6kCOyh5UDBASyDcfTxLc95MzKyM3q0JPUJtYL8wr1
  H9zu9tgM6fG153a4OQNO2LpP5V7FAWVMwLdwgB4JAAeuGL2JFC3uo0j+QOOHaeHcAHvDUWMn
  guKUIdQC7Ea6bi6WzVpArqWOl88k7rq282JPEJZG8lMgk5JKUSx1nORYUYBNFJ0AhCwnS01C
  HiQzBlyRtWmlklu7dtxaQZAqBksLOr32MCVRpxTd4DH79ENROHVj3O2Tt8ReLvmu7rin4p7h
  oRfoog3KU1swS3PN9OWwdy0OuUwJnIwZJ+40kWpVaRmM0asLsbZKj7UENgOkZ2k1W37rFOli
  qU3X+PO7Uss/Fy/ZtaicaZLoyTAIA0zNvLlHCq8f/1oMCdhfmpB+Z2DpyXcOsa1Pq5YJtjqO
  k1pEsio/iT2k8BYGpp8NsnhLcjkPnPzmFj3+qvwjPHeDc6w4ZwtyLNyFoAgGNogYBS+twlg/
  wknnmH7GoSCgDeBp9QJmU6XCEXr4aZcCfsmpltmKLDK1IQRduRUYgAdU9cBu+zw0loZkDdjc
  h5YER0iJNqBYoINIYjDPjA/Y2E4jPl1Ywos0gMiRWwnCxQ1oFEXVJV815GMigyq2aGZmVDqA
  WkDFzaamQxAIg8eaPKLzI2fINPXLl3iFCeWRSDaoJGJt5iUYdcPs8xMJacoQm2cLgsANRFIx
  oqNsuPE5ygwDDZTimaCu2ChbofnSI4ZuXmPTQOBgOYJtO1TGEAuB6XggU8skCoOR4rHk66/A
  BvvrqFHCRF2cs1b+suqDIsAqUGee0XhCNHCZ6WVmy4wpAqsEeCpeoA5iEBicnBxK21Oy2DSu
  thWs+2yQLqwrW2pb4DrTrhmKF+q+Hl7gbrnkAoxusgr8uw08PIDIiLQnDMRIpvxeeS+tIkbL
  LgfgJinunwLfqA92qdJmLj6pzBfwZfm2IKSBBFHEzcUcjHsIzBpUHLGZg2Qg5gPOUhgRyE/J
  RHME0Zg89DbHlKfrCaLcfOYgR1sQTWZQUwyS00y93MHPIj8QyZshH+AGQJfmfEObtixtgqwO
  tgxO1SvInJ3VtQlrd7AT10zjMl4bpctTY/NYNrEnzjziMQyH4InLXGLd7civ2swU3GH+OOP4
  2614QDCqIRNC1jI2iULZr74EQHXIwKqwcnTLtfJ4oBBTznSDUbe7OrhqDxp6n9mYijrhnJBe
  rumToy58w4u3aBCPLuN1OUTAZwC1mdGoLHnEzoBg9m0/RxXVOAvZ9DEFaP+CxrUjWpZFarl/
  cDvr1TiPOe7NR4/B+xWqjtfddhvbwfYWyo9M7sO7aySuSSsbA4YqBZLv1UhW7euAKC7FonQx
  D3rVel71+OAQUG1wBWxrUKQksLzHoKJcnwDGwhwItI3tjzuHSCG2alMTX3EsBCGckDySN6+M
  ZeqDJMAf/FQgp4gVCocfkwnAkpUPAoaNaHyxYQy9hBRMoWL+igiBHMY6GCpXBQgjfHmeZ04W
  AodRjXhX8lbcMpY3D2TrNizE0Dci4j03cSNS0aCgpuCAJmRALI1arFnGeITGHmZMR7IrY2r0
  lLIw8DBUE2xW3gZIHX1QcgF39JfrVnQX6M2xOnoMUe3eUkhQqccvnTnJ1SrIxTP5DYllasqK
  LJilP54pjnZZSE68pJGs9HIicTzGgqZ2JrRdakuhLB4ZKcPGEVqgig9xjumEVMhXgoBtjJxO
  lo7Dv/k4k4T+K0pjfllAsZVTAsSk3jG+YQYMZrBFiWSmLa31RA9gxlKQfKfyNre1VrooltAb
  jj5vWc8NOOWUC3zcGnx5joVh6jH+qjhV30qJs0cKg18AlGAzOSgVB0lFVjrq2UeYk81FQWWj
  X5xVURZ6ocCsQokWsNkV5Te/j37joq0qitsyKs+I4VJS0PLpQFlkzQ7IAlu9sWUgJTnPfX5T
  AtOwiJceVrBqdQINoGHhZyQKsJsNAiwUdWVBn4nSg/nQkK8pjwhkdpyCmGxuICyr4r7GOYb8
  Kx0SousEmja/oIwRrHtMa1OeikKhEjYCq8SoP+k5VrL+sTKPbew1h+pURgHTD9cqxrUcmBUH
  sopYj9wXlx61L9Hl0odTYuNNmRNSfmpgL02FbZZQm1J13EVkvNNmOmJarduhcbRBvR1uTOtG
  yrLop/b+DGrEOmSmom6gcTlNKgWRi8qmHleyF3BDbhL60M2uxbuFOOqOKhDaLjI0mYydLG23
  xUYwMqea9vOYcVji1pmCKa7rVSSUFFRXUPB3XLzFDctadd6+knReQAUoUZdpVOaa0rgriqcg
  revgh8bVupzErmNlI4drLewAdALR90BDVQrw1WkiQd/N4tsAuTlnlFVVrTttwREJu6Y5JTUP
  fsF1WJ5F5y5yyEWEzHnfvXZFhJj7BPGE2C/3KbeiB3QhelskXn6JdAMq/mdlL2xYElwrxRmM
  qmrE/CWEqGVBFS7mQRfLLxaDL1BR1mQzc7wntWg4AiGk8zlnl1/1RhEytpD+WRb3DKPSOIg/
  a0Z0g5rsgXTC080eC6uanxwq527HuBWmrpchPMMe+wI36cpgVEiihcc1btRc1LQ3HGc5rHlx
  Ay5OTRDbqwr8wZerZD1wczxdFrn6mS9BSVuwtQkdLrZtfsRDEoFxLWc2L5vBzFRwBcmB4qWa
  OM3NhfamMdxpTHRFmJ/1RAxnIYj12UwuAXZcFbFGXJIdOnLtDVSNrW3k994OxuDss7vjjSQT
  JU+IGIGmR+2MNAdn7otTzpowsZ3hOxea2w0XKMTTx+uvPPu9xjnELBhtZEl7MLHYczg63xdB
  g8Y7h2G0M6/5zWxY8ji1LBcUWX+L5CqiPLgiJ9r+V0W1KUkSLwvUuy/DMwPEaCc8a66tbpct
  2zKhpcN05fMcTds9gao4buizpLfUSI7vm08IST8c32tPLlB9jzTmBpJzW5A8Fjs3B41Owsc6
  WXlwJzNQwcKUDtYsLT2Q75NrJ514mWEOmS755kosWTteyHE/v1da2lQWO6wB3nMnb9xAL0XS
  rfWLOzy93LJoJ1B2rxZ1v7b9de+tX5yvPYqPax2q7k1mLxbj8Wy3vNeEeGvur1jxDVs55/mu
  zuVKD3jYO1usTiu6Rg8JlI2jXoe4NZBaXU7ZS5b98/AOPftu3Mr31uv0uhcs9FY/AU0ItfgX
  IIlDiS728uZT2zoTYsn+18Zpe/UeomKcZakRKjXnI54lJOcrWfd6EkB5jWRy0PJefaNsfAJj
  bZJMeQQD/tFN3oRKr0FfNwOAxNZ/pFcbU5ZmYlAab2R5R8d+6EdeARdKakQd6pZ0BiUaxzQf
  MRBrL5Za1iUGd4NeVLd1izNoLLd/kmcHLEd+C/hR2+c7DpI+wFdomhEsxTGD7vd3+LYPqpB/
  2MJTtnM9yKdYJ6h2jhN3Y/dHjIA5NJIuLbiEn0ZP90c+LDeF4gVDzVRqY3FYwcFNpaWBB9hg
  MTc0e2E6+YNJZ6Vy2bdrL+BookWAbAh5l+OC5Sc5UsQyiTdvXciBJRh58IdOT6Z+gGeI53f+
  fwwHaS5UexV1iX3HPt20OkToY4RghV0yewmGdjQjM3pShnMmiGQic3RDgXiDiI2YKb4ifwJY
  KsWVCtonDWhIIaIYKivYdy0DGugGNpXYcL33MPX1K5gzhRDAiSFHis2GgV+YRbVDWlbIf5aH
  dqnoM84zgtkAO3wyjSVjNzbmbaGHb9QSevGYGNq3gY2GdaDCiyBzCBBBI9NzOSVUjiyzRpRW
  bdzIW9G4L6B4EawIH684j//jR/7zjzzGJ+eYCMcnMRsZirp4N/fIfUFyisxxiyXJP6szAtD1
  PJtRkc7zjMiYf3wzV1iEh3wnSfl4f1kWDrqoR6o2KBVEgQvGkvn+OIkFGGhPUos9RCaHGFfJ
  eIg8uT8NiSu8h5KHp24z9UUROV6NRmQhwy1iVJOnBXFXtmk7GTk4GYfk2E84GG8w+JDFEnqS
  ApbLcnKagkSPWB3csYZpaIVBiQE96ZLHeBGCV1pyyV2sGJhSo0AzCZg5mX58CSwmtW2LWZgW
  MpkUeCA6GRSbyZnUp33T6JgYyZRkqTmDlAITGJLC0pbLN5R3wyKuiWUOhoPCIpuYqZiQ6Zfg
  E5WImZhZEIDY+Jc0mZv7IIcuuYPLl5XfiAq953Wwoz8/mJchEJzSRJp8JpqW1YTQ6UcddXu4
  x51sJBpz6JRd6ZUmYHWXmZgM4FAkIZP+tXSZU+mbUslUXcmYKHiU7OkANDhwbpeenfGeCxma
  MTeV+vmc+pmb/Nhm4CmEh/l+DYqDtGB1mghtEqqX8qmWWcOa06WgzmabwpJMiZmg+/lm0/mf
  /pJ42TgNRwWG7amik8VNrAkss0mbN3l0tlkTKLl730c9vyglqHme6CmP53maeMYcP9pBAsej
  SCpIJAqkghSC/FZqctmiPgg79eKPYKOfKNCf/tmUrHcgeLhaWvKOV1pkkplyxQmlN0afylil
  Wvg8I8lbQlqiJlSkkSkMhyelrxFNfBQifMkfRjMoT6pXGhBrushav5Y2rekcFfekJ4V2vpin
  hdoc2lJna2r+qDoThWcYjpzKboxIp+dppzmCp2nSkkgVO3EgltxJUES4oFfXoEioClXpMoPG
  a3jVo/zCp0FIqFwaP5JKnRFggK9BrH84AfM1o43KpvioGjmKpMGoFbgqFs9KL9HaFB8JPrUa
  dNmKrIroNEbaN17FonGabTcmpH2UXG5jhcXRrVT4oKByRZyXj+6KWKE3qY0Sb1bRXiRZqclV
  G5JakFsEsAUarrQJrwTFmzUCqxB6bcEhgHyKBndUL3upcc7EsE4DaVFEkzClh06xrfrXRjZ6
  lArrpV9qsEsUKBLra8VWqRpbqPtGop6qn8PZX7/aiHUTQrpCfFGEEOfospEZpj2zq4sPS6DX
  SmVrGaq4d5Q1252pibMVaxx4wLMvJLD56bKnRbIEqp02mY9d6q/kM6EA6B1NQhdky58XA7SJ
  eaZYio3WGbCg57XTArZVJ7bpcbYtZrbYiUlpq7bN4rS/sLf4YF9TqrB7cLh5wKESmLQpORqI
  +7iQSzSKexSE27iGG7mYm7mau7mc27me+7mgG7qiO7qkW7qme7qom7qqu7qs27qu+7qwG7uy
  O7u0W7u2e7sfkAAAOw==
}

set pieceImageData(Merida,65) {
  R0lGODlhDANBAMIAAH9/f7+/vz8/PwAAAP///////////////yH5BAEKAAcALAAAAAAMA0EA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWq/YrHbL7Xq/YGQAEC6XBIM0mTgOmDnjt/yDVhfbRoDeBrBL3EdjAwCAMnqHh4UK
  iIwNjIgnY4QQdWkDAhCSijR9a0SHfH4Rm0OCkzOPepupewysniSalJZpmA+yN52llbWkLX2W
  sAsBokO8lzO0tMLKtA3Nlia8tq5pAQTEgw7TudpCAccCvizAxcPmQeHJ0MzQA8/u0rTUC+XL
  2/Pdwj3ZarTjKvpUoqfgHxs1eqzJqHUITTsBDdPAg6inTolydQg6JMD+UWCDjMFC7ePRrxnA
  FPaQwVNYCtpJFO7aQVvZ7GIzjdBwKhv5QleQXhyxoSH46xKBcoXqAHuJY2NHbzDUBPW5QOrT
  dwysHpVIwmo2RVqBMfh6VQa4nEwXulMJIyXLA8fS2jiGjkVMmsrwOuvqMmu8czNjnL0pt2ea
  oIjrqsjGcbAbskOHhOUaddBUqAUtl/UL4DLWDOXaIXZYTUBQ0vWMBqW8KOQJt653wL4Ho6RB
  22yBrFXbTGZNv4FBx87cu1rx1MFbKwYxG3MOrYg9xjh82trQ1QxP8XC6lahdzVtjembnWYO6
  qgMSY+6H1Lh6BudNRKTI08ZaS95XrB18U5X+D7q5uXDfgATuhUF8B9wFmDKKzAafMvkxV9El
  oPRgDWJBZaNWZ9VZcs1VOxXGQjaHvPXCZOKB2Ft5GADTmYvomRbeJrbRaFVkyr24XAhU+VAg
  bwXGJOIKuA0wZAhBJvnXBbB54o6NBo2VE3I7BeQcD9RhiA1rLgiUIS8YIgThkSggeCJ4Wqap
  ZppiHQiedAfglh+O+DC4AHdtotCjhQQa4g6AYs5D5glmCqjkodG4edOC+M0yp0l3LvnalTtA
  x2Jt1B3DIXYfCtJoDgPRYOmapLLJJQUojlVJfXQ6gBFYb55Kwp5Y9inYTRytlaE9EcYQ6jqI
  HprBbsOsGkGrjuD+B2ty8lGqAzBp7limVBt1Eiaa4Zk4g1LSwoRtqeBeaoGXHCGrAGoPJPoA
  nKVd1isI5so2YH0pxIUdNFraM+gImq5D3ysAv2JRi1PiU5+6rgaoHIQqxFupah3KACa4oxLw
  KyoXLqXWvwF3jMjAF2QDUR2+1BIBwivdwhDJelb5w4ASN9MYoIN8ONqnGEM63bfhgpunBUWW
  rDB6nzmAsgJBt0ybDyR6ti8HS1WM4Rhr9vPutEZ2SO8JwSKqQY0QeLjucAtmYiehBfOzXwyw
  CUCzoFPjDAPNW5tQS894lyvrBGCnq+3CI/VjdpTyLNpDQrEOUvcITVOd95oZ+XphdCb+vzCQ
  x5hfjIEkbvNENuBj712Rf2iL49nibTXXSgtFImr6zUM3PODVJGRk8+NpDka7A5xDFHZdoYVe
  tCsAUPS0BEPZDKMONWZ49grk7oq7lpG7QHLulXcZzPT5fp6Bhn7LOtTQBsJzPKpZIwZ+Dq33
  RU5vzd33ut6xo03g7iLE37X3m+9dfqTZA87w0HO+CUzOeQOkAXs2FZ326Ac8g2Eg95QyIg9V
  TWwusBr3OmQkJJHvEh2EBwFOJRDx7a0CxYOQdg4ALS0VYwx0WaEJ2vek7+DnXoi6HQWJdKgC
  Am1/9ylgAIkWQgEO0B4PAJkGUjiPFbZQErlaDxOX4UOVQYz+VCxTQZY0uEH1FHFSV1STY+aW
  HcwxwnYiwCBnFHaYUznkVIIzz+zWYK3u0fF+KKDh2rRoQRXl0ItVFJ6SAkmBjGwQjR78InHY
  +D+4NDKOGXgbhVrDIUs85Y6z85aMwqVErqVnS9vrIgITKAJD9qx61gOi3ECwygSlh0uMQcMm
  2jjASnxvVYToyBQvUYdo8bKJHJKEsuzWtQdeUZXpAyUpQ/C2+OGPmX3Em9X4wkiU9Y2ItTwa
  33D5oQnhp5cjpE4tpnGIoyDuEoRsId661QHGWGwZooQd6r53NzumKYsZVGU61ZiZrcDCWj2y
  Fiz5NzieTeUYWrLN/Ex1Qqjtb57+X7OkF/e3yXc66wM01NEe82nQNWlsVvx05XBS8s+8SImd
  C5Ighnb5yesoVKWUa+gGpObR+jHHKCIL4z2TWaqQerCilIOpRXcm1JouswMlHN5hiEEPgGLm
  QlxaWgWuBy7+SPBV4RIZNYP1TOFo5hr6uxDdAEnMtWiUWGfiWMc6yaP/MYxKxfgNESFK1VJZ
  dUtgVVZWh/iBLElTph6wik7XRNOV8vUD6rSnmlCaxo6q6Wf5GwS6XBnFQlxofe5kjUBoFYGl
  5K0S1xrsBS+qAVWm4IZhpYUflRHTrlJAf2dVkKH216zh6EykBsKNcRjr2Y6Eg4GgtVgUr8jS
  +ZFoBH7+XedRPZDYvB2QYn8L7Lc4R6FcGta10i3qYwG7xJq9hTpUyRJl6kiZr3D2d0Dl5AGB
  kV5wsbUDpgVj11YLz3sF0qyqwEaJ0taC+IKUcMfBrbrcwijSToSDLumQ8uqZrZxErLHT4+5M
  eercT/aMvXy5XYN3ErfllqCwDPXkO6lBHfC5M5yAmBxlriNhFj4Xb9eJYnurylgDAtFb+xuu
  rjyD3ZORZ7vMMmYxy5q9wCRNwCs2KQonJ0mIxTicm2yyjI7rQQ37rMVumvGVtbxYLPMtuco0
  iQs9zBfthpiYle2n3sgQPV1EjytkaTE4I5y1E+OuxhLwb+2AWMkB3U6bcPj+E+QkZcMhf/hs
  7HBxcWCDNLkWUrwF2uRlmRwk7EBUAtE7pYE3YOfHgZh6eO4smINKqlDD17FnPvRmsuSROvrT
  kZviimDJTJwNeok7d/ayqLt26QrokUD0hZA8IzuvDSd6tobeaqKm1CRH/ibIPv5kaid3a81M
  28KAjiiFSfWVEpBlenPGWzmITSpX+7LXpzZz92gdSc3Ywa8SwZNE4P2OFrLLxhamczi7mG16
  do2Qv/azju8TU3ZPIKyxlW2hg2W/e8QDUEiWiFvwJ9Frp4+WjVFSnw0+gbpy+7B0EC2NUd1l
  RQY634rtcmXUbViOj+ur1vDrG3nqj4rGPH3nTeL+qMUdSu6BfANN5m8pVUk/AjHQp5w2urEd
  vXCv1ZZBMTHrk449VfAqSUahKXqBPvRzoIMYwzbZtno/vdNNW4CmbTDqyh8H2chuanzpaUNJ
  +lGSo4NwkzlPGMofR3d+m/0CQc9H2JGZpIrukJWzs+h9KkPbp8+D8LWg+mshnaRr9D3gqo0i
  upHHYFDb9OQsb3nP8wZ2eIncvZ8/dOjFNQJ11t1Dj4887G9+wMke/MWe3nnPqMyvws8Q8odq
  7dCR2WMO6Ln1wNc4k6it8eEG+0kZ/zvB9s6p0uWNP0bKdIVdPjgr79XkfFz9ZtA8nnkFyebg
  97H4x8xl1HN/8gQi5Ln+kOmpyntRBJj3M+OTrezkBxEDWhF47iZOhcd6IMAfhNULh/YhblMq
  lzN/62dfw+d9H5d6qsd2unZ2WjEPGzZl4aB4FjYYXkWB4WJufGdqfyCA4nBoXeM7zpYkdpd+
  cgRExWd8N+ZJ/mcry5dMgScjMVaA0bd5EKAvo4WCoUM/RbiC6BGB9COEdbJ6pXcm69d2jSVo
  0WFlkgBcizcsuHdhJCdG/bYBFkdQ3XVDIPgnywKETdh78MMIQtdfN0hkOYhWFxBNgXdJA2dW
  o8Q4tpNVqIRYlvFtCaWAJ8WE4eGEdaI4AEIRrQSHU5iBFSB7etVFzaFtYndlTEh3fEhRxKb+
  GnqkhAVWaaezZ6MnehYoh06HinP0CAKoa32ndcCkB3hoTt6EK9Eng/52iUV4Pk4lVFqld7YW
  hpYYJPKngY/4fvAXT1WTS6b2iisVEeeHCNgTXSenTzc1a2vhHWPoV+MWC6WoPkgHh2oVMO/V
  Vioki9yzXy6Ti2+XX+WGajCEd+GobQslTfjkUJ2RdoMIfk2TjsKobYmwK4hTjL5mjZ5kZlnI
  CiQ4fpaYj9t4P7lUCVWUf/EHiFIRP75AkZkHIr8Xd6lwDdRYQV8YUwTJEAu5QfXXTmbIc+sn
  kch1eupVg2oGjpZHjR6Xa8goamaWdymgkTVUW1NxTryGjuHkhBn+hQgnmZAY2X80iHj+cDf6
  MgvB8meQiGkwOAMiA24hyVy6p4xF6QFcJJBCeR+M4Hm42G7Ud4KnqH4YIksjJpWGeC/k11M5
  eXLth0VbeWqtyFLyU5UPghBeGZSN6AGtGGl9lWCa85fBEoOquIWYghBmhAjzKF1AlQr4VZYx
  BU1qUJhGhxHeqIvXl5dflkxMBUpSeZfQdZbCgZq2x0Mw6X7yZxsBiTtZ+DwWGZhjhoi8M4dP
  gxsbR0qcWV/DFUgpwTH98gI+qX/eFijAN5CdOIdkOGFxCWoymRFMdA2ypHOvGS7R2W4QMS8O
  Ewmdd0iiWVojiYF1KUK4+R4s6H/VKWz+LpmIvOZFuvkgLMdUyMmbjel/TJmDYbee5cl5HIif
  eoea3GmEu5lDrBMMkQkwkymG50l60heJoImSfilI/nc8NKONyBSD9dloaelLBNkAWemV92iO
  +DGbF1aLCGo052hdK8qXZxChJbiWo9A5T0kK/aiM3bmDFOFRTPShtwedQtoaeKeOQSKNa1Zm
  Ynlt/2KAIQedMhknOEo4obiY9Il8m6KU7vhq7xOiPjehS0ajVyam+DadDJWe6omb6xMC7JGg
  J5l7ZhptEEGCMPSP19igHdOiBeqefFoawSk/rXmY0KmmKZWgHZqljcWXOUGFQImbf+oe6/mV
  yAemGxSpyjH+qZTqmnEqpx9afxo3osgBoHOKbwzapdz2MUr2nGVElI+lquuIokS6oJDnofj3
  kDIkXzlYpDMZlG4YiyoaRf/5jL+aooygYZjqYmi6XbxqRQa6fbx6p/KTqyjRlXTWrGQnoYaK
  HMvackX6kMrnmrWqqIzDqE0kqskyq4c2MoTHrtuqHO6qSu5apLgWmPdmfZPao0mXHgKlmo8a
  mMmqaJoaoC/3rJ9VqlJZqFP6aM1JrsvJr5ZBsJU6h7wKrpX3X4VapNmKnpw6sDWmlOPgWXfD
  eyR6rOQGqQg7hGHEpVM0jho2qOmWO5GZaolUqO9qRQ17LxAFsiqbNSOrGDxrjnr+GjAByyvB
  mlXQmKwv2qnqk7QXCg/dum7Qs510ZnbYxw7QyHUl1qqzQyY2S4fpxqIUtV98oV8WuwzM+K5f
  +7RDmrPDtThX24bXqUzR9y9dO2GshbKremqWGqZqa61+y6ST6qgvybQ4WTfWmUvChBZFF0yA
  kl/UZaOLtLZ4aqoxsVYww6rTWqxG95JfK2TjqrMMS7nyY555C7BMZ4MGy3Pbyl6aujweFLUt
  txiAe6nSIoiv2qUkckAJaVcSS1ma6kvXmKKGm1Bj+aEx5KrSNJaSm2drS7v+Z6t/QHhn+6Au
  uhND+wrQ5lUSxLLYe6zeOnzda5loC77CN3SyG1RW0rf+trtpT/ZXyHCTMWmewSu8FgmjXukp
  H5qi63lObJix65t80iugnBiog+m8pFu5u2ZOkPejkWpJzNuCsqjAt5e+1dGskjqpb4o+Qkkf
  CYVIfUisvIA4E1a/Y+aUEMGmU+SmDPG6J4pUX4vBcAV54MhONAgIBsx9CdxIpvu8B5jAPkQi
  2eug/grDijPEAtOj4KoTiIJukInEmPO0V+vByyu2RbxN2Fu8xHpbGPW1olq9WFuLi7PEH8Fr
  eLvDB5jDxOimanyxVQidU0u5qFOnjoCkhEiidEEfJdtjO0zBo2CufAZwzkS009qecxjHazvH
  MiSZY3ILeVwhw1A8NtjHt9qCxr0hyApbsVAMMOjKspv8kVnwyXr6GlbcmdTqbaV8daQjH6IM
  yovRygGDrnwAy5hDyoB8mbIcuWN7ynPQy74cBZ7syr88zMRcBcH8CLJczMq8zMzczM78zNAc
  zdI8zdRczdZ8zdiczdq8zdzczd78zeAczuI8zuRczuZ8zuiczkSQAAA7}

set pieceImageData(Merida,70) {
  R0lGODlhSANGAMIAAH9/fz8/PwAAAL+/v////////////////yH5BAEKAAcALAAAAABIA0YA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWq/YrHbL7Xq/4LB4TIYMyugXYA1Io8/uOIrdlosBggBcu7azBgKBgntGgX4ceHpc
  fYcpgIKBhEWGSQABgQF1NniYkgyAnkaXgpo0kKelB5ynAqmrrCmrqQuCAaMCELI8lEiWmLMy
  q4pmAqFFt603rKQNr5Cuy7wmug/Rucw70kSPrMYuzsDc3kHImDbRyQzO2Avrgijlxc2RBASP
  qdy12bhH+ZDjLMA9EKckXoBz0aAtU7gMXrd5Cxv4y7OvHyk2g2pwEkb+yB/AHo8G1AtZ4xfG
  dO3y0LmkcKU2ERvXmFMXqF49lg1GnQQ242WQif80shqmwCMSoPJMJYTIiiGsacsO0ozIIB5K
  Zfwm5RFpc8AohPVW7Rnl1WeQRDYJ4FQKIG0gaG1t4lGYltOJt121cUq7llYxuWZdoMsKZHDg
  FtHGPjXi7moMdE6fMV1MAt1kyQwsl9QMhBPXtPc2xyUY0t7bI3PrOoaB16ZJNixVZ3IpQDVh
  DK9Sta53OuVf01f3Ar6de8Xgj0IN80Sc+ADQ5T4aQzcebTbs6nR8WdtQPPNS302nNhS/usTx
  ILvTEkgkmm+kSwGGvwWAPLnb8oJb3U9oO5r+7Q0GeRcXcJIE+Ak9YUljIAtv1YeQYVL1hE5Z
  kNEnhHSbGabhhodFsOABE+Y0lEScHfChCg0Skd5/NKDl3iD7nUIUD4/MRlZ7rkHW3zIsYlBj
  gkmZSA+FDlDYiQOdABfhj2pFwlwhGybHIToz7oCUgyZMqWWJFly5gFW3OYdMlSBGtYCXDIZZ
  mH7qAVnDI6Aho95G8Uw3w0RYZslmm3z26aebGQh3E0pAGbMRMQ8pICgBvVEHpYZvUjmYdpjR
  mKhSW2b6TqDfObeMoY2SeCmGaUI54JwdUhdXPKjWFtaYeQZUKVt/1mrreqk6sOhcnyDjzaGI
  HpnSqX2hqOYPHD7+CEl8jKIjF6w9dJehpplqQGpRvkYA7EDLEnKtsYzVxGeoMXDD21Yu5uXq
  jhHigCdCp94q73DWinsuMJkcqyi5DbTSrm/32RkCI0Q0Nmu53eToX6sz5fDutNRqWS9VOeEh
  wbbVWPLAt7EIrAMmbX5FQz4iDfAZfW3aq94tZMogTo0IWZfdzDTXnJ3IGZh0o678ftkwz+XV
  uMbOKBi5lYqGRYrKSAb/tnItsZoA1L+P7Tnv1XZp0BzQy41CtW/hbF00tEMwOWiQCOcBn7x5
  +EmyDeKMJKwMEWu6wURfq/Jzv5s+oBMEdaqA5k/n3RkVpYNl0ubbSkcVdQggXy05zj7+jrjx
  3t4F9rff4TkiNhAU3tdyC6OYxqzkfd7yOAhxq4v2C3VXezfL2mZUTd+i4qeK16tXILTJYglh
  dLc0gJlpvE06abhhvXdQC+ry3sKBkXnrrTySuB/Ys6K8C/4L8LZHt2yMus+BIPS1qu4yjOoR
  VK7q6PfJzei44UdN7uXdP4/HIKwNGuY6aNrBZBW7pb3odS1AyqVeYCSbOfCBUAMBry7Xs3wA
  Q3804d8H/Keu6r0JVnFaFgw4CKT4PS1XJPDanzYXA5aZ8IAerB8C9Ua5lOQLGJgI2vYuMIDs
  hKI07bveJ3zIQOOZ6QXl0ANwNHW6s+nLIVPS4AiMWMAYXsD+MxT0ibQyp8Py+Y6IRTqfun4I
  xj9A7U9Ggt2eSPZCm3gNiZFbIQDNGMf4qQ8mAJyLEPclk1ncozwsnNjhptKnUCEOFXSUHXNI
  4TpqfSZ5VpxipiIptWc88JJBwSNFgLZH6/HLgpzz4sWoVArh3ONe6jCeFAEkRjTOUU+NfGQb
  2eG5FfnpjjCw4CVtlkkRuC+D7OGbTMLEkglqb4YUEGAEFxWw3XFolRigIpXycz4FaumRtCxa
  3Zp3tzNCz0jcPKYnXkE1yigAGX4Mnww1pAjh3AKVh4RMLFpZq9CkSWEqa+MSwxmB1rVPlnLr
  JOkK+EoPQIJn9HmJuBRaDGP6BYX+IkrSnJxRunHlARUApZ75IgZNrdXRmlNq4i9PANJhhkgG
  74SePvT0Emdg76B8y2aZIFoVb9aFouLayLnKsYY2UYqSHm1i9Gg6vZyeQp8nBOoG2JeydQVx
  kyidzy6zU1CDZjM1xYJTs7xVE4d68omAk6jbeOpTGQG0fYEcQQETybDYEauqH4CQE48YVade
  Da4cOJhAzrTAvYIHrKFUIhrJWhqKnjWEotQAEK/2CALu9BKH/Z9da6VOEnjGT/mUbEc/wMx5
  7TCuF0VbaoK5L1cVKzYOrRFRzWYr6jVVrLYimi/XugJOMKstw6vOuZy1H35ykj+4mtQMYjJV
  OnzWqur+pCh5QsXTY662jl1BnmvVwqxu+fSssh1YZtmWWKuuymm32q6fVoEC6KpHvGcbLnop
  2916+aJdqdnqsBKkCa06VCftVYAgUCcWyVqNu4DllFup0zaQ8ih5wAVSgJeqHAVz6RsF3OwE
  yLncIK2jvp075wAnsN9nDeWR/e3p/BBrXYWVoLPclXAy11srFP9JphKkZ8B6ujiB1pbFmM1v
  /RJqYTb1JTbBtSF9+drQ/F6Wv3UkyzePK+C6qXjCbt0tbxX25N+iYg1c6aFJFwjhAZ/AGXu4
  sM8WI+aiIPKK5zOiYIPbRBWORM3qqjLAVKrjyv0XazhmGFQm26zENFXOMkT+H5MR0TacxTdr
  ZRoNL1pjTJB5lcN5Th2CWCu57HE2wrV0spQTHNwFr9OAfeJYC5SpIUBfJivueOgpVM2ODFeg
  wwg+jhsnHUdptm23lo30n1NAQtQB+a42dt6d+3xgVClVk4Kuc5dqYs/4yrdM9zHzuiZ42UFL
  G3nA/i6f54VFTTqZ1wUs2YbOG+y86na81QkGpknaOXfAYTvPKe2GEaVoDn2mNRWNtWGG49tE
  I9nTE4tsa2WcPrxiwJbkbjFRtbttW1mbwUA6g7PZs6jTcJBXp2yWDLE9L60SXF4YAwFtv1xF
  B+to1lAVwYbiQmp1e/kuD3HHQTBklTYAZeMmZ17+QO8V0GcCydS9vhV8tDnseQX9VuStTNFz
  pHCAI5vOpuYbV8zh7Mamq0ltOB+v8k1TnZpwPreOX8g3WLdj85Cg4t5QE5Ou1lLnfDwwILVy
  UmCmxnxVMkuxCs47zaEcDbPnPg+uqSldcGSKPOzQUzKdlR0BhN9H4WaPccPZG3UBBbSHVnvL
  r3GVL6fOJeOIroDXxa62pYP84b7bZnkjPNd978fUK3+7jFz+7dV3Qzpg6nxTkLJ3uTOS80C2
  9TN+fmLz8olCldcweCcHa18b3AIuZvqLGe+B6FMe3AOCj7M53yAs29Zqc7l4yimQ8a83H33d
  HoHvz2wC4Zfa/VZ7o/r+2bnpTqlh3Sewiu5llLiE6N/OLJcp+AR49hZx7cdUx7dSlWR6rjR8
  0MN2DCdwA0d9HWB905d8+3I6FnQPFvQP/8AGfHcyOuZ4V+NmJlRDIgB/z5cBBkZQ56ZgucZO
  toaBo/Rye+aCx5Mze3I8Z3NbAggoN+g2lZVCiMdty0KCDod6FGB8RrdwDJdsNLgoLHOE1UGF
  KvE8fpcB+UZ6DBhe4zd/U9JvEVVFmkJuFPhSBKVeNlh8OJiD0WQvLTg+gld/GnJvXwiGHDd2
  I7BYp3dGSMheZwgREiiEgcgdukZuUShGQ5EYSMEqzbR3XMhxRliI17BymVZ2IqEp2KSEUEb+
  UDRYg7VHcm24JVI0enHYYV63JSJIiZMxfZxYAf6jVYSYIp7ShejmhBKAgPICM2oIhcaiEkPz
  D/aATl1BWHKIeeUGAbooORY4gU6HZqSYfzKyfhZSi6TYH+ZRRS2CfzA3imHoUU5ziq6SimHo
  OgvYIGjEEdmIW+d3Qo5xZA9oaR8wYk2YjPM0eRf4i47DWE/1YKn3cRNoQiM1W9RCP+bGPjN4
  arEnHyXwHLzEZfe3htnojcmiWEx1ij+3RHVYjPbYZJLIJ8HTdiJhfd0TUQIpj/1TC2pGO/YB
  dcZiC8vykXjGf8+ofMsHbPEzPzcYMf3mgAZGJirYfNwgjVgoaQr+qIah2I1qd0nwxwHwQ4AX
  QQcRZw8z42c9F3kSwDuoQyFYeTs8Z5RhY4uumHzi2JEq0IyFxIqhhFT1tFMd8JTtcx2kyAZn
  pZMlEJTppnIdZmBhWYavx25IGJLJUVxUpZYOUHM0lpMnMXtO+XvrkQmDiJZDs4p36AEmmHgr
  mIt/0UOHlV1jNoiAiIH3AIJycRJiyEqx05X/KJNMYzOgmX6oqR89hJcrhzK3UHm0mZeHZ1SF
  c5gRY4aimVM0cy6nqQFbCGyqSX4XwZYgKZixKVTchZmvWIPQ6XyG94bYJliwqT3VWWnTSQGb
  F2qVCUcElZydeDLBGDHWEWesU2ozk1H+NFOHO+mC4cQyr4CMQyht1AKcUMQh5kl2RViP/1k7
  N8mcWTiPRXlTcklKdIBd+ekBhIc+njkxZcWOyORP8QNjkFOgT2UDXkOY8TmeG7p/3qgdBHkR
  oNlxW2aQRUWR3CQdIRgs+/mXSlmRxcOEfTigyhigVAmiWMYnKNiiUjWK1hFB2didyfaf0EUP
  Q8ctAOlZKHlpSPo0n5iVT3orvCiK00iaIGdcrpaS+KhPmXkBFDmmtyMjTpQ3ZfkpMaKlUXSj
  RHqPWpabh2My3xlKZWo5bJiiu3in0qY47wUcwcKngFiTS9gKC1oLXjoDitdGWaonkGmgVBmk
  oMWaJ2moK5b+p082ERZqFmu6EG0KqSjjNiZaV96oo2iYp1EqbKoKU5AqqZSqhQdWLETGoYuH
  qf3kSC10pYxlpqJ3iM5XeX/4QqGndKoaAgaxYZ8Kqvi0gMBWnFYKjD5aM/I3nzHpo/I0kYeD
  rf4oeZIKHFU2EYIaRrwKpbgYrYpKM0lUNccqiqEGomflp7cjiYR5WMXqbZoaAtbkpJ4YqkRo
  qYCBqq/Goy8Uq5dmEkilZRqKXMCYsFsmjVN6giLKgmeQLS9FqF54ro1XrkPJGq1KdOsRTwXZ
  U2YZaArriSRrmDaZpzpqMkKSn214Kq4qknOaEFjWWB4LsDNJdBj7rHvWs1CafOX+upUaqy22
  EAqNypYzy3BR+Ug1O2+rp7OTSHe1WTM2GoMhlZ5vylIfOw1Hi6f92qxYqylVyjcyM63qWrSa
  KbU7u2dsO4mVh5YZqrLRCrRJqLa6EpSZUDVvayv32nYmYbetiQyA+Rp2q2WIE7eturBEiYMy
  u6qIoLdlG1N5SrUDgrjHw6V/255rtJhRpLnyCjTfeqCjNrTeibdmEIyd915YRisGurkb2rdI
  h7qiIrhJeJrrl5Q3NoqPS7tForp0AozQeqjtCnMkWp4G2wFCc6oTemmye3qTq18EK6aQy7nI
  aKfXmSbPe4s8O7okwYaji0psuLjRO1Mx66886xnfiyn+lVu4EEKt9mat2JG2lriO4ct5XXa/
  Syt671mJI4kXeugcNGO/r+u7hAQahTkpDYqIeoJtaAuSBmy+rSqwBMq76At9/UtB/6sfAYy4
  dHCkaOtAyVuB17q9iSqsJWxCiYqrO7q9fhvBLay/1ftXFWJSzOJmt+m52SpIi0sxZKcHIque
  9EFabdcWWpuaPQW7rNrD1GTBYvur7mlS2nbDUUlqHeUOLgzBX/qWpuudFLyMShvBw6rCMCy6
  Mpwr8wOCJ7uPGpllqoQyCxorucu7evm+mAQpY+ufR/xM2rq4pUuRvRtgy6qqcsyszInFdXzG
  43vGUReeBTy5xymp+7uWLbb+wDkCkyKVHUlodnLLnKGLN4n5gFtWslB8ZaPKWDocujXVw5Bc
  ppuYK3S6LLF8lNBnyGyJyJxbndN6Vtt5sLrso7ysyvMVvrQatfdLbP9ot2RhW+izvtB4zBZV
  ZbJgu+OVVqD1C5L6sHmcr2lCkWYoyPQ5x59Tyz1sfyTMwUGcuUYszDNlxOk8l+tswJ3sktp7
  zJN8OfGBEXR5i5v0oaTaRykLTfOMVKFLCtRMqjNMpsC6lSPcZK06uXn6zeRcditLLQJdzuZ8
  kBhtrBtNAqMZwhCUvXp5tiBttQktwTWcuOQhM1e7cSXNrVWWRKH8TXvcslfGlmvMzoOsiotU
  ol7xClE2KM73nKkYzX6WWdQ2XdSGWc5NzNTLhqgJXB2ekFum7KWPI9QuKGf4ua13DBliSNVQ
  rckTdTNSzbXFS2AdvZpVmThTTUUhetVKbdQQitVR9NWz7J+4S75NzcRXlExlNBAzk0znHNfs
  rCjvPJfDa9hlSjDyS8c97dTQ59cNKgEezNi5MNiEHbeHrSWte5Z0rcCNENqizb9rHdZ/HQyp
  zE50Odqs3dq5EKKmPdkakdp2XI2ufdu4ndu6vdu83du+/dvAHdzCPdzEXdzGfdzIndzKvdzM
  3dzO/dzQHd3SPd3UXd3Wfd3Ynd3avd3czdoJAAA7
}

set pieceImageData(Merida,75) {
  R0lGODlhhANLAPUAAAAAAAwMDBMTExoaGiMjIywsLDQ0NDs7O0REREtLS1NTU15eXmRkZGxs
  bHR0dHp6eoKCgouLi5OTk5qamqSkpKysrLOzs7y8vMTExMvLy9PT09zc3OXl5ezs7PPz8/7+
  /szMzAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
  AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAACAALAAA
  AACEA0sAAAb+QJBwSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweEwum8/otHrNbrvf8Lh8
  Tq/b7/i8fs/v+/+AgYKDhIWGh4iJiouMjY6PkJGSk5SVlmwAmQCXe5qZnKChXAABmqJ2npun
  q6xOqaqtbq+fhJ6xt1qzsLhppZ6lhba8XqZKxYaaAwS+w2qZAgQEA8eBws1b1EfZgpm+pNvX
  Ybq7gJoB4djgQ96HmQcbHg7q6K7WRpkDFh4eGwe02vbkBKRHZSARg3/GkSP4JFWSVN7OIXFY
  ByHDhhSLZByUyYCGDx88MPh3sck4gAlAgpRA8uCsOK/YlYzyCuC3YL9qzmQSs6X+EF3AbHbz
  uSbixp08dQ7pRmAen0wpVU4gilQjAAEJsgogClXlB5YLQWhCoCBBAapohuqqatKo1aMJ1Spl
  e8+cwZx3vVk8oxAtnpNzvi3YSjJTgw0I/HYCEBVkBMVk+pKz+/KtXKdTNE2QyswqgQsgOSQO
  m2mBB5AZmoZNi3fvnW9z4axdCrFWT8xjIMaWXFeuTGx67faubDn4G95+kMPMhGGCgc74QD+G
  /JcUBJAXVLsJDjubctoKuzzjoHJDZ88VMIwmDWCzygTUc98u1zfwZVWtV/PhPrSN5Mn8bUNZ
  bFgMONBl57mkm358GXVTcpoQAAEED0AAH25mdINBSCP+fVIaSB5oFx8d3TTwgGoMRjZUKcYp
  yGJ3C1EWgF7iXUUeSBokCF53E7Xn1WiBschNd75EJEdw7Jw0okCAYTIgjQoaGOOCD6aTn1Bw
  iUWlfUsyGUAEG3jVQXYYqggAaB900CE+GYC4Ho9xZcmaABdgYKedFRBmlQAW3GknkFY9kIGf
  GFxI012ZMPBRBwuoQyBtBWzIQQR6DpfiKGWS+N+lZQCFV4CcMnnZdt9pCWN/ls2GETj19Wak
  NgFiecZvEAbgnlcgYaCdbMyp1AGKBHSgEgag9pGMAa7JN4CwXm1QKW0C3KiSAj5lcqtKjYb6
  E3HgBQABtX7JaWoBEqCIZZf+VSRbB6iPOnPqpu1WpOoaMlKW6oB1FbsqlC4K95BFVw6noziZ
  MmnAabiqJI+2YmiigEoQXPUAAAcPC8AAFVQgzQAtyhthBf0Qhm4uFzNbHqjR4nqhLteClO2q
  A8T8bJSY/WIMt9sGELO5ZjK8nIO+LfeKMg9MI1cEFySgjG6vtTZjwZjC65vURI7YJM1OiXv1
  tjiDYQ59AFiQsFccGM0rAAR4gMHEAlCwwQIbWkwASB3UPcECKPrcMwAPMOtBnuoSUzKu5o2T
  slcrz9LyB+Bi5FHdFph9b4rici2gdXWrCXVm3dAHNNBwBKeAosICnsqGHnTAQQMKdDwH0EQ6
  SaX+lLPT/u6+BHZdBL9YBuXSi2r1DDaaY6e5q3+ZKFCBB/C1XTyjEYytAaWBy9cNAsvjikED
  Iut9BT4mg9hAAAKUT375AjiAsEoVDID++wLYmvDL9VyHLVU2/0tA4/86OjhqA4ta5TQlteN4
  AgFiC0kHHsAAow0lAQzAwD5AYoFsDTBDVPOeFdjFHXYhaBy+S0pPJkIrSzGId1i7oBXmpQdN
  EG9svxqZlRYgrKQ9QFowXF/CKHAAFKbBEwTQR/H48SbWLCth/ODABpSoxCXqsDxNjOIG4DE/
  qwEgevdbDZx6hICJnRAh4LOYDOuhO1Q86YyysEsCxmQBAyxDITGDwAX+yNOBBuRNgwK8zal+
  eJ//eHBTqxqVrPqXNVTli4U1KqPHEvi847kLAAoIX/EmObZ4qFBw3XBA3Ci5gQjkDYNHpKQo
  Rzk2+pkEAPZzGf72kokuhgs2PQrlB4g1RhEuIAIReEAJm1ZA2aUCAQGCHQAOIIHTaABZ1RMc
  khTiQ4Jl8Jlbqgd/elQlgOySZtScmjMF8ABPJhN5CHiiV7yIxwKRggKkTKdKLPk1PlpHnJO8
  gAPLiZEB1CkDg8JnBjSAgX32E5//9Gc+9RnQfxZ0n4aqB8b2KYEAmmoepWnoKsGYKIAC6gua
  oKFKJEpPDNrum3mcxR9/YYAKfIADE5gnvaD+eckCPY0ABoipTGdKU5oeoKY4pWn1aJdNtMSk
  p1zZWtQW8JEPbICjxiKFSaWHTORdEVcVgECb1DnJBrQ0C5pIwOJGeQEGODSRTyvSB/UCPODB
  RpiBfJpPg2fNjzTVJl81h1o7+i+NegWpi+GPW2QnOggg84xU4hNIKtBM6zltamoJoTN9RMrT
  wJOS63NW58hYmx6plYT2gpWc+kgw03xgAhAgD14XMwAK4PADx9xcOg6npgOUD51UHaJVVVuQ
  5J2WqtOp5SkjFI3e+va3wA2ucAlQgO5BAZG0WQBhgxqAj/hDi3wDi3726LVMwJZsM8uDHq+a
  yG8IYAIY4AAHLNj+F+5gDCQTMK5Hv/E5ePUsAk3UwBTnS9/61le+9s3vBUR2XHwBlT1yyaZD
  F+RMdE5HAR7gQHZ5mTyvpJa2WGWMsBiFjzDFlpIL0y0ZI3lhXEXgq1ay1T5GTGISh6TEHTDx
  iE+8Yl1tDi//eoBk2ROAML21LhPIgP+4eygCbNJX/Guh7a5pBr04z3iH7aMnBLBU09GVspsS
  JoT3NaMqW/nKVY4flrdsZS1vjjuEZBUikfvQKSeFABkQTXQ8EGQzLlkAC2gWAuLHY3PCdk3B
  6vAkNSAyiTQMkpJMWHNGCQEQk+xiFHisnr3igQz8dYUUfcAHZgsQBpyGApczAAcwIGb+kP7H
  KnEeG6YL8+lOKblVrNFMebo3O2HmA70g5VzVXnGANUnGz4t9F0uZma7Num5HcEHuivR26pYk
  6gMWgE0DPjDqXQDS1FdRwGE48ETVRUABn2TNZjLgwK0uGiQU4K8zB3BbXHnAAAKYKiUtaKYA
  eOTbQ/SqmR+VCUnLs1oAkACOdKSJ63A6qFDr5bYYULxmAxvVfwYsYjV8qI5Iy9Yf9dQAwsSB
  vz75X6AqkgT/atZfV3fXUZ73FomwzEOyFdjX5K7AtUTwZGti2Qa3XKkwmryljjIDujTzoTbj
  gAEoYAASsDC8EwYuhls2lZMMEQB+XLx7X/xmHWGA1KfegKn+S73qVbc61q3OgAY0IHHmDCtA
  JM04fOsbtfwe3L+nBEvKki8ADpiA3CUg9+c45ENjq4AuygeBucs9AhwrLMlGOh++aAK+FeCA
  A/Km102R/cFPN0IHDVTjD1g8yZHHbCbiLvfOe/7zoA+9BFT6vSwZaHLeGXNLIQJ30BfAOMd2
  uWE+oHeRDqDvnvckq51pV3WOVkV0mnQBOiB0UnLgAkUdJQbUK7hlk3IZTC9e7TP/X5DH61C3
  vxzZM+wSJofmLJMBAAKExWnFsjyhu/3SFMWpRAeA3xMEz/vQFJCB4qtEiWQydK9FKlfbGf4b
  EsBnd7RdiKUX7pEB4kZzUeZcuvb+UxgUANH3bUr3ZGCGes7mNLujTSR3fW8RAAhERbiiRCdC
  GXEmewAAc3IxAAkQXmOzARqwAKQnOAjWYb/3Zw/zARfgbZTEEjc4SiIRa6dEdqLkAWcRgYKm
  c2TUABEwIUu4hEwIAU0IhVIYhVQIAQtWW3B3Y7RBdh0gOdvSg19BagCwVBeQetzkSFCnVaS0
  AUXjCaG2Q6nAAC80SdtDZN8TEYS3V3uDAPI2bAvXgAdEHh7Abh8HIx3EgCCUWWayAA/QiI74
  iJAYiZLoAFcoa213EB30O9wSHilEWViEGo7oAMTDhvNUgp6AguYgh+URijaHgzGIKTooSoVD
  fb3hAN/+xhJvOEoWwHzpQAD25yv7pAH81BQUIIz89Isg0iEeZYR6NoHpoAARY2x8Qzda2BUb
  RWrBN0uTxTUJMB1k1HukpAE5dxVzmCZ/tYKKNjYfZnQW+EFCRXOE50e0c4MWYIeQpkcCYAAF
  sI8GECYMQFwxNYAcaCXW944F8WuFJ3MnpyWgsyP6xzVfkiYXcFN5MwA3NQHkIY6aIFjZ4osd
  cI4ZkDrq4UZLFlNBN0s8gyk+pmcdgH5/RgDl5nsAkIsgMklFl3kdMUmPYVbd0RgJ40XLOHSM
  hoaQlgDsdIFCGHOacHa5cjzHlivU4HBTESrJE2ikRE7XdX+EgQAxKUrrGHn++cFB/mVYt6Zw
  s9Yd2XiTmBQgtogw67M+JkhmXiBMy5QfBmmJl2gqFVg7sABCnXhKEckBiaML7/YBGWZSs5U2
  CtZKgtiHfbSSZMKOX/htU2kmn6hnPEhVyweE/2IAk5RS0aAMpaAMoUmT4ySZUMeMHeaMJBNO
  3EZqQngB0GEjuMI/mmCL2HEM3aBv0QiYjBRbHuBFByBJC8OVLEmUpXdrfxQZUkZ4AjBPChF/
  m4maZXYqDCBe8xUaUTSV7bVe1jdNh7YRfsmQgHGW/3B6PFEKw1d2/0EADbAPXnQwGjANPMeY
  X3F54+CLH7AmNQKGF1aZ8kEAybeajXKZPqiWmOL+mUOUOaqzDBeQOimmaEAJbap5YawZYT4y
  odsihB9AP+OhMrSQCXmGHUKyLQezAV54M6ZJVcGpVM1yFsa5aFP5dPCynIYlVu74LgwwY/dR
  JKDBZkg4OWSFPjrzEQlwPuWjcLiWcK02ZDwVnrSiR0KAozrhKX4mlwqyGRgAne01k8IyMQFg
  UgygaRsAo6IVj0/FZ0E6mTJKnW+Bm4uWI7KUTiZYIxVDSkWYThpqPeUogcjZa/qmAU45jQoT
  omhzWh5KMeuzdjmzGTlCHbP3bWUjfl4Rja3YYctFow4iZWeZG2iqEA2wmPznCWT4kIfSeBDh
  VlTjnXlYXtuIVRVIni/+4Sk5KgzRtATuQB4Q9x/xZ1XhhAHR8xj9+FlUCjulkGfc94yU6aYu
  wWHfpndrFFsutqZaoqA+2BQVWqi0mC/ZyqJ/WltnN1v4QaihYTaZwJQqoaak6hVlaKiahnaQ
  UWEWClX0CRIoyjfp2FhtZk41yom5RiV/FAASsCZUmgp3SogEqWsRYWN65CAZMlcAO6oNGFKm
  QFb2wpeoaqiKBB4EJ5sZJKCu6CP8sBXRA6QZJGnThyn+GVsAqoAQOHTI8pvqNFvmB2l3OoR5
  SkoLE5RCCSLfyjlY9KjjyqEfQE7WlTCi2krhw6iqBq+XIqL5mk6NMn5FizZW2VheoYwh1rD+
  Eluz4XlWruqHBeB0idgN7pGsCftRzfUBz+Ve0GaXfFk7AkQjISexgPhQetg/plhWZkkxYTK1
  X2q1+8m13aAXqEijkLSs29oNSKdnCyOz6fSVS1p6EXmtS0dKCMisvdGni3ahETZxK8FqLeOx
  mpCVodE9RPua8Mcs+9ojIwpvamoBHHAWsUhVWvu1pJCxsVKIzLQp7lmAsxBqG5CSGAq3G4mI
  GceZDdekhIulUlC25umHgLgWurZbLcdSgRo/qrptrAYvh6uS6hZbHdC6h2agHUZOzgecxBth
  cCpKDxCjotSu25qBl9uzIfGzz0sKQnchpXCzodG2hcloFpSNp5n+DNKihfpztSFBVb46McMq
  lLcLq707wbSIeZNXbPIIAAVQVOTUBRBLeZmANALZeKxKwbQ6cmGXQa26cBe8W3vbvHqRZ1aF
  NAMwrBDXpbm7ecxGrVpSu5M0i9Wln4u2AdE4mgqcdO+HuAMwoEnXlYzmktbTrTU5hPh7KAKQ
  fPulCdZawAAQf3CYkz/pCUiHwLHExIuWbAZgVekLb8yDmj3atQNZW/06qhcclgAAW4w6t3ar
  M2HLwzcTTHCMcC5lF6iavBgMvZCaPPtwkxbsCShLCtOwbFn8TKArXTViaTRoqvlrvupExKw2
  p+nkAeKKuOQqlJP8f5zbjFXcX2sLIkX+t8UqsYt491kSwCzT174qQU5o5hXPVQ9eDLvgd2RD
  l7IUS4AxMblaUJfRe2qArAtLxQG9TLHNnBOo+n9w60FolbZxa7xKGnGkQEaIyc26QHAP5jzT
  Z7F/qJjT4LX7954sW4nm9K4W6n62EEYXtnaSKaJOHFvk60xSHMqrvCprXI9gHJw3BBLIpD6T
  UgoKwAEdQNBr2wF3Y2Ho6xUTYKoN1rOzVQBRm04z2l3KbLxeE9JSU7DMPBbMEo1gabe7Zs12
  nFhqm89Jxqk82bfPJkLjlwFJHLGpsJKJmSb+8KlXsRlGi0kLALnFkwGF5sem0rjp1FUUYc/S
  Kni1lQD7rKf+moyhqbyaAY0RB4ArS+2ZoowPdNcUAiABc3YVZwENEoBMEVABdocAYeJFMwgx
  VhQARLtoidnRupiiaZuxY1nMLCzSMf0Kh9PBebTCSYYG6AxN4KnNGczSMOwv6WdS8uvNsIFO
  F93FRtW9OKzDG/BoCghn6jNJHUDPTK0lKaMBioK5DJCApvIAfG1uCGol0fptKs1HL2u/nhth
  X41sCyABkzoAC0AtBPDaxyyHSp2kEJEAiWEADAB+fXJLINGbGHHFQrmZnNxh68G7cUvV95gJ
  BeAc0HRwL+1dFpYa1HnIkT2/1nTC8Qje+cupEae7GdxflEph03sqm0GEl8uj8Vj+AG2SWxUM
  RArMUantDh2gAWdxAOn40DutIEidTiqNzMlp1SByARdgARuu4R7O4Rt+Ix6w1N+cav/cWF1d
  DwVgASlFCneEQB7wggjg3AiAPdR2fxmgADW+4w+6AT/nCe6zeRgQwaeE4UOXsz6YdOMoDiSt
  ZIUo4LOEw3MVQrWqEG/Ytt21ZDWeFc6dFTP+5QkgkO5tQvu9qXcpx4G8zWWucKcq2z/I068A
  k/69IbvatwbQJpHD1L8gIROgaB4wAW+Ck0e70Yqm30F1AFc9SUWNSQjgAA5AXgrR6A7An4+0
  1VOM4ut9iiMxI3xuvzgYAcDUEVaockrLxki+aLmtgI7+nekAuG9ui3oj1RjdneUGEAE+nCvX
  Vmd3aMz7PWtryd5iRdOYDTXeIGkdIAE7LRnRMucfMJhly00fkeesfjEMsE/ptOARkMRRY2AX
  M+FpQl7WtN0ebY/hTQo5VVO7t1L127O9jaE+ljr7VH+eLibBSB5v42n2+W3YKoHjKOg53M2n
  lkeSJpjMyxX0jYdazCzy67LXAlAO308YYDKC+qouDdg23U6C/YftneboYg6yjVrvR98ps+/O
  3oDfRUGvWLwr2WHjq7xvgU4egDTpGgEm0/JrlTbwhizsXHoJ8AAOUEw1OUELPEEW0Ijjo7lK
  4Asnbrn+ztHzzsC0lVWJHhr+TeiEW6GEUggBJ47YWT7sglzucFfygnQucJ4K7lE4Fo4FviAh
  UEiJuVtl36AATQh27mLxfwg7xFDHjd3HvA6EsEF2jtYXDEAhFSIsuUQeExCK0JkKRybt/q5+
  3/bQ61sFvnCpK4FKdIOwvXEts83LLt8jt96CWW0llq6+btoNef30xSPt92jkYjJFhYIAB4AA
  +KkLBCD7sg8yGlBuS45JDSjl9wGlr4AAZIvxAiPOr3B2QuvdLKXuxivUp+/1BYlG++duYqPT
  Lh4NEoABdSJB6pQB3d9GxLUVJ+/jDrSWoa98k19ba1yuEa35dfHAnWtxgrPyF9YBF2Xint7u
  g5f++toTekAwEQ6Hlc8RmTxaBAAnCBqVTqlQJyBR8Xw4QkUAfAUIGGUzo1FOowviccAgmWCO
  GMe1mtdXxYGr3y1QEBCAcO9wqs9N4qLJCXDvcfCPcvFIAxJRM1Gw0xNvM1RT8rO0VFTU9GpB
  rdV1bY1hwDQTVUpswOIjgwHC41dpSY6IeEvp14OCYeJjw0DMdrRQIrjauhqDADT61smhA8nj
  ruHDY2GbD4Bg47o9aesAmlvPSeG9gwO/Az9fHxxpwpN5865Y0MehH0KE+xTmS8jP4QZtAQYe
  uvLgw0EHCjhy1KbqUUeOCzQw/GBhlsCKfxIoiFdogAACEyxUqHDBHRL+DTZtJhDw0wkBjikB
  VKSnilDSSp2MgggEqECHc24iTQLph1q5BiqbRhE0ICZYsWHFEkLXVdoYsgLAsmXbFm7Yt2SJ
  oqXSx+2CmjYtGMupBANfCxDKUjWKS9cRD/k2LBCpgOigliIN7kPC4SVXtI+ylvv7F0Obok2v
  EJBgwcIdAMwgnOUEgNxnd1sLjd7spAFPCw3m0nV7QHeElHY3XZn7cyzdmL2VA9VM/FGDAm8D
  FXh8/TGCQG4HOKhL2o0AjhP0yb6mDwNH0fKgk3q6VKmprrXdUIDgx9BRRVfFNDAm4bnDnBig
  owYcWkghiDqQoKNnAiTuLicqQ7AfhhriZx/+BRn7CMJuxsigvGA62IunEktkpxqFtHONGzcG
  mKACCxhwC6RPfttrRRYFPMA881qzTcBADOBAA9GqAoABv3oMZgOz7HJCgMweAYPKKqkUgwAO
  O9TPyi69/LJLHeebEowHiJBAA/M6IEYCAq6sDUInFsBpyR41uO9B8Pajr0b38ANypUAOMAvQ
  1/jcMz43AkjzAwDzbHGMxN5BhlJglHDm0Q7FuKBSDzrotNJPKRV1sY8KfXKMNH/ZoIFYGpjK
  FAPSiIVTZHKM0xNC+1Rly0forNOdDrTUUwwELlhPvySBrQYT9vQcAIEDEDAAgg2stVaDa7Pd
  wAJpozX11F4fqfb+2nLN3dZcbc21wNQtnRrjgmuVXDYJbV/SFAAK6DVvA0fERQq+pd4LNxr3
  BOHS4FJ0DQBFRwmGVIAMNsDgAgwCMyDLjAvIOAKKK8agg1vdNRSAATI+GeWUVT7ZSXEDQHnX
  UkzOuBZUdY15V4rwTeAfYOdVIgIxbVHUsHSQ/JneZuEcE0lJ69TgAX9H7iaACPa95oJ28RWA
  g6v/WhFfZqwxxoMJHjgb7bQf+AfpI/rNlKA/Eu1zbqFRIdpZkq3i82b8GA0I7lRKc5C/AXN8
  uD2cFf9X8YR5xfWTMBCIgHIIIoDA8sstZyCMm4u2WWyvq5Eo8IL/0K+BnkVXunRpEMD+oOtl
  PdBANcSJJUmD3HXPIPcMeN+9d981+P133XOPIL/2AqBD9PMcxFffJDrI3QEDrDfA8ysKuD4B
  DDSIHQkG7Ib0UL5xnptxggX2c32DFz2ig3tvG7h8kKa24hEDtte/gP2t3x6A+wPg//p3vf4N
  UGuJM1/jOtEyBYpBAP2LQAYsBr5reAADvrOA9cBFCnwhoHnX+FH68nAFZYXwA5hIXpAAYADK
  UY557djACyNwjhWmjz6A+JMOa4OfzvWQFITa2TUwQEMjHjECoQuGo8QlNg5IoIaCUIADqFhF
  K/JGSDA8AtjwBTD2OY4pXXxQA2vUN0Z9IAGtG0X29oS3QNz+7woJKAfZPAMMS9FxC3ms4x4/
  ELTxmc4T3aHiA6pISCseMkc36xUhBBm9Y0ARiRE4YxLs8ADCqTEUj3Caedo2uufdL0Kw6SS9
  NhCZ+QWCAAhIwCoR0MoEqHKVl/ycu6BEgJllzGS3tOUuc4lLX96yl99pzwE4YK0GqNKVHVQY
  MmGJgQ1w4AGYzCQAGDGtR5jsARfQ5gVUhw0LaHMCWXJEKiPAReiwMVftkw8o9dbGmDEMCWm0
  neDUWT93SnNoTkDAM61VzGLy05/9BOg/ESLQaykkmo/oIlsasM1vxlA2G9imNtskzMQF4AEZ
  QEIHrLUAZn6iAK985QUQgoQNhPP+j/nkGgqtYU52vksd3fzMKJHgR3yWkIENBKUfJvCLDO3D
  U57KUFA/VVSgDtWoovqUBqR2UQHoygAMYMVUpVpVVlRVnhD8Sc1s5gcBLGABPZUpJzHICtFw
  lWkB+2Ku5nZDUK6VfolSyvuOIE9U7W0/6JznSgDxE78+dat+DcBfBzvYwP5VsFvN25hyYSkl
  dKACQqCAZCk72QlM8h0bkB/kCKBRDQgBAYNV3GAHIAdddEABKb0bABxAU9FVAK0j+0NnWNqo
  001NSLBa7EudoMTamrSpNvPrAypAAQoUF7PBKu5xJ0sAxe7VFoAwAAV+VY2fXheo7chABWgD
  XXrKtZ7+7aubd4mVzrXWzQHxVC09g3iVl+LPizndrYAGUIF9NAasCgBrVkthgPyyQgGVwdRN
  LXIF/8pSq4hVcCcGsAD+hu23weBAcOE4Bt+yFHDrBeQYFKABcYDrvRG68G8nTGDUKaQaHkDN
  ilncYhQ9Nh8ToPCYBJDcI1xgxQhITnJmsuKxemCzd81ZeMlI3kAFTK+xUq+Ji+NGSoTYQ/KN
  r5ERAcG2FHlXc5kvXz0R0lVKoCFhVsgFVvnKU+BqxLVtEpPpmeYQAsitBaZE3QghAEemcD1O
  9sRt7hxhLuAHVwOIHTAk0KpWpVYVBnCAodeGjCMgmsoFFsCLt1DoGfUBTGb+UUAD1uaZLbI5
  HQtT53hJHWnTsbFvauUTj+qq4Xya19VphfVoD/ZAPSfZnm+0NRgQIAE0HYOOfNzoBCTAuVq3
  x80ofJup8zmkdrj2Lw4rHQMJ4IAXX2YC2FPYJMCT7BAuO3ENoFgCslQAs8DhANY7wLrTre51
  R0YdGzNABTAwgdgGKgAIwcADsiRdA6S7AhswnvEmloB/G2CcQvEeGkEdSiLnmhazlK37Zg0S
  AyzZyGDEK523LfG4STlm5wQXrhfoRciVjFrfC4fFMLDBjGGs3KZhOfg8wK0DtOHeA3GCATTq
  53BMhber8fkRpP2worwnDIXQ4Z9gU61nS+ABA9j+oSR8+An2XkEBmHuA1rm+dcxp3etg/zoE
  zjb2rZ/NAYo8siAW8PUH0HZefrnA2SHgAEeE8TYBqJ4Y2g6BsebEbGm/wgD43fAooxPxGq9E
  0O+peFOwmuF+SEvFh6zqvG4ZYY9wgNu5Tvawk73zoOe86Ok+EV3rvAEZkCWq18fxJ9+VABCt
  wNbFx/TIAeAAmIuAXzqQVVwtYOhJABw7nXCAv6Ow6EcqxMYewHLnP9/58+JAAiBgwSNQEPrZ
  zwAF+mfKaYIc/PgmgLzpbbHjmwf7GCD3xu4u5AEo2mLQzonFKIAxzLco1WAcL1xjfWoy5r/k
  zALyfE/5wouHMO3h6Of+zJrMDfwj+CjA9pistzgA0vaG5BrP8EBAhyRA4BgA4aTMAIxFoj5C
  Z1AF+ILvCCggA6vCHoIv+XAKAAqAASTAseqEA1aEAKprSX7hAhqAAKVB0QxNCIeQCIvQCI9w
  CCFN5+SEUnwOGSQg5zZMAXqKpTggAgRvBWFq/7Bs1I4NtwBQzxyvhTBOziwPDC2QCzXu6qAk
  AnzNDd2Q2HwtDt+QDueQDuEQwUzsDxyhrYgsPsArAzFt/yLQvN4kC+UEBY9g+CpMAHQwwl7Q
  aE6QXjgAAi6JAEhC/q6hAlTLCTAiEa0BA6IQEQCBAkLEzxajmIbjlMZglSIAITJRwkbE4E7+
  bxXdaAvP0IMqzA/xaq0M4B9+MA9YTwGJJvGGET5C4QDB7536D74YaAvXSdZuD8v0SvLaY96M
  CxuLa7mOyyaYCxuZaxt5whuzkbuyMB0G4PzerA+OxGkuIBu9UaY8oAIkQOoEQQAcoAJszAKM
  SxxtQklqZ5og4BNBkRm18CeukQI2aV8wgB95wzmYDQb3UABygx+tLxj28bja5Km8cDOecWBw
  sXzeauN4URWUBRiNpuQC8OH68CNpscqcjOSoEdZKcMMuMCVn0uPWLocgziYLUj+UsUZEMgD0
  a6quyqqsiqqSUqrAyiiLsgyeIc66IQHywQEW4O7cAAGQkgEqEEj+3CCqknIBPEcBGCADPCCh
  vGvnyuCqwOool7KqjpIt2VIttZKq2lIJl3AQnJIBSkKo+hKp9uEB5JIBlMkny1AQEmAu47IM
  vsDqJi6uEhAXqxG39M8YdxKMsjKBftIZNaNwbm8NgXJx8BJR+A8NF1CM0nA0z20ySZLyZk0k
  QbNP5MyWEpAjYUpxTCYqIxI2Q1MnAylLdAmYMiYoX1PKiE8MUZM1IZIBk5M5Ie4lu7DxjkIm
  9+y7yohuFAWdyus6j7M2cSgM24fxdrMxdVE8qfM5oSHkSsg2qeJxPrM87Wd+1rM4C/PjgvL+
  Tg45K9M0HTPipsxzJLMMnQIlXVI3d6uGOw3Tcf5TP5GzI5+gNPWTIu7TO09BQqHMQi8Uyir0
  fhwIQzt0tWLTQ0NUREeURAWnROHrRFM0ypBCRVvURV+0wlKUPmGURmvURm8UR3NUR3eUR3vU
  R38USINUSIeUSIvUSI8USZNUSZeUSZvUSZ8USqNUSqeUSqvUSq8US7NUS7eUS6MhCAAAOw==
}
set pieceImageData(Merida,80) {
  R0lGODlhwANQAPUAAAAAAAoKChQUFBwcHCQkJCoqKjMzMz09PUVFRUtLS1JSUlpaWmNjY2tr
  a3Nzc3t7e4SEhIyMjJSUlJycnKWlpaysrLOzs7m5ucLCws7OztTU1Nzc3OPj4+vr6/T09P7+
  /szMzAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
  AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAACAALAAA
  AADAA1AAAAb+QJBwSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweEwum8/otHrNbrvf8Lh8
  Tq/b7/i8fs/v+/+AgYKDhIWGh4iJiouMjY6PkJGSk5SVlpeYmWEAnJqCnKCeoqNnoJ2kfKYA
  qKytWqqnrnWwsYehsrhjtKu5cqYBt4jBvcRbu8Vvu7zCtcjOU7DAy89pyorD1FzYRqaMoAkU
  DtvZVKpLnAUSExMK40Pmee7kVvJC3YrSnPnN81LwSb/+HRFop16/KPr4vUu46BuHDx4cSDtY
  jhYSTgIwfNjIoYBCEMfw6KOYJeG0IgQNKZtIEopJAMCQ7At40aJImC2vqIpZZOb+yUIYN2zc
  eOBjzoHWuAEQ8HAogo9JfSn7efQcLKW/EgX0WZXJVKQwQeWr+ZJqm6lm/9i8ExUkpwYDjAbC
  2JSo3K72AAxQR6EBS5RLhQ4tSvUbhcMe06oJuHZQ41k7a+kby4zxPbxkGYONXLjsXTSWU6oN
  WZBTggeU8y74QMGgH4waN3ZILPUrYJ+XiaBVXAXUhKEfGCh0CPHBOHQdhmYQ8LkU49SfSNfJ
  t0/340G4q8PZjdX2bc68y+3T/p1rd5jktz93zZZ7aQUfIjw9hbHChw2ErwEooJFDAvZk7PaT
  gOdJZwwnFwAnjlnfTLAggwAYAFwHcYXn3CrXvRbWTAH+8DQLbnllGF1bZ4ll4jYvmWcdWl94
  5k5YGfokTQBzhCLiHmLtVQEFOxLWXDUAJMDRf51wgsBQxollCzoMzGdhGaAIEIAAUkr510Iw
  UTnllCMBhp6WXP6IECcWANeAUSAiZYAHQ3FQYXtPxiNgnBemCAKHoo1GnZLqsehlNF2uaGJ6
  B4KXGYwQkljbkgA8UJdsFTBHpxmcwCcbkfrExhpuk9Z4lVQZcbDBqBtwINyAnEwgKqkYSKqU
  AaWSOmp+/uRGBDAFXPDQBK6SBZ2XDXjgAQaYbibmgTSqhOendNCCp4y2angjkH7qBsye0RJY
  qzucpuWZrxky61yyhOjzAHD+6FbwZrMAWLqRBtQJthGm08IZ5K8lMoXuBwtiZR9wHPRq3QH7
  fkDrEwaGGCQE+ArKGygMSHRcvV4AeJNn43Wqy4mhuRgtjhyTm8wvGRd4nUl4jikultgqkSeW
  iHa2coDHljaAvAXPB5kAZUIknAAS3QzcUwtc8ACVkn4MmT4LdBBB0jVXPOWjGz3IMQA9twk1
  LGvue3ATYmk58cx/PhwuRkhbXFLU7Lo4I9ti7CLAAA4knY8BFkwwgJTRpJIi2aAl1O2cgG4V
  mUuDjsOpy0rD/LF3UMIdBygSFAxcBleOzEkEH2CwwFIVbIDAv04BAAFEHHDQAQQHEPCy5pwc
  kAH+mx6Y+jrNAVDNL1pZcwT1TATvqzPCS03QQeqfD2eSV4077qF1/KWOgesa9yZ5nxkrWiKM
  AhywQKkfRAqLkB+knkECB6hY0OBqt7jLtYh2mPiGV2vrxEoyZ24s8yYOlH31FQFgn0hnuQ0I
  THMBcAAGArYUCljuAxLgT8EyILH2RY4TA3hABtDlNB8JUCdL0Z0FKkQLWKHLA8nbhYSEdyzT
  AGcD+oOZVQCwgAPuz38A+M1QImDBAF7PDR3T3hpixgACTiACiTEFASBQOeBIgF4/BMP7xDIj
  IL4tZlisXxbRExlCLSFjDVPYDJvnlrAcqm8XjCIbQOHAB94nhkNcShP+66ZAN3pggw+kwABm
  UpsAaPCBG6AA1OKor31hwAIWuIAiF3kBnA2lAxdIJCMlqamheOBrzHPXBzRgQ4cxDgAUoJ4S
  rkgWHW6Eh8+TYsL8Bh4YhXExqtBIB86XAGiBAgEJWCCbOLAAUnpKi0Kk2W5sWTjCqa1aSoFj
  XuCnGD7i8G2vLBTg9AAKAhbMgGrEAkZMeTw2ufGbskENn4AIigNwDpwWQEA0VVlIcLrznYNp
  YbuAAy+jCO6T6pIL4NgIHB5mEyAYSQAuqfdBWMbsf/+kBywK8IABkHAq0qBSHe+YxIQqFGPO
  IqMqIUq4ZWXPmfe7WioFJTKTfdKMOAzNxjj+QYAEGECjS1uANy3HMItCw4XwzOkHkqTMNDYt
  pxFYZ8UCo1PLzRSemLQKATSVJG+9jI3rOiNZflo+J8WNckPBAAkBgZbs9cljQaSFAP6YgaPB
  VBdXXBbk0sjM/41HrRkFFOIWl1J5uFKfeXLPRhPYHwgQdC4wwWPBZmPTmy4VXakrqlGtVtAA
  CoCqOW1oD60HgAMcAAEIsKxlMavZzHbWs5sFbWY9O9rRfraTnyQAZ1HrOKfKVAJ2fSonDIDZ
  lxbWOgJoYlajKi2R3o5Su3DoRwXnmQLgUZBnBYNa4Zq4wHFiARiIrnQvIN3qTre61LWudrMr
  XX/K82oARVR4vdj+WoCmaRNyhMgGknMBUQLWuINN4QDbVAG9CUCwigWOB8R52xkaYHaK9UAH
  5AtcYxrYmCqbpj4SQJuanOsCL1rKAnhrssZapQEFa01/pci+nhY4HwNQAAM6QNgDgwK+4Xvo
  GueErVVutFH5jfEFhIoSr57Rqft8yUlhRDMMd4ABATiAUDRs4QsZ4AFHHfBkuwCKBEEkeelI
  Toz31dQihzRCGpjyB5S84T81QAIRCLOYx0zmMpv5zGHmb/Wm6ZYBRMSeMIYwVCr7ASLtuMt5
  cUDBLLDkNXqUIV+tz0Yy0ABJETOuBchy+N5mUPZFA1se1kbsHPAAB1j60pjOtKY3zen+SxN4
  Wzy+sXmXhxRG39jKANWznGHMZzxLky7AqXKftQkABrCJyzTUcsE8cKZZI24/inajlIX9aWEK
  Wdfu9ACQ+6xjpLjZu2DhXKuUF6QPtAPHP3QxOhy5ERTOmWJXbaWLzbCPAbQxfJzpFm5QnE9f
  /zqIwHio/cjgaBMjmLJwPpGxHnej345Radv2gJMIsMHknRrVxDNFAhYwYoAxoIb+VuW5PgAB
  sZwT2cARH0ppZoBgPzADowMnro1cgWFjHF0WOFVj86qXDnigwYA5gDeTVOosX1uqNxX3SWJ3
  VAXxQ+eudtgUxy1MTkwcIg+yZVppgeIqlwKhJuHP56540KD+K8zexiRvgr9t6hqfLceABrUW
  Y77lh2qkX2ULpvtYqg6T76sCEnASwp0AjHNtwLYAIEDPT65xPGMVnMY5Ejgx5+4r11pYEIGI
  N2mn+G4nHvHdZjybglp4smGQTRPQX+yQ9PO3sOnmmxkp/6jYIfl9avOWQ3sZSV96kAYIoX8u
  fK1wpSkY1tu3ptDzBzJQUbYW0+gfWPXQZQ+2nVzr+PJLvvKRj57S02Rt/D6Zs5Kp77p26tGt
  t8lst0xQAJw9FtjPfnJBCBMJcPuBHLDAX2F5Lj5H4Mjnx3iJuwwKyLqx4uQDfKQljY4C+L8A
  BmAAAPh/ACiAAvh/ATiAB+h/Ajj+SGtDJcrjZm/UeYK3U50HAP9iZ8m0VQmnF/43ARkQgiI4
  AQXQfRgSIQ/UL6owAATgACIYghpwAQLYfUUXV8R1OAZVbQr0AN3HKbenF23Ee8QXXl30Eg8G
  HlRndauHABrwgk74hBnQhFD4ghrAWK8gGv/jJfADP+URfeOnCgRQACBIhSRogm5RAGVnCt93
  gqDAgg4ghSIogwWgdZKWW4rVXkqYFwJgHxCWIHv3TXmDX4AYcRUxAG73QBVXgeCkgc6FdXMi
  aQ1wKs6WHB1QLFhydBoHGAS3EaAHGGJoMfkgQX/IEQ1ggnSWerDwWBgwirskd2h1Ij84hGCT
  D0LyFD7+CIt/g0UD0BStRn/ABAtHaGCwpEknB1sJhUXUJy7ItHop8TcdWAANEH+kaIoE4wFm
  tzs7AV2s6AGsM36zp1tFdQEURjPBwwHGo1hFcXEiRy9XRQCHSFPV9k4GByQGQAETYI/2uA75
  eBjrcI/+mI/3iI/rUAHFRmsEIFmoYogbUQHDAGvxtHMAgGHzcoEPYFXFBxMPUEluhAFHczej
  aDUBwAC940YbQIKyCDNJuEXf0oh6wQDyVnVu4zF+dCknKXRhFYwsNndKxQA82ZM++ZNAGZQ9
  iXf8hy8+SFLVxzLNyDF0BwqHlXgZ4JMO4HJswpF8s30dQFABsIb6IJKRJwH+DPdw9uFNHOCK
  GyWIOfVySlh/upaOOZV53phw1jQhVHgmsvOCujMUnJSHlXJy/UR8lHNtECmBH2ABzCEyDmkX
  0wAKp8OJzeCUFrB/emhK8LQBvbY5BXMBSYMAAJZT01OTXYWE4PZqvwdWoWkK5yQBNEZZLaYX
  Buh/lYMBCRiAWuki1OKI86ZN6sOM6YFMU5R2YuKU/VEB/idvBKBaq3gflwkAnMOQnKBqmwlg
  GsAAYVhuJfgAQuEfcXmRCJCX8HSZztFOMeaWOeVBG6WRwDEBXDQjx1drD0QhfDlPfnlKGwd9
  FFCWDdlyQyGJIaIAR1Vx4LcfM9WJebFUMPQj25T+X7wGCrqHcrckjd/0mXMXGoe2VuFmgzeY
  oaYpDd2pmLcVMiYhkcIyU4zHGq10LbfJUcMXV0WZFa2VGndVFu9AV0PATIEyQxKkAXi3RSEp
  ZZepaoxZmLfUFE+zGzxTPpbIZJwAjopFeDqJFQ06ZU9xcaOYLoT4a+g5FOpZJaa3JR3CAO85
  jhuTf0alWNBWKL+hnpKhFzP1ICFiTQy0DNs3FAR6J+7ppCG1QvnVAb22ifo1HwhwflWKLr22
  dloUe89XdCn5g2ECoqbQMxXAhYaqDLZGlYgnLJZKZBa6Uku3XPhzpaMnEGB1dXLFmwRBdCuC
  YRzgQcCEAL+xoPsxO5/+k2gCZyRCYQEQtx6mMAAJQmRrx6RFhaeRMwBZumuANIfn9k4dYFui
  R2sA4GS7JiscIA6BKivvKBtielXEiHFneiA34wGM2GbD1hH0QSaIdZjLVEkEikFCEa6fBK0K
  2mvqGHzA0KFTxgHuxX8qqauuJJm0BmkdpQ/1lYt3VQCUmESq1JoYcZzHOQDSdpwFwIJbU6rj
  4ojDVZ//qoy2dHUtRqq9GT/N4ZQP4ZynCROcA6ucQwEBgGEYUK9CIY5hpQrn4gGCOVQAAKw6
  JawrFaUxZhzb6k4kO6EwUawFg3/w5AH5ClxkGmAF061XGABZprIBqpBD0WDVdK7lCrXAwYj+
  oIBhavkk+oCWRYW0ITQ0nEC0RwtzyPJ7t8hmW+BobkULhxSzGCRlJPtiCiuzwaeirlcG2YFF
  W7ivbit2MCpublFvSeghW9SUAKBnagl7tFA5C6ov3/MBTxGowfeSIjUjGyS1CPd3MaazV4Vi
  WiYOq1FUkRlqkraV8JSI8ASfT1oTP6trTrs2ASAUX7sQ7ggcnusW9rcRVQYMEkmnARoAG5S7
  /GNryAaXDSq1wbK8QwiTdIuqT7uvtycAFwBFaDFxlXiSGOoTOAlGllENsWg4gqukIZFFY3GD
  cGWj5Tp9/GO8EOS+jmawu8cceta5wNBEKocyaLEaX4tqYYGzOSX+umsnv7rWskIztstps2iL
  Lq6rrNmKXrOrZbX7r4KxanlBmBxhW27BsxvBZyA2kqAHCkentuFFmVpmHAWAu5+DAINawP5q
  eKUpmtv5btN7sY0yYzUsDQXQFG4qNSuaDzh5qKDqrDHqv8NEvWOShMxYdU88vTuxen2bGRvU
  vxhqEg3Hwi4MADAcHx22bsKCsPynwsE6wzknU7q2oFRbVFbIfw4Qw5yniN/Uu4FTwVN2wToB
  DOeEn0Win+mpCiCMbqYApuhSwnkXGyMUskt6chjgEe0HE2YcumjMPMzVVhC1UrrqvlxECw9A
  LzrMCY1pmLL4i+C7t+qrdpOaxKxcOFT+h77mWz8dFrBsKKPxe8UCsnSg1Dmuc4QB8BvY9KmA
  UgBjHL2YOWUtu2H1d61p2Wu7mF8aQINK2sbfBFun607uGjlLy62lLKAcwarAqqkwti+p27WG
  BDUZMRSFepEEnF+fQwDZ681jO0HyA8tlMTiBW5Mhcnu3KA0EMLGHdnRJqq8xyaCErKL6PGoG
  xskZ5QUMnco2TKFGvJKWLL/9a3qTMRUHkByfo2cMSczBwa/iRsyP+7lL4XE6dUm+aL9a9mZs
  mV/mqaTpnGzY+U6rmodugccxpscg1DXA+wtZmkIgXTCXubsc9FAJMFPrzD/trFgMpEdY45cG
  nAUYjc9dxan+PIp1FQrUg7aaOKySEcWDWUw/OZhRkMtcJp2TMftWuQy4YBu28THWGGMfbuJH
  rlMm0VzDhQPAZLy6YvtOyLtR86pYFJBCfZlfO+o+Op1fPK2t80mfNoUOE9J9WapyerovDzLU
  +nWN6oygpuGdARYXCgAMD6xTTWWzeq3EqtsiSifM9POLljEjsRHMQgt0ceVoZS24G8rETezW
  YJW3ovnQSTlDM/m1aM0xEgK7Z4uNGEWw8otckW0azOx4+7JflYw4C6xYHYABCJA0brHN8JS6
  120VATCXWpbXsQtQ4E273fzDgQwKmsIB3pQBFRI8EGQAz8vc5ybf2Aox6DKPxYf+wBhnjBEp
  x0ebtKQJsq0cM+H20LrcycNUb1nDU6uM0CkJ4TdaYDEq3BfOlPb82s6ibj08fHSIFQQg36wq
  1+iAeRDDJr3mqXKTZQVZKAkA2kbFX+HW1N+0AbVkDs8VYxxQ4gGkxsWY0AC12GbazY37QkkU
  G5HSAE6WGA6wARZAebVmARggtfKbARIwVrJBPXuYcf6qD5Nc3e+EAUHqyBNMftL7iOil1rkM
  Elbtvix9t0OF0bjoiEMEdbatoeN9ZyTDYscN0aosKBimAT5SoTZ6LmguyoU56HiuF2WiqZtQ
  rxJg4OHTv1fVcYxtlqoRY7Rd24OcX3CZ3kf+QJjetN7+G5Ho8sgIEj6SIgANUGgw0SQQIykt
  WCEL4ABJdC7wGQDJaqLyBOwnt2qDHboo/OEqfsP3w+crSmp2mudxZdR2jtoYjps4HcVwnm7M
  7jKRbnxjvaiuvIw4qhFqam961r2HFbRtTgur0QE12+C0ZePc7YDoZcgSIDt2NBQUYAC/AhtT
  BqCFZQr5PWW8YuSn/tiNrVASGUHfMx8FcABxsQBwEQ0DkEuz5AAE8JICgACoEWT+vhQZAAHQ
  yCbVfpFEjmxnYgA2rlMLXxJU9+BVbM8NIJuqHfO5scTBJRgjZ7PlW4RkvecK/uB9vuoaeqLb
  XrLjdJHMuWWGveEuYshG+wH+5y7M+5BondPtoQoTEMpYsasPlQOX9v1xVo4NdTvPC3nETJ/f
  HrAjO8Ijh/H2h3EY8lLqpp7w84lKQjs6Vn6V6DCV97EAgj/4EwB5iuc5gy/4PSMf4F5rFQCe
  v1bwWjb1y8ta0nThAKvMhVyVIk3R+3xoPnHNkG/tYpH4pn/6ppjtHKvzwnjnNgh1jKrzaDwj
  DuRtSi8WZWIcQuImqa0KJuTqmu+UbWdHE6BOWs84JE09Y18wwE9tR0fd8ERkzYrEFN8AT6/R
  s84A3l0i623BRq8PlnZ8DiABfx1jHQBmVtIuRRqXMDLqRRXBugbdX//zIPvmXL8RaP6DR8DP
  u5D+f0E8qUCAkFQ+ReMRiZkwAE0nCBqVTqlV69WZ1QYAXKe3yRWHs9zrGQ3SftdrMLlbfsPf
  42yaWqZ8OJNDOa4NjCgCAIFPgG1LEUAgIuMjo+AOr1Iqa6AhYwOp82gjQ4JAy9KyqcDjAwGA
  AMPTCGPyqappAPIVF8kicbbUdI2gQFg4OHi4gICX0pf5rEkhNzq3sKk5LasAw8MjozsjVTr8
  VcO740ODqbf0ywFc/BUCIOE9d0LZGq9Nfz/wCx/ry4UPG/7wg2OlX5wAduhkSWBu4CgA/67V
  cmVkwwYNGjVuzOiOQ0EzFP8lNGlyziI2JGkZVMkv5T4wMScyyyJgzzn+WYL0WfgQ7xAHZSfX
  CCASSWI1lnmcIOBE74iHdEqXQjmVagODix86TOBwJJY6pg9euQvHxEtVpvIYMHiwDW5cuR8m
  LGCwgJfaZs+giqNWU6+aJgf6FpbmQOw6J+3odaAwYcJjCgoAFJAMeYIGqBTABI7iBpDKmJ31
  wikwYZXMZS1d8mwiwchOz1adJHg8QdRC3bsDPJA8FfDsinAahiY+581sRstBEzfunDnpvU4E
  TCiiwQD0LHbbMoBkYUEErg2643UNYMBRSavVOhkgQbPhIo7/jKx61dMFAAvArafasoBb5Cui
  AVI8C0PAvjyQSLhK+Brwlb+EawInCC0MK7j+6Zogy5NtPsCgOwYoaw2AA9paYAHrPPSEgnuU
  C8SLGIuDkbkMSTLIMlnmcMa1lGIS7wMPsvuvNBJJbDCf5oxUMrH7FJlpSZeg3IeimxZwRQMd
  FxoAs8jMksaCyCKrzwujxtvJxqXcE8jCIghqEp8mCIgvquys+1A2gABooM0PCgzkRQKeMiy1
  NJG8RJ4+kZBwwkZyioYDbySddFI6c/HPUMWacEBADTS4AJkCBtBCGANMNaDUY1AdlTpiJugG
  nA3sgZOl82pcsj2UmmCgg4IMROg5JtcA8oM8PaPJSCgP5fHW6GyNY0JktyjAVGqrRZVaa62t
  dihbb8yCAEg0QKD+gAo48iRSb8ipFN2NMiCXCA4S+LVIVnxSdCBfcz1g0CIqCMAATjDMVLD9
  ILLwT1pLeg3CDNJaFlho8C1CAoW/7eJOjCTYWALIyBRjIYVC3pJjjrcqYmA1qYNAgggECFnO
  BmRuAAK4jPAQ51QomLkBX7sIoAHc0GyUKNWEpfdiNzr4i70ptCMRZgCIfcDiOKNsDWJgr766
  QZkY4Pkts1IZO0icgyxiCZ6TWkNNMLIpwqwNSt5YJJB3e49j3E4GR16kcx2AzYnfJNimJgx4
  oINtKkiELP8CYBaAwC1ErGmWvJggLvBQ3JzzzSOIC4O8smZqgM5NPx31zo190VEINvf+mbqF
  Xp49AAFelv12YDZvANSqNZWDSwooOFm+DYSvAIGoK3dyxqhlJKoO33/ph4sH1ktOz9Cip34N
  CDB6OLAvag/AMuGPP74C8ymoIIHxpRv9MwAMSH99+s9Xv4L81zdf//MrWEB6VnPCAyywPlwk
  DoEeSGBcDtYJDKxvVtBi3im+4QEONABFsAsWPwiwOW1YcF7Lq1UTjjKx64hOZV0gwAoTEQDY
  VIxITguD9xTFgaGFzxALGMAKk9GaAKxwhwiYCvzitzWuISl2XzhAAQ5QLa9wICNRlOIUN3CB
  AzQRi6yqRQA1NQAEbOIrJpxPRhqwugMZTTvPUx5yckUcCAz+iQyQe9p53mALN4nhRdWB4gbC
  GI4OZIQDKSNiwRYgRkZ1zQmSCyQGMsBIDFSAh5GUJBAh0EhHYsBSHXBRvdBDgB3qY0rSaoMA
  iiFClfFLjEZI2AS1QJgWxRBRXSCWouBIOGvoowAO0OUuebnLEC5ikNRpQC+JWUxjElMij0Pi
  M3aZMQu54wK7bMD7csWhVOaHixoSpRFNch9kUdNp2yODjETJhYtoAJwjLGQnJhABd77znRLo
  oxFgaMszngKe+dTnPvkpgRHBsl4MaFkEJLCKMShPSgphBTwfkM4RchOiYUCk1K5ZhAuAj5WN
  UIAW9dSbLz3zhm0EmgVIWoEE4YL+A/kjqQQ4Cr8swCaVF2gp0SRXUa6wqlHWjIYCEdgBn/5U
  geKwQDZ9gcZFbJMfOAxW9kTTnG9eZAOmvGUtKuCABCRgAVeVFgIWoACvLuACcCRiRI2IJKQm
  q1kHYZ0PbWe72rV1di+LEiIP8FETUm6ZIvwCDU04CYw+NAEcsCtUOtBQgLKuOjYVJOtqWtEO
  4PRFOk0CBiiLAbwMALOZ9SRmC3ABynoWF/VEJDm3MJoZOSuvtNLe1Z7XhYvYkKjDaUREeWEf
  iJ2VrFuY6ElwCxMqsY4mBUDAFSHQAQ4cF7nJPW4FrnhFmjh0hC5s02BhAdm1ZioLfJ2YkKRa
  OKBVEkL+HIjAA1B4KHBBAL3pVW96x7te976XvYWy51T3RN2JTeCv1WTnAyCgxa05AAIRaOBj
  D1sr0yaktb4dZ3cXJkEr2KG3Tm2CgCYQWzmC0qhtyBo7HtBhD38YxCEW8Yg7nBThMETBRfMh
  g2/pI7cgbkDjJa8gEEmABlYUgPMd7Sy3G9IUniIBCAiygH3KgZ8a98gWCDIChuu3E+eWm8o8
  4wAOhuSfPoDJSw7ylrMsZCYzwMhWdkcIEUkW4zqAybyw3QAEMIAHXOACFoCznOEMZwoko81a
  PEACFKAB41r3nrkFmYIHKUODqHGOX9gKBSzMVBqBRloy2jCFbnzNX+q4cDf+YTNmBQDWOn8a
  1KEmb57bvCOideE9cHbHkTeQVS57+aoYKHLiYBHnAmDvniW06UDKO1ZZ2hQ4Sm1DZje95mJ3
  y8mBerWXmc3lV2uZ2c3O8pYTYID8OgkncO7vptkcYfSwuc0CYECdZYHIBVigh4OJ9ZH9+FMO
  LIDJ1HFzr9t46ERnONkQw/BB961QLVC40VrDN743zAWjhBrhCY+zwkNdS6J5cQEbQLKR7YsL
  BR4XyQ9IgImJFgAHHMwDcJ4ApzFrEGML4M0X6KMHIihlpTa2LBYiMKbrhYp3VPwVG4DuQyMa
  TABIdtf6KXCVXKIA+KqXv+mlnEF2uyv0ztNCEoD+gGFZ3AyGJBjDj340ioe+W6RiHZQB2EqF
  u74XsOtKxTQnelqzjsZQqv0XAXBmJwpLYhATLwkMenIXYNOBqU8TygroMCQuwPEz8mfXRzjk
  6Khzr4rqvOpr3+vaym7enyf+CEKHu4NqgdkEYMCzn630TitbWQe0mQDX9qa9tJEL4yoX9tGg
  bKH2jnaoQefAm5c8HVMckwSAg9GVpy/ubR/wOCEnSlPC1YuAhtwIHGC4VySXkQgw3OibCLkS
  2KSwBXBFmRAb/MRGthOEu7YnA92mmvc5ARxfUXTuvF4BQIB1MKCAoRRahuivqPrvmYANKNCu
  OqCkLKACCNAADbAA++X+CBDIAVTvodovKgYweeCKAslHpSoA6oyAAwCNk4KltbhuxSJvgpSP
  Sb4JXMCB7FwutQ4M/2LpWdIu+YSvqG7CdqBMC9oK14RNHxBAAa5KAniqp3yKpzxAybxKAXIv
  pzDPCC5KBlVGACDwmtBJBK3C25ygAcwCv6JM94BF/66J/3AoACRHgR6Al3KsNRCgl4hn5raQ
  WWqqsBwAOBYMZlICDTkFCTSpCX+nRkjw9hCsC3yN+AKREZCPFVDQ+Kbn61qwYKowomyr3tpu
  wdjguVQMuMrAAdxp9G4mGtrpAQYgJpJQCT/gCyeNMDAP8iRqeoQlJgTg8yotAxrAv54GWVT+
  pgtj6hBZYwEyQLyYaImAYYci6ReBaIXmgAAMoIkYICOoJg85DwBcIZAQAFWcIJIkwJIuyZIy
  wAKQYRjlpInMxac48BHFSUlQjA81zKUIcRBf4ji0YACAz8LurffSCuwccPhscK7CJwC0qByT
  JdLiKNB+yAAu4BtuJoo0IAGW6IpMpYmMUeUyooEaaQHSbRmTpEJC0T9cqgnwLgpz0NGuBqtg
  7hPuYvyWTw+dYAHox37Sx37uB33UZ3/8xyXzRxkBIYV24wsUgHM+joHkInG2QQI6x79yZyKH
  o8PmECd1ktakIS5+Uoe+YAFeiQ2z5+xSDAaBifHYTh3tTU4gguz+aK4PsVL5QBAEJ3IfuCp1
  zhItUecI9UFNPg9NIFHCnkQuhxINFCJsjEAlp2l2Dm18EoB+wmgboDIqgeUQQrEIlPEq9wPn
  FGUD7q8uAYEBIKNkcCNvOmZjMKcTLODjPKECKtMzKbNjIEA3RNAeS5Mui2gMFIAyH8VCLoBj
  GnA0TxMhZsIRJIA1IcQ1JUCDlGMSJfFofiYubzEVAeHrqlJO3LHsmgXrnMcNGFFYZnANECAD
  K2rp8k0xCME3e9M5rVOAAgBIMACDsnKDtAABGuAoLGCmiqQwDRMxJw0A5s6mpNBiwmCFqHEg
  BwQ9F8pCuiEbby2A3CNUjkFAB5RAC9T+QA3U8AQoG8rBprjBG35J2MrlpPDlkcxwMFkDLsdT
  ENBRNkvSOOEREAjAELsuQ/lNJjgU91A0IUohJtIjIzbCIzqiIzwCRj1lRm20Rme0IzDAZ+Av
  SVhBVFSMtNqunACFkwSAWijvVlS0FkxlVOhxYUqRPTv0FwgAJFNJPgvsC+4TQjSApbLgzaZr
  Qf6zCfZALs4UTdNUTdc0Ls6B3jJN1wxTKKgUUcQA5YQnEw0jfQ5AeSoxw0bDBFVidMSRRgh1
  SInDxtCmasQzUP3NND8x7pqTNyaVUmWHN25HNzB1NCF1YYrUaLZpLOm0iKiSexKKLc/oAHqS
  TVeVVW0Gryb+bZ2U8BS11AqRQE0trrDeshYYQAOo61ajwoyY5TYNEwkaUzaboi0aQFXFKC4i
  wC4WAEoFyCHa4v/IRinLxgMeAEVSoukOtVCVMx3ZZtKiZzmJUxAVAqa6cjhUNPcC8QMHjjsN
  TetIVSvBdQrXgrVs0EetoTdf8EMvFFjIp0CNQUAJ9hgMlhgSdkDBEZESYF0oBWIjVmIj9l/g
  zz3io5GWCGEJwDsopT8lTQtWKAIeths0AAIQdhhMyhw6gAGiNZZGdmJjVmZnVlIswGVlS06I
  AQJIdkKjARQkpQKOgVuvC1xChQAyg5Eg9lMSVklFNWDplVETyhHNCq2+sg0iYAP+4qHsUJQR
  aYMqfSR6ZrBRSRW3JvGhtmcqvXWNSvVemTFDxfO0LM8071HfwsAmN7VS7cZufkZvbZLBBoMC
  GirSeOMkRtVu+RahSuspzRA6JxVxERdv+5Zw73ZkKrdPk2ZDd+MBJDNvQBM0ReFxkyq1TrRS
  YbBuoRYezyox4dVfN5TF8vVp5DXRJnFfXdDbxnZJOLIeTXVJt0Zut2Yf//B05/ZI3PNt67Uq
  pxZDyaojIQo6Iep2N+hmPZR4hRf/cqvQjBN3By4x+5HtCBU0mFF3vdYc1zXZTpV6eSvt2rX4
  1CpX2NVfpaUFo5bpFNF+D2U7t5NMz/e3zHd5rDLT4DaVBnuu6e4Nqe5X4EQXgRe4gEP1qFaM
  JoaTgcl3dwX4X53DR8T11EAUjSb4ErbJg0NYhOVWDTy4bUcYEEkShVeYhVv4tpy2jVx4hH3T
  AwFWhm8Yh3NYh3eYh3vYh38YiINYiIeYiIvYiI8YiZNYiZeYiZvYiZ8YiqNYiqeYiqvYiq8Y
  i7NYi7eYi7vYi78YjMNYjLU4CAAAOw==
}

####################
# Merida1:
if { $png_image_support } {
  lappend boardStyles Merida1
  set pieceImageData(Merida1,25) {
    iVBORw0KGgoAAAANSUhEUgAAASwAAAAZCAYAAACLvRPfAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw8XSvuqHgAAIABJREFUeNrtfXlcVOX+
    //ucObMyO7Mw7KAgIIi7JiKkoqXW1dQ0y8wsra55s7LV22Le1KtlWakt1i0rtwz1oohbaCCK
    mSjgQigqO8MOM3NmfX5/cIY7KsvAr+73t/R5vc4L5nDmOed5zvv5LO/P53kA/pQ/5U/5U/6U
    P+W/JMx/4R7Un8P8/51Q/y88lAqADkANgIbe3nTUqFFwOByiysrKcRaLpcBoNN6YOHEiFAoF
    iouLZTwez8zj8ZynT5/+o/qtAuAP4AaA1t40MH36dOzevRsRERGRNTU1wS6XC1qttubq1asX
    IiIi9PX19bFWq5WSSqWtQqHwF4VCIaqpqRnc0tIikMlkjsDAwDyWZRsLCwu9vaUCQAj3zE0A
    GJFINF8ul9/T2Ni402az7QRAAGgAGACU9LZvt4kcwBwA3wNo/gOxyAPg/N9pYMyYMbDb7YKa
    mpo4Ho9XXFRU1LRy5UpUV1fj4sWLDMuyDkIITp48+UfiKgDA9d6O/cMPP4zvvvsOSUlJIqvV
    KgQAgUBgP3HihDk5OZlvtVolhBBQFOUEYJJIJJTFYpG6XC6Kpmkik8lMDofDefToUW9vKeHw
    UgXABIAGMBPAPQBSAfybw5UcgBZABQDL7zBWIgCJALJ+p/bukCGhoaFHJ06cWBkaGnoMwNDe
    NhQbG4uIiAi5SqXKDg4OLpDJZI8MGDCAMRgMMRqNJk2v188CgMTExD+iH8rQ0NDt999/f53B
    YNjADVyP5b777gMA6PX6dffee699xowZ9oCAgB2EEMhksgdHjBhhmjdvns3f3/8sAJWvr++A
    6OjoigULFljj4uKMgYGBowMCArrznNzPJgsICPjXlClT6v39/f/FgazfuHHjyjIyMsjw4cMv
    cqDzDQsL2zt58uR6jUazHoDAAxy98cR0ABYCyAGwCID+D3gfvgBeBfAjgNc5hdsr6d+/P2Ji
    YhQSieSYSCQ6pNVqkwkhCAwMDFQqlRsNBsPdADBr1qw/SlntAFAP4OPe4mry5MkAAIPB8IKP
    j88vEonkF51OtxoA/P397/Hx8TklkUh+8fHx2SaVSuWhoaERcrn8sEQiOSOXyzOjoqIGhYaG
    dnUL2gMLYgAfcUpoM/c5nDN2BEAhZ9gVAL7lrlvlgSuGa6+nIqYoKglAOoC7Afj83i9DoNVq
    t+/cuZNYLBaya9cuotPpfgAg7E1jI0eOREhIiCA0NHTf5s2byaOPPtqi1+tXhISEbH3rrbfI
    mDFjfhMIBLMYhumvVCp9AGDGjBm/V1/iZ86cWVtcXEySkpKKAQT1ppEpU6a4gbVhz549JC8v
    j4SHh1/18/PbIJfLM9555x1iNBrJiBEj6tRq9adKpfL72bNns2azmSxYsKBVoVCM8fPz66x5
    vUaj2RwYGHiApukEAH1TUlLKLl26RMaPH18GoC+AgCFDhlxYvXo1iY6OPgFACWDYrFmzGgsL
    C8mwYcMKAegZhrknKCjokEqleo+bVD2REAArAOQDWAkg7A/w8N8F4OImiAvAP3s5CTBw4EDc
    ddddQqFQeAAAYRimUqVSPSmXy9dTFEWEQuEvarU6QSKR6EJDQ3k+Pj4ghPxuuAJQy/Wj17ia
    OHFim1VVKj/i2iJSqTS7b9++4319fde4z9E0fd3Pz2+an5/fIh6P18L1tzUsLCypC1ypAazm
    FOtwTjnd5NqsABAJIJBTVATAGc6ADOYUMQFwmTOOowHs4vAh72E3IwFs4RTj1wD6/94KS96/
    f/+ca9euEUIIKSkpIbGxsWd6MQEAAHPnzm0bPbX6H19++SU5dOgQCQgIsI0ePdrW1NRE1q5d
    SwYOHGhNTk6u0mg0GWKx+NGhQ4fKgoODf4++KPz9/fcnJiYSX1/fL3qrdG9XWCaTiaSnp5M9
    e/aQPXv2kKKiImKz2UhmZiZJTU0lqamp5MyZM8Rut5MnnniiU4VFURQAzH3ppZecP/74I/Hz
    89sOQKHX6zcmJiY2abXaLzkPCwAmAcgFMMoNSH9//72JiYn1KpVqJQB5SEhIRnp6Olm0aBEL
    YEovumrgrKvhD/BK5Jz3RjyOXuNq6tSpbe6oTLbO3V5ISEiLQqFocX/u169ffVxcXL5UKv1M
    KpXePWnSJP7IkSN/F1x5hE+9xpWHwtrgMSZOAGYANo9zLi6UYt3nOIXVlSF8EICdu347Z+S+
    5ELBHQBk3HV/AXAWwDgPL/gAF+Zu4Pq6l2uHBXBfL7rqD2BbTxW7t2FCS0lJSVZaWtrIZ599
    FmlpaeTatWtZHJfSYzl+/DhCQkIUdXV1/fbv3485c+ZgxYoVfIVCAblcjri4ODz88MOCBQsW
    6E+cODFhy5YtydnZ2VMYhnkNQPETTzyBL774orfAUovFYklISAiKi4tl3Euy9tg1aFMscDgc
    2LZtGy5dunTL3wsLC+FyucDj8dqvvXz5Mg4cOIBz585BLBa3n/cUzuJX5Ofn17e2tvqazeYS
    AE3V1dUvVFdXuwCUceAFZ9FFAOq4z/UVFRVXKyoqzgN4B4DLZDKVHDx4EEVFRUYA1T3spl6j
    0byuUqmSampqljc1Na3oRRtdCQugFICnxijtLadRVFSExMRE0a+//qoEgLvuugtPPPGEdMWK
    FWhqaoPq2LFjVU899ZQqKysrduvWrTNPnjz5pVwuXy0Wi41ms7nDd+ItrgBIPZSXHICx14Qe
    j0clJSUhJCQEhBCaECL2xB0AihAicp9zuVw4ffo0LBYL1UUfGjilo+TeYxOAJdx4lwJo4a67
    yXm55dznOgAXOM/rDU5xunHQDKCxp/wVwzCTBQJBlNVqneJ0Ov/1u/FYQqEQAOJ5PN438+fP
    t5lMJvL444/bGIb5DsCgnraXnJwMAJBKpc8sXrzYnpqaSi5evEhYliVuaW1tJUajsf1zfX09
    Wb58OdHpdJkajaavWCxGampqr2iO8PDww9u3byc2m418/PHHTj8/v20Aeuy6zZw5EwCg1Wqn
    8/n8iunTp5PvvvuO7Nq1i+zcufOOY9euXWTDhg2kX79+RCAQZERGRgZ14TEyAJIAbATwJDde
    oGl6HcMwWQAUPB4PAoFgjl6vb2UYZiIHUn+GYQopinqJptujqhe4MGtkF6GWHMAIAJMBJADQ
    FRcXQyKRLF+7di25efMmWbFiBRGJRMtv3rzp5rZGA7if8+4UvYSXD4DlHp6DHcBbHhPfa0lJ
    SXF7JnNomjYLBAISExND+vTpQyiKavfg5HI5mTp1KklPTydXrlwhjz/+uEutVu8MDAzUd+GZ
    dEvLAjhym/eznQupe8WN+vr6vj5v3jxnVlYWOXXqFMnOzu7wOHnyJMnNzSU7d+4kAwcOvGkw
    GAZ2wY0KOK/8SwBPeJxfDeCER2g3nVNE93CftQDOAXjN4zvPcvic0IXjQ3MhZRjHfzJqtRoM
    wzz+9NNPm3/88Ufy2GOPWXg83pNqtdqNez2APtxPuqPMTJfidDoHx8bGfrNmzZqUhQsX8tRq
    NUaMGMGLi4uLKykpSaqqqsrl4l/vGFZfX7S2tgoCAwNfW758efSlS5fw8ccfY8KECZDJ2jxS
    gUAAiUTyH4ZOLMbo0aPR0tISmp2d7R8cHHwoIyPDWl9f710aRCKB3W5Pjo+P/2Lt2rV3TZ06
    FYQQDB06lNLr9bH5+flDGxoafu2J93Dx4kWMGDEC99xzz+WioqJzDQ0NsUql0p+iKDQ3N6Ol
    peWWo6mpCSdOnLBeunRpi0qler6srKx8ypQpuHDhQkfNu9CWDWQ4C3ho4MCBTTabbWJUVNR9
    lZWVmRqNpkQkEs0YN25cyo0bN87weLzTPB5vUnx8/MLW1tZjPB4v2263x3IWcb1H6HW7DPLz
    89s4YcKE11JSUh6Jjo6eLRAIprz99tshQqFw3IsvvhgcFxcHPp+P3bt3Cz/44IPIIUOGvD1+
    /PjFSUlJcwIDA2c3NDSMbG1tvQSgsgdzU8uFFws9CGqa41YiueyRyevsgE6HhIQEprCwcLlG
    o4lfu3YtxGIxMjIybrnOarXi8uXL2LdvH8RiMV599VWqqqqqf25urk90dPRRpVLpqKmp6YmO
    uZsLAUfSNA0fHx/Y7XaKU2JDAeRxGTivZNq0abh06RKUSmVBXl6ea8eOHQPz8vKE1dXVyM/P
    v+UoKCjAqVOnsH79euzcubOwubl5aUJCQhbLsqSTPjgB/MbhazGAQ5xiSuHCwJ+4DOc0APcC
    +AXAaQDjuet/4t5LX86wrAdwjGvvDu9cKBS+HBMT84/4+Phn/Pz8ZjscjhFGo5Hx9fWdsWjR
    orjMzEwkJSUxOTk5vJqaGodWq13Wv3//VyIjI5+SSCSzmpubQ5xOZ6GH59e9SKXSj7/99lvS
    kWzbto3IZLLN6EF5REREBMRisS4xMTFv165dJD4+nixbtow4nc72di0WC8nKyrrF6yKEkOrq
    ajJu3DibTCZ7GmgrLfBSogcMGFCYlZVFCCGkqqqKrFq1iphMJkIIIXv37iVhYWHHAfTYxC5Z
    sqTNxxWJBvXp06f0gw8+IFu2bCFffPFF+/HVV1+Rp59+mshksm3x8fGSbrKDnjJUpVLVS6XS
    dwAgKCho/euvv078/f3fB8BERUXt2LhxIwkKCtoIgImIiPj21VdfJUql8hUAlEKh2KxSqUo5
    i9WRBAUFBZ3ctm1b+1g7HA5SVlZGPvvsMxIREeFat24dIYSQ9evXk8jISNfnn39OysvLicPh
    IIQQwrIs2bFjBwkJCTnZQz7iJQ+yvaPj9Z68h7i4OPTv318lkUjOrFq1ity8eZMkJyff0iaf
    zycxMTFEIBAQAEQoFJJPP/2UXL9+nQwfPtykUChmAcCiRYu8xhWAi+72/f39yerVq4lMJvO8
    78+94f8iIyMxcuRIvlgsfjwxMbHx5MmTJD8/n1y4cKH9KCwsJJ9++inR6/U5Wq02FgD+8pe/
    eENnDOHohDe4U2u5Z11HURQN4Bvu8yaapnmcQiYeHtZ7ACpomu7n4cnf4rErFIpvly9fTnJz
    c8mhQ4fIvn37yI8//kiee+45VqvVWtasWUNeeuklsm7dOqLVaq2LFi1it2/fTlJTU8nBgwfJ
    6dOnyRtvvEGUSuX2nnjwguDg4H9fuHChQ4VVWFhIwsPD070lGAcOHOgOM+cuW7aMLS0tJQUF
    BcRsNt/SbmlpKXnsscdIXV1dh0pSqVT+6ufnZ/Bm4mu1WtA0/dqHH37YnjB48803ycCBA8n7
    779PKisrCSGEvPLKK3YAc3g8Xk+xBYVCAQD6xMTESzU1NR2O1Q8//ED0ev37APDSSy95naVL
    Tk6+npKSUgFgiMFgWL97924yceLEfABxY8aMyc3JySEDBw48BGDI7Nmzb27dupX4+Pi8DGDs
    fffd15SQkHCOI0074uCeW7x48S3GwlMyMjLIjBkzyMWLF8mMGTNIRkZGh9e5XC7y3HPPEQB/
    6wTAtwsfwL4ulBXhUt5e4WrixIng8XhQKpX3CYXClpSUFBIbG9uumNyHwWAgW7duJXq9vv1c
    bGwsKSsrI59++imRSqWZBoNB5e/v7+37ec3dTkxMDFm1ahUpKCggy5cvJ2FhYe57OAA80qt4
    2ccHIpFILxQKL+l0OuLn53fHoVQqiUKhWA8A8+fP94p7pSgqmMvQlQIY6KGwznNKOJv7fBjA
    AC7r6VZYIzllV8AwjI5hOowG5zz00EPW7OxskpSURBQKBVGpVCQ5OZns2bOHbNy4kcyaNYsU
    FhaS2bNnk3Xr1pHdu3eT5ORkolKpiEKhIMnJyeTUqVNk3rx5NgBz3bxcd+iyNTY23igqKurw
    j7/99hsaGhqucxxEt3L16lXw+fy+AwYMeHH+/PnCwMBA9O/fH2Kx+JbrcnNzoVar3YrgFhk/
    fjyioqLi6uvrJ5SXl7dn6zoTo9FIqdXqKLeypCgKDMOAoijw+fx2EnPQoEGMUCiMNhj+iGTY
    LWl85OTkeHt9k8PhqH7wwQcNer3+b3a7XabT6TB27Ni+AGYGBQUZ+vbtC39//xAAs8ePHx+g
    UChgt9t9+/Tps3TmzJlylmUr0UERY1RUFAD069+/PzpTMtxY47333kNUVBTGjx/f6SSIjo4G
    RVHRCQkJ3vTL5UW41wovC0nPnTsHf39/P7PZvMxqtUoPHz6MgoIC2Gy3wjI+Ph5WqxWNjY23
    hPb79+/H/fffj759+45oaGi4u6KiAk899ZQ37zLKgzqBzWaDy+WC1WqFy+XypF369QYshBDQ
    NA273Y6amhpUVVXdcXj25bfffvNWYTVx9Ecgx0X5eJQbPIC24ldw3O4MDy5OCWApZwCraJpu
    uR07HBc18q677hIcPHgQx48fR1NTExoaGpCZmYlnnnkGISEhMBgM2L17N3Q6HSIiIrB48WJk
    ZmaioaEBTU1NyMzMxIEDB5CYmMinKGqUe553aw6bm5u/Xrt27ZXMzMxbzv/8889Ys2bNpYaG
    hi874UbuAH9iYqJAp9O9sGjRogHR0dF3XONwOHDs2DGsWbMGo0aNQkfejkajQVJSEkMImTx2
    7Fg+RwJ3+d4tFouptra2zWUJCcGiRYvw6KOP4vHHH4der3crNlit1lZvebH/kpjKy8srBwwY
    gAceeGAqy7J3FxQUQCAQiGQy2YK6ujrd5s2bYTKZgnU63VyhUEhz4/GXOXPmjAsLC0NlZWV5
    R1nQ4uJiAKjwBPwdjClNIyUlBT///DNSUlLQlffU1NQEQkiFl6sTnJyHZe7k72YAezjvpEuZ
    PXs2vv76a9pkMi222+2ju8yj+/tj27ZtsFr/MxwulwtpaWlQKpUYM2aMyOl03r9gwQL6zJkz
    3eLK0xBcuXIFmzZtwq5du7Bp0ybcuHHjduX7h4s3GU5OYZk9+MbpAMa6s3dcksddIBwE4HEP
    Un0yR9qDoqhyhULB3l7HZjabAaCFEAIfnzvrQSsqKrBp0yaMHDkS27dvx/Dhw/Hpp5+isrKy
    Qw+TU/ytp06d8k5hATh/+vTpb1avXm0zmUztD7VmzRp7Tk7ONs6N7FIGDRqEs2fPIjMzMzE5
    OfmhUaNGoaGh4RbQnD59GkuWLLE+88wzWVevXr3gYaHukISEBIjF4kFXrlzxc6eru3pBJpPp
    8K5du8xuZaTVarFkyZJ2Yv/mzZtITU2tBnCCG/D/cWABwKuvvmqvrKwsS0tLw6BBg2Q8Hi/8
    yy+/xJkzZ6DX6/1ZlhWYTCY0NzeLfHx89P/+97/x+eefQ6PRRA4dOlSckZEBo9FYqtFobk+Z
    w263A8DPBw4caOxK6YeGhqJv374IC+u8ZrS0tBTp6ekNAI7f7tV0IT8C+IDLDN5itwB8COAH
    bxrJzMzEwoULh7W2tj5J0zTVSUhPxGJx8b59+64fO3bsjj/m5eWhsrISSUlJEAqFww4fPqzz
    0nAd9vQUa2pqsG7dOrjnCSfVXAbu/whxuVxwOp12LhwExw95eoAhHkkQsYe3BQAx7gwuRVFl
    RqOR3D7eLMsCwJkTJ06wkydPRlJS0h3P8Ouvv0IoFEKr1UIoFOLs2bMdVhPcf//9OHHihJUQ
    ctpoNLanz7tM6kml0temTJkyb+HChQL3BBeLxVi2bBlfoVA8n5aWpm5ubv4HF9d2SoieO3cO
    Wq12zJAhQxSffPIJnnrqKahUKjgcDnz99dfkvffeO3P9+vUPFArFzwKBYIuvr++Aroh7lUrl
    V1dXF8owTGl3bjWAA3v37n3X5XI9N2rUKE1rayvsdjuEQiEEAgEyMjLKjh8//haXEemt0CzL
    0lVVVXA4HLdUUDMMg7q6OjidThqAW1l0K6tWrQKAa3v37sWwYcOwcOFCPPzww4iIiEBFRQVk
    MhnUajWefPJJWCwWhISE4PTp0/j++++xY8cOnDp1ymW320vc3qVn+MJJdlZW1ifPP//8sjVr
    1gjkcjlKSkpQVlaGuro6tLS0oLa2FhUVFdi8eTM0Gg0UCgW0Wi38/f0RHByM5uZmvPzyy7bM
    zMyPAXi7SI/HpcOTO8hU09z5UwD2dxUWzp07F1u3boVMJhvPMIzu+eefx9GjR+HpHTEMU+Lj
    4/OFwWDYW1ZWtoYQEtoBbYDi4mLExMRAqVQGNDc3h/J4PG8ye+kA/gHg+eTkZM1jjz0GsViM
    lpYWfPLJJzh37lwpl03rNa64dYMQCoV3GDqKotzvkgDAvffei59//hlezAdw2UAIhUJoNBro
    dDpoNBoolUqIRCIwDAOHwwGLxYL6+nrU1taipqYG9fX17lC7/fuckvKUo+np6alRUVEPvfvu
    u3j99ddRV1eHoKAgyOVy6HQ6aLXa9p8PPvggjEYjmpqaUFpaCp1Oh9WrV2Pnzp3Yt2/fHi6b
    2a3C8lEoFO8uW7bsib/97W+0VCq9ZaCSkpIwdOhQ5UcffbTkn//8p7ShoWFpZ66vVqt1W3Yb
    RVGYM2cOdDodrl69itzcXKxYsWJ3RUXF8w6Ho5RhmCkjRowYGR8f32VphFKpFJeVlRkIIZgw
    YQIOHTrU1XtiLRbL6rS0NL1er3922LBhEIlEsFgsOHLkCI4ePfpJdHT0lh4sRL6FC1IoFLwr
    V67MdDgcQatXr74jlKUoCk1NTZBKpXfb7faBOTk5eQkJCcjOzvbmFoekUun1qVOnhjY1NSEj
    IwPffvstWJaF0+lsL05lGAZisRharRZjx44Fn89Hfn5+QTfW3U4IeTc1NdVUXFz8Yt++fdW+
    vr7twJVIJOjTpw9efPFFsCwLlmVRX1+PkpIS1NXVoba2FkVFRXX5+flrCSEfdeAtdcb9zENb
    bZhvR4ofwF0AvgLwikeG6g5xl8EQQvh6vR5SqRRVVVXw9/dHRUUFRCJRlkwmW2I0Gs8pFIoE
    lmWHdwgOlsWNGzcwePBgqFQqn4qKCn+apr3CFYA1AAwMwzx78eJFmM1mSCQSiEQiANhIUdSX
    vVn+s2DBAmzZsgVisThy+PDh6tdeew1SqfQWQ8jj8ZCbm4v333+/j0wmU2zYsMGrQm4ejwen
    03lMo9GUvvvuu0GjR4+GSCRqV4hOpxNOpxM8Hq8dy06nExaLBceOHcNbb71V1NDQcJxhGHQS
    4bSYzeZl69atc1ZWVs568803+S6XC01NTairq0NrayvS09Nx6dIlHDt2DHFxcZBIJJDL5ZDJ
    ZBAIBPjqq68cX3311U6WZV+Gx6J7ppNSBrS2ts6aO3fuoy+88ALNDX6HMebSpUvpqqqqRz78
    8MPTcrn8i+bmOxf0v/fee1CpVGBZ9mh6evri2NhYv88++wxSqRRpaWm/VldXv+R0OkvFYvEg
    Pz+/fyxdulTu6+vb6YALhUJIJBIej8fzs9ls6Oz5bk+1+vv7Dx4+fDjEYjFYloVcLsfIkSOR
    kZExrLCwUIgeVrwPGDAAUVFRdFpa2sKEhIR33njjDbFare5wfRpN08jKyopbuXLlZy6X6/EL
    Fy4UREZGorOEhodFLKBpeuvatWv/PmnSJAQEBCA6OhpCoRAikcgNPlitVrAsi8bGRpSWlmLf
    vn0kPz//c4lEcqObMNfscrlWnz9/Xjhr1qw3X331Vc/QoV0p0jQNHo/X/hMA1q1bh127dm3k
    lI+3s3I02tYl+nZznRpt69SKABzv6IJr165BKpVCIBBklZeXN7399tuKBx54AAzDYNu2bcVa
    rfbZ+vr6PH9//5Campp3HA6HtisOTiQSwcfHh2EYxmCz2W4P7TpNEhsMhsFSqRRNTU0QCoVo
    bGyEWq2GVqsdZjQaRZxi81rmzJmDLVu2QKvVDmttbf1ILBbrfv311w6phObmZggEgslGo/Gd
    gICA12mabikvL++SdhCLxWhtbc1zOBzfHzly5OWffvoJVVVVnXr+FEVBIBDAYDC4Ewr/0uv1
    RTab7RZq5zYpt1qtz3zzzTemI0eOLLLZbLBYLO344fP54PP5+OKLL9px5nQ6IZFIIBAIUFZW
    9rXdbn8et+0Q0qHCam1tVfXr1++xp556StSdMhAKhVi4cKHo4MGD865cufIj2hZJ3iF9+/ZF
    cHBw7pEjR/42b968uVqtNiYqKkqXn5+/3mq1lqhUqv56vX7TypUrB0yaNKnr2Ium3QSwD9C2
    Jce+ffu6nShNTU2D9+3bh+DgYIjFYphMJhQXF4Nl2TFoW7ya2xNgWSwW5Obm6gQCwV8ByD//
    /HN0xb1xXsGwysrKBx0OR0FFRff1thzwsnQ6nSU8PFxcVlaG4uJiWK1W2O12OByOdgAIhUIo
    lUqEhYVBoVA0AMjxkpMjAH45deqU7f333xeUlZXBYrHA4XDA5XK1K2A32EQiEQICApCdnW0H
    8GsPlJUQwF/hfV2SH3f9qY6MycGDBzFkyBAolcoTZ8+eXWE2m+fm5+eH1NTUyH18fD4uLS3N
    i4iICCwrK/uYZdm7u0yH22zt/XO5XFIAePLJJ73xgkfffffdg+fPn4+Wlha4DahEIsGGDRuS
    0tLS4nsaEubn5yMkJETN5/PXLl++PD4sLAwOR+f5h8TERN7evXuf2blzZ5HT6fy4u3Kf1ta2
    QKixsfHYuXPnlixcuFAcFRUFX1/fdpqEGwdYrVZYrVY0NDTg4sWL2LRpU1NTU1NmS0tLt1gH
    0OJwOI7QNL3ghRdeYGJiYjq8h81ma8/eXr58GZ988onLbrf/hA62M+osJIwcNmxYTGRkpFcD
    HBkZiREjRvS7cuVKJAeuO+TMmTOIiopyDR8+fKdAINhz9erVPgcOHPBXqVRZfD5fp1Kp3l+5
    cuUIb4pB3ROIz+f72Gw2FBQUePOYDRaLZe+1a9eGUxQVLpFI0NjY6Lx+/foVu91+Gr1Yy0QI
    QW1tLW/AgAGCDRs2QKPRdLn6n2EY7N27F4sXL+brdDrczi11IcbCwkLz5cuXxcHBwYiKimp/
    6W6uwWazgWVZVFdXo6CgAFeuXGnpzHh0Nj41NTVWpVIp6N+/P7glFO1hgacVrK+vR1lZGYxG
    ow092xctHG17IPVERqOtsrrDeP3s2bMYNmyYLTgwBpvYAAAI+UlEQVQ4+AMA/6qtrY03mUzh
    wcHBPzQ2NkrLysrWsCw7xVtMuXHlcDhw7dq1bo0JIaTh6NGjqfv37x/B9Y+iKIrQNH2RYZic
    LjKhnadQ23gpH6fTGXj9+nXU1tZ2iSuaptHY2MhzOByBLpcLHUU5nYSFVWq12qTVasV5eXmo
    qqqCyWRqN4YMw7i9Tuj1ehgMBiiVylYAtSKRCF4awwq5XG729fWVnzt3DpWVlTCbzWBZtv0e
    XMQEvV6PoKAgyGQyFp2snulMYRnCwsJkfD7fqwFmGAbh4eHy7izn1q1b2w0agEujRo26dPLk
    SSgUijmPPPJIyrRp03qkLLgshjtF78mRBAkEguEKhcLAsqyjpaXlJoCy1tbWT+rr61UTJkwI
    P3bsGJKSkpCdnX3OYrFsAsCnKOovCoUikGEYqrGx8abD4chFB8sq7r33XggEAiYrK+sBp9M5
    uLKyUr1r1647OIaOgHX+/Hk4nc67GhoalkkkknRCSMHTTz+NDz/8sKvuhsTGxspiY2NRUlKC
    y5cvw2Qywel0ttfqMAwDqVQKnU6HQYMGITc3V1VSUhKItgJBbyZedHh4uNTlcuH48eNoaGgA
    y7LtCQR3/ZpQKIRKpUJ4eDhCQ0MlOTk5URRFHfeSpwn1IhS8g7JE21q0TglGjmR3cQr6p1de
    eeWn1atXw9fXdz7LsjO95ZA8PEkJAJSUlNyOq2C0LR3yB+AghNwEUFZdXb1ZpVIpHnzwwaC8
    vDzBoEGDqK1bt56vr6//nPMqPWubbnKe/B15/KlTp0IgEFDZ2dkhdru9HwDhxYsXIRQKu8VV
    eXk5CCE6mUwWKxAIynk8XsMrr7wCd4jfiVIMqKqqkmZlZSEkJASDBw9u598EAgHcYZzZbEZ5
    eTlOnjyJmpoaOQA/i8Xym5fvL6Curk7svsegQYMgkUggFovb78GyLMxmMyoqKpCTkwOj0Si8
    LTvZrcJy9ZQo5Er6e/QlLtaXa7XaBx544AHKyyppEELcoQrj5h88xD8sLOy7pUuXjurbty9t
    Nptx/fp1282bN1v5fD49duxYRU5Ojv3ixYvX58yZ02fz5s0PHT9+/F4ej0eFhIRIQ0JC+AKB
    AOfPn3d88MEHaTU1NY/htl0pnE4nysrK+FKpdOHSpUvHBQUFeWbeupSkpCSMHTs2af/+/WNS
    U1NrbDZbl+4hn8+H3W6/Ky8vTyCXyxEVFYX4+HioVKr2bI7T6WwnxEtLS7F//34UFRW5FzT/
    7MV40gASzp8/T/n5+SEuLg6BgYHt96Bpuj08aGxsRFlZmXstGwUggRDyOTpeT9ZRSNjTpQQ0
    ergZ3uHDhzFq1CifvLy8hwghfG+Nrvvdur9zGz/jj7ZtdkbhP+VAdi7RRLMsK8/IyLBXV1df
    NhqNkXa7fTba1uPRHHXhnmtOAGlc4uEWXNntdtA0LWxpaXmvubk5JSYmxufvf/87VCpVlwqL
    x+MhIyMDy5Yte6ixsfE+sVj819bW1p3d0Soul+uuSZMmiZ5++mmUl5ejpqYGNTU1sFgscDqd
    YBgGPj4+UKlUSEhIwPTp0+FyuWRff/31CEJIt7hSq9Wor69PSElJ4T/33HOorKxEdXU1amtr
    2ykHHo8HiUQCtVqNkSNHYtq0aWAYhrd58+ZR3Hh7pbAqTp482fz9999raJrudpMzQgiys7Nb
    0YNF0EBboahYLDYA6HP+/HmUlpZ2ey+KotrT7XTHGs5ot9svFhQUxOfn5xtFIpHEYDDIo6Oj
    VS6Xi0pNTWX37Nmzq6WlZePatWv/Pm3atIkxMTFqQoirrq7Ocvjw4RqXy2VjWVbrcDjOo4tt
    gQkhsNls7RXOXs9AmnZ7Lt0qeG4hbSOfzy/XarU6g8HA12g0kMvlEAgEoGkaTqcTAoEATqcT
    ZrMZMpnMRtN0DQCb24p5oRRuEEKuO51OP5ZlRS0tLe0cpTskZFm2fWE3y7JWp9NpRNsWJAy8
    W+1QzxHQPdmNwYb/bJ/jlZhMJpjNZoPD4Yjy9jtVVVU4d+4cmpub23F12zs1om3d4ADudwna
    apiUACiLxcJev359J4CN165d+zuAiWjb18vFhYVNHA+nRduC6OaO8EQIoVwulxKArKioCNOn
    T0d3hpyiKFgsFtjtdhEAodPpFHWHP0IIxePxnOnp6Q179uxR1tfXU54Ftbe3LxQK4evr66Io
    qolhGIF7bLrCfX19PR+Az6FDh2wZGRmC+vp6dHUPgUAAtVoNiqIc3NjekbDoLJUgpSjqUYZh
    tF56TZTD4agjhHyNHqysDg4OBkVRfrW1tS/QNK3gtgHxysOyWCy0TCY70tDQsGPy5MnYv3+/
    5yVBaFt2UM4By0DTdCAhREMIKQaQyT2nFsA4mqb9CCHVhJAytBX6WTn3/zI62NNowoQJEAqF
    vJ9++uk+s9kc0lPP0t0NkUh02Gw2X16yZAk2bNjQ1bUCtBX0xfD5/HAfH58AkUikYhhGSFEU
    3+Vy2R0Oh9VsNtebzeZyp9N5FcAltO34YPfyeficGx5FUVSoUCj0d9+DpmmBy+Wy2+12K8uy
    jTabrYIQch3AFbQVIHpbLeqHti15A73FFWcEn+2JMezTpw8oigquqKhYZ7fb1d7cS6lUwsfH
    B0ajkScWi3+ora3dOGXKFKSlpd2OqwAPXPlx5zRoW2+XyXlcWrRVj/txePLEVRCX+bwDV5Mm
    TYJQKBQeOnToHZPJNNRLr/UOXInF4jUWi+XIqlWrOgwJuYw9+Hy+mKbpGLvdHu9yufo4HA4D
    n8/XEEKEDoeDZhiG8Hg8m81mM/J4vAqbzXYFwAWtVltkt9vNJpOpoxqs20ULYAh39OFoIzmH
    aQZthcI2bj5WAbiGtq1sfukobP7zv6H83y009w5JL8HtdcTfS6XckUg4oHqrsJzowTYzAOBe
    DyoUCgUURdHeeLLu+iA/Pz9KrVY7CgsLHe5/NPLfEk5h4dChQ3yTycTrbTtisdhmsVhcnSms
    jmT27NmgKIpiWZaprKykq6qqEBAQgPDwcKJQKOw2m4189tlnvweOBNzBw3/+8YiTU1q23xFn
    f8qf8qf8Kf+z8r8ABVbokewgW84AAAAASUVORK5CYII=
  }
  set pieceImageData(Merida1,30) {
    iVBORw0KGgoAAAANSUhEUgAAAWgAAAAeCAYAAAAB6YS9AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw8fRCAiLAAAIABJREFUeNrtfXd4FFX3
    /2dmd2Z7yya76YUklITQkyCIIBEJIuWLIAYVjIhSRFRsyIuovxdQFBUsgAgqCIKKIBYEBOEN
    IL2GEgKk92yyu8luts3c3x87GwOkh9fn+/s9nOeZJ9nZnXvn3jn33HM+pwxwh+7QHbpDd+gO
    3aE7dIfu0B1qPVH/m25m8uTJYFkW+fn5KolEUudyuTy7du36/3Lit2/fDq1WC4/HA7fbTdnt
    digUCkgkEkLTNDiOA8dxVE1NDViWhVwuB03TxOl0QiQSUTabDYQQKBQKuFwuQlEUHnjggdt1
    e0oAyQBcAI4DcPyDUxMh8GXu/+vPeObMmeA4DjzPixmG8bAsi48++uiO1PnnSQqgOwAPgAsA
    3P9g3zKBn+3tuVjczk67A7hP6PgPAOdvx0jy8vIwZ84caubMmXNtNpu/TCZb9eyzz144f/48
    OXDgAKKionqLRCLP1atXz7/11ltYuHDhPzXJUQAmCpP8LYCKjjb41VdfYfv27YiOjh5bXV09
    gRDCUxRF+fn5fVtcXPyr0WjsarFYXiKESACIABzxeDwfd+rUSV1aWjrP6XSGASASiaQiOjp6
    CSGkvB23EQZgkvD/JgAFAMQMw8xPTk6e43K5XCdOnFjI8/xy4TddATwMwCz8vvK2aQoUBUII
    ADwqnFr8Dy4iCYAQgZ8LAThvR6OXL1/GhAkTqEWLFk13OBzw9/f/DkD5a6+9hnfeeQfdu3cP
    oWmaO3fuXOk333yDxx577J8abycAjwj8vAlAeUcbHDduHHbt2oWYmJhONTU1MYQQUBQFrVab
    VVlZmRcQEOBnNpt78zwv8j5uqpTjuLPBwcGsyWTq53a7lQDAsmxdWFjYCZ7n6/bt29fW2wgA
    MEZYL9sBlAGgATwP4BVBMC8EsEr4fTiA0QBqAPwk8PXtpscEObvyn3q4g4KCgi49/fTT5Jln
    niHBwcFZAAbfjoaHDh0KAFCpVJ/GxcWRyMjIa0qlctrw4cOZkJCQBIPBcMFoNJ4JDw+P12g0
    2LBhwz8xXrlarf5u9uzZZMqUKTzLsu8JD71D9OCDDwIA9Hr9myNGjCBffPEFSU1NJQaDYT4A
    aLXa++Lj422ffPIJefrpp4lEItk6aNAgpkuXLiHBwcGX33zzTbJ48WISHh6eHxcXF9u7d++W
    uhQBCAKgFT6zMpls3VNPPUWmTp1KpFLpWoGRAkJCQs6fPn2a7Nmzh2g0mj8ByAGotFrtb3Pm
    zCETJ07kRSJRQwGqB2Ds4LwYAbxJUdRxAMcAvC3c73+bDAA+AZAnHCsBBN6OhgcOHAhCCJRK
    5SqapolMJvtPYGDgfYQQxMTEhCmVyj1qtfrXzp07h4SHh/9T61cO4HsARDiW3Q5+HjFiBAAg
    ODh4PsuyLoZh6liWdYSFhb1AURTCw8OHSCQSE8MwdQzDuCUSySYAVJcuXYxKpfIMwzBOsVjs
    VKlUVxITE6P79u3bGutfC0DRQNlcDoAXxrVKOOcH4FSD8R4QLEQ5gC3COQ+At26yIDUdRBiU
    AEZQFPUjgG0ARgJQt7WRtj4YhmGYma+88krX1atXY9WqVXj11Vc7syw7W9BCOkRxcXEAAI7j
    Cu69916sWbOmU+/evT86derUGw6H46Wnn346bubMmT0tFstijuMenDVrVnxUVJQCAIYNG/bf
    YmiNXq9PmDlzJiZPnkzJ5fLegsl0W4jjONK7d29MnToVvXv3Rm1t7SP+/v6b3W73v0JCQiRP
    PfUUUlNTwbJs8oULFzaVl5evZFk2+JFHHsGkSZOg0WjgcrlIS8KZpulnQ0JC9vn5+W0D0A+A
    VKlUdk9PT0d6ejqUSmV3galqzGbzxdWrV2P9+vWw2+3nANQB0On1+s4zZ87Eww8/TLEs2x0A
    A2Cov7//L0FBQX9QFNURFdAJ4BohxCRoMtdulybbAj0LYJagTYUDmA5gzu2A/8LDw0FRFGia
    LuZ5HnV1dYNMJtOGwMDA9KqqqmdtNtt9NTU1D5SVlf3L6XTGh4eH64cOHUr37NnzvzleNYD4
    Bp973g5+5nkeAODxeBi328243W6p2+2WOJ3OnjExMcM9Hs/dbrdbKZwX8zwfERkZOcrhcNzv
    drv1breb9Xg8rNvtlvI8T/naa0Y4Pw5gl2DR9hCghOQGzy1JGKv9JrjsmsDPfgB6N1BekgQZ
    lihsYDsBjO+ggB5GCIkB0BlAqiD0/6uk9vPzO5iRkUF8dOjQIaLX64820Mw6Bn4qldBoNA+k
    pqbazWYzWbduHVGpVK6IiAhnVlYWKSgoID169CA9e/Z0durUqUQikexUKBRTExMT/QD4zOTb
    SYxUKv1o8ODBJDk52UHT9Iu3o1GfBq3VahfOmzePEELI2bNnycaNG8nGjRvJN998Q/bt20c4
    jiOFhYVky5Yt9d9t376d1NbWkpycHJKQkJAfExMT04IG3SkmJuZ6RkYGeeedd4hYLP4cgEgk
    Es1PTk6uSUpKsotEotcbMHenBsxv9MEAEonk40GDBrl69eplAvAUAEYmk21bvXo12blzJwkK
    CjojaKQdoYUA3vyHtEmloFGRm47Dt2sxSaVSBAYGPkTTtMvXvk6nMyuVSrPvM8Mw7sjIyDI/
    P7+TDMOs0Ov1Q4YOHSqJjo7+r/AzgA8ETdMB4IXb0ejw4cO95ojB8CZFUfVzSVGUh6ZpJ0VR
    7pvmmKNp2inMi0/rJVKpNL9v374t8XMYgEsN2lotaMtvC2NyAXhXELwA0A3AXgHGiGiAS38u
    aM+1AF4U2tjYoN1Tt4Gf3+wIXNdWDLrWbDZn7tu3b+CAAQNAURT27dsHs9l8QRhkh2jixInY
    smULdDqd8sSJE2Tv3r04d+4cpkyZwgBASEgIGIZBt27dMH78eDY+Pj5wx44dqd9++23KlStX
    Hg4LC3trx44dh4cMGYL9+/ffLob2OByOnMLCQo/T6XTwPJ97O1cLRVFUXl4e/vzzT/A8j6Cg
    v616Qgj27t0LiqIQEBBww3VHjx5FSUkJ7HY7GIZpSdvj3G63q6ysDGazGYQQh1d555YdPXr0
    KoC5AH4RmBICFm0TNNkyn4brdDp3ZGRk9AcwH8A+ABQhxFFVVQWpVAqO45zCYmvXVAAYSFHU
    XUK7dwM41OCe/hvECeO8mWzCwu0QjR49Gjt37gRFUTUURTkAMCkpKXjiiSc0//73v5GVleXb
    +cSLFi0y6PV6ww8//NBn586dj50+fXpLUFDQUo1Gk3PPPffgP//5z+0cc54wPofwrG8bEUKg
    1Wqh0Wh8m4uogaC82XpnG/ogPB4PLBYLWtCefbwiugm+4wAsBXBdsIi+Fc4BwBWBj63C2CGM
    fZNgTS4R+B+NtNsR+CdAgMvEgqJT9t/WOEQAHg4PDzddunSJZGVlkYiICBO8jh1RRx+sXq+H
    n59fl8jIyMwxY8aQdevWEafTSTweD7HZbIQQQnieJzt27CDZ2dn1WnxOTg555plniFarzTEa
    jaMIIfU7egdJDGBqUlJS5bFjx8jOnTtJ586drwMY0dGGx44dCwAIDAycI5FIuNjYWDJu3Djy
    0EMPkXHjxjV7jB8/nqSkpBCNRkMUCsX5fv36hfXp06elfeBRlUr1J8uyZ27yGXQBkC+Y+vWM
    JRaLz4lEon0CVuejpQAuNtCqAWCsVCo9q1QqdwN4UCKRtHZeb+aXgQkJCbkffPABWbRoEYmO
    js4FMLCR6xjcpugjiqIAYIogkH1ak12wDjpM4eHhiImJCZbL5fsFbZKkp6eT9PR0IpfL67VJ
    mqbJ//zP/5CdO3eS6upqcujQIZKamkoUCsVfnTp16gcACxYsuF38/BQAU4Px5gDocPiPD4PW
    6XQLH3roIXLx4kWSm5tLcnJyWnXk5uaSjRs3ktDQ0Lz4+PiWNGgawDRBw70MoCG+GSOMaWaD
    c1p4o5H+vImfFwrXN/R1jBXOHYPXwddeijAYDL+PHz/eNWbMGLdWq/1DsEz/a6ShKGpRdHR0
    wfTp0z3l5eWkoqKCzJw5k4uJiSmkafpdALr2Nu4LEVMoFG8999xzxGKxkF27dpE1a9bUC+fm
    yG63kyVLlhCtVpvr7+9/LwC88cYbHTV/X7/77rurjx8/Xt/Pb7/9RuLj4wsBTG2oAbSVZs2a
    hYCAAHTr1k2vUqneDQ0NrVm5ciWprKwk1dXVTR5ms5lcuHCBjB8/nsjl8tNGo/H+OXPmUP37
    92+N1qGE1wn2dgMhF6fRaMqUSuWuBg6XhLi4uPLY2NirgjkJAIFqtfqsUqnMasDQrOCMWdAK
    HJMFMBTA+wzDfC8WizcBeAPAEOG+3nvvvffq5/mdd94hAN4TvrsXwEKxWLyZYZgfBWfQ8I7M
    v09jB/CNYP35BJZN0L7u6chG4NuAtVrtixRF8TKZjKSkpJBOnTqRRiAVAoAoFAoyduxYcuzY
    MVJeXk6mT59OlErliaCgoO5yuRwPPfRQR/n5X4JVdHPfhYLgbvd8PvzwwwCA0NDQQYGBgefT
    0tLIa6+9Rl599VXy8ssvN3m88sor5LXXXiMvvvgiSUxMrNPpdB8NGjRIkZiY2FKXNLzO6ZXC
    uHzUGUCJgCHLGyghxQL+7PPG+gE4KpwLbqCAvgfg32idkzAQwChBuXkGQAoAg0wmA4BX586d
    S86dO0eOHz9OnnvuOQJggVarhQCbDAUwA94Ik4cAhHYE4qAAzB4xYsTcpUuXSrp06QKx2Hvp
    8uXL6ezs7JB58+bN/emnnxwC5tJms7SqqgpJSUlMXl5ez/vuuw+HDh3C1KlTkZqaiilTprR4
    vUwmwwsvvICioqKItWvXLu7atevDX3/9dXvNN61IJPr3qFGjpr377rts586dYbVaIRaLMWLE
    CPj5+YW89NJLHx08eDAQwPvtcWZ9+umnAAB/f39TYmLiglOnTl1euHDhW9u3bw9TqVTNWhrX
    r1/nsrOzd/j7+8/Lz8/PcrvdOHLkSItGiiCIigTG2AHgBACqW7duhGGYxIyMjJ4C/hrVu3dv
    TW1tLZudnR0mmMF39e/fv1txcXFOZmamj3GHCo6UuWg+VloK4JX4+PgXxo4dq42Li4PH48HV
    q1dx+PBh64kTJ45ZLBYDTf9tTSoUCgAYrtFoeiclJSUOGDBAHR0dDbFYjEuXLmHbtm1PZGZm
    fgjgHbQvTnsYgDUNFqyP5PCGoA0SFt2v7WGg6upqPP300+JNmzbdpVAoqEWLFuHBBx/EY489
    huvXrzd6jc1mw/bt23HmzBmsXr0aS5YsQWVlZd9ffvllUXx8/BOXLl2qbic/6wAs8vkNAMBo
    NMLtdqOqqgrwhhh+JGy877VnPr/77juMGjUKP//8c0ZoaGja1q1bl9A0/cDEiRPp3r17N4ml
    19XVYdOmTbhw4UKZWq1e3KVLly+Ki4vt165da6lLXrAECuB17P4C4IwguAm8DsMeAI4IuLNO
    UEDCBIsxCUCCsB58/DwAXufjGwAsLemUGo1mYWJiYs/OnTtLeJ5HXl6e7fz581eKiop+BpAY
    GRmJlStXwmazYeDAgWBZ9l6LxcIYjcaR8fHxXSIiIhRisRjZ2dnuU6dOXbRarW8KOHmb5Wdo
    SEjIxaNHjzapwZ44cYKEhoZmNQDh20QJCQmIiooyRkVFnfn999/JyJEjSUJCAvn+++9v6Ss7
    O5vs2bOn0fu4fv066d27N9FoNP/OycmhRo8e3Z7bmT927FhPcXFxfbuLFy8mGzZsqP988eJF
    ctddd9Xi7zjidlNqaqrXe8Mws4cMGcLv2LGD/Prrr+SXX3654di5cyd5//33ib+//6WYmJiI
    yMhIPPVUm63xx41GI1EoFOuFxZowYsSIsrfeeovQNP1vjUYDAC8tXryYvPTSSxyAyUKo2Lpl
    y5aRxMTEK8JC1qjV6l1Go9EhaA7N0VP9+vWznz59+pbnZTabyZYtW0h8fDx54IEHiM1mI3a7
    nYwaNYrEx8eT7777jlgslluuO3v2LElOTrYDmCpAFW0hNbzx+6SF40B7rcLk5GT0799fx7Ls
    kSlTphCTyUR+/vlnolKpbuknMDCQ9OzZkzR0rvXt25fk5+eTs2fPkpiYGLefn99zPly7HfQv
    AXOub3/RokVkxowZN9+LrYNmPYqKiiCVStGrVy9/mUy2Ni0tjVy8eJFcvnyZXLp06YbjypUr
    5McffyRhYWGlRqNx9IIFC6hBgwa1tcvHBGHtCxPtJmjLBH+Hzj3rc0wKkBbgDa0kALKFDUoO
    YCsAF03TIxoqC43QPcHBwXlffvklKSwsJMeOHSO7d+8mR44cIQcPHiQvvfQSr9FoPFOmTCEr
    V64kK1as8FlD/IwZM8gff/xBDh48SHbt2kWOHj1KCgoKyNdff01CQ0MLW7GWGqVe/fr1K62s
    rOSbEtDV1dVkwIABlQLo3ibq0aOH1wZm2cnjxo1zlJWVkf/85z+ktLSUuN3uW/r65ptvyFNP
    PUU4jmv0XlasWEFkMlluaGhofGBgm0NaQ4KDgy8cPnyYEEJIVVUV2bp1K0lJSSGTJk0iv/76
    K6mtrSWEELJlyxaiUCh+uwnXahexLAsAaZMnT+aaGhchhBw+fJiEh4ef7Ny5s65fv37t6Wro
    8OHD7UOHDq0RzLOE1NTU8oyMDNKpU6fjAIxSqXTNDz/8QFauXEkEcy+6T58+148cOUL69u2b
    LZiET48ZM8Y9ePDgMngTl5p0lLAse/zLL79sFqI6evQoSUpKIvv37yf79+8niYmJpDmFgBBC
    NmzYQCQSyTHBGdMWihccNi0J6EoAvdo6wU888QQYhoG/v/8DNE3XpKSkkBEjRpDg4OBG+5k4
    cSLZuHEjEYvFN5x/++23CSGELFy4kEgkkvMhISHhDZ3IrVWu4M2eIwBIUFAQefTRR8mff/5J
    tm7dSiZMmEB0Ol3Dfn9vAHW1H+wWi8EwTJpUKuUMBgMxGo2NHjqdjiiVypOJiYm6VsAat/gQ
    KIq6V9hYLPCGsnUTIA4i4MgBAD5uML5FuDEK5JogoB+BN/zOJBKJ+ohETbrU5AB+euutt0hm
    ZiYZMWIE0el0RCaTkeDgYPLkk0+Sc+fOkfXr15OePXuSgwcPksOHD5MePXqQzz77jJw6dYqk
    p6eT4OBgIpPJiE6nIyNHjiQXL14k7777LqFp+lcBjqrHcVpDVqvVWmM2m5tUVSwWCywWS00r
    TINbKD8/HxqNJsZgMLw4ffp0icFgwKBBg2A0GuuhlIYm/unTpxETE4OmdrnRo0cjMjIyoqKi
    YlxpaSkGD25THk1E586dQ+PjvaGiZrMZf/zxB/Ly8nD58mXs378fdrs3azMpKQlhYWFd0fFQ
    nBvG1xreJIS0NwSrzOVyWR9//HGlXq+fAyCI53l07doVSUlJ8QCG6XS66KioKERFRUGlUnUG
    MOLuu++OCA0NBc/zBEBsRETErLS0NLHNZqsCUNVMf2F6vb7FxIOkpCRMnDgR33//PbZu3YqJ
    EyciKSmp2Wv69OkDf3//mEZgittJbZ7k33//HaGhoYaampq5PM8r9+7di507d6K4uLjR38fF
    xaGkpAQez42BI1u2bEFxcTHS0tIQFBQUV1VVNaakpATJyclt8lU2xDf9/f1x//33Izw8HLGx
    sUhJSYFgNTXEcG8LP1MUBYfDgfLycpSVlTV6VFdXwxf33IrojUb5Gd7oDLWA5xpv2oiHCo7D
    huO7H0B0g+cbAW/suxRANUVRlc1o0FF6vT5p+PDhWLduHXbu3Inq6mrU1dWhuLgY69atw5NP
    PonevXtj2LBh2Lt3Lw4cOIB77rkHAwYMwLRp0/Dll1+iuLgYdXV1qK6uxq+//oq1a9di+PDh
    MBqNiQ2dia0V0AXXrl37c/Xq1fXC6WYcadWqVXxWVtYetLGGwuDBgzF06FAJz/Nz09LSevqy
    CZuiPXv24PDhw83WnYiIiMC9994LjuNGJSQk+JeXtymTlUGDCIOoqCh8+umnGDlyJKZMmYKl
    S5fWh7wJO3hTYUT/W6mqqKioqm/fvhg9evQgwUSkJRIJ7r//fplIJHrS398/xuPxgGVZ6HS6
    nkql8vHU1FSaYRgQQlgA09PS0hI6deqEkpKScmGBNEUumqbdrYEhxowZg3PnzuHMmTP1TrZW
    CAE3vHGvbbLEBQ2qJboo4JWtpuHDh+PDDz+kq6urZ7hcriEt/b5Tp04YOHAgfvrpp1u+y8rK
    wp9//okuXbpg0KBBtNvtHtevXz91Y2uwOWW2IX+eP38e6enp2LVrFzZt2oTp06cjN/eGJdvR
    0LJ2C/O2Ek3ToGm6oYIwRIAc6Qba7hOCUPZRdwFnZhrMz1R4E1QAoIIQYmlmsxBLpVJGJpOh
    rKzxqLkTJ07g/fffx5gxY5CRkYF9+/Zh/PjxWLZsGU6ePNn4LlNWBrlcDplMxqCdSX+Rcrn8
    5+nTp3MNMcGamhoye/ZsXqlU/gIgsi0NjhkzBhRFITg4+H8GDx5szc/Pb9KcLS4uJitWrCCx
    sbHunj178mVlZc2avz/88AORy+U1/v7+QxrEZLaGYsPCwnJOnTp1Q3uZmZkkJyfnhnM7duwg
    arX6MDoQvXIzxPH444+3BuI4FRsbq2tFOmxjpNDpdH/u37+fnD59moSEhNR17drVM3nyZJKW
    lkY0Go0nMDCQS05OJgMGDCB6vZ4LDg7mpk2bRl544QUSHh7uio+Pd+Tm5pIdO3YQhULxbQsb
    lJ9MJjv+3XfftRiJ4/F4yIQJE8jEiROJx+Np8ffbtm0jCoXiuODNbyuNbWAKN3aUCL9pNS1a
    tAg6nQ49evQYLJVKS1sBoRCj0UgGDx58C7zhO9LT0wkhhHz99ddEIpGYjEZjsl7fpuHGwBsb
    fEO7SUlJJC4urrE+bws/i8VisCybJuC+zc6BXC4/1a9fvzbzs6AgyeFNQiENHOGuBp9dN+Hv
    Hnhrcvg+O9EgzJKiqG0ajYYRnNSNUZBarb7wxx9/kDVr1hCWZRsdk7+/P8nIyCCjRo0iI0aM
    IPv27SN6vb7R37IsS9asWUMOHz5M9Hp9VkOLpy07JWe3268fPHjQY7FYboA2MjIyuNra2lz8
    HRjeKrp48SIiIiIUTqfz6fT0dNX169dx9erVm7EVfPbZZxg3blzNggULfnI6ne8SQuxOp7NF
    XNvf31/pcrl6WyyWthSiuV5QUPDjRx99hIqKv2sixcfHIzLy7/0nOzsbH3/8sdNqtX5D03T1
    P6xJdCR5o85sNhctXrwYR48eRXh4uDQnJ0cklUoRHR2NXr16ifr27Uu/+OKLSEtLQ0xMDB0a
    GkprtVrk5OSgqKiIiYuLkxw5cgSfffYZbDZboVwub/K5y+Xyqrq6um2fffYZycvLa/bGRCIR
    IiMjERUVhWYwQADewlqffPIJb7PZftRqtaZ2zMNvAFag8Qgcp4Bb/taWBjdv3oyEhAQ2Nzd3
    hsPhMCqVSgghV409Z04qlZ5XKBQb/vrrL/vN8IaPTp06BZPJhH79+kGv1/s5HI7eJpOpVZFN
    AuUA+LEhf0kkEpw/fx45OTk+xaDhuL8BcNv4+TZDe41d4xAsonoFpIF2fItFLPzfEDdlG/qQ
    KIoqtFgsboejyUCWcqvVemDjxo0YM2YMpk+f3iivVlVVIT8/HxEREYiOjkZeXp4vYuaWjWzm
    zJkYN24cvv32W5hMpgMAShuaP83b+wwDt9s9RK/XvzNx4sR+Tz75pCgkJKT+++DgYKxdu1b8
    1Vdfzdy8eXNyRUXFK/AGhLfKRHG5XMFhYWHxfn5+WLJkCZYuXVr/fUlJCf71r39xW7du3UdR
    1Ie9evXad/To0UkxMTEyIZ6wSTIajQgNDUVpaWlCVlYWNWnSpNZyAAfgvc2bN0tLSkoeHzBg
    gMpisdRDOyqVCizLYvfu3RWnTp1aAeCrdmJn9fTII4/g+++/BwCR3W5HaWlpo/i6SCSCyWQC
    x3E0AKqd/fKEkJz8/HwcPnwYvXr1wgsvvIAHH3wQMpkMNpsNNE1DJpOBEIJHH30UFEVBo9HA
    ZDJhw4YN+Ouvv7Bp0yacPXsWAK43Z3IL363Zv39/rxkzZoz/4IMPqLCwMOTl5SE3NxdFRUUw
    mUywWCxwOp04ePAgKIrCiy++CKlUCp1Oh4CAAISEhCAsLAyhoaEoKCjA3Llzyd69e78HsMZs
    bnMRMj8AswXTtjFzUgJvKKJEEOKt2gCcTifKy8tD7HZ7UmBgIJYsWYJvv/0Wu3fvvtlauqRW
    q1cZDIbvq6qqBnk8nkeaxGKKilBYWIjIyEiEhITg/PnzPQkhGDJkCNrCz8JYpiQnJ6vmz58P
    Pz8/EEJQUlKC+fPn48qVKxXwxpd/dTsFL03Tt/iRbmBGngdFUYSmadJOwc77YFWapqFWqxEQ
    EACDwQC9Xg+dTgeZTAaJRAJCCFwuF+x2O0wmE0wmE8rKylBZWYna2lqfwM/xYfVNQBgcgE+2
    bNlyl9Fo7DV37lw4nU7s3r0bERERCAoKgkqlgkqlQmxsLC5dugS3241u3brhhRdeQG1tLaxW
    K0pKSpCXl4eRI0fi5ZdfxvLly/HVV1+dE/jN02oB7Xa77w4LC/t80aJFsY888ggYhrlF4+vT
    pw969OhB9e/fv9/rr7/+eV5e3pMAMlpqWy6Xw+12e2ia5iQSCWbNmoUuXbrg+vXr0Ol0mD9/
    vmfLli0r1Wr12yzLVrrdbj3DMI/ed999dHOxwr62g4KC4HK5gh566CHW4/G0JVa51OVyvbRv
    3z61Wq1+LDk5GWq1GjzPw2KxYPfu3Th58uQaeHPs+Y4y8R9//IGQkJAgk8n00Llz5+j09PRm
    BZ7L5Yqqra0dOW7cuG8AkNOnT7e1y18oipo6bdq0IKVSidzcXHz++eeoqKiAzWaD0+mEx+MB
    RVEQi8WQSCRQqVQwGAyIjo7Gvffei4KCApw8efIyvOFqLVEFgOd27txZXVhYODk8PFzqdDqh
    Vqvh7+8PjUYDuVwOjUaDxx9/HIA3JtjhcKCyshIs1PJUAAAT/0lEQVTXrl1DVVVVfSx6Xl5e
    XWZm5np4M8HaWvLUH94Kbi1lv4bCG54WDW+9ihZLzEokElAUJSKEiHv06AGKonDy5En4+fn5
    Uuwhk8l+1mq1rxQXF18eOHCgbP/+/eN5nmeaarOmpgaFhYXo0aMHgoODcfz48ZC4uDimrq6u
    LTWNywC8DECr1Wofu3r1KiwWCyiKglqthp+fH+CNCV+M25BaP2DAAMyZM4eaPHlyTGpqKj1r
    1izQNH2LliwSiXDlyhUsWbJEW1RUFOTxeMxWqxVqdeuLvonFYvA8v4thmGeef/55w4QJExAY
    GAiVSgWxWFxvkfqUGZ/iw/M83G43rFYrioqK8O2332L16tV5HMf9wTBMk/iyz/i32+3PLFu2
    bHlRUVH/uXPnYvbs2aAoClarFWazGbW1tcjOzsbZs2fB8zwSExMxbNgwyOVyyOVyqFQqcBwH
    mqaxcOFCrF+//rjL5ZoDIPNmB0KziqhKpfo/b7zxRqxv4TQ3UZMmTYLT6Yx5/vnn/221Wic2
    VNWbYmiDwVB87dq1E2fPno187rnnsH//fhw4cABhYWH45ZdffgwJCVngdrstKpVKcubMmbnJ
    yclDJk+e3OKDE4lE0Gg0EIlEBp7nFTRNtzWZRBUVFdX15ZdfRlRUFOrq6kBRFBQKBbp27Yoz
    Z850r66ulsAbmtMu+uKLL/DKK69ApVIFWiyWj5544okxTZlMDTWSK1euaBYuXPj+6tWrPW+8
    8cZmlmXJ0aNH29L18cuXL38zbdq0lyMiIuDn54fAwEBotVpotdp6JvJtCL7j6tWrOHToEMxm
    M7Kzs7mioqLVEokkuyW4ybfpAXj+/PnzVWFhYa+tXbsWgYGBN5vYTXsaXS6Ul5fj2WefRWZm
    5nJ4syHbOvcMgNd8jtHWKICC06lcuK5ZZ2RdXR0kEkkJwzAXDx06FHbs2DF07twZEyZMwIIF
    C0DT9IGIiIhnLRZL/mOPPUbt2rXrCZvN9mALChKsVisoioJOpwNN00aRSKQQii21hTRisbjb
    mTNncP369XqN0ul0wmw2QyQSJXAcJ0M7C8v7aODAgUhMTKSnTZv2hFgsfr5Xr15gGManKd8K
    6AYFITw8vNPx48c/CA0NnaHX63P379/faguBZVnwPH+E5/ktFotldlZWFrZu3YrCwkJYrVa4
    XC44nU44nU5QFAWWZSGRSCCRSKDRaBAeHo7u3bv7LL2v+vfvf+7y5cswmVo0mo653e5HN27c
    uDQzM/MhnU6H3NxcVFZWoq6uDh6P54YNaefOnRCLxT5sHgEBAYiKioLVasXx48e3cxz3Erwh
    f2iVgKZpGjzPjx81atSAthQST0tLw969e/tv3LjxYZqmVzRnhsfGxmLz5s1Of3//9z744IPY
    3bt3J5hMJnrSpEnYsmVLYV1d3VKO4ywJCQnsiRMn5vbq1ev5ZcuW3QCxNIflSiQS8Dwvdzqd
    bDswriFWq7XH22+/DZfLVb8Di0QiUBQFjuPuhrdc4eH2MvP27dsxY8YM0UcffTRPqVQ+LJFI
    8PPPP7cKjzMYDIbMzMz33nzzzSuEkJPtgDl2yGSyZz788EO1Xq+H3W5HbW0tHA5HPVP7NlEf
    U6tUKigUCuTl5eGRRx4pAfB7K4VzvQwDsMfhcMwWiUSKnJyc+lAru92Ouro6OJ1OEEIgkUgg
    lUohl8uh0+lgMBigVqth966kP9u5MQ4G8GQbfS8UgHR4q/s1+3qfqKgo7NmzxxYcHPxpRUVF
    D5vNFqRWq3H48GG43e4qg8Gw6NKlS/mffvop9cYbb0yyWCxvezweWUs4a12dd6gC7KSqra1t
    k5dfWMuDx40b133+/PnQaDQ+Bxs4jkNFRQXmz58/cM+ePb3hLVDVLpo4cSI2b96MgICARw0G
    w9K4uDi/q1evIjs7u9l1GhoaCoZhUi9duvSx0Wic9thjj5W2ts/nn38eS5cu5ViW3bpu3bon
    fvvtN1VYWBjCw8MREREBlUoFpVLpy0yFzWaDzWZDbW0tysvLsX//fqxfvx4lJSXlNE1vP378
    OFwuV6vkCyHkOs/zX54+fXqk0WiUxsTE4K677oLBYKhfKwqFAjRN39BvRUUFcnNzceHCBRQX
    F7sJIRsaE87NCmie55UqlWrMpEmTWKm09eVipVIpHn30UfGOHTtG1dTUfAnv2woapQ0bNmD0
    6NH46aefjgUFBY0/ffp0n4CAgORNmzalFhcXr12wYMGpQ4cO0QcOHJjZt2/f11esWCFrS61c
    YceWaDQaCce1yX8pApAqEonYsLAwJCUlISAgABzHobS0FBkZGRCJRH7wvlWm3QKaEIIzZ84w
    IpGo8+DBg6FSqdDaEKq+ffsiNzc3ICsry9CSQ60JKigoKKhYuHChura2FhzHgWXZ+l1eCKmD
    2+2G2+0Gx3FwOp1gGAYMw6CioqKoJQupqX5PnTpVNWXKFAUhpB7a8OGEAkxQL6wdDgfMZjNs
    Nm/RuTNnzpjRvgpsIgHWaE+Egla4di+aqXK3Z88eDB06FKGhob/u3r07zel03nPx4sXEsrKy
    QUqlcn1CQsL+kpISLFq06CGr1brM4/H4t8WBJggFVi6Xs23xP/A8L6IoKtXtdku++OILFBQU
    wG63g6IoqFQqREREAN7aE/exLHuoNQKqMfJ4PJgwYYKU47i0WbNm6adMmdKq2GaKomAymZCe
    nj7s0KFDPYQImFbR4sWLQdM07HZ7TkBAQNmqVatUAwcOBMdx9cqG2+2uF7osy4JhGLAsC5lM
    BpqmsXfvXkydOrW4urq6sLUWXQMlKletVptXrlwZmJKSAp7n6/v1HQ37ZRgGMpkMIpEIR44c
    waOPPmqtrKzMbRKZaOYegsPDwzu3p3h4r169EBEREZ2ZmRkMIKu53+7YscMnSK8aDIarHMdt
    u3jx4rs6na761VdfJRKJZHiXLl3mf/jhh4r23AtN0ywASRvjLHkAn5aVlV1av379sH379qWo
    1Wqapmm+uLjYaTKZfnM6nX8KC7ZDDhSXywWGYUhaWlqrY399DqmLFy8iKyurvZihixDiHDRo
    EO666y74+/vXwxo0Td+A3fmYsba2FpWVldi9ezf27t3rQDtLcup0OvLSSy+he/fuvnctwuVy
    3WAWUhQFkUgElmVBCIHdbsfFixcxc+ZMYjab2zNmI7z1GdpLyfCmuDe7OQivaeIJIQcoijoQ
    EhKirq2t7eTn55ezd+9et1ar7VdeXv6ux+MxtmUjb8jPEomkrQoHTwj5ZNu2bRcADFOr1fdK
    pVKGZVmqurraabPZfqYo6k8Ae9vYbmMOQYrjONHq1avx/ffftzo6gxCC7OxsSiQStdn5LVgI
    dTabzb5lyxZ88803KCgoQG1tLTwezw087LOCxWIxFAoFwsLCfFCPXaPRuMRicWvgjYZU63A4
    bFu3bsWPP/6I/Px81NTUNNovTdMQiURQKBQIDQ0Fy7Kora21N6fENiegdQaDQXNTllHrwC6N
    hg4KCtJmZma2SVspLy9HeXm5G0C50WhEQECAoqSk5NnZs2f7t1BOsznGZtxud1MahxhAPE3T
    yXK5PIjnecputxcLXuESnuePSKXSu6dNm0ZnZGQgODiYGjJkiPjjjz+uJoQcgzfzaCRFUZFy
    udxICHHX1dXlE0L+AnC1KQfi3XffDUKI6uDBg9Pr6upCAXRdv349jhw50mqG9ng8yMzMFNM0
    PY1l2RSZTPZTVVXVwTlz5mD58uWtaaJTYGBgsNFoxMGDB5GTk1PvzeY4rv4+fF54lUqFgIAA
    REdHIywsDH5+fp1sNltISxtwIxSv0+mMOTk52LNnDwoKCmC1WuF2u+sFtG9z8GnyKpUKoaGh
    iImJgUajCYA32eByG/s1oGMZcv6CkG+V9u4bQ2ZmphXAmUGDBiElJUXy3XffPe/xeDq1g499
    7bIcx0ma4GcG3uw532YigrcuRa7w9zjDMHdPmzbNnZmZyQQHB0MkEonWrl1rIYScAKDgOG40
    vPkMBgFzLxCsxOym+Pm+++4DIUR8+PDhWIqidIQQtb+/P8LCwlqtQXMch4KCAkosFsdqNJpS
    lmWLKysrKxYuXIh58+a1uBYAhFMUFVxTU4Pu3btj+PDhCAoKqrfMfCiAw+GAw+Goz/y7du2a
    z5EXZrFYgtB80lWjyBZN04ba2lr06NEDKSkpCA4Ohkwmg1QqvaFfp9OJuro6lJaW4tq1a8jM
    zARN03p4Mwez2yqg+fZ6dIVFRqGdEQ4ZGRkYM2YMOI7r3blz54EjR45s14oSBA3tdrvpJgTf
    hF69en0wdepUY1RUFOV2u3H16lVy5coVR1FRkV2tVrPjxo1TabVaLF++vKKgoID9/PPPNbGx
    sVN///33CS6XiwsPD1d07dpVEhUVBUIILly4wK9Zs+b69evXpwHY3xRD1dXVKSUSSfrUqVO7
    hYaG3uJUaA3NnDmTBjDu559/xsGDBwsAHGzD5f0KCgq0H374ITp16oSYmBjExcVBr9dDLpfX
    O+9cLhfq6upQWVmJoqIiHDp0CNeuXYPJZAqEt05FqwW0YKIPyMnJkWzbtg0JCQkYNmwYgoKC
    4OfnB5lMVh+S5fF44HA4UFVVhdLSUmRlZeGXX35BXl4eC+AuhmF+cLvb9HJm38t320tidOC1
    bmfPnsXly5e7OxyO+9vcsTAnAj+L3G63qAnBNwHeCBUj/q7Q5kvGsMMb86v49ddfqcLCwjKJ
    RML6+fnpKIp6khAyXlivCtxYdpSHN/Tsqab4WfAb6K1W6zq73R7HMIx8xowZePDBB9FUfPfN
    fOFwOJCens7s2rVrSVVVlUcmk73icDjWtGGa+nbv3l3/4YcfgqIolJeXw2Qy1b/UwscrEokE
    CoUCWq0Wffr0QWpqKmw2G8aOHRuYmZnZsx0KR3JsbKxq2bJlkEqlKCsrg8lkQmlp6Q39MgwD
    hULhS2RCSkoKOI5Dfn6+/Pjx48lN+TeaE9BVxcXFlgsXLuhCQkJanSdP0zRKS0tRWFhoRTsD
    3r/77jtUVVVBo9H0CQgI0BJCbkgaaSXuVu/9bg4PlUqlUKlU1NWrVwnDMFRMTAyVnJwsk0ql
    MsD7ZubXXnvNWlRUtKioqMgwb968ubNmzZLMmzdPI7xnDuXl5cjJyQFFUdDr9bRYLHYJGkuz
    u75EIsGAAQPQtWvX9tYhAEVROH/+fHveuBGiUqlIamoqlZKSgsjISKjVakil0hviVgkh8Hg8
    sNvtMJvNuHbtGnbt2oXr16+L7HZ7ZBs3bjEAnUaj4fv3708PGjQIERER0Gg09f3eHAblw6BD
    QkIgkUiQlZVFTCaTwe12s2hbinctOvaOQ2dzpmhz9Prrr2Px4sUIDAzs21rcueHzpSiqHtv0
    PZMmNvO8Bo5NIvylBEtPCnijQi5fvmwBsKi2ttZgMpleFjaexkxlIjhUnbgxGeSWtUYIoTmO
    03Icp+Z5Hm+99RZWrFjRVogDAJQejwcej0fS2vkR+ogpKyuj0tPTkZOTg6qqqhuiKRpCZzRN
    g2EYyOVy6PV6REREoLq6mgHQzRcO2Mr7FgPoWVVVhWeeeeaWfhtCHDf36+fnh4iICF84Xy9h
    U3S1RUAXX7t2LWPSpEmUVCrlWzvRFEXB6XSKCgsLD7UkpJrxXvmAdcmxY8dqRo8ezQulGNuk
    xefk5NAikciCpjMc/zp+/Pj7p0+fDnU6nRcAKBUKRYRKpQpXKpUhDodDVVlZmeNwONbCW2+W
    3bt3b8GRI0cm+fv768VicXVNTU2R1WrNczgc+QDcUqm0j9PpzJDL5Veac/jJ5XJ3fn5+1syZ
    M3mWZTsUS202mymWZSvr6uraooUvLyoqurBkyZK7Vq5cGR8QEGDQaDRatVotZVlWJBaLaUE4
    8w6HgzObzXVms9lcWVlZZrVaL3g8niMAdrfVjwTgjdzc3L2LFi3qr1Qq4/V6vVGpVGpVKpVc
    IpGIxGKxSBAkvMPh4Gpqauw1NTXmqqqqcpvNdkmAlvaj7fU3SuGtq2Fs5zQXwJv+3Wby1X+m
    KMpAealNvLxq1Sr89ttvOHbsGMUwDO9b7I3QEXjfehMKbx0RBbyFgCLgrdimgjftey28WZIM
    vMX60wQIp1r4nCfMlRveSKVDTZngjVjdIISQK1eudISlqdZa7z48GcCq8vLyipKSkrvdbncM
    RVF6nuflFEVJKYqihY0GQkIM73Q6HW6321ZdXV2RnZ19mWGYo1qtdruvxkYr15EHwGdFRUXm
    oqKivsI8K4XNUHSzPOI4DhzH8Q6Hw1FVVWW7evVqAYCzADY35c9pTkDXeTyeF3NycpTtgDoo
    QWOxtefpVFZW+ga13mazHTh58iRpD9wiEokgk8k8YrH4ehMaKsdx3HKO43gfcwnhMGKBuSXC
    OHyS1g1glc1m22iz2WTwhnrZG24ADodDBKDZaAwhAcQEIL20tLTDhZZomoZSqbTV1dW1RRMv
    AvC10+lcX1ZWpiwrK1PDG+GgFMYtwd+1DHzaY7WA0dnQ/oSGEgCbeZ7fYrVapVar1feKe7mg
    Rfg0J5dw2OGtkFgrzHd7+62C980vse3k52toZUbhLTtDaamPn7fK5fKcxrDl5ujUqVM4duyY
    zzx3sCyb38S1HLwp6vxN8KK4AXThm0cfP6+G9918UnjTpu03KTSt4k+Koszwxopr0PFkFwre
    UqGt8sV4PB6cOXPmes+ePd8bPXr08tOnT+s4jtPX1tb6KZVKPSFEarfbRQCgVCp5hmFcVqu1
    QiaTmTweT6VYLK4uLi52v/nmm3j++efbeq8HhQ1MC28ZXgO8b1rRNLRchDXkEHi5DN7Y+mJh
    TXU42e0O3aE7dIfu0B26Q3foDt2hOwQA/xfTIJP4JYcfOwAAAABJRU5ErkJggg==
  }
  set pieceImageData(Merida1,35) {
    iVBORw0KGgoAAAANSUhEUgAAAaQAAAAiCAYAAAAQ0q1EAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw8OLpAC3gAAIABJREFUeNrtXXd8FFXX
    fmZ2Z/tutqQt2TRSIZAIBGmCRJog3ZdmoykiVQFBpSgIqKCAICBYgwURBAtFg4B0ProCIQkh
    hBRSNmV3s73d74+dXZcIpOL7vt/n8/vND7KzM/fuzD33nPOcc+4F/sE/+Af/4B/8g3/QMLz0
    0kuYMWPGPw/iH/yDJmD+/PlYtmzZPw/iH/zHgfpv7HRSUhJmz56NCRMm+D4bNmwYjEYj5HI5
    tm/f/v/mBbZr1w5VVVVwu90AAC6XC7VajRMnTqB9+/aorKz0nQMAmqZx8+ZNDBw4EDk5ObBY
    LL5zIpEIKSkp+Pbbb+9nlwUApAB0ABz/AeNfDMAEgPx/GTNhYWHgcDho2bIlunTpgrfeest3
    bsyYMTAYDGjVqhXefffd/zdyNHz4cJ8sEEJA0zSkUim2bduGkSNHwmg0wu12g6I8UyZFUdi7
    dy/Gjh2LqqoqOJ1O37UMw0CtVmPz5s33s8scAML/a2OX2wz3UAIIAKAHUHU/O9umTZtuBoMh
    nMvl/jphwoSK5ORk/PHHH+jduzcee+wx6pdffhHs2rXLkpGRgb59+/67n20QgEAAxQAM9+0F
    crkamqYT7HY7BQB8Pt/J4XB+B1DN5XJFXC63g8Vi4bvdbvD5fDAMcx3ADZqmKS6XmwIg0OFw
    gMvlgsvl6iiKugjA2cRuqQCoAZQAqPT7PJGm6dcFAkFri8VynBDyJvsdL9TstQX385n5IRzA
    6wAWs23+v0BoaGii0WgURUVFXdq0aZPjsccew549ezBu3Dh8/vnnIIQgISHhP6W7wawsFbFz
    zP1S0rTL5eL4LBWKApfLdQIgarUaTqeTIYR4lQ5F07QTgDswMBBSqZTrcrloiqIIe84tEomc
    zdAtMSsPVQCMtcbtXADJAE4AeA9Ahd/5APbQArD8De9ICeBfAHbW6sffCgbAOIZhTgQGBt7k
    8XgnAUwAwLtvIzM4eKpAILCKxeLfVCrV8F69evESExMxbtw4Wq1Wj5PL5duSkpJaA8DgwYMx
    atSof9ez6SoWi49FRETc4vP53wOIuV8NRUVFTVapVOY2bdqYW7VqZVYoFNqkpKSHO3bsiOjo
    6HiJRHI9Pj7enJycbA4MDDTz+fx5Dz74IAYMGMAolcrvNRqNOSUlxRwZGWlWqVS/DR06VNFU
    p00oFB4MDw8vFQqFBwE84GfRfThmzBiye/du0rdvXwJgvt91fSQSyTmNRlPKMMxOAFH32TMK
    pSjqMQDZAAawypD+dzktACYDeAvAC+yEcz8V0hSBQHBLLpd/2LJlyxQA6Nq1KwAgMjKyp1qt
    fr1Lly7BSUlJ/25l1I2dcEsA/AQg7n411KpVqwFBQUHfK5XK75VK5fchISFbU1NT28ycORMd
    OnSICA4O/sx7TqVS/RAUFDRy2rRp6N27N6PRaBYplcof2PM/REREvPOvf/1L3MQuxQHYDuA6
    gO8BtPKSHABWsl4RAeAGMM/vuk4A9rPXpQPQ3G+WgaKoBwBcApD672TdRmo0muoPPviAnDx5
    kqxfv56Eh4frATxxvxpUKBRj4uLinFOnTiVRUVF6iUSyIiUlRREWFta9RYsWJd26dSNKpXJf
    ampqIACIxWKEhIQgNjYWmzZt+tsUNU3TW+fOnUsyMzPJqFGjCIA37psbFhQ0vUePHiQ3N5ec
    OXOGJCYmWqKiop5s27ZtRHh4eO/Q0NDyffv2kcLCQjJ06FDC4XDeS0lJiUhJSYmXSCSHlyxZ
    QoqKisjq1auJXC7/n4EDByrr2TQNoD2AEQCS/D77cMKECeTKlStk/PjxBMAG9nMeRVG7V6xY
    QRwOB5k5cyYBsJEdwEIOh7N7/vz55PLly2TAgAEEwCt3aKd1cw0lAOkURV0BYAZwBcBXrEf7
    dyMOwEEALnaCcQE4DCDxPo6ZpymKclMURfh8fm5ISMjE0aNHMzExMa2EQuElDofjVigUmwcO
    HCiUy+VQq9WIj4/H8OHD/26Dd6vfxEsALLlfjWk0mpkcDsfXFo/Hs7Zt2zYNANq2bduGYZgK
    /75wOJyFADBz5kxGKpXu9z8nk8nODhkyRNmA5mMB9PYzwigA79b67etZWWAA/FDr3GfsNXwA
    39Y6N+8e7TQVEgCvUxS1Bx4Kfi+A5ax8NQqNtQj5FEWNmTBhQsDUqVPRuXNnTJkyBc8++6yM
    oqgnWW6z2cHn8wtcLpfphRdeQHp6uiw5OXl2UVHRB2az+dURI0aEfvHFF2jTpk2fK1eubBIK
    hXNpmh5E03Rkp06d6IULF2L27Nl/hyBxKIqSBQYGokWLFggICPC60PcFhBAIBAKEh4cjPDwc
    IpGIbzabV1ZUVByy2WyficViZVhYGDQaDQIDAyGTycaXl5cfKi8v/5lhmI6hoaEICwuDSqUC
    RVEN4aKHhoaG/tCtW7evVSrVTgAPswIgkMlkYKkMsGOBAmAnhOzcuHFj1fDhw8m3335bBOBH
    9hoagEAulyMwMBAikQjwxJpua0epVO5i22kqDACWscct1jNZwgrV3wma9YjS/GSRBtADwDTW
    q2z+mZ5hijgcjpkQApvNFlNZWbnm4MGDr+r1+lkWi6WNy+WiDAbDUydOnFhE0/RjAFrx+XzR
    zp07kZiYeL9jjP7PRlrrM9l9bO+2se9yueiKioqxYWFhCysqKl5wu92CWnNRz7CwsIU7duxY
    6HK5omvdqyF0XS9WDn4EsAtAF/bzwDvQ4DQ8cdef/Si8KvZvwo4X5R2uA4BHWEXmbadzMzwz
    J4BcQkgOACuAPAA5AOx/t1Un4XA4v3366afEH+np6YTD4Ry/HxMwIQQpKSnBCoXi4tatW4nR
    aCSvvvoqYRiGyOVy99GjRwkhhKxbt46EhISQXr16kZiYGKtAIMgSCARrY2NjUwkhVGJi4t/x
    fEaEhIRoO3ToQAICAnL8BlmzIzAwcHq/fv2IzWYjTqeTXLhwgRw7dsx3nDlzhlgsFkIIIbm5
    ubedO378OCkrKyOEEPLFF18QhUJxqp4eEg/AjsWLFxODwUCmTp3q9YQA4BG5XH4lJSXFxv72
    tFrXTWbphKG1DKIngoKCbiUnJ1vEYvFJAG3hiXHueuONN4jBYCBTpkzxWorNRQu0AnC2GT2v
    xliYv9WyaL3HfZEjAEhJSYkQCATZ/u0FBQXZpVKptZalTxISEqxyubyIYZjdcrn8+QceeEAN
    AN26dfs7ns/jAMrY/mSzdNT98pBm0DRN7vIuGnTIZLJT9fSQuAC+rHW9lzXoz07wDgCFAIb4
    XScEMJM9P5b1mrye1QR4YkcOAJns3MNhGQD/djY0oxwlArjIymyT0NikBrPL5bqwd+/eHsOG
    DaPkcjn0ej327NlDXC7XRdwegGsec4mmER8fL9fpdJwffvgBSqUSmZmZmD59OoqKiqjY2FgA
    QOvWrREVFYVNmzbB6XTyMzIyEr766quEzMzMYS1atNigVqs/jIuLq37++ecxZ86c+zW+y0wm
    k9nhcMBkMlXjPgb6OBwOqqurcezYMfB4PNA07csEAgC73Y6zZ8/6Mof8zwFATk4Orl27huzs
    bBBCqNrn7wI3AENhYSGysrJQWloK/BlwPqjT6QbrdLo3WArhpN91dvZ7CgDV7H282K/Vaq9r
    tdrTAFbDE8TmAtAVFRXh6tWrKCkp8bbTHFlF0awwywGMYyeCG3+zQnLcwyvT3w9LMzY2FhwO
    x0ZRlJ71ljBmzBiMHDmSefXVV3Hp0qXbvvvpp5/yS0pKwr7//vuwvXv3PpqbmzsuLi5uZYsW
    LX566KGHHMeOHbufz8c/KG9gx8x9ZRpYr75RoCgKFosFhBB4EyDq4ZU57jAmwHo9wwC8yiqU
    DL/vWADksx7kTb9rCDzxpidZo28VgCz2ekcTvLi6jNPWrCwnsR6S7d9h3XVmGCZ38+bNhBBC
    Pv74YxfDMHkAujd3Qy6XC/Hx8XKZTLa9U6dOpEePHuTq1avEZDIRl8tFdDodcbvdhBBC8vPz
    yTvvvEP0er3PcyspKSFLly4loaGhTolE8lVqamoLiURS30HTUPSQSqWX3nzzTXLz5k0yadIk
    wjDMfgDx96OxiIiI5/h8PlGpVKRjx46ka9eupEuXLvU+unXrRmJiYohAICABAQGHhw0bVl+r
    vAOPx9uvVCoLOBzOeQBtap1fyVq3KbU+X8paopNrff4YPCms02o/T4ZhMlUqVT6Xy92J5gls
    R0okkkOjR48mb775Jhk4cCDh8/mHAET+G+RoBDvR+luvOnZSafZJt1u3bgKlUrmKpmkHAKJQ
    KMj69evJzJkziUwmu83SDwwMJEuXLiVXr14lNpuNnDt3jjz11FNEKpXqFArFGyNGjJC0bNny
    fj2XhwFcrvVcDgC4L+l/wcHBM9u1a0eOHj1Krl692qjjypUrZNasWSQgIODM4MGD6xtD6grg
    FEu9XQXwYK3zy+FJ6qjtfbzGKoEptT7vyRozL93h82uscXwYfyYbNQVimqbfio+P16alpTkj
    IyMr4Yl9Sf9OAeIDGM/lco9ERUXVbN26lRBCyLfffkuio6ONXC73GIBn0YxxJEIIpFLpk0lJ
    SbY//viDZGZmkrNnz5KzZ8/6FFFdcLlc5KeffiKJiYlEJBJtj4+PD0lOTm7O58IFMFKhUFx7
    ++23idVqJYQQUlVVRaZNm0YEAsEJNjbQrGjVqlULmUy2SSgUWiZPnkyuXr1KysrK6nXcunWL
    bN26lcTFxREej3cxLCys/3PPPdeQuKKcFaBDAIbXshZXCQQCQlHUq/7WlFAo3NG2bVvC5XLX
    +lEGHJqmNwsEAgLgxVptTAfwHUuvSerLwLD9mcNamOPgyQDyCsr0gQMHEoPBQAghRKvVerP+
    prPnpQA6svTHAvYeo+6DwmoBYBpFUfm1Jt4ClpIJvw8T76NcLlcPgPD5fBIdHU2kUuk9KajI
    yEgyZ84ccv36dWI0GsmqVatISEiITS6Xr2jXrp24X79+zS1Ho9jJ8079OYnmiSPehqioqL4K
    haJw4sSJZMeOHWTnzp1k+/bt9Tp27NhBdu3aRdasWUMSEhIcSqVy/ejRowUNaF7NUtsHAAys
    de4d9nfP9SdGAHzBfr7OT45oeFLACYCXa93nadZ76gxPGn29iCl4MkDbs3IeB0Dsx6I83qFD
    B8vZs2dJXl4eOXz4MElOTrYCGO33HTFrjHdm7xGJ5ik38j2I2ZGRkfr333+fXL16ldjtdkII
    IQ6Hg2RlZZH169eT2NhYEyvEzdLwhg0bKJlM9sncuXOJ0+kkn3zyCYmNjSVz5swhTqeTNAS/
    /PILCQsLc8tksve7d+/Ob6biNRGAVyIjIys/++wz3zPxwmAwkKVLlxKVSpXPDoxmC1a3bNkS
    PXr0EAUGBk6TSqXFY8aMIR9++CH58MMPycaNG+96bNq0iSxdupTExsY6xGLx9piYmFYAMHHi
    xIZ2QQDgV3aiCPXNKlzu+2lpaUSpVB7Hn4HW4JCQkPOzZ88mUqn0Z/a5AUCsRqPJ7dq1KwEw
    qxY3nQvgkwZMZk/zeLwLbdu2tQ4fPpyMHj2a9OnTxxUTE6Pl8XgZrEX59QsvvHDbO5oxYwaB
    J6trGp/P3x8bG6vt27eva8yYMeTxxx8nKSkpdh6Pd4lVbkwzvLr2rKXqvMvE64Qn5fnB5hor
    Bw4coJRK5QdeJfPRRx+R/fv3k/Dw8HrFRh544AGSkZFBXC4X2bRpEwkICLAqlcpZ06dPpw8c
    ONBccvQq6y342mWTbfyPmyzd2mwT26hRo2iNRtNTKBSe4PF4RK1Wk7S0NNKrV697Hr179ybJ
    yclEIBAQPp9foVAo5qekpMgDAxucsMlnabkjtRSGN8X7KP7MXlOw8kbYa7xyFO7nVforsEgA
    vwP4KiUlpSFxofdFItGV8PDwypYtW1YHBwffZBjmV3gyYJMBvPvCCy+Q48ePky5dupCMjAzy
    0ksvEYqi1rPn59E0naFUKgvCw8N1arW6WiAQ5ADYzJ5vMh5QKBSFO3bsuKcW+OGHH4hKpSph
    rcwmY+jQoQKpVLp348aNZO/evUQulxM+n08yMjLu2H5NTY0vkF8bbrebrFy5kgiFwpqIiIiR
    zdA9BsDrSUlJFv/+FBUVkU8++YTU1NT4FHZ6ejpp0aJFJauUmg0pKSkoKSmhNBpNHw6Hcy0p
    KYkMGzaMDB069K5H3759iVgsdgYEBLzbsWNHRVBQEMaOHduY5mkAn3K5XBc7UCnWQ1o3b948
    0r17dyM8dT4AkJKUlFS6a9cuEhYWlu3nATw/fPhw57PPPutv2fEoilrPMAxhvZQ6KXwAE0NC
    Qqrfe+89UlRUROx2O3E6ncRkMpHs7Gzy8ccfk0ceecTF4XAsXbt2JZWVlYQQQnQ6HXn44YcJ
    l8u19urVy/Xpp5+Sa9euEZPJRJxOJ7Hb7aS4uJisWbOGhIaG6gBMRNMCwkGsEq+PIjjir+ib
    gqlTpzISieR7hUJBdu3aRQghZP369cQ/3dn/oGn6L8ogPj6enD9/ntjtdjJz5kzC4/FKYmNj
    m8Pz58FToGzxb69ly5Zk4cKFRKlU1u5fFWscNEtgvnfv3gCA1q1bR8lksk9DQkIs27dvJ2az
    mZhMprseRUVFpH///kQgEFwOCwsbOn36dE7r1q0bHA4QiUQUgI9ZQ2SWR4QosHEgAqAGQD/2
    s0R4Eh0IG7OJYG/zJBu/8U/35sCTTeoGsNRbb1YHUhmGOTt48GDy/fffk+zsbJKTk0POnDlD
    0tPTyVNPPUVCQ0OLABQMHjyY7N+/n0RGRpLdu3eTMWPGEIqiSlUqVfGwYcPIhx9+SI4fP04y
    MzPJlStXyHfffUcGDx5MeDzeH2iGZK8pvXr1cplMpnt6IRaLxf3oo48678BjNhjR0dEIDw9P
    UqvVhQcPHiTLli0j48ePJxs2bCAVFRV3pOYWLFhA0tPT79q/iooKkpaWRoRC4cnU1NTQuXPn
    NonrDgkJqdi3b99tbRw8eJC0bduW5Ofn36YM169fT0Qi0WU0c+Hno48+ij59+tAMw+xcvnw5
    sdlsxGq13vGw2Wzk6tWrJCEhwRQWFtajVatWTWqboqjXH3zwQRIaGnrDj5ve8O6775IlS5YQ
    iqI+ZCeOYX369LFdu3aNtGvXrpodkAIej7fn448/9tYmeV/Go9HR0VXt27d3sRNPXUjm8/k3
    Vq1adU8aV6vVkldeeYWoVCry008/EUII2b17NwkMDCSvvvoq0Wq1d73W7XaTtWvXEoFAkIem
    ZRQ97jdx1HU40Ay1fe3atUNKSkokwzDZffr0IefPnydffvkl6dOnz12V0fTp08m//vWvv5wb
    NWoUMZvNJD8/n6SkpBCJRPJj586dZVu3bm1KF3vCs6rHbW317t2bZGdnk/j4+Dv1MxNAswWx
    ZsyYAbVajU6dOgVwudxf+/btS5YuXUrefPPNOx5Lly4lL730EgkMDLRoNJphANCjR+N0M6vA
    FrK/K9cvJrvG7/eup2maAvAoPPVz3nhjN9Yw9q9B8lLl3dgYFOFyuS/UI8s4mKKoA08//TQp
    Ly8np0+fJkuWLCHjx48n06dPJ1u2bCE3btwgR44cIb169SIhISHk119/JZs2bSKHDh0iYWFh
    pHPnzuTnn38m165dI+np6WT69Olk/Pjx5M033yRnzpwhFRUVZOLEiYSm6cNNNbbmjRgxwu50
    Ou8ZuHG73eSZZ55x19OyvSvGjh2Ltm3bMnw+f83o0aPdRqOR6HQ6Yjab79q2zWYjgwYNIuvW
    rbun0vz666+JSCRyKJXKSU1Mbnj7ySefJA6Hw6fsFixYQPr3708UCgUZOXIkWblypa/PlZWV
    pFu3bg4ATzU3Dx4dHc1lGGbnihUr6qQu8/LySGJiorFFixbNwcdPePzxx12TJ08mFEV9zsYP
    P3zvvffI6dOnSWhoaC48WW3zJk2aRMxmMxk0aJCTfQYPJiYmVl6/ft1Lm80FIGcYJuO1114j
    vXr1MsFTq1EXFrVt25bcunWrzt9uMpnIhAkTyOjRo0lVVRUZNWoUGT9+PKnL0CKEkNLSUvLA
    Aw+QJo7teWhYKvGiprycOXPmYNCgQVRAQMBcAE6VSkUSExMJl8u9a5sMw5Aff/yRLF68+C/n
    pFIp2b17NyGEkLVr1xIej2dSKpXDmjiG3qkdt1q3bh359ddfiV6vJz/++CN5++23a3tKTpa6
    azZ89NFHCAwM5HC53O/q+34YhjG2bNmyx7hx4xrdrkKhAEVR4/FngfRmlsZb69eW1xuaXusZ
    PM0qsFt+n7/Gxm92sH9b2JqyujA+PDzc8ccff5DvvvuOqNXq234rl8slXbp0IT///DO5ceMG
    6d69O5k1axYxGo1k3rx5pEOHDiQzM5Ps27ePdO7c+S9jLCwsjOzcuZNkZ2eTmJgYJ4BJtemW
    hiDvxo0bjqqqey9ZV11d7c7NzbXBk3rYaKSnpyM/P3+gRqMZO2vWLEosFiMgIABC4d3zJTIz
    M1FcXIwOHTrU6aInJSVxDQbDE4mJiY2tLGYAxLdr1w5crofO5nA4CAwMhFwuB5fLRXBwMFQq
    FWja86iVSiWSkpK4uI+V+PVRsM2cYVhYWlpqffrpp9GqVavhXorO7Xajbdu2SE1NjQLQn6Ko
    xLi4OAiFQsTExHDgSRMd3KNHD2VkZKR3EVgawBOdO3fuOWTIEJSVlRkAlNeDrouLjo6GSqWq
    Dz2CmTNnIi8vD9u2bcPNmzcxc+ZMb0HuPaFUKhEdHe2lTRpLFxnu8/dvw3fffYczZ850NplM
    0wBwKisrkZWV5VsQ9E6IiIiARqPB+fPn/3KupqYGX331Fex2O4YMGYKYmBiR0Wh8pn379uIm
    yNFt2ZN2ux0lJSWoqKiAw+FASUkJSktLa/eZg2bOXn3uuefAMAzVkHfL0mjU559/3uh2q6ur
    wdJwVvajkQBqL8gZDaAf/lxGyPsMkuDJUg2tNbcP86PLTRwOp7QeXekSHx/P1Wg0SE9P95Za
    +OB0OnHy5ElMmjQJhYWFWLBgAY4ePYpTp07hyJEjWLBgAcrLyzFp0iScOnXqL2OsuLgY6enp
    UKvVaNOmDQeeLEOqsQrp6KVLly6sWbOGMplMd/yC2WzGmjVrcO7cuXPwFP01Cr169UJSUlI0
    RVELJk+eLE9NTa3zmvLycixbtgzx8fF44IF7ZzUGBQVhwIABcLvdHcxmc8epU6c2mrGy2f5M
    u5fL5Zg5cyamTJmCyMhIvPLKKxg/fjz4fD68k7Td/rcXMt9vlN26dasmODgYzz//vJTH480G
    EEFRFHg8Hh577DEOwzBPC4XC1NDQUOTk5CAiIgI0TT8ilUqHDRw4EBwOxyvYyTKZbNqLL77I
    cDgclJeX16eOiwCodjqd9Va0rVu3RmpqKlasWIH27dujvuu2ud1ur5Dp0Ph6qDPwpMTXB1rc
    XsvVIMyaNQtqtVplNBoXOp3O8PpOsOPGjUNpaSmOHDlyx+/89ttvyM7ORnh4OPr27Qun09mj
    srKy7ffff98sA6qkpATLly/Hxo0bUVpairfffhtr1qyBwWC407v/PwGKosr8jI8AALPhyXLz
    gsuyCrUnw+7wlA/4K9E4eDJWvdZ7tcvl0tbHnhWJRCCEwGi8ezlpQUEBFi1ahISEBLRs2RJv
    vfUWNBoNkpOTsWjRIhQWFt71WqPRCEIIxGKxVwc1WiGV2my2V9asWXNuxowZ7rKy22VKq9Xi
    pZdeIqtXr75os9lewe0rOTcIYWFhPK1WOystLa39c889h3sVbJrNZvz666+YMGECfvzxR6LR
    aO7pRflI6549IZFIJDqd7pHOnRu1koYTwLXTp0/DarX+ZcJbvnz5Xyz28vJy/P77766meo//
    YaioqqqqLioqwrhx49CjR48HATx89uxZjB07FhcuXIBUKu0IoNWmTZswePBg/PTTT6BpukNg
    YGCr3377Da+99hqKi4sBYPDQoUMTBw8ejJs3b0Kv15fBE9CtC5fy8vIIW6RbJ7hcLnr27Imy
    sjKkpaX5PNy6UFlZiRs3bhAAfzThef0OYBPqLiC0wVOwe76xDcXFxdGlpaXPWyyWPg257uDB
    g5g3b57Xcv+rBVJWhqNHj4KiKPTu3RsCgUCp1+sfGTJkSKPl6E4nsrOzsWzZMmi1d5xLXWy8
    5f8E3G63FrfvmNANnrXn/NEVf60h6niHzwYDaOen7CoEAkF9VkvPKSkpASEEaWlpPmbnTjhx
    4gTOnz+Pvn374siRI0hLS8O5c+dw8uTd7SeaptGzZ09QFIWioiLAswKHu7EKCQAyzWbz3m++
    +caWk5Nz24nc3Fxs27bNYTQaf4GnyKvR+PnnnzsKhcInZs+e7XPXa8Nut2PPnj145pln8PTT
    T5eeO3fus4CAgNP++wPdC61atUJUVBTsdnvH9PT0xtANBMCuw4cPl3733Xe3WedKpRJ9+vSB
    QPBnOYLNZsPmzZtx5cqV3+FZUPN+WVp/tywZampqSjdu3Ijz588jLS2NQ9O06KeffgIhBFqt
    FhKJhKNUKjlRUVGYP38+CCGQSCQcq9VK5eTk4Ny5c9i3bx8CAgLEjz32GHX+/Hl88803sNls
    xfAEcOvC4dzc3NyPP/643h5oVFQUwsLCUN/iTrvdjo8//hg5OTnX4Km9aiwcAFbAsyKF7R7K
    aA37vUbvG7V69eqk0tLS510uV700LofDsQcFBZ28cuVKtv+qDXfyFI8dOwaXy4WUlBRoNBpY
    rdYuQ4YMETRWju5kwJaWlmLr1q13s9YvNvE9/KehBkBpLY+o9vp9XPy17IDBX0tJ5LXm91tu
    t9tUjz6cvHr1auX+/fsxZcoUjBw58q7zid1ux4ULFxAVFQW5XI74+HicPXsWDofjrspo1KhR
    eOGFF3D06FH88ccf1fCks9/24xqCrjRNL0xNTe0xZswYQe3C0jZt2uDNN9/kbd269aXTp093
    cLlcS+BZk6vBMBqNad27d1fyeDw899xzWLRoEULg2xgIAAAcZklEQVRD/6RIKyoq8Pbbb+OL
    L74osVgsW0Ui0deJiYk3Tp48+V1wcPA9Nbu/0oiPj8fly5djCgoKQhvptZzW6XSL582btzAv
    L69Fr169UFNTA4PBAKfTCR6P5w1Y4ptvvsG2bdsy7Xb7K/DUUjQbJk+ejF9//dVbt1Hn9/2e
    Dxk4cCB2797dlObNbrf71oEDB3Dt2jUQQtClSxeMGTMG48ePB0VRKCwsBEVRiIqKAsMwGDhw
    IHJzc8EwDOLi4lBTU4OVK1di3759WLZsGVwuF27cuOHl1d316MM1p9P5zvvvv/+eWCwOmDFj
    hi8mZLFYoNfrYTAYoNfrUVNTA7vdjqysLFitVhw9ehTl5eXg8/mQyWSQy+WQyWSQyWQ+qtVs
    NmPt2rV49913dXa7/a0mWuYieDLtHr6HDHLZ81fhCUybGtOQVqvtbTabIwICAjBt2jScOXMG
    GRkZd1JEVpFI9JtYLP4yKSnp6NmzZzejjlURsrOzUV1djeDgYMTFxeH69esJeXl5wWjc3lJn
    4FkVfxGAsMDAQIwYMQJqtRpCoRAmkwk3btzA9u3bYTabAc8K7c0uR15l2yBtyhqia9eubeqO
    1hZ4EhP+4s0LBAKIRCKIRCIwDAMulwsOhwOn0wmXywW73Q6TyQSLxQKr1QqXy1Vb3otmzZrl
    XL58eV19OGsymbYsXrx4ZkhICL1q1So4nU5899134HK5kEqlEIvF4PP54PP5EAqFkMlkUCgU
    EIlEkEqlSEpKgtVqhc1mg8lkgtFohNPpxOjRo7FmzRpkZWVh/vz5bp1O9wWA/2msQnosICBg
    zeTJk2OnT5+OsLCwv3xBKpVi+vTpePzxxwXr16/vu2HDhlidTjcTQINnO4FAoAgLC8O1a9fQ
    vn179OjRA4QQUBQFrVaLWbNmYefOnQfFYvGitWvXnpw9e7a7sLCwt0gkavfQQw/Vqw2GYRAb
    Gwu3263UarUhjVRILgCbi4uLr7z++uur09PTO6hUKvB4PPD5fFgsFjgcDty6dQvFxcV74Mki
    y2xuIdq2bRuCg4ND3W532J49e6DT6e4aT6EoCtXV1SgtLWU4HE7sTz/9dKRnz544fPhwo2UY
    wJHQ0NARq1evZsLCwiCTySAQCFBZWQmTyQSr1Qqr1YqSkhK4XC5wuVzw+Xyf6y6VSrFw4UJM
    njwZOp0OOTk5mDNnTo3ZbK6vQUMAbKmpqWGWLFnyRn5+fkiPHj1w5swZ5ObmoqKiAv4xJg6H
    Aw6Hg4iICGzbtg0ul8u3IyiHwwHDMFCpVIiOjkb79u1x7NgxbNmypdRisbyOPxeqbAyC2Il3
    PO69mgkHnur2FHjS419H/eNOPvD5/AgAeOKJJzBq1Cjs2bPnL9/h8XjXpFLp8vj4+J0nT540
    KBSKdhaLpc4KSi9zERgYiPj4eOzduze4rKwstJEKyQXgI1bRrOnUqVPqyy97StK848VkMuGP
    P/7AhQsXdrNydLW55ahnz55wOBxURUUF06ZNG9RVU2Q0GnH8+HHa5XJx6jHZ1ylHNE2fIoSM
    iYuL43Tu3BlJSUlo2bKlb+cAqVQKPp8PLpcLmqZ9MU2r1QqDwYDq6moUFRXh+vXruHTpEk6f
    Po2bN2/aCCGn7pSgcifHB8BbWVlZ8nHjxj29ZMkS7ooVK9CzZ09oNBrvjgLgcDigaRoikQjX
    rl0DRVGwWq2YNGkSnnnmGbhcLjgcDhiNRhQUFKC4uBiDBw/GgQMHMH/+fFdeXt5X8NRH2Rqj
    kDpKJJJVixYtip0+fToY5t6F6i1atMAbb7yBkJCQlgsXLlxlMBi0tTVhfSzerKwsMmfOHGrI
    kCEghGDjxo0YMGAANm/ejB9++CEjKChoklarvbl8+XJERUUFZ2dnz+rVq1dAfRUSAAQGBoKi
    KAGHw1E3ZSABuCAWi3Xz58/HoEGDfIPG4XDAYrFg3bp1eOutt2653e6s5haioKAgKBSKsKKi
    otWdOnXqmJiYiMrKyjppvQEDBvAOHDjwukqlqrp48eL3gwcPxsWLFxvbje+ys7NHfvnll4+0
    bNkSly5dQmlpqc868i7u6rXsXC4XnE4n3G43aJoGj8eDVCpFixYtkJycjFOnTqGsrGwXPMup
    NIQK22SxWG5u2rRp8a5duzoOHz4cQ4YMQVRUFIKCgiCRSLw75/6lLw6HAzabDUajEVqtFgUF
    Bbhw4QJefvlllJaWnmat94wmKKMAAG/DU1dVX7pcCE9qrACeIHWDtshwOBw3AeDGjRt4/fXX
    cfnyZQwZMgRnzpzBrVu3IBQKLwcFBT1fUFBwwmQy4bHHHhOdPn16msPhqLM+xGg0QqfzdCck
    JASEEIHD4QhuwlD2xub0Z86cwYABA1BTU+NTSFKp1Jv1VcrGHpoVffv2RefOnekNGzaMCAwM
    fHDx4sUYOnToPRVSZWUlxo4dKzh27NiEiIiISwzDaO8V0K+HkbzD4XCMHjRo0ENLly6Fy+WC
    Xq+HXq+HyWRCdXW1z/twOBxgGAYCgcB3eL1Vb79nz56NzZs37xcKhXt//vnnejvWAGYVFBQU
    zpw5c+q0adNUffv2RXFxMXbv3o2CggJUVFRAp9PB4XBAr9cjPz8fM2bMQFBQEKRSqY9h0Gg0
    iIuLQ4cOHfDll1/ivffeq6qsrNwIT8Fv1Z1ogbogoyjq1bFjx8ZNnTqVsCmR9fI+Jk+ejOvX
    r8d+8MEHr7nd7mfQgC2IFQpFxrlz5y5/8MEHbZ9++mkcPnwYv/32G9RqNb755pvrcrl8Xmlp
    6U2WAlJcvXp1aVxcXL9FixZ59yCqF2QyGWiaZgghwU0cz100Gs2DISEhOHnyJAwGA+x2O/h8
    vo8aVCqVfSsqKhKa27KTSqXhxcXFa7p16zZs9erVVENiIps2bYpYsWLFunbt2tERERHf15Me
    uxMqnE7n+i1btnQaOHCguEuXLoiJiYFarfYpAT6fDx6P51MCdrsdNpsNdrsdBoMBt27dwvXr
    13HgwAH88ssvZYSQ9Wj4FswEwD72uWx/9dVXxREREY1+trdu3cL+/fstpaWlbwL4pQmviWYV
    y9NoeOyWgqcKP4uNKdX7HYlEot+MRmPRzz//rAGA7t27Y+bMmRg/fjy4XK42JCRkbn5+/omu
    XbuCEMI7efLkdL1eP6Y+2YpOp9MX25HL5eBwOHyGYVo0VY5UKtWDycnJ0Gg0CAgI8DEN1dXV
    KCwsxPnz5/vW1NQkNjfTEB0dzdm4ceMTlZWVKzUaTfDu3bvvSG/WhtVqpcxm85ibN2+SyMjI
    ORKJpOxeGWr3gs1mKxMKhRt37NiRWlJSIsjLy0N5ebkvM81Lx9+JlvcafRKJBKGhoYiOjsah
    Q4d0AoFgnclkauj27zoAb+r1+tK33nprzZo1a3j+mdVeJoHL5YLL5UIoFCI/Px+5ublwuVw+
    xsELiUQCq9XqdDqdi+HZ+sJ5N566LqTFx8f3fvHFFwmfz2+QIPF4PMycOdOdkZGRlpWV1Que
    Pdfrhdzc3DyNRjNnw4YNy7dt2xZnNptFU6ZM4W7dutWt1Wo3rlu37uL8+fMhk8kUmZmZy1u3
    bj1h3bp1dF31R3fiZwkhcLvdoiaMZS6AEeXl5dLXX38dQqEQPB4PXC4XdrsdTqcTNTU1sFqt
    4fAsntisCkmn0z1js9mGV1dXY8GCBQ3iwB0OB9xut8ZgMCzIy8s7gqZtlXGWz+ffmjhxYlxq
    airKyspQXl6OwsJCH7dtsVhgs9nA5/N9Vp1QKIRKpUJCQgJ69OiBwMBAZGRk5LATcGNxraqq
    quLcuXPikpIS3LhxAyUlJaiurobFYvH1x+l0gsPh+PhwoVAIuVyO0NBQREVFoaKiAhUVFZVo
    ejZXK3jW0WOaMMZeALAHnu2i64UePXr8cejQoaUVFRVvOJ3O0IqKCixbtgxFRUWQSCRfPPzw
    w/tVKhWUSiXv1KlTM/R6/QKXy1WvhZHdbrcvgM3j8bxGTpPkiMPh/Gvy5MnS2bNngxACp9MJ
    p9Pp82jdbjdefvllzWeffTaouRXSjz/+OJTL5b43efLkoJCQkHvWafkjJCQE3bt3p/fv3/9k
    ZmamQ6lUTjMajebG9EGhUMDpdB67efNmodvtjktNTUXPnj0RFRUFpVIJsVjsiyXxeDw4HA6Y
    zWbfUVVVhfz8fGRlZeHgwYMoLi7OlUqlZx955BEcPNjgPCongCNut7tKoVCE9uvXDwkJCYiM
    jIRKpfL1w9sXp9MJi8Xi60t1dTUKCgqQnZ2N06dPIy8vTw9PToHzXoMcdfDYgwYMGCCJiYlp
    VOpWy5YtOQMHDhRkZWUNgme3wnq95ZEjR6K4uDgjJibmCkVRUeHh4Slffvnlk3q9/mpkZGT6
    t99+C5lMJsnOzl6ckpLy7Lp16zjt27dviqssacJYDgPQRSKRIDU1FZ07d0ZwcDDEYrHP8j98
    +DAKCwtpo9GYBuBD1C+Vub6ejqJjx4545plnGnwtRVFISEjApk2bJAaDoamLhlaaTKaC+fPn
    x4lEIthsNojFYkilUl9MzUuXeb0j71FTUwOTyQShUIjKyko4nc7raNq+WhV6vb74lVdeiZTL
    5RCJRAgODkZgYCCEQqEvCMvj8WC322E2m31WeG5uLkpLS2EymaDT6VBdXV3K0hhNwVA0fbXw
    CHiKHeutkLRarbtfv34f//LLL787nc7WTqez/cGDB4cJhcLcFi1abMjLy3MGBQVxz507N1Wn
    073ucrkaJAf+nhQhBBwOR9wUOaJpuvPvv/9Oxo8fTxUXF8NsNsPlcoFhGEgkEkRGRqKiooJQ
    FPUwIWRDc8qRyWTqnpaWFvTGG2/clh1bX8THx1NTpkx5qLy8PAD1ywz9C6qrq0FRVCWA/EmT
    JsXNnTsXbrcbZrMZNTU1PnmxWq0wmUzg8XhQKpVQq9Xg8/mQSqUQiUSgaRoLFizAypUr8wgh
    hkYoIy/KAZSMGTMmdOnSpXC73bBYLF4D29cfk8kEhmEQEBCAoKAgX1/EYjG4XC5WrlyJ+fPn
    l6GOUqC6FJKcz+cnP/TQQ1RTUom7d++OdevWJdlsNnl9LfDt27dj5MiROH36dLHdbi/Ozc09
    LhQKvxKLxdaSkhLboEGDqIMHD06Ii4ubtG7dOm5TlBEA0DQt9neJ76mlOZ4MS79MllIA0wsK
    Crp99NFHnb7++uu+0dHR/OrqakilUtf169dr7Hb7HrfbfRbAMfhlTHE4HK+H1qD+pqSkICsr
    CzabDW6329WqVStMnjy5Ub/96NGj+Pzzz91ms9kJAEKhEBaLpTG3crndbmu7du0wceJEqNVq
    KBQKX/KC9/CfwLyH1WpFZWUliouLsWrVKmRnZ1vQtKJHmsPh0M8++yzGjBnjo34oioLdbofL
    5fK17e2XN5nBS2fq9Xp8++23eOWVVyi73d6UXHoRPMWLzZGP3wOebTjqpawzMjLQvXt3l91u
    P2UwGE4NHjz4C61W+4FIJNJdvXq1lBACmUw22Gw2L2yoMqqtkLyGXX3l6A4odTgc03fv3t0F
    noSOPuHh4YLKyko6MDAQly9f1p06dWo3PLv83iZHjUVaWhoKCgrAbqbpPn78OPr37w+Kohq0
    mglFUd5JmnC5XHdERAR4PB5ycxvuWAsEArvFYjHu378fJSUlyM7Ohlarhclk8sVd/ecLmqZ9
    8VmRSISgoCDEx8fj3Llz4HA4xs6dO7t//fXXxj4iCwDdkSNHMHv2bGRlZaG8vPy2vvhvSEhR
    FGiaBofDua0vmZmZYI0HS1MUklQqlao0Gk2TXnp4eDgnICBAVV5eLmsIJfTtt9/6/v/QQw/h
    xo0bekIIPvnkE0yfPr2lRCKZNnfuXH5TlRErWHXRFDQ8C6K2drlcQSz1UgXPzqZl8ARji1wu
    V4v+/fvT3bp1wzvvvIN58+ZRH374IX3ixIkr8KTvWgDEwLPMRzg7CdjgSfe8gjukfd6FDg0U
    i8XDCSEiq9Xa8ffff8fq1asbPBFQFIXs7GwYjUYlTdMvCAQCg1gsPmmxWP6nEY8xVCKRJDzy
    yCMAgMOHD6OgoADl5eXQ6/U++tLtdvuy3AQCARQKBUJCQhAREeGj7fbt25fsdDoD0MAgvr9z
    LpfLY9VqNQ4fPoycnBwUFBRAp9PBaDT6UmO9SRVe2k4ikSAgIAARERGIj49HUFAQAgICorVa
    bWwT6Ew5mm8vpXB4th+ot/d49OifpR5ZWVkOAFlutxuvvfYaunXrpna5XK+4XC5FY5RR7Umb
    oihRPeQoGp5dRoPhWeW7Gp7MvDJ4tk+4RdN0+IABA+w9e/ZkVq9eTS9evBibN2/mnDx5Moul
    /i3wrEYQynqOYj85ugyguAFUqBIATQiRcDgcH/3YUDliVxvhCIXCENb4qGmM0rRYLEEA4oqL
    iyGXy9G6dWvExMQgODjYR3F7wwJ2u91HQVssFpSWliIvLw/5+fkoLy+Hy+VKOHDggKwJcqQG
    EF1WVoabN28iMTER/fv3R2hoqK8fd+qL1WpFeXk58vLycPPmTW8yioZlk6obq5Bomqbp+lax
    3yNOQ3M8bgXd2Hv4b5XcoUMHlJaWDklNTY0ZNGhQk/rm5YndbjdTh4X7cmJi4lNdu3bVREdH
    8wBQ+fn5zmvXrhlLSkp0BoOhRi6XB/Tr1y982rRp9DvvvIPS0lJzSUmJaMOGDbKVK1cuPn78
    +ASr1WqRy+UBGo1GnpCQIAoLC+PYbDaSk5NjOXbs2PXCwsIPAHwKTxrsXWG1WsPcbvfSnj17
    Bsnlcrjdbpw4caLxfNLQoUEAFp89exYlJSWL0PCsSABoY7PZwt544w3QNI2AgACo1WoEBwcj
    LCzMl9zA5XLhdDpht9thNBpRUVGBU6dOYefOnT5l4Xa741nFfa6RP6mjXq9XsnFGaDQaREdH
    Izo62pdt559gwVrIKC8vR1lZGc6fP48ff/wROp0Oer1eAc9yLaca2RcBezQHhGjC5pf+crRs
    2TJEREQMsNls7RpzL6817M8WuN1uXh1yNBee5W80rDKiWBrfxE6aNfAkUoUZDAZs3bqVLikp
    qdmyZYvUbrdL4Ul/Hw/Pmm8BrLIXsuEFN/v5dQAfwLOP1j3lyGg0xmq12o9tNpvK6XQGP/HE
    E9iwYUO96hhr47fffsPo0aPDjUbjDpPJ5BIKhWvgWZGjoUgUiURRq1atQv/+/b1KykfT2Ww2
    X/2RVCoFwzAQCoUQi8UQCATg8/kghOCbb77BxIkT4x0OR1PkKIXH47VYunQpRo4c6Z17fH3x
    xmG9ffEmOHj7IhAIQAjB3r17MWbMmBCLxdKONRgapZBMZrO5pq4U4joDC55aFGMTYwI+TJ06
    VeB0OtM6dOhAy+XyJiukerjmFEVRbfr16xc7Z84cmEwmUBQFsVjMAFAYjUZFTU0NAgICIBaL
    sWXLFmzfvj3P5XJ9sXbt2plqtVr+3nvvMZWVlbEWiwUBAQEQiURwuVwwm82gaZpiGEb84osv
    JhcWFsbUM24EmUxGli5digZsunVvvs3lwuTJk/Hll1829hYPCgQCYe/evfHEE08gMTERKpXK
    R3HW9R60Wi0uX76MLVu2YMeOHUFWqzW1kYLEAdBJKpXSTzzxBEaMGIHY2FjIZLJ638BgMCA3
    Nxc7d+7E5s2bKa1W25mN/Tkb0R87ezQH7Kh7yaF6YeLEiYzBYOhT31Uc7qSQvLEWrxwRQu7l
    olOsZxRzh3kogD18Y5FdR+86gM8PHTo0i/UMmTtc7+99ieDZGiS6nmNeaLPZEm02mwoALl26
    hHfffbdRTENeXh5sNhvPbrfHsdZ8YzN3O0mlUkl2djaOHTuGnJwclJSUQKfT+eoavd69l2ng
    8XgQi8VQKpUICwtDXFwc+Hw+RCJRkF6v79hIOaIAdBeLxTzv2nXZ2dm+5CCz2XxbX7xMg7cv
    CoUCLVq0QFxcHORyOcRiMddisTwETx2fuzEKSWc0GnPWrFnT9ujRo41aIZqiKFy6dAk1NTXX
    m+A23oaamhopl8uNzMzMxJo1a5p0r8OHD3t/171mTRMh5L1t27Y9ePLkSV5lZWUJAEahUChD
    Q0NlwcHBQoVCwdHr9bh8+bLr4sWLF61W6xIAGcXFxbemTp36SseOHaMTEhIgFApRUVHhLC8v
    N5eWluoNBkMVh8OhlEqlJjs7+wY8a5e56uF1UjU1NZy33367Xitc15eCOXXqFLhcLtXIBWBP
    WSyWjP3797fJz88Pio6OZiIiIsBmcUEoFPoEyJs95fWQtFotbt68ifz8fPv169e1drv9Ehqf
    ZecGsFOv1yvT09PbHjp0KDQiIkIQEREBpVLpS2jwZm45nU7YbDZfllJlZSUKCgpQUFBgKyoq
    0lZXV1+Fp7i7sTEtA0tHNcfK1OVoQPnEvVBVVSUjhMQ19nq3240LFy6Aw+Hg0qVLXjm6l2th
    gmeL7QdZxVLC/qtilZHQTw6dAC4AWALPZoal8Ozx41/T4PWs9PDspUSxlOYN1nhw1WfY+yuf
    ixcvNqUW7y/3bsxFDMPkVFdXn120aFGi1WptdLIVwzB6iqKuC4XCkkbGhGkA5Tqd7tbChQtD
    XC5Xo3e65nA4xO12V7I6gHs3A60+ZkB/eJY2byp+AfBTc7zljh07yq5evbrUarXGuz27sTXq
    Pt7AJUVRVFBQ0J7S0tIP7mEZceDZRKyC5ac5LPccwvKi4fAEm6/DUzzpz2G3BZDGcubV8CyH
    U8xOLjr2PcSwA7hetNADDzwQkp2d/ZTFYhGjmSEWiw+ZTKajjbxcwnL6rQHEAmhB07SaYRgZ
    TdM8mqZ5FEVx3W63kxBid7lcdrvdrmMnp1vwLLKZxcYUmhq0FrLvJoG1mNUA1DRNB9A0zaco
    ikdRFEMI8fWFEKJn+1LKTmzZ7LtqindPwbM6w3PN8Ho+BzAfzbDKdbdu3ZSXL1/+3Gw2JzX6
    AQuFvvIGu90OlUr19a1btxbWIUc9WDm6xU5O8rvI0X7cHlNt4ydHVawc3aqlpL0Kq16Uc/v2
    7VtnZmamW61WZTOLESUUCt+zWCzrG3rhxo0bsWzZsmAul5tUU1OTbLFYWgIIs9vtKoZhlBRF
    8V0uF9flclEcDodwuVw3RVF2m81WwTBMBYBil8uVRVHUH2Kx+NrEiRO1K1asaOx44bHy8wCA
    Duw8FcQaEUL2/XHYw8UeTpY6rWLf8w141h28AE+qvvlegvJfiaSkJIZhGLqyshI2m63Rwsm6
    mVRAQIDrypUrzn/DwqSNQnJyMrKysu7LVhYCgeAvq5c3FuPHj8dnn33GZQeud/DSrBfjHbxO
    hULhvNvK0s02Q3gMEMqvD7X74gbgpijK3cz7RYE1XppjK/Iy1htoMhISEiiDwRDodrsFXgXX
    0N/tn6TC5XIpuVxek5WVpftvkaP27dszmZmZQVarldPc9xYKhXqLxdLgvayCg4N9sc3ycs9W
    YL169eLeuHFDoFAohE6nk2swGDgGgwEymQyBgYFukUjkqqqqMqnVamtGRoYTgC9JJzo6GmfP
    nm2un8VjaVEp+y+P9XK5rAzZ4VkxxeR31Jti5uK/EFFRUeByuQ4ulwuGYZpls7n/FgHyc4F9
    aczNDT6f32wK6bPPPvNSK/eMvdxvZeQ32ZK6+nIflBFYa7HqP2kMsbt5apsiB3UtI/ZfAAfq
    mdn6d8GrhC5cuIBnn30WIpEIeXl5ToqijOxx1/d1/fp1PPTQQ8jKykLnzp2xe/duNDUHoBa8
    8VAd/sE/+Af/4B/8g/+r+F+rpMorQ9XqswAAAABJRU5ErkJggg==
  }
  set pieceImageData(Merida1,40) {
    iVBORw0KGgoAAAANSUhEUgAAAeAAAAAnCAYAAADXTpstAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw8DUCF+YwAAIABJREFUeNrsvXd8FFX3
    P/6eme2b7KZtym4aKaSQkIQaijQNCkgHhSAoioAIPCg+Io8oYKMqTR4RBEWpigJSBaWELiUS
    WiC9kd52s9k+9/fHzq5LaAnZ8Pj7fjiv17xeMDM7c3LnnnvOeZ9ygSf0hJ7QE3pCT+gJPaEn
    9ISe0BN6Qk/o/wJR/y/8Ec899xz0ej1UKhWGDh2KESNG3HF9/fr1+Ouvv1BWVobt27c/+eoO
    NHToUNTX14NlWfs5Pp8PhUKBjRs3YsSIEdBoNHdcpygKFEXht99+AwAMGDAAJpMJhBD7PUKh
    EN26dcPs2bOfDPITeiC9+uqrMBgMUCqVWLJkyZMBeUL/Z4j3/8IfwefzpRaLRbp58+ay06dP
    Y/z48fj222/vUMBFRUXw8fEBIQQURT358hyNGDECfD6ftilVs9kMhmGIVqslGzduxMCBAyGR
    SCiLxULxeNbpQggBIYS1KeCxY8dShBCKz+cDAEwmEwCQtWvXksf4p8QCeAaAHsABALn/wOF+
    ijN6U57MvL+JpmleTEwMPvroI3Pnzp0RHx+Pr7/+2n69X79+qKiogFgsRkrKk6FzpN69e6Om
    pgYWi8V+TiQSoVWrVti+fTsGDx6MwsJCm0zaDWiappGamgoASExMhE6nu8OAdnFxwfDhwzFz
    5szH9acEAugGwATgBIDSf+Bwe3LyW/FPU8A8AO0BtOP+nwrgIjeYLU5ZWVltS0tLF3l5eR2S
    SCSbv/3225z27dvj4sWLSEpKgkwmE2q12g5SqfS6SqWq/ociEQmcAtEB2Acg+3G8eOnSpYnF
    xcWT9Xo9QwghNE1TMpmsMCYmZhGAmq+//lqek5MzS6/XB7AsSyiKovh8vplhmFUALgHAnDlz
    xtTU1DxrMpkIAPB4PMrT0/PMpEmTvj569KjFiQq2P/fvfQCuOlzrIBQKv4uOjm5jNBrZ9PT0
    4xaL5VUHJUwD6AqgD4BaALsft4Lm8/kwmUwTuP/+E7UIDUDO/VsNwPK4XnzhwoXYffv2TfT2
    9t4nk8mObty4UTtw4EDs2bMHr7/+OiZPnoxp06b5+fv7V3Tu3Nl07ty5f6L8dnCQ3z0Ash6T
    8eJG03QgIYT6W79SOoqisgCYKYri0TQdStO0GADhbiA0Tedy3xk0TatomlYQTgNTFEXRNF0F
    IN/JCrYPN8+ONVjfIgGsA5AIgAXwG4A3ABQ53NOGU9BaAH8AKPkffOd/A+ADeMc2lv8EEgOY
    IxKJimNiYtjY2FgiFotLAHwIQPI4GAgODu4qFArV7u7uRCwW/6VQKF7p16+fqE2bNpg4cSKj
    VCqni0SiUm9v72+6devm8fTTTyMpKQmvvvrqP2UMu0okkps9evQgHTp0IDwe7wiAoMfx4tDQ
    0Jfkcjnp378/GTVqFOnQoQORyWS3nn32Wb+4uDjEx8d7C4XC6+3atSOjRo0i/fr1I1KplKVp
    emBcXBzi4uLg4eHxpUqlIiNGjCAvvPACCQoKIgqFYseyZcsETmKzrUAg+Ktz586kc+fORCAQ
    pHIK2Uaf9ejRgxQUFJCrV6+SiIgIFsDrDteflUqlBT169CBt27YlNE0fAqB8jIqtHUVRQymK
    OsEp36Hcgk3/Q+ZfJIDlAM4AOAtgJYDox/XyoKCg7gKBQMPn89Vubm5bQ0JCOhNC0K1bNxBC
    EBwc3M/Nze1cQEDAO4MHDxZ++OGH/zQF3A1ABrcoEwDHAQQ/jhdHRUUNl8vlxRKJpMx2eHt7
    n+jfv7/vK6+8gp49e/q5u7ufdLwulUqLZDLZc9OmTcPrr78OpVK5QCqVljvcUx4YGLj2iy++
    4DuJzRAARwCYOcPuFDfnbPS+w9gRTglPdLieyBncLAAjgJ0AfB6njqNp2oeiqF8A/ArAF4D0
    nzL5kt3c3DTLly+3FBUVkaKiIrJy5Uri7u6uBfDy42CgdevW0VKptPD9998nU6ZMIR4eHnVy
    uXx5YmKiR0hISA83N7eS5ORkEhYWZlYoFB/Mnz+fAQC5XA4/Pz+0bt36fz2GS/v3708qKyvJ
    zZs3SXR0tOVxjV1wcPCY4OBg87Vr14jJZCKrV68mrq6uOQkJCc/GxsbGR0dHJ4lEoswVK1YQ
    k8lE0tLSiL+/v8nFxWVG27Zt49u2bRsvk8m29evXj2g0GmIwGMjo0aOJm5vbj4+ggEMBTAMw
    E0Bbh/OzO3bsSPLz80l+fj5p3749AfCew/WFvXr1IsXFxSQ9PZ1ERUURAJMcFOA3Q4YMITU1
    NeTSpUskMDDQBOAFh+uJAGZxStvPyUMsArAaQDpFUWrO67jJWfySf4D8hnMLImlwnAMQ9TgY
    CAsLa8vn84tt7xYKhTkqlWrC8OHD+ZGRkZESieQqAMLn89X+/v4vAYBCoUBAQABiYmIwceLE
    //UYLmkwdhYA4x/Hi0NCQsbyeDzW8f2urq43hg0b5gcASUlJSpFIdKMBfyaBQDDIjqt6en7Z
    8Pt7e3v/9Ajy683J1ThOlm30r3vML8fkkAX3uP6WA7qwssE1I4BRDr+PAPAKgGEA3J08xAIA
    CwH8SVFUJYBKAOcBrHKG/PKcwNyQ5557TjJlyhR7DHDy5Mk4d+6cZPPmzcMAbIc1Ltdi5O7u
    Xl5QUFDO5/NVy5cvR6dOnaQff/zxtKysLC+LxSLr2rWrz5dffoktW7Yws2bNmr506dJgqVR6
    i8/nX+fz+Wlt27YtqKqqYp9++mls27btfxEjZliWhcViAcuyYFmW4qCOx0Y8Hg88Hg9CoRCE
    kIC8vLwtnMVJ83g8uUgkAo/HA5/Ph0gk4tE0Pb+oqMgIACzLuohEIvD5fPB4vEcdv1YMw2yI
    jo5+SiAQUFevXr1mMBheAXABAGs2m6HXW6eR2WxGA4h099mzZ4cOHTo03GAwkIyMjDMAfne4
    bjGZTNDr9dDpdGBZljj8vq9UKv06NjY2sKqqypKRkfEsIWQynBfn0QOYS1HUSkLIUgAMZ2BU
    Aqj/B0Cn42GF5xtSJwATONiNbUkmRCJROY/HqzKZTL4AYDAYgsvKyr44e/asgmVZlU6nawMA
    JpPJtaqq6kMfHx9XHo+XzePxslxcXArXrl2rj4qKwquvvop///vf/4txZO4xrsxj+YAURSiK
    Yh3fZzabFRcuXJgTEBCgTk9Pl7Esq2iIykgkknE+Pj5dAFBqtbrLPR7dVIjVF8B/AQzgeLnE
    zZ80AMJ73O947gCAMQACuP9f42BmRz3TEFUSOHjHXwOI4RTzdk55OyvUaObWkkJCyDjuvZs4
    xMP4v7b8pAB+nz17NmlIc+bMIRzcJm9pJmbOnMm4uLhsGDRoENHpdKSqqopMmjSJUBRF+Hy+
    ZcOGDYQQQm7fvk3i4uJIQEAAadOmDZHL5TqGYa5LJJKVoaGhXdevX8/Ex8ejf//+j3scu0sk
    kowuXbqQuLg4lsfjHedgm8fiAbdq1cp88+ZNQgghhYWF5LfffiMHDhywH7/99hspKCgghBCi
    1WrJ8ePH77h+8OBBcunSJcKyLGFZliQnJz+KBzwhIiKCvXr1KiksLCQ9e/YknGUMANE8Hu9c
    TEwMGxMTY+bxeGcaQFgUgC6w5h0cANAQ0ugtEoky2rVrx4aFhZkpivqVg7AYAJtGjBhBKisr
    SUpKCvH19dVxi4hzV2iGAYAfOOH9p5AE1ngbuc9xDIBrSzPxyiuvCN3c3H5u+H6pVKqXSqXa
    hufd3NyIp6enls/nZwsEgh0KheLVuLg4PwAYNGjQHclEjxGCTneAT48CaPU4XhwaGvoSn883
    NxwjTjETiqLu920feN3b27up8psMa86P43Ns8hsH4IrD+NwA0LGBQk3ilHYKZ/w50nMAChye
    e5KD+GkAXzV4Z31LyC9HPwD4yamOjxOs+8w///zz6crKSnh6egIAqqurwSVKZMMaNG8xGjFi
    BHQ6HRiGqTh37hz++usvHDlyBCkpKejbty+ysrLo2FhruNDb2xuRkZEQiUSYP38+rl+/Ltqz
    Z0/U3r17owoKCkbOnz9/Y2Bg4KpDhw4Vvfbaa1i/fv3jEuC/6uvrz1++fDnMbDZbzGbz73iM
    SUIWiwU1NTWorq6GVCpFYmLiXfewLIuqqipQFIX4+Pi7zWVCUF1dDYvFAqPR+CheME1RFBiG
    AY/HA03TNsEEgOtms/mlq1evjgTwKoC13ILnaK1ncPdbAOQ1ePYxvV6//dKlS8M46Gg3rFmW
    PAA0TdNgGAYMw4CyMu7s2CxtsVjacfAYxS0wF1ras2wEsQ+x4o0tzePkyZOxZs0ag5eX11WK
    ooYRQsDn89GnTx+MHDlSuHLlSqSlpd3xm+HDh2PMmDGSM2fOtDp48GCrixcvDrp169al6Ojo
    lYGBgb9ERkbqH3O1w2Xue0Zw43UMzk1gajI1xghxsqEiuYfX78LN98sAxnJe7nAAKzgY13Ee
    noM1gU3D3e9IhwFs4JT8ZgBbufWRuYeByEfLhHak3NpiAiADl8D2v1bAFgDfnjhx4pmVK1eG
    fvDBB6AoCitXrrQcP368EMB6zoVvMRo/fjwmTZo0lBDyilarxd69e5GcnIzBgwdDqVQiPT0d
    0dHRdi8kPDwcdXV1CAoKQlBQEPr27YsJEyZg7dq1vtu2bXs3NTU1MTQ09N/r168/n5iYiLNn
    zz4OFGG2v7//4AULFqCoqIj32WefTVWr1YUANrb0Asjj8aji4mL69ddfh1QqtZUYPZqWoWlY
    LBZkZWWBx+NRTVwAj2RkZJx78cUXE4VCIdLS0jJgTbawUQaAXwC8CaAX50k6Ztn7c4cQgBfu
    zKB0435j4CCqKgd4aev+/ft79e/f36+qqootKSk5DOBPJ8OTr8tksv+EhYWpzGYzbt261Vav
    13/GGRKW/+E6reegvmdxd8jDzF1rUQN6zZo1iIqK6pSbmzvCNu/at2+PpUuXIjMzE2r13etc
    dXU1QkND0bt3b7zxxhvYu3cvf+3atZ0vXrz49datW58KDAz8mKKo249JCbsA+A+s8Ufb954K
    oBDAdy39fS0WC+Xm5kZNmTIF3t7ed9TrPwKcDZZl8e233yI/v8n2wykObo7j/n8bwF4HKPsv
    AHWwxog7AvimgW5Qwpp46gFAwY2fjcSc0WrgYO4yB/2zk5u/Xty5c06WX5sHP0skEj3Nsixt
    NBrXc979pf+1AvYGEGE0GquOHj0a+s4774CmaRw9epQ2Go1VAMK4hbNFUsYJIQgJCWlVUVEx
    Z+zYsYrOnTuDoii7wgWALl3uDG9Mnz79jpo5hmHQrl07LF++HJ07d6Y++eSTnvn5+d+0bt36
    jbNnz55++eWXsXHjxpaSHy8A7wUFBb35xRdfiIYNGwaj0Qgej+f9ySefLK6pqREB+BYtGEMX
    CoU3CSHX09LS2iQmJmLEiBEQiURNVsIUReHChQvYunUrzGazxs/P74xUKm2K8ZVpsVheSUtL
    6wdryYECQM49rFsaQG9ubt1wuBYik8lkZrOZrq+v92+ggDvBWuaVeQ9Fk1lXV3f59OnT27i5
    uhdNq0HkcxaxkPtOmgaGQSc3N7e5CxYs8B02bBhMJhO+++47/wULFnyo1Wr/gjXz2PFZcm7B
    McBaMmVoYQWyh/M8n2pw/s8GBlCLyG9MTIxXfn7+Rzqdzi60t27dwqRJk3DlyhVoNJq7fvfr
    r7/ixo0beP755/HSSy9h7NixeOaZZ7BkyRKXDRs2TMrJyfHr1q3bDLlcntvCStgL1mSiN3Fn
    TNMLwCJYE/A2cJ5dixDDMAaTycTK5XJ63LhxEIvFzfoep0+fhl6vB0VRBqFQ2JRF4AaA1zgP
    N5yby6n30DcMrOVaobAmI9rIpnwlnCFd2EABJnJ6pKGXfYlTutXcerH9HghYcyiAz+evGjZs
    2FOjRo2CyWTChg0bRhw6dMifZdnR+B/2G2gH4JBSqdQPHTqU/Pzzz8RisRCLxUJ27txJhg0b
    Rvz9/Y2wpp93aikBdnd3nxYWFma5du0aIYSQS5cukbfffpts3bqVPAodOXKEREREEIFAcCk8
    PDy2VasWC+VEAPgpKirKtHfv3jt4MBgMZPXq1cTX11cD4Au0YMr94sWLqbCwsHaurq77PD09
    LXPmzCFVVVVNGjOz2Ux+/fVXwpWgZSqVynG9e/cWffnll4/KVkcAxbAmKzmunnGurq7FPj4+
    LIAZDX4zu3v37iQhIcHEQV2OHuhqlUpFBALBFViTRWwkgDVD+Rqanv3sAWu251ZYMySvcAvB
    VgDTYS2T4gN4OzExkVRXV9vHq7KyknTp0oVwfx+fu3c6gG0AzlMUdZ1bWH6GNTPbqwU+vQDA
    i7DGgCtwdxywEtbkk5c4RdIi8uvj4zOGYRgDACIWi8nzzz9Phg0b9sDYpeMRFBREPvnkE1JZ
    WUkMBgNZs2YN8fLyIlKpdFebNm18GxrgTqRI7vuYH8BfHYBlLSm/nTp18vHw8PhSJpNp+/Tp
    Q4YPH06GDh1KBg8e3OhjyJAhZPjw4WTQoEEkMDCQSKXSk2FhYV0PHjz4qGx1gDVmO7XB+Wju
    vMXxGmcgTefQPhOAsbaEXk7+bVnSN3Bn+SAN4CPufPAj8BnIeeTzYM1k/xDW7Ooo/J3kNS4y
    MtKckZFB/vzzT3Ly5Ely7do1EhUVxQKYyIXKBNx8GMU9axmAz2CtZAlrCQ84EMCKnj17Ji5a
    tIiOj4+HUPi3AThkyBD069cPV65c4c+ZM6f3oUOHVhJCkuHk5hJpaWmUxWJp26lTJzoiIgKH
    Dh3C9OnTUVZWhm7duj3SM3v37o1FixZhwoQJCaWlpR9HR0e/2rt376oNGzY4k/VEiqJW9OzZ
    s9PixYvRsaM1H4FlWVAUBYFAgMmTJ0OlUrnMnj37Xzdu3GjFLdZOb85x5coVUl5efikyMvLV
    zMzMf3/++ecTU1JSXJVKZaPgLIqiYDQacfr0aVaj0RxTqVTvZ2Zmng0ICMDUqVMfla1yWJMp
    pnAKwtZ0g3Z3d6d69uxJbd269Xmz2fwt5yXyAES2a9cOZWVlvNTUVMcErUCZTPb0s88+ix07
    dtBGo9FRofeENa50uYlQaziARV5eXv179uwpbN++Pdzd3aFWq5GVldXp0qVLL968ebNQo9Ec
    BeBFCLkLdeG6ij0NIE4mk/WOiIhQtW/fngoJCYFMJkNNTQ0uXryYcOzYsYHl5eUDYc1Gvumk
    z86HtTRkDue938/AeBpAZ25xWwwnZ30ajUYA6GqxWAQKhQLz589HcnIyvv/+e+zdu9d2/YGU
    l5eHuXPn4sqVK1i2bBkmTJgArVaLOXPmDCwpKckJDg6evXr1av2bb77pTNa7wlo37ZhIBJFI
    BJZlHfmWcoolBMDbaIHmHBUVFaUdO3Z89/Lly9dSUlL+Yzab/Tt16oTOnTvb8igeKr8FBQXY
    t28f9Hq93s3NbXNoaOgnaWlpuVzi4KNQGYfcTIU1fnvTQWHaciwGAfge1lgqzSk9ipPlCAfU
    wpeDmG3KmGpgqE8AUMAwTK2jjD2ERJzx/Kafn19kcHCwQC6Xo66uDqWlpaaioqKS+vr6kxwC
    lKBQKBiGYTBjxgzU1dXh559/RmBgIJWRkdHBbDZXAxgiEAi6+/n5+Xl7e/NdXFyg0WhQUFBg
    KS0tzYI1UWydM8M574aFhVnS0tJMD/OOrl+/bomMjLRw1oVTsaDx48eLxGLxnunTp5Ps7GyS
    kJBAAJCYmBhSXFx8X2+tuLiYGAyG+/JsMpnIe++9R3g8njEwMPCtjRs3OpPvWJqm/xo6dCjJ
    ycm5453Lli0jO3bsuIOXc+fOkQ4dOhAAuzhY1uk0ePBghIWFYciQIUJvb+/pPB6vvl+/fmTy
    5Mlk0qRJDzxeeeUV4uXlRVxdXXe3b9/eH7BmojaT3CmKOs1BYGsc4L32gYGBpd9//z0JCgqq
    5qBoAHBnGObU2rVrydy5cwmsmYo2C/bVjh07GtetW0ekUul1Bwvag6KoAyKRiMAaT26sMeoH
    YH9cXBw5cOAA0ev1d3wvlmVJaWkp2blzJxk8eDARCoWsm5sbOX78uP2e06dPE09PTyIUCtkh
    Q4aQ3bt3k7KyMsKy7B3P0uv15NChQ7Z5vR/Oq1FO4jxc0sijBsDzzp53c+fO5cvl8h/5fD5Z
    vHgxIYQQnU5H+vTp80B++Hz+XR4yRVFk6tSpRKfTEY1GQ5KTkwmPx1OHhoaOcDLbbWGNdd7F
    06JFi8iUKVPux/deLmTnVNq7dy8SExNBCKFCQ0P7SKXSMxEREWxaWlqj0Cu9Xk/GjBlDRCJR
    sZ+f39t9+vRxCQ4OxuXLl5sDi7vh79ryFQAEnDHQFtbYMIE1D6Mnp2hdObTFNlY7KIqyye9I
    DsInnCJXORg3W7nzP/N4vMZmbAsA/MfV1bVu+vTpJDU1lajVaqJWq0lVVRXJyckhu3fvJq+/
    /jpRqVQ6AFWBgYHk4sWLZNSoUWTgwIHk8uXLpHXr1oSiKI1CodCNGjWKbNmyhaSnp5OysjJS
    U1NDqqurycWLF8n06dOJXC7XAZiPe5djNZlEAPZNmjTprgXjfjRt2jSWs4Sclp0WEhKC8PDw
    WJlMVrBx40byxx9/kB49epAFCxaQnTt3EpPp3rZBdnY2ee6558jJkycfyHNeXh5JSEggIpHo
    Vnx8fNQzzzzjLNjv606dOpHs7Ow73ldfX0969epF/vWvf93FS0pKCvH39zdxHmGLUdeuXeHq
    6horl8vLDxw4YC8retBRWVlJEhMTiZeX17zw8HDcuHHDKeNE0/RPvXr1It7e3lX4u6ygo7+/
    f9m5c+fIiy++SDh4ngIQIZPJ8o8dO0Y2bdpE+Hz+Rc5YEQPYOXv2bLJ3714ikUgcIawpISEh
    +m7duhEuXtcYogHM9fb2thw+fPih876mpoYsWLCAuLi4kBkzZthDNDNnziQuLi5k0aJFpLa2
    9qHPOXr0KPHz87MZsc3N0GZgTf4iTTy+hxN7x3fp0gXt27cPFovF6SEhISQ1NZXs2bOHrFix
    gvj6+t6XD39/f7JmzRrSqVOnu67J5XKye/dueygqODiYuLi4nO7YsaOfk0oLBfcbO4lEQlJS
    Umz9z+91mGBtMtMi9OKLLwIAvL29ewmFwuoxY8aQ+fPnk3nz5j3wmDVrFlEqlcTLy2vZ+vXr
    6Q4dOjSbF4qi+FwslnBoVhJ3KZ4LLdnGZAlN0xSHsDh2EkulKMqbQ2p+cDh/i4sPA8BoDuIn
    DMN8GR4e3lj2hgiFwuqPP/6Y6HQ6cuXKFbJo0SIybtw4Mnr0aDJ79mzyxx9/kJqaGnLmzBnS
    r18/wjAMWblyJTlw4ADZtWsX+eabbwiPxyPdu3cnv//+O6moqCCHDx8ms2fPJqNHjybjxo0j
    S5YsIdeuXSN6vZ4sWrSIiMViNf5u/tMscgGQMm/ePGNjY4SLFy9mAZyGk2qCZ8yYgU6dOvFF
    ItGyLl26sLdv3yY6nY4UFRU9lJeLFy+SkJAQcuTIkYfeu2LFCsLn81kPD4+PRo8e7QwvOJzP
    52etWbPG/g6dTkc2bNhA3n33XRIUFEQ6duxI5syZQ3bu3EksFgshhBCLxUKmTp1KABzkxr/F
    yMXFJU4ul5cfPHiwUd+2urqaJCYmEk9Pz4+czMrysWPH2uq5f+cg0Y4qlari6tWr5IcffiBC
    ofAqp1CTQkJCtFlZWeT06dPE3d29GNbC/HYeHh6lJ06cIAcOHCBisTid8yLD+Xz+9Q8//JD0
    79+fbYJhEwDg2ksvvfRABKVhPH/WrFmkdevW5ObNm+TmzZskPDycvPvuu41+hslkIuPHjycc
    FO/fzHGVwlom01QFfMZZ8vvmm29i4sSJlKen59sURZmkUinp0aMHcXNzIwzDPJCPuLg4UlRU
    REaMGHHP6/369SM1NTXEYrGQ2bNnE5qmTR4eHm84qeQmnAsD3VGTPHPmTLJq1SpSUFBA0tLS
    yBdffEEmTpxIOATH8TiEFqyrTk5OhlgsbkvTdHlTvi1FUUSpVM4fOHCgU/iIiIgARVHLHN6x
    n5s7CbAmUtnOp3Hy2J1DWWznSzhvORLWhCrb+Qxu/ithbZlq4/+dRrImA/Bb9+7dSUVFBdmz
    Zw8JCQm5azxkMhkZO3YsuX79OiksLCR9+/YlXbt2JYWFhaSoqIj07NmTdOnShWRkZJBr166R
    l156ichksrueExYWRvbt20dqa2tJ3759Caz5UG4NLfqmkgFAcX5+PsV1FHogEUKQl5cHWDNL
    nZLNu3z5cmRmZvYXi8Uvv/XWW5Sfnx9EIhGUyoe39z1//jykUilCQ0Mfeu+gQYMQGhpKqdXq
    4adOnQpetWpVc1lv5eHh4dup0985aSaTCWlpaThz5gxqa2tRUlKCM2fO4NatW/YYLE3T6N69
    OwQCQVhLwdCPSi3Y9CCvpKQE48ePR3h4eA9YE6sYABQhBD179kRoaGg4rOVF4X5+fmIvLy8o
    lUooFAp3WJMf+sXHx3vHx8fbxpLiEJwpnTt3jhoyZAiKi4uNHCzWGAoEEJCQkACBoHGIl0Ag
    wLRp0+Dq6opdu3Zhz549kMlkmDZtWqOfwePxEB8fD4qiAtH8HsMPq/29b8gWTiqJ++mnn7Bv
    3772arV6KiGEp9VqkZKScteuPvei1q1bw2QyITv73ukQKSkpOHnyJGiaxqhRo+Dn58erq6sb
    07p1a4UT5morNEioEolEiIuLQ/v27eHq6govLy906NABUVFRtji/I4W2pPxu2bIF1COkfHM/
    ofbs2eOsuDQIIbkOp56GNeGvYY19BKw7hIU1MEzcYY2bJzUwOG0dAl/D3/F3M8MwRY38tiEA
    Enr16gWJRIIvv/zynvNIrVbjhx9+wIQJE6DX6/HJJ5+gtLQUKSkpOH/+PHJzc/Hpp5+CEILX
    XnsNmzZtume5XGZmJlatWgWBQAAOQY3jjLhmKWATgMOHDx9mG1Mje/78eeO+ffsMnPfW7JKK
    5ORkREVFtdJqte+PGDHCvSlWW05ODtavX49nnnm3MvJRAAAgAElEQVQGKpXq4attYCCeffZZ
    mM3m1kajMWnatGYjSGI+n89zTFhzdXXFokWL8PPPPyMmJgbPP/88fv31V7z99tt3CLBEIgFN
    00IOVv2/QIWFhYUmpVKJiRMn8vl8/huwZgsTAAgICEDv3r0FsJY9dAwNDaVcXFzg6ekJlUol
    BNCTpukB/fv3h4uLHTQgAPq4urqOmTFjBgQCAUpLS7VNUMCEpmlWImlaJEWlUmH48OHYuXMn
    fv75ZwwbNgz+/k1zZMViMbi4Z3PJAOD6I/zuKpzQOvP9999HYGCge21t7RyTydSkEgMvLy9M
    nDgR58+fR3p6+j3v0Wq1+OWXX2A2mxEVFYXevXvDaDS2r6+v7+WEciRxQxi+pKQEEydOxNCh
    Q3Ht2jX89ttv6NevH2bNmgWtVnsvCFv0/7rgVlZWgmGYIgdDTwBrQlYU7mxxKYC1frrhxiQC
    WBPdBjU4T2BNCpzgcL6eZdmiRn5bKUVRIqVSCY1Gg6KiogfefPr0acydOxcRERFISkrCjh07
    sH37diQlJSE+Ph4ffvjhQ/tEFBUVoa6uDiqVCjweT9wQRXrUeNKugoKCvdOnT6cOHz5s6817
    B1ksFhw9ehRTp07l5ebm7oc1Zb/Z3lZwcLCwoqLircjIyA4zZ86ESNS4+XzlyhVMnz4dN2/e
    xODBg9GYDD+appGUlASJRMLTarX9evTo0VzhUdfU1BgbfniBQAA3Nzd06NABMTExkEgkd1nP
    2dnZMBqNdXBSB5b/H9Dt8vJybVlZGcaNG4fExMRIWLcoE5aVleHMmTPo2LEjZDJZHwDPisVi
    fPTRR/jqq6+gUCgAYKRKpYpv27Ytbt26ZctM9QAwY+DAgYoBAwYgLy8PtbW1NbDGqRrFE8uy
    FQ8T3HvR888/j6KiIhQVFeFRoL6ioiKwLFuBO+ubH9UD/hHWWFxjqRjAFjihqUTHjh3pysrK
    13Q6Xb+m/jYwMBAsy2L58uWor7+/LZCSkoL8/Hzw+Xz069cPfD5fVFdX12/IkCHNjWHX3MuJ
    0Ov1qK6uxq1bt5CVlQWtVnu/DO7/M/JrsViKcGfWbxub/Da49Rn8vc2oIw3nlK0jucJafhjo
    4L1reDxeY+v2KwkhtbW1tZBKpbZ14sGKbtcuHDt2DKNGjcKFCxdw4sQJvPDCCzh+/Dh+/fXX
    h/7e29sbUqkUNTU1YFlW3YS1plFw3EY/Pz/DveKFf/zxBwkICDBRFLXZccCaS5GRkc97e3vX
    bNq06aGxM5ZlSXp6Olm4cCGJjo4mbm5uNa6urmZbokZjqKCggERHRxMej5fbunXr8GayrwRw
    aerUqcRsNt/1Lo1GQ3Q63T3jrM888wyBNbGhRS3opsaAq6qqWioGHCYUCvO+//57Qgghu3bt
    IjKZjMhkMtKxY0fi5+dHOnbsaI+9KBQKEhYWRoKCguxJPCqVinTp0oWEh4eT5557jggEAmLL
    aLTF+CmKOovG76AiBrC7W7dupKysrEm10jU1NaRLly6kW7duRK1WN+m31dXV5OmnnyawttB0
    BgLCwLrdm2NCzP2OYu7eZm8u0LZtW/To0aOTTCYrQNNj0EQmk1kUCsVDa4SFQiH58ccfCSGE
    3Lp1iwQHBxOBQHAjKioqwAnye/F+71UoFMTNze1BvG1rafmVSCRxTY0B0zRNlEqlU+WXoqhQ
    WBtjECce7D3O/SUQCBpbZ+0G4NTIkSOJXq8nmzdvJnK5/KHvnTp1KikpKSExMTEkKiqKFBQU
    kEmTJjVmvpLNmzcTs9lMXnvtNQJrcxtPZ3jAFKzBc6qyspJUVVXddUN1dTWqqqrAbRSthBP6
    63bo0EGcm5v7avfu3eX9+/fHjh077psuX1BQgHnz5mHIkCFk/vz5mSUlJYsiIyPnURRlaEx9
    oY18fHwQExMDs9nsazAYIpv5JxQD+G779u3G7du33xXvcnFxucuj12g0WLFiBU6ePFkCay1Z
    i+4s9Q+iGoPBUL5gwQJ8/PHHqKiogLe3N9RqNXQ6HZYsWYJevXohJiYGHTp0wLBhw/DTTz9h
    9erVaNu2LVq1agUvLy8kJibitddew8WLF2E0GhEVFYWysjKsWLECa9euBSGkBI2sz2MYRgdg
    0/nz57WrVq2y787UGHJ1dYWfnx9UKpUjJP5Q0uv1WL58OU6dOlUHayayM7oqWbhnrXgIrKyD
    tXf2987wfv39/XlXr16doNFo/AFAKBTCoeHCfYnH49V7eHgcioqK+q9ardY9LN5nMBhw7tw5
    EELsWxaaTKYAg8HQ2gny+y3uE0MvLy9HTU3N/X77f0p+CSHVzvT2HPROQ5SyjMfjNRZVqAWw
    948//mAPHz6MF154AXPmzHmoPKanp4OiKPj4+ECpVIJhmIdWe8jlcsydOxcjR47EqVOncODA
    AQJgH/5ug2ud24/iJAF4QygUTuvVq5f/Sy+9hAED7t58om/fvli7di1vy5Yto48cOdJDp9Ot
    grWPp+ZRR1+n0ylpmk4YMGAAjhw5gvfeew8rV65EXFzcHfedPXsWs2bNwp9//pnl4uLybXBw
    8E+7d+/O6NSp0+t8Pl/UmGQtG/H5fLRp0wYAhPX19bFdunTZc+bMmUeelwC+Ky8v95sxY8ak
    CxcuuPft2xd1dXWorKy0tYCDRCKBt7c3zGYzNm/ejIMHD+bp9fr5sGbRtRi9+uqr+PHHHwln
    wTbqNzRNg6IoUBTFxsTE4OrVq85iRwOgpKysDDt27IDRaIRcLodtE/HY2FiMGTMGarUaZrMZ
    7u7u9o0ievfujcrKSohEInh5WZtItWnTBl999RXS09MxefJkMAxjiwEV4M7WkQ+C1QBgr9Fo
    /Orzzz//FwD+O++8A5nM2svCZDJBo9GgsrISarUa1dXVUKvV0Ov1qK+vR35+PiiKwoYNGyCR
    SCASieDm5gYPDw/I5XK4u7tDKpXaww9qtRpLly7F559/btTr9as5AXYGBcNaEjPyIR61iIMN
    FZwiblYjmLy8PD+tVtuDCyVh3rx5+P3337Fp06b7Kd46FxeXgzKZbFNISMixjIyM/mazeXJj
    3nXjxg3o9XqIxWJER0dj7969Uq1WG9OnT58/jhw50hz53cg5H5MBeLi5uWHixIkIDQ2FTCYD
    y7Kora1FamoqvvvuO5hMJsDarnAerLsk/ROVJQDAyW1363CPFsQURUEkEkEkEsHFxQVSqRQC
    gQB8Ph98Ph+EEJhMJpjNZhgMBtTV1aGurg56vR4Gg+FennaRUqk0ZGZmNvb7/VBVVZU0a9as
    3iKRCFOnToXJZMInn3wCvV4PqVQKuVxul0+RSITw8HAwDAMPDw8YjUYwDIPIyEjodDo7X1qt
    Fmq1GnV1dZBIJJg7dy6mTp2KM2fO4K233sLt27dTYG1LSpqjgOUAPlIqlRNnzZolePnllyGX
    y6n7WfzJyckYMGAANm/erFq4cOFHBQUF/gA+4GIpTSZ3d3dJSUmJyMfHB6dOncLAgQORlJR0
    xz1Hjx7FtGnTLJmZmTtVKtXH2dnZaXK5HP/5z39kZrN5eHR0NB0REdE0LDQsDDweD1qtNtgJ
    iRxqAHPLy8vPLlu2bNHmzZsj5HI5+Hw+pFIpWJaFVquFxWJBWVkZamtrUwC8y8EXLZZyPHLk
    SKxfvx579uxpXVNTI/ruu+9w6tSpxhhFyM3NhdFoDPby8hJHRUXpnFQLbABw3MXF5bmlS5cy
    ERERkEqlkMlk9rGxTX6DwYC8vDx7LoJNmG3XRCIR+vbtix49eqC0tBQ6nQ5nz57Fu+++W28w
    GFKaOK46AJ/W19dbFi1aNDU/P186fPhwXLlyBampqSgoKLBn89oWNpqmwePx7FnPn3/+Ocxm
    s/06wzDg8/lwc3ODSqVCbGwsYmJisHPnTmzdulVrNBpXwbopuDO8p64AlsK6fWNjPA5/Lu7W
    BcA7sG4Fh0eUX3eapt0YhsE777yDxMRE3K9dqUgkuiyXyz+Li4vbl5KSoh0wYAD/ypUr/S0W
    S6PWrPz8fFRVVUGlUiEyMhIMw6C+vj68GcrX0TCcB2sZzOLY2NjIadOmQSQS3TH/unXrhj/+
    +APZ2dnHYe1idqEl5ddmZBYWFtImk4nu0KEDfHx8HlilYOtid+7cOVgsFmrjxo1YsGABZs+e
    7Qx2jDRNn2NZdoBCoaDj4+MRFxeH0NBQBAcHQ6FQwNXVFa6urhAKhY47oMFiscC2/7dGo4Fa
    rUZxcTFycnKQmZmJtLQ0XLlyBTU1NSZCyOmMjAy2CetyIYC3rl+//uX48eO7z5s3D1OmTIGf
    nx/MZjOio6OhUCggFovte6TzeDwwDGPfqEYsFmPJkiV2Po1GI+rr61FWVobr169DIBBgxIgR
    +PHHHzFnzhzk5uaegXWP4oK7jMwmDKgAwDsBAQGTV61axRs8eHCjIGW5XI4pU6YgKChIMHXq
    1Em5ublqWPt3NjkjmqKoMqPRWJ6RkeE7depUu+W0d+9e+Pr6wsPDA++99x6bl5e3MSgoaFZF
    RUVFr169sGvXLiQkJIxiWfap5ORk+7aJTVg4IBAIwLKsj7u7uxDNz+Y2AjjB4/Eqp0+fjlGj
    RkEsFts3QdDpdKirq8Nnn32GH374IbelhXfgwIH48ccfERAQ8KxWq/20bdu2LhUVFSgvL2/M
    N0FkZCSuXr06IjU1NSchIWExwzA6J3nCW/Lz84dv2rSp86hRo5CWlob09HQUFRWhqqoK9fX1
    MJlM9haeNn5sgsIwDAQCAaRSKTw9Pe1wZGRkJHbt2oXa2tp9sGbnNxkeBzDfaDTmbNy4cdbP
    P//cKjo6Gu3atUP37t0REBBgT74Qi8V3WPg2T9lkMsFoNEKn00Gr1aK8vBwFBQW4desW9u/f
    j4ULF6Kuri6LU7ybnKR8O8PaSKJNM347Adaa/qZbVAZDFcuy1RRF+RQVFWHBggW4cOECYmNj
    YTKZ7JnNMpnshJeX19Ts7Oy02tpa6PV6nDhx4hmtVtvoxK3q6mr7Rg4KhQICgQB6vV6RlJTE
    P3z4sKmZ42iCtdNTdV5eHsaMGYOqqqo7ECw3NzdbaUo+rJsStJj8EkLQpk0beHl5yQoKCsaq
    VCrZqlWr0KFDh4cqYK1Wi1deeQUHDx5MCg0N3TZ79uxrN2/eRFOdlHsqDIFgs16vH96zZ8+E
    devWwc3NzR4isBnPRqMRNTU1dpmwGTACgQACgQCenp5QKpWIj4+3G7ClpaU2nk9IJJKdjS3n
    c6DLAF4tLCz85O233x6SlZUlmDhxIvR6PfLz85GWloaioiJ7SMFoNMJgMCA1NRWEEIwdOxau
    rq6QyWT2w8fHByEhIejVqxekUilWrlyJpUuXGquqqvbA2vL1nmn7TVHAz7q4uEyZP38+01jl
    60gDBgxAVVUVM3Xq1Clqtfo8rK0Vm0RBQUEVV69ePbRx48bYbt26oV27drh69SoWLlyI//zn
    P9i2bRtu3LhxLDw8/IPy8vKKQYMG4dtvv4VSqexXWVk5Z+jQoeIxY8Y0eSLJZDIIBAIYjUbf
    0tJSCZyzQ81TISEh8UOHDoW7uzvq6+tRW1tr7wWtVCrxwgsv4Ndff+1dW1sbDeuG1i1CQUFB
    lL+//3O1tbWrRo8eHfrxxx/Dw8Oj0b+3WCz48ccfpXPmzJn1119/0WFhYYv69OlT7wRvo4gQ
    smrLli2xx48flwQEBCA4OBgdO3aEUqmETCaDWCyGWCyGRCKx7wSj0+nsR319PdRqNQoLC5Gf
    n4/t27cjLy8PxcXFJQBWNiMkogPwNYBii8Xy/ezZs+WDBw92ys47Bw4cwAsvvKCBtf/3bid9
    Zn9Y+zm3acYzorhnJOMR9rslhJQKhcI/1Wp15JIlS8CyLFQqFZYtW4aNGzciPT0dEonkhr+/
    /79u3bqV9tRTTyEtLQ0hISHRubm58/V6faMtZ4PBAJ1OZ5dfPp8Pmqb96+vrJVwssNnyS9N0
    nEajQV5eHlxdXeHm5mY3oLOysmz7YvcmhETD2nSiRSgyMhJKpVKWmpo6T6PRvNm7d29eRkYG
    cnJyGv3733//vXNRUdGa8PDwNyIiIq7OmzcP8+bNa94APfVUbkpKylenT59etXHjRmF9fT2u
    X7+O27dvo7KyEgaDwa54bbCzI4JlO4RCIRQKhR0dAoBLly6pxWLxyrq6uopHZC8DwGS1Wn1j
    yZIl7/30009CrVaL6urqh+Z27N59b5GUSCTw8PCATCZDVlaW0WAwLAHwOaw7NaE5CtgdwPSh
    Q4e6Nacj1AsvvEAdOXLE9bvvvpsOIAUNAtIPNTtNJktAQMB/s7KyOowePbpnhw4dcOPGDcTE
    xICiKOzYsaNKIpEsTE1Nvf3xxx9jzpw5CAgI6FtVVbWqf//+AYsWLbJbYU205EDTNMxms0Sr
    1TqjHZ8AwPC6ujrJzJkzUV5eDq1WC7PZDIqiwOPxIJfL4erqCpZl/WFtxdhiCnj//v0dy8rK
    VgUEBITGxsbi2LFjTW6wwTAM4uPjxQcPHnw3MzOzplevXsucxN4Rs9mckZycHPf+++9DKpXe
    4UWazWY7FGSLEdkgLYZh7F6wLb6k0Wgwbdo0fP/99xdgzWhtLqXq9fri0tJSOUVRMBgMqK+v
    h0ajgVartcPgNiuaEAKBQAChUAihUAiBQACJRAKZTGY3IkpKSlBfX1+Kuzcmf1RiAEyCteOQ
    MyDsyVwoqUmJWRqNxqRUKteYzebu9fX1IYC1RrqoqAjHjx8HTdN6d3f3JdevX099//33sWbN
    GigUioji4uLVWq22Y1PeZbFY7ArYBicaDAaXiooKvpPkd8TIkSMl77zzDvz8/CAWi+2ljRaL
    BXV1dcjLy8MHH3ygOn78+PMtqYA9PT3dLl++PB/AGxEREfysrCwsXLiw0TJM0zT8/f1RU1PT
    vbi4eE14ePgbXbp0afZ6c/HiRSgUigMlJSWZb7/9dhuGYSCTyeDt7Q2FQgE3NzdIJBJIpVL7
    3Kcoym44a7Vau2NSVFSE1NRUbNmyxZaLkapSqY7FxMRg//79j8piNYBNZrP5lczMzCAejwcP
    Dw+EhoZCoVBALpfbeXPkz5bTYeOztrYW5eXlKCsrQ1lZGQoLCwFrr/WtD1K+TVHAXT08PDpN
    nDiRFYlEj6yAhEIhNXHiRMuePXvaVVZWPtVU63779u0AkB0ZGfna7du3J+zfvz/S1dU1sK6u
    rt3SpUtRW1u7tW3btikTJkzAnDlzEBoa+kxlZeWXQ4cODf3888/RlOSr+wmet7e30AkxzjYA
    koRCIby9vZGUlAQPDw+bwoVarUZZWRnOnz8PsVhMaTSa4bBmX5a2hAAbDIZoiqJa0TSN9evX
    P9Km3jboVyqVitRqdfurV69SToLdqgHk3rp1K27v3r24ceMG8vPz7UaLTbnZlDAAu/K1QVmu
    rq7w9vZGcHAwQkNDbd1vbjkJ1q0ihBTs3LkzMicnB+np6SgpKbHH8W1wuG2hsy3QtjGzGS+2
    xLvWrVvjxo0bYFm2qKkG6gMojPNaaSc8i4K1D+8mNLGhR48ePTBz5syz/fv3n1RWVvY2y7Ih
    eXl5XlOmTPHUarWQyWSHfH19fxk9ejQ+/fRTJCQkRGRmZv5Xq9X2ehRYtqECslgsAhcXF2co
    4BgXF5en+/btS1iWpc6cOQONRoO6ujowDAMXFxfI5XK0atUKTz/9NE6dOjXUbDZvQAvti377
    9u2+hJBJCxYs4A8cOPCRUZj8/Hy88cYb3dLT099Yt25ds/vOV1VVQaPRlJvN5luxsbFt3nvv
    PcTExMDHxwe2vBdbEuf9viHLsjCbzaitrUVpaSkuX76Mjz76CBkZGTdKS0s1+fn5zWWzEkBR
    WFhY0Pvvv4/4+Hj4+vra+WMY5oHjaVt3NBqNPQb8ySef4PLlyyWwbvOJ5ipgCkCvhIQEl4SE
    hGbja/Hx8Uy7du0khw8f7gng16Yu0u+99x42bdqUlZeX958ffvhBcO7cOb+9e/e+YDQaqeDg
    4B/Ky8sN3377LTZs2NBBrVavGDlyZPjixYvh49P8LTkpihIwDOOMOj4VgBq1Wu2Wm5srEolE
    8PT0hLu7OywWC6qrq1FSUoKCggJSX19fB2vM2NuZCrhr164oLi6GxWKBwWAgvr6+ZP369QgJ
    CXnkZ1osFrzzzjvYsWMH0el0CAgIgEAgQFZWs3Zh0wPIPHjwIG7dumXrdIW2bdtCoVDYLVPb
    YYMAHS3U0tJS5Obm4tChQ6ioqEB2djYLINNJBoIOQP758+fBMAyCg4PRrVs3BAcHw9PTEyKR
    yO7tCoVC2LxkWwxMr9ejqqoKubm5yM3NRUZGBi5evAhYEzbqnfS5n0HzW1g6UiCsbQKbpIDX
    rVsHPp9PsrOzf4+JiTkHwDUgICDy+PHjk8RiMe3n57fw4sWLtWq1GqGhoQHZ2dkr6+rq+jwq
    kw0VME3TArFY7Az59TMYDFXz5s1z1Wq1LhqNxh6/dHA2IJfLiVAorGFZVg9rJnmLKGCLxSKT
    SCR8T09PFBQUPHJ7WJZl4ePjg8uXL7uWlJQ4xYAmhBgAZMjlcvTt2xdmsxlFRUW4ePEiKioq
    7F6u7bBBubbDlr/h6+sLpVKJXr16YenSpeDxeDkSiYS9V2Z0E0kDIFsikXRNSkoCj8fD7du3
    kZqaek/+bElYtrCXRCKBp6cn/Pz8oFQq0aNHD1tZUz4akWzcGAUsAhAdFxdHSaXS5vdyE4uZ
    hIQE8+HDh6NhLYFo0iKzcOFCAMCQIUNImzZtDHq9Pre2tnaxLe4zb948LF682OPmzZvzevTo
    Eb1w4UKnKF+bAjabzU3aUoqmadA0fUdWLIADANIqKytjUlJSuqakpEzu0KGDZ0FBAWQyGWux
    WNjs7OwcWPeRTOUWujJHb5NhmIbPbLK1y0El8PHxITweD/7+/k1ukdiQuMlHioqKnJV0QgOQ
    JiYm4ptvvoGvr69dkdmsT5ZlwbKs3bNkGAY0Tdv33aVp2q6YMzMzkZycTF27ds1ZO3MJAXiP
    GDECX375pT3Rymg02qEqg8EAi8Vi/15isRguLi4QCoWQSCR2KJqiKJjNZsyYMQOrV6/24p5t
    biZ/PFj75tJOXPNpAJ1g7cvbpISm//73v3j55Zdx6tQpDZ/P1+h0utu+vr4nGIah8vLyjLt3
    78b7778vysrKmq3T6ZKaufg3lF8hRVHOUMAHTSZTWkFBQRsbJN++fXu327dv893drT1drl+/
    nllWVrbaQX6dWhP71FNP4fbt2zCbzaBpmi0oKMDLL7/cqD2AHzReRqMRPB6PnDx5kgQFBUEg
    ECAjI+ORn2k2mymKonhXrlzByJEjcfv2bVRVVcFkMoGiqDvk1FZ+ZzKZ7pBtQgj4fD48PT2h
    UCiQmZkJhmGY6upqp9kwWVlZSE5ORllZGSoqKuwGla2CwYao3Y8/Ho8Hd3d3+Pn5IS0tDY2V
    28YqYI+AgACn7YkbFBTEhzWu3GQFbKPjx4/j+PHjmDBhArp3746amhp4e3tj3LhxUCqV/b29
    vZM++OAD+Pn5OXPe8/V6vaCRYxYFoA3Lsj4sywq4vzMX1r7DZdz/8wAM7N69u8tnn32GN998
    E5GRkfT48eOpN9980zMnJ0eEv3cDCYV1c2oVISTAbDbzOe+wiIsvZTVlsfby8oqoq6sbrdPp
    eDU1NW3r6+vppUuXNjlDvKEFfeHCBQBIEAgEn/L5fIjF4pMVFRUHmjHmHgASYmNjUVlZiRMn
    TiA7OxtFRUWoqKhAbW0tDAaDXQE7CrWtzlahUCAgIAChoaEIDQ1FREQEde3atc6cgmuuCR3A
    MEyst7c3du/ejbS0NOTm5tp7wDoqYJtCsBlQtvivVCqFSqVCcHAwYmNj4ebmBoZh2lgslkAA
    zY13iAAEtYDjFQjr9qJNTmhyrDWNiorCtWvXTDRNo0uXLhg8eDAiIyN7mc3mMWjG/uH3Mkwp
    iuI30oAWc/IbA+vmC0IO6cjh5LcU1uYthQC827Vrx3z22Wemt956ix8REYFXXnkFM2bM8MzJ
    yXHhkAwa1ib8vrAmwwXCGpd3lN+MJhpbQnBdlcxms4dQKER8fDya2qe84Zhdu3YNFRUVkuDg
    YH9iHUQNmtc+05UQEm8rs3zuuefQqlUrtGrVyl5d4ogQ2Rwpx6OyshI5OTn2MiQ+nw+tVpvg
    JPn1AtDWVi3xzDPPoFWrVnYEy5G3e/FnNBpRXV2N3Nxc5OTkICsrCyKRCBqNJpqbO/nNVcA0
    APoeO3s8MgkEApp7brOV+jfffNMwzuTy559/jh46dKigS5cuTuHXIZbHGI3Ghw1EFMMwH4SE
    hPRu166dV2BgIE8gEKC4uBg3b9403L59W1NVVVVjNBp1crnco2fPnqoPPvgA169fR3Z2tqmu
    ro6ePXs2s27dOo+FCxd+dOnSpYk6nU4jFotdPT095YGBgS4RERF8Dw8PaLVaZGRkGP/666/b
    hYWFuwAscPSUH4jr6vWtzWbz7Hbt2glkMhkIIcjIyGiWtQvABsXEEEJiLl++DLVavYzz+B+V
    QgGEbdu2Ddu2bQMhxA75KBQKREVF2WsJbQJiS3hSq9WoqKjAzZs3cfz4cVRVVYFhGFtyTltY
    GyrkNnN6xLEs6/fll1/areCgoCC0atUKPj4+8PHxgaurqz0RzGZBG41G1NXVoaysDKWlpcjP
    z8eFCxewbt06m0HhC+vuKc1VwEI4cR9uB5LACRuM//LLL/Z/l5aWYty4ccIDBw6MM5lMsmai
    VXZv0OalEEIYg8HwMPmNhnXf5V6cgnO83whrg4kaTnm6URTlxzCM5b///S8vKytLX1payhMK
    hTyBQOAOa73wBE5Zu8DaR0Ha4JkmTqnb5LdRYSaNRtO1rKxsudFoFLEsK1cqlfS6desQFhbW
    HG8Vr7/+OrZs2dK3uLj4D65pxjcAljTjUwQDiExOTsbnn39+h5drMpmg0+lgMBjsyZSAtYeE
    h4eHvRGGLRbLrVt4/fXXsWnTJmfJbwSAkJvaZsYAAAkuSURBVEGDBuHrr7+2ryG2+l69Xg+9
    Xn9HjomLiwvc3NwgEonsZYY2/sxmM95++22sWrUqkJtLzVbAegCaiooKp0luWVmZDXt3elu2
    kpKSMIqi2vfp0wePUB92XwXchMQklVgsHvDBBx/IRowYcUdNql6vF1ZWVgqLi4u91Go1VCoV
    QkNDcf36dXz66afQarU78/Ly/ObOnfvUihUr8OOPP/Ju3rwZXFVVZY+DeHh4QCAQ2KAnWCwW
    wcKFC4MXLlw4wGKxfNVYBWw2m4lMJrMsW7YMCQkJTt9WUK/X46WXXsL+/fubu4VdR4ZhPFu3
    bo3k5GR07NgRAQEB9tKwhyWc2GC1mpoaFBQU4PTp09i0aROqq6uDCCGxzRRgCkA3gUAgTEpK
    wrhx4xAfHw9vb+8mzz2TyYTy8nKkpaVh06ZN+OWXX/g6na47rBsnNGcMTXBO2VxDMjQVfm4M
    /fXXX0F1dXXdmo2R07T9G9igQgAUy7IPM/qVAJ7Dvfc+FnCIjIfj/Dp//rxtHd1pMBj8tm3b
    1ov7P9MI9IHP3dMfwOrGKmCz2eyq1+sjTSaTAADq6+tx6NAhXLlypVkxYC6G7GowGFw5Q6a5
    8bt2PB7PR6lU4uDBg7hy5YodIaqurkZdXZ0dIXKsAuHxeBCLxXB1dYWnpyf8/f3RqlUrREdH
    Q6VSgaKoQCfILwAkUhQlDwgIwNGjR5GWloacnBx7rwEbfzYDwQY3MwxzF3/BwcGIjo6Gr68v
    GIaRWiyWRDykz0BjFXD2yZMn+xw7doxqzC5CD1sQuQ5LuS2hgHk8XghFUW719fVOa4uYnp5u
    j1k0gk7U1dVt3rBhwxsnT55Efn6+kWVZysPDgx8UFAR/f3/4+vrC1dUV169fxw8//IBdu3ZZ
    srKyDgGYw7Ksz969e1cWFhYmDB06FNHR0XBxcUF2djZOnz6N/Px8FBQUQK1WGwUCAdWqVSv+
    sWPHTBaLZTUHYzV2gaIMBgN97Ngx5OXlOV0Bm0wmlJSUgKZp6lGyqh3nHyHkdmVlpd/ly5cp
    i8WC8vJyeHt72zvp2BSxo8djsVig1+uhVquhVqtRVFSErKwspKWloaKigsAak2surEMByDeb
    zfmpqakqAMz58+cRGhoKlUplLyUTCAR3xLjMZjNMJhMMBgM0Gg1qampQWFiIzMxMpKenIy0t
    jTUajWWcMUU3UwEbYM30dDZVwTl9qRsau+Fms9kpe+bqdDpoNBpUV1c/dJ9hR/kFsBnAFAev
    16Z878s2t9B+CGuy5EoA7RvxXWj8HUdfBWtiYKOXUsd5UV5ejpkzZzZ7zO4hq80SXoZhxADq
    li5dKre1lGwOSaVSCIVC0DRtpmnao2HyW1PZ40IC5KuvvqK++OKLB+6y1RiyJWhx49iKmzfG
    5ihgC4AjR48e7X/+/Hke5YROA1qt1gJrX1SzswXYYDAIzGYz+9FHH5kWL17sNG9Op9NBJBKZ
    8PDMQAOA/6+9q+ltowjDz+x6v7LrxCQkJpFShUg5kDRSoUhE8iGCA+dccqD8ASQu+QMof4Ab
    PVYcKoSEkDhw42YkUC1LwYjIoLSuImICSaF11/Zmv2Z3OOzYOK7TrO01hMqPtPLKsmdHO/PM
    vO8778ftfD4/nc/nf0IU7tIqSHENwDVVVRdSqdSs67qy7/vHAL5EFDN2CuABY+xWqVT6oFQq
    va0oSkaSpDPf909d1z3mJo0j/lsZwBuIqnx8hj68FkVRbDabzYe7u7vqMI4bzxO0fN+Hoih/
    9SpX2QfuhmG4Vy6XN8rl8k1CyJqu63OGYaR1XTd0XZdkWSaiKJJUKkUYY6CUMkopcxyHNZtN
    z7KspmVZDcdxTgCUETnG3MNgdXG7F6dPgiD4plKp3KxUKm8CWJUkKatpWkZRFEPXdUWWZVEU
    RUEURdKyPgRBwFzXDSzLcl3XbTqOY1JKH3GTcwnAD7yvw3LE5e1tJa2sjkKzDoJgmhCiDbvM
    NBoN7OzsIJ1Oo1arwbZtyLIs4vJjLxfAbc6p1tnsOf7y+znOv9+4leILLjBVALyPKO76Hd6O
    xfnai783uEb9OfrzOiboqlDVh5DRlzFhmD8bhnGXUvogDMN3JUl6PQiCVxljLzHGJoIgkAkh
    wkVHCHwdCQVB8AkhFmPsseM4913XLaXT6e/S6fT31Wp1qOkG4GPG2EMeFrvKx3ZiAOE8iGQ+
    +9S27fuIssV9dZmVKO4sN/huntRKzRA5NDSSni3z8/NZy7Le8jxP9DwvTEKzaznNaJrmrays
    3Nvb23sagxxqKpWyOzefjY0NFAoFCdE5kMEJWEPvYG0BUeiC3joGAHCmaVrQSjDQ+ax+tZG1
    tTX18PBwxnEcghGBv7NGvV43E1wMpgBMIjqfm+H3Kr+UDquNyz9NRPF4T/h9Y1ipPgZXdN7P
    mY5xbvWP8L65+OdM8THvW5NfSactvIGopKCUUHsBgDuIUqQmiqWlpdVarfZeGIYSX4AHfhe+
    74Mx1nbK0zTtbHl5+dNisRinprJ6gYUu1cFfiY/bRfx9uYu/NnonL1H7tQZev379tYODg1u+
    70sjnMtE07RvbdseONPF5ORkOyZ+a2tLLxaLM7Isv2Ka5rTv+3OGYWQJIRqlVLZtW+RaZKiq
    aiAIgmOa5u+CIJxMTU09MU3zD03T/jw+PnZ0XUcmk8EgtbkvgIzIaWoJkbPcAqIz5inOW4X/
    ppO/LiIHtRNElbJOEDnNnsRdj+Pu8k2MMBNTkgjD8LRer3+dZJutgHBCCGKaPBgAu1vzKxQK
    4BLRU1weIxaix3lQ1+bbftYgij2A48sCza/a8HYILL9e0T62NtHTK9SnHwF8iAScHjvm3Ehy
    GwuC8LNpmh+Nom3LsmIVYe/gRy9QvumaMebqoyGf9Tz8gigb2ZVGvV7H+vo6crkcjo6OLMaY
    RSk9Yoy1Kx09Q6Cu7zKZTDvnO6UUCwsLUBQldqrNmPAQeaxXeyhThAtUQsfYdtYnHhgpvGDY
    3t7G5uYmKKWoVqvthOyJiYSEtOK8/tcol8v/ynP6qb08xkgxsk0zYQEai4uL54TfpIToVk7h
    hKp1jRET+/v72N9/Vn/L5XKYmJg4F7HAGEM2m8Xs7Cx0XYfneSgUCu2cBf8hb8LxSI4xxhhj
    jDHGC4K/AYZGbSySs6vgAAAAAElFTkSuQmCC
  }
  set pieceImageData(Merida1,45) {
    iVBORw0KGgoAAAANSUhEUgAAAhwAAAAsCAYAAAAuG2H4AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw42H4mLAQAAIABJREFUeNrsfXd4VNX2
    9num90kmvYeEJCQBQjAhiIDSAgpIFUGkqIAFBEEFRQVpiiLSROSqqHC9IGABEfAiTaQEpYQm
    JBBCElJIn0xv6/vjnBknIUASJsD9frzPsx/KOXNmnz17r73Ku9YG7uM+7uM+7uM+7uM+7uM+
    7uM+7uM+7uM+/tfB/P/8cq+99hpsNhusVitWrlx5/9e+j/u4jwbJDblcjrfffhtCofD+gNzH
    fdxXOG6N7OxsxMTEICIiAq1atUKXLl3w9ttvX3ffc889h6qqKnz//ff3Z0QjkZ6ejrKyMths
    tn8mFcNAqVSif//+mDFjBoYPH47s7GxYrdbak49hwOfzcfz4cQBA586dodPp4HA4at3j5eWF
    ffv2gWHu2nT1A6AGUAqg+n/kpxEA8AFQDsB2f6Y2HJ06dUJpaSk0Gg26du2KRYsW1XvfiBEj
    UFlZiZ07d94ftEZixIgRMBgMtdY6AIjFYnTq1AnTpk3DpEmTUFBQALvdft3neTwetmzZAgAY
    OnQoLBYLiKjWPVKpFBs3brybrykDoOBkhvn+r84KpeaABICYG2TT3Xq5SZMmeSckJAxlGCYz
    JCTk5NKlSy19+vSpJSA6d+6ML774ggFAI0aMwIYNG/6Xfj8F16ru1jgzDKNkGKYFAJ5TQeAU
    2QoAV5zyAUALhmFUAOifjzJWANlui9GHYZgwhmFARGAYBgyLIgDFzTiGak4o6Opc4wMYBmAy
    gCAA5wG8B+D3ep7jxQmYirs5592QDOADAG8AOHpf1DVCu/Tzk9vt9jZisTh70aJF5WlpaVAq
    lfjtt99c9/To0QM///wzvL298d5772HmzJn/S6+oAiDn5IbxbnRg9+7duHbtWr3XevXqBQAo
    LCzETz/9dMtnbd68uf7dXiZzyZFm2jul3Fq31mPId+PkRjiAvwEsBnC8nueIAYgAGADY74G5
    EcLJvO8B5N3rCocSwFAAjwMIAHANwM8ANgHQ3umRKy0tlRcUFEwzm81ehYWFPwYGBq6eP3/+
    qaqqKjpy5Ah69uwJkUgkj4qKel4gEOj69ev3zZAhQ8z/A54OHoDHAEzkNsIzABYByLwLY9zh
    8uXLX5nNZqm7V8Lf3/8nPp//IgCb0WiUXr58+QOTyfRwHSulmM/nD+aUDhQWFg4oLS1daLfb
    GXdLJiQkZCmABc3Q/W4ApgFoAeAyJxT2uV1/QCAQfNi/f//QtLQ0bN++PeLAgQMaIhrqthgF
    3AKdwHkUTnIb/Zm7PEcCALQD4P8/sgkqAcQD0ACo5IS09m50pKioyPvSpUvLrVarISws7GuZ
    TLbl8OHDlQMHDsRPP/2EkSNHIigoiCkuLu7G4/GMb7755uFjx47hf0Ru9AMwCUAggHMAPrzB
    RtisiIuLi/L29h5ksVhEDMO4jBAvL68zVqv1FwAOlUolSkhIGGIymSLrGCo1PB5vfXZ2djkA
    tG/fvr1Wq0232+0u5YJhGMbX1/fgDYyD20VbTvbGA7gIYCWAY27XWwFYASDBTfkPAzACwFU3
    paQ7gPEAgrnPfwLg0l2eI4kA3uJkct69PJkVAJZKJBLDI488Qs899xx1796dpFKpkRt81Z3u
    UMeOHQNkMtnxoKAgCg4OJpFIdDEgIOCl9PR0eXx8PIYPHy4IDg5+QywWG2UyWU14ePgrU6ZM
    4aenp6NPnz738li3FwgEl/r160ezZs2iDh06EIBdd2NzSUxM7CuTyUxjx46l5cuX04cffkgt
    W7Ykf3//jUTEi4uLw6hRo6Qikei3du3a0eLFi2nZsmU0cOBAYhimiGGYmNjYWMTFxSEoKGii
    l5cXvf7667R8+XJ65513yNfXl8LDwxfWdZc2QdDWRSzDMJldu3alt99+m7p06UIMw5wE0NLt
    npeCg4PpxIkTRET0888/k1Qq1QNwnxzdhEJh8cCBA2nmzJmUmJhIALYB8K7PIXQHfpI+ABbw
    eLzNnNW0kVPW+uPeDaG25yyqawD0YENXPwFIvSuuoeTkIJlMdgoACQQCo5eX18/R0dGdiQjp
    6enYuXMnoqKiBspksgKlUpkdFxd3TwsLdwUaQA63eTvbbk45vaNo3bp1P7lcbuLz+eRsAoGA
    QkNDN6xevVo4cuRIjBgxQq1Wq/e738Pn80koFBZJpdL48ePHY+LEiQgNDZ0sFovr3kPR0dG3
    Kzdu5AHYV2cMj3BGixPPAHDUuccAoG+dOZ9V557NYD2ldwO+ANoKBIIpnOdrGoCkuzE3Goqn
    xWKx/t1337VWVFQ4bDYbVVRU0Ny5cx0SicQI4Nk73aEhQ4YoxGLxf3v27Ek7d+6kwYMHk0wm
    02s0mpUPP/ywT3x8fBepVFoyevRoGjNmDMlksqK4uLgeRMQAgEajQXBwMGJiYlBVVXUvjfWk
    yMhIOnv2LBER/fDDDySVSqsBPHw3FA6lUmnYvHkzEREZDAZ65JFHyNvbe3+7du0eS0xMTG/T
    ps0AkUh0fNiwYWS1WomIaMmSJcTn88uDg4Ofa926da/WrVv3CgwM/CQoKIgyMzOJiCg3N5ei
    o6MpLCysqYJDzXke1gB4n7NMnBju5eVl3bNnDxER7d69m1QqlQXAcLd7Rnp5eVk3bdpEZWVl
    tGTJEhIKhRUAHnK7Z1Z0dDRlZ2cTEdG//vUv4vP5pQA6uCkZDwFYCuBzAE+CDTk2BxiwIZST
    DMPkguVuXOE8X7PBhojuNbTkBDbV045xVuQdRa9evdQymazWpiKRSC6Hh4ePmzBhgqBNmzax
    crn8tPOaQqE4+cADD8QJBAIEBgYiIiICbdq0wdChQ++1sZ5YzxhrOU/fHUVCQkI/sVhsrNuf
    4ODgDWfOnBESESZPnqxWKpV1N3diGKZQqVS2cmkAISGT+Xx+3XuoRYsWTZUbEk5BXwDgFQBR
    btcGgA2juH+fFcBIt3ue4tae+z01AHq53fNaPb9FGYAH63hKXgcwD0BPAMJmlBtzAOQxDFPO
    KUvlAPIBLIEHIyGeepAQwGNJSUmSSZMmkbe3NwMA3t7emDhxInbu3Ck5dOhQXwDf4g6SZ7p0
    6aLfu3dvXlFREdq2bYsvv/wSS5Yska1cufKF8+fPqwHIo6Oj/WfNmgWRSIRz584Fnj59erm3
    t/dBpVJ5SSKRnBOLxZnx8fEFLVq0cPTu3Ru//vrrvSA47GazGeXl5XD+abfbHdxEuStwkr/s
    djsEAgGMRuNDly5d2oR//KASgUDgus4wDAQCgbdWq12h0+kIAGw2m0Cj0bjIX3a7HbdhoYgB
    zPTy8nqldevWopKSEly8eLEXEY3mXMlGq9VqKyoqEtTU1KCwsBA2lvlqcHvG3qqqqt+mTJmS
    Hhoayrt48aLZarVuRO3QlcFkMqGoqAh+fn4oLi4GEVnwD4+jC5/P/yo+Pj5KoVDg9OnTT+r1
    ej8AnzbD70UAlgH4EsCjnJLzLoBfwPJT7oUYcV08BSDtJp6P0QBmurnUmx1RUVG6v/7666rB
    8M9UMJlMkcXFxYt//fVXNREFGQyG1s5rer0+KTc3d5GPj89mmUx2VSAQXPT39y/avHmzJSEh
    AZGRkdi+ffu9MNb1kYcduEuk4vq4FQaDoX2/fv2WOBwOO8MwYrvd3rKez6lkMtlbGo2mAgDp
    9fp2dcmntwE+gJcAzOIMFgIwCMBzYMMnVM9cpDpr+Q8AhwB0cbv+X9QOXdFN1jA44+hLACnc
    v5/njImvm0lufA/gAhG1B/ACgPWcIXAB9yDpXA5gx9ChQ8lisZA7rFYrDRs2jADsxV0Iq4SE
    hEzw9fW17d27l4iIioqKaPDgwQSAeDye/fXXXyeHw0FERIsXLyahUEgRERHk5+dHIpHIwOfz
    z8jl8mUtW7bs+vzzz4tSU1MxYMCAuz3ecQzDZCQmJjoef/xxCgsLswL4z91wxzk9HBs3biQi
    IpvNRvv376dNmzZd144cOUJ2u52IiLKzs2nz5s20efNm1/XNmzfTtm3bqKqqioiILl26RFFR
    UU31cLRiGCZ35syZVFlZSX/88QeFh4cTt2gBNvy0NTQ0lLp3706hoaFWsG58vzrPacF5Jko5
    L4FP3SFgGOZYbGys45FHHiGNRmMBG0KUgA3lfJaUlETnzp2j0tJSGj16NAH4E80f/nqMs5j6
    4d6FFMDWG3g3yE1QK+5kp4gIoaGhb/B4vOv6IxaLDTKZTFuP1U0KhcIuFou1AoHgvFQq/Xdg
    YODTqampQQCbSVFftsUdRgy3ETrcLPN/3w25kZCQ0I/zfFN9Y+lsN5oXt7rnNjwckWD5Q3Wf
    OYO7HgzgN7f/d3AKRkSd57TmZHIpWI5H3evtOMPH/Ts2cHskA5YHVrcPR9H8IY4+nHdjUHM8
    3FMeDhOAq9nZ2SgpKUFoaKjrwrVr13Dx4kUAKMQdZkSHhIRAIpFcKCkp0e3YsUOdkJCAV199
    FYcOHUJsbCzy8vJ47du3d2naHTp0gEajwRtvvIG2bdsiIyNDun379sTDhw8n5ubmDv/ll182
    JSYmLt+yZUvWoEGD8OOPP94twXGBz+d/dvbs2TZnz56VcmO7HGzs7a6Cz+eja9eut/ajt2yJ
    li1bNutmxjCMJCQkBF5eXggLC4NcLoeb0nsNwOSCgoIjBQUFA7i1MIsTEO64zFk2UgBnucXo
    jrNE9G5WVtayrKysS5zSsoFbEwIAKpVKhZCQEKhUKgQEBDgVdFEzb+S+YDNmfLl/G3Fv4p7j
    lfTr1w9SqfQIn8/XOhwOldMab9euHfr27Sv94YcfcO7cuVqfkUgkmD59Ok+tVisPHjwYd/jw
    4bjCwsKhWq32eHx8/L+CgoI2q1Qq3fz58+tNzb9DyAbwGWc9ywEUcZvhPRUvboiS0AzcDCeU
    nGejLgK5PwvBZp9MANDDTRm5Uuf+MwAOcgr//nquZ4IlZy7m1umvnGGj5bwsQfX0wQdsJlyz
    Dj/Y8E+zaMeeUjjsAH48d+7cgCVLlvjOmzcPMpkMJpOJli5dSqdPn67mXDbWOzVp27dvD6VS
    6X/q1KmJNptNuWPHDjz11FNITU3FkCFDEBMTg+3btyMt7R9vro+PD6RSKUQiETp16oROnTph
    7Nix+O2337BmzRr//fv3Tzx8+PBDLVq0mJucnLxFoVA41q1bdzfWZKrNZnshLS1NOmDAAKxe
    vTr4ypUrrwB4Ff+woO+YW9RutzNr167FkSNHPGbF8Xg8VFdXo6KiAmq1uimPyHU4HMdWrVr1
    WHFxMbKyspCTk1OG2qz1XLBprhrOZRoM4FQ9aySRE9CJYIld1+lPnEBaCOBfbv9vA7Dn5MmT
    j7/66qtyX19fZ12AA5z3oTkQBeBNPp/fTyQSeZlMpoVE1AVsPDrnHtvbTWCzem7mhclE7TBX
    s2LQoEGQyWQSrVbb3W63u7g2AwcOxEcffQSHw1FveISIEBMTg+HDh+PFF1/EmTNn8O2334q/
    //77B3Nyctp9//33PeLi4t57++23/167di1Gjx59N8a7A9gMFbmbtf4qWJ5CwR3XNBkGqamp
    CAoK8qgCodfrkZGR0dRnFnKeB/cNXw8gw+3f58CSKt8DMI5T6K8TYWCzVJT4J1ul7sau5jyq
    y8GGQsltPz0ENvtN7PaZ4/UYRJ6CkFOgnuO+08m53AnAci9aKgEA1mg0GsehQ4eIiCgjI8Om
    0WhsYLkbwXeyM926dRNqNJr5vr6+tueee45at25NzrDKjaDVamn16tV07ty5666Vl5fT+++/
    TwEBASQSiYqjo6OfXbJkiSA19Y4T6TsC+LNjx4507Ngxcjgc9MMPP1BkZKQDwHdg06/uGJKS
    krpLpdIqACSXy+mBBx6gDh06UGpqapNbWloaBQYGEgDi8/kUFRX1VhOFRzLYLI1znKdiWZ0F
    7LSwl3CLfVU9SriGYZiDHCltfT2eCS+w4UIC8OINrJINnHV5hlNIWjTTz+EF4LvWrVvTihUr
    aOvWrTR//nxnKOk73D0G/M0Qwwnz+lzjx28grJsNzz//PBMeHj5WJBLVCpsMGzaMFi9eTB06
    dCCRSFSvCz8pKYnmz59PmZmZrjDtsWPH6IknniCxWExyufzoAw888BAAPPPMM3d6nB8ES8Kt
    23cH2LIF4XeyM/Hx8f3lcrlpyZIlZDAYyGKxeKSZzWbatWsX+fv7U3h4+AdNlBudwYZNigGU
    gOVSKOu5byE3hp/gekK2AmzmoDNUUlduyABs4a5PqefZvpzsKuLaz2DDMM0BHoAX5XJ5adeu
    XWnEiBHUoUMHEgqF5Vzf7imyuQTA0wB+YximJD4+3pVlcPr0aUpMTCSGYa4B2ANg7B1wCYGI
    EB4eniIUCgtefPFFqq6upszMTCouLqazZ89SZWUlNQU2m41+/vlnSkhIIJFIVBocHPzs0KFD
    eZMnT74T48wHy5A+06VLFzp58qSrXw6Hg7Zs2ULR0dEOADvApr/dEXTt2lURFBT0qlgsLvby
    8qI5c+ZQfn4+VVZWUkVFRaNbWVkZfffddxQXF0d8Pt/o4+PzRZs2bcJvwwKSgY2fzgVwAizz
    u67CsUwoFBLDMFmonRYLAPFSqfRK+/btic/nZ+B6jkdPPp9fxSkkk+r5/p5gY8JPc8qguAnv
    oABLoHTW+3gWQG8A0aid8dJXqVTqnRlDTnz55Zckl8v1qJ2WJ+He9VHumZO4Pna8gUu5OaAA
    kM7j8X7E9WmEDk7IPoo7xP0iIsTGxkZJpdLT7n2RyWQkkUhuyilwb5GRkfTqq6+6ssgqKipo
    3rx5pNFoSKFQnG7VqlWXOyif+WDj8Wdv0mcH59JPuVOdSk5Obq9QKP4ODw+nJUuW0KFDh+jw
    4cN06NChJrXDhw/T4cOHadOmTZSWlkZisbgqJiZm7G3IDV+woaf3OG9DdD33fMiN33nUzmQB
    Z1RcwD/ci7qcrTROmSHOW1IXKQD+4tZlm9swFpwVh4M5r019ilNbgUBwefr06VRRUUEGg4EK
    Cwvp+eefJz6fX3CDeaECmyIczr2b+E7MGymAd/l8fnWPHj1o2bJldOLECbLZbEREZLfbKTMz
    k1asWEG9e/cmkUikA5ueKG9uwREQEPBmaGio488//yQiogsXLtCzzz5L0dHR5CQ4NhV79+6l
    6OhoEgqFhTExMY/egXGWA5jG5/OLBw4cSOfPn6+3X7t376YHHniAOOEy+E5opmlpafjggw/4
    UVFRfeVy+XGlUklTp06lrKwsunr1aqNafn4+ff755xQREUEikagoODh4avfu3ZVRUVGe6Ooo
    sCG9FXWsDR6AT1q3bk0hISFWsGxwdzwaEBBQ88Ybb5BcLr/KLX53Yb6kVatWFBERQWBju+7w
    BrAdbIw8rYmbRR+wvJAihUJh9vPzs/n4+NhkMpmO85ps5BSFUADTwsPDr5sfFy9epBYtWhDn
    Pg8Dm/mxEcBFmUym9/X1tfn7+9uVSqWFc9luB5sWKGjGqRPHeT6r6lE23DfCas4Cb93cczkz
    MxMhISEv8Hg8GwCSSqU0YsQI2rZtG40dO7ZByoZ7S0xMpLVr15LVaiWr1UqrVq0itVpNEonk
    r8TExIQ2bdrcCbnxqtvGdqt2DmzhxmaXG08++SQvLi7uQaVSuUcikZCPjw/5+flRcHAwBQUF
    UWBgYINbcHAwBQQEkI+PD8nlcpJIJFkRERGjunfvLvZAqOYpLqSwCICAx+O5GyqLuXGzcKEI
    d3QBW8COwIa529Yxct51m/ev1mMkrQPLpWiqcuoNNsX/G7Ak9b85D+tusNlrQ9yiDmMCAgLs
    x48fpz/++IPGjBlDO3bsoFOnTlFISAgBeL5nz57gFJZBAD7mHAjnwNYSOcmt0efqMcg8ilFC
    oVD78ssvG0tLS+0326TLy8tp+vTpJBKJ9GDjXs2qcHh5ea3s0qULVVVV0YULF6hz584EgDp3
    7kz5+fl0u1i3bh0plUpSKBQZKSkpLdLT05vrddQAPpDJZIaXX36ZSkpKamUAVVdXu9y3RERn
    zpyh/v37E4/HK+HGudlPn3rsscdYNblt23hvb++NIpHIHB4eTjExMY1q0dHRJJfLHXK5/Fh0
    dPSjX3zxBc+DIavu3AIuA5BeR+H4dMCAATRixAjirGr3rIjJbdq0cWzZsoWCgoL0YKvoOhHB
    4/FOT506ldq3b09gY+HueAFsGvjFejwnt4IQbN2EkrZt29KcOXPo119/pWPHjlFGRgZt27aN
    3nvvPerTpw/5+fmZuUV/xMfHhw4cOFBrrh49epT8/f2d2TGZ/v7+lr59+9LChQvpl19+oYyM
    DDp+/Djt2rWLFixYQO3atSNO8Xi5meZPJCf4GrOB/96EMWy03AgODl7GMAzJZDJasGAB1dTU
    UEFBAfXs2bPRCgcA8vb2plWrVpHNZiOz2Uxvv/02CYVC0mg0G3v06OE1b9685nodL26TNNTt
    k1QqpfDwcBIIBPX1+RqndAvvhMxITU0N9fPz+5TP5+vlcjlNnjyZvvvuO9q4cWOD2wcffEDh
    4eHEMIxDpVLtio+PTwPYYytuFwzDPMzJjRLUrnPkHoolzihwN6SfxT+1OAwABrpdC0Dt2jOv
    1fnaJ7jvzOfz+YlN6HYCgB9EIpGhXbt2NGbMGJoxYwZNmzaNRo4cSSkpKeTl5WXkQmxzAKxy
    1j9asGABAaBp06bRhQsXqEWLFiSVSv/D3feXXC43JCYm0uDBg+nFF1+kKVOm0IgRI6hNmzYk
    EoksYDkfyc0xZ7wB7OvevTvdStlwoqKiwtG7d28H56JqNk3ohRdekCkUiq19+vShsrIyGj9+
    vCuFasyYMS4PTH2orKyknJycm97jLHA1YcIEYhjGERoaunjGjBnNsUAFAGbJ5XLznDlzSKfT
    1erDrl276IknnqDLly/X+v/CwkIaOXKkswDVwDvh6nrzzTcREBCAbt26+SgUim+9vb1p2rRp
    tGDBApo/f/4t25w5c6hVq1YklUr/Tk5OTgKAvn37erKLbXg8XoFcLifOgvdx8yKsHjZsGH37
    7bckk8muAejkroz06dOHsrOzqW3btnWFw6jg4GDTjh07KCUlpa5rNJZhmNNKpdJZiVDTyP4O
    5/F45UOGDKELFy7ccB7W1NTQ3r17afTo0aRQKAgAzZkzp5YSunDhQmfaJo0ZM4b2799Per3+
    hs/Mzs6mJ598kng8XgVqF0LzlIv/w5t4NW7WljfnRrh48WKhRqP5DwB69tlnyWg0EhHRH3/8
    QSqV6qZ9EwgEJJVK670WFBREu3fvJiKia9euUXp6OvF4PHNUVNSknJwcppnkxrucsntdfwYN
    GkQZGRnUunXrG71POefpaPYQVlRUFAYPHiwPCwubJBKJCjp16kTXrl1rlPH30ksvkUgk0vv7
    +3/SsWPHEIVCgaeeesojfeTxeIlgC2ARgB/dwo08bj46x6wYQBrnAakvrXW622MHgK2LU9+1
    YLBptgTgBMMwgY3scgyA/aGhobR06VIqLCx00QHMZjOZTCa6du0a7dmzhyZPnuz0zBqFQiF9
    8803tHPnTgoKCqLvv//eWVCSGIaxBgQE0JgxY2jr1q2Un59POp2OTCYT2Ww2cjgcVFBQQEuX
    LnXyxY6gGbhXyQzDlCxZssTRmMnx6aefWnk8XlkTXcy3Hu2YGMTHxydJpdKCd955h4qLi6lH
    jx70wgsvuEI+N8OXX35JXbp0aZAX5NSpUxQdHU1isbikTZs2D7dt29bTr5PKMEz++PHjr1M2
    iIg+++wz8vb2pmPHjl13LS8vz+nV2cPFI5sdzz77LFJTUyEWi99u2bIl5eTkNHheWK1WGjRo
    EKlUqt/79+8v//rrrz3dvSCBQHCyb9++5OfnZ8I/BE8+gM+feOIJysnJcXoq5nNCQwlg16RJ
    k0in01G/fv0ILIGMD5YDsXHw4MGUm5vr/Nxrbt6JxdHR0Y5evXo5rZ/GVBdtAeBkSkoKXbp0
    qUHjp9fraeXKleTj40MpKSkuQVNUVEQdOnQgHx8fWrVq1U0VDXfk5uZSx44dCSzvJdKDv0M4
    rq8/0NDWFE9Rg9CxY0ekpaVFKhSKv+VyOW3bto1OnDhB69evp/Hjx1PdSpb1beLr1q2jsLCw
    eq8/9thjVF5eTkRE//3vf8nX15cUCsXfKSkpcd26ebzQZwewWSf19sXJa3N6fW/Q9je3exwA
    /vOf/+Dhhx/Gpk2beDKZbJGXlxdNnDiRZs2aRe+8884t28yZMykmJoZUKlXGo48+6hsREYGB
    Az1qYwVya8DpqXiWUyp4YMmi7uG/d3k8HsOt9S11xnMNn8/nc7Lh8zrX3nBTYmZyoV/i8/m/
    qVSqxtSgkQH42sfHh9avX08Oh4OysrJo+fLlNHbsWOrfvz+NGjWKFi9eTGfPniWz2Ux//vkn
    Pf7448Tj8Wjw4MGUn59Pc+bMoStXrjhrBtEjjzxCe/bsIYPBQGfOnKHFixfTqFGjqH///jR2
    7FhasWKFS0798MMPFBAQQGBrkHi0fk5XoVBYtXHjRmNjFI6ff/7ZKpPJasCS6TyKmTNnIiUl
    RaRQKJZERUU5Tpw4QQ6Hgy5evEjV1dUN6t+8efOoZcuW13kN6oPdbqe33nqLGIYhjUazpmfP
    np6urfBWcHAw/fXXX7W+c+/evfTZZ5/RiBEjSC6X04wZM2j16tW1iKRERN988w2JxWJtnRBC
    s0MsFs9q2bIlXbx4scHzwmw208CBA0mpVP7eu3dvZTN0SwFg10svvUQTJkwgLp4Zyy3yNUOG
    DCGj0UjTp093hh4CuJBJ1tKlS4mIaMqUKQT2HAUlgHYikejqF198QWVlZZScnExgyxADQHeh
    UFgyd+5cZ5G5FWhcvYlJAoHAvmzZskaTmhctWkQqlYq++uorIiJas2YNKZVKWrRo0S29dnWx
    atUqEggENrCVFz2FB8HyNpqicNSgGcpwT5gwAWPHjmX8/f1f4fF4Fj6fT+np6ZSUlEQikahB
    ZNE333yTSktLnV4wqi+M8e9//5uIiEwmk5MT4tA4LEzpAAAgAElEQVRoNHNHjx7N8/ArvV33
    +0UiEQ0aNIimTZtG33//PRmNRlq5ciVNnTqVevToUd871oAl7N4RPPnkkxCLxe80cV6QRqP5
    vU+fPgqj0ehpWSZnGOa/qJ055cwyW1WnH05SeRDYdO66IUEV54G4WOea86jh9mCz6ZyZT2vk
    cnljeFS9AVSNHz+eLBYL/fe//6U2bdpQ3QJ2XFE0WrBgAZWWllJRURENGjSIfH19ad++fVRV
    VUV//PEH+fv7U/fu3enSpUtUWlpK8+fPp8jIyOvmCo/Ho6SkJNq9ezfZbDaaOnUqMQxTAw8X
    HUxiGObqp59+amqMEPv666+tAoGgBB7OpHCSg/z8/B6XSCTl7733XqMFrMlkomHDhlGvXr2o
    pqamQZ/JzMyk0NBQEggEhaGhoR3Gjh3rsbkO4Pv09PRa3g2TyUQvvPAChYSEkFqtJh6PRwEB
    ARQREUGLFi26zjUeFRVV12X3f1XhEAD4yulO5jJ6PgZLfP56yJAhZLVaae/evaTRaJwx105y
    ubzyl19+ISKiTz75hHg83mXOyp4eExPjyM7OrqtwqABs7dy5M506dYoefPBBdwumoSGHdd7e
    3nTw4MFGc4sqKyupT58+1LNnT8rPz6fevXtTeno6VVRUNPpZGRkZ5OvrSwDWwnNEwgfxD5nu
    nlA4NBoNwsPDk0UiUXZT+sXn8+lf//oXnTlzhoKDg294X3p6uquK7p49e8jb25tEItG5Fi1a
    RH/wwQcelRv1bMj0888/U0lJCWm1WnI4HFRRUUHFxcX06aefklAorK/Pb9xpuXE7CkdzyA0e
    j8cH8IXbd9nBZq6IAKyu0w89WLJ1MlgujPu1XIZhWoLNCKt7zsrbnFfkqzrzqjEEHwbAR1Kp
    lLZt20Y6nY447+pN5+2oUaOooqKCzp49S3FxcTRu3DgyGo00efJkCgsLo4yMDCovL3eG6G/6
    POdetW/fPlKr1fUaWrejWecR0YXt27cLtVptg6jAOp3OvnXrVp7NZruA6yuv3RbS09ORkJAQ
    WV1dPfOhhx7SPPfcc2A9WA3HwYMH8fvvv6N3795QKBrmDWrVqhW6desGm80WpNfrB4WGhnrK
    WhEB0Pj7+0MkErkvSrz77rvYuXMnJk2aBJVKhWXLlmHHjh0YN642F1etVjuLZgXg3j0p9E7B
    BiD/ypUriIiIwPjx4xmBQPA03IhgDocDycnJaNeunRQsObStRqNRRUSwVYlbtmwJmUym4SyR
    R7t27cpERkbC7RwHHoChSqWy15QpUyCVSlFYWGjlYsCNERwyhULRpIJnXl5eGD16NLKysrB+
    /XpcuHABY8aMgbe3d6OfpVarnetAeZuywh2lYCsrNgUVYGsSeAzz5s1Dy5Yt1VVVVW9aLJYm
    hWsSExPRs2dP7Nu3D9euXbvhfRkZGfjzzz8BACkpKejYsSMsFkusXq9/PCHBYyFvEeo5qbiy
    shITJ05Eeno6PvnkE+h0OkybNg29evXC3LlzYbXWW5PR//+63HA4HHYej3e1zp45Fv/wvOqG
    NPqDLRBYd/FqiKgtZ/XXtzH1AZtZ6IQdjSvGxgMQrlKpEBoaCmexw5vBbrdj/fr1+OijjxAb
    G4vRo0dj37592LdvH/bs2YMxY8agXbt2+PDDD/Hdd9/dsqhjVlYWiouLERoaCo1GA7ChWKGn
    FI5KAGt/++030+LFixmdTnfTm/V6PZYsWeLYvn27ntPkPFZp8dtvv0VkZKT42rVrr2g0mtQZ
    M2bA379xR1WcP38es2fPho+PDwYPHtzw1S0SoW/fvhCJRLBaren79u3z1BkZdgCW8vJysGeK
    /YOAgAC0bt0aUVFRkEgkaNWqFeLj4+Hl5VVXwUNNTQ3wD1P9/zoKSkpK7JWVlRgzZgw6dOjg
    BzazxN9Z3l6lUqFv377g8/ndATweGBjIYxgGmZmZUKvV8PHxkQN4Wi6Xt+vfv7/rIDoO8QAm
    9evXT9K3b18UFBSgqqrKhMZVf7UDKDAajXA/PKwx6N69O8LCwrB06VKEhISgR48eTXqOTqdz
    9uEqPHeAUz5qV3ttDPaDrQ7rMa9o7969maKiotF6vf7xJrnNBAK8+OKL4PP5+Oabb65bq+6o
    rq7Gtm3b4HA4oFQq0a9fP/D5fL7JZOq7ZMkStSflRn3vmpeXh8zMTFy5cgUOhwMXLlzA6dOn
    UVxcfMMpcF9uAESUh9qlvoPAFsTyqef2HpxSUV+hrydQP3cxgpND7vVmLA6HI68x3QRQ45RF
    DMPUezjedVaYzYbPP/8cBw8exODBgyEUCvHxxx/DZrNh+PDhOHjwIL788subzmuXpcQwcKYN
    c9+tR52D5m7XatloMpmWL1q0SDdp0iRcunSp3ptyc3PxyiuvYOHChWaTybQSbNVDj2HkyJE4
    efLkYyaTacz48eN5jSFh6fV6/Pjjjxg1ahQOHjwIHx8fp3bWYKSkpCAsLAwGgyH2ypUr7TIy
    MjzxWmYAly9cuID8/PoN5K5du2LBggVwWuB1cebMGRQXF9s9KaT/x5FfXV1tycvLQ2BgIF5+
    +WUoFIruADobDAZ88sknmDBhAux2O/z8/MIB9CAiTJkyBenp6Vi2bBmkUikfQJ8WLVp4lZSU
    YNasWThw4IAzpDcgNDS07eTJkyGVSnHlyhXo9fpqNK4cMQE4p9VqHdnZ2U16SX9/fzz44IMo
    LCxEp06dGq18O3H58mVUV1c7+S6e2njMYGPfWY38XDbYcz9MnpoMLVu2xKxZs1Krq6un2e32
    JhUucjgc2LVrF6ZMmYJjx47d8v4//vgDpaXsdHjooYfg7+8Pg8HQLjc3N6EhQr2B43vTEvZH
    jhzBxx9/jMuXL9/KI3hfbrAKRz6uP+W8D2qnyToRyXk56sJZuLE+Im7/uh4ThmH0EomkMXLD
    AeCCVqtFcXExgoOD0b59+4a5HEtLnUY70tLS8Ntvv6FNmzaIiIjAunXrUFbWMN/AAw88gKCg
    IJSUlKCyshJgC6J5/KRZOdhDr/Rvv/12vbHg+fPnEwCTSCSaj2Yo+vXggw/6qdXq3x9++GEq
    Kioiu91eKy3wRrHujRs30sCBA8nHx8cRGBh43t/fPy8hIYGuXLnSaO7HE088QQDI399/1q28
    PY3A43w+Xzt37lzXKasNhVarpaFDhxIn2FvdyQV6j3I4ACCZx+OVvPXWW1RVVUU6nY6efPJJ
    AkBqtZrUajXFxMRQUFCQq3w1n8+nxMREGj9+PAUEBLgIWEqlkgIDAykoKIhCQkLIy8uLBAIB
    zZ4925U1wpFMT+Gfg58aiiQA+Y899liTuBdERKtXryaxWExr1qxp0uerq6ud8yeP64+n0Qcs
    OdeOm8fn7WArLvbxdAcGDhwoDw8P31DfqbANbWq1Wufl5VXTiPtdKbI1NTXUu3dvAuAICAiY
    4sHzRPqDPQSMbqPdFbmBe4zDwW3+SWBDeXQH23kej9fYcvO9GYapeeWVV8hms1FGRgbFx8c3
    6PtSU1OpoqKC3n//fQJA77//PpWUlDhr8tyyxcfH05EjR4iIaO7cucTj8fT1KV636+HwAVtd
    rL9SqRQFBQXVe1NAQADUarXQYrE8BmA8PJhuRUS4dOlSV5vNljpu3DgIBALMnDkTO3bsqN/f
    aLdj165deOqppzB+/Hjbzp07jwN4s1OnTsMcDsc5Pp/faO6HWCxGUhIrk61Wa7tnnnnGU9kq
    e+12+48rV67E2rVrYTI1zLi7du0a3nvvPfzyyy8GsLniWbgPAChzOByVy5Ytw7BhwzBv3jxU
    VVW5XIvPPvsstm/fjgULFuDpp5/GE088gZkzZ2L9+vVYsWIFVq9ejeHDh6Nz585IT0/H3Llz
    8dNPP6F169aorq6GQCBAdXU1Fi1ahNGjR4NL7S0BS3ZsDM4B+M/u3bsdH374IZqiwPr7+0Mu
    l8PPr/FLTafT4f3338e2bdscYI8vP9sMv8VRsEfT3yqtwAS2GNufHrZacfLkyQ7FxcW9nRwc
    Pz8/qFS3rqLOMAxEIlGVt7f3lg4dOoz38vJqcN+0Wi0yMzMBAAqFAsnJyQDA2O32B958801P
    VXbdC5Y4WlvY83gQiUSQyWRQKBSugyrdqmc6YQB77tB9ucHOlTKwFII7BoFAUKFUKqsb+bEM
    Ivp948aN2LNnDzp06IBly5YhJibmlh8sLCxEWVkZgoODIZFIEBQUhNLSUhQV3ZoylZiYiJUr
    VyItLQ0ZGRlYu3YtHA7HQbD1tmq/1+1YiwDmqNXq9P79+4tGjhx5w8puI0aMQHh4OG/9+vXJ
    W7ZsaV1ZWdkLwDtgU4xuGzabLSUoKEiSlpaGFStWYN26dXj44eu9XUajEV988QUWLlyI8vLy
    TH9//9VhYWFbDx06dPXMmTPhVVVVkSkpKddxIRqC+Ph4iMVimEymmPz8fF+wpw7eLmoAzC4p
    KaGpU6cO3b9/v7xPnz4wGAwoKSmBXq+HzWaDSCSCUqlEcHAw9Ho9/vOf/+DgwYMlVqt1OYA1
    qBNHa06kpaXh5MmTDqeAayh4PJ4r7igUCh3jx4/H559/7unuVQO4JpVK42pqarB161aIxWIM
    GzYMQ4cOdR5LjpYtW2LMmDGw2+0QCv/hPA0YMAB9+/aFyWSCSCRykXm//PJLfPnll9i2bRs2
    b94MIoJcLgeXolfQhDCAFcAys9kct3Tp0gE2mw1vvPEGfHx8XEqzwWBARUUFysrKoNVqUV5e
    jurqaphMJpjNZpw8eRIGgwGbNm1CVlYWxGIxpFIpNBoNfHx84OXlBR8fH6jVakilUtdvVV5e
    joULF+KTTz4hk8n0I6ewetItygebafI62EOybnW2kgxsllVnAB+BPVTLI0cS63S6h6xWqxcA
    dOrUCe+//z62bt2KxYsX3/AzQqGwSq1Wb9FoNOvCwsIyKisrxdeuXXu1MYrO33//DYfDAR6P
    h4SEBAgEAhiNxlYZGRle8Ay3TQdgNrfuhwFQyOVyTJ06Fe3bt4dKpYJEIoFer4dWq8XOnTux
    Zs0aZ1iwmFM2vrqTcmPkyJHYvHnzvapzaMFmncTXMx8glUohk8mgVCqhUCggkUggEAggEonA
    5/Nhs9lgsVhgs9lgMpmg1WpRU1MDo9EIo9FYLxnT4XBcFQgEjc3xrQKwqLCwMGHq1KmRixcv
    Rq9evbB8+XK89NJLuHz5squf3t7ekEqlkEgkkEqlCA4OhlAohFqthlgshkQigUQiQc+ePVFY
    WAiTyQSTyQSDwYDKykpX/2NiYrBixQp069YNe/fuxeuvv46LFy/mgS18Vu4phaMzgOWJiYnt
    Zs+e7ejfvz8kEskNGSpyuRzp6el4+OGH0b9/f+G8efMeO3nyZAjY0skHbtfjJZVKFSqVCkKh
    EMePH8eoUaNQt9S4yWTCRx99hI8++shst9u/btmy5Udnz5696CbEHwUQ1bVrV8jljY/6RERE
    QKVSobKy0i8nJ8fLQwoHwMZRJ1ZVVR34+uuvZ2/evDlMoVCAz+dDLpdDJBLVmrhVVVUwm83H
    ALzJWTq2O7UqO3ToAB8fH7lAIIguKyvD8uXLG8yHsdvtOHfuHKxWq09eXl7QgQMHLg4aNAg/
    /vijJ7uoB7CfYZiHpk+fzktLS4NIJIJarYZAIIDZbEZVVRWsVivMZjPMZjNsNhvsdrtrgxAI
    BBAIBJBIJC6lIyAgALNmzcLEiRNRVlYGh8OBn3/+GbNnzzaBrdvRlA2yEMArJpPJuGzZsify
    8vL448aNQ15eHo4cOYKsrCwUFRVBp9PB4XCAiFwKG5/Ph1gsRkREBP744w/s2bPH9Q4AwOfz
    IRAIoFQqERQUhKioKKSmpiIsLAxr1qzBjz/+aLdarRvBpkUWe3D8JZyHc2Yjw0wysHV72oA9
    oXN1Azwjt5Qbcrk8uKysDF5eXpgzZw4CAgJw4MCBG3o1pFLpIY1Gs/DBBx/c9euvv5qUSiUM
    BkN3m80W15gvzsnJQU1NDdRqNaKjo6FUKqHX64NzcnJ84DkyfR7Ys30OAJjbsmXLsHHjxsHX
    1xc2m801n0UiEYKCgrB161aUlpb+5SY37HdKbgwePBibNm0Cj8djpFIpkpOTIZfLG3S8PBEh
    MzMTNpuNkUgk1AwyAwAMPB7vqMPheFgmkyE2NhZt2rRBq1atEBERgfDwcHh7e0Mul0OhUEAs
    FsPpKWcYBg6HAw6Hw6Vw6HQ66HQ6lJWVITc3F5cvX8b58+dx5swZ5Obmwmw22wEcKi8vb8qx
    8PsBvHb27NmPnnvuuch33nkHo0ePxmeffYYTJ064uBleXl4Qi8UQiUQQCASuuXDy5EkwDOOq
    ArtmzRrY7XaX0mQymVBRUYErV67gzJkzSElJQefOnbF27VrMmjULV65cyeOMiT31em6a8EKJ
    AJampqa2+/TTTyklJaXB8QexWIzBgwejRYsWmDhxYtLhw4eXgk0xOn17Hi/KdWpdCxYsgL+/
    P/h8Ps6fP4+CggJ0794d69evx5IlS0wikeijkJCQD6qrq3WPPPIIzp8/j549e0YfPXp0fGRk
    pPDxx5tEVodCoYBCoUBpaamUx+N5urKnHsB2hmEmjh07Nmzs2LGQyWSQyWQQCoUwm80wGAyo
    qqrCu+++i99++y2bm3h3TNlISUmBWq2WHz16dLrD4XjC19cXO3fubNwOwDAQi8WtcnJyPoyM
    jHzlwIEDeR7uph3A19euXXv8q6++ahsbG4uioiKcOXMGly5dQkFBAUpLS13au9VqBRHV2tB5
    PJ5rcUqlUqhUKgQEBCAsLAwxMTFo27YtpFIpNm/eDJPJ9BsXNrgdZXOy1WrN3bhx44SdO3dq
    ZDIZIiMjERcXh169eiE0NBR+fn6Qy+WQyWSQSCQQCoUQCoUQCASw2WywWq2wWq0uC0Wv16O0
    tBRXr15FTk4Ozp07h23btqGmpgY1NTVl3Ia+1IObH8Cmx00Cy/dqaqw9AMA8sLVTFqOebIzG
    yA2LxVIIsJlmVVVV2Lx5M44ePQqNRgM/Pz9kZWWBiMDj8eDl5bXV19f3taysrOzi4mLExcUh
    NjbWe9euXc9ZLJZGVVQsLS2F0WiEWq2Gl5cXlEolKisrZVVVVepmkBu/AphcU1MTNmPGDJSW
    lkKv18NqtUIkEkGlUrmHkS6BzSC6Y8oGd+4V4uLi4rKysnrExcVh7dq1CAkJaZgr0GrFpEmT
    sGHDhuiTJ092zc3N3d6/f39s27bNo3JDJBJ9bTKZHm/btm3c119/jZiYmOs8uDabrVYzmUxw
    OBwu5UMgEDgz3SAQCK57j9OnT2Ps2LE4ffr0UalUutFgMKAJvB4C8AOA6qtXr77/+uuvp+Tk
    5OC1117DQw89hOrqaly9ehXnz59HSUkJysrKUFlZCbPZDKvVipycHBgMBlfmikKhgFKphEql
    glqthre3N0JDQ5GamoqePXvCbDbjww8/xOLFi1FZWXmCMyZ+xQ1I5o1VOFQA3omOjm6/fPly
    e0pKSpM8JMnJyVi2bJlj5MiRydnZ2bPB8kCqmzob1Gr1oby8vNINGzb4zZ4922X1L1iwAIGB
    gQgNDcXHH39MAP7dtm3bD7Kzs3X5+fkQiUQIDg72OX78+Ls2m639K6+8gvj4+Cb1wakAMAwj
    k0gkoc2wNh8OCQmJHzdunIsv4lyw7ulPo0ePxoEDB7qYzeZEsCV5mx0JCQlQKpXy48ePv2k2
    m6e9/PLL0ueffx5isbjRC2bv3r28t956a2Bubi4vIiJiisViuaLVaj3Z3UsAVu3cuXNJZmam
    xGw2QyKRwMfHBwEBAS6L0/l7OptIJILFYoHBYKjVampqUFJSgr1792Lz5s2w2+3g8/koKCgo
    5VzTtxv7LeXc439rtdql48aN837rrbegVqsbzTW6kWeppqYGH330ERYsWFAF9jyYDVxYx5MY
    wlnPt0vsU3Ahlnyw/JKmuzgY5k+BQKArLS1VPP/889Dr9RCJRJg+fTp8fHzw0ksvwWq1QqVS
    7YuJiZmSnZ2d269fP5SWlkKj0Qj379//YmVlZaPJrEajEWaz+Tq54e3tHVZdXX3Uw+P+CID4
    nJwc5OTkuJRRp6vfarW6pzx25rxIx+6E3Ni2bRsYhkGbNm1aXb58eYVEIuny1FNPgYgaxB1w
    YuDAgTh06FBQfn7+8hYtWtDly5d3PProo402eG4GjUbzd2lp6b/Onz+/MCMjQ2gwGHD+/Hlc
    uXIFV69eRXl5uSucaTabXX+32WwQCoUQi8Wu5gxX+Pn5ITQ0FJGRkYiPj8fRo0eRm5url0ql
    n+j1+qu30V0CG3rMr6mpmfnxxx8PP3z4sEgsFiM/Px/l5eXQarUug6o+7NmzB3v21HZSOA0t
    tVoNX19fhIeHw+FwYP/+/VaTyfQdgAVgM1NuiMYqDAMkEkm/6dOn2zp27Hhbhyilpqby3njj
    DeukSZMeNRqNg7mYYVOEBoYNG3asvLx83cqVK1+prKzkdevWDUeOHMHRo0fx2Wef4auvvkJO
    Ts6p+Pj4hZcvX9bl5+ejdevWUKlUmrNnz863Wq3DJ02axIwZM6ZRvIO63htugxXo9XpPs6Wl
    AAZrNBrJgQMHsHHjRheHw2q1QiwWQ6lUIjQ0FAKBACqVKri0tLQf2BNEmz2P3tfXV3769OmZ
    Op1u2ujRoyXPP/88lMqmDUGvXr1QWVnJzJkzZ0BeXh4TEBAwSavV5nu4yz9ZLJZnNRpN6uzZ
    s5GQkABfX18oFIpanI2GwmKxQKfT4dq1azh58iTeeOMN2O32Pbj9cKHrK8Cewpir0+m86wtT
    OT0xTm+Mw+GA3W53eWScHBnn353g8/nw8vICp9QVcILK08pGDGf5aDz0PC9OefkTwIWmyo1O
    nTodMZlMuysrKwdUVLC1yJKSkjBgwAAsXboUVqsVEonkamho6KyMjIxcIsKjjz6KwMBAwZEj
    R8ZXVFS8brVaG00QN5vNTn4PpFIppFIpiEhkMpk0zSE3kpKSJIMGDUJUVBR8fHwgk8kgEAhc
    CnRJSQmysrLw3XffBV+5cqUfWG5ds8uNfv36oV27dgkXL178RKfTdROLxVi5ciU+/fTTRv+W
    FRUVsFgs0cXFxZ9ERUW9PGHChB07d+702DsYDAYKDQ3dXFhYOGbq1KlteTweysvLXQafSCRy
    eRdFIpHrT4FA4JLTFoullrfRYrG4PCC+vr6wWCwwm82HoqKifqmsrERJScntdvsCgDk2m+3B
    AwcO1GKO8vl8F4fDvc9Oz6jD4ajVX2eY2Wg0oqSkBCUlJTh71sUlLwRbffX8rTokaOQiH9Ol
    Sxfpk08+6ZEfcejQobyNGzcKf/3119FgD7tpUgXCv//+2xITE7Pw/Pnzoi+++GL42rVrlXa7
    XdinTx9edXU1NmzYYJbJZKsWL1586ZFHHkFSUhL8/Py8jx8/Ptdut4977bXXBK+//nqDq4ve
    aNI7PQ08Hk/q4bXZFsDDOTk5WLhwIby8vCCRSFyWt5PDsW/fPlRXV6OmpoYBWynzc3g2Bl8v
    KioqWmm12udsNpvk999/x19//YWmpvgxDAO73Q6TyQSj0djfbrdvBrDOw10uA3BeLpendunS
    BTweDyUlJS43o1arhdFodFkpBoMBFovFFUZxWilSqRRqtRqBgYHw8fFxEa84d+kpXJ+7fzvQ
    AsjNyspKzs7ORlVVFXJzc1FYWIji4mJotVoYDAY4C4Y5hYWTz+GcL1KpFEqlEgEBAQgKCkJk
    ZCR8fHxw/vx5gK3+q/XwWPMAPAOgtYefG895Rt9sagjgxIkT1QkJCbPtdrt3TU1NZyLi5efn
    46WXXsLx48fBnbC7NiUl5dCGDRswfPhwhIeHC3755ZdxFRUV852E06aEEZxEQTcFkLFarRIP
    j1GSTCbrOmPGDIwYMcLl6nda3k4OkkAgcHLCmDlz5vR3OByfw3MctBuic+fOoadOnVrm7e3d
    berUqfDz83Ov2ttouWGxWLBp06aoM2fOrPj0009Hgj211COoqqqCSqUqtFqtJ6urq9vGx8ej
    d+/eiI2NRWRkJIKCgiCXy11ywenF4PP5sNvtLq+Hs+l0OpSUlCA3Nxfnz5/HiRMn8Pfff4Nh
    mGOTJ0+ufukljx1fVAogl8/nx8TGxiIpKQmtWrVCZGQkAgMDoVQqXX11NoFAcF2fzWYz9Ho9
    SkpKcOXKFWRlZSEzMxOnT5+G1Wq9CpZUe0s0RuFoLxQK2w8fPtyuVqs9ckS0SqXijxgxwrJ7
    9+4km82WAuC/TXnOqVOnkJCQUNq5c+fXMzMzN8jl8gB/f//YzMzMCa+88opvTU3N1uTk5I2r
    V69GZGQkQkNDlUePHp1tt9snvPnmm4KpU6dCKvWcjiAUChV1Qx23KbAH8ng8/9jYWAwcOBAP
    PfSQa7I4FQ6tVovc3Fzs2bMHW7duxeXLl5PAFqb5rrkFBxGJAAi6deuG5ORk3G49AYZhkJSU
    hB9++IExGo3NcRy5DUD22bNn8cwzz6C4uBglJSWwWCwut6Ez7lqfC9pqtdYiUhERxGIxgoOD
    oVAoUFhYaAF7QJMnYQJw8a+//sLAgQNd5FZn5omTHe8Udt7e3hCLxbBYLC6FtLq6GkajETqd
    DhUVFTAYDODz+VAqlc5qkzlgUyI9iQiw59J4ukS2U6n+V1PHuqamBgKBILN9+/ajrly5MoDH
    4wUHBwfHHThw4FGbzSZRqVR/RUREfFFcXGwfMmQIRo4cyf/mm2+eLSsrW2C1Wr1vY73UL4wF
    ApmH5cYAmUzmf+7cOZo2bRpz6dIlaLVamEwml2dUKpXCx8cHcXFxqKmpsYnF4tZGo/ERsKd9
    NisMBkMLk8mU8tprr+Gdd97xjGXWti2GDx8ede3atQRPKhyccmhzOBwXfH19sXz5cnTt2rVW
    WNOpSDqbU8GTSCSuEKg7mdQJh8OBbdu2YeTIkXaDwZC1dOlST3ZbDyBHIpHg/fffR9++fWvx
    R9z77CS2cp49V595PJ7rT/fPHThwAEOGDFA1/GwAABPUSURBVEFZWVluQw2VxigcKQEBAaqH
    HnrIo4Kjc+fO/KCgIHl+fn6TFQ6GYTBjxgwcPnzYVFVVdTA3NxdnzpxhgoKCtlVXV2uCgoLO
    7tmzp5KIEBsbyz9w4MBLRPT8zJkzhVOnToVYLIaHJ6bcw4L1tMPh2JaZmdn60qVLPl9//bVM
    qVTyndqpk/VcXV1traio0BuNxmKwxZKKmktYdOjQASUlJe4WCQ0aNAgTJ070yPOPHTuGXbt2
    Qa/XIzw8HEQEkUiEhQsX4oknnvDEV+RptVp7RUUFv0OHDoiMjHQqo1AqlS4Gt9PVKBAIXMqG
    xWJxuj9RXV2NvLw85ObmIjc3F0eOHIHRaNQ3k4VYCgA9e/ZESkoKwsPDERgYCF9fX1dqq7uA
    cMIpTJx/Go1GlJeXo7i4GHl5eThx4gS++OIL5/M97UpPBRDVTNMwCmyp6CYpHAKBAP/+97/x
    zjvv5JWXl68AgMmTJ6uKior663S6UF9f311//fVXDsMwSE1NxYoVKx7XarXzzGbzbYc+6lM6
    hEKhwsNy41R5efnWBQsWtCYiX7BFF+sj/lgB6Ph8fqHD4TiOxp3hcbvGBR07dgwffPCBRwyV
    goIC2Gw2eLCImgtGoxECgeCKwWCwFRUVCU6dOoWsrCzk5uYiPz8flZWVtfhdJpMJNpvNldXm
    zgnz9vZGSEiIiwCel5cHq9VqUqvVRVyVTk/BAeCS2WxGYWEhzp49iwsXLiA3NxcFBQUuw8O9
    z07PqNOAcXpGvby8EBoaioiICLRq1cpZSRlgye0NCsM2VOHgAYiLiIjgBQYGevRHDAgI4EVG
    Rtrz8/NjucXQJPeo87TFuXPn4sUXX8SkSZPIbrefcQralStXgmEYhIeHdzIYDK+MHj1a8vLL
    L3tc2eAmvqwpnxMKhYiPj0dWVpZ7kS87Z238YLfbQ6qrq1Orq6vnpaWltVSr1di1axeGDBli
    3r9/v6CwsPAP/EPcKa47lq1atUJVVRVKS0tveRDPrdC3b18wDMOIxWKsW7fuppbb7QhkhmGY
    qVOnMuXl5RCJROQhZQMAwoODg3mrVq1C69b/ePudqV/u8Uuj0eiKtTrz6+VyuStU0anTP1WJ
    d+zYgWHDhkl1Ol2wh6eVAEB8YmIi5s6d6zrYTafTuVydzlCQ0wPjJLA6++0s+CSXyxEUFISY
    mBgwDIMBAwbgyJEjOHToUBzYMyAsHux3O7AnmDYHhGDrAX3b1Ac8/fTT+PvvvzFgwAD4+fnh
    6tWrWqvV+q1T+P7000947bXXsHHjxtiqqqp3rVarvyfndzMaKnYA64noR7Bnf6QCmJecnBwW
    Hh4u3rVrF2/EiBHYvXs3cnNz/wAwz263Z9UnNzyJTp06obi42Ml9cBARbdmyBVu2bPHo90gk
    EkeLFi1chsrZs2ebxM9yR0lJCRiG8TUYDPxXX30VNpsNWq0WPB7PVURNKBRCIpG4OH3cGVuo
    qKhAUVGRKxvEGaolIleWkMViETIM493QAo+NgJfNZsPs2bMBsGf6cGneEIvF15FaRSIRbDYb
    qqqqUFxc7Oqzk/PjcDhcfDeOi+Q83NHRECHWEIgAePn7+zMSicTTE0MQHBzMB3u6ngi3mV8/
    a9YszJo1CwDw+f9r7+pjmzr38/Pax3ZOHKeNndiktfNBSNKkBBgNl7TSuNGl3bS2jJtu9K6t
    OmlXRUPtKrRWY0Xirv+1EuoGAq3dLZcWdf1AzarSCdGqQEYhiLShdM6FJECwSYwhCbGPffx9
    PvfHeY9rAqX5OGb3VvlJ549Y8Tmvz3nP8/4+nvf57dmD7777Dm+++SZeeOEFbNiwwXb06NHn
    mpubF23ZsmVOehu3AxAdRBRFYQVBIDOMFp0UlO8XRdEzMDBgpS98CFpN/RqNPCVoXSD/vLGx
    0fvGG2/g0KFDOHz4MLq6uqyPPPIIefnll5dxHPenNLpWoOkeeADcA6BueHi4mn6epVGMH5qK
    ZGq2v7e7u7szFAr9KpfLEZPJ5BEEoWz//v0YGhoyJFLR9fgJIc9u3bp1ldlsJizLnvj0008/
    6Orqmq9nUwpgVV1dHUkmk3lxrMuXL2N8fBwcx4Hn+Typ61bbYnX+htPpRHV1Nerr69Hc3AyW
    ZeF2u0uSyWQHgG4DMwZuACs9Hg96enowMDCAS5cuIRQKYWJiIl9n1cdc+Az0cevAojPkfT4f
    GhoasGzZMlRWVurOgQez6257OzMD8BU5SPZSx2PORFd9Z9qFCxfQ29ub31JZU1ODrq4u7Nmz
    x6woyq8lSWorlrNBP2dpZmImc8ZFna376TOz0e+NFeDGJHUeqwA8Vltb696+fbvwxRdfsIcO
    HUJnZycefPBBbNmyZXk0Gv0FNFVclb4fi3TcoA6LNA03fj8X3IC2y6hCVVWk0+lqAObly5ej
    oaHBENyIRqPo6+sDlSbw0fuZUFXViLSBVVXVn9lsNrJ06VK0tLRg8eLFqKmpyWdG9RJsYSlW
    luWbSKM8zyMUCmFsbAyBQABnz57FqVOnrNlstp0Qsl81LnorB7DKYrGgtbU13/TT5/PB5/NB
    168qJI0WjrnwSCaTCIVCCIVCCAaDGBwcxPHjx5HL5R6ga1PEKIeDADBNr+MYVH4AwzCEekiG
    lms2btx4w0ve2Ni4NJlMPrJu3To0NTUZ+jv0hYmmsJmxsbEfAw4zgPUMw7zo9Xr/pK2trdzr
    9RKWZcFxHC5evKiEQqFsNBpNpNPpuKqqot1ur1y9erVn27Zt8Pl8+PLLLyGKonTw4EFm9+7d
    MJvNrp07d/5maGjo17lcLs4wDFtWVnaXy+Vy1NfXWxsaGkhpaSl4nkcgEJDPnTsXuX79+glF
    UbZDk5qesWWz2WW5XG5TU1MTysrK8rXA06dPG5UlwsqVK0EI+bkkST8fHh5GMplkGIb50IBF
    3Avg/oGBATzxxBOIx+Ow2WxwOp3weDxwuVxobGzMKzLabLa8pkVhKWVqagrj4+Pw+/3gOA6i
    KMLhcOhy6e3QiNZG5UdbANQdPnwYR44cAcMwqKysRF1dHdrb2+F2u/UWAvlS0HSw43kek5OT
    mJycRDgcxjfffINDhw7lnRQANfQ6RjkcFhShd9ItFjArDNpZU1hb7+vTKADvvvtuQzwe/2tV
    VYlR19BxtBA3FEWxXblyZSa48Utool4raHRZOC4VGlk5AU1qQKJBjdtut4sffPAB09PTo4qi
    mNuxY0dJW1sbWJZ1QttF9HfQavEsXaj0ezs9RR8B0AtgO2bJk+B5vmtiYuI3siwTAKwsy46N
    Gzdi06ZNhkyG/v5+rF+/HtFo9J+uXbv299CEIT8SBOFfDMAND4DlLS0t2Ldv301aIYqi5J1+
    PcOolydYls2LglmtVhBC0NHRkf/u+fPn8eijjyIQCLSrqmokbtQBaK2trcWePXtukjn/oTGb
    TCaUlJTkM7k2mw16aVG3cDiMxx57DH6/fwmABiMdDgFAPBaLQRCEeaembjixIOid5XgYy+q/
    yVKpVGdpaemihx9+2Chi1vf5S0oS0jFFEISZ3Pv1tbW1ne+88w46OjryUtkAkEgkTBMTE6Wh
    UKh0bGzMk8vl0NDQgPb2dphMJmzduhV+v/+6yWT64sCBA3/T0dFhee6557B27VrTt99+6wuH
    w76Kioq8J1tVVXUDMTaVSpm7u7vdmzdv/iue5/tm63DIsqyUl5eru3btIkYQRW9nsVgMTz/9
    NL7++mujpJbbANzjcDiwdu1adHZ2orm5GT6fL8/a1kHhdlGqXn7Ro5WhoSH09PTgyJEjyGQy
    TfQlPG3QmFcTQu5qamrCk08+iTVr1qCurg5OpxMsy856PmezWXAch9HRUfT29qK7uxunT592
    KIrSgTlyqX4grS+guJYrZgngrbfewvbt29ek0+lao86pl7iA78Wi6Jwi8Xh8JrjxlwDW3CY4
    LKHHDY10BgcHLYODg6CZjINnzpx59syZMzromADcS4/bxoj0vL8EcHK2Docois5sNtuol3R1
    3oXf75/zDpXC4PXChQu6tohbkiS9/GUUD6AFQE1bWxtyuRx6enoQDAbzWhy6+GQ6nc6TL/VS
    rJ5FsNvteR0Lr9eLmpoa1NXVwe12o6mpCYFAoNlg3FgBwN3S0gJCCI4fP45AIIDR0dE8h0Mf
    s67Jom+n17lrdrs93wqhcMxerxetra3w+/0uAA/MZA1hZgEcV4LBoDI6Oqp6vV6zEQsMIQTj
    4+NqMBgktAxQNOA4cOAAI0nSUpvNRioqKvLCOwY6MxBFcbZAuefq1atrjx07du/IyAguXboE
    URThdDpRU1ODmpoaVFdX47777gPDMEgkEujt7cV7772Hzz77LCNJ0m8B/I7neeu2bds2DA8P
    mzZs2IAVK1bgoYceQjabRSwWw8jICI4ePYpQKIRkMgmWZdHa2oqvvvoKyWSyD8DHc83q8DwP
    XcOgWBaPxyFJkpFOoheAWZeWrq2tRVVVVd7ZYBhmRtfSd7TYbDaUl5fD7XZj0aJFoGVHB7Qd
    GkYAhwlayVEQRdEaiURw5coVWK1WyLKcz2pYLJYb9DYKn5OegdI1AOLxOMbHx/MKq7RBnExL
    NwyMUamVUPymV1wxnZrVq1eb0un0ClmWjWqsBrPZnA8uCrfITs+w3AY33gawdgbOwa0sDU1J
    di/NZPwKtyaS/pjNFTf0DqNE//27d+82rG+SzquYts4YEqiYTKYmVVXLTp48iccffzz/3sxn
    LdRbDHi9XkSjURBCKgkhzYqiGIUb7QAsfr8fXV1dCIVCSCQS83Lu9J1teu8uOn9+Bk2GQTLC
    4QCAr0dHRzNPPfWUfa6iTreyTCZDAoGAMNsIe7bW29trMZvNHp7n8corr8xLc+OHHI5wOFz4
    oGf00mYymXdee+21f1AU5bwoipdpynKRzWZzl5aWusrKysrvvvvuErPZbKYpcTGZTJ6loPOf
    0Oqom6PR6OiuXbueff/996t1EatMJiPG4/FsKpWKpdPpKVEUx6FpUNhZlm0SRdGjKMoOaHXf
    2U46wvM8efHFF2E0r+dW2aNwOKyX3oyw/QBy586d+7NXX3211eFw3OtyuezV1dXE5XLB5XLB
    6XTCbreDYZh8XbOwnsnzPKamphCJRPTSihqPxxPpdPoqNF7M59D6qBhhCoB/U1V1aHBw8BeD
    g4NtVqu11uFwlFVVVVkqKipQXl6e7+dQuLOmsAyUSqUQjUbB8zw4jsPU1JSUSCRSkiRdATAE
    TQ7/v2GcJL5Kz6vC+G2x+Ww0ithk7PPPP7cyDFNj5DljsRjefvttVFZWIhKJYHJysjA7MdPF
    /nfQpOLPQ9slIEPjWnig8TvKaZZDx3gRmjbMb6GRbNMA/pG++39Lv0sK/jcDrRnYFDQ+yBQ0
    bkczzRjswNxLb2Q6dtKFq1hmyNwrKSk5KMtyVTgc/gtFUZbIsnyXqqrz4hhIkgSO4ySO46Im
    kylQVlb2PwzDnDRop4oK4CCAxWNjYw8AqMT8Grbm8TgWiyEWiyl0jpyFJkz4o57XrBwOSZL+
    dWBgwAljt84RaLXGU0UNgzhOEgShTxAE++HDhxUUQUmPRpaEEHJuhqQfCcBbuVzuGDQiVoKO
    y5bL5cpzuZyL4zhPKBTyUkCwAgjShazQSZgAsA1AdzQa7YhGo1X0XFehkbwmoYmqJSmYmDKZ
    jIeCx8k5evsnAfzz6OgowR0w2tjpfw0iU00A+A8A+1RV9fA838jz/JJgMFgNjShXTTMUFnrP
    Lfh+B5VII2qRzttr9LgK4CI06fRJzL5D7EzGvI8uFi5BEBZHIpG6SCSyiC4A1TQLYisYM1Mw
    Zn3c+pjH6XGZzqnrKE5Jsw8aidFVBNyIFRs3Ll26pEqSFLFarVNGEfkSicRNHWmpAFd8hpGn
    TOfvMepEJPTTUEfDSR2P6bjx1TQnYRJab5v/gra9uJJihI4b16fjBrRySjPmrnGRodc14c5Z
    wohHJ8vy5ZdeeunVEydO/DvHcYvj8fiyaDRaV1JSUpdKpVyyLDutVmsFIcSqqqpFlmWTLMvE
    bDar9JABCLlcLmoymaZKS0sjmUwmaDKZhgkhZy0Wy9jrr78e3bRpk1HviQqtPHoKWpmmDcBK
    aMq/lXSe3E3nh4keZnxPXFboXFMK8I6jzmcAmpL17ynuRWb60v6/eIu3uDFFt/r6emtVVZU1
    k8moPM9DkqRiXJe4XC5x//79QmtrK36qtmTJEgSDwXlvr52NsSyLjz/+GOvWrSuqb1PgaDAF
    h+5wSAWH7ngofwCPhJkGGvqYFTpWuQBApDs4rhJouynYIpw7C61fUKZYg3/mmWdw7NixeyVJ
    usFhMpqzRAiBy+WKfvjhh+EVK1b8ZHGjubnZNTIycg8ljd4p3IhMTk6G55uZr6+vRzKZRFNT
    E8LhMK5evQpBELB582bmk08+YWw2m6OiosKhKIolm82aE4mEKZVKEcqBUMrLyxWLxSJyHMfz
    PM+vX79e2rlzp6RrXjAMA6/Xi2XLluGjjz4q5i2xQSNzl1OHw1EQqNgo/kk0AMlRnNNJyDy0
    rHp2Luv2XNIr6h/jRH/++edx9OhRgWEYQdcmmC9J6YeAQ2/v+1M2hmFgs9nuqMNhsVjuxPWU
    ghftj8mkP9BxZYudhSimUVXIMCEkXOhsGE06L8SOn7hFZhoNG31v52vBYBCA1u1379692Lt3
    L2RZRn9/v5ROpyWWZbOEkOv6c9SvWfi3zq9Kp9Po7+/HqlWrEAwGUVFRgYsXLyIejxf2KCmW
    6fgWxYIt2IIt2IIt2IIt2IIt2IIt2IIt2IIt2ILN0v4Pz3h3nbZTKGUAAAAASUVORK5CYII=
  }
  set pieceImageData(Merida1,50) {
    iVBORw0KGgoAAAANSUhEUgAAAlgAAAAxCAYAAAAGAtdnAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw4uDOUTVwAAIABJREFUeNrsvXd4VNX2
    Pv6ec6bPJJlMGumFhDQCoSuELkooKiA1ggjSpKq0DyAIyhWEKwLXK0UMAnqlCChcFBThKkXp
    HUIIgfSQNqnTZ/3+OGfGEEJLMgn8vqzn2Y+Sc7Kzz9n7rL3Ku94NPJNn8kyeyTN5Js/kmTyT
    Z/JMnskzeSbP5Jk8k2fyJAvz/+JDz5s3DwqFAh4eHhgzZsyzVfBMnskzeSbP5Jk8kzoV0f+L
    D52Xl4fU1FRYrVbMmjULS5cufbYSGkiGDBmC4uJimM1mMMzf9j7DMHBycsKOHTvsRvHp06dh
    sVju9RIYBizL4qeffgIAvPrqq9Dr9bBYLPY+iQgsy8LZ2Rnbt29/9uKfyVMjEyZMgE6ng8Fg
    wH/+859nL+SZPJNnBtaTKxzHsXK53NvJyalo6dKlFe3bt0dgYOA9yiszMxOjR49GSUkJvLy8
    sGvXrmcrpo7lu+++e6T7vLy87AbUw2T37t1Pw6PLAfQAEAdAD2A/gD8BWJ7CaWQBDAAfEd8B
    wPpsZdfhy2VZ+Pn5ST766CNjs2bNEB4efl8nYciQISgsLITRaMThw4efvTwHOYRVHT2GYe5y
    3j7++GMcPXoURqPxoQ5h//79UVFRAavVes99Li4u2LZt2xOxbQJoCaAdABOAPwBceYqn0kPQ
    W7lPq4ElBtAYQJCgeG8BuCFMToNKcXGx+MKFC/PLysrcwsLCvnVxcfll27Ztpd27d8fBgwft
    97344oto1qyZk9FodNu5c+etV199FT/88MNTq6cBRAFoJmzoJwBkNPSgunXrFpCRkfGmTqdT
    EtHfC1MkIl9f3z1Hjx49IswZGx4ePkCn07W1Wq1UafNhWJZNFYlEX924cUMPAG3btg0rKCgY
    YTQapbY+GYaBSCSy+Pn5ff/HH3+ccvBjhQNoI/z/SQBJ1XwbUzmOm+Xu7q42m80oLCx8g4je
    A/A9AKryjbYA0BRAqWCEZTxha0sJYKLw/z8DKHm2FdedlJWVSQ4dOvROUFCQXi6X79m2bdvN
    Fi1aoH///pg/f779vpdeeglTp05l1q9fLz569Kjx5MmTaNOmzdOsr5oCaC7oq78ApDX0oCQS
    CSeXy1UWi4WpagxJJJIKAEaAj5jLZDIFx3GSagwsM8dxZbZ/S6VSEREprVZrdX2W18Oe6QUg
    WtinrwLIqmYuXgfwIQA/QT9dBTBd+N6rSgiAMAAVAC4BKHrC1pYMwBIAEgDjhHE+VQZWIwDv
    AHiNZVlvAIzVas0WNo8V1UxgvYpOp0NxcbFHUVFRv6Kiopfy8/N/CgkJ+bxz585/aLVa6+nT
    p9GmTRt4enqqf//994U6na5jy5YtFyQkJOwNCAig1atXP43KaiiABQqFIthisVgMBsMJAO8J
    BkCDSWlpaUB2dva7FovFWalUAgCMRiN0Oh0UCkX+v/71ryM//fQTOI7jcnNz+5aXlw93cnIC
    x3GwWq0oKyuDxWL5UyKRfDN48GD9Cy+8gE8++SQ0KytrBsdxUrlcDgAwGAwwGAzk5OR0C4Aj
    Daw+AJbIZLJwANDr9UkAZgH4b6V7IgG83bdvX/XcuXNRVlaGmTNnBpw8efIdAL8DuFPJEBsP
    YJZcLm9kNpvNJpPpT+HbOvuErCsZx3Fqi8UiExSvsxCF0z1lkSw/8BHFaOHfVwH88iRs6qWl
    pUxubu7zxcXFfQsKCkYHBAR85e/v/938+fNzhg8fjs2bN2Pq1KkwGAzc8OHDh+n1+rgmTZp8
    2qZNmyQiuiv1/hTpq9cBLAAQKKynU4K++rMhB3bz5s2o9PT0pTqdzvmu8A7HWQMCAlYD2C78
    W3Tu3LnpJSUlL1VxmBiWZS+IRKLpAMoB4MqVK63y8/MXm0wm2V2bs0hkDgwMXA5grwMfqR2A
    xQDaCgbWOQBzBT1kk2AAMwH4255B+E5mAzgNIK/SvPUHME8wsIwADgGYA+DaE7S+xACaCAaW
    1JEGliNEDSBRIpGY+/TpY/3ss8+sK1eupJdffpmkUqkFwCYAmoYc4MKFC0Vubm6JLi4u1KNH
    D3JzcyOJRJLh7e39bnx8vHPr1q0xevRosa+v7wKZTGZwdnYmlUp1Mzo6uou7uzs6d+6MPn36
    PE1zEgbgardu3Wjnzp301VdfUVhYGIFP56gacmCtW7eOk8vlRUOHDqVjx47R8ePH6dNPPyWV
    SkURERHvenl5wdvbG0TEKJXKRH9/f9q6dSsdP36cDhw4QC1btiQARziOUzZq1Aje3t4IDQ19
    USaTlU+aNImOHz9Ox48fp/nz55NUKjXFxMSMrQtHVjAkuCo/9wVwPDY2lr755hvasmULNW/e
    nAAcA+Bd6b5eHMfpvvjiC7LJlClTSNjMm1a6rw3DMBnx8fG0a9cuWrVqFfn4+BCAzYIXVlVp
    OKN+0/79AfzEMMz/GIYpFjzVw+DTnUOfou+jvTBug7AZkrA5HAHQuaEHN3bsWLFGo9liG5tI
    JDKq1epfwsPDO2VkZDC9e/fGgQMHEBoa+ppcLs/lOI7c3NwOtG7dOuSFF154GqNXTYQNmaq0
    nQCcGnJgLVq06KRUKkuqjo3jOIqKinpvwoQJSEhIwPLlyyUajea7ap6BWJY9LpVKXUaMGIHx
    48cjODi4l1QqNVS9TywWW2NiYsbVwbCZ++gFNwD7qhnjYSFIYpNuAMqquS8LQGyl+yLApw2r
    3rehGn0F1D9EyR9Af47jRrMsmwQgGcBbAF4Dn217KmQ4y7IV48ePNxQUFFhsG0hhYaF10qRJ
    FpZl9QBGN+QAs7OzGY1G85Farabdu3fTjz/+SHFxcSQWi3UeHh5rO3fu7N68efOOUqk0Jz4+
    njZt2kRBQUHk4uLya5cuXXw3bNjAAoCHhwd8fX0RFhaGwYMHP8lz0lcikeg3bdpk39AnTpxI
    4FO2oQ1tYMlksqIZM2bYx7Z//35ycnKiwMDA76Kjo4dHRkaOaNq06ZsqlepIWFgY3bhxg4iI
    SkpKqGvXrsSy7PXQ0NBxUVFRw6Oiol4PCgpaJpFIDP/85z/tfW7ZsoVkMlltDSxOiFBtE5TQ
    vypFPADgeQCFH3zwgf3vvv/++wQgX9jEbdIKQM6IESPo5s2bdPHiRerSpQsJ3qBPpftGSqVS
    07Zt24iIyGw202uvvUYALggRF5vy7ABgozCmr4Rx1EfY4jUhynOUYZgS8KnB4wAOAkh4SvRV
    Y/DpJ7pPOyNsHA0mhw8fZjQazfKqY1MoFLeCg4Pfmj59uqhVq1ahKpXqfKVNnLy8vNZMmzZN
    5uTkBG9vbzRu3BjNmzdH3759n/Q56SNEQKvORYpgfDWkgdVRqVQWVWNgWZs2bfoeETG5ubns
    xo0b5W5ubv+pbk1xHHc0NDTUVavVskTEhIWF9ZZKpRXVGFi11VcMgNYAlgP4TohMBVW63hI8
    BqnqGAsBdKyir6q77xr4dKBNBlVxUmztSpW/2wTAImFMi8FDV+pDJgEoAA+1sAitDIAWwPuO
    +IMiB/TX3cfHRzpx4kSLRqNhbRdcXV2ZCRMm0J49e6S3b9/uAWCLMBkNYWCRq6trakZGhiU7
    O5sbP348oqOjMX/+fNnu3btHJycnKzmOk7q4uHi999576N69O3JzczFv3rzOp0+f/vbixYs3
    nJ2drymVymsikeh8WFhY5i+//GJ54YUX0Lt3b7zzzjtPmsLSWywWc1pamlSv10On0yEnJweC
    l256EgZYGeBpS2vk5OQMZFl2gO3nZrOZs6UGAcBisYDjOLAsG5qZmfmvSr/PWK1WrnJqpCqA
    tIbSk2GYtWFhYT7e3t64du1a59zc3BgAIwGkgg/5V9y6dcu1sLAQRITbt29DCEGXVurnCoAt
    W7dunXjixAmZ0WjE7du3CwGsA5BT6T6txWIx37hxQ1RaWor8/HxkZ2dD6Etvs1EBrA8ICIgM
    CQlBSkpK5/T09DYARsDxacSdAH5mWdbNarVuFQzQYYJB+bSE3V/D33i56iQWwGBhQ6CGGOC8
    efNILpff5jiOKmN/KioqArOyspbt3r1bAcCroqIipvJ612q1w7Zv317m7Ox8xsnJKYNl2ZsB
    AQG5e/bsMbVq1Qrjxo3D2LFjn8Q50QMwV/Nzg6CznjixWq1MQUHBa8HBwU2IiGFZljOZTNWu
    K5ZlGxsMhhWxsbFGAGQwGAIsFovEAcNqB+DLSk7gIMHRGwsgU9D91el/c5X3fA3ALiHaw1Wa
    o28BpFeZn+oUbeV9JkwYU2UDrjuAUXA8aH43gNsAXMCnLaWCgVcA4OLToKwUAPa0atWK8vLy
    qKrk5+dT27ZtSfB6nRtyoM2bN28jkUhyx40bR2azmYiIrl+/Th07diSWZYnjOPPLL79M5eXl
    RER069Ytio6OJrFYTJ6eniSRSIhl2QqRSHRBqVR+Fhoa2nXo0KGyyMhITJs27UmbFw8Aezw9
    PS2vvfYaxcfHk0Kh0AP4B/jUUoNHsN577z37OsnMzKQtW7bQxo0b72k7d+6k4uJiIiIyGAy0
    f/9+SkxMrPbey5cv2/vctGlTbSNYUgDfNG7cmI4fP05FRUW0bt06UigUFvA4Kdv6/5dSqTR3
    7dqVunbtSkql0ghgFfiqwcriCmAa+LTgeQADhb9RWbwB/Ozu7m6Nj4+ntm3bkkgkKgcwBTze
    gQGw3N3dnfbu3UtFRUX0ww8/kJubGwmKo77ANy4AjoJPharx9IgUPGaGHtL2gAfyN4gQEZo1
    a9ZDIpEUVzc+iURSqlQqC1F9OookEomZ47hCiURyQaVSrffx8RnUoUOHRgAwdOhQVC4ueULE
    XdgMzZWeRQfgo4bWV/eLYNkawzD29qA19Sj31TKCJQLwRTX9mgUDC+DhIYlVrlsBbBW+6crS
    SHj/6eAzH9OquccPPHar6t9bXCmYM+c+z/thPeorlTDOE4IefmpEBOArf39/On/+/D0G1qVL
    lygwMJAA/Ac8jqVBZMaMGejdu7erQqE4GhkZSSkpKXT58mXq27cvqdVqUqvVBIAqp3pMJhO9
    /vrrFBQURLt376bExEQaM2YMNWnShFiWJbFYnOft7f1lz549WxAR2rVr90QpLpFI1JLjuKRK
    C3q3YHg1qFSXInSE1EGK0BnA/vbt29sNvL/++ovc3d0JPBi3siJaAD5VlidEahvdp89wIfL1
    O+6PS2wGviQ6G8CvACbgb9wcB2BLaGgo3bp1i4iIUlJSKCQkhAQvkaunafQE8BP4iiKvp0hf
    yYXv4GEG1n40IPZn6tSp6Nq1ayMnJ6czVccWGBhICQkJ1Lhx42rHHhgYSAsXLqSRI0dSs2bN
    SKFQEMuy5U5OTkdCQ0NHT5482dXb2xtTpkx50uamOfgKtCdKXz3MwKrLVksDSwkeHF9d3x9X
    ui8IPNThKvjU4H9x/zRsEIDLgj5yv889z4MvRCgAD2VYLOgHCE7hhvuMaQvqD5PlJOjSPxxt
    YNX1A5kB/JqVlTVk7dq18iVLlsDJiddLZWVltHbtWsrIyDAJD9cgoV4iwo0bN5j4+PiuJpMp
    JDk5Gb/99htCQ0Oh1+sxY8YMhIeHY8OGDWjbtm1lAwW+vr4wGAwICwtDVFQUXn/9daSkpGDb
    tm347rvv3K9fvz76r7/+6hQeHr48PDx8S6dOnZ6UNIncbDb35Diu0Ysvvoi8vDycPXu2GXjs
    zg9ooNSHTRiGwfXr1/Hjjz86xChlWRanTp2qbd/lAK4kJSW9uHHjRsTGxuI///kPtFqtrRTZ
    Jjng00n7AXwjGE66+/TZWFA+IsEIK7xPeN1D8BpH4e6qNguAc1lZWUMSExO5F154Afv370d2
    drZFUG6O5tTiwANgxwuGIANgjdAOovo0z5MkegA3H+G+m2jAlOfKlSsRExPjYbFYpJXXdO/e
    vTF//nwEBARg0KBBSElJuddVV6nw5ptvwsfHB9nZ2fjjjz/w9ddfK44cOdLh9u3brXbv3h0f
    EhKyZOXKlaeICE9IhbQCPA7Lv7JtA77g4PuG1lcAoFQq4ezs7DB9WF5ejoqKitqu7dT77NE3
    Kv37FoB3ATwnOGVmwZmrTvzBF/K4CI5UfjX3ZAh64SZ4rPWVSnrACp6yhqqJViWjfjgAncCn
    JKVCNPRFQVdrnxavUAVgnVKpNG3cuNEeQdi8ebNBqVSawFcRujTkACMiItoqlcrr/v7+5O7u
    Tn379iWtVkslJSX28RYVFZHJZLonCtKnTx97tKCyXL9+naZOnUrOzs4kEonKfXx8lvXr108d
    Ghr6JCireRzHlQ4bNowyMjLo5MmT1K5dOxI+rlfRgEcmtW7dOk6pVGo5jiO5XE4uLi7k4uJC
    zs7OddLUajUpFAoSi8UkFosttQSNRgM4JBaLDSqVysKybDn4MLzzfTzwTADFghFSbXBCUDql
    AHo/5J4T9/EafQH8IBKJ9E5OTlYhhfgf3F216JB9AMAQlmWzYmJiaPjw4TRkyBAKCgqyVRcN
    wdNxFFdH8GmP+0USsgE0WCleWFgYOnbs6Ofu7v5z5XSSSqWi7du307lz52jmzJm2SOo9zc3N
    jZYuXUrXr18ni8Vi121bt26lDh06EMdxpFKprjdu3Hjw7t27uYEDBz4J+up9VF+1loa/CW0b
    LIIll8u1I0aMoAsXLtD58+frvF28eJHef/99kkgkplpWEbYCXzhjxd+VsTtRfUQ9Ajw+SYv7
    V86OEoylMgD3q5QYAR5vda5S5KpqFOwX/J3+tYAvzqmP4oUgAF+LRKJid3d3q0ajsTIMUybo
    S4dUEdZ1BMtdsA6dysvLzZcuXbL3f/nyZXF5ebkRfFi+l+Dh3qnvD+S5555zunz58nSZTBb2
    z3/+E2VlZcjPz4dCoYBY/Hd6X62+F0oyYMAA9OzZExqNplpF+PHHH6N58+b4xz/+oUhNTZ1y
    4sQJaWRk5AJPT8+iY8eONYQ+0AB4h+O4aQkJCaolS5bA29sbvr6++PzzzzFlypTAY8eOrRCU
    2o6GiCoqFIrbMpnsd51O19tkMrG9e/dG3759IRKJahVxsgHcjx8/jo0bN8JkMsHZ2fm8WCw+
    V4vhXgbwhslk6mYymXyFtV50nwiVLT3nLHjj/6vioXEAIjQaDaPX65UVFRVh1fThIhheDPjw
    OlvNPRUAis1m887S0tLzgtF88D7e5aOKTGic0L++mqhBMIDZ3bt39169ejVCQkJgtVpx+vRp
    TJkyxfv06dOzBaPwZjWGmRx8CsMq9F3egBv6KZZlvyKiWURUFQNnAF+debyhBhcVFSU+fvz4
    pMLCwhcqfw9lZWWYM2cODAYD0tLuT9VVUFCAuXPn4quvvkKfPn0wbNgwxMbGYtCgQWjTpg2W
    Ll2KLVu2hGVmZn42b948Va9evb5+5ZVXzA1EqOwmRFOmonrMmz+AT4X1s60h9BXHcRaGYayX
    Ll1CUVEROnbsWOc8Y1euXLFH3BmGqc0zngbwBoCXBUesOfjIdv4DbAGbzvkDdwPWGcEI4oT9
    ojp9JRd0negB+uoO+HSkUdANt8ED6FMcHXQEMMfb23vEtGnT0LFjR1gsFuzZs0e5bt26IVqt
    1ghgMuqYJLkuDaw2AD5gGKaLl5eXIjY2Fq+88or94ssvv8xcuHBBevbs2ddyc3P7WK3W3wEs
    BA+MrTfJzMxsp9PpXhg8eDBefvllSKVSlJaWYv/+/bhw4QKGDx8Of3//6ncdmQwymey+fcvl
    crz55psIDg7GlClTJJcvXx6n1+v1sbGxHwwePLhi69at9fmooQA+kEqlA9944w3JokWL4OX1
    NzymVatW+Pzzz/Huu+8GHTp0aJVw/2rUM+vu77//nt6qVau3RSLRzMLCwjePHj2qev755zF+
    /HgoFIpa9f3999/jt99+g16vt2g0mh/9/f0/2LVr14XAwMDadJsmbLpi8JxVY8CnvH+rch8r
    EokYtVqNgoKCHkTkJyiTyh98aPPmzaHVapmzZ89GCN9j5bRaS7FY3MrFxQX5+fnsfTz3QeAB
    8vMBLKvFc0nBVx11BxADHqzOCR7tDcFYOg0+7WAB0EwsFkcOHToU4eHh9k7at2+PcePGYdKk
    SZFGo7G5oEQ5wSBrLfyNUMH4twoRvsvCOzyK+kvFseDxIiOsVmuH++hCEYBXwKdovwGPlavX
    o4wuXboUXVJSMtRqtdrxdP7+/ggNDcWpU6dQWlr60D7MZjOSkpKQlJSE77//HsOHD7fxL2H5
    8uWIjIzEokWLGl2/fv1jrVZrnjZt2ubAwEDrqlWr6jVYJ+wHr+HBQPYAACuFNbQK1afVHeex
    ajRX1Wr1uosXL04YMWKE80svvQSVSgUiqjW8gWVZmEwm/O9//8OVK1dMarV6u1wu/622S0ho
    EvD4p7FCBOloNd+DTb+8CGC1SCRKN5vNlZ2u8ErGVnX6Kgp/09Ew99FXPYUo1wrUvjJXCp6I
    Nhw8lkok7F+3hFZ5L2vGMEz/N954A++99x5ycnJgsViwYMECmEwmrFy58hWr1ZooRNPscRZB
    b4UIwSOLoA+vCwZhvTmGLQCcdnV1tcydO9d06tQpKioquieNptVq6cyZM/TBBx+Ql5cXgS+N
    bFdfg7x69SoCAwPfd3Nzo19++YWIiDIyMmj8+PGkVqspJCSETp48WSeg6l9++YUCAgJIJBKV
    hoSEjJs7dy6zadOm+nrU9gB+V6vVtHDhwmrnwiY3btyghIQEEovFevDp26D6VFivvvoqIiMj
    0bdvX0VgYOB4mUx2W6FQ0FtvvUW7d++mPXv21KgtWLCAPDw8SCwWF3l7e/+jY8eOHgAPGK4j
    YcAft0DgcWxVQ55t5XJ59qBBg8jV1VUvKJWqBvC16dOn05AhQ0gwMFyqKLxPQkJCrH369CEh
    5N6omk3pgqDkhtfiWULA46ZylUolhYeHU4cOHahz584UGxtLvr6+JJFIdILnuRp8ymymXC63
    7N69+541tX//flIqlRbwTM89AHwO4JpUKtX5+/tTixYtqHPnzhQXF0eRkZGkUqkIPCj2K9QP
    L5tY2Ghu49FBx5mChyutr2/j8uXLCA4OnsZxnAUASaVSSkhIoBMnTlBaWpqNO+2xG8dx1L17
    d7uuMxqNtGLFClIqlSSTyW5FRUV1qefAUAfwpK6P8xwG8MDo4PocaPv27fHqq6/KgoODR8nl
    8hRUStn6+/uTn5/fYzd/f39ydXWtXBWa7+Pj8363bt00UVFRdamvFgt/YyvuLdqIBY+fslVs
    Dqty3Rc8J5xtnL9Vo/P+TzBCSDDqfKpc9xYi+Vb8XclYo0AigE6Co5sCoFwikZhlMpmZZdkK
    8BCF/4GvTHxOiLi9LpPJTDt27KBbt25Rhw4dqEWLFnT+/Hk6cOCATV+NFe5tB+AD8Cz0WQB0
    IpHIInyHevCQgh2CsejwqlYnANvVarU1MTGx3Gw2Wx9mfJjNZvr2228tAm5gD+qpVJKIoFar
    /9m4cWO6du0aFRcX05tvvmkvlZ07d+49uKuaitVqpdWrV5NMJiOFQnGzZcuWbYOD60UXdANw
    0d/fn7766isyGAx3jSkjI4MKCgruoc+YPXs2OTk52eaj3llt4+LicOrUKSYiIqKri4vLUYZh
    LFKplFQqFTk5OT1Wk8vlxLIsKRSKpMaNGw8fOXKkpEmTJvjrr7/qetiTBGVRAZ4jprI8J5VK
    c5ctW0YdO3YkIaVRmaqhM8dxRV9//TUtWLCABK8orEo65OzIkSNtLO8XcDeuSiJErKyCp9at
    hs8QAeCATCajfv360a5duyg1NZUKCwtJq9VSVlYWXbhwgTZu3EhvvPEGBQQE2IgIb7Isa12y
    ZMk9a/+zzz4jjuOsNi8yMDCQRo0aRZs3b6ZLly5RdnY2abVaKioqolu3btGPP/5Ir732Gslk
    Mpuh6Whiz8FCmuRxjZMiIeVSLxigyMhI+Pr6fiLQxtCkSZOosLCQiIgSExPv2pRr0p577jm6
    ePEiERGVl5fT22+/TSzLklqt/l/nzp3964kjq4cQwazWEGzVqhU1adLkQc+xD/VMPtqxY0cQ
    EZo2bRqnVqt/Y1nWGhYWRps2baJr165RUlLSY7X//ve/9PzzzxPDMKRSqS6FhYW9NmnSJFGr
    Vq3qtOiHYZi3BX1RBiCB47iqQZLMSu/1W9zNvt4afAGP7XoywzCV37u7YNTYrl8WjLLKDuNM
    8KnBMpZla8p2KwVfVJOh0Wiob9++tHTpUtqyZQtt27aN1qxZQ7Nnz6auXbuSp6cngSdI3QHg
    W7FYbE5MTKRLly6Rl5cXubi40JEjR2jXrl2kUCisTk5O+4R7c9RqNbVt25YmTpxIn376KW3c
    uJE2bNhAH374IfXq1cv27d0Bn9KWO3RDZximaPz48RaDwWB9VAPEaDTS1KlTTQIDdK/6+DAm
    TZqkdHV1/TEqKorS0tJs3EjEcRxJJBJas2bNQ42ma9euUUpKyiM9Y1FRkY15m3x8fP4zfPhw
    1blz5xz5iBEAzgYGBtKuXbvIar17Ou7cuUN9+vSh2bNn33OtoqKCPv74Y1s0YRMaoCw9IYEn
    /46JiWkmFouvN23alNasWVMtx9WDWo8ePUgikZRFRET0JyL06NEDSUlJjhjyAJFIVCHQepyt
    En15XiwW30lMTKSPPvqIGIbJAA86tclbrq6upiNHjtA333xDHMcVAeha6foQhUJRsX37dpo5
    c6bNI6xsYHVlWTbbzc2NGIZJR83YkN0AbJPL5fT+++8/MNJp4x07deoUTZ482b65d+vW7S6D
    XavVUnx8PAEgjUZDU6ZMoTNnzpDRaHxg31qtlhYtWkQKhcJmjLo5aJn5CunOmhom5+srarJ6
    9WqJp6fnZgAUFxdHmZmZ9vc1Z86cRxqvQqGwvdNq29ChQ+20IykpKdSmTRtiGMYSFBT00Ycf
    fig6f/68Ix8xSnif1Y7Nx8eHTpw4QZs2bSKxWPyg5/wW9cyr+NZbvD/13HPPBTZq1Ggdy7IV
    AwcOJL1e/9jO+Lx584jjOItGo9kbGxvbEgB69uxq/Em5AAAgAElEQVRZ5xXVDMO8KqS1CDyV
    QmWsRCshWlO5oCCWZdnKTklldn1tFacuXvhZZfZ2vyoRsmThWg7DMC1rkkUVHNnCli1b0q5d
    u+w8lUREer2eDAYDmc1mKi4upuPHj9PMmTNtFCYWAPTmm29Sfn4+9enTh7p06UJ37tyhadOm
    3bXmxo4dS7/88gvl5eWRwWCwN5uUlpbSrl27qE2bNgQeszUJDqTEmalQKKx79+611CCNZlSp
    VFbczSPkEGnSpAmaNm3aQiqVZiYkJFBFRQXNnDmTWrRoQatWraL169fTnTt3Hjje0tJS6tev
    Hw0aNIgqKioe6Rn/97//UaNGjUgikZRGRUU58jwdEYBlcrncunbt2nsMKCKi27dvU1BQEPXv
    37/asZaXl9O4ceNIOFeuPxpAXn/9dXh6erpyHHfypZdeeuT3XFmmTp1KMpmsMDY2trVNETpI
    npNIJHkDBw4kd3d3i5AytIWMO4jF4rxNmzbRyZMnycvLyyKE0G3YhKU2/qojR46Qq6urqVLY
    XAZgc4sWLSg9PZ1mzJhRNeSuBrArNDSUXnnlFWIY5lwVb/FRUwYTARhef/31uypoHyZ6vZ42
    bdpEvr6+5OzsTPv27bNf+/nnn0mtVpOvry9t3rz5sTacsrIyGj16tC39M8FBkaKB+Bu0X5Nm
    FKJYDpXu3bujffv2QS4uLlcZhqElS5ZQQUEBbdu2jTZt2kQtWrR4JONqxYoVtGrVKlIqldXe
    o1Qq6euvv76rUlqpVJJSqcxo2bLl89UV+9RhmnY5/q5wq5bDKzU1lfbt2/cwA6tUmNd6lcmT
    JyMsLAzR0dG+IpHoQnBwME2fPp3mzp1Lc+bMeaQ2d+5catWqFclkMm1sbGwnpVKJl19+2VFD
    boO/j7wxC2kwrtK17Erv1AJgRqUo1/u4l0B0vADwF4GHDlS+fhV/02zIwZ9SYSO/TVYqlSE1
    GH87AKkRERF07NgxIiJKS0uj9evX05gxYyg+Pp769OlDEyZMoG+//ZZyc3PJarXS6dOnacCA
    AcRxHAUHB9OVK1do27ZttHHjRkpJSaGoqChiGIa6d+9Ohw8fJpPJRDk5OfTNN9/QhAkTqE+f
    PhQfH09jxoyhDRs2UEZGBhERnTx5kpo1a0ZCyjDOUZO21MPDw3Lq1KnHNt0vXbpk8vPzIwCf
    OTQG3aMHWrRoIVGr1StcXV2te/bsISKi9PR0unr16iOPt6CggNq1a0cvvvgilZWVPfJmNHbs
    WAJAarX65y5durgMGzbMEY8ZAOBi9+7d7WkEmyQnJ9MPP/xAa9asIS8vL3ruuedox44dtHfv
    XsrJybnr3jNnzpC/vz+BZ/htEDLYRo0aaTiOO/niiy9SaWnpYxtYU6ZMIZlMVti8efO2Dh5q
    YwA3Zs2aRW+99ZYNp9PJlkkQi8X5GzdupPLycurbty+BB0m7CwbUD126dKHi4mJKTU2l0NBQ
    EjYcgAeZ354xYwaZzWaaPn161ZD7KLFYXP7RRx9RQkKCjQjzcQ/t9gBwzN3dnQ4dOvTY79hi
    sdCGDRtIpVLRiBEjSKfTkU6no9dff52USiVt2LChWiP/YXLkyBEbPvMI7k9mWBtZgNqTQC51
    5KL6888/sXbtWnh7e0/mOM4IgDp06EAJCQmkUqlIKpU+0jhdXV3pzz//pLNnz9rY/attHTt2
    tOuBwsJC6tWrl+3318XHxztKBwTibiJRO/N869ataejQoTRt2jS6c+cOnTt3jkaOHElDhgx5
    ULpwE+oRH1dZnJ2dNSKR6GRt1pRSqSyMjY1t40hyarFYHMIwTHKlv3sbf2Og2+He8wYPAXBj
    GEYMHu9WddyfCgZWCO495PmasCcBfBVj5ZMGfpdIJJoapAbXi8ViWrFiBRER/fHHH/Tcc89V
    a3zLZDKKi4ujHTt2kF6vp5ycHEpISCCO42jJkiVUUlJCRUVFtGLFChKJRNSrVy9KTU0lvV5P
    27dvp/bt21f7nYnFYurQoQMdPXqUiIi+/PJLksvltvUnc8S8zXJ2djb88ccfusdVpidPnjR5
    eHhYwFdAOXrT7isSifJGjRr1yMZRVbl27RoFBgbS5MmT7cfrPIocPnyY3NzciOO4Yl9fX0ed
    tPoCwzDFixcvvgfvNnPmTHJ2diaFQkEMw5BIJCInJydyd3enylxltlThq6++akt5eTeEwnqK
    DCwPAMdGjBhBf/31l41BfYeQXu0kFosLExMTiYjo3//+N4lEohLwwEgvAOdHjx5NZrOZtFqt
    DbC8V/D23nF1dTX/+uuvREQ2A+uKEMEKBnCmY8eOdPnyZeratSsB+LoGIer2AAratWtH+fn5
    NfoeKioqaNSoUeTp6Ul//fUXnTp1iry8vGjkyJF3he4fRwoLCykuLs4Gen/eAXP2UR0YWCsc
    uaiCg4MRHBwcI5PJrtRmnAEBAZSUlER79+59YJpQKpWSbZ0SEX333Xc27riMwMDANj179nSI
    3wu+gvSeiNr27dvt6R6bMa/X60mn09HixYvvd7zMedwLqq4XUavVdWJgOVpfMQzjBp5+oSqD
    ugI8GPxOlWtaYZ7U4KlKqo77vyzLysGfxVr1kOfrghHtAR5XWfmYoO3iypxIj5iEAnAjLCyM
    bty4QVqtll566aWHvle1Wk0fffQRGQwGSklJoVatWlGzZs0oLS2NcnJyqE2bNtSkSRM6d+4c
    GQwGWrRoEbm4uDy03/j4eNJqtZSRkUGxsbE2YzW6upxmbeVqeXm58ejRo4/d17Fjx1BUVKSH
    Aw95bN++PaKiogK1Wu3/hYaGur/zzjtQKh//WDEiwg8//IDCwkJ06tQJVQCCD5RWrVrh+eef
    h8VicS4pKRncqVMnR4DivCQSiaJx47vx6RzHYfjw4fjXv/6FBQsWwM3NDbGxsVi1ahVWr16N
    7t2733W/XC6H0IcbHIeD+f+LlADIuXXrFsLCwvDmm29CJBLFgy/tv4tWoWvXrggMDHQCzzET
    wjCMd2hoKDiOg0qlQkhIiC0KGQqgV/PmzbkWLVpUTelJAIxxcnJqPnnyZKhUKmRmZkIIUT8u
    fYALAKW3tzfk8potRxstCQDs27cPP/30E4gIo0aNqjHFhlwuh7e3N8DTWGgcMGc5qP5A2seR
    TEctqLFjx8LHx8epsLBwlsFgiKxNXx07doSfnx/++OOPB7KCGwwGfP/99ygrK7Ov1ZiYGJhM
    Jt+SkpJBCoXCEUeYeAob+12i0+mwYsUKTJw4EQsXLkRBQQEuX76MadOm4e2338a33357P2yS
    6zN99WBhWbaUYZjcKj9+GTwGujpaBRsnViCqhyAEWq3WQPDcV1UjnYzg9CVUiurb9qTs1q1b
    P+4pD34AvPz8/ODl5YWbN2/iUfCBWq0Wy5Ytw+bNmxESEoJJkybh5s2bOHjwII4cOYKkpCRM
    mjQJMTEx2LhxI5YvX47i4uKH9nv27FmkpqbCw8PDprs9cTemrc4MrOMWi+V0YmKi5MSJE4/8
    S2fOnDF88cUXMJvNpwWrus7FaDQiJCRElpeXN5Vl2XZTpkxBdHR0rcL2rVu3RqdOnR7rd1Uq
    lZ0802w2dyopKQk/e/ZsnTsoVquVMRgM91xo2rQphg8fjiFDhsDJyQkBAQEYOXIkhgwZAj8/
    v7vutVqtqK6PZ1L9EgOQmZWVheLiYowcORKtWrVSgAc9NkYljpfQ0FB07NgR4HmmustkMqfw
    8HAQETiOQ2hoKBiG8QLwMsuyLaohtCXwef7hffr0YXv16oWcnBwUFBRYcffxOY8qZQB0Foul
    VmDaFi1aoFOnTti5cye2b9+Ojh07omXLljXuz2q1QuDecRQB6VkhOlZTKQBw0hGL6dSpUwgN
    DWVu3749rLy8vF9t5iUoKAhTpkxBZmYm9uzZ80j6zVaA4+Hhgd69e4NhGJhMpvjMzEz/DRs2
    1MsHZbVacezYMWzYsAHffPMNSktLkZGRgQ0bNiAxMRGXL1++rw+MJ+AInSdZAgICTAzDZFX5
    sRN4+pHA+7y/7oKBpLmPkfyiEP2qbj6ag8eVVo5WkdlsTj1+/PjjzhUDgHF2doZYLEZpaekj
    71PFxcVYunQpLl68iJ49eyIiIgI//PADduzYgdDQUAwYMAAXL17EJ598gpKSR+MZNRqNKCsr
    g1gshouLi218IkcYWHkAPk1KSsqeOHEiDhw4gErkZPeI2WzGoUOH8Pbbb0uvXbuWBx53kuuI
    BSUWi3H16tX4kpKSN/r27csOGzbssVl3rVYrfvvtN0ybNg03b95Er1694OnpWaNImre3NwwG
    g09+fn7H2NjYun5crclkMlZ3HplNFAoFoqOj0aRJk/tuqjqdDjdv3gR44GjxM7X04MAmgLSC
    ggJkZWXBz88PEydOhFKpbAn+WAmx1WrFzZs3kZSUhO7du0OlUjUGMFKtVkuKioowY8YMLFq0
    CCqVCnK53BXAaB8fH02HDh2QlpaG0tJS25p1BTDFz8/Pb/LkyVAqlUhNTUV5ebkefPXP40oG
    gLzs7OxanXmmVCrx6quv4saNG0hOTka/fv1qFCGuvP6ysrJseiXdAXN2BvzB1DWVfY4ysMaM
    GYP9+/e3KCkpec9sNteKZTc4OBhKpRKfffYZrl279tD78/Pz8dtvPKclwzDo1q0bNBoNdDpd
    49zc3HajR4+u68ctwkOY2I1GI7KyspCTk/MoTkDZM331YMnMzCSr1XqrGkPqOfBpvurSdqGC
    LqsO3+YCnmi5uqN3lOCLaMLvspIYxiyRSLJrcOZlLoDCgoICGAwGeHt72wybR5Lk5GSsW7cO
    7u7u6NevHw4fPoyff/4ZPXv2hIeHB9auXYsH7Z1VRa1Wo1GjRjAYDMjPz4ew9rIcNXe2UOAV
    b29vy7p166oFuFqtVtq4cSP5+flZwVcZJMBB5Y3Tp0/HwIED/T09PY83adKETp8+/VhYELPZ
    TFeuXKEPPviA/Pz8SCqVEsMwNGvWLPuZXo8jZWVlNqAzqdXq7X379q1rQFwYgGvt27en3Nzc
    +9JMZGZm3sODVVmOHz9uAxnvQDUh/PqQpwiDBQBDJBKJccuWLUREVFJSQgMGDCAAVolEQoMH
    D6amTZuSj48PJSQkkLe3tx2AHBoaSnK5nBQKBbVp08Ze6RUbG0sjR46kJk2aUEJCAg0aNIgA
    WFmWpTlz5tjxf5988omN56V1TWwjAPvc3NzowIEDteJ8O3fuHHl5eZG3tzdduHChVn0dPXrU
    tv724fGB+48qUXh8Yksb8D7KEQOaOnUqFixY4BwWFvYdy7K1xYiRXC6n0NBQG7fYI7UePXrY
    KRvy8/NtWDhyc3NbN2rUqLpOE4YKe8B9x8OyLEVHR9vOt3xY295Q+uppwWBFRESAYZhBuBcv
    ZasKtKL2+MTKVYiWan5eKJVKO9dg+G4A/goJCaErV66Q0Wik6dOn0+N8KzExMZSVlUW///47
    ubi4kEqlov3791N6ejpFRkY+cj8sy9L06dPJaDRSWloaNW3a1IZZbuTI+XMCTyRW0rt372rL
    s41Go40XSqdQKObDgdwlH374IePr6ztdoVCYbVUHBQUFDwXfWiwWOnPmDL333nsUFhZGEonE
    6OnpeTwuLu4bsVhcMWbMmBqTkS5evNj2MV3p0qVLQB0/sgjAMplMRitXrqyREVhcXExvvPGG
    rex5aEN5Wk+ZgdUFQNErr7xCJ06coJKSEtq3b5+9aotlWQoLC6PBgwdTo0aNiOM4+8+joqJo
    /fr1NHv2bBK4tAgAiUQi8vLyoh49epBGo7FVqVCLFi3oxo0bZDQa6eLFiza+KRuYtCYyHoBx
    4MCB91SePo5kZGRQ06ZNqUWLFpSdnV3jfoqKimjEiBHEMIwBtWN6fhSJBvDdfTab6qgZtoI/
    GslxA4qOHiKXy8tru7k5OzsXxMTEnBeLxcbH+T0fHx86e/as3RkTiitIpVKd7Ny5s4cD9NUn
    dbSpl4I/XLxB5GkxsATpCD7NTQ3UblchKH2cIM6nIpGIPvroI3uwoF+/fvcrerinaTQaOn78
    OF2/fp18fX3J39+fbty4QYcOHSJnZ+dH6oNhGOrfv7+dqmHt2rU2J+bfqCZFWBdeCQO+2mcK
    gO5eXl7Kbt26oboiAZFIhC5duuDo0aOynJycqeDL0VeCP4/QWper6Pvvv3cpKCjo26RJE27A
    gAE4c+YM/u///g8DBgzA/RiKCwoKsHbtWnz55ZfIysqqkEgkf2o0mu+aN2/+39u3b3cmooFK
    pRKVyNcez22OioJMJoPJZPJNT08PRM2wM/cTM4Av9Xp9x48//ridWCxGQkICnJ3vtmGFA0Tv
    +eWbN2/is88+w9atWy3gKRr2NJTCcmSpsgMkB0DpDz/8oD558iQiIyPBsix0Oh0YhkGrVq3w
    j3/8Ax06dMDp06dx6tQp5OXlISAgAJ07d0ZERAT0ej06deqEgwcPoqCgAP7+/ujSpQtatmyJ
    xMRELFmyBDqdDsXFxfj4449RXFxsB1mCr/wpreHYdwJ45ccff+zp6+uLBQsWoCa8R3K5HCqV
    ChzHPfCszgeJVqvFwoULsXXrVhDRL+APgHWU2AC9ecJ3I3mEb6sAfxcu1PkCjYuLU1y8eHGw
    TqdTAPy5dJGRkSguLkZGRsbDH4hhIJVKc5VK5cHGjRt/rVKpfJKTk9eYTKZHHkNhYSGSkpIQ
    GxsLhmEQGxsLkUgEi8USVFhYGCi8r7rUV+vB4wrvqhaVSqWQy+UQi8WQSCQgIhiNRhiNRuh0
    OlR5Jgv4M/b24Jk8sr6CYwpIHm5Vi0RaZ2fnksLCxz5C0gLgP2azuf+GDRsCn3/+eXTr1g2f
    fvopzGbzI+EMS0tLkZmZieDgYGg0GlitVigUCmRmZqK8/OFwT5ZlMWDAACxfvhy+vr44deoU
    Pv/8c+j1+gwAm3H3uYx1YmDJwOdnZ/j4+AQOHTrUOnjwYCYmJqZaI4RhGLz11lto164dduzY
    ofn2229fS09Pbwseh/UleKbYOhGpVOpnsVhCWrduDZVKhSlTpuDKlSuYMWNGtfenpqZi3rx5
    2Llzp4Vl2eMBAQH/joiIOLBnz56C7OxsaDSalgDE4eHhNTawQkJC4O7ujszMTJXFYolB3YP7
    kwBMzsnJmT9jxowXDxw4IOnZsycMBgOysrJQUlICg8EAhmEgl8vh6uoKPz8/FBQUYOvWrbh4
    8WKR1Wr9GjxhZllDGVdCGg0Mw9ToXdsMSIZhrMOHD8fmzZsdOeRCAPnu7u7+MTExKCgogNVq
    RdeuXdGlSxf079/fVmWCuLg4xMXdy0cnk8kQHx+P+Pj4ewzgyZMno2nTptiyZQtOnz6Nn376
    CTKZDEqlEhzHwWq1ZqHmYPA7AN43GAyeX3zxRUuTyYS5c+faqvgAABUVFSgpKUFubi4KCwuh
    1WqRn5+P4uJi6HQ66PV6FBcX49atW2AYBvPmzYNarYZMJoNcLoebmxvc3d3h6upq/3+lUnlX
    5WJ2djYWL16ML7/8EgaD4SR4UsM8B82XK3i8ydvgCxEeBZQpF6J9LwH4AvyZiXV6yLBOp/Mx
    GAwxNkU+ePBgfPjhh9ixYwfmzp0Li8XyIF2X5+rqut3T03NLVFTU2e+++07v7u4+02g0PhYv
    lF6vv+vEg8aNG8PV1RUFBQUu5eXlEQBO1fFcJIMvCJkPnr5EKpFIMGPGDPTo0QMqlQoqlQpW
    qxWlpaUoLS3Fhg0b8O2339p+v0hwBpeiHg/erU5nPS62twFFC/6YqMDq9KZcLodCoYBSqYSz
    szOcnZ0hl8shEokgkUjuMnhNJhNMJhMqKipQXFyM4uJiVFRU2PVCtVaSxZJpNBpr6hCeAfBp
    amrq4ilTpqiWLl2KXr16YcWKFTAYDDhw4ADEYjGcnJzg6uoKtVoNuVxufyZnZ2d4eXlBIpFA
    oVDAaDSC4zj4+/tj6NChKCkpgU6ns7eioiIUFRWhrKwMJpMJvXv3xvLlyxEQEIBff/0Vs2fP
    xoULF8oBfIr74DJrY2ApAExnGGZ6586dFR988IE1Li6Oexh9gVQqRevWrdGyZUv06dMHH374
    YcCvv/76D6vV6i6EjOvkQ9FoNDKRSCR2d3eHVqtFUlISxowZcw8tAQDcuHEDkyZNwsGDB4ud
    nZ0/Dw0N/feJEycyfXx8MGzYMLz44oshJpOpV6NGjdCuXc3Ppvby8oKHhwcyMjLYwsJCn2HD
    hlVWFnUlJwG8WV5ePmr37t3T//vf/3rI5XKwLGtfaFarFeXl5TAajbBYLCgrKwMRXRUU3R4h
    bVLvMmHCBDAMg4iIiID8/HzXlJQUfPLJJ5BKpY8c1WIYBidOnIDFYpGWlpYGbd68+VSvXr2w
    b98+Rw27CMDvRqOxxbBhw9CrVy8wDAMnJydIJHxgxKaIbM1gMMBsNsNiscBisYBhGHAcB47j
    IBKJIJVKIRaL7R58jx490K1bN9y5cwelpaVgWRbr16/H1atXjeDJAGszX6cAvG0wGJatWbOm
    Y1paGqZNm4aysjIcO3YMly5dwu3bt5Gfnw+dTger1WrfUFiWBcuyEIvFkMlkYBgG27dvh8lk
    gtVqhdVqtXt+IpEICoUC7u7uCAwMREREBNq1aweVSoVVq1Zh3759sFgsvwOYDh7P4AgJBH8A
    7GA8PokuA55QcTGAlgDmAkitq4G5ublpRCKRMwCEhYVh4cKFMBqN2L17932NK+G8zV88PDyW
    xcfH/2/37t2miooKDBo0SHPw4MFetvf/WBZPcjKMRiMkEgn8/Pzg6emJvLw8cUlJSeMxY8Zg
    /fr1dT0nZwQnfRSAmX5+fh7Dhg1DWFiYfR0BPN2MrXrsxx9/RFlZ2RVBX+1tKH0F8Efm7Nmz
    B+Xl5YxarUZUVNRjUfjY9p/i4mKwLEtxcXE4cuSII4dczLLsKavV2kokEsHHxwcRERFo2rQp
    goKCEBQUBB8fHzg5OUGpVEKhUEAqlYLjOLAsazckiche8WswGFBeXo7y8nIUFRUhPT0dt27d
    QkpKCq5cuYLk5GTk5+fDYrFYGYY5XlZWVtOqGosQ9XS+fPny9LFjx7rMmTMHo0aNwqpVq7Bt
    2zYEBAQgIiICXl5edh0skUggEons4y8qKgLHcWAYBgzDoFOnTvbzJU0mE4xGIwwGA0pLS5Gb
    m4tr164hPT0dQ4cORaNGjZCYmIgPPvgAaWlppYJxtba66FVtDCwRgHEsy87s37+/fNmyZQgK
    CnqsUAPLsoiLi8OXX36JWbNmqbZt2zbdYrGUgk8ZmupgId0BUJSRkeGlVquxcuVKREdHg2EY
    e4rlueeeg06nw7x583D48OFCT0/P+d7e3l+WlpYaunfvDrFYjMjISNHJkydHVlRUhA8cOBC1
    Od1cKpXC1dXVtul6eHh4cHh8/qJHkXzwIPWR/fr183jrrbfu+mBsBlZFRQVycnKwaNEinD9/
    PgnAfxtKWf3+++/o1KkTYmNjI1NSUpZxHBei1Wrx+eef1yh6JRaLFVlZWR80bty4YN++fYfe
    fPNNJCYmOmLoJgAbSkpKeq1fvz4sLi4OIpEIFy5cwI0bN5Ceno709HTk5eXZPTyDwWA3rmyb
    Z2UDyxahUqvV8PLygr+/PwICAtCkSROEhIQgLS0N33//Pcxm8/+Eea6t/AVgjMVimb9nz57+
    x44dk1ksFshkMvj5+SE6OhoBAQHw9fWFh4cHFAoFFAqFPY0jlUrtxqQtlWNL51RUVKCiogJ5
    eXnIyspCeno6UlJScPLkSaxZs8am8HTCc3wEHlPmCAkAsApAX9TuCB4JeLyPixB9qRMjy2Kx
    FFqt1hIAHi4uLjAYDFizZg3+/PNPiEQieHl54c6dO/b0mEgkMru6un7j6en5/uXLl9NTUlIw
    cuRILF68GM2aNetTWlpak8IH5ObmwmAw2L18NzeeWqqkpMRVJBI5JD0qRAN3AhjNMIzHF198
    gfz8fJSUlECv19sjKxqNpvJauw6+EKLBjKuEhATMmzeP+fXXXztbLBb/9u3bY8OGDY/NK7dg
    wQKsXr1aeefOnc7t2rU717VrV9OhQ4ccNWyjRCLZoNfre/r7+weuW7cO7dq1g5OT0wMjdDbH
    qrLTZHMAlUrlXZQyzz/PZ3ytViuKiopw+PBhTJ48GTk5ORfkcvk35eXltVlDOiHjVZCVlTV3
    9uzZvqmpqZg1axbef/99WCwWlJSU4M6dO3bHsKCgAFqtFgaDASaTCcXFxbh9+zYsFguWLl0K
    d3d3qFQqODk52ZuzszPc3d0RHR2Ntm3bgmVZlJSUYOnSpVixYgWKioqyhUzP+gdl3mpqYPUA
    MKt79+6yTz/9lPz9/WtcCejv749ly5ZRcXGxbN++fTPBU+zvrQOP8I5CoTh/7NixiOTkZPTo
    0cN+bcuWLdi2bRu2b9+Obdu2Ye/evXo3N7dlAwYMWPvzzz+bk5OT0aRJE+zcuRORkZGvpKWl
    jQ8NDeUmTZoEqbTmpzFIJBI7JkoulwempaXJHBja7uLq6tp47Nix1UbtKsu1a9dw8eLFdlar
    NQb8Qbj1KtOnT0enTp3QrFmzqOTk5FVisbj7+++/j+7du9c49H716lUsXLgwOjMz818hISGT
    EhMTD3Xo0AFHjx51xCNcBrD+zz//XDx48GBxeXk5srOzYbVaIZPJ4OTkBBcXF/smYQtby+Vy
    O2ZJr9ffFZ7W6/VIT0/HxYsX7ZwvEokE/v7+EKgfCsGf/1VXqbQkwWA4X1BQ8H8dOnRQL1u2
    DKGhoXB2dq7Vur9LuxuNKC0txa1btzBr1iwcPHhQKxhWX8JxZfZqAAvrwLiqHM2KB7AIwFTU
    QbpQq9VmcxyXDKDx+fPnMXDgQBuJLHr27IkJEyZg4sSJuHXrFliWJY1Gsz0mJmb6pUuX8mfO
    nAmNRoMVK1bgl19+aZaRkTHdaDTWiCujrKzMTrNTWV8pFIrAoqIiGeoQxlFFugIISUlJwSOW
    8LcD0ExwDupdxowZgzfeeIN54YUXXs7Jya10JF0AACAASURBVPlnUFCQ57hx4+Du7v7YOuv1
    11/Hn3/+KTl79uz//fXXX/revXuvJSLT4cOHHTJ2JyenM2azeWN2dva85ORkzkYJk5WVhezs
    bOTm5kKr1dr1UNXGMAxkMtk9zabfGjVqBG9vb/j6+kKj0eDGjRsoKCjQy2Syf4eHh988c+ZM
    bR9BL0SNUsrKyt5fuXLl80lJSVx0dDRSU1ORlpaG7OxsaLVau7N3vyjwP//5T7vBaHNwRSIR
    xGIxNBoNvL29ERgYiNDQUCQnJ+P777+36vX640Ik/NeHBUhqYmB5AngvKCjIY/HixVZ/f/9a
    A+V9fX2ZDz/80JqUlOSekpLynpC2yKlNn1u2bNE3adJkY1pa2gvvvvuu26JFixAVFYWLFy/i
    3//+N1566SXk5+dj3bp1kEgkP7Zt23bN5cuXzcnJyZg+fTqWLVuG4ODgnllZWUvVarXH/Pnz
    UYVZ+7FFLBZDpVLZIliuKSkpYgd9/04A+oWHh0vUajWOHz+OvLw8lJeXQ6/X29OFKpUKPj4+
    aNq0Kdzd3RvduXOnj/DurfWprLp3746ff/45+ubNm6tlMlnX+fPnY/z48faoSE2kXbt20Gg0
    ePfdd6OysrI+DwkJmdSoUaNDDvLAreABmK9dunSp7csvv4y33nrLHqr28PCASqWCSCSyR6ps
    H3PlkHvltKHFYoHZbEZpaSny8vKQnZ2Nq1ev4vfff8fPP/8MItoH4GAdP0eRYOi8QkTto6Ki
    Hotr5lGdDDc3N0ilUps3fB38UT+OMq4Y8IczD0XdHh7NABgE4Bz4o3Nq9c20bNmy/M6dO7t0
    Ol03g8EgsXFX+fj4YPbs2dDr9SgqKrJtkKfCwsI+SEtLy+/duzeKi4vx+eefw9/fP+T69euf
    lJSUxNR45xI2UZu+skU2iMjz1q1bEgcZWM4AXnV1dZVERETA398fnp6eUKlUd0XcS0tLkZOT
    g7S0NFy7dq1RRUVFX/CQiHrVV6NHj8aECROYQYMGvZqenv6pwWAIMpvNWL9+Pb788svHX0gM
    g5KSEhiNRs2dO3cW7du3D3FxcevatWtn/Ouvurcf9Xq9xcvLa0t+fv5rn3zySfTevXuRmpqK
    7Oxs6HS6BxJ4VtZX9xNbBN7Pzw9BQUE4efIkGIY54u3tvTMnJ6cude4BAJlms3n73r17I/fu
    3ftI77pyqxyZs1qtdxVQFBYW4saNG/jjj7ug0qngiVkfCcZQE+PoZZFIFDdu3Dhz69at6+wg
    0BYtWnDjxo3Tz5kz53mz2fwqgDW16e/MmTP4/PPPf/v1118/OXHixJwhQ4a4+Pr6IjMzExqN
    Bm+88Qa2bNmC5OTkLB8fn1V5eXnao0eP4p133sGyZcsQHh7+YnZ29kp3d/fGS5YsweDBg2sN
    ZLRhbQDAbDZLpVKpyEE6oCWA569fv46BAweiuLgYer3ejp2xfRwsy0KpVMLJyQmlpaUM+GMR
    1oEnoaw3Wb16tTojI+PDsrKyrsHBwSgsLMTy5ctr3S8RISAgADdv3ozMyclZfvTo0VfgGPJK
    gCeZO+nq6tp2zpw5aN68uf2CxWJBRUXFXfgrs9lsz/UDsKc+bEaYzYvy8/NDYCCPR+3bty+i
    oqJw8OBBi8ViOQagwgHPUQYgJSsrq31eXh5cXFyg1+tRXl6OsrIyaLVaVFRUQK/Xw2Aw2P9r
    MBhARPbnkMlkkEqlds9WrVbbQcsKhQIFBQVIT08HgFtwbEFFFIAJcMxBwBLw4PdfAFyoTUcZ
    GRmIiYnZYTabu9y5c2eIxWJhbDrj/Pnz+Pnnn/+/9r48OK7yyvd3l17Ui1rdrZZaUmu3LNuy
    5Y0YM9iCIQkMQ/SAYFM2zoRUnGAPW/FsCrJUCJUwNeBgx34wExweBJuE4AQK8HuOHZKBJLaw
    LdtjZEuydrXWllqtlnpR9+2+y/f+uAuSscGSusWb93Sqbl1Z5e6re+/3ne98v3PO74dQKASd
    Tjdht9v31tXVtanzuLCwEBUVFaVer3dfOBy+bTZduOrYBOS09aQ6woyJiYm0+SuWZW/YuXMn
    HnzwQdhstqs2t/A8j9HRUTz55JPUwYMH/1FBMvrm0l8RQuhNmzbd1d/fv8dmsxWXlpYCAGYT
    PFitVqxduxY+n88+NDT007q6OpSVle1HasplplgkEsHtt9/effjw4eNer7fK6/XCYDAgOzsb
    5eXlyMvLQ05ODjIzM7WyEvWspj85jtPS/2q5iZqaGxoawuDgIFpaWlQ5G2I0Gj/o6uoaTcPr
    6IbcLLFYDe6cTifcbjfcbvcV78NsNsNoNIKm6Sveh7qpHRoags/nQyAQ0GSkIGsOXjMj6XQn
    jA3APRUVFezmzZtJKjsnKIrCpk2bmAMHDqCpqekeyLwzY7PYEeKuu+5KVlZW/g+apnsCgcDm
    7u5uh91uz9bpdIt/97vf4eDBgyQzM/P1NWvW1L/55psoLy/Hnj17sHTp0i/39vbuy83NXfiz
    n/0MGzZsmHHn4Gc5Z4/HYzxzJuWk0AyAuxmGcbpcLqxcuRLLli1Ddna2NsAkSUI0GkUsFsPA
    wAA+/vhjcByHgYGBKsi8Tr+eS4eVTCatyWRygU6nw+DgIF588cWUUDWouxSDwYB4PO4RBCEz
    jbchAegMhULSyZMn6eHhYVy6dAlerxfDw8MagqjuEHmen7KYqdC0WuRuNBphsViQm5sLt9uN
    srIyVFZW4sSJE0gkEjEAHel6HQA6AoEAXnvtNdA0jUuXLqGvrw+jo6OYmJiAIAjajo8QAkKI
    Nj/UGo3JRfAMw8BkMsFut6OgoACVlZVgGAZ+vx+Ks+LSdC+0gjItTON7L4dcNN84GyTl6NGj
    ABC84YYbnpQkaXh8fPxumqYtHMeZduzYkZFMJtXmiaMFBQX/e8OGDQAAu92OZcuWlTY3N+8N
    h8N3zHbeqO/zchNFUZednZ0OxJ0FcLfH43F85StfIePj41RjY6NWrzgxMaHpdao1YYWFhfja
    174mvfPOO4sjkcgtCgI6ZzY6Oprr8/mecDqdxfv27cP69etT1kXY2tqKRx99NKupqelxnuf/
    qAQPqXdWkiQmk8kWANI999xD33fffVi4cCGys7Nhs9m0ppXpjp1EIoFwOIyRkRG0tbXh5Zdf
    xtGjR7lEItGyf/9+bNu2LeWAnPqMbr31VmzduhWLFi2Cy+XSyjJm8m7U+xgdHUVHRwd+85vf
    4M033wSArulsbKcbYC0EsPKrX/2qVFhYmPLJ5vF42FtvvTXW1NS0HMAiyAreM7Z3330Xa9as
    4bq7uw/ddtttRyorK3Umk8l94MCBR3bv3n2dwWD4z6qqqhd6enp4QKZqqKqqur6np2dvXl7e
    oj179uDOO+9MSwsuRVF6QogxDXPHDeDvVNjTYDDAYrEgJycHWVlZsFqtWiFgMBhEMBhUNRKh
    7PL/HsDv8DkyFqneFIqiSFatWoVnn312xkLBV5v0+/btw29/+1uJpJ9gqy0Wi8V/+MMfmgkh
    EEURZrMZ2dnZyM3NRUFBwZT6K/UghEypv1KP0dFRnD9/HqOjo4jH49p7IoQEkUbBYQAd0Wg0
    8fOf/9yQmZkJh8OB3NxcLFmyBEVFRZrjUtGpy2vJLq8nC4fD6O3tRU9PD7q6unDmzBmV5kFQ
    HFa6zK2gsunsoacgi+W+NFsk5dSpU/j2t7/dd8cddzx57NixX1qtVtPq1asXHTt27DGe5yvN
    ZnN7SUnJ7rq6unBdXR1effVVLF++PL+lpWX32NhYrSRJVCrmy5WmCUVReuNMSc4+23IB3BAO
    h8mOHTvoQCCAkZERcBwHURQhSZIWrDMMA6vVitzcXFgsFiIIgk7ZEKqEsXNioigaBEGwLlq0
    CDU1NZ9ZID4TYGDt2rX4+OOPMyKRiD5d9/DWW2+BoqhWALEVK1ZYvv71r2toO8dxGBoa0jrN
    VaRdRawJIcjIyIDBYNAQa7XYPTMzE06nEzk5OaiqqlLllyIZGRm9aQiutI0tALGyspK59957
    tY0ex3EYHh7W7iORSGjneDwOQoiGtKv3oDZ32Gw22O12uFwuLFq0SNXpJMq1rlmoeroB1lKd
    Tpe1bt06Og2IDiiKompqapgXX3zRyvP8stkGWABQX1+PRx55BKdPn46OjIzAbDaPlZWVPcay
    rMVms0VCoRDf3NyMrVu34siRIwWdnZ3/kpWVtXT37t2466670ueVKUrP83w6HFYQwPclSbql
    vb39S+3t7YWvv/66R6fTWcxms4bqJBIJieM4iuf5UcjprQ4AdZDb/oV0OihCCK677jqMjIxM
    SVtmZWXh+uuvn3YXzudZfn6+9nNRUREIIdDpdLj55pvx6quvpvJSIQB8dXU1vvGNb6CkpARu
    t1vbFaqtwp83d1R0KJlMYnx8XKvB6ujowEsvvYSmpqYo0sv7I1EURR566CFs2bIFLpdL47aa
    6bwnhIDjOI1H69ChQ3j22WclURTTOdYWQZZkSbeVQ05FzirAWrt2rTpek5IkXeJ5Hm+88ca5
    ioqKE4lEwuNyuQbOnTvnBYCXXnoJP/nJT8znzp37QSwWuzMVwdVnQoE0bYDMB5ZyQAjAk8Fg
    8O8/+uijNQAKAXgAZBoMBoiiCIZhQAhRUzn+4eHhQWWhU/0Vjzk2mqbR0NCA++67DyzLppQc
    uaWlBTRNqwzlaTODwRDkOC5x5swZyzvvvIPm5ma0tbWhr68Pfr8fsVhsSmDC8/yUrme1i1Cv
    18NgMMBkMsHlcsHj8aCyshIVFRVoaGgATdNRh8MRnY3m6efYAIDExYsXTW+//Tba29vR2tqK
    3t5ejIyMIBqNTulunlz0PrkbUj0yMjKQnZ0Nj8eDhQsXoqqqCqdOnYIyzqalNzjdAKskKytL
    X1pamraiwrKyMikrK0s/MjJSnKrvVLtS6uvr8cILL+DcuXNJhmGC0WgUVVVVaG5uRl5eHi2K
    4lYANz/88MOora1N9xzVTZcI8PI5jk8YqScvUnHIxc//AZmrrFqSpJ+Xl5evvf/++7Fr1y6U
    lpaS22+/Pb53714jz/NHIHdD9V8FtaIgpx1VuYCUvHuLxcIIgsAAAM/zegCUuutg2dSVeqjF
    4wAonU6ndzgcekmSwLKsRFFUqhf3xTqdzvyd73wHW7Zs+QTD5jgtJctxHJLJpJYeFAQBFEV9
    KkWoIkPZ2dnIz8/Xarp8Ph+amprckGkHetKxdgBY5XQ6jRs3bsTy5cs1rrSRkRFEIhGtFkt1
    XFdKdaoNHTabTWt9zsrKgsvlQl5eHpSiYL3f718JmQU5HXQlVZAbPtJtFsgyOn9MxZf9+c9/
    1n7+5S9/ieeee66HoqgeQgjuvvtuvPvuuyCE4Mc//vHXY7HY/ZIkpXS3exUESyeK4mz8FTMJ
    SVQ16gA5xfOhcmRAUfcoLi6u3r59O71nzx5jeXk5amtrsWvXLhIKhf4I4MeKv+KvMn7pVPor
    QghuuOEG+P1+EELAMIxE0zQJBAJT3lUqTQkqpbKyMhBCoNfrUVtbm5La1El+qQSA+ciRIzh2
    7Bh4ngfLssjMzERWVpaW1p+MUqsgptphqKLUKi1LQ0MDTpw4AUEQoNPpVL9g9/v9uZA7ldNh
    xQAMf/vb3/DRRx+B53kN7bTb7TCZTFNIR1Xk/UqIu4rUXbp0CadPnwbP89DpdGpAplM2AGkJ
    sGgANrPZTNlsNiZdniorKyvDZrNRIyMjNuWaKQ3mHnnkkSn/9nq92LVrF95+++3yiYmJLevW
    rWO2bduW0kX+Ks5LF4/Hp5NmpZSB9CXIBX25kItsacgdl91KJD+IT9rGywA8lJOTs+rpp5/W
    FjZCCLV161Yzz/PYs2dPbSKRGFZg9hFlTOQAyFcGUynkNndegeH7IBf0noPMtzUjGx4e3jIw
    MPDfBEEgFEWZkslkUUNDA7773e+m/NkrbcE2QRB2tba2hmiapoxGY93y5ctfSCFapwPwJYfD
    obNYLDhy5Aiam5vR2dmJ/v5+jdcnEokgmUxOqV9Sd1JqKkSldlADkqKiIpSXl2PJkiXIz8+H
    wWCwJxKJ65B6JQBArrP8ktPpRGtrKz744AM0NjZiYGAAg4ODGBsb0whTlbGk3YOKRKooqRpo
    sSyLrKws5Ofna12rBQUFsNvt8Pv9qyGzqwfScC+5yvxIt1FIk9DrZFmvaDSK7u5uEEJw7733
    5oqiuF2SJEua/NOnxncikTBM85mUKP5qifIuDIp/GYKcGlb9VUAJhsoBPOhwOJb+9Kc/5dxu
    t01d/DZu3IhoNErt3bu3Nh6P+wH8FjLXoV75btVflSljOKkEb5P9lX8Wz8Wu+EH09/d7RFHU
    5+fno6am5oqycLN5/mfPnkVbWxtjNBqLCCEJ5fdRpFbdgFbejXHlypVYu3YtysrKUFxcjPz8
    fOTm5iIjI0NruJl8BqB1PE8+x+Nx+Hw+DA4OauUAx48fR2Njoy2ZTK4G8Ld0AHEA1gJgqqqq
    UFNTg9LSUhQXFyMvLw95eXkaE/3l9wLgU/cwOUXq8/nQ29uLrq4unDp1CvX19RSANQDM15pB
    mO5Kxqi58HQZwzCU8v000ls7odnSpUuxe/fur0mSVL5hwwa4XK60XEeSJA2aJITQyWTyWp+/
    A8A2lmU35+bmLli8eHFGQUEBzGYzJiYm0N3dDa/XK46NjcVisVhEFMWwAgG7li1b5nziiSdQ
    W1uLH/3oR4hEIlJraytVX19Pff/730dmZqbj5Zdfftzr9f6TJEljAPQ6nS7TYrFYXC6Xsays
    jPZ4PNDpdBgfH4fX6yVtbW3hsbGxC5IkvQTgbcyg/iEej6/kOO6ekpISmEwmzbGr7empNKPR
    iOXLlxsAfJXneXR3d0MR/f63FF7GCWDF+Pg4Hn74YQSDQSQSCW0X6HK5UFBQAKfTCZvNptUv
    qPxSk+scxsfHMTo6imAwiLNnz+JPf/oTOI7TmhQEQaAh8wAZkfoC8TIAlV1dXXjggQeQTCaR
    kZGBnJwcFBcXY+XKlcjJyUFubq52HyrrPEVRU9jqVYkdv9+P4eFh9PT04Pz58/j9738/ue5v
    gbKwpjrAopFGMfkrWKbiT4V0X4iiKFRVVd0WiURWpxy+nJTCVusIlZ/ZRCIxHX/1z5AJWRco
    4/RT7lBB2yP4REvTCcBhtVr548ePm1pbW+lIJMK3trYyP/jBD2il+SMrHo//dwBbIKfkdcqz
    tyiL7ZXWjDCAi5Dr5H4/E38VCoUeHB4e3ipJEoaGhliO49w1NTXYv3+/Rr2TqjVi586daGlp
    sU9MTLwSi8V4xYf9L0LIzhSOLxuA68xmM5566qmrZmxUdFotW1CRalWdQZ3/6phRZcHU8fOr
    X/0K27ZtowVBWENRlJEQkmp/5QawXKfTYefOnbj//vuv+J9EUZyiMKF2b6v3odfrodPptNim
    pKRkyucPHz6MzZs3IxaLVUFOYV8TGjedAEsCMK6kO0i6gp9oNEri8TgFWTNJnAvP+Itf/MKa
    SCT+we12MzfddFParnMZz8Z0nt9CAI/V1NTkPPfcc1iyZMmUQvBwOIyhoSHG6/Vae3p6rL29
    vfkMw6CyshLr16+Hx+PBe++9h1//+tegKOqvkUjE/fzzzy9esmQJvve976G2tpauq6vL83q9
    eRkZGSguLkZJSQmKi4uRk5OjwamEEIyNjVHHjx+37dixY31XV5cOwAeYAWeZJEmS1WrF3r17
    sWrVqjkTePb5fPjmN7+J1tbWVKe5KwCUURQFu92OO+64A6tWrUJFRQWKioo0Jn2DwaChVZc3
    T6jPQBRFTX4iFAqht7cX7e3tOHv2LE6ePInR0VEoWpZ5SKFci2KroLCJr1u3Dl/+8pdRXV2N
    oqIiOJ1OmEymKZIZ12KiKCIejyMYDKK/vx+NjY348MMP8de//hU+n8+pXDPVhD8Ec9uokcQc
    8TE9+uijxkOHDv3jLEsMrrbB1RaZy/3VNOq8FgJ4VEHCPysANivHFPSvp6dHN0mS5wOO4wre
    euutpZd91j0N1DATwI3K5/48E3+VTCZdsVisdLKfSiaT8Pv91yQUPB0EKx6PAwDDcZxn0u9z
    CSGpXHNLAFQUFxdj8eLFGB4eRl9fn4baqESd4+PjWpG4IAjgeR4URWmbKoPBgMzMTNjtdjgc
    DuTn56OwsBButxv5+fmorq6Gy+WCz+erJoS4IdOypNKWAChyu92orq5GIBBAX1+fphqhou6h
    UEgraZiMvqsIu0qqq2qm5uXlobCwUEPBKisr4fF40NbWlg+5HCDlARYA+MfGxsT+/n5SXV2d
    lhxaf3+/NDo6CqRP7PXT25twOC8ej5erHCDp3HmqOwBMj/DyAoB3+/r6HuA4TtNGSiQSmoBl
    Tk4ObrzxRtxyyy3aAiiKIoaGhrB//37s2rULQ0NDXshaaotOnz79r9u3b7c+/vjjWL9+PbZu
    3QqaprVdq9rFduHCBU0+w2QyYcGCBRgbG0MgEOAAHJwN7E7TNOx2O7Kzs+dsFVTrDNIBhAJw
    uN1ubN68GbW1tSgpKYHaWHCt4wP4pI7JbDYjJycHJSUlWLFiBVasWAG73Y4DBw5AqVFcmOIA
    i1YcFut0OrF69WqsXr0axcXFsypyZxhGY3lWAy6/348LFy7A5/OxkBm5U43+EMyC5mUGFpyr
    AKuhoSF7YmKiKh3frS42kxGMGfqrdwA8MMuNeDdkfdqFkAWdZ4NIxmfprz51//X19bjzzjtT
    TuHj8/mu6fqzXIuqCSHueDyOp556Cm1tbejv79fYz2e64WUYBkajEQ6HQ+PwU8ZQCUVRSwgh
    qQ6w1gKwCoKA559/XpMnGxsb00oxZro2GY1GjVqmrKxMDXwzAPydMr4/98unu9JciMfjoX37
    9lnPnDmTctSBpmmcP3+eicViIQXSnROjKMrFMExWJBLB66+/nvIuNtU4jkNHR8dMEKwYgBc7
    Ojpu+ta3vlXJcRw3Pj4+JkmSyLKsxWq1mrOzs3UOhwM5OTnIyZE3jn6/H21tbWhpaRE5jvtP
    yHIhHwL4CAB1/PjxJy5cuFC4bNkylJSUwOl0aq2tioYTGR0dTU5MTEQIIXGapo1ut9s+MTHB
    hsPhP0LmKpNm+MzpaDSKZ555Zk4DrFgsht7eXtA0nWoEtg7A616v9+ZnnnnG/corr+gLCwtR
    Xl6O0tJSuFwuOJ1OOBwOmM3mKazualAriiKSySRCoRACgQACgQCGh4fR0dGB3t5e9Pf3Y2Rk
    JMnzvA8yuWVniu9BgsyBZmltba15+umnPfv27TOpshdlZWVaR6HT6YTZbNagdbUOZXKnTiQS
    QTAYxNjYGIaHh+H1ejEwMIChoSGEQqGEJElDAE4p4ygdaHWfgizp0zysBMwhOa/RaMynKCot
    dQzxeByXLl3SusiUze60pxmAFwHcBLmTMwY5IyFAbjqwQE7tfdbzPAu5+UbzVwC+B7m547PG
    Lw85JRhTFkO7cq1Z+asrWTQaRXNzM/4rml6vb5Uk6T96enrWdnd3pyyVLoqiJvzc19eHkydP
    gqbpSEZGxlmKovwp7iSklIB50Ofz5b/xxhupc4SSpBGQDgwMoL5eU5AbU8bXNdWHTzfAagZQ
    //777694//3305XToSBruzXN1WALhUIEgNje3i7u2LEjrdeapIkkTHNX0kgI2d3Z2bkKwFFl
    8RAAZEUikbzBwcEifNLinA+5noGGXEh6GLKYbs+k3dy/AzgfCoW+ceLEietPnDihFob6lc/0
    A+hVrjMEuajP1NHRsRjAVyCLXM5Yg42m6SGe51v/8Ic/zLlzoSiKysjI6E8xL9YFANsALIjF
    YlWdnZ2rOzs7l/7lL38pAJDFsqxFp9OZ9Xo9yzAMRdM0RdM0NbloVJIkIkkS4XmeJJNJnuf5
    CaW4dRyfFOt+rMyNTqSH++ccgIcAFIqiWBkIBFYFAoGlFy9eLACQDcBKUZRFp9MZWJZlKIqi
    aZqmFPSNkiSJEEKIJElEEASR5/mEMnYikFvyB5W/vwGy7qgX6WGkh/KsRgAUzAF6dX6uxu/4
    +HgGIcSUju/u6OjApk2bwDAMJElCMBhU5ww7zU1hE4DdAJYDOKb4EwFyoXie4quKlHdToPgr
    Rvl/70Gu7VRpL1TtuQYA9ymoRZYy/if7qx7lPAxZHcAEoBKydu4rs0Q0aXyxltLrJxKJUw6H
    Y7PRaLxOp9PdGI1Gl09MTFQIgmCnaTpTEASDJEkMdRn8PrmhZfJZkiTQNC2wLJuQJClMUdQI
    gCaWZRsYhjnn8XgaLl26NJJiTkmirENnIPM3robcOZyvBPHMLBFUUfFNQ5AlvT5WAv7TuEa0
    fboBVkBxvrY0D6aIMknmKsBq0ul0D7AsaxUEQRIEIW2clOpOX6/XJxiGaZ3mYHpNOT6L98UI
    ua7BpAyw0FUci6SgLqcV52ZSAqwJADGKopJXeQb1yk5wVvUtDMP8u06nO8gwDPkCnBXFsmxc
    kqRU8+dwkBm9G5VnZFHeRZYgCNmCIGTH4/FM5R0ZlLNRebcJ5eCUI6TMt4ASYE0gvdxXky2p
    BHCdAP6g/K0m5XAQQpzJZNKSTCb1Cjqk3of6WU45q+MpoAQhE4rD4uboPjoB/AZy6386N4Qt
    SBPj9lWQk269Xv8/rVarM9X3RQhBOBzWukFZloXVakVGRkZUr9cPTdNfHVDOwuf4K9Nl/mr8
    Kv7qI8X/2Cf5q5hyXG0un1U2l7Od66chpxi/CKMgc0KmDH0rLi5GOBwODg4Ovg/g/e3bt5uO
    Hj1qY1k2l6Ko3MHBwUyWZQssFouboiiTJEkGnudZjuMYBUWVDAaDqNfrBYqi4uFwuJ/juL68
    vLywokQw5HA4gu3t7UmTyYRkMonbbrstXb6qXjmMAFyQG3XcSiBfoPxsVXyVXkE09cpzTV52
    RJXYw6ccaof+8Ez873QDLIL0Mi9/IcaybCgUCr03l9dUO62m+7FrXOQ5BS24FhOuFMx+ToCZ
    igUyQgiJfFHvfI6K6qOTJmzrf+EpjZYAuwAAAWhJREFUogZ/Y0gvg3yqbQLAU5iZ5up0TMAc
    MomXlJT0NjY27jh8+LDWSj+J7w0Mw6RUfSKZTIJlWSQSCZw4cSJd/io4jWc9MoPxOyuTJOkN
    AG/g/xHr6ZGTGQcPHsTRo0cxODgYEwQhJoqiTx1P4XAY4XB4ir88dOgQaJrGxo0bP4n+lLFm
    NBq1AnJRFBGJROB2u8GyLLq6utDVlfbQQaXluJzwl1GCKuayg1JQKvUQlDGbsnIFCvMGlmXx
    2GOPwel0YnR0FD6fT1OUT5epQsuvvfba/AuYt3mbt2u2BQsWfMo/pXPDoEqKlJSUqNIn/18Z
    IQTl5eUaB9kXYUajEQ8++CD27NmTtmusXLkSHo8HwWAQFy9exOLFi2E2m7Wmp0gkgrExORni
    cDhgt9uRmZkJvV4PjuPQ1taGvLw8OBwOtLa2or29fX6yztu8zdu8zdu8zdu8zdu8zdu8zdu8
    zdu8zdu8/V9t/wfqGMCWUkeZjQAAAABJRU5ErkJggg==
  }
  set pieceImageData(Merida1,55) {
    iVBORw0KGgoAAAANSUhEUgAAApQAAAA2CAYAAAB5hAnwAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw4jclRv6gAAIABJREFUeNrsXXd0FGX3
    fma2ZUvqpveQENLoEFoIJUgvISpFEAUiKqAC0kH8BEFBURA+RRQ+/SFNikgREFAIUgREaaGF
    9J6QviXb7u+PmY1J2CCkETx5zpkDJzs7+77zztz3lufeCzSjGc1oRjOa0YxmNKMZzWhGM5rR
    jGY0oxnNaEYzmtGMZjSjGc1oRjOa8VSCab4Ff2PVqlXIzs6GTqfDunXrwDDNt6cZzWhGM6pj
    wYIFsLW1xfz585tvRjOa0YxmhbI6Pv30U8yYMQMMwyA8PBy+vr74/vvvazz/iy++wI0bN5CV
    lYU9e/Y038AngJiYGOTk5KC8vPyBz0QiEdzd3ausTc+ePaHRaCxei2VZWFlZIS4uDgAwaNAg
    FBcXQ6fTPXCuUCiEu7s79u7d2xRvixiALwA7ANkA0gDQv2jZpQC8AKQDUDe/BY2PLl26QKvV
    wtfXF/v27Ws2vpvxtEIJwAZAAYDi5ttRNwgbWXkVADDxR5ODXC5HRETEgLCwMCeFQvHLzp07
    M+/du4dvv/0WYWFhD5z//vvvIyMjA8HBwSCif6tQFQOQ8Bu3sck9wEKhSCwWu1p6lkUikVEg
    EOQAKAcAIkK/fv3sTCaTvaXnk2XZEolEkl/92gzDCC38rom/traR10LGr4WuhnNcAMwGEA3A
    FkAGgK8A/A+A5iEKmhhAWVNcYwsYAGA5gEUA9jWL8caHo6NjYFlZGY0aNSrRzc3NOG7cOGzd
    utXiuYMGDUJ2djZsbW1x8uTJf+stkfCHqim+Q0OHDkVmZiY0Gs0D+5RYLK4wDMxysnv37igu
    Lq7R8JbJZLhw4QIAICoqCgUFBTUa9T4+Pti/f/+T0G0MD/lcAeAlAGN5pTKdl5P7HiJbzTqM
    4Sl5JgMAxADYD+DWv0WhdAAQBaAnv3AFAH4DcBzA/aZ093/99VdkZGT0yMnJmZuenn7F19d3
    i5+f366wsLCcfv364fjx4xXnBgUFwcfHx9XBwWG4ra1tHMMwtyZPnoxNmzb9mxTJgQCeBeAI
    4C6AbwH8hSbk7crPz29x7969r0tKStxMJhMRkVnoMXK5vEggEEwD8Lv5/MTExJeLioqmm0wm
    mEwmAgCGYSAQCFiBQLDd2tr6HfP8cnJyvDMyMr5RqVQPXFsmkxULhcK3+Ge5oSEA0AfAOACe
    vPDbCuDXapuXGMAMoVA4Mzw8XODj44Nr16453bhxYzkRlQL4rtrayfn1HQ7AGsBNXvG80lQf
    SoZhQETeAFoB8P6XKSUC3giQ8oZKUVNV8BMSEp7Pzc0dN2/evIOurq47evToceXs2bPG9u3b
    V/HaDxw4EO7u7uLi4uJWUqk0KSQkpCw+Pv7fpkgO4d8jJYAEAN8A+KMpyUmRSCSXSqXtWZaV
    WvhMJxKJ/kIlD52VlVWAXq/3q8HwTrOysrpZSSGVSaXSjgKBwMqC4a0XiURXABQ20lTDADwH
    wI+Xk3sA/FltLQQAJgFYwctAAAgC0Jp/3/ZY0JMi+XVWArgBYDeApKb6ULIsC5PJ1AfAMt6R
    cOvf8LIFANgOoFQqlZKLiwvJZDLiPSHfAwhsagP28PBYxLIsCYVCEggE5XZ2dscCAgKGvP76
    68Lw8HAAwIgRIzBgwABnd3f3b8VisdbV1fVU9+7d2/6LhCQL4GUAuS4uLhQUFEQ2NjYE4DKA
    Tk1poD169Ghna2ubpVAoqGPHjhQeHk6dO3cmpVJJUqm0JCIiIio2NhZDhgwBEUGpVK5gGIb8
    /f0pPDycwsPDqVWrViQQCIhl2a/s7e3ZESNGYOLEiWjdunWIXC7PtLGxqXJtBwcHsrKyKomI
    iBjUSNMcBiDVwcGBgoKCyN7engCkABha7TxfALfCw8Pp7t27pNPpKC4ujjw8PAjAAV5prCwk
    5wAocXNzo8DAQPO7eYEXrE0NVgACGIYJEQgEHwLQ8xtCCICWvBL2NCMUwAcATvNG228AVgFo
    0xQH6+7uvoRlWWJZluRyeYKXl9c73bp18waA4cOHAwBGjRqFF198UeTv7z/V3t4+oUWLFqvH
    jRvn0Ldv33+TnJwEII9XWMzHFQDhTWmgPXv2bO3m5nZHoVCUVj+cnJxSoqKiepjPJSLG399/
    maVzFQpFma2t7WceHh6s+fz27dsHOTs737Z0vqOjY1qfPn2iGmmaXSop8ubjOm+MV4YbgIvV
    zjMf+8GFwCt7Jcfxyqn5HCPvEGvZVB9MBwcHMAzzBi8n5/wbXjYlgJ1CodA0bNgww549e4zn
    zp2jvXv3mkaMGGEUiUQEYC8Ap6Y06BYtWkwVCAT6ESNGUExMDMlkMhKLxdkeHh4L+/XrZ9+p
    UyeMHTtW5Onp+b5IJNL5+PiQXC4npVJ5pG/fvm5RUVGIjIzEsGHDnua1cwPwe0hICB07doyS
    k5Npw4YNZGdnRwDWAxA1IYWyrbW1dUb37t0pKSmJ8vLyKDs7m0aNGkVWVlaFvXv37l3loVQq
    l0mlUtq4cSPl5eVRXl4e7d69m2xtbYll2Q2Vz23fvn2QTCZL7927d8W1MzMz6bnnniOxWFwU
    ERExsB6VJS/+kFT7zA7AT25ubrRt2zZKSkqibdu2kaurKwE4WE34hQHIiImJIY1GQ0REWVlZ
    FBoaSgDOVHvXAgHcbd++PZ08eZLu3btHy5cvJysrKwKwkt8sK0MGwAeAOxqXKlNZqb4NIJ1l
    2WJesBfygj4BwKin+H3rY2EjrKyc9G9qA/b29p7BsqzRPE6BQGCws7P7NTQ0tM/27duZkSNH
    4vLly2jZsuVoqVSaB4BEIlG5h4fHqmHDhlk9gRBoQ8nJCzWs2+dNSU527dq1na2tbZalscpk
    ssLIyMjeEydOxLhx40BEjLu7+8c1zIsEAsFGiUTCjh8/HhMnTkRwcHCoXC7PtHSuVCqtTznJ
    gPPg21qQT3L8HYGpfuyrJidbAUit4dw/+HU1wxPApRrOXcF7O1HN+2lXzXhvVBUGwFwAy0Qi
    0TFe+T0GYCmAhbwB3mBoyI1hEIBhgwYN0n355ZdCNzc38wPAdOvWDVOnTjX+8MMPg8GF25pM
    nNhkMqWZTCadvb29cOXKldi/fz8+/vhjl4SEhCVGo9EvNDR0YUJCQmB+fv7k4OBg0dq1a7F9
    +3Z88803Uffu3Xutc+fOq9zc3MrXrVtncHFxgUgkgrW1Ndzc3PDLL788TYLSp2vXrujbty9Y
    lsWIESOwfv16FBUVBYPjnxQ2lcESEUQiEZycnCCXy0FEkEqlMBgMVpmZmS8EBQV1JiKWZVnk
    5ub2AAAbGxs4OjpyGpudHRiGgUQiae/t7T2P5xhRaWmpi8FgsBaLxRXXNhqNsLKyqs/hhwGY
    xVvX4D1Tn/DKE3jh2SIwMBCDBw+Gra0tlEol1q9fj+zsbH9wlJIS/tw8AKl//vmn+88//4yQ
    kBAcO3YM6enpAHAHQGml3/UC4NS1a1f06tULAJfg9PnnnyMjI6MV7/FT8UK8K4AZANqC46Me
    AfAZOH5mYyELwFl+42jFe1HT+Xmp+P8/jfDjN6YONXzeBsCH4DzSt5vQuDPBcc2sAMBoNAqK
    iop66/V6v8WLFy8dNGjQ/02bNs0rJydnnkajcQQAvV4vzs/PfyUhIeGP4cOH73RycoJQKISt
    rS0UCgUcHR1x5MiRp2ntXFAz7SKYVyoKmtB4LYbgGYYhkUgk6NWrl7C0tBQHDhxgWZatMSGA
    ZVnm1VdfFQYGBprkcjmuXr0qrCl/oB7zCtwBTAbQm5dJcbzekFZJTtbkzQ/ljWmznCzm5YmX
    hXPTwEVQK2wn/h21hPa8PDJfNwDAK7y81PJy8v/QuNS+tuA49HK9Xi/iFe/eACJ42Z0DoME4
    J8IGvG4vuVxuFRsbW+7m5lZFi3d1dWViY2OZEydOSEpKSvrwlkV5k5AQLi7pOTk5pffu3ZOJ
    xWLExsYiJCQE8+bNk1y4cGHinTt3pCKRSMAwjOvUqVPRu3dveHh44PfffxfGx8e/eezYsa5i
    sTjTxsbmurW19R2WZa+0bNky49SpU8bIyEhs3LgRQUFBTV1QqgFoU1NTkZ2dDVdXVyQmJqKo
    qAj85t0kSclmjqM5QcpoNFolJSW9Us1gsKgQMgwDnU4XnpiYWCVUZTAYzLy9iu+b/18P8ALw
    qUKh6Ne6dWsQEa5duxaiUqk8eeGZDY7/UpSVlYWEhASEhYXhzp07yM3NBb9ZqSpdLxfAhqSk
    JP+JEyc6OTg4IDs7G2VlZTcAfI2qCUQFADSJiYm26enpcHR0xPXr11FWVgYA+fibmB4K4HN7
    e/t2YWFhUKlUuH79ephOp1MCmFlNSW1IXAIwhWEY1mQyTeeVrO8ArAWX5Kd7ShXK4QA6P8Im
    MZKfc5OATCZLk0gkGo1GU+VlUqlUPpmZmauOHTsmEQgETiqVqgoVqLy83C4rK+tdT09PL5FI
    lC6Xy5NZlk0ODg7O37Jli6FDhw5o377908JF16DmRLdScOHGJg+9Xi9PSEiY9957773EiUKG
    0Wg07Ws6XyQS9dq/f/8ms+FtMBjs9Hq9XQMO0YF/9l+o5BHsBY73OJ2XV/pqsrD6fqarJif/
    r5LSX9lo/aaaTHtYAmRppb3QE8A6cHkHZvTmZfxiNF41il/AcUitAIwA8CovJ3fyOta1p9FD
    KQagtLOzY/z8/Cy6c3x9fVlbW1uUlJQowYX5moRC6e7unnrz5s2EtLQ0l9TUVNjZ2aF79+5Y
    uXIlYmNjBXfu3HmBZVljcHAwBg3iKHQBAQEYNWoU3nnnHTudTtffYDBApVJBpVKphEJhQmZm
    5q/Ozs4HXVxcfu/Tp0/ZF198gddff70py5gUAD/FxcVNmTRpksDPzw8XLlxARkZGCbjwQVlT
    HjzDMBg/fjw6duxo0UIWCoXo0OFvh1CrVq2watUqi1mKRAQfHx9IJJKGGGpfhmF6xcbGYv78
    +SAiLFu2DBs2bOjHE6q389bt9wkJCW1jY2OtgoODER8fj8TERDWAXdWsXwKwA0BJQUHBrIKC
    grYAjgL4GFxorjLuAjj866+/vjR+/HjWxcUFFy5cQHFxcQ44Kop5M3xWLBa3mz9/PiZPnoyy
    sjLMmDGD3bdv3wgAWwCcasy9j1fmzUJch8bNsm8IOdkJD4bNHnAK8efJ0ETKJPn7+2dkZmZm
    aDSaByomaDQaZUpKygcikUhnNBqrhyZRVFQUXFJS8hHLsnoAhRKJJC0lJeWsh4fHLwqF4tym
    TZtyRo8ejZ07dzb19UsFRzt5rdpeWgzgh0Y0tuoEnU4nTklJeaayrHyY0axWq1umpKS0fJRz
    6wk9wWUrC6q9EyP4+7wTXMTsEP+eVF4LI4DDvGeuwq/AK44Gfu38wfGW1/DXqIxEACcBjKn2
    9zL+XPP7OALAMxbe73H8nhnXSMtZXOm3Avg9IZ73lj61EALYbGtrS0eOHCFLOH78uDm5YDuv
    TTcJXLt2jXFzc1slFotp3bp1ZDKZ6NSpUxQdHU1yuZwEAgEBoHHjxpFOp6uYT1xcHNnZ2dGg
    QYNo27ZtNGfOHOrdu7d5jiQWi3OdnZ239OnTp7tarRaEh4fjjTfeaMpr6CkSifbjb76IBsB/
    8HdWXJNAjx492ioUiozevXtTWVkZNTQMBgONHz++vjiUi4RCIe3YsaPi+lu2bCGxWEzgeDBm
    2AKYD+Aqr0DmAniP/7tFnRrAf3mhN/Uhv+8LLlu8mBe4ZwGMx9/cLxbA/+zt7enUqVMVY1y6
    dCnxBuCYJ7Tsr/L34TU83ZCByyilRzh+QlUe2BPFzJkzRa6urv+zNFZ7e3uKjIwkT0/PGucT
    Hh5Or7zyCoWHh5NSqSSGYUggEJTY2dmd9PPzi42NjVU6ODjgrbfeaupr6AEu47eynFzCr22T
    Ac+hzHzEZ63eDplMVh9ychavBFr6jSWVznMGRxdK5WVfEbjIjMdD5ORKXv69+ZDfD+UNh1L+
    uAuOj6ioJm8tjU8PYMITWvapvJye3ZiKX0PAAOBcSUnJuB07doi7du0KW9u/976SkhLasWMH
    FRUVmQCcbypehm7dumH69OnK8vJyH51OhwMHDmDUqFHYt28ffv/9d8TExCAgIABff/01wsLC
    IBL9zbl2dXWFnZ0dRCIRoqOjMWbMGBQUFODSpUvYsmULjh496pSbmzter9dHdunS5XNvb++v
    v/vuu/vnzp1Dt27dmqKgdDAYDEqFQgFPT0/cvXtXaDQa3fA3t65JeSQ1Gg1SUlIglzesvqvX
    681h4fpAmtFo1J0+fVrcq1cvmEwmxMXFQa/X63ihWNnq/Ig3vuaDC3+moeZCvLbgkm7k4Dia
    DCzzpzLAhb61/PW3WLDkk1QqFeLi4hASEoLi4mJz/bkScCGixoQt7wkYygvKYfy/e/F0FiXW
    PsY9zELN4dXG3aWmTkVubi6ZTKb8ynQQgUCAbt26Yd68eWjTpg0mT55s5u8+gLCwMKxZswal
    paW4ffs2Dh8+jD179lgnJCT0KisrCz927NiI9u3br1y4cOHZv/76y3Tq1KmmuoZOqJrEIeQN
    NQWaaNF9lmUb5XdMpnorN53P6xTVk5xM1eRVLq/obQEXCh8ILhmxJq63BFymtg248DcLyzWy
    48FVOOnMeza/5ZVKcxSHwNGTanrHnwSP1p43atL5PduRv49PpUIJAPuJaPjOnTsHh4aGsjNm
    zIBQKITRaKRNmzbpt23bJiSik3iw5tMTAR++ll2/fn1RaWlpjJWVFS5cuIC//voLb731Fl56
    6SW0atUKDMPgmWeegZ9fVZ6uQqGAk5NTxYvEMAyUSiUGDBiAnj174vz589iwYQMOHDjgffPm
    zaXZ2dkdg4ODF3Xr1u1uEyyK3g7AZ3K5vPucOXPw7LPPYunSpcLdu3dPNHFS4p3GeDgfVZ9k
    WZa5evUqYmJiwDBMffMcqwhihmGQmZkJgUBgtkzrgl+J6NTmzZufOX36NIgId+/eNRLRUXBh
    lsow8kqmFlwyQDS4UI+lsJojuIxsgAvnWONv4nhl+INLnnPkhV6OhXP26nS64R999FHHffv2
    Qa1WIyEhwQgujHO5kT1By1mWfd7R0VEml8tRVFTkVlhY2AccV2khGjdJqF72XHDlRybg4Vmh
    av68J87JIyIcOnSInTNnzgslJSUvVX7PWrZsiQ0bNsDX1xenTp2qUZkEgKysLOTk5MDPzw8u
    Li6IiIjA+PHjsWPHDmzfvl2amJg4tLCwsE23bt0+ad269dcxMTGqJtiVqj245LTu1fbUF3kF
    aBG4RLmm8bCZTAgMDMSUKVOqOEMaAkajEVu2bMHNmzfr43K/gSvz073a3y+B4wxWV+D+5GWZ
    O2+A7oJlmpYj/k64McvJ4hpkz1DeA5oEy0kth8AVSQ+u9veTeJBu1JAQ8PJwFq8A2wJ4C1wd
    zdXgMr4brLZtQyqU5QD2ajSaTjt27HCdMGECnJ2dUVhYiK1bt7JqtbqAVyabhLfL29sbhw4d
    isnOzp4UHh4uHDJkCL7++mtkZGSgf//+8PHxqeLJfMBMdXLC+++/D4VCAZmsarRDJpOhb9++
    aNeuHTp06IDPPvtMnJ2d/TwR2UZFRc1kGCZ+2LBhOHDgQFO4FR0BfGpjY9Nzzpw5mDlzJuRy
    OT7++GOwLCvatWvXJKPRSODC37lPerBisVgtEAiKNBqN2+3bt+Hm5obQ0FCwLFuvSiXDMLhz
    5w6Sk5MBANbW1iqWZUvq6qEEMEuj0Uy7evVqR95iJgCba7B4BfibRxQOLlnDUmF1L4ZhHIVC
    IfR6vQ8vOC2NtS+veBJq5vHdALC1pKRE8Mcff4D3kv0C4As0HkfMCsAMqVT64qRJk9ixY8dC
    qVQiJSUFX375pfTAgQPjDQZDLm/oPG2cyp/BFZN/FQ+WjAI4nuh34EJuTxxBQUEQCoXh6enp
    S7RabZWSb4WFhdi5cydu376NEydOoKCgZsfML7/8gtGjR2Pw4MEYPnw42rRpg9DQULz33nsY
    OnQoli9fjsOHD3unpqYuJyKX4cOHr/L29i5KTU1tKuvWCRznrkcN++rLvMHwbg2GWqMb3kTE
    SqVSjBgxAgEBAQ36Y+np6di3bx+IiKkHwzsRXC3F2fz+JORlz1reU/iA7V9Jt+kKLkpz3sJ5
    rvwBXg461qBQRoBLAHqYzvQXLxOngfNO63jv6KpG3iejAGzw8fHx69mzJ1xcXJCcnKw8depU
    v/z8/FbgwuAHnyYBKQUwmtfY0wDohgwZQoWFhUREVFRURCNGjDBzCzLAkUXH4Qlz89q1a+cm
    l8tPOzg40P79+0mr1dKFCxfo/v37VFhYSFlZWcR3S6kT9Ho97d+/n9q0aUMMw5CDg8Px4ODg
    oHbt2j3pdRMAGAzgD1tbW/rggw9IrVZXGXtaWhqNHz+eBAKBjveOhTzpQY8dO1YUEBAwwsbG
    5grDMOTp6Umff/455eXlUWlpKZWUlNTpKC0tpcLCQtq5cycFBQURwzAkl8sT/Pz8JvXv37++
    immLeWu6B7huNd/DMj9SCGA9y7LEMIwJXPtBS8J6ilwu13fp0oUEAsF93jqtDhsAh1iWNRfq
    rYmP2JL3BPwArlyPC/45ieRRnjUP3uMwnJcXQ/jNwtL12wBIGzRoEN2/f7/KM5mUlEQ9e/Yk
    Xta0sfA7rrylPoL3HD0Pru6jL558nUAhgDCGYRayLJsLyxysfHA8sbb8c/JEERERoXBwcNjG
    MMwDY7X0t0c5fH19adasWfTXX39VrGtWVhYtWLCA7OzsSCQSaX19fT8ePHiwbc+ePZuCnByK
    v7uvPOzQ8d6xsCc96KioKD9HR8dzQqGQevfuTYcPH6a7d+82yHHq1CkaNmwYicVisrW1je/d
    u3f7epqGHb/nDADHJ/8f/uYxVlcozXU0TeD45pbk5Bhw3n8Cx8nubeEcCarWt6yJa+kKLvnn
    EIBuvKdSUY/PnARcCFuCB2twmuEM4Livry8dOXKE9Ho9ERFpNBrasmULubi4ELjGCe41fJ/l
    jXdrXo8TPunn1obf5Art7e2pf//+9N5779GVK1eqbALXrl2jZcuW0cCBA8nR0ZF4q+BjcHH/
    J4IWLVrECASCshdeeIFUKhUREZWXl9OuXbto0KBBNHjwYEpLS6u35I64uDgKCwszk9j39u3b
    1zUiIuJJTV/GW1Zpbm5utH79+orC2NWRmZlJU6dOJalUSrzV1/8hD3iDY8wYLi+kY8eO7ZRK
    5V6BQKBXKpW0dOlSKigoqPM6qdVq+uKLL8jLy4sYhjHZ29v/Ehoa2vPEiRNM//71Xm/ajVfe
    1LyHozpEAL5wdXUlf39/AhcG8rAgFFa5u7vT4sWLSaFQ6ADEWrhWT5Zlc9u0aUMymcwIy8k7
    InDcShMvvOu6ziy4MOHH/DzzeE+Dmv/XXGvyv+BC+s7896IBaJYsWWJxjZYtW2ZOhoj29vYG
    r5SO5K9znvf2lvHnqMCF96/xn3d5Qs+vBy8rE3iv6sMUk3JwobaP8TeV4YnA39+/p0Qiyak8
    vsDAQFq4cCEtWrTI3FGrVkdQUBB98cUXVFxcXLERfvbZZ2RnZ0cCgUDt4uLyn2HDhknmz5//
    pKYvB/AGqnZNeZTjAjg+3xOTk3PnzmXbtGnT0dHR8SDLsgZHR0dq1aoVtWrVilq2bEkBAQF1
    Olq2bFlxPVdXV2JZ1mRnZ3c6NDS0z+jRowX1PB1ncO10S3lj1JIC9kml+38eVXmuZixG1eSZ
    KRbOaQsgudJ5M2qQa2/z19hZT4YqCy4Z6BUAXwL4kVdY9/Ke2UngatdWVlp7ACiMjY0lvV5P
    f/75J61Zs4bOnDlD5eXl9NprrxF/z/pVotjJwVHcJoKjb/wIjl7zEzgu6kzeG/9Ic6pPDVQE
    jgg7s3Xr1uz777+v79Wrl6hyMo4ZYWFhCAsLQ0lJCc6cOYP33nvP5vfff3+D31iWo5FLCJ05
    cwYTJkwIlkql8kGDBkEmk6G0tBQfffQR1q1bh5KSEowZM6ZeC1r37NkTK1aswJQpU5CXlzfs
    7t27tyIiIv6zdOlS3ZIlSxpz+vbgEj1eDwkJsV6+fDmGDRtm5gg+qPG4ueGDDz6Al5cXVq9e
    3SU/P/8r3oOyFU+gPuWOHTswbdo0/Pe///2rZ8+eryUlJd3Ozc19bcWKFXaXLl1Cq1atah36
    ZhgGGRkZOHjwINRqtcbZ2XlLy5YtP/jtt9+St2zZgp9//rm+p1PKe+078gr+KTzYL5a1t7dH
    3759sXHjxmC9Xt+d94JUNg5aubm5ITw8HAqFQlRWVhaAqoRzAYAB9vb2jgMGDEBKSgpTw2bX
    g48eMPy46sKyF/MewvksywZ4e3ujbdu28PHxgbW1NYqLi5Gdna1ISEhwTUxM7FZSUvIi74U4
    wAs9gZkbW51vzCcZCAB0Sk1NDQaXrNPazs5O4e/vj4CAADg7O8PGxgYqlQqpqamyK1eu2Ccn
    J4cZjcbB4GrcfdOIcseX3/CGP6K3V8x/ZxbvoXkLlkN9DQoiQlBQULjJZHICuCSc6OhoLF68
    GGFhYfjtt9+wefNmlJTUjgly69YtzJ49G7du3cK7774Le3t7TJkyBcXFxXj//felRUVFb9y5
    c+f2jBkzdnzwwQfUyLxzBwALeE/+43qeOgPYCI4mtAVPgAtrMBhM8fHxf/To0WPK3bt3387P
    z4/Nz8+3AYDQ0FB4eHjUOomGZVnk5eXh6tWrMBqNEAqFWmdn5+1+fn7Lz507dy80NLS+p1PG
    y6NwXuc4gwcbG1R+r0LBhb5/qPQ3Kz7iUlkXqi4nGXAhZM/K24KF8YTxyqiQYZhsiURi0Grr
    xLxxB0eBGQvATyqVCq2trWFlZQWtVguVSgWNRqM3mUy5vFPhRwC/givCLvX09ITJZMLq1aux
    detWDBgwADt37kRwcDAYhrESCoUtDAaDNx+pMdfBdbWyshJJJBJIpVKUl5dDrVajvLz8Bf5e
    7wZXZ7PR+pd3B5ARGBioP3/+vPpxvECXL182tm5aKvBwAAAgAElEQVTd2giOa9L3CbxscHBw
    +MDe3p4OHz5MRqOR1qxZY+5tTAqFgg4fPlzvJWj0ej0tXbqURCIRWVlZ5bdt23Z4I0/dgX9I
    ynv16kXnzp17YIxZWVl09uzZCq+tGTqdjrZs2UJ+fn7Er1vsk3SRX7t2DR06dMBLL70kCQwM
    fEkul98BQEKhkKytrWt18OV7yMrKKqNFixazhgwZonBzc0MDekgE/MZDvHK+rNo9FQP4KiQk
    hPbu3Uu+vr7EK0KVLR0vAH+OGDGCbt++Te3atTO3HpNX84793qdPH9q+fbvZq/SmBUNjzz9Y
    748KIe/Zue/i4kKLFi2ia9eukUqlIqPRSERERqORtFotZWRk0JEjR2ju3LkUHBxMAoHACK78
    h2nw4MEV3iszysrKKCYmxhzaKhEIBKbQ0FBasGABHTt2jDIzM6m8vLzid0wmE6nVaoqPj6d3
    332X3NzczC0c32yk59caXCkTE2pfjmUruPBf47tVPTzeM5dOGzFiBKWnpxMRUWFhIQ0ZMqTW
    Ye/Kh0QioQULFlRESQoKCmjMmDEEgGxsbK736NGj9cCBAxtz2o7gWimW1zTm0NBQevnll8nZ
    2flhc8sD8DqeENXi888/R9u2bTF69Ghpy5YtYxUKxT2BQECRkZF05swZUqvVtTquXLlCQ4cO
    JaFQSDKZLMvX13d2dHS0rYeHB95+++36D6fJZCzfJtcsmxZWUyCF/L5W+d5/jaocZSdeEa3S
    x5tlWUW1/fHnaudUn5AUHHfSBMDIsuzsKVPqIirRkld8DQEBATRjxgz64Ycf6OLFi3Tjxg26
    ePEiHT16lD7++GOKjo4mb29vYlm2HFyL1qMAdJMnTyaVSkUTJ04kADRy5EgqKSmh2bNnE8uy
    RhsbmwvgOJ/lzs7OFBUVRUuWLKG9e/fS6dOn6fLly3T27FnauXMnTZs2zRwNM/HXbxT6Bgtg
    pVgsprVr1xpqwzXcsGGDTiKREP8gNKpiMn36dIWzs/MBJycnOn/+PN24cYNatmxJAoGAhEIh
    +fj40MWLF/9xDiUlJXTy5MnHCo1nZmZSv379CAC5uLj8Gh0d7TpjxozGmLYAHNG5vE+fPnTj
    xg2L41uxYgX5+PjQyZMnH/jMZDLRjz/+SD4+PmYOWy88YQwYMABExLi5ucWyLFseExNDhw8f
    fuzjyJEjNGXKFBIKheTr67ti9erVgsjISFy8eLGhp7BEIpGYnJycCFxWd0Q1hXJzcHAwXb9+
    nV544QUCcA9cG0IzugLInjVrFpWUlNDQoUMJXDZ2ZSv7OYZhSleuXEnHjh0ja2trS6GcSSzL
    lrm5uRHLsmXgeGO1xVAAme7u7rR9+/YKbs/DYDQaKT4+nhYvXmzuW04uLi4UFxdX5bzz58+T
    h4cHASBXV1d655136NatWxUK5D8ZdLt27SIvLy8C10pwSCM8oiPAUXzqonSpeO9Fo2Ljxo1i
    d3f3bwGQj48P/f777xX3Mj8/nzp37vxI4xcIBKRUKs194y0eDg4OtGfPnorrX7p0iQIDA4lh
    GPL29v7qzTfflDZCQW2znJz/MGUSAH344YekVqtp5MiR/zT/DN7r9URARIiKisKNGzeYdu3a
    9ba1tT0NgBYsWFBrx8jatWuJZVmytra+EhYWNmL+/PmCiIgI/Pjjjw0yB77ayiJw3G8CRxvp
    VOkUEW8AVL7vdwCEVoq8teG9bZXP+QtV2zH2A8dhrnzObAuyLY//TMswzKhRo0bVdmquAPay
    LEvDhw+nCxcuVMnbUKlUVFpaSuXl5RV0rD///JPeeecdCgoKqhhjWFgYJSUl0e7du8nR0ZE2
    bdpE2dnZ1K1bt4pzvLy8aPr06RQXF0fFxcVkMplIq9VSWVlZldwJo9FIFy5coJiYGHMN7p+q
    7SUPeA7qA3IAnRwdHdG3b1+2NqGIPn36MC4uLqbU1NQOvBXfKL2iHRwcEBcX16q4uLhDly5d
    EBAQgIsXL0KtVuONN95Aq1at4OnpiTZt2vzjta5evYoJEyZg4sSJ+M9//vNIv+/m5oapU6fi
    0qVLuH//fvf4+PixXbp0WYMa+q7WI4IBTPbz8xMvX74cISGW82uysrKQkZGB4uIHk98YhsHQ
    oUORnp6O2bNne2o0mikA/sAT7KRz9OhRLFiwgNRqdQ4AU4sWLVBbb8bNmzfBMAyEQmHBlStX
    jHFxcejcuXNDTyFFIBDoBw8eLD548KDX/fv3p4IL/Zbw4RaWiCCTyTB48GDs2bPHu7y8PAp/
    t9TyYVnWLjAwEAqFAv7+/gDHH/Lgw0ISAINcXV0V/fr1Q1FRkZkSUPmlbQHg9YCAAHnHjh2x
    c+fOUtS+/IkrgFlisdhtxowZeP7552ukU1QPowUHB2PJkiVo37495s2bh3v37uHQoUPo3r07
    BAIBjEYj9u/fj8zMTLRs2RKrVq3CkCFDHrkkilAoRExMDDIzMzFv3jw3rVY7kw8hNVRWpohX
    WutaoFzGb2R70Uhh+lGjRuH//u//3DUaTWeAq3TRunVr/PLLLygoKMD169dx+/ajtRpv06YN
    PvroI/z0009Ys2aNxVBrQUEB1q5diy5dusDDwwPt27fHlClTsHDhQuTl5T176tSpAwzD7G+E
    qYeA45c9NCFKoVBALBY/Ci3Knff2X8AT6KRj3pvXr19Pf/3110lHR8f/MgwTfvz4cXFtFHSG
    YXD+/HmwLAtnZ+dd169f/7Fly5b47bff8NtvvzXIHDQaDYgohY/iiM3yio8yqCo5uSrDF0Af
    lmVvGI1GcyTHqfp2zDCMJxGl8YbEAN5LWd15ZoYzuJC7Iy+zNFZWVve///772kZxXgcwrG/f
    vvjkk0/g7++PnJwcnDhxAmfPnkViYiI0Gg0cHBwQHByMfv36oWvXrli6dCmeffZZfPjhh9iz
    Zw8SEhJw6tQpDBo0COPGjcMzzzyDM2fO4Nq1a2BZFv369cOiRYvQvXt36PV6nD9/HsePH8et
    W7dQUFAAqVQKf39/9OjRA1FRUejcuTM+/fRT6PV6HDhwoD8/5yVowHa3TgAut27dWp+Zmamv
    jZWTl5dn6Nq1q5HfGD0a4+UaOHAg+vbtK1YqlWskEonps88+q/A0nj59+rGTOg4fPkwymYxm
    z579WBnhxcXF5rAd2djY/NG1a1fvl156qaGn/ybDMIaFCxeSwWCoMp6CggI6f/48nT59mp57
    7jkSCoW0YsUKOnPmDF2+fPmB7O+8vDyKiooiXmHpiCYAOzu7ESzLambNmlVry3v16tUkEoko
    ICBgTmM6WRmGKXn//fdp0qRJxCuSL/CfWQHYEhQURImJiZScnEyhoaHEhyLMCW3v2tjYmMwd
    qv773/8SwzBa/E1eDwGQEB0dTWq1mo4dO0YKhYLA8fPMgm25SCQyrFixgl555RVz664WtZzP
    OACajh07UnJycq3XYvfu3eTk5ERBQUF0+/ZtIiK6efMmtWrVipycnGj37t21vnZGRobZetc0
    sOfPFlyZp/roQnLJwobYYPj555/h5eU1na/wQEFBQfTOO+9Qq1atSCqVklAofOSxDxw4kLRa
    LW3dupVEIlGN54nFYlq/fn2FLE1NTaUuXboQALKzs9vXtWtXm8uXG7wU6luVPGFVDi8vL4qK
    iqL+/fvTjz/+SEajkT788EPq168f9erVi5RKZU1zywTH/3uiWLFiBWQy2XCWZTV1fR6FQiEF
    BATMmTZtWqOMXSgUPsMwTGVPfwGAkbzCLMbf1KHKxyH8XT1jhoV11QIYy1/DB5Yz+c18JwYc
    jadyQl2iUCisbUi4PYAkpVJZQa+7dOkSPfPMM+YE2AcOpVJJEyZMoD/++IOIiHJzc2ny5MnE
    siwNGzaM7t+/TykpKVRQUEDPP/88AaDo6OgKOXzp0iV68cUXa3xOZTIZDRw4kC5fvkxERKdP
    nyZ3d3dzNLJBn19bAHGBgYGa1NRUbW2EenZ2tqFDhw5GcBlZysZ6qdzd3YcKBILcqKgoysjI
    qBMn8quvviKJREJffPHFY3937969ZG1tTQKBQOvu7v5aQ1l3lRSTrQ4ODlVa6pnx2WefkaOj
    I9na2lbwCOVyOdnZ2VGLFi3o+PHjD3xn5cqVxPfmfaVZoawTOgBInT59Op07d87MUT3LW9QS
    AFuDgoLo3r17ZDAYaPr06ebSMn14ZfA7X19fio+PJyKiQ4cOkUKhMNfDA4DpYrFYv2HDBiKi
    ygqlOZQTASAtMjKSbt26RYMHDyZeCapNBYYKTugbb7xRp7JbOp2OFi5cSCKRiD788EMyGAz0
    0UcfkUgkovnz51dpg/q4MJlM9Pbbb5sF6QbUvSxSTVDyimB9KJTXHhZ6qk+0bdsW/v7+IVKp
    9Gp9jH3MmDGk1+tp9erV/8i57NKlSwVP02Qy0dq1a0kkEpFQKCzw9PQc3MBTtwKwrSYFat26
    daRSqUitVlcY5TqdjtRqNRUXF5uzamtqx/dqU5CTcrl8RH0qlI3oaa2efU3g6uO68pGATRbG
    mQuuhBrDv+fVPzcBeE8oFII3LC3dl4WVDPMb1T77g2EYl1pOaS4A0/Dhw6m0tJTu379PQ4YM
    eaR737Jly4r2vampqdSnTx+ytram/fv3ExHRzz//THZ2dtShQweKj48nk8lE27Zto4CAgEe6
    /rBhw6igoIA0Gg2NHz++csvLB0LR9VXGQAMgKTc3V5KQkFCr1LukpCSkpaUx/EPS4MXOIyMj
    ERYW5ltYWDjPwcHBadasWXB3d6/19UpKSnDw4EHY2tqiNjUlIyIi0LZtWxiNRklxcfHzr776
    qmMDcoSkALxdXV3h5eX1oEbToQNGjx6NmJgY+Pv7g2VZ9OjRA6NGjcKoUaMe6BIEcJmCUqlU
    CK7zQJNq+/OUIRdAYWJiIkJDQzFhwgQIhcLOAF7iFUYW4LhQAoEAAwcOhK2trRJcxxsnAP5u
    bm5wdHQEwBXsd3BwYMBlMDoBGOzj4yOMjLRUmhLWAKbb2Nh4Tp06FdbW1uZuJ1moXcFwIQAv
    lmXh7e1dp25QIpEIEyZMQIsWLXDgwAEkJCTg0KFD8PPzw0svvVSnzh8Mw8Db29scivdFwyVN
    lKP+2rAVoRGoJbGxsbCyslIUFBS8rdVqW9f1eiKRCJGRkTAajbh48eI/VmC4cuUKTp48WbFO
    gwcPhr+/PwwGg31paenYnj17NmTPbCmqcuoqYDQacfLkSezcuRO7d+9GQkICiAjnzp3Dzp07
    sW3bNly5cuVh74Vvs5ysPYgoDw/S4iL4iIigBt3GCVz5JvsaIi4MgACDwWAHjlJiib/A8ob9
    q6jWFUcoFObb2trWpt2miHckMCEhIVAoFIiPj8e5c+ce6ct3797FwoULceLECXh5eWHWrFlg
    WRY7duxASUkJ9u3bB51Oh5kzZyI4OBjHjh3DokWLkJCQ8EjXP3v2LG7evAkrKyu0bdvWXFWj
    IyzQQOpLodQB+LmoqEjzzTffsKWlj0cNUavVhk2bNuny8/O14GotNWi3i3Xr1sHV1VWSk5Mz
    XafTdXvxxRfRr1+/Ol3zwIEDOHHiBHr16oWwsMf3ejs5OWHo0KFgWRY6na6TTqfr0oBlMUQA
    xCKRCLw1VgU9evTAZ599ho0bN6Jv374QCAR47bXX8MUXX2DFihVo0eLBd1EsFpsVBjmeYL21
    fwFKAOSnp6ejrKwML7/8Mtq3by8EV3fsAQJnp06dzM9bP14oefj7+8NcrsvFxQUuLi7gFf1e
    ADpFRERYWkMWXAmJIUOGDMGQIUOQn5+P3Nxc8CGO2nL19CzL1kuPdX9/fwwbNgy3b9/G1q1b
    cePGDQwdOrReun7IZDKzoDQ24EavBXC7nq51szEM7wEDBiAzM3NUWVnZ8/Vh4Hbr1g3R0dG4
    fv06zp49+883TKvFjz/+CLWa26f9/Pxgrv9aXl7et7i4uPWqVasaVE7WoNBgz549eOWVVzB5
    8mT88ssvMJlM+PrrrxEbG4tp06b9k0Iga5aTddD0pdJSlmXvW1ivV/Bgg4PK6A8uidG3hs9b
    gAvn9qjJ/gRXAH1sdTlhMBhStbWrF8QCUAgEAri6co17srOzodFoHvkCiYmJePfdd5GWloZe
    vXohMjISZ8+eRVxcHE6ePInIyEgMHToUKSkpWLJkCZKSHr36j0ajQXY217jN3d3dbLxbN6RC
    CXDp9Sf37Nkj/OSTTx65FllZWRnWrFlj2rZtm5jv7f1TQz+M06dPx61btwYVFxe/3LFjR8G0
    adMgFteuCQUR4eTJk3j//fchFAoxceJEKBS1K5Dfp08fODk5wWAw2KjV6v4zZ85sqGx3HYDy
    goKCGteJZVkIhUJYWVlBIBBAIpFU9LK2hOzsbOh0OvBWo7FZ5NVJ6UjPyclBTk4OfH198frr
    r0Mmk/mBKz5ub14fAHB2dkb//v3BsmwrAC8xDGPXokULZGdn49y5c9BoNGbl0Q/Ay3K5XDl4
    8GBIJFW6/BG4chXTPD09FdOmTYNCoUBKSgrKysrMmam1KVKnB5BoMBiQk1P37nNCoRDR0dGQ
    SCTYsGEDxGIxRo4cadEoelxkZWXBYDAAXJu3hkp0MQDYXw9eykJwCTkNWs9w6tSpWLt2bdui
    oqK39Xq9dT0oAZg2bRrs7e3x1VdfIS0t7ZE9JOaEH4FAgEGDBsHGxgZ6vd6tsLCwT/v27RvK
    ACgHF32rEUajEXq9HgaDAQzDQK/Xw2g0PkpNx2Y5WQfo9XqtyWSy1IczEFxyS02Jb0HgOOkO
    NXzuBa5ebk15HG7guJNOFvbMdK1WW5t30gggn4hg1kdlMtkjJS9Wxvnz57F582bI5XI8//zz
    KC4uxrp165CdnY3nnnsONjY22LRpEy5duvRY1xUIBBUOAa1Wa44qFMJCUk59KpR5AJaoVKo/
    Vq1ahbfeegvx8fEP/cLt27cxa9YsrFixQqxWq/8C14s3ryEfxOeeew4TJ070zcnJmatQKJRz
    58616HF7FBQVFWHjxo2IjY3FrVu3YGVlVWFh1AaBgYEICwsDEaG4uLhHXFyccwMqLal5eXm4
    du3aQ0+Mjo7GggULHprlbjAYcPHiReh0OgMeLDDbjMdX9tPLyspg7lscHR1t9qAPARDOMAyu
    X7+Od955BytWrICvry+cnZ1lAIZJJBJFbm4uxo4di6FDh2LOnDlm5dEVwDOBgYGsQqHA+vXr
    sX//fqhUKjCclTCcZdlOL774Irp06QIASE5OhkajKQeXRFAbmADcAmC6d++e2eCoE0JDQxEU
    FIS8vDwEBQXVKhpg6flNSkoCEZl4z5+pAdf3LLhuGqY63NMd4FqoNRg2b94MpVJpk52dPVul
    UtVLi1WDwYBjx47hvffew44dOx654UBOTg7On/+7FXObNm0QEBAAo9HIlJaW9l22bJltA8rJ
    lEc58cSJE9i6des/ytNKhlZas6ir07Okr2GvYQDEgCufZglycC0Xa6rjau6yVZOVOhgWamUz
    DGPkO0jV1tC8bTKZkJGRAQAIDg62SC37J+Pm+++/R1paGvr06QMfHx8cP34cSqUSPXv2REpK
    Cnbt2gU+y/2R4efnh6AgrpV5amoq9Ho9eDnZKBUm2oPrRawdMWLEA4WIzSgtLaUxY8YQwzB6
    gUDwA/+9Bsf48eOFQUFByyQSifGtt96qscXgPyQQ0fbt22nEiBHmZBWNvb29Si6X0w8//FCn
    5IAlS5YQwzAkFouL/f39ezfgrZgIQPPss89SSUlJnZKRrl+/Ti1btiRwdb3aNgWB8xQn5QDA
    VJZlDStXrqwYy/HjxyvqMTo6OlJISAixLEsikYhCQkLIwcGBABDLshUJVB06dCCpVEqOjo4V
    yQ8BAQEUEhJCAoGA7O3taeDAgRUF/Dt27Ej37t2r+M1Zs2aZi35H1mEuXQBkeXl50fnz5+vc
    DMBoNFJsbCwBoNdff/2R6k0+yvMbGBhI4LiiXRphfT14pfChtQ1huQXjDjRCFQwiQq9evV5W
    KBQq1E8SkbmBQ62Kn48dO7ZCVuv1epo6dSoBIKlUmh0cHNypAW/FS7CcnPFAH/OHZaxXOxLx
    8LBso+FpTcrhlbhXeWWMmsBRyrJsXZLEogGounfvTpmZmWQ0Gunrr79+7FamNjY29NNPP5FO
    p6uoiTps2DDSarX0ww8/kFwuf6zrWVtb01dffUVGo5GKiorMiUJa1FANo745HAz/Y2kADLyH
    w7Lpp9UiOTkZRGQUiUSZvGemwTklV69eDUxOTh7r7+/PvvbaaxCJRPjzzz8rLIOH4f79+/jf
    //6HMWPGIDY2FgcOHCiTSqXH+vbtO8vOzu4yEZm199q+IOjYsSOkUikMBoOCZdn2kydPbqhb
    cRTA7z///DO2bt1qDvc9NgoLC/HJJ5/g3r17RnAtAG822891RrrJZNL+73//w9q1a3HlyhX4
    +PigW7duAID8/HzcuXMHkyZNwooVKyASiaDX6yEWiyGTyRAUFIRPP/0UO3fuRGxsbBWaQnp6
    OliWxaJFixASEoIjR45ArVZDLBZj9OjRsLe3R3JyMjZv3oxDhw4BXK28utRlvAbg57S0NKxf
    vx55eXULQJgTfFiWha+vb0Xov7bIz8/Hxx9/bCaoHwZX87OhkQEubPYJHp0HqQbXw/dN/vsN
    il69erldv359SllZWZ2TXhiGgUKhyA0PD9/p5eUVXxsu5pUrVyp4XEKhEF27doVYLEZ5ebmD
    TqcL+eijjxrqVvwMrvLIPyrgjyj7DbycvNUs5ups9KTjHygJjajcqlmWzazDJS4BuHH58mXs
    27cPDMPghRdewOzZsx+Lf15WVoa7d+9CJBKZufPw9vaGRCLBrVu3oFI9Ou3axsYGc+fOxQsv
    vACWZXHixAlzbdFbACwShOuTo+cAYAKAVwUCQYCfn59gwoQJcHBwqMmDhPHjx6OwsFCSmJj4
    Krikgo0AvgVXBqVBkJ2d3V2n0/kMHDgQAQEB2LFjBxYvXoxJkyZh8eLFNXIEL126hOXLl+PY
    sWMwGAyFNjY2vyiVyh3e3t4n79+/b7p///7LIpEIlnqXPw58fX2hVCqRlpbG3r9/P1gmkwnQ
    MFybTAAflZaW+r377rveZWVlmDBhApyduSi7uW+y+V8zf9L8r9FoxNWrV/HZZ59h+/btMJlM
    h8E1l9c1JaHTyP1+63NtNLdu3ZLPmTMHLi4usLa2ruAhuri4YNKkSZg1axYcHR3x3HPP4fbt
    28jOzoZSqUTr1q3h4+MDAPjwww/x3HPP4c8//4RGo4GPjw86duyIgIAADB8+HAsXLsSvv/4K
    nU6Hb775BgcOHEBeXh5SUlLMxmAuuM4utYUawH8BdP3+++8DFQoFli1bVpGF/jgeM7MiYuYX
    SSSSCq4awzCPvdb5+fl45513sG3bNphMppsA1jfSBqUA0BNcj+FHlcFCcKVKInljsMGKYt+8
    eRMDBgzoVlxcXFGuQiaToWfPnsjJycGVK1ceKVzNsiykUmmqXC4/4uHhsaN79+63v/vuu521
    GVNOTg6Sk5Ph6+sLgKMH2draIj8/X6TRaNo5OjpuQcM0g8gCsApcEodvlUVUKGBnZwdz/2Oh
    UIjy8nJotVpotVoUFBSgvPyBqOBPTVFOPqXI5OWL4kkPRCAQFNva2hbfv3+/Lkbm/7Rabeu1
    a9daBQcHo3fv3pg1axb0ej1Wr15dkZj2MJhMJmRlZQHgmqYwDAN7e3uYTKbH4rFbW1tjwYIF
    ePPNNyGTyXDp0iWsXLkSxcXF5QA2g+vi1mAKZSsASwEMDwoKkkyaNMkYHR0NX1/fGst5CIVC
    vPrqqxg4cCD2798v2Lx5c+D169eXA+gGYHFDWHDJyclsjx49WltZWQl79OiB9PR0fPjhhxAI
    BOjVq5fFDcloNOLw4cNYsGAB4uPjy62trU/4+fmt7969++mNGzeW+fj4IDs7u4darfbz9fWt
    EHi1hZubG3x8fJCWlgYiCrt79659AyrYRwC8npubu2Dx4sXdfvrpJ8GAAQNgMBiQnp6OwsJC
    aLVaGAyGCu+Xk5MTPD09kZWVhb179yIpKUkFrnH8CjQR/uTo0aNx9OjROvPgKj0P1LNnT5w+
    fbqxppAPoMDBwcGxb9++KCoqgkqlgq+vL1q3bo3BgwejW7duFYlkD3vuZDIZIiMjYalMUMeO
    HfHNN99g9+7dOHz4MJKSkpCSkgIHBweEhITgypUrMBgMOeAyz+uCiwAW6XS6Tzdt2uRZXl6O
    d999t0LpBYDy8nKUlZUhPz8fWVlZKCwsRGFhIfLz81FUVASNRgOtVovy8nJcvXoVer0e3333
    Ha5cuQIrKytYWVlBJpNBqVTCyckJDg4OUCqVcHFxgZ2dHWQyWZXEu+TkZCxduhTfffcd9Hp9
    KoAF4AoZNzRag+sHHAMuU/JRIQbHoe0Frjf7x+D699Y7ZDIZo9PpOhqNRinAJdPMmjULb7/9
    Nnbt2oU33njDkqJU5b2xsrLKUCqVWz08PLZHRETc2Lp1q/7gwYOtCgsLaxWuLy4uRmJiInr3
    7g2Ayzb19PREXl4etFptyLFjx6zr4TmtCUcBvMY/IxEABCzLYurUqRg3bhzkcjnkcjlEIhG0
    Wi1UKhWKi4uxdOlSHDx4sMJ5xHsmP2gMD/PjGGpPMQr4w2KugXnPkslksLa2hq2tLWxtbSGT
    ySASiSAWiyEWiyEQCKDX66HT6aDX66HX66FSqVBYWFghe9VqNdRqdY38Q6PRmKlSqYrqshTg
    6Cwdb9++PXH69Ons8uXLMWzYMMydOxfl5eVYu3YtdDodrK2tYWNjAycnJ9jY2EAmk0EqlUIm
    k0Eul6NTJ44BIpfLwbJsRQWW7t27Q6PRVJmPRqNBcXEx8vLyUFpaitLSUlhZWWHmzJmYMWMG
    JBIJjh49ioULF+Ly5csmANsBfIcaOOD1oVC2BbBGIpH0GjlypGHRokUUFhb2SNcVCoXw9/fH
    zJkzMWDAAKxcuVLy/fffP6vVap3AVbKvV4nF7TIAACAASURBVAFvbW3NyGQyq9LSUjg6OiIp
    KQnZ2dmYO3euxQ3XZDJhz549mD17NjIzM7NcXFxWBwcHfxMfH38/Ly8P0dHRGDx4sGDFihUj
    DQaDY+fOnS3WdXwc2NjYwNOTq1es0Wgcr127Jm3AF9LIW8w3ysvLp/36669TTp48aSsWi8Gy
    LCQSScXLp9VqoVarodfrYTKZzAkWSbyA3IEn0EbMEmJiYtCnTx/ByZMnA4hIeOHCBdSmrAjD
    MIiLi4PJZIJarfZyd3cXR0RE6Bq44HyFQwbAb2q1OrB79+6YNGkSDAZDheCobOwYDIaKf8vL
    y2E0GisOs2dZIBBUZO1LJBIIhUIIBAIIhUK4ubnhjTfewOTJk6t4VJYvX47Lly/rAZzgvQCo
    o7D8AQDp9frl3377bav09HTMmzcPEokEFy5cwOXLl5GQkICsrCwUFRVBp9M94H2sPAcPDw+k
    pKTgzp07MBgMFZ50IgLDMBCJRJBIJLCzs4OHhwf8/PzQunVrdOrUCTqdDh999BFOnDgBk8l0
    i1cUDqBh250y4EqWfAigXR2uowAwHhwHbyG4MH29JhEJhULW2traxezRiIyMxKxZs5CRkYFv
    v/32oclVAoHAIJfL97u6un46fvz489u2bTNcv34dWVlZCA4OHmA0GmtVjN1gMODOnTswmUxg
    WRZKpRLe3t74888/odPpvC9fvqxoQIXSxCuV8eAqLbzu6OhoGx0djTZt2lS8g0QEa2truLi4
    QCgUYujQoTh69Cj0en0ib3DvxBNsS1sdkZGRuHz5MgGAq6srvL29ayUnMzMzzV4xatOmDa5e
    vdoo42dZNg/AZZPJFMQwDOzs7ODn54fQ0FAEBATA29sbPj4+cHZ2hlwuh1QqhVQqhVgshlAo
    fKBqCRFVyM7y8vIKpau0tBRZWVlITk5GcnIy7ty5g1u3biE9PR0qlQpERAzD/K7Vauv6/BWC
    a0BhdePGjVGvv/66KCUlBZMnT8bChQvh4uICo9GItm3bwtvbG3Z2dhXzMSvG1fUr/p0EwzB4
    /vnn8fzzz1fsHTqdDjqdDhqNBgUFBUhNTcXVq1chFothptpt2rQJy5YtQ2pqqgHAHnAFzQse
    FkapC1oA+FgqlfaePn26Yf78+QIHB4dakZpCQkKwdu1aeHp6Ys2aNZFqtXo1uJ6nCfX1ADo4
    OBiNRmOq2QMXFRWFNWvWVFi9RqMRN27cgKenJxwcHHD69GksWLAA+fn5KR4eHnM7duy4JyUl
    xZidnY1169YhMjISKpWqc0FBQYxCoWBiYmKqbPi1dJ1DqeQaBel0OpnJZLJDw2cEpgDYAuDZ
    vn372k6ePBl2dnaQy+WwtraGWCyGWq1GWVkZVCoVUlNTsWrVKqSkpNzkhWSTUCZHjhyJwMBA
    wapVq0YXFRXNZFlWePHiRVy8eLFOXsq8vLzxZ86cSYiKivoSQGMolRoAG7Vabb9NmzZ5Dxgw
    AP7+/sjNzUVKSgrS0tKQkZGB9PR05Ofno7i4uMLaNBgMFRucefM1K5AikQhSqRQKhQL29vZw
    cnKCl5cX3N3d4ePjA29vb7i5ueHEiRM4ePAgiOgcbyzUh6Jl5AVSnslkWnrs2LGeV69eZU0m
    E0pLS6FQKODh4YG2bdvCzc0NHh4eUCqVFVZ3ZcFpZWUFoVBYoUSXl5dXCEaz9X3//v2KjS4t
    LQ2HDx/Grl27IJVKwbIs8vLyjABOAfgPGjhjmlcmh4HjTPrX0zXbgAvRzwZXQqjelGF3d3cT
    75kGAHh4eMBkMuHLL7/EmTNnKkJi5vteySOkdXBw2Oji4rI8Pj4+96+//sKtW//f3peHR1We
    ff/OOXMyk5nJbNkmyYSEkAQSIgSigIBL8XMB7du+WBVrFS+3Ftui4KdoC27VqlBUtLgUa6mi
    uFXEwEspBWWHQMoSEhJIMllnksyS2fc55/vjLB0i0CwzQ99+ua/ruSZcZM7JPfOc+7mX3/27
    GyGRSHDHHXcYrFbrgnA4PGzS+J6eHrFaQtO0CM/x+Xxyk8kkT4KJ6QQ3Oed2hUKh3rFjB774
    4gvYbDb4/X6Ew2ExCM/OzkYoFIJcLofT6Wzis5P/Ns5kVVUVDAaDor6+frbH46Fvu+02PP30
    08Oyj++88w6efvppuFyuyXl5eZrS0lJHU1NTwuFGDMP4ZDLZe4FAYI5Go9G/8sormDt3LjIz
    MwfSog1aF5IkQdM0ZDLZOfC12GElXq8XJpMJmzZtwvPPPw+/398ok8n+7PP54gFN6+YrGFaz
    2fzA8uXLFW1tbXjiiSewdOlS8Zd8Ph/6+/vR0dEBu92O/v5+uFwuBINBMdv67bffgmVZHDhw
    AO+++y5SU1PFbLpSqRRXWloaxo0bh/Lyctx0000AOAabV199FWvWrEF/f78PwHoAL4CDgCAR
    DqUKwK8piprz4IMPhpYvX06rVKoR7SCNRoOnnnoKoVAo+vrrr18TiURWgAOhO+O1CeVy+YlI
    JOLdvn27Yv78+bjrrrvE/6upqcEjjzyClStXYuLEiXjppZfQ2dlpHTNmzFN1dXWfT5s2jT11
    6hQAYObMmSgoKMipq6v7tdvtHisMYR+pUBQFjUYjRBgqnU5nMJvNdUmwMdekpqYaHnjgASxY
    sOBfPcg4deoU3n777SngSnj7/x2cyYqKCmrDhg13dnd3v5yRkZH3i1/8AkVFRYPhhLuYM4m1
    a9fqWltbn9u5cydRVVX1zqxZs0LCwZpAqQXw59OnT//q0UcfpTQaDerr69HT0wOfzweWZcVs
    neBoxf4sl8vPcbp8Ph9CoZDoBITDYRHOQFEUlEol8vPzMXHiRDQ2NsJisTjBYR9NcdZrDziG
    gSW9vb33ZmZmpv32t7/FzJkzYTAYoFarkZqaOmQOtgvtU6GkYzabUVNTgxdeeAEmk8kN4D0A
    r+MCWKA4y5UAXomjMylIIV8hsPCfa7yEDQaDdRRFhSKRSMq2bdvQ1dUl8tcVFhaKcIG//e1v
    gq2KZGdnv1VVVfVsQ0OD++DBg7j88ssxY8YMTJs2Tbpt27ZfOJ3OK0byR7lcLvFZjrWTJEmm
    6fX6PJfL1ZyE7/JaAPlGoxHPPPPMYN8zhQ8A9uHfQK699lrk5OQodu/e/ZjL5frZ5MmTqTvv
    vHPIuOZY27t9+3YcOnToR42NjbaKiopnq6qqHInWY+bMmWhsbNwfiUQ2ut3uR7q7u0mDwQCW
    ZeF2u0XYTH9/PzwejwibGbiEhkYBOiOs1NRUyGQypKWlQafTQavVQqvVQqFQoKioCGazGX6/
    PySVStfp9frG1tbWeFaolgM463a7H3vzzTfHtrW14bbbbkNPTw9aWlrQ1taGjo4O9Pf3w+/3
    i9hd4fmIxZxv3boV27ZtE5MKQkAmnBVCtr+wsBDjxo2DwWDA119/jY8++giBQMAI4DVwvS3/
    MgM7EofyhwBuv/7666NPPfWUZKTOpFjPUSrx+OOPE01NTUx1dfWP+CzC+3FJExAEbrjhhoPd
    3d37N2/efMOkSZPwwAMPQK1Ww2Kx4I033oDH40F+fj4+++wz7N27N5yXl/f2D37wg7/ce++9
    ojN5+eWXQyKRpJ8+ffo5u90+d8qUKXjiiSdG3JAj/I3CdViWTfV6vcmYa64D8IOJEyemXHnl
    lfB4PHC73SKehGEYcTNKpVKo1Wrccsst2Lhxo97hcNwCrgvykpH0vvjii/D5fBLBmdTr9bkr
    V67EHXfcEZfrl5WV4dFHH9WePXv22aNHj2L69OnvFhcXBwc7umqYEgHwZ4Zhbt6xY8fUzMxM
    lJaWYubMmSgqKoLBYEBeXh5UKpVoGAYuiqIQiUTEiFVY4XAYwWAQDocD3d3d6OzsRHNzM5qa
    mlBdXY3+/n6Ag0JsS5BuRj4zWE7T9HXXXXfdRXlOhyvChB6FQoHc3FwoFAoB/lBLEMSLLMva
    krA988Dhyyck6Pol/PXvjlclgyAIVFRU1LhcrjMej6fCbDaLQH+ZTIaf//znuPnmm7F+/Xrx
    99PT07+uqKh4xWg0us+cOYPLL78cc+bMQUZGBvXVV1/d09fX91A4HB5RRUyA3MhkMpAkGetQ
    KliW1SfLThIEQQs4NqGSk5qaCpqmxeDN4/HA6XTC5XLB6/Vm8xnqg7jEZObXXXcdNBqNcs+e
    Pf+3t7f3MQDKkpISHD58GIcPHx72fikqKkJNTY3MarUuqq+vR2lp6bNVVVWO2trahOly4MAB
    yGSykEqlet/tds9bv379eJqm0dfXh1OnTqG3txcOhwNOp1M8x4RG04slGUiSPKcJVTjztFot
    cnNzUVlZCaVSiY0bN4Km6YMajeYTn88Xb+5aD4C3AdRHIpG3N23aNGHr1q0ivGc4wTXDMAiH
    w99p7jEajWKwKJFIRIgby7Kt4LDDOzFIWM1wH3ADgIf1er38ySefjOj1+rjS/WRlZZHLli2L
    HD16VGo2mxcB+Hu8Mgnbt2+3TZ48+Xetra0Tnn322TH79u3DxIkTcezYMezbtw/Lly8XR2gR
    BFGTk5Pz7unTp0Pbtm0Dy7KYNGkSVCqVrq6u7nm73b6wsrKSev3113HZZZfFTX8BRMswDOX3
    +2VJsDPTAFwRCoXw4osvoqurCz09PfB6vWIZVSqVQiaTiRhPtVoNmqYJcLNR38EgCYATISzL
    Eh9//PGtJpPpFZqmcxYuXIgJEyYMlmT4X4rQVf3iiy9qLRbLc4cOHQpPmjTp3ebmZibBqhkB
    /J2iqKnLli3DfffdB7VaPWiqHIZhBt39HIlE4HQ68fTTT+Ott94KAtiCxEIZXADO2u326zo6
    Os7rUMZOHYl9FTCjFEWJKxYrKgQ/A/Xu6uoSaIuMLMu6krA1KXA8htck+D6zwI3mfJEPREYs
    CxcubHvnnXfeC4fDLwWDQRHHk5mZiSuvvBK7du3CP/7xDwBAampqm8FgeGXbtm0ivdTtt98O
    mqapvXv3/qSvr+83gUBAO9K/aWCJXS6XC3aS9ng8imTYSYIgLl+wYAHuueceGAwGpKenQxhh
    S5KkuD8DgQD6+vrQ1taGV199ldi/f/9NvIPQdqns5IMPPgiXy5V24MCBx3t7e5cCUEgkEmzZ
    sgXV1dUjatCJccBSrFbrIoIgUFZW9hy+O287rhIIBLBgwYLGr7766m/t7e3jly9ffs7/CxAE
    rVaL1NRU8RyLfR0InxGWkMF0u92wWCzo7OzEyZMn8de//lWwLdG0tLQvTSaTOUHlfQbcIITj
    ACbE7n2CIER2AYVCAblcDqlUeo5eUqkUNE2LmNDz6ed2u+H3++H3+8EwjAiZ4qUBXPVx0Ofc
    cB3KGwBU/uhHPwrPnDkzJRGf5PTp0yW33357YM2aNZfxTssf4nHdF154AXfffffOdevWPWI2
    m3+9efPmSdXV1RKKosicnBxceeWVWLduHRobGz1ZWVlvHzx4UOzIu+qqqzB27FjdkSNHnrHZ
    bA9cccUVKWvWrBEni8R9NzEMQdN0orFBNIAfAFA3Njaira0NOp1O7IrLy8tDSkoK/H4/PB4P
    PB4PDh06BKfTKRj3MgDfA4exuCTi8/lkTqfz1mAwmBOJRPDHP/4Rf/rTn+LttApZW43Vap1f
    U1PzJyR45jxBEAzLsk0Mw0SCwaBEo9GIndBOpxNWqxUWiwUejweBQEAsfcSWuFNSUsTSjVDW
    EXCUWVlZyMzMhFqtRmZmJpRKJZxOJwA4eGc20RnYM6FQiDl9+jRZVVWF7u5utLW1wWQyobe3
    FxaLRQxqBIxoIBBAKBQSHUohay50OQqvGRkZyMrKQk5ODsaOHYu8vDycOnUKgUCABXAWCR5b
    yMsEcOV9SYLvI+Ed102IE49mS0sLM3PmzD/t2bMnt6en56ehUEgNcDjGRYsWweVyweFwgKKo
    qEql+qCysrL26NGjIAgCjzzyCDQaDfXRRx/92GKxvBQIBLLjZA/PcXqEQ5xhGJJlWVkS7OQP
    9Xq9evHixZgxgxvEMrAhTqgO6HQ6GAwGTJ06FU6nE7W1tRMCgcAcxKnaNhyprKyUrFq16mc9
    PT2PlZSUyB966CFkZGTErdObIAi4XC6sX78+5fjx44saGho8c+bMeWbXrl0JzcquX78+QlHU
    MQCRtLQ0SXl5OSZOnIjS0lIUFBTAYDCI9E4Dqzg0TYMkSZFHNLaSI0CEnE4nuru70d7ejqam
    JtTV1aGurg5+v9/t9/tPxwPqdhEJ8/ZK5BiuqKhAWVkZxowZA4PBAK1W+x3dBGdyoG6xOgaD
    QbhcLphMJnR0dODMmTM4deoUTp48KZwDRgxxGs5wDJ0SwPd1Ol3KbbfdFuazVPG3kBIJeccd
    dxAffvghZbfb/wtcc8CIsworVqzAbbfdxjQ1NX01ffr0k2azebZGo5EXFhYW1NbW3r148eK8
    trY2Rq1Wf3nZZZdVV1VVYdOmTcjNzcWYMWPUe/bs+XVfX99PZ86cmbJmzRpMnTo1kZk30DSt
    EDpXEyTjAdwolUpx9dVXY968eZgxYwZycnLEqIckSUQiEQQCAXg8HrS0tGDv3r2orq5GXV2d
    lGXZW8E1BrguhaGMRqMky7JUSkoKZsyYEVcjGWss3W43Dhw4AL/fT1yIsD/e3z+AdpZlXdXV
    1TqTyYSzZ8+KQOxAIPAdGguCIMRMHUmS52T3YkslAgZToNopLCyEwWAQRtxZkRxqEyPDMP4/
    /OEPik8//RSdnZ1wuVyIRqNiA9H5om6VSiWSuQeDQXi9XrFDPTazEAwGQZIklEolcnJy4Pf7
    EY1GQ0lwlgGuEecWcDPUkyEFAP4LHJH8iDf/O++8gzlz5rimTJnyfH19fU1PT88tqampivz8
    fEN9ff3lgUCABgC1Wl1bUFCw3ul0RgmCwIwZM3DzzTdTS5cuXWA2m1/2+/05iVach+SkJthO
    TgBwY15eXqS9vZ2sra0leayxyIAhVHIEqpr8/HxhZF1Ip9NJTCbTreAa05yXwk729PTIPB7P
    VVKpVP7MM8/EDRI0UEpLS3HXXXel2Gy2KxsaGmgkuMz/4YcfYuHChW0AnFVVVenvv/++mAgZ
    igymkSccDqOlpQV33nknjh8/botEIu1///vfE3oMgGtMDhcXF9MffPABiouLh9x0NJjfj0aj
    6O7uxn333YedO3cyvCM7pO9uOA6lAcBlkyZNYiZNmpTQyLuiokJSWVkZ3rVr10RwQ9vr43Hd
    zz//HIsXL4bX623t7Oxs5VPZkgkTJuxqaGi4OiUlxZKXl/fl8ePHXd3d3VixYgU2b95M79q1
    6+c2m23RVVddJV2zZk1CcF8DhaKoRJO25gDwBoPBwMGDB2VGoxFbt25FXl4e1Gq1SE3gcrng
    dDpht9vR1tYmlg9ZlnWCK+1pL5VDyflHLJRKJZ5++mnMnj07ITdpbGzE/Pnz0drayiaRML0d
    gL2mpkZ35swZaLVaZGVlobKyUoy+VSrVOSByYdE0LXY/x4LQ/X4/HA4HOjo60N7ejo6ODjQ0
    NGDv3r1wu90ABy9xJEG3LgBOs9msSE9Px3XXXYeCggIUFBSI3ecCYbRQVhReKYoSs0PhcFgs
    1QjNRmazGR0dHejo6EBbWxuMRqMwG92VJGdZC+BG/tlIhpD8/d5BnHhrd+3ahdmzZ3vPnj37
    l4qKis1Tpkwhp0+fnvnqq6/+tLOz8waZTGbT6/WvHTp0yEgQBB5++GFIpVL88pe/nGsymV7y
    er258Q6wLhQo0jSdaDupB+Cqq6sLPPDAA0o+OPnXaU0uMCJ9Pp+DDzK0l8qhZBiG4JM1CIVC
    OHXqVEICb2HyFsuyzHAnsA1F7r77blAU1Q7A5na705VKJSiKgt1uh91uh81mQ29vL5xOp1jt
    ECoefr8foVBIDGCFYECodqjVamRnZ0Or1SI9PR0ajQY0TcPpdEIikViysrL6TSZTMs4Ap8fj
    yRBo/Pr7+0X9BKzo+XQLBoOQSCQiZVJsJSctLQ16vR7p6elIT08XM508ht6HYcDYhuMQlgLI
    mjx5MqNWq+lEfoppaWn01KlTg7t27crgM2n18br2m2++CQB45ZVXcM8992DDhg0Rq9W6IxgM
    7ggGg/D7/ejq6gJBEPjNb34DnU431+FwLCkrK0tdvXp1UpxJ3mmR84YoURx5u8GVvK/weDxX
    NDc3j2tubp4MYKxarYZA66LVahmXy8VGo1GWz4KcBkc+fwhcmc2SLMO4fPlyVFdXw27n6LCE
    WdQCrmQ4lBGDEZlMJmZAotEoK3CO0jSNyZMnY9OmTYnKdKG8vByrV69GSUkJdDqdSKMzUhEy
    fDabDYcPH8aSJUtgtVqT9VVqAMjmzZuHt956S2wwiofE0nyEw2G43W488sgj2LBhgxQcQ0Wi
    ZQyA4iT7DEV8RjRuX+C+ffvw0EMPYcuWLZG2tjaMHTu2e9y4cc/Z7fbVWVlZkYaGBu+KFSvE
    gOvkyZMVLpfrhVAolB9v5S7m/PCBdyLt5LcAfhgMBquCweAV4Dr2JwMYp9VqGZZlKYfDAb1e
    D7fbDa/XGwFwIhwON4TD4SbeTtYhgVPgBhV1kCRcLheWLFmSMDsZDodht9vjwtQwhAxcJBAI
    RNvb2/Hqq6/CarWisbERZrMZNptNzCBfLCg532clOGM6nQ45OTkoKyuDVCqF1WoFRVFBpVKZ
    jCYrN4CA1WrFmjVrEAwGUV9fL+rm9XrFKtRQdKMoCjKZDFqtFnq9HuPHj0dmZiba29sBrtQ9
    5ATRcBzKbJIk5SUlJUkZazd+/HiWJMlUhmGyE3H9ZcuWYdmyZQCAG2+8UczCGQwGEAQBlmUx
    d+7c7P379y9RqVQZTz75JKZMmZLM1Ftqf38/ieGXDShw+B+CXxGcO/YrBKCVX5/yAcMavV4/
    dtWqVdi8eTP+8pe/4Iknnoja7fbImjVryFAo9DdwBM2OizhBUj5rwvBGPhRPY5+SkpIqlUpp
    AOzBgwcVAGiWZeHz+QSy2bg798J1CYKQpKenq6VSqR8AIZFIwgRBJKoGXg5AP3v2bNxwww2C
    Myt2aTudTpF7MZaHcmB3vpDdE0DcSqVSnBqh1Wqh0+mQnZ2N9957D7t37y4F152cyIwzAWAq
    RVGa733ve8jIyEAoFILD4UB/fz8sFgtcLhdcLhf6+/tFzG5sNlLQTaDASEtLg1arhUqlgkql
    QkZGhshlqdPpcM011+CTTz5Ji0QiU8F1sSeSyLwYQAaSK+ngur6PxPOif/gDB183mUzYs2cP
    Jk+eHI1EIk6fz4dZs2bhhRdeAABkZGTII5HIklAoNDmB9vCCgXc0Gh2JQxlrJwV7G4uRDoOD
    ShjBTQYrAfBmWlpa3qpVq4Lbt2/XfPHFF1i6dCkcDgdWrVrFhMPhb8A1SjkuklWOtZNMPO3k
    888/j6+//hp2ux0syyIjIyMKgGUYBiMYETj4D5SbQMMUFRUJ8C1MmzYNGzZsiPu9fD5fEYBM
    q9WKl19+WTgjkJaWhpycHKSnp4uwGSFDJ6zYSo6wfD6fCJ+xWCywWCzo7u7GgQMHYvddfktL
    SzYuQvQdJykBkO5yubBmzRpRN4VCgaysLGRkZFxQt5SUFITD4QvqZrPZYLFYcOLECdTU1MTe
    U8Xf99tEO5RqiqLI9PT0pNT8srKy0miaZoPBoDrR99q+fbv485kzZ8SfjUbjDYFAYMZdd92F
    +fPnJ9yRjjWaDMPIDhw4MNQbSsE1y0znM7tZ4Ea30fhnw0UnuJKjmXccJOBwQouUSuX/Wbp0
    KebOnYsPPvhAAPXSjz/+ON3R0YHPP//8pwzDpILDtZp4I5zOOyEGPjtTwP8dIT7a6QRwCsBh
    PpU+kg7pFLvd/nhvb+81DMOwJpOJCoVCE1mWxfLly5Genp6QkWICoS3LspP7+/s/7e/vj5Ik
    SUil0sPFxcXPI/5NOiSAKolEohwzZgx2796NEydOoLm5Ge3t7WKpQyCcD4fDYgQ+sHkhlgJD
    cCiFqLuwsBClpaUoKysTxiHmguMXPZ3AbZ4KYJpcLifdbjfeeOMNHDt2DEajUTTgsZG3MIkk
    NsKO1VPQT8CPCmNCMzMzUVhYiMrKSgSDQaSmphJut/sKcFjwRHax6/hnLplCJ9qJjbV9Ho8H
    XV3/nLba1dU1KxQK/TCZzmRs4F1bWztUOynj7eQM3vYJdjKFt4mtMXbSxNtOin/Pz+Ry+dWP
    PfaYb+7cuWmCnbTb7Vi8eDFaW1tTvvjiiwcjkUgquHF1Jv55zuBtZB5vIwv4+wV5+9HF28lD
    vJ0eiZ0kAGSzLJsGgN24caMiGo0qZTIZbrrpJuTkJAbearfbsW3bNvj9frlarS5lWTbA/y0e
    AD0JCuSmANAVFBRg3rx5KCoqwtixY5GTk4OsrCzodDqxSSXWTgjd6QLGfOAKhUKw2Wzo6+uD
    yWSC0WjE2bNnsWXLFvT29uZEo9HLaJo+HQ4nrMeP4s/xVL1ej+9///soLi4+R7f09HQI0+3O
    t2KpkmLx9OFwGP39/ejr6xMnAbW2tmLLli1ob2+n+fv+GUOYOz8ch5IiCEIc65NoibkPiUsg
    c+bMSTObzbfL5XLZrbfeCrk88QMZBNwJPwqK5lPQg5UKAIspirohIyMjr7i4WJKTkwOVSoVg
    MIju7m4YjUbYbLaw3+/3RqNRD3+w0iRJZhQXF2sWL16M++67D3v37sWxY8dAkiS7bds24v77
    78fq1auh1+s1H3/88S8tFssCcLQQBAAVTdMKhUIh1+v1VFFRETIzM0EQBGw2G7q6utDS0uJ3
    u90tLMt+AmAdgL7hPmR+v7/S7/fPyc3NhUwmEw8bm82WsJItQRBC5loL4NpQKASTyYRIJBJh
    GCYR9R0lgCqGYbB27VqsXLkSDgeX7EhNTRUzi4WFhdBqtdBoNOc0rwh0GEJHXzAYFKcqOBwO
    tLS04OjRowgEAiBJEjqdTvgcU8FRSf0FiQPUGwBUeL1ePPfcc/D5fCAIAqmpqdDr9ZgwYYLY
    qS1gmITORYFjUzD4sbyafX196OvrFlfq6gAADhFJREFUg9VqRXd3N44dO4b9+/fjww8/hEwm
    E5gJyvn7J9JhVl8Cm0Xy9026LFmyRPrll1/+OBAI6BKmHH9ACiJgGHk7mXLy5MmhOJSTwA3N
    uAEcjvxCB1oEHJ7MyztEFB88q5VKZaSjo0P95JNPSvgRhtFPP/1U4na7BVyeOhKJPAzgdnC4
    SZJ/phX4J5TpfBIA0ALgMwDvgiO6Ho5I+/v7n+nt7Z0HgO3t7SVDoVCGRqPBY489hlmzZiXk
    e2poaMCRI0dgNBqnWq3WrbwDSUil0p3hcPjhBATegr0i//u//xu/+93vLlpuF4LT2EA1NiAV
    HE8x7Z+ejtLSUvHfwkSuTz75JBXAFdFoNJF2UgugCuA4RN944w3IZLJB6SaU+GMTCgLNmhj1
    6nQYN27cOe+nKApvvPEGwEE6MjEEzPlwvML+aDTKOJ1O8iIPRNzE4XCwDNee2n8pDKXH46kI
    BoPTpkyZgmnTpiXlnkKmaaDhHOT3eS9Jkg/eddddWLRoESZMmAABpCxQ3/DRCN3S0qJpb2/X
    mM1mKBQKlJWV4frrr0d5eTk6OzsFLEq/VCo9WltbO2ft2rXUihUrsHLlSsyfP5/45ptvstra
    2rIAID8/H4WFhSgpKUF+fj6ys7NFjE40GoXVasXRo0dTf/vb31YcOnToV+A61z4dQbaCSUtL
    w+rVqzF16tSEZCT/lXPZ0tKChx56CN3d3dEE3T8fQJnwvZWUlKCqqgrl5eWYMGGC2JCTlpYm
    TpcRjMfAGbXCikaj5xAvd3R0oKmpCfX19aitrUVLS4sA9ZgKDuOYqNrYRAAGiqJgMBgwffp0
    TJ8+XaTDyMzMFHUaChZLOCT8fj+sViu6urrQ1NSEmpoaHD58GM3NzfD7/cnIwIaFgzSJ25JF
    cuiQviMnTpwostlsVw93KtVgkwuxB32snWRZdijMCxIA9wC4f5C/q8J5cLd9fX2S998XWYBs
    AA4ajcZ5a9eupQZkCTP5NZTM6UQAywA0jcBOksFgUO/z+cYMzPQKAWYioEHCdVmWlcXeOxqN
    6hmGSUSQpQdQIZPJMHv2bDGJYbfb0dfXd86IWofDAZfLdQ6FDsMwkEgkIo2Q0IwjTMbJyspC
    bm6u2LiSlpaGGTNm4LPPPgPDMFMZhlEjcWXvIgClFEVh1qxZSElJERtyLBaL2BzrcDhE3YTx
    iwKPL0mSom5SqRQajeYc3YQxtwJcaObMmXj33XcRDAaLwJW9E+5Qhtva2mgkoYPRaDRGIpEI
    g+R0nX6nxDJ27NjyUCikKygoGPZoquE4lMLDOdTkJoAtLMsuCAaDeePHjwfDMDAajeJ4Ka1W
    i8LCQoHO4pyoRPj5xIkTePnll8HTIWwPBoNrAeT8/ve/r6BpGvfffz+uvvpqXH311SIVTayh
    93q94sPLsixkMhmysrJQVFQEj8cDAEfBEbaOrBZAURg7duw5EVayJV5NJBeQywDkarVa3Hvv
    vfjJT36C8vLyIYPpY/eRMK5Rq9UiPz8fFRUVmDdvHtxuN44dO4b3338fn332Gfx+fxlvzBLh
    UBIArgCgyMrKwo9//GPMnz8f48aNG3EFQMhi0TQNlUqFoqIiTJ06FdOnT0dhYSHWrVuH9vZ2
    IaPxBUZWUryoneSvncwsJZPAAOCidrKkpGRiKBTKTeR9JBLJOZWxWKLnYdjJagB38JnqeMjf
    wI0qLeCzn/GQIxj5WNvveIxutxvPPvtsQujVhOv39vZeaH8mQsoAFMjlchw7dgzffvst6urq
    0NnZec5owuHoKthLhUKBzMxMjBkzBhMnTkQ4HBYmI5WBa9BKlENZBSBDKpWitbUVjz/+OE6c
    OCFSxwl4yOEEchRFibplZGTAYDCgvLxcZAkJBoMa3k5/m0iHsgWAdffu3Vm7du2CVCpNmFMZ
    iUSwf/9+mmVZM3/fpItCoSgGIGFZFskYeM+yLEZIQ7CfZdk/b926ddmiRYsoYWxaNBplJRIJ
    IRDuGgwGZGdnIzs7G5mZmQiFQujp6cGpU6ewY8cOtLS0RFiW3QLgOT5Kfszlcr3y0ksvVf7P
    //wPrr/+ehQVFSErKwvRaBS9vb3o7e2F2WxGZ2cnTCYTXC4XWJZlU1JSiNLSUgQCAdTX19vB
    zVAe8Zi4QCCAzZs3o76+/pJkKM1mM1wuVyL3hBVAg8PhqPjoo4/ompoaTJo0CePGjUNJSQly
    c3OhVCqhVCqhUChEztDzZSiFEojf74fb7YbH44HL5YLRaERzczOam5tRX18Po9GIQCAgNGol
    MttVC+Co2WyuWLVqlWzjxo0oLS1FcXExioqKUFhYCJ1Od45uAl2QgH8SdBLogwSeVLfbDavV
    ivb2drS0tODs2bNobW2FyWSC3+8P87bkdIKzhzZwpT1lErdlCIlvEDivSKXSskgkIk30Myc0
    ZAlz2ocZeIN31NYDeGpAYmSoWeUwgK95O3kWwFIAKwEMlaB44H3t4GYod8V9k4RC2Lfv32Ks
    eFyEpmkwDGN3OBxpL7/88lAqeoPyQSKRCLxeL/r6+lBfXy/OxWYYBjRNhwCkJghDSQDIBrgy
    +2uvvRZX3YRqlc/ng8ViwenTp7Fjxw5RNz4YLgCHzR6UgsN1KBv2799vmD9//qDHwA3XufJ6
    veAdmuZLsFeJYDCYRhAEtm/fjkTOJY3VWaDDEZIuQ01wAnjP4/Fc++mnn2aAmx/bCg4HJADC
    cwDoCYJIT0lJkdM0LWcYhuInkITAjVzaAOBjcE07QgTeGwqFlhw5cuTGI0eOZKWkpJA831g4
    FAr5wuGwBxwusgccCL0LHO5I19jYWA4O5PsVgL+O9HMiSZLw+XxYuXJlQvfgRdNBPLBZKpUm
    yjHZBeB2hmG+19vbO6O3t3fivn37xpIkqZbL5VKFQgFhprBQ9hayODRNC12WYmd0OBweOGNY
    4KgM4J/NWvX8nvkGiRunyYIjwj/KMMyVXq93WlNT06SmpqZSABqJRJIqk8kkwpSfgbqdj4dS
    mFHrdrsRCASEzsYowzCCbq38vj4KYB9vTxJJ+XGGfwZKk7gle8BReSXdTkaj0UwiwdF2XV0d
    fvazn4GmabAsixMnTog2cxjBQQTAH8FN+dLF2Ek/OIxkPrhSag7/bwH3KJyZQf5Z+QBcc6KQ
    ktsJ4F4AjwKYC67Rh4px+H3gMOt9vG0188G1h/87hCahr3mb+58kCdkfWq327z6f7w4AP0hJ
    SbkmEAgUhcPh9HA4HHdaQ5Ik/QRBWAiCOKtQKI6q1eqtEomkZoh9DkOxk+8BcLMse0s0Gh3P
    78VElsWi0WhUOAt2APhwKImF4TiUTgDvMgxzxul0skgs9YbgUO3DJSh5A2B9Pt9xiqIOejwe
    gtcXSdBZyDQRJEk2DuN+beAA5wH+YIvdEDSfNUljWVYXDAZzg8FgHr9RSd6JEIzrQDkB4GFw
    YN3KUCiUEQqF/Lxx7OaNo4M3mL4BB7aSf5+JN9ojkTBJku9TFLUvGo2y8YzahioURREURbVi
    iCOqBuuz8o5PM59N0QEoZBhmnMfj0Xs8npze3t4c/uBT8t+thH8VICkRfoVjXl0xTn8P/72d
    5Q+2fsRpHvQgjGU7vz4Dh1HLA1AQiUSyPB5PthD4gGs0kcXoJeFXdIBuwRjdzPwh38vfo5vf
    m8naLJ18FnZsMuwFf1gfT2AQcFEJBAJOiUTijcW7xrtqYLFY8OWXX4rXFaANfJOofxj3a+ft
    pJ/f/5EBdlIBIA1cY0Quv9L556oNXCd22/l8XwC/BAdZmcwH8gF+L3bz+9PJ20n/gD0pB1cy
    7xmpnWRZliUIIsDbYhaXVggkaFRtNBoNRSKRw7///e9rtm/fntHS0pLn8XgmdXV15cvl8rEM
    w+S6XC4VQRA6mqZVBEGkAKAZhqGi0SjJMIzQsMJQFMWSJBllWTYUCoUcDMNYVSqVmyAIk9fr
    bZZKpWei0WiDQqEwz5o1y1ldXc0mGArXBWA1gD+BY0+5DFxHeyk4XK6GXwreJgr0gGTMz2zM
    Emj8ouAazZz8svHnzAlwLANGfr8OqZY+3FbtL/lMU9KSQZfqKYhEIn+urKz8giAIwuPxsF6v
    N6kOjFarDbEsO9QbsvxhdqEMZj+/OvhDaCji4x3Og0N8nwcjxwOJXwvLslsZhtl6qUNuoZyc
    BAnHOEiHY4y04GQJdCcDHcroAGcyzBv2MC5RA8cFnm8Hv+oHBJMSXg+Sf5XEOJWxukViDGX4
    UtoMXvwAfgdgWxIP7NP8c5b0x4BhmHc1Gs0uJJZc/LyzvAEQSqXSzDDMcOzkPy7yvAl7shND
    n5Hu45/Tw8N436G4bAiCCLIs+xK4TOy/g0NpSYRTabPZkJeXh7feeovNzMy02O12i8fjOe7z
    +fCrX/1K8s0339BHjhyh9Xq9TqVSaQCkhMNhOhAIUB6Ph/L5fEhNTYVKpWKUSiUjl8ujJEmG
    XC6Xvauryzpp0qTwtddeG16+fHlE2HfhcBj79+/HwoULsW7dumR8fnZ+HQdXPZTxwYeWdyjT
    +cBbOmDRMcG2sAJ8MGOL2eMe/v9G5NyMhPvnUhvsxD8BBIH8/PwwQRCO83XPJuP+ybzf/yYR
    6B3+HUQikSQdwxlzIIb45f0P/JoFouf/rfKPizgs/2m2sosgiK4LOX/xtImjMqTnp+7/B0W7
    u7vR3c01I7/22mvYsmULenp6sHPnzojFYolEo1E/QRAuYVjJYPea0B+wc+dOjB8/Hl6vFwRB
    oLOTawFIkjN5Prvv55dtdJuPyqiMyqiMyqiMyqiMyqiMyqiMyqiMyqiMyqiMyqiMyqiMyqiM
    yqiMyqiMyqiMWP4fdSwxy5pU1hAAAAAASUVORK5CYII=
  }
  set pieceImageData(Merida1,60) {
    iVBORw0KGgoAAAANSUhEUgAAAtAAAAA7CAYAAABSS730AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw4UyunK5QAAIABJREFUeNrsXWd4VNXW
    fs+Znt57JYWEkgQSCKEmEEIHJUiRqiJCABGuCtgQKZcqerEC6kVRVPQiihSpKiUQajCUUEJJ
    L5OZZCaZvr4f58wQQhJKCoEv7/OcBzKz57S999prv6sBLWhBC1rQgha0oAUtaEELWtCCFrSg
    BS1oQQta0IIWtKAFLWhBC1rQgha0oAUtaMHjgYULF2L16tUtL6IFLWhBC1rQgha04P85mJZX
    cH8YPXo0SkpKYGtri4iICLz77rstL+UJwIQJEyCXy6HVasEwd04HIoJUKoWXlxc+//xzy+dD
    hw6FRqOp9Zwsy2LXrl13jJ3y8nLo9fq72hIRxGIx3N3d8eWXXz4ur00IgAAY/5/JSWqZMc0T
    U6ZMgUajgUAgwFdffdXyQlrQghY0yULYgvuARCJxtLOzk33xxRd5kZGR1L9/f+zcufMupcuM
    lStX4tChQ1CpVNi3b1/LC2ymuHXrFk6dOgWlUlnj9w4ODujbt+8dnx07dgwFBQX3fY1r167h
    woULKC8vr/F7Kysr9O7d+3F4Xb4AhgPoCEAD4E8A2wGUPcFDRArgRV6JXsc/dwuaGdzd3QVj
    x46lXr16mfr06YNWrVph/fr1df7m2WefRVFREfbs2dPyApspnnvuOSiVyhrJBwAQi8Xw9PTE
    Rx99dAdhoVar6yQ4tm3bdgeJolKpar2GUCiEm5vbHSRKM4UdgHgAkQD0AI4BOAJA+6RPfwAs
    gLwnXYEW8A/rzP8tB1AAwNDce+jChQtdc3Jy5nXo0GG/u7v71p49e/4THBxsGDZs2B2TEQDS
    0tIQFxeHyZMn2yqVSmP//v0r6lK2nyCwAPwBBPCT9jKAouZ8w46OjnYhISFPqVQqF9zNMDK2
    trb5VlZW2wBYJHJgYGA7JyenRCK6q0MZhmFYlt2XkZFx1vyZi4uLNDQ09Cm1Wu1V0zWsrKxK
    bWxstvHzoSnBAPADEML/fRnATdTMtAYB+A+AJIlEIjSZTNDr9WMArAewsBYlWgwgmB8T5QAu
    NffxUAM8AaTw7+o3AFktak3zw+HDhzv/9NNPA7y8vP7w9fU9vWHDBvWAAQOwc+fOGtv36NED
    3333HQBg5syZWLt27f+H1yQAEFhFPmfy62+zxfDhwyGRSBhert7xnclkAhGRXC6nqgr0pEmT
    IBKJ2FqFHsOYqq7Z48aNA8MtzrVeo7Kykh6RAu0AwINfW/PrWCNceTk8llekwbf/D4APAVTU
    oXy68uMhD4DqMRvTTgA+4teaFwEUPqkKdBCASQD6gWOyACAbwB4AX/GLd7OFUqmUFBcXx+j1
    +u5FRUVj8/LyNnl7e3/zyy+/XE1MTLyDZe7UqRMSEhKCdu3a9ZZQKFR06dJlWVRUVAGebFjx
    /TuV718DgFMAlgE4iGZq/lapVK45OTlvFBcXtzaZTHcxFW5ubmdbtWr1V1UFOj8/v3teXt4q
    o9EoIKKqghkCgQACgWA6AIsCXVZWZpOdnT27tLS0My+Q77iGo6PjNV9f35NNrECLAIwEMJtX
    cgHgCoD3AWzhGYyqcuJFlmUHDB48mBk5ciTUajXWr19ve+LEiSl8P39bfW8CYCaAiQBc+PFw
    HMBSAH8/ZptCs8sK+4TPYVsArQF48X/n8UqWsrnfeH5+vtfNmzdfBTAlLy9vd/v27b9s167d
    0WvXrulSUlIwa9YsS9v+/fvD2tpaFhERMYKIyvr27fv7+fPnDU+4pdAawPO8fPYG5351BsC/
    AexrrvJ5w4YNzjdv3pyiUqk8aiIf7OzsbkRGRm6ouoFfvHhx98LCwuQ6CI4tAA6ZP1u9erV1
    QUHBFLVaHQjAVAPBUdimTZsveIW0KcmNHgBmAYjmZU86rxDvrXafLK84v8ArkmZ4AHidn8M/
    Vzu/lJf/k3ndrBLAXwA+4K/zuMARHOMuAkfMPpEKdBS/MPdydnZmAwICwLIssrKyPIqLi6MB
    dOMX8lPNtZcYhtEQkVYsFksZhgnKycl5q6ysrH9wcPD7CQkJv0RFRenOnDmD0aNHQy6Xe507
    d255QUFBskAgMOr1eoegoKC58fHxhQcPHnwShTMDzrS/xM/Pz6Fz585Qq9U4cuRIklKpdAUw
    AcA/zfHGjUYjo9frWZFIhGHDhsHJyQkAUFJSgl9//RU6nY4lImbQoEGWxXf58uWMVqtl2rdv
    j+7du4NlWZhMJhw+fBjp6ekQCoXM999/j2+++QYymQw5OTnQ6/WsQCDAkCFD4OHhYd6UYdu2
    bdBqtYzJZGpq80QfAMtdXV29u3TpAgA4evRodHFx8XIAxQD+qNLWHkAPNzc35tVXX0WPHj04
    OksgwLRp02z0en0PAD9WUboFvDCf36pVK2nHjh1RUlKC1NTU/pWVlY78eMhs5opkVwBOQqHQ
    y2g02hGRCcBQfhFVAjj8OCiWDzB/4wDM4BdtJ/7zUv45P+YVDlNzfQCdTqfX6/UGnU7nXlFR
    MaG8vLz35s2bN3t6en42a9asay+88AK++OILzJgxAxUVFZI///xzen5+/psCgUDz9ttvL0hM
    TPzqqaee0s+cOfNJlc/JABbxc9mM3vzmdnxzVZrKysqccnNzJ5eUlLSq6XtXV9ezYWFhP1RV
    oIuKijplZWW9UpWoqEZyZFZVoEtLS61ycnLGl5aWdqjpGg4ODtcDAgK2N7EC3RHAWgARVT7z
    BRDKy9a/q22OBlRTni23D6A/OOuZrtp6vZrv/6okpx9//hvNfEyLAYhEIpGNXq9nqrwHG34d
    emLcVlwBbGNZlpKSknR79uwx3rp1i7Kzs2nv3r3GAQMGGAUCAQHYAc5c2izRuXPnbiKRKC88
    PJw+/PBDio+PJ7FYTBKJpNDX13dh3759nSMjIzF16lSRn5/fUoFAoA8PD6ewsDASCoW6wMDA
    dxcuXCjq2bMn+vTpg/79+6Nfv3544403nhTmaoezszN9//33pFarSS6X02uvvUYsy5oAzEUz
    DVhNTEwMdnZ2zvT09KT09HQyGo1kNBrp9OnT5OnpSa6urmcnTJjgS0QwH35+ftMAGKdPn06V
    lZVkNBpJq9XSrFmzCACJxeIUAJb2SUlJro6Ojmlubm50+PBhyzUyMzOpVatWZGdnd23o0KGR
    DfhYInCscncA7cBZB6oLn68kEgmtWLGClEolKZVKWrZsGYlEIgKwAYCk2hw+7uPjQ2lpaWTG
    d999R2KxmAB8WQPzcdzd3Z1+/fVXUqvVlJeXR+PHjydeuKXUIug78Iqr7yMeLz0B5ADQMgyj
    55kv4hcgLc9y9HuCFKy+ADKqPGf14xKAQWjGQecRERHxEomkqOp9CwQCo4ODw8GoqKg+n332
    Gfvcc8/h7NmzaN269WiZTFZsbmdlZZUfHh4+nIjQu3dv9OrV6w7G+kmRz7X0rQnA/Obat/Hx
    8SEuLi6Xaxubbm5up5999lnvZ599FmPGjMHUqVMRHBz8CsMwptp+IxQKU1566SWMGTMG48aN
    Q0xMjKujo+OJ2to7OjpeGzJkSGQDP5ozL6O9aiAyhbxyW9t8XFdNPjuB83Wurf2PAGTVWNu9
    tbTV1SKfRQB8eCXb4REPi0B+c/GTUCjcw7KsGpzryR8AfuLXr7ZNcSNNwUAnAUjs0KGDdu3a
    tWxoaKjFDOrt7c36+fnRhAkTjKmpqb0BDATwRXOcyCaTKY9lWaVSqfTo2rUrRowYgS+//BJr
    1651zcvLm2c0Gn3at28/Pz09Pbi4uHiiu7u7cNWqVRAIBJgyZYooPz9/yi+//JLm7Ox8wNfX
    1/DVV1/p3N3dcf78eXTo0AFisRjHjh17XAW0PQAfDw8PxMbGwsrKClZWVoiLi4NIJGK0Wm0Q
    PwF1zfkhWJaF2QfO/K9Go/E6duzYgtDQ0HKOwGBIp9NF8v+HQCAAy7IQCAQWH3dra+sRAQEB
    wfxYJyKS6XQ6X5FIZGlf9RqNIJhnAHiWZxfUAHYBWA7gKt9GAsDbxsYGXbp0gZ0d5zIXFxcH
    KysrKJVKP3AmPvMuvgxAekFBQaevv/4aAFBZWYnvvvsOOp3O7KpT1eXDBYCLp6cnoqOjLeOh
    c+fO2LRpk5CI/MGZHc2MZmcA83gWVMSzH5+Acwt5FEF71wB8BsCdYRgnAIN4c/BOACXgGPpL
    T4hy5Q/gXQBt6mgTCs6/MhPN1NWOZdlCIiqvyqgZjUZWoVD0unr1asD777//7ogRIzbNmjXL
    p6CgYE5lZaU5DgcVFRXu+fn5c/v163du//79lz08PJCZmYnIyEjY29tDIpE87oGGduDcNmoC
    wytFzV4+13jzDMOIRCKJl5eXxGg0MnZ2diQQCES8MliTCwdEIpHI399folAoGKFQSAUFBRKG
    Ydh7XKehblkK4Blwrm0BABQAfuflTV4VgiO0jnO05kkRs3xW8/Ioroa2BOB8NUbWGbfdaGtS
    lMN43dBQRWGdDiCRZ3kv8Ur8zmpyv6ngxhMYngaDga2yOegBzjVJCc4NMeNxV6AFALoKBAKr
    UaNGVYSGhkqqNwgJCWFGjx6NEydOSAwGQ3cA3zTHiRwQECC/detWoUKhaH3jxg3ExMRg7ty5
    CAoKwoIFC8TXr1+feOnSJbFEImF0Op3XqFGj0LdvXwgEAkyePBnvvfee58WLFz+SyWSZ6enp
    efb29ulOTk6XieiMj49P3v79+w19+/ZFQEDAPaPHmyE0AMpKSkpw6dIl+Pn5Qa/X4/z58zAY
    DABnCn7sUp4xDIPy8nKXy5cvv1BVgNZkGqwKhUKRUFZWllC1vclkgo2NTWPfsphXnucHBARI
    goKCkJ+f73Tp0qXJBoPBlheCcn5+FajVapw5cwbR0dEAgNOnT6OyshIAcqsJXC2A9Xq9vvOn
    n37a/ueff4Zer0dxcbERXBaOrbjTP7EUgKKwsBAZGRlwdnaGWq3G2bNnQURG/vxm5TkEwAdS
    qTSuXbt2sLGxwfnz550LCwtXgPPL24ym98/MBrAYAENErYmoEy/LFgG4wN/7k5LSrj+AmPto
    FwlgCDhXvGYHmUxWZG1tXaLT6QKrf1deXu5vNBqXb9u2TSgUCp3VavVdpvqysrKYf/75Z7WX
    l9cuOzu7YrFYfEUoFN5q166d/JNPPjHGxsbCw8PjroDxx0k+1/F9CR7TlJRqtdr/77///g8R
    VZpltlqtDiGiGhViPjXpxA0bNnQz+0gTkUSr1QY0xT4PnK/yCtx2kzLPLQ8Ar/H9ZOQV69pQ
    Wk1x1QL4GkAvXtmtiuPgGOiq7lcVqBLTU4PCLa8yHtz4+x2O2zEgweDcclMA/PoIuv0Mvwmx
    BceKr+DXvjf4DX4FL6cbHY2tQEsAuEgkEoSFhclqpTdCQwUSiQQGg8GF36E1OwU6OTlZdfTo
    0dNarbZHRkYGkpOTIRKJMGbMGIhEIrz88suC7OzssSzLGpydnS3fA8DIkSPx3Xff4eLFiwEm
    kylAr9fDZDKhoqJCJRQKM3Nycg66uLjscHBwSFu/fn3Z3LlzsXz58sdJjpUC+C0/Pz/m9ddf
    l/Tu3RsqlQrbt2+H0Wi8xTOgj5WA9vT0xBtvvIHy8vIa2QciQnR0NAQCgZkBw+DBg+Hu7l5r
    e2tra/j6+jY2mzg6ICBA8vnnnyMmJgY5OTmYOXMm8+effw7mN6c7eIG7VaPR9F+2bJnL0aNH
    AQAHDx6ETqcrAvAL7mZ+0wBMMRgMM3NzcwfwAvhbnjnJqda2EMCveXl57V555RVRbGwsioqK
    wPv/Z4ALWjJjKIDYkSNH4r333oONjQ1+/fVXvPLKK05lZWUT+LEjfwRDgLhuI2OVvw14snJf
    i3jlWXyfa0U0L5+bXSq/pKSk8hs3blwuLS2tcTNQUVHhlpWVtVwsFmv0ev1d657RaGRzc3OH
    sCw7pKioSMswTJFUKr157dq1w76+vgekUunJbdu2Ffbu3RuzZ8/GkCFDHqd+VgDYxve1tNp3
    N3km8bEc1yqVykGlUg16oMWqtLRDbf7Ojb2sAJhSTXk2z63RPBGxi5fPu3jZaFd9KPP9VV0B
    /gtc0PYrADrxsmofOFeQ6spkMbjkDZG4OzA6G8D+KgRBPwCDa2jnzT/L3/z635TQ4naQfiD/
    TkwAUsGx7U+UgN4gFovp888/p9rw5Zdfmn0pN+NOX51mg3HjxqFt27YjBAKBunfv3lRYWEgG
    g4F2795NycnJJJPJLH5EXbt2paKiIsvzaTQaGj9+PAkEAkpJSaGlS5fS0KFDKSgoyOxzShKJ
    pNDNze372NjYPn/++ac4ISHhcfOPdhYIBB8wDKPFbX+qbHCZOUTN9aar+kBnZGRQU+LKlSsN
    7QPdC0Bh3759qby83HKduXPnmouezKjSVsozCBd4YawD57qQUsMiWxV9wTHIJwC0qqOdO7iI
    7hJe4JWDC9wZiNumVQbAZyKRiNatW2e536tXr1JoaCjx9xb4iIdIILgA2Ix7PO/jCCsAP6B2
    38nqx281LOjNAkSEkJCQGUKhUF/TvUskEmrdujW5uLjU+nwikYh69epF8fHx5OHhQUKh0Py5
    3MnJaW9gYOALTz31lBsAvPjii49bXzsBWMkrG1Xl8/jmLJ/v5QPdFIejo2NDyefOvOyszRf9
    X1Xa2gBYwLfX80cxz7Y63GMNuArgJOp2y/LnCZCyKue/AuClauNhVR3v5jJuZ3B6VAgG51p2
    HU3k91x959OY0AM4q9PpDLt27RKOGDHCkuXAsjVWKGjnzp2k0+nA7yoqm9skXrduHXr27ImB
    AweyDMMYT506hePHj6N9+/Z48803kZ6ejujoaPj7+2Pbtm0ICAiAvf3tYGeJRIKQkBAwDIPW
    rVvj5ZdfRnl5OW7cuIEDBw5g8+bNOHnypGthYeEojUbTY9asWV+HhoZ+snTp0lspKSn45JNP
    HgcBrQaX45dsbGyg0+mg0+n04Py69I/DA+h0Ouj1+nu6aDTk9Rr4WgoAlfn5+cjOzkZYWBhK
    S0tx7do18Ap0cZW2GgCfAzgAIIFXruUAdt+DYQwAZ9YzB5Vcq6VdAc9yDOYV0A95pftmVb0H
    QJHBYMDVq1eh1WohkUiQlZUFhUIBntl4lPLAn2dgNLyy359nf56UXNBaPBi7L0czjW7v2rUr
    RCJRBsuyqqoKBsMwaNu2LaZPn47Y2FjMmzcPf/zxR43nEAgEGDduHJ566ilcvXoVp06dws6d
    O/HXX385yuXyPmVlZV0NBsOIAQMGrBk6dOj+tLQ0w+nTpx+X/P4VNTCFRr5P9WhBU0BTx7sm
    3Mkqq8DFrRwAly1lNC9TN6Bu9w43cOywPbjg79pwA1wO/yj+vr4FVxjrLO6sy6F6yOdpKgj4
    9yLi//9EKdDgF5zxu3bt6rR27VrMnTsXUilHcGm1Wvroo4+M27dvF4JLo/Nbcxz1U6ZMQWxs
    bI+SkpJ3DQaDrUKhwG+//YYuXbogJSUFYrEYCQkJYFkWMTEx6Nixo8V9wwxnZ2eYTCazYgBb
    W1u0a9cO7dq1w9NPP43ff/8dX3/9NY4dO+aVkZHxWm5ubuf27dsv/Pjjj/92c3OjZl46XAog
    xWg0/svX11fy9ttv48qVK/joo48CKioqVvGTbH9zZa4YhoFCocCCBQvg6OiI6vmgGwO8fzWK
    ioogFDbYNLwG4K/z58+PS0lJQXR0tOn69essX1b8NDh/uOoL6AW+/+aCCybqgdvBhtXBAgjn
    BZUdOHb2r1rainjlOYhnug7Woqz9QUQTvvrqK7/CwkI4OjriwIEDKCws1POyo/gRCeX+4AIb
    Y1iWlfAZVd4H58P4bzzGZu9q/X8MnJVIeh9kyLHmqEBHRkbCxsbG8+rVqy8ZDIY7GPLu3btj
    7dq1iIyMxI0bN+qsUGcymaDVauHi4gIXFxfExsZizJgxOHDgAL755hscOHBAduvWrf7l5eUR
    169f/9zX1/fTqKioIrMMacaQgYt/eA13Wnj9wLHSOn6z2yzB+y0jLi4Otra2TXbd8vJypKam
    NiTJcROcK5xfDd9d5+dXdQX1L14GjgPnmtEJtacAZcAFGIp5+dyKV4prk+W9wQUMHgfn3ldT
    kau/wKXv86jlu8JHNCysAQwDl8s6iH+eZeCycPyEJ6w67jAA1z09PenQoUMWU+3Ro0e1np6e
    enA+lKOa5Y0PG4aOHTv6OTo67hOLxdSnTx/y8vKiNm3aUFZW1n2b61NTU+mZZ56hXbt21drm
    +vXrNHv2bLK3tycAZGdnd7Fjx46DMzIy2Jdeeqk5C+c5AEr9/f3p+++/J6PRSEqlkubPn0/W
    1tbEb44SmuPN9+3bN8jV1TUTVUxTVlZWjX7gbhNhRAM9UntwgR1l/MJYCc4vbHAdv+kE4BZ/
    P1t44VSb0NqG2ybHf6P29FehAC7ybQ/VwYYIwfnu3eIXDB0vyD+tRWg3BRIAXHN1daWJEyfS
    6tWradmyZTR48GBz311rruP5IeAJzi++1rRf/LG/loX/kWLSpElITk62DggI+EAkEhmq3/cz
    zzxD586do88++4wSEhLucLWrfjAMQzExMfThhx/SP//8QzqdziKbFQoFbdmyhTp37mxOhaZz
    c3PbNHjw4ECAq5jXTGHFK86KOvr2H3AZFpodevXqFeLk5HTZw8ODUlNTqbKyklQqVaMfFRUV
    dPToUfL29iZ7e/uGTGPXBVxudR3/7o08wTClDgY1ApzVy+zmalVLOzGvCJv7dTlqL/7kxSvs
    BM4y6FHHOefxLK9ZRmh4mRH+iIaFA7hgZoW9vT2FhIRQUFCQWTaXg8tb79wUN9LYDLQzuNrs
    CQAYuVyOwsLbG5aioiJRaWmpeQGO5zvnAJpRuV83NzdhamrqcwqFomdCQgI+++wz7Nu3Dzk5
    OXBwuP90iJ07d8Z///tfC/teE/z9/bF48WK0a9cOy5cvR2ZmZuurV6+uGT9+vHDUqFG/zp07
    19TMggsdeWZjTmBgoMOKFSvw9NNPg2VZ2NnZYd68eRAIBPjggw/aq1SqDwG8xU+8ZlO63cnJ
    qUgmk/0qkUhe0mq1NjY2NnjuueeQmJjYKGnmiAh//fUXNmzYAIVCAbFYXGllZbVDIpFkN9Al
    zoEraRrHKzyBPNNgXcdv2CqKcBw4X7LjNU0HAH7u7u5QqVSMWq0O4ZnLmtws4nHbf5mtQ9Fm
    wflnmlPXmUt+p/L/bwgw4AKazemtKutgj50AzLGzswtcuHAhJk2aBJmMI+0mT56MVatWYc2a
    NYFarXYOOHOnvA7ZasU/n+Ee13yUyGNZdg2AIJPJVFvqrGvggpFuNreb7969O5YsWTIkLy9v
    gl6vv0sB2b59O06dOoWcnBxoNJp7zs0TJ07gzJkzaNWqFQYNGoRnn33Wks5uxIgRaNOmDZYv
    X44tW7aIiouLx5w9e9Z22LBhr23bti3zvffewzvvvNOcXo8TONes2bizgEp1tAUXr/AWuKw6
    zUY+C4VCE8uyJrlcjq+++gpz585FYGDjh0XcvHkTGzduRFFREaytrU0syzbU3E0FZ/EZAM4d
    zpVngfPrkA9V5Wc3vv2pWtZj/yp/B/PyuaYy3nG47SNdl3zW85svczXDSl4+/467g8ebAgIA
    ExiGmZaQkCB9+eWXERoaCpPJhLS0NKxZs8YmPT19Mi+rVjf2WG5MBToWnFm4DwA7Jycn9OrV
    C1FRUZYGkZGRzJAhQ7B//36vkpKSqeDMo3/yO6fDaAapog4dOhRQXl6ebG9vL5wxYwaCg4MR
    GBgIIkJhYSF2794NZ2dnJCQkWDIy1LiCMwysrKzuTRdYWWHSpEkICgrCnDlzcOrUqeCLFy+u
    +PTTT9VZWVl7goODm0sASytwaWOeDQ0NlS1fvhxDhw69Q+m0s7PD66+/DpFIhPfff7+9Uqn8
    GFxQwpcNqBzVCz/88ENZUlLSQpFIdCsvL+9farXad+fOnYiKisK4ceMgFosb7FoGgwFbtmzB
    77//DqVSCZlMlufh4fFBcHDwukuXLika8LEKwGXSALjUYy+Bi85ORc0VplgAjL29PVQqlafR
    aEwCZ2qsPv/cAXhFRUWhqKgIp06d8gOXd/dWtXa2AAbKZDIxwzCoqKgQ1CGgu4IrLyzn2ZXr
    Dag0B4ML3OnAM61WPItSwCuFJ/mFIbfKswYA6BQREYHhw4dblGeAc8NKSUnBwYMHkZqa2pnf
    IMirXM8LnAm1Ez8/PHgZW8kvkGfBFTy42EyU6RAAw00mUy+GYVzusVF+GZy/5FY0o0j39evX
    u5SWlj6v1Wodq34uEong6+sLhUKBq1evPvA8zczMxOXLl7Ft2zaMHDkSU6ZMQWBgINq0aYM1
    a9YgPDwcK1euZHNycgZrNBqKiIiYvW7duubkGx8E4E1wvrP3E5jflmfuWoHzs20WJnAvL6/8
    rKysjWq1es6GDRucL126hIEDBzaoXL5LY9TrsXv3bhw8eBBCoVBhZ2e3ydbWtiEr813G7Xzq
    AwF8z29yTvCyqC4F2gtcbY3Ttcjnqmme/Hn5XH3jK+Gva3MfCnQ7cOXEhfw60tBzXwbOZ9uR
    V/aJV9iLwLmuVJeT3gAm+fr6ShctWoSuXbtCo9GAiNC2bVu4u7tj8uTJ4tzc3EngXDmu1qCA
    u/BkkDP/t4a/ZjaaSXXZXuBy9VHHjh0NixYton379lFBQcFdbguFhYV04MABWrp0KcXFxRHL
    subE30nN4UHatGkzVigUVg4ZMoRKS0uJiEin09Evv/xCPXr0ICsrK0pMTLR815D4888/KTw8
    3GzmP96zZ8+2jVR842E2R38AMMbFxdGBAwfIZDLV+hxqtZo+/vhj8vHxMZtY1oILQHvkmDdv
    Hrp164Z3331X2KZNm36Ojo7HGIYhBwcHmj9/Pt28eZOUSiUpFIqHPpRKJeXm5tKiRYssWQAc
    HBxOhYeHD501a5YoMjKyMTdFXXjBYAAX1V3TprkbgLz4+Hhq1aoVgUtN5F5Du7EMw1TMmzeP
    xo4dS7ywj6lFKc6Ji4ujtm3bEq+M12QidOBZDfNkuIvWAAAgAElEQVQ1G8rs5g2uulo6z76Q
    RCIha2trkslk5sw3BnAZQo4BWMIzMma/OtWoUaOosrLyrrGs0+lo4sSJBC64Zhi/CHUHsJQ/
    VwkAo0gkIisrK7K2tiaJREIMwxA4/+FL4IqSPEp3CCGAp/l+qTFrRS2Hkd8EPItmkrkhPDy8
    v0QikVe9zzZt2tCaNWsoIyODVqxYYc7w9NCHQCCg7t27044dO8hgMFgyK61evZrs7OyIYRij
    m5vblwMHDnQYP358c3gtceAqzRkf4nnNJnDf5vAgnTp1wpgxYyTh4eHP2tranjf3h1AoJJFI
    REKhkAQCQb2P6ucDFwx/KSwsbOKIESOkiYmN5uHSiVdwdeBcbWpa4GP4NuY+OoBqLnG8XjCI
    VwDN7fJ4AqGmzVJVt8WzvGJeHVIAH/GkQ11uHg8ro6fy8v88rzCX8oTEdZ5A/RRclphWuO3e
    0hVASb9+/UipVNKFCxdozJgx9NRTT1FqaipVVFTQ8OHDiX8PiXxsgoAnO54FZ+U8zBNJpbzi
    LOcJld24HQv0yBjoIABLBAJBRHJysmrx4sXSkJCQWhu7uroiPj4e8fHxGDVqFJYsWUKbNm0K
    1+l0S3hm68KjmrybNm3Cu+++6ykUCqWxsbFwcHCAwWDA119/jbfeegv5+fnw9PTEpEmTGiW4
    oUePHnjrrbfw8ssvo7S0NCYzM3PugAEDXu7evbti/vz5j+KVMODSmK0SCoXthwwZgkWLFqFt
    27b3ZNVfeukleHt745133rFJT0+fxk/YueBS5zwyLFu2jBu0QUGG8+fP746Li7t57dq1t+Ry
    efLq1aslv//+O6ytresVSMIwDDQaDS5cuACDwaB3d3f/zc/Pb2FaWlp6u3btcPbsWZw9e7ax
    HrGAFxLe4EyHu3km+i6Go1WrVggICMC1a9fa85uk6knyQ8VisbR9+/ZmBsgcSHii2rn6isVi
    j/79+2P37t1A7X54T+N2SexcNEx+4Ta8MjvIzs5OGBMTg+7duyM4OBhOTk7QarUoKirCzZs3
    BSdPnnT6559/OhcUFHQ2GAzPg/PVLgbAlpaWorKy8i6XK61Waw4EFjAMM5iIJgDoJhKJ3D08
    PNC+fXtER0fD19cXLi4uEIvFUCgUuHbtGv7++29xWlpaqEKheJNfFOeBc7lpSrAAksFZgnwe
    4rcR4PwPzf6Wj4xJJyK0b9++Napk3ejbty+WLl2KmJgYKBQKFBQU1Dso2Gg04tChQ5g2bRpW
    rVqF4cOHQyKRYOrUqcjLy8OHH37IKhSK0ZmZmRnJyckf7tmzx9C3b99HJZ/7g0t11q4ueVSH
    PLPhmUazfM58lPI5LS0NXl5e2vPnz38XHR195datWwvlcnlfg8EgcHR0xLBhw+Dr61tv+ZyT
    k4Nt27ahpKQEAoHA5OLicsDPz++dHTt2HB0/fjzt3bu3sR6xkJfPvgBeABfMeaYOBhrg8jd3
    AucOCQDmMR6KO131bAEECoXC43xBMzPicaerB2phoBPABekxDMMUCYXCSr2+3kk3BDz7/TqA
    TiKRSOLm5gY3Nzc4ODhAr9dDLpc7KhQK/5KSkq5arfY5nq3fDS7JhA8AsZWVFUQiEdLS0vDj
    jz/CaDQiNjYW0dHR5gxoYhsbmzC1Wq0DZ4VNAhAqEAikjo6OsLOzs1QZVSqVKC0tdSwuLg40
    GAx9wMXjreIZ7CatSSIAVxbWmJSUpMnJydE+KOtaUFBgGjZsmJ7f9axE02QKqRGnT59mXFxc
    ForFYvrggw+IiGj//v3k5+dn2bGHhYXRjRs3Gi1XcGVlJc2ePZsEAgGJxeKKdu3aTcvNzX1U
    Id8DAFySyWSUkpJCOTk5d92vSqWi//3vf3T48OEaWenU1FRKSEgwB6Ftw6PP82vBkiVL0Lp1
    awwYMMC5VatWC8RicREAsre3J29v74c+HB0dCQCJxWKFv7//0oEDB7oHBwdj7ty5TfFYzuCi
    pc1j9mtesFa3GBVMnz6dfvjhB7KxsSF+p17VTioD8I2zszOlpqbSN998Q2Kx2AjOb7IqPAEc
    CgkJoYMHD1LXrl0JnLnRs4aN9rEq97Ua9U9DFMxbRqhjx460adMmKi4urnVuKRQKOnHiBK1Y
    sYI6d+5MEomEeEbW5OvrS8ePH7/rN2fPnjWz9ATAIJFIqEuXLrR69Wo6ffo0KZXKWq8nl8tp
    8+bNFBMTY/79XjR9HtWOPONT3/y4V3k26JGBiODl5TXfnK85JiaG0tPTLe9779695OTk1KA5
    gc3j2ozs7Gzq16+fmbG8FRsbG/8IX8lgXtmo9f5dXV3prbfeokmTJlnyXKP2vMQ78Ojz/AIA
    ZszgUtgnJib6+Pj4fCCVSpUikYiGDRtGly9frtcam5WVRSNGjCCxWEwSiaTcx8fnk/j4eH+A
    i3toTIhEIieGYQ5Weeef4e4gwS7g/I2r9s9/qlmBhLzl4I4+ZBjmnWquow68Ilq1XXoNm2k3
    cH7O5gDbdVKptL5+M0KexLklFAqpe/fu9J///IdOnTpFeXl5VFpaSiUlJZSTk0OnTp2i9evX
    08SJE6l169Zmq0AhOLc7bVRUFN26dYv27t1LDg4OZGVlRT/++CMVFxdTr169iGEYo62t7VWe
    QCIfHx8aNmwYrVq1iv7880+6evUq5efnU1FREWVnZ9OxY8do9erV1K1bN/O8yAcX3yVuynHu
    BSDN2dmZfv/9d8PDDuh9+/bpXF1dTTw788gUrJSUFFtvb+/tUqmUNm/eTCqVikaOHEkALKbg
    gQMH1rlIV8WFCxfol19+ue/2Zly6dIk6depEAMjFxeXcgAEDwkaNavKkJa0AHBWLxTRjxgwq
    KSmp8V6PHTtGfn5+NGjQIFKpVDW2SU9Ppx49ephNwqvB+WQ1G8TGxmLmzJlSW1vbLWKxmBYs
    WEAnTpygtLS0Bz5OnDhBy5YtI4lEQh4eHvvHjh1r37lz5ybLNc0L483W1tbk6elJvNlqVA1M
    Q2FKSgpdv37dPNYywAWrmOEBILVt27Z0/fp1OnToEDk4OBCAr6r13zAAyueff55u3bpFcXFx
    xDMqVU2EYgCLWZbVBwYGklAoNAF4tZ7PaQfOd9PYuXNnSktLe+BFdMWKFeTv728x2y9ZsoSM
    RqOljclkotWrV1uKHwUEBNDKlSsfeAN96tQp88bCCC4Xa1Pl5hLxC29DKZRf4N7p7xoNn3zy
    icTf3/+/vPJK33zzzR3vefv27ebN4H0d1tbW5OXldU+Xj8GDB1N+fr7lOrt37yYvLy8CQN7e
    3r+88MILTlOnTm3q1xEELvC3znvv2bMnlZaW0vHjx8nZ2fle78TEjxdpc5DL5noLo0ePtg4L
    C5smkUiyxWIxbdy4kUwm00MdREQ//PADicVikkqlBSEhIbPGjBljFxUVhU2bNjW+OYhlZQA2
    VXnnRTxjWhVdcXcBlnQAIVXirpx4lrZ6H/63Wv/15JXDqm3O4U6XHZZXHiuqKNALGuBxhwC4
    JZPJ6OWXX74ji5nJZKKysjIqKSkhhUJBWq3W4jKXkZFBS5Ysofbt21vu2dbWln766ScqLS2l
    xMREio2NpdzcXNq3b98dxZICAwNpzpw5dOTIEVKr1UREpNVqSalUklwuv6PgmHkdmDNnjjl7
    WD64kuFMXTuChkQggMDAwEBTp06dHtpZt0OHDoKQkBB9UVGRH6+4NXlwhp+fHw4fPhwul8s7
    +Pn5ITIyEiqVCjdv3kTHjh0xc+ZMVFZWomvXrnB2vrfrpslkwqZNm/DRRx9h/fr1eOaZZ+77
    XkJCQvDiiy8iIyMDpaWlbTIzMyePGzfujSY0LwjBBXjGmCskVi+IY0ZFRQUUCgVKSkpqVRLb
    t2+P9957DxMmTGBv3bo1imei/2ouCvSxY8eg0Wi0JpNJzbIsgoKCEB0d/dDny8rKAsuyYFlW
    o9frtcePH2/KvLEaANkCgQDDhg3Dli1bHEpKSqaBC2a7VdVEaDKZ4OHhgcTERJw4cSKIiHqC
    C3gzMxKevr6+cHZ2hl6vh7u7OxQKRTC4CP9CXpHub2VlZde/f3/IZDLzGGCqCaHOAMaHhIQI
    u3fvjo0bN2rA+evVBwMBjHR3d2cXLFiAmJiYB/pxQEAA5syZg4iICMydOxdnz57Fzp07MWnS
    JHh5cbp/fn4+fv31V+j1ekRFRWHFihXo3bt3ncHDtcg38Bk+2JycnJE8a76lCcaCD7iMLA2F
    XrzMb3I3u3HjxuH777/3Ki8v72SWkQkJCbhx4wYOHz4MmUyGL774os68z9UUGUyePBnjxo3D
    kiVL8Msvv9Tadu/evfjhhx8wY8YMsCyL+Ph4jB07FqtXr0ZJSUnS8ePHh7/xxhsbPvvsMzSx
    fL5neWqhUAiWZS3/3gMMgBHgAkcPPGq5HBHBZfq0tbVVDx8+fMMHH3zQU6vVjv7++++RkZHx
    UKQEy7I4f/48jEYjnJ2dzyYkJGy4fPmy+syZMxg3blyjP5PJZNLgziBsF155TeMVOIt8rq4W
    AOghFosvV1ZWAlwQXkBN6gPLsvb8dQTgXBlca+jnqucPA5dWTwYADMMYpFJpEX+d+mzw3hQI
    BD7jx4/HggUL4OTkBLlcjr/++gt///03Ll68iNLSUkilUnh7e6Nz587o06cPwsPD0aZNGwwZ
    MgQrVqzAli1bUF5ejh07dmDw4MEYNWoUVCoVXFxcsH37dhQXF0MgECApKQlvvPEGunTpAoFA
    gPPnz2Pfvn04fvw4cnNzodFo4OTkhLCwMPTs2RM9e/ZEQEAA3nnnHZhMJnz88cfuer3+DX6D
    cbGpTEgVQ4cOrVSr1caHZaA1Go1p7NixOn7xf6apJ+qpU6fQp08fsZub2wcsy5rmzJlDWq2W
    DAYDHTlypEbT7r2g1+spJSWFhEIhbdy48YF/n5eXR/Hx8WZTYVZ0dHSHwYMHN9Ur8QRw1N7e
    nrZu3VpTf9GVK1fo/Pnz9OWXX5KNjQ1FRkZSamoqZWRk0K1bt+5g8sw7yzlz5hDDMCZwQVzN
    IjqyCgvNWltbb5RKpfTf//63XibCH3/8kWQyGXl5ee185plnHkWp+tlCodC4Zs0amjRpEvEb
    r3m47TLRF0DJ1KlTyWg00oEDB8y7+K24Xbr5KQBlU6dOJYPBQHK5nHr37k385tbsBN8awIWO
    HTvSjRs3qLi4mLp06WJmS7z5NvYAvhOLxbRs2TKaNWuWmXWJr8fz2ZvNjePHj7cwDQ+LP/74
    gwICAsjKyoo2bdpk+fzrr78mmUxGAQEB9Mcff9TbNevFF180Bxf+iqYpkd2Lt0A0FAOtAufW
    1aRYunQpNmzYAH9//xShUKgFQG5ubrR48WIaO3YsWVlZka2trfnd3tchFArpo48+IiKiuXPn
    3rN9x44d6erVq3e49vDl58ne3v5QdHS01+nTp5vqlXiBi2uo8V7t7OwoIiKCOnbsSNOmTSOV
    SkUXL16kpKQkio6OpvDwcLP7Um3HcjyCKm91wdvbm5VIJBsbcCyTt7f3jmeeeabJ2XaBQPAK
    7gz4rASX9YbliZaaWGNzzn7bKlbEohraXMdtf3ifWsbJedz2iZaAswpXvR+lUCgcUh+iHZyr
    n6Fdu3Z0/vx5IiI6d+4cDR8+nOzs7GrsD5FIRKGhobR48WKLu2hxcTHNmDGDhEIhBQQE0MmT
    J6moqIhycnLo3LlzFBoaSizL0siRIy2WwezsbFq0aBGFhITU6rZkb29PI0eOtNxbVlYWxcbG
    mq2Ei9FErsR9ACgHDBigVKvV+noo0MZnn31WywvogU05mM27WG9v70FCobAgPDycTp8+XW9f
    ZnN0qL29Pe3du/ehzrFu3TqSSqXEsqzRw8Nj0VNPPdVUQi0eQGFcXNwdpkszfvvtNwoPDyc/
    Pz9yc3MjhmFILBaTj48P+fr6Unx8PF28eLEmVx2zf/BeNFHi8/+nCvRoAKo33niDDh8+bHZT
    yATnDwtwgXxys3JcUlJCSUlJ5iwbZj/XOSzLmtasWWPZAPHKuBJc9DcATGEYRjtnzhwyGAxU
    VFRkVqDP4baP3bMAlD179qTMzExKTk42C/mwejxfDIAcqVRK3333Xb3nqtFopDVr1pBYLKaB
    AweSXC4nhUJBgwYNIrFYTO+///5dG8KHwZYtW8zJ/29V6YvGxEBUMcs2wKHBIyiAJZVKERIS
    0tra2vo0qmXLeBClueohFovpm2++IZPJRC+++OI924tEIlq1apXFDUCv19P8+fOJZVkSCoUV
    3t7ek5rQTas25YkA0L/+9S/Kzc2lwsJCUigUZDKZLPO8qKiIrly5Qv3796/reQ/yzGizQXBw
    MCuVShtagX4k8pll2WfAZT+p7lbRtsr6W1DDPWfjdpaNyag5o44St11Ckmu4DvEWpMAqOlxe
    te9z6ymfPAEcZ1mW3nzzTcvYe/rpp++rX0QiESUmJtLRo0ctZOLQoUOJYRiaP38+GQwGMhqN
    tHDhQmJZlrp3727xiz9y5Aj17t3b4nZ3r2PEiBEkl8vJZDLRqlWrzL87g1qyJjU061cIoDQ7
    O1sql8sf+iRKpRJXr15l+c4vaMrBHBUVhYiICH+FQvG6SCRyS0lJsZiO6oPMzEycOnUKfn5+
    CA5+uLiMvn37mpOGs2VlZcPOnz/f6scff2yK1+IPwLFt27Y1um5YW1vDzc0NTk5OsLa2BsMw
    EIlEcHFxgaurK9zc3GrM2xkcHAxfX1+A879qVgr0E4ZsABVXrlxBZGQkxo8fD4FAEAxgGm+m
    s5jwiAhOTk5ISkqCUCh0501+EgCtpVIpYy5iIBQKERoaCoZhbHnznA2AgY6OjuKkpCQIBIKq
    JlWGF1C+AKbb29vbpaSkwM7ODtnZ2QCX+aI+eWd9ADjZ2trC29u7IRY0jBkzBt26dcOhQ4dw
    +vRppKen48iRI4iLi8OYMWMapMiOt7e3OWLcBU2T1k6Fhsl0Yoa5yEKT4bXXXkNsbKy1XC6f
    XVFRcUd1OKPR+NCxBR4eHmjbti1KSkpw4cK9PVL0ej1+/vln5OfnW+bD8OHD4eXlBYPBICsr
    KxsdExPj8uuvvzbFa/FDlUwk1SGXy5GXl4fCwkIoFAoQEfR6PQoLC5Gfn4/c3FyoVKp7Mdwt
    8rmRYDKZsnF3sZM24NwoxLz8ZGtRTPvy7GhILSypWT5Xz/1cFWb57wCu8I5HNYa83NraurQe
    jxgKoJVMJkOnTp0gEAhw6tQpHDhwf15Ber0ee/fuxeuvv46LFy/Cw8MDr776Kry9vfHzzz/j
    6tWryM3NxS+//AJHR0e89tprCA4ORkZGBl577TXs378f95s9ZN++fTh9+jQYhkFMTIxZPgfW
    RvA0tAJ9E8C5K1euiPfs2fOwuYNo//79ugsXLgj5XViT+T8///zzcHFxkeTn50+vqKjo1q9f
    vwZZLHU6HTZt2oQbN26gT58+Fp/KB4Wvry/69esHhmGg1Wpbq9XqxPDw8KZwprUHIHB2dq7R
    37Nnz5746aefsGPHDixfvhzW1tYICwvDDz/8gN9//x3r1q1DQEBAjYq3jY0NwAW6ObSI0kZD
    IYDy7OxsVFZWYuLEiYiKimIADAdXwre6Dxzi4+Ph7e3N8ux0EIAQFxcX+PhwRDLDMAgKCoJY
    LGZ44d0RQGxERAQ6dOhQk4AWA5gIoPPAgQMxcOBAlJaWIi8vD+DMk6p6PJ8AAGtjY2MWePWG
    u7s7kpOTodFo8Mcff2DXrl1Qq9VITk6Gh0fDpEO1t7c3j38WTRNIWwguT3VDQY76+67f/8JA
    hMmTJzPXr18fXlZWNpqIGkT2MQyDQYMGoW3btjhx4gTOn7+/WhHnzp3DkSNHLH+3bdsWPXr0
    AABoNJoucrm8+5AhQ5ri1dijDheLb7/9FoMHD0b//v0xf/58VFRU4MqVKxg9ejT69++P5ORk
    HD16tK7zW6HuSoYtqAdEIlExwzCqGnSzUeDcN4CaA9lYcGkL/cC5z6EW5TiEVwB71tFGwK8H
    STUo+Hkmk6k+hc8cAFjb2trC1ZVzv7527dp9xyeYcejQISxcuBByuRyxsbEYOnSoOT0ojh49
    iosXL2Lw4MFITExEcXEx3n333Tvm530xDCoVsrI4ldPV1RWOjo7gSSbXplCglQC+r6ysVH/4
    4Yei48ePP/AJzpw5o1u2bJmprKxMDeBb1F4qt8Hx5ZdfIi8vb4BCoZjk6+srmDNnzn0FCN4L
    27Ztw9dffw1/f3+MGjUKItHD1SAQCATo378/HB0dYTQaxXq9fsDKlSubIoJfB4DkcjmMRmON
    9+Xi4gJPT0+4urpaGGhPT094eHjA3t6+xqA5tVptnkSGBmbGWnAnygAU5ebmoqSkBMHBwZgy
    ZQqsrKycwAWs3EXbhoeHo0uXLmYmZBgALzc3N3h7e8NgMMBkMsHX19csYNoAeJplWbfExES4
    ubndpfuAyys9ydfXV5iSkgJbW1vk5ORAqVSCV8LU9Xi+EgDlGo0GFRUVDfbSkpKSEBoaiq1b
    t+Lnn39GaGgo+vXr12DnV6lU5vtVg2PhGxs3waUObCikgitA0CRwc3PDhAkT2sjl8jl6vb7B
    FLqIiAjMmDEDer0e3377Le7XeqpSqfDbb79Bq9UCAGQyGYYMGQKZTAaDwWCv0+kGTps2rSnS
    YGlQR9VenU6HvLw8ZGdno7CwEEQEg8GA3Nxc5OTkoKioqEa5Xk3+V6IFjQKGYcpQs6XdnZfP
    7nX8vH0VJbo2hIFz4/CvbW/KK+DTcHcKPRBRXmVlZX0UaB0Ag0gksliidTrdA1uLiAjbtm3D
    Tz/9BLFYjKeeegr29vbYvn07tm7dCplMhjFjxsDKygo//vgjtm/f/lDX0Om43AwSicSsq5lQ
    S0nwxgjc+h3A5rNnz9LMmTOxe/fu+6LPDQYDDhw4gOnTp4vPnj0rAVfW97emGsTTpk3DyJEj
    A4uLi/8FwHXatGmIi4ur1zkrKyvx3Xff4c0330RhYSFiYmLuKGX+MGjXrh1at25tVkA7nj59
    OrQJfO0UAPRZWVn3VFBcXFwQFBSE0NBQCIV1+90XFBSgsLAQPPuoQAsaC+UA8hQKBXJzuUqx
    w4cPR0JCAgD0ADAGXGEQqFQqXLp0CQqFAklJSZDJZLYAngPg4+vri+PHj2PGjBmYN28elEql
    mY2NBDDS09OTTUhIQHFxsbnYiHnj5AAgRSAQtBo7dixiY2MBcNlJKisrzT529SnIcRNAUVlZ
    GYqKihrspQUGBiIxMRGXL19GZmYmEhMT0apVqwY7f2FhoXkDUYSaS6w3NCoAfIeGYaFLwOUU
    VzXFAN6+fTumTJliW1hYOEetVkc25LkDAwNhb2+PzZs347ffHmzJOXbsGG7evF0puXPnzggI
    CAARoby8PO7s2bO+TSSf7ysjU1lZGeRyOYqKiiyKwv3sFdBMShw/idDpdOVElFvL14ngipnU
    pqvZgfN/rkuBDgdX0a825k7Gn6NG5UQsFheCq6L6sMgFIDePPQAICgq6q0jV/epU69atw40b
    NxAbG4uuXbti37592LlzJ6KiotCpUydcv34dGzZsgEbz4JycTCazyPiSkhKzfFbgzkwpjapA
    KwC8B2Dz8ePHK1944QWsWbMGdaVA0Wq1+OijjzBx4kQcOXJEA+AH/hxNplR16tRJfPny5aly
    uTwuKSkJzz333D0VwNqg1+tx7tw5zJs3DzNmzMCVK1yxvdLS0gc2W1SHs7Mzunbl4ro0Go27
    QqHo0gQp0S4ByEtPT0d6enqdDcPCwrBx40YsWrQIMlnd8Rj79u0zK9CZ4LIDtKDxFKe8iooK
    i3nKxcUF06dPh6urqxW4IBXrkpISvP3220hKSkJycjJyc3Ph5uZmNgFa5+Tk4NVXX8Xnn3+O
    lStXYvXq1eArXDkD8AoODsaRI0cwYsQIjB07Fjt37jRXyPIE0C0iIoJ5/vnnLRaYGzduQKfT
    GWoTTg+iiwK4UVlZiWPHjqFa1a2HhlAoRIcOHSAUCiESiRAdHf3QMqEmpiM9Pd0sF6+j6WI9
    DoArz1sfRrGSP8f+phrAgwYNYtLS0kYUFxc/YzKZGlTg7dmzByNGjMDbb79tXjDvf+d28yaq
    Ztvw8fGxpLysrKwMKigoiG6CgkmZuE9Xmn/++QcvvfQS3nrrLZSVlT3I+Vvkc+OhkmGY/Fq+
    swEXpF1blh4GXPrCukzlnuD8kGuDF7hsOjUJNxPDMNn1fL5cAAUVFRWWzWbHjh3RvXv3hzrZ
    hQsXcPLkSdjZ2aFnz55Qq9VQKBTo2rUrnJyckJqael9xDDWhW7duFhfEnJwcMxFUyJM0TQpn
    Xgku8/Pzo3PnztVZKCQkJIRYltXY2NisRBMHLNjZ2SEmJqa7ra1tjo+PDx04cOChU1P9/fff
    9Morr1BoaCgJBAJycnKSR0VFZYhEIl1ERARdu3at3tH7P//8M1lbWxPDMOTi4vLFokWLRI38
    iqwBbGQYhqZOnUoVFRX1fobLly9TdHS0OWXPi81Noj1hWTgA4A0AppdeeolKS0stCeWnT59O
    LMsSAHJ0dCSRSESurq7k5ORE1tbWdxWVsLW1pRdffJH69u1LLMtafmv+vZWVFdnY2JBYLCYv
    Ly9LsQZra2tav379HVUA+aJEKnAmyPqAAVeIRR8WFkZnz55tsEqgBw4cICcnJ3J1daVDhw41
    2HkzMzPNFQn1AP6FOpL1NwIcwKWOrCk11r2OfP63TRazsHjxYowYMSLIx8fnJBow6wIAkkgk
    5OXlpWZZ1vSw55g5cybp9beTTn3++eckEomIZVlyd3df/fHHHzd231qBK2hEjXBU8uxks8KT
    lIWDx1zcmTquuRyV4HKB1wdicEWuaNiwYSSXy4mI6K+//qLWrVs/8D0xDEPvvPOOZW2VSCQk
    lUrpxx9/JCKiV1999aGeNTQ01FJltKKiomqa0W9RSzGhxsq96wjOGT0GgJhl2TqLDZi/N5lM
    Ao1GE8XvhppMiU5OThbdunVrpFqt9hoxYneiS7QAACAASURBVAS6deuG0tJSfPPNN0hLS7vn
    700mE06ePIk5c+ZgzJgx+PDDD5GVlVXs6Oj4U+fOnSd6eXl9DMDIsmyDFNAICwuDq6ur2b8n
    4ujRo439rtQA/ktEOT/88AM2btz4IOa/u1BQUIClS5fizJkzALAHTeiq8/8YtwDoN2/ejMmT
    J+Orr77CsWPHEBAQAGtrawCchcTf3x+ffvopvv76ayQlJcHPzw9ubm7w8fFB79698Z///Afv
    v/8+PvnkE0ydOhV+fn6QSqWWo3fv3ti4cSPmzp2L0tJSlJRw3gIhISEICQlBRkYGfvvtN8ye
    PRt//PEHwLmXFNaX0AVXjOdcZmYmVq5caQ5OrDecnJxgY2MDW1vbWosHPSjy8vKwePFiM3N5
    BsAvqMOHtRGgALAUwEo8mGm+DMD7/G+bzDooFouZEydOPJ2fn99grhssy8LGxianffv2H8fF
    xX0ilUof2kSdnp5+B3MdGRkJZ2dnmEwmGI3G9qmpqY292agAV3EuuxHO/UeLfG4S3OQ3080K
    LMtqJRJJfa1jOl7GlR48eBA7d+4EEaF79+5YsWLFA7vFERGuXbsGg8EAFxcXWFlZwcrKCm5u
    btBoNLh27cHDMlq3bo3Vq1dbgoCPHDmC33//HURUBuB/aKIYLQZcsNCPAMplMhl16tTJtHbt
    WtJoNLWyMTqdjj7//HOKi4szl1BUAfgZQByaoMBGhw4dPKRS6UlnZ2fas2cPVVRU0Ouvv05W
    VlY0e/bsO9iFmvI7r1u3jkJDQ4lhGLK2ts7x9fVdFxoa2qdXr162AODu7j6HZVlTfHw8FRUV
    1Zu9KiwsNJfCJolEkhMWFhbVBH0r4pmyUhcXF3rnnXcoKyvLUhJVq9VSRUUFqVQqUiqVpFQq
    SaVSUUVFBel0Osu72rt3Lz399NPm/IonAUQ3R2n2BDLQCbwCZMmtaW9vb55vJJPJKCkpifbt
    22fJbVtWVkaXLl2iv//+m06cOHHX2K2oqKAzZ87Q1q1b6X//+x8dO3bMUuJdqVTSsmXLyNfX
    15Jn18fHh3x8fCzX5I9/cDsHaX1lz3gAJUKhkCZNmkS5ubkP1E8mk4n0ej3pdDrSaDSk0Wjo
    zJkz1KpVKwoODqb09HTL5zqdrk65UBtyc3Np0qRJ5oT+heBydDNNPBaCALzDK+/6B2BpDOCK
    4iwE59bTVHPR3s7Obl/Ve3F3d6epU6dS//79SSAQPEjBFKO9vf0lb2/vVVFRUXFffvmluG3b
    tlMkEslDs39BQUGUnp5u6ePs7GyKiooyF7261qFDh5Amks+zwAXd33F/Dg4O1KZNG4qJiaFe
    vXrRgAEDKCkpibp3704dO3akkJAQkkqlNT3bCdxHdcMWBrpB0JPflDYrBlooFN5wdnZu1wDP
    Z8tbSUxt27a1WPlNJhP9/PPP5toE930MHDiQysrK6MyZM+Tn50eenp6UlpZGpaWl1KdPnwdm
    nnfu3GlZ99LS0qhr167m7zehjgw0wgaewKMAvC0QCEJjY2ONU6ZM0ffr10/o5uZWZyo4kUiE
    F198EcOGDcPevXuxbt066yNHjgw3GAwR4KrAbEYjlq12cHDwBeDh4+ODNm3aIDU1FV988QWC
    g4MxatSoWv0e5XI5PvjgA6xduxYqlarc2dn5N39//0+Tk5PTFi1apHV0dMSYMWOsd+/e3c1k
    MjGhoaFwcKg/GWFvb4/WrVvj77//BhE5CwSCtvxi2JjQA/gEQHFxcfHspUuXRuzYsYPp06cP
    GIZBdnY2iouLUVlZCZ1OB4ZhIJFIYGNjAw8PD3h6euLmzZvYuXMnCgoKNAB2Afg3r0Q3KxAR
    Bg4caJ5ADVl2myQSCRFRU5byNqMIQKm9vb3tsGHDoNFoIJfLIZVK4evri169eiE+Ph7u7rcD
    vm1tbWFra4vQ0Jrd52QyGSIjIxEZeTcxaGdnh3/961/o0aMHtm7dirS0NBQUFICIEBISArVa
    jRMnTsBkMhWhYfwrzZW5vAwGwxubNm2y02q1WLBggSXoFuCClVUqFZRKJfLy8pCXlweFQgG5
    XI7i4mIolUpUVlZCq9Va3pGZzZ45cyacnZ0hkUgglUphZWUFJycnuLq6wtnZGc7OzvDw8ICr
    qytsbGxgbW19h+Xt0qVLWLhwIbZs2QKDwaAAV+FtaxOyzxJw0fivAuj0EOSEAFzUf1twVsLV
    PPPfqOyMWCz21ev1FprK3d0d//73vzF27Fhs3boVhw8fRnl53UkCGIaBlZXVNVdX140BAQE/
    jh49+vLSpUuNWVlZgvz8/AidTvfQRE1RURFycnLQvn1783qCkJAQnDlzBjqdzoWIwgBcbgL5
    /Dm4bC7/196XxzdVpvt/z8meNGmatkn3nUJBQFYVUVBEUUdBvQ6KiLI4etVR0Tt63VBxuziK
    zoygI4q4jAs6iCsCLtdhVZBSukD3LV3SNvt2kpzl98dZTCsg0CRl7q/P53M+SUvJyXvOe573
    +36f7/M894FPCCMAYMGCBbj33nslpk6lUoFlWVAUhWAwiL6+Ptx+++3RJb8oAF8B+B8A5TgN
    LYFNahJldgFAJx8rYiLeP61Wi+TkZKSkpMBoNEKn00GpVEKscqFUKsGyLMLhMCKRCCKRCEKh
    EHw+HxwOB5xOJ7xeLwKBAPx+/3GT7Wiatnk8nlhUCPKCj3iNqK6uPveOO+7Ak08+iSuvvBJX
    XXUVQqEQ/vSnP6GjowNarRYGgwEmkwlmsxlJSUn9xq7VajFmzBhpvHK5HCzLQiaTQaVSYcGC
    BZgwYYI0vkAggEAgAJ/PB5vNBqfTCY/Hg2AwiLy8PKxatQpz5swBwzDYunUrHn30URw4cAAA
    9gg+2h1vAK0Cr2N9NDk52bxkyZLwPffcI8/LyzthbS5BELBYLLjhhhtw/vnnY82aNVi3bl2J
    w+H4szCpXouXo05JSVHIZDKZRqNBUlISKisr4ff78Yc//EGqGDDQnE4nHn/8caxbtw4sy9bn
    5OSsGj9+/MYjR454rVYrAoEARowYAblcPjkUCp2r1WoxY8aMmCQhKZVK5OfngyAIMAyj6u7u
    Tlu/fj2WLFkS74c8COAtAD/TNL18//791+3fv19LkiRIkoRcLodGo4FarQbLsggGg6AoCgzD
    gGVZ0el1AviLsBvtPR0dc3p6OiZOnJhOEEQmTdPYsmUL+vr6TslpkySJgwcPik4staWlJS0z
    M9M6BENrA/Cj3+/Py8nJwQMPPACWZaFQKKDVavsBeo7jpHvGMAzC4TAYhpF+JyQGStIr8d4r
    lUrpZ/F306ZNw7Rp0+B0OuHz+cBxHCiKwn333QeWZWkA32JwTVSijQKf3BamafqBDz74wGK1
    WvHwww8jKysLBw8exL59+1BbW4vW1lb09fXB7/cjEolIYyIIQjpkMhmUSqUkcTlw4IB0LTiO
    6zcfxPEnJSUhPT0d+fn5KC4uxoQJEzB+/Hh0d3fjmWeewc6dO8FxnA28DOJ1DC67/WTMAL5J
    wnIMvqscKQDwteBLZL2EOFZpMBqNRoVCoQsGgyAIAtdeey0WLlyII0eOYM2aNb/VBARyuTyo
    1+s/zszM/NtDDz3080svvcTu2rULt9xyC/bs2VPMcdwFgwFkwWAQ9fX1mDOHl/JrtVqpWRbL
    skm9vb05GzZswM033xzve0yB12seEO7zDQaDQXvppZeiuLgYHMdF+2Go1WqYTCZkZ2dj1qxZ
    2Lt3L1iW7QDwouDn+3Aa2u7du/Hkk09yHR0dIEkSOTk5pyyvIggCDocD7e3tAJ+rwX777beY
    NWtWQsdEkmQHgEMsy+aL5ERGRgZKS0sxatQo5OfnIy8vD7m5uTCZTNI6q1ar+/ndo5nou8Ph
    MILBIILBIAKBAHp6etDa2orW1lY0NzejtrYWTU1N6OvrkyqnkST5E03TsaodXyNs3lfX1NSc
    c+edd6K1tRVLly7F/PnzQZIkKioqMHbsWJSWlkpEhEqlkjYIR3m2IUpjRfwRjYPEpkHiJsLr
    9aKnpwf19fWoqqrCxIkTMXfuXASDQbzzzjt46qmnxLmwD8CfwPciQTwBtAx8GazH09PTUx97
    7DF6yZIlco1Gc8o7+tzcXDz++OMoLi7GihUr0ru7ux+DoMPF4MpdHcvB9gBwe71ei8PhwPnn
    n49Vq1bhqquukv6ms7MTMpkMFosFoVAIf/nLX/D6669DpVJVGI3G+/7rv/7ru3fffZerra1F
    bW0t2trasHDhQv3bb799k9/vt5x11lmYMWNGzL6zyWSCXC5HJBJBIBAwZWRkEAlksqqExf+S
    iRMnam+66SakpqZKWtGkpCQwDAOv1wuv1wufz4f6+nq8+uqrsNvtRwC8gdg2dIiZjR8/HpMn
    T04vLy9/LBAInM+yLDZu3IiPPvpoUKCcZVk4nc4z6+rqHh8xYsTDJpPJdqING2JkHgCv0jR9
    3rvvvptx2WWX4dxzzwVFUejo6EBHRwd6enqk2rAOhwMej0fawUciEdA0DYZhQNO05LBkMplU
    pUJsjGM0GpGamoqcnBxkZmbCYrEgOzsbmZmZkMvl2LBhg9iFaj/4EBkbw3H6AawBYOM47pEd
    O3aULV68GCqVCl1dXaBpGgaDAZmZmRg3bpz03YxGYz+GQ6lUSkyzSqUCQRCgKAoURSEUCiEc
    DkvMRiAQgFgisKenBzabDXv27MHWrVtBkiTMZjPC4TBsNpv47DwD4GMkTvOYDD6J9A7wCcEx
    c0MA7hc+/0nEqVKDQqFwsyzrB5AudsBkWRavvfYaduzYwS9CMhkIgvhVBRa1Wu0xmUwv5efn
    v2S1Wp2bN2/G/v37sW/fPhw8eJB8//33r/J6vYOSWEQiEXR1dUGMLIlkkEwmA03TRG9vr2HM
    mDGJfNYPCwTFZUajUdvY2IgXXngBNpsNfr8foVAIBEFAJIyys7PBMAw0Gg38fn+tsM6elv5Z
    vMbTp08vJAiiWCaT4d5778X8+fOlTfDJEhyffvop7rrrLvj9/qzq6ur8WbNm1SU6Srh27VrX
    Pffcsy4UCp2rVCpNd911F66//nrk5+dDr9cfN4fsNwGaTCaRAULjJgB8LpVooVAIbrcb9fX1
    eOutt7B+/XoAaFSpVG8GAoFY+qm94OtNP93V1XXpihUryJaWFtx///2YP38+5s+fL/2h6GPF
    yKDH44HP50M4HAZN0wiHw2hra4PX6wVN0/j666/R3NwMtVoNjUYDrVYrvYrvc3JykJ+fjylT
    pkjnEZUEL7/8MpxOJwvgG8Ff/mZ0PBYAeobAPJueeOKJ8LJly5Sn2ihkgOPD4sWLQZIke//9
    95scDscj4BOhtsV68rIs263T6Zrb29tL9+/fj//4j//oV6/ZZrPh7rvvxtSpU/GnP/0J27Zt
    wyuvvAKCIOry8vLurqys/OHNN9/Ezz/z1/vyyy/HhRdeSGzYsOH6rq6ua5KSknDLLbeccgfC
    o66IyckSgNbr9YXbtm1TJpDNIgCcp1AoLDfffDP++Mc//uZ/8Pl8qK6uxubNm8eADwP/7+kI
    nlNTU9PLy8uf6O3tXVJcXKy65pproNPpBhUyFMHXp59+qjx8+PBNAIjCwsKHRo0aZTty5Egi
    h7gLwAdtbW13Pf300+T06dNx6NAh1NbWwmazweVyIRwOg2VZaVcvAuRosCw6cxFMR7+K7Kz4
    GRqNBikpKcjKysLYsWNRUlKCDz/8EH6/3wfgVfAl3GJtYfCyr0YAD3R1dV0qk8nU8+bNw8UX
    X4yysjJkZmbCZDJBq9VCoVBgMAm+4nhpmkYgEIDT6UR3dzfq6urw7bff4sMPP0QkEqEAfAbg
    OSRWtqQCcLvAPmvj8PlaALcJDPQqxCFK2NnZaRVKaRVEIhG89957qK6uxqeffgqRrbvjjjvg
    crnw5ptvSk1B1Gq1Pzc395lx48b9taamJtje3i6ySyAIArNmzTrHZrMti0Qig16wPB4PokGX
    wWCAQqEAwzDQ6XR5n3zyiRzHaMYQD0ITvKbWYrVa8cADD0jP5zEIJMhkMjExfAyAceDLHZ6W
    4HnGjBlFVVVVz4XD4bOmTp2K2bNnD6o76AUXXICzzjoLe/bsGdvU1PTnKVOm3EsQRGMiQfTK
    lSuhVqu/pWl6UzgcXup2u4mysjKp8QjDMAgEAhJ7HAgEpI18KBSSDoqi+kkoxWMgITAQWKpU
    KpjNZuh0OqxZswYMw9AajWZDSkrKwVg2pxKsQtjM3+fz+RatXbs2ua2tDcuXLwdBEGhubkZL
    SwsaGxvR2toKh8MhNV0LBAL9Itosy0ps+SOPPCKROeI4o19TUlJQWFiIoqIi5Ofno7CwEEql
    Ei+//DLee+89BINBtxDBeQEn2CBqsADaAuBPcrm86LbbbgsvXrxYEQvwHP1gL1y4kGhpaQmv
    WrWqIBKJ3C9Q6jFtH7tgwQJ/RUXFZw6H44I1a9YoR48ejdGjR0sP7caNG7Ft2zZcdtll6O3t
    xV//+le43W7XiBEjnlm3bt2/nn76aakA/6233ooJEyYQzz///OXd3d0PURRlWLZsGX7/+9/H
    9GEUHXQwGATHcZk1NTWKBAJoC4DLi4qK5BdffHE/ECGGtqND4ULGO+bNm4evv/7aTFHU5QB2
    JnBB+U0rKSlBWlqauaqqamVvb+/iUaNGKVevXo3Zs2cPupW7eH0uvvhi3H333fLKyspFHMcR
    xcXFDxYUFNhaWloSNcyQwP7P3rJly5itW7dCrVYjPT0dRUVFyM3NlVhjo9Eo7eSjq2yIP3Mc
    JzlsUUsZ/d7pdKKjowNWqxVtbW3o6OhARUWFJIEAsFUAlHFbb8F33bsTwBtyuXzOvHnzsHDh
    wtjvJoWNhqjBS0lJQVFREaZNmwaDwYBNmzYhEonsIAhi+XEaJsTL5gG4N07gWTQNgLuEDcs/
    EONImMVi8TQ3N+/w+/3TWZbF3r17sXfvXunaX3LJJfjv//5vvPnmm9JGVy6X02az+Y2JEye+
    bLVag9F1YUtKSpCcnFxw6NChFQ6HoyQmYQ+/HwzDSL5CJDgAQKlU5tXV1akS6O/M4GsHy0Ut
    rChFUigUUCgUUmhbBGBRneHMAH4n+OfTqiqEUPe3qLq6+nmn0znXbDaT119/Pfx+/wlVyzqe
    XXfddWhsbCS7u7uvaG5uJqZOnbq8sLCwMVFjExpcBfR6/bpgMDhr06ZNhdOmTUNGRgYOHjyI
    uro6dHd3o6enBw6HQ/Kj0cRF9M/RPima+BAPtVqNtLQ0WCwWZGZmoqysDOPHj0dNTQ2++uor
    KJXK/Xq9/l2Kopg4DbkVwIMAymmaXrF58+aC3bt3g+M4eDweqbvnSbEm4bBUHex49dzF6IvB
    YIBcLkdXVxcYhrECeBzABziJrriDAdAk+AzyC6ZPnx66++675Wq1OubbNZVKRdx5553yH3/8
    Mbx9+/bzACwAr9GKWdjXarVy2dnZn7jd7it27NgxZ9myZbj99ttRVFSE8vJyrF69GmVlZbjw
    wgvxySefYPfu3VxKSsqH06ZN++iVV17hRPD87LPPYurUqcTtt99+SWdn5wsUReXPmzcPDz74
    IPT62Hbc1mq1EhMYDoeTOjo6ZAn0ZecAmJCXl4eamhp8//33aG1thcvlgt/vRzAYBEmS0Ol0
    0Ol0EkBjGAZpaWmE1Wq9WGAfG08Hx8xxHGbMmGGpqqpa2dfXd3Npaany+eefx8yZM2PWlAMA
    zj77bDz//PNYvny5vKam5kaCIJCXl/dQS0tLdwKHWwM+8atsxowZ5D333IMRI0YgMzNT0tPF
    YsMAQEpkCQQC6OjoQHV1NVauXInDhw97BaCViOYMXQCqQqHQnARuVCQTui0CwGGO47oTfPoi
    ATynJeBcJvC62x8R44Q5hmGYvLy8j4PB4LUej6cf4E1NTcXtt98OmUyG7du3S2F8o9G4a+zY
    sS/8/PPPfrGR1d13340tW7ZAq9XmtrS0/NnpdF4Uq2Q0UeYkEkh6vV4C0OFwOKW5uTmR/nka
    gDPHjRuHCy64QNocJyUlSc+4mIsQCoXgcrnQ0tKChoYGbN26lWhvbxf9cz1OA+M4DjqdDhMn
    Tiyurq5+vq+v70oAZE9PDx555JGYncfn84FlWcJut/8OAIqKipZ7PJ7GE23tPlgbO3YsMjMz
    Dx44cOCrvr6+2++66y6CYRgpd+RXACwqB0UExiL7KuRHgWEYhEIhBIPBfnktAFBXVyd9lkwm
    g8FgEAFsODU19R9XX311y6uvvhrPIfsBvANgOoAlQlO145IUYn7NwHFHj1fM04keb7S8h+M4
    icWPst3gu7SeVIOpwQDobAALk5OTlX/84x/DmZmZcXMQFouFvPPOO5mffvpJ5Xa7F4IvcRez
    1fCuu+4CgK6zzz77EYZhDHv37j2nvLyc0Ov1cLlcUCgUWLlyJWiaxmuvvQaWZZtMJtO67du3
    B8TObq+//jqWLl1KjBs37qKOjo7VoVCo9Pe//z2effZZFBQUxIX1Eo2maWV2draiuro6Ec+5
    GsBcAPqdO3di586doCgqmvmBXC6XGI7ohDOtVitm/I4EX1bttADQANDW1nZtX1/fzQzDKLVa
    LTZv3ozNmzfH5wLySZZyh8NxYyQSKQfwtwQOlQWfWU+lp6drL7300n7JGSzLwufzgaKoX+me
    xUQMMWtbDI0pFIp+Ug8xmUOr1UrMtclkkqongE8ebUjU+isAgXBTU5OSpmnI5XIEg0F4vV6p
    i5XL5UIgEJBY9Gi9cygUAsuy/UKiA1n55ORkmEwm6HQ6KQ+AZVk0NTWB4zgafEiQTeB9loNv
    wZ7IMpHjASwF8Ggs2csvvvgCjz32WEUoFFrV2Nj4P36/PzUauH722Wf46quvsGvXLnFeuo1G
    45ovv/yyTYyGvfPOO3j88cdhNptzGhoannO5XFezLBuzEqk0TfdbpMVFXfTPssGIWE8+GjDX
    YDDoV6xYgWuuueakxvDUU0/hySefHMGy7IWnC4AGgEmTJhXX1NS8YLfbrywqKiJKS0tBEERM
    q3GIn1dfX0+0tLT8rrm5GUVFRfc4HI6mRIyxsrISlZWVYa1W+wPLsotdLpdWZN7T0tKQmZkp
    RQjT0tKg0+kkGUb0e6VSCYIgfpWnIUpAfD4f7HY7urq6+uW/OJ1Oca12BoPBn6uqqhIxbFpY
    C1iBlJUSsTMzM6XcmbS0tF9V4xDHLOaoiEmS0WMVK3HY7XZ0d3f3G6/b7Y5+ZltwCvKzwQDo
    mQBGT5s2LTx79mx5vK/yrFmzFOeff37w888/HwXgQgDrY/n5S5Yswfr163+eNGnSMrVavdTl
    cl1A07S8oKDA4HA48mtra4nvvvsO1dXVtNls3jBt2rSD1113HS666CKcddZZWLp0KSZNmnRB
    S0vL6lAoVHbDDTfg6aefRk5OTiJ26Mrc3Fx1gnxZGYALVCoV8vPzUVJSgtGjRyMjIwMajQY6
    nQ4ajQYsy0pstM/nQ0tLCw4fPozGxkZ0dHSoWJadC77smPs08M8EwzAZLMsqSZJEZWXlb7Ys
    H6yjJkkSNE3LvV6vaQjGawXgaWho0FZUVCAUCqGhoQHNzc3o6elBT08P7HZ7P0ApJm6IR/Rm
    KVp3JpZ3S0tLg9lsRkZGBoqLi1FSUoLu7m4xka4TiWtdDQG8Bg4dOqR899130dbWhtraWrS3
    t6Onp0cqaRQdCo1magayYdHMh/herVZDr9cjLS0NOTk5KCkpQV5enpgXQQ3BZrEYwDXgk7wT
    ZTIAVwF4G3ykI2ZGURQzadKkdyiKithstjv9fv8YuVwuUyqV8ldeeYUU5WMEQcBgMHyZk5Oz
    9bnnnpNAUVZWFsaPH59TUVGxyuFwXMswDJmoi0LTtCI7O1uZQP88c8KECdzZZ59NOJ1OKb8h
    FApJWlKZTCYBEo1Gg9TUVJjNZsyZM4dbv369qr29fS6AD5HAhjnHc5ldXV3z7Xb7FWPHjiVW
    r16NiRMnxkWfzHEcKioqcN999xEVFRW/Y1n2B/Ca2ITY22+/jSVLlrQBcBsMBu0tt9yC8847
    DyNGjEBqair0ej202tiosaLBdF1dHbZt24b169cjGAx2UhTVuXPnzkQNux4ApVartYsWLcLF
    F1+MkSNHSlU4BlaJOlUTq3C4XC40NjbiX//6F9asWQO32x0WQPxJ78ZOFfiqAFysUCi0c+fO
    Den1+rg7aZ1OJ583b15gy5YtGpqmLwafJBSM1eevX78eK1euxOuvv3543rx5Dx48eDBj1KhR
    ZGlpqXndunX3vPTSS3MikQhhMBi+PuOMM9bbbDbmoosuAgD8+OOPmDZt2jmNjY0vhEKhMxYt
    WoSnnnpqUIkNJwnIlDRNJwJAEwAuA5BHEATS09MxZcoUTJ48GQUFBTAYDEhKSpI0dhRFSQ9o
    bW0tZDIZurulKPY08EX6//c0ITk4ALjiiitw/fXXxzV5hCAIbNq0CRs3bgQS24FOtG4AXYcP
    H85YsGAB7Ha7lMksssfJyckSm2EwGH6VkCJqoMVD1FK6XC50dHRg//79CIVCoGkaSqUSycnJ
    0Gg0YmdCa4I3Tl0AesvLy4233XZbv3Gmp6ejpKREctYDs7c1Gg00Go204ESXgRLf+3w+9PX1
    oaurC3V1dVLZQplMJoJwp7BpSBjgADBbANGJtkIAl4CvBBGzub1q1SrMnj07tH///rfnzJmz
    o6am5gyLxSI/++yzR3zzzTdLOzs7RxAEgeTk5LrS0tKXa2trPVdffTUAvs/Atddem7Fr165n
    ent75zMME/P1amBZwwGmVCqVifLPlwLIc7lcoYceekjR3t5OilV1xGiSyLqJUSOVSgWLxYLc
    3Fykp6eHBWb+HPDRi29PBwDNsmwKx3HkpZdeGvcSczNnzsTll1+OgwcPEn6/X5fIgS5atAgE
    QXQC6FAqlZnz5s3D9OnTf/V3Azf74kHTtKQfFmskD5Q+iIfo29LT0zFq1CgwDIM333wTJEl2
    KRQK+7GSTuNg7QA8BEFoL7nkEojP3eRLQgAAG1tJREFU7dHGGy3JEMcrJr6rVCqprN1AmYc4
    z1UqFdLS0lBSUgKdTofXXnsN4Dt5Np/KFz9VAG0GUGY2mzFt2jR5oq7yueeeq8zMzGTb29tH
    gU9ka4nl569YsQIA0NraGunr62u3Wq3w+XytZrP5To/HMzoSiXDp6elHDh065LBa+VK+U6ZM
    QWdnZ2FVVdXTFEWdefPNN+Opp56C2WxOnHchCEUoFEqUg24A8DlFUWfu2LHDvHv3bpVer++X
    dGY0GkHTNNxuNyiKgt/vh9PpRDAY5DiO8wkPzF7Erv5vzBbB0aNH9yulEy9raGgYVGm8QVof
    gB6RdT3nnHOQm5uLvLw8qd6o2DBEZJfFV/G9KNERpR3Rr8FgsF+N0fb2drS1taGyslJkr5uR
    2AQlNwCXSqXCVVddhUmTJkk1VVNTU5GSkoKkpCQp7B59HA8siYcoexFbl1utVrS2tqKiogIf
    f/wx/H6/K8EbBg34CKFyCOaWQjj3OvAdZWNm27dvx3333cd5PJ4mhmGaHA4HNmzYQEyePHmH
    y+W6jCRJIj09/evFixf/uGzZMgC8rtTn8+m+++67+x0Ox3U0Tcvi5TuOZSRJKlmW1STIP9cB
    +LSiomJ8RUVFBnjJ3W/vMLu6cPDgQQ58onEreC278/TxzvxmrLKyUgQ9cSU4KioqxPcJJzi0
    Wq2Doqger9eL1tZWjBs3DjabDW1tbbBarejo6EB3d7fUDCVatiBK78SNoxgRjK66odfrYbFY
    kJWVhezsbBQUFMBsNqOxsREURUGn03VOnDgx9MMPPyRqyDYAXRRFZbS2tsLj8aC3txetra2w
    Wq1SwyuPx3PM8XIcJ4HkgeRHUlKSVK5UHG9mZiaamprg8XgAvmTjKeWnnCr4zQWQm5eXF8nO
    zk5YiDAzM1Oel5cXaG9vzxa+Q0s8zvPFF18A4DuHcRyHSZMmOe12+y6x1E97e7u0uHIcp/F4
    PPd4vd4Zs2fPxqOPPppQ8Cwy0KFQSJWAU7Hgw3pfCezWGIZhSlwu10yXy3V+WloaabFY8P33
    30On0yEvL49qaGhQCDVcPwevva0HX0mlA3HsLnk8W7x4Mfbt2xedqcudyEIYjwWX4zguLy+v
    n4Z81KhR2LJlSzxPnwRAn5KSgtWrV2PWrFnQarUnnTyoVp/Ynk3s/vf3v/8dDz/8MBiGMYEP
    9ydKE5wBIKO4uBjPPPMM8vPzj3tfouUb0cxiNKgWk1rEn00mE0wmE4qLizF16lQAfGb9oUOH
    UF5eni5s+BsTON4RGDobASBLAHMxtXXr1knvvV4vtmzZwvn9/t1+v3+3+PyUl//SPC85OZlo
    a2u73ufzLWMYRjEUFyOBEUIWvCxuC/gE0jHC60wAF5hMJjYvL09WVVVFJiUlobS0FIcOHQJF
    UR7wicUHBIKkCnzEJIzTzLZs2YKtW7fG/0Ky7JCNkaZpOQBlOBzG2rVr8e6776Kurg52ux1+
    v/83E9tFP/5bY5DJZNDpdEhNTZUkdgKzq3Q6nYlslcuJvvadd97BN998g/r6evT09MR0vARB
    QKfTwWQyoaSkRKopjV/aeicMQBsBaAsKCmRarTZhWjKNRkMUFxfLd+3aJQOf9Z0I5weO4/Dn
    P/8ZVVVVePvtt6VFc+rUqaBpeno4HF5QWFhIPvTQQ8jNzR2KZ04eIwCtEg5SOBjwMpmBjtQL
    vnX4QfAh2/FJSUl44oknYDQasWTJEhQVFeG1117jXnrpJW7jxo0kx3GHwWd2+3+DOVNHLQas
    cH46xveUJEnSKJPJFABw5plnkizL6gBI7GmsKlEcy/x+v7hr1stkMosI0mQyWZggCBfiK+0o
    BlBcVFSEc845R0p48/l8Uh1jl8slNcKJbqQivoqNVERGWizSL7b/NplMMBgMEhNgNBoxY8YM
    pKamoqenZzz4yhBdCXo+xgMwT548GVlZWYhEIvB4PHC73bDZbLDZbHC73XC73dLYQ6FQvzGL
    4Exk4VUqFQwGA1JSUpCcnAyDwQCLxYKMjAwYjUYYDAaYzWZMmjQJ5eXlqeBbK+9O0HizhGOo
    LANATjwA9EC77LLLMGPGDNhsNpAkiTPOOANr1qwBAPznf/4nDhw4MCoSidzDMIw+nt/jeBIO
    giCUkUgkFv5ZPcA/s+DDzwNrfvkAHBKOQgCT5XJ55P777/fk5uamLlmyBDk5Ofjb3/6GF198
    ER988IEMfJ7A6zh21IAAXwpRLfimaP8c02jSrbfeip9++gkul0tca1mRCRa7pCZq7QfAFRUV
    SfdYoVDgjDPOwKZNm+J23lAolC9sfKT26mJ+UXFxMbKzs5GWltYviTmadRXLjEZLzsRDzGkR
    m2Y5HA50dXVBLIYAAIFAoLSysjI1gf65BEA+AJSXl6O8vBxqtRo6nQ6FhYXIysqC2Wz+1Xij
    xwwAFEX1k9aJ4w2FQujt7UVnZyfsdjv6+vrQ1tY20F+VAjjpKgynCqCTAchSUlLIWNZ9/i1T
    KBQys9msBZ+UY0ggg3DU3+fn56t37969gGGYtBtvvBHnnXde4rZsUc6a4zhFMBg81XBtCngt
    8lkCc5QOPgyrFK5zO/iQnlU4ugWnqQRfcP9WpVI58+abbyYXLVqEdevWIRKJwG63Qy6Xax5/
    /HF0dHQk7dy5834AeeDrLLYJTjhZWOhzhQU3H0Cm8L3CwsJgBa+n3A3gCGLQqIGm6RSXy/Vn
    u90+juM49PT0gGGYLADYuHEjfvzxx7jfP6vVKjKdN3R3d18osJuEUqk8EAqF7kN85S1nAjAV
    FRWhrq4On332GQ4fPoyWlhapC6HP55OckAggB8676GeDIAgolUrJqYmZ4zk5OSgsLMSYMWOk
    uqM9PT3FwlxLhINWAJhKkqRKJpPhjTfeQEVFBQ4fPoze3l44HA4JMB8v+nC8jH+CIKBQKGA0
    GmEymZCWloZRo0Zh3LhxYm1gBcuyUwTwk4ha7QacYOg+TqYS/EpCLDrUHP1+7ty5sh9++OH6
    cDhclugLMGCuDIbgMAGYGOWfzcKcVgk+sm2Af+4SSArRP/9BoVBMv/HGGz233npr0po1a2Ri
    V0yKovDoo4+is7NTt2PHjns5jssBn1sk+ucU8NW2BvpnUeoR7Z93Ca+xyEuSCefTAuDGjx9P
    RiKRVIAvBTp9+vS4NzjZvXs39uzZA5IkzeDb1YPgT+oXxhxPino0gAydTodrrrkGZ5xxBvLz
    8yXJmclkgl6vl7S+x5OcDZSbiRsQj8cDh8Mhgcm2tjbs378fX3zxBUKhUDGAESqVqutU6jGf
    gk0BkKxSqXDllVdi8uTJksQuLS0NKSkpMBgMR5XY/daYRa201+v9lcSusrIS//znP+Hz+XQA
    poKPwpzUfZUPYoIT0SHMRFlUNSASQ2gVFRWYP3/+GX19fRdlZWXhiiuuQOIqFf2qbBIZiURO
    9uRK8Mk+txIEcZbJZEotLCwkzGazpGEWdUg9PT0IBoMUwzABgaHwCwt0WnZ2tn7ZsmW48847
    EQgE8MUXX4DjONhsNnz55Zd49NFHsXbtWjz99NOGr7766hav13s1eG0dC0APQCeXy7V6vV6R
    nZ2NvLw8JCcng2VZKTGrpaUlEggEOsB3oVwDXgJyygwtx3EKiqLGBIPBSWlpaVCpVFIGP8uy
    A3encduUCdGKLABZ4XAYfX19iEQiQSGEF09wMwmA4vvvv8eOHTvQ19cnhclUKhWMRiOSk5OR
    lZUFo9EIo9EIjUYjdbNSqXgsEN2EgaIoCYx6PB709fWhoaFBKmwvdoLyer3ipm0igH8l4FEx
    AziTZVm8//772LBhg8Rg6XQ6ZGRkoLCwUKpCYDabYTKZpGRJkXEWn7no5hMul0uqWtLX14fe
    3l50dHSgtrYWO3fuBEmSUCqV4nM6HryMoy0BY05GYqtvHG19MAylf7bZbFi2bFlhT0/PNTRN
    x32tGNjJUqwKIvgbGUVR8lPwz5cCuFVY3E0CC3wsCwtstF94VQn/J0mj0dChUMjwxBNPKLds
    2QKO4xi73U4+9dRTxLhx46BWqyGTyfQ0TS8F33jHLfjXJAHEan9jPtHgJR/bALwsMN+n7J9p
    mjY5HI6/9Pb2TgLACS3ajRzHYdasWXjyySfjjjtWrlyJ3bt3IxwO32Sz2eYJ95JQqVR7QqHQ
    sjgSHDLhfmtGjRqFZ5999oQ6GItgMXrzNrCZWRQRCbVa/SupaXV1NQ4cOIDm5uYUABNkMlki
    /HOSsB7JcnJy8MQTT6CsrOyExht9DCRzxPGKmExMHhwx4hdlW0dHByorK3HgwAFCIBGNAE6q
    6PepLtROALSYuS+2m4y3MQwjLsAMhjjBQWCXLg6Hw1lnn3221LkwUSYK54/FDJ6AFQN4VqfT
    jVm6dCmuuuoqjB49WmoAIIaAOjs70draivr6enVjY6O6vb3d5HQ6kZycjDFjxuDyyy/H5MmT
    QRAEVq9ejT179kCpVP4cDoctb7/9ds6MGTMwc+ZMvPrqq/juu++I7du3p7W2tqaFw2FYLBap
    FN6IESOQl5eH9PR0qQFBJBJBb28vfv75Z8Ubb7xRsHXr1j8IUovbMMgEJZZl2aSkJKxatQoT
    J04cUs2bWDpv+fLlcLlc8f4iZoGZgt1uR3JyMiZNmoSysjLpyMrKkmQJOp1OymYeuOsf6MRo
    mobf75fkEFarFYcPH8aRI0dQXV2N+vp6+P1+0e9MAi/ZCcZ5vCUQwqEymQxjxozBlClTcOaZ
    Z2LkyJHSnNNoNP0y1k+UZRQrG1AUhb6+PrS3t6O+vh4VFRXYt2+ftIlgWTYffJgwEQCawdBU
    d5EuDRJb8/rXk9xsRlNT0/lut7skEecTs/+j/XOUTyFYliVOwT8/I7CRJwq4lQII6Gcej0f+
    j3/8I/pXe1mWzdq+fXvh9u3bB/55qnCcLI7IA7BMANu3DsY/cxwnpygqKxAI5BwLOCVwnTVE
    IhFDFLjPiPOGzAQ+QkhMnToVFosFLMtK9Yx7e3slyZnH44HT6YTb7ZYkZ+FwGJFIRIoIiodW
    q0VKSookLxPr8ovkiEajQUFBAUaPHo3m5mY5gMmBQCAR/jkLwFgRUxUUFEiNTsT6zTabDU6n
    Ex6PBy6XC263W4qMiocorxFJD41Gg5SUFElil5KSAovFgpSUFCmZ0mKxYOLEiThw4ADAl3/M
    SxSA9gAIW61WNhQKEUqlMiE0NEVRXFtbGwNeczWkFRzuuOMOtcPhmMBxHFlWVhaz2ownA6AH
    CfraAezhOG50aWkpMXPmTLAsK2W4ip0ERVA1Z84ciXUMBALQaDQSE+n1evH+++9j9erVCAaD
    HoIg/spx3ISmpqa7H3jgAeK5557Dueeei6uvvhpXX321lBgghqGiTeyaJIbFCwoKkJ6ejk8+
    +QRCBY//FRiWwW/1ZTKUlZVh3LhxGGoTN6IJWBxKARSqVCpccskluOmmmyC2jB00ta1SQafT
    SczG5MmTMW/ePLAsi46ODnz//fd4/fXXsWvXLrAseyb4cHC8mxRMAZBqMBiwcOFCLFq0COPH
    jz/hBMjjmThHRUbHaDSipKQEF1xwAUKhECorK/Hee+/hrbfegsPhSBG+y7cJALdOJK5t9LEY
    SftQPk+PPPKIzOVyTWMYJiHsjthM6FgExyn6513Cwh7L9dUN4BXwEaDlMf5sH4DvYuSfj7q4
    ff/993jwwQfjei8JgsCuXbuOev8Igoj3xjAfwEi5XA6DwYBNmzahvLwcNTU1aGtrg8PhkKpR
    RLVfP6mNnqgvFvM2CgoKMHLkSIwZMwapqdLeaXyC/PMZ4GVCSEtLw9dff42DBw+iuroara2t
    sNvtcLvd0nhPFvOINfpFWaHZbEZeXh5GjhyJsWPHIikpCSRJgmVZizDmg4kA0DYAtvr6emND
    Q0OkqKhIGe+FnyAIWK1Wor6+Xga+k1nPUDpohmHSSJIsIEkSer0eCdIKSeb3+wcLoH0A1gYC
    gemvvPLKKJlMhiNHjqCtrQ2hUAhyuRxms1kqbxYdzk9KSoLD4YDT6URdXR0+/fRTbNu2DU6n
    0wPgrxzHbRKcf/5PP/105ZIlS2Rz587FJZdcgszMTBiNRpAkCavVCrfb3U+XZLVa4XQ6pc6F
    Irj9/PPPwXHcV+C7ULIxuoeor6+HXq9PKKtxtLldX18vMQdxNj2/vkdQXV2Njz/+GO3t7Rg5
    ciSysrKQkZGBpKQkqUHKySRTioysKHVwOBzo7OxEe3s7qqursW/fPtTW1orzVo3462QJgUGx
    BwKBtE2bNqG6uhoTJkxAWVkZCgsLpS5XWq1WSog8nq5woJZQrEPq9Xpht9vR2dmJhoYGHDly
    BJWVlairqxOrvfjAy85kCQC3PmHcQyWjCA81wREMBk2hUKgkUZElMcE0ekMcC/8Mvs1xrDTc
    bgAvAdgMPqckF3zjm1hJxr4EsAlxjD7s2bMHe/bswf9Vk8lk2RzH6RiGwd///neJsIolUePz
    +eDz+WCz2VBfXw+xYYpe/0ueLUmSFgBZLMvGE0AT4NnnJIDPP/rggw9ElUGscJrUPKi3txeN
    jY3S/NHpdFK0HbzkaQKAf5yMfz7VB6cdQKXVai276aabyNTUVCQCQAsd7Qjw2ZJtQznRe3p6
    tACSOY7DP//5z7h2rjuaNTU1xQK0HwLwWk1Nzf8sX74cFEX1gNeyBQSgZSYIQq9SqTRKpVKV
    lJSE5ORk6HQ6UBQFt9sNh8MBv9/PgNclvypMQJ9w3AWgtampaeGLL76YtmHDBimkQpKkVAkh
    EAhw4XA4GA6Hg+A7X9nARxmMH3/8cbZCoUgLhUJt4PV1zhjNJ9Ln8+HBBx+EVqsdcgAdDAZh
    t9shk8nirdfcDuBWlmWvaWxsnNzY2Jj90UcfGcTugRkZGUhPT0dqaqp0GAyGX7Xrjq4DTdO0
    pIEWtcBOpxO9vb3o7u6Gx+MBRVEcx3Ee8OUL9wL4GDHuVnc0TA/gLQBNNE1f0d3dPam7u7vw
    hx9+SFEqlXJxzKmpqRDntjheMRQovgJ89EU8QqGQtPlzu93wer1wOByw2+0IBoOgaZoVAEuH
    wGp8KVz7RDDDncJ5LUM0pTuR2MYxvzK73W6Qy+UJqyfa3NyMDz74AHK5HARBYOfOnf2Sb0/R
    KgG8BmCVMJe7wScJBgTQYRE2SdHVi46Km4Q5+Cr4JEGxccQ9wlp+I/jk8eOxwUHhcAjkVRi/
    JBmmC+vx33B61Y7+tzONRrOTZdmHaZqe7/f7x3IcZ0zUub1eL0MQhEur1dZqtdp/hkKhmliC
    2WP4528E4Drd7XanJvJaC3JCgM8bOAjgByQoiTAI4BuGYc6urKxMdLIKB+B7HL8cWtzN7XbT
    NE2HOI7j9uzZww3xrvhUd/wMgA84jiugKKoevDzCIYBXpbBBsFAUlU1RVI7H48nu7OzMBa9b
    ErerHQC2AvgEfJ3baCRqBfAw+OSSq5xO55lOpzNZcOgO/JI5bhU+pwN8dMEnfDcNy7K5oVBo
    rrBw7I3R9YqQJFkHQGu1WodSK9rPSJIkZDJZPcdx8QRZfvDZxtvAh+hG0DR9psfjGefxePKb
    mppMwr1NAqAjCEJG8AbxNbruppAsxQk/c8K99Qv30As+lN8ogIFK8HVmuxCDaionweRtAS+d
    SAMfIh0XDofHhsPhHJfLldHQ0GAUxpskMBFyCEnKxAAKmvtlp8UKczSMXxJrXQK4aBc2+VUC
    ULEh/lrCaOsSFoSJQzSVK4VnesjMZrPJaJpWJep8u3fvxv79+/sxX1EAWoZTk0ow4Ovu5wGo
    BZ906xT8s0oAz2bwFSuyhVfxvSGK7BL9c/NRNjqPCL5grgBkUvCLBKc9yjeLr/Yo/6wWznWl
    MMd/iiUZe5pi3Lh+L4Zh7A8//PCatWvXfmyxWMb5fL4z/X7/BK/XOyoQCBgVCoWRZVldJBKR
    cxxHHi1CNpCcGfjvBEGwCoWCJknSH4lEXAqFwsZxXIVSqaxkGKbCYrE0vPDCC7YFCxYkYm3c
    JRApEwFMFubgmCiMIUdsZUbi+mQDX2bzEID94BsHdeEk5XWDCd18BGAfEl8NgwNftmdIgY/H
    4+nkOO6/jUZjDsuyLE3THMMwCU+c4TiOUCqVIZlMdqpsXheA+3H08lodA1hCmeA0NcJBCCyb
    5zj3IxAFYJIB6IQFICCAKOo37mWbMLlJxK7eqEMmk/1RkcgajCf6QMrlEQF4xtuC4PVtTcIC
    K97XJAFopgFI4zhOz3GcGr/UoBVfIcwZCr+UtKIEENknHOJiG0Riyrcdz8L4hRndI/g+VRQQ
    SRXGrBM2jyoAKo7jVFHsXjhqrCFhPvqFcfYJz4J4HSJDPNYt4BPQhoLg+CqBG6RjsWl9JEl+
    ajAYRkf5F+5oICOGvjj6OZZyRHQ6nVehUPQOwj8/eIznpxN8ac9o/6zCL4w0Ifhmz2/4ga3g
    tcvJ4JMAI8LvxWf7eBfMKgCQWPrnkACs7EO9zg8wQgBccXu2g8EgXnzxRRAEYauqqtoeiUS2
    r1ixQv3ee+/pGYZJNRgM2Q6Hw2Sz2Qw6nS5Pq9VmkCSp5ThOTdO0IhKJyCmKIgFArVZzSqWS
    USqVjEKhiHAcF/D5fFaPx9Ocnp7uMZlMTpfLZQ0EAr05OTmeAwcOREiSRDgcxjPPPAOKStgj
    7BTwwbfC3E0Hn/idCb5Gs1jX3izMT4Xgv6NfCeG+RASQHImax33Cc9QFPorTLZA6NsF/n/Ic
    GwyAduEkBdf/l0yhUFAul+vL0+G7OByOweqGThTcMMKEOxX2PwyeXT6VhSTWjCwHwDWUso0T
    WYQTbOJmxikwT8dbRIiBoOTf0OgoNsKBOHU1HUL7HMDOGLM3J7MgDqnt2rXLUVpaer9SqZQN
    5TPGcRxUKhWn1+sHA7pOxj8HcGpJfBEBaJyKMcIRsyVNIHWI0+2h4jiOjffm2G7n82+ffPJJ
    VFRUoKqqigqFQhTLsr2RSOQIy7KSjGzNmjWyrq4uoq+vj+jr6yN6enqIri6+vH5mZiYyMzM5
    8dVsNnOLFy9mCILgxM8Ih8OgaRrd3d1IS0uDXq9Hc3MzOjo6huoSBwXSrG3AmiOPInCOBqAR
    BZyjX8V1jUEctPmn3QT9d7L58+djzJgxkn7VbrdLdW8TaTKZDMnJyXjrrbeGb8qwDduw/X9v
    48aNg9PpBMuy/cLYid6gchwHjUaD4uJibNu2bfjGnIAtWbIEX375JWw222n33VQqFWbOnJmQ
    duLRlpSUhBkzZoCmabS0tKC3txelpaXQarUQwXAwGITP54PHwwccDAaDdIjlSCmKQktLC1iW
    RXFxMQDg66+/Hp50wzZswzZswzZswzZswzZswzZswzZswzZswzZswzZswzZswzZswzZswzZs
    wzZswzZsw/bvav8PFQIJ/oketoIAAAAASUVORK5CYII=
  }
  set pieceImageData(Merida1,65) {
    iVBORw0KGgoAAAANSUhEUgAAAwwAAABACAYAAAC3DfDxAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw02NKTYwgAAIABJREFUeNrsXXd4FNXb
    PTPbskl2QyoJ6Z0QSgolgdC7NCGIfAjSlKKACoqAAhZAf6IRRaUIioIoKqACUqRKr0oPhIRA
    EjZ1N3Wz/f3+mNllEzahhSRgzvPMo2RnZ+bO3nvffl6gAQ1oQAMa0IAGNKABDWhAAxrQgAY0
    oAENaEADGtCABjSgAQ1oQAMa0IAGNKABDWhAAxrQgAY0oAENaEADGtCABjSgAQ1oQAMa0IAG
    NKABDWhAA+oQTMMreHAsX74cLMsyTZo0wY8//kjr1q0DwzS80gY04AHgBiAcgBMABYArANT/
    wffgAiAUwDUABQ3TogENaMB/GGIAXrxcyOMPw39YX6e6fABhw3x8cOzduxcGg6FJRkbG00Kh
    8J/Ro0f/C0C9YMECvP322w0v6N7B8nNR9zg99Lhx48AwDJOXlyfXaDTiqs6TyWTlrVq1Ks3I
    yICbmxsOHz4MV1dXkVqtdiIimxYmwzDEMEwRAP2OHTvwzDPPwNHREQUFBTKNRmNX1b3s7Ox0
    /v7+xQzD0Oeff17Xv6mY/01Nd9kEEwC8AaANAHsASgBbAXwCIP0u9xEBEADQ1vVmWkMYAeBN
    AB8BWNqwNTx+GDNmDIhIplKp/EUi0Y2NGzeWzJo1Cx9++GHDy7k/CKzkAjW8jv8cQgBMAdAN
    gCM4B8pGAKt5w+G/hBYA+gHYBeDMf81gcAEQw78EJwAlAM4DOI3HyKumUChgMpmcr1+//rpG
    o5FkZGTsbtmy5drLly8fAlC+YsUKTJw4scJ3Fi1ahDlz5qB///4hKpXK19PT86hardZs3779
    v7ghSAF0BtCJnxOpAHYAuPA4CIjCwkIwDOOSlpa2uKCgoCUAow3Fn/X29v4jJibmg5KSEkOT
    Jk1w+PBhxMTEtLt169YCk8lkb2OsDMuyZQKB4HUiOg0A2dnZ8PLyEqalpc1UKpV9ichoS0l3
    dXU94+XlNYdl2bpaR2IA7QH04j1DCn6TOwxAb+P8ZgA+EQqFbXx9feHs7AyFQiHPzs6eTESO
    AGbwBkRlNALQh7+XA4AUAFsAXHqMlQshLyR9AATz//4vetNYXlk03sXYrJe4desWjEajb0pK
    ykq9Xp/ZokWLX/755599RJS/ZcsWDBw40Ob3hg4dCkdHR7vs7GxpQUGBys7ODgcPHvwvygV7
    XknsBMAZwHUA2wGcfRzmw7hx4wCALSgocK/KucMwDGQyWUmbNm2U169fh7e3N/7880/4+flJ
    CwsL3YiIreJ7JpZlc4lIu337dgwfPhyOjo7Iz89302g0DkRUlSNJExQUlMcwjCkpKakuXgsD
    LnKaAKAJr/AfAnC5it/UG8BiAAP5/QAAAnmdsTGAebzeaEv+xABoC0AGIA3AQQCZj/F6YAAM
    BTCX15PO2tI1nkSDgeF/yBkAugJwxe0wixLAfnBexWOPg9BXq9UwmUwmrVZrKi0t9SotLR2l
    VCp7KpXKX2JjY1efO3fu/IgRI0zr168HAHz88cd4/fXX0alTp5anT59erNFoWqjV6k87d+78
    lZOTU9lPP/30XxIKDgCmA5jCMIwHwzAwmUwmAP8H4C1eQNRraLVaMAxjp1QqoxQKRbRQKATL
    3t7njUYjTCYTHB0dUxiGEeh0OgPLsjAajSgqKnJTKBRtiMheKBRaUtmICAaDAQBKRCKRnHsl
    QGlpKXQ6HatUKptmZ2fHMgwDgUBguZfJZILBYADDMEaj0Sito1ciAjAewGwAvizLgn/+5wAs
    AvBNJaPBvBG27t69OxYuXAhvb2+cOXMGM2bMECQnJw8GsBnAH5Xu4wrgXQDPMwwjs5o7Q8F5
    5/c8jkJBJBIJ9Ho9Y/VuWNxOG/0veFh9eCWxFQC5lSPpAICbj8sgVCoVTCYTU1hY6FVSUhKv
    Uqn65OTk7GvVqtWKVq1aHRCJRGWffvoppkyZYvlOhw4dUFZWJjt//vwUtVrdNjQ0dPGePXuO
    vPfee5g/f/5/SS44AngdwMvg0hTNc384Lxe21ve1UFBQAACu165d+0ypVFbpSPLx8fk1Pj7+
    /eLiYoOPjw+OHDkCgUCQkJaWttBoNNozDEN3fo0pFQgEU4noJADcvHkTHh4eopSUlFkqleop
    k8lktJEWzbq6up7y8vJ6nWXZuvDMMwD68kp+NK/UG3jH4EIAm2wYDYN4h1Blw0kCYBSA3QC2
    2TA0JwGYCsCP/64awN/8vU8+xg4UR/49Olo5U554gyEewBcAor29vRETE2Py8fFhFAoFTp06
    5ZqZmZnIe9im8T9y/d7ZHB1hMpl0DMOUCwQCBAcH49atW55ZWVkvFxcXdyksLPwoLi5uY0JC
    QvnBgwcRHR2NHj16+F+4cGFRXl5eL6PRCJ1O95ZaraYhQ4Z8NmDAAP2WLVv+K4KhD4BXvL29
    XZ999ln4+vri2LFj7B9//BFdXl4+F8BVcBGH+g4CQAKBABMmTED79u1hMpnAMAw2bdqE3377
    DQBM5s1fKLQsOSIiCgoKwtSpU+Hq6goAyMnJwWeffYaMjIyqlERiGAaDBw/GoEGDQERgGAaH
    Dx/GypUrQZyLqa4EajyAN52cnHwTExMRGRmJ8+fPY/PmzX5FRUWzAFzkvUoWxxeASJFIxPTt
    2xexsbEAgB49eiAuLg7JyckyAJE2DIYRAMb5+/tLhw0bBnd3d+zdu5fdvXt3rMFgeAtAMoCs
    x2QdOPFGcqjBYBAIhcIuvMHYEcD/+N8yHcA62I60PAlgAfQAMBNAO14omlEG4BTvbdxRV4Ly
    AWAgIi0RQaPRyLRa7cDS0tI4pVL5c0xMzGcvv/zyNZPJhGnTpqF79+4ICAiwO3DgwCtZWVmz
    dDqdg1ar9YuLi5t+/PjxA+PHj8fq1av/K3KhL6/wuVRSOFsAeBucR/pafR5AeXk5GIYRK5XK
    pgqFIsKmBs0wcHR0DGRZltXpdBaHUUlJiWt2dnYLo9FYVWSiRCgUOpojCcXFxZDJZExBQUFg
    Tk5ORFURBoZhSgwGg8TaoVWLaAbgfXCef2vdM4r/ezq/xq0dT615+WALLgBiAfxZSdb1BTAL
    gHslI6IPv2+MB5DzGK2FRgAaiUQiERG58HJBDiCAd7wV8zKh1uR9bRoMjXkPQXSHDh107777
    Ltu2bVuhRCKBVqtlTp48aXrvvfdw4MCBVuBCL2Pqu9D39fWFwWAoO3fuXJFAIMDYsWPh7OyM
    ZcuWsRcuXGih1WqXHD58ODI4ODgpNjY2b+DAgaLvv/9+Um5ubu/AwEDExsbizz//dMrKynpt
    165d50+fPr1z/vz5ePfdd590oSAC0FcoFLpOmDABs2bNglgsxq1bt6BSqbBr164ocPnsqY/L
    gBiGQadOnfDss89a/nbt2jWzwSBQq9WS0NBQKiwsZI4fP24aO3asiIjg7u6OxMRE+Pj4AADS
    0tKwdu1aZGRkQCKRiFJTUwWLFy8WCIVC5OXliRmGYRmGQXR0NEaOHFnh/qtWrXqUHqJAAM3B
    hXmvg/P8Vg4J9wDg379/fyQlJcHJyQlFRUUoLy/Hhg0b/AF0B5eaZN7gTAB0JpPJHK0Dy7LQ
    6/UoLy83f66tdA8ZgN4ikUg6ZcoUvPbaaxAIBBgwYACeffZZnDt3LoYXRJX3DjdwnmsvALng
    wrr1QXj4gKvfCLKKLoH3xEXz/5/Fv7cn1WDoBeAzAGE2PnMAl7boCy4i+Xt9H4y7uzuMRqM6
    JSWl2Mo7ALVa7aHVal9Sq9Ut2rRps6Bfv357x4wZY1qzZg0iIiIG3bp165Xy8nIHAMjLy4sx
    Go3vd+rUaezp06dT8d+ACEDvSsaCNVqCy1Co1waDlYf/bulTxLIsGIaBSCSydkCZqrs2wzBk
    Ngz4f1scSVUZDGbH1iMgZWnEywZHcOmnN3Bn6mlv/rezhTBwufmnKym+d7NsbEUehlQyFqyR
    wOsUWyv93Y5XwN0BFPKyrbQeTCNv3mEUaTAYGJZlfa3eZST//zd52XH1STQYugPo7O/vr/vw
    ww8pISHBcm+xWIxu3bqxdnZ2xlGjRhnT0tI68C/mm/q8Mfj7+0Oj0SgZhrml1+uhVCoxc+ZM
    dOrUCUuWLMG6detcs7KyZhgMhsYtW7ac9ffffwfn5+c/Z29vL3z99deRmJiI2bNnY82aNU0y
    MjLe6NKlS/revXuzhg8frvf29taWl5fjq6++ehIFgxSAp0QiQUREBMRisUXQ+vv7A1zI0udx
    G5StzZqIoFKp4n/88cdVRGQCgE2bNpFGo/HhN7kK3yMi87+lDg4Os9q3bz/GKp+VUavVbU0m
    0x33MqcuPSIhPgRcmkAEv2cUgksV+pAXEOYN3B0AQkJC4OTkBABwcnJCYGCg2ejwBBdONWvF
    WgAnjEbj0A0bNkgCAgLg7++P48eP48CBAwCX51q5wMsRgLNEIkFYWJglLcvX1xdeXl44d+6c
    hHdOWCOW90524T1O5QCOgAuHH67jaXMDXLg8hGEYAcuyfYxGYxt+3Dt54Zte35Wkh/G7gEth
    C7vLeUEA5oBLY6jXCrSXlxe0Wm2RyWTKrfyZ0WhkVSpV56tXr/qXlpbOnT179o8KhcL71KlT
    09RqtZv1PlBUVNTh5s2bU0aPHv1WQECAmnc+PMmwA+Bxl73I90kZLMMwwszMTAdXV1dRWloa
    PvroI9OGDRvsmOq1esbR0VH69ttv26WkpAgYhkFpaamYYRhhNcbCo0I8b8S34/fVbAA/A1iG
    ikXJftXomixvcAitDA09gHP8f0U2vlPMO6yoklxoUs2z2vP3qbz3TAMwgHcoFQPYByAJXDS8
    LiHmny+AiMhoNNrzf3fiPyNeptZq+nFtGQws7xlw6NGjh7pt27Y2Q01t2rQR9OnTx/jVV19J
    AcSBC8PXW+achQsXgoi0W7duzSwsLERaWho0Gg0iIiLw0UcfwdfXF0uWLBHl5OQ8n5ycLJRK
    pSa1Wu3Tv39/DB06FG5ubnjllVdw9OhRXL58ufOpU6c22dvbZycnJ9/Q6/XnfH19UyMiIv71
    8fHJLikp0R87duxJ2Su1AAo0Gg3Onj2L/v37QyqVQqFQIDU1FbxCmf+kDDYvL88/Ly/P/37X
    ZnZ2dre7eLCq/VsNoROAD6VSaUBkZCQaNWqEK1euNM7IyJjAb1hvgIs0mMB5wk3//vsve+vW
    LTRu3Bg5OTm4ePEiwIWEb+LOIt6NALqePXt2wIQJEwT29vYoKiqCVqstBfA1gOM2hEWuRqPB
    qVOn0L17d0gkEly6dAnp6engjYGMSsLqfyzLdg8JCUFAQAAyMzPFV69e7WswGOQAxoIrmK4r
    lAL4QSgUwmQyCY1GoxPvCTsG4B08ZsxhD+hIanOP50aBSy/4sj4PyMPDA6NGjSrZv3+/whw5
    q+xEKC4uDjCZTB8lJSUJxGKxS0lJSWzl6xgMBjYvL2/0d999J9Tr9edbtmxZwLLsVTs7u1sR
    ERGFKpXK+IQZETpUH0Uz4PFKKanWuVRQUNBp48aN68yOJIZhSKPReJtMJnE137MTi8Vzv/zy
    y8lWDHtMeXl5TC0bDC0ALOH1OjNcwWWROAKYz+/F4GW5CVVHDXJxZ6rhbwD6g6t1rWBz886q
    fZX+Xs47sqqbW9YGfCPwdXC8E8v8/IG84TERdVs3dRNclo0Lr6dPA5eKux1ceqYGQBHuziL4
    WBoMEgBuLMuiWbNmUrFYbFO7EYlECA8PF/DFku68x6HeCsyXXnoJDMOgefPmJwUCgeb69et2
    ubm5CAgIgJOTE9544w1IpVK89957goyMjOcEAoFBIpEwgwcPhpsb50yKjIzEkCFDsHDhQmFZ
    WVkztVrdzJwDX15eXiIQCK5mZWXtk8vlOxMTE09t3LixMD8/3/L9x9hg2G40Ggd8/fXXznl5
    efD29sapU6dw+PBh8N6Do4+7UOjUqRPeeuutKpV5k8kEPz8/yOVyy9+cnZ0xYcIEKBSKKr/H
    MAwSEhJqa90OZxgmYMSIEZg7dy5kMhkOHz6MadOmsenp6UMB/ACOhQLgPOIjd+3aFT5+/HiE
    h4fjypUr2L9/P3ilfKeNe2SBi16klJaWJpaWlnqBYzr6hr92eaXzywD8YTAYui1btkyempoK
    FxcXHDt2DFevXgW4iMG/Vuf3BZDQtm1bLF26FMHBwcjMzMSrr76KvXv3tuW9S5+ijgsprVKR
    rNO1nvRCZyG46M+9esnE/PkS3JmqVm+wb98+fPDBB9S8efPLCoXCaKWQVLQUS0u9rl+//pFY
    LC7X6XQSW+eUlZU5l5WVTREIBJSfn1/OMEyOvb19ekpKykGZTPZ3ly5dzm7bti3/008/fRKo
    vLXg8tIHwHZa0llwkcEnAvn5+T75+fn3FUknImF2dnZ8HT+6gFde21SxRp8Hlzpojt6aSQsC
    bJyfDY6konKIPI2XC6+DY8xqxBuLG8GlL1Y2LNUA/gKX3mhv4z4XUbHouQu4yLmttdkVwNMA
    6pKX3AguPeo6/4wKq/d1oq70YmEtDl5tMpmQl5fHmHOVbSlQ+fn55nQLNeo5peBXX32Frl27
    AsA/N27cyLx27VrImTNnEBAQAK1Wi3379uHYsWPQ6XTQ6/WsXq8XBwUFoXXr1pZrsCyLrl27
    4ssvv4RYLMZzzz2HgoIC/PPPP7hx44asqKgoFkCsVqsdfejQob+joqK+XbVq1d6EhITyQ4cO
    Pc775RaGYYLy8/NfX7lypXMlofAOajEv71GhW7du6Nat2319x8XFBS+//HJ9GUIjAEESiQR9
    +/Y1p4uhc+fOaNGiBdLT0+XgqPLMBsNpAO+Ul5e/tWPHjmY7duxg+bV/CcCCSoq8NVLB5WuG
    gktr2gYuuqCv4vyNAPzz8/MnrF+/3hNcpEMLjijhfVQMhzcDIGnXrh1iYmLAsiycnZ3RqVMn
    7Nu3T0RETVExHF7XMAtOw3/AYBCDo828HzjXd4NhyJAhKCgogEQiOSGVSlWlpaU2vTssy0Ii
    kXhoNJpqryeXyxEWFsbk5+fb37p1K1CpVAYyDNOlpKQkv7S09HRUVNTPfn5+fwLIMZMgPMbY
    CsCfVxSt05P+AZe6l4IG1DXMZBRVTTRXcPVuZoPhGICPwaUeelcyFpbwBoUtnAHwGv/d/gB+
    5OVIVTUGG/j7juSf0YzL4NJn06z+FgMuvacqR5mZzam+OKzZ/9JD6MAxlxiOHTuGnBzbUcW8
    vDw6evSoia/oeSw6vWo0GrAsq2IYJre4uBhbt26FRqPBX3/9hRdeeAG//vorPDw80KZNGwiF
    Qnh4eMDLy6vCNYKCgtC4cWMIhUI888wzWLFiBX777TesW7cOU6ZMgb+/P3Q6nXtOTk7itWvX
    vlm/fv3ikJCQpkSEkycfT6YwmUxWJhQK/xIIBLlWhVsAl7v4WNDq/gegA1BuNBqhVN526KjV
    ahQXF5uV27JKyu4v4GhUp4ILn94C1x/hd1RfBOgILk9dBqApv1lXhWLeoDgDrgbgO3AMGON5
    74s1SgGgqKgIOh239xsMBqhUKrNjoqyezDUBuHC4mRUJuN2L4UmFHra51KtDST0y7mxi5syZ
    iI6ORkBAwBWBQHDF1jl+fn6YNWsW1qxZg7i4uGqv5+HhgY8//hibN2/GihUrMGnSJDRt2pQx
    Go3u+fn5fdLT05deunTp+6eeemrgqlWrJC+99NLjPCfU4OjVK9d/JOPOwtgG1A3uxsRX+XM9
    v1+PBpdOcxRcxGE9uPTC6ixmI78vNuLlQ3X3LQBXl7aLv/5ucGlHo3En0969jKE+oRSAipd9
    dfZstSmM9gBIPXr0aPjSpUvx1ltvwcHBwfJheXk5ffnll8ZDhw4JeYG5q16vGCL069cPHh4e
    jvv3759eXl4eAwAHDhzAlStX4ObmhtatW+P555/H4MGDwbIsFi5ciNatW1dIQQEAOzs7NGrU
    CAqFAiUlJZBIJAgMDERgYCB69uyJUaNG4aeffsJvv/2G9PR0j8uXL0/Oy8trGx0d/b+QkJAt
    bdu21Z04ceKx2nFKSkoiwXmDw9q1a4fOnTtj7dq1UCgUQ8A1WfkfuBy9xwImkwn79u2DrZzl
    R271syyOHDnyKO5bBOCkXq/vs3z5coFYLIaHhwft3LmTOXXqFHCbLanyBn+O9+qEgMs57wFg
    OSrWFlSGP24zXATyhkNZNee3BFfnJOH3lh+qOO8oAOXOnTtdPv30U8TGxuLixYvYvHkzwOW8
    HkPdRzJdwYXxx4IL20t446cXgG/5owBPHvTgipirKm60pTxcwJ1pavUKH374Id588000a9Ys
    Tq/X31G71KlTJ7zzzjvo1KkTSkpK8MUXX9x1b5HL5YiKikJUVBRGjBiB1NRUbNy4ERs2bEBy
    crJDdnZ2r6NHj0ZnZmZ+GxYW9iWAm49ptKEFr/RFVvr7IHDpiwtRfa56vYOLiwvs7e3r5N5l
    ZWVQqVSPQnm9AC51jKlCcb9gw/m0B1w0+h1wFMptwdGEVsdK5A2O2Q68weB0F7ngDS564M7P
    lZVVnHeaV75tRRm0vDOqPkQXGuM2y5QGXARlDLjUvVpnEa1Ng+E8gCStVrtg5cqV7i1btsTw
    4cMtH27btk375ZdfijUajQpcjlq9dp0PHDgQkZGRgj/++GNsdnb2ZIlEYufk5IQbN27gr7/+
    wvTp07Fu3TpIpVIIBAIQEb799ltIJBJrCjUAgIODA4KDg6FUKu8wJiQSCdq2bYvo6Gj83//9
    Hz7//HNs2rSJVSgUbYqKipap1Wq/wYMHr7C3t1fzueKPAyLBMRH0io+Px6efforo6GiEhoZi
    3rx5DgqFYho4b/RH/KKuz2AAMCaTCatWrcI333xjYTp61EVo5sgM37jM3P+BAcDU0L1N4IgH
    4s6cOdPrxRdfZFmWZfR6PUwmUza43P/kat6LkP9vc3AdmTfcxWAwp6d4gGOIyK5m3+rLK9pa
    VJEjzmMfgNUKhWLS3LlzZWKxGHq9HgaDoZz3cO2o4/njzAu20a6urnYhISGQy+XIzc0Vp6Sk
    NFOr1e+DS9WaxXuYnjTsBpdfHHUP516F7TqYeoMRI0bg7bffxtatWzvm5OR8pFarK+Soi8Vi
    jB49Gp06dcLFixfx22+/4dy5c9VeU6VSYc+ePXBxcYG3tzfEYjEiIiIwZ84cDBs2DL/88gu+
    /fZbpKamupeWls7Izs6OiY2NnQPg5O7du9GjR4/HyVhI4h0MlWEP4CXeaPzgcXEmiUQivPnm
    m+jTp491nVKtOZI2btyIRYsW1bQsMvJ7Z1dwTEkVbBTewXGmmu8K+COK//7Gau4VwO/z1nLh
    VjUypxf/neqKrAEuDWoDuOiDdf0QgauFqA9sAu3AMcP1ZBhGKhAIYDQavYioK4BEXm4cwhPa
    h0HGKxf/FBQU9Nq7dy8SExMhEolgMBiwZ88egVKpZHnD4jxvedZLAdmtWzf89ddfuHr1aruM
    jIxXTCaT45QpU+Dj44MPP/wQ5eXlICI4OjpWUO7MVJN3vBiZDHPnzoVKpUKrVq2q3HjatGmD
    JUuWIDIyEl988QWysrLcMzIy5m/ZssVu8ODBS+3s7Eq3b99e371KUeCiB70SEhLwySefoG1b
    jmjh+eefB8uymDdvnkNWVtYr/Pmf1Nd5wBuDGoFAcItl2Wgz3WlkZCSaN2/O7T6PyGgwc26f
    P38ely9fBhGBZVkIBIJbYrG4Jj2wqeAYGkbr9fo4cLz4EnDRBVvFatYQWAn7fuDSkmyFn1kA
    ISzLisViMTQajSu/6Z+sRoh0tRIS1QmGMt4oiDUajeLy8nIh7wHbBS4nti4VDxZc+tbIqKgo
    u7fffhtxcXFwcHCAUqnE77//jqSkJLvMzMzn+D3xK9yd2/1xwxUAHzIMs5iIqqPMvMU7EM7V
    14G89tpr2LZtG+Lj472Tk5PnqFSqppXP0el0WLduHc6cOYPdu3cjJSXlrpFBlUqFefPm4aef
    fkKfPn0wcOBAtGrVCmZq4VmzZiEhIQELFy7Evn37BLm5uT2MRmOjvn37vrl9+/Z9e/bsoe7d
    u9f3eRDN/77VWTf2AKbwCtJi1NO+JHxkhwHAGo1GlJaWIjg4uEJGRW2gqKgIxcXF1s/D1KA8
    ugjgFXAduePAERfowUWWv0PVaUbW+7UjLxe2ouqapABe5gBcIXwA7mTOg5VB0dvsxLuLXCgC
    sAZc2qcPP7eKAezl99nMOp5GTQEkicXi9nFxcejQoQPc3NyQmZmJ3bt3Sy5cuNCbiNwAjKvP
    e+KDwAFcNfomcKlGapZlad68eWQ0GonnmKX333+fhEIhgcthvAku5+xZVCxeqRfo378/nnnm
    Gam7u/sKANS9e3fKysqi0tJSOnToEGVnZ5PRaCSVSkVarZZqGlqtln7//XeKiYkhACSVSgsD
    AwNnd+3aVdq+ffv6Og9Y3vo/AsDUuXNnOnXqlM2xrVmzhnx8fIhX9lbyi7re4c0338SmTZuY
    6OjoWA8Pj9+FQqEeADVr1ox++uknKi4upvLyciorK6vRo7y8nIqKimj9+vUUERFBAEgoFGo9
    PT03t2rVKio5ORnvvPPOo3AuuPMeniT+t3kLVXv3JeD4uInvcp0GrnFaVXvET05OTtShQwcS
    CoUGAG9W8yxjAZTz19UBeKGac53B1VXkgOuo7M3/rabrtxz5eRrPKz19eQ9REO/8sAUvAMfl
    cjmtXbv2jrWg0+no448/JqlUSryQ9KpCADuBS//qAC51ox9vUDUDl/tbXz0IDDgKw6eFQuF+
    3M4rtnUcAjCUF+5sfRzMSy+9hN27dzM+Pj5viEQi3V3G80AHy7IUFBREr732Gp06dYoMBoNl
    vly/fp1eeeUVcnJyIgDk7u5+Nj4+vhsRYfHixfVVLgjApS0ew21msLsdZQBW83O+3iExMRHD
    hg1z9fPz+51lWZLL5fTyyy/TpUuXKDc3l3Jych7pkZubSxcuXKAXX3yRHBwciGVZ8vPz2zB1
    6lSXqVOnPgpHcBi/ty8BRzgxrZo1KuTlh/m3TAFR+Jp8AAAgAElEQVQXga5WhvCHEVzhdFUY
    AC5dzdz8bko159qBi5Df5I2eFvz+La2FPe9usAPwGcMwphEjRlB6erpljZtMJjpz5gz16NHD
    /E6+tjKo7rbOWNRzuAJYBCCPYRhyc3Oj9u3b0+zZsyk1NbWCcExPT6e5c+dSQkICubu7m5WM
    An5yNa4vAxo2bBhcXFwQERHRSiqVXpPJZLR+/foKY7lx4wYtXryY+vXrR8uWLbMYRjWNw4cP
    U2xsLAEgsVis9Pf3n5iUlCSYOHFifZsH9gBeBJDKsiz169eP/v333yrHpdPpaP369RQaGkrg
    8sv/4pWheqf49O/fHy1btkT37t19/Pz8FtvZ2RUCIF9fX/r888+ppKSkxn/3kpIS+uyzz8xG
    FUmlUpWvr+8HvXr1auLu7l4bw57Fb8jXUJGHu/Kmt8LBwYGCg4OJZVk9gBnVKM5H/Pz8aN68
    eSSXywlcWNuuCsX8Z6lUSmFhYcR7tSbcxbgoBccF3vERvAsfAJN5o+QC7wnP4/euLHAMUb+B
    Y3jpgYqsQK0BKCIiIujy5cs2f+srV65QZGQkgUvPsqYxdAHXw2Ae71xJBke9p+LvncML4638
    ew+qZ+unEf+7HeSfV38XJdHAe5SP8gpJveKVHjt2LMLDwxEVFeUrl8uPVn5+e3t76tq1K02e
    PJmCgoJqxHAICwujpKQkys/Pr7A3fPzxx9SoUSMCQDKZ7HSzZs3iANTH1CQHcHz31x/gHRjA
    RTg71je5MGfOHBw4cIBp165dCy8vr59FIpFOKBRS8+bNqUuXLtS5c+dHenTp0oUiIyNJKBSS
    SCTSeHt7r2/dunUEEeGjjz56lEN/nVfqL/IKuC2IeMPC/Dvq+PVclbNnV6Xf/bsqlHoxHxkg
    K4OhOuuoD79HFvLOnUcBObh6inHg6jU/AUfV+h6vD7XH7WamlaMqFz08PGjXrl1ERHTz5k3a
    s2cPpaSkEBHR9u3bqXHjxgQuAyDchnPWg7/+BHCpS8sAfAEuMvcKv27c69O6cQRHh1UulUoN
    48eP1+7du9eUnZ1NOp3OpnDU6/WUk5NDBw4coJdfftnsKdHyE6FRfRjU008/DQCIiYkZLZFI
    tAkJCaRQKCyRkj179lDnzp1JJBIRy7I0depU0uv19Kiwe/duCg4OJgDk5OR0vW3btt0BYP78
    +fXJaFwIoEAqldILL7xA169fv+u4DAYD7dmzh+Lj483G4yUAw3FvxZG1Di8vLwwfPty+adOm
    L8hkslQAJJfL6ZVXXqFDhw7R8ePHa+Q4dOgQTZs2zaxUk0wmu9qsWbOxzz77rNScBlULeI5X
    wglcvwSHKgyGVc7OzjR58mRydXUlfvO3pei1BnA9Li6OtmzZQl5eXgSOKcXWue0A3GjZsiWN
    GjXKrDhMruI5Q3jPvHljrckXJAYwjFd4ywGQo6Mj+fn5UUREBEVGRlJAQAC5urqSSCQye8fy
    wDFHvcx75QYBUMXExNCNGzdsroOcnByKj48n3hB4mhcO08Dl8ucBMIrFYnJ3d6fAwECKjIyk
    Zs2akb+/P8lkMrPw1IJL7xqJivm6dQUfcJHDkgdUmNUA1uLOzq11huHDh4NhGMTExDwtlUqL
    rJ+3adOmtGLFCsrOziaNRkOTJ0+usYiDvb09DR8+nJKTky1zRq1W08KFC8ne3t4cadjRs2dP
    37CwsPokF9zA1SIoH/IdJIPrCSCuT/Lg6aefRnx8PPr06eMRGBi4QCqVFlg/t0gkInt7e7Kz
    s6uRw97e3rzPWM+NvMDAwLmDBg1yj4yMrI1hP2u1ppdW4fAR80qz9bP+Cdt9N8J5J4z1uQeq
    UHSbVjrXxCvGtuAOjrmPeKdO60eg+w4CV5uRaZYPlQ4Nb7Ac5p3i3a2cSfEAcps3b07Xrl0j
    lUpFw4YNI7lcTj169KD09HTKzs6mtm3bEr9+eloZWF3BpfYd5J1MGhv31vGyYzc4sg2Xex3Y
    o6xhGAbgRblcLpg9e3bZyy+/7CiTyar1BJhpRz08PBAbG4vw8HB6//33RXl5ec/zCuOXqGO6
    q4wMjuilsLBQRkTi8PBwNGrE2TIHDx7E1KlTcenSJQBA7969MX36dAiFj+41d+3aFW+88Qbe
    eOMNFBUVBaSlpc3u0aNHyrp1624uX74ckyZNqsvX5Qmu4+NYFxcXydSpUzFt2jS4uNw5Pyv3
    5hAIBOjWrRtWrlyJefPmYevWrRF6vT6JX4zfo5419FMoFHByclKPHz/+mx9//PFaRkbGO0ql
    suMXX3zBfvfddwAsua0PdH3r75aUlACA0cPD428fH5/5EydOPHz+/HnThQsXamu4WfxG5MBv
    jNtwZ+EaA4A1mUyIiYlBcnIy9u3bF8V7XCozoPkCcAsICEBISAgaN24MhULhxUce8itdszuA
    Jl26dDHXCDFVeBhF4NgkYvh/56Lm6hWkvIdoNsMwnkFBQejbty+6du2KwMBAODk5gWVZFBUV
    QaVS4dq1azh06BB75swZt9TU1D5qtbo7H51JAyDOz89HdnY2/Pz8bM6rvLw8ABAzDDOZiPwA
    BDs4OIhCQkIQGxuLjh07IigoCI0aNYJcLjd3EkZ6ejr279+PP//8U5ySktKaiJJ4z9Ny1B1l
    tQtud1gVPsT7H8F75l5DPej+m5eXByKCXq/3glUabevWrZGUlISEhAQwDIOdO3eiJvvnqNVq
    bNiwAWVlZfj0008RHBwMqVSKl156CdevX8fq1auhUql6pKSkvNqjR4+5KSkp9YGq3MtqDlRp
    wIpEIkgkkrsxz4Xznls5uKhkvejPYe7A3axZs9xevXotOHToUHJWVtZbRUVFTYkIzZs3x8SJ
    E+Ht7V0j97tx4waWL1+OCxcugGVZyOXyC35+fgvbt2+/6cyZM7qLFy8+0vHysukWEWl4Gf0M
    bwhst3F65dSYGABRDMPsrVRj4cvrEJXnThNU7LUD3mMebEMG2cJQcM3gwDCMUigUFgGAXl8j
    bM1NwKXTjmIYxtnJyQlhYWEICAiAm5sbRCIR8vPzkZeXJ8nMzPS4efOmR1lZWXsiGsU7df7g
    nWBCk8kEo9GI4uJiXLp0CcXFxbh69Sry8vKsZYXYzc0tNj8/P5CXxW0BuEokEsbLyws+Pj5w
    dnaGm5sbTCYTCgoKkJWVJUpNTXUrKSnpTkRx/Lt4DxX7VNQqvAEcZlmWpk2bVq5Wqx/IxV5e
    Xm568803dQKBgMBV3QfV9UawevVqEBHj6ek5DwBNnjyZdDodZWRkULdu3SpYcoMHDyaNRkOP
    GoWFhTRmzBhzLrshIiJiwfr160V13PnTFVx+ndbb25uWL19OarX6jmc3mUy0a9cuWrRo0R1p
    amZkZGTQpEmTSCKRmFMzxqKe5uMpFFxDxp49e4Z4e3t/y7KsVigUUmhoKEVFRVGrVq0e6IiK
    iqKQkBASCAQkEol03t7eq/r06RMMAMnJybU9zJYArvORH+I9FZUlnz2ANXK5nH7//XdatGgR
    sSxL4KKOwkqb+usA9LNnzyaVSkWDBg0icKHip2wYoHudnZ3pzz//pHfeeae6XNWOlZ7xd9RM
    PZQAXJi3QCwW09ChQ+nYsWN3jSJqtVq6du0aff3119SjRw9ydHS07BNisZiWLFli83srV64k
    Ozs7y7kymYx69+5N33zzDaWlpVUZrbWO2p48eZKGDx9uXj9K/vkFdbA8GACv8sZKTXjYNai+
    jqbWsGzZMk7weXtP52vxyNfXl3bs2FHh95gxY0aN1zWAT1EaN24cqVQqy72Sk5OpXbt2BIAc
    HByUsbGxwwCu1qKOIwvf8A6fKscjFApp4sSJtHnzZkpISLiXd5ALLr1JUN9kwhtvvIFff/2V
    bdeuXXsPD48dQqHQIJFI6LnnnrNkJzwMMjIyaOjQoSQSiUgoFOq8vLy2tW3bts3ly5eZ999/
    v1bGKBKJIBKJWjAMk2b1m2xBxeZ7wJ11CeY9/EMbv9143gC0PreIYZj+lRxvTgB+tXHN12w8
    agSvS1pkl0AgcBcIamTa+IGj99bLZDIaPnw4bdmyhTIzM0mr1ZLRaCSTyUQGg4HKysro6tWr
    9PPPP9PUqVOpVatWJBaLzdHgGwB0zs7OtHXrViotLaXBgwcTAOratSvl5+fToUOHqEmTJsQw
    jNHe3r4AgFYgEFBQUBCNHDmSVq5cSWfOnKGCggLLvQ0GA2k0GkpPT6dNmzbR8OHDzVFoI7i0
    2TqrCRoMoKRp06am8+fPP1QC/9WrV/UtWrTQ8UJmVF0v/ueffx4TJkxwCggI+JNhGFqwYAER
    ES1dupTEYjHZ2dmRVCole3t7eu+99+65fqG8vJz++OMP2rhxI5WWlt73ezp58qSlALZRo0Y3
    u3XrliCXyzFnzpy6eE0MOAq8Mnd3d/r666+rVGxKSkosm92qVauqHF9ubi698MILxBuPF1B1
    AW29gFwuR3BwcJRYLM708PCgX3/9ldLS0ujatWsPdKSlpdGGDRvI2dmZnJycstu3bx9bFaNW
    LcAPwHFnZ2fy8/MzK25vVNrw7QGslcvltGXLFjpx4gQ1adKEwPFfB1cSIKskEgmtWrWKjEYj
    vfLKK+awaeWW1/0AFHTu3JkUCgXNnz/fLBgq58A2ArBOIBBQRESEeSP+uoaUic7ga3FGjBhB
    WVlZ971WFQoFLVu2jJo2bWoRcH379qW8vLwK56lUKougAEARERG0YsUKysnJue97Zmdn05gx
    Y8zr5xqATnUwbwIrCeuaOC6DK+6uUyxYsADffPONXUhIyLdmBX727NkV9j29Xk+TJk26b0PA
    09OTfHx8zL9dlYdcLqfVq1dX+N1/+OEHcnZ2JgDk6el5cPjw4X4dO3Z85JTP1ciFqfdiMMrl
    ctq2bRsREc2cOfN+0pNi6qM8ML/vPn36BPj4+CxjGEbj5eVFR44csaQzP8hBRLR//35ycXEh
    oVCo9fX1/XzAgAF+1veslR+Wo/f2AUdqYl2c/lIlT78dgBU2frsT4HL3zRCCy7e3lU5TuTYh
    HlyPn8rnzrBhrCzmPfgEjpDjR5ZlJTXALCkHVyNgcHd3p6SkJFIqlZZ1aDQaqbCwkLKzs0mh
    UJBKpbLsDQaDga5cuUKLFy+mmJiYCuv81VdfJb1eT2vWrCFnZ2dKSkoig8FA77//foXzQkND
    6c0336QTJ05YnNR6vZ5UKhUpFArKzs6moqKiCvqoUqmkzz//3JwCbOQN+WrTkx5Vrkw0AIcW
    LVqYQkNDH0pIBwYGCqKiorTnz5+X8kri2rpa9KNGjcKePXvg4eERkZub28rNzc1CCZqeng6x
    WIwpU6YgMjISRITevXtXSLOpDjk5OViwYAGys7OxefNmxMTc374XHR2NUaNGYf78+SguLvZN
    TU2d/Pzzz5/Ly8uriz4GoQDGCQQC+/Hjx2PkyJF39J4ww0w7p9frUVZWdT8Wd3d3zJkzB1eu
    XMHBgwcjwIWzZ6OepSZZdiaJBMXFxRoiMgmFQvj4+CAw8OFSrm/dumXu6WEiIr2dnV1dDa8Q
    QI5AIMDgwYOxdu1aiVKpHMNHGv6xUg5Ys+CKiIhA27Zt8dtvvzUFV4yVyp/nCMDf3t4ewcHB
    YFkWwcHBEIlEIr1e3xRcapEeXO5rL5ZlXXr27Al3d3drgcjYMCz6h4WFoXv37khLSzOBS6N6
    WErSRnw0Iyg6Ohpvv/02mjRpct8X8fT0xIQJExAZGYnZs2fj8OHDOHXqFE6fPo3evXtbzjt5
    8iSOHDkCAEhISMCHH36I+Pj4e95TrNG4cWO89dZbuHr1Ko4cORLMC/KzqF1a2Xa4s0CvJoyQ
    DuBSVusE//vf//DLL7/A0dHRq7CwsDUAuLq6onfv3iAibN68GSzLIjMzE9u3b7/vufLZZ5/B
    y8sL06ZNw5kzZ6o8t7i4GMuWLUNCQgLCwsIAAAMGDMCuXbvw3XffQalUxp07d27kmjVrPnzh
    hRfqgp43DFya4F2ZaBiGsaTy3kdKbxg4Xv2LqCepSdbjAYADBw6ku7m5LRMIBAOKi4u9ly5d
    ik2bNj1w003zvCorK4Ojo6PKz89vTWZm5k3re9aiQVSIiumB9uAY7PZXWp9sFZ7/eNzucm9X
    hcdbxDBMBMuyYpPJpCMiFhyZhK2NuPJ9OoCrgxRYyen8Jk2a6AsKClBU9FBb4TMAnnd0dBTM
    mDEDkydPhp2dHUpKSnDkyBH8/fffuHDhAnJycmAymeDq6oqQkBB07twZHTp0QFhYGGbMmIGn
    nnoKH3/8MTZs2AC1Wo39+/fjxo0b6NatGwYNGoSnnnoKOTk52LlzJ4xGI4RCIfr06YNZs2ah
    bdu2EIlEUCgUOHjwIA4cOIC0tDQUFBSAZVl4enqiefPm6NKlC+Li4uDs7AwzQc7cuXPZoqKi
    Z3mHTq2n/i8DQDNmzFAbjUbTw4bc3n33XQ2fWvAtard3RAV07NgRQ4YMkXh5eX3GMAwlJiZa
    QsBXrlyhn3/+mXJzcx80kkIRERHk7u5OR48efaBrXLlyhaKjoy0FTy1btuwDAC+88EJtv6rx
    AMpbtmxJV65cueM5jUYj5eXlUUZGBl28eJG6dOlCAOjdd9+lW7du0c2bN6moqMjmGL///ntz
    Osc51FO6VbOB4+7uHiESiW42adKEjh079tCh54MHD5Kbmxs1atToVnx8fEuzsVoHEAJYZWdn
    RytXrqTnn3/e7On/khcSZkPgR3NKEhHR8uXLzd7+H/jPwXuHk0NDQy1MQdu2bTMXdf+J22QH
    oQDOeXt70+HDh4mIaN68eeb7vmr1bAEADorFYvrf//5Hs2bNIoZhNPycfFj0A6ASi8X00Ucf
    kcn00FsbHT58mFq0aEEAaOLEiZa0vbKyMnrxxRcJADVv3twy5ofF0qVLzSlOBXh07CBVYSEe
    QToOuEafdYZZs2YBAIKCgiaLRCItOOYymjlzJi1dupR8fHxILpeb6XHv6wgNDaWLFy9SeXk5
    DRgw4K7nC4VCWrBgQQVP4t69e83RPZLJZBeioqIiwsPD8cknn9T2q3oBtoswLal5vr6+FBwc
    TNHR0XTw4EEiIkpKSqLg4GAKCQkhT09Pc2pjVcdF3nCol5DJZJDL5REikehmTa8DZ2fnW/Hx
    8S3btWtX6+MSiUSwt7cXCASCrys9l5H36ptrVaS8J9vWGNbgNoFGALicflvnbcftAmEvcIXQ
    ts6zpuZ2AcdkZ/25XiAQzPj6668hlT4Um6q/ObIyYMAAiw549epVGjNmDLm5uZFVamyFQyaT
    UZcuXejnn3+2ZJYUFBTQjBkzSCKRkJ2dHS1fvpx0Oh2dPXuW1Gq1RQdiWZZGjhxJN2/etGRr
    /PTTT9SpUydrwosKB8Mw5OHhQS+++KIlBbyoqIhGjhxpPud0XehVSwDQ66+/rjSZTA/NKbpo
    0SINv0ksRx3lrkdHRwMAfHx8nhKJRNkeHh60ZcuWGqtDOHPmDHl5eVFEREQFxov7gclkokWL
    FpFQKDRPjFWJiYnSnj171uarsoQcp0+fbjO3+99//6W+fftSVFQUtWjRwjK5vb29KTo6mqKi
    omjSpEl3pGiY8zV5KtkicP09GgyGOnCYAXiPZVnTBx98QAcPHjSnJuXySrXZYPjZ2mC4cOEC
    hYeHE+9FiuXP6wGgsEePHpaN9t9//yVfX1+zURjKnzcWQPmgQYOosLCwssEwnT9HAOBtALrO
    nTvTtWvXzExKxeBo9B4GAnD1FxQUFETnzp2rsbX//fffk5OTE/n5+VlSFI4ePUq+vr4kl8vp
    u+++q7F7Xb582UxHS+AaKNbmfvrlIzIYvkMdseQ89dRTaNKkCSIiIkJlMtnJygrwgxgJ1kdk
    ZCRdv36dioqKLI6Vux0tWrSoIEPKyspo3LhxBIAEAoHey8vrzevXrzP9+vWrbbnwdXXPPWTI
    EDpz5gxdu3aNUlNTLcZzQUEBpaSk0LVr12jPnj3UvHnz6sZfAo5wpV5CLpc/coOhLuSCUCiE
    m5sbw7LsfN5IsH62TPBFxrxDaU0VY0gF3/GdYZg4cDTRts67YBWpHACOQc7WedY9G8biTla2
    Mj4y8LAYDaBcJpPRunXrLOk+//d//1eloVD5cHFxocmTJ1NaWpolBfu5554j8H2+zLUuubm5
    9NRTTxEA6tWrl+X81NRUmjhxoiX9EPeQ6jhy5EiLw3vbtm3k5uZmTvmqkpf/UQmLTACGmzdv
    inU63UPFxQwGA6WlpZGJi9nVRFrBfeONN96AQqFAy5Yt/YuKit4wmUyNR4wYUaO81qdPn0Z+
    fj5CQ0Ph5eX1wGHPfv36wd/fH0SEkpKSnmfPno06cuQIBgwYUFuvyxFAhFQqRWxsrM1wcklJ
    Ca5fv2451GqOuEOlUiE9PR3p6em4efMmdDqdTUWcN97k9dmT9ISDANw0mUy6a9euITY2FiNG
    jIBAIHAHV3fgYWt/CQoKQkJCAsDRappbzwYDkPr6+kIu5/qbubm5wdfXF+DCzF68kOkrFovt
    evbsCScnp8rpSOY9JhbAmEaNGokmTZqERo0amVnNinmP+sNAAiAS4NJ7PDw8auxlDhw4EP37
    98fNmzfxxx9/QK/XY8eOHcjMzES/fv0wcODAGruXq6urdRpVS9imPnxUUD/C6xrqYiGUlpai
    efPm9vn5+dPKysoq5JHqdDqUlz9c0/WgoCC4u7vj5s2byMrKuqfvXL58Gdu2bbP8297eHsOG
    DYOrqyuMRqOwpKRkyMCBA/2Tk5Px448/1qZcaFrdCY0aNUKTJk3g6emJxo0bQyzmbEBHR0c0
    btwYjRs3RpMmTWBvb3+3+4SgAbUKg8GA/Px8MplMN3BnmrA3uBRIZ152VKV3+oGjBQUR+YMr
    ZraFJgzDeOF2079Gd9FvgwFMwu2oNvchy2olEknuQ6b2isClOtl5e3sjLi4OAHDs2DFs3779
    nutIlEolVqxYgXfffRe5ublwd3fH9OnT0axZMxw9ehS7d++2XPfw4cPw8fHBzJkzERgYiOzs
    bMyfPx9ff/01VCrVPd3PZDJh27ZtOH78uMUhHhwcbB5PR1TBXvaoDIYLAErOnDkjvnr16kNt
    5Glpacbjx4+LeOuwTlpgnzt3Dp06dbLLzc2dXFpamhAVFYWJEyeipnLIFQoFfvzxR5hMJnTp
    0gUy2YOTuYSHh6NLly4AAK1W61tYWDhgypQpAoOh1uSpPQAnBwcHNG5su99e69at8cMPP2DT
    pk1Yu3atJXozbtw4/Pbbb9i8eTOWLFkCT0/PO74rFoutqei8UE/7MvwHkAlAl5GRAb1ejzFj
    xoAvwu7Ke23u6CwplUrRs2dPyGQyAYDe4FiPwgUCgTgoKMiiIDg7O5sNBkdwoelmAOL9/f3R
    saPN3msMuFD2ZADBTz31FPr16weVSoVbt26BNxaUDzleAT+34enp+bAh7ApwcnLCsGHD4OTk
    hAMHDuD8+fPYu3cvZDIZhg0bZqFtrglIpVLrdeVYy+sn8xEo9iZwBY+17kiaOXMmDhw4gJSU
    lIFFRUUjTCZTjcpTBwcHPP3003BwcMDJkyeRmZl5z8rb1q1bzXS8AIA2bdqgdWuObl6j0bQs
    KCjok5qaihUrVtSmXJBXd8KmTZvwzDPPIDExEWPGjMHZs2cBAOvXr0diYiKGDh2KUaNGWf5e
    DTxRz/oy/BcgEAjAsuwt2K4r7AOujwyLqsknhAD6MAzjBi6CUJWC5UBEgeBYM7tU80gsv7+N
    xe2ItjVUAoFA9ZAMSSJwKUlwd3e36G4XL15EcfH9lY+aTCb89NNPWLJkCcrLyxEVFYURI0ag
    vLwc27Ztg1KpxI4dO1BcXIzhw4ejY8eOUKvVSEpKws8//3zfdTBFRUUw0+06OjrCx8fH/JF/
    Ve/+URkMJwAcTUtLE61cuZLheePv322kVuuXLVumvnTpkgBc6/gjtb0IRo0ahZ07d+LcuXO9
    VSrVaJlMJpw2bRqaNm1aI9fX6/VYu3YtDh8+jGbNmqFXr14PVawkkUjQv39/yGQymEwmxmg0
    9jp+/LhPbm5urbImGI1GmxECALCzs0NMTAy6deuGbt26wc2N688VFhaGTp06oUuXLggNDbVZ
    3ElE0Gq1aECdowBAcVZWFgoKChAeHo7x48dDKpXagaPttMlcExcXh/DwcIALPXcGEGRnZ4ew
    sDDLvJdIJPD39wdue/V7A2jSvn17hISEVLWP9QEwyM/PDxMnToRMJkNubi4KCgpqymAwgotU
    gIhqfC3FxcWhXbt2SE5Oxvfff4+LFy+iXbt2aN++fY3eh4hg5Two5cdVa74XcOkDNYlCcIV6
    tYo5c+Zg5cqVaNOmTVOlUjldp9O51PQ9evfujQEDBqCgoACbN2++r2jFP//8gxMnTlj+7eLi
    gn79+kEkEsFgMNgZDIYBzz77rEtNGqMP/UMWFuLgwYPYuXMn/vrrL+Tncy1YUlJS8Ndff2Hn
    zp04derUvez/5tSLBtQieGpVJcMwtjRlB3Be/rtFf2L43gAhqLqPgphhmAjeOVVdvj0DrpB6
    pC0jxWQy5arV6vzqyFbuARZyD1dXV0gknGNeqVQ+UCG7VqvFqlWrsGvXLrAsi379+iEoKAgn
    TpzA33//jb///hsBAQF49tlnIRaLsX37dqxevbpKXetuBoo5IiGRSODq6mrtHBPWpsGgBLDG
    ZDLlf//998JPP/0UhYWF93WBkpISfP7551i9erW90WjMB7AadzbreKTYtWsX/v33XwwcODCg
    oKBguk6n80xMTLR0e35Y6HQ6bNiwAUuWLIFer8fw4cMRERHx0NeNiYlBaCiX+l1eXh5WVFTU
    5vTp05gwYUJtvDYtAG1paSlycu7eT4lhGEgkEjAMY/Ew3+2dWYXmC8Ax6DSgbgyGgtzcXIsn
    c+jQoeboViS4Asc7QmXe3t7o2rUrGIZx5jfycJlMBk9PT6SkpGDfvn1QKBQIDQ01R/DaAxjs
    4ODA9urVy1Y6AoGLQrwkEAicR4wYgfj4eEHlVjkAACAASURBVAAccxmvZOXyyvHDzusUgIsI
    PmyqSWV4eHhg4MCB0Gq1WLNmDTQaDQYOHFijqU+8E8bSKwTAVXBFqLWF0+AKFGsSB3hnUq1C
    oVBg9OjRjnl5edNKS0tja/r6zs7OmDBhAtzd3bF582bs37//vpXvXbt2VVBaOnfubPEilpWV
    RV+9erXpli1bUEv9ejTgOt7es2FrfvYHULwa5EIdQKvVQqfT5RNRVXpaDO7OkuUKrilj+F3m
    Ryy4SLakGqPRDVzU2d+m8suySoFAUPiQbFJ6ALcAzmNvbv7m6ur6QGx2ANcE8quvvkJ2djaa
    NWuGnj17QqFQ4KuvvsKNGzfQuXNnNG/eHAqFAsuWLYNS+WC+MJZlLUaCXq+3jojkVrVWH2XB
    2xYAnxQXF5d89NFHmD59Os6dO3dPnrlLly5h5syZWLRokaikpKQcXCfHLbW9AI4cOYI5c+bY
    ZWRkTC4sLOzQokULvPrqq5Zc64dBVlYWFi1aZKmPEAgEcHd3f+BJZg0vLy+0adPGrGDLsrKy
    Orz44ouilJSU2nhtagCZer0e//zzz127Jzo4OGD48OGYOHHiPXlTc3JyzGE0HbiiqAbUDZQA
    8tVqNW7cuGFReidNmgRXV1cBgERw3Tuh0WiwZcsWzJs3Dxs3bkSbNm3g7u7OgCt4Dvbw8MCh
    Q4cwePBgDBkyBOPHj0dZWZnZOGgDoFV4eDhCQkLw+++/45dffkFmZqZ1JG4ggPatWrXC2LFj
    LRS+169fh0ajAWomFcYI4AoAY15ensX7WZOIj49H48aNUVRUhMaNG6NDhw41fg+VSoXc3Fzr
    8dRm7n8xOIaUjBq6Xia4QtrC2pz4X331Fb755htcuHBhkFKp/D+j0VjjclSj0WDPnj1YvXo1
    kpKS8CBR+hMnTlgbhwgKCjKnDUKr1TYuKCjoaDAYcPjw4dqSC/f8u6vVauzevRv79+/HqVOn
    7ssPZ1bgGlC74COvSgBVbY5C3hiojjOeAdCfdzpVh7YAEu5yTi/c2fzTArFYnOvn56d1cnJ6
    mGHr+H0Uubm5lnrM6OhouLg8eNDx0KFD2L9/P8RiMfr16wd7e3vs3r0bRITu3bvDzs4Oe/bs
    sdBuPwhcXV0RFRVldiBYO2KTa9mRZIEMXBfXFACmDh063JUBKC0tjbp162auLk/lvy+r7Qd/
    6623AAB9+/bt5eLiojA3xXkYKkWj0UiZmZn0ww8/UO/evUksFpNEIiGpVGpiGIZmzpx5146x
    94p169aZO7uSs7Pz8TZt2njWYqOv6QB0TZs2pUuXLt31Wc0dCO8F33zzjZkWMp1XJuslnnCW
    JIDL3VwvEAjogw8+sDxjaWkpvfDCCxVo4xITE83NYcjJyYmGDBliZkGyMER4eHiQSCQiFxcX
    EolEFBERUYEarkOHDjRkyBBydHQkqVRKgwYNohEjRlg+t7e3p+XLl1dYa3yTLAPubOz2oOgI
    IFsikdDixYvvuSnjvSInJ8fSmbdDhw4PTNFcHVavXk329vYEji+9Lpq3CcGxcOTh4Vhh8vjr
    1CrNdlpaGrp27Ypp06aFBwUFnbpXFpQHOR6WZcnV1ZX++uuvCr//xx9/bKEldXd339a3b99G
    PBFBbeBV3KXDs/VhZ2dHLi4ud21YV+m4DqB1fZULTypLktljLRKJhAzDrH1Ua6IGD5NAIJjf
    sWPHmqhHGwZA7e7uTjt37rTIwalTp97v3K1wzJgxg0wmE129etXS5DMkJITOnTtHBoOBXn75
    5YfqDD9lyhQLleupU6fI39/fzJI0pi7XiCO4Ft1KqVRKv/76a7UC7c8//yS5XE4ikajMxcVl
    bl0YC3v27EH79u3Rr18/mZeX1y8sy9Jzzz1noXPU6XT3pSyYTCZKT0+npUuXUufOnc19BMjZ
    2flWz549fw4JCbkIgCZNmnTPivPdcPbsWQoODiZwnOC32rdv365x48ZISkqqjVfYDMA5gUBA
    s2fPpvLy8hoZU2pqKnXq1Mk86VfjNmdzg8FQ+2DA0XJS9+7d6fjx45a5e/LkSQoJCbHwPgsE
    ApJIJNS9e3dq3ry5TR51uVxOL730Ev36668UFxd3Bx0dy7IkEAgoPDycwsLCSCgUWtYRAHr6
    6acpPz/fYoCeOXOG2rdvb+5CnVhDY3YCsBkAxcTE3JMxfD8oLS2loUOHEgAaOXKkhVayppCS
    kmK9fjaiahaSRw07cFSEp2DVdfUeDyO4moXRqF2GJwBAYmIiFi5cKGzevPlCiURifBTKjEQi
    MUZERGT4+fkVPMx1WJal9957r8Ic+Pvvv8nDw4MAkIODw7WoqKiwJk2aYOPGjbXx+iLANQt8
    lIrg17jdC6bBYKhtocCFfRei/hsM+hpUjJuC6/9B48aNsyjhaWlpNGDAAHpQp0KfPn2opKSE
    8vPz/7+9N4+Pqrrfx587e2afyTbZCWSBhB1BxAUEN0DQoiJUxYIIVUSoWlRQwF1oqRXcKmqr
    tIpWQFqhKlJEVFaBQIDsJJN91sy+3nu/f9x7bicRFEgyye/zy/N6ndcMEHLvPfec896fNzt2
    7FgWAHvVVVexdrudtdvt7Pjx4y/5XLj55pvZqqoqoSP0ihUriHFTBY4977zenm4zOPnQ0xxw
    DYK0RqPxnMw3HRWtxMREnD17NsHhcMwFkAquu/MRxKlAb9myZfD5fGhraxvhcDiuSUxMxJw5
    c6DT6XD69Gls3LgRV1xxBe64445fLFB2uVzYsmUL3n33XRw9ehShUIhVKBT1ycnJXw4cOPCj
    K664ovbMmTN/B7jCk65ISQK4XPGsrCxUV1eDYZgkl8tVFIlEDu7cuTMeU1gO4H2app/buHFj
    QmZmJubOndspS95sNuOZZ54h4fMSAG+A41HuQ8+ABZdiQO/Zs0d81113YeLEiRg8eDAYhhFq
    DViWhUQiwT333IOVK1fC6XTio48+wtGjR9Ha2gqJRIL8/HxMnz4dU6ZMgU6nQ0FBAd577z18
    8803aGpqglQqRUZGBq688krMmTMHTqcTjzzyCI4fP84dYhIJUlNTcejQIbjdbpw8eRKff/45
    SV3zgMvJ7Aq4wDWPvLKkpCT52WefxZo1a5Cdnd01IRupVMgpNRgMF9Ph9oL2z9NPP032Tytv
    cLt6aO0EAXwGLse4P/7XhOlC4AawGcB2xDlsfscdd+Ds2bNoamrKq6urmxEKhdod1hKJBAzD
    XHLXXrFYzCiVytp+/fptmzx58u7PPvtsJYCxl3q/DMPg9OnT8Pv9wn7Mzc1FdnY2LBYLaJpO
    lclkgxmGqdi+fXu85MJfeYWyO5T6Y+CaxvrRh54RClzOuZnX1cS99T5FIlE4ISHBSlEUGIYR
    UokuEdUA/gVg4I4dO0RTp07FjBkzkJubizVr1iAcDuOrr766aKIMi8WCtrY2GAwGgXGSFFY3
    NTVdUlqsSCTCtGnTsGbNGkKjioMHD+KTTz4BTdMsuNT/sni/DwOAhwCcAcAYjUZm8uTJ9Hvv
    vfeL3uZQKMRu2rSJnTZtGvGEMOCK85aC69bX7SBFzSkpKb8DQE+cOJG12Wys2Wxmp06dyl5o
    NKC6upqdP38+Sa1g9Hp9RX5+/svDhw8fPXny5AStVoucnJz8hISEyo6pHZ1FMBgUUkP4Jm4v
    7t+/X3TzzTfHaw0k8od3wGg0skuXLmVLS0vZUCjEhsNh1uPxsHa7nbVarWxzczPb0NDAtrS0
    sFarlXU6nazf72ej0SjrcDjYjz/+mJ04cSIrkUhIyPlXvf3g/P9BhAHgaPICFEWRd8OKRCIh
    giCVStmCggJ2zZo17dJryHutrq5ma2trWY/Hc871W1NTwx4+fJg9evQoazabhf3GMAz7/fff
    s9OnT2dVKpVwLY1GI6ThxYwK3rPZVVACeBFAUCKRsLNnz2br6uouusFiKBRifT4f6/F4WJfL
    xba1tbGtra3sggULWADsAw88wFosFratrY11uVysx+Nh/X7/JaUs1tXVsbNnzybvKABgdU94
    53nIwOUWbwZXg8RcpIeMAdACrmvrTTh/0WOXY8qUKcQZM0cikYTQocHa2rVr2fvuu+9ca/Bn
    h1QqDSUmJh7r16/fqlGjRo146qmnpGPGjElPSko62VlP6ujRo4VOsB2jWGKxmM3MzFwGcIQF
    cYIRwAbe2fOT5lXjxo1jb7rpJvb2229n586dyy5cuJC955572F/96lfs9ddfz44cOZKkpHYc
    NQBu6e1y4f96hIHHtHO93940xGJxk8FguPxCnNgXiEG8U5sdOnQo+9///ldIXz958iQ7ceLE
    i77HQYMGsaWlpWw4HBZ0ubvvvpsNBoPssWPHhCj+hQ6Kotjp06ezlZWVwnlw6NAh9uqrryY/
    cwzAkJ97yO6IMGQDWAlgdkJCgvL666+PLFiwgLrqqqvEF1JcIpPJcPfdd2P69On44Ycf8M47
    71BffPFFvs/newHAMF7Y1XXnalcoFPj73/8uXbly5UCLxSIaOnQoDAYDPvnkE+zatQuXX345
    Fi5cKFBonQtHjx7FihUrsGvXLkgkEpvJZPpn//7937733ntLy8vLo4WFhfjPf/4Dk8k0NhKJ
    pKvVahQVFXXZM8jlchQVFUEkEoFhGEgkkhFr165VuVwuT5wODTu4rrtVDofjgQ0bNgz44osv
    MH78eEgkEjQ1NcFqtcLn8yEUCpF7hEKhgFarFZr3VFdXY+/evXC5XBEA34JrM7/7/wveFj4U
    2V0eEvZSPZldiFYALpVKpSA0b83NzRCJREhKSsKwYcMwfvx4FBYWtvOWi8ViGAwGGAyGn12/
    ubm5yM3NPde8Yty4cdi4cSN27dqFXbt2oaqqCm1tbQIla0NDAw4fPkzWYVdWKPsBvAIgKRqN
    3vvPf/5TFggE8Mwzz2Do0KHtvLs+nw8ejwc2mw1ms1nwGDkcDtjtdrhcLgSDQYRCIQSDQQSD
    QdTU1AAAtm3bhpKSEiQkJEChUEAul0OpVMJgMAhR2MTERKSkpCAjIwM6nQ5qtRpKpbJd1LOk
    pASrV6/G559/jmg0GgJXdLwBPVPUlg6u7mA+//2SthW4qPPt4Bom/ZV3TDR0983zEWbxTTfd
    NJSmaYHSrbi4GOvXr8fEiRPx+eef47PPPrsg6meRSMSoVKpyk8n0Xv/+/bdMmDChrqmpifnr
    X/8KrVY70O12d1qTaW5uhsViIX1NkJCQgKKiIlAUBZqmAWDE4sWLE2pqagJxWgMOAKvApT48
    CK6TOwUAkydPxksvvQSNRgOZTCZE3KPRKEKhECKRCM6ePYu7774bZ86cIb8vDI4p6w8A/tvn
    4+8VsICLXip/7nxXqVTCIPLAYDBArVZDJpNBJpNBKpW2WwfhcBjhcBiRSAShUAhutxsOhwMO
    hwMulwt+vx9erxc+n4+s73OCpmmb2+1uIfKkC3AGwLMA1p84cSJn0aJFWL16NW699VYMHjwY
    f/zjH7F48WJ8//33kEql0Ol0gp6TlJQElUoFpVIpfCqVSqSlpSE5ObkdgyTpGZGWloZly5ah
    paUFfr8fPp9P+PT5fLBarWhtbYXb7Ybb7UYkEsG1116Ll19+GXl5eYhGo/jPf/6DVatW4dix
    YwCXLfAcuB5qcTMY8gG8DOCWjIwMLFmyJDxv3jxpYmLiRb8RrVaLm266CWPGjMGmTZvw6quv
    Ks+ePXsPuK5+T4CvTO8OJCUlwWQyUUqlUkpRFDQaDSiKwrFjx8CyLBYsWCBUl58Lhw4dwpIl
    S3Dw4EEkJCT8mJWV9dLw4cN3fvHFF4EFCxaAoihMmzYNS5Ys0W3ZsuVX0WhUmZ+f307h6BLL
    LTsbCoUCfr8fbrc76ejRo9JL4evtpNHwZwAHaJp+tKysbGpZWZksVnGUy+VQKBSQSCQIh8MI
    BoMIh8Mdw/o2cLmpb4ELd/ZqrFu3Do8++ijy8vL6OZ3OBK/Xi82bN+PgwYOXnK4gEolQXV1N
    6DwVbrc74+TJkyfXrl2LZcuW9dSjVgI44vP5popEIrz44ouCYUDeaUcQukSaphGJREDTtJDG
    wTAMWJYFRVEQiUSkERDEYjFkMhlEIhEoihIO+JSUFNx11124/fbb4Xa7EQqFIJVK0dLSggce
    eAC8N/q/6HwPho6w8k4RXzQanb99+3Z1U1MTnnrqKRQVFaG8vByHDx/GqVOnUFNTg+bmZrjd
    bgQCgXMKMfK8EokEMpkMBoMBPp8PP/74I6LRqDAvHdeDTCZDQkICtFotMjIy0K9fPxQWFmLk
    yJEoKChAWVkZnn/+eRw5cgQsy3oB/AXAWnS+6/WleuBeAMeA0lXN4tIA/J73iq0AcLI7H0Cp
    VKKkpESsVquTYx1cCxcuxMSJE3HixAmsW7eO9P74WUilUrdOp/swKyvrzdmzZ5fW1NQwTz75
    JO6//368+eabkscee2wqwzBJnb1nt9uNs2fPYtSoUcK6ycvLg1QqRTgcht/vz963b5/c7/cH
    4rgWHLzRehAcqcnNCoVCPnHiRMGwIWcFSWkkzGdKpRLjxo0jBoOVX9N/iYfB2IWOJOG5srKy
    OpWqS1EUfD4fzGZzr3Ak8Wd0LcuyZxiGSYt1EOXk5GDQoEHIzc1FVlYWsrOzkZGRAa1WC7lc
    LozYs/58YFlW6PVEnC6BQAAOhwP19fWoq6uD2WxGdXU1Kioq0NjYCK/XK5yjIpHoR4ZhLOR3
    dRF2gGtO+PyZM2dylixZgrq6Otx///0YMWKE0GAtOzsbgwcPRlZWFnQ6HRISEkgPC5yriVw0
    GhVkqVgsBsuySE1Nxf333/+Tn4tEIohEIggEAnC5XKivr0dpaSnq6+sxe/ZsDBo0CD6fDx98
    8AFeeukl1NfXg987K8GlVbHxMhjSeYFw68CBA/Hyyy/TU6ZMkUql0k6Zb0ajEYsWLUJ+fj4e
    f/xxcWlp6S28IrCkuw6JlJQUTJo0KULTdAMA2O12RCIRTJ06FVlZWbjpppuEnyWWXVJSEiiK
    QmVlJZ544gkcOnQIRqNxj1ar/X1ZWdmPDz74IFwuFyiKwttvv40FCxZg/Pjx1zmdzmsoisLk
    yZO7LBc61iOmVCrh9/vh9/uVWq1WTdO0Y+rUqdixY0e8zhAawPfgit+vys/PT545cyZSUlKg
    Vquh1Wqh0+kgk8ng8/ngcrng8Xjg9Xpx+vRpbN68GT6frxzA6wAae7sweO2117Bo0SJ8+OGH
    V1ZVVa2KRCJJkUgEr776aqc9GUSAUhRlqK+vf3rYsGH2ZcuWHf7ggw8wZ86cnnhcG4C3WZYd
    u3Xr1sQpU6YI6XyhUAitra2w2+2w2+1obGxES0sL2tra4Ha7BU9IJBJBNBoFTdOCciwWiyGR
    SIRPmUwGtVoNtVoNnU4Hg8GA9PR0pKWlwWg0Ijk5GUajEWKxGDRN49133yVUjMcBfIDuqX1q
    ARftbGVZdsmhQ4dMv/3tb2E0GlFfXy9wWut0OiQlJSEzMxMpKSlIS0uDRqMRvEgJCQmCoFQo
    FFAoFJDJZMKhT4QhEYj8XobH4xG8x06nE2fOnMGBAwfAsiw0Gg3S09PhcrkIrWYTgFd5xaon
    6hZG8B7giTh/M6ZLhZQ3QnS88nm4ux7CaDRi1qxZ9LPPPmuPPWOLiopA0zQ2btzYrl8CUQw7
    QqVSWY1G48uFhYUbv/76a8/777+PoUOHgmVZFBUVwWw2F9lstqkxDfYuGYFAAC0tLT9xiKlU
    KoTDYbjdblV1dbWMYRgUFxcLnV/jAAZc/4yNAK4xGAzJNE3jww8/RFNTE7xeL4LBIKLRqGAY
    63Q6ZGRkIDU1FXK5HKFQqIJ3IvV6uQBwzIvPP/88BgwYUOj1elVGoxHr1q3DZZdd9rPe8J+D
    WCzGgQMHcP/99yMUCinb2tpyT506dXLZsmVYu3Zt3J9Rp9PB6XS2KhSK90Kh0GgAmttvvx3z
    5s3DwIEDkZSUhISEhE7LQoqiIJFIIJFI2vXm6d+/v9DRnKZpeL1eNDc348SJE3j33Xfx1Vdf
    QSwW18tksndZlg3IZLKL7sj8C7rOZnB1c8+2tLQMffbZZ2E2m7Fs2TKMGTMGo0aNamcUkEa0
    Xq9XiIz4/X7BmRaJROD3+1FbWwuAo+Pft2+fEIUj0WciOxQKheBESk1NRUFBASZNmgSapiEW
    i2G1WrF+/Xq8/vrrpGnbGd5Y+AwXQLHdVQaDmj+sb83Ly2NeeeUV9sYbb5R2UagHEokEkydP
    hlwuZxcvXsyeOXNmGn9IPAW++2pXwul0IikpiU1PTz8jlUpDx44dk1ssFtxyyy24+eabhRce
    Doexfv161NbWCuxDr776Kr799lvo9fp9eXl5Dx89erR06tSpIMXGLMti7NixmDp1asHRo0eX
    +ny+xGHDhmHmzJldVvAcI5gE74VUKjWkp6dnRCIR84U0SOtiiAGMpSjKOGvWLKxcufKCCjob
    Gxtx5swZ/PDDD/ngmJd6tWB4/fXX8eCDD2LUqFFXVlVV/cnj8YwZPHgwRowYAZFI1GlPBinQ
    OnbsGE6fPj2urq7u1aFDhy6ZM2fO4X/+85+44447euKxdwPYZrfb52/YsAE0TaOmpgYlJSU4
    e/Ysmpub4XQ6EQgEEA6HzzkHJGpABjGMztdRmUQcVCoVEhMTkZGRgYKCAowaNQoKhQLvv/8+
    IpGIn1dGKrrx2V0A/gQu2vl4U1PT6KamJlF+fj5mzpyJYcOGIT8/H+np6UhMTIRKpYJMJoNY
    LBaiJxetZfHRGeJd8/v9cDgcaG5uRnV1NUpLS7Fnzx6UlpYSAbYfHJvVV+DSN+KN/gBeAjCp
    O5224ChiXwbXqKlb3rnD4QBFUXRGRsYpsVgcjUajEpfLhQ8//BAHDhzAZ599JvzspEmTMHDg
    QHzwwQfteiio1WpLTk7O8tzc3E1+vz8MQIgsz5o1CzNmzNBu2bJlkcfjye+Ke45EIj/p4aBW
    q0lTRADQZmRkpEYiEUtOTk48DQYiF64AYLRarVi2bJmQckLTdLu9HxuBE4lEpLNtPji+/l5v
    MKxatQorV66kRowYcZ3Van2Gpmnj8OHDcdlll7WLqlzqOx42bBj27duna2hoWDVkyBDnmjVr
    9imVSqxevTquz7lw4UJs2LABEolkRzQa3RmNRu+MRCIYPXr0T1JQyVkWm2YUiUTafSd/pmla
    iDSRVCUyyJ/JZ2xkWqfTCePtt98GAEYqlf7DYDAcCofD3dFPJwqucNgC4Gmv13v9W2+9JSGk
    E5mZmWhpaUFzczPOnj2LqqoqmM1mOJ1OIX3I6/UKspJE3kmj0D179uDQoUNCqlas44l86nQ6
    5OTkIC8vD9nZ2TCZTDCZTHA4HHjxxRexefNmBAKBKLjo+wsAvuMN+F/WxbtokqYDmGc0GkWr
    Vq2K3njjjbKuMhZilYqJEydSq1atoh966CGRzWb7DbjOoe939Ruvra1FcXExpFLp4fr6+oqS
    kpIhn3zyCRYvXiyERgHg8OHDePPNNzFmzBiIRCJ8+eWX+Oijj6BWq2tyc3NXHjx4sHTy5MmI
    ZSYaNmwYVCpVyvHjx5+yWCzj9Ho9HnnkEQwePLjLN2+sYBCJRJpIJJJC0/QldwbsBDIATE5P
    TxdPnjz5gtlf0tPTMWXKFBw4cCCFYZip4HJVw71RILzyyit48MEHMXLkyCurq6tfcbvdo8eN
    G4d169ZhxIgRXXqto0eP4pFHHsHBgwevMJvNrw4dOvThO+6448imTZtwzz33xPvRfQDeATBh
    7969eQcPHkQgEADLskhISIDRaERubi6SkpKQnJwsRL1iD7jYT5lMJijCgUBA8KrHfnc6nbBa
    rbBarbBYLKirq8OePXsgkUggl8vh8/kAYA+ArfiFEGsXIMx7Z8rApd/dMGXKFLz44ovn6krd
    aYhEIsJ33q7ep6ioCJMmTUIoFMJzzz1HDIY94MgnyntoWxjApQpdH6frTeCdSL9DN6Rd7d69
    G4WFhQBwwuFwtESj0cxwOIy//e1vsTUByMvLwwsvvACxWIxt27YJCrtcLvenp6f/8cYbb9zU
    2toa/vzzz9sZGFlZWaItW7bcZTabZ0UikS7zHvH7oZ0jicgFqVSqVyqV6TRNn7xUL3cnkAmO
    QVEcjUbR1tbWTt6LxWLBScIwjKBYxiAFXHTpm94qFwDgqaeewrJly6hRo0ZdV11d/SePxzN4
    9OjRWL58OXQ6Xac93ElJSVixYgWWLVuGkpKSkfX19a8OGzbs4ZKSku8kEkm8OnkDAF5++WWo
    1Wr4/f42qVT6F4Zhrt61a1f6v/71L0ybNg11dXWorq5GXV0dmpub0dTUBLvdLtSodDQUyHcS
    eY41FDqOhIQEJCcnIyMjAxkZGYLSnJ6ejo8//pjUD5xQq9XvezyecFJSUrc04OSV7/0AFgBY
    FI1G5//73/9OKisrg8FgQGtrKxwOBwKBwC82tv2JsPnpHjivnJBKpVAqlUhOTkZqaip8Ph9O
    nDiBaDTqBPA2uKyNi2qi2RUGQy6Ah0QikW7evHmB22+/XUF1tbUQc4jceuut4hMnToTXrFmj
    oWl6Ebh0l6quvM62bdswY8YMLFy4sObBBx/cZDabn1+zZo0sHA7j7rvvhk6ng9lsxrp16+Bw
    OHDLLbfA5/Nh48aNcLvdgfz8/Fffeeedb7dv397Owh8yZAh0Ol1SRUXFapvNdqdcLhc98MAD
    uO2227pl8yqVSqEwm2EYucvlUrMsS7rfxhNXARg8cuRIpKSkoKKiAk1NTULRJ8ntJiE2pVKJ
    pKQkZGRkYMyYMTCZTGhqarqO91aW9TaB8Ic//AFLly7FyJEjr6qurn7F6/VeNn78ePzpT3/C
    yJEju/x6V1xxBTZs2IClS5di//79V5jN5vVDhw59+J577jnyzjvvYP78+fGegqMAPqVpeplU
    KhXdcsstGDlyJIqLi4WCXGIoxNYhXMwxERttoGlaSF9ra2uD2WzGqVOnsH//fkJf18YbMZY4
    PT8LLrT7HYAbGhoaujxaeDFnJJ+XGi58CAAAIABJREFUCgAHZDJZOYlI9ADuBDALHMV2PCAC
    Vwx9GMBrXW0sFhUVweVywWQylXq93t1NTU33Ei8ggVwux4IFCzB69Gi89dZbsFqtnCtdLGZN
    JtPH48aN21hVVRXevn07/vGPfwj7edCgQaKtW7fObG1tXeH3+7Vded8ej6ddHnSsI4llWaXH
    49GzLPuT1KU44GoAg3U6Hfr164f09HSkp6fDYDAIxZ9SqRTBYFAoZrXZbGhqakJjYyPq6uoQ
    DAYnAcgDcLo3GgurV6/GvHnzqKuvvvp63lgoBoCmpiYsX768K/Pn0draCgBwuVwjzGbz+hEj
    Rjx87Nix78RiMZ588sm4PbPX60VRUREUCsWB2traz9va2hY8++yzePvtt9HQ0ACLxYJQKNSl
    z34uhVmpVMJkMiErKwtVVVUIh8PhpKSkD+67777ynTt3oqSkpLunogFcIbGBZdmFFRUVcXsH
    DMMIqaxOpxMdrv0fcEx/F22pdoXBcCeA0cOHDw8vXLhQolAoqO6cCLlcTs2fP1/01VdfBY8c
    OTISwGwAz3e1cFi0aBFWrVrFDB069G/RaHRoY2Pjr5955hnRli1bkJaWhoqKClRVVeGWW27B
    lClTsH37duzduxcajWZvcnLy5rlz5zLHjh0TDIZRo0YhJSUl6cSJEyttNts8pVIpe/jhh/HY
    Y49BpVJ126YhShnDMOJwOCxramrC4sWLsWHDhnitXTW4CJSqrKwMc+bMEXLZiYVNcnVJ2Fkq
    lUKtViMxMRFGo5EYOPkAru1tBsOjjz6KFStW4J///OfgmpqaP7rd7stSUlJw4403wmaz4Ysv
    vug2xfDGG29EWVkZ7Hb7FQD+nJ+fP+ell16q6YFpiPDK8sL09HTDM888g7y8vIv6BbE1DGQd
    xK7fWANDJBJBr9dDr9cjJycHw4YNw7Rp0/D1119j7969CAaD9QBO9MA81ALw19fXK10uV2za
    h4BQKIRwONyubiP2k3jUSNoVSV+Krekg/6ZQKH5idPl8Ppw9exYAQgDMcSY5iMUgAL9F/Jto
    JYBjYvoGXVwEvXPnTixcuBAffvhhYOjQoW+EQqHL7HZ7caziI5fLYTQaceLECXz00UeCB1Gp
    VFaYTKbXNm/e3LZkyRLhvV177bUYNmyYaMeOHTNaW1vX+Hy+jK6ekI4kElKpVDAeGIaR+P1+
    eTAYREZGRjzfkwbANIVCoXzyyScxe/Zs6HQ6aDSanzW2I5EI3G43bDYbXnjhBWzatCkPXG1M
    rzMYnnvuOdxwww2iGTNm3FBVVbXO6/UWqdVqSCQSeL1eEgXsUnmv1+vJHI2oq6tbP3LkyIeP
    Hj36nVgsjis5hk6nw/79+wNpaWlfUBQ1q6amRksY4HgDWmBIInWMCQkJ5xwKhQJSqVRgRiLR
    ZuJsJAaly+USUnp8Ph+8Xi+qqqpQVVVFrtnMMMzuNWvWsM8880w8DAaAY9U7g3P0pRCLxVAq
    le3q80i0/XxzQNP0eeeAFDq7XC6hRvA8538ZLjGVv7MGQwaAW6RSqWTevHmBvLy8hHi8gdzc
    XMn8+fNDJ06ckIfD4engChu7lGp10qRJmD9/Pj777DNrYWHhCpqm3Xa7/Y4jR44kUxQl5MqN
    Hz8ejY2NeOONNxCJROzp6elvf/vtt5YJEyYIQuGKK65ARkaG8dChQ8tbWlrmq9Vq+aOPPoql
    S5dCo4lPI2uapimFQqECOM7tOBoMQ3hPEqqrq2E2m2E0GpGSkiJEEwiNGvEikQOgpaUFFRUV
    RNjJwPE7f4yuZ73plDDmU2iG+ny+IQCX67x27dpu9aDEenMAwO12Dw+Hw9ng+Mh7AmYALofD
    YbDb7cjLyxPy691uN1wuF1paWoR6BjKIAk0OwUgkIqTbkAJgohyTwzMxMREmkwl6vR46nQ5G
    oxEKhQINDQ0k/aIOgLMH5qAagNtisSjr6urAsqzgCW1sbERrayusVqtgLMcOwg5GDAeSg0vm
    IlZwkKK25ORkoZA6NzcXGRkZsFqtxFPs6cG1IALwa/wCp3c3ogjAXeDSobo0tPKXv/wFzz33
    HJ566qnDl1122ZM0Tb/ocrmKGYah+H2I5cuXQ6vVCoWKEokkotPpNs2YMaNk0KBBWLNmDQBg
    5syZmDhxomjDhg2/am5uXuvz+bK7YzLOVwtE5AJFUQq73Y4HHngAe/fujadcuGrQoEG4/fbb
    2xF+MAwjUGjSNC3kp5P9QCiFZ82ahR07dsgcDsc0cMWmtt4iF1asWIGnn34aX3755ajKysq1
    Ho+niNCxG43GbpMNFEXBarXizTffxLFjx0bU1dX9qbi4+C6WZSvj+fz79++HRCKBxWKpYBjG
    DkBrMplQUFCAoqIiFBQUCCxJycnJgkJMCpnJkEql7ZwiLMsiGo3+ZEQiEQSDQYFgo76+HuXl
    5SgtLcXp06dhtVrBsmy93++3KhQKPP300/Gcjhr+PNYbjUYUFhaiuLgYgwYNQlZWljAHSqWy
    3XPHzkOsEU0YooizNXaEQiE4HA40NTWhvr4elZWVOHXqFE6dOoXGxkZiwFyyrtxZg+EqAIPz
    8vLCkydPlsTxBVCTJ0+W5ufnB06dOlUEruBtU1df5J133sFf/vIXLFy40Hz77bc/fubMmX85
    nc7LExMT2cLCwrT9+/ffunHjxrRNmzbh5MmTbFpa2paxY8d+NWzYMHz66acAgAkTJqCgoEC/
    a9euJ5qamhZqNJqEJ554Ag899FC3RRbOJzQkEokSQHfl7Z1PcbgZQJrBYMC1116LCRMmYMSI
    EcjMzGx3SIhEonYeVo/Hg6qqKhw8eBC7d+/G4cOHEYlExgC4DFwBZ69AzMFPsSwLpVKJvLw8
    yGSybjcYKIpCOBxGVVUVAoEAuoJVpROwA2hyOBz9PvzwQ+zcuRMVFRWora1Fa2srPB6PYBCQ
    YreLmZ9YqlXCmqLRaARlecCAATh27BhJvalDz7ABtQKwt7S0mJYvX45gMIja2lrBQCBeXuJw
    ONcgxlEsZaDD4fhJfi+ZO4qiIJfLodVqkZ6eDo1Gg4aGBvDP39RDayEHwFTELxXpJ8sFwBRw
    3ay7XFF6+umn0djYyN522207tm7d2tjS0jLHarWO0el0ooyMDGNVVVWuxWKRkPej1+sP9u/f
    /+/vv/8+feeddwIAFi9ejLvvvlu0aNGiWxoaGtZ4vd7cOJxRPwHDMJRYLE4gkYc4Qcw7f0xD
    hgwJOZ1O2eeff06Vl5ejoaFBqFny+/2IRqOCY4kUdA4YMAD5+fnQarWRvLw89tChQ6MBjAaX
    atErQIpUA4HAYK/Xm5+Tk4OXX34Z48ePj8v1s7KycN9996G5ubmIpumM7tgHvyj8OSXXwTBM
    I0VRuXPnzsXSpUuh0+l+tofVL8kBclaeC/3798fo0aMFZ57T6cSKFSvw7rvvgmGYGr/f3xPO
    RiKP9NOnT8cLL7wgOLkudQ6IIXEu9OvXT0iFjkajcLvd+POf/4znn38eLMt2ypHUGSVfBGAc
    APWECROCOTk58ni+gczMTOnEiRN9p06dMoJjWvgIF0ALdbFYuHAhAGDp0qXeqqqqL6+++upd
    /fv3Z+vr6+UzZ87cc+LEibnBYFBrMBh+NJlMr2zatMn34osv4tNPP8XVV1+NpKQk+e7dux9q
    bGx8UKvVKlesWIHf/va33VIQ+UvrTCwWq1mWpV555RU2TtdMAddciXK73Th8+DBcLhfKy8uR
    n5+PxMREGAwGaDQaIVe1ra0NTqcTTU1NKCsrE1K/eGVYzxuHu9D9xawXDYZhkJeXh7/97W/I
    zMyMi8FgNpsxd+5cnDx5sqcf3wngbDgcHvfGG28IOfNyuRwGgwFpaWlITU1FamoqNBrNecPP
    pB9HRw88GYQqlBQ919fX4/vvv49lWIqC6wbO9NAc1AWDweL//ve/UKvVSEpKQnFxMdLS0pCT
    k4Pc3FyYTCYkJCQIzak6fkokEsHDSvJQyXdiRLS0tKC2tha1tbXCfNTU1MDj8ZB114CeibKA
    36OFPbwe88EVQXeLovTWW2/hm2++YZYvX3703nvvPfXvf/9bPXz4cGrmzJlpGzZseOjMmTMz
    o9Foglqtru7Xr98fvv3227otW7bgtttuw6JFi7B+/XpceeWVN9TU1KxxuVwD4uTUOOcxIpVK
    1SzL4s9//nO83k0ygLEARF9++aVoz549glH9S04PYiCrVCro9XqKd35pwUWxv+yhff+LjqSk
    pCTk5uYKfWe6W1HPzc1FYmIiGhsbe6p+iaTDOHgF9apIJIKUlJR28pKmaSGdKPasI4NEnsPh
    sOAoImclGYSWmqQ1EQekTCaD0WgUovAJCQmNgUAgpNfr2xXYx8mZ1gwgJxgMIiUlJTYlEDRN
    IxgMwuVyCc/acQ5IY0+JRHLOOSDzoNVqQdLeSAqrwWCA3+8na9IJjhI87gZDIoAiqVSKK6+8
    UioWi6l4vgGRSCQaP358wltvvcVGIpEi/n5au+t6V111lbDOAeC+++4LDhky5FMAX9jtdqlW
    q/UcPHgwMnPmTCxfvhyDBw/Ghx9+iBEjRsxwuVxL5XK5avny5XjwwQcv2bLsAgVTCUBks9ni
    dYK4wNWX7Kdpelx9ff2A+vp6/e7du+VisVhCGF9IpIF4oPmQG8swTARAgH+vpeA6PffqLs9S
    qRTJyclITk6Oy/V8Pl88PYM/hyAvHDBgwABMnz4d/fv3R3Z2NlJTU5GUlASj0Sh07exIpfpz
    gpeMWIo5m80Gm82GlpYW1NXV4cyZM9i6dSusVmu4MwdiJ0EDiFAUhblz5+Kee+5BcnIykpKS
    oNVqhbSKLrsYH5EjOd02mw0ff/wx3nrrLTAME+oOB8oFeo/HIP61Cx2h4O/jfXQTg86ECRMA
    AO+//34IQGjIkCF44oknbMOGDVum1+v/4fV6Venp6bUajaZ83LhxuO2223DnnXdi06ZN2Lp1
    6xCPx/OMz+fL7+6J+LmUJAAQi8VqACK73R4vZbsNXFfc71tbW68EMAAco5Ycv9DUj2VZNhgM
    hoPBYMButxO5sBccRWSvcyIRBb6mpgZLliyJi6OQoih4vV7U1dX1GPkCAGg0GnzxxRehSZMm
    2YLBICoqKlBZWYmmpiZUVlYKKcrNzc1oa2sTes7EKsqkH0FsQ89YBZl8KhQK6HQ6mEwmZGZm
    Ij8/H4WFhVCr1aSpXUStVrcEAgGkp6fH22BwgWMjGltfX4/Tp0/D4/EI81FfXy/Qj8fOQexc
    dJwDwgjY0WDQ6XRITU1FZmYm8vLyMHDgQKSnp6OsTCj9bEInWhF01mDI1ev1dF5eHoMOBR3x
    QP/+/aN6vT5gtVpzACR1p8HQEe+++y74A6odyfUnn3wCAGhoaMCkSZMGer3ex6LRaOKcOXMw
    f/78HjMW+MNWWVJSQnVho5JfQgDA17ySnwSORi9HJBJNpmn6Hp1OlzBhwgQcPHgQjY2NuPzy
    y2mKovwHDx7UsCxbC64pz4+8t7QBHIVnrwJRACmK6g3CqifvQQsgm6IozJkzB08++eRFN+ch
    HY3PV+xM5loul0Ov17crrPb7/bBardi6dasCHJtWTyAVQIFGo8HMmTMFhfLnlH2SokWePaY5
    nzCIYR2b20rmQywWQ6FQCJ47mqaxefNmOByOAQBMiH9edyK4gufegCL+3IlLata6desAALt3
    73bxzg2UlZXFCmt8//33SE1N1Tc1NT3h9/vHxKPO6UIcSRaLReTxeOJlMATBUf5+w6+XDABZ
    AG4CMFetVktuuOEGHDlyRGY2m3HNNddAoVBg9+7doGm6Dhwd5BFw/RcaweVl91pQFAWHw9Gu
    T0e85VNPwOPx4Oqrr5YDMALA3r17ceutt6K1tRVOp/O8kRZCqUsG6b0R6433+Xw/m9ZKURQM
    BoMQZaEoShIIBJJZlu2JOVGAayyJkpIS3HnnnUJD00udAxKF/6U5IA3cYhjQ1LxhHneDQQ0g
    ISkpSZyUlNQjZmxycrIsNTVVZLVaFeBYF3oFHn74YcyfP188Y8aMe8Lh8IgxY8bgkUcegU6n
    69H7YhhGUVpaKurYyOdSnCbk/fPfyfv38QYUfQ5F1soPL8Mwd0ulUtn8+fPx29/+FrNmzUJj
    YyOuv/560axZs6QLFy7E999/rweXi3z4FyxiOf/uFeCiPwx/fS9vsHQpfv3rXyMajUpPnjyZ
    5vP5pNu2bUNeXh4TDAZTAVDhcBgNDQ2CAtjdnqvGxkYS+hXJZLLM5OTkfjRNiwFAIpGE+vfv
    3yyRSOiCggKsX7++O28nA0CRXq/H2LFjhQZshK3C7XbDarXC6XS2Y7OILdzqyLdN8jTlcjk0
    Go1Az0qKHtVqNTQajZDjPG7cOHz22WcihmFG8AaMO85bbDCAjAEDBqC4uBgMw8DhcKCtrQ1W
    qxUNDQ1oaWkRisDb2toEamEyD8SjRsLJZB5IzwW9Xg+tViv0XyAdpPV6PRITE1FYWIjc3Fw4
    HI5UcIWlpXGeAyOA7F5yFGfw99PUG27mtddew0MPPYSBAwdOD4fD0+NlLFzAdZSHDh2iSOpG
    J+WCJkYuiDvIhcg55IKNHwEA8wCI77zzTscTTzyhv+uuu2A2m3H11Vdj9uzZuP/++7F//34D
    //uP4efrlBL4M0AeIxOIXOh2A4MY9bwjqSetQhYA001M9xeic5BzSGDwUSqVSElJgU6nQ1pa
    GrKyspCcnNyuY/G5GIKIkkwYgch3MiwWC8xms3DGut3u2J5TlNfrLaYoqifkQjb4FE2v14uy
    sjKhZwQ5x7OysgQimI5zQL7LZDJEIpHzzkEoFILNZhMiFkTOVlZWxp4B/cBF9OovaV130qMo
    5TvN9chqVKlUcq1WC3BhTC16Cb777juUlJTk2Wy2XykUCur+++9HUVFR/E+KDoKCYRhFXV3d
    pQoGES+Ax4IL9efy3jsJOAYjFlwqiBlcNKCe/7SCSwlIADAKwHyRSDRu2rRpoiVLlqClpQUW
    C0eXX1FRQfXr10+xatUqLF682FBeXr6C91ZuBpeXHuUNlTTeI5XJjyxwIW2aF0ohXkkoB9en
    owRdlM/NK3b9XC7X621tbbksyzL8XGtZllVUVFTg3nvvjVuaUCQSQV1dHQAowuHwyy0tLT5O
    TlEimUx2Mj09faFUKrXGQWAUAzCZTCaEw2F88sknOH78OGpqagTubVL4THJVL0ZhEovFQq6q
    UqkUhA0JPw8fPhwajQYqlQoej2cwvy7iSbUo4te3VqVS4V//+hcqKytRWlqK+vp6QYDF5JL+
    rEfyQuaGsCVptVpkZmaiuLgY+fn5JIqpBjASwCfoYqagXzqWeeO9N0Dw7PUG7N27F4sWLUrZ
    vn37byKRiLq33BfDMIqamhpRxwZvF7HuM8HVEY4BF91L5GUy8WTGygUyWni5oABHZHGfSCQa
    c9NNN7lWr16trKurU/DF+zh+/DgeffRRPPPMM1i6dKnu9OnTy3gF7CNw+fE0b6ik88pZBn9P
    2eDq3iLnkAvfATiOLmLcu/fee0HTtPzEiRPZXq9XvmXLFgwcOJDx+XyZDMOI0tPTMXv2bGi1
    2rjUtjmdTmzevBkWi0Usl8sHGAwGa3JyMnEk+QsKCuqkUilNGsh1I/LBR56vv/56XHfddcjM
    zERWVhZSU1OF8+tcFNEXq+v4/X54PB6Bkc9sNqO+vh7//ve/ceDAAcGh0wMGw3Bw0WeMGzcO
    06dPF+bAZDJBq9VCo9FAqVR2eg6CwSA8Hg/cbjcsFgvq6+vR0NCAXbt2kR5Fibxc+CbeBoMI
    AEXC5T2BDtemesPh++STT+Kll15CZmbmFJfLlT9y5EhMnDixR+6lY8MmmqYlbW1t1CU0bksE
    cDc4qsRBBoNBk5mZKeSlMwyDtrY21NfXw2KxwOv10gzDBHmvkYc/qFUAjEajUTVr1iw89thj
    MJlMeO+99wjdFw4cOICSkhJcd911ePPNN/HCCy9o9u3bN4enznXyAkbN/64EiUQiNxgMFGn4
    o1KpQNq9WywWNDQ0MMFg0AbgAIC/gEuP6lQ+Mx8SVAQCgQGBQKC/Xq8XjAPeeI17J20+ckXx
    QhLhcBgulwuRSMQTiUQkcTAWRABGANCYzWYsWLAAdrsdfv//HHkymQxarRZGo1Hg3jYYDD8p
    /pVKpQI9HMnfDAQCAj2r1+uFx+NBa2srTp8+LQhIYizwa9vEC4d4GgwG3mDA4cOHcfjwYYRC
    IeEfNRoNTCaTECYnlKhGoxEJCQntmJIkEkm7lCWSz+p0OmGxWGC1WmG32+FwOGCxWGCxWFBZ
    WYk9e/ZAJpPF3tMo3sNujeM8aHgHQm9Ar3IkbdmyBWPHjr3O4XCMibecjD0DOhbe0jQta2lp
    oS7BYEgCMAdcL6SB/Nn8S95uIhd8vFxI4PeOUi6X0wqFQvX6668rSkpKSCMy5sCBA6LHH39c
    iK5RFKVmWfbX4Ji4XLwjScmPhAtYfwy4QtRD4NJev+qsXCCOJLvd/m5bW1t//hpgGEZJ07Q8
    IyMDv/vd7+LW66KmpgbffPMNWlpaFD6fb20wGCSHkVgqlR7Nysq6l6ZpWxxkw0gAiWq1Gg8/
    /DCmTp3abUYS6etgMplIR3ZukSYl4ciRI4hGoyZw0Y4zF+qU6QLI+HNYIZVKcd9992HevHnd
    NgckKpGSktIuZbewsBDff/89vF6vmH8nKlxCindnDIY2AGESFukJxFw7zN9Pj6OhoQELFixI
    /Oyzz6YAkEycOLEdx3Q8QTjdY9fUJaTKULyh8GJqaqryrrvuwg033IDi4mLodDohHzASiaC5
    uRlnz55FVVWVuLKyUlVbW6tqbW1NCgQC0Ov1KCoqwvTp03HNNddArVbjhx9+wN///ndEo1GX
    Uqmsrq+vH/7GG2+ICgoKcO211yI/Px87d+6kvvrqK0N9fb0hEAhAp9MhIyMDubm5KCwsxIAB
    A9CvXz8YDAaIxWKwLItwOIyWlhYcOXJEtG3btpQdO3ZMD4VCRQDu4Y2HzoJlWZZRqVR49tln
    MXLkyG5nvrgY5eDgwYNYvXo1/H5/vELhBv4Qgs/ng9/vR2JiIoYMGYIBAwYIvNukd4JerxeY
    sWLz9GOYjn5S7EyodgmLVnNzM8rLy3H69GlUV1ejuroaLS0tZG1r+PvZEkfvejavNCESiUCr
    1WL48OEYNmwYBg8ejPz8fGRnZwth51iO7YtxAJC0JWIYm81mVFVV4fTp0zh+/DjKysrgdDpj
    vXv94mww9Db0ikLYJ554ApmZmYoXX3xxciAQUMXz2h257EmDxJjDTEQKKy9BLjzPK+kX+n8S
    +GE8hzwXb926NTbB3AHglN1uv3Ljxo2ic/wuPT8uxcGRzBscA3lnWKfkAu9IkgcCgTSfz5d2
    LmUunmlBMdeigsFgUjsNViZLjUQi4jg4etW8I0nUr18/FBcXC55wklrjdDrhdDrh9Xrh9XrR
    1tYmFEDHOkwYhoFEImlX5KtSqWAwGKDT6YTmZ0ajUSCYIGfr8OHDkZycjObmZiIXPiWZAXFA
    EpGNJpMJw4cPF+YgGo0iGAzC6XTC4XAIDefIHBDGJDIP0WhUqGcgziWlUgmj0SjIVDIHhLqW
    zMGgQYOQnZ1NnGzDwEU8LppetTMGgxdAkH+4n3SxiwccDkfU6XSKeYPBg16Ab775BkajMdPn
    8w2USqUYMWJEjxUedRQMnRC4pwE4VCqV8o477sDYsWPbHZQAhDSRgQMHChsiHA4LC58saDIX
    JSUlWLlyJcrLywHgYDAY/CvLsuu2bt2anp6ejt///vfIzMzEggUL8Jvf/AZOpxN+vx96vV6g
    Det4ADMMA4qioFQqodfrUVBQAJvNhp07d4LfHPVdOb9SqRTDhw/HuHHjepWG5Pf7482clAWg
    kKIoFBUV4dZbb8UNN9yAQYMGwWAwXJRS/HPQ6/XIysr6yRq32+04efIkdu7cia1bt5IUrVG8
    IROvot/hAEwSiQSTJk3C3Llzcc011yA1NbXLIrCkAI5wmBuNRhQUFOC6664Dy7Kw2Wz44Ycf
    8MEHH2DHjh0IhUKpvMA+HMe10MZ7jnsDQug5atl2+O677yCTyTJCodBl8S50lkql7dbgORxJ
    lyoXSsF56jO76dYPgeul0R9cKkl3oApculS3wmKx4KOPPurWpm2xxgKJQvawEZ0KYCjAMee5
    XC58/vnnOHnyJCoqKtDQ0ACr1QqHwwGPx4NgMHhOAgiiT8QaXbFkEFKpVIhak6aeOTk5AkuQ
    RqNBZmYmmpubAWAURVFGALY47cN8cOnbyM7OBsMw+PLLL1FaWory8nKYzWZYLBZhDkgh88/N
    QawBGsuaRObAaDQiNTUV2dnZwhyQOeENhkxwhBBxNRg8ACw2m61/TU0NM3z48LhrxWazmWpp
    aaF4pcCLXoD6+nrk5OQURCIRnVwuh9Fo7LF76SLBAAD7AGyqqalZ9tZbb4llMhlOnToFs9mM
    YDAIiqKEjrOEb58UoxoMBiQnJyMcDsPr9aK5uRlff/01Nm3ahMOHDwNABYD1DMP8F0BBIBB4
    9PXXX9eePXsWv/71r3H55ZdDo9HAaDQiOTkZoVBI6AZNPM3Nzc1oamqC2+2GWCyGRqNBUVER
    lEol3nvvPQSDwVYAG8AxanSd65LPmySbvDdALBZfUJ58F6Mfr5wjGo0iEAgILBAAoFarkZCQ
    0OVr2+fzwe12o6mpCXV1dbBYLKQAHPwhHS+WIApczxExy7KoqqrCtm3b0NraioEDByInJweJ
    iYlQqVSdztUlIGwhXq8XFosFtbW1qKiowP79+3H8+HHCZy8B10RNgvhRrPrRDWQDnTAYeoUj
    6bvvvsOYMWNyvV5vcryv3bFTbMdU1c48Fjja2sfR+SawHVEOYD04NqVCAL9H19ejELnQ7UXx
    tbW1ePzxx+P63ntSJolEIrAsm8+ybBoAHDlyBLNnz0ZDQwO8Xu8lyaeO9MDk+YhO0NraisrK
    ynayUKfTIT09HU1NTbEKfCY4mOBJAAAI00lEQVRFUfEyGEbysgGVlZWYO3cuGhoa4Ha7L8mZ
    G2s8xCIcDgtMgdXV1e3eg0ajQXp6emzkWc871HZcrPHYmU1uAXA8GAyOfe2110KlpaUSKo4x
    N4qi8OOPP4r5lKQTiCOl6gW81FSKohQAsG/fPthstrinrIhEIlRVVcHl6pKGt2EAfwUw8dNP
    P7183759aG5uZoPBoB9cMZlEJBIplUqlSKvVQqfTQa/XIyUlBSkpKVCpVHC5XGhtbUVtbS2q
    q6sRDAajvAfpOXCN2GgAfwLg9vv9j3z66adZu3fvRmFhITIyMpCSkgK5XC7kcdtsNoFlxu12
    IxwOR1iWDQBgKYpS6PV6uU6nQ11dHQOuaLrL+zf4fD6sXbsW77//PnoDPSLZF83NzfB6vfEM
    gR8C8GeWZW8rLy8fUFFRkfDOO+/AZDLBZDJhwIAByM3NRUpKCgwGgzBIWhLxFJG0JOJlIopN
    IBBAW1sbHA4H7Ha7YCgSHm/SxI2vGfCDa9b1IeLHjsPya0xC0/St1dXVedXV1dotW7aICHuR
    yWRC//79kZqaKqRlkTkg4XMSZhaLxQJ1HgnJh8NhgfWDpGWR5m2tra2w2+2xQsjHe02/QDc1
    tPwZkA7T+b1gO7SgZzp+nxNqtTpXJBKp4n1dr9eLxsZGYY81Nja2q6/ppFz4G4BJ4MgwAC53
    X5AL4OoKxBf5Ow/wcmEPLxf+DK5Q9TFcPANXmDdgWXAF1oqY+/wIXP+G//MKfLydVgkJCYhG
    o5UAtkcikakNDQ0p8b4PmqbhcDjgcDhAURRkMplDLpfvC4fDTuJYi4MjqQVALYB+FouFIgQv
    8QLDMAI7VQzsvIwQX6xs6IzBEAGwH8C9e/bsUe/Zs6cnvUgH0XvC4PD5fHIAYo/Hg3Xr1vVY
    UTjDMB0FQ2c0yEoAb/p8vvSamprj4NiHasFFdpQ0Tad5PJ4Mj8eT2djYmAkuHKnF/4qUySQE
    eO/RZgCfgguLEW3bC+ANcClQC5xO55UHDhxIilmnEf5nPOBSH5rxPzamJt6IZViWTXQ4HAMd
    DseN/P95p6u9nhRFIRKJYPfu3ttHTiaTCUZDNxs0TQBe5JX0USzLDnO73cPdbndBRUVF4rff
    fquiKEoslUoFpiO5XC4oxySkSjyhpOCXGAw0TQvdLkkxNK8YR/n1YAUXqToBjhHrCIC6OCvK
    dQBeBvB3cOHeodFodITNZiu22WzJ5eXlqr179yoBiCiKApmL2DmINZr45oXtjCdS/BzTDZfF
    /wpI7QDK+DkoBXASHLNYvM9FGzjKy/G9YAsc58+EXgGPx6NlWTbuXRa3bduGQ4cOCecW8cZ2
    EaoAvAmOue4ouKiDmcgF/u8Ja1EmuKifFlydkSrGmPADOMMr8Vt42SKIVHCkFWcA3A/gKnA1
    CNIYo4DIBSd/HtXjf30aLLyBkAigAFy/B7o75EIfOEU9MTERdXV11UlJSUui0egmmUw2SS6X
    j3M4HP1omjaFQqGE7pRJFEWxIpHITVFUs1QqrWAY5sfk5OQDycnJh48ePeqcPXs2Nm/eHA9H
    0hZex5kO4Epw6XXJiC+TXBRcPVADuPTUf4HLGrlo+djZMOJeAOv4A6AnXKwUf5j0Kq3N6/We
    BnBAIpEoI5EIw4fSeswFzee5iSiKqkLnOJm38wKijFdQOkIU48XR8kaDif/UxShWR3llhj6P
    R+gr3ggcAq4oLYn/ewdvJLTwSqIPHOvGuZQiMbj8Vx26mC2HoqhGsVi8UiKR6HtJw7ZzQSSR
    SFrBe1jjEOEK80p7BTgqT0Jz2B9AKsuyqeFwOC0cDqd7PJ5UXpmQnGOI8D/O9Cg/CF2uB1wk
    sZkfrfxaqOK/e3nFoMdkJb++68B595X4X8PCZH6ksiybFg6HTeFwWMvvFdk55oDpMAcRfo7d
    /DO3xOwDK68gkT3R03OwHxynvqYH5YKfP0MivWVDBgIBl0gkCkilUnk8r0sic+0OBz73m6Io
    kVgsjhA50Qm5UME7gpznkQtyfq1rwKVopMU4lYhcOMYbCsx5zpevwUUzi8HRbSfGyIUW/kwg
    PR0CODfzkQhctFzPGyBdrSBG+D3AoHfior3Kl+Q94erIYLPZPGvXrt2TmJj47e7du7UNDQ1p
    Ho9nSEVFhSkhISFXKpXm+Hy+RK/XqxKLxQaJRKKmKEpKzkKGYUQMw1CkPlEsFrMikYgViUSM
    SCSiGYaJRCIRdzQadajVaq9KpXKEw+Fan89XrdFoqv1+/+nk5GS71+v1+v1+9tixY4hzT4ow
    fw79yOsi2fz6Hc7rNikxBjTpYULFjPM5etmYTzJICiYxnm38vizh13oNr7ddshOpswZDHYDV
    6FlKUxbx5Rn/ZRebzbaroKDgsEajEQcCAdbv97OBQIDtSSYdvvNhSCQShUnR5CWgDVxk4Xwg
    4Wg/f4jXduKWXby36rtOKC6VXT2PW7ZsAS8UP+rtnp5IJEL4p7Fv3754K41t/Ig11qT8kPEK
    hIwf0pjv4g4KciTmeyDm76K9fPpZXnn38edkrDIr4Z9ZzP9ZjP/1MyHGA93h+YkSwsT8XW81
    Vr8F8BS46GJPGQyky3yvgF6vRzAY3KHVap0sy8rOIfh7Si5QKpWqFkC4E0QJLvw8yxATo8Q7
    0bkiYzdvkO6/xP/P4BKKPS8Q9eBqLTS9eG9SvNIYt1S9ZcuWEZngBOB0uVyntVotnnvuOcmJ
    EydkJ0+elDQ0NCiys7NNKpXKCEBB07QsEolIgsGgxOv1ioLBIKVQKFi1Ws0olUpGqVTSCoWC
    BhDwer02s9ncZDQaA0OHDo0WFRWFV61aFW1oaEBWVha6oCFhVyDKz7udN4w/4g1owhiWyI9k
    3oBQxMhJeYy8jPIKP3GUhmIMBRv/+x38IA7VSFcunj70oQ996EMf+tCHPvShD73HuOs42HOM
    PvShD33oQx/60Ic+9KEPfehDH/rQhz70oQ996EMf+tCHPvShD33oQx/60Ic+9KEPfehDH/rQ
    hz70oQ996EMf+tCH/1P4f+j48JaSleoAAAAAAElFTkSuQmCC
  }
  set pieceImageData(Merida1,70) {
    iVBORw0KGgoAAAANSUhEUgAAA0gAAABFCAYAAABwkcaYAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCw0M8qgBcAAAIABJREFUeNrsXXlcVNX7
    fu6sLMO+b7IoooILuCGiAoq5r6HmVlqmppaauWSWZX4tv2mmLWZfq59pppWmaWbilhtuuaSC
    CorINuwwDMOs7++PewYBB9BCAZ3n87mfwrl37j1nzj3ved73ed8DmGGGGWaYYYYZZphhhhlm
    mGGGGWaYYYYZZphhhhlmmGGGGWaYYYYZZphhhhlmmGGGGWaYYYYZZphhhhlmmGGGGWaYYYYZ
    ZphhhhlmmGGGGWaYUSeE5i4wwwwznnCIAViz/xoAkLlLKiBhdkBv7gozzDDDjMcCEQDObIsa
    NzhzF9Q/Zs6cCaFQKC4sLPQSCoUFX3/9dcmZM2fQtWtXc+eYYcbjgxBAOIARAIIAlAM4B2A7
    gFvm7oETgDnMDqwGkG/ukicPCxcuhFarhZ+fn6hNmza68vJyDBw40NwxZpjx+OEHIBZAKwAa
    AJcAxAPIM3cNOADuzG5ngndmNjiLbWgIAMgAWLG/ywCUNobO+adISUmBtbW17OrVq8uUSqVl
    aGjoTz4+PkcSEhLkaWlpNGrUqBqvJSJkZGTgm2++kRYUFGhEIhH997//Nb86VRe9juxFKgeQ
    DUDRFBsybtw4SCQScWpqakRxcbE3THiTiAgikUjv6el5/sqVK8nR0dFwd3fHwYMH4ejo6CiX
    y3vq9XprjuNMeaI4juOKhULhYSJSnj17FgDg4+ODwMDAtiUlJSF6vZ7jONN+EgcHh9vdunU7
    W1RUpPv0008bwzzhCMCL/Z0BoKCWeUIAIA7AMgAtKv37UAAxAN5gxqkmSAC4saOU3U/xhL1L
    QQBeZP+/F8BJ8/Ty5CElJQVSqdQmISFhklKpzPX09DzyySefyIuKigzvvPOOuYP+/RrKmdkj
    NVvYlcAcGXgaYAHAhi3sS9natTZEAHgfQCR4NQMAKAHsBLAUQEod5MGaHTpmizRPWH82A/A5
    s72vALj5NBMkDkBLAIMBRFVa+GQC+BPALgDXm+JEU1hYCK1WyxUWFnrl5ORE5+bm9svMzEx4
    8cUXv2rVqtV+AEXr1q3DrFmzqlz3+eefAwAmTpzYJTs7e7ytre3eL774It7Dw0M/d+5c83QE
    OAAYC2AMAE8AWgCXAaxnY0bXlBpTWloKCwsLm8zMzIUZGRmxBoNBT0TVGQ6kUqlGKpW+mZKS
    sq5bt24QiURITU0FEbVITk5eo9FovIjoPqIgEAiEQqEwUSQSDSIipfHf7969C2tr67isrKwF
    Wq0Wpu4pEAg4X1/f7RzHTdfpdA1NDGQAhgGYDMCf/dttAF8z46I0cU0IgIUikahFREQEwsPD
    oVKpEB8fL0pMTIwFMB/ATACFJq71BjAFwBDwURY1gL8AfAbgBJ4cOZoF7smsLczTS4VNtGdj
    jmMLkWI21zRJZGdnw9bWVnznzp3hBQUFXTMzM8+lpKRs9/Hx+YWI0vft24cBAwbcd9369evx
    3Xffwdvb2z4tLa2ThYXF34cPH5b/8MMPGDNmjHmk8A6biQBGM4KkA3CVLfIONTV7NHbsWIjF
    YklGRkb3kpISL6puGJh5EAqFOk9Pz3PXrl1Ljo6OhpeXFw4cOABXV1fn9PT0njqdzqqGdZtA
    IBAUCwSCgwCUp06dAgD4+fkhMDCwfUlJSYjBYNrfxXEc5+DgkBIeHn62sLBQt3bt2oaeI7oC
    mACgPXhn3DUAWwAcrWGu8AHwNoDoav9uzdY0SmaTSkxcawtgEHglhD879xSATey+TwoZbwag
    E+tf/6eZIAkB9APwlrFDpFIpAECtVncA8AyAkYxt/9bUJhqtVgudTkd6vV5jMBigUqmsVSpV
    b4VC0bGgoGBPhw4dvrxy5cqpqKgo/ZEjRwAAkyZNwiuvvIKtW7eG37hxY01+fn5XJyenATNn
    zlw0ZsyYHRMmTNB/9913T7MxsgIwD8BrEonE2tbWFjqdDsXFxUFE1AHAawD2NaUG6fV66PV6
    rry83EKpVAplMpnQ0dGxyudyuRw6nU6g1WolAGAwGCAQCFBWVgatVissLS2VGgwGkbOzMywt
    LauQr4KCAnAcJxaJ7n/N1Wq1RKlUSoRCIVxdXSEQCADwEau8vDwolUqo1WopAIFpO/nYIGbE
    6G2hUOhka2sLACgpKfHT6/VtGWn+wsQc0RNASHBwMNauXYv27duDiLBjxw5MnToV+fn50QCC
    ARyvdp0zgOUAnpNIJGJbW1uo1WooFIoWjHTNYoufpkyKZAAMIpHIXqfTCdi/27EFnwC8J7Ts
    KZtfBGw8jADQA4ALI0h5bIzsYM6YJqdsUCqV4DiO1Gq1rqyszLKsrKxHUVFRl8LCwmFt27b9
    yt/f//eIiIjiEydOwBhNTkpKwrBhw+Dv7+9w/vz5hfn5+ZMcHR1/nzhx4rtjxoxJ+fnnnzFy
    5Min2R5ZswXtLNxTvwB8tDoEvHT116bUoOLiYkilUpu7d+8uzMrK6k1E9zmCiAhSqVQtlUoX
    JiUlJYeHh4PjOKSmpkIkErW+cePGGrVa7V6DokEoEAiuiUSiS5UddqmpqbCwsIiTy+Vv6HS6
    GglSs2bNthsMhukajaahHXYDAXwAXiZnRBfwjv4lAL43MU/0AB85qmnuGQxgswl7ZMnG0mvM
    1hnRnR1zAJx9Qt4po5SF0EjSfxqKIEUDWAUgyMfHRz9gwAB9cHCwQCAQcFevXjXs3buXS0tL
    68zO0Ta1ha+joyPc3Nx0169fLwGANm3awNHREWfOnLHPzs4er1Qqu5SWlv63W7du27p06aLY
    tm0bhg8fjt69ewdcuXLl3dzc3K4CgQC5ubnNNRrNik2bNpWcPXt2/5IlS7Bs2bKn1SBFApjs
    4OBg/corr6B3794oLS3FN998g927dwfq9frZ4L388iY1I/ALEiIiDB06FK+//joEAgE4joNc
    LsfMmTNx69YtMhocsVgMOzs7MNJDAMjW1hZLly5FZGRkRTRo165d+M9//gONRgOO42CC5BAR
    oX379vjoo49gb28PIoJGo8G7776LPXv2oNJk9agnxdru0R7AdKlU6jR27FjExcUBALZt24Yf
    fvjBWa1WzwBwDMDFag4YbwDCoKAg+Pv7V/R1+/bt4eTkhPz8fBvci1pXxlAAI1xdXcUzZ85E
    9+7dkZOTg88//xzHjh1rwxZEF2A68vQg7WlI2AB4nZFHDoAzx3F2zFP8TqU2nWELgMKnZG4R
    sQXKW2y8CU2Q7aEA/sOIUpOKJkmlUlhbW+s5jqtYlGo0Gmlubm5vhUIRWlhY+LOfn98qANdX
    rFiBRYsWYe7cuejUqZNVQkLC7IyMjFkqlcpSqVSOPXTokEV0dPS8FStWpOHpRg8AL1QjR0Y0
    B/AqW7hmN5UG6XQ6iEQiTqVSWSgUCiERmSzipdFoOI1GIzY67DiOg1KphFarFSkUCgvjZyaZ
    gEAgMeWwKysrkygUCkktBAnl5eWNwWHXHLyjtpWJz/zYZxcBXDFxnXUt3+sAIMAEQYoBMK0a
    OTKSqu4AZoOXoxU3UULkzdrNWVpadi4vL5cSEQegA3jlBgDcBa8YeezOqYYgSB7MSAe1a9dO
    88EHHyA6OlpiYcErPNRqtWDIkCH6xYsXG/76668W4HMFrgJoMhOyj48PunXrpjl8+HAOAHh5
    eWH58uXYv38/vv32W9y6daulWq3+UK/Xt/H39/8wLi5OHhsbK9mxY8eUvLy8GGdnZ0yaNAmH
    Dh3CuXPnmqelpb0xYMCAxD179jytRokDH9J2j4qKwty5c2GMtDg7O+PMmTPIyMhoy7x38qba
    SFdXV7Rr164imuPo6AhLS0uwCcNy2rRptkVFRdypU6e4IUOGGBITE20BCEQiEQICAtC2bduK
    77pw4YKRfAk8PDxsYmJibMvKygQAYG9vbzhw4IAUAGxsbBASEgIHBwej8UPlKNYjhDtbZLRk
    C85L4HNgqnsHOwDwDwwMxPz589GqFW+XAgICcPr0aSQlJTUD0LEaQdIbF/dyuRwKhQLGyFNO
    Tg6USiXA568VVbuXGHxRB1mfPn0we/Zs2NjY8BOlSIQLFy6gtLS0LTOE1cmDG2tPEHht+OUa
    2tOQsAbQB0A3AAYWPTJGkNpW8tzZgpcJPS0EKRbASlTNVUM1wt0OwIdsrO5EE5K1uLq6onnz
    5tqLFy/mV/OQQKVSOWZnZ09Wq9Wtunbt+n5kZGT8qFGjDNu3b0dwcPDAzMzM6SqVypLNDcLs
    7OxhYrH4zpAhQ952cnJS7d+//2m1R53YO18TQgAENiWCVHloPMDnBABCoRBSqRRCoRAAqIbI
    URWiY8phV9d1D/Fs/xSW4CPILZj9SAKf4lE9z6cjmwtqQktmQ6oTpPK6+CkAlYlxFs1sZU3o
    xsbZORPtacM+07G2mGpPQ8IBwEfMAQW1Wm1BRLbsswWszwTMlr7UEBygIQhSLwDd7ezstIsW
    LdL379/fsrq3q1+/fkKFQqGbNm0aCgoKugLoDeCbpkSQpkyZovbw8LgtEAgMaWlpAjs7Oyxa
    tAhRUVFYsWIF9u/f75iRkTFTr9fbt2/fftHFixf98/LyxnAcJ5o4cSLefvtt9OjRA9OnT0dW
    VlaP5OTk6R4eHp9ERUWVOzo6ljs5OZUrlUp8//33T4NBEoOXvMDDwwMymayK8Wd/WxnPaaqo
    yTOm0+mEd+7ceT47Ozuq2r/bGwwGB1PXGv/mOM5Lq9V+cejQoSoTdGlpaQu9Xm/yGR6Dhy4U
    vBShD3i5F4GvoLaVLUIzK51rC0Dk4OBQQeKM5JGRHrEJ7xqYwcg8f/6858qVK9G3b1+oVCp8
    9913yMrKAjOAidWukbL7wc3NrYpk0dPTExKJxGh47EyQuLcA9K3UnjzWnv+CL/DQGJAHYKHR
    wItEomC9Xv88I+CbmSPKAF7XnvWULHa9mMOuxQOc68vOvcA8mk2GIK1atap806ZNdzmOI/Z7
    3/Mm6PXC/Pz8Hjqdbr1SqVy8atWqbWVlZZ6nT5+eUVZW5lJtzhHn5ORMPHnyZMKZM2d+3rp1
    K8aOHfu0ESQR+NzE2tDk7VHtPIezmjt3rm1hYSF35coVxMXF0bVr12SVHC41XSjw9va27d+/
    v61SqeQAwMHBgXbu3Clt4DngNQDPAnBlc2AGgP8DsAF8MSAjXGA6amiEBfj86Or4G0BuLWMi
    ndmk6vaorjFkY4Koe4JXOoxinxnb8w2A/1VrT0PCwEghAeAMBgNVc0oJ2WdqNFDe7+MmSAIA
    YQBsOnToUNa7d+8aQ7ExMTHCzp070/79+62Yt2YLmkjVjnfffRdhYWGQSqWZRUVF5Xl5eVaZ
    mZlo2bIlIiMj8emnn+L999/H1q1bxdnZ2ROkUqnI2tpao1QqmwUHB2PSpEmwsrJCbGwsxo4d
    i48++khy+/btGXK5fIClpWWBVqtNJqIrzZo1u9WmTZuL/v7+2bdu3dImJiY+qQZJCxYZunbt
    GtLT0xEQEAAiwpUrV5Cfnw/wORNPVKnMSmSFy8rKCmTeoAciV8a/9Xq9VUZGRmSNM5TBgMcs
    WfAEL+ca6unpidatW0OtVnNXrlxxKSoqms68RksredMyAZSlpqba/P3333By4tclFy9eRGZm
    pvF3TzVxn1MAviktLZ3z2WefWW3atAl6vR6lpaUgojQAa8CH7itDyQwVLl++jDt37sDf3x9q
    tRonT540Rp4KqxE4D/a8Qz08PNCmTRtje1wrteddE97BhoAOvITjOFvsxoCv9CcA8DOAP/D0
    oSf4/IGHIfd9AHzVVBro7OyMZs2akZOTU3ppaamuvLxcbGr+KCoq8jcYDB8uXryYk0ql9gqF
    orOp7ysrK3O9c+fOu35+fsEcx2V17Ngxx2AwJNnY2OSEh4cXJycnG3bs2PEkjxkdW+zWBiWA
    nCex8Xq9XpSWlvaCXC6PRqW8EZ1O56jX623rIiNarfazffv2qSpfq1AoWppy2D0G2IDPJZsG
    vnqaEa2Y00sAPsqhqeRkUrHrTEEN01HDkwB+BPCyiXV3CSMuSSa+q641TWm1cSZj7Zluoj1v
    s/asbiRr6WL2rP8FH33sSkQfgXd6LgFwkI2RAjRQJPZxE6QKRhwUFGRpb29fYyKWvb09FxAQ
    YPzclTHzJkGQZsyYgfPnz0Mmk10VCoXZSqUyICkpCVFRUQAAf39/rFy5EjKZDOvXrxfduXNn
    rEgk0hoMBkG/fv0qZEQSiQRjx47Fjh07cPPmTZuSkpJ2xcXFABAlFAqRlJRULBQKb2RkZByW
    yWTxAwYMOL9ixYqCnTt3YunSpU8UVwBfHeb2yZMn/WfPno0ePXqgtLQUv/zyC/Ly8gC+utgT
    xRBtbGwwfvx4REVFVcjuTC1srK2tK/JsjAgJCcGsWbMqcpBqurZly5YwylsfE3oB6O3l5YWP
    P/4Y0dHR0Gq12LRpE5YvXy5WKBRjAWwDn08GAAkAzqanp8e8/vrriImJAQAcPHgQGRkZAK/z
    P21qHccMQa5er59QVFTUmhmcU+CLOuzH/XINYv8+6tixY97Tpk1Dp06dkJubi71790KtVusZ
    iagc6u8NoI+Pjw9WrVpV0Z7Nmzdj6dKlkrKyspGMfJxt5O/X01iWWGh02D3ENZbsGinuaeQb
    Nc6fPw8fHx9YWVndTk9PL2T21CRKSkp8UlJSPrKwsFCq1WqrmuaNnJycYADvisViys7OVnAc
    lyWTyW4nJib+aWtre6Jnz55Xtm3bVnD69GkMGzbsSRs3BOAw+LLMzWs45zj4SOyTSJC49PT0
    IPBy4odjljqdVVpaWo9G1Jxu4KsQSkx8Zg1e2rUX97aE+At8pD28hu9LZjYLJkjQCmaX4nAv
    snObOVu+xf25jcZ1z/ha3tnTqFrtrRuA52ppzwvMhv3VSN6jHCPBIyJn5nzg2Lt19WkzSBK2
    OKExY8aQQqGgmqBUKmn8+PFGw/0dak9wa3SIiorC6NGjbby9vXcDoHHjxlW0t7S0lPbt20cj
    RowgsVhsbCPZ29vTr7/+WqUfSktLady4cQSA2rZtS1OmTKEePXqQm5tbxbUcx5FUKs12d3ff
    GRISMnLChAm2/fv3f9LGjhjAFI7j0iot6Ii9UIfxcF7gRoFBgwYhLi7OydfX9xAAmjNnDun1
    empIlJeX04QJEwgAtWjR4sclS5bYTp06tT4XpEsB6AcMGECFhYUV901MTKQ2bdoQMyCjq13X
    hxkdXaXfXcO8cjF13JMDn8SaBV4X3v0BnDizwUelDJXupwTwA/iE0sr4AAD169evSnuuXbtG
    LVu2NF43qpEOwRjwnvBc1sdPGyzBy07oIY8fwaSYTQHLly9Hly5dEBsb6+7i4nK6rvZZW1uT
    hYVFnf3g7OxM/v7+JJPJKv5NIBDopVJphouLy+5WrVq9EB0d7fHjjz9y8+bNexLt0WTwG05X
    t0cHwOeqNCk888wzGDp0qHOzZs2OsJygRnNwHPco7BHAV4HT1nLvMgDjqtmTZwHcMHFuKoBJ
    qF1maAk+inSH2aMBuL8oTGVYga/mXFDtXgbm7OtW7fwF1eykqfY0Vk1sL/BKoQLwlQIbHILH
    fD8Nm1B0V65cMXqATSIzM5OuXLliHAi30MTKztra2uKHH34oJaILAHDixAlcvXoVer0eX375
    JSZNmoQdO3bAwsICPj4+EAgEsLW1RUBA1fWXtbU12rXjcwKbNWuGd999F1u3bsX27dvxwQcf
    oH///rCxsYFarXaTy+XDbt26tf7s2bNrXF1dO+/fv1+wYsWKJ8UgaTmO225hYXGkWjSlhC1y
    mnypy5oiPU/QMxB4LTHpdDpU3vOi0t/GcyrjEPPkvQPgF+Z1S2Lk5PAD3NOTeey8UHvCK8BH
    BTYwj95d5sX6PwBzmTG9Ve18A/OqorJEpNLfBjTO0tBiNv+nsXZyNXgdn2To8c+kjyo0oa0n
    Fi9eDD8/P7z88st5lpaWJ2p6x52dnTFx4kSsXr0a3bp1q/N7u3Xrhs2bN2PDhg2YO3cuoqKi
    YG9vL1Cr1Z65ubmDb9++vfbq1avfrl+/fqS/v7/s1VdffZLGjhbAT+BlQJWjrwrw6QB/wYxG
    b3LBK5O4OtbIFtXsyS5Gcr4CX/HTmD/0HkyX+K4+d1iDl2Z7MQJUm7awDLzE7zNGqu6Ar9q6
    GryMrnq0qq72CBvxPK8HL0lXoZFUCm2IIg3HAWRcv37dd8OGDVi6dGlFpSgjlEolffXVV/pr
    166JwGsP/0QTkoDMmjUL69atQ+fOncPLysoGA0B6ejri4+PRrl075ObmQiQSYfjw4Rg1ahSE
    QiGWLFmCli1bwtX1/kiqi4sLJBIJCgoKQETw8vKCl5cXevTogfHjx+PQoUPYsmULjh07huLi
    YueUlJQXSkpKuiYmJn7s6+u7rUuXLoqIiAisWbOmKU9mFkQ0VqVSxYrFYvTs2RNyuRyJiYn2
    bLK6DtNSqyaDnJwcXLp0qUGJkkajQUHBI8vhNICXQSovXLhg+9NPP2HQoEFQq9X4/vvvcefO
    HYAPt980cd0VdkQywxQAvtjAnjrmBhvwCfgc+//AB3hOa/ARSVfwybWLUHPRgksAFJcvX7b5
    +eefq7QnPT0d4JNjbzWyRUEQI5zR4AsPcOB14EcYOUzE0yG504CXcugewhYa2DWqptTQ27dv
    Y/369bbl5eWO1auIcRyHsLAwzJs3D4MHD4ZOp8O+fXXvrCEWi9GmTRtERETg2WefRV5eHs6d
    O4fvv/8e8fHxyMvLs8nNze37119/heXn528NDg5eR0Q3Dx48iD59mnzA0gL8JrGDqy1I7dnC
    9SZ42XeThUAgMFana1DUVPq7nhx2t9i7LKvhHAV42Vx1cnyEkZNwAJ8ysuOIutNAJMwGicFH
    oVs+SBcwQuUM3hE8gz2TqXslMVJVU3vy0fgqQnOs/4KY/RezPrrJCGGDORgbgiBdAPCdVqud
    t3HjRovAwEBMmTKl4kU0GAy0detW7YYNG8QajUYLXtaS0JQmlpMnT+KZZ55pdfny5RXFxcWh
    QqEQOp0O+/fvx8SJEzFnzhyMHj0afn5+sLe3h1qthr+/P2xtbSuS0CvDyckJUqn0voR6juPg
    6uqKMWPGoE+fPjh06BC++uor/Pnnn1xmZmaboqKiVdnZ2W1DQ0M/WLNmTdaePXswaNCgpmqM
    XgDwjkgkcouLi8OyZcuQnJyMN954g7t8+XJ35mWZg/vLXTZqGIswcByHXbt24dixY4+rktz9
    sxQjZvn5+ZVJWn2ztWMA9ufm5j67YMECbs2aNQadTsdlZGRwKpVKBV5Om1SHsbBiJOYZAF+j
    atGE6rDDPVmckP1/XfkjoYwgWYCXRNRmoQ8D2CeXy59dsGCBYO3atdDpdEhLS4NarS4Hn0/V
    WHIROPCV9t4F0EkqlQqtrKxgMBigVCoddTpdW/CSjbfB69SfBpJ0jC0YAh7w/EzweQFNpm8G
    DBiAFi1ayPbt2zevuLh4VOXILcBXaFy5ciViYmKgVCpx7tw5pKXVvYYqLCxEXl4e7OzsIBaL
    4eHhgcGDB6Nnz544fvw4Nm/ejAMHDiA/P9+5tLT0ldzc3E6hoaHL+vTpc2Dq1Km6L7/8simT
    o8ngE8lN5YZ0YfZodlN12hERoqKiMHjw4Ib1YGg02Lp1Ky5duvTIlmvgnfb9aiBQ+3Av/6g6
    ysEX7bFgxHgg+Gqg2XXYI2PemoDNOxaovQx4EHgJtDUjWPm1ELHjbE7rX0N79qNxRTctwcsM
    XwVfGt+G2an3mQNiHfhtFZR4iuADPhSpCw8Pp7t371Zo9zMzM3Vdu3ZVgQ+37UTNSZCNEsHB
    wejTp4+tu7v7lwKBQO/l5UUzZswgLy8vcnFxoaNHjz50XkhaWhrNnz+f/ve//1F5eXmt56am
    ptKcOXPI0dGRAJBYLNb6+vp+M3bs2GYAoFarm9pYsWQeuWyxWEzjx4+nO3fuEBGRwWCgAwcO
    UIcOHYz62qMAOjeVhg0dOhRjxoxx9PPzi6+s+RaLxeTs7EzOzs7k5OT0yA/jfUQiUeVcAgoM
    DPxh6dKlNtOmTavvpocwIpTNvF1K8Hkw38D05q2V0Z0taAn8PkZ1ZYB3BHCnUv8eQu37l4jA
    yxeM+UeHUUtSO0Mn8HkHeaw9pcwzuRx1S/oe6/QE4IxIJKKoqChat24d7du3j3799Vd65513
    KCgoyNhHZ9i5TwMk4KtVlaLuXAgV+ERry6bSuLi4OHzzzTeCkJCQKdbW1oWoIZfo22+/pb17
    99Lzzz9P3t7eJBQK68wLsbKyoujoaFq1ahVdvHiRVCpVFVtUXFxMW7ZsodDQUOP+OCSTyW6F
    hYVN2LRpk8WCBQua4nixBO/Blz/AeDkOfv++JoFKOUhHAdCMGTNIq9U2aE5sQUEBDR48uHJO
    rF095yAZbcofzA4Z0zoUjBy1rePaMPARZQKfO1OXBzoQ9yL0BD4S5VbHNQtwL08qAXw0qTb0
    YGNPWek+xQC2N7J5XcKcCDlisZh8fX2pW7du1KVLF3J3dyeBQGDc/mMRqsocHxsedwTJCbxE
    pqeR+BQWFqK8/B55Li8vFyoUCgH7UZuBr+d+DLzMLrcxTzAffPABFi5ciLKysv6FhYVxYrFY
    MH36dMycORNBQUG4e/fufTlGD8QmfXzwn//8BwKBoE75la+vL9577z2EhITgo48+QmJioigz
    M3P8yZMnJUOHDl3s6uqaSkSNIt/lAeAAYAqAuRKJxG3s2LF477334OPjA4CPePTu3RsfffQR
    5s+fj7/++qsHgI/B56ocQQPVzn9QODs7w9vbu/Ty5cs/WltbByqVymZEBGdnZ8yYMQMREREm
    N9WrT3AcB4PBgPj4eGzYsKEiemRjY3NTKpX+0rFjx7LDhw/X922vMI9RBzYPWAAYDl4K547a
    9w3icC930o55n/ah5ohQcwD2Xl5eKCoqQmlpqTd4KURNGwo3Ay894yp5+VDH87RiJGob8zaq
    wUeNruDRSbGE4KUIQvARrro8Hxbgpagde/bsic/+YMDbAAAgAElEQVQ//xxBQfeKUPXv3x+R
    kZGYPXs2rl692pGdu6AOz6ZRwy9iiwotmkilUSMEAoHGYDBsEolErfR6/XNEVNPvbQDwK3h5
    Z5OR1x07dgw3btwITUtLe1WpVNqbOicvLw+vv/46ABi3THgglJWV4fDhwzh+/Dj8/f0xcOBA
    jB49Gh06dIBUKoWtrS2ee+45BAUFYeXKldi9ezdKS0v9U1JSPli/fr11XFzct5MmTSr/5psm
    s8WhI3sv5jyA08S48Dbao0ON3R6JRCKIxWLiOE4PAHv37kVwcDDGjh0LOzu7x/48ubm5WL9+
    PU6ePGncYNYgEonoEaxdTrB1Rm/ck791Yu98Xb9Z5c22HcBHbv6oZR70ZePICC+2Lq7JHrmC
    j/qLHtAeGYmHFsBu5kzUgs+lPYjGVXo+CsAcOzs7lylTpuC5556Dq6sriAgpKSlYv349fvnl
    F0e1Wj2L2dNdTypB4pgnZTb4UKETx3GwtbVFv3794OZ2j0C7urpi4MCBXGZmJldcXBxGRGHg
    Q22HAXzCmHGjS3p+8cUXsX37dkRGRtonJiaOUqvVDlFRUZgwYQLs7Owwbdo0GInJtWvXcOfO
    HYSFhVVpe62roYfQAstkMjz//PNo3rw5Fi5ciISEBFF6enqcUqnUBQQELPT19W0Km0D6AXgD
    wARLS0ubcePGYenSpfDy8rpvgR8TE4OPPvoICxYs4M6ePdsdfKXED8HLMxttaHbjxo0ICwvT
    REdHf/3nn39mZWZmLi4sLOycnZ3N/fTTTwgKCsKQIUOMG5Q+EqhUKmzbtg07duxAfn4+RCKR
    3t7e/oivr++Kzp07H/7kk08M8fHxj+LWheydPswMxhDmPJkKviBCaS0GibO0tIROp4NWq+3J
    iNXVWgiSrGPHjkhPT8f58+cd2NhKrMX7FmhlZQW1Wg29Xi9A7TLDFsyJE8II0ld4dPIrJwCt
    AbRnhtaGERQV+BypO+B3HU8DH12rPE86A+hlaWkpGDVqVBVyZJxfYmJiMHnyZLz55psCtVrd
    i12TXq3v7Vn/GZ/BC7z0Q81+M+Mz/A0+QtiYpWiuBoMhBkC0Xq9vV4cjggO/M/2b4CPVB1G7
    tLPBMXjwYHh7e0t27dr1nFKpDK7JVggEgociRtWh1Wpx48YNJCcnY/fu3Rg+fDgmT56MVq1a
    geM4dOzYEZ988glatWqFTz/9FAUFBZ6XL19+Oz8/Xx8dHf1/kZGRmuPHjzd2e+TPHAbjUHN+
    hyl0A/A5+By/72uZ1xocbm5u8PDwUCQlJe2QyWQtU1NTvd58801cunQJUVFRjzUfSavV4rff
    fsPOnTuhUqlgZ2eXYmlp+Uvnzp2VR44ceRS3vANerm0kOluY8+06e+fLa7NH1Rb9/uy6muxF
    ZbbpwM6vSYbdic21Nd2vOlyYPerJ1kDv4NEWPODY/G/DiKWRnCnZUZMzyQa8TLXZ0KFD8dZb
    b8HOzq7CGezj44PmzZtDpVJh9+7dHozAHmV2zRQs2XtptIlqdijBR88MjfW948DrOy8CMMhk
    MkOPHj0M8+fPpx07dpBcLr8vrJqbm0u7du2iRYsWUVRUFNnZ2RnDhNfAe5qFja2RcXFxsLGx
    QWhoaHdLS8u7VlZW9NVXX90nlXv33Xepbdu25OzsTKtXr37kIepjx45RWFgYASCRSFTu5+e3
    8pVXXrHu06cP5HJ5Yx0vnZi3Vmtvb08LFiygrKysWttpMBjo1KlT1LdvX6M8JBe8jtUNjRyT
    J0/Gt99+y3Xv3r29h4fHVolEUg6APDw86MMPP6SCgoJHMjZycnJoyZIl5OzsTABIKpWW+vj4
    bIiMjGxBRJg9e/bj6gJr8FXjjL/b8Dq8TpmtWrWi9u3bEzNcM2s41wLABoFAQEuWLDGWL9cw
    A2JyvQhgi1gsppiYGLK3tyfwGvWaZHJiNsa0bDJ+8RF6rp8Dvx9HBmtzdTmPgS2+UgD8BmAh
    +FwqoxysC4B0FxcXOnjwYI1j4sSJE+Tu7k7sPkZ5kBV4qeJC9t23mOExmHgO46bOh1h/uDbC
    V07AFhC/MIP7sGWHS8BHLfuiYfJ468SECRPg7u6O0NDQljY2Npeqt8HW1pZGjBhBW7Zsof/7
    v/8jf3//eivJLBQKKTw8nLZv305KpbLK1h0ffPBBhT23tLRMCwoKGkFEaMRyO469O3tRezno
    uo488PJM98Zsizp16oTp06dL27ZtO9zJyekvgUBgEAqFZGtrS3Z2dmRnZ0e2traP7DDew8bG
    hgQCAQmFQr2Li8uRTp069Zk8ebLQuA/eI4YV+I1bjfNg31rODWfkqnIZ7am1zDsfmpgvX6vF
    vqypdv551C5Dn8rsgK6W760PWLC2zwOwFXzxiKuMGP7N5v//MdscwZx7lYmdL4ArMpmMNm3a
    REREN2/epGXLltFbb71Ff/31FxERbd++3ThfJINXalR+Lx3YM7wCYCN41dA1dq7xOfYAWMZ+
    w4femuFxTO5d2KBo16xZs/JFixZxQ4cOlbq4uEAkMn17Z2dnDBkyBAMHDkRubi5+//13fPjh
    h0hKSmrNJpkCxiYbDcrKyoye2ACBQODm5+eHHj3u7Yd26dIlvPXWW9i/fz+0Wi18fX3h6en5
    yJ8rMjISS5cuxYwZM3D37l1pbm7uS8ePH79y6dKl7+bPn9/YvLtCNpCXAQjz9vbm5s+fjxde
    eOG+Sof3WTGOQ3h4OD799FMsX74c27dvd1apVK+zF3EpWzg2Snz99ddwcHCgEydOXOrbt+9r
    t27dSs7IyJiWlZXl/P777+Pq1asIDQ2tV1kkEeHUqVPYs2cPysrKIJPJMj09PdeEhIT87+7d
    u4VTp07Fhg0bHlcXqMFHKgzgoxbTwGuts2ry2Dk6OqJ79+5ISkqSqtXqWOadrV5+zw5ACysr
    KwQHBxs32xUzj50I9xdfaAWgm5ubG2JiYnD58uW6PHbhzKMsYvd+FBLgIObBHArAzsLCAp6e
    nvD394eTkxOsrKygUCiQl5fHyeVy66ysrACFQhFgMBj6VPK6/craLQaA6kn6VTr3noxXxHGc
    PxF5g4/u9QDgLRAIxHZ2dvD09ISrqyucnJwgk8mgUqlQUFCA1NRUUXp6uqtKpXIFnw8YC778
    bWMpVCEAnyewAnxE6J/Ahjn9AgEsBr8RcKMq+61SqaBUKiGVStuiWh5vQEAA3njjDYwaNQqO
    jo5IS0uDu7s7bt++XS/31uv1SEhIwMyZM5GcnIxZs2ZBJpPBysoKM2bMQHFxMVavXg2VSuWT
    l5f3Zq9evVKuXr16acKECfjuu+8amz3qx+xRB/y7gjVO4KV5vsyrf7Mx2qJz586hdevW6unT
    p//y/fff30lJSXkzPz9/cElJiYStb+Du7g5ra+ta55GHfikFApSUlEAul1dEEaRSaZmrq+u2
    gICAD+Lj42/MnTsXhw4dehzdUF7JHnmCz4H+i5Fck/aoWiSjL3j1SnG1c21xf069qBZ75At+
    b6DqhL0mtGJzvjUjao/CHnHsXXiB2QUfAEKO4yAWiyESiaDT6aDRaABeqq5h/WYkKwcZgXEH
    4GBpaVmxDv7++++xbNky6PV6yOVyrFu3Di1btoSNjQ2Ki4ttGTFMBh+FiwEf4WsPPmomNY4j
    kUhUeZuNEHZeDnjp41fM6dko5K6O4BPDKCAgQLl3795inU5neFhPt16vp4MHD+rbtm1r3ABr
    b2OLDISEhEAikcDX1/clsVhsiI2NpezsbCIiunXrFg0YMKBKEv5777332BIgNRoNffjhh2Rp
    aUkAyMHB4VLPnj07AMBbb73VmIzRcPAVzCgkJIR+/PFHUqvVJttUVlZGGo3G5GdyuZzefPNN
    cnBwMG7c9zMerJxmg4KI0KlTJ8yaNcsiODh4vKOjY9Kj3rBPIBAYnJyczoeGhg6fM2eOODw8
    vKGa/wqbTI3J8LNhWm/dB0B2ZGQk7dmzh1q0aGH08vU0cW5rANc9PDzo7Nmz9O233xo3wNyN
    qjpwI94AoBk2bBjt3LmTnJycjAULTCXF2oGvWGTsy1uo/4Ts9gB+B6CTSqUUHR1Nn3/+OV2+
    fJlycnJIoVCQSqWi4uJiksvllJycTLt27aIFCxZQr169jOOfwMsZLwJQSqVSWrduXY1zxcaN
    G8nKyooAlHEcd9UYYXF0dKTevXvT4sWLae/evXTr1i3KycmhkpISUqlUpFAoKC8vj65cuUIb
    Nmyg2NhYY1/rAcSDj2Y1BnRnns76eoeS0Qg32e3RowdkMhlCQkImWVtb643P6+/vTzt37iSd
    Tleh1pg2bZrxN6/3w8HBgVatWlWlgENmZiaNHDmyohiMj4/P5nHjxjlFRUXVG0mrB4hQ84ag
    VQ6RSEQ+Pj7k7OxMDzBf68HnUrRpzLZo6dKlAIBnnnnGvUWLFh9aWlrmGdcuo0ePpmPHjlFK
    SgolJyf/q8P4HX/88Qc988wzJBAIiOM4kslkmUFBQYtGjRrl2L59e7z55puPre0s32lqJXuk
    YMTDpA8a/D5ylX/juwAiTDg0fRnRqj4m9nAc52Ti/Odxf/GYvwB4m3gOCfh9AXXgN9bNEovF
    sdX2jKyPNdog8FEsg0gkIn9/f4qLi6O3336b1q5dS19//TV98skn9NZbb9GLL75InTt3Jicn
    J2PBhXLmKFsL4HUA2XZ2dvTjjz8SEdG8efMq2jl+/HhSqVR09OhRcnV1JZFIVOLp6bkSfE7f
    VfZdJJPJqHXr1vTss8/Sq6++Su+99x6tXbuWPvjgA5o3bx4NGTKEmjVrVrkAVRJ4NUaj2Atq
    BIBCKysrw2effVZuMBgM/2ah//XXX2tsbGz0bMBOaEwTyo8//ggiEvj4+CwBQH379iW5XE5q
    tZreeOONKlWBOI6j995777FWg8nKyqJhw4ZVSCACAwM3Llq0SPbyyy83li4cyBYbFB4eTkeO
    HCG9Xm+yLUlJSTRz5kz6+OOPqayszOQ5paWl9Mknn5CLi4vRKP1Uw8TS6DB9+nR89tlngp49
    e/ZwcHA4CiaJiYyMpD59+vyrIzY2liIiIsjS0pIEAgG5ubn9FhUVFXru3Dlu4cKFDdnsZwEU
    Gw0k+DyWdibO6wtA3r17d7px4wZNnDjR+Psuw/3S234Actu3b093796lo0ePGsfDZdxf1tkV
    wH6JREJffPEFHT582FgJ8hzzIlbHWAAFbOIn8NsX+NdjfzRjRI4cHR1p8eLFFdUb64JOp6PM
    zEzavn07jRgxoqKipfEYOXIkFRYW3nedQqEwyhArDkdHR4qLi6Off/6ZsrOza3wnqyM9PZ3e
    eeedCvkmi2I1a+BXyxm8rK6+icAfqLv64mOFVsunHfj4+Mw0Lg5sbGzoiy++qCBHRrLSsWPH
    R+qE8fDwoJ9//rnK+EhISKDWrVsbpXZlHTp0mEFE3Jw5cxpLFw5lTo8629e9e3c6ceIEfffd
    d0anSl2Hgb3bfo3dYRcVFYXXXnvNKjg4+AUHB4frHMeRhYUFTZ8+nUpKSuplbSKXyykuLo5E
    IhEJBAKDs7PzXx06dBg5a9Ys8YNsWFzfYFGIEagqvz0NPppfHT1ZtKny76sDv1VCdXYSboJM
    EYArAoGgeTUyY1PNAWc8LrKoTXVEo6rUL4njuLb12C0cgJHGNZqnpyfNnz+fzp8/X+M40Gg0
    lJaWRr/99hstXryYunTpYnSaGVjfajmOozfffJOIiH744QeytbUlqVRKn332GRERrVq1iiQS
    CQkEAoNUKi03ErOgoCCaNm0abd68mRITE6m0tNTkMxQVFdHp06fpjTfeIC8vr8oEdiIaOFVH
    Cj45kWJjY3VyudxQDyUfdYMGDTLq7zeB14o2CowfPx4vvfSSfUBAwD4A9NJLL1F5eTmdPn2a
    /Pz8iOM4cnBwIKlUSn5+frRv376Havvt27fp448/pr179/7jyFN8fDz5+voajWV+9+7dRwJo
    DPpvH+ZlptDQUPrzzz9rbcfGjRvJwsKC2rRpQ2lpaTWeV15eTqtWrTJqWMvBJ9kK0QSQkcEX
    cnN2dp7PcZw+LCyMLl68SDk5OZSdnf2Pj9zcXDpx4gT5+/uTRCLRtWvXbqZIJMIbb7zR4E5v
    AGlubm4UEBBgJD2fmHjH+wHIiYiIoIyMDNqyZYvR+33CxAJ8BgDt8OHDqaioiJKTkykwMNAY
    cephgnjlBAYG0tWrV+no0aNGYmFK8+0P4IhQKKQOHToYI7OHwMto6gOWAFYB0MpkMlq2bFmN
    BuBByuR+9913FBYWVuHd9vb2pmPHjt137pkzZ4x9TxzHUceOHWnz5s0mydSDoKysjFauXGl8
    /7TgE9UtGnCMjQIve6lvEqDEo8s/+0dYvnw5vv76a8vWrVt/Y/zdhw0bRnl5eVV+o+vXr1Ob
    Nm0eus1WVlYUFBREnp6eDxI1oaioKLp161bFfbVaLa1Zs4asra0JALm6ul4eNmxYW29vb+OG
    0Q0JX/DFYx6oL1555RXSaDSUmpr6MH2pBr+Pkqix26JJkyZh2bJlgh49ekTZ29sfA0CdOnWi
    1NRU0uv1pNVq//FhMBjo77//Jn9/f+I4jtzc3H6Ljo7uuHv3bs5YWbGBEAEgFVVzhf7D1rWV
    EcXsSfXf90+O47yqRYXGwfR2Apm4X0rXqQaCfsmEnXMCL+mr7IQ/LRQKPeqxsEY4eJkctW7d
    mn766acqUWGDwUBFRUWUkZFBqamplJGRQUVFRRUONb1eT8nJybR27Vrq2rVrlS1FIiIi6O7d
    u5SVlUU9evSg9u3b082bN6mgoIAGDhxYJbDQqlUrWrJkCZ0/f75CXaTX66moqIgyMzMr7l1c
    XEyV4zEqlYp27txJ7dq1M37fdfbb1RlGflSwMXqAO3bsCFdX13+dQOHg4CDs3Lmzdu/evSCi
    EPAyl7KGnkDGjRuHP/74A97e3sFyubydlZUVunXrBqlUipycHCgUCnTu3Bnz5s1DUVERfH19
    0bNnz4e6R0JCAt566y2EhoaiY8eOD1z9rkosODISzz77LD7++GOUlpY6pqamTnvuuedOXbhw
    oSGrMQlY9CDCyckJCxYsqJK7ZQrl5eXQ6/XQaDRGnalphi6VYsqUKbh+/Tq++uorKRGNA59c
    fbmxGyUvLy9YWVlBp9PpAX7XehcXF7i4uPzr72bV6oy5JvqAgACcONHgm77LARQLhUKfZ599
    Fv/73/8EBQUFz4Lf2O63auOlwtUWERGBli1b4uLFi8HgJW5pleY2fwCigIAAyGSyity/mzdv
    OoDXMR9j50rA58q4REZGIiAgALm5uZU9Z5UhZtGjbq1atUKfPn1w8+ZNMCNXX5uM9WDGVDR0
    6FBMnz4d1tbW/3TOxLhx49CqVSssWbIE+/fvR2ZmJvbv34/w8PCKPFCdToe9e/fizp074DgO
    /fr1w7JlyxAWFvaPc98sLS3x8ssv48aNG9i4caOIvX+/g9ehN4TDrjf+QaLuA8AKvMyuUVTN
    XLZsGX7++WfY29t75OXldSQiSCQS9OnTB05OTrhw4QKuXLkCZ2dn7Ny5EykpD5+eGRMTgxUr
    VuDGjRuYOXMmsrJqL4x64sQJbN68GQsXLqzIVRgzZgzi4+OxZ88eFBYWBicmJr40evToBa++
    +mp5A9ujkWxB+MARB+N/H0LSJAEv9dkDPvrcaPHNN9/gyJEjhiVLlhyxt7ffw3FcRHp6uuD9
    99+HTCb7V9/NcRwKCgqQn58PiUSid3Nz++3w4cPnO3TogI8//rjB7VG1tfJz4KPFR2qyR5UQ
    QkSdGXkybtfhDz4/6L5pmuO4FkR0tJLNiakhUsRVs0kceNlblY1uxWJxoY2NTblOp0NxcfG/
    7QtnAPMBhHh7e2Pp0qUYPnw4BAIBysrKcO7cORw5cgSXL19GRkYGlEolZDIZ3N3d0aFDB0RF
    RSE0NBTNmzfHzJkzERsbi9WrV2Pr1q0oLS3F1atXkZCQgGeffRbDhw+HQqFA8+bNER8fj3Pn
    zvEvi0SC/v37Y8GCBejcuTNEIhEUCgVOnTqFw4cP49KlS5DL5SgtLYW1tTW8vb3Rvn17REdH
    IywsDJaWlhg6dChEIhFmz56NlJSUlsxhfh2mc50fOZoBuMokK2X1JRXbsmWLinmMb6CR5JX0
    7NkTo0aNsvDy8vrEyIiN3jKjB/fIkSP0bxSG//vf/0gsFlOnTp0oIyPjH3/PmTNnjF50srCw
    KG3Tps0kAHjttdcaqvtcwG+ySePHj6fi4uIavdH5+flUWFhI//3vf0ksFpO/vz9duHCBCgsL
    qaCgoMacpNOnTxtzVdSouYJZo4OVlRXs7e1f5zhO36VLF0pPT6+XdygpKYkCAwNJKpXq2rVr
    90qLFi3qJKWPaRwcs7Gxoc2bN9P48eONnp5fUDXfcBCA/IiICEpPTyetVkuzZ882nvtVpYiT
    A4BdIpGoIudGpVLRyy+/bDz3vUqGLQDAOZlMRlu2bCEiqiyxq675DgdwXSqV0sqVK+ndd98l
    oVBoAF+Ipj4cTlbgo+Pk5uZGe/furTeZ7YULFygiIoIAUFhYGF2/fr3is8TERAoNDa3w6F28
    eLHe7nvw4EHy9vY29vs3aJiNVp3BRxkflZTsAkxLMR87xo8fDyJCixYtponF4nKwXJ9x48bR
    b7/9RoMHDyZra+v7Nod+mOOll14ijUZDt2/fpuDg4Ae6pk2bNnT58uUqY2Pbtm3GapFkbW19
    JyQkJNLb2xsNWPbbFUzNgJrzNsne3p7c3NzI2dmZFi1aRFqtltLT0ykqKopcXFzIzc2NZDJZ
    XX2iAb+lQZOAVCqFtbX16wKBQP8o3iFLS0tdu3btprds2RIREREN1k6xWAyJROIsEAiOmnjO
    Laiav9ob97YzqH58Xmmus2ZzX03tf7+SPXJjRMzUeX+jqjSzhal5TSgUbrC1tbW0sqoXkdUE
    AAqBQEDz5s2riBzduXOHXn31VfL29qZKUvP7Klp6enrSCy+8QKdPn66Q9xYUFNDixYsrIsjP
    P/88lZaW0u3bt+natWukUqnolVdeqcjxmzx5coVaSKfTUUJCAk2cOJE8PT1r3NRaKBRSs2bN
    6PXXX69YO2k0Glq+fDlJJBJjxcEGyzHxBvC3RCLRbdy4sd7qFO/YsaPcxsbGmGzVvKEnDWNS
    u4+PT3+pVJppZWVFn376Kf3LdKv78NFHHxEAGjhwIOXn5//j71Gr1TR37lziOI44jiNnZ+c9
    0dHRzu3atWuoLuwG4I5MJqNt27aZfObMzEx67bXXKDY2lvr27UutW7cmjuPI0tKSIiMjKTY2
    lvr160efffYZlZeX33e9SqWiF198sfIEZ2MmSI2OINkA+FEsFtO6devoyJEj5OPjQ+AlCVMr
    ec2GACgwEiQiol9//dVIZpJwb+dzPwCXHR0dac+ePRVtX758ualxMA6AIjQ0lJKTk6sTpMqa
    bxvwpUsNvXr1opSUFCPh0oLf+LY+EMy8WjRo0KB6L/G+Z88e8vT0JLFYTB9//DHp9XoyGAy0
    evVqEovF5OHhQb/++mu93rO4uLgiKR/8/lOtG8ge1WdxhurHzQZqV9WVzIQJ8Pf3R9u2bZvb
    29ufqvyMYrH4X5GiysfMmTNJr9dTUlJShSwTdReEobfffruKRDw3N5cGDx5cUTDG1dV1dVxc
    nKQhck8YIsBHoWtsR6dOnWjHjh105swZSkhIoNTUVDIYDKRWq+ny5cuUkJBACQkJ9Nlnn5Gr
    q2td/bIdVffFedoJUoPbI4lEAqlUaiMQCH4w8ZyFAMZXOr0vizaZatMV45zAcZw7+D08a2r/
    D7gX3X4GfAW6mr4zoFIU8h3cKyZRuRDIW/h3VReNsAOfP0oBAQGUkJBARESFhYX04osv1khO
    qh8cx1FQUBCtWbOmQrKdn59PL7/8MnEcR97e3nT06NGKeeHUqVMVWw8MHTqUUlNTK4jV6tWr
    qWXLlvSgBaxEIhFNnTqVioqKiIjo6tWrxi1CiDnnnWoLJz8qKAHINRoNl5WVVW9CyIyMDEN5
    eTkA5OP+MoqPHXl5eWjbtq1vcXHx62q12iM2NhZxcXH1WpJZqVQaSw6jdevWsLW1/Vcv/7Bh
    w+Du7g4igkKhCE9NTe1x+fJlzJ8/vyG6sCUAN2NI1BSys7OxZ88eHDhwAH/88QcSExNBRFCp
    VDh+/DgOHDiA33//HfHx8VCp7t+XzMLCAuHh4bCwsACbsBxgRmODGkC6VqullJQUdO3aFaNH
    j4ZQKLQGX0EoqNKcVeXlCgsLQ0hIiJEUGbXcXgCcbW1t0azZPcm2n5+fURYSyAiPFXh5gqxX
    r17w8blP1VBZ0tAXwDB7e3vu5ZdfhqOjI+7evWt89px6fB88Ab4cc33vYN+7d2+MHTsWWq0W
    P/30E7Kzs5Gbm4tff/0VWq0Wzz33HHr37l2/zNfGBs2bV/iyvGA62flRw4BHu2Gi/hF//wMh
    MTERISEhlnK5fHpJSUnnyp9ptVrk5+dDp/t3FclFIhH8/f0hEAhw+/btB5bwGAwG7Nixo4qk
    z9nZGaNHj4ZMJoPBYOBKS0sH/P3338G3bt1qqC5siTr27goMDETv3r3RuXNndO3aFb6+vuA4
    DhKJBG3btkXXrl3RtWtX9OnTB87Ozg9yP7M9akTQaDRQq9XlBoPBSJQrw5457PxN2IfqaA5W
    XZWInFF7hLmFQCCwAa9CiAUf8TYFrtK6vROL7oirnMBxepFIJGeOkH+LQKPTsWPHjkY7i+PH
    j2PHjh21pjhUBhHh+vXrWLJkCT799FOUlZXB0dERc+bMQefOnZGeno7t27ejvLwcOp0Ou3fv
    xu3bt9GyZUvMnz8fvr6+UCqVWLt2Ld5++23cuHEDdWzsXQGdTocff/yxIo0gMDCwcoSyTW2O
    rUeZg1QGvuJF7zNnzkiKi4v1dnZ2/4oolZWVUUJCgohV6ElGA+cfdenSBYGBgRaHDh2aVlpa
    2tPHxwezZs2Cq2v97o14/vx5HDx4EDY2NthsxfAAACAASURBVOjatWuN+0c9KNq3b4/w8HDs
    3LkTGo3GqaioaETnzp0P7N27tyF2+XYBIPXy8oK9vb1pC9KyJT788EPcvHkTBoMBp06dwm+/
    /QYHBwe89NJLcHJygkAgQLdu3WrcL8nf3x+WlpYoLy+3Z/dMM5uCxmWXwFeXMaSlpQmJCM8/
    /zzi4+Nx8eLFDuB33V5syiC5u7ujd+/eOH78uNRgMPQDHx0KAGDv6ekJR0fHKuPA1tYWpaWl
    7mwh5Aqgu5OTE2JiYiCRSEwZJGLG7RUATgMGDMDAgQNRVFSEzMxMgN84tL52XbYFIBWLxfD2
    9kY9l2mFhYUFRo0ahZ9//hl///03Lly4ALFYjIsXL8LX1xejRo2CpWX9KuA4joOXlxekUinU
    arVlAy0Iy+rxNzLpK0PNu7w/FgwePBi7d+9G8+bNB5SUlEwwGAyPpCCNn58fevXi/RCnT59G
    UdGDN/vGjRs4ePAggoLuceSePXuiTZs2OHPmDNRqdfOioqJhI0aM+DspKUl3+PDhBrFHtZ1w
    5MgRzJ8/H56eniAidO3aFX379oVCocDmzZshl8vBcRyuXbv2IGXL7dhiONVsAhoPBAKB1mAw
    ZDDHR/UFV1fwVdCWofZ98iyY8+0HRqhqm/fcDAaDG/ioUG0eKqP9swG/X2BzE/NtmVQqzTMY
    DP/aGQK+YIkLAHh7e8PKygoGgwFnzpxBYWHhQ3+ZQqHAJ598Ag8PD7zwwgto1aoVpkyZgr//
    /hu///47pk2bBhsbG/z+++8QiUSYPHkyunbtCp1Oh82bN2PdunUoLX34ZWphYSHOnj2L/v37
    QywWw9/fH2KxGFqt1pn9NscfN0FSg9dRjjp27JhtfHy8YeTIkf9mwqZDhw4p4+PjLcBHjn5r
    SII0depUfPnllygqKupbVFT0vFAoFE+aNKneQ8OFhYX4+uuvkZGRgV69eqE+pAe2trYYPHgw
    fv/9d6hUKmg0mkgiCs7Ozj79mLuRA9PoikSiGheD1tbWGDlyZMXfX3zxBfbv3w8HBwdMnz4d
    vr6+D+T1rLRRqCXMaIyQA1BnZGRYFRQUICQkBC+88ALefPNNYVlZ2XPgC2zcR5AEAgGio6Ox
    YcMGZGRkdAIQxgiSZfUojLu7O5ycnJCZmWnPvGPNAPgEBwejY8eONT2XGMBoAN19fX0xZcoU
    2NnZITk5GXK5vL4Jkg6AQSAQ/OPCDHUhODgYMTEx2LhxI/744w8IhUIUFhZi+PDhFR7C+oaV
    lZXx/TPgfq/s44CKOdX6PqLvv4EGLtBw7do1dOnSpVVBQcHr5eXlro/iHmKxGHFxcWjfvj3S
    0tJw4MCBB/YiA7x3fu/evRgzZkyF48Lb2xvPPPMMzp07B71eL1KpVAOuX7++6ebNmw2xube0
    rhOysrLw5ZdfVvw9c+ZM9O7dG8XFxfjiiy9w9erVh7mfCA1b2dEMExAKhRCJRLlarVZDRCIT
    9mAi+AJCQO1Stq7gIzABqF1Kac/IjgVqz603RpAGAhhs6gSDwVCkUqkyHzTC8gDvg0gsFsPT
    0xMcx0Gj0Rjt3j/zJOXl4cMPP0SrVq3QvXt3xMbGokOHDjh37hxOnjwJOzs7XL9+HR06dMCI
    ESMgFApx9OhRrFy5Evn5+f+MPBBBLpdDq9VCIpHA09MTEokEWq1WCKDGSiOCRzzOjgI4VlhY
    KFy5cqXozJkz//iLLl68qHnvvfdILpeLwFefOtiQL9ChQ4fQr1+/5gUFBXNVKpVHZGQkJk+e
    bMoD/c8ZplqNjRs3YseOHZDJZBg/fjy8vOpnu42IiAj4+/NRYq1W66NQKHrk5eVx69evf5zd
    aCyRi9zcXJSVlf2jgf8gyMnJgVqtBvhy3yVmE9BoCZIiMzMTBQUFAIC4uDij08EbvLTBpaZF
    PyM4LuD3XwsGeG+3tbW10REAW1tbI6GWgc9/6y8UCkUxMTFwd3ev6bk6AnhBJBJJn3vuuYrw
    fHp6OpRKJZjDpr52Lc8EUKzVaqvsKl/fZGXIkCFwcnLC7t27sWvXLjg6OmLIkCGPhJQREbKz
    s427q5eAVXZ6zFCDr6CX/wi+uxD85uWqhnpxYmJi0LdvX1l2dvZ0hULR5VHdJzIyEi+//DJE
    IhF27dqFCxcevgDb+fPnq1zHcRyeeeaZisqsGo2mtVwu75yeno5Dhw497q58aPc021gUHMf9
    k4iv2R41Quj1euj1+iwiqkk/GgA+wb+uaLgb+LzZ1nUQKRnHcWEABqDu7WsCmC20r+HzEoPB
    kF1PtkPJ5k4YDIYqY/7f4ObNm1i/fj2Ki4vRrFkzDBw4EAaDAfv27cPu3btRXl6OQYMGoUWL
    FigsLMT69evxb2W3NbybutocW4+aIOUDWAMg6cyZM9yrr76K33//3bhQfSBotVocOnQIr776
    quTs2bM24BOxV4OXNDx2EBHmz5+PuXPnWmRkZLxUVFTU3cPDA3PmzHmgSMbDsOy1a9di1apV
    UCgU8Pb2rpA11Ad8fHwQFhZm7GNhcXFxdN++fe02bdqEVatWPc4uLQGgk8vlyMvLe+Bn9/Dw
    QIsWLR54QXf79m1jfpISjSB3zYwaCVJJSUlJxT5Qnv/P3neHR1Gu7d+zvbf0npAGISR0REQQ
    UZGqcLD8DoIF8NjFg6IoH1ZA7AUrCtgRBESRXhQOqCASWhLS2242ySbZlq2z8/tj5p1vE5CW
    TTZe33mua64NAXZm3515n3Y/9x0fjzlz5hAs//UAbgIg4GhTYbFYUFlZCZFIhLFjx0IqlQq4
    ytqVCoUCsbGxWLduHf71r3/hsccew4kTJ0hhgOL+3dDY2Fhcc8018Hg8aG1tBcMwwQ5AA24G
    Kj8/H7NmzeKLIOXl5eDmIetDGBxXA2gMBAIwmUztnFIobejQocjPz0dlZSXKy8uRn5+PYcOG
    dcm5AoEA6urqSKehEaygYTjsAFe0C7XtQ3vq32613377Dbt378bJkycntLS0/JOm6S7TektO
    ToZarcbBgwfx0UcfXVZRq7GxEQcPHmyX/GdnZ6NfP5Zfxe12K2022+iHHnpI1s0FO1Ls8F6q
    r3Y6nWhpabkc+M9//VEPtEAgAJqm6y+QvE4E28k5XxwtACs6fEFoEcMw48DSe5/PpGDnjv5y
    sxaLxVaVSmUNEVS6FoDF5/PBaDQiEAhAKpUiKyur0/Dvn376CTt37uRlJVJTU7F7925s27YN
    xCdTFIUdO3Zg27ZtnTqXUChEZmYm77tra2tJHtKC84xbdIdI2c9gVYWX/fbbb73mzp2LOXPm
    4IEHHmg3G3DOncpqxQcffIAPP/wQFRUVFFjhrP/pIid3UdbQ0IDly5dj6tSp1xqNxjsBiGbN
    moWxY8eG5P09Hg8KCgrw5ptv4vvvv+cdkNPpvCSs94VMoVBgxIgR+Pbbb+H1emGz2fo2NjZm
    ms3mw/v37+/OJT0DwNzQ0JDw888/X5TuyujRo/HJJ58gKioKEREX1uZsamrCvn37SIBW/l+H
    1GOtAYDV7Xa3qxZdf/31mDRpElatWqUHi88WMgyDffv24euvv0ZJSQmuuuoqjBo1CvHx8aio
    qEgC2HmbX3/9FXv37uUTrpMnTyIlJQVCoRA0TWcAQG5uLpqbm/HAAw+gqakJ48aN4+EEYDtX
    8QqFQnTPPfe0m52oqqoCNw9Zd6lB1QXWoA5A36KiIphMJiQmJoZ8oSMjI9G3b1++Qp+XlxcS
    ja1zVsksluDB/FouoQyHtXAFuywAocISngTwBrqmM3VB27RpExYsWICBAwdmG43GeU6nM6Ir
    z/fDDz+gtbUVtbW1lwolaxd8Hjp0CC0tLXwMYDAYMGzYMOzcuRMMw6C1tfWKX3/9Nb6pqal8
    7ty5+Oijj7rTHzWgPbX/eW3v3r146qmnUF9fz+8zl2ClCPPs2n/tbOP2/qbzdJAAFq0wBReG
    ZWbi4mDFebhw0yIBbEfqL7MfgUBgVCqVXo/Hc07Sqku0Oq6olVpfXw+PxwO5XI6RI0ciMzMT
    xcXFl/3Gzc3N2LNnD2666SZkZWVh4MCBWLduHQBg7Nix6NOnD7xeL3bu3Nnp2DczMxNXXXUV
    ALY7WFNTQ+azwlmwa5eIXQtgMwC3VqtlvvjiiwvSw27cuJGJiIhgKIryicXi7dx7hE15esmS
    JZg+fTpmzpwZlZ6evoWiKOaaa65hysrKOk2Fa7PZmL179zLz5s1jMjMzGYqiGKVSSaemptrE
    YjGt1WqZ9evXh5R+98iRI0xycjIDgJFIJL7U1NSZAC4LNtEJi+DuC+bqq6/mue5DaRs3bmT0
    ej0DFs7wyN9lk/4/RvMNsLC3HwEwjzzyCK+ZwDAMc/DgQSY9PZ2n7oyLi2MyMjJ4unqBQMBk
    ZGQwERER7XQQJBIJIxKJmGHDhjGpqamMQCBgtFptO92GPn36tHtvnU7HjBs3juHkBBhwlNtm
    s5m/HofDwfzjH/9gwM7ULAzhGgjAUrTSKpWK+fjjj5mustdee40RiUSMRCLhtaK6wjZu3MhE
    RkYG088KwniPUVyAUYjOUxMXce9FhevDvPvuu9iyZYv0qquuelUul3cJ/TKCqHoVCsVF0+ue
    70hNTeUpg4lt27aN7NOMQqFw5OTkTImOju7uvSkCwPdduY5Bhxuhkwfocvu/QvPNJRkQCoUq
    iqI2ddO9EMrjZYlEIhIKQ9JIloElPWKys7OZI0eO8FpEK1asYDQaTaeuddSoUUx9fT3DMAwz
    f/58/vfz5s1jAoEAU1NTwwwdOrRT51Cr1cw777zDxxNlZWXMsGHDgnUWledzxt1hNFeZOQKg
    zWazoarqwklbdXU1rFYrxGKxT6VSHQerNUGH66E5evQo1q1bh19//fW6mpqa0Xq9Hvfeey96
    9eqFQCAAs9lMZhIu2hwOB3bs2IGHHnoIM2bMwJtvvonS0tKATqcrHz58+JuDBg1aKRAI3AzD
    hIKRpJ0lJSWhd+/eAACfzycSiUSDvvvuO/E333zTnctqAbAWQOvvv/+OVatWEdhSSKysrAzv
    vvsuYVz5Eyy5x3+tZ5oPLJMdtmzZgvfeew9lZWVwuVwYMGAA/vGPf/BtfZPJhNLSUmRkZGDx
    4sW48cYbUVtb226Ik6ZpqNVq3H333Vi1ahVef/115Ofnw263t4OuFRYWoq6uDuPGjcM///lP
    +Hw+bN++HXa7HQBL7HD//fcjOjqaf86/+eYb/PHHHwCLYTaGcA0C3D1a5nA4sGbNGhQWFnbJ
    YhNWIqVSGVJ4cLAVFxfj3XffJfDZEi4BDoTxHmPAzgvdA2AdF6ReqnkAfMe9xxaEh3QCc+bM
    wVtvvYUlS5YMOn369K0ul6vL/LlUKqV79+59euTIkb/IZLJOd0tNJhOOHTvW7nfZ2dk8Hbzb
    7VY6HI7eZrOZmjp1anf7o2/Adhu7PKT4rz/qmcZB7LwMw9T83S6doqg6hmH8l0Kech5zgyWj
    aCstLcW3334Lj8cDoVCIO+64Aw8++CA6I0ZbVVWFhgZWISM6OpqQYyApKQkURaG2tha1tbWX
    X3FVqfDwww9j5syZEAqFYBgGP/74IwoKCsg+vg3nmUHqjm6MAcB0sLjJ/gCUcXFxGDBgwAX/
    Y15eHhISElBVVSX3er33AxjBZbNr0c0zSIMGDUJxcTGuvfZa+YkTJyZ7vV7FFVdcgbFjx4Jh
    GPzwww946623cN1112H+/PkQi8Xn99IMg1OnTmHFihXYtGkTzGYzhEKhT61Wn1EoFJuTk5M3
    7dix4+iQIUPuByARCoUhp9/V6XTo1asXfz0OhyP3nXfekbvd7u7W89gK4Ae32/3Pd955R6BS
    qTB79uxO6T0xDIOioiIsXrwY+/btA9hW8atgIQ3/tZ5pXrAQLKa0tJRasGABPvvsM/Tp0wc6
    nQ6nTp2CUCjkk5ucnBw899xzuPnmm9HQ0IAdO3agoKAAVVVVYBgGvXr1wujRozFy5EhotVpk
    Z2ejV69e2LBhAw4ePAiTycRr9Fx11VWYPHkyJBIJDAYDPv30U77YIRQKsW/fPpw+fRr19fUo
    KCjA4cOHSdvfjtDTR58A8DmARYcOHRI//fTTePnll5GZmRnSk2g0GojFYnQVY15JSQmeeuop
    8vx5AKwGC0kLtzFgYX7FYDVHLpVFzMMle+ZwJUcAUFpairvuukv4wQcf3NTa2toOEkZRFNRq
    Nbxeb6cKTmKx2KdQKM7Ex8dvnDx58tpTp06NBXBFpx90rxfFxcXwer38XEBUVBQyMzNx5MgR
    BAIBCIXCobNnz5ZbLJbuZqvdBrayPAtdV0SuBfDKf/1Rj/dHNdwzTv0dLlgoFHqVSmUTwOrP
    EbKjTtpuAIdomr527dq1GDlyJCZMmAC1Wo3HH38cPp8PK1asuKx5RKfTySdIsbGxkMlkYBgG
    ej3LfWE2my8bJqjRaDBv3jzMmzePjyUPHz6M1atXkz3xMBd7IhwJEgWW/WkegEkURakTExPp
    sWPH0hMmTBCOHj36gm8wfPhwvP322/jxxx+xZ88eRXl5+XCGYfqBFd96nfuA3VKNzM3NRUlJ
    CVwuV5zNZusrEokwcuRIRERE4JdffsFTTz2FwsJCZGZmwu/3nzdB8vv92L59O55//nkcOXIE
    FEX5IyMjj8fGxq7V6XSbU1JSyoqKinwzZ84UlpWVpXk8HlFMTAzi4+ND+pnEYjGysrIgkUjg
    9XphtVoTaJqOo2na9thjj+H111/vrue6GcBSALqmpqZxzz//vPjkyZOYM2cO+vXrB6FQiLa2
    NrhcLvh8PtA0DZ/PB4qiIBaLIRKJIJFIoFAoIJPJ0NjYiO+//x6rV6/Gn3/+CZqmGzhn9EM4
    A5r/2kUFrkYAfoFAIAZYxqsjR44EOwDExsZi7NixeOCBBzBkyBAIhULExcVh1qxZ8Pl8aGtr
    A8MwUCqV7Z5DgUCA/Px89O3bF62trXC5XBCLxVCpVFAqlfzs2+LFi5GamoqPP/4YZWVlqKur
    w/LlyyEQCPjkLGhOrisSJB+ATwHk0TQ9bfPmzRSAy0qSfD4fPB4PfD4f/H4/qYwCAE9IEQgE
    0NTURDSdIBQK+UMqlUIqleJS4RolJSVYsGABNm/eDJqmA2A7LqvBdtzCadEA/h9XsMvB5VEs
    awA8Clbj5AuuaNetc1V33XUXjh07ho0bNxpaWlpGBHdEBQIBxo4di7lz5+LEiRN44403YLNd
    GlGaVCpt0+l0x7Ra7XeRkZE/jRw5suyVV17xJSUlTfX7/Z2mamUYBmfOnIHT6eQTJLlcjoyM
    DBAClqamptSKigpdW1tb22effYaZM2d21/K2AFgGliVsAlhtGt4iIiIwdOhQREVFQa1WQ6vV
    QqVSgaZp2Gw2/qipqcHvv/9O2BuDzQxgOdju43+tZ5uJS5SkoXrDYLbDQCAQUqZShmHsFEWZ
    GYbptF5mkNUBeB9Av6qqquhFixZBJpNhzJgx0Ol0WLBgAbxeLz788MNLKsYQn0KEpvV6PSQS
    CWia5gt2bW1tBO54SXICKpUKjzzyCB577DFoNBowDIPffvsNTz75JBkhaQKwAheYP+qqBEnI
    bSzPA8jT6/XM1KlTvffcc4+gf//+oovthEilUkyePBnXX389jh8/jtWrV2PdunWqpqamWzjn
    thjs/EqXO12GYYieSqJIJIqQSCTo378//H4/1q9fj8LCQlxzzTV49NFHz9vpcblc+Oabb/Di
    iy+ivLwcSqWyxmAwfJ6enr562bJlZTt37gwsWrQI/fr1Q0NDQwJFUSMAICUlJWQU38EPau/e
    vaFQKOD1eiEQCOJdLlem3+8vvtwB3E5YIYCHAMy1Wq13rl69On737t248sorIZVKUV9fj4aG
    BjgcDng8Hvj9fj5Bkslk0Ov1iI2NhcFgQElJCQ4fPgyXy0VzSfQbYKE93r/hBs10CMi74NZm
    /lJgNwxmBGCXSqWGGTNmICYmBjU1NfB6vdDpdEhJScGVV16J/Pz8c3YYxWJxO92jc256ItF5
    Fe4jIiLw0EMPYcyYMdi5cyd+//13mEwmtLW1QavVIjU1FcePH8fRo0dJMNXQBetQx+1vGpqm
    x27evFngcrnw7LPPYsiQIe0YhFwuF2w2G6xWK+rq6lBdXY2mpia0tLTwh81mg9vtJirxPGMf
    cU5PPPEE76BkMhkkEgnkcjn0ej0iIiL4Izo6GomJiXxwqFarIZX+b+wQCARw+PBhPPvss0Qj
    hwYL0Xge4SNnANhOwBUAHgNwIy5MpXshk4FFRGSDZah6DcBBdBMEnGEYREdHg2GYlEAg0M4x
    XHfddXj99deRk5MDrVaLjz766KITJKFQSKvV6uNxcXErMzIyvh89erSxtraW+fXXXzF58mTt
    /v37B3DEJJ22mpoatLS08NVigUCA3r17Q6lUwuFwIBAIJNI0nQnAuH379u6+X86AnVc9AVak
    mu/QDRs2DO+//z4iIyMhEonaafjRNA2/3w+apnHkyBHcfvvtfOGBuzd+44q7P/1N/dH/NasH
    S/0uPV+gr1Ao+EKbWq2GXq+HwWCAwWDg90iJRAKxWMy/MgwDr9cLn88Hr9fL782tra1obm6G
    xWJBS0sLnE4nnE4nHA4H3G73eROqQCDQYrPZ6gBc8rjHBWwLWIKbhceOHVM9/PDDeOaZZzBt
    2jRERERg4cKFcLvdWLVqFXw+HzQaDbRaLQwGA+Lj46HX66FUKvlDoVBAoVBAq9UiPz+f992k
    CEmSp8GDB+OVV16BzWZDW1sbnE4n/+p0OmGxWGA0GtHS0gKr1Qq73Q6pVIp77rmH7xz5fD78
    +OOPeOGFF0hy5ATwNthZQ3R3giQGcAvnEHvl5OT4FixYgGnTpomVSuVlRXkymQxDhw5F3759
    cdVVV+HVV1/FsWPHchmGeQ3sYPfXXNW1yywyMhJ+vx8MwwiFQqFAKBRCo9HA4XCgoKAAMpkM
    s2fPRp8+ff7yPYiu0QsvvIDGxsaAWq3el56e/nJubu7e4uJi37Bhw3DFFVegpqYGSUlJkEql
    Y10uV45AIMCIESO6hGUqNjYWOp0Ora2t8Hq98pqaGjXDMGhsbAzHZlTF3TcHGYZ5rLq6+urq
    6mpRxwCYbDYMw/DB3jmqC1YAX3EPQjH+Zp2jq6++GqNHj5a9//77yQCo+vp6fPrpp9Dr9Z2q
    OAkEApjNZrS2toKmaYHT6Yy/7777RD/99JO/h3z0UwAKXC7XNRaLBS+88AK0Wi1omuady/ks
    EAjwXUZSnQsEAggEAu20SsghFAohFovPSkDFYjH69++P/Px8OBwOuFwu0DQNiUSC06dP4957
    7yXJ694uSpAA4DSAfwN4lqbpydu2bRPX19fj6aef5veJo0ePoqCgAKWlpaiurkZzczPa2trg
    9XrPe58QrLdcLgdFUTCbzaitrT1vlU4kEpFBbURGRiIlJQWpqanIycnBgAEDkJSUhN9++w0v
    vfQSwXh7wUKVnuWewXCZCMDN3HXkhPi95WCZrHoDeA5sp6zLA1+ZTAaVSgWGYbRisZhP9qKi
    ovDwww8jJycHx48fx7Jly1Bff3F5qVQqbdLr9Z8lJyevvP/++4uLi4sD//73vwGw86pZWVn9
    /X5/yLjgm5qaUF1dzcO8ASAtLQ0qlQoOhwM0TWtramqiA4FAqIO9i87hACwBcIh7DkeLRCLR
    yJEj+RmJcz1XJLjLzc3FgAEDSILUCrbb+A6XfP2tTSqVIj4+HiKRqFP+iAiP1tXVEXkFBkCn
    4PUhq6gIBKAo6gzDMCWBQIBnh1QqlYiJiUFmZiYyMzORlJSEpKQkJCcnIyYmBkqlEhKJhD9I
    4H8xRtM0nyyRhIkUvqqqqlBTU4OqqiqUlJSgqqoKjY2NCC5YCASCY1xxDSGaQSLmBvAeF28/
    UFhYqH3sscdQU1ODuXPnIjo6GosWLUJMTAwCgQDy8/ORkZGBqKgofj06FhPOVdzk1pz/N2SN
    z+Xj/X4/vF4vnE4nGhsbUVJSgoKCAkgkEtx7773Q6/VwOp1YvXo1Xn75ZdTU1AAs4uMDAO/i
    IqQ5Qp0gUWC54ZdQFJU8atQoesmSJRg2bJi4s5zp5Ma8/fbbkZmZiWeeeYbZuXNnKsMwLwJo
    4xxTlwXBSUlJsFqtYBimkWEYu8/ni7FYLJDJZLjpppswfPhwXHPNNe2+RLvdDrVaDYFAAIZh
    sH79eixZsgQtLS10VFTURrVa/fSxY8fOPProo/jiiy/4Dfeuu+7Co48+mvDdd9/d5na7lQkJ
    CRg/fnwoW6bt1pRsRj6fT2iz2XQulws5OTmXQ1kaCvNw1QolgIHx8fG6iRMnIjY2FiqViq9M
    aDQa+P1+WK1WvnJgt9tx4sQJ7NixA16vt4Sr6pb93ZzPsGHDMGTIEPmXX3451263384wDFVV
    VYVnn302ZOfgIDlUfX39rA8++KDk6quv/mrs2LH+Xbt2hfvj1wJYCWDgjh07tJs3b8bs2bP5
    Z8PtdqO1tRV2u52vHjU1NfH3AKkskaTZ7/fzB0VR/CYtEokgFAr5QJPcWzqdDtHR0YiLi4NO
    p+N/RzpsbW1tWLt2LYqKigB2nmZVFwfEJ7ngzAzgjmPHjqkffvhhpKSkoKamBg0NDTzcVKPR
    QKPRID4+HtHR0YiOjoZSqYRcLufhp8HdIZlMxne73W43XC4XL6jr8Xh4WKvL5eKdUENDA2w2
    G4xGI4qKihAIBCAWixEZGYn4+HjU1dWRgNwKdo7qFZxHZ6IbTAJgBthuXHIXnieb+6xa7p7w
    dOWHio6OJlBSB8MwvKNPTExEdnY2/H4/Pv74Y+zefWE9dW5eqUav178waNCgLw4cOOASiURY
    unQpAOCRRx5BZmam9K233vqH0+mMDdVnsNvtZxXitFot9Ho9ERYWG41Ghd/v79QgeAj80TYA
    agCDdTqdLiYmBrt374bRaOQ7sx6PBxRF8c+UwWBAXFwcevfuje3bt8Pv95dyFfi/nT8iBbtr
    rrlG/s477yQ7nU4qKysL7777LuLidjd61AAAIABJREFU4jql1SYUClFaWoo5c+agvr5e4HQ6
    4x5//HHht99+S4f7MyuVShgMhur6+vo1Xq83j2EYxYgRI3DPPfdgwIABSExMhFarveC8+aWu
    h1wub4dCSkxMRN++fQGwnWOXy4WWlhaUlZXh8OHDWLNmDU6cOAGhUFgvkUg+CQQCdpFI1BVF
    BSuAl8GORPzbbDbHvfjii6iursbjjz+O1NRULFq0CEKh8KziAU3TcLlcsFqtcLvdcLvdfKeV
    +OdTp07xiU95eTlOnz4NoVDYrvMWfJBinV6vR2JiIk/kFAgEIBKJYDab8fbbb+P9998nJF0N
    AN7iEr2L4g0PdcQ9jDii0aNHe9944w1hfn6+OJQnoCgKQ4YMweuvv07Nnz8/sG3btiTunCYA
    /+mqhyUrKwvbt2+HSqWqVSgUpsbGxozDhw9jwoQJePDBB8EwTLvq9o4dO7B27VosXLgQmZmZ
    +PPPP7F8+XKYzeZAbGzs+szMzCfsdns1ANx55538/1uwYAGSkpKkK1euvLupqWkkRVGYOHEi
    L+raFZUglUrFL69er++1ZMkS0U8//eQ/ffp0uPYmMYCBALRTpkzB8uXLg6/xvHb8+HGUlpai
    qKgohQta/jYOiWEYTJs2DTqdTr5p06Z7a2trF/r9/qicnBxER0cj1M9RXV0dSkpKkkwm00v7
    9++nxo4d+6VcLvf/8MMP4V6KnwBsdTgct3388cfIyMiA3+/HsWPHUFRUhMrKSphMJrS2tsLp
    dMLtdsPn83Uayy0QCPjkQaVSwWAwICEhAenp6cjNzUV+fj4qKiqwYcMGMAzjBjsn1B1Y1CoA
    T4OllX7EZDKlm0wmqFQqDBw4EP369UOfPn3Qq1cvJCQkICIiAiqVCjKZrF0yeKlzRJyaPO/A
    3G43nE4nmpubYTQaUVFRgeLiYpw8eRIFBQWE1Q/cdb4Jlg0snJpjFFhh4efB6od0tSVwvsgB
    tnvdZfOxEokENTU1EAqFlYFAoJ4kfyaTCT/++COUSiWCn+Pk5GSMHDkSe/fuDYZ8AQD0en1Z
    UlLSk6mpqRubmprohoYGzJgxg9+TKIrCDTfccJ3FYrnZ7/eHDOvr8XjOElYlBQuuiCOQyWQJ
    f/75J3Xbbbcxv/32Wzj90WAAWqvVimeeeYafdSTdapqmQVFUu9k9hULBd7IBpALo83dMkIYP
    H44hQ4Yovvzyy385nc7bKYqicnNzkZeXB51O1+n3VygU6N27N2praymTyXTna6+9VjJy5Miv
    r7/+ev+OHTvC9rljYmJgMpkYqVS6gabpiX6/f4JQKMSYMWPOy/rJMEy7vfNcP5PuDunkk/vm
    XD939NsEnpaQkICoqCh88cUXAMCIxeJ1arV6P03T7dhcQ2w2sLM7JgBPOhyO3I8//piqrq7G
    //zP/yA/Px82mw2tra0wm82oqKhARUUF6urq0NzcjNbWVrS2tsJms8Hj8fAzsYFAAG63Gzab
    DQzDYMmSJXjnnXcgFot5SF4wRI8U9uPj45GWloaUlBRERERAp9NBpVKhtLQUL730EtauXUtI
    Hk5zBaxvuYbKRVkoE6QYAE8CyM/NzfUuXboU+fn5Xabo3bdvX7z00kuM2Wz2//nnn7lgtUjm
    ILSUu7xNmDABkyZNws0332w/fvz4Lw0NDVdt2bKFmj59Oq8ATqyurg6vv/46iouLMW/ePDid
    TqxYsQInT55ERETEwZSUlGePHj1a/eSTT5I5BgDA1KlToVKpqI8++uimysrK+10ul2zo0KF4
    6KGHuoRlqqND4irsyX/88YfY4/GEE3KVCuAGg8FATZgw4aKTI4Clir322mtRVFQUBWA8gD24
    PCrfbrd//OMfUKvV8p9//vlftbW1C30+X+TEiRPx7LPPIikpKaTDnBRFobS0FE8//TT27t2b
    aDQaX9q5cyczbty4r2655Rb/t99+G86laAXwEYARR48eTZoxYwbcbjesVisfiMjlciiVSkRE
    RECr1UKtVvOVt+DuSPDBMAzfKQnumAT/mWzuDocDZ86cIVAxSKVSGAwGMAwDs9kMAPsBrEf3
    UVa3gh2UPQ7gBQAjBw4ciPfffx+9evWCTCYL+QmJoyaFH9JpTktLw6BBgwCwbGS1tbV48MEH
    sXXrVgA4AGAB2FmLcFeBr+ISloRuPGcsgEVgYS57u+okzzzzDPr27YukpKQWuVx+wuFwDGUY
    BvX19XjmmWcAgE8+lEol5s+fj+nTp+Puu+9ulyCpVKqGpKSkxQ8//PCGbdu2BX755Rf+7+bM
    mYPs7Gxcd911WcePH3+itbU1pOtI0/RZzFfB/ggA1Gp1+quvvipRKpWeMN5HaQBuAED5fL6z
    EszgwJgUEzweT8e5r0jOH+36u/gjhmFw4403Ii0tTfH9998TnxQxfvx4PP300yFJjgAgPj4e
    zz//PNra2nDo0KEko9G45MCBA9SNN974lVKp9G/cuDEsn7+0tBTJycmorq5ukMlkH9I0PfS3
    336LWr9+PR599FG43W5YLBaYTCaYTCbU19fDbDajubn5rHnPjj8T4g5CgkM6IuSV/CyXyxEZ
    GYnY2FjExcXx6ICICBbx9/nnn+PEiRMQi8VFCoXiU4fD4QrV93Iec3PFrxIAT/h8vglbtmyR
    19TUYMCAATCbzaiurobFYuERHZfaZbRYLBeV5JEESqVSISYmBsnJyYiPj8epU6dw4MAB+P1+
    N1hE0nIAf1yqTwpVgiQAO3d0vcFg8C9cuJAZOnSotKu/pYEDBwoXLlzovf/+++nGxsYxAG4D
    20LrEseckJCApUuX0jExMRvq6+unHzt2LGvRokV49tlnkZeXB4FAAJfLha+++goHDhzAxIkT
    kZKSgr1792Lz5s1QKpW12dnZS/fv3190//334+mnn+bfe9KkSRg5ciT1wQcf3Gg0Gp+32+2x
    CQkJePzxx5GTk9NlaxjskLiZnojy8nJhZ5WLO2mjAGQPHDgQAwcOhNfrhd1uh8vlageZEggE
    fIU8+EG58cYb8c0338BisYwBkIGeQS18Xhs3bhzkcrniwIEDpHMUedNNN2HZsmUhp3fmPXZk
    JF5//XXMnz8fe/bsSTCZTEu2b9/OXHvttV9PmzbN/91334VzSQ4BWE/T9CMmk0kQFxeH3Nxc
    5OTkICsrCwkJCYiNjUVMTAy0Wi2PbSZBfcefg1mDSNWKVK6CX30+H1/9qq+vR01NDYqKilBU
    VIQzZ84QPZ9mAJ+AhQN2p/kB/Ax2uHREa2urgCSD4TICd+AgDAyALVqt9iA3QxLO+yeeK9jl
    hOHcWQCeAkvh3GU6Kmq1GkePHvVGRERssdvt010ulyY4MSI2fvx43H777bBYLKisrOR/L5VK
    3UlJSe9dd9113/3000+B4Of95ptvxqZNm5CcnBx/4sSJxU1NTSM6A6U6lxEY+vkSJJqmY0tL
    S0VWqzWcCdIoANkCgYAfxidQb1LNl8lkCAQCcLlc/BB5MPSbY/e6BkAmWOKHHp8c3XrrrUhI
    SFDs2rXrX7W1tQvdbncE6UqvWbOG/3ehKNYxDAO5XA6hUAiHw5FYV1e3ZNu2bYGJEyd+889/
    /tP/5ZdfhmUdbDYbkpOTIRKJ9jU1NW212WwzV65cifr6ehiNRpw+fRomk4mHjV1o9vOygmuB
    gE+aFAoFUlJSkJubC4PBgK+++goMw/g0Gs0X6enpJywWC8rKuqVJGaAo6jDDMA8CsDMMc1dB
    QQFfUOwuI/66tbUVtbW1wSgGYpvBMmlfVuMkVAlSH7CiefJp06a5Jk+eLOlC1q12NmHCBNEt
    t9zifu+99xQMw9zNVWiOd8W5PvjgA9x5551IT08vaGlpea+iouLFzZs3q2pqajBp0iTExcXh
    zz//xIYNG6DRaHDHHXfA7/fjk08+gcVioRMSEr7q37//7ltuuQXr1q3j3/emm27CtGnTqOXL
    l19vNBqX2+32rKSkJDz//POYPHlyVzKY8a1cYl6vV2az2QShFGu9RNODZUCUCwQCrFy5ErW1
    taipqYHFYuHnItxuNwQCAd8dUKlUiIuLQ0pKCo9NtVgs6ZxT6tEJ0tVXX43IyEjZwYMH59bV
    1S30eDyR1157LZ544glERER0ZbsciYmJePLJJ9Ha2oojR44kGI3GJbt27Qrk5eV9M2zYMDqM
    sBY3WOz//1MoFDFE70ilUrVjTbscZ0MS67+yuLi4dmQrbrcbdrsdL7/8Mt544w0EAoFisB2k
    cFkVAEd9fb2msbERaWlplxz8kFeSxJDhWLLXXMqe09zcTMT8XABqCDNeGE3IFcvGhPOxBksn
    /hq6iGX1008/xbx586DT6fY5HI4fjUbj/+uYlKpUKkyaNAkGgwFffPEFKioq+O9Xp9Pt7d27
    98fbtm1znzhxoh3N/WuvvYbc3Ny4ysrKF5qamm7x+/1dogfU1tYGmqZ5KBFhJSXm8XhU9fX1
    go5JX3f7I5FIJJs+fTomTZqExMREJCQkQKVStSvCcAkdD6dqbm5GXV0dKioqsHr1ahw6dCgd
    wLWcP+rRpEFTp05FdHS0ct++ff+qqal5yu12R5C98Pvvv+/y8zscjoS6urqlP/30U2DKlCnf
    zpo1y0+Ssu601tZWLF++HHPnzrWnp6dv9ng8k4uKinTFxcV/mQiReZng4m3w7Cv5HYB2BV8y
    fxN8EMZegnRobW2F0WjEoUOHeMY3sVhcptPpNhcUFNAeT/fVEbjPbwbL3vlPdKDDPyvZ4ORY
    /mo9gkkczrUWwQfpwl0gGQ0A+B2dQJWFIkEixAw5vXr18s6dO1eoVCqF3fUlyeVywezZs6nt
    27e7S0tLe4NlFDqJLoK+PProo3jwwQfp/Pz8T10uV6TJZHrg6NGj+mPHjkEsFoPcoHPmzME1
    11yDdevWYffu3VCpVKejoqI+X7dunWfSpEn8+82dOxf33XcfNXv27LHV1dXL7XZ739TUVLz4
    4ou45ZZbQjoAeFElar9fnJeXJwkEAigrK0N3JbpB1h/AcADYtWsXdu7c2e4hIHMiYrEYgUCA
    p8nsWJEiBW4u2foGQFho+S4q2q2qgkAgyDabzf/yeDyRQqEQFosFS5YsQairtn9VwXM6neQ1
    0WQyPWi327f6/f7mMC9NJYAmmqZjBAIBDysI3qA9Hg8/h3SujdTj8fDVPVKhJkQF59qkSQVb
    IpHw9xH59y6Xi3wfFWDZcMJl1QAsTqdTU1VVhaFDhwIAPwTrcDhgt9t58oqOsMLgaifBgQfD
    O8gake6UTCaDWq1GVFQUtFotlEolr/1C1M45OFEzurBjcgnWF8BduDyNo1CZFMCdAHYCONoV
    J8jJycHPP/+M0aNHt4wfP3653+9PaWhoGBGcJLW1tWH16tU4efIkNm7cGPwcWAwGw8qNGzca
    x48f326f37x5MwYPHhxbUlLynMVimeH3+7tML7Hj/hacpHP+SGIwGIRkriAM/mgAgCuSkpIw
    b948DBky5KL/Y2xsLI/+EAgEzLFjx8Qul2sC2Pm0BvRQu+GGGxAVFaXYt2/ffVVVVU96PJ6I
    /Px85OTkdMv60zSNY8eO4cyZM4m1tbXLvv/+e+aKK65YO3r06AAnOt2tNnfuXAgEAtTX1x/3
    er21AHQMw0AgEECn0yEmJgZxcXFISkrikQ1arfYsiHcw/FskEoGiKH62syPUmxxWqxVmsxl1
    dXWora1FbW1tMAstH0I0NTWZJBIJujNBCrJasLOmUeQZ1mq1INqdiYmJiI+P5xmTFQrFX0Lh
    CaMs8d0d1yUYCm82m2E0GtutS3Nzc3A82IZOojxCsfHFg8XWiidOnNjWv39/eXd/O3l5edIp
    U6Y4XnvtNRlYjYs16CLmpP79+5Mf7bm5ucslEklRU1PTDLvd3lsmk9EpKSnKurq66MbGRsF3
    332HN998Ex6Px5WcnPzpsmXLTm/duhVvvvkmAFb48YknnqCGDx9+TXl5+ctWqzUvPT0dS5cu
    xdSpUy95qDpEVQFJQkKCjNsQuvv0Yi6hiSFDiPHx8cjMzERycjI/eE6YuQj9a1tbG9xuN5qb
    m1FWVoaKigoYjUaYzWb4/f4hAIaAHfzvkUbTNBiGUQYCAUWwgzh27FhYrsftdiv8fr8w1FCB
    y7BmAPUej6fv6dOnUVdXh4aGBpSXl6OyshJmsxkWiwUNDQ3t2HEIq1TH4Vjg7KHY4KRALpfz
    LHYE952eno5evXpBIBCgtLSUVKVKwWophMvqADS53e60X375BUKhEIWFhSgpKWnnQAndd/Bg
    8MUKEwZDFAkVulwu5xOlxMRE9OrVC9nZ2Th69CgZhG1G98MOz7p0AJPA0m6H2zK5gl0Bugj2
    PWrUKCxZsgRPPfVUwZVXXvkwgPlWq/UGt9ttIPDHffv2Yc+ePe2+W61W+0N6evqu5ORk/PTT
    /26NgwcPRlJSUuzRo0efbWhomOnz+SRduUAXuhd9Pp8oLS1NQlEUucfC4o/GjBnD9O3bl/J4
    PGhubobVaoXX6+WJS9ra2iAUCqFSqaBQKCCVSnkdMZ1OhzFjxlA5OTn4448/BoEls/oBPdTK
    y8shEAhyjEbjvW63O2L06NFYtmwZ+vbti+7yCUTU848//kiqq6t7aM+ePdu9Xm/YCnZisRg0
    TTdTFFXPMExuVlYWZs2ahby8PGRkZCAiIoIP/EPB2NzxGSHQzebmZpSWluLIkSNYuXIlampq
    GJ/Pd8blcjm6o5j6VzVesMKrUfHx8bjrrrswaNAgZGVlISoqil+XUMezpEDa1taG1tZWlJeX
    4/jx4/jkk0/AEYw1dzYPCEWCNBhAP4PBQE+ePFkoEom6vcQjEAgEU6ZMEaxatcrd3NycA2Ao
    upha9sCBA3jmmWfsU6dO/WrlypU7/H5/UkZGhn/w4MFRa9asuW/nzp2Tdu/eLXG73f6oqKjN
    w4YN+3rJkiWBxYsX480334Tb7YZMJsP27duvPHPmzPKWlpYB2dnZWLZsGSZPnhzyh+wSTOLx
    eGQURZ0Lz9nV1gusOj3kcjkGDBiA66+/ntdYIfCqjmvj8/ngdrvR0tKCkpISHDp0CFu3biUU
    sgYAk/E3IGtgGIYRi8WYNm0aBgwYgO7c8CiKwqFDh7BlyxYwDMMQXHiYzQmgjGGYa7/77jsc
    OnQINTU1PHsdWR+Kotq17kkSFKzbQ/Q6grtLbreb6K3wbfvgbiQJePR6PaKiokiC5OUcQiCM
    6+IAUEPT9JBVq1ZhzZo1cDgc/PclFouhVquh0WigUqn+spIZPL/0V6QV5HA4HLDZbKisrMSZ
    M/8r40LoaLl1MyG8rHUAS8gwHl0ngn4pJuT2s9Vgu45dYgsXLsScOXNw8ODBo1OmTHmgtLR0
    ZEVFRWZKSgqGDh2a+p///GdKeXl5Erk/9Hp9Se/evT88fvy47eqrr+bf54orrkB+fn70nj17
    FptMpjt9Pp+0m/a98/ojkUgkEwgECAPcNx3AOJFIxDidTuerr76qrKiooCorK2E0Gnl4ICnA
    BMsIEMFqohWWkJDgkUqlXrCQvclgRwFc6IHm9/sRCATUNE0r5HI57rrrLgwbNqxbr2Hs2LGY
    Pn06jh8/DpfLpQAgDOdMIyel4AwEAmcAjE1MTMScOXMuS5syGNp8Lkrsc/lmMusWGRmJrKws
    9OnTBxyREi0UCitomvaEcX2awHZE++h0Otx9993ttM0u9d5jGIbvsF1oXQjCwWAwoFevXujf
    vz+2bt1KEiQLV0wMW4JEAcgDoM3NzfXn5eWFzSnl5uaK8/LyPPv27VNz17Shq4OYF198EWCx
    xI0AGg8ePEgoUYuMRuN2q9WaqNFoarVa7dZ169aZiVZC37590bdvX/Tp0ye7oKDgxZaWlkE5
    OTlYvnw5brzxxnAmRxAIBGKXyyWnKKrdQG83WRyXxDgtFoty48aN2LFjB6KiopCcnAyDwXCW
    DhKhjLTZbDCZTKipqeFVlwNsBN0MVtxMhb8Be5BEIsGUKVNw2223dfu5V6xYgW3btp0FWQyj
    ucDS4tKVlZVCk8kEjUaD6Oho/p4gR2RkJB/wk4FW0hk6l7Awgd4F/9npdKK+vh7V1dW8KB+p
    FtfW1sLv9wMsm1x1mNeljQu4GZqmqeTkZKSmpiIlJQXJycl855UomHckrgh+PRd5RfBrIBDg
    9caMRiOMRiMvWkjWqby8PLiS2BbmtRmC8BAz/JX15gp2FV15kqBArQXAZoZhsHbtWkybNk06
    bty4nc3NzQ/abLZEtVptioqKWrF48eLDx44dw7x58/DJJ5/gk08+gcFgUO7evfux2trau7xe
    b09IjiAQCCQ+n09BURTRHutOiwHg9Pv9jvXr1ysCgQB1qUUrMrQuEokkDMNQXKyg4PxRj0yQ
    yFdDglCbzYaKiopuLdgJBIJ2DIcURTFhgFfyxmnruIVCYRlN03R9fb3QbrcjKioKDMPwsGai
    7dXQ0ACn08l3foJfgyHfpNNItOqCf1YqlTyaQaVSQa1W87qaTU1NaGxshEAgcOl0OhN5/zAW
    MssBjLJYLISwBwzDwOl0wmazweFwwGKxwGw288LrHdeGrAuBfJPkJ3hdyNpERUUhJiaGXxON
    RgOhUMiL6nJWh05C4Tub0KgB9AMgyM3NpQwGgzBc35BOp5MMGDDAv2/fPpK0aXCRYlChMvIA
    z5gxo27WrFkfWywWgdFoDIjFYjz//PP833PteG1dXd3jNpttVGJiIl588UV0xIKHK0Z3u91S
    iqIIY1d32iEAt3NBzlU+n69fa2trXGtrq76kpEQHFj7D49SDh805c3MBQjPY+ZXfwQ4QnuB+
    91/7+5kXAEMYHfPz8xEXFweDwdAOzx3SE3LQmba2NjQ2NsJoNGLXrl1YsWIFXC6Xvwck2iJu
    76VGjx6Nt99+GzExMXwi2BV7CBG9JM8bmV+yWCx47LHHsGnTJnCBnzCM60KBnT/S9KD7V8X5
    yPXoRtpzcg9MmTLFExcX90NkZOR+l8sli4uLc/fp08f68ssvB7Zt2wYAWLp0KfR6veDAgQO3
    O53Oe7uzc3ShBImiKInP55OFyR/9yvmjwX6//yrue4wHoAPbCRKRIkOwEDWpgnMJUDOAZr/f
    XwHgMOePToKtbvdooygKbW1tWLp0KVasWNGtiAKKotDc3Ay/39/tc9jnS9rEYrGzra0tYDab
    hb/88gt+/fVXnDhxAqWlpbxIts1mawdtDi44Xej9OzKwisViviiYkJCArKws9OvXD9XV1XA6
    nRAIBB6FQmEPc1HTA6AeQMDpdAr27dvHw93OnDnDw76tVisPfe9YjLvQvXCudSEJJBnD6Nev
    H3w+H5HhAFgCiU4NZXU2slACSBGJRMjOzqaFQqE4jA8zlZ2d7RSJRHK/35/IOaawcFXfcccd
    fOGh499NmTKF0KdOdjqd06VSKTV37tyekhyBoigxSZDCQPXtAcutXwLgOy4IjBEIBDcGAoEn
    5HJ51B133IGmpiZs3LgRBoMBs2bNatu7d6//zz//1IDluV8OlsWwhaseBPA3MIFAwHQMbsLh
    EP83dmF6wrrJuYBXNHjwYNx11128Fs+Fgi5ykC4IcU4koAlmbes4GE5mNzQaDWJjY9GvXz8k
    JSVhw4YNKC8vN4ClcT4QxnWJAJAnEAhw5ZVX/iUNPHE+JOHzeDzw+Xzt1uRc60JY/oI7cYRd
    iNwn5PcajQbDhw/Hjz/+CL/fnwN2UDdclGNqALmkkNJTHu2ggl1Ld5+cYxxjyLkLCwtRWFjI
    //2oUaPw888/IzExcZDL5fq3z+fTdef1XUSCJCYJUgddoe7yR2XcsZG7vyLBzjk/JRKJ9DNm
    zHC3tbWp161bB4PBgPvuuw+//PILOD2pAgAvca9W7rn4W/ij4O+HdI7/z1fqvF6hz+dLZhhG
    2NzcjEcffZQXDD6XLyUBPZnf7MjW1hHyHczcRoTPGYZBc3NzOzQPmefhEiR1c3NzQhjnj0hh
    LAWAwOl04rnnnkMgEEBbW9tZzzfxtQQGfy5GOwC85Ma52OvIHG1LSwtqa2t5LVGZTAaxWBws
    c5DAxRCX3VoThWBh5DKZDAkJCdJw38DJyclquVwusNvtci5563Hmcrlwyy23xPn9/tk0TWsm
    T56M2bNnd4q6OMQmIglSR42KyzQxV3FTgq0uC8FWetu4BNaBc1OeesB2D+ICgcCVFEXpJ0+e
    jJdeegnvvfceNmzYAKlUittuu006ceJE8Zw5c1BWVpbNBdT7cf5ZCBEALXdIwNLwBrhXa1cm
    VmPHjoVcLlcVFRWlulwuMReYBxoaGrIYhpEEAgHU1NTg9OnT3Q5pMBqNYBgGFEUpNBpNvkwm
    awZAcZhgd0pKShlN096BAwfi7bff7rZEQCgUYsiQIXxyROhOrVYrmpubYTab0dLSAqvVyou8
    kgpe8OYaPCNAZpVkMhk0Gg10Oh10Oh0iIiIQHR0NnU4HrVYLrVYLsViMxMRE9OvXD+Xl5QoA
    g8AyUYWrk5QNIFWn02H48OEAWKrtxsZGWCwWnvGovr4edrudh6DabDae7S94bQC0m9siyRGB
    LhAYQ0xMDM/UFBkZiaioKERGRmLIkCHQarWwWCzJYCUfKsK0LsRR9zRL4Pa/lp52YTqdDv/+
    978V33777Vyv19u7p10fRVFiAvcLkT+SgO3+KLnkVcS9En9k+4u93wu2GxQPYDQA3ciRIy1L
    liyRf/jhh2AYBhKJBBMmTMDo0aMxe/ZsVFZWZgMYCBYZYbvANek4fyQ+hz+ydmdixQX3DHoO
    FTnDWdgfF4ZhBgEQMAwDq9UKmUyGuLg46HQ6REZGtoN8BxMABUPGSHGJQL5J8SqYYIgcjY2N
    qKysRG1tLSwWC1pbW9HS0sITlgQCAanT6cwDy5oZLq2wGK4wBYZhYLfbIZVKERMTA51OB4PB
    gKSkJKSkpCAqKuos+HvHdaEo6qx1CX71er2wWCyorq5GdXU1mpqa+HXpsEdkcP7gsru1nU2Q
    VABkQqEQGo0m7O0PvV4v59qeH96YAAAWn0lEQVSxUu7aepTNnDkTn332GfLy8q5rbm4eFBUV
    hblz5yIuLi6sFaLgjYdhGJHb7RZTFIVOaE9IwTI4jQRLk5oMlixBzN1zQi5YqOOOGrDsV7Xc
    7wPc9zcMwJ0URQ268sorBU899RTkcjkZwIPVakVhYaHwjjvuEC5YsACLFi2KNJvNT3EV27Vg
    2ca8YOl+owEkcUci5+jiuAqDj3NGpFVcBhYKcZi7vpDtzG63G0KhcGBra+tbDocjitv80djY
    KPF6vREMw2D58uV47733uv1eIMxMFEWl2Wy2z+12e4Cr+AhEIlFxTEzMnX6/v7obO1zZAJIV
    CgUMBgO2bt2KP/74A8XFxairq4PJZGrH1ubz+S5bnJRoNBBSB8Jil5ycjJycHPTr1w8xMTEE
    2jmAu5/DVVbNBxCpVCrxxx9/YNeuXTh9+jRKS0vR0tLCs2pd7FpcLCEHEcpUKBTQ6XRIT09H
    Tk4OdDodIWswcNe2NUzBlZw7eprJwXYfepSdOHECN9xwA3w+3zCHwzExHAHohc7JMIzI4/FI
    AMDpvGziSBnYri/xRyncvSrikhMRlxzVcs90bZBPsnC+QQXgCgAzAQwcPHiw9dVXX5XIZDIN
    ITNqbm7GkSNHcN999+GZZ57BokWL9CaT6XEueFwLFhnh4+6H2CB/lMD5o3juWj0d/FE5gP+A
    hYzXhjJZGj9+PCnYpXQo2GUGAgGJRCLBrbfeiv79+3c7adD+/fvx448/AoBCrVbnSaXS5piY
    GFKwc6WlpZXTNO0bM2YMFi5c2C01eLDEHejduzdmzJiB1NRUft6TUFgrFIqQMbb5/X60tbXx
    DHakm1dcXIzPPvuM6M8N4IqJ4fJHudx9jKSkJNx5553IyspCfHw84uLioNfr+XUJFRyepml+
    dil4PrayshKff/456ZIngIXFXrbMQmevVsDey1RYKKk7WhAjCMUdPcoKCgowatQo7cmTJ6d5
    PB7l0KFDL0lXoSusIwaUYRiBz+cTkWD+MiwHwBywMIQUrVYri42NhcFggMFgAEm8TCYTT88c
    CAR8YKvxbq7SRnMOSa9SqeQTJkzAggULkJ+fj/379+PQoUMgDnPLli2YMmUKZs6cCbVajVde
    eUVdUFBwK03TN3JOz4X/HYqVAZBJJBJBREQEYmNjER0dDalUCrfbjaamJlgsFtTX1zMej2c2
    gNMAvgTwdaiqv1zbWOFyuRJdLlekWq3mN41gRrDm5u4fmaIoCnq9HlwiGw+w8zhOpxMikcjm
    8/lEpOPQTdYPQKTL5cLSpUvhcDjI/fK/pVeJhFe2Jyr3er0ecrmcT3gIZI6sLRGZ83q9vOOx
    2+28I6qvryeOBwB4RqqgxDCFC7jC4ZDUYGf0RCaTCc899xzcbjcfaFIUBY1Gg+TkZGi1WhgM
    BkRFRSE6Opqf2yKMfwTeQHQnfD4fvz6ENr+xsZHvTJGOXVNTExoaGnDmzBls3boVUqmUdKIE
    3LVpEB42O/KM9zST9cSC3RtvvIH58+fLVq5ceavdbo8Nx37TMW4Ihn0St+52u8UALrdglwtg
    Llg2wWSuePdXNpjEX0H+yMH9WQG2yyMTiUQBg8Eg27Bhg8xisVAHDx4EAMbj8VCrVq1CbW0t
    fD4f9Ho9TCaTCsA/ANzA+TYPlyApuPtCepGxyt1B/ugrhGimtq2tDQKBYHBTU9PbDocjiqIo
    BgBTX18vdbvdeqVSiUmTJmH69Ondfn+KxWJs3boVfr+/V0tLy9cURQVIjCIWiwtjYmLu8Pv9
    td1YsOvHBd0YP348nnzyyS6Pe0UiETQaDQ/3JrpaDocDJSUlWLduHQCkhtEfCcBqV+oAYMSI
    EViwYAGUyq4FcBGGWZVKhejoaB5mTtM0GhoaSIIk55LHr7lCebcnSDYALpqmQ9X+7tzF2Gwk
    2CeBdo8yu90OnU43oK2tbYhUKsUNN9xwlvhldxuB2pyrqncZFcU4AMsATExLS6MmT56M0aNH
    Iy8vDzqdrh2+tLGxEaWlpSgrK8OZM2fEFRUVYqPRqG5tbY2iaRoajQZ9+vTB5MmTiXAdLBYL
    Pv74Y9TU1EChUBR5PJ7onTt3GjZt2oSZM2fi1ltvRb9+/bBp0yZq165d2rq6Oi2hU4+MjERS
    UhIyMjKQlZXF69solUpekdrj8cBoNOL333+ntm/frtu5c+eVdrs9nwv4XkHohqwZhmEYpVKJ
    hQsXdnt17qJ3PoEAP//8M9566y34fL7uLi+rwELZRH6/HzU1NRAKhfywanp6Onr37o309HTE
    xMTAYDBAr9dDo9FAIpG0mzMir+SeJhhmQrfq9Xphs9nQ3NyMlpYWmEwmlJaWoqioCOXl5air
    q0NTU1PwdxQNtku5LwxfSywX9BE6XkRHRyMrKwt9+/bl1yQ1NRWRkZE8LpskQxfLkEnw8SRp
    IvovlZWVKC8vR3FxMU6dOoXi4mKYzebgIktvLrkOR4LU44piPfnafv31V9TV1aXV1dWN6ubC
    R7sgOPie7OiPGIahaJqmyN9dhj9aClbI/pLiL7AQPCXYynw78/v9gl27dsl37NgR/OujABKO
    Hj0ae/To0Y5zpBTnQzpDHqIB28HqxxVJlofCH3EFO3lbW1u80+mM+KtENiwPDHfeQCAgcjqd
    MR3uG4vP5xN1IzmBhEsE5EqlEsOGDWuXHAUCAbjdbtjtdjgcDp4V1Wq1wmq1wuVy8XspOYLl
    KUgRjxT41Go1T0JEEoFg6muVSoWBAwdiw4YNoGk6irsvwuGPtFxhgRKLxRgyZEi75IjACMm6
    EPIjsi5tbW1nrQvDMLzPImujUCh4TTHCZqdUKqFWq3lxWZI4DR48GAqFgsyGDQTbLb4sUc/O
    JkhOAC6v14vGxkY/wqw90dDQ4PN6vWKweGIHepA999xzWLx4MQQCwQC/3x+l0+kwePDgsF/X
    uRKkzuSAYPnwqdzcXCxYsOAv4YN6vR5ZWVkAwDNiORwONDQ0wOfz8UEvedhsNhtWrFiBDRs2
    gGGYNq/XuzIQCAxpaWm59eWXX0ZMTAyuv/56Pki8//770djYiJaWFn62hGw456v6REdHo3//
    /khLS8Phw4dht9v9YGF2IWegEovFGD58OEaNGtVjozq73Q6RSASv19vdp44iiYDBYMDgwYMx
    ZswYjBgxAunp6dBoNCEV5YuOjj7ruSCK3cXFxfjPf/6DXbt24ciRI3A6nSIueVOi+wVjc8DB
    GbKysjBt2jRcd9116NevH58chio4IQ4qeI0Ik53f74fD4UBhYSH27NmD9evXo6CgAAzDJIKd
    AywMR40MPZM62YXw60OdZZzA4lCfz5ccrmsIDm5IoBnCoNd+uYHRhaxDQcsJ4DOwdO7/vMzi
    4sWaHyz8L9T+iPmr+ODkyZOIiYnpdohdSUnJ+daxuwt2Oi7YRlxcHNLS0lBVVYWysjIUFRWh
    qqqKh303NDTwUiME9k2KcsHFOVKEJMQFwSxtwbOxMTExiI+PR2JiIlJTU5GRkYHExETk5ORA
    o9GgpaVFDGAwRVFKAM5uhsomg4XCQ6vVom/fvqitrUVlZSUKCwtRWVmJuro6GI1GNDQ08JB4
    QsAQvB7BIubBhU3yc/BsLBF0J+uSkpKCjIwMJCcnIy0tDZGRkaiurgaANLCwyLAkSF4ATR6P
    B8XFxWHf8AsLCxkua2xF+AbWzu0hXS6sXbtWNn/+/N4+n09E5irCbUTnJETmAPABgBG//PJL
    723btuGmm25CZWUl6uvreYpHIryp1Wqh0+n4SoBarT4roXI6nTh16hS++uorrFmzBk6nMwBg
    m9/v/xosW92gwsLCjMceewwPPvggxo0bh5SUFOj1egIXa1fN8Pl8/EB/S0sLzGYz3G43xGIx
    VCoVUlJSoFKp8OWXXxKY1Q/oQtXzHjB42lMtE+ysGHQ6HYYOHYqRI0ciKyuL1/bpShMKhVAq
    laAoCmlpaXA4HKisrMSpU6fILEQ+2E5SdxMSZAPQEKfhdDp5KBxxUl1Ji0sqgi0tLWhsbER1
    dTVMJlMwHJcwyXW5Dt1fJCJtPfBebkMn9ThCbUuWLMFTTz0lSEhIyHK73WGb2+ooCBnigh3v
    j8CSh3SJCwU7c7cOLH33YBIwdpF935X+6Fxxy2uvvYZ333232+8Nt9vdkzT50sEK2cPlcmHZ
    smWorq5GZWUl7HY7PJ7Qh5smk+ksn0QSp/T0dBgMhuBnJZ9hmGiKorrbH/UjfpqmaXz44Ycw
    m80oLS2FzWaDx+MJeYwTROPNJ5kkcUpNTUVcXFywP4oFi/b4TzgSJBtYCssb9+/f79m+fXtA
    pVJJujvooygKXq8Xv//+u4SrLBzvaQ7JZrPh1KlTCplMlgiwAmEmkwkikShs8CqBQACTyQSr
    NaTFzWMAPrVarc+/8sorsk2bNuHEiRP87Ahpn2q1WsTExCA2NhYxMTGIjo7mBUClUikvKlZc
    XIz9+/ejqqoKfr/fC5Zu9VmweNsGAPMAvFBYWNj/8ccfx+eff44RI0YgMTERMTExUKlUsFqt
    aGhoQENDAxobG1FfXw+TyYTGxkY4nU4EAgGe4Sw5ORkpKSnYs2cPGIYpAfA+uogu3uPxYPPm
    zSguLuavoScZRVE4cuQIDwfoZqsDS6U9pry8XPPKK69gzZo1yM7ORnp6OtLS0pCRkYGkpCQe
    jkAGQUmX8ELXTKhEyewRITewWCwoKytDWVkZqqurUVpaioqKCrS2tpJn1QrgzzAVYX4BsIlh
    mFFFRUWG4uJirF69GvHx8UhNTQURjU1LS0N0dDTUanW7dSEwu2Dx2OBCCdHu8Pv9cLvd/Jq0
    tLSgoaEBlZWVqKysRFVVFaqqqtDY2BgMr3aCpdr/HeEhafCAFeLsadbc0wp2DocDH3zwgUql
    UmVxAqZh2V86doAJrDPU/gjACwj9fJoHrL7V88D/b+/8fuO4qjj+nR93vPbueDapE9shLbJb
    JAxVQERIiERIUARFQuIlfchjJF4I6lOf/JQ/gDdQHkEg0pcoSqBQEiVRSR4iJUobFdrihNRx
    m7Vlr2V7dmd3vfPzXh7umfXEjRs3Xu9Oyv1II6/XK3t85557ftx7zsFiVh9B7jB3m/+Sw1fv
    9VzZQcGmZxpd12FZFpIk8QHMx3H8lYWFBf38+fM9v5ckSdBqtdBqtR5xnugeG3EcM13Xe3na
    QwPlHgGA67q4cOFCz8eFc95pPLu8vPyZR0gOHIMskNJTBymGrPRVv337tnP8+PGuN238IoNE
    iroJ4F08ZVLWbrG4uIhms2lGUVTSNA2VSgUnT57sexO0KIrSrchuEUMmxf14ZmbmJzMzM0uQ
    UfY5WtgLAA6srKyMzc7OOqCz2bqum+k5XE3TOsniFCGJAXwEmZx6BhvJiDGAtyEbgr2xvr7+
    6s2bN51bt251zq6mjfvSKmdCiDY59h4ZLgtkVGkARldWVl68c+fOBMnGH2l+74JtoGntdhun
    T5/u2jGx3VqUoygCY6zXhU8+AvArAD8A8KMgCL5dqVReqFQqI1evXmWmaXa6jdu23SnJ7TgO
    SqUSGGOP9FpgjHUanKbb+1EUwfd9eJ6HWq2GWq0Gz/Me6exNwZ6QjJ95yFyDK+S8rfThkdyE
    TDr/HoDvCyG+43neS57njd29e7cIyKh8WozBsiwUi8XOuKQysbnvRDbfKJW9RqPROT+fFrVo
    t9vZgI5P4/KQglI3IEvsz/fJQWqQQfxz5KcXkqAgYq5yYufn59FsNgc4531LghVC4OLFi3Bd
    N20vAM/zcP/+/a4uYaSPXoEs1LAAWRXuE9JHA5CJ92PYKLVtQ+acbKk2ab6fod9dzfytSyQT
    b0AWKio/YW6kxy89yIp5WX20H3Ln4kUy8HZLHym2YGRkBJxzMMb+5fv+L8Mw/IUQ4me6rk9F
    UbQ3CIK+rDOGYURCiKplWXO2bf/Ttu2/rK6uzjLGetlQWVCAwADwGuTx7zLykW/ZhqxG/A8A
    f34a56gbDhIgI6m3OedTruv2O9Nco8XvTt4EjZp6xUEQtACg3W6LbMO+nKB1ybCZB/BbMnL/
    Bnn0oEUKRCflU4LMMxkFMMY5P+D7/gu+7z9P7w1CHl+Yh0w+/Ctk2e7kMUJ6G8BJyP4Urwoh
    vhmGYTkMw4SMprRsa4UifVXIM6lrJEjpPjWDTMp9hYT9zacVrM/xjKBpWs0wjPcNw3gujuNc
    Nw6kSlO6YRizmqb5aTCiRywBOAvgLVp4J+m5fCuO40nP8/Z7njdMlaJKkJWhjMxc3u4inzrb
    LQqwtMhoqUJGbf8N4B4ZVXk4vrsC4O9kjJXIwJuCzP35ahzHY41GYx8FH4r0mQE82ocMm8ZJ
    bBoTTrIWZsalSX97CcCnkFW17pGD5OVgXBIKjtUgE3PzQIPuKcqTXK+uriIIAi0Mw77lDQsh
    cO3aNVy/fv0z72fWn06l3B2wQProA1pLZjLrfqqPipBNYEfJWTqAjTLc4xl9VCF99BYZYPwx
    68l7AH5NwZ2fQh453UPz09tCH1UhK6X6mbmS9hD8If2ONzO6ajd0/7NETxyTzI4En5qa+o+u
    6/cKhcLvR0ZGXp6bm/takiTfdV33G/V6vcwY26Npmh1F0UAcx511Njt3tzpdtXl+0+cEYywx
    TTMQQjSiKForFApLnPP3TNP8gDF2Vwgxd/To0drDhw/jBw8e9OM5LAH4HTlKL9N1mPT0AdJB
    1i7OL0Eykea/36cg2YcUmJrDDjZLurE4zgN4Hfnp89CihStXNJtNWJblBUHwm+Hh4cuapiFJ
    Ep4kieCc56EJWtrsLjAM40MAO93ZuAQZaQ+38O7rpLiyC7RFSqFACkmQQmnhyUmpa5B5D2/T
    XCyS4Kxjo+nsk5SLT4L2B7qXru9CkoP0rmEYr5mmqeftWN2WC4VpJgAamqb140ioTwvxEmR/
    KosM/gIZHvvIuBmhZ5/+bCDzOo0IB4+5fDKqVzJXDRulfuOcPpaY7rNGwYjzm2RoD43Jc+Qk
    WZmxy14azfXsmEQZ52iVxmSN5Cn9WZLDMXkfsjHn19H7HKjHGXEPKIiYx4Bdk3N+zbbtwW0E
    xnZFQX2ewSiE0IrFYp0xtgZgp3mHlwFc3cJRTaveLpITleojtmmtEaQfWtuYWy4F9S6RLhoi
    efVJdrajjwIKSpzZLX1E1CGDynuRn+awT5Krj3sdkKGAdgJgWQjxDoB3Tp069adz584Nlkql
    8ujo6PO+7+9fWlqyXdctO44zUSgU9uu6PiSEKHDOrSRJWBiGRhiGOgBYlsXpShhjsWEYMee8
    tb6+vuC67uzw8LA7Pj7etCyruri4WOGcrx08eLDlOE585coVAOjLsbZNcGz0tLxMsrIPMo94
    PBN0GKeve+kzBjb6YhqZ79NxTkhGsq9Dkq1qxiZYJtn9mL5f75ZuetaiBgqFQrHdtU3PfNVJ
    +ae7I9mv/8/oGQP4yzAWBmSfqlIO/h+NlPUneXOyJyYmUgdlUNf1wS/ivPSSwcFBPjk52QIQ
    DQ0N4ezZs2pl6yJHjhyBbdvsxo0bpUaj8czYg4yx5PDhw40oivixY8cwPT2dm3ubnp6GaZqo
    VqvM930WBIHu+77ebrf1ZrOp1et11Go1DQDK5bIol8twHEc4jiNs2xaMMe44Tnjo0KGw3W7j
    xIkTX4apZmYCDmkJ/TRgxza91rARhEuvbMCulXkvVjpcoVAoFAqFQqFQKBQKhUKhUCgUCoVC
    oVAoFAqFQqFQKBQKhUKhUCgUCoVCoVAoFAqFQqFQKBQKhaLP/A/1jUrIk8gMMgAAAABJRU5E
    rkJggg==
  }
  set pieceImageData(Merida1,75) {
    iVBORw0KGgoAAAANSUhEUgAAA4QAAABKCAYAAADjRJpiAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCwwvSdRBQwAAIABJREFUeNrsXXd4FFX3
    fmdmW3Y3PSGVhJBCgNAJEDpSQzEgqBSVKuBPQFT8EAREQEQFUfRTsaICFgRBuvBRlKp0Qggk
    hPROdrMpu9l2fn/MbNiEbBKkbWDf55lH2czdmbl75572nnMABxxwwAEHHHDAAQcccMABBxxw
    wAEHHHDAAQcccMABBxxwwAEHHHDAAQcccMABBxxwwAEHHHDAAQcccMABBxxwwAEHHHDAAQcc
    cMABBxxwwAEHHHDAAQcccMABBxxwwAEHHHDAAQcccKBhg3FMwf3Dhx9+iKKiIoSFhXHDhg0z
    ubm5gWEcP4EDDjRAcADCALQA4AogG8AFALmOqbGJIACtAFwCkOqYDgcccMCBBg9XAI0BuAvy
    LwtAuWNaagULwOwwCB9hjBkzBu3atWP3798/XK1WKwICAg73798/64cffjAdP37cMUGP6Ev4
    bzF9+nSIxWKoVCqpTqcT2TqPiBiRSGR2d3fXATCLxWI4OTkhJycHIpGIKS0tlZnNZo5hGKpl
    nzByHKcjIvzyyy8AgHHjxgGASK/XS81mc43ODSICwzBQKBT6Vq1aGTIzM/Hhhx/a4z5oWRtU
    j/NdAIwFMBVACAAJgFIAfwNYBeAvAKZ6rkemnuc2dIgBLAMwBcDXAN4AYHBsSY8enn/+eSgU
    CkatVoeUl5dre/funafRaMyvv/66Y3Lu315nckyFA3cIEYCeAF4E0AaADEAJgEMAPgGQUE95
    +qghGkA/AH8COGpvP6g9eRnCwXuRFYKHIR1AEgD1w7AKsrOzMWjQIKSnpz+WnZ09MTU19WJi
    YuLvnp6eW+bOnZukUqlMX3zxhc3xs2bNQuPGjdmTJ0+GlZSUaPfu3ZthUbgdsAkFgPbgIxNu
    ANIAnBLWVYM2DouKiqBQKJxSUlJm5efn9zKbzWTLIFQqlYXNmzdfASBBoVDA3d0d2dnZUCgU
    vleuXFmg1+ubMgxjaz44juOOcxz3HhFpLR+mpKSA47hehYWFL1ZUVMhqMiiJCCKRyBwYGPhL
    165dN+r1entSRGQA2gLoAMAXvHfzNICzACpq2TOfBbCUZVl3FxcXODk5QaPRyMvKyoYCCAQw
    o46N3hNADICWAJwApAA4ASD5YXJYVINc2N89hP8qHpZ93YHbQ2ZmJkaMGMGsXbt2Ul5eXs8L
    Fy5s8/Pz+2P27NlXPvzwQ/3PP/+Mp59+2ub4lJQUhISEYOHCheLMzEyzk5OT6bPPPnNMbO1w
    BtARQGsrOfg3gCsN3TicOXMmRCIRSktLJXq9vladluM4cnZ2roDgGHVxcUFeXh5EIhGj0Wik
    ZrOZq8WIYRiGMbIsWwEA33zzDQBg6tSpAMAZDAap2Vz79u3k5GSIjIw05OTk4N1337UnJ0EQ
    gGYAvIR9+aogl2p7oFgAKwFEVPs8QviulwGcr2W8VDg3VJCD2QAuA8h/iN9DKXhH8hQAGwCc
    A1DmMAhvQgygB4DxALoDCBA+MwoL5CiAdYI1rW/IK6GsrAwqlQo6nc5cVlYm12g0nVUqVcf8
    /PwnCgsLvwkKCtr67bff5ly4cIFWr15dZezYsWOhVCrZ3377rX96evpCsVisGj58+JsAzixd
    uhQLFy50iLxb4Q9gpqDA+wsbn0FQ+lcC2N6Q11R5eTlEIpG4sLAwJi0tLdZgsB1w8fDwKAwO
    Dl7HMEyCSCQCx3EoLi4GwzAuGRkZj2m12kgisiUFIRKJDGKxWGR9zo0bNyASiUIyMjIGl5WV
    SW1dWyKRmJVK5VWxWPyzyWSyF+XDFcA0ANMBBIP3mhN4KuOnAL4AoKlhXCSAaQzDuA8YMADT
    pk2Dt7c3EhISsHr1aly+fLktgBcAxAMormF8MwCvAxgh3AOENXkKwLsAdgp738MGa6+Vw2ts
    Gy6CwRwIQCkoC5mCs+ChMKBVKpXlv265ubk9WJbtmp+fPykjI+Pn3r17/5SXl3dtwoQJpnXr
    1tXk3ELnzp3h5eXlk5WVNclkMt0YNmzYjzKZrESr1TqcozUjAMBsQQ76CJ8ZBWV9JYDfanGA
    2T1yc3Ph5OQku379+tTCwsKugmO0pj2GVSgUBZGRkWsAJCmVSri7uyMzMxNKpdLv4sWLr1RU
    VATWYgRxHMed4jjuYyLSWT5MSEgAx3Hdi4qKJun1eomNazMcx5kCAgK2tW/ffktZWZm9yEEJ
    gMGCntRO2H/KBPn1X2FtaGsY11gYE1HTPAPoDeD/AMwBHzWsDi8AkwE8B6CJcB83BD3/IwDH
    8XA6R2XgnaIAT7F1chiEVY3BMQAWAAhjGIZxcXEhNzc3RqPRSNRqdRMiagLem75MsKgbrALP
    sixCQ0MJgA4AMbz04m7cuNGhrKwsUqVSDcrLy1vTq1evg9YC8ejRo+jWrRu6d+/eMykp6f3C
    wsJWIpEIZrPZpW/fvq8eOHDg1Lx58/DOO+84RN9NyIUNabZUKpWFhYVBqVQiJydHnJGR0YWI
    lgsK+/6G/qBERCaTCe7u7ujQoQPEYjGICCzLIiMjA/Hx8SAis0Whqg6z2WwmIoSFhSE0NBQs
    y/Iag9GIixcvIjc3F0QEIoK1B1T4jMxmMyQSCdq2bQtPT89KmmhZWRlOnToFvV4PsmVtPhhw
    AMYBmMdxnFtwcDC8vLxQWFjIpKWlhZhMpnmCEPuyBqHUHEC4t7c3XnzxRQwdOhQA0LlzZ+Tn
    52PhwoUgoraCQl/dIPQEMA/As05OTmxoaChkMhnS09PF+fn5MQDeBlAA4NhD8g56gfc8cxKJ
    xNVkMnkJ/gB38FH7EmF+MwHkPeL7FQegK3jHaE/wEWtOmJ9c8I7R7wAcQQOn2lZUVECn08Fg
    MOiJCHq9nisqKmpRWlo6v7i4eEBBQcG3ISEhW9q1a6c6e/ZslbGtWrVCUFCQd2Ji4qLc3NyJ
    IpFI98cff/hOmzZtzeTJkx0R51uhAM9YmAk+OmGt+3UQ9CoVgL0N9QFLSkoAQJqXl9cvLS1t
    mNFotJm+4O7unhcUFPQrgCSxWAyGYVBcXAyWZT3T0tLiSktLw2pzKnAcJxeJRJ9bG4S5ubkQ
    iUTNsrKyRpeXl0tsXVsikZhlMlkux3HbDAaDvRiE/cCnOTSt5pTqKhhqJgC/1GDkNgcfca7N
    AdhD+I6LNRhF/wfgNcHpZYEPgCcFY3M6ao8uNkj9n2EYMplMlrm05bh4ZA3C/gDeAtDEz8/P
    OG7cOKZ79+6Mp6cno1Kp6MiRI7RhwwZkZWU1BbAYvIf0t4a6ILy9vTF06FB67bXXVACMHh4e
    4lGjRuGff/7B+fPnFfn5+cO0Wm1zrVb7QYcOHX4cMmSIulOnTpg1axb69esXevHixYWFhYWt
    nJycAACZmZk9jEbj0qFDh76wY8eOVIfsq6o7ABgnkUhkU6dOxdSpU+Hu7o6EhAS89dZbOHr0
    aASAieBpM5qG/rBmsxnNmzfHF198ATc3NxAROI7DDz/8gFdffdViCJoBwNnZGS1btsSOHTtg
    oYmKRCKMHDkSr7zyCsRiMQBAo9Fg9uzZ2Lp1q+UaNdp1RARXV1csWLAAPXv2hMlkAsMwuHbt
    GkaPHo3U1Pu6NBlB8dHX4mFsCmA8wzBuQ4YMwX/+8x8EBQUhPT0dK1aswM6dOz2IaCKAfeBp
    M9ZwByBxcXGBr6/vzY1UJEJgYCA4joPRaHQRhGp19AIwTC6Xs7NmzcKzzz4LhUKBEydOYNGi
    Rbh69WpL8B7TU7U4vkSCoWDvHn0ZgLkARoGnU3EMw3gKf+sM4HvczNncIZxb+ojuVSIATwFY
    CD6CXF2jDANPq+oGYHlDd4wqlUrMnDmTPvzwQ42gcHIAoNfrpYWFhd1KS0uj1Gp1l4iIiFUq
    lSpx/vz5+OyzzzB69GjI5XL5X3/99VJ2dvYknU4nA+B0+fLlVzQajSk6OnpVv379dPv374cD
    VeTg2GrGoDVCAUwAT1kvbqAOURARTCYTGY1G1EZCMZlMlTLM+r9msxlGo5HqQ2BhWbaKY9Rs
    NsNsNtd5baPRCFu59g8IvoJh1tTG3/3B5weexK1FwBrZkHHW8BSO6ugk6F5KG+M6AZgkGIz6
    Bv7+McKe3tlsNkslEomS47hmer0e4JlJ4wW5ZwJPHz2LBxgZfZAGoT+AlwA0CQkJ0b/77rvm
    YcOGSWUymeVtYQYMGMB06tTJNG/ePHNSUlIQeNrDGfD89waHwMBAMAyDli1b5hKR0Ww2ix97
    7DHMmDEDX331FX755Rfk5OSE6fX65UajsUVQUNA7O3bsyB44cKDk119/nXzjxo2eCoUCL730
    EsxmMz7++GMmPz+/79WrV59/8sknl3To0KGiJprNI4rmABqHhIRg4sSJiIqKAgAEBAQgISEB
    x48fh9lsbiFsipqH4YHFYjHc3d3h5uZW+ZlCoQDDMCAizmg0+kilUv+cnBzu+++/R0BAAJWV
    lQUwDCMGACcnJ3h4eEAk4rcFjuMglUot/+/Uvn37AF9f3zKTycQAgFQqNV6+fNmdiBiWZeHi
    4gJXV9fKa7u5uVVGG++HngngMcEr6QM+0nYEfAS4OmWlCYBQFxcXjBs3Dt26dQMANG7cGM88
    8wwOHjyIsrKyJuDpMNUNwjwA2hs3biiSk5PRsSPvJK2oqEBiYqJFISgSjip6hKCcuYaGhmLi
    xImIiODZNn5+fjh27BiuXr0K8FVLvcFXaqsuvIeAj6wpBAG9VzAe7TFixAm/g5egdLFEJBb+
    JgVPmzFbPRv3CO9VAwAsrUUxsygWYYIDVQNgCxoo9bZRo0ZgGIaaNWtWCJ66yFkr91qt1jU3
    N3eiwWBoNmjQoOVRUVH7YmNjTT/99BNat249KDs7e6pgDAIAysvLXXNzc2dcuXIl4fTp0799
    /fXXmDx5skMC8ogATxmtDS0A+DVUg/Df2JAWJ57FuS4YafV6n+yL7FLjvttcMKoCwefjnQJf
    Abu6cdUUfDGY2hAprI/qBmExeKabopaxJaiZLtqujjXJgo8++tWg68sFGdgOfMrFdcGZkWKn
    +6EMwHzwkU/o9XqGYRiJ1Xv3tnDfLICD4GndhY+iQdgNQGeJRGKaNm2a4YknnpBzHFfFdSKV
    SjF8+HAuNTXVsGDBAkan03UAz03+riHuQkFBQRCJRNDr9SkSiURbUlLilJqaiqeeegrLly9H
    r169sHTpUpw9e9YtMzNzmslkcm7duvW8ixcvNi4oKHjKaDSKBg8ejFmzZoHjOGRnZ+OHH34Q
    5+bmjt+7d+8Fk8l0uG/fvkalUlm2cuVKbXZ2Nnr16vWoCkI3AKxcLoezs3OVP7i6ulq8fM7g
    k+0fDilnI3pnUZouXLiwjOM4DRExFiFoNptlRqOxseXc6t9h+TfDMNHp6ek/Z2Zmmq0+I51O
    56XX68U1Xf8+Ck5v8LkK43EzRwbgo23rALwnGIjWm7TIUlTAGs7OzuA4DuBzGpxquNYlABdV
    KlWXDz74AAUFBfD19cXFixexfv16CzX2GHgaZHVBLQfAODk5QS6XVzHkre5DJhzWaAngTfC5
    HhYBbAYwGjzd53vYX8SwDMASAFsBSDiOUwJ4wWQytQdPIVorKBUG4d+aR3Sf8gVP52taz/Mb
    A5glKHkN0jHq7e1tcaKkMgxTgRqiVyaTiSsoKOhhMBj+q9Vq5y9atGiTUqn0+euvv14oLy/3
    rn5+aWmpb2Zm5ktPPfVU/OTJk5NKSkpu2fcfZTlYxzkPlRysAywRubm7u3vo9Xr29OnTCAkJ
    MavVaneGYerUh1mWlURHR3uEh4dL9Ho9AwAuLi6mQ4cOKe3g2ZzAp2DNAp+HLBOMwAwAnwP4
    qto+q6zDoLMYYO41fJ4EvvBMu1rGxoMvDFldDnrUY0264tYIZCPw9Ofxwr4pAl988iyA9wHs
    hv3l35vA539nCDoFS0Rewm9lAE/XNgpH8oOW4w/KIOQEz4RLcHBwxZAhQ7jqxmDliRyHIUOG
    cOvWraP4+Hi5sAB/RAMMJb/xxhsYNGgQRCLRjeLiYnV+fr5HamoqDAYD5HI5hg8fDh8fHyxa
    tAiHDh0S5+bmjpXJZKyzs3N5aWlpsL+/PyZPnoxGjRoBAF544QUcO3YMSUlJAfHx8Wvkcnle
    bm6uSqfTXe7atWtCcHBwaosWLc6Ghobmbd++Xf/qq69i1apVj4ogzAVQkZWVJbtw4QJCQ0PB
    MAw0Gg3Onj0Lo9EI4WVUPQqTYTAYRNnZ2WE2NwKRqFaDUqfTuWRlZbW5HWP0Pu5h4wG8KJFI
    FM2bN4e/vz9yc3Nx+fJlL51O93/go3UfWG22eQBUGo3G9dixY+jWrRsUCgXKyspw4sQJlJeX
    QxhT3agDwzApRPQJgMB//vkn8OLFi5BIJNBqtTAYDGbwLSfW4tZEcYOgwBsyMjKkp0+fho+P
    DziOQ1paGk6dOmU5L7ua8eoFPu9wlKurK9OyZUs4OzsjOTmZvX79ejOz2bwAfDRxlx0uu2Th
    gMlkchMM2vaCkvAzbo2iPoroCp5CeztoD55+/H1DfGC5XI42bdpAJBLlFBcXl6tUKhdbe4pa
    rW5KRCvee+89KBQKZUlJSWdb+5RKper6999/rw4ICDjUvn374s6dO2cbDIZENze3G927d9dc
    uHDBbKG/P0LIFfY9WS3n3BCOhx5ardbt4sWLS1iWVcGKmm02mxUGg8GvTsWV4zpmZmauz87O
    tuaGUnl5ub/FMfoAMRQ808Df6jOZYBwuAB/R+8LKaFILh3st36lBzfndyeDbBy21MT5ZMEIL
    ajCQclGNGVADCqutSSfw9NVXBSPV2qjtgZvVUf+ysyWnB7AawHoALMMwzkS0Ajwr5BSA/whz
    RMLzljyKBqEEvFefCQ4OlgUEBNSqTfr5+bEBAQGIj4+3eAlkDdEgnDFjBvR6PZRKZfY///xz
    GUDTa9euoaioCD4+PmAYBl27dsXHH3+M+fPnY/v27eLU1NSxYrFYbzAYRDExMYiJian8vnbt
    2iEuLg4ffPABdDpdI51O10j4U0+RSISysjIVy7JXcnJyDvn5+R08derUma1bt95IS0ujl156
    6WHf+88AOJ+fn995yZIlSEhIgLe3N86fP2/po2cAcEBQwB9aNGvWDJMnT4bBYKg1d4FlWXTs
    2LEKxVMikaB///5wc3Ordawlh9Df3/9BPGIwgDEMwyhGjBiBhQsXwsfHB4WFhVixYgXWr18v
    N5lM48HnHicKY64AOKDX6yeuXbuWyczMRNOmTZGSkoIdO3bAaDSawdM3Emt4VjN4up4OwFSd
    TtdRp9M5A8gB8Af4ymwXbNzrEQAJOTk57d544w389ddfcHZ2xsmTJ3Hw4EEIwmA3qnpxewIY
    rFQqmf/85z949tln4eTkhISEBMyZMwf//PNPEPgcoaOwf8qXowRktdcON6lPtwNLK52fGqIc
    vHbtGqKioiCRSAozMzMzwHv7be4txcXFTa5du7ZSLpdrtFqtzUiMXq8Xp6amDmFZdohEIjFl
    ZWWpGYbJdnZ2TomPjz/s5uZ2MiYm5vLAgQPV2dnZVFuLp4dMDp4RHA81+grB0+qzHoXJ0Ol0
    kuvXr7f5t+O1Wq3H9evXu9vho/mCb2VgSwi7gy/Ush98ZA8AroGvoRBSy/eeA99PsDqMFiMH
    fBuFpoJerxXW22pBv6oJ/4CneDavZU3+Vc0QbQfe8Su3MaY5+LzE87A/tkkldZaIXKzurxR8
    pLXgUReEUvCNK6l79+6UlZVFtSEvL4969+5tqcjzHWwno9o9nnzySQBAixYtlnAcZ2rcuDH9
    +eeflc9aUlJCe/bsoaFDhxLHcZZnJrFYTB9++OEtc7Nr1y7y9PQkjuOoW7duNHjwYGratCnJ
    ZDIS+sIRwzBmqVSa4+vruzUqKmrM2LFjPQE87FVJWQAjwfe1IYZhrOdTB+BXwXPWYDF06FCM
    HTvWJTw8/DcA1LNnT1KpVFXWh5DsXq9DKBpTBUKifr2O6rh69SqFh4cTx3Gmtm3brvzmm28k
    06ZNu9vT0BdAtlwup19++aXK9bdv306enp4keA6HVBvXFnzRGL1lfQhrowLAHvD5fnWhPfgy
    2SrwnleveqzJsYIQMFW7bjGAj3GzJLUFbwIwRUZGUkJCQuWzGY1GmjdvnmX8GfB5kfYMN/D0
    UQLwO2r3Sj8qcAIfTaZ/cWxE3UUd7BJLlizBtGnT8Mknn4gbN278pbWcs3WIRCISiUR1nsey
    LHl6epKLiwuxLGs93uDk5HS9UaNGv7Zs2fK5gQMH+p08eZJZv379o+B0GA6elm2qNl9a8Gyr
    0Ib8gAMHDsSoUaNcQ0NDt9VnLT2oQywW30s52BF8Pl1t91AM4PFq47rjZiGT6udfBjCojuuK
    hO/4A3yE6wfwRVTqGjMDNyNj1ocBPNulOptphqC31fZ85xrAWnYRdE8C7/z1trfN4kGgAjyn
    1piWllZnFcK0tDRKS0uDsGgzUHNflAaBTZs2QSKRQCKR/CkSiYqzs7Oxb98+mEwmqFQqvP32
    25g0aRJ27twJ4GZREJlMhsjIyFu+r0WLFvD39wfDMBg0aBC+/vprbNy4EZ988gmef/55tGzZ
    EizLMhUVFb55eXlxKSkpn5w8efKjMWPG9GzVqpXo008/fVgFoRnANoVC8ZFEIikVqpBZ/nYW
    wBK5XJ70MD1wTVE8hmHAcVy9jprGsyxb7/H1uZ97ADEAjmVZyGRVWVFSqdQS8WQE76U1zoPP
    3XoPwCEiSgGfgL8PfMXLi/WZcvARSjfwCfxMPdbkJvD0nUQAeUR0CTerbC7BrTRKCQBGLBZX
    mWOWZSGRSCxzbKk8au/vYx54KlAeHs4eU7cLiwL0b2BoqHO4aNEiaDQazJgxw+Dj43OE47hy
    W+fK5XL06dMH//nPf9CpU6c6v1sikWDChAn45ptvMHfuXAwYMABNmjQBAJFWq21SUFAwMiUl
    Zc2ZM2e+eeWVV55UqVQuc+bMeZjXmBl8v93VuLWP5XnwVWuvOV7FBg8X2K4ka22IVXfEHRfk
    4AbwzJkCYY8+Dp6eWVfJXqMwxhe8M9MTdafhGMHn9i8EH620yIQL4Bk2r0FINbCCO+pmNMrr
    MQf28k7CykFjN3iQRWVOAsjLyckJ2LhxI5o3bw5391udxhqNhjZu3GjKyMgQCR6IE8JENkj0
    6tULzs7O8vj4+J5ms1luMpmwf/9+TJo0CQaDAXv37kVhYSHatm2LuLg4iMVirFq1Ch4eHggI
    uLUwk1KpRKNGjXDx4kVoNBp4eHjA19cXnTp1wtixY5GcnIytW7di8+bNSEhIQHl5uUdaWtpY
    rVYbk5aWtjYyMvJbLy+vgmnTpuHtt99+2DbJsPLy8gFE5OTi4gIvLy+kpaXBZDKFAxhYXl6e
    goeo1L3BYIBazcv8B10JjWVZqNVq3Ic+9OkAisrLyxv98ccfiImJgaenJ4qLi7Fv3z7LfBSD
    955WV8YTwVdtbAS+Wfxo8EU76mtchQrGIMBTZrxQN/3DIAhNb/B5hh+Bz6crs7GvXQVQnpmZ
    qThy5AiCgoIglUqRlJSEY8eOWcqfX4f90kWl4Olqj4OPqJrA03+Wgo8U/oUG3BT7DqEDn6dq
    vE1ZbOnd2GAdo5bCMiUlJWqWZWs0ips1a4YXXngBTz75JGQyGZKS6vbfERH8/f0xfPhwPP74
    41Cr1bh+/ToOHjyITZs24eLFi9Bqta46nW6QwWCI/vbbb7dGRUV9pNFo4g8fPkzDhg17GNdZ
    M+H9q05NjhA+T8OjW9TpYUGu8BvWlgepxa1FXkzgUxniwbNMpgF4Arxz07I31YUAq+uGCbIt
    v44xpcI9i4R9cC34fodZqLlR+3XUXdU0387XMSsYzTcEJ0yB8G817KQYzoM0CE8D2GE0Gp9f
    v34926RJE8ycObOyzD0A6PV6+uabb4zffvutyGg0Evgy6w22afPrr78OHx8fybp166YXFBS8
    ZDAYpABw6dIlHD16FCNHjsSiRYtQXFyMbt26ISQkBGq1Gp6ennBxcUFo6K3RcJFIVGlIl5SU
    WFeFhJOTE1q1aoXmzZtj1KhR2LZtG3788UdcunSJyc7ObqpSqRZnZmZ2CAsLWz5mzJjzKpUK
    D1HEMBLAu0Q01NXVlX355ZfRv39/LF++HLt37/Y0m81zhfM+tbEBNSiwLIvExES88MILlY3p
    HyQYhkFpaSlyc3PvdaQwDcBes9kc/t1333EpKSkIDAw0ZWdnc0eOHIHQAHgHbHvBDcIGTYJB
    5w6ehnq2Ht67CNzMafASBOPlOsZ4ABgmCE05+CpptQmxvwD8o1Kpei9evBj79++Hi4sL4uPj
    8ffff0MQJr/DPotCeAN4XjiCcTOC6gO+KXYcgC8FZeBRzaM4Cd47HnAbY4rQgB2jY8aMwZo1
    a9CtW7foK1euzK+oqLglhzIqKgpr1qxBz549wXEcUlNTK51ddRmE5eXlICKIxWJ4e3vD29sb
    HTt2xBNPPIFt27Zh06ZNuHDhAtRqtWd8fPzE/Pz89oMGDVrl6+v7W/fu3cv/+usve+oVd6do
    AeBd8AWdqjPC3MFXZyYAa/AQOEeJCF5eXujUqVNlP90HjeLiYpw8eRIGwz3tDpQpyIoI2Gaq
    nETN+YAWORIPPrrXSJBTj8F2Prw1wq0MNQ9hL7tUxxgnACPAO1L14Gs5XK3l/L+F+7NVgEsP
    nt2Tb6dL0xvA0+Dp280EfWEQeKfyDvD5mA91PYv6eq52AzCGhobS0aNHq+QAnTx5Utu0aVMd
    eI/oIdTdM8Vu0bdvXxARWrduPcTFxSWNYRgKDw+nJk2aEACaMGEClZSU1JhDWVNulwV6vZ7e
    fvttCg8Pp88++6zGXC7rvKNz587RM888QwqFggAQx3FmHx+fPx977LEYIsKWLVsehnUVAb7w
    h8nV1ZXefPNNKi4uJiKiixcv0uDBgy35JYUAXkbN7QXsHkIOoWt4ePjWBpA7seoe5U5YjP+t
    gge0eo7M78Lf6xJO/7Uatw91c/slAL6xGlMOvsmg4bAnAAAgAElEQVRvXXgMfAEaEhwRz9Vj
    zEDweYLGas93A8BK2Gc+ngx8FFArlUqpY8eONGHCBJo9ezY9/fTTFBoaankHtcJ5skdUBroI
    RrEJt5c/6NkQHzY3NxdhYWEYOXJkSHBw8B5b+9agQYMoPz+fMjMz6dtvv6W4uDhycXGpc24Y
    hqGYmBj66quvKCkpifR6/S0yMDk5mZYtW1YpexmGIaVSWRAaGjp/6tSp7vWhpjYgp+jWeqyt
    IvA0PXlDfEjrHEIIufSZmZlUUlJCxcXFD/QoKSmhY8eOUXBw8L3OpYfgZDtSg5ywND3vWw+Z
    9p7VuN24Nae9OlhBBlnL3Bn1uNc2ggFIgjE3tR7XeRo8lbR6vqNeWOdN7fg9XA++sAxJpVJy
    cXEhsVhsrTv8ivrVLbineJARQjfwvW8uAOiamprqkpiYiK5dbxbDSkxMFKelpTHCIjsvKG4e
    ghejQTXlTU9PR3R0dEhmZubLGo0mKDQ0FB988AFSU1Px7rvvQqFQ2Izq1OatFIvFmD59Oh5/
    /HEEBwfXmMtlAcdxaNOmDVatWoWWLVvis88+Q3p6OlNQUNDj0qVLHwwaNGjunj17/srMzKTA
    wMCGKBsYAFHgaYCPe3h4sLNnz8asWbMq+7xFRUVhxYoVYFkWe/bs8TQaja8L46r36LF7KBQK
    eHp66sVicYpIJKowmUxSy5qIiYlBixYt7mukUOhriFOnTuHcuXMWOiPEYnEJx3HXmzRpYjpz
    5sy9uHQigJcAjALQRfC+GQVD6SzqrqDHoKr3vK0gXPfUMsYTQGORSASpVIqysjInwVPKwnZu
    lwhAf8EDW9N1bSFZ8I5amgFLBI/wH4J30R7pogMATFIoFLLnn38eL7zwAgICAipbdJw7dw7v
    v/8+du/eLTOZTJPAV577/RE0CDUAVjMM05iI+texHixtTd5HA20TMGLECLRt21bx999/T8/J
    yelji1J+4sQJzJo1CwUFBfjnn3+g0dRvayYiHD9+HBcuXEBkZCQGDBiAYcOGoW3btnBycgLH
    cQgNDcVrr72Gtm3b4v3338fRo0dRWlrqZTQa5544ccItLi5updFozH/77bcRGxvbUOVgK/A5
    yUPrsce4gy9/D/DFsRpqg3qGYRhkZ2cjISEBffr0gUgkeqA3VFZWhrNnz6K4uNiixzH3UCaf
    FYyxSeBp+h7g6fhGYW89WY/vsFYg24Gn+deWR6gAz/6wdgTWJQcBvp948G3IQTP4Am5/AOiE
    m/mClrz/78BXLrU3eIBPRxnr6+vLxMbGol27dnB3d0deXh4OHTqEgwcPOpWVlY0UDPdZqLnN
    x0MLBXiq0Ebw1faKAZg9PDxo69atVbx5O3fuJG9vbxIWg0b4wX8BXz2ywVRYGz16NNatW8cG
    Bwe/JBKJymUyGa1YsYKMRiOp1Wo6dOgQpaamEhFRRUUFlZeX072GTqej3377jTp16lRZnc3b
    2/tEx44du69YsQIDBgxoaOuKA9APfLsAo5eXF73zzjuVkcHqiI+PpxEjRlgq11kiLUEN6YEX
    L14MIkKfPn2aBAUFrVIoFEWW37Jr1660c+dOKisro4qKCtLpdPf0qKioILVaTd9//z21aNGi
    0vPu7OycGR4ePrdPnz4eALB8+fJ7OSWs4GjyEwyvFPDRuGH12JPW4mbVTzOAD3FrIRpUMxqv
    eHl5Ua9evSzraAtqbyEQAp76Yu1NnVjHvUkFxU4LPtcwAnxp8XtdaVksGL3NBOO4O4Buwr8b
    ofZogkK4Vxo4cCBlZGTU+A6ePXuWoqOjLXPxM2rPD1GAL1zQEkAf8N7ubuA9zb5ouBFGVwAd
    RSLR+yzLlqD2SE6ZsC47C+u8QfEa587lWfqRkZGxCoUiG/eBncCyLIWFhdHLL79MJ06coIqK
    iipr8PLlyzR16tTK6KNcLi9r0aLFqueee85LqVRi165dDW09cYIz5jBujRTVdRSBLz7TpCE9
    8KhRozB16lRF8+bNPxeLxUYA1KxZM/rss88oLy+PNBrNfT9KSkooLS2NFixYQD4+PgSAnJyc
    Sjt27Pjq/v37RTNnzrzXOnZTwTk+HHwaw1Xw7JS6ZM1qq/VgBh8xrI17GyYYmtbraCtu5tbb
    cqbutjrfCOCFesj2yYI83yoYvFGCUWmvhWQsLTk0jRo1os8///wWFmBOTg7NmzePlEqlZX9/
    GQ+wQNz9dp80EizgSQD8GIaBs7MzwsLC8Pjjj6N796rtXWJiYvDSSy9h69atTFJSkrNGo3Em
    ohDwTXl/AN9s2u55t8eOHUNiYmKj4uLioUaj0alTp0546qmnwHEcXF1d0atXL+j1evz999/Y
    vn07ysvLMX36dISH37uuCFKpFHFxcWjUqBHmzJmD48ePo7CwsLNer1+2cePGmRcuXLj4448/
    YsyYMQ1BJjgBeBJ8A+9IX19fvPbaa5g2bRoUipp1zJYtW2L58uUQi8X47bffPAwGwwxBEC4H
    T89rEAbhm2++iatXr6Z27tx58ZUrV65lZGS8UlJSEnrs2DHMmTMHr7/+OkaNGgW5/N6ygdRq
    Nb788kusWbMGmZmZ4DiOXF1dzwQFBa3o3r37jqSkJF1JSQmcnZ3v5W2YcbPZrhd4KkkI+P5L
    pwRhYnPz5jgO/v7+yM/PZyoqKnoJ68FWXoMvAE9XV1f06dMH58+fh1qtDhL2OFse9q4AIjw8
    PMBxHAoKCth6eEY7A3hGMHjKwRcF0N1jA6Ur+Ga/7cBXT5VZKQWlggPlMvg88Avg81KsG+q6
    A2gmEonQtWtX+PnVXOcgKioKQ4cOxdmzZ2E0GpsJ3lTrfF5n8DlQbQBEg+815SF8zoDP/6wA
    X5zgPPjo2SHwVHB7h0wwbMcB6GQ0Gn3rMIghGOGTwUd9ToPvQ/gHGkgO9MGDB9G/f3/FxYsX
    R2q12hoXhZ+fH1q2bImsrCwkJibeMcPBbDYjOTkZa9aswa5duzB+/HhMmDChck1GRkbinXfe
    QXBwMFauXAmVSiVPTk6eqtFoKnr06PHewoUL1Wg4kIOn1b0uOI5uF+7gae9NALwt7Jl2j9DQ
    UPzf//1f2YQJE9aUlpZKCgoKRl+5csXpjTfewPbt2+Hh4XHfc+oZhkFWVhZOnjyJ8vJyKBSK
    Il9f37Vubm4b+vXrZ7zHjtEy3IyWmcFHniLBF4w5D9vsguqROkYwIhvDdvStEW7Nfw4Cnydu
    691pj6rpX/WJEEYIxpWvYEBewr2PZLPgA0+hgoGttHI8Fglyx1KdvMKGYT4EgHOvXr3w5JNP
    QqlUoqSkBDdu3ICbmxt8fX0xc+ZMXL58GVu3bpULzut1qLlSq0SY11DB6e0tGI9aQTdIA18p
    thB2UqSmLmPwv8JipbCwMMMrr7xi+v333+nq1as2o2JarZaSk5Npx44dNHfuXIqKirJ48XXg
    c3gC7P3BPTw8EBMT08/FxSVXJBLRqlWryGQyVemzuHTpUoqIiCCO40ihUNAPP/xA9wNms5kO
    Hz5Mbdq0seQUmoKDg38aM2aMX/v27RuCPPACX744x+IZ/P777+sdZU1LS6MZM2aQs7OzhWt/
    QlC4xGgguHHjBrp164alS5eK27Zt29/Hx+cvjuNMAMjHx4feeustys3NJb1eTxUVFXf10Ov1
    dP36dZo5c2all10sFlcEBgZu6dy5c/t169axD6hyX5DgJSfBiJlVi9BRAvhGLBbTyJEjqXnz
    5iQYOJNqERQzAJT36NGDduzYQY0bNybBOdW9lmusgxA169KliyX34flansENfO6BxZP6Hu5d
    qyCxIPzXC+9SZXSBZVkSiUTEcZx130SzIISugY/uPSsY36xgvF2XyWT05Zdf1vr+bdy4keRy
    OQFIBRDNMAwrCLwJ4Nkg14TrmGHVT7SG+zEJgnAL+AIaUjvfsxaDb6FUU/+v+h65wprwawj7
    lKenJ7p06dLW2dn5cvVncXZ2pmeeeYb++OMPysvLo82bN5OXl9ddjxjK5XIaPnw4HT16tEpu
    vkajoSVLlljkAEml0qLw8PAX33jjDZGld7Cdw1tYU7l3YZ5M4JkMj6N2loTdwNI6ZNCgQZ6R
    kZELlEpljtXe8MAOlmXJzc0tsXXr1pPHjh2r7NevH1588cX7OTUBAP4n3I8atTNSnMD3wq3e
    u/DZWsaMFs6xHpMDoKeN8yUAVqBq9NqE2vMOpYKj3iCc/ynubTCLA8+EeQ182oiFyagTDL8K
    8IzFDOE9+UaQ41HV5E4QgPNisZiWLVtGZrOZcnNzafr06dSmTRsaPXo0XblyhYiIPvnkE5JK
    pSQYdM2rzVdzQRf5Cnw0NlPQT/TCUSHIyAwAR8EXker4b3TY+xUhlAle+oksy0p79epVsnTp
    Uknnzp1FdXG8ZTIZQkNDERoaioEDB2LkyJFYtmwZ7d69W2owGMYJSsBb9uwlZRgGer3eC4CH
    j48PoqOjLf3RkJmZiUWLFuGnn36CVquFTCbD8OHD0aVLl/t2bz169MC8efMwe/Zs5Obmsnl5
    eXGnT5++EBsbuzIiIkL/008/2evUBoLv6fYsy7LymJgYvPXWW+jdu3etuZRVrIagICxZsgSN
    GzfGRx99xGZnZ3cWNsWl4Hvz2H1JfE9Pvr5EcHCwYcWKFftWrlyZeeXKldcKCgqeysvLU7z3
    3ns4cOAAPD0977qn1JKzcebMGej1eigUiiIfH58vWrVq9cm2bduyunTpgu3btz+IaVELypHF
    UzcJPJ34oi3PqNlsRtOmTeHu7o7Lly8rwRdz2VKDp1MseCydQkNDERYWBk9PT2RkZCjBU1iO
    2PBwxigUCgwaNAg7duyoj2d0qGDcQFiH2bg3vefk4KOQrwMIYVkWvr6+aNGiBaKiouDn5wd3
    d3fodDrk5+cjNzcX169fZ5KSkpxyc3Ob6vX6poIn9DL4fI50ACKz2QyVSgWz2Vy5393yI6nV
    llxTEcuyncxm8xPgKW+RAORSqRR+fn6IiIhA06ZN0ahRI3h7e0Mmk0GtViMvLw/x8fGIj49n
    c3JyPE0m0wjBIH0XwLd2KBe8BQfWZNx5EQ8fwdHhKXynXbNlhJ6oTUUiURVqfmBgIF555RVM
    nDgRbm5uMBgM0Gq1lTnIdxPl5eXYunUr0tLSsHLlSvTp0wcWptLMmTORn5+Pzz//HBUVFe6F
    hYWvHjhw4Orx48f3bdu2DXFxcfY6tY0BLAIwFnenMIzFqfMxgGXg2Vg6e15bK1euxJkzZzBn
    zpwbcXFxK3fv3p2YnZ29oKioqLXZbGYAwN3dHR07doSzs/NdX1ssyyI/Px+nT5+GVst3gxGJ
    RMZGjRodaty48VtTpkw5vn37dtP+/fuxf//++zk1xbiZk+YKPsp2RDBybP321nABMJBhmG1E
    pKnBfmiGW4vyKcBHmf+0obP1wa20yNro712FtS0CYGYYJl8qlZr0ev292CPcBQN4vGDgSSw9
    jhUKBeRyOYxGI0pLSyU6nc7ZYDAEms3maOH+rgs6xi7BUPQB4MyyLNzc3MAwDBISErB582YU
    FBTg2rVrGDhwICIiIuDj4wORSISKigqlRCLx0+v1heBzJQeDZ0SGApBZ+kqLRCLI5XKIxWLo
    dDpotVoYjcZAs9kcCJ5VNEJw7n5hpQfZDfoIVi099thjxVeuXCm9k6hWSkqKOS4uTi94C/IF
    xcluIRKJEBgY+KxEIjE1b96czp07R0RExcXF9Morr5BEIqn0rnh7e9O+ffvofqO8vJzmz59f
    WfnIxcUlo2vXroMA4B7z3f8tgsFHXPQSiYRGjBhROa81QaPRUFZW1i1V56wj0evXr6fIyEjL
    b5EtbJ4NKjdp7dq1AIBRo0Z5REZGvuHi4nK/cnXM7u7uCe3atZsyfvx4ebdu3R70VHAAPkHV
    PIUPUHNFWWcA33McR4sWLaJNmzZZop3XBWFUHR4AtgGgJUuWUFFREQ0fPtzSZHyJDc/cywB0
    HTp0oGPHjlG/fv3qyp0IAU9/tNy/Cjwt+m5DAb70fL5l/5k2bRodOHCACgsLyWg0VomkmEwm
    MhgMdOPGDTp27BitWbOGRo4cSf7+/paqoSZBAdECoDFjxpBGo7G550yZMsXyfBUMwxQCMLEs
    S4GBgfT000/Tp59+SidPniSVSkUGg6EKs8JSNVKlUtGff/5JM2bMqMzXERyFc1E3DfN+QiY4
    sMru8vunA/COnT3rLXBzc0Pz5s3HKpVKg7W8W7t2bZXcvr1791JYWNg937M6depEx48fr7Ke
    rl27Rv3796/Mgfbz89sXFxcXEhYWhtWrV9vjtIYIBpse9ajAGhoaSt26dSMPD4/biUK/iAZU
    ifupp57CsmXLmN69e3cMCAjYIpFIdADI1dWVXnvtNbp+/fpdzZ/XarV07tw5Gjt2rCXKQ3K5
    XNOkSZNPhw0bFgoAL7zwwv0XgBwHqVTKsiy7BlUrctqSUXIAn9ewBpIF46Sm87+v4XwjwzBv
    cxwnrsEROA63RhRN4IvCwYas3Wh1bplIJJoK8MGiuww/QUcoEXRg6tq1K82dO5d++ukn+uOP
    P+j48eP0559/0vbt2+m7776j+fPnU9++falx48bWerwKPJX/fQCZHMfR3LlzyWQy0bFjx8jP
    z48AkJubG23ZsoWIiD744AOSSCQklUqLAwIC1oOPTFbWhPD09KTo6GiaPn06ffTRR7Rx40ba
    vXs3/e9//6Pff/+dvv32W5o9ezZ16tSpkuUgyIWvhD3CbiCxLDJ/f3/j7t279XfDgDl48KAh
    ODjYsgmut1dhuG7dOhARFxYWthAAtWjRgs6fP09ERJs2bbqFFuPq6kq7du361/TPO0FSUhL1
    7t27UniEhITsmDZtmu+oUaPsbVo9BM+HXiKR0Pjx4+natWu1GoOLFy+mQYMG0Z49e2pty7Fr
    1y5q1aqV5ffIEJTwBteUKi4uDq+99ppTu3btxjo5OaUBoJCQEHr66adp3LhxNHbs2Ds6xo0b
    RyNHjrQUfSJPT8/4mJiYxz799FPR9OnT7SIgISjfRpFIZKEWpgnRJ9TgBV3PsiwtWrSIUlJS
    KCYmxmLgza/BaxoJ4LRMJqPvvvuODAYDvfrqq1TLXuQNvhoovfTSS5SZmUmPPfaYRRDW1KpC
    LFxXa3XvOeCLqNxVx7YgoKvQrW0VYrK15xQVFdHu3btp4sSJlcLOckRERNDJkydrHHvhwgXr
    d40AkJ+fH02ZMoX27t1LKpXqtvY0jUZDGzdupJYtW1pTl8bh3tFsbxe9hDV4LwycXPBRWrtE
    SQmfYtqkSZPpYrHYBD49gebMmUOlpVX9w9u3b69Xi4k7PRiGoSeeeIKysrJuKWYXEBBgoY7q
    o6Ki3vrvf/8rHjdunL1Nqyd4upqhPs/r7e1NP/30E+Xm5tLMmTNvZ66yhfeowcjB119/HQAw
    dOjQxqGhoSuVSmWRYKjR4sWLyWAw3DVnukqlonHjxlVS2F1dXVOjoqJeGzZsmHvHjh0f2ByI
    xWI0atSI4ThufrU1chVA9xqq1ysEvar6718Bnj5ZfUAAgAM23q0fGYZRVruGEsDXqJmiPNuG
    DH8WN/sEWxx9g+/BdHmDT2nTsixLrVq1ok8++YTS09NtBhGIiAwGA+Xn59OhQ4do+fLl1KNH
    j8qWbsK8mQDQgAEDKD8/n/Ly8mjAgAEEgKKjoyklJYVKSkpo9OjRlXuSJd1HLBZTy5Yt6eWX
    X6YtW7ZUOjJsoaKigq5fv04ff/wxtW7d2uKgrQBfgdVuUuvCAZxnGIamT59uLCsrM9+Nl7Ci
    osL08ssva4WHvgKgtT1uTKNHj8aECRPcw8LC9gCgfv36UX5+Pt24cYMef/xxSzSOPDw8SCaT
    0cCBA+n69eu3NRdGo5F2795N8+fPpyNHjvxrw9BsNtMvv/xS6WWXy+Vl0dHRM9LT01kLR99O
    FP3xAIo4jqPRo0fXOV/JycnUtm1bAkCvv/56reeaTCbavn07RUREWF7qQ7Df/ja1IiAgAJGR
    kW5KpfKQJVqTm5tLxcXFpFar7+jQaDR09epV6ty5M4HPCf71zTffFFsqCdoJJgIoCQ4OpvDw
    cMvv+St4Koc1XAFsZFmWFixYQFqtlubPn2/ZUP9Xw2baG0Cuv78/HTx4kIiIPv74Y8v5h3Fr
    tdq+ALLc3d1p586dVFBQYG0Q1pQ7EQPgskgkoi5dulg8flfBF1i5m+gIvigMBQUF0c8//3xL
    BO52UFxcTFu2bKFu3bqRpb+cVCql995775b+qCaTidasWUNOTk6W3GXq0aMH/fbbbzYjivWB
    yWSirVu3UmhoqOX3Pg++Sqo9OEY/u8dGzgbc++qz/wqLFy/G559/Lm/duvW3ltyu1q1bU3x8
    /C2/4Zdfflm5Lm7XwPP19aV27dpRo0aN6jVGoVDQBx98UMVAKC8vp3nz5lV6/D09Pa/Hxsb2
    AoCTJ0/akxychJstuOo8mjVrRpcuXSIiorVr19Jt9q49IuhyDQp9+/bFpEmTXFq1avWCVCrN
    AkDjxo2j4uJi0uv1d3wYjUZKS0urrJbs7u6eGB0d/fiCBQtk48ePt5dpeLZaVM4Mnl1VvX+t
    Uogo1fT772UYxreagdcRPPW0pvP/QtV2FABfpCyxhnPNAF6xYTscrvaOZ3Ac16G+KUH1hExw
    wJaCZzHS0aNHq8gss9lMxcXFlJ6eTklJSZScnExZWVlVqoYajUZKSUmhzz//nHr06GHdZ5B8
    fHxo586dlfmCLi4utHjxYjKbzXT8+HFreUUMw1BUVBQtWbKEzp49S1qttvIaJSUllJWVRcnJ
    yZSUlEQZGRm3yEuj0UgnTpyg2NhYizNZC75IVJ1Bs/uRQxgCIFAikSA6OhpyufyueJkkEgkb
    HR0NmUxmKi8v9xeU9gv2tBmNGTMGu3btQnh4eFRubm6USCRCTEwM3N3dkZOTg/z8fCgUCsye
    PRvt2rVDYWEhunTpgiZNmtzWdbRaLX788Ud8//33yMvLQ/v27eHkdPsMD4ZhMGDAAMTGxuK7
    775DeXm5PD09ffKMGTMOXrly5ZKdTGuIoOi7t2rVCvPmzatzvkwmEwwGAwBAr9fXHjJhWcTG
    xiItLQ1z585FWVlZJ/BtTlajAVRusoZer4dlsxDeGbi6ut41qkVZWZl1nyfKzMzE5cuX7WkK
    MgFoOY5TPv300/j0009RVFTUD3zbm69wMx+vSi6fRCJB//798fXXXyMvL6+1YFBY9zIMAuDi
    4eGBxo0b8x8EBcHV1RUqlaoxeOpJulW0ry8Av7Zt26Jt27Y1KXbVjdPJACIiIyPRr18/y5zm
    oWolzzuFpWJlK6lUimnTpmH48OE2c/3qAxcXFwwfPhxhYWFYunQptmzZgoqKCuzZswdjxoyB
    dW/TnJwcbN++HVqtFhzHYdSoUVi4cCFatGhRa9/VusCyLIYMGYKUlBQsWLAA5eXlrYX9IkEQ
    jA8KQYKhfy8RDT7X5Lw9vYSLFi3C5s2b4e3t7Zubm9veksccExODsLAwFBUVYdeuXXB3d4dK
    pcIXX3xRmYd1WwtaLsecOXPw1FNPYfPmzViwYAHKysrq3MO+/vpr9OzZEx068H4DJycnTJgw
    AYcPH8axY8egVqubJCUlTZ84ceL5ZcuW2UvV0abgCy+53Y58t7xbLMve7nvWATxb5n0h2tQg
    8L///Q+BgYEaLy+vn1mWHQ3A//Tp05g3bx5EItEd59QzDIPy8nJkZGSAYRh4eHhcio2N3WMw
    GPTfffedPclBHW62amPAp1ntYxhmI1WdBFsCoA0RtQPfLsICf9zsqVsdjVmW9SeiNCICwzAM
    EfWE7ZYm1a8rBZ+X17mKwSISaRQKRSnA55/fJfQBT4tWtG/fHkuXLq3sh67T6XDhwgUcPnwY
    586dQ2pqKtRqNViWhZeXF4KDg9GlSxf06tULYWFhCAkJwdSpU9G7d2+sWbMGGzZsQHFxMQoK
    CrB//37069cP/fv3x4EDBzBs2DAQEfbv34/0dF5dkEqlGDJkCObMmYPo6GiIRCIYDAYkJCTg
    0KFDOHHiBDIyMnDjxg2YTCa4ubkhJCQE7dq1Q58+fRAVFQWJRILOnTtj2bJl0Gg0OHr0qAx8
    0ZvT4GsiPFA8DUDr5eVl3LNnj+5u5r0dOXJEFxgYaAQfFp1kb5tRr1698Nxzz8kaN268hmEY
    Cg8Pp2PHjhERkV6vp507d9Inn3xCOTk5dzQParWannjiCQJAY8eOpbKysjv6vr1791ZSZiQS
    iT4iImIBETGLFy+2h2kdDaBYKpXS6tWrbUY0jEYjlZeXk06no/Pnz1f2xps5cyZptdrKv9mK
    pubk5NDAgQMtXpsD4MsdNyh4e3vD39/fVaFQHARAzz33XBVv050iJyeHunXrZokQbpoyZYq4
    R48e9jQFbQCkenp60q+//lpJywBfSTbS6jw3AJssEUKTyUT5+fk0ePBgy/kf4WYuqVjwthn7
    9+9PhYWFRER05swZi5evBHxCtwVNABwXiUT01ltvVX63VYRwVrV7HgWgwBJZe++99yyRil9v
    R/mrB3qAp0RT+/btKyMHdwuJiYmV74+bmxv9+OOPVf6+YcMGcnV1JQhVVy3V1u4WkpOTqWvX
    rpbfLw0154LeT/QHT3e6lxHCEvB9x+wKQ4YMAREhIiJimkQi0Vrut3fv3nTgwAFatWoVeXl5
    kZeXF7m5udG/rQ7p6upKmzdvtqSU1DtKyLIszZ07twodyxLBFirgklwuL2zRosVwsdhuik+P
    BV/psNZnk0qlpFQqSS6XU/v27SkxMZGIiNatW0fu7u6kVCpJqVRa+qjWdRxGA6jqXh3Ozs5w
    d3d3lclkB3Hve15umjx5sthiUDxoiEQiiMXi1gzDXK/hng+gan6ZM4RK2DaOlbhZdZYFT/Os
    qGUvGmnldPAG8LuNc83gKanW6Ioaoo8sy+5XKBR+ttqJ/Qu4g69mTUqlktauXVupE+bk5NCC
    BQsoLCysSrQPNVQubtWqFS1fvpzS09OrUE9pvQkAACAASURBVImtKxe3adOGLl26RDqdjg4e
    PEgajYaSk5MtFcdJLBbTtGnTqvTtTUtLo2XLllFUVFStrAmJRELNmjWjJUuWUH5+fuX4n3/+
    2TotbbvwO9h2qN6vhclxnMHJyemuVqpSKBSsVCplrCbGrozBw4cP4/Dhw30KCgqe4DgOcXFx
    lRECsViMwYMH48UXX4Sv753ZGkajERqNptIIuNNwekxMDPr162eJMokLCgqG9ejRo+nmzZux
    bt26BzmtUvA0BZewsDAMGDCgxohGWVkZ1q5di6lTp2Ly5MlYsGABsrL4AM++ffswZcoUTJky
    BTNmzMD+/ftr9BL6+vpi+PDhEBSA8GoGhAMNAyUAbmi1WpSUlGDKlCmWKFUH8FU1pVYe08qF
    RETw9PRE3759IZVKAT73K9hqDYYD4MLCwir7O/r4+FiqvUrAN+u1fF8nAM19fX3Rp0+fmtar
    9QeBgifPKyYmBiNHjkR2drYlqp0HntJyN8CCLwvux3EcYmNj73rP02bNmmH+/PkIDw+HWq3G
    5s2bUVRUBAAoKirC5s2bUVxcjLCwMMybNw8RERF39fpNmjTB4MGDLe9vgPC8DzIHyg33Ps9d
    Ar6lhV0hNTUVHTp0aFpYWPiMXq+vpCccOXIEzz33HJYtW4bCwkIUFhZCrVb/66gNy7KW9xU6
    nQ4VFfUrEG02m/Hbb78hPj6+yncNHToU7dq1AwBotVrPvLy8Uf369XPp06fPg55Sixystalr
    UFAQ3n77bfzyyy/YtGkTVq9eXRml79u3LzZu3IhffvkFGzduxIsvvlgfVlE47j5t/Z7jThgH
    //Z6d5nS+O83epYFy7IlQsGu6ugGPmAjspKDtd14bwjpEAzDSASdSFLLGg0jIsv3RQlr1qaJ
    YPX/LuDZK2E1vKvZZWVlmroi/7eBzuAjhOjQoQMGDhwIhmFQXFyMd955B++//z6Sk5MrGWY1
    oby8HBcvXsSSJUswefJkbN++HRUVFXBzc8PMmTMxadIkiMViJCQkYPfu3RCLxejZsycUCgX2
    7NmDc+fOAQBGjBiBBQsWIDAwEBUVFdi2bRsmT56MZcuWIT4+vlbWhF6vx5UrV/DOO+9gxYoV
    lfZAnz59rPu7d0cddQjuh0FYCKC0rKyMuXHjxl394hs3bphKSkrM4MPh+fa0CSUnJ6NNmzYh
    arV6jk6nC2jfvj0mTpz4r6icdSEnJwfp6ekQi8Vo2bJlpVC8E4/ayJEjK9sZlJaWRmVkZAy4
    cOECtm7d+iCn1RVAWwBo06YNgoODazwpLy8PX375JdavX48NGzZg+/btKC7me5gmJiZiw4YN
    2LhxI7766its3rzZJo20Y8eO8Pf3B/icswYnCB1ACYBcnU6HlJQUdO3aFU8++SSEXjfjAHSx
    ZRCyLItevXpZFKgw8BE1i0cxkGVZhIaGVr7PSqXSQl2WgC/FLREMgIEAXDt27IgWLVrUtgdz
    gnDu6enpieeffx5eXl7IyMiA4EHNw92jLMsFQ5VzcnJCVFQU7kXkIyYmBs8//zxkMhkOHjyI
    48ePAwBOnTqFI0eOQCqVYsqUKbgX3nSO49C8eXMInmROEPzyB7we6VF7AVu1aoUmTZo4ZWVl
    TS0uLu5c3ZGZmZkJlUp1V67l7Oxc2XA+NTW13gahRV7/+uuvVWRBSEgIRo0aBalUCiJCeXl5
    n6SkpOiDBw9i4MCBD3Ja3VC1sXeNaNGiBSZMmIDY2FgMHjy4UgkF+FYfgwYNQmxsLIYNG4Zx
    48ZVyvta4I2qPdIcsHPo9XpUVFRozGZztg0H0njwuX31sQmaQWBaEJECtVevFDMME8myrEQw
    OPvj1tx9W7bIQPBpHTUZuAVKpfJuWYOcYCR5isViDBo0qDIF5NChQ9iwYcNt7SE6nQ779u3D
    jBkzsGnTpkpK50svvYSePXvCYDDg119/xfXr18GyLHJzc7FlyxbodDq0adMGc+fORWBgIIxG
    I3766SfMnDkT+/fvh05X/ziaVqvF999/j8OHD/MvrLc3hgwZYnFcu4F3bosfpEGYD95LL42P
    j5eazea7JhTj4+M5tVotAl+eNc9eXsJ+/fohJiZGlp+fP6WkpKS7q6srpk+fbkshvCOYzWbs
    27cPqamp8PPzQ8uWLe/K93bq1Kkymmk0GuVqtXp4+/bt/VJTUx/k1CoANOI4DuHh4TZz4Xx8
    fDB+/HiMGjUKcXFx6NOnD5RKvtZCeHg4Ro0ahZEjR+LZZ59FXFwcJJKanVx+fn4Wg0AMPsog
    ggMNziA0m81IS0uDSCTC+PHjERUVBfBUzqm4ScO8ZS8MDw9HTEyMZd0NEDyXfgD8nZycKoUH
    wOcdhYVVOjRDhDGhAHrIZDL07t27JoWLwc2oVRsAExiGkcXGxiI2Nvb/2fvu8KjK9O37nGmZ
    lkwyKZNGGkkIIRCpoQmEDgIrIoo069pZ3VWUqsvSVVyVBVxd0BUbKKgIqEgHRSC0kEAq6ZlM
    ZjKZ3ud8f5yyk0gJkGSG3+dzXeeagSQzZ945532e+yn3DaPRiLq6OjAJr/bUmBOAmf0IDg5G
    eHjHFJUEAgHuvfde3HXXXdDpdNi/fz8sFgsOHjwIjUaD7OxsTJ06FR3VhhcREYGQkBD2nyr4
    V6xeh/ar8F4zJgkkP/jyyy8jPz8fRUVFY41G41yPx9Oh/Za9e/dGYmIi7HY7Tp8+fVOBlNfr
    xQ8//IDKysoW/z969GgkJ9OcYg6HI8ZgMNw3efJksdvt13FyCa49u8VZYWEhPvroI+zduxd7
    9+7FsWPHuJnK6upq7N27F3v27MGuXbvw6aefog0Jez6z//3hB+9AP3gdkPc4aEKZG+nistq8
    Mub6i7lu9ouiErxer4yJnXJv8Nrsz+IBPAmaQbelsyQIT1BQkNZsNrfnfZQNgBCJREhMTARJ
    knC5XDh8+DButYBVVVWFlStX4uDBg1xi6emnn0ZERATOnj2LAwcOAABOnDiBvLw8yGQyPPXU
    U8jOzuZmCleuXMkmg2/atFotjhw5AnaPSkxMhELBTZtk4zqdKp0BCCsAnPN4PPj++++p8vJy
    T3u8aHV1tfPbb7/1Mhm98wDKAuHOGz58OH7++Wfk5+ePMhgMc71er/Cee+7pMFHb0tJSfPHF
    F3A6nRg4cCAyMtongRcVFYWxY8dCIBCAoijY7fY+ZrO537lz5/DCCy/4ExAG8Xg8hISEXLMV
    RCqV4tlnn8XmzZuxdetWvPHGG1zwPnbsWGzZsgWbN2/Gxo0buRaBq5lYLOZaApkNSviHb7mj
    jAuQa2trodfrkZWVhdmzZ7Pf63jQVP28q+2FwcHBvsmEgaCz4zEAIiMiIlqQpAgEAiQmJrIk
    O9HM7w0DkBAfH4/rzFYS+B/BS0ZCQgIeffRRjmCDAYS2dgaEJAtEZTKZr7Nod0tISMC4cePA
    4/Fw7NgxnD9/HocOHQKPx8O4ceNumkDrZiwkJAQ+syZ8PweyjaATlx1pTQgQEeKXXnoJmzZt
    wpAhQ7rqdLq/2O32Dp3BDg0NxfTp0xEWFoaioqJbYgMtLi7GkSMttbS7du2KYcOGcaDRarWO
    qqioyNy/fz8uXPAbh520LcmNqqoqLFmyBA899BAefPBB/O1vf+NGJw4cOIDZs2fjoYcewuzZ
    s7Fx48a2EvmE/uEH7zhz8vn8xuv4nz8BGNUGQAjQLYdpBEFE3wgQMj9Xge5GuVE1hGSuqxm4
    xrw3RVFWp9NZ1c5+MIT19RER9HidzWZj/e4t26VLl7Bs2TJcvnwZAD1CNmTIEDgcDhw8eBCN
    jY04ePAgDAYDBg8ejHvuuQckSaKgoADLli1DcXHxbb1/XV0ddz9HRkb6+vhgXEdTtDMAoQnA
    dgDNZ8+eFb///vsk2996q2a1Wl3/+te/LMeOHeODptP9HDT9sl/twoULOHToEEaPHp2i0+le
    sFqtsd26dcNzzz2HsLCwdn8/vV6P9evX48yZM4iIiMC0adMQGhrabq8/atQoLuj1eDxKl8s1
    9tlnnxWzPc9+MBd9Kh7o9frrzpoIBALI5XLIZDLI5XJudksoFEImkyE4OBhSqfS6rIoWiwU+
    2SgbaBKQP+zOsjoAdhYQkiSJ6dOnY/DgwWxw8yToit5VswKDBg1CSkoKC/ImgJ6bCIqMjERs
    bEt+hdjYWHbjVYGWwRkDQJiTk3O9GT2Sccb38fl83oMPPsi1UNbX17OtztZ2BoROMC32drsd
    Vqu1wxZfIBBg/PjxSEpKQklJCT788EMUFxcjKSnJd8avQ8xsNvsGuVomQeAvq0THs3+eAy0i
    7XerrKzE9OnTxZWVlX82mUyDb5fN8XrG5/Nx//33Y8KECXC73di1axdKSkpuJa7Anj17uPEC
    4H+sf6z/drlcCW63e+R//vMf3ooVK/wW4KON7eM2mw3Nzc0wGo0wGo3weDzs54DBYOCOG7Fv
    +77kH37wzjOCIOqus/9FAngKdCXvRgOXsQDGUBSVwoKp65iKIIh0AONwg3lXxnqDZs69FmCx
    ut3u6nZeGg9Ai9yzyX+32432mFH85Zdf8OGHH8JutyM8PByTJk2CVCrFqVOncPToURw5cgQS
    iQT3338/4uLiYLPZ8MEHH7SLtI3FYuEqhBKJxHeMjLred9xZpDKHAOx2uVzeDz/8kHz77bc5
    goGbNYPBgPfee4/YtGmTzOVyEaAFn/cHwk23dOlSLFy4UFxdXf2owWC4WyaT4emnn0ZHiJMa
    DAZs2rQJH3/8MdxuN0aMGIHRo0e363skJydzVNzMTTLwypUrXXwdZiebDYDD4/GgtrYWbW3b
    4fP53A1xM5ILzc3NbNsABbrly4k/7E6zWgA2rVaLxkY6SRoXF4fHH3+cbeHsB5pg5qpOKDEx
    kR3K5jGAMBcAERMTA6FQiLNnz2LPnj0oKytDdHQ0IiMjwTi/PwHoL5PJkJubC7n8qv7Qi/+1
    yET16tULc+bM4a5VnzmoZrRvdckOWteQYimxO9J69OiBESNGwGg0Yvv27TAYDBg+fDjbutth
    ptFofKnJi/wMCI2g2ewMHfj6XyAAEqPz58/Htm3biMLCwonNzc2z3W53h7aKRkZG4pFHHkFI
    SAhOnjx507M/vnb69OnfZefvuusubtzD5XIJtVrt4F27doXdbhb/Nv3gTV/LXq+XA4Rut/tW
    yHv+8IN3qLlcrmrQicVr2VDQDO43qv7yQXfVDGsDfpBSFDUONKHXja6rUNCtq+nXAbVWHo/X
    0I6EPU4w8lBms5nzFWKxGFFRUbePND0efPnllzh+/DgAukrYs2dP1NTUYMOGDaioqEB6ejpH
    +nLkyBF89dVX8Hq9t/3eUVFRHL9Bc3OzL8Ctx3XkqzoLEOoArAZwqLm5GW+99Rbmz5+Pc+fO
    tfnDe71eXLx4EYsWLcLq1av5BoNBwADN1czr+9WOHj2Kb775BgUFBaO0Wu3DLpdLMGHCBMyY
    McNXq+22jaIoXLp0Ca+++irWrFnDsQnJ5XLf9sZ2seDgYOTk5HBZfJPJ1PXSpUvdz549i6ef
    ftofy2xlLmhcvny5zYFsVFQUpk+fjj/96U8ce2pbrLi4GPX19ezG0YD/D0kh/g9YHQCbzWZr
    MR80duxYjB8/HqDlJB4A0J3V6aqsrMSWLVvwz3/+ExcuXMDdd9/NgseeAAbzeDyoVCps2bIF
    9913H2bNmoWHHnoIRUVFbNsJH3Q7akTXrl3Ru3dvnD59Grt27UJRUREXlDH770QAd0skEjzy
    yCPo1q0bd59fuXKFDWzVN3DmNx0fgKb09thsNvYa7zATi8UYMmQIJBIJzGYzxGIxhg4d2u77
    VWtTq9Vs9dPNAEJ/B7JHAPzQAfsIBVof7EAg3HBnzpzBs88+m6pWq583m80dLtdjNptx4MAB
    7Nu3D2vXrkVRUdFtJRFOnTrV4v8iIiLQv39/7r602+19ysrK0hobG9GRlc8b+MHam/2jxsZG
    HD9+nGup9dmHbiaAVv/hB+9cP3idn0tAC9i3hYSiN+julxtVE3mgyWGS2vCaI5jfvSYmEQgE
    zSEhIZZrJFdvxRygE6Nes9kMrVbL+atevXpdk1viZqympgZffvklXC4XEhMTMWrUKLjdbhw6
    dAgmkwk5OTlISkqCw+HAl19+edutqgDd1dCrVy+u+KHRaFjiLgr+T4y2sP4A9gCwkiRJ9enT
    hzp06FCbdKVOnDhB5eTkUDwej2Iu7D3M6/nddu/ejSlTpuDll1+Oz8jI2EcQBJWenk4dO3as
    3XS13G43VVVVRW3evJkaOHAgRZIkRZIkpx80ZswYSq1WU+1tx44do2JiYjidnYSEhHdmz54t
    8BP1Nh/A6wA8MpmM+uKLL9r8OaxWK6XT6a6pW9jazGYzNXv2bFbO5Ar8r2N20/aHDiEAWsD5
    EqsD6GuHDx+mEhMTOS0fHo9HzZkzh5o2bRoVFBREEQRBde/enVqwYAGVmprK/R6fz6eys7M5
    nTM+n08RBEH169ePysrK4n6PIAhq8uTJ1KJFi6iuXbtSCoWC6tOnD7Vx40bq7rvvbqEjdM89
    91B1dXXcuTmdTmrWrFnszz9C+0sW5LJJjilTplANDQ1UR9rx48cplUpFAaBiYmKoEydOdOj7
    GQwGas6cOez6NQAYGSDXY08GuLWnBtoB5nX9ahRFYeXKlfjuu+8kI0aMeEsikbjRsbqLLbTA
    IiMj26qnd91j5syZlNVqbXE9bd++ndPN5PP5zpSUlJd37txJzpgxwx9LzQOwlEl03NRni4iI
    oFJTUympVHora1MOepb6jrLg4OBO1SEMND9IkiR4PF4iQRAFbfgc3s66Z2/2ffl8/nfBwcGh
    LEFgO9kMADY+n08tWbKEcrlcFEVRVHl5OTV69Oh2+WwDBw7k9Al37NjB7SNSqZTasmULp5vb
    p0+fdnm/UaNGUVeuXOH2rn/9619UUFAQxSR0ngi0+zMedFWvGQA1b968a4qD+9qiRYsogiAo
    gUBgCwkJWQ9GDyUQbObMmWhsbCR69er1gkQisUskEuqtt97iLq7bMafTSRUWFlLr1q2jhg0b
    xolcymQyR9++fYsTExMbAFB9+/aliouL2z2wUqvVrIg2BYAKDQ396d577w3Lzs7m2JI62YaA
    JhCi7r333hYinO1pP/74IxUXF8d+7s9BtzT8AQjvPEAYCeAgAGratGktgI/dbqcWL17Mic4S
    BEEpFApKIBBQIpGIioqKovh8PiUWi9kN9XcBwIgRI6j58+e3SJr4/k5YWBgVFBREkSTJiVzH
    xMS0EM2Ojo6mdu3a1WJtKysrqWHDhrG/swrtr6EXzlarQkNDqc8++6xDAVpxcTEHlvv27UuV
    l5d36Pvt3bvX9/7di8DS5xvAnJPtNp2/jXmdAYHwoTZv3gwA+NOf/jRGpVLVdUYgSZIkFRkZ
    6VQqla72es2srCyqoKCgxfVUUlLCXb8kSVIqlWrbqFGjJCwzoB9sEOh50c4M2j/F/1iZ/wCE
    dwggZDpfIgDs8xPYa6/jX2h/puje7H3Ur18/qrCwkLvnf/rpJyo9Pf22zzs2NpY6evQoRVEU
    9euvv141Mfrjjz/6Csjf8pGamkr98MMP3Geoq6ujJkyYwP68+kaFDbKTr0056B5hFZPlgsPh
    uOGGSlGU7++RLpcrCjTjX4i/b7Ynn3wSGo0GM2fOjKmqqrrfarWKcnJycN9994HP58NsNuPk
    yZM3TSFLURRKS0uxbt06zJkzBwsWLMDhw4dhtVpdoaGhF7Ozs1dMmjRpHkmSZQCtu3U9gpRb
    NYVCgaysLN9sU6rL5UoSiUTYvn27P5b8DOi5Uc++ffuwefPmtrKjtdmKi4uxbt061NTUgLmJ
    tiIAZnP+sFsyG5g243379mHevHnYuXMnSktLYbfbkZuby+lZUhTFzRHMmDEDH330EZ5++mmE
    hobC7XZzbLQ8Hg+xsbH485//jPfeew+vv/46VqxYgV69ev3uHmxqakJISAiefvpprFu3Dj16
    9EBdXR00mv/Jpk6YMAGDBw+Gy+VCU1MTDh8+jOXLl7OCtS7m/Ns76tSCJvsy6/V6fPDBBy2E
    udt945fL2flKREZGoh3bfn5nBQUFeOutt9j71wjgS+bzBor9Bnpu9F3c+kyhiQmQnmJez6+2
    fv167Nq1C3PnzpWcOXNmTmNjY3RHv6dIJHKnpKScveeeezZERUUVtdfrVldXo7S0JTdPVFQU
    evaki7Berxd2uz1ZLper+Hw+/EQucw7Ad+g8gpcqxg82/+FS7iyjKAoURbU3MVmnG4/Ha5BI
    JI52jnMvg27lx4ULF/Dtt99y3BS5ubl4/fXXb5sJu7m5mYv/Q0JCOCkkhULBsX9euXIFtyun
    0bVrVyxfvhwjR/6vGebAgQP45Zdf2H8eB5B/vdfoLBpuEkAfxnmNYwAhGRsbi7Fjx94QyBAE
    gZEjR+Krr75CRUWF0OVy3Qea/nYfgI0ATsFPzFd5eXlQq9UQiUR9rFZrpkgkwqRJk5CQkACz
    2Yx3330XH374IcaNG4e1a9eiLeVuq9WK77//HuvXr8epU6dgt9shFovtSqXyrFQq3aFUKn98
    8cUXL73zzjuxNpstlHGO7dLzfBWni5SUFPB4PHg8HthstqhLly5FNzY24siRI9i4cWNnL7kV
    wH8A9DObzQPXrVsHiqLw+OOP37aemtvtxpkzZ7By5Urs37+fDbreBfDzH27ljjXOERoMBmzb
    tg379+9HQkIClEolfGcHWODy4IMPYtGiRUhISMDgwYNx//3349KlSygvL4fT6UR8fDz69OmD
    Xr16cZv7rFmz0Lt3b+zfvx9nzpyBVquFUqlEamoqBgwYgEGDBkEmkyEqKgpLlizhwBdBEMjL
    y8Pf/vY3eL1e1NXVoaSkBNXV1eyMDzc32wG2i9mPpx05cgQLFy7EmjVr2k26xteEQiE3MygS
    idp1rtrX2PlqVgMKwG4mgRRIJgOtURkJJjF6iz41EkBX0IRDJn9+oO3bt8PpdKKuri6rqalp
    aOv5NIFAgOjoaJjN5lsmlPPxSQ6pVHohIiLi20GDBu2IjIx02e324e2WQbLZUFxcDK/Xy8Um
    MpkM6ekt+C6S6+rqEqxWa3lrqYpO3Nc2gybFGtLWP2LnpBmQ0NY/MwJ4BwEyo/qH3dpl3YF+
    pDPAoFMul2tdLhfCw8NbJFTb4T76DMAYh8MRu2XLFmRnZ2Ps2LHg8Xi477774HK5sHjxYlRV
    3ZrihcvlQkNDAxdfREREoKioCHK5HEFBQaAoCo2Njbcy08tZeno6VqxYgSlTpnC+NS8vDxs2
    bGCT3GoAn9zIT3QGIAwB3af7HIAMHo9HZmRkuCdNmkTl5ubyWIadG1lubi62bNmCAwcOYPfu
    3UR+fn60y+WaBXqO8F/+yl4lJSXhxIkTZEJCQk+HwyFPSkrCgAF0B89nn32GN998EwaDARRF
    XVPvzte0Wi3Wr1+PjRs3QqPRQCAQ2GJiYk7GxsZ+GhkZ+WNsbGyNx+PxPvTQQ4iIiEjQarVK
    gKa87whpC4BmGw0LC0NjYyNsNlsQRVFZBoNh97x58/w1XJ4PYAmA5RqNpt+KFSt4eXl5eOyx
    x9C3b18IBAKYzWaYzWY4HA643W44nU54PB4IhULw+XwIBAJIJBLIZDKIxWJUV1dj27Zt+OKL
    L1BSUgKv12sA8CFzOPCHtTkbGWDmYQChhyRJnlwuR3NzcwsQSBAEwsLCkJmZiTlz5mDq1Knc
    vSSXyzF06FAMHToULpcLFEWBz+f/LonF5/PRs2dP9OjRg7vmeDweRCIRfFnRJk2ahKioKLz7
    7rvYv38/tFotzp07B18pFzZoY8zSgY5cA+ANAPEej2fA7t274fV6sWrVqhZdAW35zpmqCWw2
    G/f5PR4P3G43vF4v9Ho9J29hMplw+fJlKBQK8Hg88Hg88Pl88Pl8TvtTIBDcdMdDfn4+FixY
    gL1797JkZceYzxco1UEeaGHgp0Az1kbj1luBpaDZcUcB+JFJjOb5KzGqUCjw7bffIjo6+i6b
    zdaiOigWizF79mw88sgjuHDhApYuXcoFSDcDZIKCgoxKpfJkSEjItujo6J8zMjKqd+/e7SZJ
    srtWq223ln6Hw4GioiI4HA6OqY8gCCQnJ0Mmk7G+Ra7VapPKysoOvvDCC/jpp5/8sewXASwG
    sBz0bF+L5EJoaChGjx6N+Ph4BAcHc9UJgUAAq9XKSU4YjUZcvHgRhw8fvlpQ2gzgAyYJ+4cf
    vHPNy/hBF4DbYv31jWMJggBJktzh8Xjg9Xp/RxZ5u3GB1+s1ezyeWvb129mOA/gYwF+Li4uD
    Fi9eDB6PhxEjRkAgEOCBBx6Ax+PB4sWLOR3PtqwPSZKcT7NarfB6vQgKCuK6Y8RiMQQCAbxe
    L1wuFyQSCWw2G3w/Y1vWLTk5GStXrsSUKVPA4/Hg9Xrx22+/YdGiRfj111/B3LefgB6dgT8B
    YRSAl0HTyYbEx8d7Zs2a5Zw9ezYvLS2NdzP0sUKhEMOHD8fQoUMxc+ZMfP755/jvf/9LXrly
    pRuAFUymdE1nZ0H4fD7eeOMNkVwu7wqAjImJQdeuXdHY2IidO3dCr9dj4sSJePnll31Fkq9q
    VVVVWLVqFT7++GPY7XYqJCSkRKlUfpiamvrZSy+9VHvw4EGsXLkS77zzDux2O5GamjqKoqgw
    Pp+PzMzMG77+7QBCpVLJsnoSALLnzp0rrKur85eDYIkUHgfwnNlsvv+rr75SHj58GH379oVE
    IoFGo0FjYyNMJhPsdjtHs02SJIRCIcRiMUJDQznRzqKiIhQWFsLlcnkZR/sv0K1mhjvZCzDg
    guoMwCaVSqng4OBAXIYaABaBQBA8bdo09OjRA5WVlTAajZBKpYiKikJWVhb69euHqKioawKR
    tmjmkSTJBZHXyHRi4MCB6NatG3777TccOnQIly9fhlarhdfrRWRkJLp06YKDBw+yVURjB+9p
    p5jkyhter7fX3r17YbFY8Nprr2HoMdawqQAAIABJREFU0KEtwKzL5eL0zBobG1FRUYGamhro
    dDo0Nzdzh9lsht1uh9Pp5A673c6B8GPHjuH+++9HUFAQ19kgEokQFBSEkJAQhIaGIiwsDOHh
    4YiIiEBsbCzi4uKgUCg4bVHfCqPH48GRI0ewbNkyHDlyhHWmecznOhcg16AcNJvtPNBMfu3R
    90QwoHIukxh9D/S8c6fvWRKJBJs2bRK++eabmV6vV+B7z8yZMwfLli1DZGQknE7nTbPL8vl8
    Z0hISF5cXNy/u3Xrtnvu3Lna0tJS6vnnn8d7772HjIyM/larVdmen6eqqopjxGUtJSWF6yrw
    eDx8Pp/fb/Xq1Z8cOHDA5cfr6ghooohnmesrgv1BRkYGVq1ahYSEBC5wv1oiBwC+//57nDp1
    CiaTyRdAXACwHsBXd7of/MMABhBacZ1RK4IgIBKJIJPJIJPJIJVKERISgrCwMCiVSoSFhSE4
    OBhBQUEQCoUQCATcI0mScLvdcLlccLlcLfZ+vV4PnU4HrVaLpqYmmM1mWCwWmEymFpp510k6
    mkwmUxWbsGlnszHxXhyAh/Ly8vjPP/88FixYgOnTp0MsFuOhhx6CzWbD66+/Do1Gw8WPISEh
    iIqKQkxMDEJCQiCVSiGVSiGTySCRSCAWiyGVStG9e3cONLM+lU38spXI7t27w2q1cgerg22x
    WGAwGFBbWwuNRgOj0YimpibY7XZER0dj6dKlHBh0Op347rvvsHz5cpw/f55NiH8NutPthkzl
    HQkIuzAOeSaPxxMPHjzYsWDBAt6IESMEIpHolgkSeDwe0tPTsXDhQgwePBhvvvkmDh48GOxy
    uZ4EEAzg76BFgDvFlEolYmNjIRaLCYBuh5JIJLhy5QouX74MmUyGRx99FMnJydd9ncrKSixa
    tAjbtm2D1+t1KBSK3enp6W+lp6f/ptPpPKzGIEVRGDJkCE6cOJFsNpvHud1uXkxMDCdm3RHG
    AqfLly+Doig0NTUpTpw4QXakoHUbQWEBgFcB/ALg+cbGxj579+4lWwfoQqEQQqEQPB4PLpcL
    FosFGo2mhQwBY2YA34BujzmLO1iAt6ysDCkpKRg4cGD0hQsXQgGgqKgI77//PoRC4W0DQ5Ik
    YTQaOckCq9UaVlRUFJqXl9fY1mp4J9oZAJccDseAsrIyLFiwAAkJCXA6neDxeBAKhW06X7bq
    5fV6uWCKbb1iAy7fRz6fj2slvUJDQzFu3DiMHDkSVquVE4YWCoU4ePAgdu7cyV7jBzsYEFKg
    dVz/CmCZ1+sddOjQIUKj0eDVV1/FqFGjoNfrcf78eZw9exbFxcW4cuUKGhoaOODXllYXtgIo
    lUrh8XigVqtvGASQJAmBQMBlVWNjY5GQkIDk5GRkZWUhKysLYWFh2L9/P9asWYPCwkL28xxj
    fM9RBAZFfgSAF0FXBjuCnIoA0B00+VBXAG+CbhHqNBOLxbDb7XyJRBLGtiQCdCvTs88+i8jI
    SJw/fx7Lly9HRUVFm5NZIpGoVqlUfpSYmPjRxo0byw4cOEBNmDCB2+OWLl0avHXr1vEURbVr
    NrSurg719fWsjAwAICYmBuHh4ZzfMBgMsd9//z3PYDD4ExBSoOegFjJ+8C8A+hEEQfbv3x/x
    8fHX3IPYYBQAevXqhYyMDJw8eZL1gzsYP3ge/weE6H33d5lMBqVS2W4+yuPxQKPRwOVyAQx7
    5PWSgv4w5nsuoCiq0uv19vRN2ISFhSE+Ph7p6eno0qUL4uLiEB8fj/j4eCiVSgQFBUEgEHDH
    1TpkrmdsBczpdHKPFosFarUaVVVVqKqqQk1NDSoqKlBaWor6+noYjcYWMQpJkvlgNAM7oELI
    guVloGWo7i0qKhK88sorqK2txZNPPgmlUomHH34YJEni4sWLyMzMRPfu3REXF4eQkBCIRCII
    BAKu4+Va1xZbNWS/E/b3evbsyc0o+yZsfDtt7HY7DAYDqqurUVhYiMLCQtx111144IEHwOPx
    YDKZsGXLFrz55pvszKIb9FjI35mk+I39dAddf+EAFgGYKxQKBdOnT3ctWbKETEtLa7f3EwqF
    GD16NJKTk7Fq1Spq69atQQ6HYxaT2VoIuh2q4z19RARmz57tWrdunRoAZbFYCKPRCKVSiSlT
    poAkSQwePLhlOsJm4wAKQOsDrVy5Etu3bwePx7MplcotISEhq06cOFEzfvx4fPzxx9zfHjx4
    EMeOHSNzcnImNTc39wCAnJwcZGdnd9hnFAqFCA39XxxjMpmkMplMTBCEbfTo0di3b58/9zoD
    6HbhIADdwsLC5Lm5uYiOjoZUKuVaZRQKBYRCISdAajAYYDabYTKZkJ+fjxMnTsDr9ZYDWIsb
    DN4GurEAZeTIkV0LCgqW2u327gBw8uRJnDlzpl3fhw3qtVrtgIKCgvm9evVaPWDAAF2ALUkZ
    6HmbzF9//VX22Wef4ZVXXuEqFRRFwWazwWKxwGq1QqfTob6+Hs3NzTCZTFyWzrcF2ffwer0t
    2h592x/ZTKtcLodcLodSqYRKpYJCoeCyh+wcIkBrBn3yyScsIUoRc94dnXnxMsDzL6Dp7McW
    FhaK5s+fj+7du0OtVqO6upqrHvB4PK7dWqVSISoqCqGhoRCLxRCLxQgKCuKqf2zlj/2ZQCCA
    y+WCzWaDzWaD3W6Hw+GAw+GA3W7nDqvVCr1eD41GA5PJBKPRCLVajd9++40L6mJiYhAVFYXi
    4mK2BdEOmjl1GZPQCQSLZ/zRHNBaXx1pCgDPMKDzHwAqOutDhoWFITs728Pj8Qy+gVy3bt3Q
    pUsXOBwOfPDBB232FSRJIjg4+HJYWNjrAwYM+PbcuXN234DplVdewZo1azB8+PAcnU539+3M
    31zN9Hr972YdJRIJBxCZxKgsPz+f5/F4oFKpoFar/XmdmQB8AbqVuLtcLpdnZmYiPz8f9fX1
    MBgM3L3Fjk6wyWulUgmlUons7GycOnUKFEWVg261vog73FwuFwQCAXJyciLPnz8fDADDhw/H
    0qVLufmt202MNjY24i9/+Qvy8/Nht9tDioqK5B6PJ6BI6Jj9uNhisWx1Op3/oChK1LVrV8ye
    PRs5OTlITU1FeHg4ZDJZuydzSZLkfIGvpaSkcLGx0+mE0WhEQ0MDLl68iF9++QXbt29HfX09
    eDyeVigU/sdms+kkEgk6sBBRBmA+6JnsWQ0NDbLVq1ejqqoKf/vb35CamorHHnsMXq/3qnwd
    Xq8XTqcTVquVA75sAtnj8cDlckGn03Ha4RaLBZWVlXC73RxQbH2wIJz1eeHh4UhJScHw4cPh
    dDq5v6urq8N7772H999/n9UctILucFsOWi6mTdYRgFAC4GkAswQCAX/mzJnOZcuW8ePi4jqE
    0TQlJQXLli0j+Hy+96OPPhI4HI6HGLS/BvT8TcdGUl4vCIJw9+jR45JIJHLX1tYKLl++jBEj
    RmD58uUgSbJFi0xZWRk2bdqEiRMnYvjw4XC5XNi6dSs+/fRTAHDExsZuSktLW15SUtL0xRdf
    4MEHH+T+9v3330dubi7uvffeAVVVVY/Z7XaxUqnE9OnTWwC29jaBQADfVkCBQBCWkJCgIkmy
    qSPZAm/CRKDbsGSjRo3CO++8g4iICNyoJZkFNIcPH8acOXNQX18fDVq37o4FhPPmzQNBEBg9
    enTX/Pz85Y2NjVNJkhTEx8e3iwO8WubVYrGgrq5OqtFongZAdO/efVVqaqqupKQkYDAy6Mrv
    RIfDMXnr1q0YOnQoEhMTcenSJRQVFaG4uBhVVVVQq9XcXA0LWJjM722tEQuMJBIJgoODERYW
    hpiYGCQmJqJbt27IyMhAcnIydu3ahZ9//hmgNYP+i85reaRAt1nOY0DFI2q1OkKtVoMkSYSH
    hyMrKwupqalIS0tDYmIiYmJioFQqERwczIE93yzyzYwEsFlQtt2IBY0mkwlNTU2or69HZWUl
    SkpKUFpayj0WFxezL1EPes7p36CZgQPBwkB3MDwKQNhJ7ykGLTDNJkYbO+NNHQ4Hhg8f7khI
    SCgkSdLt8Xj4AN16efLkSRiNRuzZs6cFuOrXrx9qampQVlb2uwAyIiLiTGxs7KtTp079+ddf
    f6UuXbrUYt+eNm0a5s+fH/Xtt98+ZjabVe39edi2rdZ+0Dd54/V6g6Ojo8NIkrRER0f7GxCC
    SYpmAZDZbDa88cYb8Hg8MJlMsNlsV01gsWRPcrnc9/PGgK4039GAkKIoxMTEYPz48Qnnz59f
    4HQ6u7PjNdnZ2W0aAWiLNTU1ISUlBfn5+dBqtTmXL1+el5WV9fbgwYONx48fD4i1CAkJgV6v
    94hEoi89Hs94t9s9QiQSYfLkyW0uJrDdMOyMYOvnV+uUaf38ekWH8PBwhIeHIzMzE1FRUfjm
    m2/Y++5bqVS6PygoiAU7HWmVAF5j/MnTJpNJtXnzZlRVVWHp0qXo378/F/NYLBY0NDSguroa
    1dXVUKvV3PhEU1MTmpubuQQMe7hcLq6r6sKFC3j44Ye5JKpvmy77nE0mh4eHIz4+HnFxcQgP
    D+cSriRJoqioCCtWrMDXX3/N3sMa0NwX63GT3UXtDQgJAPcAeIYgCMn48ePtr732Gq+jwCC3
    e8XEYPHixURTU5Prq6++ElMU9SSTXf8MHdwy9Ntvv2H8+PFwu92F1dXVdWq1OuGHH37AwIED
    f8coarfb8Z///AcbNmzgUH5eXh7+/e9/w2azUSqVamd8fPzaK1euNJWWlrYAgwCt8zR69Ogu
    Z86cma/RaDJ5PB4efPBBjB8/vkPvED6f3wIQ8vn8UIIgogAU2u32QNjvugIYKZPJiAkTJkCl
    alt8QBAEBAIB+vTpg4EDB2LHjh0RoFkXfwLdV35H2QcffIAnnngCY8aMSb5w4cKyxsbG+3g8
    Hn/GjBl4+umnIZPJOgQQarVarF27Fnv37mVBIZWdnb2yublZ/9VXX2HYsGGBsDwaBjAMKC0t
    jXruuecgFApRV1cHnU7Xon2Rx+NxpEMsExibUfc92P/j8/lwu91wOBxwOp1cxcv3ucPhgM1m
    g06ng1qtbtH2EhQUxM0hqNVqNoN4AsA20G0fnWmVTFbxAuj5714qlQpr167FkCFDoFQqIZFI
    2l3ihm21aZ1Fbp18s9vtaGpqQl5eHl544QW2/fAcgJUA9nRGEvAmgvMnQVcGhZ383gIAD4Fu
    E3oDHV9hxqFDh5CRkQGv11ssFAoNNptNCQBnz57FE088AafTyQEmgiAwZcoULF++HFu2bMHq
    1atb3H8KhaI8KSlp4a+//rpvwYIF2L17d4v3euCBBzBgwADh559/PqempmZie1cHAbpi0VrO
    qLUfFIlE4eHh4dEEQVTf7FxkR/pBAITL5fJNllw1AcNWM1ipHR8LBzCe8YNW3IG2YsUK9jqL
    P3369GsajWYmQRDCqVOn4oknnmg3MAjQ1fGXX34ZWq0Wx48fD25oaHiRIAj069dvXVZWlik/
    3//55bq6OoSFhWHFihVVL7744n88Hs9dly5dUnz++efIyMiASCRiWeTR1NSEpqYm6HQ6NDU1
    tagu+3ZytH7udru5Fn8WsPg+ss/FYjEUCgVXmVYqlVAoFBzAMZlM2Lp1K6qqqiAQCMrFYvEW
    nU5nuueee7Br167OihXeZDDEyy6Xq/fevXsJjUaD+++/H1arFWVlZaisrIRWq4Ver4fBYIDL
    5eLAcVvMYrHg8uXL14yrfI+goCAoFAqEhoYiKioKiYmJ6Nq1K0JDQ7Fjxw7s378fbrebAj0e
    8xboVtGb1rFob0CYBHq4WZWVleVYsmQJLyEhQdAZ32BcXByxcOFCory83JmXlxfJnMcpAMUd
    +b5DhgxBQUEBlEplfmlp6T6j0fj4J598grS0NMycORNBQUHc7/7666/44osvEBoaip49e8Ji
    sWDLli0oLi5GVFRUXq9evdbu3btXvXv3bkycOLHF+zDSD1GFhYWLGxsbJ3o8HmLo0KF4/vnn
    0dFEHmyF0IeuWmYwGEIAtLsG4C0YCWAEgJSMjAywgrCtZ7x8M1jswWatQkNDMXHiRPz444+w
    WCzDQWtlBgoZRZuzd2yb6Pnz55c3NjZOFQgE/Dlz5mDp0qWIi4vr0PcPDw8HQRDYu3evRKPR
    PHPu3Dlvdnb26pdeekkfQDOFhwB86/V6Hy8oKCAJgoBUKkVSUhLi4+O5+Ym4uDjExsYiNDSU
    G5hnj9b/ZucG2Ozf9Q4WzNTU1KCmpgbV1dWorKxETU0NGhoafGda9aBbRcv9tE4m0OQkYQDe
    cLvdIpVKxek1+u1GZ7otJBIJqqurWRDhBPCJQCD42uPxeDtovuRWE6PPg5aY8IexnTolzHfZ
    oYnR4OBgWK1W8Pn8s1Kp9ILNZhsB0C17rWe1u3XrhmeffRbx8fG/S8bIZDJdly5dVk2bNu3n
    Rx55BB999FGLv2VazMjNmzdPqaure95isXQIk5rX6/2dLljrThmCIILtdns4QRAwGPzOucL5
    Qd/7hc/n/y6Jxc7St05aeTwe32B2OIBuTIB5R9mOHTswdepUTJ48Of706dN/b2homOl2u4VJ
    SUkYOHAgioqKrhmI32pSlKIo3H333SgoKIBerw9Rq9V/O336NNW/f/91DofDXFRU5Hcf2NTU
    hCVLlkAul//I4/F+tlgs07Zv346MjAwIhUKcPXsWBQUFaGhogMlk4sZqWHZtNs7wlSy5Hvjx
    ZctuDXCEQiHkcjlCQkIQHByM2NhY9OrVC9nZ2aivr8euXbtAkqRbIpF8rlQqT0ul0s4Cg6xZ
    Qev0XgY9RvGnvLw88vz586AoCh2RhGodz/muLcuaX1NTg/z8fG7+lyRJ9vuhQI9LLAI993tL
    jrA9AaEAdKvKQLlc7n7hhRc8ffv2FXXmN5idnc1/8cUXHc8995y7ubm5H2j2tb8zQUOH2OLF
    i1nKWFufPn0+NJlMg+vr6zMWLlyI8vJyTJ48GcHBwbh06RLeffddVFRU4LHHHkPPnj1x9OhR
    fPfddxAIBHqlUvnuBx98cG7BggUtwOD8+fOxceNGpKSkRBQUFCxpaGiY43a7Bf369cM//vGP
    1tpIHRaI+Q5JezweodFoDCII4rbFNNvBIkFTuIvT09NRV1eH/Px8XLlyBfX19TCbzRxjk8fj
    gUgkglQqhUQigUKhQEJCAhITEzm2qPLy8iQmy5qPO2SYfvHixSAIArm5uV0vXry4srGxcapQ
    KOQ9/PDDWLJkCWJiYjpcDiIzMxNr1qwBSZLYs2ePRKPRPAeASEtLW5Oamqr785//jH//+9/+
    Xiojs8lPBBA7efJkPProo0hMTERcXBzkcjm3yXa0sXMFer0eNTU1KC4uxhtvvMHOeJ5mNnd/
    EqJ4AZSCFq4XscK6gWJVVVXsjJcNQBnDDhwolgRaZinaz+cR5ZMY7dD+7ePHj2PZsmVYsmRJ
    Y8+ePT8zm8397Hb778CwUCjErFmzMGjQIFy+fBnHjh37XzDC57ujo6M/HTly5LaDBw96WlcG
    J02ahNTUVHL37t2T6uvrVxmNxviOvD/NZnMLuSg+n8/NWDFBodhgMEgJgvBl5/SrHyQIIigz
    MxN9+/ZFYmIikpKSEBER0WKmlw0iWUDItPzjypUrKCsrw4kTJ1BbW5sAWtLkjiKVYb+vSZMm
    JZ45c+bvDQ0NM9xutwCgifteeeWVDvGF7DXCjhfYbDZ5fX39S6dOnaJ69uz5z7Fjx5oCYX2G
    Dh2KnTt3art27bqttrZ2ZEVFRei8efM4RtC2rE1rcOfbJuqbiPcFNq0rZ06nE2azmWufBIBv
    vvkGEokEfD4fBoMBYrH4Ulxc3Pbm5mZHW+QeOsgHngfN0jkOgORGRGjXWyf2/3zX6nrHja7z
    VufiBPA9QRBnb+f6bk9A2A007bFgzJgxtilTpoj88Q1OnDiR991339m3bdsmA3A/aMrkDiUY
    EAgEeOaZZzB48OBT//znP1e43e41Go0mdu3atfj4448REhKCmpoaGI1GdO/eHY8//jgcDgc+
    /PBDaDQahIeH/xwTE7Pn4Ycfppj5Ic5OnTqFESNGhJ8/f36hRqN51O12iwYOHIg1a9Zw1bAO
    T3e31EWD1+slKYoSVlZW4u9//ztee+01f+5xvUGzquHnn3/G/v37odfr4XQ628RGxefzIZFI
    EBoaygaYIubm/xyMoHmgW3NzM5555hn5zp07X9JqtVO9Xi8vOjoa0dHR2LNnDzqjasJeIykp
    KVAoFNDpdBKNRvOcx+NpqKureytAKjcAPTheBSA2MTER48ePv2H7UOtsKOvg2Jk3j8fDzeSw
    9Nu+DsD30TfJQpIkIiMjERkZiYSEBKxfvx4MCCxGYNC81wNocLlcyoqKCu5ztg6cXS4XrFYr
    1zbkcrmu+shWI9iqBTtryK6b73OhUAipVAqhUNhiPX0BITMvoUUnM2q2ITH6EGhduECwfqDb
    Vv/RkYlR1saPH4/evXvvdDqdAysqKuY6HA5e63upvr4ep0+fxhdffIGioiLuZzKZrDAiImLz
    W2+9Zdy7d2+LVtHJkydj9OjR5Pvvvz9OrVavNhqNKR0NLFoHfyxzsM+1z3M4HEKDwYChQ4e2
    a9XpFqwvgH6RkZFYuXIlxo0bd9PJLY/HA7vdjrVr11IrVqwQeTyecQA+BVCLO8DWrFkDgiBw
    7733Jp4+fXqZWq1+0Ov1CiIiIjgSkI5OjLL7FCMVIK+vr3+JIAj07t37nZ49exqfe+45/PnP
    f/bbGu3cuRNhYWHweDxn3G53CUVR/X2TGewYDVu9YyUnQkJCOBkF30f2OctEyrJhsqRhVqv1
    d88NBgN0Ol0LHUwWjFosFt/7q1Cn05X6mc0eTByox1VIwYRCIWQyGVfpDA0NhUKhaLE2rZ+z
    a8WOQLAkaldbL5PJBK1Wy60T2757levYBqD6dq/v9gSEowGkKJVK16OPPkqEhYXx/PHNKRQK
    /uOPP04cPHjQ2djYmMgE9+fQwdn2DRs2ID8/36tSqbbbbDYPj8f7i8lk6l1bWyusr6/nqO3T
    0tIQFxeHL7/8Ej/99BOCgoJqIyMjt/z000+6sWPHtkQ6vXsjOjo6/MSJE6/U19c/7na7xcOG
    DcOqVauQk5Pjt7vD7XYTEolEBgC5ubn+BIRBoFuzlBRFQa1WQyAQQKFQQKFQcFlRdkiXz+fD
    ZrNxVUOXywWz2Qy9Xo+qqirfm6wPE0h9eyc4QrfbDavVGmqz2bI8Hg+PDZhXr17tF6F4VkLB
    4XCItVptN7fbjQAhHwIAHQsgrly5ArvdDoFAAIfDAb1ez7GKNjY2cpo/rINjN282m8rOULBM
    duysBNtW6suwyc4AREVFITw8HHK5nHMgAoEADQ0N7IyVgwGtzgBYKzUDCruXlZVBr9fD4XCg
    oaEBFRUVqKurg1qthlarhU6n42QoWjOGskCRJSDwbXcRCoVXnTmRSqWc9lVkZCRiYmKQlJQE
    lUoFiUQCH8IidYABwjQAD6Lz5wavB1DZxOj5jnyjpUuXshUa3YQJE1ba7XZhQ0PDvXa7nWvr
    dLlc+PDDD/Hdd99Bp9P5yq04FArFp9OnT7+Ynp7eYi5+9uzZ6NOnD/nRRx+Nra2tXWM0Grt1
    xsL5jhlcA0ARJEkGGQwGzJ07F99//70//eBEAGEDBgzAwIEDf5fkYgEumyj1bXdnjcfjQSqV
    YuLEicTWrVtRXl5+F4ABoCUoAt60Wi1efvnl4G3btr1UX1//IEVRgjFjxuCJJ56AQqHoVF9Y
    VVWF9evX4+zZs8H19fUvnT59WldeXr7hvffe8/s6GY1GmEwmjcfjqQbQn8/no0uXLkhOTkZm
    ZiZSU1MRGxuL6OhoREVFcbETu2ez896+mnpXM7YLpvUjE69Ao9Ggvr6e65ApKChAUVERampq
    4PF43B6Pp8zhcDhvpirXgYlRNYBYkiQRExODlJQUZGZmolu3boiLi+PYtuVyOZcUbr1ON1qr
    1uvEHjabDY2NjVCr1airq0NpaSkuXryIoqIiXLlyha1KN7aHH2wvQKgCPYQs7Nevn3XgwIF+
    dYYDBgwQ5uTkWHft2hUCYCyAT9BGHY7bsaNHj2LZsmXOYcOGbdu9e/fZioqKcXa7PUOpVNq6
    desWeebMmdwTJ06o5s2bh9OnT8Nut3sSExO3DR48+FBGRga2b98OgCaqmTVrFgYMGBC6b9++
    l2pra59yuVyy3NxcrFmzBn369PHr3UFRFMHn88UUReHbb/2KmdIA5AJAbGwshg4dit69e6NH
    jx7o0qVLi42M1c5hNyS2cqHT6VBSUoILFy7gt99+w8mTJ2G1WkMZoLkPd8BQPePouKiFDab9
    NbPg9Xqh1Wpht9vh9Xopdj4lQMwOmo7fe/nyZXL79u3Q6XQoKyvj2owNBgMHAFnq6FbzNTdl
    vox+rPyCQqFAbGwskpOT0bVrV+h0Ouh0Ol9AGAglVSOY6sCxY8fw4osvcnOPBoOBqwpeLUvO
    OkD2/vP9N1tVZBkQ2XvyWmvMVvHlcjlUKhXi4+Nx4cIFX0DYFEC341gAqQG2RSQz/vlCRydG
    CYLAiy++iHXr1pVNmjTpZZFI9Kter5+h1+vjwsLCqMjISGl1dbWyqqqK5/s3CoXieHJy8pcb
    NmzwPPDAA9zr/eMf/8CMGTOIuXPnjq2srFzb3Nzco7MC+xu9j8fjIXg8XhADaP35/aYDyBWJ
    RBg0aJDLbrcL8vLyOOILg8EAh8PBMae63W4EBQVBKpVy+1FkZCRSUlKQkJAApVLp6dOnj6e8
    vFwBYBKAHxE4ZE3XNCYRFWE2m/u73W7BoEGDsHr1avTq1csvPlmlUuG5555DeXl5sEajyaYo
    Ckql0u/rxMyWOrxe7xUAHqVSyVu5ciVGjBjBic631/tcq0KtVCoRHx/f4rszGo3Yvn07FixY
    AKPR6HC73UV6vd4VAJdWAwMKIZFI8NJLL+GBBx5ASEgI2ktvkl0r3w4EX4uNjYVvUs1kMuHw
    4cN45pln2EQym7wNCECYDqBRhxADAAAgAElEQVSHUCikJk6cSISGhvL9+e0FBwcLJk6c6Pnh
    hx9cLperO+h21prOeO+lS5cCgPf5558vSktLK05PTw+aO3euhyAI8Zw5c6YcPXr0hZ07dybx
    +XxHaGjoT0lJSeu3b99u89U82rhxIyZMmCDdt2/fszU1Nc+4XC7Z6NGjsXbtWr9sbte4gKUA
    yOrqan8Grrlghuibm5tRWlqK4OBgiEQiEASBiIgIBAcHc7NhPrMfXPm9uroaxcXFKCkpQWVl
    pW+AO5wJ7M7jDrORI0di0aJF8AfzHUEQYEXNGZFjBAUFBZJQvQt0S6ajrKxM/OKLL3Lzpez5
    s1UrhUKBsLAwrsLXmi2Nfc7j8VpUDK/GxGaxWKDT6WCxWKDX61FZWYnz5+lLi83WMwRNTQic
    ipeTAaeuiooKASsozjLJhYWFQaVSIS4uDjExMS3W6mqHrw6hbwWx9WGz2aDX61FbW4va2lpu
    FrixsRG1tbXIy8vjcg8MuHcEyHqpQHekCANsSxD4JEY7vP3v7bffxttvv40FCxaoBw0a9P43
    33zzrdlsDs7MzKTuu++++C+//PKJU6dOTbbb7SKSJCGXy6u6deu24cCBA5WHDx/mWIk3btyI
    p556CuPHjx9RWlq6WqfT9ejMKk8b3osQCAQyiqKIDRs2+HPeNxdAMkVR7u3btzu//vprfkND
    A2E0GmE2m3GjCgvLYhgcHAyFQoHo6GhPXV2dnbmOhzGJ17MIcGMqugSbHM3IyEBqqn9yMwRB
    oGfPnoiPj0d5eTk8Hg/FBvN+39TpDhcnaMIUh9vtlshkMkRGRl4zwcsm7FpXstifsfOorG9g
    E/Bsdcz3+dWE7UUiEScV5nA4QJKkTSQS1RIEgQBoGbWDkTJyuVwICgq6JpM9RVFcspNdH981
    83g8HIkTn89vsVa+68Qerav4rP8NCwuDUCj0vZ7qATQHAiAkGMAVEhkZiZycHH4gbA6DBw8W
    RkdHo6qqSg6gO4AD6MSsO9MaQAGw/fWvf8XIkSOd8fHxn8XGxv4qkUjiXC6XSaVSlf/0009N
    /fv354Ln5ORknD17ltRoNPc1Nzc/73Q65WPGjMHatWuRlZUVKHsvQRCEFACp0+n8BQgJJrjZ
    AyDbYrGoTp8+LTx9+jTXNiqRSDhpgJCQEIhEIhiNRlgsFu6mZFsEmRk3L5MJLQPwK7MR3HEW
    GhqKzMxMSKVSv71/a8mVQIobQM8QOjwej5jH4yErKwsqlQrR0dGIj49HQkICEhISEBUVBalU
    2qLK7CtAzz4nCKKF0/R9ZJ+zWb3a2lpUVlZy7KJsG4hPy3IDaNrrQDA342hcQUFBgmHDhiE7
    OxtdunRBQkICYmNjoVAoOK0ktkpyO+CfDcJZ0gGW7a6urg6VlZWoqqpCfn4+9u/fD5vN5mLO
    zx0g65UGWg81EI1NjHbaPNiqVasAmpCkFkDtgQMHAODygQMHCiMjI0+o1epMmUxmCQ8P35uU
    lLS/vr6eA4N9+vTB66+/jvXr12dqNJrXdDpdz85uf28LIOTxeKwf9FcLBMHsZ7ucTuddeXl5
    Kvh0i7T1c7JzSw0NDSgqKuIz++QZ0PI3d5oEEwUA5eXl2LFjR4fo797ISJKETqeDRqP5HVAM
    BBOLxaAoqsbhcNjNZrOkpoaul9hsNjQ3N3PjAA0NDairq4NWq4XFYmkx38bOu9ntds7PsT6R
    TQC2njuUSqUIDw9HdHQ0IiMjERUVhdjYWISFhUEgEKCqqgoOhwMikagpKiqqiSRJlJeX+3u5
    nKBJuRwOh0PEtLTC5XLBYDCgtrYWDQ0NXAusRqOBxWL53Vyg1WrlRkw8Hg9XEWQ7h9g1Yg+J
    RAKlUono6GioVCpERkYiNjYWERERkEqlqKqqYsmsPExi9LbHTNoDvIkA9AIQFBMT40xISAiI
    7GhsbKw4Li7OVFVVpQDQkzlPv21s+/fvZwOsEuaAVqsFAA4MRkREQKfTwWaz9dXr9fPtdntk
    dnY2li1bFkhgkHVCkuPHj5OMZpq/Nv2vAPwMukrYn7kOE1wuV2xjY2MyAAm7OVmtVni9Xpak
    wmO320nmc6iZm6mW+V5Ogm6tqkHgVB7aO5D5/+IcrucPAZAymQwLFizAtGnTEBwcjODg4FvW
    p2K1C29kviLAdrsdBoOB03H873//C9BZ+UCaP0sEIIyPj8fatWvRs2fPjs400U6FYURkW6x8
    OyOKi4tx7733orCwUAAgjjnPQLhX0wEoAvSaD/ZJjPrl5mS/26eeeqp26NCh63bt2kWqVCrq
    /Pnz1Lx583znQlFZWQmJRBJZUVHxqtVqHRKo+wlJktKioiLSZDL5CxBSoFkQWT/Yl/GDiaAF
    5lMAyPl8vlculxMmk4lwu91cstRgMLB7dR1omZs6xg+eAs20XXun+sGjR49ysZW/fKDdbg+k
    7pj/Bf506z5FkiQcDgf27dsHvV6P/Px8lJSUQKvVorm5GSaTCR0xw8fj8TjimvDwcKSlpSE9
    PR1Hjx5l9wovn8/3dgbjdxvMC1qKyQPQjMpr165FUVERioqKoNFouLXqiAowSZKQSqUICQmB
    UqlESkoKsrKycPbsWXYGmz2/2y7OtAcgFACIBU0xTwVKZUAikXjT0tIcv/zyC0DTf/sVEN7I
    Vq5ciUOHDiE8PDz4p59+esZut2eGh4fjhRde8PvM4FXvEK9XXFZWRvqyQvnJGepB0/SfBq3F
    FEyS5GiKolbzeLzkmTNnIicnB8uWLUN9fT3Gjh1LTZgwofnNN9+UlJWViQEcBS3EXYw7tCLI
    tOFRbKDn7020FSut93rD1H4COb0BSFQqFcaPH4+kpKRrOvSrtX60bgGhKOp3Q+RXO1qvA9tK
    GRUVhdzcXHz11VewWq1dQM98FQfAWoUAyAbA79Gjx1XXybciyrKNsgc7O+p7sCQdvnOFPB6v
    RZaUDVZ9K7Gtkn3IyspCYWEhCSALtF5ivZ/XSsQkHsUBuk0EMUAhyN9+cNOmTdzeUFhY2CKR
    cvLkScyfPx9Dhw4lN23aNMPhcEylKMovG1pbKoQAJIWFhf72gwDNSnyGOQgActBEf2sASCdP
    ntw0duxYxbJly/i1tbUYM2YMpk6dijVr1qCgoAAAfgPwGmgxbif+DxjboeFvC0RAaDKZCCaB
    JQGAr7/+Gl9//fVVgZtIJOL8mq8eL0uexj7yeDyOm8HpdHKP7HOWWIz1Fc3NzWhubkZlZaXv
    GAAAwOl0RtbU1MQKBIJAGNkRAejB7J04ePAgDh48eFXg5rtWvozZrdeKz+fD4/G0WCffR1bk
    np23N5lMMJlMqKmpwfnz57Fjx47WMU06c563dcG3ByAUsxdVdHR00LWGIv0QJPMTEhIUPuco
    Rjv02HaUDR8+HAsXLsTQoUMHOZ3OiTweDzNmzMDUqVMRIFmS1oAwqL6+nmhHR0gw11E448x4
    PocDNDtkE/Pce53XSPV6vXMBdBk8eDAWLlwIo9HIOXeZTEbMmDEjVCAQ4NVXX4VWq81lMqHv
    XwcQEsx5SAEoAYQy4NPNZI1czLk1Mc87MtGBIUOGqEpKStKcTicPAHbs2EGFhYVFeTyeEABo
    aGjAiRMn2m3g+WadX2NjIyfUzOPx4lQq1Qif4Irg8XjWLl265Dc3N1tyc3Px7rvvduYpypnA
    nd+9e3dObN3pdEKv16OpqQlNTU3QaDRoaGiAXq9Hc3MzDAYDTCYTRzLDgiAWMLYmTuHxeBCL
    xdxcjkKhgFKphEqlglKpRGhoKJRKJYKDg8Hj8XDXXXdBpVKhvLw8mAGs++B//a9kAGkkSWLg
    wIGQSCRc2yu7PtXV1aiqqkJjYyMsFgvMZjP3yLKLtgaFrLP0XTNfNmBWJzQ8PBxdunRBfHx8
    i5YZuVyO/v37Y/v27fB6vV1BV0H8DQi5xGgAx8gBnxj98ccfodfrcfbs2XSv1/uo2+2W+Otc
    2lKVpChKXFtbSzLzv+3lB6UAInz8IJ85WD+oA012di0/SIJuD34MQEJGRobu9ddfpywWC48F
    SAKBAPfccw+sVisWLlyIpqam4aDZaDde514imPMIZvy0gvk/11X8YKdXFZk9hbqaGLq/rh8f
    uSUqwIChGDSbehC7ViKRCAqFAuHh4VyrYmJiIrp06cKxYbPgRiQS/e45n8/nQA07U9j6udPp
    hE6n48Ym1Go1R6jW3NzMiqzD6/XK7HZ7L4fD8WMA+MEwJvFIsmslFAo5n87O0rOjJkqlklub
    a60VK9FxvbVyuVzQ6/Worq5GRUUF6uvrubXS6/Wt5Se6Mf7nthLJ7YHeJCwgZIObQDCSJInQ
    0FBR63MMVNuwYQOeffbZoD179txvMpnC09LSMHfu3ICg62+1sQEAPB6PSKvVku0w8Ktgbrbh
    oOdvYpn/4/scVtCzVWrQLSzVoFs660AzIRKMkxoCWgOsZ2ZmJrlgwQKkpaVh8+bNnHBwaWkp
    dDod+eCDD+LKlSt47733lEaj8WXQlZDtTIbUzgR4StAtafHMoWKOCGZzYB2hA/TcVxXomYtf
    0EHSAUzlZZRer1/hcDgk7IbQ1NREOp3OYDaDdebMGb8lEljBdQBwuVwjtVrtAB/ASPL5/CsR
    ERGzTCbTZT84ySQwLJBxcXE4efIkCgoKcOHCBVRXV6OhoYEDN+ymfDsMo2zm0DdbGBISws1P
    JCcno1evXoiOjkZ4eDjKy8uFDCCUwf9ahBkAYoKCglBTU4Nly5ahsLAQpaWlqK+vb8HE2tZk
    AUmSvxMpvkFij5tHiYqKQkpKCjIyMmAwGNhW8CjQrZDHAiCLHNA+xscPBmxidNiwYVi8eDEx
    YMCAqWazOeDmMVtft16vN6ixsZFoBz8YyiSqhoOuRsQw/8f6QGErP1jH+ED2YPcK1g/OANAj
    NTXV/MYbb6BHjx7Kd955h2hupr/6goIC1NfXY+bMmVCr1Xj77bdDDQbDi8w5fAWacIQllonw
    8YGxjA+MZkAhwfi51n7wN+aeLOkIPxgWFobBgwcrS0pKujidTh5BEPjmm28olUoV7/F4pACQ
    k5ODBx54wC8MsARBQK/XY/PmzSgtLQVJkpEpKSl9W/kFR0JCQoler7ffd999WLhwYWcnh7oB
    gELx/9q79tiorjP/uzN3nn5cz+Dxc2yH4IkdwAa8pJSGhZJEBWmzm6ZaUm2yWm1btapokKps
    9g+qRFWLVClSq2pbqd0tqw0hBfIoKGGzZZMlwQ5JbAgJD29qBzODjWfssWc8T3vuzL1z790/
    7jmXYTAJjR/3sro/6WhGZjScOfec833f73vV4Dvf+Q42bNiA+vp6NDQ0wOfz3dBfcDHltKIo
    EARBK7ZGWypMTU2hr68PR44cAc/zdgB/oSiKEeRggOoMbrcbTzzxBLZu3Yq6ujptrSoqKjRj
    b7HXShRFrUrwzMyMtlbnz5/H/v37Qc70Sqg57LobhFpVJ1rJ0SgoMU4ZIzO33/72t3HmzBlY
    rdbu6enpB6hg7OzsNMT85mvSK0kSOzs7y8xXdv42YYdaHe0fAdwPoKG6uppdsWKFxrxYrVbw
    PI/p6WnEYrHVmUyGhkYWyOCJQWghCrTH4XA4tm7dir1792Lbtm2YmZnBG2+8oTU8HR4eRl9f
    H771rW/h6aefhs/nw69//euqUCj0DUVRHiKXT47Mr4owaA4ANtrAvr6+XmPMcrmcFm8fj8ch
    CMLfQc3F+E8AB4hAXFQUi0VXPp/35fN5l9vt1vZ5aeiV3iFMNBySrJ+TGIfgeR5WqzUjiqJt
    AXtnIeggSg2OHTuGo0ePIpVK3dQ+gYYwVldXa6GMHo8HLpfrJsaPZVmN7SsduVwOiUQCc3Nz
    WgVN2veSVuxkGEZrqVCiVHYAaIOay6ong3wfABfP89i/f7/mHS01dKurq9HY2IjKykp4vV74
    fD74fD6sWLECLpdLY0NL+57RMBg6qHKQSCQQj8cRi8UwMzODbDarVQPOZrOYnp7G4OAgjh8/
    DrvdTp+ZE2r+8CHoWxrf8KQjmV+FkSd46NAhHDhwYFUkEvlGeVN7HfUHTQ6Wt8+RJMmeTCYX
    YhA6ADxE5OBXANR9jl5WWjZTIHIwD2AW16NsOAAOhmEUn8/HfvDBB5X9/f2WP/7xj/TMKKFQ
    iNm3bx86OjqQSCTAcRzS6XQlgEeIXM6WGITuEjl4u8/km1Bz898A8B+EaF00JBIJ8Dy/Mx6P
    /zSfz9Nzp8RiMTafz9cAapXR7373u7pU2waASCSCN998E1euXIEgCDunpqY2l+rNLMtera2t
    /YdUKjWig958D0iF9o6ODjz11FNoampaNmOZ5ohzHIempiYtR7y7uxu9vb20kMw9BpCDDCFo
    6gCgqakJe/bsWbaaHtQbabfbUVVVhfr6eqxevVojz3p7e/Hhhx9SImg9gP/GAsJGF8MgnKOC
    mPaUMkLYqKIomleodI5GhN1ux6VLlyyrV69+iOd5v9frxY4dO3SrElkOqsCV71XauPcL4msA
    fsUwzMr29nY8+OCD2LZtG9atWwePx6OxerIsI5VKaa0hrly5woRCIWc4HHbG43GO5/kGqlR3
    dnbi4YcfxiOPPIK77roLsizj+PHj6Ovrg9VqnXW5XFPZbPbuF154gdmyZQsCgQB2796Nnp4e
    HD16FKdOnaqempqqLhQKWtJzQ0MDWltb0d7ejkAggI6ODrS1tcHhcGh9DXmeRzgcxsDAAHp7
    eytOnz7dlUwm7yUXyR7CmC72/lbcbjd++MMforu7+yYPrpFA2328/fbbePHFF2kJbj2qRDhA
    8gcB0P49sNvtGtu3atUqBAIB3H333ZphQ0kK+szLB/19NOafvqclplOplBbqEY1GEQqFMDIy
    gmAwiOnpaSSTyXIDvpUYhXoKQi/UnDOGViG02WxoaGhAe3s7Ojo6EAgEEAgEsHLlSni93hsM
    ZWr83Y6iQ9erNOdEEASkUimMjo7Sc4/Lly/j8uXLiEajyOdviO5eA9VboecdbyHDyLAYmRjd
    t28fnn32WXR1dW2emZnp1PvOKveM0ErCZbLRIggCs4A+q18D8CuoRWC+CKlKiUvfPOeKGRgY
    cA8MDKBEVicBXMvn890vvfQSQ0Mqy+R4FRkLJR9WE6V+NYAnAYwt5jMSRbFybm6uked5l8HP
    HURRdImieMM8WZbNCYJgW8Rw49vmOqBGZXGAWtF3vv6IgiCA53ktUobneWQyGWQyGfA8r+UF
    lua80UiY0vDSiooKrRo1/Ttt21QexdTW1ob29nZqELYBuIdhmEs6FpVyE2LUAQCdnZ1obW29
    6UPFYhE8z6NQKGhtleha5XI5bY3oepVWZC0lTN1uN2pqarT2aaVrVU5Qeb1erF+/nhqENqgh
    wJVYQATIYlhuPGGnEI/HpWKxaDWCQVgsFhGLxYrkN84a2SC8ePEivv71r1fyPL9FlmW2ubkZ
    GzZsMMz8qIdwkQ9lHoCdZVns2rULP/rRj25pANfW1qK9vV27pLLZrNbXbWJiAlarFc3NzVp+
    lsVigaIoOHXqFH75y18ikUjAYrFcEAThGIBnP/jgA89zzz2Hn/zkJ1pD+/vuuw9TU1OaO76i
    ogJNTU3gOA6VlZWorKz8zBDMlpYWbN68GTt37sT3vvc9vPfeeyyuh9MsGZHwwAMP4MEHH8Sd
    gGw2i8OHD+uZ6M8RI8dis9ng9/uxfv163H///di0aRNaWlrAcZzWt3Kx4Pf7b7qbqLAYHR1F
    f38/+vv7cf78eUxMTECW5QoAGwC8Dv0KPASg5hCC4zhs2bIF27dvx5YtW7By5cpFbWBcWmSm
    9Dubmpo0NpSe+3A4jP7+fvT29qKvr48a9XcTxXNMx+1taNLxTpjj4OAgfvzjH7teeeWVBwRB
    0N3bWm4Q0vCtRZaDOSxeP+h5ydwyXIKaGvFTAN4Fkrq3q2NKWLqCbbecfCaTQTAYhNvt1qXt
    xOTk5Of10NPLyqkgxoPN4XBg48aNUBQFk5OTGB0dxdjYGCYnJxEOh7U0inQ6jdnZ2ZsKxJQS
    oLRgWHn/QdpagRo7Ho9HI9qbm5u1dk+NjY1wOp3YuHEj3nrrLTrPHgDHdZSD9YRwBMMw2LRp
    E2w2G6anp3Ht2jWMjY1hYmJCW6toNKpVHKXEJl2rctK4dK3oe5ZlNQOQ4zh4vV7U19ejpaUF
    fr8fjY2N8Pv9aG5uBsdx6Onp0Xr7EhnYordBWCSsE0ZGRmSe5y1Op1N3FpLneXloaIgahGkY
    p1fVvEyMLMtNiUSiGVBzMT0ej2HmdwsP4ULxPoBDoig+deLECfaxxx7DmjVrtCIeNDyNXiK0
    Gbjdbte8Nq2trTcZzoqiIB6P45133sFzzz2HwcFBAEjLsvx7QRBeB7BZFMW/PXz4MCOKIp58
    8kmsXbsWLpdLSwq+lVFMG43Pzc0hlUpp3nC32w2v1wu3242+vj7acHwcaoL+kjYZN2IFMwPP
    926QMBmPx4NHH30Uu3btwtq1a5e1byLLsvB6vfB6vfD7/Vi7di26urpw+PBhvP7668jlclao
    5eNroF9PwjVQc4Xg9XrR1dWFnp4etLW1aezlcoIai5QcunbtGs6dO0f/2QeV8T6po5JFw/YM
    zckYeY5DQ0NIJBLc5OTkPXpHPFAP4TLJwd8D+CeoLP9SIgk1tPo41DSNb2LpvdrXiBycWu5n
    ePLkSXzyyScaQbzc+0cURdD+fgZDM9T8cLhcLnz00Uc4d+4cLl68iPHxcS3NYTnWzGazgeM4
    eDwetLW1obu7G9FoFFarFZIkUTnIMQwT08lL2En0BthsNgSDQTzzzDM4f/48xsbGMDMzU9rH
    esn1hqqqKng8Hvj9fnR1dUEQBDgcDnovtZHnOqinQSgA+BOAQjAYtA4NDfGrV69269k3iGEY
    jI6OWkKhkANq4Q9Dl1Im7nS/zWarBVRGnoabGQG0qeZify3UHLutly5d+vLPf/5zBAIBvP/+
    +5ientZCEKgS2NTUpLEkPp9PY5o4jgMApNNpJJNJBINBnDhxQvseAHGoITkvE+bkpwBYnucf
    PnTokO3s2bPYsWMHtm7divr6ei20IZ/PI5VKIZlMIplMIhqNYnx8/AYWiFaXdDqd6OjowLp1
    63Dw4EFks1kRwBEAfUv5XIrFIoaHh8FxnKFDRilGR0chSZKeRqFC9oAUi8Wszz//PHp7e9HV
    1YUNGzagtbUVTU1N8Pv94DjuhkqYX7RAD/Wu0wqb+Xwe0WgU4XAYkUgEwWAQFy5cwPDwMCKR
    CD3zEiGwnHpdoVALVQwB6BgdHWV/8Ytf4MUXX0QgEEBnZycCgYDG8NbV1YHjOC2fkq5XaZW/
    +Z459U7QQddIFEXMzc0hk8lgamoKkUgEV69exeXLlxEMBhEMBhGLxWg+lwy10FSaKLd6VaST
    oFZXNDLSWOIqyAsBKWixkhIRehuE5ZFO1EO4BETCCwC2Adi8hD9pGsC/AHiF7IN9xAD9Gyxd
    31MRwGGorZ2WHbStgYkbDTBZljkADkmSkE6n8dvf/lY3/UEURcTjccTjcYyMjODkyZNa4THi
    NfMqiuJlGCa2gLDshWAVSG9ZQRBw8OBB3daqWCxq+mgoFMK7776rrRVBBVRi9Bi+oANsMQxC
    CcBHAJITExN13//+9y319fW6b/xsNotr164x5PI7BwN7CHO5HFKpVJWiKC4A+PTTT7F3715D
    5GJSpuu995akiN+nAP5dkqTOl19+uYZcEDyAGNRQGgs5jDUAbCzLMrREfVVVFaqrq7UqrLQA
    RTKZpMZrEWp4zL8R44wmlP4vgKcBTEiS9Pjw8LBnZGQEBw8eBMdxqK6uhtvthiAIWljf3Nxc
    aZiETIzZBFS23Qqg5ty5c95XX32VJRUXPyZCPreUzyWXy+FnP/uZYXJNb0dAC4KgZxuVjwHs
    BrBLUZSvJpPJVclksvr8+fPMkSNHtFYHtHIY9eLR0tI0j7C8xQTDMFp+ETX+BEHQwpqpwKOl
    tWOxGKamppDJZEoLtShQCyRdhurp+gMxdPQynE+Q//+vFUX5qiiKqyKRSH0kEmH7+vq06p+0
    RDkNq6YFoehaleaTzNerio58Po9kMqmt0dzcHLLZrNYGhOZmEMjEsL8KoB+qx6Mf+pYnL0Ct
    zFgAyTcxGAxPjMqyjHw+v8JiseheWluSJAwPD+P06dOapySfz2NsbEmiki8D2A+V3a8h5y9H
    5OAclTFQq47a8eflgQoALgD4V6ihotRDPATgn6G2mHgcakXtP+txkbnNlM1xBa57Oj9aajlo
    4s+D2+2GLMuXADwjCMLfy7L8FUmSaufJI9XzHlDsdnvS5XJ95HK5DiYSiXGn04lMJqPHdN4n
    svghACuMRryXzIcnds4AFtCgfrEsjk8BDEuS1DA4OOgkYXpGQRCLXOFqsUEqL1plWbYAamuE
    YDBoqDnOc1kshptHhpontV0UxQ0AXiOberLMIKwD0FQsFluKxWJroVBozGQynkgkwpF/d5XM
    ZxbAFQBHCVMyMg8ZEALwDDnsj0uS1JNMJmuTyaSdCGOJGJBponjOQPWYjJHXyRJBaAFQoyjK
    qkKhsANqzPt+InCX7B5gGEZWFEWJRCK4k8AwDGOxWGRq1C6zEBLJhXkBaqz9PQDWKIrSIwhC
    pyAI9alUqvrKlStO8lxv8BDSIimlsf/lBmF5XkXp30t+KyUVUmQvDZE5DRHlcNwAirtAzuJF
    AL+DWvSiA0CPoijdgiA0CoJQlclkqkKhkLZeAG7Ii7iVl7DcO0g9hGX7QSHzyJIxBTUa5QK5
    00PEaDVCXz2ZEA4JqCXdjYY0gA+NTIySInBuWZZ1N6gFQcBvfvMbHDhw4IY9u0RKqUxIje1Q
    KwUeJXtpqszYqoXajqIFauGpZmIk1kDNj3aTc6iQ8zJCjMDXiB5UTpiMAngWanuIx6Hmlflw
    3WNYJCQVlYNxcjddI69TRA5SYpQj98QOqJV/f7cMutedlTOh9yWQTsPv9+fq6ur+MD4+fspi
    saz3er1/WSgUNsbj8ap7Uf4AAASjSURBVFZFUdpyuVyFJEnLxtpaLBbJarXOyLI85na7Q4VC
    4SzHcYPNzc2Dmzdvnh4eHpbnawS/TPgYakGkTVCr7/ZADc1sIHqnngRfnMi/PwF4B8C7REfV
    3SCMEiX4AvTL4Zh3r5E5hY18SGdnZwEgVCwWz7As26IoiqwQzUgxCG1TqsxZLBaGYZirWBxG
    Pg41jNNOhIf4GRc/C5V9dBHhWEfGCqjVlSQirD4hyuJnKT8pAC8BeBNqvtQq8j0iEYIxqGE2
    U+SzAq73HZwPvcQA7YDqKVCW8Fm8y7LsbpZl2Tsth1BRFIZl2QzDMOGycIflRJ4oSyNQPWFO
    qMnjd5F95SNKfaMkSY2SJNWRPWctGWzJewu5hCVcD/eUSsYs2UeT5K6Mkn0fJUrZDJmTYsBH
    JhKhE8H1fCeOKKYryKjF9R6dDbIsVxOl3lG2Tuw8a0WHCNW7lsH1XmtRcg4TJcIvSc6iEdfq
    UzKMaBAGscAeVUt+KPN5WK3WDIA5lmWrGIbR9Rnncrmb2veU5hYyDMNYrVYJi9N0fIbIQdst
    SMxyvc1G7q0VZXKwgpyncaIofp6cTkMNI/0fqB7KW8nBafJZoeR+mw+niXEbIL9jKZ8hJW8V
    g94Hn6ebynrMOxwOIxwOKwDiP/jBD07u3Lmz98iRI85cLtc4PT29ZnBw0Gez2VorKipWFYvF
    hnQ6Xc3zvJtl2WqWZd0AWIZhWEVRrIqiWBRFYWRZZmRZpmSgQgfDMArDMJIsy2KxWJwTRTHl
    crlyHMdlWZadzGazQUEQRj0ez9V0Oj3c0NCQrKmpyVutVmVgYAAXLlwwwrOKA/gvAG9Bbe9A
    ydH15Mw0kL/Ttj5ucj6/aLs7pUT28oQUyhE9IkZkzCVcJ0XjRHYueC8tpjZJhb7RIEHfUKLP
    hc/nAwBrVVWVt7Gx0VUsFpV8Pq/kcjmlUCgoRnNTMwyDxsZGfuPGjaloNKocO3bMpN5M/H+D
    lZAUNmLY0D5c9pJX+t5KlCRhnlEgl3mBXPCi0e+jBayXjQxqJFtwvak2XTNadVAoIVkEohzJ
    JcahCAN7sz5DyXsCwJexAJZ2ieb1MYCDMHAOYSAQgN1ur00kEn8ly3ItwzC3w4fqZQgwDMOg
    ubn58pe+9KU3Jycnxddee828NZcR27Zt6z579ux2nufZO3D6DMuyyXXr1r02OTk5s2fPHuzd
    u1f3SR04cADbt2/H888/z4ZCIcfo6KhteHiYzefzVS0tLS0ul6uWYRinLMvOYrFoEwTBls/n
    rblczlooFOBwOBS32624XC7J5XLJTqezaLfbi7Is53ienwqHw6OSJGXuvffeYltbm9jS0lJ4
    9NFHi2fOnMHu3bvvtGdInRMOqG2afISY8ZFRXaYnlL6ncrBQpivQ11li6MVKXmeILpEnn1n8
    TWleKyZMmDBhwsSiGcZGVFClO9DANmHChIk7FTQixoIbSVI6FFwnQeWy99TzbcKECRMmTJgw
    YcKECRMmTJgwYcKECRMmTJgwYcKECRMmTJgwYcKECRMmTJgwYcKECRMmTJgwYcKECRMmTJgw
    YcKECRMmTJgwYcKECRMmTHwx/B9kAkyD9yO5qgAAAABJRU5ErkJggg==
  }
  set pieceImageData(Merida1,80) {
    iVBORw0KGgoAAAANSUhEUgAAA8AAAABPCAYAAAAk2KwLAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCwweGApBeQAAIABJREFUeNrsXXd4VNX2
    XfdOzSSZ9J5AEhIIoYQEQug9GCSAFEFAEGzoU0EU6YICz/LswM/yUOEpiiJIE5WiKAKhSwk1
    hAAhvUzJ9HZ+f5wz4ySkoZRJmPV98xGSe2fuPXPu2XvtvfY+gBtuuOGGG2644YYbbrjhhhtu
    uOGGG2644YYbbrjhhhtuuOGGG2644YYbbrjhhhtuuOGGG2644YYbbrjhhhtuuOGGG2644YYb
    brjhhhtuuOGGG2644YYbbrjhhhtuuOGGG2644YYbbrjhhhtuuOGGG7ceAvcQuOGGG264FHgA
    3gD8AcgA2ABY3MNyU+DYGEoBmNzD4YYbbrjhxk3YYBkAIQCreziaJ4TuIXANTJs2DSKRKFSj
    0QiTk5NLZsyYYSaEgOM49+C44ca9A18AwwHcByAGgAFANoDvAGS5iXCjEQ1gHnNkXgNw2T0k
    btSF1atXo7y8XNC9e3dSWFhoGzt2rHtQ3HDj3oMIQGcA/QG0Ag2engOwA0AOAOIeokYHEMJA
    k6wFrhpEcFV2xQEQs8nIsUloBs2ENEskJSXBx8fn4fz8/Cc9PDx+DwoK2h4dHX16zZo1WgCN
    IsLz589H165dufz8fM5isdh4nseMGTPcj+Lfe3i9QDNwRgAKRkSaFSZNmgRPT08+Pz+/tUKh
    aGGz1ft4cXK5vCI6Ovq0wWAwfvHFF1iwYAHOnj0Lb29vcV5eXgez2RzYgIHgeJ6/LBKJcqxW
    K/bt2wcAGDZsGAD4qlSqJJPJJK3nPTihUGgJDQ09v2HDhoLnnnsOK1ascOX5E8j+Xw6gqhHG
    MxTAfACTGBG2w8aM7+sA1qHhjKaIne8HQAegAoD+HnuGHwTwGfv5MRZAcMONWjF48GB4enr2
    LSwsbC+TyfbHxcWdf+qppwznzp3Dww8/7B6gOwsBqHojgNnfCmZ/3eTDjcZCCEDCfrbzh4Yg
    A/AIgBkA4vCXQtYI4DBoIHVXI8gcx2ywBDRgbcK9l0XuAOBdxuNeBHDUVSeJqy18sQB6A+gK
    IIRNplI2AfcCyG2Ok0mpVAKAV0lJSVeTydSzuLh40tWrV3/o2bPnFykpKSeGDx9u2rp1a53n
    z549Gx4eHsIPP/ywv1KpjAsODv7+zJkzJe518KYRDGAkgEEAItjidxbAtwAOohnJKTUaDYRC
    oaS0tPSpa9euTTWbzajDyeAAcOHh4YfCwsKmaLXaAgAQiURQKpUQiUT+V65cWazVavvX92xy
    HCcQCARvSSSSZWaz2cG2y8vLAaB9YWHhp1VVVSGoO9DFSaVSrVQqXUQI+fTpp592xWGVA8gE
    8ACAFmzsrgLYBGA7AHUd54kBTATwqFAo9IyPj0erVq1gNBpx9uxZvqCgoA2Auey9fqvH8Eaz
    9+nNCLgWwBkAa9kaar5HnmMvJwfGy72s/S1bLHZyIJu1A5ebmws/P7+2ubm5r/M8X3D9+vWd
    2dnZG1u3bn3U09NTt2LFCjz66KP1vkdERAT69u0bUFJS4q1Wq/OFQqE1KyvLPZNuDqEAxjD7
    G8bs7wUA3wDYx/7fLDB16lRIpVKutLQ0oqqqKtBms3H12F94eXmpo6KirhoMBsuqVauwbNky
    XLp0CV5eXsK8vLxoo9HozY4nddhfjuf5YpFIVGSxWPDzzz8DAJjawVOj0cSYTCZRPYEGTiAQ
    WIOCgq6vXbu2cubMmXjvvfdckdMkAxgKoA1bxy4B+AnAoXr8Nw7A/QAWAgiv8TcJs6dLQYPZ
    9ZE5HwADAPQDEAUa+D4J4AfcWxnkRADd2Ph3dBPghuHJnMan2AT2rPH3iWwifQzgewCa5jRb
    zGYzjEaj1WazWS0Wi6SysrKlWq1+WqVSDSwrK1sbHh7+7ZEjR3KzsrLI9OnTq507cuRIeHt7
    C3/44YdReXl5r+j1+pYGgyElPT196X//+99r69evh1vS1ShEAVgEYCwjMnb0BTAQNAK4rrmQ
    CLPZDJvNxmm1WrlCoZCbTPVze7lc7mexWISMKIPneRiNRlitVoFSqZRXVVXVSzQ4joNAIJB6
    eHjAaPzLj9Hr9QAgViqVviqVyru+95DJZCKDweAJAIS4nC3xBfASgMdZIMWOrswgtgWNiirr
    mHvjAXh2794db775Jtq1aweLxYIff/wR8+fPR35+fgJoZvMwaGa3JjqwOToQtPbVjl7MgC8B
    sAHNWEnD8zyYksHm5GzY3Etbo8CBBv26A+jEAigcc/pOgUrw85vjeBoMBuh0OpterxcaDIYE
    lUrVurS0dHhpael3Xbp0+So3N/fMuHHjLN98880NaqyJEyfihx9+QGJiYtTx48fnaLXajrGx
    se8sWLDgx/z8fPNjjz3mnlmNQwsAr7A1zqvG+tUPVAGzFs0kCK1QKCCXy6UFBQXPFhYWjrdY
    LPWqp0JDQw8FBgZO12g0xXZ7qlAoIBAIgi9duvR6VVVVWn3PJgtALxeJRO9YLBbHcQUFBQDQ
    sbS0dKVWq61PxcVJJBItx3GvEULWPvPMM642pGIADwF4AUA7J35jAzAKwHIAa+qwnSEAptRC
    fp2RwnjI2TreIxw02zmR2X/7QmFkhPw1AL/eIyRY5HT/Yle9SFchwB7MaZwLIFQsFiMkJMTW
    okULTiAQcFevXiVFRUUyk8nUHTRDHMSIsLa5zBaBQACxWKyzkys/Pz8YDAZeoVAkaLXahSqV
    asDUqVPf7dev3+6YmBhjXl4eCCGYMWMGpkyZwj377LODcnNzl1RUVLQhhODSpUuTdTqdKDMz
    c8G7775b4LatDULKFsCJEonEo2vXrkhMTIRWq0VWVhaXm5vbBsBs0FrCfc3K6+U4EELg5eWF
    CRMmIDQ0FPb6c5PJhI0bNyInJwcAiLPzV6NGnQBAfHw8hg0bBi8vL8ffr127hk2bNkGlUoHj
    OMer5ufbf+7Vqxf69etX7e+nTp2CXQHBcZwrGhABgNEAnhIIBP6JiYlISUkBABw7dgznzp0L
    slqtzwDIY0aY1GI8oyUSCYYMGYIuXbpAJBIBAIYOHYoNGzYgPz8fjET712KA/UClW/fLZDKu
    a9euiI+PR2VlJfbv388VFxe3Y+vrJQDHm+HzGwOgm81m8+Y4jojF4j5Go1HMxrkfM8IcG7fD
    AC66l7xqkAAYDBqATmNzjHN6thWgUfxPQLMpzUpSb7PZYLPZTIQQKwBYLBZepVJF6/X655VK
    5YDS0tL/xcTEfA2gYs+ePejfv79jDUxOTkbPnj3Dzp8//0phYeHDJpNJbDKZQpcsWeKZkZHx
    3bhx48y1EWc3bvABH2VBQGktgZk4Zn9zAfzeHG5Yr9dDKpXySqUyrLi4uEVDAWiJRFJssVjE
    9uAxx3HQ6/Uwm83isrKyyMrKyqiG7LxAIPCrGYBWqVQA4FlSUhKtVqv963sPmUymj4yMDAAA
    q9XlRCF9AbzM5oozeNBs8EIAJQA21nJuGGimsj7wjAQH1GJ/JWz+PgUqpa75twGMFBYAON+M
    Sa9QKBQSABKLxeLMM6XsOba6UgDLVQjwENDMSWhISIhl6tSpZMSIEYLQ0FBwHIfi4mJs3brV
    unr1aq6oqCgEwCwARQC+bi4zRywWQy6XqziOM/A875OZmYmWLVti3bp1yM3NlZaVlQ3Q6/Wx
    u3fvXtGrV6/VLVq0UGzcuBG7d+/GmTNn4nJzc+dVVFS08fT0REBAAK5fvy7Oz89/SCwWF2Rm
    Zi5r27atfvXq1W4zWzdaABjLcZxHZmYmli5ditjYWJjNZuzcuROzZs1CXl5eO9AGRYfRDDvL
    ent746mnnkJycjJsNhs4joNarcapU6ccBJjneUfk+KmnnsKuXbvA87zDErZp0wazZs1ykGie
    57Fv3z7s2bMHKpUKhBAbyzzfYJwBmsHr27cvXn31VQcp5nkeX375JbZv3343sr72jswi0Mxt
    XU2oQkCVA/4dO3bEBx98gJSUFHAch8OHD2PmzJk4ceKEP6i8bztoWUdNIykUCATw8vICz/N/
    WRWRCB4eHs6BmtrW7Y4AMnie58aMGYNXXnkFYWFh0Ol0+Prrr7F48WJUVlZ2AJV5/Ym6o9AS
    UPWDlpGcphCtFgGYBuAZAGJCCMxms5B9dwDwMHOs7Qb4c2ZDDO5lj5ofABOY8xhTm+/MCPFg
    FoAJAbC6OY2fSCSCTCZTcxxnhFP20WQyiSoqKjrrdLrWVVVVSRkZGe/t2LHjjH0duv/++5GU
    lOR14MCB5woLCycYDAYxAJSWlsYbjcYlBoOh8ujRoz936tTJPcsatr9jaiG/zmgDYBhoKVKz
    kULfREDXVs96TBr5WXUGoG/iOu60TeCd7r8u+IH2zoir55gwAJMB/FGL/fVE40pl/FiwpiYS
    2Boqq+fcNNDyujfruReOvQiaVqY4mT2bflarFVKpNMlqtUoJXSgfAE1c2gPQO0HLWe/6/bkC
    AQ5nzktEYGCgecmSJZaHH35YIpPJHB5gy5YtuQ4dOvCxsbG2hQsX2oqLi0PZOQcAXGkOiyAh
    BFKptIQQorXZbCCE4JlnnkF6ejpWrVqFH374AUqlMvrq1auLTCZTXIsWLf79wQcfFEyePFm0
    Zs2aKZWVlT0EAgEmTJiA4cOHY9myZTh48KCkoKBg8u+//5516NChHx566CF88803blNbtwGO
    9PLyQmZmJhISEsBxHCQSCQYMGIBOnTohLy/PboR9AJQ1x0GwEy/nf+3G0mazySorKxOtVqvP
    kCFDuClTpkAulxONRhPK87yX83s4G1lnYyuTyUJ69OjR3tnYCgQCa0lJSQwhRFiXkb5La2NX
    0KxuIiNZ+QB+BPAzaG2PMwIAxNgz2N26dXNkcHv37o2uXbvixIkTYAQjrBYDXAKgQq/X+xw9
    ehQKhQKBgbSH1qVLl3DxoiNheQ2AqpbrjQPg4+npiczMTMTEUB4jlUoxYsQIrF69GpWVlUI2
    f+W1vEcYu9feoAobFYAjoLXvl13cGBP2fSgYgSeEEGeHxoC/MpZmNK4h2b2EQaDZkZhGHBsF
    2l27HM2osRjP85DJZJWoI7Ot1+u9i4uLH7FYLK3T0tL+/dprr+0+ePCg9aeffkJycvKgoqKi
    Rw0Gg9TZnqvV6lYFBQUvZWZmXvz0008vv/baa5g/f757ttWOCPZqCAmgpSbNpr/JzQR1nW2i
    l5fXzRJoACC1Nbt0UXVCFGg5TxIjqJcYcTqOGxMQAaAlQI2ZP5G12F8lgEpGcOtDMWovv4xD
    /fJpgAYak9j8razxNxlozWxPAC1BG78dBZVMV7j4FBay4MNzAHhCCPR6Pe/09wGgXbXtwYxY
    9h1WuQkwrTdK5TjONmrUKOOECRM8nMmvk+PMjRs3jj9x4oT5ww8/FNtstmTmrDULAgwAWq02
    XyqVKg0GAy5dugSbzYY+ffqgffv2SE1Nxdtvv438/HyfgoKCxwkhHomJiQuPHTsWWlZW9qDZ
    bBZ2794d06dPd9QOPvvssygoKIi8fv36Cz179iw7f/584cCBA01isVgtk8n0ZrMZ9TXWusfg
    BUDAHKFqBkEoFEIisTcUhBx/dRe8p1BeXh539OjR/3IcZ65hwIVGozGkMe9htVrHnDx5sn/N
    31ssFg+dTie/WYfgNkEEWks0jxFG5/XofgCfAnibES4Hj2cvSCSSahlcVt7gvObWtu7mA/iN
    EBKzbds2TiQSoWvXrtDr9fjpp59w6tQpgDbQ+rUOAiwG7bLt/Fn0D2IxhEKh873V3P+9DYDF
    AEagegR7CKh8+BUWbHRVWAB8BCqN9AJAhELhELPZ/BRo1Pl/oLJdAprZPodmlEH6hwgCle3F
    3qRj+hRoTfD15jAIRqMRRqMxnz1bkbVOMotFWFpa2ttkMq3YuHHjvPHjx28ODAwM2LFjx5Na
    rfaG9c9ms6GioqJXbm7uE4sXL/73pEmTNPPmzXNLoeuxv404zhsuXFN4m4my2GAwhHh4eJjH
    jx/PZWVlITAwkBiNxlCO4xo1JhKJxPv+++8PEwqFNrud9fDwsJ46dcqfEMK70O32AN0RoS8j
    vxxo8DIPf9XyamvYP89GvK8nas/SFoIq+1rVc66ZrXm1EVJ5I/mUTy3+YwCAfwF4AjQQLQTN
    EKsAbAHwH2azXBU2UFXZcUbuCcdxPoQQ+5pYyQIMNtAA4xG4iHrobhNgAWixutzPz8+amZkJ
    Ly+vOhdBLy8vLjMzU7BhwwZSXFzsyaIp65uDM5OQkIDw8HBdZWVlsVKpRGlpKUpKShAeHg5/
    f39MmzYNoaGhWLZsGbKzs0VFRUUTJRKJwNfXV63RaFp6eXnh4YcfRtu2bQEA6enpGD16NFau
    XImioqLeSqXye5lMpiopKSnT6XTZHh4eF1u0aJGXmJj4Z3R0dOn27duNs2fPxltvvXWvGuBi
    ABqdTud98OBBDB48GL6+dCeanJwc5wxcEZpZA7ZGGF4AgE6nk+h0uhaNObau32u1Wl+tVutb
    1/k8z7sCAU4DrZdtGxwcjHbt2kEikeDSpUvIy8sLZrW8haD1kHZJtAJAESEk9ujRo8jNzUWr
    Vq0c84cRWLuhLa3lM5Xs/dopFIqun3/+Obdu3TpYrVZ7vZYBtAHb96hdPpUHQK/VamX79u1D
    v3794O3tDYvFgv3799vrh23sOOdO1L4AZgIYK5FIBAkJCYiMjER5eTnOnDkj0Wg06aCypecY
    SXdVVMKpNt9sNoey74YHbZ643c0x6pzr3f7GeSmgzYmahaQoLCwM3t7elYGBgYqCgoI61yCb
    zQaFQhEP4I2NGzdyPj4+HlVVVd3rOt5kMokLCwsfW7JkSeDcuXPP+vn5VfXo0eOaXq+/GBgY
    qOjRo0fVoUOHbPaOvPcwSkAzQt4NHFcAF8gc3Q0oFIrWhw4d+oTJ9Lm/TCuRGgyG+MbYcYFA
    MOrYsWOpNf9kMpl8DAaDt4vcajxo0DW9xu9FAFqDlmqoQEsg7bZQw+xqQ+NQVgeBrQTdNi8Z
    NEt8w/AB2APgqzr4RiGz0Q2R8Os15q8YwFTQElDn8edBs9GT2c+z4LqqQxvjYbvZd2QjhIwB
    8G/G8VaClswQ0BIkBVykkezdJsBi0G5pfFBQkCA+Pr7B62nVqpXA398fxcXFHDtX0hwIcGRk
    JObOnasdMmTIUZ7n71MqlYKcnBwkJyfbI3cYM2YM/P39sXDhQhw6dEh09erVCYWFhQaj0Shu
    3749hgwZ4sg8yWQyTJgwAVu3bkVeXp5Qq9WGa7XacNAarj56vZ5oNJpKjuPOl5SU7A0LC/ut
    ZcuWf27atKlCpVLZpkyZcq/Zl1wAB8xm8+i1a9fCarWiS5cu0Gq12LJlC06fPg226NaVgWuW
    EIlE6N+/P3x8fKplNesysCkpKZBKq5dxBQUFYeTIkSgrK2sw+8HzPO5yvZwEtAtp25CQECxd
    uhRDhgyBSCTCmTNnsHDhQmRlZfkwo7UNVJJsD6D8DCAlKyvLY/r06ejTpw84jsNvv/2GQ4cO
    2UnsThZEqQ3HQLtI/osQ0ken04UwQ3EZtHvzZ6DS09qQDeCQxWK5f82aNaisrERiYiLKy8ux
    detWFBcXgxHYX1G9jjkNwAihUCgYO3YsZs2ahbCwMKhUKqxatQrvv/8+TCZTP1CZ7Bo0Helw
    U62lupPgQYPIfn/jXDlzFr9HM+iHkJCQgKSkJENRUVEOx3G96gvCEUKgUCjiLl++/JaXl5fC
    rlypCyqVKkilUj0uEomgUqlMACo4jisoLS29fOrUqT0BAQHHu3fvfiEjI0OtVqvJO++8cy/O
    xVzQ2sxx9RyjZE626l4cIJVK5aVSqVL+4XtEqFSqCBe+TXszyd71HBMCqkD5DX8pUMrY/OnK
    SFhdRO0Q6g7k7gXNOs9ka5sHsx0qNu/eAd3KqDZcBHAaVDFVZwwDNEjrnEBpC9o8y7ueNXoo
    6P7DX7mwLTPW8GvKnWxvOej2jS6Hu02ACXPGiM1m44xGY4PaIJPJZO8+5zi3mSxuuP/++61y
    ufyop6dnlVKp9D169ChGjhzpqCXU6XSw2Wzw8vKydwAU6PV6T47j0K1bN4SHVy9BaNOmDVJT
    U5GXlwdvb2906NABWq0WeXl5UKvVnMlkCgDQk+O47mKxeKrNZjvw6quvbmnZsuUOACU1uvw2
    d5QBWAGgZVlZWeeVK1dyQqEQrKEOQLNgXwHYei851FKpFDNmzEBtdUN1EVj7fLUjPj4e//nP
    fxqd2XWS694N+INKgrkuXbrggQceQFBQEAAgICAAw4cPx/Hjx2E0GqNAayavORmALwB0MBqN
    I3bs2CHZvXs3AEe3TAPoXsBfou5GWlYA+5kxnQPavEkNuv3HxgYCfddBpWEtysvLE1evXs0L
    BAJ7d1u7QfqIGXlndAAQ4OvriwcffBAdO3Z0BC0eeughbNiwAZcvX/YBbbIlakJkh3P6102A
    6w72hKJx0tPaxjcUtGlRkyfAcrkcc+bM0SckJGSJxeKJ9mZW9ZHgsrKy6PLy8ujGrI1CoRBi
    sRh6vV5ss9nCAIRpNJouIpFohEajuebp6Xlkw4YNP7Rq1eq3S5cuFX3yySf3mhqrHDRTFA0g
    FdXLTsBIwxegQUf389x84QWqLpE2cFxr9rITYD1oqUtX0HpTrhaucRTAKtSt4DODSo7PgsqR
    H2Lv+yFotrm+uvM8Zl/jUHsJhQk0WFhTiZQM2n+mPgSw+1rfBO0vXPl5vdsE2B41sJWXl/Pn
    z59HUlJSvSdcuHCBlJeXcyyaU4Rm0okyJSUFZrMZIpHo5MWLFy/YbLa03377Dbm5uUhISMCV
    K1fw3nvvYcuWLbh+/ToIIRAKhbBYLBAKhQ6ZZk2jnpqaiu+++w5yuRwvvvgiEhIScOrUKZw8
    eRJHjx5FdnY2SkpKeKPRGFpeXj5Ko9H0r6ys/DUzM/PTjRs37l24cKFu2bJl98ri+4dEIpnP
    8/z/GQyGeKdtCcyM/L7OcVypC+4/e1tRk9De9ErIcTfUpboy5werEfL29q6WzRYKhfD19bVn
    woW4MWp7DbRu+ByA4VarNQRUEmUEbSb1HhonI1YyIh4EGoWWoXEql18ATAewkBDSme31qAbN
    rnwBYHMt7+MNQCAWi+HpWV295eHh4fw7z1qcCleGFn/tt3xPlSzcJOydZbl/cG6TR3FxMYYM
    GQKO404WFxdfNxgMddZESyQSxMXFISUlBdnZ2fjzzz8bfP/g4GBMmzYNQqEQR44cwdmzZ1Fc
    XIyqqiqJ2WyO1+v1cRqNJqOkpOTg2LFjvx4+fPiPhBBFaWkpQkJC7pW5uB+09OR9UGWCHRbQ
    8o830UybT7pRjQA3RpFiV49WoweggeNnQBVLPiyQYgHNDv8HwIlGrGk57JxARoCr0Limaz8w
    v+AFRs4FzHcsAw1g/xc3KrhC0Liadn80rQA03AS4cQNzAkC5UqkM/fHHHzFo0CAEBATU7hUq
    lWT79u1WhUIhBJUlHIeLaMn/Kfbt24dt27ahS5cuwWwRwIULF7B//34kJCTgt99+w+rVq6HR
    aBASEoIBAwZAIpFg/fr1EAqFiIqKqpV4REREQCaTQa/Xg+d5JCYmIjExEaNGjUJlZSWOHTuG
    9evXY+fOnSgpKYFOp/MrKCgYZTQau6rV6rVt27b9BMDVJUuWYNGiRc2e65lMphY8z8sJIZDJ
    ZDCbzfYtVVoC8CeEXG+uN08IcWR6G5vxvRPgOO5OXo/CbuxycnJw7do1tGvXzr7+4M8//wQL
    jGjrILOXAbwBKhfuwwhpKGh2t7iR1xAAmgnh2VqQyIxpQxsvWthnSBjxPcacyQvMCNd2/jUA
    epVK5Xn27Fn06dMHduXDxYsXUVRUZA8AXW/E57sC/EAbK3Zj3yUP2lBFzxzsCrhhh5HNdRtu
    zLg1xnaXoJk0E9u8eTOGDh0K0Kaa11FHU7Dw8HBMnjwZEydOhJ+fH2bPnt0oAiwSidC3b1/0
    7t0bCoUCBQUFOHnyJHbv3o2dO3eiuLiY0+l0/nq9/n6DwdBt69at912/fn35lClTjq9atcr2
    xBNP3AvzUQxaw1mT2AhAM2uhoLWWbjRfaFC9uWRdMNVCSu1Z3lmgvYW6g2ZxfUBLhI420oZ5
    gKrAJGxOtm+k/dWByrJHM/J8BTRx8idoUFxbW+yNraENZRkqmxDX4di9XmYcU83si83VLtQV
    ukAfBfAHIeTBzZs3o02bNpg5c6bzvpcAAIPBQP773/9aN2zYIGDO8CHQjmzNgnhwHIchQ4a0
    PXny5OsajSYRANRqNXbu3IlRo0ahc+fOGD16NPz8/JCZmYnOnTujuLgYNpsNUqnUIV2sCW9v
    b3h4eECj0dg3PKeWRixGaGgohg4dip49e+Lw4cNYt24ddu3ahcLCQq6kpCSqqqrqhWvXriWn
    paW9VVJSsjc1NdVy5MiR5rrwSgBMJIQsslqtITExMXj++edx5swZfPnll5xerx8EKkWdC1rr
    0eyg1WqxZs0a7Nq1y+UI8PHjx2G1Wv9xNroRqGJEacipU6c8FyxYgBEjRhCZTEaysrL4jRs3
    2iXNWfhL/lwTBmb8IhghiwCQAbr/bGPmTgRoN0gwwxENmpnWNuLcgaASMg/Q5k9nUX/W5DCA
    S1qtNmnFihVQqVRISEhAQUEB1q5di/LycjCifwB1S7ddxeh2BG3WNQw0Ym6X9saD7tH4A4AP
    2Li4ZZTUITnFHM6gv/GcnEAz2Q+9T58+CA0Nle7du3ey0Wis1ZgmJSVh8eLFyMjIgIeHByor
    K+0N6hqE1WqF2WwGz/MICAhAQEAAOnbsiOHDh+PQoUPOthdqtdo/Ozt7YllZWaecnJz/a9++
    /TcAVAaD4Yb+Cs0IUtCtVBY5rX3Oz3Y/0MDibDScxWuSPqBAIEBISIhLqqWqqqpQUXFHYoca
    0MTWMNQvg76IuutxK0BLfTQApoCWKg0FVUE1pptyqNMc5Jj99UDjlETdQTtXB4Bu27QN9Qe+
    TzD72raeYyqYnW4KBLgVgEzQGm67DX6I+TTb2PjDzuHhAAAgAElEQVS7jO11BQJcAVr70Uat
    Vnf4+OOPuXbt2mH48OHV6k9/+eUXw8qVK0UqlYpjk385mkE0cOrUqRg4cCAeeOCB0NOnTy8s
    KysbwHEc5+PjA5VKhaysLJw+fRp9+vTBypUrIRaLHSTAx8cHK1ascBDd2hASEgI/Pz8QQiCX
    196rw9fXF4MHD0b37t2xf/9+rFy5Ert374ZOp5Ncu3YtQ6VSxZlMpsXjxo3b0LdvX9Pbb7/d
    HI3veNDOg1ExMTFYsmQJxo0bh/LycggEAqxZs4bX6/X3sSjgPABnmovhtZPMqqoqrFy50vH7
    uy31dt4LuMb13E4prhW04VR3s9k8asuWLcIffviB4zgOVqsVbGP3E6D14g1FqgX4K7MWzZy4
    7EYYgAhUz4KEM6Oc28B5gaCdMz2c1veGxuosW39fvXjxYviiRYsgEolgsVjsRF/NHIf9Lj6V
    e4JuTdVFIpEIgoOD4ePjA5vNhrKyMmFFRUWgzWabzByNWXDqGH2P4zCAg8zhvBkcg2tvjdVo
    zJw5E7GxsaK1a9dOKi0tna3X62/oUi8UCjFmzBiMGDECFosFZ86cwY4dO+x7ezfs1Ws0OH78
    ONq2bYuQkBBHnwMfHx+H7T148CC+/vprbN++HWVlZXx+fn4HlUr15rVr1zoMGDDgLalUerWZ
    9uWQgvY7WIQ6tqBi69hA0CD0HBa4aVYICAjAW2+9hfj4eJcKQAPApk2b8Pbbb98Jn8AKKhce
    gBu7QNtRDODjRvj/jq0JGTHr20gCHFkjCBPO/p/TwHkyAIMZ+QVoVrcx9vcz0M7WPnWMx3YA
    O+DaQVsJ+76eZ+TXOYozFMB9AIYDeJfdj3sbJCccZQ7lq/n5+eFbt25Feno6ZDK6XZderyeb
    N29Gfn6+AFRD/39NwCFrFIxGIzp37ixav379w8XFxcPNZjOfkZGB/v374/3330dZWRny8vLQ
    p0+fG2r06iO+diQkJGD+/PkwGAzo2bNnvcd6e3sjIyMDbdq0wcqVK/HFF1+gvLwcSqUyLi8v
    7/VNmzZJnnvuuXWPPfaY4bPPPmtOxneCnfzGxcVhyZIlGDNmDEQiEcLCwrBw4ULwPI81a9YI
    tFrtUNDMyYKmToLZ/saE4zhHFsdueOVyOQICAu4aCbbLnktKSmAwGJx/b7Nf7210BK+D7otb
    AmCI1WoNYIbU3rNgBagCpSHwTgZQyozjOtTdydnu6EUDkPE8b/8+wphRbogAtwNtIFPb59fn
    cHwPKjEcabPZfI1Gowe713zQpl1fgkqIXRWRjNSmRUZGYtKkSRg0aBAiIiJgtVpx/vx5fPvt
    t/jxxx95jUaTxo69gmayh+0/RCmzp21A69Yag6ssaHKtqd/88OHD8d577yEtLa37pUuXXlCr
    1bVmwi0WCw4ePIht27bh5MmT+P7773Hu3Dk49YmoF0qlEq+99hp27NiB9PR0DBo0CImJiQ4f
    x9vbG+np6UhLS0OvXr3w/vvvIzs7G2q12kev1z9ptVr9H3300SUAzn/11VeYOHFicyO/i+sh
    v87r2WD8FYRuVkosjuMQFRWF1NRUl7oui8WCPXv21LRRtxM5zB/TgpYR+bLvXs9s8n9B+1nY
    GjFfuBrkdD2onLg+2JVbdtjtb0MEOJZd783YXzNogDkSwBjQILaUzfEyRhbfhWvXvnMAhrDr
    jPH390erVq0QEREBm82Gy5cvIzc3V6jX63uANvwSg/ZEuetRnrtNgANAa7X6AOgEtoeWXq+v
    FgGz2WycyWSSsoGWAhgBWpO5FzQK3SQbIyxduhQvv/wyOnTokKRQKKYYjUavNm3aYN68eUhO
    ToZAIMC1a9eQlpb2tz/D29sbU6ZMwc1EjmNiYrBo0SIkJCRg+fLlOHPmDFQqVYsLFy68unz5
    cuGgQYPWjhgxQr958+amHo2277M2E0BUmzZt8Morr2D06NHVpLbh4eGYP38+BAIBPv/8c16j
    0Qxlz86/QTMo1qZ48yEhIejQoYMxOzt7k7e3d0eVSpVmsVh4AIiIiMCzzz6LlJQUcBx3R4kw
    x3Ewm834+eefsWrVKgcBlkgkBl9f359EItHvHMdh1qxZt/MyzrMgx1rQ6LGYEczRoA02fkbd
    2xk5G0Dn2souoDLdX+s5RwYq2RVFR0ejrKwMVVVVdhl1Y5zD4Js0wHbi3B90j833GflVsjG4
    hDsjc+XYMyVgP5vROMk1D7ptVbqPjw9efPFFTJs2rVoJTWJiIrp164agoCCsWrUKJpMpnZ3z
    QSONsBg0mk/Y8eam+szXgf1CoXA5K/8IbuDYchYA+qU53LharcaAAQNCz549+5xCoWhT3zq3
    Y8cOHDp0CFVVVY2WPjtDpVLh119/xf79+/G///0PQ4YMwejRo5GcnOwgwnK5HI888gji4+Px
    zjvvYNeuXdDr9aKioqIx+/btk02ePPnlL7744nRQUBAGDx7c1IffH1Si+nwjyK/z834fWyeW
    gaoXmuSzKBAIIBKJCM/zNo7jUF5ejn//+9946aWX0Lt3b5eQQqtUKqxfvx5r1qyB1WoFx3GE
    4zir3U7fRmQBeBY0m2ivxw0BDfCGs/9rGjFXnC8yDbSed28954iYvXfuKOvbCPsLUIVX9N+w
    v/6MGOaA7qtuYfeWDdq8q9LFp3IC6PaNMcnJyXj++efRq1cvyOVyEEJQWlqK77//HqtWrUJ+
    fn4kaJOwc3CBUoa7RYB5NpGfZk5biN1RDAwMRJ8+farVukilUvTr14/bsWMHSkpKvEDlEf1Z
    5HA3qBziQFNbCHfu3IkxY8aIDx8+PFyn07UWiUSYNGkSunXrBrFYjGeffRY2mw0eHh7Q6XTI
    ycmBVCpFfHx8g3uy1kYqbgY+Pj6YOnUqWrVqhcWLF2P//v1QKpVR586de7m8vNwwcODArx94
    4IGm7ARGMcM7FYBfQkICli5dihEjRtRaZxoeHo65c+eC53l8+umnQo1GM4QtxG+Abo3U5LqR
    f/LJJ/jiiy9sv/76686+ffuWFRcXzy8uLh5qMpmkOTk5+PbbbxEbG4v09HQIBII7dl0ajQbr
    1q3Dd999h7IyGtuSyWTlERERa1q0aLH8l19+yb948SJat259uy+ligU4DrP/P8HmTTCohPaT
    BggUD4ATCoV2Uh/MnLff61mrPECzsYLOnTsjNzcXx48fl7DfiVB3HVALAH05jhOIxWK7g96Y
    RcIPwJOg2yycYtd29g591RI2nq2Z0+HHAgAiRsAL2esC+9eIG2VgUmYLZElJSXjggQdu6B9h
    f36feuopHD58GEeOHJExG/Jf3FhXzbHrimSGPYK9/JljYgCVvuey68pD0+0y7Qm6DUdfq9Wa
    ynGcpJHBgH6gzdn2gkqh1U3x5h9++GG8/vrr3KBBgwYqlcpBNpuNq8t2CgQCWCwWe038P4LR
    aMT58+eRk5OD7du3Y9iwYZg0aRI6dOgAnuchFArRp08fREVF4cMPP8Rnn30GhUIhysvLy9Tr
    9bYBAwYs2LNnz7nx48dj3bp1TdX+tgQNPD/CCMbN+q33gWbm3gTNBuqb2gCEhIQgNjbWlJOT
    86tMJuujVqtb7dixA/n5+Zg2bRqSkpLuWoLBHoT+6aef8L///Q9lZWUQCoVELpcflUgkhzmO
    w0svvXQ7L4GABmS/cfrdFFC1XiSzyWtRvyyYq0FAQ0Bluvvqsdv2BljiGmtePPvXVA+JHYjq
    dcuNsb8SxmOGM5syB8CdbrTDMZvLs2dLyGytBQ1vNSsADSanBQUF4aWXXsJDDz1Ubd4GBQWh
    VatWkMlkWLJkCdRqdQq752zUH+S2B8XFzC+w1riuJsk/hOzLPmofXD8/P5KamkqefPJJsm7d
    OlJZWUlqQqlUkvXr15Onn36apKWlkYCAAPsmy1ZQKcw4NK6duMsgKCgIqampsT4+PvsAkC5d
    upCzZ89Wu2+j0Uh+//13Mm3aNJKYmEjS09PJmTNnyJ2CzWYj+/fvJz169LCPN/H29j6fnJx8
    3759+7gpU6Y0tfnHgTYJ+o4ZTdK1a1eybds2YjabGxyPkpIS8vLLL5PAwED7eFwFbcwR0FQ9
    kfnz54MQguHDh7do1arVv728vMrs33V8fDz5+OOPiVqtviPzrbCwkMybN88xvhzHEX9//4tt
    27Z9bMqUKT4A8O67796toXoAVG1i7zbZroHjBwO4HhwcTJKTk4lQKCQsst2qgWjqnwKBgLz6
    6qtk0qRJ9nm2FrXXCNnxIIBKX19f0r17d8JxHAGwBzc2lKn5LDwM2oeBMEITdgfG0QNU9bOS
    OTLF7Fk04a/Mrwm0q2YRqNz8fwCmgUbwxTUc6WMAyGOPPUaqqqrqnFs6nY48/fTT9vE8zs51
    dkTas8/4H3NCSp2uy8quy8yMcCULGKwBbXDm3cTWwHag2f5cp8DCzbxMbO37hK2nfFNb91q1
    aoVu3br5BQUFreN5/oZ75HmexMXFkWeffZa8++67pFevXvbn6pa+RCIRSU1NJWvWrCEqlara
    nFUoFGTZsmXE19eXACACgcAcHBz8VZ8+fSKSk5PRBLfk40GVMBvt9vcfvuxbzwU2tYHYtGkT
    AGDWrFmS5OTkIaGhofuFQqEVAJHL5aRly5YkOjr6rr0iIyOJRCIhAIhEItFGR0d/3aNHj46v
    vPIK17NnT5w8efJOD1kmqASasEBtbAPH9wLtROw8X/bWWPdviJWyY2rOs3UN2N9+bC46n3MQ
    DSsb+rI12O5TtLqDNiCUff4M0DrkzQB+ArATwCbQ0piZjNhHo3pW3DmI+gMAMnDgQJKfn08I
    IaSoqIh8//335NtvvyW5ubmEEELy8/NJRkaGfWx+Q+3bXYnY9zMANKv8MWiCaQ9osvMnUBn7
    K4xHxuLv7WN/1zAErBOYSCQyDxs2TL9hwwbr5cuXiU6nIzabrV4yptPpyJUrV8jmzZvJuHHj
    iEwmsw/oZVB5YpPR5IaFhaFPnz73BQYGVvA8TxYsWEBMJlM14/fBBx+QuLg4YjfQMTEx5MCB
    A+ROY9euXaRt27YOxyAyMnLv0KFD20VFRTUlIyxk8+8AAKtIJCJDhw4lBw8eJFartdFjoVar
    yUcffURiYmLsc08Full6LJooLl++TEOsU6Z4duzYcaq/v/95juNsAEhgYCCZO3cuOX36NLl0
    6dJteeXm5pKsrCwyceJExzMtFArNoaGhu3v06DHwmWeeETZUw34H0ANUEkwYEXoLfzWcqg0Z
    AAoiIyPJ7NmzSUhICAHNHk5oiDT7+vqS9evXkyVLltjn2B+oW4YlA7AKgK1Xr17kX//6l91R
    38sMep0cgDkS9s/4Ho3bg/Ef8Q42bpftAVCO44iXlxcJCgoiERERJDIykgQFBRG5XE5EIpH9
    2myg2dqToNLlDOb0dgZwgeM4MmPGDGIwGOq1H/Pnz7ePzQV2bhCA+xkZP80+gwAgYrGY+Pj4
    kODgYBIZGUkiIyNJcHAw8fb2Jk6EyQZaS7ySZQ6aAgEZwOaG+RYQEAsLFmTCdXqKNAohISHo
    27dvN39//9ya9+Xl5UUeeeQRcuDAAaLVaonNZiMffPABkUqlt5wA21+BgYFk+vTp5OrVqzcE
    /xcsWOBYF8VisT4uLu7fkyZN8urdu3dTS35k4i/Z8q0aOzULxMQ1Rdv78MMP45133kF6enrH
    Fi1afCmVSrW1BWN4nidMgnxbXvb3d/5cjuOIXC4viY+PXzR27NgwAHjxxRfv1lB1Zes2AVXi
    LGog6dUHVKHjfE8VLFhcF1JBS39qzrH99ZBZEYAlLChYkwBH1fNZ/iywbd9PfTduvhv/31r6
    QJUXP7IAs6GO58rG/lbCfOaPWBIg1Il0tmTBZPLoo4+SqqoqYjabydy5c4mvr69jHa2oqCAW
    i4XMmjXLPsfOOiUQBOw9M0GbG+9jn2lwGpvaArAKllB4gRF07u8sSHcSbVm0LkEmkxkfffRR
    /dy5c70iIiIaFT3mOA4eHh5o2bIlWrZsibS0NLRq1Yp89NFHUCgUMaBb1FwG3XfL5WGxWKBW
    q2Vms9nTx8cH7du3d8hvNRoNli9fjnfffdexfVFMTAwWL16M5OTkO36t/fr1wwsvvIA5c+ag
    srISJSUlPc+cOfOvtLS0eUOHDlU3ge6UIvbwLgXQRiaTYfz48Zg3bx5atbq5oJu3tzceffRR
    hIWFYenSpTh+/LicEPIEc8gXgnYpb1KIjY21z0ntoEGDvjhy5MjlnJyc+ZWVlQPKy8uF77//
    Pr766ivHHrG3PBzJcTAYDCgtLYXVaoVMJtOEhIR806pVq7d27dp18ZtvvsH//d//3e1hKgWV
    5trXzrEsWrqrHrLBAXQLlaSkJOzcudOHkdytqF062wJAgL+/P2JjY6HT6SCVSmEwGIIYmS2o
    5ZzWALoJhUKuf//+EIlE9u+Ib2DtHw9aFwXmkBbg9u7r2gm02c1QACKZTIbExER0794dycnJ
    CA0Nhb+/P3ieR0VFBcrLy1FUVITTp0/j+PHjXH5+vkyj0XS02WztWRDhCDO+YkIICgsLodFo
    IJHUruTVaDQoLCy0j42U5/mxNputI3N6/Hme57y9vREdHY2UlBR06NABoaGhCAgIcHTSVyqV
    KCkpwYkTJ3DgwAFkZ2dzGo0mAjRzHMOcsmMu/Kj3Bu2WfauMiAA0o/cfRoZ/biprHs/zsFgs
    QTzPhzj/PjAwENOnT8fTTz+NwMBAEEJw4cIFZGVlwWy+fTuRlJeX45NPPoFOp8OSJUsQFkbF
    GD4+Pnj22Wdx7do1fPXVVzCZTNLS0tInzp07d3bLli3rVq9ebVu4cKGrD7cYwChGFOJv8Xt7
    A3jUyf6eQxPC2rVrkZSUhF27dp0aPXr0C9nZ2TlFRUX/UqvVjnmZnJyMjIwMeHp63vIO0fbd
    FgoLC7F582Zcv37d/nwQf3//7Ojo6Dc7duz4/YkTJ/QA8M4777iC/ZUAmAjai2B/Q/bXCX4A
    BnMct50QoqvD/tZGQoNYALq2xolhoBngmrVzDTnEQ1nwlWPjXSIUCk2EkNu1zthLT+eAZnXl
    ACCTyeDr6ws/Pz/4+vpCIBBAoVBAqVRyWq1WotFogk0mUzBov6axzObuYGMvZN8FhEIheJ5H
    VVUVjh8/DqWSflUnT55EcXEx/P39HbaZ53mZXC4PVyqVAtCeKkOYTfIHwAkEAnh6ekIikcDT
    0xO+vrRKQqVSQaPRoKqqSmQwGHzZNXVi5PkN0P4qLrldowS0aZBZKBTann76aWVFRYXhn2Ym
    lUqlbc6cOSaJRGJjTtxysGZargw7iYiIiBgnFotNwcHBZOvWrYQQQsxmM/n888/tGSPHq1On
    TiQvL4/cLSiVSvKvf/3LLuUkXl5elV27dp2q0+m4J554wtWN78NgUpOgoCCycOFCUlRUVOe9
    arVacurUKXL16tU6s8NWq5Xs27ePZGRk2MfEDCrtSkQTht3ADRgwID4uLu4TDw8PhXNEWCAQ
    3PJXjYhzXocOHeaOHj06cNCgQa40NEEsSuv8XH5bT9Q2E0BRREQE2bVrF3nzzTftGc1zjDTU
    RkqXArClpqaSwsJCsnv3bhIWFkZAJbdjavNfADwDQBsZGUn27NlDXn/9dfu1HUDdWePuoF3M
    iVNEfe5tDIp2YoECK8/zpH379uS9994jOTk59WZtCSGkqqqKnD17lqxbt448+eSTJC4uzjkz
    rLNH3hMTE8mJEyfqfJ/z58+TlJQU+3lme+RbJBKR1q1bk6effpps2LCBXLhwgWg0mnqvyWQy
    kcuXL5OVK1eSpKQke0bYxpyCZBd9tFsxgnq7sph70XBZgMvA19cXbdq0GSWXyx0ScJlMRhYs
    WFBNSq/X68n06dMddu92v2QyGZk5c+YNpWAnTpwgaWlpjuOCg4OPZmRkJAPAyy+/7MpDLQHt
    tZHX2DEIDw8nEyZMIOnp6cTDw+Nm1AhbASQ1Vdt733334fHHH/dITU2dGBAQcEwoFNoAkIiI
    CLJ06VJSUlJyW3y7M2fOkMmTJxNPT0+75NkYERGxrXfv3j2///57wbJly+7quAiFQohEokCe
    53c6fd9WUOluXdLkAbhRlkxAlT6d6jhnJv6qe3V+KTmOG1fHOQ+CNgesec5hRqjrWov3OB1r
    5nn+PzKZTFpXAPefDiEjmYeYnSJRUVFk3Lhx5KOPPiKHDh0iFy5cIFevXiX5+fnk4sWL5MSJ
    E2Tnzp3kjTfeICNHjiStW7d2zA8WKD8P4CvQ3SJIeno6KSgoIAaDgUyYMMExDhkZGaS0tJQo
    FArH72UymTosLOxX0EywEUxxFR4eTvr3709mzZpFvvnmG7Jv3z5y6tQph1IwOzub7N27l7z3
    3ntk1KhRJDIy0nm8sxlBd8lS2ES7w9WlSxdTdna25VY9vBcvXrT07NnTnsa/XIdz6VL48MMP
    8eeffwoTExMXA7CFhoaS7du3OxYjJ0fN8Wrfvr1DT3+zsFgsjapxbQgnTpwgXbp0cVxTVFTU
    oQkTJrTp06ePKw93Bph0NSgoiLz11ls31FrVxIYNG0hKSgoZP368o66hLpw9e5aMGDHCTuSs
    oG3tg9DEMXToUEyfPt0/JiZmqVAo1PM8T3r37k1efPHFW/qaMWMGad26NQFAfHx8Kjp16jTp
    nXfeEc+ePdvVhkQO1nhDIpHYSU8lgMdQe7Z1OIDi8PBwsmfPHnL48GHSsmVLAlr79nwt5/iB
    1hqR0aNHk6qqKnL+/HmSmJhod+5eqIOUbwJAhg8fTsrKyshrr71mfz6zULtsyxfApwAsEonE
    Pm81oB3RbwfCAXxtdyzS09PJvn37/tZ6pNVqyR9//EGef/555xIEB3FYsWJFneeuXr2ayOXy
    aufExsaSWbNmkf379xOdTve31tWDBw+SoUOH2qVdFnav4S42d0Wge03+nXrfmyEgDZUFuATy
    8/NBCEHr1q2fEIlEDin4Aw88QAoLC2/4nqdPn35HyK/9FRAQQD7//HNisViqSfi/+OILR/8T
    kUhkS0hI+HjWrFlemZmZLm1Kbob82ssZ7IGvjh073szYWUEbJ4U2Vbu7efNmKBQKvlevXt1b
    tGixTSwWG8DqgletWnXLya9OpyPTpk1zlHXIZLKK1q1brxg6dGiMc7Lmri5eIhHEYrGc5/k1
    qC6LLQbt/1NbtnWQnZzVeOkAPMPdKFmUgXa4r3VecRz3Ui0qRy9QaXBtUt0jqL3eWAKqVNA5
    HasH8BxuXwnnANCuy0QikZDBgweTLVu2NLq3i0KhIIcPHybLly8nAwcOvMGOAiAtWrQgv/32
    GyGEkE8//ZR4eXkRkUhEXnvtNUIIISdPniTt2rWr9pzbr6dTp05kwYIFZOfOneT69euNKklU
    KBRk8+bN5L777nPUqoNK5IfBBUthJwPQSKVS8vbbb5tvpuayMb2aPvroI71MJrOyqP5zrrzA
    EUIwatQoTJw40T8+Pv5nO7k9d+4cIYSQN954g4hEIiISiUh4eDiRy+UkICCAzJs3r8HMRF2N
    m9544w0ye/bsG5ps/R2H7+OPPyY+Pj72yWvq1KnTsg8++EAyc+ZMVxzuUDtBkMvlZOnSpQ0+
    9Farlbz44osEAAkNDSVHjhxpcFxOnTpFBg0aZH8IK0E7FjbpPaIIIQgMDERwcPB9EolEIRKJ
    yNKlS4leryd6vZ7odLp//NLr9aS8vJyMHDmSACBhYWGX+/fv36ZLF5eMYUlBZTbm1q1bO2ri
    QWtWEmo5/gEApeHh4eTXX38larWaPPjgg/ZzfqjFSWsJ4BDHcWTu3LnEZDKR8vJy0r9/f/s5
    K5mRdkZ/AFdEIhFZsWIFsVgszgS4riYcDwIoFgqFpG/fvsTf35+ANve6/zZlfxzGvkePHuTQ
    oUP/eMHXarXk559/JoMHDyZisdhhVEeOHElKS0tvOL6yspKMHz/ecZxYLCYZGRlkx44df4v4
    1sTx48dJv3797EZdB7qFlitFolvbHaDb/LoI182AOzB//ny8++67nikpKavtjn9wcDDZsmVL
    rY7WqFGj/vaYSCQS0r59e9K1a1dHM6vGvLp160ZOnjx5wzx+4oknHGTF19e3dODAgaMB4Oef
    XVJ9Hg5gy82Ml0AgIMuXL3eMfXp6+s2OuRK0sz3fVG3vm2++CUIIHnzwwVZRUVEf8zxvFIlE
    ZMmSJcRoNBKDwXBLXkajkVRUVJBhw4bZVX2qtm3bznzmmWf8AGDVqlWuNCwSUIVUzVrbHai+
    /ZAdg0Ely7XNkc0cxwXXILQhoE2W6ppXH+JGdWlH0GaItR1/tI7r6gnWB8mJDKoFAsE4ofC2
    CLASwJRrIpGITJ48mVy4cKFWpdXVq1fJuXPnyOnTp0lOTg4pLCysptCyWCzkypUr5LPPPiMD
    BgyoZnuFQiFZtGgRsdlsJDc3l6SlpZG2bduSP//8kxBCyMqVK50zyITnedKpUyfy2muvkZMn
    TxK9Xl9NdVNYWEhycnLI6dOnyblz58jVq1drbXKZk5NDnnzySWcSvB+NVIHcyRrgjgBkXl5e
    tpSUFP5mt/FpAFzHjh15Hx8fs06ns3fz5FB/++67hokTJ2Lz5s3o3Llzx+Li4vYA0LlzZ4SF
    hcFqtTrqILt164Y5c+ZAqVRCKpWib9++8PS8eXX3lStX8Omnn+LKlSsIDQ1FQkLC367XFQgE
    GDZsGH788Uds3boVRqNRdP369Qnbt2/fsXPnzj9efvllLF261FWGmgeVofbnOA5Dhw7Fk08+
    CW9v7waJn8VCywisVmujam46dOiA2bNn4+LFi7h27ZofgEmgDYbymqoR5jgOQUFBsNlsICwM
    LBKJqm1RditgtVqrbbNktVpxi9eHWwUjaJ2sWSgUCseOHYsPPvgACoUiFbSe9nVU3w7LsQ0D
    IQTe3t5IT0/Htm3bYDAYOrFFuriGAQ4SCASIi4uDSCSCh4cHYmNjsWfPHoDWzskZwbJn9foC
    iIqNjUWPHj1qblfF1RKEaQGasQ5p27YtBv9B0+IAACAASURBVA4ciDNnzgC0oUTFbRizZBb8
    9AgLC8OLL76I1NTUf/ymMpkMgwcPRkxMDF5//XV8/fXXMJlMOHz4ME6cOIH09PRqxx8/fhz7
    99NSMbFYjEmTJmHOnDmIi4u7Jb0LkpOTMXv2bFy+fBnXrl3zYM//Ttz5LS3qQhfcmQ6jUaDy
    epftw7F06VJ89dVXiIiICC0oKEiyr+9t27ZF586dQQjBr7/+iuLiYoSGhuL333/H3r17/773
    mZCA5cuXo2XLlvjPf/6Djz/+uFE25ejRo/jyyy/x6quvOvYJ9vPzw2OPPYb9+/fj7NmzUKvV
    QZcvX35ywoQJBz/88MOCZcuWwYXqge32t9/fsT21/dxI+ICWPP0KqvxqcpgzZw5mz56N7777
    LjcoKGgrz/PjrFar+KeffkJpaekt/SyLxWK3AfD29q4MDw//6cCBAwoAcLHSNrv9taB6vW1v
    FtR9H9W3CaxvH94UQkh7Nkfs8EPdkmVwHNea4zg5AK3NZgPHcSCE9ETdzddqs7++oMmRNjX8
    ar2np6cSgKPnzy2CF+h2s/14nkdGRgYWLFjg2ELSbDbjwoUL2LNnD/7880/k5uaioqICJpMJ
    crkcwcHBSEhIQN++fZGWloaQkBC0bNkSU6dORa9evfB///d/+PLLL6FQKGCxWPD777/j2rVr
    iI6OxtChQ6FSqZCYmIiysjLs2rULWi3ddVAqlSIzMxMvvPACUlNTHb1lioqKcPDgQezduxfn
    z59HWVkZ1Go1xGIxAgICEB8fj5SUFPTv3x/x8fEQCoWIi4vDvHnzoFarsWHDBlgslu6ge0jP
    Am1Qe9cJsABUqscFBgaag4KCuFsdnQsJCUF4eDiKiorsE9nDyVF0KeTn5+ORRx6R/fLLL2O0
    Wm1EYGAghg0bBh8fWsowZcoU+Pj4IC0tDenp6f+YDBiNRhgMBlitVhgM/3y72vDwcIwbNw77
    9+9HRUUFlEpldG5u7vhFixYdLisrM7rQUPuDZrV8oqKi8PjjjyM4OLheIkYIuYH02smwxWIB
    z/N1fh+9evXCmDFjsHz5clgslhTQroVNlgDXFRy4h0FAuyaaVSqVR2pqKtLT07F+/XoxI8C7
    Qbs112mAu3Xrhri4OGRnZ4eDZm/34K+9BWMB+Hp7eyMigpbuSiQSxMXFged52Gy2KLa22Ulz
    GKi0iU9LS0NcXFzN76emARYyR6G3VCrFpEmTIJVKodPpAKpauNUEWATgITvx6tevH/r163fL
    muVxHIfWrVtj8eLF0Gq12LBhA4qKirB161b06NHDESzUaDTYtGkTrl+/Do7jMGLECCxcuBDR
    0dG39GZ79uyJ9PR0fP755yCExINK806g7r2b7ySScGd6Y0gBdIALB6CzsrJw7tw5tG3bdpBS
    qXQoN4qLi3Hs2DHk5ubi5ZdfxpkzZ+Dp6QmFQvGP7KaPjw9iY2MRGRmJ9u3bQywWN+r9LBYL
    Nm3ahJEjR6JHjx5/fZFJSRg5ciQuXboEk8mEoqKiHseOHRt6/vz5/w4YMMCVhtoftO5Q3iBT
    ZvsfMzJQLZAnEokcv7MHpxthh5JAG+Q0SQJsX9+8vb1hMBisAIjNZkNWVhaysrJuq40zm81E
    LHa9Mko2P0qtVquZEOJcZuHBAh57QLOujSHAESww84fT+hyFevakJoTY94MvYv8PYPbXo54A
    UM3PTwctjar2e5vNVm42m8tuQyPZbqDN5wQtWrTAjBkzHOS3vLwcn3/+Ob766itcvHixzjVp
    x44d+PLLL5GcnIzHHnsM999/P7y9vdG6dWssWbIE4eHhePPNN6FQKHDq1ClkZWWhZcuWGDVq
    lIO8ZmVl4eDBgw7y+/jjj2Pu3LkOP0etVmP79u347LPPcPLkSVRWVtYaJNy7dy++/vprJCQk
    YPLkyXjkkUfg5+eH6OhozJw5E9nZ2cjOzuZAZdBbQJV2d50A83bCK5FIjB4eHjxu7Jj2jyCT
    yXhvb29ST+TFJdC7d2/88ccfyM/P71teXj6SEIL+/fvDeTuDDh06oF27do7ufP8UBoMBer0e
    UqkUfn5+t+Q9Bw0ahJ49e2Lr1q2wWCxcWVnZgF27diWpVKrDZWVlCApyiRLYFqCdx9G7d+86
    M08WiwW//PIL9uzZ41gE9u3b53CeV6xYgQ0bNsBmsyE4OBijR49GmzY37nji4eGBUaNG4bvv
    vkN+fr6cEeCtoPUdbjQPVAKo0mg0cp1Oh0cffRQHDhzA9evX40Azq2fYMbUawJiYGPTo0cO+
    SPdjRjeXHRsDQB4WFoaQkBCHMxgdHQ2RSASj0ejL5rS9y2kXAO28vLwwaNAgyOXymkaj5jrY
    EXT7A1mPHj0wevRofPbZZ9Dr9QDNAJff4rGKAo3O8/7+/hgxYgT8/f1v+RcSHR2NefPm4cqV
    Kzhy5Ah27tyJiRMnolu3bgCAU6dOYceOHbDZbOjSpQvmzZt3y8kvAMjlcgwfPhzbtm1DaWkp
    D5qdDwfdL/duggcLQN+hzwuECweg8/Ly0K1bt9jy8vIJRqPR4bzm5OTgueeeg0QiweXLl2G1
    WqFWq//x5wkEAge50+v1joBqY3DlyhV8++23SEpKcgR0pFIpRo0aha1bt+L06dMwGo1epaWl
    o/r27buV47hiFxrqaLv9rQ++vr4YP348unbtCoFAAI7jHLZaJpNh5syZGDt2LDiOg06nw7Zt
    2/Dzzz/DarXW97ZezP5uAt3arMmSYEb276g/64qBbhYUqbRarapagirtQDOrF/6fvesOb6re
    3+/JTpOuNOleaSkbyh5lU0YBvaAMEUQBRVEZCuJFvcgPRVAR9YpXvS5AQVS4TJFdASkFgTJt
    Kd1NV5rVJE2aec7vj5NzTBmlQEfk4fM852kpbU7ON+d7Puv9vC8AcyMSYA7oAvS3AIp5PB5c
    LldiQwkwgCCKomI9Ph6ee7tvQx8f6jf6GPTVDV0YkiR1Vqu1uomXTASaFCqaIAiMHDkSffv2
    ZRPOtWvX4tNPP0VtbW2DL0KSJLRaLQ4dOoSLFy/i6NGjePbZZ9G1a1cEBgZi7ty5MBqN+Pjj
    j2EwGLB9+3aMGDECHTt2hNvthsFgwLZt26BWq8HlcjFlyhQ2+XW73bh48SL++9//YseOHdBo
    NLe9qLq6unrd6ldffRX+/v7o1q0bHnroIVy9ehUulysMdPH9N1/Z/x8CoOLj440XLlwwN/Ug
    f15enr179+4MEdYXvpgAP/HEEwgPD0ePHj3ig4KCDsKj63vw4MFmZW/+6quvKC6XS0VGRjbp
    uTZs2EB5ig4Uj8ezR0dHvzVjxgyBD1WhZwAw8vl86osvvrjldVRVVVGjR49u1HwRn8+n3nnn
    nVsO6et0Om+x73T4HhnOHZlCoYBcLh8tEAj0fD6fWr16dZPfn7W1tdSkSZPYGeDBgwe369On
    j68uSVcAl/h8PrVmzRqqtraWWrhwIcMQq/U8dJlnz2MA9JGRkdSRI0fY6/3pp5+YGfoa/KUJ
    LAatZUmNGDGiHhHP0aNHqcjISGa29Hmm5ud5zlE9e/ZkZ/vdbrf3DPBZ/EXCIQVN8OGWy+XU
    5s2bKbPZTM2YMYP53e+buijpqXRrAFDt27e/Z/6BBkkgSJJav349FRwcTAGgli5dStntdsrh
    cFBvvvkmMy9Jff3111QT809cT8joTdpTDRoC2trGA03M11IkTttBS9P4nPXr1w+jRo0ShYeH
    v8Xlcu0tsR6TJk2iLBYL5XQ6vXUwG30olUoqMzOz3n3mdDqp1157jWWmFgqFOqVSORUApkyZ
    4ivL/SRojd4Gr69Pnz5UcXFxo/fYrl272H2O27OSR/2d/W9AQAD8/f1H83g8Q0vcq38D/9sJ
    9HjFzd5/xXXP23+ARkvd6nr18GgCEwTBA/DebdanjiCIeZ4ROh5oSb+GtNQvguZeYH5/sacZ
    cjPit50EQfg18Voled4D5a0yw+whhUJxV/cIQRBUv379qGPHjrGvV1ZWRk2YMIG62bkOHz5M
    RUVFUQCo/v37U3/++Sf7f+np6VSfPn3u+JmIv5jwqT179rCvd+LECSo+Pp75/6ueeK3BynBL
    WSkAe01Njai0tLSpAy2oVCqioqKCA3o+gGF+863WkV6PoUOHiiorK2ebzebBQqEQ06dPR3My
    KOv1ehw+fBhutxvR0dF3rHnbkA0cOBAdOtAFXpfLJTAajeOysrLanD171leWPAJAQEhISIPX
    LZFI0K9fP3Tp0gVt27ZFYmIiC0dnOnAdO3ZEhw4d0K9fP3Tu3PmWXXSpVIpOnVglEAVoyOoD
    u39MC0DvdDpRXFwMsViMJ598kvnMQwDMwV/EFzetQPfs2ZP5/UAAoz3JQhBoCDQiIiLqzakz
    4x2eJFkJeqQkAcAggiCQkpJyq46mdwd4GICJBEFw0tLSkJaWBrPZjPLycoBmTq3wfG1KC2cq
    9QqF4q74C+6kUzJ27FgMGzYMAHDo0CGUl5dDrVbj8OHDAIAhQ4Zg3LhxzTpfLpFI2O695/P1
    hQKYC03f3W/IdPDB7u+AAQNw6tQplJSUjDSbzbPcbnez4zyFQiF69eoFPz8/aDQaXL58+Y67
    ayUlJfj111/rdTx5PB7Gjh3LPBfgcDhkJpNpYv/+/eVZWVm+5H9vWwjRaDTIyMhAYWEhiouL
    UVJSArOZbuKRJImqqioUFhaiqKgIV69exZkzZxjUyu0sBDRk9YHdX/5X18D99qzXM/d2SNBg
    0ERZEoqipLg5YZW3iSiKUtbW1vLw1/hRQyha7w5wMmj0lehmvksoFBpCQ0MdUqm0qYsF0QAg
    k8nQpg09qmy323HgwIFGdVtvZhRF4dSpU3j77bfZ2fGoqCi88MILUCqVqK6uxrZt22A2m2G3
    27Fv3z6Ul5dDoVDgxRdfZHOGS5cu4e2338Yff/xx14iD6upqHDhwAA4HPUUWExODmJgY5r8j
    QfNB+UQCfBmAwWQy8X/99VdXXV1dk6l5OxwO8uDBg6ROp+OD7qpc8LVdO2PGDOzduxeXLl0a
    aTKZZrrdbmH//v3x1FNPoZl0vwAAv//+O9LT6Tn/wYMHsw6zKSwuLg6pqansvI7NZutoMBiG
    GY1G/Oc//2ntJecxyWdAQAArpH2rpPWll17Czz//jK1bt+Lnn3/GuHHj2ArsqlWr8L///Q8/
    //wzNmzYgNGjR98yAebz+d4BsD9oOOADu3+MJYsqLy9HTU0NkpOTMW3aNIaoJgV0F5h7qwQ4
    JiYGAwcOZPbNANCV2mAACQKBAImJifCewVIoFAgPZwmjlZ5keQCARLlcjuHDh0MsFt/KAQM0
    udazACKUSiVmzpwJmUwGg8GAiooKgK5il4GWcmhKEzA+JiwsrFkTYAAIDQ3FQw89BKlUivz8
    fGRlZeHcuXPIycmBRCLBww8/7L2OzWJ+fn7ePAMc3Mja3VpWBppEprnN7TmX25c2rdPpxMmT
    JzFw4EClRqOZb7Vao1vivO3atcOYMWMAADk5OcjOzr7j1yBJEocPH4ZKpar38y5duqB///5s
    UGq1WgdqNJq++fn5vgBh5aFhOGm9BH/JkiWYOHEiJk6ciClTpmDfvn0AAIvFghUrVuCRRx7B
    xIkTMXnyZHz66aeNncmWeJLgB3YfmGdOvIbD4Rga+LXhoGdemeTzdjnOINAcFUFoHEmgkqKo
    AAA9cHumYSYBFwOYjVtopFMU5SJJUiWRSMgm3rcyzx6AXC5n/a/JZEJxcfE9v/hvv/2GVatW
    MbxLSElJwdixtJDEyZMnUVBQgIqKChw9ehQAMHLkSKSlpYEgCJSVlWHlypX3RC7o/fxgRlWk
    Uqm3jxfjNnKkLZkAnwdwnCRJYu/evcL09HSqiT5s8vjx4+Yff/yR9MzW/A5a7Nmn7NChQxg8
    eHBidXX1fIvFEhMaGooXXniBrco0h5WUlOCrr76CRqNBbGwsRo8e3aQMvjweD2lpaWzC53K5
    RARBjHr00UcVPiLJQHjeF5zOhnlogoKC0L59e3Tt2hXJyclsEMvj8ZCYmIj27dujc+fOSEhI
    aLBgQZIk7HY7Hth9aw4Aau8EmMvlYvLkyczMqQg09L7HdUnoX1mhQIDhw4cz91gc6O5sHIAQ
    kUiENm3a1EuA/fz8EB3NxusJoCvVowEIO3bsiO7duzd0/wsATAQwhM/nY/LkySyhjlarhVar
    Za6pvBnWyuJJrkFRVIsE5YMHD0bXrl1hNBpx8OBB7N+/HwaDAV27dsWQIUOa/fwMiZ5XMugr
    ndA/QcNRm9uMoGF3PmVDhw7Fww8/LFapVHPMZvPglrgXJRIJZsyYgQ4dOsBut2Pv3r1MwenO
    P7w//8TJkyfr/SwwMBBjxowB0zVyOBzhLpdr9Lvvvuv3xhtv/G0eqCRJoqKiAhcuXGCLVmq1
    mt5Abjfy8/Nx6dIlnD9/HleuXEFNTc0dbckHLuv+MKFQiKCgIBePx1M38LlKAMwC0PFW/vc6
    SwAwGDRaqTFFsQSCICI9/jewEfEnB0AqgEcayLfsDoejpLCwkGRYkpvIuMz1y+VytkheV1fX
    JEzTbrcbO3bswA8//AC32w2JRIIxY8YgPDwcpaWlOH36NE6dOoVr164hNDQUkydPRkhICFwu
    F77//nvs2bPndnP8jXM4RiOLCBGLxd78QwRu0nFvrQRYD+BnANUqlYr33nvvcU6dOnXPQVFW
    VpZrxYoVRHFxsRj0zNUPaFm4121t3rx5mD17tlitVs82mUxDeDweHnvsMaSlpTXfYuv1+Oij
    j3D48GHweDxMmDCBHYBvSuvYsSOSk5PZ4M9sNnczGAwd9uzZg08++aQ1l90FoBYAampqoNPd
    HcHtnRKGORwO7yCnDrehYX9gfzsjQY9YOCorK6HX03xXSqUSs2fPRkhICEDP/cxGA/C7Ll26
    MPuGByANdEdXKJFIWAiP0WiE1WplizCe4lUkaNhWTy6Xi0GDBnknxzcL/rp6AgL/5ORkzJgx
    g3WEpaWlTCfFBg+zZRObCnTHHGq1mmGbblaLiYnBmDFjIBQKsXfvXuzduxdCoRBjxoxBbGxs
    s5/fYrGwwbvn2kt85L7NAq2P2NyWCVp/2mfsxx9/xIkTJ1BZWTlar9c/6XQ6hS1x3rS0NMyY
    MQN8Ph+ZmZn3FPCZTCYcOHDghj00cOBAtojudrthsVgG/vLLL1E//PADtmzZ0tr+13y3f+zt
    d+9hZOGB/72PzOFwwOFwkE6nswR/KSfc1L2ChhsHNOJl+R7/2xuNY8mP8EgfDWisSwI9FhXR
    wO84AZQ1AwN0jWcPwGazseR7PB6vyVCndXV12LhxIy5epGuevXv3Rt++feFwOLBv3z7s3bsX
    ZrMZvXr1YtEq58+fx6ZNm5pEkYYpjPD59FSty+Xyfl0nE3/4QgIM0LqIGwDUnThxgli0aBH2
    7dt3VwvhcDiQnp6ORYsW8TMyMgI8H/QG0KLYPmNvvvkm1q1bh6ysrFSdTvekw+EQ9O7dG889
    99xt9Wjv1oqKirBy5UqsX78edrsdEokEQ4cObZbzBQcHIyUlhXVSDocjqqKiIuX999/nbN26
    tbWX3wzAbTQa76jyzuVykZCQgKCgIMTFxTUIn77ZA8ELXmJ94IDvO6NAQzwdXjO0AICxY8di
    1KhRIAiCC2A8aMkDggnoXC4XNBoNNBoNQkJCMGTIEKbT2wN0l1YQEREBLpeLL774As899xwW
    LFiAgwcPIj4+nklcg0B3mKPCw8PZmdfa2tqboRwCPIFAV6lUipkzZ7LzNxRFoaioiEEr6Jrp
    Pi32vDY0Gg2auLp9U2O669HR0SgvL4dKpUJUVBRSU1PREtIe1yXAOs8a+IJpAGy9XUBwj2YA
    8CPqa1u3qjmdTsycORPPPfdcolqtnldbW9tipEjR0dHw8/NDSUkJPvvsM+Tn35siz7lz526A
    QUdERNQrbNfW1iqtVmuvoqIi7N27t7WX34S7kAAjSRIajQZOpxNms/leulWWB/73/jEPKztF
    UZTqNgkwHzT7cWMhP709v98YByH1+N+2jUyuJ+M2Otg8Hq/Oz89PJxaLm5qfQgWPIoVarWb9
    r0wma1IVhOzsbHz33Xew2+3sGJK/vz/S09Nx4MABiEQiDB8+HHK5HDabDRs3bsTVq1eb7PxK
    pZJVl7BYLCwk2+OPGvS/vBa+h82gxaplFEU9cfr0adELL7yAWbNmYc6cOY2eT1Wr1Vi/fj2+
    +eYbFBQUEBRF2QBs9ry22Vc2LEVRGDRoEOx2e3xpaekCg8EQLZfL8eKLL6Jjx47NEnwdO3YM
    H3/8MY4fP85CcZ1OJwN1bHLjcrno3bs3FAoF1Go1bDYbV6fT9Tly5IjMaDS2dic+D7RuXER6
    ejoee+wxltzqdjZp0iQEBwcjMjLyjh4Wly9fxpUrV9haBFoGdvjAWtYqADjsdjuKiorqFYOe
    eeYZZGZmori4ONyTAPM5HA50Oh0+/vhj7Ny5EyRJ4tFHH0Xbtm0RGRmJ4uJiGegZYEIikWDL
    li3YvHkzC/U7d+4cpk6dymiSigC0B0B07doVIpEI77//PrKzs9GmTRtMmjTJ+33GgmZBFQwd
    OhSPPvoo62DdbjdKSkqYpLm6me5TNejOche9Xo9r166xCXhzWmJiIuLi4lBQUMA6yOYcNfG2
    kpISFhXgufZKH7pv9wPYAlqKo6m7oHbPa//qSxt16dKl2Lx5s9+6devm6HS6QS05G7t161ZY
    LBbo9Xrs27fvntFu5eXlOHfuXD0JPj8/P/Tr1w/ff/89rFYrHA5HUEVFRf9JkybtyM/Pt7Xy
    8ud5Ci93RDxCURR27NgBqVQKlUqF3Nzcuz3/A/97n5mnS1pJUZTjNr8aA2ACGtfVZchKG0PO
    6wda+qgxuVMM6HnkgNtck04sFteSJAmpVIrq6iZTQyoFjYZVGgwGdk5WJBJh2LBh2Llzp7ev
    uqfCxNGjR1FRUQGlUom+ffsiISEBFy9ehNlsRnx8PCtxVlBQgGPHjt1U4/duLCQkBEOHDmWL
    27W1taiqYuuvty1A81rhHq4EsAKAmqKoaSUlJfHvvfceAQCvv/4628puaLG///57rFy5EhaL
    BQRBlHqS3898Kdg4cuQIli5dihdeeIH/3//+d5pKpRrC4XAwdepUjBs3Dk0JdzAYDMjKysLu
    3buxa9culJSUgM/nQyaTkUajkeNwOJCfnw+bzdakM8CMderUCYmJiWzngyTJHpWVlREWi6W1
    E+AroKnQIzIyMnD+/HkMHTq0UX8YERGBJ5544s4iQLsde/bsYbqCdtBwwBo8sPvNtABMdrtd
    xuiFMkRwKSkpmDJlCtauXUu43W4Rc19s2LABJ0+ehMFAN+CuXLmC/v37ewfFBAB21q22thYR
    ERGora3FpUuXoNfrWWZUAASHw4HD4cDy5ctx/PhxOBwOcLlcnD59moFhw+PQ+WFhYZgzZw4i
    Iv5CYVmtVm9HUd1MhUMrgDMARmi1Ws6OHTswZMiQO0JU3I0xhSu2ChAbi+Dg5idjt9vtOHTo
    EMOuSYLmovAlDVI9gPdBQ/MnNaH/d4HWW30ff2lgt7qdP38e3bt3h8FgGFZQUDDDYrEIWvL8
    lZWV+Prrr5vs9YxGI86cOYOJEyfWgzB2794dMTExyM3NhdvtBkVRA/R6fRRBEAWt/BFke/zv
    HTNvXr58Ga+++uq9FmQe+N/7yPz9/ZmYX0+SZA1FUQ0RnHHQeAUOAncmAdjY56bYczT88HS5
    Ksxmcw1FUWhiGLTBkwT31ul0OHPmDHr0oKlJHnroIWRkZODrr7++I13yW5larUZubi6USiUi
    IyO9iWDrFaCvXLni3aG9J+NyuZg4cSJLMgjQRINeqLzy2+WEnFa6l8tAa279G4C2rq4Ox48f
    bxREzmq1IiMjAxaLBWKx2BIeHr6eIIh3Pa/pM/b1119j165d+PTTTztkZ2dPN5vNguTkZMyZ
    M6fJAkCtVosdO3ZgwYIFmDlzJj799FOUlJTA39+/rmfPnqcGDBhwiM/n20iSRG1tbZMMnN+q
    CsNsLACwWCwyPp+fVFBQgPXr17fmx1ABYDcAa0VFBb777rsmqXjdyk6ePIldu3Yx1a1cAEfg
    Y2yoD6xJzASgmqIonDlzBqdOnWLHOEQiEZ588sl6xFQ6nY4lY4qPj0dKSgooisLBgwdRUlJy
    Q5BbV1eHlJQUfPLJJ1i4cCGCgoJQWlpaDwpIkiSOHj2Kw4cPQyAQoF27dhAKhTh48CB2795d
    z0lMmjSJhUoDdBHxwoUL3ueuhmdevonNDXokpZyiKBw6dAgnTpxo9g+Hx+MhPj4ePB4PAoEA
    SUlJ4PGav9abkZGB3bt3M8/ZMtAdV1/b/yUAlgH4HE1TMK4CrUe9DL4z74zXX38d77zzDp55
    5pnQU6dOPV1ZWdliclQEQSAiIqKuR48e1SKRqMmY1ZnnjVfhCgAQHx/vLb0Hs9kcV1FREXbh
    wgVcuNCqghgVAHa1UhHo6gP/e3+Z2WwGSZJwu90miqLU98t1URSldjgcRqfTycr5NJHZQJMC
    O8xmM7Zv386OUAQFBWHhwoUYOXJkkyTdBoOBhTX7+/uzkGSAHgWRy2kxlOzs7Dslsbt54srh
    YOTIkViwYAGbTxmNRmzfvp3poLtA64BbmqKS0aTxCWgI38OghaoDmQ+kMUEKl8tFQACNKLDb
    7SKtVvswRVEuT6KT7bnwVjej0YhDhw4Rw4YNS9Pr9W3FYjGmTJnCOqri4mKkp6ez2rJ3chNa
    LBb89ttv2Lx5M44ePQq1Wg2KoiAQCGqlUunFNm3abJs4ceKvR44cGU9R1FCPzliz6V+KxWJ0
    6tQJPB4PLpcLFEVJKYrq/d577+06ePBgazogF4A9AP5BkuSw7du3o3379njhhRfQxHpruHLl
    Ct577z1mzssIYCNo5tUHdn8mwGoAOHv2LObMmYOHH34YKSkpUCqVcDgcaN++PbKyskCSJAv3
    SUhIwFtvvYV+/fphz549+PHHH1FUOHO4HwAAIABJREFUVASr1Qq32w2hUIiIiAikpaXhySef
    RHJyMkaMGIHIyEh8++23yM3NRW1tLSiKAo/Hg5+fHxITE1kt8a+++gobN25koU4AjWQYPnw4
    y/xYWVmJ3377Ddu2bWOghS5PItRc+/SC59n8fGVlJefDDz9EREQEevbs2WwfDofDgUKhAI/H
    A4fDgVwuRzMQjNSzc+fOYdWqVcz+d4PuiF7w0fs3H8ByT8K6FHcv1aYHPXb0FXyo8wsAKpUK
    27ZtQ5cuXQaUlJQMb4ouR2MSX4FAUBcVFXVx0KBB25xOZ2R2dvYLuA0T6Z1YaWkpysvLERcX
    x/5MIpGge/fu7HiFw+GQBgQE9LRYLJnLly9vTRZkN4BfPHFeakuGX6D5YLLxwO5r//t3N4Ig
    wOfztTwer87pdN5WreROc2tPESgHQPLp06fxyy+/4JlnngGfz0f79u2xcuVKOJ1OpKen3xMs
    2eFwoKysDE6nEzweD2FhYSAIAgRBQC6Xg8vlwmKxoLS09J4bcTweD6NGjcLKlSvrFf5OnDiB
    w4cPe/u4A7iNtGNLJ8DBoIfNnwWtiSXkcrmIj4+npk6dSjRGJ9LPzw+TJ0/GH3/8gfz8fK7T
    6ewBWux4IoCvQZNw6Frzph4/fjyKioowYcIEf71en+p2u3mxsbEYOXIkuFwuSkpK8MYbb2Dn
    zp0YOXIkvvrqK2/q7gatqKgIX3zxBTZv3swSO0mlUpNUKs3w9/ffHhAQcGTkyJGl+/fvd5eV
    lQWTJCnkcDiQSqUsTLM5LD4+HgEBAdDr9XC73URpaWncoUOHBFVVVXWVlZX14JctbIUA1gFQ
    Go3G+DVr1qC2thbPPvtsQ+y5jba6ujqcOHECa9asYfSWXaCZyDfgLghAHtjfwsygu6ZwuVzI
    yclBXl4evv32W3bG3GAwsPBmDoeDXr164ZVXXsH48eMhEAjw4osvYvz48SgpKUFRUREsFgtC
    Q0PRoUMHJCQksJp9QUFBeOaZZ5CamorTp08jJycHZrMZoaGhaNOmDbp27Yq2bduCz+dDoVCA
    oij89NNPLFxaq9Vi+fLlUCgUqK2thUajgVqtZmUDQFeJy5t5rb4BMICiqG5Hjx7Fa6+9hvff
    fx/dunVrtpNKJBJwOBwwxb/mtAsXLuC1115Deno685mf81xzrQ/euwRoya00AGNBk7rc9TID
    GOO5zn2g561IX7jI3NxcPP7444KMjIyHrFbrDcQPUqkU7du3h06nQ3Fx8T3N53I4HAiFQou/
    v/9ZmUy2s1u3bns3bNhQMHTo0DdwZ9DK20f+JhPy8/NZKTMmIExKSoJAIGCQKEKDwdBl1qxZ
    vKqqqtb2QUUAPgEt3ZbYUCLA4XDYPcvhcMDlckFRFFtE9D4aMCeATaAL0A/8733uf++DBNgl
    kUg0BoMBERERTQYP9n4UeuLR9kajUbhu3TokJCQwZJ3o0aMHVq9ejVdeeQXHjh27+0yboqDV
    alFXV4eAgABERkaCz+eDoigEBQWBw+HAbrffc/eXw+EgLS0Nq1evRufOndmfnzt3DmvWrGHg
    zw7QfBRXbptMt6DTTQawEDQ7arBIJCJ79erlmDhxImfAgAG8rl27NrpKP2rUKERERCAzMxO7
    du1CZmamwGKxdAPwDoB+oKvSWWglDbjg4GCmEhubnZ0dAwDJycmIi4uDzWbDxx9/jG3btgEA
    4uLiWFmS291gWVlZePvtt7F//37Y7XYIhUJTeHj4sZiYmO+ioqJOtGvXrrqyspLcv38/xGJx
    YHl5eZLT6YRUKr1BW7SpLTIyElFRUdDr9SBJEi6XK0ksFkcFBgbmr1ixojWfMRRoYpYgAEu0
    Wm27tWvXcs6cOYMnn3wSAwcOhEgkgslkgtFohMViAQNFcTgcTHADPp8PoVAIf39/BAYGQiwW
    Iy8vD5s3b8aePXugUqlAUVQtaMjXWvhYR+SBNak5QHdNKQ6HQ4SEhKCurg46nY4lm+NwOJBI
    JFAqlRgxYgRmzpyJTp06sUUoPp8PpVIJpVKJIUOGgKKoWyI0+Hw+2rZti6SkJGbGjw0Qry9C
    rVq1Ct26dcPGjRuRk5MDi8XiTcrGBswMWsOTAFc083pdArAKwHsURSnT09Px6quvYvXq1ejR
    o0ejnvtMIExRFFwuF6xWKywWC2w2G5jKucvlgsvlgtvtRmFhIRso5+Xl4eTJk+ByueDz+ez1
    CwQC+Pn5QSqVgmHgZILvxjr9rKys65PfPI8f8kX0hz+AcQCeB9ATjSOIaciEoJlWewF4HDQU
    ejdamXho9uzZTKdUodVqu1yf3CoUCrz88suYMmUKrly5gsWLF7OEaXcYvMLPz0+nUChOyGSy
    n0NDQ3/v1atXxaZNm9zjxo3jZmdnhzscjiatOtfW1iI/Px/XzwtGR0cjLCwMJSUlcLlc0Ol0
    cTweL5ggiOp169Zh/vz5rel/94MmAvonaH3WehssMDAQEyZMQLt27RAQEICgoCAEBgZCIpEw
    zL8wGo3scfz4cfz22283K1rUgkZefIjmZTt/YK1r3qilu95fTIeSObyTLB6PxxZgGJ/C3G/e
    X++VzImiKKvdbi8B0GSyQDcpCP0IWrbpoZycHM6yZctAURRGjBgBHo+HXr16YdWqVXjllVeQ
    mZl52/Xy/p7D4YDP50MgEIAgCLaD7e/vDy6XC7fbzeY3brcbAoEAwcHBcDgccDqdrE/3XtNb
    FSMJgsDw4cOxcuVKNvl1u93IzMxkuVAo+o8PA/geNA9AqyfAHI+TfBtAfw6Hw+nYsaNr9uzZ
    5MSJE7kxMTHcO4WnCQQC9OzZEz169MCECROwa9cufPPNN7h48WIgSZLTAbQBPZOUjlaoSJMk
    CYFAAA6HE8Xn84NtNhuSkpIgk8lw6dIlHD58GA6HA5MnT8bixYtvC8d1u91IT0/H8uXLcerU
    KXA4HJdMJrskl8u/6tChw7affvpJ++WXX2LBggX497//jZ9++gnJyckdeDxebwAIDQ1FUlJS
    s15zREQEYmJicPnyZeZmjdNoNJE2my3/2rVrrf3AtHs2RC6A+Varddz+/fv9MzMz0aVLF0il
    Uuj1euh0OphMJjaoJkkSBEGAy+VCIBBALBZDJpMhJCQEEokE165dQ1FREUiSpAAUAPgSwHe4
    T+A5Xg8eiiAItBSDKkEQ1N9gWUoB1HG5XL/Ro0dj+PDhKC0thU6nA4/Hg1wuR3x8PLp3747E
    xMQGi0/XO+CGfu92YyIKhQJz587FqFGjkJGRgT/++ANlZWWoqamBQCBgA+UdO3YgLy8PoKvp
    5c28Vm5PYSgIwJtutzv68OHDsFgs+Ne//oURI0bUIz8kSRJmsxkmkwkGgwHl5eUoKioCo7tc
    U1MDk8kEk8kEi8UCu93OJsFM4YpJjgHg448/xldffcU6aoFAAD6fD5FIBH9/fwQFBSEoKAhy
    uRwKhQIRERGIjY1FaGgo/P39ERAQAD8/v3qJscPhwOHDh/HOO+8gMzOT2RslnkR/H3ykE+pl
    bQG84ElUQ5v4tSWeAKudpwD9H9Cwu1YxkiQRGBgIh8ORwOVyw7z/LyAgAIsWLcL8+fMhkUig
    VqtvS7x5i6KULTg4ODMmJua/Xbp0Obh+/XpDRkYGBgwYgN9//x1utzvG7Xb3aGreDYa93Wq1
    whsxFxkZiZiYGHaun8PhJGm12lin01n9+++/+0LB8EcA1wDMBw2JZplxY2JisHjxYnTp0uW2
    BScA+PLLL5GRkcGqXHiS7DwA/wXd/b0vuoMPrEEr8xRvGyzicblcSCQS+Pv7QyqVQiKRIDg4
    GCEhIWwsFxgYCJFIVM83MIVSiqLY4qq3f7FarTAYDNBqtdBqtWzsaLVaWd9lt9tvGzNRFGW1
    Wq0lAFiCzGaKVVaCHnVJOXPmDBYuXIilS5diypQpkEgk6N+/P1avXo1FixYhKysLPB4PwcHB
    CAoKQnBwMBs3+Pv7QyKRQCKRQCqVws/PD2KxGCKRCJGRkazUKo/Hq5ckM4Wul19+GY899hhs
    NhtsNhssFgssFgtqa2thsVhgNptRWVmJ8vJy1NTUsIfL5ULfvn2xcuVKJCcngykY7N69G6tW
    rcKlS5eYtT7jyTWLGrMwzZ0A80BDpN4GkCwWi8lx48bZXn31VX6PHj0E9wrJJQgC0dHRmDt3
    Lvr374+PPvoIu3btIsxmcz9PFfBN0DMoLToXLJfLIRQKWTy8p1IMACgsLERpaSlkMhlmzZqF
    2NjY2zrzAwcOYOnSpbh8+TL4fL5ZJpP91LFjx3WJiYlX9Ho96c3szOVycf78ee4jjzzysMVi
    iQLo7rO3dEJzmEQiqce8arPZJKWlpX5ut7vZJJjuomp4EvRsQCaA54xGY4cTJ07ckHkw3SGh
    UAiSJOF0OmEymVBTU3MziEodaH3rfwPIQMP6dH8rmz59OjZt2sRRKpVxNTU1ApIkcerUKXz+
    +edNW6J0OlFYWEhXKux2cW1tbfi5c+euNQMrYlPaKQB5Tqcz+fLly3j55Zcxc+ZMOBwONlFt
    bCfR7Xaz0D6mAsp8z1SkvZ0JU5S51etzuVwkJSUhKSkJU6dOhd1uh9vtZt/Xli1bmD1JATgK
    Wi+wJYLgTZ6g5TWKojqcPHkSCxcuxOLFi/GPf/wDdrsdV69exfnz55GdnY2ioiKoVCrU1NSw
    RanGFGGY6xSJRCAIAna7HRaLpVF/y+x9kUgEuVyO2NhYxMbGIikpCV27dkX79u0hFAqxZ88e
    rF271lvbNdsTZOyA70EvewN4C8CIZvb5cgDPgeb4eNPznG3xYpZIJILL5QKfzw/g8Xj15m8H
    DRqEp556ChKJBBcuXMBbb72FOynQEgQBsVicr1Aovo2Pj//h6NGjJceOHWOfU4sWLcKxY8cw
    aNCgFKfT2bE5rq+0tBRarbZeAhwSEoKoqCjvAo2spKQk0OFweLPHt7b//QPAIs+zcy7oMTii
    W7duSEhIaNTaA0Dv3r0RHx/PcBhYQc/6/dvj3+872LPnuikmiWA4cJrKKIqCTqdjx2KYgrev
    Gp/PB0mSFyiKKidJsq33Ovn7+yMsLAxJSUmIj49HVFQUoqOjERMTg6ioKAQFBbHJLfOVy+Xe
    cZzBdIa9k+O6ujpotVqoVCqUlpairKwMZWVlKCwsRElJCXQ63Q1dXg6H8ydBEIVMHNCMdgbA
    vwCsBtD72rVrnKVLl0KlUmHu3LkIDQ3F4MGD8e6772L79u2IiopCly5dkJiYiNDQUIjFYvD5
    fHC53AZjD28/6nWNAP5qXN6M/4OJgZi1tFqtqK6uRkFBAS5fvoyqqipMnjyZ1Tw3Go349ttv
    8eGHH6KsrIzZH+dBNz5P30mC2pw2zFMR7xwcHEw999xzznnz5vGioqKaFBbE5XLRo0cPrFmz
    Bm3atMG6desonU7XBTQUzQYagtNiFhERAS6XC5vNpidJ0kJRFIxGIyiKQkxMDEaNGoXw8HD0
    7t27vodwudjglrGTJ0/izTffxJUrVyCVSmv8/f0/USgU63777TdtcnJyPZkFh8MBgUCAp556
    qrNarR7ndDp5EokEY8aMqcfK1hzG5XIhk8lY2IPNZuMZDAZpXV1dsyffd2jVoKF6IgDLpVKp
    X9++fREREcFWtZiOkL+/P1wuF1uFMplMqK2tRW1tLa5cuYKcnBxQFFUM4F2PU78vjKIoPPvs
    s4iLi+P16NFjrF6vn+90OiUA8Msvv+DAgQNNfj4GOmM0GkNVKtU/+/XrZyAI4vITTzyBTZs2
    +eIy5YDu9r995coVvw0bNiApKYmtgDLFK4fDAbvdzjpHjUYDs9nMVj6Z6qfNZmMhvN4Hk8xd
    f/j5+cHf35+F8EqlUgQGBkKhUCA4OJitaItEonrSZwxs31NtLgCwHi0HWa0D3QnSe5xx77y8
    PO6yZcuwe/duGAwGFBcXo7q6mg0GuFwuhEIhuzcVCgUCAgIgFAohEonYr0z1XiQSQSwWswdA
    z+kzh81mY6v4NpsNdrsddrsdNpsNJpMJWq0WZrMZdrsdhYWFbHLE5XKhUCgQExODoKAgXLhw
    gZE7cnsC+7cBHIKPkDAysQe80FdoGdUHHoDhoGeLl4OGo7UoEaJcLgePx4Pdbrd4FyT5fD66
    d+8OuVyOuro6fPnllzh48GCjkS1cLheBgYHnQkJClqemph7Kzs52XB84u91uLFq0SLZz586J
    Vqs1oDmuT6vV1iO6Y4LLsLAwcDgckCSJuro6YVlZmYiZzfMh04AmTfMDsFwsFkv69euHyspK
    aDQa1NTUoK6urt4IElOM9vPzg1wuZ0m/PAlwCWhFkdO4z+yZZ57B119/jejoaIXVauXzeDxM
    nz4dM2fObDI0FkEQsNlsWLZsGY4dOwan0ymw2+2BPnbP3JAA8/n8Kzab7WeHw/FPiqL4CoUC
    EyZMwIABA9ClSxdERkYiKCioWWQ/mXVjOsXeFh0dzXJbMEgmvV6P4uJiZGVlITMzEwcPHoTZ
    bAaHwzEIhcL1XC63yu12e/NyNJcdA/CyJ0kcUV1dzf/ggw+gUqmwePFitG/fHqmpqRgwYABE
    ItFNk1yKouB2u+FwONgYhSnWM80izwgGW8w3mUyorq5mmwJMAs0k094Hn8+HWCxGQEAAwsPD
    0bVrV4wfP76ehKtKpcK6devwzTffMKoubs+1rQBw4k6Krs2ZAHf1BDmdAwIC3C+99BK5cOFC
    QWBgYLOVlsLDw7Fo0SKIRCK89957pMFg6Oh5D2pPdaBFTKlUorCwEHa7vUggEFRTFJWYm5sL
    rVaL7t2749NPP2U/ZMbMZjO2bNkCmUyGCRMmgMfjoaSkBGvWrMG5c+cglUpr4uLi3o2Ojv6P
    Xq+vBWhon7eNHz8e7777btCPP/44x2g0dgSAbt26ITW1+QkYORwOAgICwOPxmOCVK5PJ4lUq
    FTFmzBjqHsTsm8MEoAk5xP369cO6desQFxcHPp9/07kQZiMzm99ut2Pnzp2YN28eTCaTHEDU
    /eR4P/zwQ0yfPp27aNGisYWFhe+YTKbOfD4fAQEBzdqRdbvdMBqNHL1ePxoAUlJSXt20adOf
    er2+2Qs4d2EuANsAjHG73SN27NgBBgpdVlaG4uJi5OfnIz8/H2VlZVCr1TAYDOyceV1dHex2
    e2NIXW7pgL07nUwiHBQUBJlMxsJ427Zti8TERCiVSgQEBGDr1q34448/ALpTssWTvLWkOUFD
    hKsALAYwXqPR+P3666/1Es24uDjEx8ejTZs2iI2NRWRkJGQyGYKCgiCRSFioGvPVG3LlPafE
    7N/r97E3pI35arFYYDQaYTAYUFlZCZVKhfz8fBQXF0OlUqGysvJ6CRozgO2gu08X4Xuw576e
    4lyfFj4v4Tn3ak8B+lhLnlwkEsFiscBqtRY5nU4NgGjm+VJeXo7i4mIUFRVh//797L1BEASU
    SiUIgkBhYeENCQaPxyPDwsJOhIeHv3727NmMFStW4Isvvqj3O2vWrMG0adM4s2bNekin0w1q
    ro5ObW3tDcEyh8NBYGAguFwu00nhisXiULPZjG7durW2HNLN/G8SAD+Xy4Vvv/0WGzduRE1N
    DcxmM7snmQIgk2iIRCIEBAQgMDAQ1dXV8IzlKADE3m8J8IcffoipU6cSOTk5A7Ozs+e6XC6J
    WCxG586db2ic3HNVsq4O8fHxOHbsGIxGY5hKpZrXoUOHfxUUFKi+/vprPPPMMz61NhKJBCaT
    yS4QCDa73e6RLperr0gkwuTJkzFy5MgmK8pf72/vJiYODAxEYGAglEolhg4dih9//BG//fYb
    U7TaLxaL97lcLqpnz573RELVSCNBo3KWgJ7Hn2g2m/02btyI8vJy/Otf/0K/fv1YtCpJkrBY
    LKiuroZarUZVVRWqq6uh0+lgMBjYcaS6urp6RXu32w21Ws1CwDdu3IijR4+yRSymYM8cDDyd
    iV9CQ0MRFhbGImk5HA77nnJycrB69Wps27aNeQbaQPNOrMZdqC40VwIc7lngASKRiJw7d65j
    wYIFzZr8Mubv74/nn3+esFqt7g8++ICwWq39AbzmqXyUt8QGnTRpEkaOHInu3bubQkJCsjUa
    Tf/z58/j0qVLSE1NrScSzdi+ffvw5ptvYvjw4Rg7dixIksSmTZtw6NAhCAQCu1wu/yIpKekz
    m81We+bMmRv+ftq0aYiNjeVt3bp1amFh4eM2m40fFhaGefPmIT4+vsUSYK/uNdfPzy/hjTfe
    4MH3YEkdAAwRiUREWloakpKSGmTI9g6qmXngwYMHIzk5Gb///rsCNMz/AHyT9fWObMKECair
    q+O9/PLLYwsKClaaTKbOUqkUTz/9NEaPHt1sUloArfG9adMm7N69m6vT6UYDoPr06bNUJpNd
    efjhh7Fnzx5fW65SAN8C6F5eXh6yYsUKfPfddygpKUFJSQmMRiMcDscN0CYGfuXn53fT7q73
    AeCmneHrD61We4M+KBM0KhQKJCQkICwsDJmZmQwM6yzobqy9FdaNBM2U/Apogqw5AJQCgYCY
    M2cOHnnkEcTFxUGhUEAkEoHP59/TfXez4IWpNDcUADmdTtjtduh0OqhUKuzbtw8fffQRs375
    oGddt8A3Z/7beir9vVvxPXT3FKA1aEFJmrq6Opw7dw5BQUHVEomk2GKxdGc6FDt37kRubi50
    Ol09De6uXbvivffeg1qtxiuvvMJ091nfplAozrRr1+6Vhx9++OyMGTNuikpZu3Ytfv/99+45
    OTnzzWazojmvz2v+9YYCtNPpBEEQ3JCQkDarVq3i7tmzx9e0cDuCRiYQTqcTWVlZ9/Jaco//
    3e8pSP3t7fPPP8fcuXOJAQMGDMrNzV1VU1OTIhQKicmTJyMtLa3JzycWizF79myGLJCn0Wim
    AqCGDh267MUXX1T52iiSRqNBUFAQ/Pz8rjmdzg1ut7tTWVmZdMuWLejduzerCwvQ41XXo60Y
    xBWD/Ln+qzcyiFERYJBU1yOPvP+PSeK852O948qqqir89NNP0Ov14PP5JUKh8Fu9Xq9PSkpq
    ieTX2/4ELX+XC+A5p9MZs3//fqjVasybNw8ymQwFBQUoLCxEeXk5qqqq2AI+E88wR2OQCEVF
    RSgqKrrhecUcTEeYIZyVy+UIDw9HeHg4YmJikJiYiISEBDgcDqxbtw7p6ekMYlAFWvlnPe5y
    jKs5EmAOgEcBjCcIgjNu3Dj7ggUL+EFBQdyW+nQDAgLw/PPPc/Ly8uw//fSTkCTJhzyVj0/Q
    QnCsmJgYvP/++47evXvvlEqlE1QqVch//vMfKJXKG+ZdysrKsGHDBmg0GnTo0AFCoRBnz57F
    999/D7vdDqVS+Wv37t0/u3btmvkf//gH9u+vj+geNGgQlEolZ9euXQ+Vl5cvMZlMMj6fjyee
    eAIPPfRQsyYt11e7vLH/TqczPCcnh2cwGHwpAeaBhuYr4+LikJqaelfyUFFRURg1ahROnToF
    p9M5yOPU/8Df2BYtWoQOHTpwP/vss7EFBQWrTSZTR39/f8ybNw+LFy9GSEhIs7+Hzp07g8vl
    Yvv27VydTjeGIAiif//+S3bv3p29ZMkSfPDBB762bIcA7KMoavqZM2cIpjjF4/EglUoRFhaG
    yMhIREREsAcDUb7eqd7se4qi6sF1b+WorVYrmwRXVFSgoqIClZWVMBqNKCkpYWesPcboZLY2
    LKMCwEeewtFqANK2bdti+PDhrR5weTRdIRAI4O/vj/j4eBQUFDDvy0IQxGcEQXwGwHGvTKDN
    YCGe4sJI0N3Y1rRhAF71HC1CTvTOO++gV69eOHbsWF1sbOwRPp8/zuFwCACaaObkyZM3FM2f
    fvppjBo1Cnv27LmhYBUYGFgUFxf31pEjR86sW7fuhuR32rRp+PPPPxEcHBxx4cKFV6qqqno2
    5z3hdrthtVobLEATBMHhcrmx58+f59ntdl9KgBmIfPyt4gjvgJi5XpIkGwq4Gf/7t+8Cv/ji
    iwgKCiIGDRo0MDc39129Xt+foigMHz4czz33HIRCISt/2ZSWmJiIefPmQavV4tq1awKNRjON
    w+Fg0KBByyQSier5559vcu6Pe7Gamhq43W5SIBDsBvCQzWYbt3fvXqSlpaFPnz7Iy8vDlStX
    UFBQALVazY4fMUmct6TW9VJb3v++Plnzluu6/mcikQghISEIDQ1lyRTbtm2LLl26IC4uDrt2
    7UJ6ejoIgnCJRKL/yWSyk0KhkCGjbA3fuxbAZQCvUBQ1ICsri7Nw4UJmfLNRJF53XQG/BfLN
    bDZDq9Xi6tWrAP4agWLgzx55ScoTa68Czb9ju5eHUVNbJwBPA5B06NDB9uqrr3KioqJaWm8Y
    4eHhnCVLlnBzcnIcFy5cEAOYCRqKldUS5+/QoQNGjhwJhUKRUVVVtcdqtT61d+9ewu12Y8mS
    JejWrRt4PB40Gg2++OILHDt2DEqlEqNGjYLNZsP333+P/Px8BAYG5sbGxq7dtm2batu2bZg0
    aVK980ycOJFJftNUKtU7RqMxgcfjYcKECXjhhRduyzDdlAEjo73pVan2LysrI4xGoy/5mHDQ
    GpiC/v37IywsjIU2ajQaWK3WeomF9wYUCoUIDAxEeHg4FAoFUlJSEBERgdLS0nhPoJcF35oB
    bLRNmjQJ7dq1433++efjCgoKVplMpo7BwcGYP38+XnrpJQQHB7fI+0hKSsKqVavA5XLxv//9
    j6PVatMAoF+/fq+ePn36T1+rRvN4PK3L5doEYCiA6PDwcIwcORKdO3dGp06dkJiYiICAAHYm
    VSgUNtv7pygKNpuNnXfV6/XIz89HTk4Ozp49i0OHDqG2thagydr2oIVnM29hdtDwYZPD4ZDm
    5+fD7Xbflu26pc3tdiMvL4/pvFkoirpEUZQvEt4RoFl2p6DlZA4bMi6AiZ4C9FdoIZh4TEwM
    UlJS0K5du3SbzfanRqPpfquEa+zYsZgyZQpcLheOHj1aT6vSz8/PnJCQsO7RRx89Mn/+fKxb
    tw4LFiyo9xoXL16Ev79/WH41lIBBAAAgAElEQVR+/psajeZRp9PZrA8ol8t1A7HV9Qmw51kQ
    UlhYyPXMyfmKRQAYDUDg7++PqKgottvDsK77+fmx7LJMsl9XVweLxYKamhoWjsn4bbvdHgcg
    1eN//7YkWJ999hnmzp1LDB48eFBOTs57er2+L5OAnD17FrNmzWq2hIQgCJAkyaKIHA4HX61W
    TwNADh48+M3PP/+8bP/+/c3Sgb6XGDslJaVy+/btP2g0mhSNRhO8dOlSiMViaLVaGI3GG5AS
    d5usNdY8hEzezw8EBwcjPDyc5f+QSqXXIiMjt1itVmtiYiKqq1uNtNzG5XJ3u91uMYAuAIJ8
    hDDvhmLfdQU/C+iu7y/36k+a2kEKAcwA0FUkErlnzZpF9uzZU9hai5ecnMybNWuWdenSpdy6
    urpOAJ4EDcWyNfe5lyxZghUrVmD58uWGtLS0TyiKal9ZWdlvz549yMnJQc+ePSGTyXD58mVk
    ZWXB7XZj2rRp6NatG44fP46dO3eCIAhbUFDQ5p49e55xu903JL8vv/wyBg4cyHnnnXdGlZWV
    rTYajR0FAgEeffRRrFixolHMik1pTDWMMafTKSAIgiOVSuFDiUs/AN04HA6uXbuG+fPno6Ki
    Ajqdjn1gMgQczAySNz0+Q6MfGhqKoKAgplsg8Dj1H9AyjLpNav/85z+xZMkSYsSIESMLCwtX
    mUymjjweD2PGjMHYsWOh0Wha9CFNEASmT5+OoqIinD59mqPT6cYAILt167YwOTm5sGfPnjh3
    7pxPrJ1HSzcbdDc1OikpCStXrrwtu3tzrZs3+RPD5PjII4/g+PHjyMzMRG1tLQmaD8GXYLta
    0DPBkSqVClar9bZMp96M2Tc7GIgWSZL1mD6vP7wZtht6PjkcDhQXFzPBkNZz+KK1Aw0pD/Sh
    9yQF8Iyn8HKlJU64Y8cOPPHEE+jYseM1nU73pd1uX2UymYJvUsBCu3btIJFIkJWVhQMHDtTr
    /AQGBh6Ji4vbsnXrVvuZM2fw6aef1isanjp1CjKZLLSkpOQNjUYz0263C5r72kiSvGlgz8zL
    ed2z4pqaGqKZ9EXvyf8GBgZi6dKleOihh1jeAmbO73YBsdlshsFgQHV1NdatW4fNmzfzAYwC
    sBk0KdbfzubOnYshQ4Zwhg0bNvjq1aur9Xp9P0bvnSCIehrzze1DGP1bh8PBr66ufoIgCIwY
    MWL53LlzfSq2+eOPP0AQBBUeHn7aYDDkUBSVcj3U1js29SZZul72iPn++oN59ntLIF3/b4YB
    2mq11iOGAsAmb+Xl5d73cEbbtm2vcDgc7Nq1q9UTTNCSQTWgZQobXD+Gc0QsFt927TxSsLdc
    M+/vbTYbrFZrPYh1A2YFLXt2z8XUpk6A24Cex+D17du3bsqUKQIul9tqWQ+HwyEmTZrE27Vr
    V116erq/J0nZgLsYlr4bW758OSZMmICtW7eeHzBgwGtOp3OVTqfrk5eXx70e9tCzZ09MmzYN
    tbW1+Oqrr1BeXg6ZTHYhNDT0px07djiWLFmC48ePs78/Z84cJCYmctauXTu8uLh4VU1NTVeB
    QICpU6fi//7v/6BUKlt1Y3mCUMHQoUOFcrncV2ZjJZ77M5gkyRugcIwDYGYnBQIBy+RrtVrh
    druh1+uhUt3UD3QHTfzyt0uAbTYbvvzyS4lGo3nUbDZ3ZNbh/PnzWLhwIVoD4kkQBKqqqsDh
    cOByuQi9Xj/Sbrf3piiqsCUg/XeRwKkAGqJzqyqqy+Vi9e7sdns9GQXme8YZMAcT2DJOmyF8
    Yg7m38wMkkQiuWn3VKPRwIPEsAAohm8RNjHr16OyshI6nY5NgK1WK0uOwwS+Go2GJQJimJ29
    v3oHKowmu7f2L+PAvb/6+flBoVAgJCQE/v7+rD4wQ/xWU1PjDT0s99EEmA9a57enD763bgCe
    AC2P1CKd802bNmH8+PHuYcOGbXG73fEFBQUvWiwW6fWFjY0bN7KEZ95kjWKxuFIul2/43//+
    V/X444/jeu4Ns9mM5ORk+Z9//rlUrVY/Y7fbRS21mI3QF4XL5eK3adOGx2jW+0ABmvG/QZ07
    d6Yee+wx4k7jFC6Xy6o0KJVKqNVq6tChQ0R1dXU3j//92yXAK1euxBtvvEEMHjx44NWrV9/V
    6XR9eTwe+vfvjy5dujBkXy3qe202GzIyMpCTk8NXq9UzCIKgEhMT/xUbG1spFApx6NAhn1i7
    S5cuAUC5zWbLAZDCPgj5fISEhCA8PJzVyI6OjkZkZCRCQkLg5+d3wyzv9d8zcYa3moO3aoD3
    vy0WC/R6PSorK1FWVgaVSoWysjJUVlZCq9WyKhcEQTgpiioqKyuzFxQU+JL/LYfXWAKj6hIe
    Ho7o6GhER0cjKioKkZGRLBP7zdbN+2cMGoVhh/Zer+vXkEGsVVVVoby8HCqVCiqViiXeuq6I
    12QF6KZMgAnQOoOJIpHINWXKFDI2Npbb2p9sZGSk4PHHH7dnZmY66+rqlKDnoi61VAC4c+dO
    PPXUUxg6dOjxEydOvODn5/eEXq9Ps9lscrFY7JLL5SK1Wh1kNBq5WVlZyM3NxYEDByAWi41h
    YWHr16xZk799+3bMnTuXfc1169bhkUce4UydOnXYtWvXVhkMhm4ikQjTp0/HsmXLEBcX5xO7
    iiRJvkKhECsUinqkIq1oHUCTb0AqlSIkJAQhISFQKpVQKpXsbObNIFgMDMtqtaKyshKFhYUo
    KyuDwWCATqeD0+kM8jj3fZ4k429jHmZrLkVRIsbROp1O5OTk+Mx7dLlcHIvFwmUctK/VEACU
    ASDVajVHpVIhLCwMGo0GJSUlKC4uhlqthl6vh1arhVarZZNgbyfg3bVkvnpXX72lAxhpEObw
    9/eHXC6HXC5HSEgIIiIikJCQgNjYWAQEBCA7OxsOhwOgiWIKfWz9ajzrh/Lycpw4cQIXL15E
    Tk4Ou88Y9kkmwWXYJhsxH3jLQPp6KQZmTRk5qaioKCiVSiQlJcHlcqG0tJT58zLQUk6+ZgkA
    HgaNSPE14wEYB+B70CQsLWJxcXHYs2ePsV27dh84HA5DVVXVU1arNcnpdPIEAgF4PB5UKhXW
    r19//f1BBgUF7UlMTPxt7Nix2LdvX/1svls3BAYGKi5cuPDPqqqqOXa7XdzSBebboaqcTiev
    TZs2AoVC0SRQ0CawjgAGC4VCjB07loiJiWGlYrzZnxm4s81mA5fLhUQiqcf8zihoiEQi9OnT
    h+jVqxf166+/Bnrur334m5FhmUwmrF27VlpeXj5dp9P14fF4GD9+PJYtW4Y2bdq0aPLrFbvh
    +PHjWL58Oc6dO8errq5+zOVy7aUoavvdcKY05/sMDAy0u1yucqfT6SYIgpuamoqxY8eibdu2
    aNOmDWQyGVvovNvRmsagE5hYiinEGgwG5Ofn4+rVq/j5559x6tQpUBRlstvteefPn6d8aB3Z
    Aj4A9OrVCxMnTkT79u3Rtm1byOXyegX45v48meK/yWRCUVERcnNz8csvv2Dfvn1MTFQGQOdr
    CXCoJwEWJyQkOFJTU3lofQIOACBSU1N5bdq0sV++fFnqeY+bQQ+Bt4j98MMPoCiKfPLJJy90
    69bt2okTJ77z8/OLUiqV1l69ein27t074/Tp02kvvfSSwGq1wm63u2NjY3/p37//jtdff508
    ceIE+1qvvfYa5s2bh7S0tJTc3Ny3dTpdb6FQiKeeegrLli1DVJRPKfLwLRaLiMvl4uLFi74Q
    gKWBlkxAaGgoxowZg2HDhiE5ORnh4eGQSCS3Ta4YXeCCggL88ccfOHDgAI4fP85U+IZ4kuyz
    +BtbQEAAUlNToVAoWsX5elei8/LykJGRwTCbwkcTYMqTVNYaDIaAjz76CDweD8XFxdDr9TCZ
    TKirq7vpWt4Kjus9TsCQcnh/Zb6/1bpJJBIEBAQgJCQEsbGxKCkpYWBFNS357GtsfQM0o7aj
    srJS8M9//hNOpxM1NTUMxPyvTczjsR1d7yr0zSrQQiE9fXMzErHrCcUcDgfq6urgdDrrwdUI
    goBUKoVIJGK0k0lPtdzmg1t3OGj2Z1+1RI//zcYdaDXei33yyScAAL1er50+ffq/z549e1Sl
    Uo2pqqoKVSqVVNeuXSMvX76ckpubG8rsJ4IgIJfLryUlJX176tQpkze7OkVR6N+/P/r06SNL
    T09fUlZW9pzNZpO2+AOnEc9lkiT5BEGI3G63L8ggMf43TiKROAiCcGzZskVaWFiIwsJCFBUV
    oaamhkXBeOugMzJITPIbHh7OMsNGRUU5w8LCrFwuN9Dtdv8tybCcTiesVqvA7XbLSJIkYmNj
    MX/+fHTt2rVV39eYMWOQl5eHnJwc2O12gdFoFPma/3W5XKiurqYIgsgDYCYIImjQoEFYuHAh
    WgMp5l2wkcvlSEpKQmpqKq5cuYJTp07BU5wp5nA4aC6ZtLuwWk9SCQDo1KkTXn75ZdZ/tqQx
    kkd+fn6QyWSIj4/HkCFDoNVqvcl/fTIBjgTQHgBSUlKcSqXSz1c+3ZiYGOGAAQO0ngS4LWhd
    wBYNAr0eGlbQHehLe/fuhUwmQ05OTlZ+fv4VtVrdjyRJUi6XZwUGBv73m2++0ezcuRNMArx4
    8WK8//77+P333zsVFBSs0Ol0/UQiEZ5++mm8/vrriIiI8KkHO4fDEdTW1opJkkRxcXFrvx0Z
    aIgUDwCKi4vxww8/4MiRI4iOjkb79u0RGRmJ4OBgBAcHs8RFbrebJd8wGAzQarXIz89HYWEh
    S5zlpckYBlp25G+dAIeGhuLVV19Fjx49Wj0B3rRpE86dO8dCiHzYCgGYXS5XwMGDB+tdg1gs
    RlRUFNudDQ8PZ6uqzMHAca+H5pIkWQ/eezPIr8VigVqtRnV1NbRaLavTV1lZiYqKCly+fNn7
    fZaBZoH2xQKC2e12h1RWVkIkErFagMyaxcXFIS4uDrGxsQgMDGTh37eChd9MRup6yDnzvdFo
    hEqlYiWsqqqq2G69Xq/3RrDUgp6ZonxsDeWe5FLiw3tEDBqB9SNaeAZdp9PBZrPZMjIyTvfr
    1++cwWDgjBs3Dk8//XTAW2+99bBGo1lkMBjaeYpHhZGRkWtmz56ddfnyZaxZs+b/2/vyILmq
    +t/Pvbf3vWfrmcnsi5PJJMYkYETCI7H4gYhARIgsKvV8hTxFrQKeaDCKv4KSoCIUovLDpwYJ
    Ji8hgr8XoFhCcMjySGImkJBtMmtm757pvft2913eH/ecmzudSYhkZvqOv/upOtUz01M9Z849
    53z3z1f9nBtvvBHLli1zvfvuu9/r7e29O5VKuQpyYC7gXmZZ1pJOpx0cx6msqgWWv5cC4GKx
    GPuLX/yCodFe6tT7ZxVlwrTPCYJgI8ZEKfkb+3R4Pi/0HoTVaoXH4zmvk3O2jBHaW5rMTdab
    ASyKIk0R7wYQlyTJNzAwAEEQ1Bpeel5oa7toNIpEIjGJ7yV/UNkqy7Iql89V52q1WuFyueDx
    eGCxWM7ik8hkMmr2kNVqjbjd7ogkSdAROZ0EpSwqBcAxNDSEXC43yQCm6ycIAmKxGOLx+AWt
    nyiKk2qG8wetIXY6nfD5fLDZbGdxCQmCoHXgUwf0tKS0TKcBPB9AsdlsxuWXX26zWCy6OSUm
    k4lbuXKl+49//KOUzWaLoJOWNcXFxbjtttuwadOm7uuuu+6nFRUV3omJCcnn88WfeOKJ3Je+
    9CW8/PLLAJS0hD//+c+oqakpO3LkyAOxWGyl2WxmvvGNb2DdunVT9hYuNBiGMadSKZskSbNC
    4PARiENpg/UBgCskSWoMh8NF4XDYevz4cWbHjh2TiHJo3SVNgaYpljTdkt6/RCEegcJk+3cA
    b0PJfJiLAlg12qxW6yQBUiiYzWY9RnynQpAIEJSVleHyyy9HQ0MDamtrMW/ePJSVlakpyi6X
    a0pSpgtRLrRET1MJ9mAwiFAohNHRUdWgO3bsGN577z3KpDgMfabopwGIDMNg9erVuO222xAI
    BFBaWoqSkhK43e5JZFYzZVTQtOpEIqGu5djYGLZt24bNmzdDkiRBp+tXAWDhHDgnrVAc0LNO
    wvbss8/i2WefBQhT/+OPP47R0dGQy+V6vqKioiOdTrewLIvq6uouu93+/q9//euctu63rq4O
    x48fNx04cOCWcDj8rXQ67dGr8UsMGHMmk7EzDFNIplmt/H0SwCFBEK4Ih8PNxCi24WNkCpK6
    THFiYiJO5G8HgHYAOzHHMTo6iscee0wXGVhHjx6FzkjUzoLL5QLDMBPpdDqZy+XQ19eHWCyG
    bDaLYDCo9qE9ffo0hoaGEAqFJhlwWkMuk8kgl8ud5ZShBFpUL6KGL311u92qo7aqqgoNDQ1o
    bGxEIBBQeSsAwGKxDFdWViZyuRx0xs7eS3RZRygUwtDQELxer7p+XV1dGBgYUNcvFoudd/3y
    S5Kow0q7btrhcrkm1RzT0qPy8nKYzWZtH+EklDr/aTkY02UAM0SweT0eT6axsZGB0v5AN6iv
    r5c8Hk88FAp5ibHOQgdEMJs2bQIAvPLKKzloCrtXrFgxWXrE42hra+MOHz58eyKR+LIkSeyq
    Vatw//3369L4pfYLz/M2SZKgg1ZIaSg9W3dC6ZPZRvZBPcMwTQzDfEYQhHKTyYTKykpEo1GM
    jY2BZVlUVFSIsVgsF4/HbUR5ep8Y0t0AuqD0Uuslh3POGr4fR8kyoKIUgAMAbrrpJjz66KMq
    IdV0GmznYis2m81wOByTskAkSVK9p7fffjsOHDgAKFkKDgAxPflcqPyw2+249dZbccsttxRE
    4aORY6vVOqnvtdlsxvbt2xGLxTxQmJb15uRqJvea3lFE7l1d0LiTnr4CFGLMQwBw5MjZRNXN
    zc3o7OxEeXn5Z8Lh8AOZTKZM7/czwzAWSswVixX8uKehOIfbAfihBCFaANRDqV2/DEA1x3Fi
    bW0tYrEYFwqFwHEc6uvrkUgkaIsekeydQ1AyMbqhMIv3EQfknBZclHBv8+bNupqXnup+8yEI
    AmRZtkqSZKLn97777sPg4CBOnTqFiYkJtb58pvUaKkMsFguKiorQ3NwMt9utRoBlWXakUimz
    jtKfVRuVftHX14d169YhHo/j5MmTCAaDqmE7G3ohXT+v10vLHPLv5GlTqKbLAOaIAsjOmzeP
    DQQCugvZBAIBed68eTKJRBZDYczMYA7gBz/4AV566SUAWJTNZv+HIAjO5uZm3HvvvQVne/6o
    /cXzvFVnfT0FKNGHUSKQzQBWybJczzBM+dVXX421a9fiySefxLZt2+B2u/Hwww+LyWQy/sgj
    j5hHR0dlKDVGj0GpW5zzsFgssNlsMsMwEr3E9cK0nGfsSQzD6IXQJR9LABTZbDZcfvnl8Pl8
    51Rctam3U5E5aX9G21LQFgRa8iZqrFHvdP4ZY1kWFosFtbW1WLx4MTWAm6FE4EZ0tHYuAJ8E
    YK2trUVra+tZv0CI2lRBzPM8EokE4vG4ygitXTv6SpU37fppWbPpoKzvtHY4fy2bmppQV1eH
    Dz74wAQl0uoDENaRA2EB9NX66FzwkLnqwgF9Idi0aRN++tOfYuXKlcUdHR3fzWaz8ws9pws0
    gE3ZbNYmyzLt/60X+RuEki31d6KDriLGcNWll146/vOf/9z59NNPO7ds2QKXy4Wf/OQnEAQB
    P/rRjzA8PCxDcUD/DHO05ZGBafasKCVorUSvx9DQEJ5//vkpjXjKEUFlgTaaS4c2SknaQamy
    J/9rGu2kLMc8z6t17Mlk8qyOIalUqqavry/AsqyeuoUwABYTxxQmJiawdevWsy1kloXdbofV
    aoXZbFb1jvOtH8Mwk6LEWhZoGjWm8pr+nOpFlHA2D24owSszpqHn93RZJVaQ2iO/3292OvVX
    huRyuRyBQEDWKFzWuWIAh0Ih3Hfffebf/OY3azKZTKvL5cI3v/lNrFy5Uu9TN/E8bxEEYboN
    KjM5COVE6TMRJ4yJCNgQgDEoUa4cUbSm0hgsAD4jy/L3ASxasGABHnjgAVxyySWq8iyKIux2
    u+UrX/lKycjICPOrX/2KS6fTt0Ipwv8PKPUI5/KocVBSvIqgkMR5yTzonDJEGQhC8ZDPqEL4
    9a9/Hc899xxWrVo1v7u7u1UQBJYoeLLL5XJmMplaAEgmk2hvb8fQ0FDBU7AOHTpE639Zi8Xy
    Ga/Xy8uyzGkuZb6qqmpPZ2dn+I477lBJb2YZDiiN5O3V1dVYtGgRAIXcJBaLYWxsTE2npbT+
    kUgE0WgU0WhUZUDNZ4CmBvBULNB2ux1erxderxc+nw9+vx/l5eVqynBpaSlKS0vhcDhgsVjw
    2c9+Fhs3bgSJXC2GvurUq4hRiYULF6KqqgrpdBoDAwPo7+/H2NgYhoaG1NYSExMTaj00HZlM
    5qy10xrA+WtosVjOqsH2+XyoqqpCdXW1mrZeU1ODqqoqVFRUoLW1lbbdaINCpqcXA5iFEtk3
    Qf9gyV1owiy1Q7pYHD9+HCdOnEBxcfG1uVzu83MoO8acyWSskiTR8ofplL9eKGn3Xo3steTJ
    3zB5xuJ55O9lAH4AYFFZWVnkwQcfzC1ZssRGeTVoT9U1a9bg+PHjeOqpp0w8z98ChYX9tzi3
    E5ojwwGlPp7KX5HIX5HMLUjmmiyUQ4YYETLDMLL2Z3oo/aFZRPRblmX1kE0w1bNeQpySquOX
    svmXlZWhrKwMVVVVqKurQ3V1Ndxut0qsRnkjphrUYa0d2paFdEQiEfT396Ovrw+Dg4MYGxtT
    5T6V76SErliSpE/qTP56if5ipnqXxWKBx+NBWVmZuoaVlZUqB4fP5zvnmmnXlGGYs9ZqqrWM
    x+MYGBhAX1+f2vWBjkgkorY0JGgj53lQLwawjRiVan8tvcFms3Fut5t+6yRz1t1Jngp79uxB
    V1dX0+jo6PXZbJZbvnw5br75Zl3UaJ7PKy3LMsfzvGWaor8sgHkAlhKP8SeI0ufRGMBmjQAO
    QonyDkCheB8g36fI7/oB/DcAawDMr6mpYe+//36sWLECJ06cUEm70uk0Dh06hDVr1jB33303
    Tp8+jS1bthTxPH8vmcs2AMfI57LkEq4EUE1GBTmspWSuWgM4izM92A4AeBcKQ+qM7EtiEDDp
    dPqmcDh8fy6XM9PnNjExAUEQ7AAwPDyMH//4xzCZTAU3gDOZDFXeOJ7nvykIwn+XZZkhygHD
    cdxwcXHx7ePj4wcKqDBUQolqIRAIYHBwEB0dHTh48CBOnTqlklNFIhHVwylJ0rQRnGgjnHa7
    HcXFxSgpKUFFRQXmz5+PJUuWIJfLwePxIBgMuqH0ZLXoyABpAlBD+08+/fTTaguk06dPIxaL
    qQLzn1kzbR/CC93HWtZZt9ut1iO1tLQgFotRwpUKcv+8r5P1M0Pf5Fdn+aOJ/J0TBvDhw4dx
    3333lW3fvv12nuc9epjTBe5nUyaTsYiiOB0OaJY4qi4BsJLs/3Ii0/IN4HFiVI5BIRul8pfy
    D7Aa+XszgJaKior0v//7v2euvfba0o6ODu7kyZMAlD7ge/fuxW233Ybvfve7mJiYwMaNG308
    z3+HOPK2EZmZJp9bROZJ5W8gT/7mG8AhMsd/QEnPPoIZIgm866678Oyzz+Lzn//8vJ6enmpB
    EBiGYbB161a5tLTUm81mS2RZRnl5Oe655x7U1tYWnASrvb0dGzduRDabZc1mc0tJScmnS0tL
    tQ7obFVV1fHOzs7kt7/9bTz44IOFmGoJkb8My7L48pe/jBtvvHGS8eZ2u1XSpZnKbqOR4Gw2
    q5bQBYNBNSL9zjvvgOxBvcnfGmJUAgA+97nP4Wtf+xoqKipURzol+KKR35lytlA5H4/HEQwG
    EQwGMTY2hr/97W/YunUrPQ8tUHoW68YAZsgAy7K6SZ/MP8yaeanz1TtWr16Nl19+GZlM5tqJ
    iYlmi8WCq666ClVVVbqbK6nFmGQA53I5TpKki90TxQC+TIzVxQD8LpeL83q98Hg8KCkpgclk
    Qjabxfj4OEKhUBUlOSACOUMumxSUQn8TlAiy12QyWZcuXYrvf//7+OIXvwgA2LFjBzo7O9VL
    befOnejs7ERLSwseeeQRVFZWYsOGDa6RkZHriBCPEsHOkQvORi44K8uyjMPhgM/nQ1lZmUqA
    lEgkMD4+jng83kLaP9xAFIR3APwJwF4y92m9YAAwgiDYeJ735HI5k7ZhufZioxd5ocEwjLYH
    nx2AnbIR5nI5cBznzuVyhY581ZOBI0eO4O6770YkEkEqlTpLgaGRR5p6Zbfb4fP51D6X+UP7
    LGjKVTabRSKRQDgcVvvi0vShVCqF8fFxUAWSMKXCbrdrPfdtRHnVQwq/hSjVHlmW8eabb+L1
    118/a++ZTCZ4PB44HA61h7e27zGNdOd7oRmGOctjT1seafsyk7OIVCqFRCKBZDKJZDKJkZER
    HDhwQFWcyP3mJnP+T+gji8hOHdBzyAC2zxUH9F//+ldcc801V46Ojl5WSINEeydOpYTmE88Q
    B7R5GhTWEgC3EPm7iBiv5xPo5RqFWtLI37TGAe2AEnmyAEBJSQnT3d1dsn79etPhw4epA1qW
    ZZl54403sG7dOng8iu/B6XSC53knlLZKK6AQbKXJ5zqhZPdZcWH9sJvJ6xehlIX8ncjfXdMt
    f0kklUkmk7eHQqHvZbNZM436hkIhNpvNegDA5/PhhhtuKHgbJLqntmzZAp7nTel0+r6RkZFv
    abcix3HDPp/vzuHh4Q8K6ICuhZICDY/HgzvvvBPXXXfdrE+COqJtNhs8Hg+qq6vV93ieV9s5
    6kz+UoNyHtVPbrnlFtx5550FsdFoGrXL5ZrEZ+J2u/Haa69RHaYISsbYXlxk1sZ0KY5pcgmp
    7Tn0BkrLTUAvTN2jqqoK3/nOd8o2b978b4IgWBsaGnDVVVfpqaZ2ktGUryDks9Z+DHgA/C8A
    3+Y4ztPU1ITLLrsMV155JT71qU/B7/fDarWCZVlIkoRkMolTp07h2LFjOHXqFHp6ekyDg4Om
    0dFRZyKR8NN0bJvNhvfc+JAAABOqSURBVKamJlx99dVYs2YN2trawLIsDh8+jL/85S9IJpNw
    uVyn0+l08eHDhx2bN2/GAw88gOrqaqxbtw7Lly/Hiy++iD179rhDoZCb9qm1WCzw+/2oqKhA
    TU0NGhsb0dLSggULFqCiomKSUp5KpXD69Gns3r0bu3btsv3jH/+oD4VCdVDaNT0A4JXpVpy0
    z8Vut+Ouu+7CwoULoQfF7p8x5N944w1s374dkiSp7RkKBBPx6HoBIBKJIBKJAMCki7y+vh71
    9fWora2dxAjt9/ths9nU9Nz8Qf9fbR9gmpbG87zamoumV/f29qKnpwe9vb0YHh5WW3jROWmU
    vjqdCGAakTbTexqA2gapoaEBDQ0NaGpqQktLC+rq6tR2CdpaI3quLkQJoynS+bVJsVgMfX19
    OHnyJDo7O9HT04Pu7m4MDQ0hmUzmP/NFULI9RnWwhrTcYq6A+wgDSjdYv349PB6P9bHHHrs2
    kUj49DAnmqWQv9fzSWpkWWZEUWQ0xtfHlb/fB/AtclY/zt60k3HOGvWjR486jh49qp5P8n8c
    A1DZ09Pje/zxx9XznUcg5Jom54+VGFJfg9JK6QEA22fCAZ3NZn3xeLwil8tNeWb1mmLP87yX
    53lvntEnZTIZawFJnRgopHqlANDQ0IDm5uZzGvM07VaSJGQyGcTjccTjcbV2N5+bA8CUadJO
    p1N1yGq5Jc5Ferlw4UIUFxdTIrdPMAxTC6BfB8/aCsWZ6waUDha0hGuq/UvXiMpPysORTqfP
    Wj9652g5Suig6+dyudSosjZtOh+NjY2Ug4Oe+UsAbMRFdmSYLiuKp97cWCwmpVIp3UVYU6mU
    HI1GZXIhx+aCASzLMq688kqwLNsiy/InAWDBggWYP3++Luc6QyxxMvH4uIuLi7F27VrcfPPN
    OF+deWNjI6655hq1NiMcDiMSiWB4eBgjIyNwuVxqrV91dTXMZjMAhTzhiSeeoGRBvCAIz8my
    vDydTv/b73//ezQ0NODWW2+F0+nE6tWrsWrVKvT29mJwcBCDg4MwmUyorq5GaWkpvF4vioqK
    4Ha7z6uUNzU1YeXKlejv78dDDz2EjRs3MqIolkKJIs/szWe14rrrrsPVV1+NuQRBEBCJRPDa
    a6/pwXB3QKk/stFodVVVFRYvXowVK1Zg0aJFqKioQHFxMXw+34w7rnK5HMLhMMbHxzE8PIyD
    Bw9i9+7d+PDDDzEwMECdgH5iwO2GkgpYSNRBSaeEyWRCQ0MDli5diiuvvBKXXnopysvLUVxc
    PK1lNdS5QBUZLajwz2azCIfDGB0dxfvvv4/29nYcOHAAJ06coGv4CShRfz0YwGkomS1zBXPG
    AX3s2DEAaOZ5/hK9MLdSA3gW5a8HM5xhMMXapgFsglIj/IVZvOcZKBlnM1lfdl6nrSiKCIfD
    mJiYKKgxzLIsEonER8nYQjugbQCW0f25cOFCVFZWqvc3bdtD2wJSTolIJIJEIgGe51XjN5+D
    g/7fNLKrJZ+k2Vu0fy1t4VNTU4N58+apDm6a9VdXV4fm5mZqAPuJPr9HB/LXD4WAkgOA+fPn
    o76+XtWxhoeH1VaA/f396O/vx+joKMLhsOo40K5f/hrSfZTPw2G1WmGz2UCzI4uKihAIBFTO
    DcplQtPX582bp+XgAJSIfwAKC3zBDWCZGsDDw8PS+Pg409jYqCuPdDAYFAYGBiTi8YjrYON9
    JB588EG0t7ejsbGxnrTfQHFxMbxe/ZF9zqAAjgP4I4ArJiYmWvfs2YMbb7xRNYTS6bTq0aOe
    Jdor1Gw2qzUMHyV8u7q68OSTT2Lz5s3Uc9XB8/wmKO2Nlg8ODnoefvhhCIKA1atXw+/3w+v1
    YvHixVi8ePEFeX9piiolNqKXAMMwGB4exv79+yGKogTgZQA7ZkuZmovQ0byroKQQweFw4Prr
    r8cdd9yBK664oiDn1Gw2q6Qfra2tWLVqFb761a/ijTfewIYNG9De3g5RFJ1QohzP6cBwaiVG
    MPx+P6699lrceuutWLBggZryWAhYLBYEAgEEAgHU1taiubkZfr8fo6Oj1ACuJnP/fzrYgxnM
    kXRigihIz2w944knnsAPf/hDNDQ01CSTSd3UHE1lAFPn1wzJ3z9BKfVZMIv/5kEAL0KpHf4s
    CMHRLGBW5e9UoBwcHo+n4Bwcp0+f1nsfYJoOy1J2/23btuH999/HsWPH0Nvbi7GxMcRisbNK
    BGbKaWCz2dSewDU1NWhtbUVLS4u2lMsJ4FKGYZ4DkChwFLie6i+A0lN5586dOHLkCI4cOYLe
    3l6MjIwgEomcVeI4U+tHW0iVlZWhuroa8+fPR1tbm1qGRAzrZgCNejGAJWIoJEKhkLOzszPz
    6U9/moWOosD9/f3c6OgoRwRvP+ZACwZBELB//37zPffc09rf328DgKKiIt32ZJvBPmEHAWwQ
    BOGhl156ybFs2TIEAgHs3bsXQ0NDKrGQ1WpFIBBAZWWlOsrKyuB0OlWmV5vNBlEU1ZT4kZER
    7NixA6+88gr27dsHnudlKEQYvwBwHEqEpwnAtzo7O71r167Fjh07cMMNN2DZsmVqC5X8z02n
    04hGoxgZGVEjxMPDwwiHFeJYk8kEt9uN+fPnY9myZfjDH/5Aow1HAfwBQGQ2nBbRaBThcPif
    IgoqtPInCAKSyaRe5usDSe1Lp9N49913kUgksG/fPixduhQNDQ3w+/3weDxwu90zzo9A9100
    GsXw8DBOnDihEnKdPHlSG2mpIHMvpAFM+8UnANjC4TCee+45vPXWW1iyZAkWLVqEpqYm1NbW
    ori4GB6PR62Vnm4HCG2zlEwmEYvFMDo6iu7ubnR3d+PYsWPo6OhQU8rpdUfWjzLPF9T/SIxK
    GXOD2yKugzX76EkqEQ526dKlrTzP2/V0B+brANQBPUOR0g4AGwA8hJknW5MBfADg5wBOEPnb
    COAekDYtM4wjRP5GC/V8k8kk3n33XRj4aGNJluVKANWyLEMURWzevBmbNm1CMpksSHYYZVyn
    5W379+9XAzOk/pee4UYAJQzDFNoAng/FiQ8AeOutt7Bz504kk8mLKZu4qPXjeR5DQ0MYGhrC
    oUOH8Oqrr8LhcKglYATFUCLXOy7GlpvOfLwOAGOZTKbhd7/7Hd/V1WVhWVY3wvjAgQMs8WSF
    yFx1j1AohF27dpkkSSoBwLEsi/7+fmzZskV3c5UkCe+9996kQz6dtjWALQA+FwwGr3nooYcg
    SRKCwSBkWRaIMsWQ/cxZLBY4nU44nU64XC74fL5JNZe5XA6hUAjBYBCDg4Po6+ujUZ0slLSU
    nwHYSQ7WOBHG4wC+Ozo6WvPCCy/g9ddfR319PQKBgPq5GhIujI+PIxaLqYQ6GkIkykDJADDb
    bDY2EAjQ/yVJFI2Ds6FEpVIpPPXUU9i6dSvmUGsPyLKM48ePF+SCPofC9AiAr0qStGxwcNAz
    ODjIvvbaaygqKlLbEzU0NKCurg5lZWXw+Xxq+yJK5Ebr2Gmdm7aeldbQ07SsXC6HVCqlpvaH
    w2H168HBQfT09KC/v1/9maYHqEQcK+9ByaoY14Hh9n/JvNYIgvDpSCTii0Qilg8//FBtZUHb
    MdTV1aG8vFx1KNA2UHT9aA2RttaIOhLz2y/wPI9YLIZwOIxYLIZoNIqJiQmMjIygr68PoVAI
    sVhMjRxQO5kYb0cA/A0KA60eNqEEhWk3Cf2TYfFQonq6d0BHo1Fs377dbLfba1mWtejp/kul
    UojFYuq9LYoiotHoTCn9VP6uAnDtFO+JGn3yYnTKDJROCD8jrzKUVkqPQ2l79D2QbJGLuG/o
    fFkovANaj2QSSlbMQRjQPRwOBwRB6GNZ9i+5XO6roijWxmIx3ZHjCIKAaDRKjXbRZrMNm83m
    nYlEIma1Wqe7Rdk/pQZCYWo/ASWbidNRv3AVoigiHo/n/3gMZzqv6MIA7oMSvWrYvXu3b+/e
    vbpTmomw6ATQMxcO+Pj4OLq7u5l0Om0HwEiShFdffRVvv/22PjUbnp+KOXi6nCC9AH4P4FMj
    IyMeAIehRGh7iHBkoRAhVGWz2UA2my0Oh8MlxFPkIO8zWqMiT4h/CCXl6kUAJzG5riUM4Ddk
    7/xPAJeHQiF3KBSaZFBqPlciinGMOFzoGIJC3R6HUmNUyfN8c19f3yehpFT+HcBWTEOD7wu4
    +JhcLjfnPc0cxxU60yQG4HkAb0NJK/4MgDZRFBuDwWBVMBi0nzx5kmlvb1frTmk2grYHLa0z
    0pJB0D6ElP2Zkk9QAop0Oo1UKqVmHNAIkGZvy1Ciq6ehpAodhVL3ewAK46kejJAJKGQWb0JJ
    ZVsChen9E9lstj4YDPqDwSDX1dWFvXv3qo4BjuPUEgKbzQaTyTRlrRF1ztF1o4OSiOWvW969
    IJHn269Zv4PEgdoPfUUxD0Fp/aZ3Azg0VwyMwcFB7N+/n0un0z5ZlnVD2pVIJPDLX/4SGzdu
    VPerLMvo6emhnQ9mAn1E/i6BUhPcQRTnHiIfGSp/oTDclpBRDCVqzJ3nns4Sef5/APwVQFfe
    +xEAvyPy91tQmJ89H+EQykGJ4gaJoy9I5O8AuRPNULJgmqCQ8NVD6cCwFXMgO8EAcNlll+HN
    N98cq66ufiwajb7KsuwXGIZZYTab22KxWHEmkzFPoevNuulhMpl4lmUHRFHsLCoqOuR2u1+v
    rKzsuOmmm+LPPPOM2rGhEHMjcjcIpcvKKnIevNAfSaFMHGSnoQSpXiLn9aLO6nQawCEoUbNW
    ACxhZ9UTGLKI7xDvge7B8zxEUZQymUxEkiQRAEuV3TkAhgii6dwHbwN4Eoqn9lUi0GhPXYYc
    WhMRuH4o9SFFRCDXQEmVqSbvWaB4ggfIgXqFCPRzGZ9pKG1P/gHgCii9EFuJIJZkWeaJUdGP
    M30Px4iCHyZCPEP+Jp0vbduwGIpn/R3MPDOvzDBMH8dxeyVJMs3VGmC6xziOCzIME6OGTqGc
    lJrn/p9EgFRDSd37JBEq5ZIk+TOZjDOTyVDmUjsms/de6MPQnikBZ9p7JclrhDhajhHHTi/5
    PjYLzpWPK9xGyNhJ1iWAM/VJ9eQMl8uy7Jdl2SVJkjOXyzkTiYSVrCFVsJkL/HsSeW6ZvPWL
    Qkm7HCHK/XGi/A+R93M6PQt9ZK71Oj+zXbjIuq3ZQjweRygUAs/zurokc7kcJWo8/+WoXO7T
    OfedAH4FxYH7GjkngsaRRuWvI0/+BnCmL281zpBMieTO3E0+r/M854snMroDSk0wlb8+8vfT
    UNoIauVviMjfCXKus+Rvyhp9wU4cb1+AEnU+PUu60ZwWvB9DZk073nzzTQDA9ddfn25ra9v3
    zDPPHGxtbS1NJBJNXV1dNRaL5bJgMNgWDAa9JpPJZzKZvKIoOnK5nFkUxY/tOJ8iiCJzHCdb
    LJYsy7JpQRCigiBMuFyuYVEU98my/IHX6+0Ph8N99fX1sT/96U+5Q4cO4Y477tDD88sC2Ael
    7OB/A2gg52EZlPTocnLGrLO4b2UyrxjRoXsAvE/m2ElkyLRwXkynASxB8eS3Q78tDmRyweXm
    ws0Sj8dRUVGRicfjzzmdzn6O48yiKMqSJMmiKEo6dDKcWWhZZiwWS5pl2cPT2BuapkOJmDp6
    RZvcp4nw04KD4vU1Q2EOtJHfp4zg4gXunwEAm6F4qp3EkBHIZ2TI17kLNPxFctDfIUrAbFhw
    Msuyz5vN5i34FwDHcTLDMKkCG8Da/ZHDmYh/BxRCFbrvPFCiIqVkFENpP2CdYljIKxUGmSle
    eaLYhaB4cenfjZN5ZDEHUk2nOBMJMrqg1PiYNMNN1o1GmFxkrbRDu4YMWYv89aPrk4QSIaJr
    FyXvCXnKvd4RJAr8p6DvFkO7MEcc0IlEAgzDZLPZ7AGHw7FMlmXLue51HZaQME6nM2wymaI0
    s2QaEAHwBDkT0jl0QIHcSxOYHMllNfegRSN/E+T3L1T+DkKJ0v6NGNpOzd/Mac7uhcrfHBQH
    +P5ZPOvjZG3mUuuycxm/oyhwL/Tf/va39Evh8OHDwy+88MLw7bffjvXr12998cUXrSzLuqur
    q6sZhpkXDof9Q0NDTp7niz0eT53NZitjGMYOwCbLskWSJLMgCKZcLsdls1kGACwWi2w2myWz
    2SyazWaR4ziR47isKIrJdDo9HIlEus1mc7CysjLl9/snJEka6unp6Xc6ndGamhqe4zgxm81i
    165d2LdvH9ra2vT4LHliXHZCiQpbAZRB6XYwj+grATLKyXsecp5pWzvt0La6kzRDzPs+R+6A
    INlLY+SVZmx0ktc0ZiAz41/FC/UvjXMJ1zkevTNgwMAZ5ZDNEyT0azlPYOQLEgPnXk/q5f+v
    sl5VRGHRswO6C0pGgu7R1taGI0eOoL6+3iXLcrHmPM4FAxgOh0Nqbm4OchyXtlgsuuQO+a+E
    O++8Exs2bMDy5cv9HR0d3lwuN+cVOI7jhLa2ttEPPvgg++ijj2Lt2rW6nu+9996LPXv2oKSk
    hDWbzU6GYayiKHLE4GV5nmdSqRSbSCSYWEwJMpJ+tTIdTqdTttvtEsMwAsdxfCAQSK5fv16M
    xWIF7VwwC6DOZRpA8uKMA1/rfNa+Mji3Az9LjN8ozgSQ6JCNG8OAAQMGDBgwYMCAAQMGDBgw
    YMCAAQMGDBgwYMCAAQMGDBgwYMCAAQMGDBgwYMCAAQMGDBgwYMCAAQMGDBgwYMCAAQMGDBgw
    YMCAAQMGDBgwYMCAAQMGDBgwYMCAAQMGDBgwYGCu4v8DcpGqQ14QukUAAAAASUVORK5CYII=
  }
  
}

################################################################################
#
################################################################################
if {$png_image_support} {
  lappend boardStyles Merida2
  set pieceImageData(Merida2,25) {
    iVBORw0KGgoAAAANSUhEUgAAASwAAAAZCAYAAACLvRPfAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh4R8YNGDAAAIABJREFUeNrtnXd8VHXe
    7z+nTG/JTCaZSe+NQEINvQcEKQKiKCiCbXV91qu7urq7PruPZVdRd1376ioCgiBNeu+BQCCQ
    TAohpPc2md5OvX8ksCCEJLj3ee69L3+v1/yTOcz5lffv8/u2cwB+bj+3n9vP7f+RRvw8Bf9v
    NlEUr62fDEAAgEgQPy/nz+3/b67o/g3CA0BOcVxFEst4QxUKYzmIqHYAuJvBiGI5AELndNoe
    IwnpBbVmaD7gYQEFALsBoF2AyBBE0L95MQIApPD5StMcDmuGTqcvUSgyLgMsCEI64N/juIsg
    CNrs9/nTGZYFTVOEQiGrJQiC8fkCySzLiRIJTSgUsgpRFKU+XyCW43hRKpFArpCXiSLXQtPD
    +tHvVgASZSDQnEWSlEMiSbsMtCmKik4/V1XZOCsxKfrIkCHZH4ii6GHZ8jRBYHUyWXghwHoJ
    wnyXc+VBRcUh6HT6sS6XZ1pMTNhHJEnYaXrov3E9eAAkIYo10SzjjJVINbUEEV8PCCJBUD+N
    K1J6Qa3+7+UqELic5nLZMjSaoBKZLP0ncxXwB9IFQQBBEIRMLqklQDD+AJMsCqJIkAQhl0kr
    RIjSgJ+NFUVRJEkSMrlswFxxbGsWCNJB06mXgVZFcXHeczXVzbPiEyKPZGSM+kAURQ/PX0kT
    BE4nkZj+R7mi+75BJQClpKhwy4sHDuQ9b7e7g5KTo0rmzp38YkNLVW5t1VnEJoweYLcDAIzy
    HTu2Pnv5cl1YTk722ilTpr7T2dkce+LEub9FRobtzc5+8M+ieIUhiJR/G1hWax4UCnnWpo2H
    NpSWVaempcWWL77f8bA/ECi8mHcGw8aMHdDvFZdcgVajzdmw4cAXbpcXNE1LFi6askqpkLVu
    3HjoPb8/wMvlUmrJkpkveX3+sG1bj73McRyr1iix5KEZT3o8nnV3OOUowGkCKDeg8p06tea/
    jh298IxarXTOnTfxeYokhC2bD75O05T00qWysQoFXS6VymTbth1e5XB4tFOmDP980qRlvxdF
    txzg1YC2FQDfnwNGFCtAECqIYnFKXp7lD8eO5k966un7WnRa3Waf77xDoRj5b9jk1SAICrW1
    Bx7Y8cOxN8rL66OSU6Ia5s+f/Fp8/KxNbMACiWzIT+CqNuyee8aunThx8judnc2xubnn/xYR
    Ebp35MjF/0e4UiqVWWvX7t5w6mRh6rjxmeXLls162Ofz3T1XWl3O2m92f9HY2IGgYI3k4Ydn
    rlIq5a1r1+x5r73dxoeGBlPLl9/7kscbCNuwYf/LdpuLjYw04pFHZz/pHiBXe3affkYXpHYu
    Xjz9eYmEEj75eNPrJ05ckk6cOHTsK69KyymqRPbdhv2rurqc2tn3ju3hyiMHuP92rvoUrM6O
    RkilHRN27zn1u6jIUO2MmaOwZcuxkfsPnPnL0qWPz3W7a+wDX2IFgFDB6fRxDodTt2f3yf9o
    b+vKlkppxZ7duYNNJn26ViMPxMXHHxdF21UgqPNurbkb2/FjFxAZZZp6+XJN2rJlM7Fh/cG0
    0rLqqQ31rYVJyeMH/HttbV3w+3iqrc0qW758Ntat24+8PMtchULma23toJ94cj79+ec/4Exe
    0VKfLyB3uT3kI4/cI1uzZi/a2qyU1+u57aICkBQXb//1qZOXngzWa2oXLJi6/uSJi48mp0Sp
    6mpbVadOFiwblT3otMfjkz64JAebNh6S+H3++Lw8y4SAnzGnpUbjxImLj2ZnDyndtevk0ra2
    rrixYwd/OWzYnPcZppCVSrP63PiiKMpPnVrz961bj84sKamGXCH9/L77JpEdHbYvLl48g2HD
    xv6kteB5F0SxJuHDD79485NPtib6fH4ojsiTOJZ/UxRLLng9nqq74ypMcDi83MbvDunO55f9
    x+NPdGRLJbTirbdWDx4xIi1dq1EEYmJj/+1cJSRETd2y+WhaaWkNWtusadmj0qdWVjbcNVei
    QFK5uUWyy5frEKzXIC09dq5SIfMdPHiWbm3tok0mPQYPiV/q9THy/fvzSFuXS5aaFoM588ZT
    Npu9V65KSn749ckTF58M1mtrFy2aun7//rxH16/fr1KrFapws2HZmDEZp6urm6ROpxfV1U2S
    QCAQX3DBMmHNmj1ml8sLksSjY0Znle7ceWJpa6s1btz4IV8OHTr7v42rPgWrpLQSwcFBgz1u
    n3b+fVOg0+nQ0eHE6dzCBKDDIJcTdyFYEgCUzWgMqkhJiU2fMWMU3nj961Ecx+M//7gSe/ac
    Ub311jdvxsaafXFxEeXZowd/k5ExZiPHFXdSVMZdA5aUHAedVtMulUr47zYcomiaZkJCgurk
    MuldbUCCICBChEqlQGpKIqZNHYWyy9WD3G4fJk8ZgcwhaZgypRGNjW0jRFHEtKmjkJqSCJVK
    AYgicbtx+P1OSCSWIbt3nXxJrVHoCwuvxIeF6iWGkCDvlSv1cLt9SEiMtGVkpO9LTbU88s7b
    6zJmzBhVOnhI6oGa2tbMkuIqMAyLkJAgV/45y9IL50unRceYsGf3qZdSUqIP8zx/oe+RUQAQ
    SEiIWT1p4rBQnuMHT5ky8hu1SnWCpmh0tP90q8Tt9oCifPFFRVejfb4AaJqGzxeAxVIZ3dnZ
    ES+KYtXdcUXaIiKMFRqNMr28vA6ffLx5lEajRGenA8eOFaj8/sCbUdEmX1xs+L+VK70+qD0u
    PoKvqKinYmLMTLBeW5eUFH3XXIEgQJIECAJwOT14b9X6QQQBeDx+kCSJjg4H3nh99QhRBLxe
    HwgC3deD6JUrmaxkyPebDr60bt0+vTE0OD483CAxmQxetVqBoCANNFqVLTklZd999016xO9n
    M+bPn1Campp0oKqqOTMoWANBFBEWZnCdP1+89K9//XZaS4sVDY3TXkpLiznMMOx/C1d0XwG4
    xsbDoGl5I0VTXHl5NZ2dnY7y8hpERITagCCPwPsHvCCCwIAkRaVEQssLLlxGclIk4uLMkMmk
    SEqKxuCMNgiCQE6cmKU6cvjC8I8+3Dh0ytTKeffdN+M3tTXrLaLoB0HIB+iC2AHo0NaWq5ZI
    aaGqqpFKTokRlUp1UFLSXIhiDgjCMOAAJUkQsNtdOHnqPNQaOUaOTLv+/clTF2AyBcNkCr4O
    4slT52G3uUCQ5DUT/Ue/SYDnRQXL8VKNRgmaptDRaWtZvDjns6+/3vFua6vVPHnSsL0bN24p
    zspK3X38+KWMQRkJ+z79dF3hgw/O27t//5mHDAZd3eLF0393KrdoNkWR0GiUaG21Sniel4mC
    2I+RaQFAlMnknuaWDqW1y0F1WR2yoUOHepiAD/fOEX6yYCmVGkgkivbY2HAnTVMhoiiCpinE
    xpmdISH6dp/vJ3BFU3KKImE2G7B8+Wxs23YcAMAwLBISIsip00aoTpy49G/lqrPzjNpg0AoE
    QVIhIUGiQqEKioubfddccRwHg0GHlSvuhVIp/xcrN4pRz98IgoDX58fVq43gOL5XrgRBVAQY
    VspxPFiGRWenvWXp0lmfsSz3blVVo3nihKy9mzZuKR4xMmP3mTPFGVlZKfs++2xd4QMPzN07
    bFjKQ7Exprply2b97kxe8WyG4cBxPAIBVsLzwoC5amnpUFq7nAPmir7TpAUCRYiMTNSfOZOb
    43Z5ydO5FiQkROD8+TKMGjVIX3m1cEZi0j3rRbGNJ4iwfi5GFVat+hvmzs15NO9MUU5aeixM
    ZgN+9/sVoGkSUqkEWUOTEZ8QgZSUeGRnD0ZenoVc/fXuHKfDvWbFygWPtbXlFomiAwSh62fW
    owWATlZ+eddTm74/8Eez2SB54YWH8N13B2Rff73tnQcWe1QpqXP+IYqNASCi3ydteHgYdFqt
    ByDYP/95jeTee8dCp1NDhHibdCwBh8ONvXvOID09PmAKM7jc7ls3h0KhBBBTMH3ayE+OHj3/
    pMPu1qelxpYZDOM3p6WVPnbs6IWI4uKqoUuWvLZ5xw8fxXd22tHc3Bn37LPvUCdOrMmsrWnB
    pInDTxtDJ29KS3On5J66hPr6VnvO9FFfabXJF1mm8aa5aWsdhdCwzeb6+opsW5czSSaXOp2O
    5vNanchv2XLgA5fTkzBv3gTs33/mEYqmbJMnL/nLoUP5I1tavk8jCLDR0VEWvX5QHuDzAqHo
    XyyjGoBZVlqyfxZNEQq5XAqfLwCZTAKJhFZYLFX3DhmSc1UUa70EEdt/rt79G+6bf8+j3313
    IMdqdSAuLhyHDuejuroZJNndr7178yCKIhbdPwUOh4f8+t/A1dWr+576+qvtf9y2/bgEELHj
    hxMyiiLfeXylV5WQOOuuuFKr1J7oGBNrMhskqakx4HnhtkJEEAQoikR5eR38fjZgMOhcCoW0
    F66iC+bPn/SJQi59sr3dpk9Kii7T68dvTk4ufezQwXMRxcVVQx9c8ofNP2z/ML6mphmNjW1x
    zzzzNnXixJrMqspGTBifdTrEOGlTcpIrZcKETOh0avvUaSO+UqniL7JM04C4sjvcCbNmjb6J
    q8OH80c2N1/jKtqi16ffwtUdLKwrkMkyJYcP//ONPXtOPTV+QiYxYUIm9HodXvnto9h/4Kxp
    w4Y9f398JW23O2w7D+07j5xZ/QnGCnj55Y/otWvfnhEeYaRm3TMGn3++DY88cg+GDR8EgIfZ
    HAKzORQAB4oiMX78MOj1Wrz99rosvUH3l/vvX/4wz9X10xWtBRChOpv33Zvbtx/5RVZWknzh
    oqkARKxYMRfbtx83rP5mx6q5c+1R48YteQ2o6PfRnjEoHQQRvO+ZXyz808ZNh37j9fiDBw9O
    7BXM/HOlmDBxqP2BxdPfjY9P38exttsAmAZR7PKNHffIHzKz0i7+9rcfrD146Nx9QzJnrBcF
    DnKFFKWl1bOmTHWs7eiwJSYmRsLaaY8HulJKSypnKpUyAIIU8MUePHhmIcty/hdfeORppWrk
    dqCNlcqG9UDlwmuvDcLzz381/auv1q4qLa0aLJXSdPcJKLVlDEpoLympjH3++SVITIyDzxtA
    Xp5laVlpVU5jY3s8x/MyjuMhl8t8UyaP+D5nxrRXmEBza38yeQQRj+Libb947711fzp1slAm
    CCIIgoDfz+Cb1btV1VVNf/r1r8XA4CGL3vd4LkKlGtY/rl76iN6w4d0ZJSXV1PLlsxEXH453
    3l6HQIDtdt9FEc3Nnfjqq904eDAfL7+8DC+++BBWvXO3XIWr8vM3vfnFF1t/cfxYgVwQRAQF
    aeByevD9psOG9nbbqieesEWNHr34rrh6aInzT2vX7vnNxo2Hg+fNmwBjaPBNokUQBDo6bNi5
    IxcjR6bZly2b9W5iYsaduRq77A9ZWakXX375g7WHDp27b8iQnPUCz6G5uRMFBeWzpkx1r21q
    7kjs7LSjqbkzHnCkFBSUzWxu7oQo9HB1KG8hx/H+l19e8bRSNeKOXJWVVQ2Wy2U0AFEqoW3p
    g+LbL5dVx06dOgJV1Y2Ij4/A2bPFS0t7uOI5XsayHGRymW/a1JG3cNWrYDGMD1Lp5eRzZ4sX
    zZiRTcyaNQEAD0BE+qBEJCZF491Va4PO5Zc8tnDR/9oXYS5g+2vuEgSjt9tc8UFBGnz2+TY0
    N3cgOsYEQABAgGEYVFc3IykpChRFAuCRnp6IRYumYNvW4zNSUmKW7vjh4Cd+fynk8kF3vN/5
    83kICdHP27btyLMTJgyRzp03CS6XB19/tRMrH5+HBx+cAaVSLtu+/eizZnNwXleXY1t/wSLJ
    dIhipysza/HbYWHGqlWrvvnHiROXdEOGJEC4wUQmSQIWSxV4jne+9PJjz5rNkzcBXYJEGtdL
    DEMPjiviVKogS5BO4zh92pI1ffqhpwmCoAYPTkBHhy2lrfXS/VarwzR0WDLa222m1lbLg21t
    1qSsrGSQJCkpLNz7+MkTFzNTUmPalCpVUcB/gZXfkIXxeMrxxhtbkz/9ZMOHdfXNaU89OQ8R
    kaFgGJaorW3RHzlyQW8puoqW5nYkJkahtq4VlqKrhgcfnG6YeU82QkKCwPMCzp+/rNi16+Ry
    iYTyTJ6y8vmAr5CTKbLuKCyiyOg//+zNR07nWq6L1bUNKAgicnOLJCNGpj0kivbVHFPZNRCu
    Wlut8ZmZSXj8iXn49tv98PsZkD3ut0wmRWJSJCqu1KOxsR1vvvUNvvziVSxePA2bNx8dMFdh
    YcZ5//xy27OHDuZLCQIIDtbgl8/dj08+2QpblxOHDubLCOBZs1mf19HRNWCusoYufttsNlb9
    9a/f/sPvZ3T6YC0EQbjhOhINDW0YPSbD+eILy5419ZMrpTLYolYpHAf2n82aPv3w0xRFUizD
    4cKFyym2rqL7GxvaTF5vAE2N7aauruIHz+eXJbEsB1pCSyxF+x7fszs3MzMzsU+u6hta0h58
    cDr8fgYet48gCOiLS6r1RUVXMSQzETRFobGxHRZLpWHBgkmGsWMzQJIkQkKC0N5mU+zYeeIW
    rnoVLJ/PDZfLofd4/dqkxOgeMbm2CXlIpTJERZvR3t6lB0Bpg0m2b6hs+G7DOxgxYsLMlpb2
    pImTpmPw4HhERYXBYAjq+X0CXVYnvv12P373u+VQKhU9fxcwc2Y2Dh44Rx08kLfyD6+9tpVh
    2lrvfL+rABLJbVs/mK1Wy6X3zhmP9nYr1q3dh4MHz4FlOSx75B7Mnj0Oly5VKIsKr85fsPB/
    bfe4UkSVZkQ/A6QhEMUywWROP2MyGe2DMuJ09947qUfc/xVs3LPnBEqKq21mc1Iuz1sEms68
    c3iSkgAwd4aHh3S6XO7Qo0fzVwzOSOTMZgP8PkZuKa54MhBgQgZnJGD/gbMhxcVXVup0arlO
    p4JWqx559Mi5MVFRoYgIN1qB0C6Sqr8pYzRkyAx8+eWfHyksupL26qvLERcX2bPGQEiIAUOH
    piIsTI/jJy7BHB6C9vYu/OXtZ5GaGt+zHt0szJw5DjabC0eOnF88bnzWFwEmUHTnGWMAtKpa
    WjpCWJYDSZK3uDgsy6GttSsEsKsoCdnVH642bXwX2aMnzbxYcDmpsakD//Hc+6isarwuhqIo
    Qm/QYuXKOfivP3117R74Zs0evP76U9i3P2/AXO3a+fHs8+fLpAAQG2vG8sdmY9KkYaBpCmvX
    7ENVVSMKCq4oiy2V8+fMfe6uuAozDToTFKS1f/HFdh1JEDcFGwgAgijiySfvs5nMyQPgKqLT
    GBrcWVXVGLpz5/EV48dlciJElJfXyi3FFU+2tXeFCIKIjk57SHFxxcrL5bVyURQRFKQZuWvX
    iTGVlY2YNn2EFTB1kVTNbbmyWCrSVqycg29W78Hp0xYIgoC4uHCsfHwujMuDUF/XinnzJ+DT
    T7fhhReWIPe0BV9+uQMOhwc6nRrP/nIhxo8b0s3VuH9xRfY2MJVKDYPBZNVqlPbSsircXBRP
    wO/3oq62GeZwYysA1uMR+jwBHY7LeOjh5an7D5x+JSk5SpqVlYLhw9MRGqrviSUSAChYiqsQ
    HKyBUim/QSRFyOUKzLxnNAoKyjMrKi5OvXSpDE315+5wVw8A0B0dNpPJFAKSpKFSyTF4SAJC
    QoIwZEgi1GoFKEoCsykEHZ32cAAShZIcaLj3BkEXe8Tqxx8RIgQAPASB72fGS+VWKGVN6YPi
    4HF7DUWWK2ESiRQjRqYjN7cwiuN5aWycGYIgSk6euBQ5fEQapFIpSkurwp1OT+iQIYlQKGTt
    QIiHYcSb5sViqUFFRW2sWqVAVFTY9T5e6z9FkXjggRyIoog1a/Zi1qwxPWLF/2isAjIy4tHW
    3hXc2Nga3WntS1+kAMzuqGhTm0wmuSUu020JSRARaWwBjK6b+907Vw8uWZ66Y8fxVy5cKJfW
    1jSjqKgSPm/gBsEChg1NRmeHHW63DwABkiRx6mQhGhvbMHvW2AFz1dZuNfl8AZBEd3zy0qUK
    dFoduHSpAg6HGwRBwuv1o7296ydxdW2OfjwT4k3xNGEAXMndBoOuiaZp7N1zxlBcXBlGEATs
    dg/278+Lam7ulJIkgZbmTsn+fXmRDrsbBEHgcnlN+J69Z0IpioTBoGsHgnvlSqtVobPDjoMH
    z8Hl8sLj8aOoqBJvvP41TCYDaAmNbdtOYMqUYTh+4iK++ucutLRY4fMF0NjYjq1bjyMuPgKd
    VntwQ2PLda56nUGaVgJIKZ88ZeTGkycKsXv3ieti5XS68fe/b4JcLrNmZw/5qqpqN+9x9TVZ
    VdDpxij27Dn2h6bG9vQFC6bcsCgiABIMwyI39wI++3QrMjOTr6VBb1qikSPT4PczVGFhxczs
    7GWkNpjqoy4HvN4Q5Gxt7QTPc1CpVJg0aQSeeGI+Jk0aAbVaDZ5n0dLaCa1G1QGA4xgR//NN
    BIBAcJC2w+sNYNbscTh2tAAlJZUoK6vG2bwSXLp0BR9++D2KLZU4l1+GxsY2XL1aj8OH8zF7
    9jj4Ayy0WlUrAJ+UIm8SDYIgoNNp/LwggOM43PqUlgilUoHMzCSUldVgzOjB1y2wH6cTfL4A
    pBKaVSrkHqVC3qdgffzxq7apU7O/njFzNENR/8qWiqIIiiKRk5PNTJ404p+fffqWnWGEfnF1
    4MDxP2z+/ki619ud9qco8ibrCgD8AQabtxwFz/PXywCsVgdO51owctTAuQox6J0ymRQgAJvN
    hQP7z2HDt/uxf/9ZWK0OEAQgk0mg06n/r+PKaAzuUCrlaG21YsN3B8Fx3XOyd88Z1NW2gKa7
    3bU9e/MAABzH47sNB9HS3AmlUo4QQ9AdueI4HlqdGlqtqsddJ0DTFDo7Hdiy5RgSEyJRWloN
    hUKOH7afBEl2HyAE0V3KYTYbIAoCpBIJq1T+iyuy9xQxCcCu1mhUnEwm4U/nWsDzPAACbW1d
    uFhwBSaTIUAQUn1CwhzCFH7nymFrZxMcjrwR585a5o0eMwhNTW0Qxe6YlSgC+fkl+PNbq7Fv
    bx6r1apuc550i1toaDAiI0NRcaV+GNBhkNyxMCMZR4/+gx8+fNBOj8fv37XrFDweL/x+H7KG
    JsIf8MHj8WL37lNwOT2+zMzE7ceOfSlIFcMHWjlzw4YneoT2xx8CxPXr+pMtCsITj88XDSFB
    tZcuVqCk+CpoCY1iSxW6upyIiDAiIT4CGRnd1mKQTo2rVxtw+rQFcrkMjU1tuFhQjmC9pnbe
    /Ilil/XGA8WAs3lrMDgj8ZTAC3zemeIeFMhb+hwba0ZsjBlBwdoffU/2HDIMDh08h4TEyMow
    U0JZUFBfGTY1nnvuXalUKiGTEiMZuVx207dyuRSJSZGMQqlUP/PsWzKNJqpPrjye/BF79+TO
    a2/vQnR02E1CJQgC9Hot5s4dx/p8DCqu1N/khgqCiAsXLkOrVQ2Yq6HD0naOyh7kvxYfM5n0
    KCurhSnMAJlMClEUMWJkqm/wkJ/GFUEAPC+A4/hbPgIv3EBT/7kKDtbVqjUKCIKApsYO+P0M
    BEGE0+kBw3Qnu1iWg8vpgSCI8PsZNDS0QxAEqFRyBAWr78gVy3J8IMDgV88/AIVCBpblIAgC
    KIpETU0zVCo5YmPMaGvrgtvtA0EQ4HkeLMshMzMJz/xiAU6ftiAuPrzSZEq8ztVtl0Xgq0GQ
    Jt3Roxs/2L3rxCMjR6ZRM2Zmg6JoAAISEiLw+htPYsuWY+Grv9786cNLPdLo6OnrfP4KKOTJ
    t52msrIqGAzGYQzLadrabHC7fMjISIIg8PjhhxM4c9riHzVq0M6xY4c43lm17nGjMYi8nWhR
    lARx8eGouFIfzjD14RzHdtyxsFOsBJCwccGCjugffjj+yt69Z5TBwZrrqm+zuaAP1noXLpr6
    Tlr6rJ1p6bUDrpnx+c5BoZCH+nwB5eXLNQgxaMHfEBylSBKXL9fA5/MrAU9oIOBuuHb/3vse
    ioKCDQgO1lcRBCHW1DQTC+6bhKHDkhETY4LD7oYgipBKJBg1Kh0QRQTrtZg1awwKL1XgfH4Z
    vF6/GBERVv3Kyythihx5w2+bIAhlIIiwXTk5V3dv2HBwvggRKSkxqK5qQkuLFTabEwzDoa29
    C/UNbfjoo02QSiXQaVUICdEhPMIIozEImzYeRleX0/3koql/V8hjW61dhXeYq1YAYXRJyc7f
    fvrpplcPH8pXsCx3U9Dd4/Hj00+2qiuvNr7zi2dcuoyM+/4iCBUsSfbOldlsHlZR0aCZN38C
    IiOM+PDDzdettbFjB/sXLpyyM3t0huO3L3/0+I8PaYIgUFvbCr+fQUJCBMrL6wbAVfzGJ5/s
    iKYp8hWbzaV84cUloGkKHMfjw79/D41W5V25Yu47yck5O5OT6+6aK47jlVOnjUBct/t/UzKn
    pqYFHM8rAXcoE/D0myuDPqQqLFQvBvwMMXhwAuLizDCbQ6AL0kCplIGmSLAcD4/HD4fDjbbW
    LtTUNKO4uAqhocGi2WS8M1czru7etPHQ/IULJ+O3v12GmtoWREWGQa1RQKdVQalSQCqlMXr0
    ILz33nPw+Rl0WR1ob7dhxsxsHDp8Hi2tVvcTT8z/u1wW29plK7y9YIliHYBoFBVtf2rv3lOP
    Llg4iZwwYdhNgVaSJJGSEodf/zocX/xjm37r1sN/efrp4DKv11fQa21JRCgktKyVIAg+IyOe
    Sk2Nw8mTF+H3B1BQUO565NE5v8/MnHzgiy8+Xq1QSMmUlJherCwgNDQYF85f1nR22sIFQSy6
    c/AyEaJ4NTB6zD3rLJbK5Uk0kTBzZjYEQQRJEjhw4CwYhmsYM2bGV0BVgCAyBgBVB9zuAqjV
    xsRt23a/b+2yGxXKUJzJs9x6rSDAZnMaN2/e+/7ixTOfcDnzKl1WCzSG3p+Xy8hIhVSqP5Sd
    PchSV9ecaXe4sHXLMQQCLEiSAEVR6C6KJsALAgRegEQqgcGghSgCY8cOLk5NTd3n894+ze12
    59rmzb/3V4IocmvX7FsAgBw0KB5mswExsWaoVHLIZFKQJAGG4RAIMHC7fWht68KFC+WwWCqR
    khrTuWLF/D+lpM5f4/PXgSBiepmrRpzO3YXExJR2cxK6AAARF0lEQVQ5q1f/8NK+vWcU16u6
    bxN037PntEwmk7z8619rixsam7db287CEDb6tlwplYrWoCA1X1PTTJUUV2PmzGycOlWICROy
    XL/61ZLfp6WPP7B69eerCwuvkrcG+dETY/EhNCwY+fllA+Jq1Kh71l24cHn5+m/3Jax6Zx14
    vtuKaG+344EHpzeMys75Cqi+S65CE7//fvv7+edLjYsXT0VsbHiPV3KtDyTi4lqw+fsjxu82
    7Hr/oYfvHQBXhkMTJmZZOtptmQ8vnQGr1QmH3QWPx4/ODjt4QQBJkpBJJTCZDEhNjcHCRZOx
    dcsxqDXK4uSU1H1RUV29czXv3l8JvMBt2XJswaBB8aTJZEBVdRPa223weHywdbnQ0WFHZVUT
    1CoFtDoVzCYDUlJisHNnLiQSunPFY/P+lJo6f40/8C+ubhEslu2CRMLFHz6c9/TgwQk9YiXc
    1j2Ty2V4dPm9ePsvayNO5158NmfGk0/Zu/L5IP2oW66OiYkCTQflpqbEFu7ZfXp4bm4R/L4A
    AGD6tFEfZ2ZO3Lp+/ZeflRRfHfvL5xZDLpf3cl9AJpPC4/GTLpdHzrJsH4vfiDXfrML0nPnT
    zuWXxMrlUtRUN4HnRVAUAbvdDY/HHz9zxoWpp8/lrnNYL0Fn6O8bCXxQq4erjx5b89c1a3ZN
    mjZtBFJToiHcpsCPJAgoFDJ8++3eSSqV4vXZs59Z7nJdvGPnpVITPvro7YapU6dteO+9NZlj
    xgzG5EnDgB4XgecFiBC7nVCKgoSmIALdqeKiSjz08Mz1b7zxXuNrf3i5FytBBMta6u+775dP
    +f2M4sqV2tkvvLgEgQADj8cPr9cPhuEgij3wyiRQKuVQq5Vobu5Ex9828ssfnfdaUvLszwFn
    r2LV3VwYN/4Jzfr1q547eOCs5nZi9eO+HTx4TjVsWOqvli775ZGA/4rzdtd2cxWcO3NmduGH
    H34/XKGQQa3uTiQsXTbr47T0cVs3bfz6s9Vf7xzrdHpuyUoCBDiu2xWRyWQD4KoGhZfOIihY
    P7GxoTX2N795GObwENAUBV4Q0NpqRf65svjy8jPjPR7fJjZQAIls+MC4OvrNX0+evDhp8qRh
    cDm9sBRdve1cTZ48DGfOFE3S6VSvz7732X5x9fHHbzfMmJGz4b/+6/PMV1/5FMF6LYKDNNDq
    VFAoZKApCoIgwOfvPqTsdhc6O+yQy6X4/e9Xrn/zjXcb/9AXVwuee4rnRcWbb349W6tTISI8
    BMbQYEREGKFUyEDTFFiWg8frh8PuxsWLV7BlyzHExpr5995/4bW4uJm3cHWLYBUWliAoSD+t
    va0rfsmSnD6zGEFBOozKTkdpadXknBk1kRIpUXf7IH48du18rWnO3AefU6nOvFxb15zBMFxi
    QkKEZcrUSdu3bPn2byUlVfP+41cPICkpplex+pewchTD8GF2u7PPPi5/7O+KosKtC+02F6XX
    axFiDEZwkBo2uxsdHXY4XR6Jtcux+JGH39ji85T6+m+4kwCgqq5sSgkLDYZKKUdtbUvvoVqF
    DOHmEJSVVhtnzwYpkVJ9pLbDYbefBkFQ+cFBGm9ubpHyUmFFD3SS7sAySUAURPAc3y0uEOH1
    +KHRKL2REaHnnn12GQgy6Y7C8NRT93WtWPFA8d69p2e/9dZq8Hy3QElo6voG53gePNcdoCdI
    Eg67C1KZ1B0VbTzvdp+FRjPmjmNxux2QyQqz8s+VZDudHlBUX2Mn4HR6kH++dNii+y+nMwx7
    tjeu9u7+Y9O8+Q88pwtSv1xT05xRcKE8cc6c8Zbx48dv375t/d8+/3z7vLq6tp6avl4D0QPk
    yo+soUvI/HMbpttsLurY8YtoabGCYVhIJTTM4SHgWF5i7XTMGjf+0c0MUygMlKuqqqYUgiCQ
    mBh5kyt4y9UUgcbGdpSW1Rhn3zswrswmg3fipKHKtLSYbkvT7UOAYdHzWhtIpRJo1EpotCpU
    Xm3A4SMXvCaz4dwzA+Bq0KC42b98bhG0WhXcLi88Nx6EBAmZXAqNRgmtVoVjxwpQWlLjjgg3
    3JYr+sf1LHPunYtf/WpZglIpI8LCgvsUDkBESnI0LEWVIR5Pm1Hs9ilvMwjNtWzN2fsXj17i
    91eaGurLs8PCjGUFBRfvOX++dPHKlXORlBT7oxqm27ceC0PXZXXd0P8mdHRsg9E4M7KysnJ2
    TXVD0o4d2whjiIE1h4e0/P73Kyp+2HEiubSkCg6HBxqtEhHhRjz9i0WVOp2q+ejRb15wOd3B
    Bw58jri46KtJSUl729uPNIaGzgVBRNx0//r6w3C7mnI6rWULPR63MSMjHvPnTwDH9z5fNEUi
    EGDgcvniKyp2/zE8PPR4ff3hg9HR03uvWmIYEARJEgRBTJiYhaSkKHi9AbAMC5bjIPAiCJKA
    hKYglUmgUMhQU9OC48cukizH030/DdKAL7/cgSeeeEiu12swd+4EqJRyMCwHjuPA8wIgdm9p
    miYhk0khkdC4dKkCFwuuEAIv0rd7ZOSWXF5VPVQqZUxdfat6ILGcxoZ2TXNze5zfz5zti6sF
    C7KXBPxVppaWq9lGo6HMYim859tv9y2urm7qUyCvuYe9cdXZuR0hITMi6+pqZ9fWNCTt3XuE
    CAu1sOERoczTTy+s2Lb9aLLd7oLP64dCq0KQTo2FC6dW6g1a/+nTa19xOFzBx459hdjYqKux
    sXF7OzqONhqNc27LlcvVlGPruryQZQLG7uLdTtxpjgmCQGenHZFRofFVlXv/GGYK6RdXNE2T
    Xm+A2LTxEIL1WkglNBQKGSRSujvMwAlgGBaBAAuW5WC3u6DRqkiIoDme7xdXTz+9VG6zOfHF
    Fzsgl0shl0khl0tBS2gQBMCxPBiGhT/AgOcENDV3IibGRIgibsvVLRbWnr25WLFyIUVL6J4g
    O9HnaSBXyMGyHPx+PyR3SK/c4AIwAOpFsaQeMIfmndm5bFB6PJGSEne92v1OmRMC3QouCgJY
    lrvhOweMxgdD9+/buj7/fOnEuFgzaJpCfnUxnE4PD0Awmw3IGBSHtWv3YdH9U2C3ubB718lY
    EXhCo1FSZpMBTqcHZ8+WYMSI9FOzZ993v8Bbb3mOfPXqH5CTM2r8hvX7f3HNCvngg019bgqX
    ywsA8R/+/btXly67hz10KP9g725HA15//Q3MmjVlVG1ti8Lr9ePokQLIFVIoFTLI5FJQFAWB
    FxBgWPi8fvj8DBiGRWuLVV5d1Tji9JmCI21N+QiLGNWr+yGKomL9+ncG11Q349tv94NhOMhl
    EsjkUtA0BZIgwQs8OJaHzx+47o5aOx3aq1frh7S3W8/1NW5rlw0AqO4wQH/fikDAH2AIr9dP
    ut3eAXIVHnrixA/LLJZK4lY3sLcsb+9chYQ8EHpg/7b1X6/eNbGhvhUsy3dny4LUvFwuE5qa
    OhAZaQTDcmA5HseOFaCioiHW7w88YbO5KK83AJqmEB0dhsdWzD01a9b8XrmaOTN7/Ad/2/SL
    K1fqsGRJDhYsnNyToe+tGJTCzp0nsWH9wfjz+ZdffeHFJeyBA+f65OqeeyaP8nr9imeeXYTo
    6DAwDAeGYcCyfHdGjyRBXxMxCYWGhnZ88Y8d8vIrtSPy8i71i6t1694erAvS4IUXlkCtVvTc
    gwXHct0vD6AoSKXd96AoCgUF5di08bC2/ErtkPa2W7m6RV3+87WnYTTqXadPF8FiqYBMLu0t
    9n3dHC0tqYZEQgsqlYZnWRZlxWVIH5ze9+Z1OgHCmVBd3ZQ4Z+54OF1OCLzQR6UuBbvdCZ7n
    u9MAN6mwAMDoYli2ze32AkT3E/pRUaGQyaUUSZKU0+nB6TMWfuGiyW3nzpWGjRyZRkVFh9GC
    ICDgZ+Fye0HTFDweH/z+QBsQ5ua4Wwufm1ussDvckEgkeHjpDFAUhf5YGtfStxvWH4Td4UZz
    i/WOBYr/+Z//II8c/udwny+AiAgjpk0bgajoMGi1KkilEpAkAUEQEAiwcDo9aGxsx4kTl1BT
    3Ywum2P4Sy99QLmcF/je4WVAEE3hLc2dyT4/gyFDEjFqZDpMZj0UCjkkErrnkZnu09Zud6Ou
    tgWHDuWjoqIebe3WsTNmPP2lzz0KCnXvz/zFREdAJpO6tVoVz3E81Z+54nkBapWC12o1foWi
    /29RcDmdoGhPQrGlMtHnC/RYV2Kv2ThBEOByeWC33ZkrjuPaamua4Xb7QFEk7HYXamqaKZ4X
    KJqmoFDI+KVLZ7Rt3HQ4rL6+naqpaaEpqvuBfrlCBtEnoKamGYEA0waYeuXK7fGhqroJbW02
    fPvtfvzww8k+LSy324uuLicomoLb4+8XV4cP/3O42+3Ft+v2oaGhHTabCz5f4PobHwiCgERC
    Q6mUIShYg4hwI+wOF+w2V7+5am21Jne0d+Fvf92IhoY2OBweBAIMOI6/numUSCVQKeUwGHQI
    1mvQaXWgra3rtlzRNw88GPX1ByGVSgsiwkOdG9YflNM0dUe0iB6/Pzs746pcbmzguf6/LCnA
    MgAIqVwho7ZsOYp9+8706991dtgRHx8BhUKGoKAbPYxB6OzM9U2ckP2+IIiDvB5fOUVSnq4u
    V4zXFzA7HG6NhKZ8c+6dsD1nxsS1hw/nrjxzpmieIAhynVbtUirlrWqNopYkSFV2dkbKxAnD
    3m1tPeaVUwm39GHY0GRERhh5jUbJb9t6nBvIu5REUYRGo6Qjwo380KzkPqqSIaSlxX35+ONz
    pVVVTZn79uUZeV6QUXSPjyOK3TX0vAieF3iKphiNWtG+YsWc0kGDEjYCEHihrxXkrOMnDF1H
    kMSCioqGiPLyehVNkyRNUaBo8tptwDIsGIaDIAicXCFzPrRkRkV8fOSB7geXC+445qiocEil
    0rLp00c16rTqcIIk+ozpiIJIjhiZ1mQ2hZQwDDsgrgiWlAbrNVR6ehxIirhjHaVEQuOf/9wB
    p9N7R67GjR/1/qu/4wf5/YFykiQ8dpsrxtrlNHdZHRqtTu0bN3bI9qnTxq+Njglfefz4hXlu
    l0+u12tdeoOuVa/X1AKESiKhU8aPH/pua+vRXrkKNer5qVOG81arg7s29/1xZwHAYNDRocag
    fnGVnhb35cMPz5RaiiszRRFGlUoh8/sZKhBgwPdYWDKZFDKZlA/WaxiTSd8+ffrI0kEZif3m
    auKEoes4lltQcbU+QqmSq3hBICUS6vrbJ0iSBE1TUChlkEgoTqNWOh9aklORkBBxW66IWzfT
    ZQCE0uN2DPf5/TLc0b7q/pYkSUImk3eyfnkRL/rEkNBR/QKLYQpBEISxrr5hgd3mlIv9OXZ7
    yirUagUZHh6Wy7JMgV4/6aasIMBRgBAHaOsBA9tdmtyoA+xqQOEHYlsAjgMoGqgxA345EOQB
    ouwAfIBVAtiiAL4WoHiCuDW46POeh0RKx7vd3niW4cQB/Xce3ZuEkCtk1RzHVavVo+8oboAL
    gEomijWmrq6m6M4Oh9nr9Yd5vB6C43iCpilRrVKJCoW8zWDUtRmCzXUEmdACeAKAps/XmvRM
    OwnUGa3WuriW5s7ELpsjxOsJ6EURJlEUCJKkWJJEvS5I7Q0x6BrN4aYqpTKpBtA575Tx+9c9
    ygFIaL/PNjQQCOj65KqHT4lU4pDLlJd4juOk8qwBcVVf37DA7fbKQfTN8LV6sDtzxVKA+COu
    Wnq4kvmBmBu4qjMDgR6uwnu46pIAXX1zJaHjvV5/vCAIAy6NJ0iCkMmkA+IKqDO5nK3RVqvT
    7PX5wrxe73WulEqlqFDI2/R6bZtWG1ZHEHF3wVW90elojGtt60q02RwhXo9fLwg9XFEUS1Go
    1+nUXn2wrjEsLLRKJo/vN1c/t5/bz+3n9nP7uf3cfm4/t5/a/jdp5Kj/xqYsuQAAAABJRU5E
    rkJggg==
  }
  set pieceImageData(Merida2,30) {
    iVBORw0KGgoAAAANSUhEUgAAAWgAAAAeCAYAAAAB6YS9AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh4HBVfzXQAAIABJREFUeNrtvXV0VHf+
    //+89467RCbuRkKM4Fq8uC9aoKVCabvd2rbdtluX3cpWoAWKtoUWd3cnBgQIcc8kk8m4272/
    PyYE2iIJsJ/P53t+vM6Zk3NyZuZtj/t6v94veQ/wUB7KQ3koD+WhPJSH8lAeykPpuBCd/QDD
    MH/+EoJ4IJ1hmGoALgLgRgESHWCxAFEP7PvvrU/t42W3/fU8qDH7v5u5l2X4g/gAUJ3u063W
    EqgBwJN5PfqhDECw2cojgE4HpPypnw9yXRimFABFAvx4gNEBLB1AgCCC/6fW94GP7SHP/1d4
    5sq8XsNQAASL9f8Wz6zOA9ck1emqJjTUNw/i87ktMbExmxiGyQdMDEHI7nNYTgDJVHPz0Y/r
    65pDk5ITVkokyk0MU28HwuHxFA9i4KM47C7HAL2PIIL+yzCXAABstvzRRZdKn2LAID09eZlI
    lLPb47kKNjv1vr6fpi8CoLpWV9W/otOZWAwAsVhAxsSErmOxyNbqKvVzBqMVBACZTISY2NDv
    vF46oLpaPdNisdMEAKVS6omJjfgC8F3u3NjKAdRJNM21C1paWrNUIUGHAgOTNwB8bl7u0aU7
    d56YzAAYP37gnpycvvOAVqeutXyqukkzNCBAcSEkJPZHhik3E0TCA5lrn88BAuzuBw8d/k2l
    UuzMyJjygtt5jfnvrm8jADPHbisfWlenHsswDCIiw3aKRCmHGKbZTRCqB8dzfXNoUtKfeQZ8
    FPt/geerVyueYhgGqakJD57n6vpXjAYLCwAEAh4ZfRPPFosd8HPeznNNtXqm3e6kAUAmF3ti
    Yu6d55aWugWtWl1WUHDgoYCApA0Aj5uXd2zp9m3HJzNgMGni4D3dcvrMA7ROna5iqkajHapQ
    yC+oVNH/J3lmdW5HUIiuXDn8+S8/75mv0egpAOg/IGvm1CnWZ02Osp11NecQGd3rPoblBgCU
    llaL//2vnwd2SY3tOXHiI+N69+73D5erKGbdup0r7XYHf8aMR1/4/rslPzFMFQgi9r8GtNdr
    B4t1LXPLloNLjh8rjPSPtzZz7lzOeK/XdfF+v9/pdIIgqZAfV2ydXlvTzOFyOWCxWfj76/Ou
    yCSiusWLN840GMxgGAYKhRRvv/PkfqPJGvHVV+umez1euFxuREWrXP9464lfCIK5fPu187Qt
    dUsYYE8EeBWASl1atvPNVSt2vNrYqCWTkqKmzZs/hqVQSDQbNx6cEBYWQBEEgV9/3T8yKSmi
    r8Phjf1x+ZZ/X7tWzQ4JCZg5f/7Y4JQuE95kmKYQwJkACMqAoEbAB4JgdeLBKgNBJMJmsw+w
    Wh0vb99+PKpvn67Dw8NP/FUuk/7i9RRqWezs/4KyKgcQSjQ3H120ds2uDw4ezBUyDIOhw3rM
    mTNn1NthYUP/43RcYHj8rAfC8yefrB3YNS2u55SpQ8f17Nn7Hy5XUcyv63ettNsd/OnTR77w
    /eLv/8d43rz5wJKlP2yLBBg8+dT4zHnzHizPy5ZtmX7y+EUOQRCIT4zAO+88eUUqFdV9882v
    MwsLSwEA2dlJePe9hftNJkvEp5+tmV5RVg+GYdB/YKbrnXcW/IJ74LmsbOebixdvfPXihXJy
    4KCsaU8/PYmlUEg0637ZO2HTpqMUAFgstpFJyX6elyz+7d9HjuSz09PjZz67aGpwSsr4/3M8
    d7hlo7EZfL5t5OZNh+YEBSmov700A+pGLf7zn9/Cg4OV/xg1avIpi6XOcH9LLAUARiAQueUK
    CYKCZLwlSzZO1moNqWKxgDp8ODeEJEjwuOz3/vbK8zyG8ZYyjKkEkLQ86CMKALS26kBRVNrZ
    M0WRTz41HgQBrFq5O/LRkb27+mjffQNNMzQIHwGz2Ya580YjKkqFt99ehrzcy3N5PI5Trdbi
    tddmwe2l8eXn63DyZMGrTqebazRa8MEHT6G2thnbd5wgfF4fQVHEbYCphcFwEmKxst+hg2e+
    y8+/lpySElM+ecrwz3LPXRnHMAz56WcL8cH7qzgFBcWjRozovcXhdLGiokJAkATKyuopiiJl
    eXmXB9XVNbM//fRZfPvtRvLc+StjU7okX9qx49Drly6VJ2ZmJJQMG97rObvdcspqLYRI1FEI
    XWAYhltY+Ntbv/y8b9iFwlI0qbWJJrP18zmzRzX6fN6N/x1r0gWCqO+yb+/pl378cYfQ5fIA
    AGrrNEKZVPTSMwvDDjDwXn0QPIvFErfJZMVvvx3iXbpUPvmFF/SpYrGA+u67DSE2mxNcLvu9
    F19+7n+EZzabnXboYG5kXV0zAODwobzIMaP7dfV4PQ+EZ5IhodOZUN/QAoIgQFIkLl4qmcvj
    cpwVlY1oaNACACRSEc6dv/iqy+XhVlU2or6hBQzDoLXVSPhomiAJ3JXnw4fO+nnuElM+adKw
    z/Lzi8ft2X2GdLs9MG6zcHr1TB01ZGiPLVaro13P2W1OikVRssLCokFbtx5nGwxmNDZqyR49
    U8empCRd2rnz8OtFl8oTMzISSoYO6/m/znOHFfTp0xeRlBSf0dSs486bPxZBQQEICgpEZmYi
    rlypiBo1yh4gFFL3qaBF2L7jU19616wLIpFg0vjxA9GzZyq+/GJ9MkWRmDtvNCIjVXjj9cUx
    53OLl0ZEBDsyMhIrunXr8lNYWOo6hxNqHpd5YGCLRBJw2GxdeHiwc9vW4zyCIBASEmAWi4V1
    Ho/ngQUACIKAWCxEWFgQkpOjce7c1QSSJBATE4qIyBB4vTRiYkJx4sTFNJpmkJwcjbCwIOj1
    FpB3HasbcvkjnLNnfnl5w4aDGd27d8Gu3SfTlErJsyqV0nXixAUcPpQPj9cHuVxi4vFiTz76
    aO+zq1bu6sswDObPH3NOIIg8qVRWjvD6fDh8OB86vRkqldJx+nTu09u2Hu3avUcXbN12LEMo
    4r06ePCCXLer0N1JBN2REeFr5z8+njKZrI/07t21acDAbhvYLHYem0X9V6xJh8MONtsdf/Fi
    eYjT6QarrR2X041LReUhNpsx3uv1XX0QPGdldrsQEhIwSd3YipKSWnz88arkkBAltFoTAAY/
    /LA1Jje3eGlYeNB/nWcuh6OLj49wnj59mQcAMTGhZpFYUOd2uR8gz35/MEkSqK9vwVtvLk0g
    CMDhcOH6UEpLavHaK9+lMQAcdv//abojm5Kf5/Pn17/86WdrMirKGxAaGpAWECB9NjhY6ZJI
    hNBqjVAoJZDKxCYuN+bkxEmDztbWNfelaRoTJw48x+OHn1QoKkcolRLo9WZIJEIEBykc587m
    P/3Zp2u61tY2IyYmJEMsEbw6cOD8/1WeO6ygY2LCIBIJtSyKQmODBlFRIfB4XGjR6hETHeYA
    KJfLeb8uQzd6dO8KqVSpY7FIFBRcA0mSYBgGdrsTGRkJCA5WQBUSgOzsJMLn8wk2/HYwfdu2
    Y/+aOHHQjKFDL7wJ4CDDaGmCCHwAAY8yAFJ1r95d9e/+c1koA+Dtt58w8gWqFo7XAIa5v4fn
    ejiF8R+9wDA0XnhhanuwgyAAiqLA4QB/f302rsdACIIAw9Btn+nQnPO1rcZIDpeNqdMGo6Ky
    AVVVDdzZc8YuaW01vLF27d6YYcN7mAcMyPrJ6SyvGzVq8EfnzxVvdblcxOgxAz91OutqevXK
    Xn/+/JWJ69cfFMyePbJmyOAeX6//df8CsViAadOG4OrVamhbDKGAl09S6DDQBJEChqliAgJT
    Drjcrt4giMEURdoTEmL3ASn1gP6/oqDZbA7YbJ5BJhc7CYIQ3hwoksnETh6Xa3Ay7gfEs1wX
    GhqAPJpGSIgSkyYNwsmTl9qVmMPhQt9+6YTV6vgf4XnAwGz9tm3HQxkGGDiwm5HHC2phs4wP
    hmfGP4fx8eEQiwV/4pO46b1/DNBZLHaQJNGROCNfpzNFapr1oCgSer0ZNdVq7vQZjy559dVZ
    b6xZuydm+PCe5r5909t4fuSj3PNXt7pc7naeu3fPXD9+woCJ+/aeE8yePbJm0KCcrzdtOrSg
    tdUIiiLR0mJAa+v/Ps+sji2uHoAcwDVbZmaC+8cfd3AysxJx+nQRysvrMXPmSC3AkCwWc1+L
    TNN6hISkJW7etGVhZUUDWBSFJ58ch+8WvwKLxQ6VSgEWi4XsrEQM6J+FLqnxmDFDj23bThDL
    l23Lrq5W/zxntv0VuaLPWh9dyVBk3D3C7IFe91colC8nnDq5//N16/aFvvTyTNA0jQ0bDkYK
    +Nyv+/Uf9Jxe+0wZw3hAEOx7aockSJAUBYZm8P57KxAbFwY2m3VXpUsQBDweD6oq1cjITADF
    okDgdp/hAYA5IyNh79kzlzKfe+4Lkstlo0fPtHwuN3s5w5R5CwrLV9RUN/HsNqfY6fLA4zXJ
    Gho0bJfbQ2pbdAqGYcAwbFZFRQM7KSnKMWv2uBcJInV7efnu3qdOXuz/3KLPERERzGRmJe4H
    WCaLxXeHCDvDApxyACTAMwIXXECwqKjowPerVm2b5HQ4sf/AuXi32/vLrNmjH3M4nHsYhiEB
    lxywswGuGxAYAS8NsO6JNYaxw+U+DKCr75FB2ea9e84INRoDCAIICpJh8OAcM8US+ZyufDCM
    HQQhuD+eN29ZeP7cFTAMg5Qu0VAoJdC2Gtv77nZ7UVfbjKnThmDq1MHYtfP0f43ns2cPfv7t
    N7+FGgwWAMD332+OFAi5X/fuPeDB8EyS4HDYmDx5EGbMGNZmTNydZ4ZhsH79QZSW1YEkCdx+
    Wf08p6XF7R0xolfm8eOFZFxcGLrlpNzguaBsRXW1up1nr9ckq6vXsN0uD6nV6hQM3cZzeQM7
    MSnSMXPW2BcJInV7RcWe3jk5yf2vXatFnz5dma5d4++bZ7fbjQMHz98zz6y7L24VADmhVh+Z
    /9tv+z8rLCjhxMWFgyQJiMUCCAQ8LF68qfvECa2rh48Y/IzHe7lEXV2I0JjsTkJUiukznsBH
    H745d+++s11f/Nt0BAbIYLE6EBoWhPDwEAA02gIbYLNZALxQKKR4/PFxSE2NwccfrwkAgy+e
    fIprbm5q3Gq2XIVEnNrJflgBsAix5KlB27dv/8+2bcfSJ04ciLFjB7RZXhRWr941zGCwbBwx
    ctGLAOsYw5gZgpB0GmguVwiK4tVOGD/wUn2dpjuPx8GiRZOhUEhwO6YJAjAYLFi8eDMUCgnG
    jumfLxIKq9wu123eHwGaLmWiopL+9eyiadrS0uqnf/v1YIpeb5YzDMMCSh0kAeLq1SpO0eWK
    cb16zd5RXr4nzeP1kWAYNDW1pqRnTCEKCjZMKS6uZqenxfnsNquDYRjOuXPrlRwOC08+OaEk
    Lj5iZVRU3HKGLoZC0eMPc6oGYBXrdJenXrxwdXRzsy4FICilUlKZmBR1JDZW4Ny+/eiIxMQI
    fPLJQtTVNePTT39SpnWNf7Z791RuQf5vj9bWNvXS600CsURoTU2Nz09J6fIDRSXmM4wVBCHq
    xPo2oPjqBsTEpPVft+nXZZs2HgkxmW3tCsFicWDxd5tCNBrDysmTh79QWrZ1v15zCYrgjHvi
    +ZOP/zH31/UHuzY365GUHAWL2Y6PPlwDn4/2W4sA3G4Pli/fiR07TmHipIF44vGxSE2LxYcf
    rnqgPO/etes/3373W/q14tp2RZCffw0fvL9y2HPPWzYOG/rcA+F57NgBl37+aW/33NxiSKUi
    cDjsO6Yzut0emExWgCAwY/rwfJFIdFeeIyMT//XyK7O148b3f3r9uv0pOp2pnWeapomC/BLO
    5cuV43r2mrWjqmpfWnOTjnS7PdBo9ClpaZOIgoKNUwoLS9lZWYk+u/0Gz1KZCF9++deS2Njw
    lRGRMffFc3x8OJ58cizMZjuWL9t+g+eC3x6trekYz3dV0E6nATyeo/emTYc+qalWK9/55xOI
    jg4Bm81Gr15pSE2NxaFDecT69fsHyuXit7v3+MvjIsVFV/HlYnTp2qUTyyvEr+tPsvbs+b6L
    QMBFdnYyvv1mA/Lyr2HJklchk0naD0ZcLu+6jdL2PwY9e6bjtddm4/33VyojIoLfnjhpXKHX
    Y67tHMx6NDaeRoAyZMwvv2xfduJEoeqppyegT58MmExmAMDYsQMRGCjHsmXb0jUt+p9mzXI/
    ZrXYjnz9zgb89f1pnQKaohJhNe8oHT5iyHSFUvLhqpW7pn7zzUZWXFx4u+vjVn6+qqpG8Hgc
    3z/ffXJTTk73Nx32piq+oM/tLRsyCQzDmN3u/K8zMpJsK1bsWL5167ExGRmJ48RiKc3ncxAa
    FoALF8qG9OqlS25s0CQrlVKQBIGGxpbE9Axjcl7e1X4R4YEQifikz0cTNlv+o1s2Hx6r05uR
    lhb/udHUsAIYAfIPwUp/aqZAdvXqkS/Wrtkxp6lZxw4PCwTFomA0WBJ37DgxMjkl2lV8tYr3
    9DOTwOMJEBcXAalEhAMHzg7ftfPEUL3exBWLheALeDAaLNi181TG0KE9hk2dOuz5opJTO6zG
    qxDJOqq4TOiSOjJk586NX3315fpkjcYAiiLbFZbL5UZhYSkaG1sSJWLBx+PGTyjyuHVNnbcn
    /TwfPLi8S3l5PYYN74G3356PTZuO4sKFsnbl7J8jv0Wm0ejxw/dbcbmoEv/69/N4/e9z8O57
    Kx4Iz+vX71j2/febVc3NepAkgWCVAgSA5mY9rl6txnvv/pje3KT7acZM133zPGLE0OkKheTD
    H77fMrWyspH18iszERgo/7O7gyCgbTXgi8/XITQk0Ldw4eRNOd07zrPLlf91WlqCzWLdtnzb
    tuNjMjISx0klMhpg0NioRX7BtSE9e+mSm9Ta5NZWI7xeH5rUrYlpacbk8+cv92ts1KJbdhIJ
    Bn6etxweq9OZkZIS28bz8Lvy3NysY0dEBoPNopCXZ0ncufPEyKTkaNe14irerFkj8a/PfsHQ
    oT2gUEhx8MDZ4Tt2HB/a2mrkSiRCCAQ86PVmbN9+ImPE8F635Jl1t1SkTz/5NyZNHjfm8uWK
    oJdemomEhBj4E8n9ky0WCzFx4iOormrEmTOXRnTvkZXCZtMXO5dhDQAOAG6l0WhNlEpFOHw4
    H7v3nMHIEb0QERHcpoz9KqqxUQOT0YouqbE3qTEf+vbNwKOP9sb69fuzUrrELExOHve6zXYR
    QmFmB/vQjLCwAdK9e1e/eujwedWbb85DWloCABJbt54AAMybNx69emVALBbgow9Xh4WGKF8f
    NXpu3rNvlVv++n4ngyoE73pqTlVOzuin5TKJ/h//+H4Rj8dGYmIEaJr5A5gEysrqYbU68PfX
    5y5PSOj/GqCxCIT9O9AWAYcjHywWSxseFuQ9f/4q/8iRvJcmThyxBQzoXr3SyIqK+kijoWRy
    Q2NLtEqlAJvFQpO6NcZsKplWXlYX1bt3OhobNAyfzwvbs/fkY+fPX+F1757ikUgELUJh3J+O
    ZwxTAYKIQX394aeXLt00T6mQkF988TwCA+Vtm78LVVVqYvu2E7yS0lpcvFiKQYO6obXVgOJr
    1RCL+Ow5jz2K3r27IiBACpIk4fX6UFhYiq+/3hDJ5bL/NX3G+FK3y1ra0Tk3m03gcJx9Dh3M
    zdBoDO3BwZvnicWioNEYcPDg+fTBg3P6eLyezfcQhgTgVur15kSJRIjnnpsCmqZx8sRF0DSD
    61k3NM0gIiIISqUUly6VAwBOnbqEJUs24d13n8KoB8Dzgf1rX12yZJNKozGAJAl4PF5MnDQQ
    BIBvvt4INpuF5mY9fvhhS1hoWMDrI0fOuW+eu3cf/bRCIdG/8/bSRefOXkFMTChomv4DzySq
    a9SQyyR44825y+Pj+70GtHSKZzabrQ0OVngPHczlDxiQ9dKECX6ePR4vef7c1cg5s8smNzS0
    RJvNdtA0DbVaG2O1lk07f/5KlNvtAQCGx+OF7d594rH9+87xBg3K8kikwg7xHKCUkB98+BRM
    Jhu0WgP4fC5YLIo4cqSAV1JSi9KyWnTLSYFCKcaVK5VgsSj2nDkjER0dAqvNATabhcjIYDQ2
    tOLbbzfekue7qFEr3nhzGfr06x3E4bARHh70O+V8w91PIjEpCufOXeECbi6H21kfmRoFBbsR
    G9tlRGHhtfgB/TMRHROCZcteR0JCBCiKuqlNEnm5xbhypbJNQf/evpw+fRgOH8rD/n1npycn
    J69ms70lHe2H1+sCi1WdfubMpcwhQ7ojLS0RdrsVO7afxMED50EQBDhsCuPG90dqaiIGD85B
    bl5xz1Gjq7rStO/MPUe/iUSo1YctMpn0ckCADAMGdEP37mk3bUq4Mfa8K6iqVEMqEV9UN+ZZ
    wsKHdbgdHo8DQFgdGCQzZ2cnKg4fzuvRv1+WyEfTSEqKQm1NM1lcXDlfrzMFhIcFgcNlobam
    Oa64uHwuCBDJyZFoaGhhGQzGpw/sP5fZrVsygoIUFh5fXEfTzlsGyRimOXbNmhVPmowW8v33
    n2w7CdFt/eGhS5c4JCfHICY2DNu2HoN+jgHHjhUiJESJjz56BmFhqptOSgCLxUKPHumYM8eM
    9esOJPXokfqXY8dOv2+xFEAs7nbXOWhpaYVAwA9o1uhZxB3XBGhpMbDMFmuAzea4J57j49NG
    nD51Md5ssWHJks24VlyDlhbD76xnmqbRs1cXdO/eBUVFFe2Ka/eu05gwYSBmzhqBQ/fFc036
    yZMXMjUaPUiShFwuweQpgzB4cI7frPHR2LzpGFpbTdBoDDh39krPkSOrHwjPCrnsMpvFwtKl
    29qe4T+Lz+fD4ME5kEklF9WN+ffGc4DMbDBYFJs2Hu4xoH+WiAEDkiRx+XIlWVJSNb+lRRfg
    dLoBMGjRGuJKrlXMvVxUQZAkCZIiWQaD8elNGw9n6vVmKJUyC48nrqNpxx15Npus5Et/m45v
    v92Egwdz4bC7wONx0L1HCp59djJiYkKxZcsxfPTRMzhz5jJkMhH+8Y95OHP2Ml5/fQnUah0I
    AggPD8Lrr8/B3HmjbskzeZcpwNgxPRAUqLB43B5otQa/L/wWcdm6umbIZGIXwHW4PQS6pHTc
    veH1atGt2/CkvXtPvQownMFDcpCenojs7BSIxcKblDMBgEZJSS2iokJu0RcaKlUAhg3vgWPH
    CqIqysumfvftT7BaCzrUD4PBAL2+JdxosIjTbrLOSYoAQRIAAZAU2b4xdU2Ph15nluh1hiiz
    2XJ/AX+GAQOm7RhIt718f3jR7cqNYeiOZnDc7CUEEGCQSsXG2LgwKORi9p49pzJomiGFIj6y
    shJx7tzluJYWvTQySoWIiBBotUbx6TNFMRkZCRCJBGAYUPsPnO0hEgs4iUmREIsFNkBkdrv/
    3Be32wagOfTSpbLQzMxEyGTSP2w6/rGSJIEpUwYjMSkS27Yfx7lzV7Bw4eQ25Xwrg8CHrKwk
    WG0OlJXXZj3+xEe3e/7/JIGBCshkUmuAUkozd14OKBQSWiQSmhQKaadm+TrP+/effvX48UKO
    0WDBwQN5aGzUwuv1tVtmDMOAzWYhNTUOVZVq+Hx0u3Wo11uwZcsxBAbKMGx4z3vm2WDUhjc2
    tohvoMKA9tFgaAYMzYD2Mb8LfNXXayRGg/HB8dw2HpK89cs/FwxohrlnnoOCFEYOh43c3GL2
    3n2nM2gGJEEQMBjMOHnqYlxdvUZK0zRomkF9nUZ84uTFGL3e7A9OgqAOHDzX4+y5Kxw2m4XA
    ILkNEN6V5+xuSSgtrcOWzcdgNFjhdntgNttwYH8uXnttMZKTo5GcHIUTJy4gL68YTz89AQcO
    5uKzT39GZaUaLpcbTqcbJSW1WLVqN7p0iYHT6f4Tz3dR0CFYt/5rJCREb49PiDQuX7YdLS2t
    v8uBoWkaJ47nIze3GL16dj0DhFQ6bEwn1rECLFa6IC/v7DuHD+WlzZo1EiKR8CZldMNyBkic
    PHkBhRfK0L9/5m28tMDgwTlobTWhsLBk+N9e+reEw+lYpJ8kKZAki2EYMHaHP0ghEPAxbdqj
    GDmyD0aO7I3p00e19c+f1wmApijK22EN8b8qHgASg0wm0hoNVkyZMgQHD55HdY0aLqcbXVJj
    UFhYhrKyOphMVhgNJlRXq1GQX4Ls7CT4fDRq65qwd+9Z/OUvQ2E22SCRijRAuI68TeIVw/gf
    DB6fc8cELRaLjXHj+mPbthMQCnjolp10ixPEzWlyFCiKhNfrIwD/rQcdEYlECoEg/MTAQd2u
    yeXiP7mRrrsdZHIxBgzMuiKRhJ6USCSd5rmg4Pw7a9bsTjMabSBJEhTlz3C4+djMMAwiI4MR
    FCTD/v3nfpe5QJIEzpwuglrdiqFD751niqQYkqKY64rfYLBg6dJtOHw4D4cP5+H777dApzO1
    W/UURdEURf4/xbNSKdFyuWy4XB6sXr0btbVN7ePZs/sMCgtK2jeES0UV2L37TPtGWF/XjNWr
    dsHl8oDL5UChkHSIZz6fC4fDCZqm2zea6+6xstI6rFixA0OH9sC2rcch4HPhoxmsXr0bPh/d
    HvO4/nI4XaAoEmw26088k3c+qsggEsWAotLPzJs7fjPD0Hj7raWw2extSprC8eOF+OyznzB1
    6pDWfv1zvna7y2wEpB2EuRr19eWwWvMnbt1yZHJOTjKysv74YBJgGAL1dU1Y+sMmfPLJWjgd
    bsjk4tsoaAaxsaEIDQvApaKKVKA0wev1dqg/crkSMlloRUxsmOHE8Qvw+bxt4/Rh8qSBmDRp
    UJtFR8Dn8+L48UKEhAZopLKAa35L//+6iAHAFRgg1zQ1tSIqWoWePdNQdKkC3367Cf/618+o
    qGhAUVEFFi/ehG++2YgrV6pQVa3GV1+tx7JlW3HxQjm6ZScjKSka6qZWKJXSZoBwup23UqIC
    EESQLiY61FReXg+fz4PbJ7j63SwsFoXklBiw2JzbbsAAibpaDQgQCA8LLguLCvTRHVtiEEQ0
    cs8vrh83bujSRc9N9gqFvN9ZbgzDQCjkYdGiyd4JEwYvvXTpaCObHdwpnm22/IkbNxyYfOVy
    ZdtDfatNgAbDAE6nG59+8hNqa5tBkuTvfKxNTTrVd1iQAAAdKklEQVRcLqpAXFz4PfMskYRU
    pKbGGiiKbO8HSZLY8NsRbNxwpL1NhgEoikRcfJhGLFH+P8VzQKBCIxDwQJIEKisaceVyZftm
    WF3dhOrqpnZlWF+nQXWVGgTh3zSLiipQXt4AkiTA53MQECDrEM+lJbXIyk5GRmYCfD7f7/gh
    SQKnTxeBy2WDoigkJEbi1KlLMOgtv9uEGYYBj8fBuHH9YDRaQdM0wsKCfsczeWfgtACCSYsl
    f/Tx43l9W1oM8Ef3bxzz+TwuxGIBcnOLJZculT7P4ahixVIfGEbXkcMgIiJGsvPyiia1aPXc
    nJxkrFu3Dy6Xu+1BJmG12vHjj9vw9ttL0dTUaps1a4RLIOD+KeBws4IWCPjomhaHysoGeZNa
    k6lWN3fQ4hACiLk8fHivjY2NLVi+bCsaGzWorVWjrq4ZdXXNqK1phFqtwfLl29DY0IIRI3pu
    AJKvMfT9VnsRIED4nZ9tpwWA+sOLvGnJrr+3M77BKHC5hCcoWNFYW9uMV1/5BmXldWCzWYiP
    C8OCJ8Zh8uRBeOKJsfj0k2fxzj8fx8SJA/HIoGyMG9cfQpEAIAC93oR3312G8rI6BARIGwmC
    cK9Y33SL9igAYRX9+mcera/TYP/+c23rSvxhTP5jLo/HQYhKieBgOW5UK5B/Gr/JZMaqVTuR
    nh5v7tIldl9x0TbwRd06oEDtuHLlVeR0n51z7lzBhIsXyqnrboU/+kUvXiijzp27PKNr1759
    1eoTneK5sODqpKNHC7gKhQT9B2SCy+O0K0eGYUDTDKKiVZg/f5Ttxb/9xWW12m8ZBHO53Lh4
    sRxcLu++eB42rNfGXr3TwDAMuFw2evTogpzuKcjJSUGPnl3A5bJB0zR69U7DiOEPmOe2zcjn
    893y5X+O2955jzwrldJGiVQImmbg8/ngcnnaLVs2mwUejwOhkAehkAcejwMWi2qrXKThcnna
    +sFALBZALhd1iOfa2mZcu1aNjz58CqmpMfB4vKBpGiwWBT6fCzCAw+5ESIgSgQEytLYaIRTy
    weGw29xgPrDZLDz33GSMHt0Xa1bvRmparLlLl7jf8cy6k3ImiECYTPlzVq3a/J+iogrZ/MfH
    oEePVHC5nPb0tl690/B17EvYsOEQ59tv1o2f/7hV2LdvnyeBxpoOJOYAYOSaZkNqcLACp89c
    BofDBpfLBUDAYDDhyy/XwWF3OZ5YMH5n//49jvz446Z3lEppqFh8p+IBEvHx4Th6tABNza3x
    y39cB7X6PEJDe96lP9Gg6WJX167d31q40MNe+9Pux3Y9/RkrKEju34FxIx0qITHS+8zCyWuz
    s3t8yDBXfM1191GBxbQCUJKNjUdjzGYrCvKL4bDbbxn1Li6ugdlsg8frjQkPH0oyTDVNEDEd
    buvSpY1QKgOuqVRKKJVSZGYmYvpfhiKtaxxYLAoZmQlwuz2g2xTXc89PAYfNAo/PQ79+GRg7
    th/q65px8WI55AoJVKqA0osXfkFm1phbzqfDkefp1i3z39OmDe22bNm2RIfDhYEDs1BT04zq
    ajWam3Uw6M1wuTzwen0oK6/HunUHcPRIAVhsCjKpCEFBCqhClIiNDQVJEvjPVxvg9fq8U6cO
    WcLhZp9kcKkDc8zAaj2LtLSX+mzZsmf11/9Zn1Bd3fS7FLsbitGD3bvPEMXFNf1efNG8avyE
    wY87HFdPNTQWITws/e48txhTDQYLnnt+ClQqJfJyr7XltfsV5KhRfRxTpw7Z2btPtyOrV299
    x2p1hN7OD15V1QjAd188p6XlvPX3v3vYP/647bHqqibWu+8tgEDAA8DA6XDhjTd/QER4kHfB
    gglrM7NyHhjPavWxGJqh8ZfpQxEeHnTLrKSGBi1aNHq43Z6YsHvkOSAg4Fp4WBA0zTqkpcUh
    ITECUVEqqFRKSCRCCIU8XC+xdrm9sNkcMJvt0LYYUFvbhLKyOly5Uo2wsEAEBSk7zPP3SzYn
    zp49Eu++twAH9uf63VXBCggFPHA4LEREBgMgwGJTeO65KZg5czicTjdMRitqapoglgjQv18G
    3nt3BWia9i5YMH4Jl5t9EjfxzLodzE5nARimLHv1qg3vFxdXy959dwEiIkL+4Bf2S3BwABYt
    mobNmw9jzZpdQ2Uy8TupqROedjsveTi8jLsATdjEEkGz3eZMGjG7J0JDg1FYWAKJRIgtW46B
    y+EYFi2a/o5KNWhNaemuD48ezQ+dPOkRsNncNnfD7YJBMv9kmKwJ3y/ZTjqdhXRHUnf8FtTV
    1qzsPsvOn786ITo6RDF37iiwWP6p8nm9WL16N9hsjjE7u8d3NG1ppaiM+4BZC0BJGk15z/z6
    695nvV4fyivqUVnZcGtHAMPAR/uwft2+Z594gl+nUPb9gfZV0iTVsSqzpKREEIR8z6BHshee
    PHkxxevz4fCRAvz22yFYrA4wNA2CJEEQbTYQQYCmaRAEwBfwEBykQGxcGJwuD4YOybkWHh6z
    TRWsuM188sEwLeARQRdsnstPECSxeN26A+lr1+xFsEqBkJAAhIYokZoWC6GQDy6XjbHj+sHn
    o+F2eeBwumC1OqDV+tPuflqrRYvWgMzMROez86d+l5jY60Og3MPjdeTGuVKIREmBR49u/+jr
    /6xPqKlp+lOK3R9T7WpqmvDVf9YnSKWiTwcPGTkxhKvT3t3gIGxiMb9ZIOAmlZfVY9fO08jM
    TIDBaEFVpRqz54w0PPXk5HeCggeuKS/b/eHePadD3W4vqFs40a8XJblcbgQFye+L58zM3su6
    dLk84cSJS4qX/vYfuFweMAB4XDZatEYMG9bTmJnV/TuaNj8Qnk2mvGfWrt3xrNlsw8KFkxAY
    KLtlHnRrqxGfffYz1qzZ9ewzz9w7z0OHdl/o83lTPv54IQRCHvQGC6wWO2xWB3Q6E7xeHxgA
    LIoCj8tBWFgAkpOjME7eD263F2+/tQwDBmR2mudfft6X3qNnF/TqnYb6Og1yc4uhbTHAYrHD
    7nChvKwe5RX1CAkJgFDAg0wmQkiIEjGxoeDzuXj/g1UICwt0PvHEhFvyzLodzDxeN1ZB/oYX
    jx0riHzp5Zltytl3W/8hSRKYPHkwqqvU2Lbt2JS4uIjNHrdr952nl8arL4+y/fXFlw4c2H9m
    4MkTF9E1PQ6bNx1FRmYCWloM3heen/6ZSjVoWWXlnpd/+H7jU7ExoXh0VO87BpAAgMfnwu3x
    wmi0cAGQbjdDdwywZqxd8yWGDhs5PDf3iiI8Igg//rgDJpMNACCVCGE221Bb1xwwenTp0PzC
    yxes5lyIJD3uAeYmrFi5CNOnvz5v1cotnxQWlohfe202VCrFHSuvNBo9vvh8nXjlym2fLFgA
    Y6u2aZ1aXYDQ0Lsf8wlCiPXrl1aOHDli9dGjBZ+dOnkJIx/thRCVElKpCFweBxRFtQdZGIaB
    z0fD4/HAbLajuVmHgwdy4XK5MWJknzWLlyypfurxeXdoLwgM44Cu9adTU6ZMmC6RCJevXLmz
    73vvPYmQkGDcuI3kRtHRjYydm106DH76aSdyc695X331sbcVim7fAi0ugkjqwDyXYv/+/cjM
    TJ/9y897B9ZUN4HqwKU1FEWitroJv6zb1yc5OXremXPn/2005EEm735Hnv/20isHunVLHrhl
    yzFIpSLMnz8aa9buQf/+Gd7588Z9FhQ8cFl11d6XFy/57an8/JLfpd390U3gdLrh9frug2cN
    1q75HEOGjBx+9WqlYsb0oUhMjIREIgBAwGK1o6K8AdeuVQdUV18bWnjhyv3z/JfX561Zs+2T
    E8cLxUqlDGvW7Lkjz2KRAKdOXRTzBbxP5s+7N55HjBy++syZos9eeOErOF1uuF0ekBQJHpcD
    DpcNFuV3pdE0DXfbVb00TYOiKIiEfAQGye+JZ7lMsvyNN5b0XfHjLhAEIBDwIBDyIBTyweOy
    ER0d4r+lT2tAvcMNu90Jm80Bp8sNhmbw6Kg+3r//ff7bCkX2LXm+pYKmaTdIsjbl+ImCYamp
    MUhPj7+jtXr9ASJJCtOmDcWHH64Sl1yrmpiZNXVPS/M5JkjV6zaDjQFNl4EgJD/OnPVotw0b
    Do47fuICSyjk40JhGQY90u1cZNSAVTXV+xb9uHzrW3K5mPfXF6eDz+fdIYB0Y+G9Hh9cTrcI
    AMfr9XUwjGTCY3NfTj14YPvcosuVAIDMzESkJEcBBNDaaoJa3YprxTW4fLli/uRJk3Y7HYZi
    xmsEwer8DxY88fhvVEnJ3gm7dp2UiEV8LPluE+i7PHsE4ffN7913VjJgYFbGZ59+v+6rr7/o
    oN8uHj7fVZCk8GhCQpQJjE/K43JQeKEMep0JVpsDbpcHXtq/3v7rXf0+PLlcguiYEAQHK6BQ
    SAwhIaGHn3xiFu58SrpueTRj27bV19LSuu7j87h9c89fBUEWo7GhBSaTDTabo90fCAAkRYHP
    40Ao4kOhkCA6OgTXimsQGRncpFBEb/b5Sl0sVsfvah4x4nnZzp3fTc7NvUqQHU35gD+tMi+3
    mCi8UDJxyuS/LnM5S0y3H+cNnp9+ZnI3kVgwTt2oZS1fvgM2mwPPPz/tXFh431V1tfsXLVmy
    8a2dO07zaPrOd9cwba6Oe+dZj8fm/jNg187VEx12J7rlJMNotKKuTuMPs4mFyMpKQEVlAwoK
    rk2aMmXBCre75p5vqHri8d+o4uI9E86cviT54IOnERWtumWmzB9dHfX1Gnz+73WSXr3SMj7/
    99J74Fl0NDxCZTKarNJnF01CWFggZFIx2ByWP4Pmemoj/LnfPp8PFosd9fUarFm9B6pghSE0
    NKzTPKenZ+wLCVX2fWRwNsaO6YegYDlEIj5YLFab++xGZhBNM/B6vDCarKioaMDi7zYhMiK4
    SaGIui3Pt1TQOl0rOBxbr9raJtW0aUParBhfB6aLRmSUCiqVAhUV9T0zs1oDuDxKe+fBJgAo
    aunZc8CCxITIESazOfXMmcsTKJJkBvTP/sBovDh87U+7P5DJxIIXXpgGvoB/V+v5RrCHBs0Q
    KsAgYrNY9tu5c+D/+R9ee8oGmgQSidA8aGC202Aw83Jzi6FQ+EvNdTozeHwOBg7MciqUUicA
    Lo+fwwfAavsuJwDP3S7x8b9XywUg8HgYdlhoAN56ax5kMnGHLpcxma34+OO1cLt95OYtJ7mA
    lmAYxtmRy4NI0gNAqJbLRbptW49LASA0NBBd0+Mhk4rA43Paj9w+HwOHwwmzyQad3oSiSxU4
    fuIiRo/uYwQUTSyWs4MPkgqtrcfB5QpsbA4bW7YcRdf0eAQFyhEeEQyJWABem/VOMzScTg8c
    DicsZjt0OhNOnryE8+evYv78sVZAZPO6tR1WGn6DozrxWnF1V7PZ9rtsiY64CcxmO64VVyeO
    GdMQAzAXO8Jzjx7923k+fvzCBC6HzfTtk/mB0Vg0fOXKnR9s3Xpc4HcdddzXe48800BTaECA
    1BsUpHB+v2QLr7S07kb+Mc0gKSkKySnRzpCQAAowBnM4WQ7/vRa4J54ZhmC3tprw/gcrwWZ1
    rKTY6/XBXxBE3iPPArVCLtYFBkilJEnixImL0DTrYTJZ4XC64fN6/SV1BAkejwOpTARVsAIJ
    iREICJBBGSA1AvImFsvRaZ6DAuVQKiQwma3Iz78GrdYAs8UOl9MDX5uhQ1EUBHwu5HIxwsIC
    ERGpglgiRHCw4o48s/480QYcOvwrwkJVUofd2VY92PG8ZpJkITQ0EI1qrQCg2RIJ1SE/GQAD
    w+h/lSv4RHR0xnf+iwsF2Lhx5c5WrUH4/gfPdEo5X4fG6/FQAE3+MXeUYSoB8Hl2e/6o4uLK
    6dVVDTEWqw0CPo8IDg7Qq0KUdc89N1W4c+eJxIrKRj/pDKAMkCAhPgJjxgxodLncdXv3HnhD
    pzOGu90etkgoQExMRHVqavyvDNO4B3A6CeLWvrQffngX06cPf6m05NC0+vqmGJPJioYGLSwW
    e4cUtMVih8lkRW1NwzQCa3ulpsXv+fXXxZ907EFnQFGGYF2rUTZyZE9MmToYanUrWloMqKvX
    wOlwwePxF4hQFAW+gAupVIjExCg8MqgbhEI+9DqzDNAFezx0Y8fWogYrV3yNYcNHBQIMFj03
    BWFhQdBqDdDrzTCbbWjRGtqLNVgsCgIBD6FhAUjrGouAABk4bBa0WoMYMIkpFlo6yoHLZQWL
    5RTV1Wu4NM2gE/q5PQtB3dQq9PmsUuYuhsofeZbJ+cRjj2V853+IBNiyZfXOLVuPCr1e+g6u
    jfvjubS0enpVVUOM2WSBUCQkQkMC9cEqRdOzz06R7t59MjEyMrjdqiVIAslJkRgzZkCj2+1R
    Hz586L2WFkO40+FkiyUixMZGVicnx3SY57LSw9OamlpiGIZGbEwouDzO3dUHAbhcHly5XIn6
    usZpDP1Try6pcZ3mWa83y/LySmCzuxAcJEdQkAIxMaHgC7ht8QYGbpcXdrsTZrMdanUrLlwo
    Q/G1Gshk4nviecTIMYEGgwX79p1HVZUaQcH+2Epycpu7kEWCoRm4nG7Y7S4YjRaUlNbh+PEL
    qKxogMFgviPPt9zeTCYzgoICKJphbtoBOwqTP7XF7nB27mMACELRxmGdpqmpBAKBaNbZM0XZ
    o0f3aysR9qGz0q7s/tAPm7UFXB5/6po125dXV6u5XbvGQamUQKMx4PKVSlgtdtAMg6hIFWbP
    HomvvlwPmmHwysszceLEBXz51S9xJEnECYUCREepEBwsh9lix8ZNB3Ly8q6MW7Bg0lMut3Pt
    7fq1cOF7yMyKj/ryi3WZSqUEwSoFtm491mGLimEYBAbIcOrUpcjt209GvvTKzKsLF77Xgc+p
    UV5xHnKZIqe8okHRpNbi3Lmr4PO5UCglbVFvfzoQQQAeuxPNGp3fgtaZYHe4oNeZERQsl6vV
    6my73VZYV3kOkXF3+6kzDx5/4kvesaMrc8pKa/HF5+vh9fnA53EhlYkgEvLbLGgSDAN4PB5/
    UMxsg9lsA0kQ0OvNiIsLV5lMjRlej6ey4wqWgddHk/fzIwsejxc+2teparc/8iwSiWcdOpSb
    bTRYbxugvG+eV29fvm7dAe71wBhFkRAKeZBIhCAIEs3NOsTGhl4vsgJfwMPhQ/nYvPlYHMMg
    zmz2u5v8BRUUlAHSnBkzho17csHku/KcnZ0Q9fbbSzNra5sRHhGMJ5+aAKlU1AGDA7BY7Hjx
    xa/wz3d+jIyMUkV+8OFTneJZJlPklJfXKRY8OQ4zZgyD0+mGxWKH7brLri13nM1mg8vjQCzi
    Qyjig8PhYPPmIzh4MO+eeD56dGWOXm/C++8/hcysBNjtLn+A0O6Ex+3PTCJJAmwOGzweFxKJ
    AAI+DyRF4KOP1uBqcc0deb6lgk7tkgixWGSjSBLVNY0Qinh3yDv+swXRqNYiIkLFAGDsdhqd
    v9nOjNDQ4cjP/7WPVmtkpXSJBtDx0maCYPypYnfoM4vNAovFLvN6fXo2mxUik4lhNtsQFalC
    17Q4cHlskCQJp8ONn3/ai+SUKBebzcJPP+3jTp8xFIOH5ICmGbhcbpjNNrS2miCXicHjcuD2
    eAwcDlXu8dxZ2bqcbsjlYrz22mwIRfz2VKy772r+9xAEYLM68OmnP8Hl7Oid4nYkxE9A8dVt
    fZqbWhEZqcKsWSOQmhoDuUICNpuDG7nJwPWKTo/HA4PejJLSWvy6/iBKS+vQpNb27pYzY4W2
    JbcDC+MB0BJWUVGfbDbbMWq0vzIzKkoFsVgAkmTh5pzo64FDl8sFXasJV69W4eef96OqqpHV
    2KDp+9gTr2wxGQogld89kMThcMFmc51KpczLMAy3oyzfrBRlUpGPw+J57R7XPahVP88XLmzo
    c624hkWSRAdZZkC1XTHQMZ45ZT4frW9s1Ib4CyZIuFweWK0ONDf777smCALFxTV47LGRLjaH
    jZUrdnGvlylfR48kSbBYFDweL9SNWng8XgOHe3eevV4fzBY77HYXamua8PjjH924gP8uFjRD
    M2htNcLl8sBsscHr9XWa55YWA/Jyr+L0qSI0NLTAbLbB4XTB6/GBoZm20gECHA4bAgEXYokQ
    oSEBYLMpNDfr7plno9GKLVuOYsmSzWhq0sFqdcDpdMHnu6GzSIoEl8OGUMiDTCZGREQwdDoT
    DAbLHXlm/Vm5yeFyFYDD4VyNjQ2zf/ftJkFHdsEbMAM2uwNDBvcoB+RGp60SQGcvPA9rs+Qt
    Aq3WgK++Wg8+n4uOGi8kSaCxUdt2B8Ctz7NcrhJAzPnRowet2LBh35vl5fUaDptlq6vTyG02
    h8jt9nAsFjvh9fqQkZHQOH78wE85HA61cePB15ct3abicNkQi4UMl8NyC4QCq0QiMGhbDEK+
    gBs8alTfpSDSz/L5V+7YT6GID7vdhY8+XoN7La31+XxwOT0Qivgd/ATXn96kPXF87txRvS9c
    LI/ZuPEIe+tWCgIBF0IRHzwux5/lwAAetwc2uxM2uxMOhws+rw9iscAz57FH1SpV4BXAn/rX
    MXG40rrGXx0xomdgWWk9r76uBRKpEHKZGAIhD1zudd83A7fbB7vNAYPRAoPBArvNAS6Xw4we
    3dcgk4kbGhqM8NEda5fN5gMIqumaFqeOjQ1NuFVxyt2yOVK6xDSAUFQT0NzDKvl5NhjMApfL
    03Y6ITq0MWg0erz/3gq0tBg6xvOYgStMZuubXC5Hw+dzbAa9Ra5tNYr0OjPHYDQTIpEAA/pn
    No4bN+BTNpdDyWSi148cKVDZ7U7IZWJGoZC4AwLlVqVSYnC5PEKH3Rk8elTfpcDdeebyOAgN
    CQDZVrV3O2v/ljqaJBAc7M9eUqmUftdIJ3mePWdk79OnimKaNS1s/0nACa/X2x6gAxgQJAma
    puH1euH10SAAREeHeGbNHHFPPHftGn91yJCcwKKiCp5Wa4TN7oTX4y/AuTk4ShCAnfC7VnQ6
    M1pbjYiMVDFjx/a7I8/ErcEoAcCV1deVv9HY2BLJMEyniOYLuFRkROhWt8f4m8ulRHR0705a
    LAbExqtw4tiuN86duzTHanV02rdBEATYbIrMzk6pTk5OnetyWXQ8XvYf2rkGhkG4222bwWaL
    d5OkUgdoVQ67KchotMYbTGaxSCiwh4WGnaBYaZcBhvC4r6TXN9T3czicAqlYZJErJJVCgVQD
    MkhD03q522Ucw2Jz1lMU0UCSt48Gb9zwJUaM7PV4ZWXjcIfd6cN9CJ/PpWJiQw8c2H9+5bS/
    vNyB+b3+q8g1EWZzU466sSVd06JPMOgtKofTFWy12uF2uQkA4HDZjEgoBI/H0cgV4pbgYEVl
    aGjgBak0rNDlkNZxeAxNksoOtGmHPwE/Wma11PaqqWnIaVRrU3U6c5xebxI67E4+TUPh8/lI
    giBAsSgXRRI6qUzkVcgl2uBgZVVklKowNDT0LEVJLwO0hyCiO8hTFYAYSqs9uai+Tt3Hdz1y
    0/FMDiosLOi0StV9sUFfQiuU3e+Z5/Pni+Y4HC5fR11/DMPA56XvgWfJbpJU6IBWldtlCTKZ
    rPEGo1ksEPDsISGhJyiqjWfPlfTGxoZ+LpdbIBWLLDK5uJLHk2hABGpo2iB3uwyd4rmqqnG4
    2+W5L545HDYVHRNyTzxbLM05zc3adK3WkGDQW1V2hyvYZrPB1cYzl8thREIBeDyuRq4QtwQG
    yipDVAEXxJLQ++K5vl6d09SsS9W1muJ0epPQZnXwfT5G4fP6SIIkwGJRLhab0sllIq9cLtUG
    B8urwsODC1UhIXfkmbijn+sexWp145tvz+PNN/p06ifLf99+KwAfFxAK73qn0x2/q8Xn8WjM
    DMNh/gy0AzdKj60AFPf8c10Mw8Bt3QgWryfAePzHU07C3X2JD0A8Hh+uXKlHVlZ0p/v/+344
    +YCP759vpu1sSjJ+NwflAHjO6xHj+/rtuvY2aRJwiQAfBzAIAbsSYNr8HKQTUGgBnsffNt9+
    i0DcPY6z859t0VoRHCS+j7Yf8vz/b54pJyB/YDw/lIfyUB7KQ3koD+WhPJSH8lBuJf8fR0E3
    8a/jG3MAAAAASUVORK5CYII=
  }
  set pieceImageData(Merida2,35) {
    iVBORw0KGgoAAAANSUhEUgAAAaQAAAAiCAYAAAAQ0q1EAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh03CKOQMgAAIABJREFUeNrsnXd8VMX6
    /z/nnO09m91N76QHUoCEDgm9SVUQRZpYEMGr2FDErggWFCugCCJNivSeQICEkAAhvfe22SSb
    ZJNsO+f8/tgQQUGSEO693/tzXq/962RzZmfmPc8zTxvgn/ZP+6f90/5p/7T/gkb8X+osyxYA
    4IAgPP6Zuf+DjWVLADDtHxIA+R+dS5YtA8ACEIIg1P8fzcM/HP3D0X8nR5zud6IQQAsAqr0z
    aB+YwAf40yXkrh3jGYbJAEGIQBBef+pP27+hD50Zm3wARjCMESTJb5+oXg/oXdkALGBZ9g8t
    gyD+9DddecYFQQTc5+JsA0C3A0MBEANwgrHtOgTCUCnQqgbEtW1t15tZlgZQ1b6W6HYdiWof
    M7cHOEcVAJwBgLJYEuj/3FphAWT8STdkAQT/Za7+4egfjv7XOeJ0rxP1MJvzwOOFCoHaQMAq
    AyS5BKGuZNkcEIT/A/nxDKN7YtqMjf4EIT4AeKSwbLYV4ALgA3ChgEZJZvqhRpYtug2yfy9E
    1QCkXKAqiiRJX4CXBzhfZtkSy4PRYlgJwIkgCIp7y0gBsGYBbBvACSMIivzjGc0A1usAIQQ4
    gQRxyyPQFoC9CsDQ/f40AXCXAA2hAEMDjqlAYRvwGgTCN4ZmZBxeVVRUFezp6ZQdEhK4DiCP
    AUYAnkKgOhQgKECZyrIlhgc3RzkAOJRen/C80Wjur1Yr3mLZjAIg6AEKgbu1TADBFFDtA7S5
    AcIywLEAyHhgQvIfjv7vc5SVeWRVSUlNsLuHQ3ZQUMD/DEfdPCFdAo/nHZGZefTdC/GpA5ub
    W0UBgR5FTU2JPwB23zQ11phlcoceHwC9Xue/9ecjr9mr7J4aEd1vu5trwDd1uuJse1U4T69P
    eLGysmZKL7/gzwCX31i2giUIl/8ASlxufX3Ki0eOXFiZkpItjQj3b544aeiH9vbenwGw9Pz7
    ON7nz1/YlXgpTcmCBcsCfv7umDBh0NMCAb/gxPH4g9eu5fEIwqZ4h4X7mseOGzTBaDT5HD16
    6fvcnFIQBECAwIBBIXXDhg0eAyD93ouxAUAVarUlSLp8ESNHT4JA4AqgzSk3N+6zc3FXJ9EM
    zcTERJ7w8wtZDqzBubiTX3377W+h7f/C+elnpmuio3nXAY4gP//U2jNnkkaDJcjhIyIOBwQE
    r2DZqgqTqQyxZ48gLCwKjk4eACT3ZZ5gWQsADgkYwpKS0l87F5fs8PY7T12gqOAtQFrbv3fT
    rQLQqqmuOvP2+fPXJldU6DSOTkrt8OF9Dzk7e7/NsuVagnB9YByp1MqnRozou931v5ijo0cv
    rUxJyZJGRPg3T5gw+EN7e58HytHlxHQlQRBgGAa9fN1u4yg1NZ9HkiQYhkFoaK/bOMrPKwNJ
    kmBZFlEDgnuEo7y8uM/iz1+fRDM0M3x4vxN+fsHLgTU4f+7kVx999HNoYWEFvL1dnF999QnN
    iA6OTq+Ni0seDRbk0KFhh/3/j3HUTYHU2zPpcvzG9et3RigUUtjby7F508HA3NzSjxbMn8qh
    LZ6ftasXPdoIgmRTUnJQVFylPHsm6fnZs8eOGjV66JtGY77r1p8Prf7993OCxU9N/Wb2bE7T
    l+t/ONFiuAiRWNn+M1UgCLt/A0gNfY4dvfjq8WOXZDEj++PEyUQZC/bVefPUpwGk9Pz7eMSF
    i9f5F+NTeUOHRSAntxj5BeWYNGmQG8DH2bPJgqKiKk5E30AkXclAXUMjOXbciF4kaXY5evQS
    j6Zp+Pt74vz5q7AyVsGwYUPJzr23HoDYXSZ3mDFw8ERPgUByAlDE1dTkvfjN17tnt7YaARDI
    zi5++OWXeakuLi6JZ89eCfLzd8frry/Amo+34OyZK/7R0QOCa2vrR3377W8zmhoNIEgS6Rn5
    s197dX6li+uQN/n8hhFRA8aPEwi4xYBkL9BYep8DRtbVxS/PzCx+6urVLE1qaj5OnU581d2t
    ZFxIiP8yACX/vk1XJCwvT/943dqtC44eTYDJZAafz3MdO+7Gsy/+6zGBl3fEc+12mx7n6MqV
    LCQmZigHDgx5ft68iaNiRg6xcbT10Op9+2IFzz4z45tZ/2GOjh9PePWD93+UGQxtOHYsQQaw
    r86dq3lgHF1KSOV/vWEvjyRI0DSNseMGYNKkwW4AH7GxKYJffjnBoSgKNE3j8cfHdnB05MhF
    3onjiaAoCgzL4DlmRrc4GnQLR1pt/otffL5j9pEjlwCwSL2e+/DKlQtSnV2cE+PiUoLS0wvB
    4VBITy9EbFyy/4joqGCdrn7UV1/tnPH7gfMAgHHjB8xetWpxpYvL4P8zHHVLILW26h7Zs+dM
    RHhEAJYtmwUej4vU6zlYteoHQZ8+votGjnT9BUB1j1u+JXK90l6GQYN7g8/j4qsNuwO12oaf
    Xd3U9LlzKYJp00dg185TKnul7Itp06cc4fHFBYAkGXBNNZuvmVm2GATh+YBBalVlZBZK+/YN
    wKJFM9HUaEBOdokUMD8wrznLsmxUVDBee20hDh2KxZdf7sZ33x94mSIpJjOrhPPQ5MFY/NQs
    fPXVdpw8kcj5+utfP6cZmiwpqcayZY9g8uRomE0msCxYgO3kWZvrmpmZ8tO2rUdiysq0GDos
    fNHs2RPeLy+vjiwtrcGXX70IAHh+6acoK6sOdXHxPafW2JlSUrK5SZfTUVamRWiYrwkQmoqK
    ynwKCirw1VcvgiRJPL90HSoqa/opFJnL9vx27M0zZ65IHByUePzx8ZPCwvrMA1B+P6qwXt80
    7vDh8wFpN/JRVVWHnTtOe0yYMNAlJLjX+n+vQKrvf/pU4sOHDl0Aw7CgKAoWixVHDl9CUKDX
    w0897b4FwPkHwZG9Sg6LxYL4+OvIzSsLXFZT/7Obm4be8tNhQW2tHl9v2KNSqf6zHGVlFkoN
    hjZwuRy0GNqQm1P6QDkiCYKlSBIkSYIggLIyLTZtPvgyRVJMXn45h8OhOp7l5Zdzvv9+1+c0
    Q5NlZVp0PGMAgiS6yNHVn7b+fCimuLga0TH9Fs2ePeH9ioqayOTkLNz0zyenZKO8oibU2aXX
    ObVaYeJyOVyGYcHlUtCoFSZAaCouKfdJupx5U+3A1ZQcVFVp+8nlmcv2/nb8zeMnEiTOTirM
    fWLCfy1H3RJIdXU6n8oqHeY8NhY8Hg8Ag5DePnBxUaOgoNx+5EhG9iAEEpcrTHJ3d2yprq4T
    v/PO03Bx1eCjD7dIuFwupk8fjqefmYmGhmasWvVDgKeXU4CTkwp9IwIbIqOCT/j5+W4EPOJZ
    NscCuIAgJA9oWcurIiNDdNu2HnGsr/8S2dkleGTWyBJAmP9AzT/tNm8vTydERgahqrJOCgDB
    wd4IDvYCQKNPHx9UVNSivFwrB4DIyCB4eToBYMB28X3GNt3kbduOxlhpGo8+OhqbNh8SOzoq
    FwwcEJqrtJfj6w2/gaRIqFQKODqqSgCHq5MmDf2pprrumRdfXM8dOy7KMnny0K2AOsXFxbHW
    wUGJb7/dBwCws5PB1dVJd+HC5bm/HzgnWbhwMhIvZ2Dr1iMx3t4uUwB83f2RajG7u7t/+Ppr
    T+UeOnx6zvFjCco1Hy89KhQK97M0e+XfaZRimCa33NxSicVCg8Oh2k8vBKxWGnn5ZRKGaX0g
    3mguV5jk4+PaQlKUmCQIVFfVYd26XyTOLmo0NDSDy+WgrFyLVat+CHD3cPyPcdSvf7DuyJFL
    jvX1TVCrFQgN9X3wHLEsaNpm3MnJLsGaj7ZJb31+81nylWwkX8mW//VZ10gyGnWTt207ErN9
    +0kQBIH09EKxi7NqQWRU71w/P3dUVdWBJAn4+rrBQWNfAjhcHT9hyE+Vlbpn9u8/z33oocGW
    iROHbAXUKc5ODrUBAR6oqtKBYWj4+rrBxcVBl3DpytzPP98hqa9vBMsCDMv+13LE6fqEFaKh
    oaZKJBIiJ7sY4eEBAAjU1emhq2uEWqVoBQtTzy+USgBo9PF2advy8xFxcXEFEhPSIBDwodc3
    I6JvAEiSi8GDeyPpcgbmz5+AzMxi7Nt/1m7XrpOzJ04aMn7KlOgtTk7+a2g6p4plS3o0VJJl
    6wFoAcirBw0KK9+9+5TjwUPxCA72xpAh4VWAqI5lswBoQBD2PQ6S1UKjra0NPr1c8MYb8/9k
    ogHa2loQGRmE/v2D7vCsDdZbNsXONLPF4qGtaUB0TARiRvbHnt/OoqamXqxxcN7y3JKZkp27
    Tg45F5vCee+9p4s9PX02A0yrp+fADyZProuJj08NHjsmKt/LK+p9wNrq4uKzac6cMdNWrvzW
    cfDgPvSSJTMuOjt7fH/ixKUNYrEQMSP7wmS24Pffz8NoNDl33j/T0u5uEALwaHe0WsDleidL
    JEWBZrOVw7AsaMbaqLBzPt/SdKX5j++XAmgGYAUgAkH49vB6zgVAalVqOyNJEoLb54SAyl5h
    JEm+lmXzevTdNzny9nJuE/C5YqPRAk9PJzw6ZzTi4q62n9QIMAwLtcYOTz01Bamp+f8RjgYO
    Ci0PDPJ0PH/uOgICPTFo8IPliGVtypBGo7gtaq7zplACWq2+S9+1WCwe1dX1AACKItHaaoRW
    2yDWaJy3vPjSHImbm2bI+fjrnPnzJxZ7eHrfztGF1OCxYwfke7Zz5Ozss2n+gknTCgrKHQcP
    7kPPnBlz0cnJ4/vTpy9vMBhaQVEUrFYaNTV1MJn+OznidG3CtADUsLND/sSJgywbNx7ghob5
    wd/fE99/dwB2CgmGDA3LBcE1sGwegF49GLUklBXkX3rl8JELqrLSaiQmpmPUqEgsWDAJ5eVa
    BAR4AKCh0djB09MJw4eFY9y4IZg3bwKOHk3AL78cl1+7lr38+aWze4f07vMMWH1ezy3kOtB0
    HijKU11RkbLu6w07w+3sZNiy5S1s3LgfG77aFbX0+Uc+c3YOeZVhCrQsW9ejMHEoijx2LAF5
    eWXgcikwXQCCJAhYLDTy88vx8CMxpE1E3btJpYrYUaMjn96186Ti0MELIEgCA6JCbgDOh8Mj
    7JMEQsGhC/GpoWWl1WIAfFsoqklcVFwlq6qqQ3Fxtd1wNMkBryqgmSksrKTAAo8/Pj6uT5/B
    jwFq/ZAhYTlnzyb7LljwPswmC6ZOHdaoUNjF/f1cFAFoQW7uJfj5TbclaQB0VdVJlmXTAYh5
    dXWpn2zZsu/J2NhknrHNjFWrfnj0qcXTeg8cFPkIy2Zmmc0tANxu5cNqMCRCLLaDLZzX+z7X
    iwWtrckQibyrx48bqD129JJ7XrtTnGEY+PRywcSJg7UUJa82GuvBWhkQHLLnOCpIeGXv3liV
    yWQBQQBTpg6Fh4cj6nSNHbNPEEBDQzPMZgsWLpyM+fMn4sjhi9j2b+KosvLquq+/3hmedDkT
    HA6F5CtZ+Obr3VHPPTfzMyfn4AfCEcMwZESEH959dzEkEmGXv99saMPq1RvBMGynOZJIFLFj
    x0Q9nZFeoKira0RIbx/06x9k4yjcPkkgEBw6d+5aaFlpzV84qr4jRxUUzTB4dM7YDo4GDuiT
    ExHh75uSkgMHBznGjIlqlMt7mCNjz3DE6fxiYQGAaGm58uipUwnvHzt6iWsyWmAwtIJhaNQ3
    NKGiohY/bj44ZOrUmA3+AaGvAcUlPbNQWRiNqVN37T41UyDg4aOPn4NUKkRAgBfkcjHcPVxw
    M1HMz88Dq95aAIGQD4CGvb0Cc+dOQFRUMN5990e8++7GmDfeXLg5NDRsPsuWFN6vdseyVtRU
    H4CDY7/w9PSLn3z7zZ5RFIfCW6sWwsFRg7feUmLNx9u4H320Zf6zzzzsHhTcZ0VO9sZrLGsB
    QXB7YHQIw6BBYTXnz11T5BeU49kl0+Dv59EpHFiWQG5uCb79Zh80DnYYEBVSCjCNndMGNaen
    TBm+zNVFvTQtPT/86tUcrkgkaAZMLKBstFrBWCxWnD+fqp44aeh4tZqbCtDe5WValUIhRnm5
    VsHQzR4klZ7d0NA87UJ8qtpisaK1xWgCnOoBI8nn8QwKuRi9+4RZ/P3cM/r2DdjA46li/x4i
    TwIo7mdvrx6WnXWov8lkloklIp1GI08CpPGATn3o0JnHbtzI561ZsxRyuQR79pzBpk0HQtw9
    HJ9ycQn/lMerjKqqjB2lra3zsFgssLe3q3B3d7kEqA6azQV19x8OnQyRSNY3MfHUV78fiHNp
    bm7tUNwIgoDB0Ibtvx53aWkxfj9ocP9XmkxzLvQkR3v2nJoZF3cVAKBUyrBvXxzqdI0wGi0g
    SZvgI0kSZaU1eGH5FwgK9sZjc8ZgxswYRA0IwTvvbH6gHGVmJHyyYcPOUSdOXAbDsO0nCSt2
    7DjF1Tc2z3/uuVnugYG9e5yjIUPCa66m5Cg+/Ggr/P3cAbDt7/87BY8ASRIgQCA7pxRNTa0Y
    NKBPlzh66KHhy9zcHJamZxSEJ1/J5IqE/Ns4MhrNOHfumnrSpCHjVe0clZbWqFpajCgpqVIw
    tKGDo/jz19XGNvNtHPF4XIODgxIr35hn8fN1zwgP8+sxjtJu5PM+/PBZiERCHDp0/q8cVcWO
    0mo7z1GXTkgWS9pDO3YcXX/oYLxq6rTh+NeLc9Crlws4HA7efHMBbtwowL69scJ167bOfuWV
    Jyhfv4iFuK9Y/D9aWVnZ0OvXczgrVsxBQIA3vvv2N/zw/QF89PESODqqOxYNj8eFRqPCH0F+
    tgXl5+eBtWuX4qWX1uOrL3cNXf22bJWbW/izLMsa7+8UR0Gj6T395MkTH/y4+WBAn9BeeO65
    mZDLpQBoODlp8O57T+Hbb/fhvfc3xsyfN+nX0aMfWkHTvx7pGb2OXzR06IDHZVLhxxs3Hog5
    fPAi4fCkPTQa5d+aDgiCgE6nx9EjCYgaEMIuXjw1NjQ06CXQbcWddDrTxRVp24YOm3nM2eXC
    xr17Y6fuPxA3Yflylwk8nvK01WwiHJ2UaNA34caN/OkjR/b9prU1J6iqSiccPjwCWm29oLGx
    MdBOGZmQeuPXCfX1jXBxUYOmaSEAwmotmrD/QNxDmVnFWL780YO9fAc8+/qahbUfvvzZXSCq
    AUCJ9frEFUeOnFty+nSSprmpBRRFAQSgtJPN7RPqWxsd3bcpOTlLPmxYGIKD/QAwmD17FJ6L
    v468vJLHr6Zkjbt8Od2npKSa29ZmBsMyIAkCwSE+C6dPj4kNDQ151WyuSLkfoQCU+MXFndn4
    /vs/hufm2E5GJPmHQNLWNGDnjtPU1ZScgW+usv4wYsS7U1l2Yy5BSO97tVRUlA+Ni7vKIQgC
    kyYPxhNPTMC2bcfw+4HztrH6UzOZLEhJzkJ6WgHOnb+G11+fj7XrnsdLL37R4xypNb2nnzp1
    8oOvN+wOuH49DwRBgCAIUBQJmmbAMAwOH7qI8jJtzHPPPfzryJE9zdHAx2VS0cfffbcv5ssv
    dxP9+gWif2Rwh5C+86mKReLldCRfycKYsVHsqjcXxYaGdZGjyrRtg4fMPObodGHj7t1nbuOI
    tloIggBSU/OQll44PTq67zdtbblBJcVVQoZhUFpaI2hqagxU2PVPSE/fMSE1NQ8kQYBl2Q6O
    DnRwNLvHORo+IhxFRZU4cyYZTz01BRcv3LBxdDVrXGJCuk9RcSW3tdUEhmFAURT69O61cPqM
    u3PUBYFkFmdk5Cw9euSS6sUXH8XwEf07NnuAhb29AtHR/dGvXwBee/VrHDp0/qElS1zHAfit
    J+wMTU2tXiwLaByUWLt2GzZvOoR588dDrVb+SYNhQVutICnyT+ZCBk5OGrz22hNYsmQtdu44
    Oeu5pfZnRKLNv9zf5lIadObM+bXr1+/0fuThkZjz2FhwuTw0NOixe/dpPPLIKNjZKfDSS3Ow
    c+dJfPnV7gAAH40dNzKTZY1FxO3ug+5odgxgTQ4NGzj7jTftV/7008EnV7+1Uerr5waBgH8X
    7Y6A0WhCXm4Zps8Y0bJw4ZRNTk5+H7LQagmC30l7uQsaG67CaCzWSSSibHt7BQ7sPycZPKjP
    K4MGe2QxLMsqlTJ4eDjicmJ6yIgRA4fodPqQ+vpGTH5oKA7+fh41NfXBCruyEZcu3QgODe2F
    ykodGIalAb1DcvLVZXv2nBY5O9lDLBHkmY0VtW8ueRMk5XuXeTCTev21Fd98s2vVpYup1MyH
    YxDZPwgisQAthjbk5Zch4VKa+o1Tl9XFxdVwdlaBZa0gCALNza2oqa7Hxx9tVTk4KlVRUcGY
    OGkINBo7EASBivJa7D9wjnz/vU0jX3zpsU0DB0Y9wrK6PIJQdWvC6uvLH//11xPhuTlld/Tb
    kSQBkqSQm1uG7b8cDQwK9Hpc4+C0usse8zty1OZVV9eIceMG4OOPl6K+vhGp1/PvaFrncjlg
    GAYEQcBiseJmOPGaNc/j9dfn4dlnP+lRjmJj49e+++4m79KSalsINcPAwUGJJxdPxuZNh9od
    /CSuXcvFu+9uDmDBfjRmTEyPc7R6tf1Kd3eHJxMvZ0inTh0KRwd7sHcYegIEqmvqEB9/DYsW
    PdSycOGUTY5Ovt3nSCrKVipl2Lc3TjJ4cOgrgwZ5ZAEsSxAE6uoakZhwI2TYsAFD6ur0IaWl
    1SBJEuUVWmhrG4LlirIRFy/eCNbpGuGgsQNY0IDeISXl6rJdu06JnJ1V3eLokVmjEBkZBB6P
    A4OhDcXFVTaOVn6rLi6ugpubxtaPMi1aWoyoqanHhx9sUalUClXUgGCMHhMJjcYOHC4HlRU6
    7N17lnzv3Y0jX1rx+B056oJAqnNKS8sPcXRUYvCQPu0nkNsFAUBDKpViytRh2LXzNL+puSHs
    fgWSzclZ5ZmfVxIokQjRqG9BXl4Z5i+YgClThrVrdcxtg/rdd/vQu48vhg2LwO3pUDT69PHF
    3LnjsOWno8L+/QOfHxH9xHGWXa4jCFG3+qfTFc/Yueuk96hR/TD3iQkgSQIWsxlVVTrs2xuH
    oUPDIBaLwONxMGfOeFRXN2Dvvtje/foFPWqvkn5434YGQoPk2MPoO9xD5+QU8+b48br+SUmZ
    Q956ayE8PJzuKpBKSqqw8vXvMH7coOtOTjFvsGxGy9VzBPpFh3T63XK7CJsWbbxa7uqiZlsM
    rcTevbFRwSEBiwmCIAkQGDYsHD//fExYWlo6V1tT72Cx0ggJ8caBA+eg1dYFiMSF8uysIuGC
    +ROxc8dpcChK1NyUu3jf3rPDFHIJXF0dWKlEXMIXht1jnZQMPvj7mefPxV2lPl6zBOHhgbf9
    9uAQX0yePBRXr+Zg3brtOH/+GhYvfghqtQpnz6bAaDJj4aLJmD49GnZ2MthKr9i+7+3thv6R
    QVi79hf88P3+MFcXh5fc3DVL2z21XW3c4uKKAelpBX+red80m6WnF6G0tHKAxkHEBWC+P44q
    PQsKSgPb2kwYER2BuLiruHw5HVXVur8IJIIksOS5GcjOLsaxo4mgKFs49PFjiRg+PBwzZozq
    YY5KZuz49bh3SXF1h5AWCPhwdlFh1OhIHD2agIYGA8xmCyiKQklJNX7bc7Z334jAHufI0Snm
    zQkT6vofPZowZMmzayEQ8O76PaPRjLY2EyZMGHTd0Sn6/jgyXSt30NixCZfSiN27TkeFhPgv
    JtpLQLAsi4uX0oTlZWVza2sbHGpq6kGSJGq1etRq6wNEoiL5xYs3hCzLggULikOJDE25i/fs
    Pj2spKQaoWG+rEwq6RJHaz5ZAplUgpMnLyM3z5a2FBTkhYWLJkOvb8Gn67bjXNw1vPveYjg5
    2uP6tVw0NrZg1qxRmDhxMHJySxAXm4K69ijJYcMi8P77z+Dzz3fclaOunJCo1tY2jlgiAofD
    +VtlTS4Tw2qlYbVacX/CyAiAz6msPPPMqVNJTuPGRiEw0BOffbYMTk4q2OaKuW2jpWkrrl3P
    g7OLGnerHTt9+ggcOHAehw5diOjbN3i6VOb8Q/d6aBUVFlSMrNPpMXXqcJAkCbPZhC/X78Hl
    y+morKzD6rc2IaKvH/71r0chEAgxffpwvLxiAwoKy2PsVeGfowcSH/tFT7opjGmGZi0USUIk
    FkIsFt1VIInEQlAUCauVNgOwkmRIt9/P4wuK7e3l9NChoZyMzCIy9mzCYl9fN9JqpREY6AmJ
    WID0tPxJXC6HEouFcHFRQyYTo66usZ/B0GoVCPgIDPKCxWqFWCLsHRubGFBaWk1GR/dFa5uR
    EUtEZffaT6qqy+cdOnTBfsqUIe2Rn3+tvENRFPr3D8EnnyzFCy98gdOnr2DMmCicPpWEF198
    FI89Nq7jNP3nJhDwsXDhJDy56EMkJKROcHP38wTQnRBkwmg08swW6z39fARh852YzBYeQBL3
    y1FV1bVnDh2KdzI0t2L9+t3Q65vR2mLsMI3dqtRxORz07euPhvqmDtMvQRAwmczYvv0EYmL6
    YcaM6B7jqLi4YmRau5BmWRZSqQgrV85DULA3HDRKvP32YuTmluD997egUW9oF9YFKC6ufDAc
    MbBYrTSys0v+NjCLZVm4uKhhtTL3zxGPX+zopKJJkuScOpVExozst9i3lxvJ0AxIkkRebhky
    MvMncblcSq83gCQJNDW1oL7exlFebilI0mbeFIsFvc+dTwo4fvwySZIkHDRKRiTuPEfTpg2D
    Qi7BkiXrkJFRiJvWf5KMw/ZfTuDFl+ZgzSdL8K8X1iMvtxQjR0Xi2+/24/llD2P06Eis+Xgb
    zp5NQVvbHwHXO3ecxmuvzcXixVPw5JN35qgLoTsSg5eXS31pSRVqauphK9z3140OAK5ey4Va
    rYBUKtPep10XVmv6jL2/nZkvFPGJSZOGgsfjwtnZoX2R/HmzJZGfX4G6ukYEBXndZTNmoVbb
    Y+LEwbhwIZVzIy13DtDUzWQKmlddU6sQS4RQqRQAWHA4HIwaHYkpU4dBoZBgytShGDt2IHg8
    DgAGarUdRCIBtNp6NcDl4wE01kbKbT602z+252yPvVFRbqeU6S1WGjNnxmDPb2dkFZW1EoZl
    IZGIMGRIKM7HX5Pk5JQIXZxVEImkcHXVIDe3VBBZLs4iAAAgAElEQVR37ppk0MDekEpFYFmg
    Rlsv3b37tHLGzGiAABRyqYEghLp7gVRaUqGurW1AeLg/7l7E3naK9/BwwZNPTsapU1dw4MA5
    ODurMWNG9K2jd8fvOjgooXGwQ25uqQpo6W5tLNrZ2aFao7HrcNj/nX9Co7GDRmNf1s3T2G0c
    7d93dv6li2kEQRCorKhFW6upPdHzr4VCAwM8oLKX46Yv57ZTW1ohLly4DpVK1WMcabV1ipYW
    Y4eQNhrNOHgwHocOxaOpyYBDh+JxYP95GNs3OIIAmppboa19cBzd/L0UdffPvU65XeVIrbbT
    czgUDIZWbN16VFZZVSthcDM9w4SzZ1MkOTnFQpPJDIIgYDZbkZNTKjhzNlnS2moCQaCDo61b
    jymbm1tAUSTUGoWBIASd5iiirz8SEtKRmVkEkqTA4VAdyb+FhRV4e/VGlJZqsfipKTh1OhmH
    DsXDQWOH8eMH4r33fsSRI5dgNls6vmf7TW04eiwBcrn4Jv9/4agLo2lfGR4euMvJSY1P1/2K
    ykrtn8AnYDFbcfRoPI4fT8To0QMKREL5ie5rda0ASoJOn77wXmxsiuzpp6ZCLLmp8TN3EIQU
    6uv1+PLL3fDzdYePj9vfnuKioyNgpWkkJqaHs0x5n+7kHQAkKxQK2NYWI9rajB32/7CwAIwa
    1R+BQZ4YNao/IiKCOhau0WhCm9EEoYBvthVo/F9ofJ1GrWipqtQhekQElEoZft1+AizDorm5
    Fb379EJxURXOnEmGTCbGhQtXIZWKcD4+FQUFFYiMCoLZZKu0vGPHSUikIowdOwCVFbVQqRSN
    gLTyXj3gcCjbBkF1ZkkzGDYsHAQB7N51BpMfGgKBQHgPFw0LgsDNDZwFiO7Kc6uLi+vWyZOH
    tvB4nLsGnrAsCx6Pg4kTBzW7u7tsv+ORrwscnT176b0tPx+VWSw2v9mdBNGtm/Do0ZG4eOlG
    +4ZE3BYMYzSacfrUFTCMFTExfXuEIz5fwN6cu5s+q/j4VJw6lYTSshqcOpWEc+euwRaqbusP
    h6Ig+B/jyEGjaOHxOCBJEtev5WHnzlNg2xUXkiRxIT4Vp04mdey9BAHExqYgPj61Y55YlsXu
    Xadx9WoOSJIEj8eFWm3XJY4okgTNMB3v+LOlQafTY/36XQgM9ARFktix4xQmTx6CkycvI/bs
    VVDUXdZX+/rgcKg7ctQFgVTGqtWBPyx/4dGkpuYWvPTiF9Bq628TStu3H8Nnn+7A/PkTW0dE
    R35Mccw5trs7utO4gpKS3JW/bDvmO2nSYIT09r/DhkEAINHc3IKTJxPw0kvrEXs2BfYq+T3M
    iiy8vV3g5+uGqynZMq2uZijQ2J0+tvj28swiSRIJCentp0YCN/OhPv74WWg0dri1HHzCpTQw
    NAM3N4cUgDX8b4AkalFr7LU6XSOsNINnnp6G3LwyFBVV4eUVG/DhB1ug1TYgLa0Ae/fG4o2V
    3+C332KRnVWM+romfPbpDqxevRGlpdXISC/EU4ungsOhoK3VQ6220wGclnvtuy4uThVKpQxZ
    mUWdOkNKpRKEBHvDZLagT59enYgXIFGna4S2ph5eXs41gKSie2NVAi43+PRjj03a/uySaYxQ
    yP+LUGJZFkIhH888O42ZPWf8L3y+3ymgtNsclZbmrvxx80Hf6qq6W5zXdxaCNG0LYti7Lw5f
    frkHViv9l42FJAmkpuajuroW3j49w5GPj1uWp4cjmI5NkABFUagor8Vrr3yDygodKIrq6AvD
    MHBxUcPVRf0/xZFKrdQKhbYDH03TiD17FTfTAkiSgFbbgPT0P5QEkiSRlVWCmur6DiWjubkV
    Z84kw2ql283NPKhUXeMoPaMQw4dHICjYCzRN31FpSbtRgNLSagQHe6GtzQQvb2ccPZIAmv7r
    mmEYFiIRH1OmDIXFYkVVVR0878BRpwSS7c4LBd9kKospyC93MJstaG01wmKx3iaQmppbQdMM
    KitqOZWVtZMBeajZWAXamNuNySkPOHMmcSyXy8HIUf1x9Oh5GAwtt7yPgNlsweHD8Vi6dB1+
    +vEg+vcPZEZER8DQ3HrPDUkoFCIiwh9FRZUoL6sZAFDdSWawunt4bps0aUjDTz8eRnx8MiwW
    K6xWK0wmC4RCAUwmCywWCywWKy5cSMGPPx7GhAkDdd7e7tsA9oFodgRu3sly9w9BED14O6Og
    Sa1SVOj1Bqx683skp2TBTiGBVtsAR0clxk8YBH9/dwwa3BszH47Bhg0rEB0dgcBATyjtZejd
    pxesNIOioio4O6tQWaXD2k+2o6amHkqVtBxQ32uXYx0cHA4OHxHRundvHIqKyv5kUibalzp1
    y4cDdw8HODurYK+U/ek5ectY2b5vtVrxy/YTEIoE6B8ZHA8ouyyQWFYHwANATv+iorIQfYOB
    IO/iGiIIAnp9M1FaUhkOVIwG3NurO3Sdo9jYpLHJyVkQCHjw9nEGX8D7i1CiaQY8Hhfh4b54
    5ZXHmKBAL+gbDHfUcgmCQHV1HXKySyHgi3qGI3fPbTNnxjRIJKKO8jxKpRTOziowDAsnJxWU
    SmlHX6VSEWbMGKHz8npwHN0UfDRNg6aZO3zoDgHaUxwplbIKuULaMT9Wq7XdH3/rx/qXfvz5
    b2x7s03JkMlEUNiJu8TRb3tiwTAMPvjgafj7u8NqpW97D8MwaGszoaSkGk7O9nB2UoFhGFRU
    aMEw7G1/S9M0RCI+li1/GFOnRWPnztPg8TiIjAz6C0ece0OkB0CKGhrSV23ZcmB5XOxV4cSJ
    gzBu/EC4uKhvM58tWvQQwkL98OuvJ3jJKdkPLX1ullvffqGLAOO1LluVrS0DM9IL7ftHBuLA
    gXPIzSnFsGHhHZtEY6MBX321G9ev52LihME1I0b03W+nlPk8t+ST0X/vR/hjkwoI9ETTz8dQ
    VlbTq29fRgagrksbP0GAZZkT02fEvMyw9Meff75TpVGfAEnaTm1WmgFFkZBKRGDBQqvVY/JD
    Q3UzZ458ncP1TCQIWc/5jdgKAOCQJMGnaQZNTS1oamzGnYuuk2hqagFN0+BQJB+wclg2x3Qf
    91ixKrV9qVwuQVWlDkmXMxEU5IXFT03FuHEDIBCI8fhjY2Gx0uDzeQBIhPT2RouhDRSHhFAo
    QXNzMwYN6o3ExHQc2H8OpaXVEIkEUNsrq3CPyvG2edCfmzIlekdWZtGiN1Z+h7dWL0JAgDcY
    xgK93oCGhmbodHo0NhpgbDPBZKJx5UoWamsbsWv3GQiFfAiFfMjlYtjby6FQSCGXSyAQCNDa
    2obNmw7iyOGLeOFfs4t8fb2+AroWsXMzsdxqzZh95MiZT7/6ao9TXm6ZLanyzxFuBIHWViO2
    /nycSLiUPmD58ke3jhvftpzLdd/JsrYw4E5zRLcMTLtRaG8yWbBo0WSMGRuFFS99iYoKHSiK
    6Nj4evf2xiOPjKoZNix8v8JO6hN/Ye3ou73mptkuN7cU0TEDe4yjqdOiX2YY5uOfthxWGZrb
    sHbdUgQEeIGiCNA0i7zcEqxYsQEiER/zF0zUTZ8e80A5oigKgUGeEAr4dwkNAtqMJts6poie
    4UhlX6pWKVBUWAEejwuFQgK1xg4atR1UKjkUCgkkEhH4fC4o0jYurUYTmptb0djYgoaGJtRq
    G6DVNkCvN8BkMsPeXgGV0q7LHL36ygasXv0k1nzyHN5550eQBAFXNw00Gjso5BIIhXz06xeE
    vNwSCAV8ODoosfyFWdDrDWhuakVdfSOqq+pQVq7FzJnRmD17NH74fj8OH76AZctnFfn6ev+F
    o05E2cmJlpZrz2zevO+lK1cyue++9xRCQ/06bPG3njrEYiGGDe+LiL7++P67fVi7dmv4q6/N
    2xwWFvawxVRQwOX7dH6RkhRNcShwORRkUjGeXDwFfD4f9fV6CAR8rFv3Cxrqm5jVqxcf6927
    92rAznrk8O5dlVU6DBrUG51J2XB0VIIggJqaegeg1qmrINmajpHJBv64cKEdQ9PMtyeOX+Y/
    OmcMVCo5uDwOrBYrdLpGbP/lOMaNHWBatGjmy0DQz0AJ27MQOfJaW68tOx9/Lay0rAbvvL0Z
    fP7dlVWTyYKyMi3OxV8L8/XzWiaV+X7KsjnmbsLEKpWqk0OHhi64ejVHunLlPJjMFmi1DThy
    5CK02ga0GNpgMllgNtt8RSRJgs/ngs/nQiDkw14ph2d7YVg+j4vVb29Cn94+zRqN+kCnJhOm
    Nje3oLdeWvG47Isvdsz817++IMaMjkJpaTUqKnWwWKwQCnjgC3gQCHgQCPjgcEiEhfVCSnI2
    TGYLTCYzrBYaZosFDMNCKhHCp5crKsprkV9QjhdemF00YfyQJUBwcteTQPMAkMOOH4v74oP3
    f9LU1DSA+ht/l81kRSAvrxzvvbdJzbLM+kmTo+sItuBk1zZ7iqYoEnw+D2ERfvj9wHnodI1w
    cLSDrrYRBEFg/IQBzJJnZx4LDglZDdhZjx7dsyvtRsE9E0KrquoA0D3K0fwFdgzN0N9u2nSI
    f/lyBuLirsFoNIHP50EsFkAo5OHxueNMCxbMeKAcxZ1LCXNyUuKTdc/bApbuZOYkCNTp9Hjl
    5Q2IO3c1rFcvzx7haNiwsAVFRZXSl1+eg8jIYNir5BAK+R25WQzD/OWG2lsTiI1GE3S1jUhO
    ycKn637F8OFh3eZo6fPriBeWz8KGDSvAsgwsFiv0egOaGltgaGlFenohrlzJRFVNPU6dvgKZ
    TAInR3uIJULIZWKIxAIwDGC1WvDWqh+Qm1uKZcseuStHnL+fIAuAosijR2JfunjxBveDD55B
    UJDP3/hXbVFMEokIy5bNxiefbMWmjQfC33zTfqna3udVdCGPgiRFV/393JuSLmfI16xdCoVC
    gd8PnEWNth5yuRg11fV49dV5v/r06vcCQHDS0+M3bd16xH/UqP4ICPDsxNjbIsA4HAr19U1i
    wCzvlnmM0MBq/ZxtbR3smplZzJ0xcwRGjuyPlpYWWK22E1KfPn4wmky4nJjJbaivk/GxiBXb
    /9iDR31KYDBcX/Hz1v1vXLp0Q/DxR8/Czl52Z4hugUnf0IwfNx8ScTjUW088MZmSy9XrYLt2
    sotjQIBljacmPzT8QOLljLlPP70GAgEPPD4PdgoJ5HIxJFIRxBIBlHxZR4SkxUKjpaUNDXoD
    cnNKUd/QDLPJAqPRBJFIgIceGn6QL+h1tnN9cADLZld6e4c9/dqr/Mpdu04s3LbtmPSRWSMx
    e/YYODnbQy4XQyQUgOL8NTrqpv/EYrbA0GJEQ0MTiour8OX63ZDJJXj77cWXoqLCXk+7fvh8
    7zDHLm50LACd+vq1uHe+/HLXPYXR7Q5kEjU1DdiwYY/axVXzet++/a+ybLmusxf3kaToamCg
    RxPAyj94bwuamlswc2Y0XFzVWPvJdkTH9MVLLz3+q7d33xcAgpORcWHTz1uO+Ov1hnv2sbGp
    pYP3nuDIbJ7FmkwrXFNT87nNza04fiwRIpEAN+8gamszQa83IDU1j9vQUC9TKgkWPdpsHG35
    ed8b3323X9Cnjw9+3X6iE2NM4IfvD4goinxr3ryH7pujSZOHHriclDG3rq4R2dmlKCgoQ3lF
    LRobDWhrNcFstrkAbv54LocCl8uBSCSAws5m5uzVyw2tLUb4+blj8uSh3ePoNX7lrp0nFq5Z
    84vU19cNdXWN0Gob0NjUApPRDIax5Tzd7Mjrr33b4evicDiQSISwU0rh7uaIWl0DFAop3n3v
    6b/l6B4nJA5VUpK36MCBOOfHHht7D2F0+2bP5XHx9DPT8PzSTxEblzxv1iynnfq6ossK+87W
    AFNmjB49YF9KStaCV17+Cn7+Hki6nIE5c8bg4KELePyxcVd8eoW+BlipGzcSvv103S+T3Nwc
    sGTJjPbEOrYTsNuCEJoaDQRg7VZBLJZtAdDqlJlxYtqVpEyyrKwGB/af73AoAraIEoZhUFml
    I9MzCh4fOvStHY2612vlqp6p4syyzXP27Dn+xuZNBwWzZ48ESVFoajR0YvGR6NsvAD/9dFgg
    4PNWLlg47RqAbpZiaTN7eQX9OHx4xPQzZ5LEH320BBqNHUjSdp2C2WztsIezYEHAptFxubZT
    EodDgWVY1Dc04Y2V3yEqKtjQy9d3E9Bi7qxJhiACwLKZDc4uI15+Yh6PupKcvbRfv0BEDYgA
    YILVaoHRaIaxyQyLhb7NWcvhUODxuBAIeFAqZVCr7eHj44JffjmB8eMGlkdFxTxtNpenB4ZM
    A0E4d3l0jMbyCfv3nx2Sk1PaaWF0q1DKySnF77+fGxbg7zVDLBF93/lvKzNiRkbtS0hIW3Ds
    eCIokoRcLsali2lQq+0wf97EK97evV8DrFRaWuK3n6zZOuny5YxO9fHmptgTHAEAl/uN+sqV
    I9Pq6xvJn356A64uGvAFXJCkjR+zyYKqah2+XL+HvJGa+xjLVm4jCOeGnjshNc/Zs+fYGydP
    JAqeXDQZPB4HDM3i7xIkCBAYPLgP+vcPwpkzyQKBgL9y4X1zFPzjiBF9p6/95BexVCaCXCaG
    RqO0+dRc1JBIhBDweR2BDSazBa0tRjQ2taC6ug7Xr+eivq4Jzc2tWL58lqGXr1/3OHIe8fK8
    +Xzq0qX0pWfPJkOttoNGY4fAIE/Y2ckgl0sgFvHB43HbhSkLo8kMg6ENjY0G1Nc1ory8FgkJ
    abBYrPjwwyX35OgeAqnG58qV9HEESWDMmEh07RJYBiqVPSZNGowLF67bjR8/aCxFOCWh0+VP
    6DZ3j6B3X/jXHOGhg+cfqqquEwoEPOLkycvw83O3DhwYugHgWtLSLnyzfv3Oab6+bli2fBZk
    MmmX+skCMBrNXJPRpO7coi2BLZLI2u4AFwFs+bOJiWnh9Q1NCAv3xaBBfeDmqoFAyIPRaEZ5
    uRaJiRnIzi5BYkJ6xICosCW0RfEey15jbD3gAZCBINy7AE9hBxI0bex15UqGQCEXIzenDDlZ
    nY9sJEgSSjsZriRn8eY+MYFnteSBao9Q7EpVa4KwA8sastzcHEqMbeagY8cSUFWpQ422HsY2
    M6ztDmCWZTuKIBDt0To3hYHGQQknJxX0+ma4umqKAWUGQWi6qGUGgaazLFKpOMXRQYlDhy4g
    L68MZaXVqKtrgslkhsVKg6EZW1grazvBEAQBzi2appOTPRR2Uuhq9XBwVBYCTvmx5wowbkx3
    TDFWYUFB8axz565R3S33RhBAfHwqOeuR8knBIeN+6ry1gW5zdw9896UVjwsDAz0fqq6uF8bF
    XSMKCysxb/4Ea2RU7w0A15KRfumbtZ9smxYfn9qtCv09wRHL1sQkJqSFWi0MkpIy8XtVPBoa
    mmG1WMHhcGBnJ4GTkwoWK42EhLSwwYP7RhuN1/fx+Tfz6+6fo6SkDMHQoWGY+8REEEQX9hGW
    BGWrSs6bO/f+OXJx0ZS4uWmC3nl3MYKCvCAQ8G4mstuCKW4RlDdzpbhcTrvyY0ZOTglWvfkD
    nJzs74sjiUSU4uyixqBBvfH+B09DrbbrcAXY+sLcFhl5K0cMw8BgMOL06SSs/2IX1BrFPTm6
    h8muISwrs8i1T59eXd7oby7TyKhgHD58EVWVusF+/r15QOfuSiIIBVi2oNjPL2rRSyu8omjG
    7JJ8Jeux8rIa9/79Ag8LRZ4HS0tT133zzW8zAvw98PyyWe0lProe9WK1MhyzhXH9+7HIR11d
    GQB3PtAmbg+aIACaBcGUhIX7Z816ZJRnaWm18KcfD0OhkIBhbBnW9Q3N8PRwxMMPj2zrHxlU
    zOEQJXaOnkLAyP/DPcpv0dWeNtmrPEEQvTqjSQGwkwAqHkWVCAiCwBPzJmDu4xPBdiGHkgCF
    XbtP4vz5VPD5PAFBeMuAGgZo6EYorbhRaSepbWw0oLS0GqGhvhg1qj/slFLIpGLw26G6mbxH
    0wyMbSY0G1qh1xtQVVWH9LQCNDa1wl4lrwfUzd0yvFCBYNkCi6OTCieOJ8DNVYOwMD+oNXZQ
    KKTtGibXlhpA2MAyGW2mQkNLG+rrm1BRocP1a7kwmczQqJUGgKHHjRnaTd27waUgvyysqqqu
    29exEASB6iodCgsrQoJDdE7o5I2ct3Lk5+cVRdNml5TkrMcqKmrd+/b1PywU2jhav37XjPPn
    r4MgSHRXaN4fR1yGIPIEffsF5jQbWj0vxKcKU2/kg8vhwGg0QSDgw2yxIrRPLwQFebUNGNi7
    mOJAxCFCxT3JEUVR2PrzUezfdw5dKx1oC7cePKQPBIL750ihENVyOBQKCsqRmpqHwsIK1Oka
    YTC0wWiygLZaOxLgSYoCn8eFVCqCvb0Mnl4usLOTgqRIKHuAI2dnFQoLynD9eh4KCypQVlaD
    hoZmtLQaYTZbwdB0e19slgaBgAeZTAyNxg6+fu6or2+CQMiHWmV3T47+ViC1tjaIKipriVGj
    +gPdChJmoVYrwOFQqK6p5QcFy4muQegDU+u1VoKyxjIsjaioKfuiolqkgIOOYTKH/f772YcJ
    AM8umdEujLpxqRYAi8UClqHJu0NUC8Ckksnk09PTD0wpLi53b2xsBkGQhLOThlEqZYX+/p75
    wUHe1JmzSX779sZh4sSB2Lz5MObPn4hjxy5hzJhIxMREVTIMk3ftevYYrTZhXn29XsWyLGRS
    KTy9XEv9/T1/BwT7WLZWRxD3UjTlpMFQ+n5B/qWRVqvFoU6nR1WlDllZhWC7EAVLECQqK2rR
    qG+mEi6lviUQ5L7g5++RKJG4/qvr0r1WVVGpc3FyUuGFF2bBamVQU12HiopaZDeVoLXNCJPJ
    ApZhQQDg8rgQifmQyyVQqRQYODAE0SMiUFJSjcrKOhegUg2grMurjrXCZEpzrK7WYdaskZj7
    xETU1zWioaEJDQ3NqK7Woa3NbDOrsrZFwONxIBYLoVBI4OfnjgEDemPs2Ei8sPwLVFXrVCG9
    DSJ0L8kGQCu/pLSSZzZb7ksgmUwWVFbpJAAjvB+OIqOm7AP+4OjwobiHz5xJbtdwu3mC6zRH
    RpVMJp+enX1wSnFxpXujvgkESRDOTg6MnVJaGBjonR8S7EPFxl3xI0gCo0b2w5YtRzFv/kSc
    OX0F06cPR3R0ZCXDMHk30vIm6LRJi+rq9SqGYSGXS+Hp4VLay9ejyxwVFlwaSdO0Q0N9I3x6
    ucK3l3vXcp9ZAgWFZWhuaqGSEtPe4vLuj6OamgaXykodft1+Ek5OKjg62SM01BcKhRQisQBc
    HgckQYCmGZiMZrS2mdDc1IJanR7Xr+eiqqoOFeVaaGsa7osjsynNsaa6DgUFlfhl23E4OCjh
    6GgPf38PyORiCIV8cLg2ky1toWE0mtDSaoReb0CttgFnzySjrFyL5qYWaLX1KuDvOfpbgWSl
    WcpstkDA53XbLsvhcDpg6k7ji8JtOqY2BVZRTittZVplcgdOYUHRoosXb8ief/5hSCQSdDOR
    vX3ggb8vpMMRarVZX/74475ZFRW1pKuLGvb2chgMrUhMSEdzc0vvm98WiQT414uzkXo9Dw0N
    zTAazVjx8mPYuvUYjh1L9CEI+LAApFIxvL2cIJWKUF5Wi5OnEkOcndXjFi6YNsLBMWAR7lmb
    ywl5eRe8Vq/eGMKhSFgsNJKSMnD9em5XFbt2/w6NTz/9NcBqpfH2O4urw8P7d9lxz7LZIdnZ
    Ja5FRZVYtuxzNDe3gsvhQCYTQywWQCQWQCDg2eqVMSxMZjNaW01oaWlDU2MLzGYLJBIRyspq
    kJVV5Gq16oJY1lTW2crJt+h2nIqKqgHZWcUoL9fi1OlkNDYawKEoiEQCCIV88HicjkRLhqE7
    /FwtLW1oazOBy+VCrZajslKHnOwS39GjqnwBJHfXW240WoibpsHur1MWJpO5W4rX3TgqKipe
    dPToJZnVSnfZt9Udjmq1RV9u3PTbrFMnr5C1tfqOepcCAR9yubi3XCEBWKCwsBKeno6IO3cd
    jY0GnDt3DU3NrXj77c34+edjPiRJ+ugbDWhqNHTUS+NwKKhUdiGjx/Qf9+STM0Y4OHSSo/wL
    Xitf+zakslKH1lYj3ly1AI88MqaLewqF/fvP4O23N+PpZ9YEODmr8OFHz1aHh3WPo/S0Aldf
    Xzds+HoF7O3lIAhbZKzJZFOkrBZre7QqAQ6XAw6XAwGfBz6fB4ZhoNc34/nnP0PajXzXKVO6
    z1F5RdWA9PQCPPLISKx4+XEIBBwwDAuj0Qyz2dLRl5s+eQ6XApfLhVDIA5fLgcXCICurCM88
    vQaZmUW+MTF/z9HfCiSBgGeWySSordXjj+TCrulNNxNoZVIxATDITMtEUO+gLi92O03fWyat
    xjn1Rt5glmERFuaLm1UQutZIWz23ziW2WWna0pKTXUIOGRqKESP6oanJAKnUlg9AW2kYWtps
    V0+rFLh06QYO/H4eTy6ejCNHL8HBUYlVqxahrk4PmmEgEQvB4VAwmy1oamqBTCZFfPw1nD2b
    QpotpjaAsHam/yazhREK+Hj33SehcVB1y1x56//T1ujw1lubYDJbmK7PNaDX143MzCgU8Plc
    jBzZDwMHhsDVVQOpVNxudyZv0af/qLVnNtsueqyq1OFyUga2/3ICGRmF/FqtdrCDo7Ib5af0
    DjnZxaHl5bXo08cH48YNRHCIFxwd7SEWC9uDKDgdG7Atwo6GpV3Dq69vRmlpNS4nZiAzqwQ3
    buTbNTbqBnZfIBEWpVLGUBQJ9j7iwkiShEwm7qa14s4cpaflD87PrwDZjdqtHA4FArYAok5z
    xFhasjKLyZs35N70j/2/9s7vuY3qiuPfXWl3pdVqVz8dSbbsJE4cO+CQWDAtCT8aBhJKh2nC
    UGgnAfrUhwbK5KEP/QfaaRkmHaY8wQPNAIHSkMl0Ck2GFKYPDNBgcAgkVjy1LcWyHa1+rHa1
    Wu+v2wfZTtKH2JZcHlqdf2B3772fe/ace+73NBomKhW16dUWJZrGxibQtdhG/fjxv6FQkOH1
    enDhwsQN66i5hpbGtVrVkEpFactcPUe27U4YRv0AAAgRSURBVLiKUseSYOkfX3sP77//CdY0
    WRSFYrECy7RRMSzwvB+27bTO0TeTvmDQj3Pn/onseA75/DzKZRWapsMwTDiOe9PZp5/nIIoB
    RCIiUskYtmzpAc9z7XM0PnVHoSCDYTw4ceIMsuM5FAoyFEWDpjVgmhZcxwVBs9qQYRjwAR9C
    UgDRWAjpni70pLvA8xy+HLuyIke3dEgcG5nctDFZP3Xqo8DcfLmlcF6Wq7BtB4lEfM42FQfr
    Yo1APj8nqFoDH300unyYt1ZnOTk5g7purAImYiWTAy/8+Cf7977yyqn+L0azoOimUgQFgOVY
    CIIftu1gbrYEwzBx+PDD0wcPPvBOT8+GJ0+8eTZ98uSHSKViYLze5l+4YS6nrSgKmJst4Zln
    fpBNpzf9FiDW6higUFU0vP76GQhBvu1R1VQdVUVrPa0Eyto1MmDqDYP9/PxlyMUqurvjiMVD
    kCQBPM+BY9nFqKR5X6Km6qjV6pDlKmZmipiZKUIUAxge7icUTUdBDHrtnlaJ+vxcsK8vAdOy
    MTU9C4/Xg2pVQzQqwedbipCuOyTLcmA0FlCpqLhWrCCfm0cuN4+g4Ec8HoLtOL2tj6xU3NiX
    KvC8L1ar1VtOf0uSgJ7urhlgRZHMVXM0nZsTdN1YM9uu62K2IOPs2Y8xOVlYNUeJxMALTz39
    yN58/lp/NCpBFAOoVlUoigZFqaPRWFgW5bztts04dGjf9A8P7H0n1d315Kuvnk5fyV6F67pg
    GC/8/uYmHAoFEQoL0FQdsqzg0KH92Z41cAQCuISAEALXBa5cySObXbtU05JGICEELiEtd6+i
    QFmZzKB55uyn7G9+fRzqogLOkvO+nlq9frG5GVktnf1QEAQeqVQM939vpD2OfE2O3nrrAyhK
    fbEylbrpQveNa4eQpXchi+9Gwe9nEQoFsWFDZEWOVtjJxUsPPvTdC7btZiyztZRbJCxhz+47
    nK6u2PszXzsW1qPbMAjV3R2nQ6EA3n77XMt5b9clGMlsw+D2jfDcIl9BUREQQrL33ps5lkrF
    jgoCf1IU+VxN0WO1mjZYKikxRdH6inLFOzKyzdq5c+CzwcEtLwG3n3/sMf70wEDvz0dHxzO6
    bjCRsGiHw2IuGpWKUih4ORQKyKraSNcU7fHevuQxYEt2dQ6hhp7uLuvggfvNSkVdFmBsx3je
    jwMH7kd3Km61clwSCid+9+yRJy7u33/3oxMTV4ev5ueT/5osCGNjE6xpWlgwLdiWDddtpho8
    nmZ1HcN6IYoBM5GI6Hfs2FrYujX91ebN6b/7fL6/EJgthH3upd27dz2dTEQOf/Fl9q6pqdnk
    p59+HVRV3WMYC3Bs94YgjSw7CAIChvEiwPsQjUrW8HB/6YknHpwY2r7pAykktnFxTCrfPrz1
    H88//+S2crlGtVTFRggiEZEMDW3+EEiU18chESqZjNJ79gyjFYhomsIbb5xZM0f33JM59vLL
    8aNBMXAyGPTn1FojVlPrg+WyElOqTY7EYMDasWPLZwPb+l8Cbj9/8GCTo7GxKxldN5hoRLLD
    4WAuHJaKUki4LEm8rGlGulpVH0+nE2viKJWKWz995hGzVqujva631+dKFANIJmMtc3TkyI8u
    7n/47kcnJ2eG8/n55NxsWSiVFVavG9AbCzDN5j0gmqLAcgx4vw9+nkMkIprJZEzv7o4X+jd3
    f7VpU0/bHL34YvTwha+u3DU1NZssFORgSVY8iqItZ7+WorWlilnOx0AQeIRDQSQSEau3N1Ea
    2NY3MTi4cUWOqFsP7LsAhoYAN9XeFLmuUlkYY5ApT89cailld/N7TQi6XnpK09R4OymQpa/3
    cSx4PnCWZTOf3GqRARMBgOrZc/d3xv/05zcxnh3HA3ufAwA/UJQA1wtQDtBVtOzPbeL6QFEN
    MMydXqAYB1wP4LGBWA1A48wHvye3DW3HkeeO4vS7pwYAswBs0ihKWMUYfAOA7AAQx7oagePY
    RdclF1h21xrn5TJUdR7B4H00UI4C5R5Aj2uq0dswFlKWbcAyLbguoWmKIizHEJbxgePYghD0
    5QB/BYhOA9FSSf7QEYMSWF+mhfUxhYUFGRyXoQE5BpT6DKPWpyjagKbpXEM3JMsmmx3b5shi
    lRLj9cgs550WBJ8ZFISKJAnjoMRpIHn1r+/9Qf3+ww/B49nZ8gYFjCcBZ6itfBsIcR33Eu0Z
    nl2fjfM6R+2cG3U4+nY4qmtGb8MwU7bdjCJd16VpmiYs6yUsw4FluYIQ5P7rHNVq9YF6Xeca
    +sIyRwBZbFNByyznnQ4EfKYgCBVJDKyJoxUcUhaEGLBtG6TFnZ+iKLgOQUN3EY7etT5TTbKQ
    5QLm5ottN3UmIJBEAYnEBnBcZoXnXoTjWNB1B6J4J0rXzkMM0ajXXAREevnWsmURGNoCook9
    uHZtFH7WBeenAAqgCAWt5sJyXNCgEUvcCV0dBbNYN8JwI6scg28ANIUW11Pg8fpGR2GtIAFA
    qfQ5QAgCQlO81csAlmmhKFchlxRoah22ZcLLMBCEpm5cPB4GyzIwTcBDAZrqgmFoBEOZlr/j
    WuE8WNaGP+AF7aFAiI3p3DwmJ3PYt+9ZEEK8/7H+ndde+5W7ffsQ+vvTCIdE2E6z+CKfr2Pr
    wH1trtmLsG1zxR5IK6Ry4LqAzz/S4ajD0f8kR+sn+Nyxjq3CRr/4eDlvPzKy+1t99i+e/hg/
    +yUQDgM+PwuOo+H1AKYJmKaFSsVGw6CwY8fuzkR1rMNRh6OOdaxjHevY/6v9GxLPw3erfBqh
    AAAAAElFTkSuQmCC
  }
  set pieceImageData(Merida2,40) {
    iVBORw0KGgoAAAANSUhEUgAAAeAAAAAnCAYAAADXTpstAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh0vG88IZAAAIABJREFUeNrsnXd4VEX7
    97/nnO0lu5vee09ogST0Ih2kKk2RpoKAWLBiAUVFBawoRbCA9A6CQAjFEEoIJCE9Ib33bLbX
    M+8fGyIoICXheX7v41zXXkq2zc7MZ+577jbAv+3f9m/7t/3b/m3/tn/bv+3f9m/7t/3b/m3/
    C41qrw8ipASACQAHgB8oinpkP4KQ6wDY1ocUFOX5f24iCCkEIG79lxYUFfCIvrcAgBkA6YCl
    xQVFBbZjX8sB6AAwt3wuIbkATGhRloElgELhA8AOFOVz02tKARgBCEFRXv+B+c0FQLeOcxAo
    iv4vW38VAAyt/xI8Uob+5fdffv/l94E7VgxCMqFRx4MQgoQz74Fl00BI/qME2IOQot6EGAQW
    y2UQkgtCim+Bg5AMEJIJQir/C+HNQHHxJhBChIQQYWHhehCS8Yi+uwGEkA56NLZTH+tBSBaU
    yt9ACEFl5T4Qkg1CqkFIEwghFCGFvSyWzNUWS+ZqQgr7EdLMJaQShJSCkGzU1uwEIQRNTYdA
    SBYIaX6E82sFIYUSQsqGE1IaQMh/09pLBSHZsFgut/3NbE4CITmtgvFffv/l919+O4pf6uE6
    VoWjv03F6DE/h7NW9UhlizaKz+dWi8V2xwHHc0CdkaLCH8EAZU2qrKzcSAh13tPTZw/guptl
    y3Q0zQMgAMAXA6ohAJ1lMtUWsKwMQmHEfwm8TQCaxFar7uWysqpBAIG3t/sZhpF+AzDam7XA
    Dtr87I3GlvfVKq0zuWlliUQCiCWirwGqWq/Xf6BR64V/atkUJFKhXigUfgAQN61G94pOZ/hz
    UVEUpHbiOj5f9hFFBTXdX390ABpbT0MCUJRLq/bJdyVEO1upVEVKJIIcLlexw6A/XygQPg2r
    JXNWQkLSiiNHEt1YQjBubP/mvv1i3+dwfL4HqgBwAywW1TS1Whsmk9ll0rT0Z0BTQ1GRIKS2
    9eRHA3AARYnaeXxzAMyFVrv63fPn097393e/EBjYdQKgagGMoKio/+DaMwAoASByAlTDVC3q
    xwgBZHLJaUARR0hjPdEJwEhC/uX3HvgtL68aRAjg7e32H+K3dUcn5O/8avTCm98nkdyGX4oC
    iO0/HcVvi1IVKZYIc7hceRu/FkvmrMRzSSuOHr3gRgjBmDH9mvv0jfmf4JfzcN2zp0aP+X5s
    ytXLX+7Zc8q/pKQaYrEQw4bHvjBqVN8NEonjMqulQcNwHDsYAysOHDgjiztxedSYsf2GjBzZ
    Z4ynZ+CX9XUF552cO0s06uLlZ85eXujn55EfGRk+H6AS1Wo1pFLpfwPCPLW6/r0d24+9cehQ
    AkMIwdhx/QdNnTrcTi73WAqbXb8Dm1x8+XLixHVr93lbraxtNK0sHh/TB7NmjdkLSLQn405N
    27r1hOiGW4EQgunPDNeNHTv6K0DruWfvyaeO/HYeDGMzyzAMjRcWTCzt32/E6vs18xiN6eDz
    Y7kAKLU6wURIEQDiUlaWt2H7tuNj09Kuw8/PHbNmj5kQEhL7JFBtSElJf/3zz7e4hUf4gaEp
    fPrpL4oPBNwFvXpJ9gE854KCrG0//XQosrCwEp07BWL6MyN7+fgEzSOkolqnK4BI1JsHgOj1
    F82EZIGiItoJXgKgRATs6Gw0lk37+qsd/OeeHxsaGNi1FyC+BnhU/2fXXgEAXlBFec7a/QfO
    DEq+nM0QAvToETpz4hODz3h7+yygxIJHcAy28XsyLnnU2HH9hwwf3us2/CYv9PNz/+/ld8fx
    N/bsPsUQQjBp0mODpvwH+DWbraAAWFkWY8b2/ZPfk6embd1yQkRuOnVNn3Erv78dTgRD0yAA
    uFymQ/jdsSNubFpqPvz83TFz5uNt/KalZry+bNlGt6KiKgBAUlKWYtVqzoKePf/k95efD0cW
    FFSic5dAPP30iP9v+OU8XOeKO129mvntsmU/eHt5umDU6D6oqqzD2rX7xA31ypdmzBynbarH
    B63qUAc2AUVTDKqqGrBnzynemdNXJj73/Pgegwb1XmG11jns3HVi0bff7uaEhHhHfrDsue8i
    InuMv3z5xxKr9TJo2g4Ap119HffXtP5JSRlz9u49zbz44pOgKArfrtnD+Pq6zR4xwm0zgJyO
    /X4eqa9vthSXVOP558dBIhVix7aTKCmpAcCNACh5TW0TGhtb8OxzY2BlCTZtPIyamiYAiAW4
    7iUlNbBaWUyeOhgatR4bNx5CfX2z1aZL3+t6agag5vH57FMNDYnjrFZW4Oys2AMI91gsdc9s
    335ibFJSFkaN7oOEhFRsWL8vaunS52bKFZ4HMzILvHk8LpYvfx4UReGpacuQlVXk3atXX3et
    tn7aDxv2R1ZW1WPIkBj8fvQ8CMiY116bkSgU+q8TiWQT6+sTpxJCDM7ODocAu+1mc5GJy/Vv
    h7G10GpV9dvnL1x7saKyVtbY2IIrV3JcTKadu6OigtOCg8OnAfgP2lSdJfV12atXrvp1yOFD
    iWBZG6aJienM9evlQ5Ysmb3a0zP4aQCajuaXoTlISclDZmYR7/Sp5InPPT+ux8CBNn537Tqx
    aPXqbZyICL/ID5fP+y4iovt/Fb/JyZlzvluzh2lp0QCg8N13exk/f/fZw4Y9Gn4bGpWWy8nZ
    MJssbQp0l27BbfzW1TUj6XLWLe8aNiKmjd+ysjokJWW1KdBcHgeTGh57SH7JLfzu2BE3du33
    +0AIQWLiNRgNxqj337fxm5VV6F1eXgcu1yaOKirqkJNT4t2zZ193rbZh2g8/HIjcsT0OFEXh
    4sVMEELGLF7s+P8Fvw91AjaZVNN37TzpHRjgiRWfLoBIJAbAIjDIC99/t4/Ts1fnSd26OawD
    0MGavkSpsLcz+Pi4CD76aC62bY/D8g83eTc2KL8NCvI079oZx3n5pcm4eDED69ft67L4dcHW
    qKgeKTQtzQckVwHnFJZNM1KUGIArKOpRatZmUWlJldDOTozhI3oCALZsOYbKijohQESPpAsU
    IJdL8MQTAyGTyXD5UjZSUvKwZs22pQBFkpOzeC4uCkyZMgRmsxUHDyQgMTFdZDT+sg4gSEnJ
    Q3CQFyZPGoaWlhbs2XPqATrBZYzGxsVHfjuzbOfOOIHJbMHYsf2GTJ48wtdqNfmkX7uOMWP7
    Yc6cKfD2csY33+xGU3OLv1wRoPfyctGoNTrp779fBE1TUKu18PJ0VgN8XX19o0tmVhEWLZqE
    kSOHgMuhceJEEvR6bTghJa8dOHDy3X17T3MA4MlJj40YO3agK5/vugqAtR0GltYbDFGJiWmK
    3NwyqNU6XL6UTVdWNEidneVdgoNCHf+zAljXNyHh6tBjv18ERQEcDtOmXB8/noSePTsNnT7d
    vS+A4x3Nr4OjzMDjcQQsyyIx8RoKCyu9X3+95dugIE/zjz8e5uh0RiQn52DD+v1dXl3M/6/i
    t6y0WqjR6MAwtvHTag2oqmx4ZPxSFAWGYWClWVAUBYqikJFeiHXrdi4loEhqSg6PpmncbMFK
    SsoWmcy/rqNAkJFRCA6HAU3TIISAYZgHCKK18Xv0yJllO3bECYxGM8aNHzBk8uThrfzmgxAC
    DoeB1coiM6u4jV9PLxeNWCKUqlVaAIBUKoKHu5ON34YGl/T066BpGgxDw2KxIjOzqI3fgwfi
    392z5xQHACZPGTxizJgB/6f4fRgBTDU1NfoXFVdh5sxRrcLXAoBG3z5dsPGHQygpqXSMioq1
    63gBzGT5+LhVqtX6AC6Piw8/nIdffjmMr7/eyXN2VvCCg70xc9Yo9IgOxQsvrMTzz63o4+bu
    2Cc4yBsxMRGqLl2D411dPTcBHmeAfAMhRaAo/0cEsEjZuXNI4969p6UrPtkMmqah0egRGuZb
    CHAenYmSABaLLRI1ItIfDY0tSE3N5wIATdPo3j0MhBBQFIXo6HDk5pYgNTWPBwD29naIiPQH
    wNo+44GCFDReaWnZ89au3SsYP74/JFIRtmw+xnFxVswZOXLAyeBgbxw7dhEioQBnzlyBv787
    FAq7asClMDa284bJkwYv+erLHXyWEMyYMdLYq1ennwCHAgcHe1VQkBf27zuDlhYdjh2/hOBg
    b0gldqaEhOTZP246xJk6bSgIATZtOixwc3OcP2iQ824ARQ8/qHqLk5PrF2+8PqehobG+97y5
    nwXNmfO4ZuTI/sf5fCbOajLk4z/a1F65eaVCo9HcJnxvbOhmswX5+aVCUKZHEHLKZHl7u1ba
    ySQBymY1GIZBdXUDPv74Z15gkCevpqYJDEO3KgaXkJNT3MfV7b+H34jIoEYPT2dpRXkdAMDT
    0xnBId6Plt+bLJMUReFKcjaSL2dzbfN5s9XS1uJPJuNkXDLvxvMURd3y/IPwey0te97nn/8q
    qK62BW8VFVVxXF3t54wY3u9kcIgPzp1Lb7OyBAV5tfEbHR25YcGCiUt+3PQbnyUsnnt2jLFn
    z0gbv/b2qpAQH+Rkl7a9NzjIC1KJnelc4pXZq1b9ymlsVAEASsuqBW5ujvMHDvy/w+/DCGDC
    Fwg1AgEPtTWNuHnXbVFpYDKZIRGLWHJj1DpswakAcBsCAz2KxGJBwJUrOZDJ7FBaUgOtVo9r
    1xowYeIAcDg8hIX5IjjICxwug65dg3H5cjZ+//28nbu708QJEwcOGzmy31EHB6+lGk1uPmHz
    QNEdHXxSCMC1NDqmS8LYcf19V63cBgB47fWnEBvb5fyJHT5VFksGOJxOj+QUzOHQABjMnv04
    Zs4cdcvTNE0BYMDhAG+/PR0sS/7yPN36PP2AoX0mr7LSGlcOh8EzM0ZCKhXj6NHzKCqqVDAc
    TtKMGY/7//Tzof5ffb0DDg4yfPrpghyFwuEngFjs7DxXPfvc2NDMrKKper0Bs2aNPigWO34K
    UBap1OnnefMmjn3zzW+9vvpqOwY/1gMzZ4xK5PIkx/LzyydKpSLMnDkKFosVv/2WiLKyamfA
    7N4+AHPAsg1nZPKgEr1es53H5wQJBDyDwt7tJ8D3OGvJ/Q/HQwu0EomYpSjcNqdCIhGyAKN9
    FPz6+3sUeXu7BDQ1qkDTgFwuxWOPdUdRcSVYlsWNE5zJZEZoqA9CQn2RlJT1X8Fvj+guCU8/
    PbyVX4JpTw1DTEznR8ovBYDL5dxqNL7Z6XsHpfuvzxPyoJG5Jq/y8jrXhgZlmzKnVutQUlyl
    YDicpGemj/a3Wqz94+KS4OAox/Nzx93K77NjQ7MybfzOvA2/RUWVXvV1Sgwe0gPPPGPjt+B6
    5cTmZk3b9zU3qVFeXvN/il/Ogy08AiADCrnjqcGDe0zfvesUEx0dji5dQ6DRaLFp4yF4ebsg
    IjIwRachFYQUdKCPRgogb3D6tetRZWW1iDuRhKBAWx7h5ysXIuVqHjp3CgRAwOPx4OyigEgo
    wKJF02Aw6JCRUYhdO0/iq692SM6dS52ycOGkiM6dO72dc/34Ub0uC0JRRAfBW4aXnvXGtz9e
    f+Ly5dShx45dxDMzRoAQghMnLqFH97Bxw6dxz3z55Vv7HoVG39ykxsYfDkEg4N9REyatxFJ3
    QJSiKBgMRjQ3qR/El1UTFOxTzxLi9eWXOyCRiNDcpEZomF8DIDjpHxBwdNmyecvt7MQzT5xI
    okQivhKQVFMUB4SYDUplga66qgF6vRGNDSqIxYFGiuKDkKYCHo9To9cbvSZOHGh99dXpG3k8
    0aeAkOrcOUi9f/9Zh9Wrt4NlCfR6IwIDvWoAuure5rAKQDNsEZhWAEzrbkaBokIACMAwnhOy
    sy4t37bt9/CKijrs3HnSkcfjbhk6tO8PfIHrhwBMthx6Q+uOaGj9HA4Afoflk9qiO2VJfft2
    qdq7J96zvLy+Vcmy8e3p6YR+/btVURAnEZIDigrrWH7T86NqqhtBUYBQxMd7789Cr16d8f77
    G25Zj4QAzi72mDt3PObOHY/09OvYueM/y++V5LShv/12DizLghDg96MXHim/LMvCyUmON994
    GnKF9IFPshRFQalUY+XKbbDe97mJVxMQ6FXv5u7kVVpSDYqi4OQkR3CITxu/7y+du1wiFdn4
    Ff6d36rqBuh1t+dXpzN4TZg44BZ+IyMD1K6u9g5VVbZULA8PRwQEeP6f4pdz/wuvEUAqAHlg
    S0vtSA93J6jVOhw8+Ae6dA1FdnYx4uKSMXXaEJjNZg+BSDcaiNxHSAmhKN8OWH4VPtnZuSu/
    +WaXQ6/ekWiob4FMLsVHHy8AAAwdGtN6OmMBUBg9uk+rf8MKgYCP6OhIdO0ajISEVKxatQ1v
    vfld5JJ3Zv3Qv/9j86vL+YdNxjzw+CHtCG4TACUAO9G3P2Y8fTLu9CdffbXdqWevTnj99afB
    EharVv6Kjz760XPx4mnrFi9e6QTIthBSpAPLA8W0d5ECC+wVMsZkMuPbb/cgsrM/3N0ccb+G
    C5qmUV3dgIyMIjg6yGCvsKNtLol71gULIyICP3r11amf79t3VlFRXotuUcHoGdspyWwWlzJM
    rYXLdUjm8/mzCgsqcOliZqegoOBuhJT9ARgk9fXNXerrm2GxWFFb2xTq7UNLCdE1AzU9Ll7M
    CCssqAQ9nDbzeMJ9rMVQRnNCme49Ii726h3pe/FCBpycFHjppUmq7t3DVwLckrvPoQlAMSjK
    HYSoFYAqADDYAxwWEBcBDtUtLQl6oNotLy9r+fKPfoyUy8R46aVJKC2pwRdfbHfUG4wvPfHE
    sEtW6+XfAB8aaHQHtCE2ejkAJKWArJxlr+goSt7OBREKAIgkFkvVFB6PIwqP8ENFRf0trwkL
    84VIJBAZDM1TBEKPbwnJ0FBUp47hNyd35epV2xxqappAUYDZZMG5c9fw66/HkZNd0srvn5aY
    PbtPo6iwEmPG9sPQoTFYuWoRzpy5+uj53ZTxdHz8mU9WrvzVKS+3DAxDg6KAjIxCLF26wfOt
    t595JPwqFDLGyrK4XlCBqdOGQcDn4H5lMEUBBqMFO3bEwcqyUMgfjN933pn5+YH9ZxXlFXXo
    FOmH2JjIv/FbdFt+lV1qqhthMllQW/d3fosKq0BTt/Ib1T384vARsb6pKXlwclJg3Lj+qqhu
    YR3Dr1yMV1+ZguKS6nbl9wFOwCoA0tDMzNRN69ft65OdXYyAAA8MGNgNAIG3jytGjuqJhD/S
    kJiY3mnOnDEbxoxhPblc129TLhA2qjfVjjBYYTZnTtq7Nz5cIhVh+fJ5MBpNqK1tRNyJCxg4
    KAo8Hu8mWwvBwIHd/2J/sYLL5WDw4Fh4ebng3XfXY/nyH90/+0ywrkePLs0A91z7AlMHQO6s
    0eS/s39//Pwtm4/wxozph+fnjodIZEvVe+utmfhx0yF89tkWp6qqhm/HT3gsVCLxWQG6oa79
    N0BdY3R01x3vvjtrwXff7bUT8HmYP38iunULaTsR/bMGTpCeXoCPlv+EAH8PLHzxSVWP7hGb
    LCbVfWTysyyX6/DjqFEDCvv3j5568WLq9E9XbBampuZ2GzDQKQiQ5gBWvsVsgVDIx7lzaZLR
    j/cdZ28fmgDovCoq6j2lUhEYhkZ5ea17dIzKE3BUqlR1kxMTr0mEQgFYq5VrMRtdOFwPAJVh
    aam5PS5dzMKSJTO0PXp02iO1E+wGJHGAlb3zmmuB1ZoEhvEP1GqTX0hKOjSouKjKT63WSXh8
    LnF2ltcEBnjn+Pl7HQe08mPHEsMZmsKq1YsgldqBECvWrNmFvXvPSAYN6jHb0dFZUFt7dlRR
    YXmfktJq74YGJS0Q8ODp6VLXpWtotrub59eAaxwh2Zb2yKm3RatqpQ31+Su2/Hp0/u7dp5iG
    +pZbgm4oisIff6QhM7PIftLkwctnzRob5ujo+yohJQ3tqUTf4PfA/tPhOTmloCjAw9MJUVEh
    uHghA7W1za1C7da+qdU6nDp1FYnnM7B3z2m88MJ4DB4cDS8vV7z77rpHxu+hg6fmr1mzh1dd
    3djmowZsaXg5OaV45531TosWNX47duzADuU3NrrrjsWLpy1Yt+6A3d69ZyAS8iEU8e9ZCFMU
    Bb3eAJ3OCKFIgFdffUoV3eNB+e1f2L9/j7vyazJbcPZsiuTxx/uNU9jbJwA6r6qqBs/mZg2s
    VisqK+vdo6P/5DchIVViMlluy+/FC5kdzy9D4eOP56GmuhGjRvWBk6P87/wWlfcpKanxbmho
    pgUCPjw9neu6dg3NdvsHfh9AAAvlFRUZX61a9WsfmqLwzTeLERTsBYGAD4CFq4sDPv74BdTX
    K7F7dzy+/WaXvVgkeH/Y8P6pnbrX/9G+i49Gc3Ojb2ZmITVhwgDI5XbYseM4vluzB506B6B3
    n86tAvivjg/qNn+zIjjYFytWvICFC1bhyy+3u3/6qWKZj0/wFEKqGinKvR02HC2AFs+6uuu/
    fP/99kHnEq7RC198AuPHDwDDMLBYzQCxFcFYsHASvLxd8P33e7n518sWLVwwNcDF1X8mIU1N
    FGXfnoOoF4s7vffEE5xMTy+X99as2R362mvfYsyYfhCJBP9ozqJgMzsfPXoerm4OWPn2i/nR
    0Z0+BuWzjbVU3vMxmqKcQSzXWCuxnJZIgpMdnUoi1Rp9r+074gIjIgNec3QMmQdowFqtiOoe
    jNq6JmRlFj3er7//SsDkVVRUofDz8wCPz0VxSZWMsAYvim6oz8jIG1hT3YDYmDCwhLROvBOt
    VF58eeu2Y0HNzSrIZJI0qV3XRRZzhsZqVkEgvlty/TgwzKb+2dnJ3//448HIK8k5kEhEUCgk
    MJutUKm13gzDeHfuFDh86LAYY25OCe3j4wap1BakSFEMunYNwuHDibh6NefxnJzfH790MZPb
    otRAIORDLBbazOiNLR6OjnKPGTNGRY8Y0WcVjy/7gpiLjRTX72GZoQyG+kVbfj06f+33+xiL
    xXrLCfNms2ZtbRPWr9vP0BSeeuGFyXkisdcnaNeahzSam5t809KuUyzLom+/LnjnnVmQSAR4
    +aWvUFenvGM0LofDwGK24Pz5dOTklODVxdPwzDOjsGLFfCxcsLKD+c3/Zd26XYN27jhJ6/XG
    NuErFtsUaK1WD4ahUV3VgI+W/8TNzy9bNP+FyR3Gr0jc6b0nn+Bkenm5vrdmze7Q5OQcPP3U
    cAweGgPmH5RoK0tw+tRlbN16AtHRYXjppSn5MR3IL2n156emXUd2dvHjffr6rQRMXiXFlQqD
    wQhCCEpLqtv4zcq6PjAlJQ80TYGgg/m1WKFWab3pv/Dr7+eBqqoGPPfsCkyZOgSxMeE4dPhc
    G78XL2RwGxtV4PO5bfw2N6s9XFzsPWbOHH1Xfh9AAGt7xccnDayvU2LDhjfh5e3Rat7985TJ
    MDRcXR2xcOGTUDarsWvXSfuoqPD5PMYlCX8WnG0P84vQaLSGa9R6uLk54syZZKxYsRmqFi0W
    vzYNEokEf41GJ4TAZDSBx+feBm4rgoJ88OriaXjj9e+wc8eJgYtesp8nEnX6tH02HhGj0eS8
    sm7tzsGJien45JN56NW7c5so27L5dxACPPvsODAMjfHjB8Hd3RHvvLOepil65OLXpi+ws+vW
    rpsgRXmBtWZYTWbDttjY6Gw3V8dPvvhy24hffj5KxcSGgcvl3uXrKJjNZlxOysGgQVHktden
    n/Tx8VtisdanwFQCrqjL/fWF0wW/Hz+BkcOj9I4O8hYnJzkSEtIQH3958tSpjnsBvoUlBN7e
    rrCYrTh//ppfr97dBlMU41FaWsPzD3CHWCRATm4pz2DQdBIIGedz59L8A4O8IJOKbpjVrUBB
    /5MnL0w8feoKfH1d4eBgpwb4huLragRHPHaXDbgBgC4oJydt3dKl68NBCJYunYOu3UIgEvHB
    sgQqlQa5uaU4deoKPvroJ35tTRMiI/2hVmtbhTBQVFSFstIafLBsIzck1AfjJwxAj+6hcHax
    B5fHgdXCorKyHrt3n8Lnn2+R6/WG9598Ymg9ZfLd+PAzbnXJzb3+1J49p+8ofG92K1gsVuzZ
    c5oeMCBqavceTpsA1LQ3v02NLQgK8sJHH81FYGAgdu/+HVlZxbe1wFAUBYGAB6PR3HraZNDc
    rMaXX+yAj48LBgyI7XB+N6zfPXjzL0dBCNpSdwDghfnjQVEUVq/aDkIIaJqGwWDCls3HaJqi
    Rr7y6tOPjN+U1DxqxoyRkErvngmlVutw9Woehg+LbXd+HR3lOHP6Kk7FX548ZYrjXlB8C1oj
    rpsaW3DhQrpfbM8ugymK8SgpqebdKARUUlrNMxg0nfhCxjnx/DX/xoZbLDRWoKB/fPyFiSfj
    LsPf3w0ODrIH5nfZB88iKioUHC4Ds8kCnc6AwsJKnDx5uY3fzp0DwOVyIJNL4eXlguKSapSX
    1eL99zZwAwI8MHp0H/SIDoOjoww8Hg8MQ6G8vB47tsfh008335Xf+xbARqOyR2pKnqBfvy7w
    8nbHndOtWDAMF2PG9MWSJetQVVUT2qlTiKC9BLDNgV7rl59fHGIymeHiYo9ffz2OqKgQhIX5
    IjY24jbrnEZFRTVWr9qGVxdPg6/vDeXh1tPwkCExGD26N3bvPsX07Bk5Z8BAh12EqAsfPr+w
    2vPq1WsT4+Mv4733Z6NX764AWBDCwmolSL9WaNuWLOZW0xuNmJjOWLx4Glas2MwMGRo9s29f
    ty0AytrVjsB0AiGZAIyp3j6hn/WMjRhUXFwl+PzzhZDL7/6blUo15sz5BD17Rhp9fAI/BbQp
    Bq0QUlmXB+rLqBHDQQhhZTJZiZurPQR8HvbsPiWN6hbyRnBIeBIhBFwuB/37d8OPmw5zKiqq
    p8jsJA3l5XXo2TMCQgEfp09fhUql6dLQoJReuZLDmT17NNKvFYAQQjEM411UmDd79+54+7Bw
    X8jlEkgkkgKz1WS9G7ytuHCVyoo31q3dHW42WfDtmsXw8bl1DYnFIri5uWDgwCikpRXgqy+3
    IyUlHxkZBejduxs0GjVOnLgEBwcZXnt9GkaO7A2xWNy6Vv9cryEhvliyZCZkMjHWrd8vCAjw
    fKtHtPgsgIesTNXin51d7FdX23xX4XuzEK6ra0bb8joxAAAgAElEQVRObolf9x5R/u0lgG/w
    W3C9OKS2tgljx/WHwWDE5s37EReXBKuV/ZsAZlkWPr5uWLJkBr79ZjeysorBMDRomkZzswrr
    1x9Ely7BGDIkFqNHp3QIvykp6RP37z8LlkVb/zgcBhwug8DW4E+BkAeL2QqrxZaXy7IEBw+d
    YwYPjp7Z5xHw27tXp0FxcZcFc+Z80lZc447qmJVFdU0jpk4Z0q78yuWyEkcHO2SkG7B16wlp
    t6jQN4KDw5JuLHFCgIRzqZzJk4dMsbOTNBQXV7VGcFMoLamBWq3p0tColCYmpHFuGOEoiqIY
    hvEuLsqfvW3rCXutVg8HBzmkUvED8bvmu9fA43Jx8OAfSE8vgKpFC0cnGXr36oRXXpmKJ598
    DF+s3oaUlHw0Nioxf/4ERET4YfmHP0EsFmLhwicxenQfFBVX4PDhcyguqgJF0wgI8MDwYbF4
    f+kcKBRSrFu3XxAY4PlW99vwe98C2GzWM1qdAQ6OMvxzwDqBTC4BTVPQ6Y2EZkSkfeCtA8Dn
    qlTZ8w4dSnDvER2GwEAvzJs3DhKJCA4O8jYl4G8an0qL9PQC3Fy7+K995nK5eGbGCMTHJ+PQ
    oQT/qO5hM6RS5w8eXnPV9L6SnOvl7eOKAQO6tQpfgh07TuLsmRRkZ9sK0C+YvxK9+3TCjBmj
    QdPAoEHdsX3bCVy9muvTp3eP6PYG2La4I1vHtoRlWRCKosDhcMDh3H2JcDgc2yZjZQlgtVBU
    l/boDisUcQvsHWQIDPRCbm4pdu851W/JEp8AlmXBsgQx0WHY+MNBpKfnD+jaNcSibFbD29sV
    QiEPKrUWKrVmxPXr5YzJZEZMTDjSUq+DEEKxrGXu7t3xngxNo1fPCFRW1UMiFZRwmHvxmKk7
    X72aPfH8+XR8/PHcVuFrvY07A6AoGt26heKzzxdi0aIvsH//WcTGdsbFi5koK63FRx8/jyFD
    erWuUett1yGHw8H06SMQfzIZR44kBnTuHDLi4QWwkdFodDTLsvdcbIFlCXQ6Aw1QTHvze/jw
    OXeVSof4+GScP38NZaW1rSdL6jbvA2QyMaKigiGVim5xj9A0jSvJOTgZl4RJk4d3GL8pKXle
    TU3qNrMzh8Ng4YtPtir+trrPGza8jYyMAnz7zR6YTGbQtO3El5qW79O7d/cO55ewICajCYWF
    91YjgsOhwbKkffkVcgvc3BxB0zQyMwuxb9/pfm++6R1gZa2t80Xhen45MjMLBnTuEmSprKxv
    XY8UqqoboFZrRxQUlDP5+WWgaQpWKwHLgmJZy9y9e095XrtWAJqm4eyigFhy//x+smIeuFwG
    L738BS4n5dwSdLpv71n0iA7DW289g89XvogXX1yNgwcT8Mkn8/HH2RTk55dj6dI56NuvK9as
    2Y2DBxKgbFbfsrj27jmNjz+eiznPjsHZsyn47UhiQKfb8HvfdyoJhXZ6N1cH5OWWgWUt/yCE
    KRQX20LSHRzkerNZbX14eAkAJwDVo48fT5xVWFCBZ58dAy6XCx8fj1bhS+7AGoVr6QWQy6Vw
    c3O4C48EoaF+GDQoCmfOXKXS0/OnAY2e5CGvwTCblW7l5TWc0FAfCAQCAAQUBbi5OSA0zAd2
    dmLY2YkREuYLDw+nVo2QQCgUICTEFyUlNVyLRd+x1d1vOybkDo+OaRRFgcuR1Tg52ZvVGh1m
    zhqF06eucNPSsn0sVttNLQ6OCvSIDkNCQppdWVm1PUsI3Nwc4OSkAIdhUFJSpTh7NsWue1Qo
    HB3tQQgByxJkZOYHnIhL4s+cNQoGoxEODnYslytooSnBPfRM7Zx+LV8uEvHRpWvQP4yBLa7A
    3d0Fr746FWmp15GSko1DBxMwenRvPPZYzF9cN7e3Itnby9CpUwDS0wvQrGzu+vCjKzR6eDiZ
    uVzOPaWr3LA4uLg4GABK2578xsVdmBUXdxkMQ6OuthmlJTWgKOouwX8EUd2CoWzWoKys9pbX
    3cgRPnDgD6jVKoSFdRC/ZbWcW9OiCMrLa5GXWwaVSguVSou83DKUl9WCvel1LEtQUlL9CPml
    wDD0PT3a8VbatrngcGQ1Lq4O5hsFPg7sP8u9di3Hx2olba/R6YxISEi1Ky+vsW9sbGmt4gUo
    m9UoLa1WnP0j1U6rNbQpiixLkJl1PWDfvjN8mwIJuLjYPxC/Ud1CcOrUFVxOym6rAnfjYbFY
    cT4xHW+++R3Uah1ef/0ppKVeR3r6dRw6fA4jRsTiscHd8dlnm7H5l9/R0qIFc9P7OZwbZZHP
    QCIRIqp7yB35vW8BzDCy4337dam5mpKL06ev4Ebe1G1eidraemzefBTdu4fCzc3lTIuqsR2S
    +isBlIWlpaV/vvmX3+2enj4CQUE++PM+0TtBxqCioga7dp7CwEFRkMtldxXAHA4Xw0f0hFZr
    wLmEa35Wq7Kv7eaYh1ALWQKLlYX1L0F6gwZFY/Hi6QiP8EN4uB9eWzwdQ4f2uuWEQkBA/pvu
    sevwRmc7OdnpKyvqERMdju7dQ/DDhoNQqTSt48LBsKGxyMkuQVxcEhwd7CAWCSAQ8ODkJMep
    U1eRlVWMocNiQFFcUBSg0eiwYcNBREb4Y0D/bigvq4WDo1xH04Kse/SfEp3OQARCPkQiwT0q
    ISxiYyMQEemPn346gurqBkyaPLjV/Hsv76chtRNCbzDCaDS1wyWkdtkREQHnw8N9/7YO72Si
    DA/3RWRkwAVAlNd+/GZ8vmH9ATuNRt8mdO9mEmdZFj4+rpgydSiOHb+ImprGv93JStM00tML
    kJ5+HQzD7xB+b86PpSjbyWzf3jNYseIX5OaWIje3FCtW/IKdO+NhMVtuZfh/jl+ZnsNhQFE0
    amubsXHjIbSotG3FQmjaVts5Pj4ZBoMZtvxbWxW2M2dScOliVpuSdYPfjT8cQnV1U2tpSgaO
    D8ivUMhHbU0TWJb8zRJkUyAY5OWW4ssvtiMszBeRnfzx809HUFZWg2lPDcWhQ+dw8OC5OyqM
    NE2hqbkFVqsVCoUUBqPptvw+ANDyzH79Yn4dM6av5eOPf8YPPxxoNefe2omkS+lY/Oo3MJut
    mPbUiKt8vnQjQLfDChSK6+tLl37//Z5gPz83TJgwAHcLErL9RAr5+cVYuvQHFBVXoX//bvdw
    oTKLrl2D4OPjikuXMjnV1bUjAMVDbYA8nkzp5enC5uaUQKvV3zRmNuVhzJg+ePzx3jcpE7bf
    oNPpkZNTDDdXBwuHI6j83wBYyLq7u1gbGltgNJnx3PNjUVRUhYQ/UqHXG7F/fzxKSqshEvNx
    6OA5mM1WvPPOerz91lqAonDkyHkI+DxotTqcT7wCi8WKhIQ05OWWYt688bCyLOrrlfBwc7IA
    jO5e++Tm7kT0OgPUat09nhwIuFw+hg2LwZkzKfDxcYW/v8c9WxAIsaKxQQUHexmkEomyHQZW
    4+sbsHb+wieVnl5OdxUKtoIcznh+7oRGX1/vbwhhte3F7/p1e4Pz88v/0Uf5143x2LEL2Lb1
    hK1iE/X359VqPS5cyOgwfj09nNlb06LQpjgk/JGKc+fS2pSBv77uf49fZyuXxwFAQNMUzp5J
    wblzaW3jRVM0qqsbceiQrYjJjeFiWeDIkfOoqKhr26dpmkZiYhpOn75qi4gmBDweB+5ujg/G
    r0aH0DAfCIV3LjxE0zTO/pGKtLTrGDmiF06fSYGXpzPs7MTYvi0OZpPljm4cQoDgIG9wuVw0
    NChhr7C7Lb/3tSBtpdcqrSKR67YX5j1RNGhgFL75ZhfSr12/6aMo6HQGfPHFdjQ1qbD0/TmG
    sLCgHSWFVKWjA/8hzVctAJqGx504P7G4qAovLnoSQuGdTiI0WKsVGRn5WLHiZ8yfvxJVVQ2g
    aapV+P3zxungIEe3bkEoKqpEUVFVD0D9UPcqUpT8fP8BUbWVlQ04eTLpJuuB7TFgQDQGDor5
    y99pxMUlobysFtExYYUULb3wvwGwvMxeIS3W6wy4dq0AAQE+ePLJQWhoaMHpU1fx+Wdb8N2a
    vbh+vQJNTSpcu3YddXXNqKxqwKVLmVA2q1FcUo2Plv+Mt976DidPJqO2tgnjxw9AZKcgFBVV
    oUWlhVwhLQOkVffWJ7u8iAj/MrPFiksXM+/DdEcQFuoDmZ0Y4RH+YBjuPZ9+q6rqkJqaj67d
    gi1yuSzx4fiphy2P1fns8GF9T7799jNwcLC77QZECIHC3g5vvT0dI0f2OU1RgksUpYHtermH
    4zf+5IWJp09fBUVRsFis/6gEWCzWthSPTRt/Q12d8q5m6qtXcqHVajqE3759u9Y6Oclv8Rmy
    rM2qtXNnPHZsPwmrlf3b8w6OcnTvEfo/xa9cJimWSoRtypHBYEJ1VcOfQosCLBYrmlprOd+s
    rCiVGlgs1jahTFEUKirqodcb294vFAkgk0seiN/z59MxeHAsJk957I51sCmKgk5rQGpqPgIC
    PSARCxARGYCKigaUltbccQ1arSy6dQvCs8+NRVNTC65ezbsjv5x7h0cJZfN+yBX9+xQWpi3f
    vv14QOL5dER1C4ar663+VB6Pi+iYcBz7/QJWrd4mmD591JK+fWMcAMVHxJqlp5gHLQ9HMfX1
    lY8fOZLIGzOmL3g8Lj5avgkzZ46Gt4/7TadG28a1ceMhnD1zFf7+Hpg3d4LO2UUhePuttfes
    dNA0B127BWPfvj9QWFDu3b+/JqR1B3vAZl/QtWvE2ieeGPj+N1/v5hkMJgwf3hMMQ0OnN8Bi
    tgIE4HAZCIW2tJa4uCSsX7cfT056zBwT0+kHwLe8w7xHpB6AltwMwj/5+NsUwNYqhk1NmbC3
    j2yH3vD1Ts4OWpphsOTttegWFQKT0QSGoSGXS/DGG0/Dy9sFf5xNQVOzCsHB3hg6JAYGowkH
    DvyB/NxSSO1EGD6iF+rrmrFq9TZwOAzKK2rx4Qcbcf58OixmK5ycZCpAdo/X7THlkZFBhwYN
    jFq8YcNBhEf4IyIiEH8GUVF3cMlQkMmlkEiEcHGxh61U3d8Fx82pfAANjUaHNd/uAShg+PDY
    LIoWX3jwuSWw3f8r8FIqs946ciRh9O5d8VCp7nx4UKt02LTpMJQt2tHjxg5aL5d7vUtIc3F1
    cSbc/CIfiN+G+srHf/stkWcwmKBQSBETE4Zr6YWor2v+m7mWZQlkMjFiY8MxfHhPnZOzQvDK
    y1/Ter3xLkKSRlFRNaqqGhAU5N/u/HbpGrH2mRkj3l+39gDvRg6wt7crPDycIBLxAdiKWlRU
    1KGioh5WKwuhkI9ZM0eao6MjHw2/lO0IZrVa/1nPowAOc6MEY/vy6+hkr5XJpaivV7YxwbIE
    hLBt+wtF0aBp6pa95oYwtK0BFjdFQbeefm3PyWRiODjYPSC/hxAREYA335wB1mrFzp2nWpXB
    P60aN76vpqYRQoEtR9/VxR61tY0wt7oXbqxT2yUYtpNvdHQYVqx4AW5uTli5cgssZguGD7s9
    v5x7hxeQK/qPTk5OXrdq5RYvgMKbb05HbGwk7OzENwk/W/TmK69MxZgxfbF163Es/3Cjw+w5
    Na9PnjSMIxTJPjSZ8rU8XvCDACxvaFDF1tY1o1tUMNavP4DGRhWkduJbTg0ZGfn46KOfwTA0
    3nhjem2vXp33KOyd6jdv3vcuw2F4Hu5O92wCDPD3AIfDoKioSsKyhoe8GcZsFQqdvpgzZ5xI
    rpC+/PPPR0XbtsWBx+VAo9HBbLFt5BwOA4lYCIvFCpPZjBkzR+smThj4jUTist5Wiawj4NXC
    FpwplAKgLWabZvpPfsIWpQYW20XgNEBJFQoW+uZMCBUPCzFlkstleS7Oin5iIR8uzgqYTGYs
    eWcGBg+Oho+PJwAKnTqF3DSXtr6++OLk1v9nWi0zLMIj/BB3IgkFBRUwGc2wWCxwcpJDIZeV
    AkrzvboBxWLntXOeHdevtLQm+s031mDZB88iJqYTABYajRaNjSo0NChRV9cMpVINvd4Io8EM
    ZYsWTc1qnDiRhLLSWvD5HIhsGjycnRVwcJBBLpdCJhODYXioq23A6tXbkJCQirffnqEMCwta
    ARx7iNt1cgFIA6qq8jd8992Owfv2noXBYGrd/G6fb2uxWJB+rRB5eWWi/LyyaQsXTnZxd/d8
    3tWXesBjMCVvaFDHFhdXg8tlsPDFJzBkSDQWvfgFalqrSd3Y1CiaQt/ekZj+zMjanrGd9ijs
    neq3/Lr/XZVKw7tb9DZFAS0tapSX1SIoKKBD+J01a6xIIbfxW1vbhPfen41+/brhhgubZQku
    XUrHS4u+gqOjHLNmj9aNHzfgkfFLCKFFYiHCI/za7ti94y8yW5CdVQyAdAi/ri72/fLzysDj
    UZBKxXB2VsDVzQFurg5wcJRDLpNAKhVBIOCBy7ENoMlshbbVzdPSokVzkwq1dU2orm5EXW0T
    VCodzGYCF2f7h+B3Q/TixV9j6dJn8dZbMwHY4gcCAjzg7uEER0c5ZDIxhAI+fH3dIBTxweVy
    IBDw0L1HOL7//jUYDGaoVFo0NqlQW9uI4uJqAAQffvg8JBIR3nt3LS5ezMRbbz2jDAu/Pb/3
    eAKuAEC6pKdf+2758o1ewcHeePPNZ+Di4ojbBz7ZIidDQ/3wwQfPYd++M1j7/T4uCHl1ypSR
    Ro1e8UFifIq175Co+5xULsvlcAw8LgcUAA93Z0x68jEoFDIolS2ws5MgP78Y7723AV27BLHP
    Pz/hmKeX3weAa2F9fcoPx49f5HXuHAgPT+d7FMAE9g4ySKRCVFXXQ6XSPtSqpCgxCCnQ28k8
    P5w1a4LRZDJ/sG/fWTz11DC4uzlCKOIDFGDQGVFVVY+ffjqCiU8MwuzZ4z8HxCsBg4GiZB0A
    L4Ht3nBpeEVF3tLTZ5L5JaXVWLToS3D+wUdnsbIoKa3GqTNX+H36dVvq6+tfJpAbs0sv58In
    JvRhusXK7BQnhw6Lmbb112PiaU8NhZeXK+rrm1Fb24SUlFzU17dAq9FBrzfBaDSBbVUWOFwO
    +HwuBAIuRGIh5HIpPDwcMXZsP9g7yFBVVY9Fi77AsGExWnsH+30Az3Jv8+cAQvILg4MjX126
    7LlvPv98S/dXXvkGkyY9huYmFbJzSqBstl1EIRLxwefzcCPgg8vloF+/LmBZFtfSr8NoNMFg
    MMFitsBkssBssUAo4MPDwwmhYb5ITLyGmupGvPvurOYRI/u8yeE47QMWAlj0APOrAmCSNjbm
    rf76q62D9+w5bTsP/MPc2u6ZpWA2WbB92wmYTObH3nh9xncKe4/pOlVmk8gu8r755XA5Bg6H
    hkQqQpcuQdiy+Riyskrg5CSHWq2D2WwBl8vB9OnD2Tlzxt7C78kTSTyLhb1rv23R0BbU1jV1
    GL8ymeeHM2aON5pM5g/Wrt2Pq1dzkXw5G2qNDmitiMXncyESCTBt2lDMnDnukfIbfyqZHxMT
    hi++fBli8d2jg7VaA15/7VvEn0rm9+7Tpd35HTYsdtrVq7niGTNHYuyY/vDwdIREIgKHw21T
    jv+eWXGrC852zakZGo0OFRX1OH78In7+6Ui78PvyS19iwYKJeO31p1rvQqah0xmhVKqhUmmh
    UetQUFiBy8nZUKt1uJSUBQIKYjEfCoUdAoM8IZWKWgue2JTZoqIKvPnmd2huVuOdd2Y1jxjZ
    +4783qMAFkprajKWfvP1dl9vb1csXfosZDIp7n7nMWk7DU+ZMhQUgLXr9nO8vV3nDhzU+2if
    wU6X7n9OGaWDg12Ki4t9VEpKPl5cNBkMw0VpaQXWfr8XC198El9/vROREQF45ZWnNinsvd8F
    fJqMhpQlu3fFTSguqsIXX05qLU95b1XWxGIBxCIBmpvV0Ov1rjfu23xwiANBiMmq0dR6pqVe
    x6RJj2HatJGwWo2t9/Ha8vIYhgeNVo+rV3LR3Nxsp1BEGh7me+/eqgBII8rL835csWJTrFqt
    x8cfzwOPx7kXCxbMFit27TyJzz77pdeSt2f95OPr95x3ND/z4TY7CoQ0HR4xos/Rs2dSJr/+
    2hqIxAI0NalBUYBYJIRIxIdILLAJOwHftjETwGS1QNmigU6nh05ngEFvglqjB2EJ7O3toNcb
    4O7miFGj+hzncFyO3V9pQG8Qknk+LCxi+ocfzF2z5dejgzZtPMyEh/ti7Jh+CAryhLOLAnK5
    FCKRoK1QxJ9zR0CIzS/IsgQmkxlqtQ5KpRrl5XW4fDkLa7/fh+49QvHZ5wsLYmO7LKMo/x0W
    Sznhcj0ecDR5tNlc8uzOHUcf37//bNv43t9cEBzYfxbe3i7D5s59YrrVLFqD+85FY5QODtIU
    f3+PqISENCx5ey1qa5sREuKNl1+ehE8/3YLy8npMnToEixZNuYXfvXtOTkhKyr6n+uSEECiV
    GgCkQ/m9ejUPjY0t2PzLUXC5HDAM0+YDNJst0OuNuHr10fL7yScbY8+dS8OTTw7C4cPncC82
    aBdXe+zdcwafrvi51zvvzG5XfocN73X01KnkyWqVDmKJELk5pSgrr0VNTSNUKi20GgMMBiMM
    RlOb35yiKPB5XPD5PEilIsjkEri42MPL0xmeXq6wWKzo3j0UI0f2fnh+txwd9P33+5mrV/PA
    5XFRUlyFxkYVlEo1jEYzrFYrrCwL0mpm/nXLMWzbegI0TYFhGPB4HNjZiaFQSOHp5QJXV3sk
    JKTB3cMJb789syA2tvMyigq4I7+cf17MjQDqR/x+9NyYysp6rFv/VqvwvdcyobbayxMmDsK1
    awXYvPmoc3h4wBKBgJ4GQHd/08oShb3zwQkTBkz5+uudUrFEiC5dgrFr10mIRAJcupQFZbMG
    b77xzEmFvecywKdZr7/y2q+//rZk69bjzPwFT6BXr0731XeGwwHD4UCj1kGvNz50IQJbBaAG
    38yMvJGpqXng8jhY/OoXaGpqgaG1vJ6Az4XCXgaWtSI19ToyMwon9uvvtlaryigS23XI3aKB
    5RXXf16x4sfo4uIqLF8+F8HB3vexv1Lw9HDCsqUb8cmKn2PfeWf2Bm8vv8mw5Zw8RDMZXF0D
    1j85afCI999bbzdr9mj07dsVbm4ObWYrhmFA3c7teqPiDktgNNlMRXV1Sly+nIl1a/fjvffn
    qNw9vNcBZsP9bSwC2K4fU+f6+IZMW7xY8XVJcfXTfv7umDFz7G18unfOSQcAgUAEOzs5PDyA
    iIgwuLk64FT8Vby4cFJ5bM/eTxuN+ZcZmgKX9zDz3uiVfi170datJzj/VH7ybhuqxWLF9m1x
    TGxs5OIe3bsfB5B/3/wqnA9OnDhwSkpKnvT69QpQFIUhQ3rAaDSjtrYZ3boFY/bsMbfwu23b
    kSVr1x1gTCbLPQpgtKa1dAS/KgAN/hnpuSOLiiuxdNkchIT4wtFBCqFQAALAaDChqUmN/Otl
    2L0rHunpBRMHDHBdT0hJQcfcCGfj97PPforOyS5Bj+5hKCupQUnxvXksaIpC9+6hyM0txaef
    /hL79pJZ7crvpMlDRrz37jq7U/FXoFJrQQAIBDzw+TyIxTZFms+zKTAUAJYAJrMZer0JWq0e
    RoPNWmS1WiGTSWAymbFs2bPtwu9rrym+LiuvffrAgT9aLT4MRCI+JBKb5UwsEUIo4IHH49i4
    IYCVZWE0mqBr7Z9eZ0BdXTOu2artwclJjs8+XVAeE3uDX/qO/N7DCZjLr66qmHr09/PciRMH
    ws/P4z4E2M0maS5mzhyJ+QtWIyU1p/+IEa7hQPwVYMh9DB4fhDSdGD5iwFqlUv3yzp3xgu3b
    46BWafHSy5Nx4MAfGDe+v9o/wP9zwK/OYEh+eeuvR5Zt3xYnfvnlyZg0ech95F/+uY9SwA0z
    pwhQcXB/93T99VdIrdam106fvuKuVGpQV9eEwEAv+AfYouwAQKszoLa2CUWFlVCrdTh9+opv
    dHTkC4TlfwKgpf35ZbpdSc6MPnH8IpxdFPho+U9tgRL3Pjc0mppUyM0txcCBUYG+T4U9bN0/
    UJQrCNFcCwjwKnZ0lHeJjAyAQm6H/PwyVFc3oqFBCaVSA73OAIPJDKvV0iYsBHwexGIR5DIx
    FAo7uLk7wtvbDQEBnrC3t4Ovr9t1wO7qgxTppyhf1NWlQS5XN4jFsiO+fu5TiwormZqaGiiV
    GlRV1aOhXonmZjVaVBrodEYYDSawrWPKMAwEAj6kEiFkMgkUCilc3Rzg4eGC7JxiCIV82DvY
    XQT8rrCsCgJBp4cSGFZr6ePHj1/wrapquK+0n79t1DSNqqoGxMdf9u7aNXSE2ZyXz+WG3De/
    w4b3X6tsUb+8desJQUuLFocPn4NGY9tHp00bqvbz92vjd/v235d98/VusVqtu0/FgXQQv1JY
    LKUTT5++4h4c7I3hw3uiuVmNunolbgSHCQR82CukGDK4B7KyinDm9BXfnrGdhvIFAQUdcwJm
    ul1OSo++lnYdG354C0FB3rj/Qjk20+n8+auQmJgaOH16+/EbGOhdLBIJu/j5uWHYsFj4+LrC
    1dUBcpkEfCEfPC4DhmFuuo8asFqtsFisMJrM0Gp0qK1tQsH1SsTHX0Zubil8fd3bhV+RWHbE
    z9d9qlQqYiZOHIju3UPh4ekMZ2c5pFIReHweuBzmllu5bkTmm81WGI0mtLRoUFlZj7y8Uuzb
    ewaEAPb298bvPQhgXXBubknflhYNBg/p8TC6IwICvdC5UwCuJGfLBw3q+RiAK/f/OVqLUOj6
    4fRnHs/u3afLTLVaZ5+Wmhd2Kv4KPzLCD0MGR+8CXC6YTNde2rHjxEf79p0VL3lnBoYP69ka
    1vtgqchmswUWM+sHqCSwXQj6DxtfTqv/wnCTqZ4DAAKttmVQbW0jJZGIYDZZwNA0+DwuhEKb
    D9hssYKiaJhMFkikQtTXN1Pa/9femQbHVV15/P+2fv36db/e1VptyZa8LzKWNwFhxyQsBgIJ
    gZCQIQkwk0xCkYEQapKQCSEEQoAkkIonRQLYBAfM1EDMDIQdyyu2JW+y9n3rfX399vnwWi2b
    IiB1t6ZmqnQ+qVyuV6/vfb977j3nnv/JZLzpciAAABUcSURBVK5y2ByPGsaR+NQGiAbAAaBB
    EHUzXJhPP7wQWjYrG9XVZcS/fO/GXGermQMsyTIee+wFiKKkAoys6+35Oj6CWFTgd2ONulx8
    m0Owrb7/x/8O3s7BwjDgbFa43XY4BR42mxWCk887F13XIWVVpJIZjI4EEYulkM5kIcsKUkkR
    Xq8Ar9d5EmAL3syUlTXmxrE3unBBtfHSi2/h9tsegqpqsFhos0RC4GG3myd1G8/lbkkakCUF
    iUQGwYkIkikRyWTGdNC6jlgsjYaGani9rglgTLfZihW/IrlgMHjVvv3HyVIt9wcPniTCoci5
    /jLhyZk7M5Pfm266/ETzplVfTaZEz8EDJ5bu2rWHbWpajAsuaMrz+8ILr/9bYc53NvmdYNOp
    2CXBYJToODWAG7/0Q8RiqXy4cnKDxVoYOF08eJ5DQ0MNkUymr7ewsacN40h2NvhVZNVQVY14
    553DaGk5VhC/imJeTpQkpaT8Op22tkDAvfqii5vwla9ejlgsgUgkjuHhCcTiJptSdmr8SIKA
    hWXAcVa4XXa43A7U19dg3boVoGgKkWgSHq9QMn7n11YavJ3Dl2/ejIaGWkQiMUSjCXR1DSGR
    SEMUZciSnFc2oykSFtYCu52D2+2A2+VAU9NSnHtuI44d60UsmoBnmvxOwwEnqzs7Bz2BgAeV
    M7g9/LHhXIrB6sZ6vPvOEWTFdEFdsgmiBrreIerIPLNo0fl/ATTrqpUNV1x22dmbXC6+n7cL
    fwBiK957b/99zz37Gn/X927E5s3NKEo+MRd+U1XdARDMJ4PRDyAFoNYCxBcCRHlujSIBxgAg
    CYLriXvuueXyrq6Bte3tfYEdO9484xmaroMA8MUbLsaypQsmFi6sOej1uF4FYRUAsh5QGHNf
    bwEgjAH+bsM4LgN2EMR0le40EmAaAasT0FaTJAin047Nn90Ei8VeEMCqmsEf/7QLDE2zgLKK
    IPgAIEmAcqSAsEnORFAUYZAEiXPOWY1bbrkCgYALHM+ByansTOVWjTNCpqfvVkUxi4mJOLZv
    /2+0tnaCpikDEEvgjmjCxlthd3C47farsXRpHbweAVaOBZ1/v4/K/U1dPNE0DYqiIh5PY3Q0
    gieffBGapoGhaQByCd4vax8eGq8dGQ5Nu7/zp4Wih4eCGBuPLC+vWOICECqU34YcvytX1l9x
    xZWf2eQUbHl+33//wH1PPLGDTyTSRTvf0vKry06Xd+fdd38129nZv7ajYyDwlxffgqEbZ+Qw
    CQK4/gsXYcni2on6+nkHfT73X0EQ8wBb2WzwSzMUMTYWxs8ffKa4szRFwWq1lJxfiiKx88W3
    0dJyFP19Y4gnUtDU08aLJDD5eZp11FMsUzQJwcGjZl45IpH4pNxjyfh12DmkUyIeeXg7xKyM
    keEgMplsTiXLXNtIypxTQzffb3KtIQiAYRh4PAJq5gXQeqQTK1YsmDa/03DAMjM0PE6U+V3g
    CjoZnWmVFT7E4ykkUykDYAoMhS3C+NABeMu6RYohRVnJPltVffGz0dgu8NhIhYLvPbht23/5
    Lr64CZddtgml0C7WdR2qquLTtUsYCmAuDIX2f7OtrfPs7q4hTyyegKZphFNwoLo6IDpd/ECg
    zCdXVvqVjo5+2Hgr7rzzi9j23OvQdQM3fflS/PqJHSAJAoGAT0smM4H33v/w9omJ6D3jY2Ff
    MpWhKYqEU7BjwcKayOrVi3f7/eW/B+i38Mk3404zuyUa7XvkwP7jzaIoU4c+PIFwOIGXd75T
    4DwTyEoSghNRHDvW7X3pLy89zds5asPGlQfd7trPoeAuWLJ7aCi4OhyO47zzroJBENi3/yQG
    B8dzJQlpiKIEWVYwpTNr1qJznAVOlwNlfhcqq/yora1Cc/NKvPP2IQwOTKyqqFjuARAu9Jsw
    jAwMo3/JiePd1LKlddiwYTlCwTg+/LAdwWAM4UgcyWQGmYwEWVLy0JIUCauVgcPBw+N2wOtz
    obzci5rqclxy6Tr8YesrGBsL1TmEBhaAVGR4kh4bD5HpTCkWK3NsM5ksQqEoRRAcURJ+5eyz
    lZUXTfEbeu/Bbc+95gtOxEDTJen/UBC/x493n93dNeSJROLQNI1wewRUVwVEp9M24C/zydXV
    AaW7ZwhWqwV33HENtm97HQCBG2+6FE89tRM0RaKyskxLp8XA7pbDt01MxO4eHw/74rEkTVEU
    XC4HFtbPi6xcWb/b5yuU3xPNkqRQhw4eR2WlH7d+/UpYWUtBYyTJCv749Ks4drTb+/LOl5/m
    OCu1YeOKovkdHgqtHhycgN1hgyDw2LLl3FypjxNCLoJF01MhaE3T83nfVEpEJJLAyEgIvb0j
    GB0JYTwWwdBgsGT8njzZQ6mqjmRKxJLF83HxxU0IBLzwegU4chEsiqbydb6qqkLKyhBFGYlE
    GqFwHIMD4+juGYam6ejrG502v9O5BU1qmg6SoooK4Z6+w5osXJ5J/vejFqhel/+7q7sFNm4P
    dMUNt2toxf4Dx7eMDAfxw3/9B5AkVfjmDShApzzb3Nba+ufHHn/eQ9MUFi+ah6oqPwzdQGfn
    IFr2HLWkkpmVWq4bTZnfhXvu+QpomkR7ex8MA7DZOHz/3lvw9NOv4NVXd1cAqCBJEna7DQsW
    VKJhUQ1IksDoaBg7drxRsX3ba9d9+ztfunBNY+M1AN6b3ntWE319u5mfPvA0y1nNcI/f78Tz
    z79R1Py6XA50dAygrbXTJWZlPPb4nYzbfU4RR6/k4lOn+uvGxyN46BfPQcxIIEkCLrcDfp8L
    DsEsAfD6nLnexSYg2ayMdEpEV9cQ9u4xlbFUTQNrtWBsNIxTHX116zckFwHYU4xzi0SCG9ra
    uoihoQl85eafIJTrXWrP5Xftdg6cjYWFMS9xGAA0RcWErOTLHeLxNBRFBWdjwdusGBkJoatr
    qLFhUWIeiu5+RJCarpOGXjod4kmBhGJE/D+J34MHT2zZt+9EUfnq4vht+/PDDz/jOXKkE6mU
    CMMw8h2aGIa22O3cSp7noGk6xscjKCtzY9u2N9B+sg8Age3bXkcoGMMDD/wJW7e+UsEwVEUq
    JSKdNtMgkycrgiDA81zFmjUN1931vZsvbJwpv/27mfvv38qOjoah6zqWL1+Aq68+71N7AP/d
    5EBaxCuvvI/nn38DO3a86Sqv8OE3v72raH7bT/XVZbMyHv3Vd3D22WcB0KFp5vcvihIURYWi
    qNBUDQQAhqHAuQUzlcOZTJiNuEgcPHgU3/jGz3Gyvbdu3fq1JeH30KFTxOLF8/C7390Nr9cD
    QIcsy2YFRVY2872yAk3Tc/2oWTgcfL7UjGVZTJZNPfnbHfj1b15EV/f0+J2GAyZlr9eFY0e7
    oGkqKMpSjPtFOJIAx1nAslbi9psj+Oe7x7Bs5bKiGKtf2Jz/W1XbzmvZ3earra1AbW3F5G8o
    nF7DgKHPyIGHJFlO9vePee6558toalqBeDwBlrXgqi3nQtcNiKK58DIMjfJyH/r6RvCDe59C
    U9NS0AyFnz/4DB742e341aPfxdhYGJKswO12wGplQVMkkqkMxKwEt0vAoQ/b8cDP/gQxI6YA
    fUa7Qd0wwNuseOCBb2LxkgXFbVTO/GZwqr0H9933+zM6whRiqpq48OjRbsEwDDStXYKLLlqH
    xYvnwetzguPYXLtE4u+svAZ0TYWYlRCNJtHdNYw33zyAnTvfRWtrp+P662Lrg8EP9/j9awt8
    u4y3t3d4dXf3MHg7h02bVqJp3VLU1pbD53OZ4gKM2dLR3N1PaX9rmg5V0fI3tMfGwmhv78fb
    bx9CZ+cQ2to6A5s3n31W8Q5YUVxOQWPZqSb2xRpjYWC38wQgl6Su5qP87t1zzJdMZgo+/Zph
    wZwISwH8yoqc7Owc8iQSGZAEcUZKQ1FURKNJRCKJ/L+PjoYwMDCOL3zxQjA0he3b38inH0ZH
    wzC7nhEfSY+YTj2ZzODUqQGIYnbG/MIwGZ5sKNDfP4bbbnvoU2v3/25QW9PR1TWcbzKh63rR
    Tc9UNXFhW1uXYLOxCE5EsHXrS+jqHMLoaAjRaAqpVDrv5KbCuiQYhjLvduRKfALlHtTOr4Df
    7wJvs6K1tcORFUvDb2fnENY0NmDv3mPo6BhAT88IghMxxOMp8xa2ZDpfUyYLIEnKFNLhObic
    PDweJyqrfFi4sBoECaiKirbWzsDmSz+d32k4YPtoXV1l7OWdb3ufeWYXeLut4MkgQOC1XS2o
    qPCB4/goZsFSqbh3YHCccLkc6O0bLfLETmBkJJjvHWxM61l17WvWpB/esuUzj//yl89TgYA3
    n8BnLQwEgYff74LP54SmG+jrG0NbayeWLJmPf/ynz7dYrSwee+zPzd/+1iNY3bgItbXloClz
    4zIZcpVkBQRBgLOyGB+P4IrLm9W1a5f/wsjwJ2a835dkvP9BK3p6x6b5+6Y3z8PD48hKxecw
    NU12+bwCPB4Bg4PjOHy4A+mMiMoKH9weB3gbB9bKwGJh8jfcNVVHVpIhihLSmSxi0SSGh4Po
    6R5Gd/dQrnCeJ2RFXQbQZOE7j0RtNivNY3LCHxaL2RtZ1XRIkgKLhc7lixQQJJl3v5MnSEXR
    kE5nkUqJyKSzkLIyYBiwsAzErEzLsrQGwAtF5rjSVZX+IY9HaIhEEkXngXXdgMftgN/vHtLV
    bHo2+O0fGCMmx6lQi8VS6OkZxMjI+Iz5bWxMP3zTTZsff/zxHZTbbYcg8MhkJKRSGUiSkj/F
    TnpBnrfiiiubcMcd17ZYrSw03Wh+9dXdEDPSaWF3wPxOLHA4ONhsViSTGUQiCdzwpUvUs85a
    NmN+DQC6Zt4jIEkCiUQae/ccLW7rTJIgCHOeda34FUHTZJff54KuG7j33t/l6/Gn5B7N9QLE
    1KbZnPvcjOWlHs2NFc9zsNs5CIK9RPzK86xWCz7Y3Ya33zkMWVbz0YnJXP7Hb/CNvCRmPrVE
    EmBZi6l7LU2P32k4YGf36lUN3StWLPS+tmsvii0mN3QD11xzvmKz2f42Gw5Y0wyWokjsaWnD
    yZO9RT9PVTVkszJcLjsoMqf08IkWNGi64vnPf/6ia8vK3OfX1ARaPR7nSCqVYeKxVP3ERMQe
    CiecwWCMDYZiEBw2fOtb14fPOXf1C35/1UMAiHt/8LX73/rbviv37T/hOXVqAG6XA36/W1pQ
    V5nw+71Jj9fe5XDYlGg0VdnfP7Jq48aVb7GsextUaQa8pOFw2OjFi+Zh9wdHQRDHSjoPhmFg
    8aJ5cNhtNFD4Os2ywlO33rpF2rRp1XVHjnTUdPcM2Y7t7EYikYau6bnyAPIMx2LkdHBV1RTF
    p2kKgtOOsjIXmptXZW6//drBpUsXvGdl2SdluZhjv9re2Ljsp48++t2bW1ra6o+f6LHs3n0U
    YlYCTZtF+pacoICFofMca6oOKRfikrIKFEUFRZFwuwUsXFiFH/3o1vC6dct3UzT9QvEzkUlX
    VZXv/9znNl2wd+/xoi806bqODRuXoaLc9z5JK+nZ4BcA7HauqBD0rr+24N13DxfM77XXXnht
    IOA5f9788la32zGSSolMIp6uDwYj9lAo7gyGYmwoFIMg8Fi/blm4+exVeX6/f+8t92/csPzK
    /QdOekQxC7fLgbIyt+TzuRJ+vyfp8ti7HHabEoulKvv6RlatX7+8IH7tdo5es6YBNTVlKLXI
    h2EY8PmcsPNcCfi9StrUvPK6traump6eYdvg4DhCwTgSiTRk2fz+T29gYbYaJEHTNCwsDcHB
    w+d3obLSh4b66szy5QsHly6tKxG/S3/66KPfvXnPnqP1J072WkZHQgiF4vl0gaqqp6VMp44Y
    k+84KU/pdPIoL/eivr4GjWsawk1Ny6bFLzGdidD1tuuj0djVsqQW3buZIAjwNltEVZgf/+DO
    unApQtCnmywf/trx4x33hULREpVdEPlLPStW1Pc6HN7rGWZ55JPHrA9A5hxA3wC4nwUqwwAo
    YDAAGByQrochBbKSzDIMrVAU3wrYWw09rBAkAcDLAslVipJapWkaY2EYiaTYcYDvAag0UD0O
    QANGPEDkZkPX9hMk+wFBLJ0BYCcZVRHvjUYTi1VNw2wYRZIQnPZTDGN9kKZXKoUtBB2oPWcR
    +j4YqATSSxUlvSgWS50ViyWcqWR2fkbM+tPpDLJZiVAUFQAIhqYNK2cxeBsPzsYG7XZrv8sl
    ZJxO/hDL2k8AjpNA9YiqHDZicQOFhrAMowdd3XtQv/CcCiDWlEykFofD8cZwOD4/Ek1UR6MJ
    Ip0SWVlWXYqiWswcEsBYLLAwZNzKsSlB4HWf15V1OvlTPr97xOt1HqAoxxHA0g4kMwSxsuiF
    FDi5PJmI3ZVKi2yxy7QBgONYibfxjzCWxhOlXvgn+Q2HY2Qxz546mcwSv5AC2ez/DX61WeKX
    pEgIQun5jcdNftMpaX5GlPyZzMfza+Ns4Dg2yNvZfqfLmXEKttnnNxJvjIQT86OxZHUsliAy
    6Swry6pLllWLrmkAQcBiYcAwVNzKWVIOh033el1Zp8Cf8vncIx6vMCN+p+GARcQS78MlXFLy
    CZ4NaTbD6LcApBegSvxwHUBMzWTEEM+v0z/5HeIAjgGQkck4YbMB0bAGq5UCywGqYoCidBCk
    uUBoKgFZ0mAXNmDr1kO44QbAYjFAURoIwoCukdA1EjRNQMoCoqTB46WQyQA2G5ELZHgxk6J0
    w9gK4Ov43zC3m0CsiE62ofEDsDspkKQ5JhWB9RgPGiQwKgCabarpAnInHMIw9XQ0mL1CKxIE
    QeiGvg+qSsMwCGRSOlzepqJ/29joIbjdAAENGTkJp+MCACEHIDvMmyNZHsjWAAY/qQoHAwYI
    ahSwTwCkChgKUBn57OVVyl//8z+gaRSkrIZESkVVVXMJmDgAglhXsobwpszgYRDEmjl+5/id
    47cIfqf1kRv6CSiKBK0ENykJApAkHZGggrr6TbMAcAfM+rDS7wp1HZBlgOPOmsZ7RGFevvh4
    ndJjx1tgtQCGBjQs+fhJ6mxvMT8BGVix/OP/T2/XHgAG6uqbZzhOJwEYkGX5jN6ls2XTGbPp
    2ODAPvj8DHp7Rfh9Flg5CjyPXN5qKt8GA8iKQDqtIRSSQZI07IKBysqNs/YbY+ED6OoRUVvH
    wek0yxYURYemmU0NKMp8L1U1AOigaHNRVmQgElax74MEGtc7ULdgQ4mZ6ISmTnXbKoUpigFB
    WDvH7xy/c/zOMr9zNmdzNmdzNmdzNmdzNmdzNmdzNmf//+1/AFKZD6F0Z5YKAAAAAElFTkSu
    QmCC
  }
  set pieceImageData(Merida2,45) {
    iVBORw0KGgoAAAANSUhEUgAAAhwAAAAsCAYAAAAuG2H4AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh0mYhOwwAAAIABJREFUeNrsnXd0VFX3
    97/33uk1k0kymfQeCIQSepEWeq9KERBFHwUUREVFRX1oIoqgCCod6Sq9SA8QWkgISSjpvfcp
    mT73vH9MEkBAKQn4e17OWrNgkczh3Hv3/ezv2WeffYDn7Xl73p635+15e96et+fteXvenrfn
    7Xl73p63/+uN+l++OELSAdAAhKAo9fOn/bw9hg2lAuACYGttyQqKCvk/MO48AASAABTl9vxB
    PufG8/acG8+cG5yGHewtALzam0xquzeAopo/o9tHU4CIy7LpFgdEhKAoz/uMOw2ACRQV/n/I
    oHMA6GvvMwVAAoryfUbjMNSOozF1sajBr48QAiAfgAWAtdZ2uaAo7/pre2WcLzZuL/EHWFeA
    LouPW5RFSC4oyueOFxQAjACY2j68QFHUM7SNRABeDMAyhFy0EEKe6XgefsxiALZabtSAolo8
    58Zzbjznxv8QN6iGG2x8bXeBfEDvD7BcQJin0yVWS6UqUFTTZ3AD0wIA69cAdQyQnwDUmcBN
    ALI7jCMZQCjlsPxbgJ0LihP8L4dGEgA3KWAKBZgmgD0ZEKQApbqlC5pjzmfUUxxLJg1YAwCB
    6F5zMhPAkgkQM8ANBoTce3swWgFrGkDxAV4AwP9LJwSAyQDwMinKn23Ye0gB8OMBukDAxgDi
    bKBADzgBcAdQKgMqp+bmFE4vLq5UuakUJX5+3lsB5x8AZRlQB01/CaDzA2g74JQBZFgAeb2N
    PQNwKK1Wdr7dznoLBNzPgeZX/62Cg5CE2hlgcwoo8wKsCoBbBbjmA7eIY2bY8jk3nnPj/yY3
    SipVbm7PudGggoOQXAAaBuCMqqwsnZSbU9LGbLYK1B7KFB8fz72A5BegpvJpRzoISQvPSL91
    PibmprhDx5a5AQE+6wDZz0BmGeBRG/IizoTovyDExtK0bAVL9FkM3fJfDI0yAGVKs9nwbVzs
    zeHpGfmSgAAPfbt2zffy+aL3AE7F0wzdEVIo1dfk7rl1I7ON2Wy5bVgUhcBAT6u7WjUK4KRX
    VZadTEnJUdts9tvhNQ6D0FDfIoWzayRgCyouKvkjI6OA65hBOBqfz0PTZv6xErHPCIry1D/e
    GEsAmGrNXQmKEoOQDAB2b5NJMz8jo2CA0Wjk+Pt7XlIq3b8Hgo8CetjtWe8dOnh6yY8//s6U
    l2ugVMowY8YYDBzU/WMOx+srwEQBNf0qK0vezsoq7Mjj8WyBgV5HRCLFZ4Apj6LCQUgNgIo7
    wpSqRnYoLADxtN927f0xJSUHM2eN+14qpWcC4n/dTNzxnDMB2FuZzdXTsrOKuldV6dzkcnGp
    f4DnGYHAeRVY47XiDCvUIRHPudFQ3Ii7OTwzs1Di76/Wt23b7Jlzw2Kx3vWz+3HDbr+tGRiG
    vi837uyDx+M+JW6YOP7+HvflxsqVvzGFhRVQq5V4++3n3AAabEmFyyGETD8ffWnhTz/tFqen
    FwAgkDtJO0yY0K/DqJG9wsQS2buEkIqnO9OiSUFBmW3x4s20q6uT34gR3b8cOSpyiIeH70Ig
    aD9QoDAYipfv2HFoYnW1HpMnD2mlVLq+RkhKBkWF/kvRQXHMZuO8jRv3T9686TAYDg2bzS6f
    NHHA5EmTh2j4AuV7tXHpp9RcqdzcS9KPPl7ppNXUgMfjgmVZGI0WfPjxRNu4scMoQMkmJEbL
    P5yzyqlOaNQJjyVfTzP06BFCgGrq9Jk4xZLFv3KEQh5omobFYoVMLsb338+WNmsWQj36i5QB
    oBqAGweoUgO0ubj4ZKkjTA7nysrilatX/z700MHzsNnsCG8RNHDOnElNQkPtgwBpeWZG9pTv
    v9/FRLQOxuCh3XH4UDSWL9+BoBDvl8OaKtYA9pCMjIyNXy1er0pISAfD0BgwsPPk6dPGuChd
    1K8Qkleu0cRALm/pBoAPKIoIibE5wq+tG8F5pyoBe1+AfSklJRcXL13HTJBwQPoKwIkhhNz8
    d0U6/gQQ0CE3N2v9urV7w06ciIVOWwOxVOQUGdk25PXXh3fx9fV+1T1Ydfnpjus2N9xUzn4j
    RnT/csSInvfhxuGJGo0Okyb93+HGpk37J69etRtVVTo4OUnkb741cvKkSYOfKTdKiqtQZ5YU
    RWHuJ5NtY+/kxgc/OhmM5vpvioR8LFk6vZ4bUWfiFIsWbuLUTVQIAVTuikblxk8//TF0396z
    MJstaNsubOAHH0y8zY3M7CkrVuxk0tLywTA0UlJy8d132xEc4v1y0zu4seSrDaqYmJvgcrkY
    Oqzr5Glv/e9zg/PkA9YCKO2UcO3W559/vkbs66fGl/99HRKJEKdOxmLF8p1g7ezEsWMHlqcl
    x7+Hxl20+0vjUxKphBIIeVB7uGD79uP0uXPX2s94+8U1L7xgawmw/H37Tk5csWIX7DYWBoOp
    +7uzxs0TS4KmR0f/pO/cuRUoSgGAA4oK/JeAw+ienp49bNvWo5gyZSAGD+mBQwfPYuPGQ+jU
    ucXwli1dvwGQ9xRBBtbOkpoaEyZOGoDOXVqhvLwS//1yA2xWOwWQ3kBVOECJWDuL9z8Yj8Bg
    X9y8kYHly3cCoESAZjRA3GxWOyUSCzHv8ylwcXHGhfPXsG3bMdhZlgLYx7B4BoC8k8kU+25x
    UWV7Lo+j9fDw2wHw1gCmPlFRcQMP7I/GtOkjoVI5Y8XyXdjy66GATz59dYJAIP+loLDUWaPR
    Y8SonujUqSN4PAYnT8aipLhCGdaUI7VaK0du2XJIlZdXhv/Ofx1lZRqsXPkbQkN9+r/04sC+
    oGTH5XKXV4uLEiYYjGaZSuUcIxYrvrPZzBcb40GYTNpPjv55flZ6ZgEVd/UWykqrsWrV7z0V
    CmnPgQM6X/Ty5g0FUP7vcYLN5dVVOV8t+3Zr2J49Z0BRFCiKglZnwK+bj6CqUhs27/PXv3Jz
    4w0HoHm63JBSDEMjNSUXy77dRkefu9Z++vQxa7rWcmP//lMTl369FWazBYYac/dZs8b+67mR
    kZ49bOOGQ6is1IFhaFRV6bFxwyF07hQ+vMUz44YRer2hPleAoijY7Ww9NyiKEun0RhgMZlCU
    Q0zY7Swo6g5u2FlKrzfituAgkNQIGo0bZ85cHbhh/SFYrVZQFIVjRy/D00MZ8PHcKRMEAvkv
    hYWlziUlVWAYGhRFgWFolJVVo6SkUtm0lhvbth1W/fnnZTAMDUIINm08jNBQ3/4vjhlwmxvF
    CRMMhv8tbjRAhEPI2GyGV3btOq5wdpZh6dczoHRRAiDo0KE5CCH4dcufeOGF1gObhAd+A6Dw
    6Rm1SO/kJNMIBTynEcO7ITDQG0u/2YKPPvzRbc4c7Rfh4QHWdWv3Y9TInggI9MTy73YgJNh7
    XGSk1a1t2zb5FCVPBITXAGksa79mpGjRvyHTmFNWVsUzmSxo1ToU7u7uaN06BGvW7ENVlZYL
    MJxnMSiKAkJCfNC2TThKSorB43Jw4sQVpqJC+xkAZGcXwma3o3nzADRrHgYQFjarDXv3nnW6
    di39BwCIj08Bj8tB61ahUKncUV5WWQsi6pFFKiGVAKra5eZmbFi9aldodHQShEI+xk/oFz5m
    TF+VWCyoSEnO5nh6uuKll3qDz5ci5vJNJCVlwGAwqQQCmL29VVUKhVS9Z/cZEJbC/v1n4OQk
    gaeHayVAGWsMevmNG5no1Kk5+vfvAYtFj337ziA9PZ8BZQs0mQrb7993aubGjQeh1dagffsw
    3+kzRrcIDPSbSEj2FYrya8i5CmWz2fxu3sqm4q+moqioEnq9ARfPJ8FV5YyuXVr6ENYu/ncJ
    DjbyckxSpz//vASapu9yOoQQHDsWg8je7TsNH+4aCWD30+WGVCORCJ2qqnRgWYILF5KQk1vi
    9tGH2i+ah/tb1/yyFwaDCTRNYevWowgJ8RrXq9e/nBvlVTyt1gCadtxnmqZQU2NEVbXumXHD
    sVxB3ZWceOr0VaayUvcZQCE7pxA2mx0UhdrfIbDZ7Nh/4LxTYmLWDwDBtYS0erupi3DU9vtE
    3Php9W+h584lQiRycGP0aAc30lJzOGazBRwOU/+9tLT8em54eamq1GqlOiUlFxwOA7udhbu7
    MzzULvXcuHUru37MFEXBYrEhM7OgnhsH9p+auXHjIWg0enTo0Mx32vRR/xPcaAAjs9PV1ZV+
    ycm56NmrDZQuzvWROQ6Hi8jIdjh48Dzy8otdAoPCZE9XcEjylUrZDbWHq++NG1kYNboPfvjh
    PSxdugWLF2+imzb15XM4DKZOHQI3lTNiY29h4cJN3C1bjvb39lGhWbMAtGsbpgtt4ndGJnNZ
    CyhOEHK9xjFzafKMuME1Bgb4VLmrleqfVu9GZGQBTp2OhUqlhK+PezpAqp6Z+2BZAHZwODTC
    wvyRn1+K06diHWYNICzMH0IBD4AdIpEAzZoFIDOzEFmZhfWgCAvzB4dDA7DX9vf4cXGLRTtp
    /bp9oZcu3cRrrw1BQUEZVq36AyqVYtKAAZEbg4K8TXv3nRPs2H4cKnclLl66joiIUIhFwjJA
    Xubr5/3jjLfHLF2xfKfo4KHzcFc5Y+asl4z+Ad6/AOISsUhiCAsLwMVL13HkSBTKyzXIzyvD
    qFG9TABXFnP50sTvvtuGPn07IDjYG1t+/RM//bQn9IsvXn9VLJbFwbFw2lDgZiUSxfdvz5jA
    Y4nN+5ulm1vExt7Cj6s+yBeJJQliEf83s0FfiH9V03sn38rmG41mMAzzFwFLwWKxIiUlh0/T
    9qecRefgho+v2jcnpwQcDg2GYVCQX4Yvv1xLt2wZxM/JKakVSYDRaMaiRZu5v/767+ZGgL9v
    VWCgp/ratTQwDAWWEPgHeMDHW/VMufHXdurEFZw8HlMvSGiaukuM2mx2/PHb6fpgeZ3Tbqj1
    NItFO2njhv2hW7YcA007xO83S7dCpXKe1L9/r40BgV4mPp8rsFpt9f9vULD3Xdx4+50xS5cv
    3ynKzipCYKAH3nnnxbu40aSJH6LPJYIQAkIIeHwu/Pw8TABXdiXm8sTFizehvFwDmqaQm1sC
    DpcJnTdv6v95bjSA4OCCw+HZOFwOamqM96yY1CUScrkc3JkM+JReMqtcLr/YqlXwwMuXb0Cn
    08FqtUMmFaOyUoPjx69g4qT+8PB0BcBgwICOOH4sBr5+atTUGLHl1yNYs2aftFXL4MHDhnfv
    1bNn+1NyuftSkzH/LCHJTx0ehBQBuFni7eO/8/33J3z57qzvcOzYFbi6OuHbZTPh4+O9nbVV
    VhcVXYVaHfHUQUHTNAAaSqUTvl32Nu5M9KoDg1gsAEAQFuaHtevm3mMTDENDKBQAoGv7e+wm
    0Gi0bePiktGnb3u8MmUkdLpKREXF4XpShqxPn66VvSLbn8vIyO+zavVuaKr1aNO2CV6e2P8W
    XyDaBnAJh1GsHTq0l4rDMJ/NmbOSmjZ9JBk2rOc3VjP1MyAAlyvdMWnSwJEZGXnes99dAalM
    hKFDX0Cf3h3OAXTStWtpYh6Pi5kzX4RSqUJlpQaHD19AVZWmtVjsJ4AjVb0BmyVKIg05DxR/
    JhDywvl8LiV3kl6QSDxeB3Rai43Fv6txbTw+D3+Xu87jcQBQtmfBjZYtgwZGn0uoD9PzeBx4
    +6iQl1cKu91+l5MzGMwICPSAVmv4F3PDb+f7H0z48r3ZK1BSUgU3NwVmzx7/zLnBsiwePLcg
    +Od5R4P6lVpupICiUM8gjaYGN29kynr37lLZs2e7c69NHdJn9x9RqKzSoUuXFhg/oe/9ufHh
    j9T0GaPI0L9w4+WXB4xMT8vzvnjpBuQyMUaM7Ibeke3PAXRSQkK6uKJCWx9BIYRFYkL6/wQ3
    GkBw0FaZTH6+bZsmfY8evYShQ7oirFlg7UPS4vc/TsPTyw2+Pp5ZNTpTOSEZT2Vdk5BiAHkS
    o9HUzmK2ICU5F4mJaUhPy8eZM/GYMmUwYmJuIjTUF45teQS+vu4QiwUYOaIbur4Qgdy8Ipw+
    FYfdf5zG3LmrRL16XR78xhsjwlu0aLYEEK+z25MsDBP+lKBRUBvVCvIymys63LiRAQ6Hg6HD
    uuLKlVtITs5B+3bN+4qEkr3u7igiJB8U5fVUwZGTU4ybN9JAwOKBEw5yuwLAg/wMIQAFGjk5
    xU8AE5oVicR6d7ULridl4MaNZOTllUCjqYG7u9LOMJw4pdJt68xZ4+b26tV23PsffC8KCvJC
    UKB3dHmh4aarJw+EpFgAXkxllY41GMxMeZmGgDDXCbGYKIoPQqpj/Xz10QH+HuNyc0uxZMl0
    Q3jzoF1iiXQ+wHh7eatgMJgQfS4BIaF+SEhIh4uLE4RCYbnZYrQ/ug1kwbH3n4WjZoULKEpU
    +9MMAC7ednv2OwnXbr2SmpZLVVRpEXU6dlD37m3WS2UeS7ki/eXbfekAVALQwVFLQIDGqFnw
    907QHhseHlSlUEgU1dU19aH+Ogcvk4nRokVwFcvyYkvyrkHl3eqpcsNsqssbcIxl+oxRGDWq
    F77++lekpubeFZXh8jh4eWJ/RLRuiuzsQpw6Ffuv5MbNm5nQ642gaQp6vQGpqbnPlBsUBfTr
    1x4B/h5gCYsn2zhJQFM0MrMKcf1G5hNyQwn2KgHDOJ4/l8vATeV8mxszx83t1bPtuPc/+EEU
    GOT5YG7UmB7IDX9/9bic3JJ7uOHp5QY+n4u6CArLsnBx/d/gBufJjDkVQCmPpgWVL74UaTl7
    Lp739ddbsfLH9yGRiLFzxzFcungd8+e/AS9vVYrVxFCAHcSiBcWTNbIp8xmWLZ+za+eRAQcO
    XoBAwMOe3Wfw4YcT0bdfB6jVLsjLK4FUKkKdC5RKRQgM9ASfzwOXy0VggDcCA7wxZEhX/P77
    aWzYcBA3b2b5fvTR5OWRkR1VDCNYrNckmCXylo0MjTQ48rncfPS6otUbN+4duHnzEUybPgov
    vdQXW7cdxs8/7YXdbhv+yuQhXLHEZQaQk01I6lNYOyYABYplCb5btgM//7QHHC4Dln38WQdN
    U7BZ7TAYzJA7iUEB9KMLD55JLJaunzx5UIeFC9ZLX5n8X1isNrRv1xSRke2TCKGvUhRbLBT6
    furnb+7K5XJDz56JR/bLBZ0Cg5q5ASgB1GDZnC4Z6XmM2WxBVlYhbTYbOnN50l218TvnzKy8
    8Kgz8eByuPD1VZ8RS3ymg5QZQMn1Pbq3TYw6Hdvu88/XgMvjwkkuwUcfT9Q7K5y2Xk1MNz/c
    sy+vdRgW3Lq5CE3DfqnzcnbgGgi5UQsRAc9iLV2wa+fhST//tBs8HgcCHhdffLFO3K/f9VGz
    3h3v7urqPpqQomKgCo7tqC0AR4Zc7UTqAuuoLyGqL1DUOPacCYPhJESibrb27ZtVjhrVQ7Fx
    4xGwLFubvwFQNIXhI7qhS5eWlTQNm5sXAbHng2K8ngo3ftt1ZMD27SdqgU/gplIgvHkANm06
    hLNnrt0VfaNAwWg0Y8O6g9COqkH7Ds0wdepIDB36An777dS/hhubN+8fuHLl7zAazKBoCiaj
    BSuW74Tdbhs+aeLgZ8INu51F06Z+ePudl8Dl8p+4V6vFjJUrdyIhMf2JuDFx4sAO2dlF0oz0
    fAAUevaKQK+e7e7DDc6DuZGRz5jNFmQ+Ije6d2uT2H9Ax3Z/HrkEu92OkBBvvPLKwP8JbnAe
    35hzAFBiqzX/kwsXEqbv3RPFKyvTQMDnwWazASCoqNTBbLZi+47jqKrWj+nfr4uLi8BrNtZK
    bza+dq6KSEy6+Z81aw9wR43qgbZtmqCouALOSnnt7JuCt7e6Vmw4Pmq1C5aveBcKhRS3q6UC
    KpULpk8fjdatQ/DFF+sw9+PVvM8/t3w0YOALFMsyi6qqEs0KRYtGgkYhdLp8SKWqASXFWV/+
    +OPOdkePxuCdmS9i3Li+YBgGr0weDD6Pi1U/7kZ5WfWgt6a9qHZz8/5Upyk5cmd1u8ZpOpuH
    Wq0dMKATtm05Cg6Pg2lvjUBQsDfIY4gOiqaQnpaH1av3wK43on//TlCrXcusFo3tkYEG/m9d
    OreSLl/+7uyUlNyAixcTuJVVOnC4TA1Nc4wO1S/kEEIJQYDc3BJcvnwzJCg4qKPVmr4PoJma
    Gp1vTk4xPD1dkJdfCq1O7+HiouYQUmID9P2uXLkZkpdbigB/D5iMZonNXsPhMB4AWBNFU3qa
    oTFyZHe0bBnKBgd75YSG+i23WOnfXZTKh3j26Xh/jgu++brAH9B1CA6Z3karvdQEhEAoEpVw
    ufyLgDQWcE0BitpfiYkftmrVH5g4cQBGjuwBDofG+ehELFq0CT6+7p2nTh05mmFCVwHwAXQd
    WTapr06rdbVYzRCJJFaxWH4REMYBTtGEJFsaI/TviGy4QyQyDUlLS1i2b29UwJXYZFA0dcfK
    NAFNMYi/moofftgZMGxYjx1NmgR/BFp4QFOeZJG7hDc6N35Zc4BrqDEBlCOHID+vFP9582vo
    tAYQQu5e7qMA1k5w8OAFnD4dh1atQjB+Qj/06d0O06eP+VdwY/Xq39pt334CZrO1PpJE0RRq
    akxYsXwXyss1g/7zn9FPnRsDB3bC5s1HUFhYjhbhgSCEgCWkLvPzocMkdG0eR0JiBk6ejMWw
    YS88MTe+XzF7dmpqbsCFi4ncykrtA7hBkJNzf25kZxWBoihkZxU9kBv+/ur7c4OmMHFiP7Rs
    GcoGBXk+HW5MGoDRo3qCw2EQfe4aFi5seG48QYSDFpqMxfO2bD00+5df9nICAzwxedJAvNCt
    BZycpACAKVMGwdfXHZcvXceK5Tt5V+OS+82ZM/lX97dsL5pfS8ng80Mb6WWrASGZff88fMFN
    KORh0qQB8PBQIzMzG4sWboBAKMCsWS+By+XeFbKnaRoqlfI+64KOUF/nzi3x3Xcz8cEHP2DR
    ok18pdLpg06dI5Jp5+TtjXMdWQAEIqlUPiEhIe7z5d9t88zOLsKX/30dfft2qIceh8PByy8P
    hEqlxJIlvyInpzhi5qyxa1q2DP8SEGwh5Jax8Sq9ag3OzuoP3ps94TNfH/eRq3/ag91/nMFH
    H09Exy4t6jPLH4IaIIQgLi4Ze/ecAwgwd+4kjBoZ+YdMJvtYU1b2SOuWjiI9abaje3i/9B8V
    dji0SZPJNI157733A+/QofMdJ00eMpGhZSsBK2xWMzgcGp5ebog6c5U3aHDXsSKxx2HAIK+q
    0rcuKalCZK+2uHYtHRXlmhaurnYFwKnW68uGnj4dx/PycgOXy8BuZ0WAnQO4Akh9/eifF7oc
    /fMyli6dYRw8pN8CgN1enqfNlroxxM+nzT88+zwAtPSbr2+9lpZ29Z3Tp6/4JCakM1VVOoAA
    MicxgoO9X2vdOrSsZYvQK85KJSc6+ppc6SzD+PF9IJfLABAMHdYNly5dR1TUVWrChP5TCLnY
    Oi0tt2fCtVSfpKR0pqioAmaLFRKJCCEh3iMiIzvo2rRpupXHUyxl7UmZNGMF0JDr+hIAyYMv
    XYz9eeHCjeqkpAwQgnsSA+12FklJmUhKyqTOnrkW9Nm819Z17BAxrSzftq3xIgIObhw7eskt
    L7cYFEXB3d0ZL43tDS8vFyz5amvtWOn7Lg0wDA2j0YLz5xMRH5+Kfv3a452ZY9G5c+tnxo3E
    hLjPv/9+h+epU3G1QsnxnrGs4+80TcFstmLD+kPIzSmOmPH2i0+ZGy9/5uujHvnjj39g586T
    cHKSwsdX5bCFh8MGCCHIyy1BVZUO7u7OeGfmixg9quG4QVEP5gbLsrBabTh9Ou4ebuTlORKL
    i4oqUFGuvYcbVqut9ln8O7gx9qXeOH8+ATnZxZjwcj9EXrrxQG4UFlXAYrZCIhEiJNR3RGRk
    +4fiBufxDFoLoHjImbOxM39avYczbnwfvP76cMhk0lrn4ogOuLu7YPz4AXjxxd44eSIGX3yx
    Dhs37IuY+e6EdynKaRYardAMRZnMRufUtDy0b98UHh4qxMQk4bPPfkZ6Wh7enT0WNM08eggQ
    BM2aBWHevFcxY/q3+PbbrcJvPFw+IVXNY4DUjIYPQ8oooGz62TPnF86fv44rl0vw3fKZaNWq
    CQACu92GykotFAoZOBwO+vbtBLXaBQsXbMB7s1d4fvrpqz/06NnBAnA3NWIIGkDNNalY9p9J
    kwZdCwjw/OCbb7ZKZ81ajvDwINAU/bDcACEE129kwMXFCV8vnaHv2rX1MgbCH0BZyhmO4DGs
    IBilJVdhNJB8ochts8JZ9h8ej+O9detRbqdOzd9p0qTpEYAygBDQFIVevSIQHZ2IW7eyu3fq
    pPYDCC8/v1hpNJrRqUtLXIlLQX5+qXOTJjZ3UAanGzfSO2dmFKBfv/a4eOF6XZyYBfJ90tJT
    X9+w8SCPYRjInSTlgPBXk7EkjydmIRC0eghocGVWa/FX+/ef+s+qH3+n9Xojwpr5IyjYCxwO
    g7Kyapw5E48//jjt6uPjPrBP73ZIT8uHQMCvTTari9zRcFbKEBNzE7t3n4q4cD4p4ubNLAiF
    fPj6qhEc4gMej4Oy0iqcjYrH3r1npaNH9nxz6hvDW7i4uL1GCCe5YWuFVXrn5eUt+eabreqE
    hPS7thbez4EDQFJSBr5eslm+bJnzJ/7hgefQaPUiKMpsNjqnp+fDbicICfHCgoVvoFOnFtiz
    J6p+e+b9n9ntHRMMw8BksmDPnrPIzS3F10tnoFmz4GfAjQsLFyxYx01JyasXdIQQCIR8+Pqo
    kJtbAqPRfMc25CvIzi7y/OSTZ8CNQM8Pln27XZqbW4yxL/XG4KEvgH4Iw2MJwcED57Bs2Q60
    bh2C994f//S4cccScHx8KpKTs7t37OjgRmFBsbKsrBo0TaOyUouCwtvcuHkzo/PVuJQ7c5Ya
    nRvBwd7gcBmUld6fG0IhH1arDZs3/4mkxHR069YKLq5yXL58A3t2n4o4fz4pIikpA1wuBz4+
    Kvj4qMDjcVFWWo3jx2OwZ3eUdMyYXm9Off3vufG4EQ6eTlc56rddJ/mtWgfjrbdGQSQSOpaH
    7nHSdnA4DPr174z8glJs3HAIg4e8MCYoULgOQHyG6u6pAAAgAElEQVTjGLNVaLWw7aqrdQgN
    9UV5eRW+WrwZSYkZUCrl6NEjAgzDuc94KVgsFtisdojEDzJUFh06NMdrU4fgm6VbsW3rkWbv
    vvvyHJFYOb0hBZQDYLltrl+/8e4XX67lenm6YeGiN+Ht7V4r6BgkJ+fg009/wYL5b6BZ82AA
    doSHB2PZspn4eO5qzJ+/ju+slH3WokX4RUJIamNUmaw7TdNivFrOFfgu6N6Dn6L2UP73ow9X
    hd64nomevSLu2fJ4v2a323H61FWo1Up89dW09NAmwfMAyU6LsZrlix4/WdBNFVEXyre5uSlt
    CoUM2dlF2LXzZPAHczxnC4XOXwOOTPmwMH9kZhQi+ly8e7t2zQdxOExOfn6pG1/ARViYH6QS
    IXJyi11BWYJZ1hp89my8WqVyRnh4IM5HJwIgYGge12Ipm/HbrhMhGRkF8FAr4eaqsAJ2m1DU
    7iFHzaUJW/mfAwei/rNg/nq6U6fmmDZ9FEJCfMHjcesWTqHV1iAtLQ9Hj17GuvUHUVpaBSe5
    GFlZRWjePAgAYDIZceNGFjIyCvDdsh3o3KUFPv10Clq1CobSRV77Hjj6Kympwu7dUVjzyz4Y
    TebOs2e//LVU6vJy2vV0bXBtf0/eLD2iomJDr8an1guKf2oMQ+PatTScORMXOnGiVw8AvzYW
    NywWtl1VpRYSiRCz3xuHzp3bgBALTp++ispK7X0FEiHEkffF46BGb7xLeMTGJmPFip1YsmQG
    OnQIf6rc+O9/13LT0vLuev/sdhbh4QH45pu3MWfOSlw4fx0cDlM7XgqpqXlYsHD90+dGd36K
    Wq3874dzfgw9fOQSOncOB5fH/YfoKAWrxYojhy/B18cdS5Y0HjecnCRIS83Frl0ngz/44DY3
    6p51RYUGF84nurdtW8uNgnI3vd4AmqJgMJiQl1dSz43o6AR1eXn1Xbuc7uRG8q0c+Hi7NQg3
    ps8YjSZN/EDTjm3mLEtgtVqRkVGII4cv3MENCUpLq9CmTSjsdjtcXBVISspARkYBvv56K9q1
    a4oPP5qINm2aQC4Xg2EYcLkcMAyDwsJy7Np5Ahs3HoLR+PfceEzBYZAXF5W1zMgowIwZoyAS
    ie7jvP8qPChE9mqLTZuOIDU117lZ87BGOTPbkSil8c9Izw4sLqpA06a+uHEjE6VllZj8ygB4
    e6vg56fG/bcyU9i7JwrnL1zHokVvQiwW3cfgHSfojR3bB2eirmLXrlPo0rXlqB49uqwnJO0y
    RTXUAU5psFprBu3efUptt9nx2bxXanNO7HVxRGi1BqSn5UGrc6wrOwyYhYenO+bOnYzXXl2E
    gwejA8OaBr5EM9HzGzNjhieMgNl8jTA02RUU5NMtwN8jVCaX4IsvXwefx/3H75stVhQVLYHS
    WYaAQO/jgGW70VgFkaihSvmKdE5OsmxXV7m/l7crjh69jG7dWo3r0VMRRwA7ASCRitCjZwT2
    7j1LjR3bd4S3N/dkamou3NwU8PBwgZvKGZkZ+bDbzB2Ki8s6REcnUIMHdYZMJnaEqylaSFGW
    7ldiEqcePnSB6tixGYxGM6QySabNZtM+nP1WANC2vH795rvffbeNbtO2KRYsfAvOzk64M6+I
    oijI5VK0bdscbdo0xbBh3fDdd9tx8mQcoqLiagUHjaSkDFyNS0FQsBc+eH8CevRsAx6Pf0df
    t/tzd3fBW2+NhFgswLffbEOz5gEDRo/sN2HB236rG8pOLBZtu4RraYzNan9gdONe5+SovZCQ
    mM6MtRraNYbgqOdGRnZgdnYR/Pw90Dw8AHt2H0dRcQXi4pIfuE2bZQlGjOyGyMh2mDPnR1RV
    auudCcPQOH4sBr17X8awYT2fGjf27j2t/qvYqBuPVCaCq6sCYrHwHtHHMAzSUvNx+PD5Z8KN
    4GDv0L17z2Ly5PkPtV+FACgtrcLw4d0alRsKhdQ/J6cEe/ecRbdurcf16KGIA7nt9AgBoqKu
    UmPG9B7h7c09mZ6WC5uNBcPQsNvYu7hx9sxVihBHFA8EoChKSFGW7rFXEqfu3h1FEULgpJBC
    9kTcaIKFi6YBIPj991OIjU1GcXEFOAyNoCAvRPZuh9nvT8Cw4d2xbNk2nDwZh3Pn4jFoUGc0
    aeKLgoJSxMWmwMPDBbPfG4/+/TuioKAUe/dEISkpEzqdAU4KCVq0CEL//p0xc9ZYSGUirFi+
    62+58ZiCww5djZ6y2exwdVU89JKETCaCSMiHVmcgFEWzDQ+NDAA8jsVS+sa+/Wc9XFyd0KlT
    OLhcDlaufA/NmweCw+HcJ0fjtuAoKqpAamou7jxk7H7XolA4YeKkAZg1cwX27I5Stolo+rpU
    5hnXcLMVpayiPL/PpYvX0a9/RwQH+9aKDRrno6/hwIFolJZWwmKxYc0ve7F3zxn0698BvXq1
    A2BHSIgfBgzohMuXb6C8vLKni6vPMgA1jQkPPr8VLNZ4wA5SH2K+vWDytzMV6o4Zms3G2qvK
    hVB5NGvA0Y3UC4XzMxXOsp7+/mrIZRJs2HBQ0bJVkw8pijjXjbdHjwhs3nQY1xJSW6vcXXyz
    swrh56uGSCREUKAHrsanwmo1j0tMTHWqrNCge/fWKCysAMsSMByOZ3VV2Wfr1+1XBAd7IbxF
    EG5cz4REIsjlcIIe8t7zGLtd/9qevVFqQ40Rb789ulZs2B+AW0eYv3nzYCxa9BZmz16BQwcv
    4MUxkXBxVWDPnjOQykT4ask0tIloVis07A+M3lEUhdGje+H48SvYueM4p9sLrcesPy7bvInT
    MLZjtZponf7xygjo9SbY7Ha6Mblx8EC0R1lZNQxGM95/73vcupUDQ42pPnT+IB54eLogLMwf
    PB4Hdx7rTVGOHSxbtxxFt26tnw43KvL7XLp0HXduMSWEYMSI7ujeIwIqdwV4PA5ef2MYhg7t
    hgsXEvHbrlN39XL58s1nwg0QwGSyID+/9KG/z7IEhKBRueHuruwJpKOiQoNNmw4pWrYMvYsb
    NE0hNS0PiYmprVXuSt+srMI7ltmAzKxCWK3mcdevpzmlpObekUvDglPLjY0bDinKSh3nyri6
    KSB+Am68886LYO12fPTRjzh16mrtFluHMDp1Kg47d57EgIGdMHPmWHz11XTMmvUd9u+LxrBh
    3RAU5IX5/10PDpfBwkVvon37Zti48RC2bDmCwoJysCyp7+vQwYvYu+csFn/1FsaP74czUfF/
    y43HfnmVzk7g87nIzS152CAayiu00OuNcHWRg2UbtgqY4+FaAZgGXLgQP+X48RhMnToErq7O
    cHKSolWrJrVigzxQbNhsFqSl5cPf3wNisfCfzBxdu7ZC69bBOH36Kq4lpA4HzK0JaSgdZZQV
    FZf6aDR6RESE3vWoagxGlJVXo1qjB0sIqjV6lJVXw1BjvGNWSKNN2yaoKNegpKQikMPhKfAU
    Go/bGnjyU4iphoUGAJyASOykc3NzRmWFFlOmDERqai7+PBIdwhJWbrOxICyBr68HwsL8cSbq
    qrS6uto3v6AUAYGeALjw9fNASXElysurvE9HXZWGNvGDf4CXI3HMbgdFw+n4iUstkq5nYsqr
    g6DV1ECplIHPF5oA88NeOlNeXtrsSsxNNGniW1sn5mFsyg6VyhXvfzABWm0Njh2PQVp6Hs6d
    S8Cbb45Am4iwWqHxT68dgVgsRpcu4UhNy0dWdmFzmrF5N5xzEREXF/ljfddFKQOXy7M2Fjcu
    Xrg2Ze/eswAoGGpMuBKTDEON6W8rWRJCIBTy0TTUD1mZBdBqau75XZqmkZCQhujoawDQ6Nwo
    Li7zcRyIdvc4JFIhXF2cIJdJHBEymQSuLgrIZeJ7IkqFBWUoKX123KBp+qE/jc4NkZPO3d2l
    Xlicj07E0aPnQ1jY5bbaolgURcFoMOHs2XhpdXW1b05t0nHdz/LzSlFeUeUddSZeWmdTAGC1
    saBoOJ04ebnFmbPxoGnacficSvFE3Gja1A9Hj17CyZOxtaKGAcMw9X/qdAbs2H4Cc+euBigK
    c+a8DK22BqdOxSE3twRRZ67hjdeHoVOncKxYsQPLvt2GwoJy0DR9V18UReH69Uxs33YcIpEA
    3Xu0QlpaPrKyC+7LjccUHEKNm5vrhWbNA3DgQDQqKqpwe1vu/cUGy7I4eCAaQiEPISF+eWYz
    SW1Yw8gHwA/Lzc1c/OPKP6SdOjZHv76d7hAY7D/AlkZcXDKuxN5Czx4R4HD+af2QQC6Xom+/
    9tBo9Dh1MlZpsej7AmcaCoUwGs2EZQlEIv5dQqdPn/ZY/dNHmP3uOAj4XMx+dxx++uljDBrc
    9a4xi8VCsCyB0WSm/v75/O83iqJAU3S0u0rB5uaWIKxZAAYO6oKNGw8hM7PgjhA4H336tsfV
    qym4cPE6ampMCAjwAEAjwN8DZrMVly7fQFxcCiIj24LLdQhTmqKRk1OIdWsPoHefdmjXLgw5
    ucVwVTkTHpd3laIeNnlNDJ1Oz2q1NXBTOUMg4OHhawnY0aplCPr174AjRy5h394zUKuVGPwX
    u3iY5qZSwGqxoqKimgHoBjMeDkd2tk1EE7NAwHvoysOEEAgEPEREhJo5HGFUY3Fj1ao/pHXl
    pOsO3fqn/AVCCDp1Ckebtk1w5Mgl1NzhTO60PaPRgkOHLsBsNkEulzUqNwxGE7k3wZXCtq3H
    MWXKAixfvgNmsxXLl+/AlCnzsWHDobt2oVIUYLHaYDRZnnODokDTdLRK5cTW2YXZbMXGDYeQ
    kVFw17OmKAoXL97AxYvXUVGuvUtwVFfX4PLlm7h06cZd36FpCtnZRVi/7iBMJmu9uHVzUzwR
    N7hcLlJT82Czsfe1Yap2u3fU6atY9ePvCAvzR//+HXHkyCXs3XsGSmcpxozphQMHzmHTpiOw
    29kH7s6iaQq5ecWwWq3wULvAbmdRUaG5LzceN8JhEQqd1k6Y0L+6oLAM8/+7HllZ+Q+Y2FLQ
    amuwfv0+bN12DGPH9SVe3h5by0uMDVaf3QEuRqzXl33y88+7m1VrdJg2fRT4Av5DgJYCwCAr
    Kw9ff70FKpUSvSLbPjSgu3ZtBTc3BS5cSEJhQfEAIKCBKpoxkEolAMg94UWKosDjcuHkLIV/
    gAecFBLweNx7DKuwoBQ0TUEmlVgAmx3/3zeOWe3hgspKLXQ6AyZNGgCWJdi48RDqtqhVV2vQ
    okUQBAI+tm75E3QtYPbuOQadvgYCAQ/bth4Dh6HRsWMzWMwm0BQFm82GzZsOw2Kx4tUpg2E2
    W1FaUgkPtQuhaM7Dx4dhgkAooIVCASxm6yMfB0BRDIYM6Yq8vBLs23sO/ft3hJOT/JEFh9Vs
    BcMwtcngpAGXP3nHunVve6FXZFvYWfYfr48QArudRc9ebfBCt4gzNM2LagxurFmzt1lMzM1H
    KqfvqEDJQd9+7RF/NQWHDl144LILTdOIj09FTk4RAKpxuSGR1C7t3O0Y7HY7jEYzNBo9qqu0
    0GprYDSa79l9QwiBUMCDTCp+zo1abrirXcDhMCDEkQeTmpqHzZuPwGq11fs9mqJRXFyB7duP
    Q6erqb+nFOU4JG/HjhMoLCi/g9OO3KStW44gOSWn9uRYgMNh4O7+BNywOIKAYonwH8UURQF/
    /H4asbHJGDb8BeTnlWDPnrPo168jLFYb1q3d7ygU9zfCmxBAJOSDpmlYrXbQNPVAbjyW4KAo
    ZwCuMe3atf7hs8+mmJOS0vHmm0uQkJD2ly4paDQ6fDJ3NX7+aR8mTRxARo7s/afFyvleLOE3
    8JbYmgknT14YffjQebzx+lAEBPjgn/IGABoajR7790fhnZnLEHvlFkQiQe1ZHg83m/Dzc0dY
    M39kZRXiVnJOGGAKBk40wPWIKr291Mn+fh44deoqTCbjXwQdiyahvli79mM0aeL3l7A7BbPZ
    iJMnY+Hp5Qo3N8VVgFv+HBziSnd3V71eb0RZaTV8fX3w6quDcenidZRXaHDiRCwmTfwcS5b8
    CpGIjysxydDrjVgwfwPmzFmJxYt/hdFkQVxcMuRyCX7+eQ9mzFiCmCs3UF6uwdmz1zB+fD8E
    BfujrKwaWp0B7u4uBhCm+uHHaGddlC7Ffn7uyMoqRHWVDo+2QsUiJMQH7u7O0OuNaN8+DI9e
    bZFFSkoOnBRSeHq45gB0WQM+BI2Li/eiD+ZMzOvfv8PfnrFU59D7D+iI996bkKdUuswHuJqG
    58bF0X/8fqreMT+SMLPasHr1bsyd+zOqqnQPBDNFAeVlGlyLT30q3AgO9r7n8MO6qM31xExM
    m/YNkhIz7xvFYVkCPz813Fyfc6OOGyo3pZ53x64ZQgjORMXjrt0mFGCz2XEl5hYsFttd99Vu
    ZxEXm3zXv1MUUFZWjZMn4+4okug4s0elUj4+NzILoanWYsCATvD1c7/nTKu/2kR1tR7Hj8cg
    MNALbm4KaKr16NQ5HJcu3UBqav7f5C45bEUiEWLIkK5gGC5SUnIgl0seyI3HEhyOMqZGiqKY
    6hbhwcYuXVsgNSUXN29k/QWOFEpLqxAdnQAfHxV69mrDSiSCSg7N9XR2uQrCpjSQQZTISkry
    X9ny6xFet26tMWRoN1RVVcNisdwH1g6hUVZWhc2bDmLqawvx5Rdr4aF2RZeuLVFZoYHV+rDL
    xAQCgRBt2oTCaDTj5s0sJ0JMEYRENsT8y+CkcN0/cnQvEnvlFv74/VRtuXC69kOBw+HBzc0N
    HA6v/roAGixL8NtvpxATcxNDhnQ1K12cdwIa63Nw0KlOTuICDofBb7+dQnp6Nrq+0BJNw/xg
    qDHh4IFoiCUCaDU1SE3NA8PQMJnM8PNTY+GiNyGVilBaUglCgMSkDFyLT0NJaRXWrz8Mvd6A
    Zs0CMGhQZ1RUVOPkyVhYrTbInUSFoMSPsHwotgkEksPde0TYsrOLcepUHB41JUYsFsDfzx0u
    Lk5wd1c+ouBgkJqagxMnYtGpUzN4eakuVlT4NYjTISQNGk0SAGJWu7tW9e/fEQqF9IFFJQkB
    5E4S9OvXHp4eLlUcDs0HfGjCJjcoN7Zt/ZNXU2MEIQ7H8E9l+VnWEXWRSIQYOLAzvL3dUVhY
    /rdgpigKVqsNV2JvgWVtjc6NESN7EKGQf5eYY1kWdjsLrc6AK1duQautqb1e9u7ohpCPQYM6
    P+fGHdyQO4kL5HJJva06itPZYbeTepHqKJDqSAStOwX2r58738W6AneOCNNt2xFLhJDJn4wb
    J07EonnzYMydOwkqldM94vOvY6pbHvLydoNSKYe7uxIpKTm1ERzc8bt3vwdCIQ9vvjUcgwa/
    gNzcQhw7FoOOf8MNzqMBgwDIAkB5m0z5n/755/lJa9fsFWh1NRg9uic6dwn/C9wIvLzc8NrU
    oThy+AKmvbWUGTmq54SJLw/uIJZ0ecNOSk5rKxMgc37SMwVMLeKv3mqWmVmE2bPHIz4+Bd9+
    s612H32rO8bkyAw+cyYOP3y/C4WF5ejRIwLTpo2saB3RhPfppz9J9TrDQ9cHqOuzebNACAQ8
    JN/KorQaXWcuz74WT3iEIUUpQEj+9j59OnZOTs4av2LFLuTnl2HkqJ4QCLjQaGpgNJrq19aE
    Qj7kcgksFit2747Cb7tOYsyLvTBkcNffCREcbfyzEepsJBc2a7njoAQKtS8S9Y/3sC68VysJ
    iV6fCImkocs+03ZXVxebVCrG9u3HcPLkFcjkYhQXVUAul2Dw4M6YOeslCAQ8JCako7SsCiqV
    Ai1bBEMqk6Jbt5a4cOE6iovK4ayUoUuXliCEYMlXW3DiRAwKC8vx8dzVKC2pQkZGAby9XeGi
    dLIRYnsEaNsBCI/26tUu4dDB6DbLl++El5cbOnSse7fqjr570H111GgRiYQQifkQCvl3zC2o
    ewTz3UnUNIoKS7B40SbYbHaMGRNZIRLLt4slDWEXeQAonlzuPzUhIe6DtWv2+Z0+HQeDwfQ3
    yxAUKso1+HzeWpw6ebXFf94csaNFC863oITLrMZEC1fY4om5cS0+uVliYgYoioK3txtGjuqO
    s2evIf5q6j3LK3UVOt3dndGnTztERrataB3RhPf5vDXSh3unKaQk50Kr0cNJoWhUbvTu3bFz
    SnL2+E2bjsBksoCmKYQ28UWrViFwkovB5XFhs9qg0egRH5+KlJRcsKwjV2byKwMw+Flxo7Zy
    qN1mf6SCgdRT4IbcSYLCwvJ6P+L4sPUihOHQ4DAMGMaRzHqniGBZUiv47LDb2fqow21Gor4A
    m5NcAhel/Im54enligEDuoIC8Pnna1FUVFn/f95OIHUcN2M2mWG12iARCSAS8SEUcFFdrYNI
    JABN3xZGNpu9XniIJULMmvkiXn1tKCoqNFi8aBMMBtPfcuMRt8VmAOB763RFP69b98eAX389
    gh492mDq1KEICfGu3fPN3gU0oVCAGTNG48Uxkfj9j1PYvPkI0tJygz766JW1vr7eU6UK8ent
    269i3LgnKZ/MRuTnlclkMhHc3Z3x2bw1kMklCA31u8MsAZvNhh07jmHlyj8Q0ToEH344qaRl
    y5BdfIH8WEZG2rfXr2dK27cLg1AofJRXBd7ebpBKRcjNLUV1tVbt62fiwJH6/oSNqpTLVbNn
    vjOW8vFxH71p4yHu3r1nwRdwYTJaHFt3KcfxqwyHgVDAh9lihUjIx4y3x1iHD+v+u9zJ5T2A
    0jdG8Z57oZENQMyh6AoFS1iYTVaUFFeC9xAFfCwWK8xmKwjLgmEohY3lcazGBBtX2JAHXAm0
    Uqk40cPTJdxms2LEiB4wGEyQO0nQJqIJWrYKBofjcNAvdGt3hwN3fFQqF4wYEXmXTQEE33z7
    Ni5eTEJc7C2Ul2vh76dGdbUOarUL5HJpislUU/PwDoMDQkipSmWc9977EzZ88vFqtzlzVmLu
    J5PRp08H0DQDs8mE8goNysurUVJSiZKSSlRX62A0WmA2WWCx2hFz+QbKy6sxf/5GiIQ88Hhc
    iCVCKBRSqFQKuLk5Q6GQQqmU1+7IYnDrZgYWLNiImzczMXfuJEtERLPvNNXmiwpn9ye0i3IA
    dj4hle+fPn3+s8WLNvJTU/PuKmf+t4seNSYcPBiN1NRcl7mfTPm8e/e2DGGZpRZLuoXHC3oi
    bhQUlMkMBhPEYiE+/PBltGsfhtgryfUOpW58LOsIdfft1x6TJg0sadniDm7cyJD+XXTjTsFR
    XFKBikoNnBSKRufGjLdforx9VKM3bjjEzc4pxtSpQzFmTO+/MJrGgf1ReO+9lfDydsMrrwyy
    Dhva7Zlxw2azw9VNgXbtmj7UxM9uZ3Hlyi3Y7PZG54aXl1u4I4rviCK6uSmg9nCBp4cr1Gol
    nF3kkMvEkErFEAh44POYenFiMttQU2OEvsYIraYG1dU6lJZUIj+/DIWF5SgprYTRYAbLEnh4
    NAA35q52e/+9H/DhRxPRv39nWG0s1q87AG9vN/j5qaFSu0CpkEIqE0Mo4MHJSQq5XAY+nwcO
    hwOBUIAZM8bgpRcjYTZbodUZUFWlQ1lZFfLzSpGSmose3Vvj1deG4sb1DCxevAmpqXn4+OO/
    5wbn4Y2iFAARmM1Fizds2D1g8+YjmPbWCLw8cSAE9cmZ5L4OGQDcVEpMmzYa4eGB+PKLdZg/
    f13Al1++sdbZWT127FjJlXHjnsQguAaBkGenaZphCUFggCf69usApVIBk8kAiqbB43Kxc8dx
    rFz5OyZNGkjGj+t3yEnh9iUQfA1IeefcuXi/kpKqv6lC+mDBIZNL4OwsQ0VFNaqqdQG+KHZH
    A5RfpihPEHKrRCpzeWvSpKGGmhrDa9u2HceUVwbBx1cNmUwEhkODtbPQaQ3IzinCml/2YeSo
    Hpg8eehmgPs+YK32kxY+BWjkABBxgIp3Ll1OGnYl9haqKnV4dcqCh3IqhBAUFJZDoZDg4qWk
    YV27tH2HsvO+txgSbDxRQ8GD2MVi+anBg7uMWjB/gyAgwAMDBnaFXl+DigoN4uNTUVRYhvIK
    LWr0BhiNFphqlX9dxIbH50LA50Ig4EMkFkAul8Bd5Qx/PzXatmkCsUSM48cuYf/+aAwe0tUo
    l8t/t9k55kcbpxlgQw63bo15Cxa9OW/Bgg0ec+b8iPHj08HjchAXl4yCgjLo9QZweVyIBDzw
    BDwIhQIIhTzw+Vz4+bvDP0CNstJKmC1WmE0WGI2Oa7FYbLBYreByOXBRyhEW5gcPT1f8tus0
    jCYzvvhiqnngwC5LGUa0jOEY7UDJE9hFHRMyXrpw4cqn8+b9zM/LLX2kKGJd1c7U1DzM++xn
    waLF9KedO7epSrpOVsVGn0TbrpGPzw0B3w5QjIurE5o1D8DGDYcQHZ1Yvy3eaDSDEEAqE2L6
    9FFk7Et97+JGdHS8X052yUPZOEUBhhozKiu0CAxEo3NDJnN5a+LEoQZDjfG173/4HSmpufj2
    my2oqtY5nj+HU78lViQSYPTonpg0acgz50bfvu0xf/4bD1UYzmazY968NYiOTmh0bgwa2HnU
    ubPXBC+80BJTpw6Fn78aSsdW7VpXSv0lakj+Eouh7og2EgA2WK1WVFRokJVZiC1bjuDEiTgM
    HtLlybmx0MGNjz5chVu3svD61GHo2bMNeDwO7HYWOp0BOl0NtBoDdPoalJVX4/qNLKRnFkKr
    rcHhwxcgl0sgFgshkQjh6qaARCKEVCoCl8uF2WwBh8PBkSMXsOSrX0EIeShuPEKEQwYgZ8LR
    o2dHbd50BG+9NQJTXh1aC46HSWJ3HID2wgsRWLSYi9nvfo91a/cGzJ49cZbVonj16PGT5n59
    Hhcc1BUfX3dtTY1JUVhQjnmfTwXDULBaLbXlnMMhl4vx00//r73zjpKqvvv/+9bpMztl22xh
    +y4sLL1D6KKiWKLwGDVqHkvUqLEkMcUG0diiUaNGwBIVDeKTBBWkqqAgsHTYpWzvs7vTy507
    c9vzx51dIJqcLbOc3+88+zmH4zkeHL/3ez/f1/18P/WfuPmmJcottyx9RaO1PKko/i6CaJh/
    /Hj1w++9+zk7eVIZZs4a28fnOSs6LQuTSUyLfkAAACAASURBVI/2djf8/rAdIIzJOpAEMRKK
    4pVDoTNZlftPYsmSGbjp5iu/V6HnJBqXHThwCj6vz6Qz5Ed02twLAA0XAJoGvPft3vPtikd/
    94Y+Py8TP/nJ5f3yEBMEgZ07D+Gxx1YbV6wgVs6eNZGgZOYln/eYaLVVJGEvzVCU9nXz501e
    umXLviuef/4DbN22H3V1bXB3B3rr1WmaglbLQqfXqC2rGaoXFrIsIx4TwXFRRPk4ZFFCXFAT
    wRyOFBQVZ+PEiXpMmFCChQum7CBJ0wYNVdDPdWqhKE1YOLn0je2VVS1PP/2zP6964+/577z9
    GfR6LebPn4hFi6Ygv8CJ1NQU2Kwm6A1aUBQFkiJ7J2eeGwaQZTVWH4sJCAYj8PmCaGtzo6am
    BV9+cRDvv78VU6eV4557rvFMmzruBZLS/1GWYzGzebAdG1cCuLm0vu70b5979l1tS3Nnn1rd
    f59QFInmZheef+597R//6Lh/bEXxNoIorhmERlTm5KYHzWa9tdPlxQP3v4SmJhf0eg1+9asb
    cPRoDT7++Cvo9Rrc87NrlZtuuuw73Fj7/hZWFMU+V7eIooRAIHJBubF//0kE/GGsfmMDejol
    93wcVYNQzUc4cOAkfD6fSae/8NzYs2fvit/99nV9PCagbOQIfJMYFdCXoErZyFx8+cVBPPrI
    KuPvf0+snDVE3Jg3b9LSuXPHX+HxBFBQ4ARF0airbUN7hxsebxCRUBThcBTRKA8+FlcvKgkb
    hKYpaFj1omIy6WG2GGCzWZCRbkNaug3l5QWIRuOYPXssFsxPLjfefXcz6uvaMaaiCLW1LWhr
    7YbHE4Tfr05zF0UpwYez+Se/+tVrIEkCFEmBpkloNIy6ZqsZmZkOFBfnIhaP4W9/246Sktw+
    c4Pum2IIADrzmhobfvmX1/+unTN3Am688dKEsdGfkKP6kZwyZQx+ds81eP65DzBt2pgr582b
    8fmf/3L6/YGrhK6+YkzJwbFjixa++OKHsKQYkZ+fjZ07D+KLLw5i4aIpWLPmM0ycWIbrrrvk
    Uw1jXimJXjdF2+dUVx19/YnHV6cCwEMP/QgWi6nfBgdF02BYBvG4gGAwDICgk3coWwEExx8/
    dnr2iap6jB1fgnXrNqOjoxvBYASCIIOmSVgsemRkpEKn06DqRD2OHaudP2du2sRw4NBeo2XC
    EKODYGXZfe/evQeeeOzRVfrComysXHkH0tNT+6kfBC67bBYee/QNPPbYKv3jj922YuaM8QoU
    8mUkrROjGEmxZr1+881L5t3502fNp042YcGCSSgtG4GMDDvsNjPMFtUlSp0Tjz33xi7JiXhm
    XHWTuj1BdLo8qKlpxY4vKuHq8OB3v7slZHekvQbIkYFBbkQiOZvbVFJSdPXjT9z+rCjJi5qa
    XHjs8f+G0Wga0NMbjYDdbkN+PjBhgrrnsiQjwvFYufKO9qKiols5jtzKslGJYZJxQ/w1FY9X
    PbT2g89LjhypGbCxcdbooHD0aC0++HBL0YMPpv2aC/N3DDwMoasfM7r44KyZFQs3btqDQ4dO
    Q1GAS5dMx+KLp2HTpm+hKAqWLJmJ5csXf4cbv1/5ZqoaGuq7t0aSZUT5GABliLnh6eVGdXUD
    rrhyNkpLRiAjwwaL2QCKoSGJEkIhDp2dPpw504QDB07h6NHa+XPnpk1UlNq9BFF0wbjxyCN/
    0Xd3+WAw6LBm1Sf9KgcnEm0v3e4AHnlklf6JFbetmDl9iLhxy2Xz7r33BfPttz2NCMejs8sL
    PhrvLatnGDoxY4QERZK9Tg21E6qaA9GTiEkzFLQaFmlpNtjsJjQ1ufD88/cmnRuKgkXr1u3A
    pk3fJnJHFJAkqa6RosCyTCKUcg7rFPRyLh4XEQ7z6OryQ5ZbElOHSRAEUFKS2y9u9FHBaQDB
    izZv2VMYDEZw+21LE2GUgZbnK1i6dDY2f74X6z/eoZ8yZfT1a9/+wcfm/wE/sN/zh1Ks6Svv
    uXf5uEceed1x+21Pw+l0oK6+DTdcvxhutx+nTjbimWfubrWkpP5GluJuik6ZUV19fNWjj64q
    CYc5PP3M3agYWzrgZyISNdUcx5OAwiTxUBolKfjjHTsqDR53EB+s3QKNhoXRqINep1V79csy
    ohyPcDiKWEyAxxvEl18edEyfXnEdFOrokd1KdNzMoYzFUgVdXV2/fOqpt/QNDe0oLHDitdc+
    PqfUqx9PSxKQFQXNTZ149rn39K+95rw2NXXkGgDBpOwmkQtFCX5TUpJ/Mi8vc+q06eV46Bc3
    AxAQiUQRiUTh9aodcfloDDFBPK9qiSJJaDQs9HotjAbVxVhQ4ERFRSEuWjwbkixBFEQUF+cc
    lyTDLpoeMYi1lsHjOQyjMX7EaMx4e3R5wdyDB04xwSAHo9EEUYyD43jwfBzhMIdIOAo+Fkcs
    JkKWzoYEaYZS16zTqHpj0EGn00Cv1wIg0NTUieysVKSl2j6NcmWfy/IRMMz4JHz0eACuwuqq
    04s2frYnqRq3ceMeXH757EvGjKkoBXBiMNy4+55l47q6vY6DB2sgyxLqatvw1JPv4PDhM8jO
    ScP11y/+Djcef2JNyb691f1MMFdBfl5/jCHjBk1LUvC6bdv2GUaV5+OFF+6DRsMiHhcgiqLa
    IYEAGIYGyzAQJQkP3P8SvthR6Zgxo+IqlnXuHXIXR4IbTz75pj4zw4E//OEuGA26fvee6TE6
    IhEer776MZ555j39668PHTdystOmHjlag6LCbEybWo7snHRkZ6fBbjfDZDJAb9BCw6r7CkK1
    OSRJRiwuIBqNIRzmEfAH0eHyoq62BXV17Th08DTK1amuSefGqPL8uQxDMSkpJhQWZiE7JxU5
    ORnIyFDzuIwmNY9Do6FBU1TvFVEQRMRiAjguhnCIg9cXQEtLF+pqW1FX14bm5k7k5KT3ixt9
    NDj8ep/Ps3zHjkrqB3PGobikr+2W//2p0+sNuPbaeXjyqXfR0NA6cfRoWzGA4wP7vVwA4jej
    R0t3P/fcvb/dtnV/hdsTgM1mxtFjtfjyy0O4+OJpqBhb+mEszlZrWGLGqZMn3nzi8TUlFEXi
    T3+6H2UjCwb5TAml4uNaRY5mATg0cFD7EopjBSAubW/vuGHHjkoYTXosWjgZM2eORV5+Jsxm
    AxiagiBJCAUiaGzswLd7T2Djxj3YvqMSN954yc25OXmVAN4fWnCQDMfxTKfLi6KCbES5GOpq
    Wgf+aySB4qJsuN0BBIORaG5uapJL87RRs9l4KCvLMXXvtyfw7LNvo76+De3tHrVUMJGJrSiK
    OuD9nG+KLKtJugRJgCRIkCQBo0mPLKcDeXlO7N9fhUynAyaj4UAwGOMGu1K7fXxCJ+rqs7LT
    4hzHM2vXboYgSGhq6kCny4twONrrFu0JoShnlQnqoCh1GihJkmBYGna7BdlZqcjJTUdVdT1m
    zaqA0aSrA5pBUckafEUCCF+259tjuS6Xp1+egP+sHyQ6XV7s31+VPm5c2RxFcZ0giIxBceOZ
    Z+/57fZtlRVeXxB1da345JNvQJIkrr1mHsZUFJ/lxqkTb65c8dbAjI0Lyo1AdnNzy+KdOw+j
    pCQXL7+8Hi3NLnh9IUTCXG91m9Goh9VqRE5OulrBt+sIrq9vu6K0zPw6gMYLwo1OL665Zj7K
    ygrQ99y57zVgsGjxFPz5lY+HlBsZmfapFbKCV197CE5nKgiiJ39DhqKI54UozhoB6G0LTpI0
    1A6uah5Hd5cP9977Akwm/dBwIystzrIMc/0NF+HOO69J5Cf16K4EWZYgSjJk6fxGfCSZ4AVN
    JZxvakRDFOOo3F+FO+54Brm56f3iRh8Njmhxa2vX2JbmTvzkJ5eDJKlBKoZqdIyfUAqdlsWp
    U0320aNHzR6owUEQDBSlUwaUj4qKSiqLivIWAoqprc111fZtlZMYhsKCBZO/YBjtSwBKm5vr
    1/z+ybfLaJrGH56+C7m5mYM2NnokHpe0sqKM6DskzgAQoCgxhMMcolEe7W3fgiBl8Pw+ALFG
    mqa6L798trOy8iT51VeHsXPnYbAsi5QUI2iahCjJ8PvUeBwAFBVlY9KkkbJGw4ZISqkvm7gP
    XV07EpAhoNNqYDQZQRDGREy3ZABwOwKgp7yLEgkCCkEAt/zkMvzwhwvOHaQ4AG8RhU8/24k/
    Pv8hCAIKoBNF8UAC8CQIYtxg35LMMKjNyU3H1q2VoGkaxSU5GDe2GI5UK+w2MywpBhgMOrAM
    A5Ime9O91BwONWs7lMjcdnf70NzchWPHanHyZBMmTR4Jg1HbogSQxHlBOiU31wkFCj7Z8DVK
    SnORm5OOiRPKYE9NgTXFCJNRn0gaZUEmQheSKCLKC+D5OPgoj1CYg9cbRFenD03NLmzbWomm
    Rheu+6+FEkEwPpIckURA8ywfjSw8fOg0IcsKyCSOXZNlGUePnCG4CDc9LnS+MRDX+b/nRudV
    c+dMnMQwJObNm/gFw+h6ufHUU38t27PnRNKMp6HjRjRNq9XQS5bMlCsrT5Lr/rYdwWAEPB8H
    w1C9BocoSr3x+eLiHFx++SxZb9AZJCmWzkf3NgZDkSHnBkWS+Mvrf8cHa7dicFXBhNrhkySG
    lhvZaag50wy/L4iWli7U17ehra0Lnu4A/IEwIhyPeFyAED/HM0qRYFgWer0G1hQj7A4Lspyp
    yC/Igs1mRiTCY+SovCHhRk5OJnR6DYKBMFwdbjQ0tqGhoQOdLg+8niCCIQ5RPoZ4TIB0jmeU
    YWiwLAuTSQerzYTUVCtyczNQWJgNnz8EUZSQlZXaL2700eDgtI0NbVqKIlFQkDlIpThrcNht
    FjidDtQ3tJEMKxkG5/JKRyRyHDQpNNCMspogSWRljVh7083Z4wGFBKj9gpDikeX2P/71r5+N
    dHV48Nprv0BurjMJxlPPYVIQj8dBkgrVl7+rlhkXAmgqIQjtDwwG5NM0CVnWERRFERqNWQbo
    uszMnCcffPCmScFgcG5rm6vw5Zc+Qk1tCyZOLMV7723B9ddfhO3bKzFq1Aj8/P7rkJ2d2Wg2
    Gb8AQR8AYCMp3Z0mE7IlSW07y7I6EISlAdDsApxnFKUWiIwAYeyPR1ejA4jbKYpMA9ypFKXo
    AIBlGTCsdpChUzWumGixXABU3U1RJgOgRADlDQDRwb2pGCGKcrq7249x44qwevWvkWJNSdxS
    JEiSBFEUVWgI0nmTg0mSAGOlkTsiAwzDJHIS1NtKJBzGT3/6LLyeIGIxMU0RFSJJhwWASHER
    DhRJ4uFf34hLL5kFgqQStw45cVORE5Mzz2/S0zM3QV0rec7tSkZzcztuvmklAsEIBcgWRXFh
    YN6C7w+QdXe7Ta2t3Uh2eSVBEGht7UYwGB6RkVlID1Thvp8buWtv/HHWd7jx3nsbR+7aeThR
    zpu8Zxk8N4h8te22niBJMsENpi4zM+epc7nxyivrUVfXhrlzxmHt2m340Y8W4auvDqOgwIn7
    fr78X7mRSpK6O024MNzweIK9ybSDOiWiBIfDAp1OM2Tc8PlDqK9vx623/gHhSBSyJEOjZWE0
    6KHTa6DVMKAZGqyWAUBAkRXEYwKiXAjubj/OnG5GJKKGQglCnXnl8QRQUpo7NNzgOMiyjH/8
    Yxe2bNmPUFjliE6nhdGog1arls0zGhoswyYuKhJ4Po5QiENHhxscFwPHRRGPi2BZBjRDgeN4
    hCPRfnGjjwYHpen2+EiappFiMSXN4NBoGVgsRng8gfMsq4GKwTDmrE+GPwgKXCdBKZvVHpwE
    GEZXUVl5avmWzXtxzz3XoKQ0L2nGxrkKT/R5f+hcSTp2b01Nw/K9e09k1da0EIFgGOoBJ+HM
    dCAjwy6lplmjqam2UDwmGLZv34czNS14+OEbEfCHIQgCioqzMWXqSDz7zPv48MOtWLhgqkOr
    ZS/tdvuudbm8bHt7N9vd5SMkWQJFUTCbDCgsylamTx/bVlKSt46iLC9DoZv796SsvrOz4a5P
    P/m6JBjk4PMGEAxy2PjZN2iobxtQHPbcj8npM03w+8N4+61P8ywW44uWFBOWXjH7THp6/nuD
    Bwet6+72TT95qgkVYwrR1NyJ7Tsq0djQgda2brjdPoSCUQiC0OtqPHdtaqIVDbNFD7td7eaZ
    l5eJ0tIRGJGXgYMHTqGr0zMtKytbB4Ab/AfJA8A75fiJGp1Op0F2VhqOHa9FXV0bOtq70N3t
    h88XQjgcBReNIRYTzlszzZDQalmYTAZYU4ywJToJ5uc7kZXlQHFJDo4drUMwGBpjsaRSyTsU
    OiYYjtCBYHgIDA6opX3hiJJJWC8INzZs+Bo9TeqSLQPiRm3D8sr91VlnTjcTHm8AoqB+DHJy
    0pGRYZMcqSlRh8MWEuKCYceO/ag6UY/77lsGny+IaDSGtHQ7fnrn1Xj5pXX4aN12zJ8/xaHV
    spe6Pb5rOzt9bEd7N9ve4SbicQE0TcGaYkZJ6Qhl6tTRbcVFeesoenDcCIej8PvUfKk777oa
    8+dPHqSnmcTXXx/C6lUb8O47n+WZzcYXTWZD0rlRXdUAR6oVly6ZgZKSXOTmpsNms8Bs1kOv
    00KjVS8ivR1IASiS3FuSzvNxhMJR+HxBtDR3oqa2FRs/243qqoYh4UbViTpdKMhh4qQyzPnB
    eOQXqOfeYjHBZNJBq9WAZeneRNBEJBaSJEGIi2ruCRdDKMyhq8uLxkYXjh2txcaNu3HsWG2/
    uEEP6KQnDxtqh8lk/ywAnXbiv2w+B1GsWbZ5816nIzUFiy+eljwj8jumVF8kQCpK+Hcfr998
    25tvfYrCwmyMrSjC9JxyaDQM2trcOHKkFocPn6ECgbAxHheNNE0hv8CJJ564DWVlubjv3hdh
    MGjxz3/swsuvPIAVK+/Au3/dhJUr1xhFUTYyLA2z2Yic7DRMmFiG7CwH4nEBra3dOFB5klj3
    t23ZN99y2YPLl19iVpiun/bvtBuUDpdHXL36ExgMWqRYTRgzpgD+QBi79xxLyg22vDwf1Scb
    4fOFEInwmDx1lJiePjoJLy1Q0NjYXtLW2oWOdjc2ff4toABmswFp6VZkpNuRm5sJUw9ANCwI
    Us2Ej8cF8HwM4VAUPn8IHR0eHD1So+Z+yHJvu+KGxvbSnNzCAgw4ofE8g5/hY4GJhw6dJru7
    /bj//pfgdgegKAoMBrWZl81mhtlsgM1mTtxA1KOtyHJizeptpfpkE7zuAEKRKERBhMViQCwm
    gGVZtLZ0TrFYMtIAdCT3JAxRwjJx4bixbdt+Z6fLm7S8jcFy438+3nLbK6+sR1ub+zwPXC/Y
    aZJiWcbIsoxRlmWEw1GYTAa8+95muDo8SEkxYM3qT5CRaYfPF8I772zC+vVfGimKNMbjAuJx
    EZIkfaftPE1TRFaWI/uuu68ZFDfeWLUBrg5PwltEIisrFaNGDT6Ho6GxFRwXw9vvbFK7wmba
    k86NpiYXli1fiEceuS2h2yLi8TgiER4RTuWCmEg2VxSAJAgwiTOp0agN+NLSraCoEZg6dRwA
    AgxD4e23Ng4JNw4cPElqNAzuv/+/MHPm5MQei+A4HlyEhz8QghCXIApCry6pISCmt4LF7rAg
    Jzcdo0YVAaAQCvlRW9eK06ea+8WNPhocUsxuTVEEQUTAH0ZaWhqSkfMgCDH4/WEUF2eDIChs
    /rQWF18+VKVYQUt3l3vBnt3HMHNmxTkTNIlBU09N1OvvQYnJBIHTsbgoR8JR8s47r0JZWQGC
    oRBIgsScuTrccOPFiMUEhEMcPL4g9DoNnM5UhMNRPPP0u6ita8N99y3DmjWf4vnn1+Lhh2/C
    n199CO3t3YhwPGyJDGSthoEsKwiFVNfaIrMRNTXNuOP2Z8BFYzIUuZYX/fIAeA+NhsVvfv1j
    zJo9AQrkIfimkPjm60NYsfKdJH62+DnV1Q2poRCH8eNLcPHF0zFufAlyslNhthih0TCJPCXy
    P34eFEVCLCYgEuFUw+NoLTZt3IN9+6pw4kStbfasSePq63efKCiYOdiTYulo7xp/sroRNEOj
    oDALN9xwMUaOzIMzy4GUFJPqEmXoRLOkf217rvbgEBM9QyIhDp1dfjQ3u3Dw4Cl8vesImps7
    cepU44jy8vKRyTM4YoLRoJdMJn1vv4fkhSHUZlVarZYAQhg6OcuNZHlmaJocNDficVHu6PCQ
    iqKcZwSd7b8CxGJCb14XSZIIhzkcOnAa6Zk2/Oxn12DVqk9w8MApMAwFkiTBcfx5J6+nhPlc
    b6WiKGhv94Dn44PgBpEoOadAkuo+rF69Af/85y5gEJ5REGorfHVPKBBE4p9J0zuVG9FoDKkO
    M7Zs2Y0zZ5pQX9+Ork4f/P6wmk/DxyAKEpRE8nZPwijD0tDrtTCb9LBa1ZyInNx0FBXlwGI2
    IB4XhoQbVVUNcDgs4Lgo1q//HDU1LWhucsHjCSAYjCAcUasbJfFs0mhP+3OtloXeoEWKxQir
    1YSMTDvyRmSioMAJp9OBqhP1/eJGHw0OrT8nN90nCKJ++479CIS4JEysJuD3B9HY1IEFCyYp
    QozkMaSisK7ObrvHE0Raug2RCJ8kzzEBvz8ELhrrp6tVAWB6a8mS2Yt37Tq86PHH31TDS4kD
    k5JihM1uhtOZiuwsB2x2C0RRxoYNX+PLLw7C1enF3XddHVu2/KLDlhTT+Bdf/JvmlptXYv6C
    ScjPzwTD0Dh86AzaWrvQ1t4NrycIny8IEATSUq0IhTiUlOTgsiWzdpCkYfVA9kIBICsymppc
    sNrrh8hjRKCpyQVZkZP264oSzYrHBVKrZRGLCfAHQvB5gzAatABBQK/XgGGYRBMw8jzDQ1FU
    12jPH47j4XYH0N7eDZfLA46LgqYphEJRWhBjUwGsHfzGRMo8Hn+RzxeCwaDFqJF5GDkqD3l5
    ThgTiaI0Tf+HD7paX8+ylDqvwaBDWhogCgK8eZk4fqwOdXVtaGvv1isKPxPAF8nZaUFyOKzt
    zkwHTlY3QFGSaXDIyMiwwWQ0dgR9XdJQc6Ory5+UUApBEBBEEZEINyhuXHLprMVf7Ty0aPPn
    +6DVqg2lJEmNu/fk8fT8/wgCvf8uK9uBO++6OrZs2UWHLVbT+Bee/1DT0tLVO8+mZwBZz39L
    kkRvjwaejyMWE7Bw0XhceunMAXPjrBGj9NoXtTUtqDnTnJT9PZvQqwwqtPt93BDiAqnRMHj1
    1b+Di8YgxMXeFvhn50F9f46PupSeHCv0XngZRv2wEwSRdG54Pf6iQCCMYCCCBx94GaFwFJIo
    nxOs+PdrPrtenGeIEASBnsGAoij1ixt9NDhsTSNynafLykZkrVn9Cd5/b0tSXqAkyWAYCuXl
    BT6FYHbjbG/YIRCWCYejJMfxWPv+Zmzfvh/J0EWCAIS4iPo6NW9BkZU+2dMEkQFFOeOz23P+
    dOutV0z76KPtpvHjSviMTEeI52NEd7ff4HH7de0dblRXNcAfCMPvD0GrYTFuXDF+8csb2qdM
    Ln+ZZnRvX3nl/NudWak/X/fhVvuGDbsgimJvvN5utyA9w47y8sJoeoY1otdplU6X13Tw0Cnt
    VVfO8aWnpz0LJe4zGsv7bW5QFEloNCzeWLUBDLNxyJAvCCJMZgNoiiKSoR4EofvnsmULSwvy
    nYu+/uaoad/eKmz8bDdkWQFNUzCbDTCZ9NDpejqN0r2QFOICImqyFMLhKCLhKCRJBkESSHVY
    MGpUPm655TJu4qTyY4pCbkyOPouNJSX5G1esvP3iHdsrDRs37cH69V+C1ag5UGmpKUixmmAw
    6GAwaBNNy8jeNcdiArgIj0AgDK8vBI87AK8viBgfB8syyM934qFfXI/Fi6edlBUlif0XdgkG
    Q/GeKVNGXltZWQ2CIJMIfxmTJpbBbDHuJMnSIZxoqnIjwkUTZceD88rEYnG88vJ6rH1/K4S4
    MGhupKSYTBMmlPIZGfYQH40Rbrff4HYHdO3t3XC7A/B4AwgEwjDodRg3rhhLl85un9zDjSvm
    357lTP35R+u2248dq0MsLsBo0MHhsMDhSEFmpgOpadZoWlpKRKfTKF1dftOBA9XaK5b+YFDc
    ICmSMBp0iBj1GMpZLYqiwGjQgaLIpHHj2mULS/PznYu+2X3MdOZ0M9rauhEIhMHzZ5t/AQpk
    +fu/FT1GkWpkaGA26+F0OjBy5AhMnjwq6dwoLsnfuGLF7Rfv2H7AcOx4DQiSRCTCQ4iL3zF+
    egyPcwN85/WMSXhqdDoNrFYTCovUPiQX9YMbRN9eHEcAdXefPlX/eGenh0qea1SBTq9BSXH+
    blE0Xntgrz46VCEVRWlLb2w8uuuj9dtLIuGoqhhJegwCRMI1CMybNxGzZ098gKLGvdg3K7+L
    lmXXc4IQG6/RGF8AdKfUVxvNA+JFoiCYIpHY1HAkbAyHOUar1cgZ6fbDGq1hPVC+VxSPIhjQ
    UDa7MiMc8l/V2eUdJwgCqdNpBJPRGDaadJUswwRBaGoAXaOqQnwpzwcfIEnqqCRqHtQbRkv9
    3QxFOWkNh32fV1c1lPJ8DEMWp0+IRsNg5Ki80yaT9RKSLPcNThdOA6B1QGyGJEan+nzB6V3d
    vuzubl9eR4eHCgTChlCII0Mhtdwwxsd7bzGsRi1tMxkNMJr0ssViiGRkOuS01JQ2hyOl3mGz
    7KMYwz6AOggI3oYGPwbrGlWUkwAMeiA8mwsHFne43GM7XJ7C1tZOW6fLa+zs8hE+XxBcJAqe
    FyAIIqRELJYgCTAMAw1LQ6fXwGwxID3NBrvdwufmZsScTke9MyvtlNVq2QVQW4GceoKwJPHc
    1Y7p7GxeX3OmOT2Zt02CIFBQmOVKdaRfzWoqTg7dR+ssN6IcnxQ97+mX0uNB+L/KjVgsjqEW
    lh16bnjc/rwOl5cKBsKGcDhKhkIRRPnYd7ih02lgMuphNOpls0Ufyciwyw7HheWGq9NT2NbW
    bevq9Bq7uv2EzxdEJBztnbMkihKgnA0BaTQM9AYtzGYD0tOssNnMfE5ORiwz0z4gbvTR4OAB
    NOkBFABJLkCHjKA/Elz/Zm3jrQ8tcRzTOgAAAsZJREFUG0Jrt44C+KVArHiolVwQsINlJxzs
    27oaAEStsiwxgNDVs7tCXFEbNVEAkEkBRgoQaUCWT535A19ccB1CYRlW62S43YdgNhN47sMf
    4jc/PqwFZBLQiUBUAuolWSIgSQoYlkh4lgBIVJqsyAJBMT6GGT2A/TxBAXQhQGlxQUSBJEV5
    URTqtNpJ0uB0oRZxzg+SJUHTMoBiGvBaATETgBZKrEyUpAyKkqAoMiRJ7pntrlAUKZMECVmm
    QJKUC4TmFIAYgG4guws4LkoSjVgUkIQYzLbpSdDdFghCABQp4IjwV0zQrtQA3VmA6JBlvkKB
    YBcFkYnHRCPPC6WipDiU3twAAjRNiSxLVWm0TDfDUHGaZgSArQKYdoBtB7I8wHE5GhXR0MCh
    vHxOMs8dAYgFAGlIvk7EIt0dofo053RlmBvD3BjmRhK5QRCgqeRzg+j74k9DEqMQpeQnBnJh
    CSePhTBz/vwhNDgaoSghSBKf1LjevwpJklAUEgzT9/klsnwUoigjGJDhSD0/S76taR+sNgaC
    LIOPqvXwBiMFvX7id/cxdBCRqARZVqDVUWAIEl6vgDN1POYvOKsMQfchaA0ASIDVTBjgflYD
    UCAIAmRZxoUUrTZ5s2G6Oiuh0RCQZcBoIkGSgCQqkBUFJCkn5gaoZrZaKqberhWZVEtkaQKS
    BITDElga4DkFjowpQ/bsId8+8AIBs4kCzRIQRRkEIYNhFABBRPlCRqdlKIDpNeglIYJ93xTx
    M+btg6IQEEUSJKGuPxxRoNUCbc1xFBRPH4Jz1wpF9qkZ+0n/bZUd9tTJw9wY5sYwN/4/4AaB
    YRmWYfmPcujwnp7LEiABEybP+H9qfXt374EpBTCbAK2OTnQIVGEnCoAQV6tp/EEgHgcqKmYM
    v9RhGZZhbgxzY1iGZViGZViGZViGZViGZViGZViGZViGpd/yvxXB9da6YhzyAAAAAElFTkSu
    QmCC
  }
  set pieceImageData(Merida2,50) {
    iVBORw0KGgoAAAANSUhEUgAAAlgAAAAxCAYAAAAGAtdnAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh0d0xhZ5AAAIABJREFUeNrsnXV0VUf3
    sJ9zrt/cyI27khAgaAjBirtWoC2UGnWhLvStvIVC5aUutLTUKNAWd3cNbgkhECEJUeJ69Zzv
    jxsCKbQFGqS/L7PWXayQcyczZ2aevWfP3nugqTSVptJUmkpTaSpNpak0labSVJpKU2kqTaWp
    NJWbuQhNr6CpNJUbU2Q5F7DU/aRGEPybXkpTaSpNpYlXTQrWv3miZABWQAYUCEKz/4/6nu8Y
    eMG3afbf0HFIBsqAznVrMEEGLyACQRD+Rf2QgdQ6lMiACkEIbRrgJl418aqJV//f80p5bTqR
    CtSAbMVmtyOKIqKoAtRAMwRBdaPfswrEfqDIhNBkWT4ugwuCEPiHfliBM0B13UA0/5cujoOA
    GkkyA65aALv9gEkUNYAFQehwA9tWDqRjt9uQJLmh9i+AQqGoW8ACIGGzSXWL5MLnBJRKERAB
    GVmWsdvt/OExRFFAoVAC4QiC6zXqTx5QAdRit1sRBRFB1AI6BCG87plEQOsMnnfCiaGOb3qu
    BPVS2FF0vq50oAZZNmO321Eq1YAOx1z1u0lmVy6gCgBFd5ASITjp37c+0oBawIrdbgOomyeq
    unGLaOJVE6+aeNXEqxuvYMnyYSBCgNRoBKfeSqUlHJTFIKwHzTFIMt8ES1hvt1d/UFVV4ebq
    WrUGXBaD60ZZPmEBdwTBu+65k4CbF1j9wDlZlpOtgtDiX4gsLaANFEXlwzZbSjsApdLpMKhn
    gnzmBjdOBPV9CoUqRqG45FiZgelAEfCMUin4/MmYFgCfA56CwJNKpaC59HNCEvAzIF2b7hQA
    rp6gvk2hMDUHRQmoFkHoCcduvBpQBFgtJR/t3n1o1JYtBxWSDL16th/RrXvcvSpV8H2ydDQL
    wQvwBTKjBUF7u1JpdQdtCgiL4WzRzTO3atyqqoqmp6RkjggK9kn09qoeBaT8u4R5KhDhA7p4
    hcLWvW5nuwPYA9kFsnzwhgr1Jl418erm4NXhUdu2OXjVo0e7Ed26NfHq74rQuLA6DVQYQHg9
    IyPzvr17kvxycooEV1cDsR2jS1u3br5KodBNtJlNZ1TaWcBHNwiq6a5FRae3v//ez619fb0Y
    Oqx7VVTz0J8UosvXEHYcsqk7aw40myu+KSoqivf395knCPo3a2trS/T6Nv+i3WA2UBFYW1s5
    Y/36vUNWrtyJLMsMHdKVAQM7r9LpnB6zmTij0rW+Qe2zqEymQ4uWLN4yLDu7sMGE1GjVDB/R
    3RQaGtYNtKnFxZl7Fy/a2rystLJBHW5GA7fd3uukh0dwHJibnT6dsXP5sh1as8ly/u8AQUHe
    jLy11zKdrtXtgmCw/4P5U7f7lOqWkAlBaIEsZwKm0Nra0q8SEhIHHTuaKhqNLvTo2SEzJCT0
    f9B8OqQJslz7nwUL1k754L1ZaLRqBAFqa8288so4Ro0e8JZCFKYg+Cuh+KmsrMzntm8/GHL2
    bDkxMRFSl86t1zgZPJ4GZYYghNftLp3qeuho07md57UbMxk4TE1NJXp91NBjx3YtGv/gJPWr
    E++1jho1YmhpafZ6o9G1bucadROvDQsOq/rx4ZUVpf/duzexdUpKllqSJKKaB1vi41sfc3X1
    nAR3Lz+xexYturZp4tV15NWG9XuHrF6zG1mSGTSo803Fq8WLtwzLOVOIWHc8JgNqteoSvNrS
    vLysmnOnaLIMrm5Ol+SVxWKt558kywQGXk9eJQ1KPJYqGo3O3NLjYl4tXLB2ytSpP1FcXAGA
    u7szr7123yV5tWP7oZCionJatgr//55XjWzBEnWSJE/ZuHHXhI+mzRFzcotwczNQVVWLRqMy
    PvLorfeMHTvQXSXqHpflD7ME4aMbuJDh1Kkc5s/fzOLFWwyj7+zz9F13DRru52d5A4J/g0JP
    s7nk219mLR08f/4mnnjy9ieHDu1pkGXVBIslpUKt/reY3yWlLEuvzp+/ccinn/xGaJgfgiAw
    adL3lJRWDLl33NBXQf08YLsx7VPJtbVm+7z5GziedBpfXw8QoKqqlpoaM63bNLOFhkbbIMBW
    XHTE9uOPK6iqqsHo7gIyFBdX4OKi55Ye7WweHu0kKLGdPp1v+/qbxej1GgwGHciQn19My1ah
    9B8QL6k15cLVz5vjgLcCqoLBHgZiAQQdl+VTMtgMZnPltO++Wzrkh++XoVIpMZksrFy1M2TK
    lMdei4iw7RIEl7Sis1l3zJm9hujoYKa++ziCKPLG6zOYPXstPXp2uM3PL+JTqIzLzMyc/NrE
    L52PHktDp9NgNlnE8eOHDXn0sTs+Uimc7pPlrCpwF6CiJcg+oMwAlyxZPmoXhGstVDXBer3i
    abv9bJzFYlFXVNRQW2sR7faaZ93c3EeC+mew7bu518YaIGVYVlb6jI8/nOO3bv0+qqtNAOj1
    GnW/fnGxL750z4zQ0FlydBeXFTdW8XDw6vd5m1iydJth9OjeT99558CLefXLssELFmzi8cdv
    +9fz6oP3f6GysgaArVsP3VS8mr9gI3sTklEoxHohbjDoLuLVTz+tJCurAEE495xEcLDPxbz6
    ehFVVbX1/kx2u0Snzi2uC69mzlw6ZMY3S6isrEYUReI7x4RMeefR1yKanefV3LnrKC6uQKl0
    mO1KSiqZ++v6i3n12pfOu3clIkkSBoNOfOTRkUMeffT/X14pGw8AXwCmAYnHUh6bPOl70T/A
    i9def4Cw8ABKisr46aeVfPrJb7i4OA0eMbzPawnb9jxZp8beCBO0oNM5CXonLa1bRxAeHsDM
    75axa9exkOefH/tVly7Eg2RZt27H4C++WIBSpWTq1J8xOOnG9uvfbQ94zRw7BsucuTsAnzpN
    PPwmdfaTnQsLC3vP+30Dt3Rvw5R3n0AQBF5//RvmzdvIwIGde/v7hbsAJTesiYJjx9atWxsm
    v/MoCCKbN+1n2rQ5KJSiGkwPQmqJSq3ylmWZu+7sy/3jh4MMX3+9iE0b96FWq7wg5wUQ3BRK
    Ua1SKXnuubvp3acjyBJvvfktJaXlIPAPYJUDVBkhc3J+fv6QwsLyIDdX/dnAIL/FoujyEUhB
    p06eHjr7l9WMHHkL9943lLS0M7zx+gyWLd0a+NzzQbcLgvrDisoqbXFxBbGx0YSEBgPQvHkw
    q9ckUFNTqwOtEooHLl2yxfnkyWymTXuKyMhQZs9exezZa+jRs/2g9m3bxoIyCzKezckpGF1c
    VO7l7uGSHRjotwpUb9XUHCvV66/dLt9mM01cunTzE7t3H6O4uBy73c7ixVsUR46cGhoc5M29
    9w1pYzQGD8Ph4HGTlhjPqqqCtz7/7De/xUu2IYpivcA0mSwsW7YdwO/tSY++ZTQqE+qOfW4Y
    rwzOOgQEzmQX8Nmn80jYnRjy7LN3f9W5jlfr1+8c/Nmnv1NeXsWUd37C4KQf269/138lrxYu
    2ERlZU29QK+uqmXhgs03Da8c/lFiAwVLVIgX8UoQhDofZMdzkuTwwfojr8S6ui4cE1EUrguv
    Zv28mqrKGhQKBbIss3vXUZYv3x747HPneVVSUuFozwVtKyurbMCrZcu2Ou/ccazu3SiorjYx
    a9ZqevS4BK+Ky73c3f/v86oRLVhjtZKU9+DChZu0giAwZcpjREaGARLBQX68FeJLXl4xv/yy
    hlu6t+8TG+flD+TcmBWis+r0+nxPT5cYi8XCpMkP069fHJ98+hsvvfS5yxuvP/h0hw5Rtm9n
    LCEmJpznXxjDu1N/4rPP5ilDQnyn+vhWDJo953AlGPaA+hio9sHhKllOATQ3WRSVkqqqaqGi
    vJrAIB+cnV0ACAr04tDBFEy1ZgWC6saTVga9XktAgEMAeHi4YjZZmPf7RvWehOPPAZSWVlBa
    WomrqxMB/r6AjLOzE6WllcycudzLaHSZBJCZmYfZZMHDw5UAfx9AQq/XUlxS/k/JarDbzNPW
    r9/x0Jdfzic7uxB3d2f/8Q+NeGrUqAEuOp1h/ZmcAq3ZbKVP3ziaNQsnJNibgABP0jPysNlt
    riqh2u7j41kY3SKkxeYtB+nYcRuiQmTz5oM0jwrG08OtQJYtFkmyaE6mZuHv70nfPnGoNQb6
    9o1l8aKt5BeUaAWFHGixFIxfsXzzfd98s5iCghJ8fN3Dn3zyjqcHD+6mA+1zQNU1NLsbzxaW
    kXm6gKrqWmQZCgtKQXYIFKvF5i7ZrUpu6iIOPnI4pd2aNXsQxYbC7ZxgXL9+H8OGd2/Xr1+3
    wcAvN5JXXp6uMQ7hJiLLMrt2JXLmzBcur7/+wNPt20fZvvt2CRUVNSiVSs6eLeWLL+YpQ0J8
    /pW8Ki+vbiDQBVGgutp08/DqEsVitrFg/mb1vr0nngOZ0tIKysurL5pX5eXV/PDDijpeCWRm
    5mExXwuD3HleTZ++gKysAjw8XP0fGD/sqVF3OHiVk1OoraioQqh714IgIMtwOrMhr5pHB7dI
    S8upd+C32yWiIoMa8Co17QyyLNdb6wRBoLqqloILebVi833fzlhCfn4xvn6e4Y8/ftv/aV41
    YoUmZXl5aXBSUgZxcS2JaBYI2OuNVO7uRnr2bM/33y8nr+Csp69fsMuNU7CkaoWo2B3dIqzf
    vr3JlJRUMmJkbyKjgpg06XsmT55J166tlbm5Rbz88lg6dmzN0xNG8fxzn/HII++7+fl5Do+M
    CiImJnxsu7bRNaGh/tvVGufvQb8GyitlOe1miDyqKwq7j7dPRctWYaxatYvQUD9EUWTV6gRa
    tAjF09Mt1WY11dwUe1fH2NQd0WgJCPDiyJFUjhw+Vb9z9PQ04urmXPecjLu7M56eRrZvP3ze
    HioIBAR4oddr656TGslUam53PPnk3e9M+YHgYF8mPHMnBw+c4KOP5uLraxzdv3/vnIAAn7N6
    vcZ7+bIdGN1cOJWaRXZ2Ib37dESlVJWYzbXVBoPbV48/flu7Ke/86PrCC5+DIBAVFciTT95e
    6eLqOqOyOr3axeBlbtkinF07jrF02Taim4exYsUutFo1/r6eRchyiwP7j45+772fadOmGXfd
    3Y8dO47y7rs/4+vrMSY+vsMcYPM1OyhROX39wAO3BY4ZO8Tj2LGUiIcfelf94IND5VGjBmQI
    opDlpNN8ba6pKuemLjVBySdOq6qqaustDA3EkyBQW2vmxIlM1cCBXYNuXDsdvGoeHdpPFLfX
    t02hEMjKKuDt/35H125tlCdOZNUrJaKoIDExg0cffd/N1/ffx6sWLUNJT8/l3LBIkkx0dPBN
    xas/zhWbzcbSJdsaHMwIgnCRglVZWcMvs1Y3MImJonANLIoOXk2Z8iNZWQUoFCKlpZX874Nf
    8PVxH92/f68c/wDvs0ajs3deXgkKheBQkESBiIigBrx69NFb2xUWlrkeO5qKLENsbHMef/y2
    BryKbh6CKNbVIQhIkoyrqwG/C3g1dcpPFBWVIYoipSWVTJ360/9pXikbtSpBAMHxgpHli1zo
    BVEE4WZIvpULOO2OiYmoqTVZ9ImJaUREBOPr60lsbHP27jnO/Pmb6dSpBe07NAfsdOrUipat
    Qkk8lo6/vycbN+5j/ryNuLg46bt1azNw2PBbesXHt97q5OQ1HYwrZPmkHYwIgtcN7qtbhcG5
    etYzz90dO/HVL1UTJ04HBFq2CuOpp0fXGgyuM0SFpvYmQhUgEx/fkl9m//ei0GUAV1d9PcTu
    vqsfQ4Z0vQTwwN3dpe65xppxlu6HDqU4VZRX88ILdxMX156sAZmMGvUf9u1N1vbsGa+NjAzd
    8cgjI2+fPn0ha9cmIEkyneJbMnLkLamgXKxWawFhUWxsa49PP31u2oQJHxvMZjMffjihKjIy
    7LXKcusCvSEQUKwYMaLHuIMHkv3f/u/3iKKASq3kicdvo3nzsK0IFCYkJOlsNolXJ95LVFQk
    neJbct+9kzl0KEXfuXPbztcSWFC0zWYNGuDqYumi056eL4qCu06vtbq6eU0sKTIvU7gozNTe
    RNPq0hYs+e+EmgNpMtwwd4bzvGrVKrzGYNDrq6tr6ywNct0xkkBCQiIWi7VeSAsCSJJEWVkV
    fv9CXk145q7Y3NyzqiOH0wBo1z6SJ54cdVPxypFeQfrbZ/5+6sjY7fIlvvfPeXX40EmnnJyz
    9UetCoWC8rJqDuxP1vbs2UkbGRm649FHb739y68WUFpSiUKhoHfPtowY0f1iXn3yrINXFgvT
    Pnz6Il4NH95j3IEDJ/y3bjmE1WbH6O7MI4+MqOfVnj3HdWfPlp1vi1LB2cJSDh/+v8urRlSw
    dGYXZ9ek6OiQ9lu3HCT1VDZRzcPqBVxZaRnbtx0iwN8LH2+viory2hu4C3EDLB7eXm4KtUrJ
    5s0H6Ns3jinvfM/GjfuJi4smNTWH4BBf9HodIGMw6ImMDKIgv4QPP5qAzS5x5PBJ1qxJYP2G
    vaxZu0czeHD8gPEP3dq1VcvI98BpBnJ+8Y0FwAnALIKgMTjpZJVKibuH44hQqVTg6uqskCWC
    D+w9LUjWQ7Koan9DgSVJEjabDZBRKhX4+Lj/KbRsNqtj1uk16J20f/OcgCRJjaBnKRQ6nRa7
    JNWd41soKanAarWh0aoRRaFArXb+7J5xQ2jVKmLQvPnr9es37Gfs2AGEhoZ8u39f3tG4Tv1x
    CDPdAo1W86LVam9mMllRKhWFYFgoKkqtKmUUsizvCAy0/3rX3f1ePHjoJL16dWD0qL6m9u2i
    1mh1upeAPjq9FrvdXt+W0pIKbFYbGo0arqkTsATEKgwuOXFWS+ULxSXl7mqNktKySlVl5dln
    3T2NdnBbq3OxVt/cCpY+OTIy2OTkpNPW1JgusiDIsoxWqyEqMsQky8rkG80rLy83hcGgo6qq
    FpDR6TXcemtP7rtvCL/+upafflyFQiFcsJ5kwsL9+eKLF6mpMXP4cMq/ildqtapeOVGplDcV
    r2RZxuhmICTEt9E1bwHIzMznn+tYDl6dU8YvnN8azR94FRMxaP78Dfr16/cxZszf8KrW8ue8
    uqvviwcPptC7dyyj7uhjatcuao2ujld6vbbBse/5tmj+z/JK2TiTLRnIsioU+g13ju571/Zt
    h1TvvPMjH330DN4+nlgsFr78cj5JxzN4680H8fJ2X6FUts6U5WSud54WWT4FWFqfLcx85asv
    F2iKiyvYvesYqaey6NcvjiFDutIxrgVbtx7Cw8MVhULBuZBSNzcDFosNhSji7+9LcJA/AwZ0
    5uDBE/zyyxqWL9vJgQMphhdfHDNl0KAe8SqV/gm7dDRXIba5AQDIBDQCFDxxOiN98ptvTlfn
    5hbxwQdPIyPxn9e+4a03v1G//fbDk2O7hMhYbTNkOUMShLAbQCuH0+SehCTuv/8dJElClhoH
    W4LoOGM/kXyakDAfkP+J36hyS6dOMSVRkcHukyf9wKKFW8nIyMXN1UCvXrEVSqU6AaTTGrX+
    3vjOcVPPnCl8bv68zezYcYSePTq0iYwIEgFJEKKQ5cyw3NwiY35+EWazldycIu/QUFOMwblj
    ft1LEazWau9t2w5TdLacrl1a27t0jX/HbrZ+Lsv2KkHQp/Xu1aF88aLNrq+99jVRUcGcSM4k
    MNCbrl1al8h2ceuVz5kCoBhHgk2pjnnqup9rAB8EwQPYBQQ+kJeXNe2br+cZV63eTWVlDd9M
    XyQcPXKq24Sn7+oY07rl25Jk/8Ahi2QcOZrUONIJ1NSJES0Oh2stghB8HdeGDUgGFHtjO7TM
    GjQ4Pmrxom0Oo/sF4fSCINB/QBzxnVtn1VTb98pyEtAcQVDeEF5N/2qBprCwFEEQsNvt9O0T
    y2uv3UdaWg6pqTn80RgnCFB0tpxTp7Lp0CGa4cP7MHBgFw4cSL6peZWRkT757f9+o96390T9
    0e2BAym8/fa36rfeHH9T8Mpms9MhtjmffPIMSqUSuZG0LEFw1P38859RUlb5j3kV16lVSZu2
    zdwP7D9Rp9wIhEf406Nnh4a8io+beia78Ll5v29qFF516RxzEa969mxfvnDhZtcTJzJxJFiF
    Vq3C6NI55v8sr8R/vjCO1DVG089kKrlfkiRFq5hw9u5NJjHJkX8jL6+IVasSCAjwws3NmfLy
    ip6SLekxMChkKeN6Lw+dzVY1edYvq1pv336Yhx4ahsGg58SJLAYP6UXffl1wdXVmxIiedOvW
    7gLzrkzfvnE8PWEUbsZzPkB2NBo1Xbq048MPn2HS5Ecw1Vr4z2vfCL/MWj7CZCqfgSwEmGqO
    XGdYJQEqTyh++ciRI++99NJnzukZeUyd+hh9+nSkb59OTJ78MKdSs5n42nSPxMSUD1CpJ4LS
    0/Hd60srlUolenkZKSwsZcf2I5SVVaB30qDVq9HqVVf90TtpKCuvYOeOIxQWluLlaUSlUoom
    s+VqG5sQEhL49uR3Hq3q2q01BQXFWKxWBg/pQpvWkUdqamwHBKEFFrO5BvQ1Uh11d2w/QvaZ
    gu6u7kLoefN/bVTOmUKPmlozkiRxOjPPIAhyYGVl/rljoYgz2bm37Nxx1GHst1jtsmxPNFvM
    VYLYBnA6ENU8/MDIkbeg06opyC+hfYdIpkx9rCYqKvQ9i0k6eGVz5iSOfEq+GnAOBm0f0PUB
    TW8wBICPGtLrNlNBAaWlOc9Pnfq9ce26vdw5ug8ffPAUDz08nOPHT/P669M1aWmpz4Iqzmw+
    CBwD/JWgCwNNH9DXfXRdwM0VgkRZPt5IxyKXU44Bej8onVhcUurTIjoEF1enPxzlyDg762nR
    IpTy8gofjdY+EVR+cOqG8OqX2atbr1y5u8E7Ono0jYkTp/PIw++ya+fRi/zIRFHkzJlCnn7y
    Q55+ahorV27BYrHSpUuHm5pXE1/9wnnr1sMNFEYB2LzpAK+/8Y1H0k3AKx9vD/bvP8HcuRuo
    NVloDPcpQYBak4U5c9ezf/8JfLwbiVeTH60aM6YfsbHRhEf4M+RqeZVT6FFVVYvJbOF0Zv6f
    8OrY3/CqOzExocTGRnPHHb1uGK/+97+nefg68KoRtmLBgGrkqVPJX3737ZLADRv2gSDQITaK
    8HB/QMLd3YWuXWPYtu0wzz33Ke3aR7V+4ok7Pu7SpUOIIBjfP5t7tMLL/3rtmsS+h48c7z93
    7noGDenCiy+N4+4x/XF1NZCWloFWqyYgwJuLE+fKtG0bRdu2zetAfF7xAjt6vZYxYwYQFOTD
    W299y4cfzkGlVg67686BX0t21QOlpYdLjMZ21xhUhTjuixID7PaiLzZt3HXr++/PEhRKBR98
    8BTdu5//+wMHdsXZ4MTkyT/w/HOfOk+ceN+UXr3jOyoUThNk+UhOdWEbDD7Xw1vOYjcYjNtf
    fHFcN6VC4b5mzR4MBj0TJtxJmzbN6szaVyN4BY4dS2PKlB9QKBQMHdqNCRPuzHXS61eeySu9
    yqR9oh1Zmt6mdXTulHeC7zCZLQOnfzXPffmynQwc2KV527at2snyse2C0BpZPq22mK1oNCpy
    c4vYsycpMCIitLfsyPongBBzKvUMPj5GdFoNaak52KyWNmazTZDlXTLU3rpv//GQ3LxiNBoV
    VotVbbdbQpxcYpHlQ4BrbEpKessFC7cwdEgXHn74tnK1WrlRp9UuqamSf1OpZNvlzZkTOIJ3
    nPxBvquy8sTA/PyisLLSqjCLxYper5ON7oYMby/3NL2T2zpw2gimfrt3H2mxdctB3p70CHfc
    0Ydz1350jI1mwoSPWblyl+9TTwXdqlY7F4Dcy2Y9OajwbEmnwoLSoJKSMhAFvDyNJl9fz5Ne
    Xj77Qf+V1brxmCwnIggx13CNZAC2oMrKgumLFm4aNnv2ajIzC7Db7Rc7I1fV8Oknv7NgwWbX
    cfcMmjB6dL9oZxf/10zViQe0TjHXlVe//rq+/moWhQLCwgOx2+wsX74DURAvefRyrhQVV7Bm
    zR62bTtCl66teeThYXTu0oYxYwbeVLzavGn3rR98MEs4efJMXVQnDcYDBHbuOMYLL3zu/Mqr
    46b06nXjePXSS+O6ffrJXPfPPpvH4sVb0WnVyMiOMZKvGFUOJ3cEak0WsrIK6NO3I88+c1ej
    8Wry5MvjlSgKZGbms/cSvEpLzcFisSHLMhnpl+ZVVlYBoihgtdpuWl7dfnsvKiqq0Ot1xF4O
    r0rLEAQBz6vglfKfLZBUoKhfamral6//5+vA9PRcxt07iN69YwkN9cVodAEknJ2dmPzOI2Rm
    FrB/XzKzZq3m1Ve+1L/33hMvd+vWsaCqXPqc6+BEKssVQFbLPbuPOYHMPWMHotXqcHV15ofv
    l7JkyXYGDurMW2899Cc7kr9yWHT8f/fu7Xj//Sd58cXP+eTjXwkI8BrSq1f8Y2dz7O8f3HtY
    7tCp3TXq27l2mQZWVOS8PWfOqs7ffruEVq3CeOONB4mODqtTGsX6c+mu3drx+RcvMnXqj7zy
    yhfCQw9n3nbPPUP9XF39XnfyPrrp+hncPT6Nimp2dNKkRz8PjwiI+unHlTz37Ce8+PJYYjtc
    3RHysaOpfPjhHAoKSnh6wh3cO27oKU8vzwmnEkvWqpwNV7nDDEa2HbYjSgsLzqqWBgc5vRwU
    7DslKytfnDt3nXdEROBrBoPxcELChkqwy3abFS8vN7y8jWzcuE85bNgt96jUyb/rtC2qzaaK
    Nmmp2bRsEYZOpyE9PZfa2poW7ka9CEG6mpqcgevX7xUimwVRXl7p8CEDwRFa7+RsNhf+99e5
    a33TUs/g5eVmd3Ux/re0yPSNViuZZSTU+vaXOWdKAGX/isrstzdtTOi6bOk2kpNPU1trqXPw
    FjA465o3iwhs3q172yG33NK+ICIi0HzkyCnRxcWJuLhz42MDBDrERhMVFcyxY2lYLLXj8/Ny
    Ru0/kByxc8cRMSkpg6KiMiTJcf+aQiGqQkL9YocM6Ro7YvgtvX39Av8H/FRaeMhu9G5/jdZI
    pmdtbekX3327aNg3M5ZgNln/PIJLBqvVRlpqDu+/P4vS0or+Tzw+GqXCaeyRPaeL2saHXhde
    7d2T6HS2sBQQcHMzcP/9g7ljVF+SElN56aUvqam91KQfAAAgAElEQVQx/bWYFQVEUYHZbGHT
    xn0cPXKShx4awYPjh9O9e/ubgldz567q/O2MJZSUVNTnlgLqncgdOaIc/6akZPLKy18IDz+c
    edvYG8irdyY/9nlEs5VRP/64krLSSlq2CiM+vtUlI1L/qkiSxN69x0lKzMBoNPD447dy773X
    n1eS5NDjrFYbGzftVw79A68yMnLqxy0jI+8iXm3csE+wWusCLRzP3XS8io9vyYYNe/nk418Z
    ODCehx4ecRGvDhxIjtix44iYlJROYWFpPa+USoUqNMz/inj1Dy1YCoPJVPzqd98tCUxLy+G9
    959gwIDOnE/Lf97Ko9fradEinBYtIojtGO1Y0J/8qgwPD3wuICJg1ZplqacGjbjmt8QLgFdm
    dgGBAV40axZIcXEJb74xg1WrduPt48bAgZ3+xmoiNFCoLrFc6NQphuefu4vXXvuGTz7+VREa
    4jchPCxsa3CQsOvadu90RHl5/ueffPxz1Lx5mxkxojsvvDAGbx/PeotcZqZjkYSGOqyLUVEh
    fPTRM3zx2e98+cUCCvJLOj/3/NjJRqNnsiyn513rawwEQYssZ1lBvdbTy2P4hAl3vRMVFTz6
    449+Ff4z8Ws8PNyuqt6SkgrcPVx4970n5IEDuyxUq3RvgT3ZL1iLs+vV3ysnKB0CJy9vj00Q
    DNu9vd2tOp1Gs3LFTvr1je3bf0D3O+Pj+34P6YDjup8BA+KYM3sdKSdOt4vr5N4SSrMqKmtC
    T5/Oo0/fOAwGHfPnbaSktDJUrz3rrVCrgo4np3dISkzn/vsGsXjJ9gtmmxqoHrpz54FuixZv
    RavV4OXtbkNUHpEFq1lUdL6C3swFug4oKMj+7qOPfglevmwHgYFeDBnSlcjIIPQGLTVVJrKy
    CkhMSuPbGYv56ceVPt27tyEvrxi7XcL8h+MLSZKQZYmcnELef/9nn507jvqUllQSGubHLbe0
    JTTMH1c3A1arjZzsQnbvTuSTj39ly+YDUW+88eCnLVtFaswmecbJwwftUe0a+/6/QsA8avuO
    gyN++GElFrOtgTC/9Px0pEOwWGz8+OMKWrUK7ztw4C2jWrYt+eY6CHMB8MrOLsRud5wEvP7G
    A9xxRx8qKqtZvmInJpPlshSZc+kCFAoFxcUVfPrp79jsNp56cvSN59Uns6LmzlmHxWK7SDnp
    1Kklogj795+o93MSRZHi4go+/vg3CgpKOj/73I3hlccFvPrk49+EkuIK4jq2YMTIXldU57Jl
    W1izejfNIgN58cUx14FXWs2qlRfzSrjgaPnQwZOcTMls1zHuPK/S03Pr5hFkZuVTegGvkpMz
    Ohw4kIIoiMgNZON5Xi2+lryKCsLJSUt1tZmsrHwSEy/NK4vFSmZmPkeOpBIa5l93yfY5Xs3y
    2bHjiM/ZwjICA73pEBtNRLg/bkZnLBY7Z84UkrD7GB9Om3PZvPqHCpYy5uTJjNhNG/dz95h+
    9O8ff27f8ZfWn5YtmzFhwmhefeUr9uw5FjJ0iNcYQVC9czz5uNyyRctrad7VWq22rmUlFagd
    kQvMnr2G1at3Y7fbcTe60Lx5yF/WUF5WiUqtrI8u/LNNzpCh3UnYk8Tvv21g1qyVfq+8cv/r
    KrXznTb7iWqlIvoa9M2ssNurX/vtt9WRv/66gXvvHczzL4zByUlfr1zZ7XbenfozsgzTv365
    LlxWwtPTnYn/eQC1VsPPP68iMMi72/jxIyYoFPrXr4dlURCCObRtNa06eZ1E1j45fHivI15e
    bv95ZsLHTq6uTvQfEHdF9a1ft4/aGhPvvfdEbdeu7aZZasUvrLKtKGmvRPsenRqlzX5+8chy
    Br6+nhgMeiora/jxx1XqNm2bv+TtdXyjIGolAFmSiYtrydKlO9i+/bCxfYcWtymVlb+dPVsS
    VFxSQUSEPy7OTpSXV1FQUBwQEhLpI8nmsdu2HTbqdBo6xcewaNG289o7BBUV5b30008rNWaz
    FRdnHX4+7pIsSxYP78uHlePON2tgWXn+ux99NDt46ZLt3DNuIA8/NBz/AG8aumfKmEy1ZGTk
    snp1AgsXbiEvtwhBENi3L5nIyJB6fSAjI5f0jDyKzpaRn1fMoEFdGDmyOy1bheHs7AQoGtT7
    4PgqVq7cyUcfzuWtt741TJv2zNshISHHfII02xt/pil0JlPlsGVLtgkVFdX14eKXddgiClRU
    1LB82Taxe/e2w0T0PwPXOF1AHa9KHcmlR43uwx139EYUlVRX1ZCUmIHdZkf8CyXRaHTGZrNT
    WVlTb6UTRRGLxcqPP6ykXbsoevToeMN49ftvayJn/7IWu11qoFzJsoxOp+bFl8cgCvDAA1Op
    rTE36IPNZmf27LUEBN4cvJrw9EdOq9ckEBkZdNk+WbIMq1cngAzvvvv4deGVXq+hqKicH3+6
    mFfnlPHi4nJ27DxibNc+up5X5wIsBAGKi8opKCgOCK7j1Y4dR4zn8ltdEKBUz6uff1qpKSur
    wtPTtVF59cjDI/Dz9wFkzGYTkiTVBR3IZGTksnLlrga82rv3OF27tiY4xJcBA+LJyTlbz6us
    zAL69onl9jd70bZdJHq9FrtdQhAEtFo1oKS0tIylS7by2WfzeOu/3xmm/W/CX/LqHypYNa1S
    TmQZ7XaJvn06IgiKP1GuLmXlaYm/vydHjqYKI0f29KuqUctgvYbm9hNAVURWVnZoSkomvXp3
    xGq1sXnzAeI6taBFi1Dat4/CxcXwp5vJmppaJk78iohmgbz00j1/aUXWajU88shI9iQksXDR
    Vnr07NCrT58u/UBaem16mBeUmXm6z++/rRc6d27FhGdGX6BcCfWLuays6hLWOAmdTseTT95B
    YmI6837fwMABne8KDnb6Fjh9Pezu7XsMdsyoqkPF4P5DcJDfoy4uTk6t2zZjwoS7r6iu/IJS
    ampMhAT7ngXdDLtUXaTXX4uQbudco9E5x81oCG/TJoJDh0+xfPm26AceGPmEUtQoASRZwsfX
    g65dWrNp8wHuvrv/EP8Ap9TM07kKq8VGSIgfWq0aGcjOzlfHdTT3LCjI7bll8wG6dInBz88D
    SZYRBQFBxk2WKh9atXJH7LGjqfTo0ZbMzHxc3ZzP2O3yFSbtlUVZrn122dJNHRYv2sLdY/rz
    8svj0OnOJWdtuI61Wi0tWkTQokU4gwd34bNPf2f16gQ2bNjLbbf1RKfTIssy69bt5WxhGR1i
    m/PSS2Pp3Lk1CoXyAot2w3pdXJwYM2YgGq2aN/7zDT/+tNxr4qsPviHIwiigsnHHq8r9bGFx
    zIUJOa+kiKLAiZQsiorLYkKCPd25homSL+TViROZGJz19O7dgaSkNA4cSCE1NZv8/OL6DNyX
    slw5GXS8/8GTFBSUMmnS90i286lKRFGkpKSC775bSrt2Ubi4uNwQXs2bt0GwWu0NLInnrG6i
    QqwTbNSP14XpBgRBwGq1s2DBppuCV0aji9PmTfs5sP/KMnpUVFQTEux7XXjlbnTJcXHRhxcX
    V7B3z3FWrNgeff/9I55Qihql/AfFb9Om/dw5uq+DV5l5irKyqjrF0ZEsNftMgbpjHa82bz7Q
    IJJSuIBXq1ftiN216xiCIODs4oSrm0uj8OqVV+7FbrezYsVWEhKSOH06D5PJjIeHK+3aNmPA
    wC688MIYhgzpyqef/Mbq1QmsW7eXwYO78MD9Q4iPb8lvv67nbGEpUVFBvPjSPfTtG0dJcRkr
    lu9g797j5OeXICpE/P096RwfQ7/+nXjgweHo9Bre/u/Mv+XVP1SwLJSWVaDVqvHwcL2izYPB
    oMPNzUB5eTWSLEkz5wXzUZvkawSrY4BWIUkVj6xevcu/qsrE0KFdcXY2MHHiOHx8PAgL8//b
    4z/JLpGZmY9ao7oor8illMjIZsGMGt2baf+by8IFm/QdO7Z8zMXFcwNwLfIDdU88lhaUn1/C
    iy+NxdXVtU6YCWSk57BkyTZMtWbOnCkEYNoHs1FrlAwb3p3mzR3+WUajG2PH9Oe1174h6Xh6
    UHBwcMfrBaxzRW9ojyxnKSS5DrQyXBxw8HcD7vjY7bIMsqA3XKt8OcpcJydNjpubITw0zA9P
    LzdmzVpNly6tx7dq1SLrnFOySqlgwIBOLF++ncOHU5r5B/g9c/p0nt7JSYeXlxt6nRqjm4HU
    1DMaQbQ/fOzoqfDsrAJeeP5uVColyCDJICrEYSkpaQE//riSW25pR0ioL6UllTgZdAVKpWvh
    Fe7DW+bk5o6dM3utEBrqx+OP33qBcsVfWqBbtIhgytTHsdnt7ElI4uDBE3TrFktOTi4rV+yk
    dZtwpk17mmbNQi6prF1cLwwd0o0tmw6wfNkORgy/pVuH2A7tgW2NO14SVdXV1NaaubqkaAIm
    k4XaWjOCqL6Gm8HzvFqzZrd/ZmYBgiDw2WfzqKysJuVEVgNB9qezUykSGuaHXq9FFATsyA1C
    /kVRZO+e42zYsI/bb+9z/XmVmBaUmZnfQNmVJIm4uJYMGdoVjUZJYIA3AP/5z/1YLDbWrdvD
    rp3H6q1doihw+nQ+ScczbjivAGprLZhM1iscb8eh2vXgld5JnePt4x6ekZGHzWbjl1/W0Llz
    zPhWrVpkyZLUYDNx6mQ2h4+cbOYf4PdMVmae3my21L93q9VOelpOPa9OpjTctEiOuxmHpaSk
    Bfz882qsVocPu6eHK04G7T/m1RNP3IbZbOGdd75n+bKdjjVZp/zJsszaNXuZO3cD94wbyL33
    DmHqu0/U8yrxWBr33TeUs2dLWLFiJ2Fh/nzwwVO0aRvF0qXbmPndUk6ezMJqtTdI27Js6XZW
    rdrFO+88yq239mLHjqN/y6t/mKZBKXi4u2AyWSguqbgiaFVXmygvr8ZoNCCKCrH5NU1n4gRY
    hyYmnrjvt183MHJkdzp0aI4oCnTu3JawsMCLhMil4FpUXE55RTUREYF11rq/tSUzdGg3QkN9
    2bbtMIcOnegFtv52+/Fr0MeqtidPZimNRmdiYiIu6IdA9pkCVq/ezeq1CZSUVFBSUsGatQms
    Wp1A5um8BsKuTdtmuLo6ceLEaRVYO9JU/sqmITk7u9m8vIycLSzl3nEDsdtszJ69xtNmq+lg
    tlgdypEkE9M6gqAgHzZtPqi32UxtTqVm4+1txN3dFYOzE94+HqSeysZiqY3ZtOmA3t/fi7bt
    Iusjx0xmK3bJGv/rr2sDK6uqufe+QRQXlePu7oyTXm+3WC1XGGlU0+J4Upp/WloO3bu3+ZPI
    2T8rdjw9jbz00j0Y3V2YP28TFouZtWv3UFpayYsvjq1TruyXuemS0Wg09OkTS3l5Nfv2J+sF
    wdL7GqguODs7iXq9hqtLXCSj02lw0muRZet145XDr01m757jnEjOBISLrmC5SJWUZLy9jBhd
    nUlPy8FqtV30vCA4FMaFCzdTUVkJgnhdeZV6KltpsTRslyzLREYGMmhgZ/r2jcPV1YCrq4G+
    feMYOLALLVuEXmQpMdVaOHky8+bglXAuqODyP9fvahMHr7y93ZFlh4KdkZHL3Dlr63hla/Be
    a2pMbN1ySG+zmdqcu2fwwnFKTT2DxVIbs2XrofrbBc4Vk9mGXbLG//b7usDU1DP192d6ebs2
    Cq/8/X1Yt3Y3SxZvw2KxolQqUCgUKBQiSqUCURTIzi7gw2lzmDbtF/R6HS+/PM7Bq/mbUKqU
    bNi4n4KCUl54YQxt20Xz44/LeevNbzl+/DSSJF9QpwKlUoHdLrF58wF++209Go2GgQPjqKio
    +Ute/UMFS58eGRVSKYoCWzcfQJaly1SyRA4eTCE3t4iYmAjZblee+PRDgWvhfyXLaYAUXVqW
    /9b06YvcnJ31PDh+OErluSzB0mULlvXr91JVWUtcxxaX2U+ZkBB/evRsT2lpJWtW79aZzdUj
    7HZB0fj9tMglpZU4O+sxGC70D5Pp2rUNs355i1m/vEV0dAjR0SH8/MtbzJ79X3r36djgWWdn
    J5ydnSgqqkDGrqCp/NX8t6rV6u1+fh6czswnJNSfu8cMYPWqXezceZhzu1pZlnFxcaFP31j2
    JCSRlJROVmY+wSE+6HRq1Go1EeH+5OYWk5SUzu7difTo2R6j0a3+qg1BENiz5yhLl27nztF9
    adUqjKysfLx93NFoNUlZWWevMBOynZwzBdhsNsLC/WnoG3V51qDo6DDG3TOAXbuOsW9vIqtX
    7aJP34507972yq2OQECQFyq1guzsfEGWbU6NP166Si8vz7SoqCCkq1CwJEkmKioIdw9jcnlF
    ddm1sV6d59WMGUvcHGHvDkwrFOJF6QsuXYdDyA8a1BmDs44tWw+di0K9mMSiyOHDpzh4IAXg
    uvKqtKzqIgVcFEWWLt3OmDFvMv7BKaSmZpOams34B6cwdsyb/P77xksc78oUFzfx6rJ55ete
    P4dEQWDFip3s2nmEP+Z2FgSBhN2JdbwquOh3ublF9by6lPK+Z89RlizZ3kBWens3Fq8EDh06
    xR8V9D/OJbtdYs6cdcydu4bmzUPrebV37zFWrtzJLbe0YdCgzqxbt4svv5hPdXVtXcSqcAl7
    ieMi7OPJmdjtVoKDfdHp1GRn/Tmv/qGCJe1u1ix4Y//+nZgzdz3r1++pEwjCn6v3iCSfSOOz
    T3+nefNg4ju1zjWb5I3XBlYFgEJvtVa9MXfOmtiE3Yk89fQogoL8rkAACICCw4dTmDVrNfGd
    Y2jbLvIyvy+jUCjp27cjBoOe3bsTycrK76lSCUFTpqxp5J2TKKjUSqqrTXVh2+ePO5VKJb6+
    XgQF+qDRqFFrVAQF+uDn541KpWpg7aqtNVNba0av1yIIotAEpb8TZXK+n58HpSWVVJRXcded
    /QgI9Obbb5dSXFTusDSIAqCkd69YLBYrixZvIT+/mKjIIGprLVRX1xIe4c/Zs2UsXbqd2loz
    /fp15Nz9noIgUFJSwbczluDlbWTM2AGYai3k55fg5+eBUqlMcXd3vmKNRlQoEEShztn76oZ6
    0ODOuLg4MWvWagrPlnLHHb1RqTRcja+xQqwDmyzANXFWrqnQaJx+GHlrT6urq6HeOni5ypWr
    qxMjR/awGAyGOW6umupryatf566NXbd2z1X5ikmSRExMOOPGDWLLlkPs3ZP0p6kDBEGgqrKG
    jRv3Icv268orpVJ5yfbU1JjIyMgjPSMXi8WGxWojPSOX9PRcqqpqLyH8BHQ6TROvLpNXvr7u
    9fNBEATKyqqYOXMZRUVlF+WCy80rYsmSreTmFSEI4gXKi0B+finLlu0gN6foou+VlpQz87tl
    lJZUXnABuYivb2PxCsxmy98m+xQEAavFysyZy0hMTGXI0K71vMrLLWL06D6UlVcy/atFlJVV
    /22KDQGwWMzIslRvMZP5c179QwXLWqvXe3796GN3VISF+/HaxOl8/PFciovLLwFsAbPZwvx5
    63nm6Y8oK6/i2efutnl6eU0rLa25Rul4DYB53K5dB0d/9+1ShgzpzMAB8ZfJbocyKEkS+/Yd
    4623ZpCdVUj37m3Q652ugP8SrdtEEBHhT1ZWAUeOnAwGe+fXXx/YyJZp57KIsAC5qLiMU6ey
    //D+HZY6URS4976B3DtuUN2kl/7QD4FTp7I4e7aU4CBvCVTpTVD6q8WrAvRCQIA3VVU15OQU
    4e3jzSMPjyDxWDqr1+xGVIiknjrDrFkOgEW3CGHp4u2UlVWRm1fEww+/y0Pjp1BQUILJbGHh
    gs00iwxEEGDTpt0UF5UhKkTWrU3gwMGTPPzQcIKCAsjLL6aioprAAB+QRZuH+5XmC1MREOCD
    UqGgIL/kqixOIBMY6EunTi1Zs3YPPj7utG4dfpV1QXFxOTarnYAAb1kUlJWNP17NAKcVPXp0
    2jn+4WGoNcrLUrIkSUatVvLAg8Po1r3DeotFWCoILa8pr2Z+t6zOB0S4in46MoIvX7GDzz79
    7U+Ukob0OHAgheLiMkC+brwKC/OTLyXQBMFxvZXNamfd2j1sWL8Pm9VeZ70TLmGpEAgMbOLV
    5fLK39+7QVCBKArs2XOctWsTLlKUbDY7y5buoCC/5A+WU4HS0goWL9ra4JLxc99bu24Pu3cn
    NtggKJWiwxWhUXgFbdpFoVar/lbJEkWR3JyzLFq4BX9/L+LjW7J2zR7cPVxo3yGaVSt3kZSU
    8bfKlSw7rl5rHROBUqmmrLQSk8lC4F/w6h8qWBogbGNYWPiHH374jCWuUzTTv1rIggWbLqFg
    iezdm8Tbk75Hp9Py6SfP0a1ru3nFxZYfg4L9rlFobWmz4qKCV2d8s0jt7e3GI4+MRKXWXpZi
    ZbFYOHIkhUmTZvLkkx+SmVmAQiFSWFh6xULIw92Vdu2aYTZbOXjwpNJkqrklN7eykXdb6vWd
    OrWqdHN1ZuHCLdTW1lw0BoIgMHRoD4YN73HJXWBtbS0L5m/GYNDRpm2zAlBubcLS366BQx4e
    bqWyLJOZ5bg2YsiQrvTs2Z7srALKyqqYNOl73pk8k1de+YrMzHxKSyuprbUwf94mMtJzSM/I
    47dfN2CqtVBZWUNJcSXPPfspTz05jUmTvqe8vIqM9Fzi41oydFh3QCQ3twizxYqHp2uFLCuu
    YoOiTwoP98/z9XVnz54kqqqqrsKKJSOKSjrERiEKAjGtIupSMVx5KmtJsrN71zFUahUtWobV
    gKbR554snwZyS7Va43+eeGJU+iuv3IOHh/NfAlqWZYzuzrzy6jiefOKOdJVSN1mt1lc66mp8
    XhUVFbz63beL1cXFZYiicMXXB51TGIuKyvjfB3NIScn+W8EhigJZmfmcPJkFCNePV3GtKgMC
    PC95fHnuTr4ZM5bw9fTF2Gz2Sx6NSpKEj687bdpENPHqMnnl7uFaqtdrG+RIs1isZGcVXFLZ
    PXehfUMlyjE+5eVVl1R6T2fkYTafV7xkWUatUeHu7tIovKqurmLE8FsYMDDOESRwGevk0KEU
    TCYLMa0jAIiODsXJSUvCnuN/Or/+ONc6dIhizNgBACQkJKJQKv6SV+LVwyoDxzU5qZ2rq8uH
    pZzIVFRXmZAkGbP50g6gVosVm9WO2WzhVGo2hWeL4z3cdS+ZahVGWUpuZJjmAlW9tm/fH3Lk
    SCqPPnYbQcF+HDqYREF+0Z8IEwFJkkhIOMqLL3zG+Aensmb1bvr1jeXNN8djMOgw1ZqvYueg
    pH2HaNRqJcnJGZSXV7T18iowNO7C0Sc2iwzfdOdd/di0cT8zv1tSFy2lqBtmBY6gUbHuo2zw
    u9paM999t5iNG/dz1119ad48dBGoTzQB6e+KKcfd3bnG2dmJGTOW8MMPi0hKPk2bthHodBqK
    i8opL6vilVfHceutPRAFEQ9PF3x8jPTpG8v06S/z8cfPEBUVhEqtQqVSkp9fTJs2zervysrJ
    LULvpKPfgDhKSirZvDmBn35ahVajxs1oqBJEdeaVt9uSFhzsv71f/zgOHjzJihU7L9hgXL5i
    BODv54lKpSQw0KtuTl2pLBbZtz+J5ct30rFjc9q0aXbCbLElNy4PSoFywMsLqu9LTz3jWZBf
    wuWcEsqSTH5eEekZOZ6iwv4ICFFQjFyb0ei82rH9QMjevcmIoogkyahUSgRR+Fuf/HOO8H5+
    7owdO4A33ngQg0F3Wb78giBQVWXiyOFTOPz9rh+vRo3uW+8AfaFSK8tyA8Xw3DN/fE6hELlz
    dJ8mXl0hr1zdDA3e5aWCJi5833JdsI4kSXUfuX5uXUrB+WN9sgwuzvpG49Xy5TswGl15/fUH
    6d27/SXmxvl2nWvr2bNllJZU4OfngUqlJCTYl5oaEzk5Zy94Tr5kfyRJon37SCZNfoSQkCAS
    E0+xePE2YmP/mlfKq4NBFqBUQcozaWmpz0//akHAunV7CI/w58UXxzBqVJ9L7GIl4jvH8Pbb
    D7F2bQJTp/7MsmU7Ip5/fuybnTq1CwTli1bTkVKVtm1jIUtjtdYM2rjxgMLPz4MePdqycOFG
    pv1vDi+/Mo677hrwhzaKlJdX8sP3y5g9ey0uLnruv38wffrE5jdvHl6xbfvBKLPZgtHd5Wra
    QlRkEG5uBrKzCsnLK2rh7R0QiuNWyUYq+TVKpf7Ve+8d7F1SXNb122+XkZycxV1398PV1UBR
    URkVFdWOzNsyqDUqXFyc8PR0pbKylt9/38C2rYe4Y1Rv7n9gWIJG4/Q5SPbrvfwdgrDKcS8M
    5yJshCsQ2vIfHhWk9PSDhId3uEYt1kkeHp5mN6MzyUkZfPTh3PoFrlQq6BTXgscev5V+/R1H
    0w+OH05JSTl6nZbAIG/UakfC25iYcNJSc6iqrMLo4UpUVDAqlYqQUH9mzFhCVmY+n306j6++
    XEhpaSXV1bU0bx6Mh9HVbrOZr+JMTjYrlYaZY8YMHLxzx1Hnjz/6FXejCwMGdq571xf6Ul74
    /s8HRMiyhN3uSOynUIjo9FpsNjsKUUIQFX/4/rnvXniHp6PeY8dOMvntH7DZJcaPH24zGo0z
    V2/QFTXenCrCcTGsU9eqqsIPFi7c0P37mcvJysr/22g8QRAoL69i5szlrFm71+Whh4Y9fOed
    g7s4Obk+irZkV3H+ITx82zcarzZvPqgwmRzh8O3aRfL4k7exedN+5v2+6ZJr4JwgaNYsgJG3
    9qB3rw75zZuHV2zffjCqpsZ02QkvJUkiOfk0NpsFpVJz3Xg1bpyDV7//vhGTyYogQKtWYQwd
    3h1Pdxd0ep1DLag1UVxSzvLlO0lKdJwEqtUq7rq7L/fdP/Sm4NU5Zdwu26+wjhvAK1cDWXJB
    AwXpvFLhiG5UqZWoVSpUqvMRevVpCySw2SVsNhtWmw2rxYbVaq8PyrlQyXI4h8u4uRoalVdu
    bgYGDerG1KlP8Oab37Bhw0FkWaq7sUB0+BvXbVoVooirqwGT2Ypep0EUBZydddjtElqtCv//
    1955R8dRX33/OzM7M9uLpJVWvViWbFyQ5YblBqYYF0wnFIPpAR5ChycJoecJSWjBmGIIAUxs
    MG50V8CGuPeCZcnqXbur7buz098/Zi3LOP5cYXwAACAASURBVCGWvVJy3qN7jg/n+LDenZn7
    +8ztNysNiqJCFCUIggieFyFJcoJzwJixQ/CnP96DwSWFOHSoCk8+8Q5isfi/5ZWu9woVBBBj
    gfBD+/YeePKpp9/Wt7Z4cPc9V+Lqq6fB6UzBPx93oMJkMuK66y/GnEunYP367ViwYDkefOBl
    PPHkbTdfdOEEQlXJhzyefX6nMxn7r3RmjhOHt7V5UVSUCUVR8bd3v0RZWQkuvHBcD8CrAEi4
    3V784Q8f4NtvduOyyyZj7twZHYOK85dRFLtIFOP3bNt2sERVtRfh6UiGKxV2uxktLR60trr1
    ZWWkNbkHZxiAquq0tKw7H330xjdLh+RX/O3dL6h77n4BrJ6BJElQFK0bjUgAlSS1llZeEJGR
    7sBjv75RnjN78lab3XEXoKvWalb6W9wA2AxA1cuKgkiEQ2uruzfmFSIRDrKigCShV1U5o7BQ
    1953v9fSaTKx+wryXUWdHV24994ruwdr5uZl4uyRxbA7HN36lpXlQlaWC8e7VzXW2O1WjB7j
    OMmQueqqC1BRMQK7d1eirrYVkqR9ZtGiNcjLy4DVaj4UjQq9NkYIogiq2rKxpGTQ87/73S3P
    Pv74W7rHH38LXb4grrxyGhiGhSQJ8PvD6Oj0wevxo72tC53uLgQCEXCcAIEXIAgyunxB8LyI
    JUvW4Ycf9oOhKTAMDbPZAEeKFa6MFLgyU5GSYoPTaYPDYU0Yliq+/XYnnv/DB+jo6ML//u8N
    6sSJoz4URWXxqBHxJD6jVACRCr+/9d3XXlsyZPHf16HnTJ9TifAQBIGW5k788fkP0dLiGXbv
    vde867A7bkvJcCZpnYzGq472LqiqitRUK377+DwMKsrByhUbT0jn9DSKzGYDrrrqXFx//cUn
    8Grr9kMlsRj/b1cB9bzGxsZOcJwAi4XpV1498siNb5aUFlS8/96XVE1NK2bNmoh77v4FThzz
    oTUb6VkalYcbUFSUiZtvuUS+ZPak/x5eSTIGFedg3PizesWrHdsPaxP5+5NXBZlF+/fXdBuq
    TqcNWdlO5OSkIyfbiczMNNgdFlgsRphMBuj1LFiG6q6pkmUFcV5CLMYhGo0jHOEQCkbg7vSh
    pcWDlhY3Wlvd8HgCEEVtQXReviupvPrtb95CR0cXrr12Op5+5k4w7CIosowhQ/ORk5OO1DQ7
    7DYTzCYj9HoGegMLpzMFkXAUBEGA1TNITU3Fm28+hjgXRzwuIRKJIRiKwu8Loa3dg6qqZgQC
    ITzy8A0YVJyHr77chBdfXAKvN3hKvDqNCNYPAIZeUVVV+eRvH39dH4sJePGlX2HquaMTQ+yU
    f6NOKkwmAy677DwMHVqAp5/+K558YiGh01Hzpk4ZJ3Fx3K+qR2MEMfgMFYmWGYaJGo1sd+iv
    tDQPl19xLlJSHFAUEYIgQq/XI+AP4dln/oaduw7jqadvw8yZk74wGKxPP/fMkD1PPFU3uamp
    dvqG9TsxuCQ3MWOqtwa4CqORhTPdgZqaVnR2+vSyHB8NYHOyjs0xLwE49KPJ7PjF3Lmz3/P7
    Q9MXLVqD22+/BMXFeXA4zGBYGgAgCSL8/giqqxvxzjufY86lU3Dj3FkboFC3Amhb9HZjv2JK
    ++1HAbCDBcH/6pdf/pDR2uqBxxPA/j3VvQpgebuCEAQRn3/xQ8a8eSmv6hnj7aq8/2jLERm5
    w5LtGQZElrVsvPTSKbM3bdrLeDwBPPLoTQBUhEIR+AMRHK1pQWdnF3y+MGIRTuvUjAsndMFo
    HhcLg14HvZ6FwWyA1WJEWpodWVlpmDq1HJfMngQQFF55ZQkURcXlV5wrmMymlaGQEDu93y6L
    UO2vT5l6Tub/PU/c/vtn3zM8/dRfUVnZiIL8TOzcdRg1R1vg9QYTZ4WBXs+AZWkYjHoYDCxY
    loHBwOC8aeWQZQXBYBg8L4KPC4hxPKREF5ggCCBIEnabGbm56SgvL4WiKFi0aC1omsQzz9yu
    zr5k8t91OvYxQRDDOdnDkqRXzQCac2JR74IFCz4a8sH7X0NR1F4v5D2WphIEEe+/9xWgqkMe
    eOD6dwhSvUJVa6sIYlBSeGUwslBVFTm56RhckosPF63GurXbQVEU9HomkfbXjCtXZioee+wG
    zJgx8QuDwXYCr779Zhd6Ux9PEEAwGEE4HIXFYv4P8GrWe4FAaPr8+cvQ6fZjwWtL0NUVQoyL
    J9bmsEhxWBDjeOgNDC6ZMxlz5878r+JVW4cXd911Oe6//zqceh0igfnzl+KtN1f0K6/mXDp5
    9rff7maystLw0EPXYujQAmRkpMBg1ENL81MnRaz/1e8/8b8KAAlcLI6OTh+OHGnA6wuWo66u
    HZddPjW5vHruPcPvn3sfhw/X48EHr8NfXn0QNEWCpCgIgoBIhEM0wiEYisLnC4GLx8Hz1ait
    bYGiqDh4sA5O53awrA4mkx4WswHOdAfyC1wwmQxgGAaqqkAQJciSgtcXfIKFCz+FyWQ4ZV7p
    eqdUdQDIYo+n8YkX/vyhPuCP4MWX7kNFxdk4PlPqVO12FaWlhfjTn/4HDz34Kv74/AdEdpbz
    hpLSwau4WOzrJGhSkGWpzYOKc8q/37QPUIE/v3AfGIYCoGLNmm04cqQB9933C7z7t8+xbdsh
    PPXUbeKsWVPmyzL9gqKInU881ZjJxXx/XLJ4TVZLiwe/+90tvZ5Yf0wYhobdpuW9PZ4ASZJy
    VrIPPUEQkISjoGiTGgw2pG/b+iOmTCnDL395xb981OdNG4Pqo83Ysf1H+Lr8FqMpLRoNxDDv
    l9P7EVY+ANUA2BKe9737waJVk15/fQUmTR6JIUMKoKi9M2hJgkRNTTPefHMlSJKccvO82X+j
    SONtOWcR1TU1e1BcnExo5QBofL9iYtnMCy8cd/GSj9ZDlGS0t3ehtqYVHq/mwSmyViND0zro
    9TQYhgFNU+j5FpRlBaIggYvHtfku0PZ+0bQOTqcDxcXZyM1NxyeffIdp00Zj8qSyTQTBLiV1
    9tPUl3yo6pEQQD88ZfKYphdeND81/9WPzUs/1oZblpbmoby8tNsjTE93IDXVCqPRkBjCR/6k
    s0tN1GlotRqSJEPgBQSCEXT5Quho70JtbSsOH67HO+98jmgsjlGjSnH//VfzEyvKPyAp+teA
    7DcYkvl8LJQiN//vypXryhb/fW13FPdMzpiiKPj739civ8B11nXXzfxVJErct3f/D8qosyef
    Ma+KBmWVk9+SqK9rwyMPzcfhww1QVRWzZ1dgVHkJ/vynxYjF4khNteHxx+eJs2b+E14tWZNV
    X9feSyNSGzoaiXAAiP8Ir7ZvPwwuxuP997/S0jMnFZBpLfqKomDXzsr/Gl4tWvTppPnzP0Fp
    aR7KygajsrJ3zYxlZcUYMqQAr81fBookp8zrD15VlM2cPn38xZs3H0BOTjoKCrMRjcbQ0tIJ
    rzeEcDiKWDSOSJQDF+PB8wLignDCq49haDC0DgYDq81ftBhhsZiQmmKFI8WKwsJskKRW33fh
    BWOTz6sXNF59uuoHtDR7cM0156Ojows1tS3o7PDD7fYjEAh3p/tkRdaCLYoKSZKx7JNvsGrl
    RpAkmRgmSoJladgdFqSm2uDKSEFhUTaKCrOwZcsBLFv2HYYNL8T9919zyrzqZQTLxCiK99cr
    VmwYunXrITzz7B0J40o5zYetIC8vG7/57Tzcc8+LePudTw3PPHPnb1XJuA9A25keA4Iw77jg
    /LHi55/9QL/77ud44MHroNfr0dHhxtsLP8WECSNQWVmPlSs24uZbZqsXXzzpUyGue1bHSCGS
    tLjicff8997/dMLixeswc+YEXHHluaf9ayiKAqvXvFO/PwxFkcm+OPwUzQDgZhzYf2RkZWUD
    ZsycgOqqJnS6vQgEIojz2nRxlqXhcJiR7kzFqFEl+O7bj7Fvf3XZtGnOKXrX2C/6P9ROl/C8
    /90PP/xs0oLXluOS2ZPwm9/eBIvldAxaApFoCH/+0yK88cYKAJh0042z32Vp/e0AqpL9klDV
    2rDJlPbGLbdeMnnbtkOm99/7CsWDczF0aD5mFE5AZmaqNlXbYYHRpAdN67S6AB11QmBOSRx8
    XhAgxEVEIjG43X60tHrQ1NiBqqomrF+/E1arCbfccknMYrW/AZBhqzn/DH7/EKjKIdHnx19G
    jBja+OJL9z/1wp//Puzrr7fiqaduw9hxw3C8F+b0mn1dmeknpD69Xh9unvd7GIwsXn75fk9u
    bvZvuJj6Mc1IUZouS+KLsB6Af9aPh6tuXLjwU+JYbVMynjnPC3jn7c9RdnbJ9WedVbqubOSk
    z5PBq2nnjRFXrthE+/1hrF+/E4BWX/XAg9dix/YfwfMidDodbrv9EvXi6Sfz6v33P52wZPH6
    U1jldXIESxQlxOPCf4ZXB6pGHjxQg5KSXOTlu5DpSoXdZgKr12oUeUFEOBhFR6cPDY3tOHiw
    Fvv2Hy2bNs059ezMcZ//53j1+aSXXlqCOCegrq4djz36eq+H2JLE8RVMr7zyMdR+5tVzz72H
    wsIs1NQ0w+0OIBiKIhaLQzxhsvvPaW7ijtAUjEYDrFYj0px2DCnNh7crAEEQccutfcerF19Y
    POzDD9dg+/bDkGUZsqz8ZNp/z3qw42l2SZIhiTJUqCcUxDc3u7vv1bHJ8KIo4ZwJI/Dii7/q
    Fa96aWBFC+vr6mZ8/NF6nHdeOWbPnogznwcoo7x8KObOnY633/4Mc+ZMmTBx4pg5qlr5FkEM
    PYMHkQpVrf9i7LgRX82dO/2yv/71c9TUtGLEiEHYubsSHCfgkjmT8N57XyE724mrr5q2n2ZM
    D1OUEiIp1slx3tfe+9tnVy1YsALnnTcav/7NPFgs5jMwJrUXv6oC4XAUsqwwHZ274MoYk8QX
    ihsA9JLUOmftum1UIBDBG2+sxEvcR5AkCSRBnhTmpigKBiODQCCC9et3GidWnH0Lxx1YHfZV
    SpaUof1FKxMgvrxs2ZpJL730EXJznRh/zlDs3VvV6zb1niAZO2Yodu2sxCsvfwyDgZ0094ZZ
    /wPgQZzaRvJePVfA/E1paeG28tGl5x892oKFCx9DTo6rxxFToKoSRFGCJGmelKwoJ3jpJEl0
    r2WgdbpEoTjVfU68Xh/uuON5pDsdKC3N3xaLEeuMRuHMfz05HH7/XokgiGUOR65hQsXId1at
    2sQEw9HE9/esg1GhKNo1yPLxjqKe/4/mEZLQ6UhowyR76h0JLsbD6w1g1qwKZGVmLAXi70oy
    BSOd7HQIqRPFriuXL//G1tTUecr1SKeaLmxq6sTyFd85flNScFswvG8tAD4ZvJo37+LL3nhj
    JQRBKxpm9QwqD9fjo4/WQxBETJw4Eldecd7JvHrv86sWvLYCHCec3oBSVf2JcdC3vNKENUqS
    b87qrzdTefkuvPHGI3C50hJ6c6xR4ljpiQJZkuDx+vHA/a9g7ZptxokVI2/eFTiw2mofKaL/
    xASIL3+ybM2kV19dikmTzsbEipHdFb2nSxAVwNZth7DgteX9xquxY4eev2LFRmzb9iNMJj2s
    VhPy811IS7XBZjfDZrfAajFCr6dh0LMnOCiCKILjeG3tXSiKUDACvz+MtlYPqquasHdPFWRZ
    wZw5k/qUV+dMGPHO0qUbGFlWoKN1sNu1PccZGSmwOyzd65bMZj1YloaeZbodPlVVEed5xGLa
    iJxQKIpQMAxvVwhtbR6EglHwvAhVBUYOL+o1r07ZwNKmDPt/8f33e7P8/jCuv+Ei6PWGpDx7
    giBwxeVTsXLFRqxZs5WcMGHkDL9ffO9MgJX4l4Ms6/z97XdcmWcy68uXL9+I6uomxDgeY8cM
    wcEDNfh+0148/vjNcoYr468A0UxSbBof9/7lgw++uOrNtz7FpXMm4eFHbkCaM+UMjavjEo/x
    kETZykUJKrmHh6SB8AONja3Tv9+0HxarCeWjBmPEyMHIzEyFw2EByzAJj1BAIBBGZ4cPhw7V
    4R+bD+CH7/ehtq51Uklx0T3hIN4GEO8nYNGKohT88P1eRCMxtLZ48OTv3jlt4+qEFIQsg+N4
    bNlyENdfP10AqKR3GmkFmG0xhrFUFRRknb9r5xF0dvrh9YbR1NiGxqZOeNx++HwhhMIxxKJx
    rVNFEBPrpRI3QUdBr2dhMhthtRhhs5qQlu5Afr4LBQWZUBUV7k4/xo07CzTN7BElKUYQyem6
    dThGIRLZA5OJOpCR4YiwLJ1SVdmIoUMK0dHhRUtzJ1paPfD5Qgj4wwiHY4jF4ojzEkRBOOGe
    M4m6LLNJqyOzp1iRnriOnJwMNDa0IRrlkJObrlI65od4hIDVWtYXejW8ob5lRm/rkXoT9fnu
    u924ce6MioKCwrMA7E0Gr267/co8o8lQvmrlJkQiHLq8QTzwwKsQBBFGkx433DD9JF4tWvTl
    VQteW45YjD8t4+o/wSvtfDeObGhovWDz5oOYNasCPC9jx44f4XYHEukdLS3F6hnYbWY4nTZk
    uJyYPGUUln68ATW1zeOHDRtW+t4bqw/dcs8M9CevNm3cDWeaHX/84z1IS8tAMpYPXHbFVFx/
    3ZP4xw/7+4VXefmZ5xsMLObdPBPnnTsa2TlpsNksiRokOuEc/VwXd8+uYgkxLo6gP4ymZg82
    /2MfFi78DHn5rj7nlclkSHGkWHHXXZfhrKEFcKY7YLGYYDSyIEkaJ9eK/bPr0P5IkpYq93oD
    qKttw+LFa7Fp015knwavehHBImyxWOjCH37Yh9LSPIwcWZw0gwNQkJWdjilTyrB160F4vb4x
    VktGEYDKM1OkAqjqvt1Wq+Py22+74qGZMybOjnI84+7sSl/41qfsO+98jlmzKjB1avkWRSaX
    kRSdxvOB+X//++rrFr71KebecBH+596rYTabknatBAHEeQFxXhpicxhNAEK9B1MTgAAAKaEw
    PVeT8KNrjjYaIpEYZFlGc4sHZrMJOoqEJMkwmbRBq9FYHD5vCDW1rWhs7IAoSJAlGXW1Lc6h
    QwZNJUm8rSgHQBA9V0axAKwgiNwkwLVnnQKhJhxoFBZk4de/mZswBM8UWAQEUcRLL310zFhT
    UlOVE76bIIqS9FyzoKpV3+Xnu+4OBMLEffe9Ao7jQZEkGEYHq9UEh8MCq9UEZ5oVDMOA0dMg
    jw3igwqBl8BzAsJRDj5fCPX1bQgEIxAFLWLEsgx8viDy812qjmZ30sywpILXbC6HqjZGXa50
    2eGw4oNFq7F8+XcIh2OgKBIMQ8NiNcKW6MxJSbGCYWj0XHkiCALivAQ+rhnwzS1uhEJRcDFe
    G+GgI0HTNBRVRX6eS1JVImSwlPXRe1A8f9/+Kmd7e9cZ1V39nAHf0d6FHw/Xpg0qLhp+pgbW
    iby6/KGZMypmRzme8bh96fNfXca2t3swffp4TJ486gReLV685rr5ry7rE+Oqb3l1EAA9s6am
    0RyNxvHVV1uwbPl3CAWj3TV8PbsndToKJEHAYjXBbDYgGuNQV9uWddZZJdfdfPfFj8+7q395
    RRIk4nEBX3+9FTStS8r9liQZsVj82HPsc17l5abfzTA0MbFiBCZOGguARyQSQyAQQjgUQyis
    pQt5QYLIiyc4vTqaAk3TMBpZ2KwmWG1mGI16uDKdyMrOBUWR+OCD1cjLy+hzXtntZtisRlxy
    ySSYTBZIYhzhKIeO9giCwQgi0ZjW+SxIkMQTU580Q3d3PdvtZpjNRphMehQX56O4uBDbtv2I
    bdsOIS83o9e86oVWcBm+rsCQ2rpWTJ8+PmF0qEk8yBRGjxmCL778B9rbvJkZZVklZ2pgaWIH
    EGrytbMP5eYN+jNAUUNKw7MHD86/KRKOObKyUitNJvMTqlroBhqeWrdu83WvvbYcV189Dfc/
    cC30ejaJhuSx0KoMWVYdNhNN9+6QVyWCetk6gGI0gFAACBIQFYBWAGXp+HNGOhe8/uiIfXur
    bPsP1FB79lRh6dINyMhwdA+BZRganZ0+DB6ci8KiLFx40Xh5VFlJsHRI/iFZlpe89gYZ/90T
    mXpARx5LUWn3wSmo6o8SoANBlJ5JaoAASD1AE4BiJAiQqqrCajPj/AvGgWEMSTGwJCmOd9/7
    MvFPETpJtugAuw7gCUCJJ1WJIfM2qwkEQWDkyGLMnDkBeXkZSHfaYTAaEsXtNMgTUn8nOxuy
    LEEUtVksHMfD6wmgucWN9et3YM3qbbDbLQCIPoou0rDbrbBYDVAUGdffcAGKinKRlZkKu8MC
    g0EPltWBpunETjDyJ16hAkVVIEsyJEnurusJh6Po7PChobETn3/2PcKhKBwOqwD1TKPU/zo6
    oiiHxhypbIQgSN37y5JtYPG8iKojjVBmCecB+LDPeFWcd1MkEnNkZp7Mq7/8ZSmi0XjSjau+
    5xWjAq2Hzjln5Kb5rz00Yu+eKtuhQ3VUQ0M7qqubkZ7uQDSqHVGjUQ+vN4iiklwUFmZixIhi
    edSokuDQIQWHZEna98VnVcTlV2Wy/ckrkiTQ3NyJ3z2+MIkY0borzxpa2D+8spmhKAq++WYX
    ampacbiyHs1NnfB4AohEOAiCCFGSoSTKAXqeL5Iku0f9sCwNo0mPtFQbcnMzUDqkAF1dAciy
    0i+8stnNaGvz4JOlG+D2BFBztBmtbR4E/BHEYnyP0gzlpGG1JEV1O8IsS8NqNcLlSkVBQRZK
    SvNw4EANGIY+LV71wsCSibZ2N0KhGIoHZSfAmszopYr8vAyQBIHWNjfGjtOpyYFgAQDA79ml
    yBLfRukIAJ43MzMLFktOwa6jmE4QHE8QbcXNzU03vfnmCpSVDcb/3Hsl9Hp90o0rQFs+qcgS
    GJY6RVC5AbQAMGQAmAFUTg8E/IXhSCTRnQbSZrUoNpsjTlLERrs9ZVNFRerBioqy6X6/f/DL
    L3+EaIzD00/dhjdeXwkVwJ2/vBTPPvs3jB8/FI88Mhd2R0ojQK0D5HZAKXnk0eBrXV3tQ8Ph
    qJkAQFIkzGYzHA53PWBYCzCrVWV3J5AOgjwdDzGYASjvyJLsVAFKVcUiTfkVCLyQtHoZnhe0
    4X+yDEWRrjTq/S5J4jJ1Ol0EIO8A0JGcF7oPQMeI2tpWgmUZ3HP35RgxchgACaLIIxLh4PNp
    KY84FwcXFyHwwgnFmDSt6waVwaCHwcDA4bDA5XJi+IizUFycg+837UN9fRuhyNJwUTzyOU0P
    SbJ+Sma320N5PUFceulk3HHHL3C8Q1gCxwkJgykGUZIhS8pJq06OFYayDK3NmkmzIT09DYMG
    FaJiIgWHw4xHHn4NjU0dxlHlI1O4eAMM+oKko5fjQkRjU0efpAfRI+HQ3OKBKPGGvuSVK7Ng
    sSSdzKuFC1chGIyAoqg+u8Jk8CoY8BeGI1GNVwBptZoVm/WnvBo13e/3D37llY8QjXL47ePz
    8PqC5SAIEnffczn+8IdFGDd2CB5++Iaf8qp4xuzQ/K6uzqGRSMRMgNB4ZTLD3oe8kmUFrJ5B
    bk560iKkqqqitdUDqZ94VVfXRsQ5AUuWrAdNa+MKbHYz0tMdGDq0ABarCSajHgxLg2F13RF3
    SZLBxzUWxOMiolEOfn8YXm8QW7YcxLp1O7TuaUVBY0N7n/PK7wvB7Q7i97//AKyehsWsRdgH
    DcqGw2GF2WyAwcCCZnUnRBuFxEgZrclAcwS7ukJobOzEwYN1iH3CQ5Zk0LQOTafBq14YWIw+
    Eo2RAi/A7rD2yVE2W4wAQSAUSvqiejicx4szucgehDlfCIoaIlQSaRkFhKp6blz99ebCzg4f
    nnziFtjtdiS9trBbuRP5sFMe7kTqAMu1wWDzgzt3HirbuHEP2djQjlhMmxEjihJMRgOc6XZk
    ZadNzcp0Is3p4MOhKLV27TYcrmzEI49cj+LiHLg9fm0mWEkeHnvsBrz4wmI0N3tw0UXj8x0O
    661dXQGmpc2LtlY33J1+cFxc8/4JAgY9i/yCzPHnnTfmmnHjhu2z2dJfgUJ+nIj991Iotqmp
    Yexbb67M8PvCkBUZ+/cfBR8X8cD9ryTthSHLCo5Wt4CmKdx370t5JEnmOVIsuOvuKzrz8gax
    yXuqJMVx4TEHDhxFSooVbe1e7Nv/GaqqGtHS5Ibb60cwGEEsqq2T+rmKWJIkYDTpYbOakZJi
    RXa2E6Wl+cjK1ob/7d1TjVgsPJbVZyW5JqYDQHRidXWjIxCIICfXiR079qG6ugk1Na1wd3bB
    n6i/4rg44nGtjkyWlBMsDprWPFqTyQirxQCrzYy0NDsGFWWjeHAuDAYGRiOLgwdqidmz4xP1
    rGdFHxwznSjKbDB0OjsWe2dhhcNRiKLU77xas3pL4eFTWFL7n+TVrl0/ln3//T6y5mgz/P4w
    JEkbOpmWaoPLlYrMrNSpmZlOpKbZ+XAoRq1btw27dh3BnXfOgcVsREdisa/Vasa1112Edxau
    QmurFxdeOC7fbrfc6vMFmdb2LrS1etDR3gWfL6g9fB0Fm82M4sG546dOLb9m7Njk8SoYiEBR
    VezbV43S0jy8/favYTAkByU8L+Dee1/Evr3VePD+V/IIgsiz2c19yisdo8NNN83AhAnDkZ2V
    hjSnA0YjC5alezQbkP80KALIUNXESBZBRDwuwOsNoK3Ni107j+Dddz/Hnn7gVXt7F4qLc3Dz
    LTNRXJyLzExtUK6eZcCwPbMGxD/NGgAKFEWLuAu8iEiEQ0dHF5pbPFi18jts2LALBw/VEbN6
    yStdby6HJMjEHJi+2c2sKqo2r7ePgWEwl//kypoyPB73NV9+uYUYP34YRo8Z2geRqxMfbe+Q
    HyoJhzwv/f65t9M3bzmEiooRmDFjAnLzMmAxGxAMRnHwYC0OVzZi29YftWXCcYE16BkMG16E
    F1+8F+PHD8fLL2ueIQB8vHQDHn10LjJdaVi8ZC3efnsVFeMEyqBnYHNYkJ2VhkmTzsaIEUWw
    2cyIRDm0tLixZ08VnnrqbfKcc4aVP/XknS/b7Gl7ABw+ndCuPxCR1q7bCYtZD1dmGkpL8wAA
    kWhyo8nDhxdqPmg4ho52L8KROH5xOuycUQAAD9dJREFU3UVSXh6dxG8J5nZ2+sqrqprQ3OLB
    Aw+8ClEQYbWakJpmQ3aWE2ePHAy7w5yYjqwHwzLQJQxJRdEid1xiFpE2SVh7cVRXN2HVqk0g
    E3V0oiCio7NrVFFRWh6A5C3EA02ramz0rp2HCY7jMf/VZQiHOfC8AJNJq09IS7MjMzMNNpt2
    HXo9DYZlEkOGVUiSBJ4XEYvxCIWi8HgDqK9rw9691VgZ0uYL2u1mBAIR7D9wFH5fcLzF5rRD
    K9JJdg5P1Vii9ilPqBNmgfUfr77+eishy0pSuyOTy6t30r/8aguikXjicRDd0cTq6maoqvZ3
    WvcsySqKCi0qROOTT77DokVrNcOVAB56aD70rDZo9ZtvdmHjxr0USRLUsc9oYynQY7mwZgBs
    3nwQy5d9S86cVVH+5BN3JIVXnR2+RDpWRWaWE3o9myglSYrOgtJR6Oz04cuvtkJRVGS4UvqM
    V4cPN6Ag34V77rkCqanpCdtTQizGw++PIB6Pg+e19JrIC93RakpHde9OZVnNYTKbDDCZjEhN
    TUFpaQlGjxmCTd/vQeWRhj7nVTwu4MKLxuLGGy/tNvxkWUQ0GkcgGEE8zkMQZUhioikn4S8w
    DA2KpsEy2nWYzQaYzAZYrBZkZrkwqpyGwcDgH/84gH17q3vNq14YWAJvtZoVVs+is9OXOHLJ
    PNgk/P4wZFmBxarVd+3bdQRlY4ag70UaX3m4vrCmpgVXXzMNNM0mPXqlqipk+XS9XKWDosij
    kqSmG416PPDAL5CT40IsxkFRVBgMDKaeOwaqKiMeFxDwh+EPhGG1GpGZ6QQALF26Hh8u+hqz
    L5kIHUVhyeJ1KMh34dprL8L4c4ahvd2LYDAKu90Eu8MKPcuCJHWQZW0+C0mSmDx5FM6dWo4D
    +2u0QZgEamRJdJ8+S7Q5I3fcPgdXXXNRn0UMe7wKsfyTdXj1teV98EKUJ9fVtWS1tXUhLdWK
    adNG45xzRmDw4By4XKkwGPVgGB0I4lSXIcuJnVgCvJ4A6urbsHXrIaxbux0dHT4crW5MLy4u
    PuvQoY31w4efm6RrEE0+n6/swME6kCSJlBQbpk8/B6PKS1FYmAmXKwUWqxm0ThslQZI92+h7
    erXH9xSKojbV2e0JoL3di8OHG7B9+4/Yt7cadbWtaGxqLx09Oiu/DwwsiWV0gsNh6/PojsVq
    gk6nI9BvovGqqqopqXVXJEl0O7fJ4JWiID0WjZ9S/dsxp52iSIiChMbGdsiyghtuuAg6HYVF
    i9aAJAgQiVEmP/3Mz0k0FocoyUnhla5b97Vaqfq6Ntx115+RHENeG2JbVdUEiqK6gxk6HdWH
    vPLi3KllaG7uxA/f70NNTQsaGtrh9QYRDEYRiUbBcYK27kZRT2g6IEltMKfRqIfFaoTDbkZq
    mh052U4UFGahIN+F/HwXqo7s6HNesSyN7Kw0bN6yB7U1TaipaUFriycRcY8iEuXA8+JJJQ3H
    RsoYDCyMJj3sNjMcdgvSXSnIy8tAUVE2KIpESqoVtafBq14YWDTncqWF7DZT2u5dlTj/gjEJ
    rzVZ7hKJvfuqoUJFpitNjcckFWD6CVics66+jVUB5ORm9AWGIUkyuLhwmvUgu31G06jnbr1t
    zkf33vuC4/nnF0Gv16OxoQ2CKCI9PaV731tOTjqyMlNhtpjQ5Q1hz56j+O7bXVi/YSfKRg3G
    HXdc5jUYWHi7gmnP//FD7NhZiWnTxiA/PwMMQ6O11Yvdu6rQ1NyBzg4/2tu70NHZBZrWoago
    G5IogeN43HrrJX6r1f4MCLv3zJACRKIcuroCfRI1/KkRH4lyfZQwiuW5O320JGkF1XaHBVab
    CUaTXqtJoo5FOXrOlDrZMDluoGiDIiVJhppIeRgNDGhaB54X0NziMaiqMBHA18kL0cRHuN2+
    0ob6NhgMLCZOGoGZMytQWJgNu90MgiBxYlG7+i+eKAGCoKDTATodDYOBhd1h0XQ0IwUso0Nz
    Uyfq69vQUN+WMmb0yCkA9if7iej1Fl9+XkafahRBANnZaSAI2td/BhbnrK9vY+NxIYm1P1ok
    jtKRSePVLbfO/mjHzsOOxoaOEwwhWVZ+EtEiTvgebZURgXPPLcPtd1zazau1a3dAVZR/6ryq
    6vHF1yRFdu9bVVUgNzcdt9w8Kym8+qnBFYnEsGP7Iahq8vSJ7OOIaE9eqYqKbdt/xO49VfD7
    I93RwJ97Pj/Vm2MLkXt+jqJIWK3GRKOL3Le8atCM8QULliMajSMajXc/++NRTeJnddnnCwEq
    EkNHj39OK3o3IRCIQBSlXvOqFwaWvSU1xb53+PCiotWrt2H/gZokG1iAryuE/PwMZOdkNAqC
    eihBc6UfiKXr6gqCjwtYsngtNn63+4xnMP304iRJQmVlg6aUitJLDZsFIPzN8OFDV95444zb
    li7dgNHlpZh2/lioioL2di+83iCOHGlEIBBGPC5CkiXEojwURYHT6cAtt8zCL6654EhObtbD
    AIFnn/3l/A8//HrQqlWbsG7dDuh0JAx6FjqdtvPMZjMjNdWO9Aw7zi4rBkVRaG5248D+o7j+
    +oswcsTgJVJc+aa+uvOMqK4oKhYu/AwffbyhrzM5AAGEglHNA1aT/WXs3vMvGLsPUM9ev2En
    sWrV91iyeB0MRhapKTakpFjhcqXA4bDAZDLCZGTB6lkQx5anSjLinDawLxiKIRiMwOsJwO8P
    a8uU4wKMJgNKS3Jx/fUX4YILxreLolKV3GsQA5mZ6Y1333PlkNVfb8XyZd/hk0++RaYrTTPe
    s53Izk5DSqoNRqO+u6OwZ4RClhUIgmaEh4IReL1BtLd70dbuRUd7F9xuP0RRQoYrFTfdNAMj
    zy4JqJCTvtxWm1hd892IEYPuYhiaEAQRya/FUsGyNIYPG6SwjH5D/xlY0Hm7ghBFKWnpQVUF
    QuEY3nxjBVJSbBDFM+fVsGFDV86bN/O2Tz75BmPHDEVevguKoqCjowtebxAetw9uTwCRMIcY
    F4fAi6BpHbKznZg5awKuufo4r5555s75gwfnDPrs0x/Q0eGDrCigdTotNWUxIC3VjvT0FDjT
    7cjMTAVFUWhpcWPH9h9x+eVTMWJ4kniV6KZT1RMTqMm0h3oO7lUT39mXvNqwYRdRWdkAg0FA
    NMr/xGAiQEBN/BriJP0/pjvHjSvtD5OIbJWW5mHSpLP7lld3XzFk9ddbUVXViEiE6za4fzq5
    /d/xQqtyILodXEBFPC6CIDjk5bswsWJEr3lFnLpuBQG0P7Bjx+5Xvt+4pw/ehdpNOWtYAc6f
    NuFDt1e9LeCxiv2RIlTVH3/1/gcr5y/64OseVnkfxE9IEqqiaEuun7mz2pmWWUGQQ7pO7Tc2
    AZBLwuH2LwKBoDMj3bmCYdld2hmUUqJRflKcixnDYS4tFIoWebx+MhAMExazEYOLc9tyczPX
    04zxBUCqBWjIsny2JHH31dY2nVdb25oZ5wWYTQY1Pc2h2ByWeovF4DWwxqjZavgHSep8AAhB
    EEZ3dLivtJhNXRarfQYk1NKGkad5zyvzOztbd6xauSk9HI6hOynet08aAAGLxYjLLp/qdrmy
    xhHEWY3J0aEPAYzOAaQrg8HQrI6OrpKWlo70hoZ2Q0uLB21tXgT8YcRicXAcD0HQRjEcS3OQ
    JAEdrQPL0NDrWW22jN2MrKw05OVmIDvHGS/Iz3RnZjqrbXbrt6pMruls9e3vCqtKskLuqhoB
    0FasKrG5bnfXjOrqxrNq61r1dXVtuqamDng92hoNjtM6a455ev/KqyVIAnqWgdVqgt1uRm5u
    BoqKslBSkhcfVJzbmJnp3M0wzPvRqPSd1TpGSv65bip1u2s3rFzxbU4kEkt6VEBVAbNZj0sv
    ndqQkuqaptePqEc/yHFerU7uaAY1EfVJOq9Czoz0tJN5FT/Gq1iR1+sjA4EwYT5FXvG8ALPZ
    qDrT7Irdbqk3W4xeA2s4iVedHe4rzUnmVV/o0s/rmKGfeNWZ3tTYYWht86CtrQsBfwixWBwx
    jofAi91dgcd04xivDAYWRqMedrsZrqxU5OakIyvbGS/Ic/U7r+rqW/UNDR3dvAoEo4gl0oPa
    71dPCJ5oBpi2dYJhtWuxWc2w203IyUlHYWEWBg/OPW1eEb27oCMZgHCXoigWVe2bF6KqqoiG
    5TUxDhuyssf0kxIfmRoKdc33+QOGvv8ybQ9gampKnarqrtXrywK9uTfA4csBRfF6zV+kpQWU
    42mlDB1gIYHOFEDOA0gG2uAZCVCbY754A2tTlWBAAQjAaqUQ5xyk2RIvAJCrRTMJHlAEgGoG
    7D5AkoE26Vjax++1kY7UyGxFUXTfftC58sLbpp/BPT/oAHCvosi2vtKlf2PMBwF1AU2X+5Oj
    Q0Eoch1ISgXgZIBoPkC4AGGMooi5giAa4pw4Ih4X03kxDlEQIYoiociaK0yShEoztKotUGXB
    sozbYKQPMQwdA3RdJMlsAYgWgG0CunieA7goD4ezIonnoBWqGgRBeAE47QAxGJALFDk+juOE
    9FA4Vh6JRPWxGEdyMZ6Oc7yDFxWzwAvdXomO1oHVM5Ke1bn1BlYwGFjZajUrJqOh1mTWH6V1
    TDsIdguAOiFCtjHmqByLqTCZyvvgXO8jAd11qiqN0gzZ5EewCIKALKt7pZj6kdFergzwaoBX
    A7z6L+QVx9NxTnAIomzWxuOoIJBwallaYvuIV700sA5BVYUzKH48ldA+CVUmQbPl/ae+qo8A
    fA6A0PXPwZGhqlGJ42S/yTRaPfXf2QmgFpJEIRzW4c3XPXj8iYvR1rwddgcNhZAhxAiYzSRo
    /fHP8byKaJBHasZx5c7OW4/d222wWymwxuPXLHJAJKJAJRQYWB38ARHZeePxf8+twa/uTYfF
    TgKgEPKKsDnLz+Ce7wFUFZKsoO9zgyervU6n1RMRxKik/std3t0gSQUMQ8JoIhIvGQkkGYW/
    K9VqtjJ6nY7W3g8gAag9NimrKiADiohgkI/bU6IhVaWgqjptqGVMRTgsQ6+nYLGP7rO7Ew3t
    hiiKMJl10DEEAAlVVSxVUqq3E2ApgKABSa/I8XwVRDrVY2e1ogCqIscpHXsEoKMAIQKiHItK
    EaMpElMUEgRBgospCAUkhKIiSksn95UhAoCHJAl9qGMECJWCwBEw2kYN8GqAVwO8GuDV6RlY
    /7+KqjYAiAAQ+u07FQXgecBoLO/lb/UnDFHHf+Re1R3dAgAoGlwxoDinKAf3bQFrVJFiZ8Ho
    SRiNwLExX7J8PFRNkoCqAFwM4OIKggEe0TCBkeX/2Xu9+R+bkZdPwm5nYDIREEUVsqxCklSQ
    JAWKSlyPCkgSoMgyCFIrNtbptErNcFhFWzOHYWdPGlCIAV4N8GqAVwO8GpABGZABGZABGZAB
    GZABGZABGZABGZABGZABGZABGZABGZABGZABGZD/3+T/ASi0aonoanHFAAAAAElFTkSuQmCC
  }
  set pieceImageData(Merida2,55) {
    iVBORw0KGgoAAAANSUhEUgAAApQAAAA2CAYAAAB5hAnwAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh0R2q4VzwAAIABJREFUeNrsnXd0FdXW
    wH8zt5f0HtKooYYOoQuhgwiCgg17BXzv2fB9PkWxg72hgnRRUBGl9x4IJZBASEIS0kkhPbm5
    feb74yZIBFSUEt/LXot1FzB37pw5Z//OPnvvsw80SZM0SZM0SZM0SZM0SZM0SZM0SZM0SZM0
    SZM0SZM0SZM0SZM0SZM0SZM0SZP8LUVoegUNRZbTARWCEN70MprkbzyGFYDz/KcgtPovaVs5
    UAPIgBZB8G/q8CZONkmTNHGyEXBSuD6NywGsdY3TNGoIyXKmFnyckG0HIxCBIAj/owqXBZgA
    TV3/GRCEiEb4nCmAdAN+WUQQ2l7HdhYCZsBSp0ctfvX/pUAJ4KEBR2+QvEAsB2UcmKzgiSD4
    /Oo7eXXg0QA6BCGwkY/Jw0BbDUgiHDKDD4LQ7b9E34rq9M0OqOr0LaCJk02cbOJkEyf/FpxU
    XttGpQG1gEEN1JnI2jJZPl7r6pTIRtgVtlcgPwI0C0F1FA6WyPIpBKH9JdqXWjdwFYARQWj+
    XwTJY4DKCwz+oO4GqnhQFsvysXJB6NrInlbwBLXR1Q8XtQSwSyAVg+QAhR+oNCBe4loJsFvB
    eQ5EJYj+oBIvve5yAs4aoOL69EciECBCfiAggqpSlk9Ugx5BaFl3lbcAZZFQ/FJxcdnY8rIq
    o6eXW01AgM9mcP8MvHfVvRBkOaNONzVu4PAAlQQBhbKcKAlCVCM2uIo7Qu7bgB6CX4ba3X9/
    XcsBqoFDQFc/UGhAaYW4c7KcBLghCGFNnGziZBMnmzjZqDkpXNuONXiDdTzYR1RXmaOdTqfo
    5q4/rlBotoLmO6gswAqCtlsj6ozE75KSUicJiDWRka1SFUrDVyAuBofZZQSHIsvn6lY4+rZQ
    ex+wFwxbZUe1TVR1+C+AZBpgbQt8lpd3tl1FRY2Ph4ehNDQ0OBnEJ5xOKUWp7NSInvfkq6aa
    mvssFuuvhrSMQqHAzd1QqlAoxoMiV5bt31ZXm6LtNsdF16rUStzcDAcFQTUFnKFOp3NNdZXJ
    x+l0XnStVqvBYDQuFoSOL16fSUsfDtLTFkvtGIvZrHb3MKaKonY+aNc7HLYapVIHWCMsloof
    f/h+S5flyzdRWFiGv78X99wzkkm3jUjXat1H2m3mDJXaDzAbwD5WlmwPV1ZVRWo0WptOZ1gP
    4rtgzW5MsHTp2zkgUCdJOctWrtw8sbKymqlTx+zSabxvFhSOmsZpdP2hsQv4iVARDbZJVost
    xmSq9dbptWU6nXY7aL4Hv4OQJTUmA6WJk02c/G/g5IoVWygqKsfP35O77hzexMnG6KGss+qD
    Zbl6XmJCypgf1+xWnErKwulw0rxls5Bx4waM6dOny+0qlfrhKostuXFBwi79+ONu1q+NNQ4b
    3rv7hFtv6hQV1TpGofCaByE7ZDlbhmJA1c5mK1kYF5cQHRoWOD0iPOIdEN8sLT1h9fHp9DeG
    ZD5Q2cpsNi1a/cP26MWLN5B/9hyBgT6B9983JnDSbTGLtFr9VJvpVJra0L6RPLXTe8mytSGb
    N8Yh1C8t69AWGOTDC/+5TxcaGqGEYMFkOhH4ztxlIQkJGRetrKI6t+S5mfcEGo2RApQoz57N
    D3n9tcU+hQWlDe4pAyNG9eaJxye5Xz0QlF1wdxeMBaFtXTjNHGGxVC7YuvVgzE9r9lBWVkX7
    Ds1D7rln9IDIyDZvKxQes8ABWO/btHFPlzfeWErziECGDutFakoWb765FI1W3WrixGFTQXgZ
    BC3Y3ko7ffqRZcs3qk+ePIOHu4Fx4wbOGDW6b3ut1uPRyrKEDA/vzrg8ZMpftd4bQfC7jv2b
    D3hrwBIpSeKgNT/upri4nIkThxj1eoUbaJ1wxizLZy4KbzVuXbMAp0UofqSkpOS1Dev3+ezZ
    k0B5eRUeHsaQ/gM6R908duBUP3/7f4Av01MPSq0ioxsXJ9fVcXLC/zAnV2+PXrpkEwUFJQQE
    egdOnToqcNKkxsvJLRvjEBXiBW2RCQjwviQnExMzEMVfrpUkiaioS3OyqKisQdqD5JQYfoM4
    +fPPe6koq6Zt+/CQu+++FCf3dnn99SWYTGYEQeTcuXLeeGPJZTm5/OtN6lNJmbi7Gxh7c/8Z
    o0Y1cfK6GZQgesuy/Nm6dbvHvfn6UiqrTLRs1QyVSsH2bYfYuvWQMGPGbf3uvnv0QqWofDAp
    Ie5Uh869G4XKCYIClUrF2YJSVqzYzMaNB9R33Dl04h1TRg0JCja/AF5LQe9tt1cuXLFiQ/Q7
    76ygbdsww+zZj/67XbtW1n/8o/rN2toC9PqgvykqRRGEaVu3Hox+663ldOrUnJihPTh5IoM5
    c5ajN2ijb50QM02W5Ke4MQk5l2RNdnYBiYnpdOveBq1WgyzLpKXlUXSuArPZJoFCAnenw+GU
    UlKySUvLJapzSxQKBXa7g8SEDNQaJQ67E6fTQ1YoKpxms01KSMzA4XDQunUIgiBgsViJP3qa
    tu3DkCRZ8dcheRrwBWqagaI/SIEgngBnrCyftgAqsL/48897Yl57dSFubnq8vd358cfdpCRn
    q999d8ZDIaEhmxSK0Lja2vzBmzbF4eGu5513Z9C2XStOp57h/vtfZ/PmQ4we3W+wXt98NtQM
    Pns2/4EXXvhcnZR0huYtmpGWnsdLL32JxWKNuX3yiJdkSXxIljPtYNeAph/InUBRCM594Jsv
    y/HXMXdR3d7pLJ1TW1sb4XRKXk6nhCTJWCzWdlXVhRt0Wm25SuXzErDv76VrGkBzb25O5pzZ
    sxe47dhxFLvdiSCALMOuXcfYty/R56UXH5gT0TzU2iryzkWN5cnrOZmfX8KSxRvYvPmgesrk
    oROnTBl5SU6+9+43REaGGV6Z/ch/HSffeH0pNTVmRFGgstLEm28uRa9vvJw8fDQFpULRwEgM
    DQu4iJOpqTkcOZKC4oJrnU4nGo3qIk4mnsggN6eogfHpcDqJvAGcfHX2V1RWmhAEiDt0iuTk
    bPW77zTk5JbNcdTUmFEq6x9NQU2Nha1bD1/Myf98rj54IOm8XsbFJbk4eXsTJ6+5QSnLMpB6
    y6mk1DFvvbkMTy83Xn/jMbp2i0QUBU6n5jBn7td88MFKmgX7R8fE9Hmq1mJ5+IIlxw0WneDu
    bsRg1HL//WNISMjgs09XcyjulNezz93zfvfuXQaCXL5v3+HoDz9YiYeHGykpubzx+iL13Lkz
    pi1bFrIfNAdkWXbAThnCcOWsuF+U6Ns4xak0myu7b9gQi6+vO2+/PY3QsBByc/OZes9sNm06
    yKiRfXroDQEqXBnojUa8vNx4/Y3HCQ0JRJKcvPzyV8TuT0ShEI1gmQWJ1Uqloo0sy7RpHcLH
    Hz+FXq+nvKKahx58E1kGhVLRSqE49RGIBoVCNCLLDBzYlZdffhBRVJCbV8idU15CuArZIrLs
    AA4C0hSLpXp2RnpOREWFSRUc7FMV0TxkqyAYXwOHqby88uZvVmwmNDSA996bQUCgH2vX7uW1
    Vxexa9fRoHvvC4mRUR90Oh1ijakWvVGPj68XIOLj44HBoMVca8bpdIqCYBShrM/OnYf0CQnp
    /N//TeWW8TdRXFzK0099xNcrNhMztNcoLw+/5qDSQ83zubmZo3Jzitz1Bq29devwLIPB/BJY
    v922LYGhQztfh551/POnn3aOWb16F06nk7S0XGx2B88887FBqVR06dOnIw89NP4/Wq37OMD2
    9zFK0oIslsqn5s37zm3LlsOIonDBBOdi6fZtR3B3M7jNnv3oU1brmk0aTVRBY+Gkh4cRQRBQ
    KESKCsv4+OPvOXT4lNfTz9z9fo8LOfnhKqqqTBw6dIo331isnjNn+n8NJzdtOkh1de35flMo
    BGqqzWzZEtcoOSmKAgpRRHGBh1IQQHT9WwNOuvpW0eBa1/XCRZys+34Dg1KW5QZ/vx6cXPnt
    ViorTef7Q5Zl4o+msHt3fNDUey/gZK2ZX+8hEwQw11p+xcnD+iOHU1AoxPPe16qqWr75Zisx
    MZfiZLG73qD5n+XkNfBQFrvLkuO+H3/cpayqruXtt5+g/4AeuJJzoUfPjrz44n089OCbfPPt
    Fvr2jRrYpq0xFMhpJF6DvIAALxwOJx07tuChhyawdOk6Fn61jn/98wPNrFkPTOnZq708f/5P
    6PVaPnj/Hxw5msx7737D8mUbmz32+MQVKnV1glZrtEHYDlClgPogZFTLcsp13e3250SB3W6n
    qtKEwaDD09PNZax5GjEatdSaLDicThAUjW5LpyAIGPQ6DAY9suxEpVJSXl7N55+v0Xl7u09F
    AKvFRmZWIeGhARgNenR6PTa7A4VCJCurkHff/SZQo1U/AVBWWkV5eTUqlRKjUY8gKDDodQiC
    gHxV1j/bgJa3nSvO++Tjj1f6rF27j9paC/4B3u6PP37rxIkTh2k1Gv17FeUVmuLicqKjOxLZ
    NgJQEd27PWq1krz8EkA2mEylgsHgnt+9e1vmzfuRJYvX069/Fw4eTCQ7u4jhw3uh1+vzLFa7
    pNVIyqzsQlQqJdHRHfH09MTT00DbtmHsjz1JdbVJ4+8XEORwlDyzYf3usR999B25uUWo1SrV
    yFG9W//rn3d8EhQc4IyJkb67To4VjcPhxGa1I0muSUJ2StRUm9Fq1djtTmRZ1jkdlr/ZNmP1
    LcnJZ9pu3BiHKAoX7ZIWBAFRhK3bDjFp0uC20X263QJ83lg46R/ghUqlwOmUEEURWZaJ3X+C
    grMfal588f4pPXu1l79asJay0koUCgWyLLNvXyLLl29s9thj/x2crK4yXdIwqTVZGy0nL8XN
    yooavvjyJxcncXEyO7vokmMyO7uI99779hdOllVRWVFzDXf5/8LJTz5Z6bNuXSwmk4WAQG/3
    Rx8d34CT586VI4pCg+d1Op3k5Z9rwMlu3SLZves4siy7eC7LgECXLq0bcDI7pxCn09nASyuK
    AuXl1Rdx8pNPvic3twiNRq0aPrxX63/+c8r/HCevgUFZq6mqrmx28mQm7dqG07lrmzpjsn4C
    loiMjKBLlzacTMqgpLTcr2VLH2PjUS/79ojmwf/Q6TRiYkI6o0b1Z8aM24lsG87rry1m1qwF
    jBzZWziRmMFjj42nZ68ORLYN48DBkyxavIH9sSeaBQb5NGvXrjnt20eM79ChlSUw0DdWFD0X
    gnq9LCdVuMoSNGukeNHIer2HqUuX1ixavJ7Fi9fRu3cUh4+cIj3jLPfdOwq9TpcBTntjfHoX
    GGRkZIxGPVqtmu3bDnNhZqUkyXh61g85GQHw8nTjbP45fv5pb4NrtVo1RqMeGRkBue7+V0u6
    aB2OwocWLvrZZ+Wq7YwY3puWrZqxe9dx3n5rGYEBXiOGxPQ/6eXtlRsa6t8hPj6V2P0JhIYG
    sWHDAWw2By1aNAPEyv37Fc4RwzWf3zYp5qaUlOyA+fN/YtGi9TgcTgYN6syUKcNKFArVgs3b
    9zrHjGhlbRsZjt3uYN36/ej1Os6eLSb+WBrBwb54errlSZJj+LH4EyNefXUh/v7ePP7EBPLz
    S1i/bj8ajcrnhRcefAxJuwFXvZRrLNovJ4wf1nHYsL7+suzwfuSRN/XFReV88snTNk9Pz3Ma
    jeqMRqWea6ow/Y28kwC1oamp2UpXeE647GRfa7KQkpqt7D+gW2ij4mRE8D/c3Q1iWVk1glDn
    uVIoyMwsYNasBYwY2Vs4fjztvJeqfuJeumQT+/f/d3CyU1Qr9uxJ+JVhAp06tWjUnPz1GKup
    qWXVt1svsaARLvJwFhaWsmjh2ouuvXYGpYuTixb97LN06SZk2WW0Z2UW1HHSe8SQmH4nvby9
    ckNC/DtkZRVS7xiVZRmlUknz5g05OWnikJtSkrMDduw4gt3uRKlUMHhwNyZPHtqAk5FtwlGp
    lDid0vn2SZJMUNCvObmI4mKXMWux2Fi1ajtqjcrnhRce+J/i5DUwKAUkScLhcKLVqlFcwuUt
    iq6JWnJKOJ1OGUS58aiX+nRYaGBOSIh/xLFjadTW1qLX6xk4oCtHRySzYMFaFi5cj5+/J0OH
    9QLA3d2dceP6E7v/BDm5xZSUVLJ92xEEUSA0JEB7001dh4wY1bd/56g2BzQaj3mg2CDLp6pd
    9coaW01OT7tSWbH8jjtH3ZSckqX9+OMfmD9/LRarnb59O3Lb7UNLFErNlw6HwtmYISkKAg8+
    MJbx4wde5EGQAYNei1andvWfm57Zrz5Mrdl6USBblsHH2x3xmsDS3Kq0tKz79m2H6dSxBa+9
    /iju7p707RvFvffOZu/eBOWAAT0C3NyMsQ88cHOHl16az6OPzMFg1FFVZWLgoC4MHtwtB5kN
    I0eEIMv7doeEhv7z9defeP/LL1YHLliwlrvuHsaTT04p9PPzeS7rTNXOsSNvQZbT9w0a1L1o
    8JBuAQvm/8x3q3ZgMpnR6jQ88/QdeLp77BIVSLt3H1dVVZl4990ZDBzUG7O5huLicvbsOc7D
    D5d0CQuLaAGcuPa9eW6/09lshEoheirVloUqpXKAUqXAzd2Q5OnpM6WiwlqoNyiqZLnRz90X
    m2X2P6ZGDoejsXlXXZwM9Y8oLa2ifndvvUFVUlLBj6t3U1traWBoCIJAdY2Z3P8WTt4x8qaU
    lCzt7l3HkSSXp/amwd2YdFtMo+bkpRbGlzIIL7eAvpJrrxYnd+w4iiTJ50PwCoXLU7hv33Hl
    gAHdA9zcjLH33Temw5kzZzl7tgRRFFEqRYbEdL8kJ197/fH353+xOnDBV2uZcsdQnnxy8iU4
    2a1o2PBeAVu3HMLplHA6JQIDvbn/vtHnOblnT4KquLisgRdTkmT270ukuPh/i5PXwKA0WN3c
    PArCI4Ja7t1znOyss7Rr35r6kDcInCsu51RyFoGBPnh5ejirqyxS41CyFECs8fZ2N3fq0JyN
    mw6SlpZLcJAfL788n927j+Pr60FVlYmgQB8CArzPe7OiOrbEzahj6NAeTJs2idOns4mNPcHu
    3cf5auE6Vq/erR4zpu+gqfeO6dOmTcuvQfN/kF8oW9MQNK0bEWqSAUNyaGhwxcCBXQL37U3E
    7nAgOSUGDOhCeHiIzVTlrDR6VDXKCdqVO6MARIKCAwkKvlzSv3y+70SFghYtwrl8Fa36axvm
    uF2NWdnplHV2hxO1RoVarQQEdDo1oiBgtzuQZVnllJSvxQyNDnb3MIxc+/NexaYtcYQ3D+L5
    56c6AgMDXs3IrEqs0z0ZMr719Y242d/f+06Hw4mnhxE/P++d4LnM3aM+lcu6zdc3eMFzz939
    77P558SCwjLGjO3LzWP7S717d9pkdwhvKVQ8ZLM7EAQBnU4DCKhUSjQaJQ6HhMMhqUFQXp9e
    DUCrE8zo1OMsFkuUQimiUimRZTlMEJ1Dvbz138pyEh7+E/5m5qThbHh4kEOjVSstZutlJ2mV
    Skl4eKBDksSzjY2T7dtFcPxY2vlnVShE+vXvxJ13juBAbALLl2++RLtkJtw6kKlTx5CamvW3
    5+SAAZ0D9+1NrDMoBfr379yoOSkIoNWqr2khelmWqa21cJW2RqidTlnncDgv+Tt1odwGnFy3
    dp9i85ZD+Pl5MnPm5Tnp5+99p93uxNPDcFlOPvvsXf/Ozy8WiwrLGRLTnbFj+jXgpNXm4Ne2
    tCC4NiX9r3HyGjS0pEqp1GwYNapP3y2b48R33/uW1157lMBAXwBqakx8Nu97srMKeG7m3Xh4
    um/JTK/OXP1tHrdOCbnBqqZQQNXDx+JT2h46nEx5eTVbt8YxZfIw7HY79903iqHDevHD9zvR
    6bUY9Lrz33RzM2AwuHIMw8MDCQ9vxrBhvcnNLWLjxli+W7WTZcs3EXvghHr69In3jR17Uye1
    2ucZ1GWNpjCzLCcA2nayXPPp2rXbAz/55Ac6dGzBQw+PY8H8n/jss9UEBHgHjxzR932HTZgu
    yydSBaFTI4GkgNlsZeW32/D0cufa7PESqCivwnyZyf/KRVPs6+ud06tX+7Zrf97HBx+spFXr
    ULZujsPhlOjeo52s1qiPgZDhdIr3Rffuck/nzm1ezMjI90o8cYb8/GKhRfMwZXCgUXa9g67I
    crHK4cjRJidnIUkSyck51NaaAy21Dncfv151s5sbYDmbk1Mkp6fnExXVipdnPXxOq9G9JTlY
    qlJTAuryPtEd5RVfbxE++vh7xo0rIi+vmIMHkujfPwp/f+/02lrr2Ws/JjMBdTtZrnw/MSFx
    0KpV27VJSWew2Ry88soCnymTh37Up2/3uxWKztNqahKOGY2d+TuIq5i5c0vHjq2Ke3RvE7xn
    TyKKS6TbSZJMly6t6dKlbbHNJm0pKYjHN6hbo+Hk4cPJDcKit08ewnPPTUWn03LgQOL5UPCv
    vf5Wq4OwsEDCwkL+9pyc99mP2GwORFHAbnfyxRdrGi0nZRk8PIy89dZj+Pt7I0tX35cjiCLF
    xWU8//znXB2npYuT3Xu0a3v6dG6D9AK9QUv3Hm0vycn09Lyrx8k0FydnvfTQJTjZQV7x9Wah
    fqe/LMtIkkxUp5b/c5xUXr0G1QCZgMYHHOoB/bs4b588RFy8aCPff7+D6dNvBwR27TrKqlU7
    GTykG7dOGCSr1arK0DB1YGT7ohx5Uj6CstkNUjQTUHhH5pmsZ155Zb5QVVVLaGgA69bGMvn2
    YXzw4dPodGoEQUFkpCv8olKrzhsuOr2Gdu0iCI8IrBvwrtVUaGgQjzxyK8NH9GHJ4g2sWrmN
    //znSyE3p6jH/Q+M/8rd3e0BqzVxT025A5/Abjeo7WeAPEDd3mGv/GrNmh3Rb765hOBm/rz+
    +mN06NCK0FB/Zs78lJdfXoDdbh96880DF4PuYVnefBLCG0USfVV1LR98sBKFQkSpUiBL8lUy
    LAUEUcDhcOJ0SDh/gfBftCoP5anVbT98+OHx7xYWluoXfrUOSZJQKBVMnjyEoTG9Mq1Web1W
    2wGrNb7Ebvf5CexPS7LsVVVZw47tRxR9+3QeJTmVi3GdMwZUh9fUWKLPZOYjIJCXX0xFeXWn
    wIDgIKAOlEq102kasX37YUVVlSu1x+GUjmblWD8JDlDaNJpuyHLmuj59uz5xz9SRrZct3ciB
    2BMgQM8e7Zg+Y5LFzWj8ShCjiq58rBXgOnnCguvwjL4NKyFzDDDU9VskkKWGqlk7d8aOmDXr
    S2qqawkLD0AQBPbuTeBA7AnF889P7TPh1qEzRVF81GKVKzXqtLrbVQI96vrJBBhITxXkVpEn
    cB0XqEIQQm6AvqUBbiKUBXt7e9U89vgEzmQWkJ9X0sA4kySZoGAfHn9iAkFBfjUOuxBssygz
    ZDlTulGnzVzIydmzFwjp6fkNdgG7uxs5ciSZlSu3cfBg0iUXXqIosGXzQWRJYvjw3nTr3pbQ
    0GY88sjEvx0n58xZRnFxxfk8UVEUKC4q49XZXzVKTsoymM1WKqtqGTGyHwqF+qr/htNp47vv
    tmKxWOvD4FeJk7e8W1RUqj8QewK73YlWp+b22wYTM+TPczIzMx9BEMjL++ucXL5sEyaTBaVS
    QXSfjkybPvF/jpNX0UOZChjDJKnqk+PHkketWbNbefhwMk5JusD1LVBZZcJuc5CaksOrry0S
    bh434JHo3l16gvsjOL0TuGE1uwoDrNbyGYuXrHdLScnlldkP4e/vxY4dR9EbtOj1+rpHk9Dp
    tL/0a92np6cbb709DY1GjSCIXLgJCSAiPJj/+7976dixBXPeXsZHH31PTY2l5ZP/mLxQq3V7
    8Mf5N2YFLsvZgIcIPFBTXfjUkqVr2837bA1t2oTyxhuP0bZdC8BJ56hI3nlnBrNenM+slxZQ
    XFwefdddY34wGlvOBf1CWc6Wblyek9IaM6QXB2JPknmmAP9ALx599BZatwpDkv/6cBIEkfT0
    HOZ/+TN5uedo2SKYIYN7AELtX7vzCCB3QatWEeq5c2Y8l5CYFlRcVC6u37APU40FwCaKgqm8
    /CgueBXpzWbRKDldK/Q9e46Tm1fYJyKieeuS0mMnfH26AhiLikr054orCAnzo7y8mqLiUnVw
    s1AvV3/nAfbe+flFA2L3nwAEHA4HdpsjwM1N5aXRdytyFW221sqyZK2urqVfv44MGdILDw8D
    nTu3KQ0K9JtbXeP4siDrFEERVzrWvgLu9wC6g19LSBkhyzalS2d0RWDcAup00KbAcSu498/O
    zh719ltLcXMz8M47M+jYoTkgcDLpDG+8tpj33/+Wtu2aT+zQoe0ySTq6HlqooLo5ePaCtLGy
    XKt1Ou0oFEpaRabkgnIHqE5Bs1RZPo0gtLmO+pYCaFRw9tGioqKXN6zf67N793HMtbZL7ha2
    WGws/God6en5bcaM7v9DcETwy6BeJDmSTKKyww3lZEJCBgqFeN4jIwjw1VfrWLx4A+Za6yV3
    rtdHE4qKKli2bBNr1uylZ6923HnnCAYO7EJEeMjfgpNLl65t9/nnP1FVZbqoNI4oihQVlfPK
    y19xrrgi+s67RjcaTg6N6cWB2BO8OnsRyaey6NEj8oI1919ZfAvnPw4fSeX773cSGOhLzJCe
    V5WTc96e/lxiYlpQUXG5uH79Pmr+IieLisoRRZHi4vLf5KQg/D4n+/bt0Gg4+e67M4jq1AoE
    SEhI5/XrxEnl1VG2ZEAItZgr56/4ZuPweZ+txmy20q59c+6/fwy33RZDffX4YcN6UVhQwrFj
    aezaGc/WrUcU99w9osdjj0/60cNDfEiWk7YJwo0ApRyTdCq9208/7WHAoC7cfPMAjEYDgwZ1
    I+10NitWbKF3rw706t3hsorn4+P5G4opoVIpmThxMJ5ebrw8az6LFq3Dw9PY8uGHJnx611N+
    4+56OuGMTt/5OkIyF1B4wNn78vLyXvvk45XGH3/cw+Ah3Zk58566nXHS+efv2KE1773/T+a8
    vYz3319JdlZBmyem3fZ+SEiIAbwWy/Kpykud5XvtRb9o2LB+/r4+7pPfeWeFKjb2BGt/3s9z
    zzUnOrobrpxK+U9CUiI+Pol1P8dSWFBG/wFRPPuei62BAAAgAElEQVTM3Y4uXSK/sFnlT/6a
    oapDllMdSQk1n7Tv6L81JsZ/qCzZnz9xIi34++93039Al9a33DL48Zoa24uAE+w47FbZ4XAQ
    GOhNUXE5cXEnfZo3DxvvdMgnXY1UDCkqLHerqTEzelQ0mzcfIj/vnLFLF0d/4CDIAlhGxsWd
    9CooLCMoyAen04nT6dSLCurcFWoV1Dy9e9eRdt+t2sHEiYO47bZh2SB+LopsLMqpSjL66B1B
    ER2ukBFaPUyZZLUWTk87ndn+ZFKGIT/vHKYaM0qVAl8/T5o3D364deuwwmbNAo+rVO7rwN4r
    7tBJ96yss8x+9RF69+5MfT52796deWLarfzjHx+wd89xZadOre8SRY2HJOXcWlR4rkdGRl54
    amo2+fnnMNWa0eu0NAvxp1OnVtM6dmydbTCYPgLNIlk+XiEIXa5nuHja8eNJr7/11lL94UOn
    sNudDercXWh4VVbUsHv3cfbvP8G2rYd9nn9+6pyu3Tq3kSXl04D9RnFy7dp9gHA+/BjVuQU3
    De7GkUPJHDiQdFHtwkt5KUGByWRmx/YjHDmczKiR0Tz62Hhatgpv1Jz89JNVxh9+2FUX5hYv
    0z6RigoT7777DdnZBW0ef2JSo+Lk3Llfq5Ys3cjyZZsQRBGNVu3Cx5/EpICA1WpDckpIskyf
    vp147tlrw8kh9ZxM/G1OOh2uNIScnCLi4pIu5mRRuVtpaRWiKFBdXcvZ/JJLcjI7pwhRFOo3
    Ef8tONmzZ0fi409hMlno06cT06ZP5B9Pvn95Tp7OJj/vHKZaC3qdhpAQfzr+CU5eJQ+l1s3p
    rP7ghx+2DZ8792vat2/Ok0/eTrdukbgZdXCBx87fz5unn76b2lozKSnZzPtsNQsW/IxCIYZP
    mzZ5tixpT27dkFg4bHTUdQRGDZDjmXwqS2kxW5l46yCMRiNWq4Xlyzey8Ku1FBdX8Oyzd9Gr
    d8ffME5+TxtdXtqYIT2x22w8P/MzvvxiDS1aBLcfOaL/zIpKefr33x6xT5rS43pBMhiqPz16
    9Pjot95apj6RmM5ddw1n+ozb8Pb2rDMmBUA8750NDw/mtdcfo9m8H1i+bDOnT+can39+6pwe
    Pbv1RdI9ZKo5Xm0wdrnOoLSflGzOh7t2aXdmzpzpj37x5Y8B336zjSeffI9HHh1Pq5ahf2oH
    oiAIZGUVsGD+TxQXl3PP1JE8+sj4c0GBQfNstdLbChW1f/XJBSGSnJx4yW63J6u1bZMl0iOC
    mvk9ZbXaWLJkg6JHz3YPh4aErCsvPxzr6gMZWZKIbBtGVVUtW7ceFkePHnCrp6fXJ7Isl0OS
    T3p6tkKpVNCvf2d27T5OenqeKAiS31uPfQ1Inqba8pFbtsTRPCKQwAAfSksrzg9eV25fVfei
    ovz7Fy5cqzCZzPj5eyGI4jJBcLxls0qodCJGt+5XMNbygRovKH33VHLqlCWL1+l2bD9CWVk1
    Wq0anV6DLMmYTBYEAcHPzzOob7+ooBHDo0d17RZpzco8i1KpJCIiiAs3U4FEq9aheHgYyc4t
    pLKycuLx42mTdu08qtq/P5GzZ0uw253o9RqUSgUWqx1TjRmDQSv06xcVcf/9Y9/q2avzYFFU
    Tq+tSsjRu3e+xjonA2m3Z6Rnzn551nz9sWOn63aiKn5zDCqVrhqOBw+eZNas+bp3333y/pYt
    WhxcvnzNirvvHn/9OZmcpXTVHQSNRsWddw3noYduwdfXi/nzVxMXd+qKdEyhUFBTY2blqh2c
    TMrkpVkPEB3dqVFycs7by9WHDp26wCj+pW/rjev6hYEoCthsdlas2MrptFzjzOfuaTScnDt3
    xqNffPljwKqVO7Db7QwZ0o1Ro/qiUl+5SWC3Odi8+QDr1x1ArVEzeUpMo+BkvdjsDrZvPyKO
    Ht2/AScz0nMUNpsNoW4DZHrGxZzcvu0wdpv9okm+MXOyRYtmZGTk8q9/fUhFeTUffvQUbSPD
    G3Ay4XjapJ11nMzPP4fFYkerVaFSqbBYbFisNtyMOqFf/6iI++/745xU/nWFqwYKBmVkZI2e
    9/kaWrZoxpw502jZMvwC75Z8kdGl1+vo1q09r7/hz/PPf8qyZRvp27dTr97R3SZbLI6PuL4n
    54igbFVcXIZaoyIiIhhJkliyZAPvvfsNJpOZvv06Mem2wb/zWAK/nOP520bnsGHRJCdn8eEH
    q/jww1VC69ZhU1q1bL594mTDKqZcjyarVFDx6KFDR8bPnPkJZWU1zJx5N3feOQKNVnvemDx3
    rpxDh5Lo2bM9/v7egISXlzvPPHM3zSMCmTN3Bc8++7H6rbeeGN87uus+m9XxKdc5bUEQ2uC0
    HbNkpZe+EtGm2c6ZM++d16plSNvPPlvNm68vRaNR/alNNLIsY7PZ8fb24IX/3MekiTGpBqNx
    enaafUdQMySV9urkcoWFue6TmXmIiAhDelCQHxqNmoTjaaxZvcv38Sdue0qSnMd/gZkrxaJn
    z3YsW7qZ9LScNt26eUZD1W5Zpv/p07l4e7vTqnUoAYHepKXnYq619Jn+dhcD2Pumpma2PRZ/
    mqlTR3A2v5SSkvILRqZRIUm5j61evcPr8JFktFo1wcG+CILiXEmBiF9wpyt8h9mA0g3kD2L3
    H5760qwvyc4qpHfv9owY2Ye2bcMwGvVIkkR5eTVnzuRz+HAye/ccZ93afbRr11zjCjc5KSos
    vUjfKipqsNnsJJ1I5/HH3lYnJmag02no0qU1t98eQ+vWYXh5u6FSKzHXWsnJLmTHjqNs336E
    Y8dOq1944d6bx4wZqJYFxQNVpcfPuvtcy0k+091ur3li2fKNbseOpTUoM/JHDa+EhHSWL99o
    fP7fDzw+4dZW6zif73V9OWm3O1CplNx77xj+9dQdaLU69sceY+nSzZfciPNrvfq18eVqn8DJ
    k2d46cX5fPTxv2jbtmWj4uS/n/+U9PR8RLGhN1mSJFq1CmH0mL5s2niA06dzL6q9eSjuFM/N
    /FT95huPNTpOzpu3mth9J7h57ACGDxtwhY8lsnnzbvbtTcTX14Mnpk287pxMTEi/JCfPP6Eg
    ciz+9EWcTEvLxeGQUCoVSJJMRnrerziZ1fbI0RQEQUS+IHXq15w8ciTlunDSzU2PJMkuTmbk
    cehynCwqxdfPE5vNid0hodGoqaisruNkxnlOiqJIx04tGDmqDx06NMfHxwO1WoXJZCEz8yzb
    tx9h185jHIv/45y8Ch5KhQKkkbt3HdGeKy7juWfvrDMmf6/8lgw48ff35YnHb+XBh95k08YD
    iq5d2k/t0lVY8eLLhedefTnwOqmaRS1JtmjXiQ5KVGoVJ0+msWD+z9TWWpBlGV8fj7pTY+TL
    GpN2u52ScxV4+7ij0Wh+06hUKETuuWcUhw6dYv++Eyxa+LP7C/95cKZSKe8Giq59m809CgvP
    /mPO28s5V1zBS7MeZNKkmLpVd73yKDhw4ARP/etD3n3vH9xyy2DAAcio1Sqm3DESlVrNrJcW
    8N5736o/+DDo2YCAoG24ampcV1Gou1JdcVAC5y610jB+6tSxs9Vq5e0vvbSAQTd1ZfDg7ld0
    uo2AwM6dR9m+7Qj//Odkptwx4ienTfw/cJ7y8QeN4epvDGjevBeynC42C/JFq1WBoObbb7cz
    cFD3sZ2jIieCePhCA2PgwK4sW7qJPXuP6zt3jrxNoSyLr6quDM7KKiAwyIeQZn40C/YlO7uA
    6urqIP8AfzdZNt+zd0+C3umU6D+gK6tWbr9gYsQJJZOSk9MmfLNia10VA5ngIF8kSbZeKSTr
    nxTKp5w8eeqOl176ksLCMp577m6m3DEUNze3CxZgrrfev39X7rxzBDk5hWzaeICVK3eQnV2A
    JMns2n2MUaP7oVIpzy/aDh06RUVFDWVl1QQF+3DP1NGMHduX1q1CUWs0F92/W7f2jBrdj23b
    D/PmG4t59dVFuHsYRwwc0HN6Za38Ut0Av1bGSVROztmOO7Yf4c8WCRAE2L79KHfdNbJjy5Yt
    oriu55e7OFleWoXTKdGrdyQPPzwOrVbnMu7Lq6kor/7dRZparcLP35PysuqL6lQqlQpSU3P4
    fN6PvPnWE+h0ukbByblzviYtLf+S3mRJkonu04Gnnrqb8vJKUlJyuDASXr8YSE/L4733v1V/
    8EHj4+SLL85n955j+Pp6XrFBuXv3caqra3nllYevMyf9UKuVmEyWy3Ky/v2XlFSwd19CA05m
    5xSd3y0kCAK5uUUNOLlvb4K+5FwloiDglLk0J7/ZitMpodVqaXYNOHnHHcMwunkAEhZLLTab
    g/DwQPr27cTd94wiO7uADetjG3By5854Yob2pHfv9hQUlBIV1YrlyzfWcbIKb293xt8ykAm3
    DqJ9x5ZoNSrMZisOhxOlSoFep6NPn65MmHATGzfGMvft5X+Yk1fBoCxXWq3VXRMTzxAc7EvP
    nu2u0Lko0a59c9q3i+DkqUxqLSbPoJBAxfVSMFnOAqpaFxcXhsfHp9I8IpCgQB+WL99EjamW
    u+4ZjlKpZNzN/RFFxW8alMePpTJz5mf8819TGDfupt8xqmV8fX145JHxnDx5hp/X7mdITM+o
    oUP73CrLefOu5e5TV8JxxcgdO+I8jh8/zYMPjmXixMF1xqR8gafVteKx2x04nc4Lwt+uyVwQ
    BMaPv4mkk2dYunQT+/cnhE6aFDClqjJulrtH7+vNStw8o11TgOlYqkrQbQ4M8L1doRDp3Lk1
    t98+4grHpUBZWRW7dsYTFOSDKKrX2SXbKUHoeo1boUv28fGoNBp1Hl26tOHw4WQWL1qnefXV
    Zk8ajNpZ9bSv94p07tyGbdsOM2XysCEB/rp+ZSUV2ry8cwyJ6YHBaCAsPJC4uCTKyqo8vb2r
    h5eVneu5fcdhOnduSZs2oUiSVHfyhGA06pURFnP5v5Yu2eBeVWVi0E1dOHr0NN4+HhWSdOWT
    nyxXAiWtKytL//3xxytVmZkFPP30HTzw4Li6HLtLT1yusE0oT0wLYeiwXnz00So2rD9AXNwp
    cnIKaNkyDICysnK2bIlDkmQG3dSFp566g86d21ywKe7S99doVIwZ3Q+1SsEzT3/MRx+uIrJN
    +GO+PgGbMjPj9jRvfq3Gbm27M2fyPYuLK/502an6yTEz66xnmzat2l0vg/JCTh4/fhqFQiS6
    T0f0Bi0rVrhOZFq3bj82m/032yZJMl27tWHu3BnMX/ATSxZtuKhckigKbNlyiGHDezFmzIBG
    wcljx1Ivygut97TKsoxQH/4W6svGSBedHqNQiBw9mtr4OBnoe7tSqeTH1bvYuD72CikJFqsN
    lVJ5AzjpXqk3aDzMZit5ecUsWbJeM/uVhpysN9UkCXbsOMLk24ee52RubvH5fhMEgfz8kgs4
    Wdxz584jrjSVumtk15x3npPLlm50z8kuRBQFdHotXledk7fgcNjZufMg+/YlkJaWR2WF6yjg
    sPAA+vbpxJCYnjwxbTJDh/Xmo49WsmH9AQ4eTKKosJRbb72J+mjCls1xOJ0SXbu25pln7qZf
    /yhMplq2b41jz94EsrIKMJutuLnpadEimEGDutO/f2duvXUIGo2a52d+xkcf/T4nr4JB6cRq
    s8jlFdV4ebnj7mG84mi1TqfBz9eDlNO5WKx2WaFUydcHkjJwVoTyR3buPBKckZHPczPvxmg0
    MGxYT0JC/IiJ6YFWq/nVzu1Lq5bZbCUnp4hzxeV/2Jju2zeKoTE9+e67HaxauU3Zq2eHB93c
    xdXFxQlF/v7XKqdL7WG1Wobs3n0Mb293Jk4agkKhpD7MXVlZzaaNB6muqiXxRDqCILJrZzxl
    JVXoDRpGjIyu24DkOtbq9tuHsm5dLHv3JnDzzQP7W6yCgety1NRlxpOhK7KcJkp1eZOuT+mK
    V97nv19Xfkhr6Hodnt6aanTTVbi5Gzz8A7yYMGEQ33yzlZhhPbqPHnXTDFmSVDKu8h8Gg44R
    I3oye/YiEhJSg0aMDHyysKg0uLLSRMsWwYBIZOsQamst5OQW+bRt1+bhEyfSQjLS8pj0/D0Y
    jfq6PHwZQRA91Brxn9u2He22bl0sEyYMwsPTSGpKDm5u+iqlUp35Zxwi4Lx3375jzXfsOEr/
    /p246+6RdROz/DvRC9f/t2kTwauvPgrA+nWxbN50kCemhQKu3ZsnTpzhppu68uZb0wgO9v+D
    /ey695AhPbhl/ECWL9vI1q0Hve6+Z9wEu13ayzVLt3HI5RXVrjOe/8pdnBKVVbUgCDeEkykp
    2YiiyOZNB8jOymf79qNYLDYkp3zRUX2XupfBqCU0NIDgIN/LGs0mk5mvv95Mv/6d8fRwv6Gc
    3LPnODabo4F3UpIkwsICGXtzX1QqJV26tgFkhgzpgbeXO06nxIb1BzhzJr9B+NtutbNvX2Kj
    4mR9eTWz2YbFYvsT4wN0OvUN4aSvj6dHyblKRFFg86aDxMT06D5q1KAZsiSpfr1IOZ2aQ0Ji
    atCIEYFPFhWVBpeXVV2QcgHV1bUXcDI95HRqzq/yZLmAk/Hd1q+PPZ/S4O3lhpub4apy0mq1
    8v57K/jmm23Uly2qf95Dh5JZ+/N+OkVt45FHbmH48GheffWx85zcsD6Whx6+BYfDydYtcSQm
    ZtCla2vmzp1Om8jm7NkTzxdf/MiRwyl1dZU5H83ZtzeR1T/sZsKEQTz73N2MGNGHuLgkli39
    fU6KV8OgVKtUgpubjuqaWmpNliu+g81mp6rajF6nQaNWCjRwMF9LSQVMY7Ozsu9csngDnTq1
    ZNzNAwCZli1DGTt2IDqdrq4Tf/+RCovKUKtVhIT6/1FVRKPRcuvEm/Dycic29gRHjp7qLAjy
    JLv9Wr6C6tDS0oqO6Wl5tGvfnLCwwAZhwZzsQj799Hve/+AbtmyJq/MYxPH+B9/y8cffkZGR
    d8HQkWnRIpgOHZqTnJxFWWlllJenLowbLo3oNM8rY4vk4emJj48HRYWlTJ48hKAgbxYuWCuU
    lJwbKeF0czgkEFyTc6/eHfH0NLJ9x1G1LNsHZWWdFSXJSfMWzQCBZqEBAGSeyRdk2d5/+/Yj
    ajc3A336dDrvYrDZnCDIbuXlZbd99dValYeHgSl3DKWosAxPTyNGo0GWJOefCAVXauz2mmG7
    dx/DbrMzfHhvPDyutOi8hJeXB089dSdtIsNYs2YPZ88WYzLVsnr1bgL8vXj2ubvqjEnnFdxb
    RqFQMWxYTzQaNTt3xlNTUxMTFubmew07V3B301/yONorGiGiiJtBi2sn6vXl5LKlm7DWFfFO
    O53Lzz/tx1xrRZZ+35isn7hdKRRO8nOLL7tZThRF4o+msnfPsTqv8o3j5JmM/Iu8rpIk06FD
    c6ZNu41p0yYxcEAXQGLggC5MmzaJ6dNvo2u31nVGVsMXkJqa3Sg5WX8e+5X/uXGc9PX1rCtZ
    JVBTY2bRwl84aXfUR9Vc7aqttbBzR7yLk9kFotlsbeAQslhsZGa6OLlj51F1Tc2F6RgCNvsv
    nFy0aJ2qqqr2vEHp4+OO0ai/ipz0ZNvWOJYu3UhNjRmlUoFSqUChEF11lpUKHA4nR48kM/O5
    T1m6dD0eHkaeevoXTpaWVgICq1fvxsPDyPPPT6VNZHN+XL2Tp5/6iH17ErDZ7HX3rb+363fM
    ZivffruVn9bsRqlUMXJkb3Q6ze9y8ioYlAZZq3UvjWwTxtn8c6SmZl/hbV1FRVNSsmnRshl6
    vbGstKzGca3zJ2V5PUC4xVL5/MKF67wLCkp4/IkJ+Pp5X+Ahkf7gBCVgNpvZtu0wnp5uREaG
    XcHEJtG1ayTdurWhvLyGtWv3KWtNpsm+vlr9tWu9g4qKSqqqawkL9UejUTUATJvIMN7/4J/M
    ++J5pt47GkmSmHrvaOZ9MZOPPn6KTp1aNvACqTUqQkMDqKiooaKqWqVUqRU0yZ8UfZXRoD8R
    GOhDdk4xPr5e3HvvaE6ezOSHH3a4iqs7ZVdBI0kmNDSAvn07cSD2BNnZZzlzJh+9XkdgkA8g
    4+/vhbe3O5mZBWRn57NvfyI9e7U/v/Gs3mskyRKr1+wi/thppt47ioiIIPLyivDz90Sv052t
    qDCb/9SUXF1JVuZZtFp1nV78GZFo0SKEhx8eR05uMTt2HCU+PoVj8ae5667hdOzYit/P2b70
    ZBoWHoibu57c3CIqKio9NRq19hr2bUVERLDdw0P/p08QkWVwd9cTGhZottvlgmvvnfyFk4sW
    r/dOTs46bxCLoni+3NEfCeHLsoy7u4Hhw3pRXl5N/LHUy36v/uSrdetiMVvMgHzDOFldbb7o
    ORUKkYMHTzJj+jtMn/YOq1ZtB0RWrdrO9Gmuf9uz+/hFYfL6UlBNnLw6nPQP8GrghTx2LK0B
    J3/97g8eTDrPSZvN0cBD6XRKZGUWkp2dT2zsyYsMZVlycfLHNbu48IQoWQZfv6vNSSf79p/A
    bLZddqFWn5tbWVnDhx+sYsvWOFo0DzvPye3bj3A0PpkjR1O4ffIQ+vSJYt++eN5+eynnzpWj
    UCp+U//sdidHjqYiSXYiIoLw9nEn53c4eRUMSj8bKJcOGNjVotVqWLBgLSUlZS4v7u+KiNls
    ZsmSDVRVmRg+vDcqpXprekZl+bWFZDXQQSPLjrkbNuzr8/33O5k4aQgDB3T7E54tV67htm2H
    2Lc3kSEx3QkJCbgiT4nRaGDIkO6o1UoOHkgiMyu/k1otda6qtlyrN1C3XdjlHW7oJZDRaDR0
    796egQN60rZtOE6nRNu24Qwc0JOePTui0+katE+WweFwXjCxyE28+9MimQRBOtUs2IfSkgoq
    K2sYd8sgunVrzfLlm0g6dQaxfpISQKFQM3hID0pLq9iy5RBnMs4SEOBFYIC3Szv9vPD39+LM
    mbPs3BnPueJKhgzpjkqlbuANSknOYtmSjbRrF8HEiUMwmSwUFJQSFOSLRqs6lpycXfFnIjmS
    JLt2BKuV6A26vzRmY2J6EBkZxvZth9m08QC+fh6Mu2Ugf+UgDo1ahUarxmq1Y7c7rg4SL8+K
    HWFhQUf79o06b8xf8VuQJPr27UREeFCCLAv7rysnv9v5F+7l+hwzth+9ozuxZUscaWl5v+nV
    FASRo0eSSU3JBrhhnJQvM+FWVNSwbdsR1q+P5VTyGUAg6dQZ1q+PZcuWQ5SUVF7yHHOaOHnV
    OBkc7HN+DAmCgNPp5JsVWxpy8gLO5eUVs3XrITIzCy55kEBWViG7dsWTl1tcl+Z2wfcVLk4u
    X7b5/HxX/7uBgT5XlZOy7KS8rPIPfVsURcrKqvjwg2/Jycln6NCeDTjp4a5n0qQhlJSU88EH
    qygoKL1s/dRfv4/yiiokyYlGo8Jo0P0uJ68CPa2AYktUVNvYe+4dxb79icyaNZ/k5DO/swoX
    yM8vYu7c5az8djs339yfvn07J1ks0nJPD8011rRkwHZ/YsKpMe+99w2tWjXjoYfGoVSp4IpT
    kgXiDp3gvfe+QavTMGnSEJRK1RXDom+/KAIDfSgsLOXIkWRPWbaPsJpeuEbtV+Pu5iYYjToy
    swrqTjL6dRhEApyu8jOtmuHl6VbnBbrYa2s2W8jMOou7mx4PN6NNdjocTcD7s+IPqMWgYF/M
    Zht5ucV4eHjywIM3U1lpYuFX66g1uTwmNpuD0tJyoqJa0qJFMGvW7CYp6QzNmvmRdCqT+V9+
    T3x8CuHhgWRk5LNq1XZCQ/3o3bs9Foulbm4TsFisLF60ntLSSh55eBx+fr4UFZVSVWWiWTN/
    BEE0R0T8mYiBDY1GLXh6umG3ObA0CDFd+eTu5eXJqFF9OBqfyqZNB7lpUBeaNfPnr1Rfsdns
    WM023NwM6HVawHkNS7mEl+r1ngvvf+BmR0TzIJxX+FNOp0RYRCD33TfGYnRz+7xa7Fh+vTj5
    4Ycrqamu/Qtn2MtotWr69+9MRkYuCxeu+90NPKIoUFpaxaG4pBvKSYMrveCSRmV9uNBUY6Gm
    xoTZbD0fNrxU22QZ3Jo4edU4GRjo08A4EkWR3NxiFi1cR22t+SKj0eFw8vNPeziVlHmRwSgI
    IukZeXz/3U7sdmeD7woCWMxWlixeT3Z24a9+UyA42O+qclIQFHj7ePxxk1Qhknwqi5Urt+Hp
    6c7o0X2Ij09l48aD9OkTRYsWwfz0026Oxaf+IWPyvOfV2x1RVGC3O6mtteBm1P8mJ/+yQSkI
    WkBVpVJ5vnP/fbfkT506gh3bj/DII2+yafOBy3gPROLjk3ns0bf5evkWRoyM5sl/TClXKvVz
    rRb5VNvIftdw1Z0B+Heuqip5et68H/RlpRU8+tiECxL6/9CaABApLa1gxYqNPPfsJ2Sk56FU
    KtBq1X8KtmFhAbRvH47d7uDAgZOYzbV9RdVdumvzFvQFAQG+KR06NCclOZukpKzLDAWXN2TF
    N7Pp1z/qMu9H5NSpTJJPZdK6TSieXu7HqittOU3A+7P6JAD68qAgP8lms5OXXwzAoEHduHls
    P/buPU5+fjHV1bXMeWsZd0x5kc/nrSYk1I+U5GyKiys4e7aEp//1Ia+8spCZz31GcXEF5eXV
    JJ/KokXLEL7/fgdPPDGHZcs2YLfZyc0tYufOowwe3J2Yob0AyMs7h9Vqp1mwnyw7xcqQkL5/
    ZpxJBoNXeYuWzbBYbKSkZP3l9xMd3QFBEKiuNhMd3QlB+CtRQ4HsrAKqqkyEhwfh6emeYqu1
    Vl3bvjX+EBXVce2sWQ9ILVoGX9H3W7YM5qWXHpA6dIz8sbDI8oOviuvDyc9/0GdlFlzk8bnS
    d20225gzZxkzZrxHelreH5rYJEniYFwSVqvlhnGyfbsIZEn+zcl8185jPP7YW+za8f/svXl8
    VNX9//+8986+TzLZE0hC2HdUdtlUBBVwt251r7bWalvbT1vX1qWt1VZrba3a2tZaV1xxBQQF
    2SEQIBBCQlaSyTIzmX279/7+mEkAAS2QhH5/D87jkYc8MNy5Z+65z/M+7+1V/rUqQaqqUlZW
    cIqTvcTJ/Lws5Ujf9+rVFWkvo3CYN6+ysm4lLmsAACAASURBVD7tpTu8u0BnRxeVlXWH/b/u
    tkIrPy8/wvOXyM939TInRRYuPJOsbMd/Hc1QVVi5cgter5/Jk0bS3ad32vTRRCJxPvpo3SGe
    1a9/71QcDgvnnz8VUdTS3NSGxxOguDj/azkpnjh43KREzPVr7PaMj374w6vUq6+ZS2ODm/fe
    XU0ymTyCUanyycfrKS/fw7nnTuK++26goCBndSyqfJ6V05Ru8tlnrnIBYjesWLGhbMWKLVx2
    2Vmcc87EgzyO3+SRhIb6/bz44nvcdOMj3H/f80iSxPARJfh8Abq6ghx7CC5VnHP66cMQRZGq
    qnpaWz1j7XZ9cR9tF506nfXzhQvPJB5P8q9/fUAwEDjCclDR6XTk5WWj0+mOcEoXCQQC/OPF
    D4hE4px77iTFaDS+b8uwhU8h70SGvCzDaQ2ZTHrKt+yhq8uHXm/kxpsWUFiYhSwrbN5Uxcsv
    f9pTnPL559uQU9JgbN9eg1ar4eZbFpJIyqxZU5GCkiCwbt0OnvnTW2zevJvfPvYyq1ZXkEwq
    5ORkcMstCzEaLchykl276hBEAYfTEhYkzRfHNw9XVBQ1H06aNBKdTsuyZZsIBo/n/TiwHgsL
    s8jPy8ThsFBSms/xhw0FkskEn3yynlgswdRpozCajGt1psJA3xlp1YDHIwj6l846a3LbAw/c
    SFFR9uGFG0eAe35BFvfdfwPnnDO5TUB8IzcnJ6iq+/qFkytXbEEUBVQ1dS/HojqlqiqyrKCq
    CmWDCzEY9OypavyvPZ2CILCnqpHm5nZAOCmcXLBgOmaL8Yjz7v67To+flSvL6ejoOuTvv/q7
    RpOeuXNPcbK3OOl0WkNWq+mQ71sQBOSkfFTj6XjSTbrVdJKJQ6+pqipGow6Ho7c5GWDKlDH8
    4M7LMFsM/9U9i6JAy/4OWls7yc1zkZXlwGY1M2hQIS2tnTQ3t//XxqTBoOWWWxYwa/bpKEqS
    pcs2Eg5Hv5GTvdA2qBmwFUPn/ZU7q69++eWPheXLN6HVahhUmp9Wg1APg3lJSR42u5lVq7by
    yMMv8q2r5p43bszIkkR85IMwYLEa24ugL+uLs43L72+fs/jNFbhcDr593XlEInFe+c8nlJUV
    MnXa2KN6JT0eH6+/vozXXllG8/4OSkvzuO22C1l04Uz+9sL71NQ0n9AJftToMqxWE60tndTX
    73cUFw8cQR80vxWEPFS1/p9Tp024aNGFZw5e/OYKnE4r3//+5eTmuo6y4R9+KHC7O/jT06/z
    yScbuOzyWcyaddpWJSm8WVfbdYp1JwbKaEamE6fTxttvf0Fjk5vp08dhtRrTqjIqoVCEWbMn
    8NOfXsPevY18/NE6mprb0WokRo8p4+KLZzJ8eClz5pzGyy9/QuXOOhKJBK4sB+edN5XTTx/G
    E4+/wtq1OwAoKMym0+Pngw9WsnbtDj78YA0Ou4WMDJsM+I5vHhFA+mjixJF3TZ48YuAXq7bx
    +uvLuf7689PeqWNNL0m1SjIZDYQMUZxO60H/79iv9ckna3n33VWMGFHMrFmneRMJ4aOqbdv6
    yJisBQQjaC72elseW/L+5zlvv70St9vzjVWyggDtbV7++NQbNDS05Zx33rRnsrIMQ8D6V1Wu
    9AnSiD7j5NtvrSQSiaXVYSA7x0ksFqfLF/7a+071YlTR6TUMLivkrLNOY+GiFCd376o/BlYJ
    tLV5qd7TSGlpqqjrZHBy4aIzB7/66jL4igqQTq9Fr9Oi1Wp6Kn6TySSxWIJ4PHmY4XnJxTNP
    cbKXOWl3WPAc1ALogAfzUO8dqD3/PdohrvufCSmB8q+9ZvdztdvNvc7JN95YznXXXcBVV84j
    EYvz1FNvEghEjiD3eWBeqqqmc987GDAgG5NJj8mkx+VyUFfXgr8rlD4QKj1tgoSvzFFVVSwW
    A7d990JuvvlCtFo9n3++iTff+Ixhwwd8Iyc1xw/IDqATEM+MxXzPvPPOZ6P+9PSbQqfHz1ln
    TWDu3EmcOX3cUZKPVRYumkF2TgbLlm7ko4/Wsmp1hXTLLQtHXXvtgmcs2iSCfuhiX0c5Dldv
    97TSTGpocJdu27aX2bMnUFiYxZNPvcbzz73HnXdezpSpY48ASpEdO6p57Lf/Zu26HYwcUcJt
    372IadPGtuTlZW8LhcNn1tQ0mu02M3a7+Ti9JirFA3NxuezU1bWwd2+TbvYseVIsWr1Ybxjc
    By/jgEqLJXzrXXd96+8CFL/xxgoqKmq59NLZ5OZm0NLaic/jJxCMIMsKkihishjIdNrIzc2k
    vcPHm2+uYNeuOi6+eCZ33vmtOrvd/nOWLGkdtODnJxUzKZ3VqCIcAoJjdcYfkFdLX0fp7Cwn
    M7M/eqyZoxkZzoDDYbU2NrjZU9XIhvWVqQ1ap2Xw4ELOO28q11wzj5xcF8OHl3LuuZMJhaNI
    ooDVak6HglVmzDiNiRNH4vP6icfjWG0WnE4bIPHbxzJ59tm3WfHZJrZtq+aO7/+eZFJGVRXi
    8SQjR5bgcNgS0VjiuHK9BMGCqlZWOxyu/9z23Ut+truqQfjjk69hsRi55JLZXzlsCgf9wMG5
    uoqikEx7HGKxBIgCWp2WeEImHI6g0QhIkiYdavzqdQ4uq0g17VdVlWVLN/DIw/8A4Lvfu5j8
    vJxX/AF18+gzZvfBeqwDcED4scrK3Vc/+eSrphWfbSaebr/zTV6DVL5sgi1b9rB9R62watXW
    vLvvvubXw4YNGYIg/TgaLPcZLH3Dye3ba3ukBueeO5Ef/vAq1q2r4JGH/0kikTyKByilBDZ+
    QhlXXH7WIZysrW0yH0seZvfcd1fVce68VDjxZHDyzjuv+LuAWrz4rc+JxxKgwtDhA/judy8h
    JycDi8WIKEkoskIoFMbt9vCXP7/Frl31CALodFouumgGP7jziv85TnZveN1N2o/XIZK+zEnh
    ZPd9qwf1DlZVtef90mgkjEY9BoMOvV6HTqdFk27BkzbJSCaVNGPixOJJotE4sWgcRVHSP6nD
    xMFFQKoKdruldzm5u0F46snXMRh0XHbZ2dxw4yJEUeSJJ17tMSpFUcRo1GMyGzAadOj0OrTp
    lDtJktBqUj+SJGHQpdg4ZGgRsWgiNcdonEg0TjgcJRqNI8upOZpMem7//qXccvMiREli6dI1
    PPzQP4nFEnzve5d8IydPwEOZCfhnhMOdf3vuuTfKnn32HUpK8rnn3huYPXsCer2Bo7fdUTGZ
    jMyZM4mZMydwwYLp/Onp1/n9E6/Qsr8z564fXvVnVY7z+sKdi/tgFZZ2+ULmWCzOkKFFuFs9
    LH5jBbNmTeCaa+YdBMgDG92aL8u5/4EX8HT6+cEdl3LpZWe35OZmvy4Imr/JsjJq9+66s6r2
    NDJ0yMB0kcDxGZQZmXYyM23U1DRTW7sfRUla/vPKzj56GWuAthV5eYW3/PwXNzw9ZuzgIX97
    4T3xwQdfQKfV9hjVkkZKhbsUlWRS7nlpE4kkxcV5PPDATcr550+vttttd4BlKefOB35+EiG5
    D7CKEClNKjKqCp5OP/V1TcfsxfJ0+lMhO0UGKPV4VTEaLFd6f/P+6rDVGY3abYMHF+bv2rWP
    W29dxODBAwgFw9gcFgYNKqCgIIdUJ4WUkaTTGdHpuosHFFRV7jHTDAY9uXk5X/EyqxQXF/LI
    I7dSXX0ee6rqaGvzYTYZaGxy8/zz71NaWoDVat4eDJxIrpekgPj45MljB917z/WX//KXf+eh
    X/2dtjYP119/ARaLBUVJEgyGaGnpwO324G71sH9/B23tXrq6AkTCceLxBPG4TFKW2b27AVmW
    +f73Hken16DTatBqtVitRpwZNnJzMsjPzyTT5cTlspObm4HFYkaStMRjUV59bTl/fOo1IuEo
    P/rRFZxzzuTPVMTfSJKa6P31WAVITgg9vn79lhsefOA5obKyHlEUvjbX7kiGlSSlQnlLP91I
    c3OH8Mtf3nLDxDNGoSr8uL293JeVNb7XORkOR1EUlZKSXH7602tx2K1s3lTVw4IeYa1un5Gs
    kJXt4LpvzzsiJ6v3NiEeY2GPoqjU7WtFVWUEQTxpnPzZz1Oc/Pvfl4h7qhqYNGkUF17YvbHK
    hxxGQWRPVT27d9czeHAhN9y0QDn/vP9RTspJVFXl9IkjmDBh6DE7+7dsrmLH9r1pNbWTwMmy
    wvwtm3enDBqNhN1upqgom8KibIoKcygszCYnNwObzYzJZMRk0mMw6NFqxEOa1ccTMvF4gnAk
    SjQSJxSKEgxG6Gj30tzcTkODm8bGVhob2/D5gulUDpVBg3qXk/fde/3lD/7q7zz0qxfZ39zO
    TTcv5NvXLSAWl9m4oZIRI0soLc0nJzcTu92C1WzEaDKg12vRaTUYTQZURUGj1SJJIlq9jsmT
    R/PKKw+RSCSIxZKEw1GCwTB+f5j2dg91+1qo2F7DyJHF3HTTQmLxBP/659s899x7xGIJfvxf
    clJzfItxP9BQlEj4n37xxbfL/vzMW0ybPoZ77rmesrKBdFcIf5MBBTKSJDF16jgGlRbwu8f/
    zSuvfAoC2Xf/+No/z/vP6GjCVPGBVjumFxehNmyxGBWNRhKTCRlRFBgxopjLLpuN05nSMY3H
    4+kQhsSmjTv5+c+fRavT8Icn72La1LHvabSGB99/J7B1wYX52dD5nY8/Xqvt8gWZMXM8Vuvx
    eihBq9OQk5uBqkJraweBYHTCxReXOG+4gV6v5hSEsvRmF11mMtkvvuqqea+2tnaM+ceLH3LD
    jRcwcmQpLpcdo1GPKAooikosGqOjo4vtO/bytxeWMG/+ZK66at4OVRavBKkyEvZgMp9+EiFZ
    D+hE6Pyu273/B2+++RnhSJRXX13KRx98ecxB0a5AmHA4xhtvfsaIkWU/KB6Y0xL0xP8S7tqi
    mOwT+nAmlrhGY1l94UUzz/700/XatWt3cM218zGZbChKFL8/TENDK21tHtravPh8QcKhCJFI
    jEgkTiwaS1UQp/dtnU6LTqfDaNBgMOoxmo1YLUacThu5uRnk5mYyZEgRoqgjEgnxgx/8HqNR
    z6WXzU4YDIa3jcZPA8e/zoaiqlUeUXA8dMGC2XZRI8799aP/En7/+1epqWlm8uTRbNu6hx07
    amlp6UzlIAtg0OtSgNRpU6dwox6dToder2HChJQnKuVNSBIKRYhF44TCMZKJJIlEkng8iYqK
    xWwkK8vB0KEDGD9hKNV7GnntteVYbSbuf+AmLrpo5mdarf5WFRrt9nF9EToWIHpXRcX2G+69
    9y/CnqqmYzIkj2ZYVu6s5d57nhV+97s7bhg5cmgg5JV/CsR7m5NarUZU1SilgwoYMCCHf7y4
    hPffT6k96vU6EonkQV4hhWHDB/CLX1x/VE56PYH/usr04OHx+IlG4xiNhpPKySuvnPdqa0vH
    mJq9zcTjCZ5//i062n34A2GSySSSJGG1GMl02YlEU/vI3HMnc9WV/7ucfOPNz5AVlcsunc0V
    V8zjWBXFXn/9Eyq2VZ9UTi5dulGrKArfv+NSpk8bS25eBg6HNR2pEQ87TB/5z8JR/nzgkO7z
    BWhp6WT9+h38+U9vEU8kueTS3uXk+Qtm2wWNOPc3j74kPP30YnbtruMnP7maW2+9iO98ZxGS
    lKpnSCRSRm8kEsXrCRAMhYlEosTiqZQLrzdAJBLjsxWbycywodfrMJuNWMwGTCYDTqcVk0mP
    RqMDRBQlDoJAm9vD4797mbff/oKMDCs/+9m3/2tOHqeHUtSihn6+9NPVo579y9tMnDSChx++
    Ne01OdYGwynDMifXxX333YgAvPrKUgYOzMm+5poFP47FNKuqd230Dx5+Ri8tQnmVM8Paluly
    5O7eVU92dgZP/+lujEY9oNLQ0MILL7zHbbddhCRJPPLoPxBEgUcfuTU8ceLYP0fCPCGIcuuC
    Cws0EP7Z+vUVs957dxUlpQXMnz+FY8/nOuhhSBIZThuCAB5PgEgkmpeTk2kC+qQ9iCAMTQOm
    UfV665xr1+xg/IQhfO97l6DXH71w8swZ49i6tZoN6yvp6PCardaM0LLFXVxwxdSTCMkGQCdB
    8LutrQ2PPvTQ89YVK7Zw1lmn43I5UNXjS8T2ev188flWHnroBev999/86/zcHKGrI/GXiL9c
    Ntr68gQu/fm000acvXDh9NmvvLKMJ598FYfDyo4dtezbt5/OTj/RaIx4LEE8kUQSRXQ6DRqt
    Bk3aq3zwMkzKCnI6nJOUFTSSiFarxWDQ4cpyMKgkn1GjS4lEY3zx+VYuuWQmkyaOWgOafzY3
    zgd+wonBsnKHKGiuPm/+mU9kOO3ffuLxfwvvvbua995djc1qYmBxHnPmnMagsgLy8lzk5mTg
    ctkxmY2HKDkckkMEKOlQjSynw1XRGF5vkI5OH62tndTXt7JnTyPr11fy7rurUVWVkaNK+OEP
    v8WMGad/LIqa2wSRelEY0wdrcj8QntPaUn/bY7/9l7CnqjEd5u+F1SFJVFU18Nvf/lN4/PE7
    ryvIz3uvs23DZ5nZE3uVk1lZjtyuriA7d9Ryzz3PsmF9JbKsUFZWwK23Xcg///EhlZV1oMKg
    sgIe+tUtR+Xkkve/5HgKslJV/eEeg/Jkc3L9+kpkWeb115aTSCR7JAe/Gv7VarUoisKmjbv+
    Zzn58MMvWJct3ch5501h3LjBuN1tx3zNceMGM2/+JJa8vwZJFK33nQROXrBg6uzXXl2O02Fh
    +IjBgEwymaDL7ycQiBDp9joGI4QjMeKxONFY/NBiF0HAqNelwuMmA1aLEVPa+2e1GLHZzDid
    DpxOF60tnYQjMRYtmt7HnHxZWL58M/ubO/jBnVeg12mo2tNAQ4ObNrc3lZrmDRCPJUgkZWRZ
    RlYUVCXVEg1Ufv6zv6RUcMSUR1arkdAbdDgzrGRnOcnOcTJgQB6DywoxmQ08++zbrFxZzqjj
    4KTm2BfkPiC8aN++fVc/9eRroivLyb33XJ82Jk+khZuC3W7jRz+6in11rfz12XcZO3bI1PHj
    Rt4QC6tPn+DFD16AbXm5OfXTp43OXbJkDZ99tol586en5ybz8sufsHpVBd+97WLeeGM5tTX7
    +fVvvssZp49909sh32uyijFJcmmh647t23fe/MjD/yAUivDTn17NoEGFJ3CbKqIoYjIZDwJo
    FEGQxL4FTDUQu668fHfR7qoGfvrTq1EVFbfbjd8fIhpLoCgqoiBgMGixWs3Y7VbOPmcij/3m
    35Rv2V1yztwzzz3/cuNzXMFJHKoEwe+1uhsf/dVDz1mWLd3EzTct5Hu3X0Kqj9zxjUgkxl+f
    fYtn//oOAljvu++mX+fn5QoRf+IvQLJvNjABVV3n0+sz/vzt6y6Y8uWa7Ybnn3svZfy5HBQV
    ZTNyZCn5+S5ys504M2yYLUYMhlRukFanRfpK24tEUiYRTxKJRIlEYoQCYdxtPhoa22huamNP
    dSOrVm8jGIxQXJzLt687P2owmJ9NClJX4YBhvTCnEahKRWc0Kt41adK4hj/9Ke+2Z555M+uV
    V5Zxw43nc8ONCzGb9Yii9gjeg//2gJaa84CBB3sYZOJxmU8+XsNPfvIMM2aO5957rw8MGJD/
    ZCTCX/QGuUUQ+mrDU3ITieA9L730QfaXX24/Ls/c1x7rRYG1a3fy0r8/dtx155U/12rFnYC7
    Nzk5ZcrI3L17m2lr8/LKf5YiCAJms4HbbruI2XNO48W/L0FRVJxOKz+++6qjcvLRR/5JW5v3
    uL+DSDROPJHqGHIyObm1fHfR9u21mEwGMjPtuFx2rFYTBoMOUZRQFIVYLE4wGKGzs4v2Dh87
    dtRQXl5Vcs450//HOPm85f33ViOJElu37uWOO35/XDmU3apGqqqy5IM1IGC9797+5eR1110w
    5cvVFYbnn3+PaDROfYOburoW3G4PHk+AQCBEPJY8JB/ySJXT3TmXopgywHR6LVabGafDSla2
    g5KSfIYNG8ibb3xGVpaj3zj58sufctedTwIQi8WPqYdtIhFO65Af6nndt6+l53lrNBJ6vQ6N
    RiQYjDB33mTuvefYOXkcHkpBK8v+6994Y7ltX10LDz9yK8OGl/aSvaeQl5/DD35wGbd/73Fe
    eukj/dChJXflFuuWkEpm6YWR79Xqgs9ee+38UevW7jA/8MAL1De4GTOmjIrte3lr8edcdfVc
    fF1BXn11GQsWTOesOROXgfgze6YcE0WHBF13VGzb+at77/2ruaZmP9///qVceNGsXtA0FRDS
    wA2HI0SjMQlUXd9Bcj8QdSQS3lkffbiaaCTGxx+v5913VtHZ2UU4HO05gQuigE6rwWTSk5Fh
    x2DQEo3G+eijdeKZZ064MR4V32xr3ejJzj3jJIFSudHtbvr1ww//zfzJR+u57PI5XHX1uSQS
    SXy+wAk9kyu+dQ6dHj+v/GcpiorlwQdufjQzM6cJeLvv5pMN6D4dNKj4i7lzJ8392wvv86O7
    r2TeuVPJyrJjMOjpztU6uuefrwnnHJAXTSbjdHYGWLWqnPvvfY45c05j8OCBq0Mh+UOzuRc3
    AHEMwWC5b0u5/v4zTh/Yct750/+wePFKfTjdWPxAQ33hK/d87G24DvwI6HR6wpE4yaTMWXNO
    UwcMKPqn1x3/pd4iyL+ep/bRu+UGus7cubNq1ptvrkgXCPSuzSMIAoqisPjNFZxzzsSzR48a
    vsjtXvdcTs7kXuPk1dfMH7Vu3U5zdXWqz66iKGTnOBk3bjAffLCGmpr9CILAokVnHpGT27bt
    /NUD9z9n3rGj9rjnn5LFkw8yAPqXk2lTw5lI+GYt+WAVGRlWfvmrmyktLcJuN2M269HptOl3
    USURTxCOxPB3hahvaOXBB17ggyVrxDOnj78xHhUXk6pmPbmcfOhv5mWfbmDqlFFkOG3Iispx
    64GmH9LIESV4vQGWLd2EovQ/J+fNmzz3mWfe4t57n0+vFxWdToPZbMBg0GO1mLBYzVgshnRx
    jhatpDmIGiqxeKInfSYYDBMORQkFw7jdnZSXxwABrU5CVVRuuWVhv3Hy7be/0IdC0ZQBmC42
    0ht0mIyp3EmjyYDVasJk0qPXazHotD3vCKRym6OxGNFogmAwQigYIRqNEY3FCQVT+3wkEkUU
    RURR5OzZx8dJzbFBsgPwT6yvb5y8ZMlqTj99OPPmTaZ3JaQUpkwezTnnTGT5Z5upqto3cNy4
    kWfu27uupqRscq9AWFVr/jNs+PCzHvzlzdc88sg/eOy3/8Zk0hMMRigtzefCRWfy0ksfoyoq
    V151TkhvMP5RTdIiShoR/Lfu3LHnl/fe+1dzXV0Ld//4W3z7+gvSQFF7baOIRRMk4kkRVE0f
    0sUA6g31dc0T1q6tBAT21TbjcjkYWJxLRoYNq9nUk0MZDIfxegJ4PH7q6z2IosjatTuoqW0a
    PXzYoHOTHvkdUj0QTsKQ5n65ptz8wfurEUSB9et2UrHt1730TASi0Sgq8MnH6zjnnDPMl1xy
    nrkvZyMIpahqhV8QTFuGDh04FyAzw05RUT6xWAh3awe+rhD+riAdnV34/aFULmEsTiSaIJlM
    HLKejAYdRoMei9WMzWbCYbdgd1hxOKw4nVZycrIwGvUoqsqQIQOQRM0aswW/IIzu1XlZLOMJ
    BLYA2etzcpx+u92cVbdvP8lkAo1GQzQaxecLEgiECYWi+Lz+1NzCUaLRBLFY7JAm06IkYjCk
    2mNYLUbsNgs2uwWzxYDDYcVqNSNJAg31LYiiQE5uRkRVxXc0OlU2W/oyv0s1yHL4yiVLVgut
    rZ4Typv8ei+liNvt4cMP1zB69OCFgqC+1BvvYDcnhw8fftYDD9x0zaO//ie7d6VqDpqb2rn9
    9ifw+QLE4wmKBuRwxbeOzMkH7n/OXF5e3SvzF04SJ1NV+soF9XXNE9au2cn886cyZ85E5GSS
    QDBCR0eqg4Kabimk0+kwmfTk5WVSNCCfRRfO4D8vf0pNbdOokSOGjvb5tq50OMZxMjm55P1V
    fOvKc/jJT67Baum9vvDBUJQ//vE1/vmPD/udk0OGFs+VJJHMTDsTThvK8GEDKCjMJj8/i8xM
    O3a7GZ1Oh1abSp+RJBFREA8WEe6pdk4kZRIJmWgkis8XoLm5g5qaJnbvrmfzpt243R4GDynq
    N046nJasSCTGsOEljBtbRmlpPvkFWeTmZuJ0WjGbjWjT6U7dcxMQemrmFFVNhcPl9NziCYKB
    EB2dfurrW6iubmRHRS1bt1UjCALZx8nJY3wJNQLEFm7aVJnZ2trJd793MTabjWPPm/x674JW
    p+eii2bwySfrWL9+hzB+/LB5sZj6L3ot7B2Og/VnZ844o/nPhTm3fb5yi93jDRCLxfjow3Xc
    /8DfqKio5sYbLmDY0JIVgWDyM6vFIkHy1l27dv3mnnuetbS0dvLLX97EwkUzj9Jr88RGurTf
    mkjEioGqPjqtjoHYz5YtW69raHBTUJTNeedNYdrU0RSX5JOZaUenO9BfLSX110V9fQvr1+3k
    o4/XUVvTzLJlG0zDhpY8rtVqGoAvTxIplXg6PD9y+ECyshxfq25x7KdGO2aTkR079xGPJVVB
    EKN9PSFBGIOq1q/Ny8uM6HQa45Ilq9m6dS81extxuz10dHSl82QO7SUmSSKiJKSViFMV+nJS
    4WANd0hV8LsyU1XQA4vzcbd2IIoi+QWuiKKK6yRxVJ/My2qdgKpWN9tslmBuritr9+56nnvu
    HerrWmlsdON2e/D5gkfQmf+qEyXdf+0rvkxREjGbDWRnZ5CXn0lZWSFr1mwnM9NOXp4rKQhi
    uy3jtD5+eqGhbW0ds1at2tovi3/tmu10dnqnZGZmlwCVvc3JZwpzbvv88y12rzdILBbl5ZeX
    4vUEMBi0XH7ZHIYNLT6Uk5W7fnPffc9ZesuYPLmcVPUQPmfZ8g269nYfkWCEBx98gYb6Fjze
    AMFgmGg65CsIAnqDHpvVhMNupmhgHqqq0tHZxbJlG8zDhpXeIAjCWlKaxSeNk6oKZ5wxApPJ
    1N3FoleG0Wjk9DOG8dJLH50ETmZEo2J4NwAAHvBJREFUDAadccaMcTzy6G0YDEYOFOQckBNW
    FKWnTyqoqZ6TaXpIUkoy0yJ1R38kBgyAMWPoieY8/NDfefXVZeTn9x8ns7Mysjydfu6++0pm
    zz4jXWx0cEea7rmph4gPpBtCISGi1UrpcH66UClXpGwwTJ48HpCprWng299+CFlWjpuTx2hQ
    hjMUJXn2hg07cThtnHHGCPpG4F5h5MhSBg7MY+PGXXz72thUu0McBFT3zvVHAbXNQY96b2nx
    wKWlNxbNB1UTj8dPGzmidPrGjbuYPWscixbNqBIl7SNWizsE5htra2t/+8D9z1taWj38+tFb
    OevsSUfps3niQ5YVonHZDELx8Z+sVaCJVK76wWksImABgnWqGvPm5bmyZ8wcR/WeRl7618e8
    /tpysrIcWCxGrDYzBr2WaCxBIBAi4I/Q0ekjFk3gcFqZMXM8hYU5qKoS0BnEmmh0G3q9+hXb
    Xw/YgMIT0AL+6txqD/urbjB85zuLOP/8GfRu6o7E0mXruOOOP/QsUq93Cw6H45DTch8YJrVZ
    WY54ZqbduHJlOVlZDooKsxkzZhBZWU6ysp1kZtpS3jiLAb3BcKAo56B1kEzKRMIx/IEQ/q4Q
    Pl+AtnYf7W1eGhrb+OKLctrcHnJyM8jOzkiIotTct9uASXA6Mykqyqa8vIq/Pvs2mS47BflZ
    nDZhKJlZDlxZDpwOCzarGbPZkKpW1+t6coEUWU5VtkcTRCMxwpEIAX8Yj9dPe7uP9jYPdXWt
    bFi/C683wOjRpTgdNk88mgj2/b4tnltb22RvbGg7TMat972UAg0NrdTX7Te7XNlje8+g/Aon
    Sw/m5KDpmzbtpqQkl4ULzzyMkw/+8gXL5s1VfWpM9h8nozZVTZyel5vJ5MkjWbZ8Mx0dPuLx
    BJIk9hxyRFHsCcunwvQKWp2G7CwnkyaNSHFSkSfq9ZI9Gt3WdrI5+cenUp7E3t2/BPyBILKs
    dlsz/crJjAybsbPTm/LABcO0tHTQ1NROZ6ePzo4uurqCBIKRdLQjzsFy6oIgoNfr0Go1WKwm
    nA4LmS47mZl2CguyyS/Iwmw20ukJ4HRa+5WTBYUutm2rpssXIByO0dnpobmpnZbWDrweP52d
    ByI58XiSaDR2yOlb0kjodTr0Bh12m5nMTBsZmTZysjMoLMohOzsTfyBCOBwjPz/zuDl5rAal
    tsvnzdhX28rAgTnk5Dj7yKAEm93CkKFFbK+oIRAI2HNy8nrNL9/9sga9W5OyklguiSwHEY2o
    K1p04ZxLzzvvzDFanbQfhMWQtQVMhaFQ54+ffvp1S2VlHQ89fAtnnzP5oNNBX4RZVBKJhCCJ
    gvbY4VgHdK8FsRT0zhSsREDUgKKAVgGTXxD40cKFs8+eMfP02U2NrWV79zZZKiqqee215eTm
    ZmA2G/n0041MmzYanzdIq9vDFVecxbhxQxgyZECwsCBnr91uWSnLLI3GlKjdbikGMg5oDijp
    ZSZ5gVpV3ZaGZvGJQtMIwlDoLptVMroPbBqtFq1OT6pPY28NDVqdtnv9CKAMNBr1A0DjBFkA
    tYo+CfdLGllWhEQiyRmnD+ehh79DXp4Ls9mQLmARD/LNfV0Ri3DEaICqJgiFonR0+PjVr/7O
    zp213SoSUt+CUlZFAURJwOGw8Lvf3c74CcMxmw2YTPr0mhGOcN9Hnsfhf5aJxWKEQlFqapr5
    6U+e7vbemtU+z7cDCLtqaprFaCTWa8bB1/EsFIpSX+/WT5ykjgde6WtOLlw0+9L586cfkZN/
    +tPrlnVrd/Z6zujJ5aRw9+Gc3MsbbywnJzeTvNwMvvxyO9PPHJvqp9rq4bLL5jB23ODDORlV
    43a78aRyUhBI9/BUev2ZdOfhnQxOqqrKjh37+PGP/0h9fSutrR5C4SiJeKKnMKVbBUfSpO6z
    2/BKJpXD1GcAtFoJs9mIy+WguCSPysp9CAj9ykmNRiKZlHnppY95663PqalpxucLEonEegp0
    BOGA8o2kEXvm2d1D+qtqQaIootfrsFpNFA3IwWTS4/cHKSzKOm5OHmvIW/J1+YW2Ng8TJgzF
    bDb2FSbQaDQU5mexalUFHq9fzS8Y2OuWm8V5aB5LIr6lUUT4w/YdYWnChBIZtQ2E/YDhW6tW
    bR76ycdrufLKuSxYcGafGpOQWgTxWALxmJdrHWA0gjwbdlwVCoWnuN3tOZ2eLhLxVEjYbDaQ
    l5uDzWZp1WjVCo1GLzocjs1OuzUzO9tp2bBhJyaTnnvuuY7m5g5Wr67g3HMnk5Obwb33PEsi
    HmfatDFkZ7n8IFWqSrJAURM3C0L8J42NbQWdHb68YCgiSKKARqvB6XSQm+tyWyxda8HyHzCu
    gL0nCJXwIFlOfhgMhGyKoiCKoj4YCKXOqsEwXT5fT3Pv3tlcRbqvHw5HBJ+n44F4PPEdnd6b
    a7VZIpKkmQvs6N3NsgkIz6yp2W9qb/dx7bXzGDJkEIoSIxKJ4fV00OnxEwyGUv3IQhHC0TjR
    aIxkQukxMQUh1dzcYNBhsRqxWk1YLGYcDktP/k1x8QBmzBjH6lXb2Ldvv3nokLIZfn9Fuc02
    pq+WeH5HZ4epek8jw4eXMGPmaRgMJhQlTjSaIBL20+VPzSsaiRGLp9ojJRNyz2unAqIIWq0G
    vV6LyWTAbO6enxGdTkdGhpOMDAennzGcTz/ZQENji3H8+Bxn3xuUCbGx0Y2sqEiS0OefpigK
    zfvbEQT15HJy9eahH3+0Ll0t2/ffcm9yss3dkePxdpFIpDhpMhrIzc0+Kic3bqzEYjFyzz3f
    pnpPA198sY2ZM8dRVlbEL37xLInEN3Cy05cXDkcFsZuTDjs5Of3HyQkThpKbm9nrHsq2Ng9b
    t1YTDvU/Jzs6ukgkkqxatY3sbCdDhw0gJyeDnOwMsrId2G0WLBYjekMql1Ls9qCrkEjIxKMx
    ItEY4Ugqx9DjCdDWnhJbaG3tZN26HUQjcTQaibp+5GTN3iYEQWDbthpsNlO6d/CAdETJSWZG
    qtuA2WJI5VKmw9sAipwqFItGY4QjcSLhGF1dQTo6fLjdXtzuTpqb2/B6AsiyQlNT23Fz8hgN
    Sp0hnkhK4UgUk8mIKGrotbTGw885mMxGEvGUa/rYpfOOfWh1E6jdt4b8fElubmzgtX/Aj+7L
    H+D3N1//6iufSjk5mVx77Tw0Gm0fzvuQN+V4XuiBitL5eN2+ffOXLltvXrmynPY2LymYCMhy
    SlZPr9eSlWUflJfnGpRfkE1mpp3Oji4++2wTbW1e7rjzcqZNH8dPf/I0sqywceMunvj9Hdz1
    wyt46sk32LKlmlmzJuTn5GRc5fUGaGpuo6XFQ3ubh2g0ji4t99QtV+VyOUpnzT6t9Ny5ky4s
    KSn5SBANdwP1x//lSGJtbY3lwQeeN3d2pDRxvd4Aqqry5B9e4+9/X3JiVYuHW5QEAmGSCZnn
    /voOb7y+3C4Igj0j08aDv7xFGjx4SB8sUEGE6LCKimqNKAqEIzFeeGExlZX72FfbQkeHj0Ag
    3cw2luiRGBPS0lwHb7rdvRoVVUWv02A0GrCYjTicVopL8hg5spRwOIogCGzetFs699ypwxVZ
    Evtioad64cXPqKvbn1Pf4Gbu3AGsW7eDPXsaqN7TRH19Kz6fP2VMRtMqOYkkyaTckwt6sBdN
    0ojotBr0ej1Gow6jSY/NZmZAUS5lZQUMGTqQLJedYCjCnqpG8/jxI+eoauVaQRjRhy+vYAiG
    wicgZXfsqAiH+6ce7us4+fqry6RQKNJr/Tb7mpP79u2bv3z5BvPnn2+lscFNMBjuqRDW67U4
    nVayshyD8vIyB+XlZ5GRYcfT2cWKFZvZU93It644C4fDxurVFSiKwqovtjFx4ijmzp3I668t
    o3zrXmbNHJ+fnZ1xlc8XoKm5ndbWFCc9ngCxWDytmiRhNhsoLMopnTVrQuk5Z0/sdU56Ov0I
    ooDPG0QQBG6+ZSELFsykd2sgJD5duobbv/c4L7zwHm8tXmlXUe1Op7XPObm9oloTicSYf94U
    brppIXl53QUrBgShOyfySNKsHCXK052SoBIOR/F6A7jdPl761wcsXrySTZuqpLn9xMm6ulas
    VhM/+vGVTJs6hoxMK3a75aDuAsJRolVwuAztgR9ZTuL3h/B6A+zeXc+vH/0nTU3tx83JYzQo
    FUUUBFWSpLTMUt8aVcmknOoHJYl96g08eJSWTP1KWKR6TkVF9fBNm6q45tp5FJcU9I8xedwj
    dlPV7t2X3nXXHwiGosybN4nxV8+lsDAbm9VEMilTu6+FbVv3UlvbzN69zWzcuJtwJIbVamTc
    uCHcc+/1TJs2lmXLNrB6dQUlJbmsWbOdFSu2cPXV5zG4rIjX31jOx5+sw+8PYzTqcDis5Odn
    MXbMRMZPGEJJcS5arQZ/IMz+5na2lO/h5X9/zNtvrTA/8fu7Lh09esQu4P4TOXBEonF19+4G
    zGYDgwYVkF/gYtToUpQT0qT9mqOi4GL48IE9YZGammba2n1EonG1bw48MZffH5xZUbGXaDTB
    X599h1gsgcGgI8tlJyvbyfjxQ8jLc+FwWlInVLMBrTalMNPNkWQiSSKeIBJNKSt4Ortoae2k
    sdFNe5uPlSs28/77q9FpNcTjSXZW1uLz+WfabFkuoK3352XVQuvkLVt2EQ5FWfrpRj7+aD2h
    UBSdTkNGhg1nho2S0oJUfqg9VbltMmjRGXRIaXeUiko8GiMaSxAOx/D5grR3eGlze3G3etm9
    q5533omi12vR63UkEzKbNu0WFl04a3IsJpmAcF86DTUaTb946dLnHTQaTb9R5kic3F5RPXzT
    pt19FupW+4CTP/zhk1RVNaIoSjpkeOCBBYMR2tu72LOn8Qh+OAFJI/HBB+v48MN1hEIRSkry
    2LRpNzff9AiqmgqhVu6so3LnviN68rpDlN3D4/HT0OBm3dodvLV4hfnxJ+7sNU5WVTXS1uY9
    JCwajcYJhSK9HMmRiEZTgk0tLZ3s39+JqqpkZzv7nJPbKvai1WpYcME0Jk4cTyofVkaRkwTD
    EUKhCJG0GEQyLbOYTCR6bC69Toek1aLTHYh4WC1GdHpdWrLRQkFBEV5fFx98sIYdO2v6jZNd
    XSHGjR/MJZfMxm53pg8BMvFYnGAomtbkjpFIyCSTKU9rdzqDKIno9XokrYRep8Vg0GM2p6I5
    Go2UbtaeSWlpAR9/vI7a2pbj5uQxEigaNZkMssNupaOzi1gsntbs7gtjT6azswujUYfJZDhJ
    RlybVlXi5635cpsYi8WZPGlk+hQg99HnqYckCR/nNdabTMawomAqKszizjuvwGZzIMspVQCN
    RmLI0EHMmzcdVU0QicTSnq4ITqeFvDwXgqBly+YdPPbbl9BoNHz/9kt46o9v8rvHXiY7y8Hk
    KeOZPGUULS0deDxBLFYDrkwHRqOhpzG1qiZIJJNIosS4cSOYMXM8u3fV0by/E6NRH1Hk5JYT
    D66kxsIF07jzh1f126Gj+9Of+sN/eOXV5fSdzSBOaW1tL95b3YTRqGPChKFMnTqGseMGM3BA
    Di6XHV1a2SGVyvNNBWIHqh1lWSaRSNDlC9DY1M72HbV88Xk5G9ZXsre6ieam1pyMMa7BGzcu
    bzvjjLN6eV5hfSTiH71tWw2KomAw6BkxspjTTx/OqJGlFBRmkZ3txGQyImlSDYZTHoavehc4
    iAupdh/pql86Ortwuz3U1DSzeXMV5eV7CIWi7NhRQ0eHd0x+/kAX0NCHCyTqynSk77s/DEoB
    u90CCCeNk19+uU3s6gr1eiGOqoIkHqyS1HucVBFMiqIe1aP6dekKqqri9fqRZYX8fBe3f+9i
    nv7TYhob21JtW4Rj12uHVDGPrKi9yklREtO5jQfm89xf32Xx4pW9HfHG40l9J90HC0VRESWx
    zzlZvaeRzEwbLpedDRvK2bevierqJpoa3Xg8gVQKTTBMOBIjEU8eUg3dE+2QRAyGdJTDmk4L
    yrCRn++iuDiP4uJ8rGYjWdkO9lY39hsnZVlhSFkhbW0eNm6spKamiZqaZtytqY4YgWCYUDBM
    LJZI5Usqas9jTR0gRDRaCZNRj8lsxGFPzc2V7WRAYTbFJfkMHJhP2aACNFrpuDl5jAallHQ6
    HZH8fBc1e5tobm4jJ8fVBxu5QDQaY091I06nFZvVKgS6IieBlHExHAkUVlc3YbNbKCzKTr+e
    fRXOkXtOd8c/DMsHFg948Tu3Lrr9oV+9yJ+eeROtRseeqnoCwRDZ2RkMKMomO9tJbp6L3NwM
    slwOcnMziMcTlG/Zw5q1Fbz26jK83gB33nU5F11yVpeokey/+c2/ufPOJ7n8irOYNm0Mefmu
    9L9LUrevhf0tHbS2dtLm9tLQ0EqruxOr1czwYQNRVaioqOH/fnYtpSVFLwa8yse95gpS1LSK
    Rn8eOsTupOw+HOGixka30esNoNNpGDq0iNFjBjF4cCF2uwWjUfeVV/ib7kftuXdJStV6mi1m
    CgpSikAt+9vYtrWaTo+f2n0tztGjR0ylT9pAJcZ3dviG7dxZi0ajYfqMsVx95VyGDivGZrPQ
    3SD6yPNSjzJPEVGU0rrlBqw2OyUl+YwaVcq4cYN5/71VvPLKMurrW2lqdOfl5xWcCbzcd8/O
    EB44MFeVJFHoj7C3JInk52eqifjJ6AOb4uTevc19FuLXaCV0Wk16bSi9xsmbb154+333/pVo
    tLtF1eESil/1XB5sUCqKgt1h5qabL+jh5GO/fZn2di+iKB3RQ90d4Tjy56UUyW688YJe5+RX
    R3V1A1VVaq8aeiogCsKB3MR+GSlOdnWlCqzu+cWztHd0EQxGkGUl3c7pyF7hI42AP4zKwc8o
    1UkhpdCkJyPDRmdHF4mE3C+crNy5D0kS+XLNdtZvqMTt9qYFR5R0oZFw0Py+/oo+b5BUYY56
    oEhJENFoRDIybKkOIUKqa8TxcPIYDUptp8lk+mLkqJLhX3xRzo9/9EdMfVSYk0zKbC2vZsGC
    adjt1qqOtvj+yu2V4ojRI/rRatCJsVgMj9dPMpHkk0/WsWVLVZ95wmRFYXdVfarv43F/ihoF
    y1Pz5k2/4LPPNg/8z8tLKS3Np6goh1xzJu42L1W76wkGw8TiSVBTuUJanYZoNEE4nbc2YkQx
    P/2/a5g7d/IHkqh7/JJLzrrb6bSd/8enXuOpJ1/j+efexWROKQ4kEzKxWAJVVdDqtFgsJlwu
    O7m5mciywqrVFdTUNDNl8kjOO2/qXgHpSUFMRnsDXgDvvb+abdtr6O9RX9fyX5lxJ7CNto8e
    XdZ2112XZ3/00Tpee+0zXnv9MzIzbAwcmEt+vovsnAxyczNw2K2YzEbMJj0Ggx5Jk+6NqpJW
    7oji7wrhD0bw+0O0t3lpa/PS3NxOc3M7vq4gqDBwQA6XXTaH0aPL4olkMs6JiNMf3QDBbDWp
    8+dPYcn7a/jog7V8sbKcAQNzGTpkAAWFORQUZJKXm4ndYUkbiVo0GgmNRuTgnKCUhrdCPJ4g
    HI7SmfZMtrR00tTopnbffhob3HR1hTBbjMycOR5XllMQJbmPD6jaZYNKC+52OizmtnZfn1Z6
    q6pKTm4GpaWFYVGUlvW/QXmAk4JArxuVgpDKj37r7ZVYLWZkRe5FTk67YMVnmwauXFnOkCFF
    DBiQStVxt3nwpAUDurpChMOxHsm+bgNFr9cxclQJN954wWGcfPrp19m5o45kMtmz6aeMEgGj
    UY/NZsFmN+Ny2cnJyUCWZRob2qiqqmf69DHMnz+lVznZrXmvqsIRvaK96KBETXtZDzG8ZaXP
    OXnnnSlOVlc39khBquoBOcWj3/FXdpR0JXi3kdbds1JR5B4jtaQ4jxkzx/cLJ+fNn8wHS9aw
    f38HiUTyoLUkHvNzPDCnA/NV0wVJLS2diJJIhtPK5MmjjouTx2hQqgqYdsyefbqyaeNuMRaN
    Ewz03YF45Mhizj7nDCRJt1IUuzrA2s+glHSqik5RVQKBMH986nX6OqSkqEpPJ//jedkFoQRV
    VfdarfEnf3jXtx677LI52lGjymrsdnOlKApKLJbQBgORIV3+YF5Hh0/wePzGNrdX2N/Sjtfr
    x26zMG78ECaeMbI9Lz/7KUUR/yaKams8LuyeM2fy/w0bNvDq1au2ZlVsryUai2Exm8jPyyA3
    z6W6XPZIptOh2hy2FrvduEev1yZUFdHvjwzfvn1PWU5OZsJutz/x4G/XV//qFzef6DclmE0G
    cezYwXg6u/p0HR5tuDIdZGbaMZsMIih9sDBsi7Ozc+puu+2Smy+6aPb5e6obMvZWN+p3V9VR
    X+9m06bd+HwhEuk8oK9C4kiGRzd8tFoJh9NKdraTyVNGMmTIAEYML46XlQ3ozM7OXC8gvR4M
    RN/tG3vZuM7pyLr6Rz+65raLL5o9fdOmXbbyrVXU17eyatVW/P5Qj/dXlMR0TpMeve7Q6kVU
    SCSTJNJ91yKR+CFtUCwWI3l5LqZMGc24cUMYO64sOmTwwBqT2fx8LJZ8p48NyobSQQNa582f
    POhEZAf/Ow+9wqhRJRQV5u4LhRJ7+9+gTHFSVhQkSeqTubpbPTz2m3/38Lc3OXnXXd967NLL
    ztKOGjXoME76/cG8jk6f4PUEjG1t3ZwMYLOaGTtu8DdycvvOfcSiMSwWM/m5GeTkZaqZmfZI
    ZoZdtdkP5+SOHdVl2dkZvc7JMWPK8Hr9fd7C6mgHHqfT1q+crNnbpK/aU09Dg5uO9i483gCR
    cDSVTiDLqTz7I4S8u421lNqMhMGo6y7MoqAgi8FlhQw/iZzcum0PjQ1u9u/vwOsLEIumQtyH
    HnaErz1eCGkPskaS0Go12KwmcvMyGTgwlzFjBzN27PFzUjj2xVFVmEyGXvb5/KWK3LehHEEA
    k9kkywnN9f6uzJWBQJgRo0f044vQZIjH3UvWrtt6lru1s19fRlESmTRxFHl5OT/QaMY9fWz3
    3QV02CH8D1D3JxLiE1qtow5cKnKTFI9HinRGTQaIOpTEGYhyKShaWVZ0oijIgqDZi6J8WrWt
    fXvJSJdaviHIaZMtNDQIUkmJZrQgKOcoSnKQqqqSKApxATGBKtUiareAGklEZI+oMzRK0gAZ
    2oREwlus1ao/UhWlMBpWrtvvDneVlU0/wWdTMVxOJhf7fAHrwafh/h6iKGCzmwMajeYSSRq/
    q3fXXz3gp61ZkbLypSGCIJZBYlYymcwOBsOTAv6gqcsf1oRDkcxAIKLxB0NE0snZ3Qo6ADpd
    qigllWRuxmI1Jc1mfafNZk7abNaExWLYotXq9oFmNahVIT/VZlsk6fcp2J2T+uC9qgeCREKy
    xWiWRqAk5slqsqDLFzrd4+3K8Xr8Nnebz9rR4aWrK0gwECYUjBCJJYhH4wf6romg0+sxGlIJ
    9KmWSOZUw3eXPeF02jpdLkfCZrdsMhgMe0BaGY8lduj05qZoNIjROKEP2aECO+8J+P23RiJR
    oW8Poip6g041GU3PanXjH+1vo+EUJ09x8v8ZTgYjmkAglCpiicVJxA6kTmh1OvQGLeZuTlpM
    SdP/OCfb2nzW9k4f/q4AwUCq8Ch6GCcFdHodBr021efXYsRqMZGd5cTVy5wUjg+UVZkgG/sj
    ATwYjNHeGukoHTw12t8vgaq2Av6JEJ9yqMBbv3w6IBOPKyv1+gnbjv3ed6MockYslozIshxZ
    uybI3HNnsKdyNdk5Rix2gUPz0G3ceftW4alnLlFhK/FoEr83Rlb+AaC1tm7G6RTQ6QRgHHfe
    vlh46plxKvh7fkeOQzCg0uyOMHLkdD795AumTLUgyhiQFJMg4jGZT++FZ7NLC0I2qEL/FuMc
    /pxi8YSqKkqb0Tgh0Ref0FC/gaxsDQZD6pWtKI9Kw0dZsrVanQ4UczKeHC9IguvQ56keHBtW
    QFC7YwxyUu3Q6LRbgVAikUg0NYU6Skr0MVBJxsHTmSQnf2Kff3M+z2asNgFREhCE8ajKTheC
    1qoo8VJRVLsr4PSgGKOR2JCkTKmiKBpBSOV9pUKJEpJEu9Gk2wRiBIQYCElVpU0QNOWghNra
    Eu3Z2URRZYKBJFb7xH7iR40Boq7+4IaiyAQD8Q6747RTnDzFyVOcPMXJk8LJ4zAoQ6DuI56I
    o/TxgUcQIB5T8XriDCyZchJc9W5SclxRTkaVeSq3AfT64/OkyHIFiYSMwTD+G3+3snINeh3I
    SQj6YcLEqUf93S0b1mCxgaSBWBxGjJj6jdePhraAqGIwntZL301KXS4ej/eJ2sNxBSj60OPV
    PV7+92fMmmXG4dRiMEAioSAnU7DRajlo8xMOA7qqpNZTMqmi0ahIGpFoVCUQSJCXN/kkfnNr
    qK6CoiI9Gp1APK6g0Yhotd0FcBpqakIal8uoNxg06PViz3xCoSThYFjOzrNFIQEkUVGIx0AU
    UuGdri6FpUvaufKG+f38/lYhJyMkkv2zPuWkisV62ilOnuLkKU6e4uRJ4aTAqXFqnBr/vxpb
    ytcc6sBRYMLpU/+fm8emDWswWcBiAp1BQKfTodGkNgRBSG3qiQTIyTjhsEIoAvE4hL0weebU
    Uwvh1Dg1To1TnDzFyVPj1Dg1To1T49Q4NU6NU+PUODVOjVPj1Dg1To1T49T4/9qDAwEAAAAA
    Qf7WIyxQAQAAnACLFTY+QQXw6AAAAABJRU5ErkJggg==
  }
  set pieceImageData(Merida2,60) {
    iVBORw0KGgoAAAANSUhEUgAAAtAAAAA7CAYAAABSS730AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBCh0IvsW9DwAAIABJREFUeNrsnXd8VNX2
    t59zzvSZJJNeIRAg1AAJXUBAelMsCCpWUK/dK+pVf9fernjtFRECKlUFpfdeQ0LoLRBIQhLS
    22SSaWe/f0yIRMB2KcE36/OJwcxkcvY5ez/7u9deey1osAZrsAZrsAZrsAZrsAZrsAZrsAZr
    sAZrsAZrsAZrsAZrsAZrsAZrsAZrsAZrsAZrsCtsUsMtaLAGa7A/akLkAM6a/9MhSRENN6XB
    GqzBGqyBz//fmabhFvzRjnms5l8qYESSGjXclAveq9Pe1ZkU1nAz/jbPNAOoAsIlKDR6fxpU
    JcQRARYkKfJv2ObjeH0Moua7G0mKbegMDXxu4HODNfC5gc+XzwMtxMEauGkAD6AAbkBCktpe
    DYDuAFIbkFdBQSEEAKYLrvCEqAayADvQHkmS/uYD+MwEVg200nn/fdgJhhpYN6/H136opk/+
    gQEjtUOI/Rf9vaAgSa2vQNuPADLgqgGRXOO5aHbWe0qBU4C+DTAB1N7eV+RNIE2H0AMej8Oj
    0QT9Cm6emi8BaAEVSWp5FfVpAWR2ADqDtBca7fy7jmPv+K06j0/FXSNIm9f362/gcwOfG/jc
    wOfLeg2ay9PQfYBeA6IRKP3B7QuaStCuATVDiIMu7yqpcT1+XO7hHk/Va7JEiiT7LwTtcgjf
    LcQBz68XAEJkAslAVABo3LCvXIjDSFKrvymcdwHFQEgCaPvCia7eV7RJoF0PebuE2IUkJdTX
    JnQBpbd3UXc+UwFPOYhZ3hlXGgFKrBdmF3z/UWAxYALpdlB8L/x+DyBtAnZe3ue2H9BZQBkE
    NPFeiGY9hO8V4pj4ZVL1AwrHOp0lLyQnH4hLST6MKqBTp5adu3ZtP0anU971eJp8VEPimgkv
    SoK89uDqC6oC+pOgrhRiv02S2l0Ffboa2B9fXV01Oy+vuGVwsPWEyVR2D7Dx7zd+dwOVgD4A
    9B1B7VgzAe8Gx26wFwtx6IoIiAY+N/C5gc9/gs87D8TtSjmMp4HPfw8BLcQBQImD6ifKy0sG
    52QXRhYXl0m+vmYio0Ky/f0D1oH+I8hPrt+PzKPu23tYWbliR9d+/bt2jWsf+7hBX/Up6CYL
    sT+/7grVDoR1g8pPwVkB5k/A8lNl5S5hNifw9zN/oKKbx1OcuHdvWuudSQdQBXTp0mpMx46t
    DimK4V6PR95Rf69fHZCbm/NW+vHs8/Rf8PMz07p1kyyNVr/Y+3Cd9x0/durG3Nwifr3gFQLC
    wwNp1jxqAUxZDKqv2+V46dChk43Kyio53wI5plkk4eERz11OQHshSguwvZ+fXzggMyPXoNVq
    aNI0ItvPr3ouGD4QYs8pSeoAHG1WZS9786uvfoiZNm0xZWWVAPj4mLjvvuERDz54y5Mmc+VS
    4Kh3p0kTCcefKi8vGZNxMieyqtpB48bh1WFhwatB+5THszdNUdrXYzgfBvSA6f6NGza3fOWV
    r3n6mdub3nTT4F5CpG4EI9AYSTL9DcTVQcCjB+19qqfy3pzcjJanc4t8VVUlLCywPCIy5IhG
    Y0kE7TTh3uWQNAkNfL7a+byzhs+dG/j8d+NzYuISSkoqEIC/1Yd77hnawOerVUDXhG10cLtt
    0zdv3NVx+owl7NlzHIfDhVaroXXr6Mi77h42buDA7p20Wu29+aeTdoSEda2vg5jTecVMnbaY
    2XPW0KdffMiYWwe83LVru2GK4jcZ1G+FyPJACSB3c7nKEn/6aW3rktIKxo4Z3NHX1+3jcqnf
    1Sx//2bmaOl0OhJnzVzW+rPP5lNQUIIAgoP8ePiRm1vffvuQREXR3wQcrp/Xr1HXrU/mrTdm
    IMSZraFfrFPn1nz08ZOeAH8fIYQLIfZ6vvtuKXPnrq2NwAKQJAkhBGPGXMdLL09QIQnoLcpt
    xZ5J784kJflQnc+VJAlJghf+fTe333a96+KOvcwar1NpzU98AB8kKbbmtfLGbrdj8rJlm/tN
    /vJnTpzIQVFkEjq1jJw4cdxTcXGtEEI86/V6eO7Zti01ZvLkn4mJCefZZwchKzKzZ63iq68W
    0rpNk+ghQ3rfCbwIVguUfXzgwKGb3ntvJinJh3G7PTRuHGZ44IEbRoy8vo9FUQwTCnNTjweF
    x9cwwg7Yaq7TCgRcsd0o79amGgVpIaC9pqCwjBMncsnPLwXkINC3A6UC9mdc/eJZAEcM4P53
    Tnbmv777dolm+YokCgvLQAgCAv18Bw7q2uXuu4fHR0WFR6Lyxo7NG6q79epTr/k8Z+4a+vRN
    CLn11v4NfP4Vn7/4YgF5ecUIASEhVh56+Karis8g1Yk79ajqb/JZkeVfLbPU3+Tz2e8XNf+9
    0nye8tVCTp7MRVFkOnZsEfnUBfj81Vc/U1lZjaJ4PfVlZTamTFlI6zZNo4cM6XUOn99/bya7
    UtPwuD1ENQox3D/h+gY+1xcBLUQ+UNFIVW1Tf/5pfcc33piB0+kiPj6W6Cah5OYUkpJ8hH89
    +xl5E4tb33H70ESdTr533coNO/oNqo+A1mA2m9BoNDidLhYv3MKmDbuV224b0H3cuGFxYeHh
    kWD6GsxmVa2cNm/eytb/efsbKiuryT5V4P/UU3d8aDT65Hk8+1e43U3R6bIBK5IU8ncg9H0p
    yQdbf/DBXKKjQ3lq4lhk2SuwPvhgLjExka379uly767ZW/+VcNs19bIBLpebigo73bu3o2Wr
    aIRQsdsdrFyRhN1eTY2mlqFSAuRqh4vqaieDBnUhODQAgJzsQtauSaHa4UIIlC1b75VAkoQA
    u90LtkGDu2Iy6ZEkmSOHM9i+fT8ulxsQ2osz7o7ijZeTQsGSAH7XgzCAWAlihRAHi0HSgPxw
    UtKBfq++mogsQZ++HbFXOti+7QCvv/Y1H3305N2BQaGLgXVQ3Wbbtv04nS4mTBjJqBsHAWA2
    GXjqqY9JSjrM4MHX1Oz72QYUFuYNfeP1RHalHqFbtzb4+JjYufMwb7wxnYBA377X9u76DxDP
    C3HEDcIP/IdBwCCQqkFdCNIuIVLyvOcjLrdH0NHN5ar6qqiouIlAMZeVVaAoMuXlleTm5j4E
    nvsCAqy5er3xodTd29bHd+xxNUtoQHos+1Tus//3f19o1q/fDYjaeODyCjtTvvqZw4dPal5/
    /cFnmzSOKlOF+916KrJq+VxcXM78H9ezedMeZcyY6y7I53f+8w12u+P/Hz5/OI/SEm9/Bigo
    KOXDD68ePtsqq34ln8HjUamqcpzDZ4fDhc1WVdvWs9/vOA+fq6oc532/QFxRPr/2WiJ5p4u9
    1yVgzZoU7PZqPvzw13w+gM1WXef6ZVmmstJB0s5DDB7coy6f30hky+a9yLKMJMGB/em8+WYD
    n+uRB9oEFI05fCgt/v33Z2M2G3jzrQfp2zcBg0GPy+li06Y9vPb6ND76cB7NmjVq3b17xwc1
    OnfSWU69emQG2dfXgixLdOvRlj694/nmm2V8+ul8UlPTzM88e+frHTvG9QE1Jzllb5uPPpyL
    0WggPCKEmTNXEhYa4H//Azc+K8vKCZ2u+AT4q5DsESIdbwC/chWnnKlqvnHTbqrs1Tzy8E0M
    HupdAFmtFh595D22bt1Lnz4JsVKsJNXPZ/sLLEeM7Mm4ccMBlZzcQnbtOoqqqkiSFAJVUyHd
    AXQXqoper2XC+JF06uJl0/r1KWzatAehqgDde3Q3zAe7TpKkEFVVCQq28tTE24kIDwJkvvtu
    Cdu277vIrfABSptC+ZScnOzehw+d1Dkcblq0iLqjWfPoHZJkfhlcu9xuxw3Ll27BVlHJO+88
    wvARvXA6XUya9C3ffbuclJTDgSNGRnTxAhpUoQISivaXeUSjUbxiS6i/OGxwt03aud+YnHyI
    sWMG8PwLd6PT61i2bAvPPP0py5Zto3v3DiPMvvo3wZgAZS9mZJzoefRIhlYIaNkq+r7o6KhN
    oLu/rMxz4vL3As/4FSu2tP/s0x/xqB7KSivRajV8P28Nq1fvNCCEYdy4wX533T3yIQTrr25d
    lRbtclbc//XXC7Tr16fW7IjIdXZIhBBs3LCHzz79UfvyK/ffH98laB6QUV/5fKZPyrJEQUEp
    n3++gNTdaeZnnjmLz8l723z88TzKy+1IksSs/0/4fLZ4PiOwykor2bZ131XBZ694luqEWUiS
    hPAu+urwWVVFTX+uK7glSUJVxTl8rvmMOu8X4lJkWfgTfF62lbzTxWg0ypkbgCLJ7N6ddh4+
    /8YhS/VcPqfuOoKiyLXtVRSFvPwSli9v4HM9EdAVoUKody5ctEnOyyvh9dcfYNiwXjXPUaA3
    6BkwsBslpeU8/9wXzP9xLV06t7m2XQe/aOBk/Ru+ulP+/j4Oi49Rb6uwM2bsAHpcE8dnn/7A
    okWbeeqfH0qvvDJ+YNeubZmRuITy8kpee+0B2rZryuOPvc+0qYuJj4+9Lj6hzSKNpjpLUcwu
    aL4EpOMgb4Pjpd4Tt1djyhmh2iurkGQZs8VUO1Z9LEZkWcbhcHkHsFz/HXK/iAiBLMnIkkRm
    Zh6TJn1nMhkNQ0GgqoLklKNIEnVEx5mQjOSUo7z22rQwWZZGgYS9qprMzDysfhZk6Qy0ar5f
    xOlKiBKgorEQ7qkbN+7o9993Z3LwUAZCVYmIDNY++tjoXjff1P91rdbwiNNZpeSeLsLHx0zr
    Nk3QaLRoNHpat4rG5XKTX1AKSN4j+ugzu3Zty3ffrmTmtyswG/RIiszMmSuQZYlOnVohSUrN
    mJWU06eLcDpdtGkXg8nsA6i0bhWNr6+ZgsIy3G6nYjIZojye4tdWLN/U8/3355Ke7o1vbNYs
    Uvf007f37z+g21StTrnHZtubabFc1ng8yWg0YPX3QfUInA4PQoBGo8FiMqLRaTCZjQhVyEKl
    XguO3zf96OPpB2OWLt1eI0zOlQteMQqrViUx9rYBMXFxbUYD/62vfPb1NenLymy1IloIweZN
    ezmd+5H08sv3efk8fQmnc4trxaTL5SYxcenfns8X9Ok5nFcHn89jsiyRlZnHu5Nmmowm/VCv
    XlRJ2XUUWZbO+/6UXUd57fVpYbIsjwKosjvIysw77/sv6lM4m88bdvR7771ZHD6ciaqqhEcE
    ax959OY6fM7LKzrv5zid5+fzzO9W4nC4asexEAKjUXdePtfMyecsUAoLy8/h8wcfzuPkiRwE
    EBMToZv41G0NfL70ArrSZLOVhe3fd4KIiCB69mr/i0qp/S7Ro0ccjRqFcuRIJqVl5UFhYZHm
    erqK32C1WoqbNo0MzziZS3Z2Ps2bN+GNNx+keYsoPv98Pi+9OIVbbu3H1q376Ns3npHX98Jo
    NPLAP0bx0otTeOaZT4mODo9t1Cgktk3bGFq2bDykZWxTp5/VdweEJoJ2IeQWCXECaHIVpVYy
    OtvFNUedtYpZs1diNnuF86w5a/B4PLRrF4MkySeFdHWJDUkCg1GP7VQ+P83fcM5rJpOhBrqi
    ZhUvYzIayD6Vz5zZq34FT0FYWCCX9pFqgOq+aWnp/V59ZSolJRXccnM/jCY9a9ek8PZbMwgL
    9e/et2/3ew0Gc2azZpEt1q1LZc2aZIICrVTaq9m0aQ8Go55GjcIAURP4ppnW85oOo+68a0iT
    775dzsOP/NfryRGCceMGcW2fhCzVI76pcXdUN4uJwmQysGHDLnr2jMPHYmHdhlSKSypoEh2O
    TmfIFB73vXv2HOrx6qvTkCSJO8YNQghYtTKJV1+dSnBoQL+OHdoMqqhQv768T107tV/f7t06
    JbSLVrSydt6cFcaXXprC6NHXcd99NzgEOMwmfarq5guRp17F4hmgKirtaJZSXFzxO/1SoqLC
    zpEjGUqnTu2i6jOfo5uEh2dm5tVZACiKxLFjp3jppSnccks/tm8/UEcsybJMQUEJzz7zGY2j
    w/62fNbpNLjdnjoCS6tVrlo+n3m+BfklzJ614pyfX2hBeCwti7SjmX/o/ZeKz6+9No20tFO1
    XuDMjNP85+1v6vC5aUxki1pP+FmXdiE+j7tzSJOZ362gutpbSEWv13HHHQMuyGe7vfqcK2zc
    OKwOn197bRq5Zy02jxzO4LXXpjXw+dILaBW32y0cDidGox6jUX+exYBAr9dhMhmornbgdruF
    JEn1dBArpT4+1iNt2zUNT9pxkKNHsmjatDFms4V77x1BRkYu8+au5ZOPfwBEjXg2ASr9+3dm
    1qyVpKYcpbi4gp07DzJz1kr8fC20aBGl69+/c+/r+nfpERvbZIL3wIvuZ9hdJkQ10plFZv32
    ZM0dOKD7qK1b95iWLNnGju0HkSTvpDt8RE/69e2UKYQ0PeEqixcNCPDh1VfvPy9ozgjm5i0a
    1fbruHYxfPLZ06ie859DMpkMBAT4XMIrrtSAZ9SWzbtJT8/h8SdG8+STtyPLCh07rmPiUx+x
    evVOuVev+CZC6DaPHHlt/7Vrd/HhB3P48Yd1uNwecnIKGTKkG506tTolVHWN93Nj91l83I/8
    85+3fd4ytlH0pEmzcLvdTHz6NkaN6pthMpr/eehw+Z4avO+Nj29dNHhI98Ali7dw6OBJdDot
    mVl5NIkO44YbeiOhWSVrROzq1Tvl/PwSXn/9fsbdORyAJk3CefON6WzelEp8x9iRJpNuOt5k
    xJfLp7Xd4/YdrtHgZ7EodxuMuqdBYLYYsfr7f1Zl90yrrHTn+Fikkk7Del7d+hlVVFVXn3No
    9rzeM8SZible87l1mybhmzbuOe9iOCPjNJMn/4TdXn3erf2sU/kUFZf/vfm8eFtNTC9otRqG
    DutB36uUz39V/F4esfxbfN7D8eM5v4Rm1MwlxUXlrDmbzyN691+3NoVDhzKQZan2ugcM6Pyb
    fH733Vm43G4mTrwwnwcO7hq48KdNCCEQwrur2rJlo3P4fLZ49l6nwqlTBQ18vvQC2sdlNtvt
    oeGBHDmSSWZGLsHBQdRNiC5z+nQRuaeLiG0Ridlkkqrsrnq6rG9SLkkH93Xr2qbvjMSlbN6y
    h0GDu5NxMpevpvzE6lXJaDQKLpebkBArzZv/IqwCA6x0bN+c/XuP86/nxhEWFkRq6hGSkg5w
    8MBJkpMPM3PWSs2IEb2uGTOmf+dmzZrdDcZHYf+h+p5/1RsjaNpg9Q/Z/8gjN3fds+c4x9Ky
    AIiJieKRR27G3z/oROZx59Gqil0Yfep3mihZkvHmA1XR6fR07vx7RX5E7XO2+lvp1TP+dxeW
    Z/q+LF3sPVO3BO7AkhIbsizTqFEosuyNWY6KCkbRKJSV2nC5PGFOp3t+m7Yt2k9699EbE6ct
    YkfSQQqLyriuXydeeOFue0CA9XVw1wRo7wY6LjWb90/q2bPjZ0bjj1RXC7p0aYXF4vcutF3g
    65dUcw1FK319A798/vm7nnNUO5R161Px8zUzYEBn7rt3BHHtWvxsq/DM9PGT3igsKgcgukkY
    Z3K8RkYGodEq3lRMQrVqNJd7pvNB0VRkWfQmD7i7aDQaNBqlZrKTIo0mShWdvkSjMXL1m6ky
    PCJEaHVayVPl+M0+rtFoCA8PFKpKZb3mc5c2fb+dsRyHw1kbwy2AZs0bMfL6XpzKzGX+/I3n
    4ZjAaNTz4kv3EBBg/dvyee/e4xw/ll0z1oKvOj6rqvqHFnwXS5hf3LCOM3yuQAiV8+W1Li2t
    rMPndyY9emNi4mKSdx6ioKiUvn0SfpfPn3zyI/wen5/z8nnDhj34+ZnplNCSe+8dXofPRUXl
    F7jXgpISWwOfL62ANuTqdPo1fa6Nn7B86TamJS4hukkkQUFWzoRvlJeX8+23SykuLqf3tSMx
    my1783Ltp+sfhLKBY208btf1x4+fwuNR2bp1Hzk5+ezdk8aSxVto3745AwZ0YemSrVQ7XAT4
    +9aKK0lSiIoKQVVVfH3MDBrUi0GDulFaWk5KymHm/7ieDRt28+UX89mwIUX34D9uvG740J6z
    9Qb/D0H+1m7f5TGZEuohnA8BOh1UPFlcnBc3ZcpCMjNOM2RYDyQJVq9MZlriEp595s6u4Y18
    Hnba1U+Fa5dT0tZTSEsS+fnFHD+eyR+tfPU/eMzIzy/m4sZ0+AuoKo5tGY0sy6xalUT79rEY
    DFqWLd2G0+GieYtGGAz6I7Pm7Nl7z10JD3fs2GrvO5OaPrBu3c7wJ574gMrKaowGfbXbJVI0
    Wln13pZ4vNvWSln6iWxKSipwOJwcP55Dy5YtmTZ9AePvvekMUlTQrtDptI+UlVdaEYKXXxlP
    v35d84x6/VSPS3wqS+SCxt2uXQw/fL+WhQu3EBkZihCwcuVO3C4PLVo0QghNxunT9sucVkyn
    lxXlPper5P49u4/Gbd22F6NJz46kA7RqGT26c+dWbfWGwGmgm4w3v9NVbGJJy9jGj7eMbeST
    mnq0Nv3VuaJFEBMTSbu2zW2qKi2p13xOP4XqPTSFEAJZkRkypBv//OdtNG/elI8/noUQ4oIe
    yMioUHr36va35XPGybzatmdm5l1VfFaFYODAznRo37xGgF7KqUBmz95jrFqdfEn4rNfrcLnc
    dfqhJEs0bxF1Lp/fuUR8Lqu0qh4PL75473n53LZdDBqNUme8CCFQFIUWLaIa+HypBLQ30XWG
    ByxJgwd3v2vjxl26pUu2Exrsz3Mv3INOp0VVVb79ZikL5m/k2ms7MHr0dW5Jlhc0bmoqEuIg
    ktSmnkDoKOCJVFX7pJ9/XhX9+WcL0CgyJ0/ksnzZNm65pT9ff/08LVtG4+vnS7fubamqchJY
    u1Co8fWYDMiyjMvl4kz5TKvVl/79u9OzZ3u2bt3PtGmL2bplD/9+YTJHDp/s8OCDt3waEBBo
    dTrFJ5WVuzz1Jcm/N0+lAKp0UPFCUWH+8+++O133/ffrGTSoG6++ej8gcLvdzJu7GhlhnPj0
    HW8EBgT44na/JUSGE6rqXeUvWZaYMmUhM2YsQxUq4hKFUHm342ScTtdF9nCY3CCmXXNNh8GD
    h3QzLlu6jdRdaWi1CgUFpXTq3JKRI3s7hJBWDezfHEniNDhe0el8rBaL6QnVrbJ791H27Tvm
    36tX56EQknLWglgDjgEnT+Zgs9nxeFQyTp5G9bgG3zSy6VfjvbmZgFXADX327Enz27P7GG63
    B4vF5DEazS/N+9z21c0TdFj84hHi6KpBA7vdsXLlDsOC+RvYtDEVARTkl9K3Xzz9+nWuRpXm
    Xttvkeev99MioADvyXdHzZfAW77YVPOaX21OUyG+AsoeKi8vmPTV5HnamTNXUlpqQ9HIrFy+
    g21b98tjRvdr9/CjYyYFBATlCHFqriRF1fxuFlCGN09qJd6y2AJvwn8D3hLKfvXiAJr3MFw1
    oEsPCQk99uCDN8S/+OIUiorKzxGWQggCAn156B+jiIgMPVZcXJXuzQ2r1Iuyv3X4/NOq6M8/
    X1ArTIQQmE0Gxt83nOBgf+bOW8aqVUkX/CyHw8XypVvxt/rQokUjrFbr34jPM3Q/fL+OM9kp
    znhzf5i3BlkSxokT6z+fharSsWMsjz42Bi7D+d3PPp3LypVJl4TPAwd1NS5ZvAWPR631dHe5
    iHxW1YvD51Urdxg2b96Hx+Op9cr365fQwOdLJaCFSMObM9H4gNtd/mBubqG2Vcto1q9LZdv2
    A5SWVhASEkhFRSXr16ciKzKtWzchL69Y4+NjecDtNFmqqpX/Ck9mlaTUh9LeqgxV4w8dOj7s
    gw/mYTIZePiRG/nu25Xs3n2cu+8ZSZeuHWpEsUqrVjFnfu/sz6D3tR156eXxXNOzw1mvCcCD
    wWDguuu60qFDC775ZinTpi3mq8kLyTtdan7uubveCAsPoazc9Ul+9i5PSGTCFYbzMaAICIwG
    14MZJ7Mm/uedb3Qrlu9gyNDuvPTSeEJDAwF49ZX7keWpzJm7BltllfHZZ+58PqpRlBmkz8B1
    Uohj/FKO9Iq2SrL6+WDQ67y5QDUOWrVuTHh4UK036+KJZ5nTuYUcPpSJ261iNOqwWn0AcZHi
    OTQrAgICX/r3v+99NSYm3LR1y37s9mr0Rh0jRvSkSZPI9PIKz1pFAUWJo6piFwaLJQAkkCRs
    FVWsWZMsde/efpAkF31SQx2gTFZVe2x6ei4Aer2WtGOncHucwUaLUUstoMdaPe6SwevWJks2
    mx2tVovH40FV1cqht2nQ6ONrcsQ7NoSFh6WNuqF3XHlZBUJIKIrMDTf05s5xQ6vDwoLfqrZ7
    Vr/5ehfGjvmzfdRWc9nZQJwEWQaQ24NW7x1zMsBRsObDSdXLLBVQopzO0vunfP2jdvLkhXTu
    0orrR15DYJA/uTkF/PTzZqYlLkVWFM0/nxr3qhDOnZWVO9NNJiuQBUTLIFqBEgS6mvGtAaQs
    CMyAA6oX5FcuJZoQqTWThykOyl8sLMhvadDraN6iEUVFB877OzExERiMegoKi1uGBAV+orrF
    67JG2SecxUi6gHrD5w8/+p7SEludmM3KymreeWcWqlDZs/sYDqfrnOIaZ4SBqqrMmbOalSuT
    6N69LTfc0IdresZhMpmvaj6/8863uuXLt5/jeZckCY9HZe6cNdhs9Z/PRqOe6dOXYDDoueWW
    fhiNukvy16qqnHz//VoSE5diNBouKZ+3bd2P3e6g2ulg+EXi85k56/f5XIVWq/lNPt9wQ++4
    srIKJElGlmW6dG3dwOdLJaC9VWI8enA+k5GR+cKMGUuMSxZvoaioAkmSaBcXg6+PN82ZyWSg
    XVwMe/ceY+rXi/h+3loGDe7W6r57r38xtmVMAKrxQ1fV7gytseMVHrz62KqqgnumTVtIdnYh
    z/zrDiZMuJFuXdvh42NCliWKiorw87Og0Wg4fyErQePG4dx99wjOjpk9+3XwEBjox2OP3Upk
    ZDCTJs3kpwUbkCTM/35x/Bs+Fqtamuv+LDt7lyfyCkFaiEK8lbzMcWCbmpqyv8ubbyWye/dx
    brzxWv713F2EhATW3oOoqHDefOMf+PvNYsGCDeSdLtE9/8LdT3fs2K4PaMeDuk+op5HksCvt
    e07r379H2b+eK/P74vMF5OWV4GMx8dhjo4lr1xz8RxUJAAAgAElEQVTvEar/cUuwZltw795j
    vP32DNwelZBQfx566Cb69e2S7na6D/zv244SQhxz4nZ+GBocUvTkE7ePHX+fvXNJSVnAxIkf
    MWvWKq7tndC4WfPobv7Wm+cDGH0SECJD9saMglarsHHjHu65N79dk+jGccXFuzcHBHQEfCMc
    jrKgY2lZREYF4eNjJv14NpW26lCdXh8FHPV6vtSBObl5XTZv2VczHsBZ7VDcbncL34BONVca
    DBzteioru+n06Uvx9TUx6d3HMBlNdl9f404ZZUFZkWuywaw4xo4Z+Sf76CEgB7AEgbE3HLzB
    5bI3ttkcnSor7UYEGE1GYbEYDuj1+lMQugTkHWA5ALZb09JOtJw9cyUdOjTn/fefIDz8l0Ia
    fft15onH32fe92sZMvSa5gkJba4tLZUyTCalNQR3heLrnU5np7Ky8tCKchseoeJjNmO1+pww
    GM3HIXoVMBtKTl+JXTYhcmrEs7NLdXXplytXbkuYMWMJBw9k4HK5Lpi1YPfuNJ544kNatYo2
    3X33sNFDhvRsZtToX0G1LM3N2OUJj0648nxOXEhmZl6teFZVFZ1Oi4+vmZ07D+HxeJBl+bzi
    ue74gcLCMhYu3Mzadan06dORe+8ZTufOrQgMtF51fH7r7ekkJx+pbd/5Fw6CRQu3kJ9fqnvu
    ubvqLZ+ff77M79NPf+SdSd+xZMkWzCZD7SG4i2FnDulV2qvZv/8Efn5m/vXcnfWCz06HE284
    KL/J5+PHTtXOOCdP5Pwmn73dQVzlfA6t0RvSFeGz5uIN3iKgXAfV/zx06Oi/X3zxK31K8mHa
    tm3C2LGDaNOmCZ06t8JgNAACrVbL44+PoXfvDhw8cII1a1KYN3cN+/ela99444EnO8bHmT1u
    8cTp06lVYWHxV3DwOhPS00813bB+F+3iYrj5pr4oikLH+Nakp5/kzTenkbTjEI8/fisDB/Xg
    wrGz4g9sOQk0GoXRo/tjMul55ZWp/PTTRkJCA8z/fPK25yKaGrZpdWLnlYFzJd4KSWWjnM6S
    F5Ys2RT//nuzKCoq5+FHbuL++2/Ax8d8juc9JCSIl18eT1RUMJO/+pnHH3uPp54a22XosF4z
    9PrAtxCahULscUpShyv4jIMWWiyMuuvOEW/FNI3sMendmWzbeoCnn/qEpyaOZfDgHmguEB/6
    R83jUdmwPoV33pnJwYMn6NChGROfvp3evTrtkJAfTV5XlHxxJoHmCFeqW9aoiTnZrjk+vpbr
    wsN1U4OC/EJ37jzM7DmrzM88M+4Fu31pklGvnJKUDoALj8uFoii0at2YtKNZ7Nix3z86utGd
    qmCH13tR1a60xNYiMzOPdm1isFhMbN6yl4KC0ibNWwS09HoM0EJV3+079ulzsgto3745hw9n
    nJngvLgWR4CMJqrH8c6871db9u49zpAhXQkK9D/hceueqCh1rPMPDrZJcgHGPxlXKsTJGgHh
    3xuKnz15MmvYxg3J8rbtB8jKzMNe6c2mYjDqCI8Iio9rFxMfn9BqZPv2sbn+/gGpIPnv2nVE
    KS4p577xIwkPD64znqOjIxkxoidvvfUNe/akKQkJbR709ZEHlJZm9zl0MD1q9+4j7Eo9SlZm
    HjZbNapQMZsMhEcEx/bo0S52wMBuQ1s0bzwUTO9A8zVVVbswGi+n2AoHjsRVVZV/NfXrnzp+
    8eUCbBX2mmpk0m9snQsc1U5Sdx0l7WgWWZmnE+5/4OYvjXrnjRvnVSRxRc3L500bdtf0gZqM
    OO2bcccdg2jbNob/vjuT9et3/+HMC97Udwr2yiqWLN5K8s5D3HXXEO66ewS+vparhs8ffjCH
    jIzT3sPK0tnvFXUE9ZmFw/Zt+3nyifd58skx9ZLPd9454q2mTSN7TJo0k+Rkb9Vxo1FPRGQw
    sizz10M6vLsPOTkFVNkdSBK079Ccp5++g969EuoFn91uV22dgqysvAvy+dSpfM4Um8nLKzkv
    n3fs2KfPzDhdJ675auPz+PEjCQ8PYs2abSxbuo1ePeO44cZ+jBzZkzffvBCfj9byubKyCo8q
    avgc9Jf5fBE90AagaEhBQcHL/3l7hj5111HuvHMw/3joZiIiQmqej8rZeaD9/X257rruXHdd
    N8aMGcTUaYuYNnUhb7yRyAcf/vOusNCI9TpLyKyFPxzn+luaXQEoVQCntJlZuZSV2xk3LqHG
    wypYu3Yn/3l7BkeOZBIZFYxOr+HixGR5t9mGD+9FYUEpb7/9Dd99u5yWLRuH33BDv9ds5WL0
    ybSdtiYtulzG+yBqnt/JRyoqCt6eMuUH49SvF2OxGHj5lfu46aZ+aLVazhwO/dUWK2aLiYcf
    uYWoRiG899/ZvPB/kzl27FT8Aw+O/sbX132H2yUWXGEXhwvk9a5q7uvTp9NzEVEhYz75aK5h
    0aKt/N//TWb79gMEBFj/sqdDkiRKSytYsmQLRYVl3DCqN08+OcYR07TxDw47bxgs4vCB/WUX
    rTGS1rvgTD+xpcpTLFZENdZtDo8MvlmSJH76aQMDBnSO796943jQvHq2i1wIQbdubSgvq2TV
    qmSGD+99ndXHJygrKyUX3L5Zp3IoKiojtlU0Pj4mlizbRnZOvhQb2zzQu4AuDbLbywauXpVE
    SIiV3r3iOHL410XrZAmq70jdvT927tzVCCEICQ1Er9fufP21U4sfejxQSFKTv9jyAKD6xqqq
    gs9+/nlN+NSvF5GWdgqDQUdwiD+Bgb4oikxFRRUpKUfYtHE3sqwQFxcTPmhwt/C+fTtRUFAC
    Avz9fc47NkNDA2rzBh89mt59w/pd3Vev3sn+/ek4nW4sFhOBgb6EhPojBJQUl5Oy8xAbN6Qy
    Z/Zq7rl76MDRYwYlWCzVj1e5HbMP7dklWne4XCL6pI8Q7vcWL9rQ8Ysv5lNZ6bjgwcHzi0qJ
    yspqvvzyJ0JCAiJGjx70xtD7/W7mGSqujGg8m8+VtR7EYcOv4Zln7iA6OoKTJ7Nrd0L+yrhV
    FIn8/BI+/PB7snOK+Ne/xmG1+tV7PidOW0p5eeW5pamF10mDBG6X5ywR7d0hO3HiNP9+ccpV
    weclS7ahaLyHREeP7l+TKvfPW1WVgx9/XMv06UvR6bQMG96j3vFZOqtPOhwu1qxJuSCfz5yr
    KSmpOC+f16xJxul0XWDsXx18tvr74HK5SExcyto1O9l/IJ0Bg7oRGhp4QT5XVXlTKvtZzVj9
    LAigpLiCzO0H2LB+F3PmrOaeu/4cny+igC7RgPPe5cu3GjZt2sv11/fimWfvxMfHwvnDGs54
    Zb0enuCQAJ54YgyVdjvffbuCRQs36h948JYHHbk5KzQan6IrM24VBfSdS0ttuF1ugkMCAA0p
    Kft59ZWpnDyZA8CQId3p3TvhN9pZZyP/D3ijvSJ69K0DSN2dxvwf1/PZpz/Stk3T61o0b/ZI
    0sGqSVz2fKzHO1RXFTzx2aezjVOmLKRZ80a8+O+76X1t/FltknC5XMyfvx5VqNxy83W1wlqj
    0XDTTdcRFRnMK69M5YsvFuB2e4yPP3Hbi3qdcd/pvKRjYaFdr8hTliRvHKerOvUwkjSheUyT
    I6+++uDE5s2jAqdNW8L0xKXo9VoUzV/zQnvcHhwOF/7+Pjz62C3ce8/IEj+r9b/lFc73zAbF
    IUmXZoclpmlPsrO3uCXJUhoREYKiUcjPK2F64hK5TeumEwwGyzK7bU/S2ZNrREQwvXq3Z/Gi
    rRxLy4ru0KHNoLIy54yoKM2oU1n5ktPppmnTCKxWC0IITp7MxeNxDVM0uhngGXo0LaNJUtIh
    hg/rTpMm4TUlwGuWUmoaIJrZbCX3JE5fIufmFKHRKERFBSPJclXHzhYREnHNXxQRx4DiwVVV
    ts8nT/4+7IvPF6DTabj11usYOuwaYmMbYzLpkSQJp9NNaWkFR49msn3HAbZt2cek/3zL9MQl
    +PpaADh6JAMh1HOSpJzM8MYYrl61g59/2sjp00VERgYzcmQvevRoR6tWTbD6+6DTa0FAdbWD
    zIw8Vq7cwc8/b+LNt2aQk1sY+PgTYz826yyOVu2VHy+PyCoB8nvl5ub0SZy+1Bvr/xd2VWTZ
    K6JnzFhCz14d+oSFR/aqKNm5zMe/yxXjc1mpDUdN8Yg+feJ5+aXxBIcEIITKDz+sY+3alN8M
    3filfwoQAkmumydYlmU8Hg9z564mIMCHp566HUVR6i2fp05dhMulnlc8m0wGHntsNIoGPvrw
    Byorq+q0VVFkbBVVTJ7881XD5zmzV9OjRxx9+nT7A3PwOT2ajRt3MGvmKrQaDRMevf6K8Vmj
    USjIL70gn8/uj6m7jpyHzwVSebm91gNttzvIOA+fk3ceqq2eW7f/1+Xz6dziy8tns7cwmcvh
    pqS0nCNHfoPPRzOQZIlGjUKQZZmYppHo9TpOnsyp5fPCnzeSnV1IcLCVvv060b1HW+LaxhAc
    GoBBr0MIgd1ezbFj2axYsZ3ly3bw5lvf/Ck+X8wY6FibraLz8mVb8fe3cN99I/Dx8eGPpwNT
    MRoN3Hv3cNau2cXy5dsZPXpAD4Pet9f6LU1+PnjoIG1aX+7MHOUKVCcUFZQgSRJ6g56qKjtf
    f72Q9PRsZFlGCEFgoB+yrPxOWyWKi0uxVVQRGRX8ByYvgdls4qGHbmTf3mOkpWUxZcrPuldf
    feChXoPNP6WnbT0S0+Kay3QfbBI4n1uzdkfTGTOW0Tg6jDfffJAuXdr9alcBKiur+PKLBQgh
    GDqkB1ar9qzFkkTXbu15/Y0HmTjxY775ZjkdOsbGDx3a88HqatezXOEiDVpDPELscmef8Lwb
    1cy65eFHbknMzy+JmT17NffcO5wePeIQfxLQEjLbtu1j2tTFjBjRk8ceHZ2hKPoHck641gZF
    4tYaLm14UmRkT4Q4KkdFBqHVKPj7+7Bhw25Wrt4ZdfON1z3jquauX+67QKvT0KdPAvPmrWXj
    plRtXPsWtzRp6vO9EDa/4+nZaDQKjRqFYPWzYNBrOXYsCyS3VYgiM7jHbdq4W2uvrKb3tfG4
    HO6zn6hA6AxItpfWrUtqtnZNCkFBVmw2O40ig1E9ksNkkv8inE8BlaGqWvnMvO9XhH3xxQIC
    Anx57vm7GDasB1qt/leLVomQkEBiY5sybFhPcnIKWL58G7Nnr+bgwRMAbN22n9zThUSEh3Am
    xq68ooJtW/fjdns4ciSLZs0ieeaZcQwe0p3GjUNRFC11K616/1Z4eChdurahT79OvPH6NBIT
    l+Dv7xt4//03Puewk1x4OjUj6JKHqRllkIfuTDqgSzuaVbPl/ddMliWOpWWzM+mAbtSo0KFV
    la4Vf0G5XDQ+FxaU4PGohIYG8I+HbiI4JKiWS6WlNlRV/V3eCiEICvLD189M9qkCnE7XOQfu
    VI9g9uzV9Lgmjl494+sln7/9dgUul3re7D5CgJ+/mVE3XgvA9JqF1K8XibIs4XJ5+Pa7FVcF
    n2fNWsWWLfuQkODPpreTZDZv2Ud5eSW33TbwivG5UWRQTel5ifXrUy/A51/6YmFBGZs2767D
    5/T0bDyeXxZOQgiOHT9Vl8+b9mgLCsvOF85Uh8/r1qQgy96D3Zeez0ZApara7g3jkCAiIpgW
    LZowYkRvsrPzWbZsa10+b91PUVEZg4d0Z8WKHQwd1gOny83WrfvxeDwcPpxFREQQ/3hwFNff
    0JsWsY3RaLTY7fbawmhms4GgoAAaN27EtdfGM2BAF95+a8af4vNFFNC2xqdPF0Wmp+fSrl1M
    TYW2P8tUQaPGYXRKaMmmzXvIPV2obR8XrLPbNZd9sAqRB1S0Li8rbrpt2z78/X1o0zqa06cL
    SNpxgA4dm9O9e1sUReH663v/Tlu9A/uLz+ezavVOPvvsadq2bf4H7o9Kq1ZNGXfnEN56cwbL
    lm1nwMCu0UOG9Lz7dG7VC5fvbuQ0Kysv6jZ37ipUVeWxx26hS5e2Zy0YpNqvM+mjzgx074la
    UeerU6c2PPTQjfzfC5P54Ye19O4df0t4iGUycOwKbxUiSQmUFu5yQ9RmjSbrpJ+fJUaSJeLi
    mnPddd3/kofDZqtCkiWsVguKokkvt2lW+1g9qtHY6TK1Sr8zJMT/ToNBq+l5TTuSk4+QOHUR
    Xbu0HhoZGTn87EapqiA+viXNmkWxalUSY8cM7O7nG9Lf6XBYjxzJwupvISIiCK1WS1CwP8eP
    ZVNld0QpStFQW2VpmzVrdhITE06nTq3YumVvbf+QBP6SXNEnNyf3xqlfL5QsJgM9e7dnzZoU
    gkICPAIpaeCwPn+xfaoE4v59e4/0//KL+WgUhWeevYMbbrj2zB7AeXa+fvHmREWFMWHCKPr1
    68Snn/zAokVbSE/PJnnnIa6/PrT2OaYkH2bv3uNotRqGDO3GE0+MITY2+qwdGM8Fdtm8f+fa
    3vG8+uoEnnziQ6ZOXUTnzq07d+7U/lGh8jyXvJpXjiJEZfuDBzN+Y/v2j44RCafLxcFDJxk1
    ytM+MNRHudwC+mw+b9++HyEgpnkU7ds3Y2fSHvbsPU51dTWrVu04r7ft1+JZkiUeevgmhg/v
    xVMTP2TLpj3n3CNZligqLGXq1wvp0L4FPj7mesXneXNXYbPZz7nuM5Xmfqk6pyLw5lX2FiaR
    aivdnd3WinI7P9ZzPlv9fGI8HpVvZixh1nfL/9Jnuj0ePB4Vf6vPFeNzcIj/nXqDVmOvdFBV
    5WBG4pLz8tl7D8CjClav3smYWwfU8vloWtavtLYg/XhdPq9bm4xQBbIi1dEnZ/N52tRFks1W
    hSxL6PS6S8jnPrjdLpKSdrN+fSqHDmVQkF+M2+MhwN+X2NhG9O/fma7d2jFhwk3069eZTz/5
    vpbPSTsO0q9fAqNH96NHj3YkJx1k377jSJJEz55xPDXxdjp3bo3T6WTH9n2sXp3MkaNZFBeX
    IcsKwcF+tG7dhIEDu9GhYwsGDuyBTqth4sRPmDrtj/H5IipTVZRX2FS7vVoJDw/6i/FI3sOF
    YaH+OBwuKmxVIClCli/vIBWiFLDL4Hhg67Y94SkpRxk8pDvNmkXh8Xh49tk7iGkWSefObfCG
    W/zeAQZvMfvy8kpysgsoK7NxbqzwhcX3yJG9WLx4C0k7DjF3zip6dI+7vUuPoBmb12860qtv
    78ux9u96Ij278e7dacTHxzJgQNc6HjaXy8X27Qc4nVuI3V6NzVYFwE8LNmAyGQgK8uOanu3R
    63/xBA4a1J3v561jz+5jZGbmNmrVqkX7+gBoAGtQAkKcUkCSzjTTmxpI5a8UWPmlsAN4VDDo
    hWK0JlxGwVGd6udncZnNRo3Fx8TYsQP4+OPvmTdvjfnxx8c+J8vayloftBAEBPjTv18CX329
    kD17jwYOHBByf15eRdypzDwiI4KxWEwY9FqaRody+GgmhYUlMU2aRN+9ffvxkCOHMxl352CC
    ggLq5tGW5DiPy9H6hx/WWPbuTecf/xiFKgQmox5fX4tTUZTD/0MDmzoclXfNnLWCU1n53Hnn
    YEaM6HWOWL7g2Kx5T7NmjXnl1QkoGpnv561l0cJN9O/fGbPZjMNRzc8/baSiopKbbu7LSy9N
    wN/f95wdmN/7O927x3Hr2AF8/OE8Zs9eQbt2zW70DbC8D+Re2j7gwO12qkXF5RftE0tLbXiE
    qmo1mivK59TUNDQamWNpWbz99gxSU49y8MDJGr+F+EO51iXA6m8hLCyYAH9fLnTcQZZltm07
    wLp1yVx/fV+AesPnvfvSz9lZEELgW1MJ1MfPjNVqwWwxATBmzABKS21U2qpYvSqZ0tKKc0JX
    9u1Lr9d8PpMfyeFw4TiTqe0v9ytxRfnsb/XRVNocyLLEgQPpdfhcG+YuflngpB3NrMvnrPw6
    ukKSJHJzi+ry+UjmOeNBkn7N5+M13nDw8zVfMj47HA6mfr2QqVMXUVhUhqjJUS7VnMfZsmUv
    P87fQL++CUx44Abax8Xyyqv31/J54c8bua5fAg88MAqTUc+CnzZQWmqjb78E3nrzHzRqHMH+
    /Wl8/fVC1q1NprS0kjPhsUJ4/8b6dan88P06xo8fyX3jR9L72gRuu30QH3809w/x+SJKU4FB
    r0Wn01BeYcfl+isOFe9pWHuVA41GwaD3bol+/Mnlru59GKgYVliYP2Z64lIsFhN3jhuMXq/H
    ZDIyZuxgunRpW3to5fcnUAmn00VZWSVWqw+BgX5/YjdMJTg4kBtv7IPeoCUp6RBJSfsbazTS
    3QajfJlujHPwsbQsxVZRRZdubfH19akjoPPzinnzjUT+9a/Pef31RIqLyykuLuf11xN57rnP
    eeWVqZw5HXymr/j7+9GrZxwlJRWkHc1SJCEGUa9M8PcxWfX3t0pWfx9O5xYyalRv4trHMHfO
    KvbtO9pJwtXL6XSdJRpkevbuiFajYe2aFEngGl5QWGzMLyyhUeNQzGYTOr2e8MhgCgvLyMsv
    0iO5h65dmyJJksS1veM5U6ZWAPZqFxqN1Gf/wWMDZ81aSZOm4dx8cz9ysgvx8TXi52dGCOH8
    H8ThtTk5ec23bN6Ln5+Z4cN7otPp/8IzVPHz8+XJf95GQkJLtm3bz86d3qIhBw6ks3Hjbjp2
    jOXpp28/Szz/WQ+azOBB3QgJ8Wfr1v1kZuVFKhr6Xvo+oCLLMiaT/k8s3n+baUajHlmSpcs/
    Vn7h8zczlmGzVSPLMiXF5Xz7zXIO7D/h9baq4g9n3tDrdVj9LFRX2ykqLPtN73uV3cEP36+j
    vLwCEPWGz+VltnPaq6oqzZpH8cqr9/PKyxN48omx+Pma8fM18+QTY3nl5Qm88fqDtG3XBM+v
    ct57U7SWXRV8PjMX/9Wv+sFn39pc3aqq8v33a37hs8tVZ1EnSd6zCOvW7qrlc2Fh6TlhR2fz
    ed26FMlWUTfmXQiwV/3C5zlzVtcWdxFCYLWasV4SPhvZsCGFzz7/kaKiMhRZRqNRUBRvvmlF
    UVAUBVuFnZ9/3sgTj7/PunVJ+Pn58s+z+Zx8iMDAIA4dzmDTxj20iG3EC8/fTaPGEWzYkMzj
    j7/PgvkbKC+3oyjez/V+/i9/r7CwlE8//Z51a3YiyxqGDu1OaGjAH+LzRRTQOk9YaLAaFGwl
    7WgmxUVlfwnU5eU29h84QYC/hZCQALWq2uUBLlv8szeXtW+Ix1P1f3PmrAxMSTnEHXcMJD6h
    9VneKpU/lpbul8kmI+M0e/cdo3HjMCIjg//05N6vXyeaxURSXl7J/AXrJZvNNrZ1O5/LVGmm
    Ssk65c2xGtsi6hyQBQVbeeTRm3n62dt5+JGbsVotWK0WHn7kZiY+cztPPHkrEeFBv2qzREzz
    RsiyREZmLpIifGiwS2TmEj8/37zgYCunThViNBm4997hlJVVkjhtCVXVdrna6akjttq2bUr7
    9s28Jeuz8+Xs7ALKy+00bRqBomgAmWbNonA53WRk5JGTky9v2ryHNm2a0r7DWeFJQuDxqDhd
    1fL06YspKCxl/PjhBIdYyckpIDDADx8fs91mczn+BwEhHz+eJRUVlREcbKV58yj+ekSBSlRk
    KA8/cjNCwI8/rqfSbmPhos3Yqxzc/8D1REaG/Q+fL2jUKISQUH+KS8o5ceKUQVGkqEvfB/yF
    opjdMTERF6V6vCRB0yZhgFKWX1B92bx1Z/N57pyVgUlJB2s9at6KbvKfFkaqKmjWLIKOHVpw
    8kQOJ2rOt1xw0pQldu06QkrK4doptD7wWT1P5VRZljl5MpePPprDZ5/+wDczlmCz2bHZ7Hwz
    YwmfffoDH300l2Npp7zp7n5lHo/awOfLxOegIL/a8Edvwa1ipifW8NnhOe+iYfv2/bV8Limp
    OEdAV1ZWk1nD5y1b93G+0Gd3LZ+XkJ1dUNv3vbuRvlguCZ9drF6dXJtG87cWrIqikH48h9df
    S2TPniNERoafw+dFizZTVl7J/ROup3WbZuzefYjXX5vG8WPZKMpvp+mUZW+Y5cbN+xDCRXR0
    KJFRwRQX/z6fL6KANu3ys/omX9OjHenpOSxetOWs9Dp/TGSCxLp1yRw6eIKuXdsSFOSfZrO5
    d1w+OJ8EJB8QH27bltotMXEpHTvGcse4ITVxZX/F0yLhdnuYP389ubnF9O/fCYvF/Cc/SyUi
    MoRrerZDkiR2Jh0m7WhmhMkkdysr3XV5fBwu7xypUeRzBqBer2fkyD48/NBoxt83El9fM76+
    ZsbfN5KHHxrNTTf1x2gyntNmnU4LSDicnpq1cINdGvM9ptWSEhUZTEFBKSUlFQwY0J2+fRNY
    tSqJNWt2INVxlgksFgt9+yaQnV3Aho17vIdmJalGNHnHapPo0Fq4bdu2n6ysfHr37oCfn885
    gmPd2p0sW7aDbl3bMmLEtZSX2zmdV0xYeCBGo35PSXFV+v/iDbXbq3C53Pj4mjBbjP+zd6t3
    7w5c0zOOHdsPkpy0n00bU+nWrS39+nX6n3cnDAYdPr5G3C5PTTjX5XDhhntA82N8fKwaEuL/
    P1XYVFWV4GAr8QmtVFAWfJ2Y5bncfN66LbXbjBnLcLvV/8mDKITAYNBxyy39CQzyZ+XKJPIK
    Sn8z7EOSJCoqqli+fDtut7fAxZXms8ulnpegkiRRUlzO11MWMWnSTL786ifKyyspL6/ky69+
    YtKkmXz++Xzy8kouePCwgc+Xh8+REUHnxKGvWZPMmjU7ap6NdM6zzczMY+OmPaSn5+B2e84R
    yKqqkp6ew7bt+8nIyDvPWJFq+bxyRdKvBDiEhAVcfD77mHA5XeScKvjDvUpRZI4fP8UHH8ym
    uLiYa3t3rMPnjRtT6dChGUOH9aC4uJQP3p9NWlrWOZlofsvy8grxeFR0Oh1BgX643b/P54so
    oCOKFcWQePMt/T2RkSF8Ofknli7ZUlNDXf7dGyyEYMOGZN57bw5Wqw9jxw7wyJJudXGeK//y
    dWSdDM6bMjL+H3vnHSVVeffxzy3Td8r23mgLu/RqQxB779g1GqNGE001iSWJSYh5X1M1ASyg
    xhIbTZEmKohK77CU7b3vzu7s9FveP+7sCs9HxGUAACAASURBVAqKsLvwnpPnHM6cw+zM7nPv
    cz/P7/mV76/62qeeel3QVJUHfnAdKb1V3d9+4YDA6tUb+c/rHzBkSBaXXHomx+OZFwWJ886d
    jMtlp62tk8/X77bounJpNKIOQAKiKNhsZlRVo6Gx/YgGR49XXueLsKn+FY/94Z9pampD0zQ8
    7jhAEP8L0v4aii4IkpaalkAwGKa+rhWHI44777wEq83CvHlLqattOQSeIiBx9rRxJCS4WLHi
    c7Zu3Y/NZiEvL4O21nZaWlpJz0jC5bKzc1cpy5etx+1yMOPcicTaoxpPgCjQ2NDKvPnvYZIl
    7rzrUtxuN60tHXS0+0jPSEaWRW9rk+A/EYPXYrEgyRKqqvfBVq9jtdq44oqzCIXCvPzyMtrb
    urj8sjNxOOJO2N7VAVUxlBLMZvMAGdDowJLhwwfvvOLKqb3MPR6DE+DKK6cyYkT+DlXVV4rC
    wPP5L39+XTia0fdtDwOnnVbE9TNnUFxcxjsLPkY/hsOFIMDGjXtpaGg1jJBTgM9HO0cc5pmP
    GWaiKMa6owpf20hHECDe818+DwSfU9Lij+hBfnH++9TWtXzl/gqCQDSqsGLFerZt23/ECISu
    6+zcVcaKZRuIRpSv3GdBgMaGNua/+F5MkeXw35+eltTnfEY3asIE6ds9u6Iosm7dThYuXIPl
    MD4vp6XZy2WXnonH42HJ4jV89tnub6U0pOtw6I+r2rHxuQ8fiirAumhEYcGWn/zsRhRV4ZFH
    5jBr1kvU1DR+jdEo0Nzczj/+/gY//9k/6ejw8cOHZlJYNGx3VNF/n5xmUQZiCet6BRAoDAZ9
    v587Z6F5546D3HDDeUydOvY49jcBkIhGoyxb/imzZr1IW1sno0YNJisr5Tj3S40RhXkMGpxJ
    NKrw+We78Ho7p8Un2tP7/+o4OocNyUIQBLZvO0gkEjnq/XTE2bj/gWu4//6rcThsR70+0WiE
    7dv2o6OTm5Oqo4vF/wVp/wxB8AA2ISszBUVRKK+oBzQmTiriumunsXtXOYsXrzFQHlX47NPt
    vPTSEvz+IJMmDWfL5v1s3FBMQqKL1R9u5pZbfsNNN/2a1R9uIS0tkR3bD7Ju3U7GjRuG3W5h
    7dpNVFbW9P7+JUs+Yfu2g1x++ZmcPXU8oFNZ1Ug0qpCdlQKapNvsJ2L2SkJGRjJxDitdnd14
    vV2ceJ6vzuTJhaSlJ/LRR9twuRxMmVLYJ7Zut89PV6cfm81KZkayhq4H+n8NCICp0WLxvPDd
    716lnXPuuF5lhm9jPGuazrRzxnPHdy4Pi6LlL3UNwcZfPXzlwPJ57kLzju0lx6TtfCzXpbm5
    gyVLPuF//+cVqqsaj2nj7SnQ2rWrlJ4mYSeTz0OHZGEyyV97PwUBvB1+lr3/OStWbKCzM/D1
    6Tw6yCaJnOz/8nlA+JyR8pUDoSgK7NpVxuJFa45qVG7beoBNG4uPeJgURZGdO0v47LNdRz1s
    Llmylu3bS79aXCiKhr3Sx3zu6OjCZDIzYWIhkiQeM4MEQUCJqixZ/AmtrW2cNqWI9PREPvpo
    K3a7hTPOHE1LSxsLFqwhekiToGPhmiRLjB83HFmWCYXCtLZ2YrNZvpHPct8tgjx0XW8VqPjR
    5ZdPn2+SpRH/+PtbzJ/3Hl6vjz/96YFYyP7wi6WqKnNmL+Cll5aRl5/Bw7+4jSsuP7ssGpV+
    Ehc3tGlfZfUALWN7HHifWLt2U9aSJZ8wduwQbrvtoli+57HnOoNIOBxk544S3lnwEcuXbyTg
    N5oW1De0EolGsR2XhJROQoKbsWMGs2N7CQcOVFNb25LsdMaPA2r6+RFfNGrUsO8OHpRh/nz9
    HvbsKWP8+EKOJA9mkk1cf/35hzi99COe2/bsKeOzz3aTm5vGsIIcn47w0X9R2p/DtDElJeFq
    URTFnoJOk8nM7bdfwrpPd3FgfxVms4nVq7cw+1+LaGpqZ/CQDFJS4gmFImiajiiKPPfsElRV
    Rddh/gtLkWWRYDCMKIooqsbPfvo0xcVVDBqcwejRg9E1nQMHqsnJSeXWWy/GbDEDRuhR03VS
    UxN0TRd3Fo46Ec1cR3FmZkprdnZqUnFxBbt3lZGdnXnCVywx0cPIonyK91QwrCCXlNSEPjCg
    Rfbvr6K2toWsrBSyc9Ka0YW1A7MGooDr3+npmcN+/7t770+M/49pxYqN+P2hb9xsjCYcNi6+
    5DQeenBmNCM9bXZtXWhRSrJlYPm8ZlPWe++uiykHnKhRbkQrSkpq+O1vXohtusdmlAuCQCgY
    ZsuWfVx88RmIonjS+Zyfn27ev78aSRKO+jcHAiH+539eAwEiR/BIHrY3axp52en/5fMA8tls
    lsVwOHpYm3VFUamsaDjivRIEoVf16mjv97TIPtL7mqZTUlL7lfd1XUc2yf3I5yxmXj+D9et3
    seHzvUdds189EAhUVNRTXl7HxImFDBuWzd69FeTlp5OTk8a2bQcoK6v7VpEpTdM4/fQirrzq
    bECitKSWmuomMjK/mc994oHW9Y5YcceHgJ4uiVJo9OihnHb6yJjcSi3BYOiIBmckHGH//ipU
    VWPcuKFMmFCAySyFBFEfVFFaljxipBk9VN6vS9cQ+vYN9nqbp776ygpBFEV+8MOZZGalE41G
    UBXlGGgt4vP5Wb5sHT/60d+5554/8dZbH5GXl849912N02Wnq7MbTT3+3ENBEJk4cQRms0xn
    ZzcHDlTaZVmYoOv9neVi3ZmRmb73qmum097WyZzZ79DU1EKP0sJX3Bb0SL4dydiQaGpq4Z/P
    vENraydXXHEW2VmpH/i7I9v/C9H+HMHNCQku1eNx8um6XXz00QZ8Ph+5eZnceOO5mEwmVFXj
    8892E4lEuerqs1EVjU0b96GqGrqu09XlJyXVwxNP3M0vfnkrOjr19W1omo6qqnyydjtl5fVM
    mFhATXUzCxasJaqomEwmbr31IgqLBgEa5eXVbNxYjN1oq6qKkrTpxOYW2e12e4pnnDuBcDjK
    okWf4PP5TtDCMrpnDspPRwfyctOwWMwnaEALBIMBFrzzMT6fn3NmjCMpyXOgOxCp7X8PbiuG
    lGm3EwRbYoI7et55k0lO9hxTyouuQ2KiixnnjCcxyRNF0BwJCaYkh12LfffA8Pm111YKPl+g
    VyVAVTXUb8nUns+JosCYsUP43j1XYbNZYwoI3+5+7txRSldX9ynB5yuvmoYkH9mj16P/rGk6
    wWCYYCCMpmoxHegj/7wkiVx++Zn/5fMA8tntjvvK8/hNBbH99b7DYe1HPneRkZHKrx+/k1Gj
    8o/5GRYEgWAwTE11M6JoNPVCh9zcDKxWM+Xldd94MDzUVNE0jdFjhvDIo98hIyOVaDTM4sVr
    aW/vYsYx8LmPPNDtAAmQebPXW//bRQs/TnzttZVUVjXicjmYMWMCzjg7R8qDtdmsnHveJEpK
    anl3yTq2bTvAzJnnFl177bnP5uZnXAmm+zD56isqtpGfP76fFq8CRM/dsX1/8tatBzj77LFM
    nTqWysoannn6LQYPzuSee69Blo/kjTZAvn37Pp6du4g1a7YhSSLjxxdw0UVTmHHuFL22tkl4
    5eXlmE2mE5bMGTosh8QkN/V1rezdW84VV4QnBbtr4oDu/nu4O5sFwfo/1888b86OHQfiP1i1
    GYRnefDBGxg5Mh9BMPoo9Aj1H/nhFNF1leLiEv7x9zf56KOtXHTRFG695SKvJJrmapoS/C9E
    +3PIYkJivBAf72T//ioefPBvFBXlk5GZTEuTkdeuKCqpqfH88MGZzJx5HlVV9XyydgelZbXo
    ukZBQS7Tp4+P6aFrjCzK572ln1NWWouqKGTnpnHppWcyfvxwXnt1OXPmLKKz04/VJlNeWc+c
    2Quormli08a9lJfXk5ubSmKCG9AiJzY3Z7ckRV+6/LIzT3/33XWmjz/exksvLuW++67BdISo
    1zcZRYce9hwxbnni42K4VL90WDz271UUlddfX8myZRvIy0/niivOjkqS5e01aw509eed1/VG
    IBHoOEPXfY9t3bLv4vnz3+XTT3fS0zDhWDw/tbXN/PKXczjjzFH279x5+T2TJo6cCK4nqUp8
    J+Lfj9kxvN/5vG3bwd4OsCaTTNHIfELhCAf316BpX28A96SguFx2xo4dyoUXTu7l8+uvrTwO
    h4YRSamva8XjcZ8yfF61cjM9erfoIEoCLrcTp9OO3WYx9jHBSNcKBsN0+QJ0dnajq7qhnR3z
    zF908eT/8nmA+ex2x9HY2P61h39ji9WP8vrlNWqwJ/bylRznr3tWPG5Hv/H55Zff5957r6Go
    aCh//ON9PP74c+zcWXaIAkiMr3qssuqQOeq6Tn19CyDgdNrRgcQEJ4IgUVvXTDgc6c3xJ6Yr
    /cVchV5GaLrOpEkj+O0T36WoaCiaprJo0RoWLFhLbm7aMfFZPnE4VwBCEgT+VVpSet1f/vq6
    uGrlJlJTE7j//quZMWMiBQW5iEdTsRAEbr/9Yk4/fSRr12znzTc/5KmnXmPjxr3SL395x+Uj
    RgwT0MV7MlKEfmw0YHZA8NLt2w7iD4Q488zR6LrGn//8OkuXfs699111lK1SIBJRWLDgQ57+
    x1u0tnZy9tljuPmWC5kwobDe43Yu13VhyqpV60f6fAGyc1IwmU7kkuukpyWRlppAXW0LpaV1
    BAPBES5PUj8Dehjw2pvJSROdv/n13X92OR3uJe+uY/fuci655HRyc9Oob2ilrdVLZ2c34ZCR
    tm62yLhcDpKT3KSlJ1FX18KyZetpaeng6qun8ZOf3tSZnJL0i5bG8JrkNOspgzLjEFB/mAvd
    UKn4poY5R45M9ChcCIAooEtmUfv4422cc874AZyVvd3jcbd74p0psiyRnORhX3ElW7fsRxRF
    0tISGD++gJtvvoDJU4oQRZGCgnwKCvLQY+1xjYOSsQ4lSWLylNFMnjKSaCSCqmpYrGYEwThk
    fu+eq8nJzeD111eyr7iChe+s6dU4tVrN6LqO2x2Hx+0K+f3R0InNLQxYluYPGrT+gR9ce/Zv
    Hn+BOXMWYbGYuf2OS7BarXxRBCwc8o/YwU/r9bKrqkY0EiWqqKiqQCisIEpGekpLSweypCHL
    Miaz6RDNUjG2NoRDDOtD05dEwuEI/3ltBf/421tIosD991/LsGF5n/r9kTdHjkzvx7XcCKQB
    B28IhrxPv/P2qpTZ/1pEfX3L1xaPHW34fAGWL1vP9u0HeeAH146fef1Fc625IcnEw292tGwj
    Pnl8//F5+0G6uw3ZK1mWuP32i/nBD2eyb18593//KTo6uo86H00z1ufUs0Zz080XfJXP3d8+
    DV0QoLs7QHl5LYWxzrKnAp/dLof73Xc/JRw2GotkZaTxu99/j9zcdOx2S2wPElAUhUAgRHV1
    I79+/DkqKxtBB7PFxGWXnvn/is/fSvDrCJ7IHjScCnzucUJ9YTTqsRQ6Q9JNlkUsFhNxcXbs
    Dis2mxmb1YosS5hN0mEGYiSiEo5ECQZChMIR/P4QAX+IaFSJRW/U3u8WBKHXwOx3Ps9eiCDA
    nXdezugxhcyadS+/+OVsdu0sj2k0y5jNMk6nHafTjsNhxWqzYDGbkGWJzKwUACxmU+zVWNM5
    OWmcf8FkNFUlHDYOiN3+IN2+AL7uANGI0jv3ceOG8odZ91JQMJhAwM+bb6zi6affRlVV7n/g
    2Pgsn9hCLgOkFAg8s3PnnpmPPzaX3bvLufCi07j//usYPXoQRpj/67p16ZjNZoqKhlBUNJhz
    z5vEs88uZMniT/lx09/5wx/uvWzChKLnNI176+q312dm9EdvetGka1pGh7cbm9VM3qB0Dh6s
    4aOPtnLuuRO5//vX8NVuWwKhUJgXXljM7NmLSEp087snvssll06td7rcb2mKPt/nU/wI4VWf
    r9+DruuMH1eAyWTmRDRknS47ySkeBEGgtrYFn69bdrkSHP35aAuCjK7vBOJfzMzS9cd/ffcf
    Jk0uTH3++SXCvBfeO0yUXJIlTCbD0FKiKopitEk1XlWGDs3mD3+4V7/oojOa3E7n4431gRcF
    QVIFYfQp5A0oAeQc0NIiUQVd12mob6W0pLLXmDz2ayfSUN+KrutEolFEkbRAMJJzxulCxQB7
    OPabzdK+whF5KZs3FXPuuRO48sqz8fn8mEwSickJZGelYDJZvuSN1Q9pFqQdMcJgMlswHWaY
    6phMZi67bCrTpo2lsqKemppGAoEICQlOPvhgE6+/vprhw3NxxNl2d3ZG9p3Y+sxG19e1CELi
    /ZddOu2Nrg7fyD//9U3++tf/0NDYygMPXEdSUiKgEgyGaG7qoKa2mZbmdhoa2mhoaKOltYOu
    rm5CwSjRqIISVVE1zdBWReCVf6/g/fc+R5IEZFlCNsm4nDbi412kpMaTkZ5ISkoCiUke0tMT
    SU72YLNZAYmOdi+zZy/k1VeXIwgCD/zgGq66alqxIEgPu1w5rW53Qj/ed7cAZTf4/W3PzJ79
    RtK8eUsJBcPH3cq7R5e1qbGdJ2e9QmtzZ+K9913zT6tllhBoDr9JvyiKGHz2ervRdSNvs6ho
    EN+//1o0TeOdtz/GSOug95B2qGGl6Rp5eWncd8+VR+Tz+vV7UBX1W18TI5wcoa6u9ZTi82OP
    3/2HiZMKU+e98K6w/0A1hYX5nH32WERRRtdVw8MMiIKAIEjk5aUzeXIhFRUNDCvI4e67r/j/
    w+eIgigKXHTJ6YwsGoT+LZefgMCeveWsWPY5kcipweeNG/b2Rn0cDhuZmUnk5qaRnZNKVmYK
    2VkpJCZ7sNutWK1mrFYzFosZSRIxydJhB41oVCWqKIRCESIRw6AMBsO0t3XS0NBKdXUT1dVN
    VFU1UFvbQiAQ6lXz6G8+//1vb1Fb28IPfnAdI0cN5ze/+R4vzl9KdnYKBcNzycpKIT7eid1h
    w24z5mg2m4w5ChIgYbVZAT32KnHLzRdzy00XoCgK4YhKOBQmEAwTCITp7PRRV9dMcXEVtbVN
    3HXXZRQUDKahoZGn//EmCxesRZIlfvAt+HyCHmibAP5flpQcnPnYY3PZv6+K791zBd///rXE
    x3ti4c5jkQj9wltTUJDH7353Lzk5acyds4hHHpnDX/784GUFBUPnKop257MvrGq79+6+bopk
    0QXRFHS77WiajhJRMMkSI4bncuWVU3G7XYCOpqkYuokiiqLy4vz3eObpdxg7biiPPvodRo0a
    tlhH+t0Hy9p3XHjpZNlprvjD+vV78jdu2Et6eiKnnT7yxLcSUSQjphfZ1emjvcOXlJKqTATK
    +hfSY9D1CvWjD7vmnT3NefCGGy54s6KiLr2utoVbb72Q8eOHk5wSj9NpQ4o9xJqi4vcHaWr2
    smXzXl55ZSXTzxnPDTdc0Khrwo37NvvXpQ+T9fiEyaeQd6MUkPIg9MKGDbsLP1i9iWhEYc7s
    hbz04vvHJcgSDIaJRhQ++GAT02dMKJwyeewL4ZDw3Uhwe6XZNm6AZmYLS5J9/VVXTztr2bLP
    pdWrN3PTTeczanQRoBKNhggEw3Q1ttPU1EZbWyder5+gP0gwFCYYjBAKhQ/rMCoKAharBYtZ
    xmo1YbVZsNptOOOsuN1OUpI9JKfEM6Iwn1GjCwCJisoqNs16meRkN9dfP0OVZdO7n6xt9p74
    +pyKru/ZazbH/+6W2y57XLaYR/3tL28wf/77VFU1cvnlZ1NWVsu2rfuprm6ira2LQCCEySQZ
    UDbJmM0mw6NjNWO2mHGYZeIT4hg8JNPYjCJRIlGVUDhCuLOb6uomlKiCoqhEowqapmG1WvB4
    4sjKTmHs6MEMLchh1cpNrFy5icREJz/+8U1cP/PcYlm23CNJri2CkNCPa7kcaBwXiXT97YV5
    7yQ9/9y7RKPKt5J3+joOhUJh5j67CEmWku6955pn4rLMrX7/ztUOx5h+4bPL9YUdOnxELomJ
    8bz+2jIWLVqLpunIsjGvLzqpga4bBUKPPHLHUfm8aVPxMRcPHskb2tLiRdfVXkm4U4XPlRV1
    6QcO1mCxmnj11RW0NnvxdvkJhsKg61gtZlwuO8kp8ZjNJkRJYvr0/198XvXBRhwOKzOvP4dz
    zjntOJxTImvWbGLd2u2sOkX4vGLFBqml1csNN5zLtdecQ2ZWMomJ7pjz7UjVs0fflCzWQ50h
    R7a7otEIbW2dVFc3sWL557z66irc7rj+5/Nf3+C1V1dRUlLDz392M1NOG8X48QXIsjkW0VOJ
    RiOEwxFCwQhNjR34/QGCoTCRSJSoAvsPVCOKIqVldXz62VZMsojNasHusBLnsGK1WUlIcJGe
    bkKSTEyaJHHVVRqKEkaWzezatZ8n//gyn3++m6QkDz/+8Y3fis/y8S/kGsB/RXt7w+1P/e8r
    7N1TwT33XMlDP7ox5nk5Xm19jbg4O/fffx1mk8Tf/vYWTz75En/5648uSU1Nufae76Y8d+/d
    fY4fH2jLs3PSxqmaRklJLefMmMTzLzyC02nkQIbDEebPf5dhw3I499zTWLXqM2bPXsjoMUOY
    Neu+riGD8uf7u7U/WWxa04wLIkD9VZ2d3ode+fcysb2ti9vvuIQhQ7JP2EEjCCKJCW4EQSAc
    jtLe3mU1maTMgXjEBSGfQOAgsmwSWlurrBvW72VYQQ4P/OA6PJ74r5mbwJQphWzevJ/Nm4pp
    aW4zuT2JwezhNt3pGXeKwVnMg+C8jRu3znj00bk0NrQxdepoLFYz2nEq+QiCQDQSZfu2gzz2
    yFxm/fH7M06fMmaeEhG+Gw1trzRZB+IaVOtgeWrkyKFTr71u+pmzZy/i6X++w4UXTOHAgWoO
    7K+iqroJr9eH3x8kGAj3hoBl+YsWqEfKl1VjxUhRxfBumU0yFosZu91KQqKL/LwMCgqyGVGY
    z9q12ygtq+Pe713BmLEFmzRNeO6ccxL7aI5FwIa3Taa44htvuHBecqJnyp///BoffbiVdZ/s
    Qtc1PB4n2dmpnHHGKDKzUsjISCIjPYHk5HhsdiuySUaWYtEUSSSWRmoIlWl6b9hTVTWiUZVw
    MEx7RxctrV4aGtuoq2mmvLKBivJ6tm87iKqqKIpGwfBsfvrTm5hxzpRdoiR/VxTtW1Yvr+nH
    tdwCdMfruv+JFcvXpr3w3Hv0eOv6jgcCkbDC888tISc7OenyK2b8IhBgc23FZ51Z+Wf2A59T
    x5nNJhRFYcvmffz5z6/w0YdbUFUNu93C3XdfTn19K4sWfWLkTuoaU6YU8Yc/3Ps1fF4utrf5
    vlXDhS8Pr7cbVVWRZfmU4vOmTftAh/ff/5zFiz6JFQx++TM9USQZXdPYumXf/xs+P/bYs9TX
    tXLnXZdSVDQIv9/H8Xg4ioryufW2C5k/bymPPfoss2bdd1L5fPU1Z585Z/YizCaZ8eOHI4hG
    WpyuKwQCYcLhCNGoQjgcIeAPEQhGiEYihMJRFFU5zD4WJRGbxYwkSdjsFuIcNmw2CyazCYvF
    jM1mxmSykJaWSlpaChs37EFRVK6+auqA8XnTpn389KfP8NOf3cLo0YMpL6+nqqqB+ro2Ghpb
    qa9vpcvbbcwvqhBV1VgBrE5UUZBliXeXrGPlig296V0mk4xJlrDazCQkOElNSyQ1JYGs7BRy
    c9LJzUunuqqBWX98if37qhg+Ive4+HwCHmgtW1MDP/vPa8sTV6/ewqWXncn9D1yLzWbh+FMU
    vrjzZrOJ79x5BTU1Lbzxxmpefvl96Sc/vuVef5f44YGDn5UVDOtLQOsamMvGjRmqpqUmSEuX
    fsZFF59Gbm52bC4iGzfu4dm5S/jlr26jvb2d559fgifeyWOP3sng/LyX62q6f56YYldkuRoY
    NjUQaPnds3PftqxYuYmRI/O5885LMZlMfXBtBOLi7AgxGaIOr69PWvMe8znZFm+Bju9v3Voc
    f+BANT988Do8Hg+qaoSGjPyiWDtSScBskrFaLcTHx3PhRVN45um32bylOOnii6ddLsryFgam
    icQxwLkaUPIh/MKGjdtm/OpX/6KxvoOf/OQmbrjxfExm6fj/UgEURWPBOx/y56de55FH5vDk
    H++bcdqUsfOUiHC3t3NHhcc9tp8311Hout4uywdn33zzRZPXrdtpem/Jp6xYbgjsu90OPPFO
    cnPTSE9LJCnJjcfjxOmyY7dbsFotmC1mzKbDw9yaDpFwlHA4Qrc/gL87RMAfpK2tk7r6Nurq
    Wti2bT8ffbwVURRQogpFhfncdPOFUVmyPCtIC9oTeaLPDLqurhBO54G90aj0nXPPO/3JQYMz
    L37+ucWWt9/+mDPOGMmjj36HzKwU4uLsSJLpa7w4+tff0KP+n0ogEKKjvYs//ekV3nvvU664
    6mwe/OHM6ODBWf8Jh4S/WE36rn17Wjj/kmn9ecdlUG4tOVh24dNPv43vG1rmHr8nWqC7O8A/
    /7mAEYWDpg4bNugmb5P2/Al4UL6ez2nxUm0sv/jpf7zVm7t5/nmTeOCB65g7d2FvMXNefjqP
    PHLH1/L5gw82n9ChQhCg2x/sNaBPJT4XF1cCOpIo4k5yY7OaMZnl3lQVVdVQogqhcITu7iCB
    qEpxcSVbthQnXXSK8/mRR2ZTcqAWp8vBzp2l/PznzxxXY6AeboRDEUwmmdKSWh57dC6zZt17
    Uvn8ySc7TIsXr2P8+ALsdhslpTXU1jRRV99Ka2snXT4/4WAEVVFRVA1N1VA17SsdRo2UKxFR
    EJBkCVmWsFrNuDxOkhJcpKclkpGVwojhuVisJt566yOGFeQMOJ/feWcNjz76LE6HDV93gGAw
    HIskfcFWI0XryM9gKGRESI/G5561IcsSdrsVp9NGKBzB29HNlVdPO24+y8e3mHcB4RuK95Wc
    9cprK8nNTeOhh2bicjn7wED8YgOz223c/8C17NhZwltvfsiMGRPHjxld9BNN13+EIWraRzc2
    Hl2vXJ4/OG/3jTeeN/bvf3+LXz/+PA89dAN5+Rk0NLYwe/Yi4hNcTJlSxKJFa9lXXMVjj9/J
    6NEFS0J+fZbLY1Kstipg2FS/v+P5hOEU7AAAIABJREFU555dUPDCC0vJSE/il7+6nSFDcvrs
    2ohSTxGIis/nB7AODMS6gaa0UMh72rL3P0PTNBobO/jTn16iob6FtvZO/P4Q0YiKDphMEnFx
    VhLiXaRnJBMMBtE0nfff38D0aRNvFDC/3N+hzW8xBul66PlNm7bPePzRuTQ3evnZz27mju9c
    iiyb+mAfEbj99ksRRYE/P/U6v3rk2RikRz8vRYXvALX9P8VywP5+dnbOZ9dfP2P6E3vnU1SU
    zy23XMCQIdlkZiZjt1uN07tJ5tCOgt9uqGiaQiSiEAqFaWnuoKKygf+8voo1a7Zz7bXTycvP
    Wh/wq0sc0WvA9ESfzdDlssaMg+37/QHx1qFDh/z8jjsu/dWHH241q5pO/qBMbLaetvLqUYzg
    YxUZ1o9gbAvY7TbMZhNRRcXhsHHzjeczZEj+vKA/+lNR1ANtNTonpqt6LMOXFon4fvLqq8tN
    paW1/WI8f2FEi5SV1fHaaystj/zqzkfdybYPMRJV+5zPM2eeO/Yf/3grlrIhoes6LpeDO75z
    CV2+AB+s3oym6VgsJr5395Vfy+f585fF2h+fmIUbiSiHdYE7Ffj8/tJPMVtM3HffVQwryCUt
    LYF4jxOHwxpTpjG62AUDITo7u2lsaqekpIb5895j6dL1TDvF+Vxd2UROTiqCIFBT3dQXeCY+
    3onHE0d1dRO/+tVcZv3xvpPG55kzz53++9+9yC9/OQdd1wkEQr1rVRAEREFAlEQjL9hiwmwx
    IckiJkk6jF26bnhpFUUlElHo8hmFdHplY6yRkoYkSTgcNsxmCb8/xK9+dfuA83nNmu3mpqYO
    Av5g7O8+REEkFiWRY5FNs1lGlkRkk4T0pdQrRdVQVIVoRCUSiRIOR3sNcV03+o74fAG6Y0XD
    LpfjhPh8nB5oh1NVuy5+d8knNDV28Oijt8fSE7Q+XlAa2dnp3H77xTz26HMsXLiWUaOGXZqZ
    bvsfoI87rJgbZNn22K23XfpGRWVd3OLFn7J7Tzm5OanUN7Ti7ejmiSe+iyxLvP76B4wfP4xL
    Lj69WRDEJwRRa3I64wHrpGCw87lnn11QMGf2IrKykvntE3czbdqEPj/IC4KxGGIi6bIvsBmn
    fVI/wjkEmATQbqyoqM3ZuLGYcFjhtVdXounGhuWMM8JDcszD4VNV6uqa8XeHCIXCCIKIqmls
    2riXsrKaIUVFBefpem2Zv1MnzpN9kvksXtPS0jzjd7+fZzT+yE2htraRp556hT7oDd1706KR
    KPEJTsrL6pg16yXmzXt0QkJCWuZAAFoQBqPreieULi4qGjzd4bCSmZnMtdfOAHqiI4YqRSgU
    prvbSygUIRyJxnJ9NRRFOWwtC6KA2WTkEFssZmxWM1abBZvV8FpbrTY8ngSGDstjzZpt2O1W
    CgvzEJAXfbze7738gin9s9sO0tm7V/VDwvupacn3Jye7k2trW+jo6MJms9NT7GjMFxRFIRgM
    EQrG5ntIXvOhQ5KkWL60jNVijhXxWBClHhUOA+iBQIia6ibi4mwkJXnalaj+qiBoAat9wgAY
    UjpwcPr+/aVpK1du5MS7Mh6bBbJy5Uauu35GRlFhwQUvvvhayZ133tIvfK6srI9bvORTVMXY
    GP3+IC+++D6iKHDwgBF2nTy5kIu/hs/Pzl1CJBI9YeP5aNGKgeZzD8R6+Lx+/V6mTRvHgw/d
    gCBYOLo2v7FmRyFy/vkKTY1tLF36eQ+fpxeXbikrHDLxFLCfY3z+3QtUVzdx331Xc8WVZyP1
    cR95VdNZtuxznnt2MbNmvci8eY+dND7Hxdnwerux2izk52eQnOwhPTOZ3JxUkpPjSUhw4ohz
    GBFCS4zBZtPhFpSuE45EiMQUKfyBMN0+P+3tXurq2igtqaauvo2mhjY6O/24XI6TwuekJHdy
    U1MHkiTh8cSRlppAcqqH7Ow0MjOTSUx04XLH4YiloBhpgjKSeHhENKooRCJRQqGoUTzoD+KN
    pdhVVBipdY2N7bS1daEoygnzWf72cPYC7ec0NDSf/sEHmxg8JIOLLz3jkA2pz7cDLrhgMm+8
    /gGffLKdhobm9MzMzOn79qz/94iRp/fh71GA5A8TEsS/PfbYd+8ZOjQn9f2ln9Pa1kmcw2Zs
    rOEoc+csoqGhjZ///BYSEuMXNNcHdqZkuIHoxFCwe97zzy0e/uzcJRQW5vHrX9/JhIlFR4Vs
    X2yUoWAYTVOS9u8OiP1wgvkSaWvu0PXQwx+u3iQ2NXWQnpnE+edNYNLEQlLTE/G447Dbrb0G
    tKIaqgder5+W5na2bjvAhx9uobKykVUfbBIKCwf/xN/VHIHkf9OnId/jmp4cCIRoqGtBlkSa
    m7y8+sqqb13VfaweO1k2lAy6u4Oh9HSTd+Dm2QVYa9PSkoIul8NWWdnA3r3ltLf7qKluoKKy
    kdraZrwdPrq6AgRiRmUkEkVR1ZgBfSj0jTQds9mEzW7FYbNid1hxuRxkZqUweFA6ObnpxMe7
    2FdcidNpIyU1wReNqvsnjzH12yyrqsYTFwe6Xlpit5saMrNSkzdvKubAgSq6u0PU17dQW9NE
    VVUjTU3teL1+urv9BAJho2I9aoBY6w0jxgxo2ZirxSJjt1mMcKDLQUKCi5zsVPLyM8jMTCYY
    DNHa6iUjI4n4BGeHbJJLTeZRA3SPa+N0PXrXxx9vsTY1tR93gdy32/wFmps6+GTtDnFk4ZBr
    r7xy6L8BX3/w+dHH7rpnyNDs1GXvr6fbH8IkiyxfviHmoQO7w8qtt110VD4/99y7fWg8nzJ8
    Bjpsmha+5qMPN4vd3UEmTy5kz55KGhuaaWhqp7XVi6+rm0jYeIZNZhmXK46kRDepKR7SM1MZ
    M66AdxasMfhcNPjK3HT5DcB/8g1og8+1Nc2MGJHHDx+8Hoslrh/2VYFBg9LZuGkP+/ZWnVQ+
    x8XZbZIk8cijdzBhwghSUtxYrJaY0dhzYD9Wu+vLP2ccqMLhMM3NnezdU8pvfzsfQeCk8Dkj
    MyV59+5yLrjwdO699yqys1JwueOQZSkmn3q8MrL0Oko0zYgI1dW18cbrK3np5eWkpZ8Yn4/H
    Ay2DMn337hJbbW0Lt9xyAelpSf3IBp3ExHimnzOe2bMXsndvmTk7O3uqjv4mhsBgH20AOeh6
    cygQUH+bmJC07r57rvvFNVefMyUYCguCgOWZZ96Rn37mHXRN48YbZnDWmWNqlAjPpWRM0eDg
    xFDYN//FF98fNXv2QiZOLOA3v72bYcPy+t2mNfKO1VGZ2Q5bX4NO1zti1ya+53wX7/d3JWzc
    VIyu6/i6/GzfXoKvK8iwYdlkZqYQ57RhtZoBjNy6riC1tc2UlNZQXlZHR4dR6LF58z58Pv8w
    m82VoIQ5ZYau6wweksXDP7+5V46vr4eiqPzj6bepq2tDEBBAH7AsSUFwo+tln8bFWeqzs1MH
    b9t2gLu/+yRt7V3omhHOs1hNeDxO3C4H6elJOGxmzBYLJrOMxXo4MpSoSjikEAqF8PuNfy2t
    HZSU1vDpp7uMwiXA446j09vNqNGD8HicraIk705NHTMAM5ZVqzVOz81J46MPt/Db384nFAzj
    9fp6DzM2uxWXy44zzo7bHUdamhmzScZkNh+WH6tEFcKRKKGwQiQcMTafFi/l5fV0dwfRNA1V
    1bHbzbhccbS0eDnttCLsNltYUbTowK1ibWhXl7dww/q96LqOKA7EujK0lrduLSYQDOQ7nS5X
    XxrQ38Tnfz6zQF65ciOSJHDNNdOPyue5cxYRDkX63Xg+OXz2nx7we4s2btyLoqg8/9wSuruD
    dHb6jS6Euo6ufVWGUhSNf06XA5fTTjSisHnzPro6u893xHnOAlaeKnwWRcHo+Lt8A7LcP3xW
    VRWvtxvRKCI+aXzOzEweXFpaw/DhOeTl5QAquq4QiRjdkX3dQfzdQQLBcCxKqBKNRr9yvcxm
    Q+LObrfhdNpwOu0xOTgZi8VOdrYTTdMQJZHU1PiTwuecnNRDmhwZik2goapRFCVKOBShyxcg
    EAgSDEVQokaKxpcjhCaTjCzLWCwmHA4bLqehlS3LErJswu2Ox+1OIDl1M5qmkZ2dfEJ8Pg4D
    2m9DV2bsK65A13UmTCpEFOV+diCKjJ9QgCRL7N1bwYUXnDUtIUF29qUBbSzeFAKB/RqC+kF3
    l74pLT19MIgCKBMef+yux265uS5ZlkUGDcqsczgcP96/y7djxNjKZE2LPL1o4epRzzzzNmPG
    DGXWH79PXl5m/zscgFBYQdNIsdvNpuMHcTkQAMxA8JB7WQpE0fXdQI0OkY1Wq7X8Fw/fljF9
    2gTr5s3FFBdXsnr1Zt577zMURSUuztprdGi6Tnd3CEkypGXiE5yccfooJk4aweRJhSGbzVov
    StoG0dyiGjqmX15DJqBHl9iFIKT14eZz4Mt3X+3pKunxOJlx3uQvFZn1oXmjRXnl9ZXU1rbG
    jseCEgrtwGKxHbIWC/px1QQVq9WiJSd7iESiZGQkcdVVUxk0OIu8nBRcHmdvqMzQGDX1Fkgd
    9VAQjRKKFROGQmH8/hC+Lj/VNU2Ul9ezdct+NmzYS3JKPDabRZEkITQwW5IdUVTIzkkxYKvr
    nH32GPIHZZKbm0ZaajwutzNWKGnGbP4ix0760pw1TY2ldhieeKMSPkooFKary09rSwdVVU2U
    ltezdfM+olGFtLQEzBZLo88biTBgQz29tdWbWl5RPyDe50ONsZKSWtpauzLS0xzTgdcGis+P
    PXbnYzffcmGyLAlH5fO//rWAQDDSp0okpw6fLUA0bLXZmh5++LaMDRv3WjdvLObAwWoURe0t
    8LZYzSiK8XlZlohEojHjyorLaWfEiDxuu+NipkwqCtnttnpBUIOBwFZsNvmU4LORb1/Pj3/0
    9352pEBysvuk8jklJZ7t2w+wr7gCvz9CSUk1FeX1VFU30Nbaia8rYEhqhnsaP6moivqVZ1KW
    ZSTJaL5is1qIc9pISHCSk5NG/qBMhgzJob2jk64uP6NGDT45fM5OQZIEqqsa2bO7hLr6VkpL
    a6msaKCuvoVObzd+f8goWI9Gewtgv1w4KklSLM3OKJa02y243A7SUhPJy8tg8JBM8vIzKC2t
    BR3SUk+Mz8dhQPv0ULhLLytrwOVykJ+bSv8X6upkZ6eQEO+ksrIBVY2Qmubun9tpN9rRdrVv
    7dTUyDZRgref6dp6/Q8SPxwz1j1a10EWhV2I9rIRY1cBV1x9YH/JpH/+cwEpKfE8+tgdA2Y8
    A0TCEQQ07FbxOCDRDHQAeQKUpoFlBijpxukPQBCNo6AcBbMO1s9kWb6tsGjEaUUjh4676aYL
    Lurs6k7atbOE3//+RURR4K67LmXOnMUA3HPPlcyb9x66rvPrX9/J2HHD8bidrVabbbWgCxsR
    hc26JrRIkj4d5GGgug4PNZkAawPoH0Fqo64f0CEeQUjpi0NZNgjXQ49+VfgcUewRLOtpSd7/
    61qWpTiIzjSbzbvBPBhUEfS3gZp+hJYzGGwzNza2kZjo5vHH72TsuMLDPGc9f1/MIEPXjRCY
    pmqHvx+r8pZNMnEmC3FxwmHfMWGi8R0lJRXcfvvvaW7qIBiMmM0mzQW09f8TEtR0PaS3t3ch
    iiL33HMFN99yScywFL7x/hzuzZEwm8Fs/ubPLF26lh//6O+0tXcRjUazrA67hX7tRnfYnPWq
    qga6fcEBVYAQBAFfV4CGhhZzXn6Oe8D5PObIfN6/v2TS7NkL8fkCJyRXd2rz2aKD7TA+33zT
    hYfxWRAFvnPHJTz33GJA4O67L+fll5ejqur/Iz4bz5ih9NRffBZ6dcRPJp+bmzsIhxWefPJV
    olEFb2c3qqIhyyJms6lXv97oQCgj2S2Yzb1rA03ViISVXtlNvz9EZ6efSI2RnqYoKqIo4HI5
    sFrNdPsCtLV6TwqfOzq6EASB7dtLuOuuP+L1+ggEwr1e5d7iwZiSiCzLuNz2WHG0DrrQ22VQ
    0zSiUaN5TEuLl0gk2lvoa7GYiI93EgqFESWR9hPk83EY0DZzONwsNzW3YbdbSUr2DIih0VNB
    3NLSSTAcxmaX+/U3uhK+SCb3tm5FiUbLEPUyA2wCkYCKxX51ciTivfs/b6ySm5vaeOKJ7zF6
    dMGAGc+AUVGra0gm4VvCuRVIFKB7NBy8KxLxXd7U1J5VVVlvamhsIxQK9bb4zMtJJy09GafL
    3mazmnw2m0ORZKlUFEVTS0s77yz4mM5OP48/fgeTJxfx7NwlaDpMnz4em93Cn578N4sWrSUz
    KwWPy2XWVD1LUSIjQ+HgQ4FA2Oz1+hxNje2u2rpmQVWNh9pqtZCSkkheXno0PT211mwOvQem
    +ZC8S49W64Ip50Qf3KGhUOgvdbVNKIqCgEBtbSOqqhEIhNhXXI7Z1D9rLKqo+H1BFEWjvKwm
    LhQM/15T1ZDFYrFnZqdhtVp39Begdb0B6JpWV9eaVVZWx7Ch2RQMzyUSCdHZ1Y233Ud9Q2ss
    L7gLny9IKGBojUYiUSLh6GGmoiQazVSsVhOOOBtut4N4j4ukJA+pqfEkJnlwu+LIzU1j2LBs
    du0sob6uJcvjTpzW1razIjGxv8OEYmEg0J2+c8dBEhPdTJxYiCBIRCJBfL6AUWTi9dHW3kVn
    l59Ad9BoGhOJEA5FvxLylkyGJ8dus+JyOoz5JjjxeFzExRnhUYvFxtixw0jPSGLXzjK83k5H
    arI9ZWA2JADR0tLSTjg8gE7v2AiHo7S0dQKCfqrw+Y03PpCrq5oG3HjuDz5XVzWYGpvaCIXC
    RthdEMnJSSUt7eh8XrDwYwKBEI8//h2GD89lzpyFgMBZU8eSlBzP73//IosXf/L1fG5qd9XX
    t8T4bKg/pKQkkJs7sHyORhUyM1OYecOM4+6m+U1DUzUWLlpDe1vXyeVzeR2SJKLrOjk5qUzN
    HUN2dipp6YkkJrpJTHAR19Pm2moxUnEOXeO6oVsfCYUJhY3COr8/iLezm/a2Lhob26ipaaKy
    qpH6+lZEUaSysvGk8HnXztJYrb6AxWJi4sQR5OamkZmVTHJyPImJbuI9cdgdNhx2C7LJhCgJ
    h6ViaaqOqkQJhSIEQ1HCoTBdXQE6Ony0tnqpq2umqrqJ6qomQrE0rt27y0+Iz8dhIZhMqqZL
    wVAYWZawWMwDAiKzxYwsy0QikS95wfp/eJImHGGhdwAt1xw4UDF+2fvrmTS5kEsuPYMBl808
    7ssgyFB+ZzDY/sTGjTvS3333UzZv3kc4FDGqXM3G0ujuDuL3h5AlkcQkd2JSkjsxPT0Zl9sx
    pLXFy6ZNxYQjCvc/cA1XXzODhQs/prWtE12HjZuKmTnzfPz+IHPnLOauO2cxceIIV1pa4ln+
    7gB1Da20tnbR1uolGlVwOKzExRlhskgkSigYwWQ2mSZNKsy/8sqzHpxy2rjrbTb1N2B5EaOq
    6ASGpJeX10QfevBvppaWDtBB1TR8vgD791Vx661P9OMaM1JbVFXloYf+gSSJooBgT0r28I+n
    fxwtLBzen4tIBG3w/n0VUmenH9ks8eyziyk5WE15RT1Nje0EAmGi0WhvG3ZRFDGb5V4dUaGn
    uwhGoaimGhJJiqLG2rrLmEwGG5KTPeTnZ1AwPLc3t3DXrlJp5Kjhg2MetH47ber6p0CwoL6+
    KfXAwRo8Hgf79pezbPnnlJbWUV3VSFNTB8FgiGhU7fXUaJoh0v/l/LoeL6sgCkii0VxGlsWY
    vJKJ1JR4cnPSyBuUwbBhWSTEuygrq6OysjE9LTVtqq6X7BOEoQNiQIfCEdQB5iQYedDRyMAa
    7t/E55UrNgyoJ74/+Lx06Wds2rSP9rYuIpFo7N4aETObzVCDSUpyJyYleRLT0hJxuRxDWls7
    2by5mIbGdi44fxKeeDcLF62lo8OHrsPChWs588zRTJo0nFUrN7FjRwkTJ45wpaQmnOXvDtLQ
    0EprWxetrV6CgRChUCQWmRNiGv8mEhJcpsmTC/Mvv/zMfuNza6sXQTA8qp2dfoaPyOPBB2ci
    CP1V5Kawbcd+Kisa+MmPn0GSRFHXsScmuQeUzx3tPrKzU/nT/9zPiBF52GyWmL0lnfD+E3ta
    iUQihEIRysvrefjhZyg5UHty+HygGkGAG248j7vvvpKEhDisVkssjVLok/nquqFE4vOFeOed
    1fz1L/+hoaHthPgsHw8iRUHQzSYTPi3Qm0/V36NnM7fZLV+47U/q8Jt1PXj9Bx9sljo6fFx2
    2Vl4PD3ty/8/jI4ECD68aMGq9D8++Qpp6YnMnDmDMaOHkJ6RhNvlQBAEWlo72Vdcyb59VVRU
    1tHY0EZl1W7CoSgeTxxnTxvL1VdP5/TTR1Fd3cgr/16Bx+MAQeCVV1Zw2mkjufvuqxgzZigL
    F65h86Z9bNpUjMViwhPvJCMziTPOGMnIosGMKMwlOcmIaHR2+WlsbGPnzjIWL/6EVR9s5OFf
    3Jp+x+1XPKwJ0SVA84nubIqq0d7hw2G3MnrMYMMwjBmHej+vLwEhFg010kV27SyjvcOH0u9G
    T9CuKMGLd+4qJRpVWP/5Xj5ZuxNBEIiPd5KaEk9KWjw5OemkpyUSHx9HXKwQ48uFdT3trQ0J
    uAgBf5j29k4aGtqormmiubGN5pZOVn+4hZUrNyLLEqqqsWdvGZFI8GKbzfa/9KlSw5dHkQwt
    5+zeXUp7WyftbV08/PM5McF9AafTTlKym8ysJFJSEkhMdOF2x+F2O3HYzUaI9JDUVU1VCYfC
    BMNR/N0hvF4fzS1empvaaW/ror6+jeLiSlRV6y2kjSoqO3eUiJMmjjyjrcP7In2oX/81G4Zq
    MpkQRSGmTzxw1qMoCphk+RTgm8Hn1as3S01N7f2qg93ffH7yT6/Q1RXo1QD+snHg94fo7g7R
    2tqJrlfFWpjrva3FRVFg3bpdfPbZbsLhKElJbkDgxflLef21lWia8bNNTV7ee++z3s/26O/2
    aPEe+js1zSgg9vuDVFc3snLlBn7eD3z2ertpb/cdxhwlqtLe7sNs7h8DOqooRKNGV9XOLn/M
    6tGRZGlA+RwKRRgzZginnVYUOywYhyZFMe633x8kFAwRikSJRoy6jEgo3JsXLIoiFpsFSTbk
    7SxWEw67FWecHavNgiCImM0WzGYbY8fGMWHCcPbuqTwpfG5p8eJ02jnvvIlkZ2fG7CgdMIze
    7u4QgUCIYChENGJIqkZC4cNUocxmE7LZaBZkMZuw2iw4HVbi4uzIJhlBkLDbHdjtLs45ZwIv
    v7SMpqaOE+Lz8RQRhkxmOZqY4Kamupm2ti6SkxP72aAV6OzsJhAIkZGZFGvycLIN1ciYTm/X
    mHWf7MDlsjNu3LABM+pVRemLv78ThE+SUxKG6DoUFeZx333XYDLZACPsKAgCySnJFBYOo6c5
    ht8form5nWAwQmKii7S0RATBTHNzM0899Qq7dpVy112XIkkiz7/wHk899Sq//c3dTJ48lsmT
    C2lqbKeltRObzURycjwOhx1JkumpujXybXVSUpMYOnQwp58+moqKOurqWkhK8qBp2vqoEurs
    O4+ZzmmnFfGn/31gQAuuDj+Ja/zy4X/x0ZodA2HiTPb5/MN27SpFFEWys5OZOGE4EycVUVCQ
    Q1ZWMm5PHLIsx4yOnlzhY13bho60oqgE/AHqG9opK6tj06Zi1n++m7LSOnbtLMXb0ZmWmGDN
    W7du1e6pUy/op7l6ZVXtLty5s5RIRMFms5CVncLo0UOYMH44Q4ZmkZ6eQGKiB4vFfEircvEQ
    96HwJU/GF3mYuq7FWnordLR30djcQV1NIzt2lbF920FKS2qIeLvZtu0AoXBwfLzbFYeR1Nrf
    hAgkxLswm02EQgPrDZZlmTin4xRwcPTweWefybgfdc6SeJhh29d8BhFRFL9WOeTQjm1HGj1d
    2jRN5+KLT0eSROa9sBRVVWOf71Hj+HbeTSNXWEdH6Bc+9xwaeuYuSSKlpbV8//v/i9hPvNZ0
    nYMHqw+75l8cJgaOz7IsMrIon+rqZmpqGigrq6O0tI6G+ha8Xj9dXX66/UbXvkhYibVq/6rK
    iiRL2KxmbHYrTqedeE8cHk8cqelJ5MakN7MyUxk+PBezRT4pfA6Ho+TlppGS5GbX7n3UVDVw
    sKSWqsoGWlu9dHYG6O720+0PGkWTESX2XB/exEoUBcwWEzarGbvDhsftwOOJIzHJTWZmCvl5
    6WTnpOF22cnOSaG+vvWE+Hw8KRy61eJQc3JSWbNmG2WlNRQU5PWrx05Ap7a2mY4OH2lpCUii
    WfS2hyjeXUzhqMKTBOhwYkNja0J9favRZz0t8ZCNt399l8FDTpkn8KCGQfr71LMnnHHFlWcN
    f+/dT/nPf1YiSSaKi8tpa/PidjsZPCiDtLQEEpM8JCZ5SEpwkp2VgihJKIpCbW0Lu3aV8Prr
    K/nssz1MnFjAbbdfrFstFsrK64WVKzbi7fBx8y0XMmbsUJJj36PF0iXKy2ppbumgvd1HY0Mr
    ZeV1tLd3/V975xobx3We4ffMfWf2SnK5uyQtURfK1J2ibJmypUp2o6BOJdeFm6BGEKCOGjRo
    4z8pekGQoEXrBEVatGlQGUhQ6E8ao0nqOk3jWJYNW3AVSrYUyZUoUaJEkRTJJbkk936Z6zn9
    MbsUJdtRwmh31WBegAAJcEGemW+e+c75boi2hbF581pwHMEbx8/gE5/Yjf37dl6xTfIPDOQe
    d19xQ/KkSadUjDYKzgBQ7pqemgtOTabAccCmTd04+NQebNy4BtG2MDiev8OO2R2O44fb5DK7
    AiFuFXQoLEDza24T/KAPpWIZ4+MzuDk5h4mbs52xWPxhABfr+KzszOeLvefPuxX9ff0b8Lk/
    fAoDA1ugaVoVf7VQOL1jzbirE0gIVy1okeDr1NDR2YH+HQ/iYwd24f33r+HfvnMMrx07jeHh
    MSzMZ9d1JAIDAF6r/z0Wnc6OdqaqCqlUzIalLzAG+HwSotEwM3SHoqly+Twzs1jf9TNAksVq
    SzX3ubnXfD506LHel146vuzqnPCVAAAMgUlEQVQ6s6VJbcu5cTeGUMqwcxmfb4wlyYm3z/8C
    95X93L/NGMOTTw7Ujc93srpQKOHd00P1dWPvsmFpDJ85vPLKCfz7995EcmahWhBIqxGCj4oO
    fMSxZ7ECxrJL0Ql3jW6EQhQFxGIRyJIIAoLJyVQT+EyQyRbw539xBFPTC8hli3AcWk1Vwh0R
    kbuc4VcMZFntkANL18s9IOHgD/jQkWjF/HwWHMfhypXxFfN5JUWEZZ43j/X2ru6nlOHo0R/j
    9HuXUc9tPiEEY2NJ6LqBjRvXgOfFkfm5sg7IDWhO/3P2T9kCK5V0RCJBDF28jkADTl4YgLHx
    JH7VwTWE9IKx0YuKEj7y3HOHvnHq1BD/ta99B5oqo6UlBEWRcPnSOI69dhqWZUMQePhUGX7N
    LRSTFQmViolMpoDk9DwEkcfTT+/FF/7kGX3dulV/DxB89auf/7N//ub3lP/64Ts4c2YYnZ1R
    hCNuyzDTsJDPl1EolFEqVeA4FILIIxTSEAr6MT4+i9ePv4dySUe0LYTDhw9ZmqodAZEuENJ7
    D0FJcP78CP7qr/+1qa/78+dHGtReS3Di8Tbrk596XPzJq6dw/PgZvPXWOXR0RLFuXSfWr+90
    Jz9FI2hrdQvjFEWB6pMgVqu+l79MLcNERTdRLFVQLhsolSrIZAqYT2UwOZXC2FgSkzfnMDOz
    CMuy0doawp6929He3gLTsqOo3wQmAHqIEBLesGE1bowmcea9Ydwcn8WaNQls3boe69Z3oT3W
    gmhbCG2tIaiasgRZjnNznT9w/kzdlJvayXMuV0Q6ncfCQg5zs27rpZGRSdy4kUQqlYHA81i1
    OgHCEVWUWUtjrEk8GU+0zHR0tHbMz2cbFlmhlKLrgRji8dYZRtnJJh9BV/lcqa8jRAiymQJ+
    du4KJFGoC5//4LmD3xg8dZEfH5tBJBJAe6wFPkVCNltENlusdhmwqmkHFLUuErXxxYwxBIIq
    Dhx4+DY+v/CCy+f//tFJlIr6bfxZ7igLgjt5U5JEhCPu6aVhWEilskgv5rF6VQyf/exv14XP
    tW5Id7oXjUjJuX0TxO7BpuiX5/PVkamlVoOE1CJkZEU2vfwztbVQytwc6LEZiAKP9vYw9uzt
    awqfFxZymJvLgue5JVtcaa/vj4rIUEqRy7pFlADg88l4YFV8xXxegQPNKCBe27Gjl3Z3x7nh
    4QkMD080BIiJRCt29G2gAPd6a4Qrzi2ANA/PolbRDWKYJkZGbuL5L/wjSIP6i1bKenVoA63m
    OK5UawFM/aCnZ93vP//87z02MTGH/fv7nVisJS2KAiuXdeTzJV96MR+YmZnHdHIBqbkMUqkM
    UvMZFAolBPwqnn76N/Dxjz+Cgd1bhgKB4L9kM+ZRgCKRiKe//OXDX9y3d/sDr756CldHJrGY
    zqFcVhBtC2Lr1rWIxVuQSLQhEY+iLRouRCL+iqrKsGyHzKdyLSdOnOHj8Tb09HS/ncvp3/2j
    Z+/dhpjnOAT8PszOpvHyf5xo6sueAIi2R8DX/cWg/rgtGn3uT7/46c8/88wTu86evSJd+N9r
    GB2dwoUL1zA4eNHt3kCIO5ZbrjnObgHh8sp3xhic6shrwzBhmLUxqma1Ut/tzNHR0YqHHu5F
    3/YN6OvrMVd3d03Kovh93bCP1nfHKQ+Gw21/85WvHP7MoYN71rzzzvs4c+YShq9M4OzZqzBN
    C7IsIRDwQdN8UDUFmqZAVX3Vgh230vsW+RgM3YJumCjky8jni0ubhmLRDS3KsghVVbBqVQyP
    P/EQBgY245Fdm7MtrS3ftyz7jcZYk5CMRFoXdu/e0jE6mmyATblyKMWuh3sRDAQmDL2ebb5+
    cT5XdLeYsl4+NGPAu+9extDn/q721q4fn2/O4fF9/U401pKW7uDz7OwCkjMLSM1lMT9/i89+
    TcWmTd04cOAufP7JKVy7NgXTsuBTFETbQojFWhCLRRCPtyL+kXw+y8fjrejpWVMXPmuagkrZ
    16TT4NudaU1TmsPnC9cxNjaNiYk5lApl6IZVHSLC7nAYP8xpZEt2WvuZEAJJEqEoEjS/D4lE
    Kzb0dGH7fcDnsz8bRnJ6HqWSjkrFWKqxW55Kc/tbEx+63tvWzNz8CELcgltVVdDV1Y4t29Zh
    4JFfjc9kZcZ0o8uyMm9cGrreWyiUGxIiZAxQVRm9G9cuOJa4NxTZeaWZKRyMXf2dqyPDP3zx
    xZdRKev4QDpOPZ2taijDcSj2PLYNzz775BVOUHZL4tbsL7+OywD4/ZSWfgRglBDpRUL4EwAc
    gONA7VUgdA8IBICFTd3cb1pGsFSp8KZhcZIksnAokJIV5XXbdL4rSOJQqeRG8I4do+TgQblf
    lvGpYqF0IF8otjuOQwRBoKpPdmRJySmqfBLgsnBggHAnwQk3AUoB8Iw5+xg1/5hS2sMof4gX
    yQme77tH9+/CE8Vi8c3Ll24Q07CA5vIZYO5o3U2b1zK/X/sYx/W9VR+7nQJQABwzQgn3WxyH
    baZh7CuVi52L6UJ4IZUOJmcWkUplquO8S8gXSiiXdHdQim6A1kKAhECWZciKAE11W7gFg27O
    WbQ9gkSiDbFYix4K+edDQf+0JCsnAXKO2vgfTohM6ZUUfGp/HZ/RUQDr4NgXH+QFPGUZ+m9m
    svmNi+lcYHJyLjI5lcLs9DymZ9JYXMyhkHeb9RuGCdumcBwH1GFLZzCEq/a8Fnh3KIEmI+BX
    0dYWRrw9gmh7BN3dHejqarcSida5cCQ0JknyTy3TeXNhsXIyHlcMjutvAJuyHJD8UjI5/bc3
    RqcaasZr1nQgFov95dlzma/vHni8aYnQd/K5ng4YqxUdM4/PHp8bxecs8tk88oUKcoUSKjU+
    G8ZS601CCOTqICxV9SHg9yEQ1BAOa4hG/5/wObmAmdk0FtN5FPJFlEs6DONWhyi6LFPM5bPb
    GUmSRaiqDL9fRWtrCLH2CKLR8D3n8wod6HMEUJ4AnO2NtuN8zixk5s2Xunt2l5r6PLFrayit
    fNswSp2UNS/dTxQEcJx43TTJpzVtR2GFD63AWOaTlmVflCQM1XKGGANsi4FwBIJA0M2fw6X0
    Q21aSBABIgMQAeZQSnMcxy1aZgXXb+jYtHEPACCZPIdIBFAUFWBOKwgJwa0WtABiVPKW9U8v
    TC9+6esdzDYdMDAI4q3QI2UE1MRmwrNtlSL7QbBlp33v7t/QAwB5BmBc8wuebsm2bUope1mW
    ++t2gnf0yH/iM4dXAxyBIEool52wqgp+26YPCgK2A0QCWNAy7T227bRRZoMxBw6loI5DGHO5
    QQgYx/OM5wh4zq1y5nhhQZKEkwApAMwwDDouy+JZx7ALvOzLUKcAywSyiwzxB3bW/XoaxiiA
    PAQBKOdsvz/iD4E53SB4BGB+0zB3W7a52rItYpk2b1k2r5f1sG46Ecu0iW1bqHnQPM9DkiXI
    El/0+eSUIApUFAVHlkUqCFJRkqS3AW6RUTZNOG7Qsuy0KCo50yigVLbR0jLQIDalASwkAPN3
    AaY00n4dh+p62X7l+nh5pm/bXo/PHp89Pnt8/rXl8wod6CsAHFDHhEMba9x6hSKfsdHVPdDU
    B8kdPe3XAFNq7iPtwHEsxzBYQdP62crWch2UJmFZKkavU2zesusDv3Ny8Kfo7uQRCkrQgoDt
    uK2Mak9qdtFErOPD78n0xGm0tUsAcUMpHAcIHEGpAGSzJnJ5ii3bH/3A58bOn0PXRgJB5mFW
    HMjqjnt4/4ZqQKzmDd5fkqT+hvyd0ZFBxBIyZAWwTECUSDXvbBvmkuf8akASA4FagRSpnq9R
    7ta5DKG1Ijy97CCfM632RH8RuAjbtmEYFLJM4JjA1KSB9b2PNu2a5tOn4dNEOBSQFQ6Aitm5
    khqPKbI7EppKjFFJr1irCcev5jgii+KtIh3bcnOfGXWmZJ80TClxeJ4zAWaXy7ajadvyjL0P
    06AAYaAOcHNMx4ObH2vGCSzAdNi201j3gwGWyaAFdnp89vjs8dnj8681nwk8rfABX4Tb6lJH
    s3fIjgMYBqBp/Stcy63OLYRE7qvrfOPa4NL3a3se9QyvARq6MAgtAAT9EgSJg6LcGl9NKWDb
    tRAhIFYnrFEbqFQAw6AolkwUc8C3vg1888j9ec/OnhlEMMjQ0iohGOTBce7/TimDbVNwnOD2
    NBaxDNCAQxmYY4MTODBGIEnuV7kM5NImMmmKLX2enXp89vjsyeOzx2dPnjx58uTJkydPnjx5
    8uTJkydPnjx58uTJkydPnjx58uTJkydPnjx58uTJkydPnjx5un/0fxIE7tLND9KgAAAAAElF
    TkSuQmCC
  }
  set pieceImageData(Merida2,65) {
    iVBORw0KGgoAAAANSUhEUgAAAwwAAABACAYAAAC3DfDxAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBChw6bwndzgAAIABJREFUeNrsnXd4VFX6
    +D/33mnpjfRCCqGEkNA7SFFAARUUsItixa677g8buiv2VdfeK6ICIii9956Emt476ckkmUy5
    9/z+mIAgwS+6CMHN68ODD5nJnXPmnPfztvMe6JAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO
    6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO6ZAO
    6ZAO6ZAO6ZAOuYAidUzBHxch8gBNBzo38GgC2SFJ3h0T0yEdctZ7qBiQJLAHgXQ5qK6gNINY
    BfpyEEKSwv4H5qGudR5aPJxq2Wh2jr1Dn3RwoUM6pIMLHVy48FyQLswklAAYwW50fgS9FYRV
    kkIvsi8zFxDDwf4+sBykb8AnB/wt0IAkeXXs/DPOXT2gSFDjBqoPGIaCbScoteDbBKpo7/Mn
    RC1QCLjrQXUB+bdeDWgq0NT6DyaQDL/9Hg0QNqCl9R/cQFZ+e9tqgGKBRjtEIEk+F+i7RYEm
    V2iWQKeCaxPYOFnJC1EIeBqh9AohHM+ZzY1xDQ1NioeHq+rh4XFUlk2fgvc3YK8+WTc4f787
    UOEGTQq4CnBrBtSLcc8JkQN4SFA1GXRznYvFMReCV0ANkhTzP2YkCDewyWDQQGoCxMXEhg4u
    dHChgwsdXPgrckF3fp0Eb6AoEOouA90EsPdGAJI+BRxrhEhfC+EVUMfFAQgrILnbbOa48rKK
    +ICATreYXDgC5g9BWi7EUZsk9WxjHkKAgmFgjgLjOog5BuVcbA7TH18LyU4dR84E0J6zWs0+
    qkq4olBkNHrUgmUuxKwSIhlJ6tvOR2MBDOPA/qIQ4jc8c0lBktKAW1vh8DCIW4UQ9jO+R5L0
    IH0JvNQ6YV8iRA+BUH/jPYBhDlhWXIh4gBD5QHk4SLeBmIqw6pD0ldDyGRiXC9FSC9WtU1AN
    tDxmbqj9x5IlGz1/+mkblRV1+HbyUq6cPCxh2rRx//HwlAJefLHLUw5HJoriBvgBZT5QPgm4
    XQirvyQ5HNC4BPhciPwiSYq8yHZEE6B2sVrNb6xdsztaIBg/fsibBr2WDo6cv74+SAfc9FA/
    EOSRIF0thM1NkvRNwFLQtgpRtBea7JLUvYMLHVzo4ML/EBd+/mkblVX1+Pp5KZMnDe3gwgXk
    gu78KQIXIG8iaPOKi0rik5LSlfy8MjRNo3NUSM/+/btfHxERdhiK5kDA6otDKVgBHVVVNcyZ
    8z5+fl4hV08ZFTJoUK/hbm7uH4HuAyFEFtRrkuSNEJlAHWCe4HBYP24w14Z4e/mmyHLWY6Bu
    EbZMJEPXvzgUMoEC4OgEu9324ZYtyeFLf9xCfn45YREBXaZePZLRowd8qDccuQv0qxsbk3F3
    b8/rwAZI3oVFJb1+WLQRu81xmj4WmqBXQgzjJwxxKIpObo32hCTtP9x944ZkhBCnvkc4FfyY
    sX3p179XCKiALKuqI3bN6l1xhw/lIMnSaYEqvUHHNdPGEBHe2Q1A09Tz/N0mAc3dhLB9kJdb
    PGrnrkOUl9UQHhHA0KEJQ8PDw3dC8Wwgzbl33LtZW2pnvfPO956ffbYCWZHx8fEgI72AF4/k
    UlpayWN/u+mmOXMs84VQ06ESaEoA21vFxSVDdu86aMjPL8c/wIchQxISu3SJGC3L6j1CJGVI
    Ur+LaFcYAddpNTVF0S+88BWyItG/f5x3cLCbm6pqf3F9kAI0+4D6QovFPO3QoSy/o0dyqaqq
    x8fHnZ7xMQMTe3erdnWVF4H6hBBHa39tbHdw4S/MhYJywsMDukzp4ML/JBc+/3wFVqsdkMjP
    L+PI4ZwOLlxALuj+/MVSBqwDjFfYbc0fLl++LfSjD5eSnlGEpgkkCZAkunUNV+66+6rekydf
    8rFB33Inkm61uToZD7/2rBQ0QKBpgoqKOrZsOcCGjUkMG5bgOnPmFQ8PHJg4VafLfgfq3xQi
    zQ45QJcJDof540UL14QtXrKZWbdP6jd+/LBPhdDdrtNLW8vKkgkO7stfV2xAZFdVNb+zYMGa
    8Dde/xaz2YKrm5G09Hx27TjMI49cF37jTRPe0VQmGgxktP8x6SgpqeDDj37E0mxHPklpCwGq
    qnLttDFcOm4QiiK3rhuZAweyePe9H06KAB1/jzMi5evnQb/+icrxWJSqaaxbv5/FizaiKAon
    vQVNE7i46hk8NJ6I8AgZXMjIzDjHe7kREApYWlPmLlZAkyQ3hEjFGSK1v79x495RL734FVlZ
    RQghkCSJxD5dDc89e+eo3r2731tTY3nUx8fgkCT79UePZkd///1GgoJ8ePLJW+kRF01mRiHP
    /+tzFi7cxNjLBnYeMrjPdZIkPwudPcD8xuHDaZc8O/cjkpKzEJoGSETHhPDEnFtGXTZuyMd2
    m+Emq/VwodHYCyGaAGSwGJ3z7mIDSZUk93YQTU0H+g0Gewy0DBMaOBwqipAByShJTNTpDAlC
    ZGZB1z1wMQRRfs8cpAJ6b7C8XpCff+u77y6S1q7dR21do7NSA/DydufSS/v5PfDA9LujosJN
    mlV9RDiS6yTdxcSFg2zclMzQob06uHDWXPiOmpoGJAkOHcxm50XNhaVYmq382mPQVI1rp5+Z
    C20lJiTpN7iwcCOy8utSJoGLq7HdcOHll74iO7sEVVVRFIWExC6GZ8/AhUWLNmGzOVAU5cSz
    7HaVxYs3/wYXPiblQBaqQ0WWZaKigpnTwYWLx2FwRk6G9hCi6d0fftgQOm/eVzhUlUvH9qdP
    v25IkiAlOZNt2w7x7NxPsVisYTOmj//Y3CBud3dT1rV/D1DBaDJhcjHi6mrCw8ONtWv2kLQ/
    nRnXjY249dZJ/wwMDAoGz3chVgHLuz8u3Rj20kvzqaqqpyCvDIdDi7niihHvWq3aeG9vSv/S
    oSRUQLs+J7sw5oMPfsTVxcjTT99Gz/ho0tMKeOXVb3jv/R/o1797TK/4bteB9lzynp30HTS0
    XY/KabxLxMSEcONN45FlCUmWSU3NZ/HCjccdAhmECTxbwGKUJAlJkhg5IpHRY/shWqNOa9bs
    Ze+eo63vEcaaGofB11dvkkCWJAm9Xse108cQFxeJ0DQ0TfDN/DWUllUdR5JybpVYIaAzQtkY
    YAyIUSDZofp7kJYIIYpgIxA6vaCgZPTLL39NcXEFV189gqjoUI4cyWXjhiRef20+b7z58K2e
    Xt7LVVVbq9OpvqmpedTXNzJhwiAuvWwIOp2esLAgNm1KZsGCdWRmFjFkcB9/55AaxzU0VA97
    /fVvSUnJZuyYviQkxpKfX87q1bt58aWviIwOGRETFXWtrPC60/FKDwJ5hvOP0EP9ZpA2ClG6
    ERxWSYq4QCtGBvpcZrM1fFZYWBZms9s5dqwGVVXRNI2srAKPuvqGF/Q6HRHhQYVG05Hbwb7h
    r+MslAFBQMbjpaXlM5955iM2bz6AJEko8i9GT0N9Ez8s3kJ5eY304guzZ0ZEhJZJSuITZSV7
    CQ4d2O654OJiRFFkmhotrFm9m+SkdKbPODMXXn55PtXVDRTklf/PcuHDD5dSW2tGp/tFjdXV
    NfL+B0suSi7IsoQsy6cEhU78XG6bC4oitwZUTw0kybLUNhdkCVmRWx0PTn9PO+DCKy/PJz29
    EEWRURQFIQTJSRm8/u/5vPHG6VxoaGg6bc4kCcyNljNyYd++9JN+P+TklPDiS193cOFicBiE
    qAH2A2G3pKdlRb7zzg8oisKcJ25lypRLcHFxAcBisbBo4QZeefUb3n7rB3r0iAlL6NV9ht6Q
    uC49bTvdewxvp+rADZBwdXXF1dWIi6uROXNuJienhG/mr+G9d5dw5HCu6f/NufWRuLjuQ0EU
    HjmSHvXOW4uw21XGXDqApH1pvPrKN4SH+cf16Rs/x243zBMirRY8BQTZwSEkyfgXA4PNL+VA
    BsfKq7n5pglMmToGWVbo2jWSlJQMvvpqNUeO5NCrV7cA0J3Hkzb/ZVxR0wgO9mfmzMmtURGZ
    1Wt2smTxJjRNQ2giRtMcK6HeIctKjKppaJpGYmIXZs68yhmVFIKiokp27zqCqmmAeqW3t4jX
    NLusCRGtaRqyLDFmzAAmjB8KaKiqysYNyRSXVPwJezgdsOhBfdRqbZp75HCGMSOjGL1eISEx
    dnBsbNQ9spz9DMQvhqpJ+/elkpVZxNVTRvLii7MxmVyoqa3j3rtfJik5g4yMAs+hwwK6gmMt
    gMGoR5IkLC02VFVDp5NQVRVLixVZBpNB5wyTUQ/4RqSm5hr37Uujb99uvPLK/fh18qOlpQVF
    kVi8aBNJ+9PoEhN5RX528xuRXQqvFcLxTH5+Yc+DBzIls9lCTExo/959uj/g6tryHDS9JkSy
    /cJE7QWgTtm+PSXs2Wc/xWJpQdME9fVNSBI89uhbSLKM0aDjiSdujbhi4qirQdqQnLKTvn2G
    /gX0QDXQEKupTTd8/dUKtm49eJIxdBI+ZQkhYOeOw3z08TKeeur2G+z2A58LoWW137GdxAU3
    U2uWzWnEVFXV88H7S9vkwrtvL6a2thG9XkdlZR2vvfq/yYXKitpTsrTH10FNdcNFyYXfZIYq
    2uTCb85UW1xQxXl2+H8/F7Kzi09xAo87RgcOZJ2RC22a1JLUJheSkzJOccwkCRRFIS+vtIML
    F0eGoRmIj4a6GStW7qSoqJJ77r2a62Zc1po6c24MFxcT198wnuycYj7/bAUrlm+jV3yXMTW1
    +6P9A/S57Xe7mwBR5epiqg4K8vHfuzcVV1cTjz56AwMH9OCNN79j85YUjh2r5dm5swYNGBg3
    6NsFaykoKGfmrRN5+NEZ/OethXz+6XI++GCp/NJLwfd4enmMEkJukCTZCjmLQVcoRMlOCKmB
    aiSp018CDzarHU0T6PS64xETZFnCaNA7y3gc6vHd89sNI9pf7LT1j3ZcsyJJMikHspj77Ccu
    Op3S//grjx7NbU07n/x6DSFAIFi5ajcFhRX+gD8405IpB7KQJNlZ64TG8fKHEzUc5xQKpUCw
    All/q6+rnvvue4uM33+/gdoaM5IE4RFB0kMPT+s+5eqxT0PLYZ1O1SoqahBCEBUVislkAjR8
    fTwJDvZl795U6uqbWjeOAugKe/fuKoKCfaWtWw4w/6sVJCR24cjRPLZsSSEw0Jf4XjECyHeO
    U1JqahtoarIQEtIJv04+gIbJZCQqMsRphlY3IEmaFhruGq1pdU+tW7sj/t///pbMzCJUVcXD
    042pUy4xPvzIjLm+vr7k52uv1Ncnq15eF6TcQ/P28SQ6OoTmZhs2qw1zQzMAAf4+uLi64OZm
    wsfXC9CcBc1/mSMNdsD9xqKi7M4rVuxsjYTKZ4jSOp2GtWv3cOMN4zp3iY2+0WDg2YuBC4GB
    3v4n2z2yLCOEYMuWAxw7VsvcubMGDRwYN+jbBevIyy87ESFWFJmCgmP/k1w40yFhIcRFzIU2
    DF9Z4sCBzDa4kIcQtBFdd+6Dlav2nMaFAwcyT3Oy/jxn4XQuLFq4kdraRmQZQsMCpAcfOp0L
    mqa16QRYWmxtciEg0EcqKapEUk7Nsvj7e7fJhWZLC236GOIMXHj9W7KzS9BUFXcPV66+amQH
    Fy6sw9AA6N2bmho6HT2Sh6eXG6NH90dWdK0RhV++Ub3ewCWX9GPRok2kpuXT1Nzk6+vj56Jp
    tna85U1AwwFJ4mD37lGXLlu6naNHc7nsskEMH9GPyKgQXnttAcuWbeXpZz5i5swr2LQpmS4x
    ocy87Qq8vX24Y9aVJCVlsHFjErfdNk8XHOIXHxUZQs/4GGJiQkdHRYapRpPrfmj8HOQlQsyo
    hJeQpKiLVE0qgFHr2jUSL293Nm9OYviIBLp27Ux2TgkbN6fg5eVGbGw4QCnCcZFHVJ1RlKKC
    cgrzy04DQFtRVUmSkCWJgylZHEjOPO1nBoOe89PpwgxY/DSt+bYvv1ph/PSTnwkPD2TC+EE0
    W1rYuuUQLzz/JYEBPr1Gjhj4OBjLoqLD0Ot17NlzlMLCEkJDAkhLLyA9vRBvbw+CgjoBotmp
    4XTfdYmJuHH27Km9X3vtW56f9yXu7i40Nbfg4e7C7Ien071bVKpqUxcqBhfA0RQeFoifrxdp
    afkcPJhJfHw05eU17N2XiqwodI4IBuQyRaf+4/ChzIRnn/2EhoZmJlw+CB9fT5L2p7NgwVqM
    Jr3x8cdvuSM4SPe1pmnF539d6AH9l30Se45+6z+Pd0fSRFlZpXLjDXORZYnX/v0AISGBqiQr
    Dnc34w5Hi/q1ooO+/f4K2YXjNp8lJDunhMrK+jNGE39Z91Bf30hGZiFxPWNDwHHRcEGWd5y2
    hxVFIj09n2daubBlSwoSp0fVt2xO+Z/jgoeHG7W15lOMPyHAw8P1L8QF5zooyC8jL7e0TS6c
    SVKSMkjen37aWvm/9tCfw4WVxk8/WY6qqicc/sKCY7ww76s2uWC3O04rs/L0cG+TC/fcM6X3
    6//+jtraBmRZRtM0vL3duffeq9vkgo+3BxUVtaf9fkV3Oheee+5TSkqqTpSLNZotHVy48A6D
    09i32aw0NbfgYjLg4+N+hmiowMfHHZPRQLPFisPhACRkuT1veG+EEA5oyY/vFYPBoCc5OQtr
    ixWjyUhYWAhPPDmThnozGzYk8fLL82lssnDPPVfTOTIEUAkNDeTKK4dz9EguKSlZHDiQjRAa
    siwTGOhLXFyUcuml/QeNGNlnQFhY8J2S9M+PQVooxHu1cO95VBLnSgyAYVnvPt1vnzZ9jMeX
    X6zk0UfeopOfF9U1ZsyNzdx883h6JcSmqw5toXRRXy0oSEyI4aWX70M7Q5pZ0wQ94zr/EjST
    JaZMGUnP+CjkM6VkZZnEhJg/JavQxhgurayojlyxfAeubib++a87GTmyL6qq8t57P/DqK9+w
    cuUuhg3rHatp4vN+/XrcOGxYL2XzlgPcftsLBAT4UFJSRVHRMabPGEP3bhHFmt2+XVUFelNz
    sawY759x3fjPgoM6dX3//SXs25dOYmIM9z1wLWNG9c8WQn5EZ/AodIbc5MJu3SIbJ1w+2P2b
    b9Zw/32vER4eSFVVPVlZRQwf3osBA+M0u1Ws1hulW1at2k1JSRV3330Vf3/8JgwGI8kp6dxz
    98usXbuXm26aEBEVGT7a7uDr8782etLUcHifydXralkm1M2d2IZ6879lWfaQZQkvb896bx/v
    RxrNWlZDvf0IRrnO55jgryUq5sYmNO3sxqVpgqbmlovAEDyVC0aTAWuLrY1yK5nMzCJefnk+
    Fov1NCNRkiSsNvv/HBeumTbK46svV2G3OVqj6gKDUceUqZf8RbhwuvP4e8S5Ti70BDi5sHLF
    DueB3JPOTiiKTHVVPat+xYWhQ+OVjRuTURT5xPcqSRKXjevfJheuu378ZyHBTi7s359OQmIM
    991/Zi6MGz/I/Zv5a9A00VrKKFBVrU0ulJZUnfKZZVlCVTXWrdvXwYUL5zB4ATrh6tqCr58n
    hw5mU1R0jG7dotuMxJaX1dDU1IKvtztGgwGEht3W3rf8fsBtaUx06O1RUSFyRkY+uXml9OgR
    Q1FRCYsWbiQ9vRCQMJubcXExkpDQBUlScGZZJAYN7ImXpxsmFyMPPjiNwsJyDhzIIj29gHXr
    97J5czLRMaHy5CtH9Js65ZI+4eERt8G4f4F5xcXRfvZk8QFcdxqN9lUPPThjek11PUt+2EJV
    VT2yLDNlykgefPA6FNllvaIzZhfkNVzUDkNwSCeuvvqSs4i2/rLpExK6kJAQ+7ve8+eIA8C/
    prZBX1trJjDAl5iYCJxZEwM9ekSi08kcK6umpcXWuaVFy/X377TomWfuuM7zzW/ZuvUAWdnF
    mEwGrrlmFI8+er3F1c319aoq2xEPDwVn3/ltO/S6LveOvXTQkqSkDK/du48S3yuKsWMGNuh0
    +nvAdUNxQSVhnX0Bw0qDwe3tRx65fo7dZmPFil0UFVXg6enGhAmDePSx6/Hv5PdDdZVju5+/
    PKuw8BhCCBISu2AwuAAa4eGBBAX7Uph/jOrqel1UVGdfvV497yvjuEEnhDXLYMjL0lT/fkiK
    uyQ5fyYhyUCpu4d+e1aema7Rw/lriQS4igB/X3R6BZvN/n/DSpHp5OfBefKUzx0XIoPl1NT8
    E4ahEKAJZ6nesOEJ2KxWNm5Mbi1D+dUuF4Lo6GDunT2V/Pyy/yku/Lhk64nypCsuH3JRc+F4
    t7zzLfKfEnE9lQtnct7Ky2tO54Lnt2zffoj6+iaMJgOTJg49ey7Enx0XVq7aQ3NzC57ubgwZ
    2pNHHz2dC5oQp1W0SZJEY2NzBxcunMPgChjzjUbzzv79eoxfvXI3S5duZcCAnnh5nZxpkKiv
    r2f58m20WK3069cDk8mtvLjIYpZl0Y6VQCXgrUDe1TpFll1dDaSn17Bz52G6d+vMe+8s5psF
    awkN9eeqq4dz6EAO9Q2Nx9NvvxiVwX74+XlR39DEiBF9CA0NoKmpiaysIjZs2M+qlbvIziri
    jX8vYN2aXfKdd189aML4oR8bjfbnwPd7IXLrJCn6IlCaAkgFmnqDPSE1LY+DB7IxmQzEJ0Rz
    5HAuBw5mkZVVSN/ePcfW1zYlBAXrD+WlJBPV5+KBnxDipDrcP7J+xR+Y1z8r6qcIX19vvL09
    KCuvIiengNDQTqiqldTUPBwODf9AP0xGQ3FDg/0g6B6IjglveOGF2dfk5hX7vfraAvbuTaVH
    XDQBAZ1yGs32r/Q6NJOpX+tnTwakvKamZmt2TgkA2VklmBsarT6+nQqEMBMeOaT1tYUC3L70
    8/O5PrZrRKRYsZORl/Tm0UevJyYqrM7Dw32xateeqK231/sHmkR4eCCSJHHwQBbjxg3GaDRS
    WFBOWWk1bu4u+Ph4oaqqRVXFBdoLe4FjsQaj/202W/OthUWlksHkPOyXX1Dq4efn8YnR5Ppl
    bFTEF0IkZ0OfizBy/Jt4XB0VHXpLRESgKfVo/m9GWoUQBAV3onu3qBZNZbWscNFwwWTSnzIO
    WZbo368H998/nWHDEvn0s5/YsCGpdd+fPgeuri6MGTOITp28/ne4cDAHIZyRX00THDyUfdFy
    QQjw8nKjW9fw87p/hRBkZBZhs9vP8W/+hQte3h6Ul9e0+Sr/AN8LwgVtxU6GDev121xoo+RL
    CIGbawcXLpjDIEmBCJFiBved48cPHv/Tsq2sWrWbwAAf/vb3G090SbJarbz33mLWrN1LfHwU
    EycNA4mvQsJ0hYrSt50qATuwA/CfYm6onfrGG99x8FAOQghWr9rJlKtHMvbS/nTy9+byy4cQ
    GxvOsmVbycsvJyoy+BSjUKdTcHU1cuxYDc2Nzhvi3dxc6d27O4mJXbnmmlH89NM2Fi3azMGD
    OTzx/z4gJTkj+L77pr3v7x80Egf3C0dqraSLa+dQyAB0g8H6xd69Sd2eeuJ98gvKmTXrSu68
    czIff7yMTz7+maee+pDnn7+rR/9+Pb9w2JkdFi/tFvnJSJHtGw6KIpOWVsBdd77E+bTrhIC0
    tIJT+lWfoxEByiZ/f5+Sy68YEvr2W4t4+qmPGTBwB83NFnbuOIKvrwcTJgxG0emPhIQcqxUO
    T7Ab7lZkl7Xx8d0/8PL06GSub2bVqh1MnTIy2sXkNliWxIpfnqEHXKPM5nKX/LwSZFmmqKiS
    +rpGLzc3r+FIIvuX1zYDckxdbb3/qlW7qa1pxNPTg8TEHoUtFvX+sgL7StVgU7t2HQakZU2Y
    MPDSpUu3sODb9RQWHnOeYUjKoKqyjltvnUBYqH+hpdmxSVyQgHU+EDoImr7Izsru/tGHP7Jp
    SwqVFXUAPPjA64wa3Sfs3runPBkTGzsJPCZDZtFfx1nwAuTk4KDAwmnXju76cs58bDZHm+AT
    QqA36Jk+bTQhoYGFjU3W5PZ6+rstLhw+nHsi0qtpguiYEF599X4iI8MoLiknN7e0tflBW0pD
    oq6+ibLSY3Tq5IGbmzu9e/f4S3Ph6Sc/ICe7+MScybJEbm4pTz/9Ef/6150XHRdUVaNHQmc+
    /WROG/ck/GnWF5qmcsesF0lKyfzzuHD54NDsrKITZUDONa7h6+fZrrmwbNkWyspqWrsq/eLI
    jx3bt4MLF8phEOIA4N4FHAHh4UHigQeukZ586mMWL97MuPGDGTSoFwCHD+ewaOEmfHzcmT37
    GqKiQjVNkwPNdXQ/ciAz09VN06Jju7czNVAMdB6gqg3vfP75T74LF24gOioEECQnZ7Fz5xEm
    TR7JpZcNwtnOQXDttLEnNvPJDoPJZCAhMRYfH0+8fDz4pdzEebFdZGQYDzwwg7FjB/LBB0tY
    vWo3X36xiqKiCunJJ267ITomgpYWx/0WS3Kti0t7VZ5pgDZYCPWLzZv3dXv+X59SUFjBrFmT
    efDB6bi7u3L//dNosVqZ//VanpjzPnOfndVn2LA+X0iSfDudf9wpRB2S5N1OyefsYFFTU8/6
    9ftQFBmdXjnj2YVzIbIsoTo0HA5nq1WTi+EcqzgBuB+V5ea3br114j9raxuMPy7ZwsLvNyDJ
    Ej6+Hsy+ZwrDh/WubmrSPm9uDkbS9cRmK0SnuJVIUoPxeN11RnohR45kuw4fMWDIs0+ZVzQ0
    JOHp2a/1GfbxFRW1HpWVdciyMy1cWHzMEBkd0RNg86adXDIqDAgHMoYePJTpltHaz9tpX6qZ
    FRWWDUiokWHDECIHkD/t3afnNQ89PD3grf8sYt26faiqhru7iWunjebe2dfYDAbjR+ZKLdft
    PC8pIXIBvREaHzuamtb9H4+/w9EjOURHh9Cje2eEEOTmlrFo4SayMgp56eX7e/WI6/LQsWOO
    J2trk60+Phf3BV7O8UsuYE0EWZo6dRSHj+SwbNl2xGm9552p+Msu68+06WMxGPSS0Ehsqlcr
    aiuTLT7+fds1FxYv3oSqaqe0erRYbBw5ksuaNXtYvHjTiUOvbQUZJEmivKyKxx79D0OG9uLy
    y4eQmNgVF1eXvyQX5j3/GdnZJaeV0iiyTHpaPk9JD/46AAAgAElEQVQ+8QHPzL39ouKCJEFO
    TikbNiZx+RXD0ev1f/pj7XYbq1btIye31GkQ/8lcWPrjVszmZiRJwsfPk3vvvqpdc+HBh6YH
    vP3WIioq6k5w4fIrhnRw4UI4DEKYgY8Bw3BNa/4kMyO727JlW0lOyaS5yQqI1ppVp+HcbLE6
    r/+W4IsvVnLoUI585VUjHuzeNWZyz0THTDi6VQgzkuTRToBXD9gUqLg7OflI4Ndfr8XLy51n
    5t6O3e7gyy9XERDo2zo+mdPrzU/9f5PJxOOP34TD4cDHx5PTt7eGJEHPnjG88MJsunaN4MMP
    l7Ju7T6aGlt4ft7dN8TEdKa+Xr2/tja5tj0ZFELkAYor2P9mt9Xc9uPSjZH/fs15i+edd17J
    /fdPw83NWV/u6enO3/52E64uJj77dAWP//1d/v73G7tNmjzyO73+5s+g/GUhUi2S1J4iZi6A
    6oiKCmPMmH6sW7sXm00lMNiXu+66ktjYcDRNcC6rhiRAViSyMgv55JPllBRXodPpGDO6P9FR
    oWiqwyEr//0DJSkGIdI1ML7u6+sr/vGPW5+YPGmYd05uKRUVNSxfsYPS0mpUVXPodJJNkpzP
    1OsNAEaQTM5Wes4ON+vW7WfQ4MSr/jHH9IaqatXO2zb3A366vNxSLC02QkL9qKttIi+3lJEj
    cY8MTmH91u6te6IoSlNbZmzelILZ7MzEOWx2bFZ7mIurEqhpWl5rTA8QNofD4SgpriQ01Jc7
    77oSF5ORzpGB9OrVtcXd3f2l6irb63oPSXX1HPBf6gKhQFVPUDuBMgUcYc52d7IExjKw/QBK
    FfgcBRxQCqgjzOaay//z5rekpeVx+6zJ3H77RAIDfZ1gyCnhvfeWsGzZNt7/YIk874XZt/v6
    us0H9YCz5EXSQU0cKANATAQrzlalMmAQoKwAKQ3cMkCqkaSgdqIPkgCrO4jnGhrq7t+0aY9h
    /bp9pKcXIiGdpvmkVqM5Pb2QZ57+mEsvGxA7ZszARb6BPu+Afq4QyY3tpVb/11yY/8067HbH
    CePXWaoIJcWVPPro26gOB6qq/Wad+fHoZ2ZWMRmZxSxbto0hQ3tx/Q3jGDIoHr3B8Jfhwhuv
    L6S0tOq0y8d+CZDIZGQU8f/+8R6PPXbDRcOFsWP7s2b1Hp568iOOHs1j2ND4P/3p23cc5ttv
    19Pc3ML48YPaJRdkGerrm36bC3mlWCxWJEmiudlKXt7/xQWn03I2XAgJ8eWOO9s3F26bNZk7
    bp9IULAfQgiys0p4770fLhgX/qQMQx4wYazVav742+/WRn3y8U8UFh7D08OVgEBfRl3Sh96J
    sRxPK/dOjGXGjDFs3XqA9LR89u5JZcXyndx511VR11837hujqe+rcPNbzbXVuPr4tQNlUAUw
    sMXafPWXX67g2LEa7rz7KoYNS0SWZYYPT0SnKKSmZVFSVMngIfF4eLhxZj9ftP4cfjsWoOHh
    4co990zF28eD119bwM6dh3n++c958cX7bujk5990rNhxX2lhsj0kom87gEIS0OIOPFdXV/3Q
    558tVT755GcMBj1/f/xGbrrpclxdTZx8nsPTw52HH7qeAH9f3nlnMU8//TGFheXhM2+78ikv
    L18PUNqVkQC+gLw9MDB04Qvz7r28S0yox+efr6K8tJpVK3eT+I9Y+vWP51x3tUhJSWPVyr2U
    llTj6enKzNsmcvttk81eXp5v2yzqlnN11k2SumOuS3bYbOJNV3eXnf36JVzSr3/CjWWl5XFL
    f9zKokWbGDa8d+Bllw2+t7TU8kBp6U6LU0nJ2O3OHtIuLkYMBh3btx+ktORYTOfOYZcgaUuc
    t8D3CIKaMbl5pSBgxPDerF69h7y8Uuw2+/CjmX2DhVDLoAWQx5SUHeuybdsB3NxdcdhV7HY7
    DofqJkmKi3PNlQOyJ1Q/vHfP4ZD589cQEODNpInD8e/kn4skvrfbHJubai1b3Eyy1dVr4B9c
    2zWAKkH1VGi5ubm5YXBJyTHfY+U1usrKWkl1aBiMegICfERAoM8dQUH+da6uLbvAuBrct4L1
    urS0XPedOw6TmNCFe+6Zir+/T6tOlOjWPYbH/nYDqal5bNlygOysIp/evbtf1dSEWa+vmwDW
    8S2WpiFl5ZU+JSUVSllpNebGZnQ6heAgP0JCA67s3DlEc3e37wX9v4XIWwbeQpJ8LqA+OArI
    7uB4JSMj+67XX/9W2bIlBUuztbWVpHx6lL3VYM7NKSEnu5iNG5MYMXK34bHHrn+gR4/ucaid
    HhBqXrakRLUvLny1gmPlNSeMX03TcHExEp8Qg6uLkT27j2LXxFkfSj3+uvr6Jlau2MWuXUeY
    NHEos26fSFRMxEXPhc8+W4HZ3HxGZ+FEpkGRKS6u5NlnP7nouPDZ5yv58MOlfP75L5U35zID
    ffJastnsuLu7Mvu+qe2SC3ar/QQTt+84dEYu5OWWYbHYkGUZu91Bfl7Zmbmw/eCJz+dwOP4S
    XLj3nqn4+3tTVlaJpgm6dY+8oFzQnXtlcBAwDbXbG97/7NOfov7z1iL0OoWbbxrPxEnDiIwM
    ISDAG51Of8JQ9PR054knb+POu2rJzytl+fJd/PTzNl5+6SsszZawWbOufkQWz+138S7a2T6U
    QSOgDygtOeZ38EA2YWH+TLt2DIrinM6aGjNffvkzy5Ztx2y28PbbjzJ69EBOvXuirTTf2aUD
    9XodN94wHlmCF174is2bknnnnYU8+cRt04IiXNbodOKH2tpkLmREyZleMwwF89zs7KzRr//7
    W2XV6l2EhwXw+OM3cfkVw1rhIE6KJTozMSYXI7fOnERIaCdeevlr/vOfRWRkFCqPPDLjgdiu
    3eIRfs8Jkb9TkiIv6CqQJB+EsACmEqi+0dur5eb77pvxRnRMmNebb37Pnt1Huf/+f/PQw9OZ
    OHH4OTljoGka69ft5c03vycnp4TY2DAeeng6E8YPqzfqjX8rLmn+Qq9XHEEhA8/ZOD28+wLY
    k5I27ejbN3BHfb1ln6eX+0JfP0/vzMwivvhiBb37dL0uMNBntSw7FrdGcpyX0GkCb293eiXE
    sG3rQfbsTXUJjwibWVfLSm/vmhZZNrhZWxrCcnOKcXUxMnJEb7ZtO0RubikWS3OQp5e3u9AA
    TCZomrh3zxEKCo8xfFgCmRmFxzM3J22eSsA0saGh9pbPP19OdXU9sbGhuLmaypoa1Ru2r96/
    p/eQHoREjvgv1nY6UB0IPNPYWHXdpk37fX/+aQsHD+VQXdVwihGg0ymSf4C3vmtsuP+QoQlX
    jh7Vd2LnyPAag8FgO3I0j8ZGCz3jY/D19TjFcQaNkBB/+vXrzqJFm8jOLqZPn7jbZbl5Zlpq
    Xvj2HQeVPbuPkpFRyLFjtTgcDjTNGZGWZRkvb3ddQkIMU6eOHj5mdL8e7h5+l0LFP4VIq5Ck
    HhcooqxzAfNzmRk5d82Z876StD8DWZbOal8cN4ZsNjvr1u6lqqpO/9JLsyd07xbzUFWZ/fH2
    UZ70CxcOH8w54fyomka3rhHcdfdVjBs3EKvVxt13vUxSUubvbql5vA1nfV0j38xfS1JSOo/9
    7UbGjhlwcXLh9W+VtWv2tJlpOd48QpJOvWNAlmUaGpp5++3FFyUX8nLLkCSJ0DB/Bg6MQ6/X
    /VeNKyRJwu5wsH9fKsWFlWhCEBMT0q65cHIb8YKCsjNzIbe4dQ04dVteXtmZuZBf1toeVbRh
    VF2cXOjUyYtduw7xwrwvsNtV5sy5mZEj+54VF8qP1eKwO050XzsXXNCdeygortD4xKbN+2M/
    +OBHXF2M/L85NzNlyujWlFRbN9MKpwcU7E9wcCD9B/Skd+8uvPjiV3zwwVJiY8Mjx44d8nhl
    pXZ9bW2ypazYRFyvuAuk8BqAo4CvqbKqjuqaeoYNjadLTKgzt5JXwty5H7Nt6wEEMH78QGJj
    wzi3VYTOyNS0ay8lL7eUTz9dzpIfNhMXF+V93XUTnrA2i50uelF24aCQD0hdoPHT/fuSuj/3
    z085dDCX/v268cSTt9Cvf8/T9rPdbkev1534d0WRmTBhKMHBfrz80lesXLmLwsJy/dy5d4wb
    MDBRIORpmnrILCsJFxgOzoP75upkh9Vi+8rNWym86qpRj0dHh4579ZX5bN12kOef/5KfftqB
    Xqf7r1aBhPNWz6NHc6mtNXPJqD48/vhN9OoVu8nazMtVVeo6b3ej5uH35xgE/fqNpsVyEA93
    /R5VU/NCQv37KIrM3r2prFi+3W3mrZMeaWmWNxlc1OqT4S9JEqNG9yE5KYP16/cxceKwBA9X
    1wihqZnIhrAGc6MxN7cUHz8vunbvTGCQL/kF5TSYm0yurl6dZUXKkrBGWCwNfdet34ebq5EJ
    4waQk118yhoyGHSAwQta7vn55y26rdsOOPvWB3XCYDJUNDWqWX1GJhAcOvi/WNs5AAHQ8m5e
    Xv41b765gDWr99JssRIS3ImBA+MIDfXD5GrEarFRWlZDWWkle/amsWlTMp9/9rMybHiC/8Qr
    hlJX23AChGcyBPQGpzFRUVHNmtVbI1as2Mnu3UepqKjFzc1EcHAnBg/uSUCgLy6uBiwWO2Ul
    lRTkl7F5UzK7dh5h/PhBfo88cv19UdHh/jar435rS3Kl0XS+jUYzYBrdYK67983/fKckJ2Wg
    yPLvTrwdN5gPpGTynze/48WX7p/l1clzlSypKy9s1PxULtTUNiBJEpqm0b9/D/71rzuIi+uC
    1Wrl5+U7KT9W+191NpEkCSRITS3gqSc/xP6sncsvH3ZRceGf//yMQ4dyOd4y8tfOgsGgx9vb
    nfr6RqxW+ymvkSRn3/yVK3dTWHhMP3furIuKC9u3H6LR3ER0VDAzZ07Ezd3lD5WtShI0NVn4
    8ssVbNywHwGMGJl4UXDhhLNjc7B+w/42uZCXV3biMLUkSRScgQvr1+/DZrO3GXy4mLkgSRKH
    Dudw6HAumqpx6FAOl4zq3yYX9uw+SnlFLQa9jsBAX+LjowgM8MHVzQVLs5Xy8hqKSyrYuGE/
    u3cdYdy438+Fc5xhaAQMCWZz7ehvF6ylwWzh8cdv4Nprx7ZGD/6vvrYCUNHrdVxz7Rhqaht4
    5eX5LFiwhgED48e5uLiNdXevWV5WHHyBo0iDZcgf19zcgqXZiourKzq9jvp6M6+++g1bt6Sc
    uK1x/PhBhIWFnMXYT76M5WzSlAKD0cAdd15FyoEs9u5J5aMPl5KYGNu3R/fYZ9MPtdyXfmS3
    o3v84PMMhWOA3gUq7ks9erT7k099SHpaIeMnDOKJObcQGRX2q/HJbN+ewnffrmP6jLGMHNnv
    JKcSEhO78+prDzJv3hesXLGLp57+kDfeePiyuLjYv7/9ryPPFGRvp3OXC9+jvlUZqy0tKRvs
    djLi47q9+cqrD0z46MMf3b77bj07th/Cw9O11Sn6I26DhN3uwNzgvMvj5lsmcN9911gC/ANX
    26zaQ4qBIv/wP98INJoEINklWecICQ5AkiRsVjvfzF/DiBG9h0RFht8jy/p5p2RFhEbX2Aj6
    9etG0v50srOKOycmdrsKyfAqMKm2xuxRWlpN337d8A/woUtMKFmZhZSVVbmHBIeMQ1Ssh4hr
    MzILO+/dk0rv3rHE9YxG+xVhDQYAdUpBQdGwr79ejbXFhixLhId2Qq9T7B4estVkkv+LtV0C
    mIxQ8++8vIJrnnzyfbZtPUh4eCD3zBjN+PGDiYwMwWDQnbiYyOFQqa1tICurmN27D7Nl8wF+
    XLKFFct34uPrhSRJpB7NxWxuwtvb85TnNTVZyMoqAiGY//VqamrMOBwOunfvzPRpoxk8JJ7Y
    2Ai8fTxQdApy6zPtdgclxZWsW7eXBd+u48cft1BVVce8F+6ZHhEWZq0s1+6qKk9u6RTU9zzp
    BAFMAebdsHvXQZfNmw6cMHj/G4N5y5YUdu865DLh8uE3QJeV27duZvjIUe2GC0IIOncO5umn
    byMurgvHz+st/XErxUUV6HTKWc2dpgknHdq4zVdRZMrKqnnllW+IjAymR4+Yi4ILTz31IWlp
    hWcsQdKEYPKVw5k1axJffrGC777bcNrYnZkHOHw4l6ef/pDXLzIufPvtOr74YiVDhibSv388
    f6zzl0xGZhaff7aSFouNm24e3264YLf9NhdO/h4PJGe0yYWysuoTdpEkSdTUNLTJhaT96Wd0
    wE/ngv0CcUGPrDhPoDscKjU19WRmFv0mFxoaGonvGY1na8l6n77daGpqJiurCCEE33y9mppa
    My0tdqIig7jppvEMH5FI924RBAd3Qq/XtQYuBC0tNnLzSlizeg8//LD5D3HhHDsM+UDPSdlZ
    ha5J+9OJj4vkmqmjW52F32MkOSPoU6eMYvnPO9i/P52c7EKXuO7dZqz9OWhVeBTqhVMBZsCu
    QEvssbIqNE3g7u4CSKxfv4/16/c509CqhpvJhI/P2R3UVlWV/LwSDAYDYeEBZxl90ggK8mf2
    7GvIyS4hP6+Mjz/6kefnzb4mNt7ta0UW2xf9sJFp14w5j/NTA+j6NDXV3/3++0tITc1n7Jj+
    zJ17B6GhAW0oRZm9e1JZuHAjUdGhjBw54Fev0QgNDeLpp2/H3NjClk1JfPTRMnnevHtmPvh0
    3PdCaEdpR2Iy9cEqdhdXlCg3BYcG3Xnv7GveSEnJVDKzinn88ZtITIxB1X6/w6DIEgcP5vDS
    S1/TtUsYDzwwXQsM9H+uorTlHXcvtcnkMuR8IRAwIUlmwkL9kGWZgEBfcnJKmD9/jfTEnFvv
    tVurlxldO6m/KFTw8HRj+IjebNyYxNZtB4iP7zLFXN/8rrefpssvKKWpyUJYmD/eXu6EhvjT
    bLGSn1/KgAGJBiF6uEP9jB3bD1Jb08DQYYl4ermfksbXhKa6u/v3VNWGud99t1bJzCgkOMSf
    imM1hIX7I4Rsa2oUqvPc3R+NjoYCqVOrq6uufOWVr9m+7SA946N55pnbGTqkF847dU7NniqK
    jqAgf4KCAhk+vDe33jqJ1at3sWDBOtLT8gHIyCgk5UAmo0edXLYok5qWT0Z6IZoQlJfX0LNn
    FDfcOJ6xY/sTEODbevmjdlq2TlF0xHSJIKZLOIMGx/PcPz9l+/aDvPLKfObNu/dq/0Dv1YqB
    BTUVyfgGnA+noQR4L1bTKgdvWL+Ppqbm/7o8T5IkmppaWL9+L2NG9x9st6bHJvYyZbUXLqiq
    hqIoXHfDZfTu3fX4N4Ol2UpTk+WMt7j/2lnQ6RSiokNQHSoFBcdQVbVNpyEvt5SPP/6JefPu
    xcXF2O658FvOghAgSxI94iLp0SOO2G5HTrkZ+DTdqMikpRVelFxIzyhk29YUKiuqz/rG81Po
    KTsbAjQ1t9Cje+d2wwXnZ5PJzi4+Ixd+GYNMZWUdW7efzoXm5pZTMrCWM3ChsrIOWZLbcDxP
    5UJGegGKIqNp2nnkgjNI2NJioaG6EbvNgaJT8PJyZ8SI/owY0ZeZMyexatXpXEhOyWDggDgS
    EmMAib59u3H4SC4Z6YUIISgprSYqKphp145m8uThhEcEI8kKFkszdXWNOBwqeoMeL09X3N3d
    SOjVnV7xXRgxPIF//esLtm8/9Lu4cI4dBglo7paZWYTZbGHAwDg6+fvwRy+v6uTvzaDBPTl6
    NI/M7CL69uupH39lnJqalkZqWipxPS5EWZIEWBNaWsyx27YdQJYleveOBZwRL7vNzozrx6LX
    6YmMCmbAgLiziBzIHCuvYPbs1/D28eD99x/H19frLOdNMHJkHyZNGsrXX69h3fr9XHpZkt+k
    SSNnFRfZdg8b5OY4v/NTBgRNOnAg1WXz5mQiwgN45NEZbTgL8kl/S85WaCe6Ssm/yrRoBAcH
    8NCD08hIy2fTxv2kpeWF9+/f60ZJ4gnamRilwQhxuAX0+w16vcOg1yuKIhPbNYLExJ5nkW1q
    02Wg2WJHUWQMBj16nc4O0lbfAF2TXj/wPI7ODQi0Qu7OoCDfAXq9wuBBPTh0OJcfl2xm7Nh+
    oUOH9rlPE46lvw4hDx+eQECgH+vX7+W6GZf2cnfzuwK0wJycEuwOB1HRIYCBsIgAJCTycksR
    whHX0lQ+ubmlofOGDfsI8Pdm5MjEU4wLCcnLoMg+YJmenJQauWjhRkJD/RkypCfLftpBQGAn
    JFne/t472dbb7gj5g+O2Adlh0PTUkh82eK5ds4fgEH+efPJWhg5N5Pi5g7azpgAqkgQBAb7c
    cstERo7szbvv/sCypduor29iy+YULhnZ90Qdt6raWbN6NzU1ZlxcDEyZcgmz77uG8PDgk5wS
    9Yw64fhz+/TpznPP3sFDD73B2jV7GTgwzuOWmyfNaaiVtig6Ss7PmqkF9J1qa+vCsrJLOHcN
    ACSyc0qpNzeG+fsHdBKa/QI6DKdyQVU1fHw9GTqkF2ZzM4sWrcdoNJCSkkF2VlGbF0e15TAE
    Bfvx7rt/QxOCO2a9QFHhsTaNZkmSWLtmD+PGDWLChKHtmgtbtqacKDNpK5sCovU/DdAQaKia
    2soHZ+b+13MgyxKbNydfVFwwGvSKpdnK++8vaXM+zlY0TWC12jEa2xEXDDocdhUhYNmyrb/J
    heNj2LB+H9dNP5kLzg5JJ6/xlhYb+Xllp3Bh48Z9aJo45TxQW1xYvHhTa9mXQKfXnQcu9EZV
    HaSnZbJh036OHsmjpLgCi6UFvUFPaGgnEhNjGTduELGxkdxyy+TTuLB5UzKjLunL1VePRFM1
    XFxMrF61i9paM4oiM3Zsfx5+5Dri4qJxODRSU7NZu3Yvhw7lUFZWjc1mx8XVhdBQP/r06c64
    cQOJjg5l8JDePPvsLB555K3fxYVz7DCEAg6qqusRQiM8PABZVvijl+xIkkJosC+aplFb29i6
    EH4gLf1CnV8oABQZ6m9PSUkL2rHjMN26RTB0qPM+iWnTRtG9ewTTZ1yKr48HkiS3KjZxFr/b
    2QHDYrHS0mLj13c1/JbDoNfruf6GcWzanExhwTG+/24dw4cnTg4O9hqMJLZ/+vYKZj0w8TzM
    T2VrqKFsyPbtB6mrb2TatDH06hV72hrIzCygIK8URZHJzS0+8ffG9TtwqBphYf70iIs6CQyC
    Pn26MW7cIL7+ehW7dh2hf/+e/fMLmlzy8ndYoiKHtTM8KICCOPl7FMeNyj+yHyQ46fZoIUDT
    wOE43+NyB/Zq4JHk6+eFi4sRnV7HjTeM44UXvuKzz5YTFxdznZenzoH45ZSXpmlER4cyZHAc
    q1bv5uChTPdLLx16rbWlZVR+Xikmo4GIsEBAEB0VjLu7idzcUpqbLH3d3N3r9uzP8UpLy2fU
    JX2JiQmjuKjiZEXh4uJimNhQXzfts89+pqqqgX/8YzLVNQ2YjHp8fDyRJCV9ztMxIi2t+Q+s
    6wpgPdB/Rk5OUdy3361D0zRuvmkcw4clcvqZrN/er+C8W+Xpp2dhMOhZ8M1aNm9K5sYbxhHb
    NRKA3Jxi1q/bi04nc9PNE3jkketxc3P9A2tHIzGxGzfdNIEX5n3J99+uY+yY/vFhYaGXIrQv
    v1qwgltu+LN1g7OtX1NzMw0NTefsUkNJgsZGC5YWGyAjybQLLuzadRRZljGbm/nii+V4erqy
    8PuNOBwqDod6wvg5Ky2ik/HycsdkMuLmduY6d0mSMJstLFiwhiFDeuHl5d5uudBQ39TmAWe9
    XsfAgT0IDPJFCOga2xlQ6RrbmWuvHYMkQVVVPXt2H23zTEN9XeNFxgWnPrDbz4USFycQ0x64
    4OnhSnW1GVmWqK6q5/PPVrTJhZMdvqzMwlO4UJBfiqpqp5Xu5eaWnMKFzIzCNhwu8SsuLKfi
    WB2KIiOEsyrkT+XC8N6oqp3vv1/Hu+8sobS0svVg/y+fM/VoPhs3JPP99xuZOvUSbrhxPJGR
    4adxIevGIi6/fAgOh0p2dhHr1+9FCLjiiiHMnTuLTv7+FBWVMn/+Kn5ato3y8prTnnX4UDbr
    1+1n0cINPPzwDCZPHs6AgT25deYVvPjC2XPhHKtYZwTAaNKDJNHUbOW/O+wrsFqdt3+aDMcv
    O4kmrkfcec8uOFNPaYB5XKO5btrXX6/GbLZw3XWXEhYWCGiMGNGXe++9hk5+3siy0grGsxt/
    U5MFh8OBf4A3Li6G3zlvGnE9ohg3bhCyLJOcnMnuXYf9dHppVlmJXXfFNf7naZbqAXNCc3PL
    oKNH8nAxGRkyLAFZ1p2yRszmZl568Svunf0q985+ldWrd6PX61i9ejf3zn6N+2a/xrPPffL/
    2Xvv8KjKvP//dc6Z3pPMpIcWCL13xIZKEVRULFhA197ddZ91v4/bXLc9gnXXta/YC/aCFKVI
    b6GTQCAJSUidyWRmMn3OOb8/JglEQRFCzP4uz3XNFZ0wMzn33Pf9+rT7/cHr9R8VjVSRJC2T
    Jo3AYNCzY3sJoWB4ZIbTONCVoufnq5PiqIKlJZokJFLTHNhsZqoPNzBt2jjOOGMQ36zaxpLF
    6xyilLhFVuJ6uZUNKoiijrPOGkEiLrNiRSGJRPzScCScXl5eg8ViJDcvuY4yM9OwWE2UllUT
    DIVSVTVxycoVhUQjcc48axharSFZngDE4jKqquh0Bv5n6bKNucu/3srgIb246KIzqayox2wx
    kJJiAdQIKAw4qW3DD0y0gXzthg27KCutJi8vnQunn4EgSie5xyX7jtxzz5WMnzCI8vIaFn25
    rs2hXLxkA5WV9Uw6byR33TXrJJ2FI9cFF4yhR48sSg5UsXlzEYLINLdX1c2c0VnnwZLlNTpt
    x8aotFoJTYd3OD95Lrzx+mI8Hj+iKJCIJ/joo294/bUlRCIxEgn5O4o/PxREspiNaLUSfl8z
    zYHQ9zpboiiweVMRa9Zsb9k3uyYXjlV6o6oqGRmp/PVvdzB//r3Mm38PZ04cCiQ4c+JQ5s2/
    h/nz7+XRR+8mr1vGMd9DlhV27Pjv40LrnDiVR9fjwpGSUVEUWL1m+7G5cNQYNDeH23PhUM0x
    M0ll5bXtuOD3h7/z777NhZUrC9sMaFVVsbFFuPUAACAASURBVFlNp5cLgoZ1a3cwf95bVFXV
    AwIajYQoim2PpCMkcLiqnn/9631+/cDTFBeXYrPZ2nNh0VosFhMOh43Fi9dTWVHP6NH9+M1v
    5uB0udixo5hf3v8kLzz/CTU1jcf9LFVVOXjwMH/72wIKtxYBElOmjKVnr2xKSk6MCx3sMOwB
    tHTvlolGEtm/7xDxeJyTS0ELxONxilo69+XkulqiKz8VHMJAz3RI/PmLL9a4vlq2mTPPGsLM
    S89pB8UjTsKPMyK2bd+P1xugT5887HbLjx8tUWL69Ak4nXaam8N88MEKAoHARWku3VizRdNJ
    YxQAwhpvY6OmpsaNy5VCzx5Z3xkLg0HL2WcPY8rUsUyeOpZevXLaItCTp45h8tQxnDdpVEuP
    hvZGR163TJxOO1VVdTT5/FajWWc2W0R+vjrzsgHmcFpqatyRYqW2thEVgRtvnIHBoOPVVxdR
    WVmjlxWVaExut/rHjO1Pjx6ZrFu7k9raBp0/0CwcqqgjIzMNV7oDULHaLOTmZlBX14jb3STU
    1rp1q9ckD5GdMWEwR0vxxhMKCFBbW69/9dUvEESBX/xiOharkdpaN3abGYfDRiQiy8EgwIST
    nNc+IR4P6HfsOEAslqB371yys5ynGBBRyMx0cvfdV5CaZuezT9dSWVlHTY2Hzz9fR3q6g9tv
    v4yUFPspOQugkpmZSs9eWUQicXbuOoAsxwalpYomvU7thPmiBzQ47DbS01NOqlb7mKOnqLic
    DiwWkywrCSUWU3+CtdCeC8u/3tpO1SV5EJ2TMuwEAcaM7k9Kio2i4jK83sD3vkdrc6tPP1lN
    OBwGhC7Hhbq6xmOW3wgIBJpDLF++mdWrtrF61TYOH24ARA4fbmh7buXyLfiamo9bllV9uOFn
    LnQBLtgdlraMuCAIRCNxXnvty3ZcONb3t2H97jYuVFbUH/Oge/1RXFizducxw9btubCIUEuv
    lzaHwWY6rVxQ1TiLvlyP2+M7qvP0sde4KIqgwprVO/nzn/9DVVUNmZnpx+CCm88/X4fNZuau
    u2eR1y2bAwfK+PPDL7F5c1GbQ/V9qntJkYRGvl6xDZDJzk6joCCXSPTEuNDBKyoPMKzu0ztP
    zcpKY+PGvezdU3rSDkPJ/go2bdpLdraTPn26KZGosvqniSKVAgYrJB7es7dk1PPPf4zNZuaO
    2y9rUTY5NbFMr9fHp5+uQRRFxo8bjChqT+I9VQYOzG85T6GyZcs+9uwpTTOZpDFWq4TXu6VT
    oohJ7z5BNJpAr9ccI1uiotXquH7OhTz++H08+cT9nH/BaGKxBOdfMJonn7ifxx+/j5tvmYnB
    YPjOa81mI3qDnlA4RjSWSM6tn7nQyZcIKKsNBmlPdraThoYm3O4mJp45nKlTx1NUdIi33lx8
    jFS7Qk6OizMmDqGioo51a3dxuKoen6+Z3FwXRkNyrhiNOnp0zyAcilFZUceGjXsoL6tl9Oj+
    LVkItR0cEgmZt99ays4dBznn3BFMnXoGjY1+3G4frvQULBZTeTAY2xUMxk9hXgtJZQtPEwDp
    GSkYjEZOXS5ZZcyYAUybOpbS0mrWrt3J+vU7OHjwMNNnTGw5MHvqn6HX60lNs4EAdXUeEgkZ
    VRUEWemM+eIAbOUmk3XPsOG9k3KqHTELRZHhwwuwWMyFNTXhkgZ3vFNXwbe58MKLnxCORI9h
    5JyMM6SQkZ7K9OkTURSV5csLaW4O/aDTIYoCW7YUs2dvK3e7GBdiieMEvJIlRf/4++vceed8
    7rprPstXbAE0LF+xhbvums+dd87nj396Gbfbd2ynQ+BnLnQRLmRmprYz+yRJpLj4KC6oxzZo
    23HhOOWL4fARLlQcqvvOXFC/xYXdu0rb7TmqCmkux2nlQjQS40BJ1Qlv3a2qmuvW7uKZf39A
    JBJi7He4sJMDJZVcMHk0EycOxe/388QTb7Nly76WppfCCf/dhypqkWUZjUZHZqYTURROiAsd
    vKRcQOKjvG5Zu6dPP4Pa2kb+9c8PqK/z0Hq49UQMaBBpaPDw9D8XUlPjZsb0CWRlpjdFgvKa
    Rnesk6FQDISAyBWNjZ5bn3ziHaG8vJbrrpvK6DEDT9lZUBSF9977mo0b9zJ0WAETzxxyktFE
    FaPRwHnnjUKn09LUFOCbb7Yjy/ErmgMJi6B2XvRNEEVESSQcidHcHD7G955UwdLr9UiSHklM
    qhZIoogk6VueP1apR1IZJRqJotfrTkGi9Ofr1K4GwBeUJDGamZlCKBShrtaDVmtgzpypZGWl
    8cEHK9i6tehbtcoCoOXss0ag1Wr56uvNbN9eQjQSo3d+LgaDiWQ61UD37plEozF27jzINyu3
    IUki5543CknStp9Hksj27ft4b+Fy0tLszJlzIUajGY/bh9vtIyvLiUYjuiVRrErEOmRyIwic
    kCTmia5bjUbLRRdPxGIxsnJlIStWbMNmNXHJxWe1NIPsiDmeTFMLHDkKI4rQGdUMgpADHK4B
    8c0pU8ar3bpnnHKHW0VR6N4tnSlTx6qiKH2Um6NvzM3pPKnQb3PhqSffEQ4eOHzCnZu//71b
    mnNeP4WRI/uxdUsRX3+1BeEEDmkIgoDH42fliq20Zrq7FBe+Z3wEQSAWSxAMRQiGIi3nPZIG
    Wetz0ej3G3d6vfZnLnQRLnyb+6oKH364KsmFYyhkJb//OF8tP8KFY+1jkUiMnbsO8s2qbUQi
    sWMayq1cWPj+8uR+J7T/nMyMtNPKhUQiQSwW/5H7ZDI78OnHq1i8eD2SRs9FF5+JuYULK1cU
    YjTouezSs9Fq9Xz6ySq+Wrb1e7MKx95fIBIOt60RvU6DIJwYFzrUYUh2igtWSpLptWuvm6aO
    HN2XZV9t4qGHnmP7tuK2DeD7oCbLMnt2H+D3f3iBxV9uYOSofsy+Zqqs1Wn/tmF10y53fayT
    F4EI6EclEuGHFiz4TPxq2SYmTBjE9ddPPQm52PbvqygKn372DS++8DECcPnl55CWlnpKm92Y
    sQPJzEpDUVTWrd1Jo6epn8EojdQbOyPcYgL02KwWUlOsNDb6qaysO853rrY98rqlk5+fQ06O
    q93zx5ofVVW1eDx+0lKtWM2mOLIcT0R/3qo799IDZkRRR1aWE1lWKS2rAWQGD+nNrCsm4Xb7
    efnlz/E1BUAQUBEIh0N4m7wMGpzPoEG92Lq1mKVLNyGKIq50B0uWrOMff3+Fzz5bQXpGCkaj
    nhUrC1m9egcFBXmMHtWPeDyKorTsI4JAwN/MK//5nLraRmbNOpcxYwYAMpWVtcTjCXJyXGgk
    SXZYpXhmhvakje3WDuvp6SmoqkqzPwxqR6k7qwwe3JsRI/uyevV2Vq4oZOTIvhT0zesww0dV
    ZZoDYVRVJd2VgkYjqaIoyDpdZ9U/WwDpvf79e+/6xc0XYTDqT7o0SVFUDEYdN/7iIvr17bUl
    nlDe7vxypPZcSBr0HTeWRqOegQN7UVvn5tlnP6C+vvGElXRUVWHTpiL8vkDb3ttVuJDiMH9v
    V+PW0i1VhSZvAFkO4fcFW1VvvneMFUUlNeVnLnQVLhzr/EF9vZeXX/6cpqbAccrKRLYX7mfZ
    0k1tZ37a/z7pUK9YsY21a3Ye0wEVBIGAP8gr//mc2prvrhtB4PRyARmDQU96ZtpJ2NDJsxxv
    vrEUr9fLkCG9GTWqH6tX72DlykIGDOrJ4CF9qKmp4513lh3XYfqhz8jOdrXZrz5/EEU5MS6c
    hiLGNEB8Ky+v+8V/+tNNZ/7+dy+wdOlGduw8wJ13Xsb11087bvRYUWTefmsxzz77EYcPuxk2
    vIA//PEm0jMy1tTURt4cOtai5GR3ZhSpEtBYwXvvhg07er3x+mLsdjM333IJac4UfnwmIHko
    XFVVamvr+eDDFbzyyhe4G5rQ63UtXvepQEelW7cMBgzoQcWhWkpLq9lfUpFyxhmuIRqNZtWB
    g2vpnX86VSNsgHAoNTWlZMiQ3gM2by5i5crtnHvuqON85wAKF198FmecMYS0tO+r1RZQ5Dir
    VhYSDIYp6NsNm81c6K6P7lV/DiZ18mUBrCp4QjnZLlRUysuqSZ7hkbj22smsXbODDet3I0ki
    TqedbYXFPPXEu1TXuJk2dQx9++ZSWLiPJm8Ao9HAN6u2s3lLMe6GJtKcdqZMGYvJZKBobzmq
    CuefP5qVK7awevVOcnOdTJgwBI1GYt++Q8iyQp8+ecy+ZjIaTbKcr7SsBkVVyc1NR5GliM8t
    q3rjqcxrvarRNIf79MlDkiQOVdTg9vhwOlM5tfMFyXVrNpsZM3YAK1duQwBGj+6PyWTqgPdO
    rh1vo4/qqgYkSaJPQXc0onaP352IaA2ds3gEoT+quqMKrP++6srJj/ma/Obnn/+UYHPkR0lK
    JvveGLjplouZdcX5gVCYx81moUbUdl7n6m9z4c03liDLcodkF9oMnkCIR/78HxwpFvbsLvtR
    RoEoihwoqaK0rJphw/oDSpfhwuDBvQds2bLvh+9BEHh/4QpK9leya1dpsm+F8MPj1qcg72cu
    dBEuHLtsTGDTpr2IonDM37eqKjV6/Me1hURRbOtVcLz32LfvELKiHPesy2nlgtuH05nGpTPP
    ZsO6XQQC4R+1x4miSFFROdt37Ofcc8YxZnQ/Vq4oRFEUxozpj81mY9myTezbV/mj5XgVRcHp
    sjNt6jgEQUOwuZmqynoEUTwhLnR4eEEQuoHqqQbtr4cM7r/vqSfv55JLJuJuaOLVBYuoq208
    7mtrazy8/J8vqK5u5OKLz+SpJ+9n6OC+RfGo8uvcHHPt+pUy77+1thMXQBQIzfD5vNf+5+VP
    afQGuOaaC1qkFPmRxr2InJApK6vkpRc/4qZf/JXH5r9DLBInPSOVSCRG9WH3KRseOp2ekSP6
    otFo8PuD7NlTRiIRn9bYGDaku3SndbQEIRsYUA76bVOmjCUlxcpXyzaxffv+7xkrFYNBT05O
    5jHOLLQfv23b97N48SYsVhPnnjsSQZRWOTNTmtZtDP68V3fqZQTWRUFZmOZ0YDbpqaioI9hS
    fpaVlcHcG6ZjMGiJx2UCgTDP/PtDvvpqE/v3VfDU0wtZsXIbgpjsQBkOR1ixohBVUTjn3OHI
    CZkvPl9LIJCUutNoJLYU7uPhh1/hk09X8/zzn/LovDfx+4PEYgkkSeK666bQs2dum1FZXlaD
    RiORmmZD1IiLUjLtUX/zyWoN2oF3/SB8OHhQL5xOO/v3V7J710E6sqfA4IG9sJgNWCxG+vXv
    1aHR8J27DlC87xCu9BQGDeoForTc5hRjJSXNnThvXID9VYPB8s877pilPPzwTXTrlv69Eef2
    hrpKTm46f/rTzdx95yzFoDc+4/HEPq6v7+xQcnsuuN1NHeYsqKqKLCtYbSaam8NsK9x/zEjr
    DxnOTU3N7NheQmu2tmtxwfKD37kgCFRXe1i0aD1VVQ0/eP+tUpk/c6ELccFs+M73LAgCckIm
    Hksc9ztVocXYP3mHOxqNk4h/d92oqopOr+0ELsD5549i7o0XotNJJ7zHtQUMmkMcKEm2Q8jv
    nYfRqMdo1NO3bw9Apbj40HekhX9wb2nJzN588wzGjhsEQFFxObt3l5Ke7jghLnS4w6CqRSBo
    e0H8j76mptzdu0vxeptRFAVZUVC+Z+BUVUWRFRRVwdPoY9fugzT5fHl6o/gXj1u94rwpvbVT
    ZvTspChSMaDTQOT8Deu3i+vW7mbY0N7MmTsDjVZLXW0D9XUnYuALqKpAcdFB/v73Bcyd+xf+
    9rfXKC2tZtzY/vzj0bs466zhJBJyi6LUqRsew4YVYDIbUBSVHTv2k0jEu+v0ojUhd0bIZS2g
    eXX4iIHBqdPGU13jZv68NzhwoIKkwpVwjAccrXzz3YfIwQOHmPfoG9TUuJl+4ThGjepfiqq+
    oSgBZl567s97dSdeguACsoGYOy3VTkqKncJt+/nzIy+zYcN2PJ5Ghg/vQ/8BPVFVlWAwTM1h
    N+PGD+Jvf7uNiROHEPAnu92KoohOr2XAwB784fc38q9/PsDd98zC4bC06dZHozFKD1SRnePk
    gQeuZsiQfHZsP4DXG0BVYdjwAqZMHUc0GqG2toE33/ySTZuKsNvNpDgsgFABfsqqoyd5v05g
    OiCsGTgov3HsuIH4/SFee30xXm9TB22jKunpDnQ6LXq9lswMR4c5C42NXl59dRF+f4iJEwfR
    r1/36mBzdFPAF2PoiLM7aT/dDpSI4J6rKPHZB0oqhfKymh+sSf+OqR6NcbC0igMHKwWVxOys
    LMPNaalZtoa6JT8JFzZs2NPmLCSNfRlZln+UgdDq5MqygsmkZ8KEQfz9H3dy9jkj2hSWfuwl
    yzJbtxYjy4kux4XJU8ahqup3xijZQ0BBlhVkWUZRlLbnEgkZWVbanvu23QAwbdrYn7nQhbhg
    tx/bMTwRxbBTLe873meoKthspk7hgk5n4PbbLuXaay9okzX9Mcb94ap6IN7CBQ0Gg57sbBex
    WJSKyrofd25BUTGa9Nx22yXccMMMJElLMBjkzTcW43b7mDhx8AlxQdOxG6lGA8qlihr85aaN
    u8c//9xHrF27E1Dp3TuHq6++gMzM1OO+R0ZmGnPmXsg7by9l86YitmwuYvyEwZZbb5s5Zezo
    YeMlTXMq4ciLaqxYEXT9Tvf0B2KueDw0ccmSjUQiMS68cAKZmS7Wr9/GX/+ygOxsJ/Pm34vd
    bj1OZFwkEGjmww9X8sp/Pqe0tJqMjBQuv/xspk4dq44Y2T9qMBgNn378DZIkHkNG9OQMj9y8
    DHKynfiamjlQUkVjo69nqsM1DlH9rHOiiIZVBkN03l13Xf6/VVV1unVrd/HrX/+TO26/lIln
    DsVkMhCPJ9oMwlYItEqMSZKIVqtBq9UQDkdZs3Ynz/77Awq37uess4Zyz71Xxq0W6ysIQ4qL
    t6/6eaf+SS4TIJKaloLdYaGyso6331rGl1+ux+l0IEkCNdWNLQc4tZx//nD+5zfXUVDQg6nT
    xrN3TxkHDlYRi8XJyXEybGgBGZlOQODWWy/l7LOHsXr1LupqGlBVlZ75OZx11gh69MjmnLNH
    8shfFrB1SxFqi+LDww+/jCzLlB6spqKijkgkRt++eaSl2gFFTiRUJow585TWFUgbjEbrkuuu
    mzx748Y9rFq5jWef+YD7f3X1CZYPff/5LaPJiCSJSBoJq83yLYf6WH/PDzsLweYQzzzzPqtW
    bScrK43ZsydjMBiX1Neruy3mzpGRUdWDgNoDdHc3Ntbe/PZbS+3vvLOMysp6BOHEjQNBEPC4
    m3ju2Y/57LO1wpVXnt999jWT57mcpulOV/6dqnqwTBDyO5ULzc2RNmPAZjNxweQxKKrC4i83
    Eg5FTygynuxBkMKZZw5j6tQxbVz47OPVp3AoXaD0YDWNHh+u9NQuxYU7W7iwft3uZD8VIVmm
    m5RPzyIzMxWb1YzJbEAQko22wqEw/kCQmtpGDlfWI8tK2+tUFc48c8h/LRfU/1/VTh3hgs1u
    Rq1UT+i+Wzswt//5w/tB8ifHlF893mc67JbTzoXnnvuQ++67GovFyq8euBatRmLBq4uJt2Q9
    Wu/96Ps9ejwSCZm6Og/xeAKzUd/ST0HAYTMRi8WprXG3KBp920lKrv3W22/9LLPFyD33XM4N
    N85ArzcSDod48cVP+OKL9WT+CC5oOmbCVwJ2EWpvjUZ9/3jvvSXWZ/71IbW1jYwY2ZfrrpvM
    +PGDcblS0GiOp2CgotFomDt3OtOnj2f9uj28/vpiVq7cRlFROffee6Vt1qwLHtUbTXga/C8G
    g4WK2Xw661YlQDzP7wv13rO7DIvVyOjR/fB4Gpn/2Nvs2XuIcRMGozfojgvrmpoGHnvsTT75
    eDV6vZbZs8/j6qsvUAcMyK/RG4zvxeOK7XB19S+Ki8sxmQz06JHZIQ6D02knJ9vF3r3luN0+
    aqrdxuzs7AxU2LZpHcPHTDhtoyYILlS1OAbSo92755n+7x933v/442/rPv9iPb/81dOMHz+I
    /Pwc6uob8bib8PmaCQajqEoSHEaTDrvdistpJz0jlUOHali3djfRWJxLZp7JA7+aHevePedJ
    d0P0cbu9kIEjzunKKGhv7glHMiYnZai0bYJCi3MFmp9MOtACqJU2m6UhNdXmEgSBQQN7Eo3F
    qa/3JksErEZGjerHBZPHMP3CCThS7ICMzWZh3PghjBs/9Fsbb7ITtkYjMXBgHwYOLGgxwtWW
    9Zj8N0OH9eWJJ+9n4cKvWLF8KxWHaln+9VYAjEYdNpuJaDSG3W7BZreG/f64/9S1//sB+2Kg
    fXbcuGFn33LLRdlPPrGQBa8uQtKI3H7H5S2BA+VbzkHrd66iKAnicZlEItEWSY1GYsTjCWJx
    hdpaD7KsotWLVFTWEw7H0OtENBoNeoMejUZEkqRkszKNhiM9aY4WCVCPyiz4eOZfC3nttS/R
    ajTcdvtMRo4ceDAUij9pNqsJq21MJxhDJYDYE0IvlZQcmDRv/pss/2oriUQCQRB/tEHceiC2
    qrKep558l127SgwPPjh3au/ePV9SEtysKkVlgti/c7iwp7wNyCaTgV/+8mqunzOdiopatheW
    UFpa/QOHdBVsNjPTpo09JheK9pWfdKRVFATq6j3U1nlwpTsBpctwocdRXPjyyw1EIjFAwOGw
    8sgjtzJsWB80GimpdtQi35JIJNfOrl0HuP++J2lo8KGqCgaDjslTxv4Xc0Fo0ekXTkreQGgx
    glv3t67FBbvraIO21blrVUjTajXodMnAoMlswG63YLWYMJn1WCxmtBoJg17TlsETBIjHZSLR
    BJFIhEBziHAkTsAfIhAItu2lyf000dIsVGznUKiq2ilceOU/XxCPJ7j7ritITUvhvvtnE0vI
    vP7aEhIJGb1eh16vxWw2kJJqw2azYLeaMJoMGAxatBoNgwbnI4oiWp02eeZDENFok+viwgsn
    MHhQPolEgnA4TjCYdKj9vhBer59Qi6JYJBLDYNByyy0Xc+MvLkan01Nf18Czz37IW28tQxAE
    bv8RXNCcOhCaWpyFg7eHw75/PP/cQuuzz32EViNx512XM2fONDIznccAG8dxGiQyM9O59LJ0
    xk8YxOuvL+LVBV/yl0cW4PUGbDfddMmjDoeeAzvlF+sPFyrpOafLaRAAwRwOx8RINEZqqhVX
    eio7tpewvXA/55w9nHvvuRKDXn+MexIpL6/i4YdfYsWK7Qwa1JN77r5cPfvsUbU6vendRFx9
    sbQkUN6rj+WFbYXF1NR4SE9PZeDAfDpKcz0jMwVRFPD5gtTUugHVjmDEmhI67VuGIPRDkfeE
    BUH/cF633NifHr7tvjMmDrW+8PzHLFu6ia8EAZ1Og0ajQaOR0Bu0iGLS23a7myhL1CQXfSyB
    qqr07ZvHbbdfyuTJ4wI2q/WfnoboX61WTUinH9E1caBWATkC7B+dkBPaeEs6vaz0MCl2M/JJ
    iN9LkkhZ6WFkWSGeiKOoslaW1dGffZHYUF9fqKand/ZYSEBkl04nlQ0ZnO9a/c02cnKcPPjb
    65FlhUQ8jsVqJjXNgcVi/tbrBNqrYanfkdk8klI+2sE6okqWl5vJ/ffNZs7106g+XE9DQxOC
    IGC3m3n6nwtZsXwbgwfnYzYZir1Nia3xuHKKc1pIlltStlqScu+ZO+eil0PBiOOFFz/jxRc/
    pbrGzQO/mk237rmASiwWxesNcKiilurDDdTXNVJT05h0lD1NBAIhopFkna3c4jwkEjJerx/R
    J3LPXfORJBGNlJQo1mg02GxG7A4rLlcKWZkpZGSkkeZ0kJmZSm6uC5vNgk6nA0TKyir5v/97
    k2VLN6DTabnl5hnMvnpynSRpfm2xaHY2N8c6YR0UA7qeEH5p5869k37/++fZvq2k5cCjeArf
    RVJNRVEUli3djNvt55FHbp00cGCfl0LNsZsT0cIyjf70cyEajbUY/irjxg/kyqsmEwgEee7Z
    D6msbGhzJo5VR60Cg4fkc/ddlx2XC7U1jSftMCTPMYTwuH1dngsvvvAxxcUV9OyZzdChBdhs
    VlQ1caT8SBQwGPQYjRLDh/dn6NDeLF26mX79unHrbTP/a7kQi8voDTpmXzOZPr1zv7dc+/sc
    w5IDVbz91lJi8USX48Ka1TvafmM06ElPd9CzVzbde2SSl5dBdpaTnBwXdocVvV6LTqdFq5XQ
    6XSIooBGat9fQFEUErLaJluaSCjEYnGisTh+XzO1tR6qDzdQUVnPobJqDpZVU1/nbSt9FEWh
    07iw4JVFHK5q4Je/upp+/Xpx333XoNHoiEVjDBrci+7dM8nISMVoMiQdCF1SEjgpzSq1BZ20
    OkNbx2ad3oBeb+K2265oCU4luRGLxZPjEI0TCkVoqG+krKyGHbsO4HLaueWWS9DpdOzatY95
    j77B6tU7MRh0P5oLHZBhqAcaz03Em/+24NWPrf/+94fYU6z85n9mc8kl56LTaflxSh9HnIrM
    TBf33TebHj2ymPfoW/zz6YVotRrbDTdcNC+jlybhSOHl0zfxdYCQsFiMGI16mpoCJBIJjEYd
    BQV5TLtwPDZbMqLYmk5KnlgXqa1p4E9/eolVK7cxY8YEfv3ra+nRI/fjREL40/7i5t3demxU
    evWZeqnf75nx+WdrCIWijBkz8ChZ0VOHWlaOE0mSiMcT1NV5iMXi59TXR55xpOginbFtiNJA
    GhsLQ6GQ+ufMTMvOyy8/79W9e0qNZWU1zLriXM44YwguVyopKRa0LTrAqBCPJZILv76R1d9s
    4/33VzFu/GAuv/y8MKp4c8CT+Mhs0MQNxq4MBYcAB2+Kx0OPfPDBcnHfvgpCoSj/94/X0eu0
    Jx1JisbihIIRivdV8P77y8W5cy9+ZNo0Xchfk3g56C1UzSmdOSYRoHsYopsumXnWmM8/X8ua
    tbu4dH8l0y5srX9MEI/HCTQ30+QNUFfXiNcbwOcLEg5FiIQjhCJxotFYUnP7KGBKGgm9QY9B
    r8Gg12Iw6jEYDVjMBqw2C840G670EwfzNAAAIABJREFUFOx2S4tSUXKDXbpsLVu37Kdbt3Rm
    zToXQZQ+f/vtUvecOd07wODpjyofBlX3hcms/ecdd15xq96gz3j+uY/56KNvqKxsYM7caXjc
    PrZsLmJ/SSVutw+/r5lEQk6W2WkkJI2ETq/F1BJNMuqN6HUaJEmkW/eMNjDGYi3RsmiMYDBM
    g9ubdDBkhUQigSwraHUarBYTaWl2+vTJZcSIAlJSbbz+2mIKC/eTlmbj3nuv4OrZU2r1esO9
    D9xx4OMbb8ti6MgzTvM6qAR0OvD99cCBkkl/+GPSWZCkjgt9CoKAgEDh1mJ+97vnmffoPZN6
    9Oj2z5p6Zc7h6sLGnOwRp50LrUbIgAH5mEwWlixeygcfrGgLCkiS2FY+0+osCKLA9AvH/SAX
    IpHYyY+XkDzHUN/g5ejzYV2SC3tLjUVFFdgdFhZ9uZZGdxNebzOBYAilZewsZiMOhwmXKxW7
    3YIgCowdN+i/mgtFxeWkOCxcdtk5DB3Sj5NTRBPZuXMfS75cT1FxeZfjwhdfrKO8vJozJg7h
    5psupk+fXFLT7JjNJtpn2tXvId/R8wd0Euh0ekymb9uNAkPbktYKwWAIj6eJ/fsreeftpXz1
    1VZy8zqXC19+uYGy8hp+9csruWDyeB787RwkSUIUNG2MTCSSj0g4hruhiWAwTDgSJRZPkEgo
    NNR7icUSCKLA5s17yc6pQxIljEY9JlOSiQZD8r9tNjOiqKFnz+6MGTucK66KoyoKgiDy5Zdr
    mT/vTfbvr8TptJ8UFzSntggOAEIOhH6/7Ku19mef+QizxcgffncjMy46syXdfCoenIJOp2XW
    rPMxGvX88Q8v88y/3icvz2WdPPnM+71edVljY2FFaurpWBAJQFxpMOrKcvLSe5aVVVNV2cCE
    M4awYMHvcaTY2ibp6m8K2V9SwQ03zCAej/LkU++yauV2Lrp4Ir//3Y1NqamuN0NB5RFRVOv6
    DUgHZhZAcN7XX220r1+/h9RUOzMuntgSHeyY1qvOVDuiKCDLUFfrRRCUHhkZOp2qypHO2jpS
    U0egKJvjguDQ1dWWazZtLCI3L5077phFt27ZfF+/BRAYPKg3GzcVsXVLMXW1btGV7opZnWJc
    EIZ3USgcBlIEqL4pHvc99tprn9meemohoigwbFhvEOBUylVbAy1lpdU89dR7CIJgmzvnosfS
    cnQ0VCVebvYWqpZOgoMg9EBV9yog/KVv3x7jrrpq0qh5897i+Rc/wZFiwe3xU1x0iJKSSg5X
    u/H7mgkEQoSCEcKRGKqitDScERAE8ZjycK0pbFVVUJSkodV6+CtpQFjo1i2TgoI8+vXvTnZW
    Gq+8/CmBQIhbbrmI/gN6bZVl9Zlbb+2FwTCqY+5byiEcro1WH+rzh169IptvueXS51wuR/aT
    T7zLls1F7N1TSiyWQBQF7A4rvfOzycvLIDPbSW6Oi5ysNJyuFIwmQzKapJXQSBKS1H4MVEBO
    JIUi5JazPvF4gkg4SmOjn7oGL7W1Hqqq6qmuauBQRR1ff72VxYs3otGIRKNx8nvn8MADVzNl
    8hmHJY32jksmbf7s4Uf7n3ZnIXlVAs65Td7aix+b/wbbtnass3D0ViFJEtsK9zNv3mv8/R93
    TcnISL1Gq1X/dXru6ygu5Kb33Lu3DBDYsH4nr732CZ9+urpN1eicc4YzeEg+CxYsIuAPtS3i
    GTMm/CAXNmzY+6MlE4+1fryNgXZ7bFfkwpZNxQiCwDertvH111uQE/JxO/eILefbBAS2Fe77
    r+XC008vRKuVuPXWSyjok4OsRE8ODoJAnz453HbHTJ58/F2eemphl+LClVeeO2revLdRFIWh
    QwtIcx7dZ0puOcyezLLGY4lkICkaJxaNE48niB5DCEajkdBrtUiSiMGox2zSozfok+e/JAlJ
    khAECbPZitlsxZmWxgfvrwTgilnndjoXiorK+X//73kOH/Zw2eXn4Hb7qKyso6qqgdpaD5WV
    ddTUeAj4gkSiceKxOPGWgFDyoL9KKJRcng899BxarQZJFNFqJbRaLTqtBoNRR1qalcwsJ5kZ
    qeTkppOd7SI3N53UVBtffJF0Furrm+jdJ/ekuaA5+UUQACwC7J978EDZ2U88/g6RSIzf/r/r
    mT7jjBbjpiOi5cmU7vQLJ1Jf5+Uf/3iDp55cSN+CHoPy87vN/vT1ukdLi5ervfpNOg1g0NUb
    DOamsaP7sXzZFr74Yg2jRvcnIzO9xbAXaGho5LHH38ZsMjB3zoV8s6qQTz75htFj+vOb/7me
    tDTXk2Ul3r/YHQbZmWkBlAwI/3nTxm35Tz/9HqFQhLlzL+SMCYPpyO6UVqu5Jaql0uD2oaoy
    Op1OkOXO3CirASkFvLds3LxHu29/BdddP5Xc3Iy2dJrSqpzV1mWwtVxBJDcvi3POHckbry1m
    48bd+ktmTrpsx87gp3V1m5SMjDFdDArVLVCouikeDzy24NVPbI8/9g4GvZ4HH7yW8y8Y0yEK
    nAKwalUhf/vrazzxxLsIAra5cy56zJWro74i/nJzY6FqSe2siJIAxOtA/+/LZ5330oqVheLW
    Lfu4667H8DUFSSRkdDoNRpMBp9NOQe9c7A4LNpsZi82IxWzCZNSh0xswGLXfKb+QEwqRaJxI
    OII/EEyW8YSi+PxB3G4/tXXJtOuyZZuQNBIOuxmvt5kRIwqYNWuSKqJ5SdQKdeEOLr8xGjMJ
    hwsJh42fCYKiXnbZ+Y/26Z3b76WXPxOWLN6EM83O/fdfyagxA3A6HdjtlpaOzSeSXT1+dO3Y
    806huTlEU1OAHdtL+OtfX6W+3sslM8/m9jtmKv0Keq6OxcS/x3TSkoWfj8BoHdEJa2E3YMpX
    Ff99b7zxpXnZsi2nbPz+YJxVFFi+vJC33lyqufPOWTcnosJnqrrjkCAMPW1cGDO6L19/tRmA
    zZuL2bSpqM2p79YtgwcfnIMgwLvvfIXfFwJUxo4dcEJcCAbDpyzVqqrgD4TazauuyYVKVFUl
    Fouj0UjojHokUUQ42oFW1bZzP/FYHEVRKSmp/K/kwhOPv0uwOUK37hnsL6ngL39ZgHoK7BcQ
    SCQSOBwWystrW4JJXYgLKwrFTRuL+PDD5Vx00UTKymuoqXZTUVFHZVUDbk8TvqYAkXCceDxO
    PCEjt5TxJo4xMSVRRCNJCKLQdg7CYNDjSLXidDrIyXKSm+siNy+D3vk5bN5SxMqVhQwb/tNy
    Yf78t1m4cDnNzWG8Xj/BYOQoH/FIO+ojGPz2/0OzP9RurrR/fetrVERJxGY143BYsdtNlJfX
    EgxGmHnpqXHhFDIMdUD9+Hg8+JtXX/2C4uJDXD5rEldfPfkUOyAfx2kQRa6ePYUdu0r56IOV
    vP7Gl/z2t3N/e/ZFqRsNenllx096G/BKAC57Z8qUCcM//mg1H334DdnZLubeMB2bzUwsFuO9
    hV+zd08ZD/z6GppDERa8ugiT0cAvfzmbnJysD/1e+d+udJtsS9UBugyIPFW4dfdVf/zTy5SW
    1nDOOSO4/Y5L0ev1HZZdAJC0Rwwwv78ZWVakcETWJuJKJ26XPkBbEA77xiz6fA2CKJCekcLK
    lVuoPlxPfUMjXm8z4XAsGUFuOfSc4rCQ7kohOyed9HQHgijw+RfrmXTe2Mn9+5mHS5K6tWtB
    oQZwCVB2UzweeOz11z+3PfXEe5jNRh566AZmzjwXQei46Oqll56PTqflkT+/whNPvIuiYLth
    7ozHXHk69pXwcm3tNjUz8/RH25LNuMoAPsnMzLl1ztwLx+3eVUokHOO880cydGgf+vfrQVa2
    E6vVhMVsxGjSo9PpTno8VCVBOBKluTmC3x/E4/ayf38lO3eVsnTJRiRJZM6cC8nOydoUDMjv
    A1jsozr83o3GEVRXF7JtW/zzs87S7x4xcsiff+2wXb9ndznhcJRhI/pTUNALaAWeclL73olk
    naxWM1arLSkegEpOrov77ruK/Pzub0UjifsNRslTuK2RkSPO6qQVMVCAkj/u3Llv4JtvLiGR
    SHRYn4Ljj4NAIiHz5ltLOfOsoUMHDSj4y/Kv6uYuW7VcueDsSaePCx+vYc+espbsidD2t9x4
    43T69e/Fc88upKHBB6i4XCncd//VP8iFsrLaDhovlXAk1uW5ACrnnT+S3r1z6d4ti/R0Bw67
    GaPJiCiKKKpKJBzF7w/idjdx6FAtBw5WsfqbHf+VXAiHouh0GmprPLy/cOWppZ2P2gjElgxs
    OBTlya7EhRsuHLd7dynPPPMh77zzFR6PD78/1CKdq7Zlko8OGB7JPgvfEUZQVVBUBVVJlm4m
    z7okDW5REBDE5EFym81MRkYKgeYQiqIw9yfkQtHeQ5SWVlNUdKjtEHbrvSXHoDU4TosYxJHf
    H1seNjluigqqotB6BFAQWs5MqeDzBWlqam7jSM+e2afMBc3JLYR9QBHQ+8w9u/fbv/hiHbm5
    Lm6+aQYmk7FDDd+jNz+TycjNv5jBxvW7+fyztcy89BzHgH59bnj79YZVHeyhIAgZLQofvN69
    R+6ke+6dNeWhh57n6acXsmHjHoYN6U1ZRQ2rVu5g6NA+XDrzbJYs3sCWzcXMnXsho0f136uo
    yi+NNrVBp9UDpEHkqa1bd131v//7HMVFFZw7aSR//OPNZGWln6Yxa/FKm0PIsiKpqqpLxNVO
    2izDQAqw/er9+8qMmzcXIScUnnv2Y4LBMNFIrN3G0D6SlIwmtaoIyAmFLZuLKNlfnjF8WP/r
    BUnTpcCQvBrGQXzeO+8usT3x+DuEQlEmTxmNy2ln/bptHTo5BcDhsHLOucN5f+FKnnrqPUwm
    ne26a2fc7XQpixSF6s67byOwrxGyXxo6pGBcVpaTaDTOAw9cQ9+++e0iJ0eMYBlVlds2vW9L
    yn3bEGwHD1GDyaTDZLKSnu6id+/ujB03jMOHa9lWuI9IJMaQIfkIgviy2aa4GxtP33zPbqmR
    V+QD5cDTmVnpk3NynRkbN+ylttbNgAG9jrr/b3dxT6o//dAYtN530n5sfY+je5ccedTXN+Jx
    +xk2tDepKbbGSFj+dyKueAzGwZ1oJO0DSvJkOTDi/feXU13tOT2lSMfMMojUVLtZuHAFA/6Q
    P2n0mNTBiqLuOK1cuOfyKb/73fO43YF291lX18jyrzfy0UerUFo6zl5yyZmM+R4uPPS/z1Nc
    fKhjnSu1q3HBD/QHvr56/74y44YNexg5qh9PPnk/drvjqB3ueEVJyb8zFArywK+eYu3a3f81
    XHj88XcQBZHbb59Jnz55nA41VUGAAweqeOutZTz15Ltdhgs5OS5KSqrweHxtsukWi4E0pwOX
    y4HVasZuN2OzW7HbzJjNenR6LWajHkmU2sXPY4kE4XCEcDhGoDmMzxek2R/EH2imydtMdY2H
    ppazco2NfgRBoGePzJ+UC9nZaRkHDx5GkkQURW0ZAwmb3UJGegqOFCs2qwmbw9qiFmXEaEwq
    KRn0unbrQVVVQpEo0UiU5mCUgD+I3x8i4A/gD4Sor2+ittbTohqV7ECvKAoul+OUuXCSGQYL
    MDwHPNcu+nIdDQ1N3Hb7TAYM7HlaDV9QGDioFzNmTOCFFz5lyeINDBnUe9TUGY686sPrK7Jz
    xnfw5+UClTVge3zylLNGJeRE2tNPvc+6tbtYs3onophUwZg6dQzxeJw331hCZlYaV111Hhqt
    bgHoKiQpDCgOUOfv2F501e9/9yIl+6u45OIJ/ObBOeTmZZ3mMYNwKIqqKJIoqlpRVDsBCg2A
    KkLRLGi+ZvnyrTQ2BjCZDWRnpZKXl4kr3UGa047DbsViMrQpigTDEXy+AI2eAA0NXior66mo
    rKOpKcDSZZsZNrzvWSF/5IL62r0rIR5PzxzaVchgV1XFtvzrTTQ1BZA0El8u2sjiLzd1tC/b
    tnWqqoqsqDQHQqxatYNrr53ml5WEPxbrPF1vQchEVcsBapzOVMXlcoi7d5e2RDZkmpuDNNR7
    qW9oorHRj7fRj9vtw+dvprk5QiQcJhaXCYYiqGr7daDTajAZ9RhNhiRQbMkUq8NhIc3pICM9
    hfSMVGw2M3V1HhoavOTn55CSagvHIomDsiyTljbq9I+BaAHi5Uajrqpbt4yMtWt2Ul5WDSQb
    eYVCERoavLjdPgL+ED5/EG+jLwm6YJhIJEY4EicejX7HWdAb9Rj0eiwWPTarCbvNgiPFhsVm
    Ii3VRnp6Kg6HBZ1OoLKilmg0RnaOE6vNWNTQEN0VbO68WpNkrfZWYMClBw9WDly+fCuC0LmL
    UBBgxYpC5s6tzu6d3/0cUTLuqKtbT0bG+NPLhac/oPRgNYqqIqDy4ouf8sYbS9rKDrr3yOTK
    7+PC71/qeGehi3EheTUC6waA/4rly7fSHIxw4YUTsNuthMNBfL4gPn+QUDBMLBqjJXCMTqfD
    bDZit5mw2cyYzCZmXDSRFSu2sXTZJoYP7ztaiSWcsfBWt844sit5DEe44PXz4IPXc8edl6PR
    aE8bF2Q5QarTziMPv9JluJDmtIslJVUMHdaHs88ezsABPcjJdeFw2LDbzUmlIJ2Wk22CqaoK
    kUiE5mCYRo+Pioo6ivaW880329m0qQiny/GTciGvW0YGgNVm5txJIxk1ooD8/BzSXCk47BYs
    FhNGow5RlDi52mWVRDxOKBzB5wtSV+um5EAV27eVsHTpJjweP9nZp84FzY//YrxAFaAdWV/v
    GbR2zU5SU61MnToWQdBwJAV/uqJIGqZNG8/7769gw4bdNDX5Bzid9sECcsXeor0ADOg/oIMm
    vbEltWZeJkmRmdOnn/OHwYMLzl+7ZofQ2NSM3WbknXe+5o03lvHRJ2vYX3yI+++/ivz8bnuD
    zfK7Gm0Uvd5sh8QTu3ftueGhh55n//4Krr9+CvfdfzWpqY7T7iyAQCwWJxKJpun0hnxZVg6d
    /u1CBpozQf5tbU1N+pIlG5FlhfSMVM45dwRjxwykZ89sXK4UDEZ9u/pmRVWJhKLUNzRSUVHL
    5k3FfL18M3v2lLN06UauvWby8MyMzHlyQr0skYiVdrV4kkAynTh4UE9cTsdpbcojCAKNjX62
    7ziAeMQyUzu/EZARkAtNJt3OvG4ZwzZt2svbby/lvfeWc/BAFR5PE43eAMHmcLuoeWuTGVEU
    0Oqk9sWaqkoioSAnWiPxtIvGG0160lJtpKbZ6N49CwHw+0Pk5qZjtZr3etzRXfFOK7OoA6Qm
    QRAD3fKyEASBVd9sJxyOsHt3KYcPN9DYGKCpKUA4HG1TzTl6DFqN3fZ77ZFU/ZFGP0kLSqeT
    sNkspKbZyMhIoV+/7uzbV4koCvTqnYtGo4mkOJTw4Vq1k+fBMC34L968aS81NZ5ONYBbswx1
    tR527NhPQZ9uFwW8vn+l2LVyx667E+PCrp0HEQTQ6rRcecUkeh+HC7/73fPs3VPayWP1U3AB
    wANYh9XU1GQsXrwBk1FPeVk1Dz74DOVlNTQ1BQgGI0QiUeLxRFtDT41Wg9Ggx2wyYLOb6N4j
    m9RUGwajjiVLNnLN7Mljc3JzL9LqeIUueAkkZTH7FOSh0YgkEqdP1lijkcjPz0GrkxDoIlzI
    yxi2Yf0epk0dx113X3lUvuDY2dL2+913R/NIqY7Q8v8SRqMFo9GMy+mkb998LrhgAk6ng02b
    isjJcXUBLkBGuoNfP3ANeXlZR40BR42B0mYTtsovc5zM85GfyYdGa8CmNWCz2cnLy2bU6CFM
    vqCRPXvK8Hj8HcKFk8gwBIFBIpRdUVZWJZSWVjNkaG8KCrp3gvGbHNS+fbvTt293iorLqayq
    E/R62xUrl0cW9yowyx3ttQtCT2oqi9T09NCaaEy6rnte7pU9rsudDKoCQuagQb3HvPH6YqGx
    0cf0X17F7GsmN4H4d7PFVAE6K4Se3LfvwA2///2L7NtXwW23XcIdd85q0abvnIkbj8tEY4rD
    ZNZ2P1kP/sddWiC1BuoPxmLx4T17ZdHU1EzloVqeffZjFryyiPSMFNJS7JitRux2Mzqthlgi
    gc8XJBgI4/H4qG9oIhyOopEksrLS6NMnj2g0jlYvlOzd1VyW6pS7IhkAuOOOy5g2bRyJxOn7
    GyVJYsXKrdx266P8tL1CDwM0CILg6dUzG0EQ+Pij1Wi0Es40GxkZaRT0ySPV6SA9PQWn047D
    YcVmNWAyGdFok9rb3w6syAmZeEymORCkKRAi4GvG6/XT4PbRUN9EY6OP2jov+4oriMUSIECP
    nllotRqf0UBTQ6cZyzaS5XdueuXnoNFIfLNqG6u/2YbJZMDpcpCVlUa/ft1xpafgdNpISbFi
    s5gxW4wYDdqkyodG085ZiITChCJxwuEYkVCEYDiM3xfE2xSgrt6Lv6kZt9vH9m0lrFm9s2VO
    iHTLdaHIYk35flnp31vXifMgBAijFSUxfMPG3ciyjCRJnT4bEwmZzVv2ctmlkzLDMSW7OSxX
    dryz/sNceOvNpYK30cfIUf2Oy4U//OEldu0q7XTH6qfgQjLY6BBgz8x4LK7plZ+NzxfklVcW
    EWs5yNyqFnZ0TfvRTckURUWWFTZs2ItOp8WV7qCgoBuxWEKSJGXah+94FpSWrFN79ZnQJbnw
    xOPv8OqCRac9kNTUFCAekzs9w/f9XICamoaWTLJKIBBKZl49Ppr9zTQ0+PH5AvgDQUKhGLF4
    gkgo3L4jNMnMs06vR6/XJIMmKVacLjt2u4XUFDvpGSnYbBYkSaK2zosgCF2EC5pkBq0pQF5e
    JrFYFI/HR0ODF58vKT3ucfvxB5oJBMNEIwmi0RixWKwdGgVBwGA0oNFIWCzGZOfqNBupackx
    cLmSjDUYDDT5ggQC4Q7jwkk4DH4gLkI4t/TgYcLhKAMG9MRqNUGnmC0qFquJgYN6smnTXg6V
    1zB4UN/cUSN1ZOb2Y29RUYd/YlZesnPo6jcL6yUt/xp4Dv+yuSARJ2PEsP5/GjK4z4hYPOE0
    GvWVILywc5f/ncGDQBTVi71e97X/93+vsX17CdfPmcadd12B2Xy6znkcJ97f0hRKEgUd0qlL
    9bXgRoRSIWkkQBI4KRoIJlp2RwU0/+nWPU8zf/59IyoqanP37CoVt27bR2VFHftLKth8sIjM
    zDRUVaWurpGMjKTkWm2dl8yMVEYMLyCvWwajR/dn4MCecre8zMMGg2FbLCL/Z+ioMWp9Q7Gg
    qrtE+Lb0WhqQrYJGOdob7zjwBVsR0ELZQhl6tyOuRqtFEPVodafTqZHQaLVH+qABiYQiehrG
    o6rBVktNAVRBMHcKFfUGHaqqMmp0f26/bSbde2SSlmbHYbcgStK3okrHq1X+7pr/didjRZGT
    h549PmoON/Diy5+zamUhRpMREFR7iqQMTuksY1Vs2UoFHA4zGo1EVnYad991GX379iA1zU5K
    ihWzyYDYppYkfNea+N775zv3Hw5F8DYF8Hj8FBdX8O9n3qeqqgFHig0BQUKFzg0q+gEpxev1
    OQ6V1/1kZoqqqpQeqCYQCA1w2M1DZFmpPB2fc6pcePTR19hauA9J/Gna8nY+F9RWLrzSrXue
    9ttcqKqs48CBKmpqPGRnp6EoSS5kZqSCALV1jWRmplJQ0I3cvPRjceGVy64+r0tzobj4EJ0R
    6U+W+CpHOdE/PRcEBPbsLuM/L39K8b5yDh6oxu3x4W1MdiZOJOR2meTjbYUqHHVwOPmzVWI1
    xWElLc1O9+6ZDBjYi82b9yIIdAku6PQa/P4gi75Yy+Il69m9u4yaGg+NHh9+f5D/r70zj5Kz
    rPP9511rr+qq3tekO6GzEsjO7sKmKKAzoKNcZxMGBvVeuc4I3jvLGWd0PO7jKCMiXodxVERU
    QAQUBIIsAkmArJ1Od6f3ql6rq2t71+f+8VZ3usOihPQy99b3nJzknJxTVc/7Ps/z/a3fn2FY
    c5rAxevsM2Yz0zPr97L0x2fy1NTGWbeuFV1XmZiYQlGUU8ILJ+EwOEAaEHR3DwOwelVD6eG4
    i/YiVrU1AHCsN4msCGobNYTYiyQtnHTY+dcc/+yevXvQzpxK5fdFbozW2HFZpnpyrDgQVXy5
    MzYlADkOxnX33fuE9thjezjvvDP46EevKg0sWTxnQZK8iJtlmui6hGWf7GVoAg8BXQ1gbgb/
    1WA1H1+Mi2BclZBtGJVgKgnSj0C5XZFD7atWtcbWtLfdeNX7Lqo5sL+Tv/7kN3Bsl8997gae
    emof3/zmz7j8ivPYsWMdN9/8b1RVRfnUp/4bG09fg+tIo44t/o9tu32SpB7T/bLiuvtuqK6S
    V4OzCYQ03+jKytDfD87dENwrxHNDcCaSdKqirf0A54P0dyALqBcwXQWOPN8YPtVqYa92IXnf
    5wqBLCunN9QHvt7UeGgYtM3gKiA+Dexa2F1WCbj1jjNZc/jQMVxX8M5Lz+Kii885wdCdKS/y
    VC0s08Iw7dL+tF5xiSmKjKqr+DQVn09Hnp36KSHLKhUVcSoq4qxa1cpQcpInHt/DkY5eLMuu
    zU7J9Y4tBhbnlAWAcQmKdHcNUCia7Ny5nve9/1IkSXkN5+e4E3Q8/X6C6yDNNwDm7i1Z1giF
    NULhCE1NDWza1M6ePYf44Q8e4fDhY7z1rdtrVqyRZU1jUdJwXvS4B/AxOjrB2NjUKTfIfu9T
    IUuMj2eYTE9REY8DCzvV+mR54YnHX5xbSri4Ae9lzAumZfMPn76WXU+8yHe+8wCXXLqDc87d
    xN/879uIJyLccss1/0V5QaCq6qJE/YUAy/IERJYLLwhg74tH2b3nyOwEdFVVCEcC1Nd7Snp+
    vw+/35v4rKgy/oA+7x6xDBvDKMmvWg6m6UXhc7ki2ek8yeQEAwMj7N17hHvv3YUsywghlgUv
    GEULIQTfuPWnuO7xwY5+v040GiIaDc/OlPD5NE9i2K+iaXMyz66gWDC9WQ2WjWl5U54LBYNs
    tkAmk6Ozs5+Ojl52PfHiHJl6TgkvnITD0ADkfa47oo+OTQBQXV3BKRGZfwMeayIRRQjB+Hhm
    kTIb89G6ectcC2Ci9Kd0WJ8+EGkrAAAbQ0lEQVQHwpf19w+e+/0f/IqKaJDrr7+SmprKRXUW
    5m4yx3FB5g1fVp4xMwCM1sGaq6Fw7fj4+Omdnf1S55E++gZS5LJ5hCuwbIeG+kpWtTVRW5cg
    Ho98MBINiVg0WpQVtX9qKh9/9tl9fPO2n9LVNcjHP/4+zjn3TH5+/9NIEoyPTXHBBVu58S/f
    w7/8y4/4m7/5Fn9x/XvYsX1jPBqNfMAVVjSZGotMZfLS5GRGSg6N0tMzTDI1PqvbHQoFaaiv
    Ys2aFtrXtP5xVVXlPoh9G0bvFqI/CU2nwJApAHJtJpO5cO+ew5iGBXiX89jIBJIk8eKeQyiS
    i7OAAueyLLNvfxdCwOjIBL986KkKSZL+GMDn19m8ZS3RaPS2hd9zGUA5Y2xscuPBgz1UVIQ5
    c/NpOI7B0PAYI6k0yaEx+gZGSKXGmZjwZPVMw8Qw7NnR9idGVVRF8RwGXcUf8BGLhUgkYtRU
    x6mr84bT1NTEaWioZu3aFSQqYxw6eIyxkckN9fW1Z9iWvUjEYALydte1Nu3Z04EEnLVzI5Kk
    YJoFRkenShOfM96/x9Ok0zly2Ry5vEmhWMQo2tgnWm2ShO5T8fs1wqEgsViYeIXXt1BVFaei
    Ikx1VQVV1RX4fAF27NjAXXc9yot7OzFNs0FYSnOh6HUeLg4MIByazuak6WxuyRwGCYlstkA2
    W2BxSjDfOC/84IePLIrc7FLwwtHOfgYGR8jlCgjXxbId6moTtLU2vi4v3Patn9Hfl+Kmj7+P
    s88+g7t/9CiyLDGcnOC88zbzkY9ezVe+8n3+9m9v57q/uPJ1eeHYsSSpkQlURUaSZYJBPw31
    VZx2WvMS8oLMRz92FTt2rJ+dBL4QUGSZvS8d4Stf/gEjy4QXDh06Bgiqa+KctrqJtrYGWlsb
    aFlRS6IyRiQcJBIJEvD78JUchtc6G8J1sSwL03KwLIdi0SCfN8hO55mYzNDfl6S7Z4ju7mGO
    dg4wNDTK4UO9S84LjuPg9/tobWtgVVsDbW2NrGytp76+kmg0RCQSIhT04/Pr+H0ayus4l45t
    Y5qew2CZFvmCQS5bIDOdI5kc51jPEN3dQ3T3eM8gny+eEl5QT+Y69gZ0o5ilKXy6z7fol93M
    d9q2xSulG5cOQvTiDaVJX/fII8+pRzr6eP/7L2TnWRuXxLF51XDlG8IAIDXD9HdSyYGL7r3v
    CX7xi2fo7BzEp6vEExHCoQCqIpMvGDz6yAtMTeXQdZVoNCjFYhGpuroi6A/41kyMT9HR0YfP
    r3PjR/6QP/uzy+nrS/HC7kMoisLuPR309yX58z+/Atd1ueOOB7jppq/RflqzWl1d0WwYFmNj
    aSbTWTJTOUzTIhoLUVebIBj04Tgu2WyBdHqaYtFi1eom6d3vOnvTlVe+5Wu1dY1X4Pr+HLpP
    QXmCABSO9Sb55M3/xujIxCuaU2/9t58tTv2o8LIL+/Z1ccNffnE2elhdk+D2b3+KTafHWUgh
    Am+/vwCcflpv75A0MDCK36/z058+wde/cQ/9/SmSyQmy0wVc4c47ArIi4/draJqKpmrzZCld
    4ak+2LZTmn5pz4uOgkQw6KO2NkFzSy2NjdWoikJvb5KungGpvrGuXfUpD4yO7aa6ausCrl8A
    +wG1eiQ1mti/v5tAwMfgYIrPf/67HDx4jKGhMUZSk0xncjizmuHzo4G/T0R49hxLnrMYjQSp
    rU1QV59g3doWAgEfoWCAzqMDJJPja1esaNwEzrHnd+9i+9bFmMFgAbLmlKa3LuVV57oOrrN8
    epxO5IXOI/3IkrwMftmp44WHHvotHR19FPKGV1rBiQM5JXw+jYqKsBSNhqXKqlgw4PetmZzM
    0NHRR2Y6z7atawmEgvzwh79kz57O0hTvDu6661eEQgHa25t54fkO/uoTX6e9vVmtrIo1m4bF
    +PgU6ZLuvFE0StLcYu6RQZa9aG77mhbpsneetQi8MDkr6DHTn7Fhw2rOPXc7CysOo2C7Ak1T
    OXCgh4989Muzv6G6Jr5EvDCCqql87GNX8/73X4hvVir0jZ8BSQGfouPzv9578Mrt7r//SW65
    5VZ6jy0xLxzowXEEGza28dWvfpympuo5ikhv/AyqmoqqQfA11+9l8kdG0tx005d5ctfLHD06
    +KZ5QT25Q+G6IDk+3TPaDcNc9GvOLA2k8en6snEWPEwB/g2TkxM7H3zwWYJBP++87Bw0zQc4
    y54GXoksIF1cLExe9E//dAc///kznLn5ND51yzWs39BGTU2caDSEqsjk8gZ9vSkOHuyhq3uI
    7u4Bxsem6OkZwrIcKiujvPcP3sIVV5zPtm3rMU2LO+64j1RqkvXrV9DTM8ztd9zHp//hOq6/
    4Sp27tzIvfc9yXPPHeTll7tQNYV4PMyaNS2sWFnP6rZGTt+0iubmGm9eg+N6w31G0+w/0MNP
    fvI4//zP/8HeFzv54hf/+0WhYPRiIcR3TuGtgOO4VFbGOOecjaUGz6V1Ch3H4emn93sRrEUp
    YM8AG1QwLt9/oIfp6TzT03nuvPMhXFfgD+jUlqJK9Q2VtLTUU19XSUVFmEg0iD/gQ9dUNM1L
    Qc+NflrmjMNgks8bTIxPMZwcp68vRXJolNGxKZKpCbq6BmfrWAEOHuzm3HM3XzGSMr8+PGRa
    C7v+PjwRiK4rO4/20d8/gmFYfO1r92BZNq4r0HWVysoK2tc0k6iMUVubIJGIUhELE41FiIR9
    aJo3rZR56XcDw7DIF01y2QKT6Syjo5Mkh8eZGM8wPpGhp2eIQ4eO8div98ymrgcHRuk80ic1
    NdW/7VP/6+h9t9y8YpH2nwq4tqqpqKrKQpcCvc6xRFHVUhO5WHa88NCDv/V6BxZlPoVYNF7w
    yl+k2ZrqE79NCCgUTPL5CQYHx2cdZ6nkAKuKwssvH+Xll7twXRdNVVi/voXu7iSf/cx3ZwfC
    KopMoWCwd2/nnM+YP+jKuw9eudpCwWDvniO89GLngvLCPIdl3vcXyBcyOAsshpHP52f7AWaG
    es3+piXghampHA0NVezYvgGfb0bwxZvJY5om2WyRXK7gzVcomtiWg207GIXivGyMpnsNz6qq
    eDMK/DrBoJ9oJEgwGECSZ8p0FVRV5cwz11JfX0VP99DS8kJfCkWR2bplDS0t9RxvPBQI1yaX
    L5LNFsjnixQKRUzLxi5lD8yieXwOhQSBoB9ZVfHpmvcMAj4i4QCRcBBN1ziuHKVSW1vFtm3r
    efqp/QwMjLxpXjgJh2EUKBZlGbOqyhu0kkxOLPLFLBgdSwMSsYpwyVCCxdOSfj0MAquv7DzS
    5z+wv4fGxirWrV3JYpciObZ9qj4JkJ7Wdb23paVuhSRJbN++lg9c8y4UWZ1z8CEYilBdXcPW
    bRsBL8KXyWQZGZ3EslyqqjzdeEnSMM0i//7d+/nZT5+gpibOjX/5Xr70pbu4794naVtZz3XX
    XcnWbZvYsnUd42MTpEam0DSZqqoYsVgERVGBGQ/9+G+IRmM0NTWx6Yy1DA+PsXdvJ01NNaiq
    NlAoms+f6oYz13VZt6aZz33uRnx+/5I7DEaxyLUf/iyHOvoX6TujgNhqO4XT9+7uwHVdYhVh
    1ra3sHXbOjZvbmflynrqGyoJBgMlI0l+FdNF/B5urxc5cl0X0zAZGUnT159i//5unnrqZQ7s
    72ZsbIrdLxzmjz9UqA8F9KaVK3w9C28IHlDAWf3yvm5yOQNZ9u6lVasa2bp1LadvXEXzilpq
    a+LE41FUVZ1jUEm/w5QT86JGnkqMw1R6mpHRNMPDY3Qc7uWF3R0cPtzLSGoSwzDZvfswb79w
    +5ZP3bxSnxizF8ly9wNWLhwKinA4KGWmlqosSZTmd/iYkeRdTrxw8OCxeVLSC+rCqfIi8EL9
    ipnhgr/rfZ8oJfzK+9Tb547j0thYxQ03vJcvf/kujh1LvuJzlJNo1D4+BJEF5YXXwq3fuIcf
    /uCRBVdJmkxPY1vOkpUFnsgLtu3QurKeeDxEx5EuBvtTHOkcoKdniNHUBOkpb/hYLp8nny9i
    mrYnqS1AzOEGSfLSRVpJZjcQ9BEJB6moCBGLhampTdDUVENbawONTdUk4lFWr27iaOfAEvNC
    EZ9P48xNq0ilRhkYGKGne5AjRwcZ7E8xMZElk8mRzebJ5vIYpoVlOK/aAD3jkPv8updVLs0n
    qagIE09EaWioZkVLHStX1tPQWMWGDa0Egz5yOeNN88JJOAwuEAGKtLTUIQQcOtSDEC7SIqVY
    hRB0dg0Cgvr6BKYJA70WldXLIcWrA8WVnZ395PNFmppricUjnGzq6WQNx3wpLfzmEQBaO2Tl
    8Dc++MF3/OOuJ1/y/eQnu9i2fR2ypHDwYA/J5Dg+v0brykaam6pIVMaIxsLEoiEikSDtca/H
    RQhvkFdPzxA//vGvufvux5AkiQ9ccwnvfNf5ZHMGn//89/j6v/6Y4eExrrr67bS1NVJZlaCq
    uhoA2zaZzuRJT02TyeSZmJhioH+Erh5PsauhvooN69tQVZl77n6MNWtauOaaSw1d1b/20tPu
    /rYNC3NBK4paSjEuLRRFXTSS8Bpd+wC1bnRkrLqjow/XFbQ01/Anf/pONm9dR21NVcm5EycY
    wO4bdBhm/q0gywr+gEbLihBNzbWsXt1EW2sdd9zxc8bGvLK34eR4e2vris1g93Qcepo16xZK
    ZjEGsNUoptbtfv4gjuPQ2tbEh6+9nHdcupOqqsqSYzuzZneeA/BGHUxvOrpCVXUVVdU1rF/f
    zoUX7iQ9mebxx/dy++33laK0neSyhQ0BPbS1rlZ5ZsFvHCmOEAcBWVRWxkkkogwOjC5RhkEQ
    i4WIhEPkczaW7S4rXijki4tyRiXJMyqOl30sFC9c+o9P/uYl30svHkVRZiRRxWxj66yh9/ta
    GI5LMOjjjz54CZe963xyOYMvfOE/mZiYfkNZmeO/43jiTpI8mdYNG1oXnBdeDR2He0vD/RbS
    XWZexnXxz98reUGSJIaGRvmfN32Vo91DTE5MY9sOjuOW9slxYYe57+q1YJk2+VwRMTZfVUmW
    vXUrikIsFmLFijpGRiaRZXlJeWGmJO1733+Yr996D339IxhFE9t2Zxug52bIftf6hQDDsJhK
    55g7p2gmW6coMj6fRkNDFdFoCLf0/2+WF07CYQgB37XhA79avbrxraGQn2ef2c+3b/8pqq4t
    SrrLdQXPPP0ykWiItrYmLEMMH3ixwOp11dLBfQfF+tPXLyExJACHZGocx3XRfRpjo5Oo2uIZ
    k7IskRqZOCVRDElahRBHBQS+2djUdPF1111x8S0338on//pWDMNEK2nom4ZFsWiCAN2nEk9E
    qaqMUZmIEE9EUXWVfNZgZGSSI0f6GB+forGpmhuufw9XX33hUVXR/v1977/oT4LhwOovffE/
    ufPOh3jgF8+wpr2ZquoKwpEAtuWQnswyOZFhZHSSyclpDMNCIAj4fei6huN4jVB+v04+X+Sv
    b76GFS0Nu/JZ+7bTzhQiHD21Ex4lSWJoeJx/v/MBVE1lScOZkoRt2QwNjy9iZMkFFD0Q8HH2
    2esZH09z4MAx/uqvvkFzSy3tq5tZu34FrSvrqaurJBoLEw4H8Pt1NE1D11RUVZkjuTpzI7o4
    joVp2hQNE8OwME2LXK5IejLD4NAYPT1D9HQP0dU9SF9fimLBJBoNctqaFnRdk0zTWq3rDYQj
    CznfLwvIDYZlVvn8XmlRb2+S22+7lyef2Mv6DW1s3LjKm7IZCRGJBAiHAyiKdhJM7FIoeqnr
    XC5POp2l99gQRzoHOHzoGIcP95FMjnuGgiRTKBYrIzXhOpDYs/dptmxeaG36EOAeSCQiHW2t
    9Wv3vXx0qQ4CbasaqagId0xNGAfsZeEwzOeFxTqfuVyBkZExxALzwrXXXn7xzZ+8lVyuWFK+
    CeL36RiGRTZbwHGcWQN+5h3NNY6OZxegqama66+/8pW88KXv03csOVtyNNd48tYk5gUXFFUm
    HA7i8+lYpsX0dB7LcggGfXz42ncvKC+8JjcrMsshrLkUvNDdM0zn0UEUWZ5VvDtZp+a4UQ2v
    VvpmWTYjI2lveKQkEY2FlpQXVFUhny/ym9/sKwV9jpfunUygca6s6qsFol1XkM8bHDkygFu6
    b2RZetO8oL7xH7oCIQ4BvLSqrVm0tdVLL73UxWc/+x+Luh1t22HLlnZamuuFqsl3XX615hza
    v+A6lr+HZ62p0OWbyuQQruDZp/dz3XX/vKipwRnFIddxZ3V939xjCQGZafDf+vYLzzr/iiv3
    +YeHJ7j0ku20tjUSDPrJ54uMjU0xNjrJ8PAY/QMjDA2N09U9TPblbi/NVjTx+XTq6yt597vP
    5aqr32ZvWL96tyQrt0jyGY9L4sXfXP7uC/61dWXdxu/9x4M89dR+XtjdgW3b6JpGMOgjEvFS
    b2vWttDUWENDQw219Qmqq2OEggEKRYNjPUkefvhZEokIF1+0M22a7he0gJuZGjvFutulv48e
    HeSzn7lz2VzVjuNSXRNfpINQDchPV1Q4n7nllj/90BVXXNDyxBN7+e1zBxnoH+GRR5/n5w88
    haooKKpCJBqistJrlPf7/QQCOnqpFnNu/b5re7Wt+bxBLlcgVzDI54qk09PkCwau7WI7Dpqu
    UVkZZdPpq9i2dS07z9rApk3tuVAo+GvTcu/R9RTOgtqLPkDaE43Ef/H3f3/t284/74zAo4++
    wL59XTz88HM89NBzaJpKMOSnurqCqsoYiUSUSCxEOOQnFAoSCPjw+726/7kmj23bFA2bYsEg
    M5Ulnc4ylc4yPplhYnyKiUmvwdOyvHr4mpo4O8/ayAUXnMGll57tJhIVj9mOs1dVFuvuyQOh
    fkXxD+7csX7t/fc/taBqMK8XMNmxbR2qonfjWv3xhLqk5/FEXnAcd5HUkQQ/v+8pfvvswUXh
    hSvfs8+fTE5w6SU7WNnacAIvpBkeHmVwcJTh5ART6Ry5XIHprKeWpus6tbVxzjvvjN/NC0/v
    Z3w8g+M4aKpGKOTxQjQWorY2TkN9DQ0N1a/ghd6eJA//8lkS8YXnBdf1IsdCLJ/+Sq+fwV1S
    Xnju+UMMDowyMTFFNlecs/2OC9fMGsOvbgtzvJd+brnhfEfUH9BJxCM0t9Sybcty4YXd7D/Q
    xdhoGi+IIWbXPS+zgPSGilHmynIfDwZ4n6tpClVVlbStajwlvCCd3MYbAtwWIcYffOCBXesP
    7OtCkhf3YAhXsPH0Vi659Lx9qZR1meuKgRUrzl4GxKDq0P2Tr3z1e+/63p0Pe9MplyDqPBOF
    Ea6gvr6Sr371JtpWrbzBtu3bfL4tJ7G2JBBSoe9z6fTYJ2RZng6Ho4/LsnI/YHrHXMRsy36H
    JNsNtu1gGlY8k8k3TaanpPHxKXL5AqFAgBUr6+3GhupjiqL9xLT4fDrtjkciEqapoyjW6lBI
    /phlFi7r6RlaOTA4olq2hU/3Ea8IU5mIiVg8MuD36xlVVh0hlCHNpz0E0lQpvK+7rvPubDbz
    NsuyI7Fo7IuHDuduqapSnYaGs0/Re94DqFcnk6kf3fPjx8hlC8ur715AKBzgD696G3V1te8D
    ++6FnE8ixF5Mw5ZlSVmnqMoHXeFclsvlK0dHJpr7B1L09iYZHBihf2CMVGqCbCZH0TApFk3M
    0hwG27ZnU7Oe0Sejql72Ycah8Pt1wuEANbVxGhtraGmpobGphqbGahobqgfD4dCYLGu/KBbc
    XflscVfv/qF8w2nV1LWcv4BrHwLqEaI3LETmUgn3w7lcbkPPsaGWnp5BentTdB7pY2BwlInx
    DJnpmRpdC6c0qGgm2vRaZDDzR1EVfLpGMOgnEvFkCFtaamlra2T16kZOa2+hubluKBwK9jmu
    9MNs1rzLNO1kOBwgHN6yCPffEPAYsOMTg4M9X/zx3Y9iFM3FPRulDOdVV72d2tr6GzRdva2/
    P0tLy1nLixfkxSobnF8DXeaFMi8sN14YGBgjNTJJbjpH0bBKcxbm8oKYFwxQVBVVldE1DZ+u
    ofs1QiE/tTVxGkq80NRUQ+Ny54WS/HB6fJrpXIFCwcA0TCzL+b3LBiXZ6+fQNY1A0Ec4FCAS
    CdDcvDC8cJIOQwfwEeBf/odwzetsx12SnK+EpOZz7q6jB/I3bj3vgiXPO3vE8DjQ/s1UKnn9
    0PDIsrgkdF2jra0JVfXdIIQ4KWLw1tcJiHYo3Gpb7l3pKXFnVZVjSNJ2AH78hee58E81fyQu
    qa7jKkZBtPh8yjbNLysgBQEVhOs67qhRtB/PTxdSsiLbiZrjKbHxib24LlrQJ2r9AfUtsiJX
    4xXh2kDeMhzHKDovIDMU8CvG9KSwH3vMLv7BH3kp5fb2x3nmmZgeC0sfkhXxQcsUHxGCw/7Q
    llP4nvcBXALiS7ZtC+GK5UUMx40DCfgE8EtNW3iDMZXaiyShRKNSwLbcVn9A3aEoUtS2nYsl
    7BbLtmXLcnzFgtGUyxt6Ll/AKBqlciNznhSnoijouo7uUwn4/QQDfgJBvxkM6gOappqqqtpC
    KFOqqjyIkIaNgr3bRfTIjpxzXeGE4psX9VnnJveSGXNJNKo+2zbbfAH9HEVVYo5tXmI7Voth
    mP70ZLZpfDKjZTI5crk8hZwxO4vBMC2K+eJsxEySPM10n99HKKgTCgcJBH0EA34qKiIkElER
    CQeHQiF/RlU1U5K1h2RZ6ioWrOfNonN0KJXLuS5iw4a3LvId2AHQCsattm03C1e4S3A2ZJD6
    raK4EegJRbeWeaHMC2VeKPPCfzlemBnKNqsGxtLygvQmiUFxHTNo2e6SHAvHcRkbKdqPPJjK
    f/jG9ywPJ150AkozFN4B7rIpVxTClixLPAEcOnli6AJUyXWL0WKxkAXXCYW2LLsLMTe5BwGK
    gxt2cTMSkqiIbz/VRpEKUtCyjHmR8eXzvimlxEVekrCX6j0JcYh02ghEIpJWLLq64zhRCXmn
    qikxn08SJ6gFzWlkEKUOYU8G0DSFZBrOlMD9rSzLWZ9fKU5O2s7oaDYvK4h1a9+y/JiZGzEL
    1weELPzFop3QNPWsQEAJe3Eh/IBu29ZpjiPWC0GFhBCzuhECXCEhIVxFlfaoqn7Qex4UQNiW
    5dqm4T4vSWJQVRUjMy3ygFNdfeYy2HsjIMaClmWprliyvW/btp2PxbYvk/NY5oUyL5R5ocwL
    /7V5YRn6v2WUUUYZ/6/j0wjxtxw4sEurrAz5dF1VdF1C10uVrTYUDHAsk3S6UFjd3m5K0reA
    vys/ujLKKKOMMi+UeaGMMsooo4wyyiijjDLKKKOMMsooo4wyyiijjDLKKKOMMsooo4wyyiij
    jDLKKKOMMsooo4wyyiijjDLKKKOMMsooo4wyyiijjDLK+P8L/xclujTw6Wz5JgAAAABJRU5E
    rkJggg==
  }
  set pieceImageData(Merida2,70) {
    iVBORw0KGgoAAAANSUhEUgAAA0gAAABFCAYAAABwkcaYAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAsTAAALEwEAmpwYAAAAB3RJTUUH2AUBChwuddMJswAAIABJREFUeNrsnXd8VUX6
    h58559bc9F4JLUAKLfQiCAqIiqKgqKgrllVRd9e+6v7c1bWvq66uFRXErmCj994hBFJogYSE
    kN6T28+Z3x83IEhZdy0QvN/PB4XcOzl35s68z7zvvDMDfvnll19++eWXX3755Zdffvnll19+
    +eWXX3755Zdffvnll19++eWXX3755Zdffvnll19++eWXX3755Zdffvnll19++eWXX3755Zdf
    fvnll19+/UcJfxP45Zdf56qkLAICQsBl9f3E7AB7gxDt/Y3jl19++eWXn0d++R2kX28QFAIi
    GDzXgrEQWlZCgVuIy/2N87O0byOAufWfLiGC/Y3i1w/6SBZgNoA6DtTbQU/3vaLkgf42BC2C
    ODs08VvrP762cQhIGgt6N1C2w/0r4BGEyPR3Hj+P/PLzyC8/j37zPBJnpkGKABtQrYA1HvTB
    oClg8IJYA45K6CaFEG30Cy8AREdwbQKPEQzLwfwVWNdCfBF4EcJ4irKy9XvZr0K9Bor0T1qO
    9JkQAY3DoCUBrGN9rzgXgK0UAldDo2xrkRgpcwBrNHiGg6L+h3cL0CuB5YAEUkH0AkWevpwu
    QGYDu1rH/EhQokH8p3IaGFeBo1KI7mewjRqAYKDMANowcMb4qm/YBwFZoOtCxB/z/sMA7aH+
    Nper+Z7cnL1Bu3YdRNMhNTWZHj26tFgsYWvBdg+w78SyugJkgiPF11yWClBWQ5wXGhEipI2P
    pfWAbYqm6f9sbrGHBQYGlKmKeg+4ZjsawggI7fwbtS9eAYoBxEDQEn2vGA6BvhF0LxhkW4z0
    +nnk55GfR34e+XnUBhwkXzTLYAD7IPDcAVove4sz1W53CKvVotkCrXmgbgHT2xC0FZxSiA5t
    7AvPBdSOmubYuHrV5iiBQp9+GQQFhe4B41ugv+mLNHU9SdvYgJbrwH4HyIVg+xD0EiE6/caj
    L72B3MlSitcqysvDikvKkRKSkqKJi4urE4J7IONj2N6mouBSbgMCR+q6fV59faNF8+qnfK8t
    0EpAgGWNDyh4QfzZ43Y/29DYgtRPzhahCEKCbRhNpkdAPgcYgOV2u/O8lmbHKZ+lGhRCQ4Od
    ihJwCTQvF6LPGWqfJmAO0Ksz8LQuvWOaGppCpNQJDgkqUxTTN2B6B0qzhRiOlPmtY8j+RH1t
    1eOvvvoZX3+zmurqBgAiIoK5/PKh/PFP1xIeHv2Cx+N6xGAw6IrSHSlXAQm9wP17KT3jGxsb
    43QdQkJCGhTFsAh4DLILYBxCBLXRsVQByHiomvvpJ/N6f/75Um65dRzjxo349nCxMslska7I
    mO6/MftyACAGnBPAO0nTtLTGhuZIKSXBwbZqg9GYD4bPwTIbqBCio59Hfh75eeTn0f/Eo2+/
    W0NNTSMAYWFBXHbZED+PzlIeGX59w1KtgukOTdOfzc/dE/jdnHXk5xfS3OwgMNCqZmR06nHZ
    5UN7pKZ2uUJR3H+A/R9LmdXGUj/cgIqmeZg3fyPz5q6nf/80rpw4ouuw83o9ExYWPhzMr0pZ
    sxG8DiFikHJXa9mayR6P87Xc3N1hMbGR58XHxV+CLqZIPX+vUNJ+gzDaBXQCcia73c7X5sxZ
    HTZj+nz27i1BSujcOZ6bb7407LLLh79mMu0gN1d+7HBkYbW2lf6iAwo1NbX85S9vUX641he2
    OIYvQggQcO21o5l0zahjXjSSk7uPfzz/EXa766ThjwCrmQcfvp7MzO4CvK1lBXO+W82nny4G
    eTRK/H3IREJsfDhPPXWHjIoyO89s+xwEeqaC+73duw8M+uKL5eTs3I+UkoyMDnETr77gzoz0
    buMg4TIp92z31c+Zrmn2KW++NYvpM+YTFhrI8PN7oyqC3NxCZs5chGpQ+fOff3edlMaZLpc7
    T8r5QMIYcLy3e9fehM+/XEZezgE0TSc9vWPIVVdfcHX37imJ0Ps22J4v3RJhaosr3BKQmWDo
    vWXrLtavz2HgoAzGjRuhlVbW6uFhht+YfdkO1LcD47SG+rrRy5ZtZsWKLMoO1yClTnRsROT5
    52cOGz16wLCwsMjLwXublDuKhejp55GfR608WkBBgY9HHTvFcfOUc4tHJ83k+U88crgQP4i/
    S+SP4tGJbS7Pah7l5vh4lJ7+n3k044P56LpEURQA6uqa+fDDH88jXZekpXcIueoqP4/OKQdJ
    yoNAsApldzqdjc9++vGiwLff+YbS0mqMRgNWqwmHw8XqNTuYO28dd989IXzChAtfNantxME9
    +sf2xiwZENxWjIwABEIIzCYTLS1OVqzIYuPGPAYMTLfeduu48QMG9h5jMukvQ8vfpcx1QpKA
    ssler+Nfs2ctDvvnS5/StWs7Hn30psGpqSnT7XbvzW739j0mU+/fGJIMQElnkE989dXKsGee
    noHL5SEpKQoUwYEDZTzxxHTcHm/YpEljnuiQLDZrXva1rToK3B4vefmFFBdVYrNZUFTlqP1w
    uTx4PF7OH9GH470nQWNjC9k7CnA63QQEmKEVZlKX2O0urFYTjY0tgFC+LysoK69h+/Z9GI0G
    zGbj0bVkXdNpaXHSrj4at9sjdN1rPmLMf36bcKQelTHgCvSl2JpLQfcKEYWUeYASDp5Xt27L
    H/TYo2+xK78Io8mAADZvzmP9hjyee+7OxMzMtKllZS33xMQEOBXFcHHxwbKkhQs2ERBg5tHH
    buCyy4aDEMybu5b/+79pLFq0mUnXjE7s2qXjxVLKPEgPAftfd+zclfDnh/5NXl4hRqMRIWDz
    lt1s2JDDM8/eObhfv+5vut2hEwQ7q311qAKEAdwJ4DWAuRmiK446tmdNxLtaQLuJUDUSlGQA
    RSioqurLlEHp3a9v6L9BkVLu/ha6Lmhr0e//bbJrbAf2aQUFhaP/8cKHrFy5HYfDfWQYISUs
    XbqVJYs38dDDN4zu0qX9NK9Tv013ZxUrprbLo5Urt7Npcz79+6f5efQTePTsMzOor28+OuHN
    yy3kySfPHR4dLKzgRPvvY8jpeNTS4jzB/kkpsdksp+XRkX8f57LpOsn1MWcdj/7y2Fvsyj94
    tNzmzbtOy6NFCzehaRJV/f7zK4pA03QWL95yah49/G/y8w8idYkAtmzZzYb1fh6dUw4SNAPu
    /rru/tusL5cGvvjPT/F6NS65ZBDnn9+HqJhQKsurWbpkG6tW7+C5Zz/CoBrCr7hi5Cu2CM3h
    1ZXZbcfAmAEVVdWxBVoRAjp1TsTpdLNyRRZ5ufu56uqR1ptvvuz+6Oi49hD8EpTZwfn0okXr
    wl944WOqqxuorKjjr653ef65qYPbd0j6sxD6LVJm6b+tHHA3oIyrrqrp9NFHC3G7Pdx//zVc
    OWEEihB89fUqXvzHJ3zwwXyGD8/slBAXc6ku9ZezNq8ns//gNlVTRVGwWEw89Ofr6dAhHl3X
    0TSdTz5ZwvKlWxGKADBJKaKFMDpBhAl8PlH75FgeePA6rDYLQkBtbTMv/fNT6mobWnEjQz0e
    GW40qhaQxtbfxXnDe3HddaNQVQVFUSgsPMyzT8/0QVEI5C+QhivlbiARONgfHJeBMha09qA0
    g/4ZGD6WsiwPajRQL25pqTvvrTe/Zs/ug1xwQR/GXjwIIQTz5m1g5Yos3n7rK/7xYuK1kVG2
    WVJqi8CgHjpUQXVNPXFxkQwa1AuT2QLAgIHdiYuLpLS0ioryGrp26WTygcM+yulo6D/t7W/I
    33WQ4ef35tJLh6CqKvPmbWDF8q3869Uv+Pdr7QZYTMEXG03KTCkdKhzsDPoUUK8FbyA4iqB2
    AVi/k7J5MxxCiG5nuGeZgHZXut2Nb+bl7o+oa2hCEYKy8moUReFgcRmrVm/soGv670NDg8hI
    73Sp0ZR3IxiXn7vOUTEQo8CBB8vLK0Y/9ff3WbFiO4oiMBiO33rh9XhZsnQrDoebZ5+dOjop
    Me7BnBz7H6urtuiRUf3aJI9UVcHldLN65Xby8g5w1VUjTsuj2tpGKivq/Tw6hkcff7yQurrm
    4/qLqqo0NDQzc+aCc4JHqqqiKCdHwKl4pKoKqqqc1EFSVeWUPDrWcTjuOUKclTzalV/km9Af
    oz27i07Jo5raxpO2paIIGuqbT8mjvLwiDKoKxzTP3r0lfh6dKw6SlNVABJA/Zd/eAxHTpn2H
    16tx9z0TuemmSwgK+j5/8qKxQ3n1X18wY8Y83njjK7r3SIno1Cl5lMnUe/bePWvp0nVoGzAt
    NsCAokjCwgLRdUmfvl2ZfO0YZsyYx8KFG3nn7e/YvbvE/Ogjv7uuS9fOPUAWHzhQ1O7112dR
    W9vEhaP6U15Ry6aNefzrlc956pnbJwQFBW9zu62fSrnTAZE6RLjAI4UIPIeB5AKM5n0FJRws
    KqN9cizjx59PZGQ4AJdfdh6zvlxOaWkVBw+WkZAQH6AAqG2ztqqq0r9fBunpKYCOpmmsXrUD
    KSVer46ue3vpulwLXmkwqOFeTUdKSVBwAMOHZRIY7BtL5RU1vP3WN9TWNODVdMBzqxDalV4v
    QlFEnNfrK5eYEM3IEf1bDb1CVOS+E4z+z2sL9gJ2oHggaNPLDh/utn79ToqLqwgPs4X3G9D9
    wW7dOt2gKOIp8LwByuDCA6Xm7O17SUiI4oEHJpOe0QWAjIyO7MovZPv2vRw6VGFLS+uWeCRt
    w2azYjabaLE7qW9sJoG41ghnM3a7E7PZSIDV3BoObQLCIgsPHla3bN1FdFQYDz90AxnduwKC
    9PSOFOwrJj+3kKKDh809e4QMBNdMqLwdPI8eOHAwfuPGHFFRXktMTHj4oCE9Mju0T5oA9VNA
    bDzzKcIeQA7dsWNvxL1/+hc1tQ0oQuB2ezGZDCxbupXVq7PRdZ3IiBD+9eq9CZmZPQaAtjxr
    +3oyew8+B+1KA2BPlbpj4pdfLGH16h0oijhplFUIgQKsX5/DBzMX8NBD109MTbe+JYTMaxt1
    PZ5Hx9ZLqIKa6kbenTbnpDx64/XZ1NQ0oqoKUkq2bM7386iVRyXFJ1tdASEUyg/XnBM8Op1O
    xaP/WO4UPDozgZL/jUc7d+w7ha1QyM7edwoeGWlqcvDDYlKCyWQ4KY+2bduNcorn7Mov8vPo
    XHCQwAkc7gVi/MJFGygqKueisYO4+eZx2GwBgHb0nWFhIUy960pycgrYvHkXy5ZuplOnpLHV
    1Zs7CyEL2obpMAOmFrAXx8ZFRBmMKmWHq8no3omnn76Dfv1SefOtr1m2dAvVVXX89a+3ZPTt
    m5bx9eyV5OUWMWhgBs88fTv7C0u570//YuGiTQwclB50zTWjnzMam+8AxQ0tLdD0MZjKpSxb
    B2VVkIEQpnMSS7rum9AjRGsamTgafRHCZ2RaV/p9/1XadG2P+3Nk5X/p0i1UlNeYEXSgNcO7
    tLQKl8vbWvEjZQCpIwCX28P06fNZvHhLqIRQ36YjyMk9cNQ4+8qIY579i8YlgdAB0DR98+bs
    bs8/9yFZ2/ficXtRFIXk9rHi3nsnxV1++Yh7pTStMhi8sq6+EafTTWxsBJGRYa2fURIbG0FA
    gJmKyroje7DMYARESeeUJFe31GTzurU7eeuNr7jqqhEIVWH2l8s5VFrFwIHpdOyY4NI07aCq
    SkBRWpodNDXZiY4KIy4+6mhbRISHEBYaRF1dMw0NLQghdLfL3MVoqr932dINCS+++Cm7dheh
    eTVUVSU9vT0PP3xjt2HDe093OsVNqsqmMxwLBoSMiAilZ68UqqrqEULhwP5DVFXVERoaSEJ8
    FAajgZjYcMLCQgBN/Crd4YypCYi7uaKyJHbOnPXous7pUneEEOi6zsKFG5h83ajY+ISEmy0W
    5f62yCOTyYDXqx+d4B1J81mxfNsJPMrPLzoa9T7y/kWLN/t51Mqjk27PEa0QOmd4dBLH4rQ8
    8pzavXSdmkcnJtf9WrbxeB7t2FGA2+1FVRUSk2LEvfdefQKPXC7PKRwkaLG7TsqjLl3bmSsr
    d6L84HBAXddJSUk6OY+a7ad8jsejnZJHe/cW4/VqqAaV1G7JPPzw9d2GDc/08+jsdJCaADXI
    bm8My8stwmAwMHJkH2w223HO0ZEJWlRUBEOG9mLLll3k5R3A7XaFRUQEWnTd20bMhwXIq4DQ
    Bcnt4vqEhQVz+FA1paWVJCcnct3ksXTpmswzz3zA1i27ePz/pnHn1CtYtGQzgYEWbr1tHLFx
    0UTHhHPDjWN5+eXPeOXlz1m0aJMtOiYsPaVzO9LSO9CuXczQ+Lg4qRocWRA4Ew7OknLBYRiB
    EJZzxBRbAZWkxDji4iIpKirj229WcNllwxBCMGfeegqLyklKjCIxMVoDvUpq+jkV9ZZSous6
    +bn7yd1ZcJKJm/Q5j8cctiABXUo8bi+rV20//iCG1knRUafzV6uHE9gOhE44UFjc7Ykn3icv
    t5CMjA6kpCRSXl7Ltm17eOqpD4iOCus8dEifp8C8JyEhhrCwQIpLKti2fRcXjRmIlLBxQy41
    NY1ERoYRHh4C6PbWiPmc4OCQtVPvvPKC0tJqvv12DcuWbUUIQVOTnfbtY7nnrgmEh4dutDd7
    5wYEmQBPc3R0GHGxEVRU1rFu/Q4uuXgwiqKyI2cfJYcqCQqyER8bha7rzQaD/nRubkHnv/3t
    XcrKaunbtxvt2sVQdLCcndkF/O2Jd/nXq/d1657R5RohYjYVFqynQ+cz1SdNgPpxx47tR73w
    /B8z3B4PRqPg0Uff4ssvljFmzEDuv/9apFQwmgxOm9W80tWifS0EZPY5F1ePjqDPGVt44DDl
    5bU/Kj9fCEFdXSN7C0ro2Ck51hcdbns8Cg0Nap2UiOPqpqqCnJz9R3m0eMlmpJTHOY5CCFwu
    N6+84udRdEw49fXNJ6y467pOTEzYOcsjWh2ZXXmF5LcG2o6duCuKcsqVWE3TWbsmmx9ix5dG
    9+u6Rz/k0ZNPvM/OnftRVQWDQUVKKD5YxtNPn8ijkBAbDQ3NJ9RT1yUREaGn5VFR4eGjY0rX
    Je3bx3L3KXgUGxPBvsZDqOqJ6YqBNutJePQeJSW+lU1fHSS5uQd44on3eOXVED+Pzk4HyQUY
    cLlcNDXZMZkMxMVFctIjS1oVFxuOqqg0NjnwahogUNpIFEaI0Na8VqW6Q4dEGRcfIfbtKaGg
    oJTk