// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "context.h"
#include "fitsimage.h"

#include "alloc.h"
#include "allocgz.h"
#include "channel.h"
#include "mmap.h"
#include "mmapincr.h"
#include "share.h"
#include "sshare.h"
#include "socket.h"
#include "socketgz.h"
#include "var.h"

// *** Fits ***

void Base::loadFitsAllocCmd(const char* ch, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsAlloc(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->load(this, ALLOC, fn, img, ll),ll);
}

void Base::loadFitsAllocGZCmd(const char* ch, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsAllocGZ(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->load(this, ALLOCGZ, fn, img, ll),ll);
}

void Base::loadFitsChannelCmd(const char* ch, const char* fn,LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH, 1);
  loadDone(currentContext->load(this, CHANNEL, fn, img, ll),ll);
}

void Base::loadFitsMMapCmd(const char* fn, LoadMethod lm, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsMMap(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->load(this, MMAP, fn, img, ll),ll);
}

void Base::loadFitsSMMapCmd(const char* hdr, const char* fn, 
			    LoadMethod lm, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsSMMap(this, hdr, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->load(this, SMMAP, fn, img, ll),ll);
}

void Base::loadFitsMMapIncrCmd(const char* fn, LoadMethod lm, 
			       LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->load(this, MMAPINCR, fn, img, ll),ll);
}

void Base::loadFitsShareCmd(ShmType stype, int id, const char* fn,
			    LoadMethod lm, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsShare(this, stype, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->load(this, SHARE, fn, img, ll),ll);
}

void Base::loadFitsSShareCmd(ShmType stype, int hdr, int id, 
			     const char* fn, LoadMethod lm, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsSShare(this, stype, hdr, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->load(this, SSHARE, fn, img, ll),ll);
}

void Base::loadFitsSocketCmd(int s, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->load(this, SOCKET, fn, img, ll),ll);
}

void Base::loadFitsSocketGZCmd(int s, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->load(this, SOCKETGZ, fn, img, ll),ll);
}

void Base::loadFitsVarCmd(const char* ch, const char* fn, LoadMethod lm, 
			  LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->load(this, VAR, fn, img, ll),ll);
}

// *** Array ***

void Base::loadArrAllocCmd(const char* ch, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrAlloc(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->load(this, ALLOC, fn, img, ll),ll);
}

void Base::loadArrAllocGZCmd(const char* ch, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrAllocGZ(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->load(this, ALLOCGZ, fn, img, ll),ll);
}

void Base::loadArrChannelCmd(const char* ch, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrChannel(this, interp, ch, fn, 
					   FitsFile::NOFLUSH, 1);
  loadDone(currentContext->load(this, CHANNEL, fn, img, ll),ll);
}

void Base::loadArrMMapCmd(const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrMMap(this, fn, 1);
  loadDone(currentContext->load(this, MMAP, fn, img, ll),ll);
}

void Base::loadArrMMapIncrCmd(const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrMMapIncr(this, fn, 1);
  loadDone(currentContext->load(this, MMAPINCR, fn, img, ll),ll);
}

void Base::loadArrShareCmd(ShmType stype, int id, const char* fn, 
			   LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrShare(this, stype, id, fn, 1);
  loadDone(currentContext->load(this, SHARE, fn, img, ll),ll);
}

void Base::loadArrSocketCmd(int s, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrSocket(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->load(this, SOCKET, fn, img, ll),ll);
}

void Base::loadArrSocketGZCmd(int s, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrSocketGZ(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->load(this, SOCKETGZ, fn, img, ll),ll);
}

void Base::loadArrVarCmd(const char* ch, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageArrVar(this, interp, ch, fn, 1);
  loadDone(currentContext->load(this, VAR, fn, img, ll),ll);
}

// *** Photo ***

void Base::loadPhotoCmd(const char* ph, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImagePhoto(this, interp, ph, fn, 1);
  loadDone(currentContext->load(this, PHOTO, fn, img, IMG), IMG);
}

void Base::loadSlicePhotoCmd(const char* ph, const char* fn)
{
  FitsImage* img = new FitsImagePhoto(this, interp, ph, fn, 1);
  loadDone(currentContext->loadSlice(this, PHOTO, fn, img), IMG);
}

// *** Data Cube ***

void Base::loadExtCubeAllocCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAlloc(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadExtCube(this, ALLOC, fn, img), IMG);
}

void Base::loadExtCubeAllocGZCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAllocGZ(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadExtCube(this, ALLOCGZ, fn, img), IMG);
}

void Base::loadExtCubeChannelCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadExtCube(this, CHANNEL, fn, img), IMG);
}

void Base::loadExtCubeMMapCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMap(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadExtCube(this, MMAP, fn, img), IMG);
}

void Base::loadExtCubeMMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadExtCube(this, MMAPINCR, fn, img), IMG);
}

void Base::loadExtCubeShareCmd(ShmType stype, int id, const char* fn,
				LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsShare(this, stype, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadExtCube(this, SHARE, fn, img), IMG);
}

void Base::loadExtCubeSocketCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadExtCube(this, SOCKET, fn, img), IMG);
}

void Base::loadExtCubeSocketGZCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadExtCube(this, SOCKETGZ, fn, img), IMG);
}

void Base::loadExtCubeVarCmd(const char* ch, const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadExtCube(this, VAR, fn, img), IMG);
}

// *** Slice ***

void Base::loadSliceAllocCmd(const char* ch, const char* fn)
{
  FitsImage* img = new FitsImageFitsAlloc(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadSlice(this, ALLOC, fn, img), IMG);
}

void Base::loadSliceAllocGZCmd(const char* ch, const char* fn)
{
  FitsImage* img = new FitsImageFitsAllocGZ(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadSlice(this, ALLOCGZ, fn, img), IMG);
}

void Base::loadSliceChannelCmd(const char* ch, const char* fn)
{
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn,
					    FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadSlice(this, CHANNEL, fn, img), IMG);
}

void Base::loadSliceMMapCmd(const char* fn, LoadMethod lm)
{
  FitsImage* img = new FitsImageFitsMMap(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadSlice(this, SMMAP, fn, img), IMG);
}

void Base::loadSliceSMMapCmd(const char* hdr, const char* fn, LoadMethod lm)
{
  FitsImage* img = new FitsImageFitsSMMap(this, hdr, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadSlice(this, MMAP, fn, img), IMG);
}

void Base::loadSliceMMapIncrCmd(const char* fn, LoadMethod lm)
{
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadSlice(this, MMAPINCR, fn, img), IMG);
}

void Base::loadSliceShareCmd(ShmType stype, int id, const char* fn, 
			     LoadMethod lm)
{
  FitsImage* img = new FitsImageFitsShare(this, stype, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadSlice(this, SHARE, fn, img), IMG);
}

void Base::loadSliceSShareCmd(ShmType stype, int hdr, int id, const char* fn,
			      LoadMethod lm)
{
  FitsImage* img = new FitsImageFitsSShare(this, stype, hdr, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadSlice(this, SSHARE, fn, img), IMG);
}

void Base::loadSliceSocketCmd(int s, const char* fn)
{
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->loadSlice(this, SOCKET, fn, img), IMG);
}

void Base::loadSliceSocketGZCmd(int s, const char* fn)
{
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->loadSlice(this, SOCKETGZ, fn, img), IMG);
}

void Base::loadSliceVarCmd(const char* ch, const char* fn, LoadMethod lm)
{
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadSlice(this, VAR, fn, img), IMG);
}

// *** Mosaic Image ***

void Base::loadMosaicImageAllocCmd(MosaicType type, Coord::CoordSystem sys,
				   const char* ch, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicAlloc(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicImage(this, ALLOC, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageAllocGZCmd(MosaicType type, Coord::CoordSystem sys, 
				     const char* ch, const char* fn, 
				     LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicAllocGZ(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicImage(this, ALLOCGZ, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageChannelCmd(MosaicType type, Coord::CoordSystem sys,
				     const char* ch, const char* fn, 
				     LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicChannel(this, interp, ch, fn, 
					      FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicImage(this, CHANNEL, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageMMapCmd(MosaicType type, Coord::CoordSystem sys,
				  const char* fn, LoadMethod lm, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicMMap(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicImage(this, MMAP, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageMMapIncrCmd(MosaicType type, 
				      Coord::CoordSystem sys, 
				      const char* fn, LoadMethod lm,
				      LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicMMapIncr(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicImage(this, MMAPINCR, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageShareCmd(MosaicType type, Coord::CoordSystem sys,
				   ShmType stype, int id, const char* fn, 
				   LoadMethod lm, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicShare(this, stype, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicImage(this, SHARE, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageSocketCmd(MosaicType type, Coord::CoordSystem sys,
				    int s, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicSocket(this, s, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicImage(this, SOCKET, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageSocketGZCmd(MosaicType type, 
				      Coord::CoordSystem sys, 
				      int s, const char* fn, LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicSocketGZ(this, s, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicImage(this, SOCKETGZ, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicImageVarCmd(MosaicType type, Coord::CoordSystem sys,
				 const char* ch, const char* fn, LoadMethod lm,
				 LayerType ll)
{
  if (ll == IMG)
    unloadFits();
  FitsImage* img = new FitsImageMosaicVar(this, interp, ch, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicImage(this, VAR, fn, img, ll,type,sys),ll);
}

// *** Mosaic ***

void Base::loadMosaicAllocCmd(MosaicType type, Coord::CoordSystem sys, 
			      const char* ch, const char* fn, LayerType ll)
{
  FitsImage* img = new FitsImageFitsAlloc(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaic(this, ALLOC, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicAllocGZCmd(MosaicType type, Coord::CoordSystem sys, 
				const char* ch, const char* fn, LayerType ll)
{
  FitsImage* img = new FitsImageFitsAllocGZ(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaic(this, ALLOCGZ, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicChannelCmd(MosaicType type, Coord::CoordSystem sys,
				const char* ch, const char* fn, LayerType ll)
{
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn,
					    FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaic(this, CHANNEL, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicMMapCmd(MosaicType type, Coord::CoordSystem sys,
			     const char* fn, LoadMethod lm, LayerType ll)
{
  FitsImage* img = new FitsImageFitsMMap(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaic(this, SMMAP, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicSMMapCmd(MosaicType type, Coord::CoordSystem sys,
			      const char* hdr, const char* fn, 
			      LoadMethod lm, LayerType ll)
{
  FitsImage* img = new FitsImageFitsSMMap(this, hdr, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaic(this, MMAP, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicMMapIncrCmd(MosaicType type, Coord::CoordSystem sys,
				 const char* fn, LoadMethod lm, LayerType ll)
{
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaic(this, MMAPINCR, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicShareCmd(MosaicType type, Coord::CoordSystem sys,
			      ShmType stype, int id, const char* fn, 
			      LoadMethod lm, LayerType ll)
{
  FitsImage* img = new FitsImageFitsShare(this, stype, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaic(this, SHARE, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicSShareCmd(MosaicType type, Coord::CoordSystem sys,
			       ShmType stype, int hdr, int id, const char* fn, 
			       LoadMethod lm, LayerType ll)
{
  FitsImage* img = new FitsImageFitsSShare(this, stype, hdr, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaic(this, SSHARE, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicSocketCmd(MosaicType type, Coord::CoordSystem sys,
			       int s, const char* fn, LayerType ll)
{
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->loadMosaic(this, SOCKET, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicSocketGZCmd(MosaicType type, Coord::CoordSystem sys,
				 int s, const char* fn, LayerType ll)
{
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::FLUSH, 1);
  loadDone(currentContext->loadMosaic(this, SOCKETGZ, fn, img, ll,type,sys),ll);
}

void Base::loadMosaicVarCmd(MosaicType type, Coord::CoordSystem sys,
			    const char* ch, const char* fn, 
			    LoadMethod lm, LayerType ll)
{
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaic(this, VAR, fn, img, ll,type,sys),ll);
}

// *** Mosaic Image WFPC2 ***

void Base::loadMosaicImageWFPC2AllocCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAlloc(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicWFPC2(this, ALLOC, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2AllocGZCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAllocGZ(this, ch, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicWFPC2(this, ALLOCGZ, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2ChannelCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicWFPC2(this, CHANNEL, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2MMapCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMap(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicWFPC2(this, MMAP, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2MMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicWFPC2(this, MMAPINCR, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2ShareCmd(ShmType stype, int id, const char* fn, 
					LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsShare(this, stype, id, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicWFPC2(this, SHARE, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2SocketCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicWFPC2(this, SOCKET, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2SocketGZCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::NOFLUSH, 1);
  loadDone(currentContext->loadMosaicWFPC2(this, SOCKETGZ, fn, img), IMG);
}

void Base::loadMosaicImageWFPC2VarCmd(const char* ch, const char* fn,
				      LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn, 1);
  setScanModeIncr(lm);
  loadDone(currentContext->loadMosaicWFPC2(this, VAR, fn, img), IMG);
}

// ***

void Base::loadDone(int rr, LayerType ll)
{
  if (rr) {
    if (ll == IMG) {
      if (!keyContextSet) {
	keyContext = currentContext;
	keyContextSet =1;
      }
    }
    alignWCS();
    if (!preservePan) {
      centerImage();
      crosshair = cursor;
    }

    // set default scope
    if (currentContext->fits &&	!isCube() && !isMosaic())
      currentContext->frScale.setClipScope(FrScale::LOCAL);
  }
  else {
    reset();
    Tcl_AppendResult(interp, "Unable to load file", NULL);
    result = TCL_ERROR;
  }

  // adjust current slice if needed
  if (currentContext->fits && isCube() && 
      currentContext->frScale.scanMode()==FrScale::CROPSEC) {

    // context->slice() IMAGE (ranges 1-n)
    // params are in DATA coords, edge to edge
    // setSlice() IMAGE (ranges 1-n)
    double sl = currentContext->slice(2)-.5;
    FitsBound* params = 
      currentContext->fits->getDataParams(currentContext->frScale.scanMode());
    double ff = params->zmin+.5;
    double tt = params->zmax-.5;
    if (sl<ff)
      setSlice(2,ff+.5);
    if (sl>tt)
      setSlice(2,tt+.5);
  }

  currentContext->updateClip();
  updateColorScale();
  update(MATRIX);
}

