// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "base.h"
#include "context.h"
#include "fitsimage.h"
#include "NaN.h"
#include "blt.h"
#include "bltVector.h"
#include "projection.h"

#include "sigbus.h"

int Base::markerAnalysisPlot2d(double** x, double** y, double** xc, 
			       double** yc, Marker* pp, 
			       Vector& p1, Vector& p2, int width)
{
  // does not extend across mosaic boundries
  // uses currentContext

  Vector cc = pp->getCenter();
  Coord::CoordSystem sys = pp->analysisSystem();
  Coord::SkyFrame sky = pp->analysisSky();

  FitsImage* ptr = currentContext->cfits;
  while (ptr) {
    Vector img = cc * ptr->refToData;
    FitsBound* params = ptr->getDataParams(currentContext->frScale.scanMode());

    if (img[0]>=params->xmin && img[0]<params->xmax && 
	img[1]>=params->ymin && img[1]<params->ymax)
      break;

    ptr = ptr->nextMosaic();
  }

  // default
  if (!ptr)
    ptr = currentContext->cfits;

  FitsBound* params = ptr->getDataParams(currentContext->frScale.scanMode());

  Vector vv = p2-p1; 
  int num = vv.length() +1;
  Vector s = vv.normalize();
  int cnt[num];

  *x = (double*)malloc(num*sizeof(double));
  *y = (double*)malloc(num*sizeof(double));
  *xc = (double*)malloc(num*sizeof(double));
  *yc = (double*)malloc(num*sizeof(double));

  // main loop

  SETSIGBUS
  for (long ii=0; ii<num; ii++) {
    Vector t = p1 + s*ii;

    (*x)[ii] = ii+1;
    (*y)[ii] = 0;
    cnt[ii] = 0;

    Vector tv = ptr->mapFromRef(t, sys, sky);
    (*xc)[ii] = tv[0];
    (*yc)[ii] = tv[1];

    Vector z = t * ptr->refToData;

    if (z[0]>=params->xmin && z[0]<params->xmax && 
	z[1]>=params->ymin && z[1]<params->ymax) {

      // check for nan
      double v = ptr->getValueDouble(z);
      if (!isnand(v)) {
	(*y)[ii] = v;
	cnt[ii] = 1;
      }

      Vector ss = Vector(-s[1],s[0]);

      for (long jj=1; jj<width; jj++) {
	Vector tt = p1 + s*ii + ss*jj;

	Vector zz = tt * ptr->refToData;
		
	if (zz[0]>=params->xmin && zz[0]<params->xmax && 
	    zz[1]>=params->ymin && zz[1]<params->ymax) {

	  double vvalue = ptr->getValueDouble(zz);
	  // check for nan
	  if (!isnand(vvalue)) {
	    (*y)[ii] += vvalue;
	    cnt[ii]++;
	  }
	}
      }
    }
  }

  // average if needed
  if (pp->analysisParam() == Marker::AVERAGE)
    for (long ii=0; ii<num; ii++)
      if (!isnand((*y)[ii]) && cnt[ii]!=0)
	(*y)[ii] /= cnt[ii];

  CLEARSIGBUS

  return num;
}

int Base::markerAnalysisPlot3d(double** x, double** y, Marker* pp, 
			       const BBox& bb)
{
  // does not extend across mosaic boundries
  // uses currentContext, firstSlice

  Vector cc = pp->getCenter();
  Coord::CoordSystem sys = pp->analysisSystem();

  FitsImage* ptr = currentContext->fits;
  while (ptr) {
    Vector img = cc * ptr->refToData;
    FitsBound* params = ptr->getDataParams(currentContext->frScale.scanMode());
    if (img[0]>=params->xmin && img[0]<params->xmax && 
	img[1]>=params->ymin && img[1]<params->ymax)
      break;

    ptr = ptr->nextMosaic();
  }

  // default
  if (!ptr)
    ptr = currentContext->fits;

  // if more than 3 axes, walk it forward
  int num = currentContext->calcSlice();
  for (int ii=1; ii<num; ii++)
    if (ptr)
      ptr = ptr->nextSlice();

  int srcw = ptr->width();
  // will be incorrect for multiple ext/file cubes
  //  long srcd = ptr->depth();
  int srcd = currentContext->naxis(2);

  // slice jump vector
  FitsImage* sjv[srcd];
  FitsImage* sptr = ptr;
  int ii=0;
  while (sptr) {
    sjv[ii++] = sptr;
    if (sptr)
      sptr = sptr->nextSlice();
  }

  // init
  int cnt[srcd];
  *x = (double*)malloc(srcd*sizeof(double));
  *y = (double*)malloc(srcd*sizeof(double));
  for (long kk=0; kk<srcd; kk++) {
    (*x)[kk] = 0;
    (*y)[kk] = 0;
    cnt[kk] = 0;
  }

  // take the bbox and extend to lower/upper pixel boundaries
  Vector ll = (bb.ll*ptr->refToData).floor();
  Vector ur = (bb.ur*ptr->refToData).ceil();

  FitsBound* params = ptr->getDataParams(currentContext->frScale.scanMode());
  // main loop
  SETSIGBUS
  for (int jj=ll[1]; jj<ur[1]; jj++) {
    for (int ii=ll[0]; ii<ur[0]; ii++) {
      if (ii>=params->xmin && ii<params->xmax && 
	  jj>=params->ymin && jj<params->ymax) {
	// shift to center of pixel in DATA
	Vector rr = Vector(ii,jj)+Vector(.5,.5);
	if (pp->isIn(rr*ptr->dataToRef,Coord::REF)) {
	  for (int kk=0; kk<srcd; kk++) {
	    (*x)[kk] = ptr->mapFromRef3(kk+.5, sys, 2);
	    if (kk>=params->zmin && kk<params->zmax) {
	      double val =sjv[kk]->getValueDouble(long(jj)*srcw+long(ii));
	      // check for nan
	      if (!isnand(val)) {
		(*y)[kk] += val;
		cnt[kk]++;
	      }
	    }
	  }
	}
      }
    }
  }
  CLEARSIGBUS

  // average if needed
  if (pp->analysisParam() == Marker::AVERAGE)
    for (long kk=0; kk<srcd; kk++)
      if (cnt[kk]!=0)
	(*y)[kk] /= cnt[kk];


  return srcd;
}

void Base::bltCut(char* xname, char* yname, Coord::Orientation axis, const Vector& rr)
{
  int size;
  if (axis == Coord::XX)
    size = options->width;
  else
    size = options->height;

  long length = (size+1) * 2;
  double* xx = (double*)malloc(sizeof(double)*length);
  double* yy = (double*)malloc(sizeof(double)*length);

  if (!currentContext->cfits) {
    for (int ii=0; ii<=size; ii++) {
      xx[ii*2] = ii;
      xx[ii*2+1] = ii;
      yy[ii*2] = 0;
      yy[ii*2+1] = 0;
    }
  }
  else
    bltCutFits(xx, yy, size, axis, rr);

  Blt_Vector* xv;
  if (Blt_GetVector(interp, xname, &xv) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xv, xx, length, length*sizeof(double), TCL_DYNAMIC) != 
      TCL_OK)
    goto error;

  Blt_Vector* yv;
  if (Blt_GetVector(interp, yname, &yv) != TCL_OK)
    goto error;

  if (Blt_ResetVector(yv, yy, length, length*sizeof(double), TCL_DYNAMIC) != 
      TCL_OK)
    goto error;

  return;

 error:
    result = TCL_ERROR;
    return;
}

void Base::bltCutFits(double* xx, double* yy, int size, Coord::Orientation axis,
		      const Vector& r)
{
  // Widget
  Vector rr = r * refToWidget;

  // basics
  FitsImage* sptr = currentContext->cfits;
  int mosaic = isMosaic();

  // variable
  FitsBound* params = sptr->getDataParams(currentContext->frScale.scanMode());
  double prev = currentContext->frScale.low();

  // main loop

  SETSIGBUS
  for (int ii=0; ii<=size; ii++) {
    double vv = currentContext->frScale.low();

    if (mosaic) {
      sptr = currentContext->cfits;
      params = sptr->getDataParams(currentContext->frScale.scanMode());
    }

    do {
      Vector img;
      if (axis == Coord::XX)
	img = Vector(1+ii,rr[1]) * sptr->widgetToData;
      else
	img = Vector(rr[0],1+ii) * sptr->widgetToData;

      if (img[0]>=params->xmin && img[0]<params->xmax && 
	  img[1]>=params->ymin && img[1]<params->ymax) {
	double value = sptr->getValueDouble(img);

	if (!isnand(value))
	  vv = value;

	break;
      }
      else {
	if (mosaic) {
	  sptr = sptr->nextMosaic();
	  if (sptr)
	    params = sptr->getDataParams(currentContext->frScale.scanMode());
	}
      }
    }
    while (mosaic && sptr);

    xx[2*ii] = ii;
    xx[2*ii +1] = ii;
    
    yy[2*ii] = prev;
    yy[2*ii +1] = vv;
    prev = vv;
  }
  CLEARSIGBUS
}

