// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framebase.h"
#include "frame.h"
#include "util.h"
#include "fitsimage.h"

#define BLT 1
#if BLT == 1
#include "blt.h"
#endif

#include "NaN.h"

void FrameBase::bltHist(char* xname, char* yname)
{
#if BLT == 1
  int num = 256;

  if (!currentFits)
    return;

  if (DebugPerf)
    cerr << "bltHist...";

  double* histogramX = currentScale->histogramX();
  double* histogramY = currentScale->histogramY();

  if (!histogramX || !histogramY) {
    histogramX = currentScale->initHistogramX(num);
    histogramY = currentScale->initHistogramY(num);

    for (int i=0; i<num; i++)
      histogramX[i] = (double)i/(num-1)*
	(currentScale->max()-currentScale->min()) + currentScale->min();
    
    for (int j=0; j<num; j++)
      histogramY[j] = 0;

    if (currentScale->max() > currentScale->min()) {
      FitsImage* ptr = *channelFits;
      while (ptr) {
	FitsImage* sptr = ptr;
	while (sptr) {
	  sptr->bin(histogramY, num, currentScale->min(), currentScale->max(), 
		    currentScale->scanMode());
	  sptr = sptr->nextSlice();
	}
	ptr = ptr->nextMosaic();
      }
    }
  }

  Blt_Vector* xx;
  if (Blt_GetVector(interp, xname, &xx) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xx, histogramX, num, num*sizeof(double), TCL_STATIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* yy;
  if (Blt_GetVector(interp, yname, &yy) != TCL_OK) 
    goto error;

  if (Blt_ResetVector(yy, histogramY, num, num*sizeof(double), TCL_STATIC) 
      != TCL_OK)
    goto error;

  if (DebugPerf)
    cerr << "end" << endl;

  return;

 error:
  result = TCL_ERROR;
  return;

#endif
}

void FrameBase::bltProjection(char* xname, char* yname, 
			      char* xcname, char* ycname,
			      CoordSystem sys, SkyFrame sky,
			      const Vector& p1, const Vector& p2, 
			      double width, int avg)
{
#if BLT == 1

  if (!currentFits)
    return;

  // the number of pixels sampled is the greater of either dimension
  Vector ap1 = p1;
  Vector ap2 = p2;
  int xnum = ::abs(ap2[0]-ap1[0]);
  int ynum = ::abs(ap2[1]-ap1[1]);
  int num = (xnum > ynum ? xnum : ynum)+1;

  double* x = (double*)malloc(num*sizeof(double));
  double* y = (double*)malloc(num*sizeof(double));
  double* xc = (double*)malloc(num*sizeof(double));
  double* yc = (double*)malloc(num*sizeof(double));

  if (!isMosaic()) 
    bltProjSingle(x, y, xc, yc, num, sys, sky, p1, p2, width, avg);
  else
    bltProjMosaic(x, y, xc, yc, num, sys, sky, p1, p2, width, avg);

  Blt_Vector* xx;
  if (Blt_GetVector(interp, xname, &xx) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xx, x, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* yy;
  if (Blt_GetVector(interp, yname, &yy) != TCL_OK) 
    goto error;

  if (Blt_ResetVector(yy, y, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* xxc;
  if (Blt_GetVector(interp, xcname, &xxc) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xxc, xc, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  Blt_Vector* yyc;
  if (Blt_GetVector(interp, ycname, &yyc) != TCL_OK) 
    goto error;

  if (Blt_ResetVector(yyc, yc, num, num*sizeof(double), TCL_DYNAMIC) 
      != TCL_OK)
    goto error;

  return;

 error:
  result = TCL_ERROR;
  return;

#endif
}

void FrameBase::bltProjSingle(double* x, double* y, 
			      double* xc, double* yc, int num,
			      CoordSystem sys, SkyFrame sky,
			      const Vector& p1, const Vector& p2, 
			      double width, int avg)
{
#if BLT == 1

  int* cnt = new int[num];
  Vector s = (p2-p1)/(num-1);

  FitsImage* ptr = currentFits;
  Matrix mm = ptr->getRefToData();

  for (int i=0; i<num; i++) {
    Vector t = p1 + s*i;

    x[i] = i+1;
    Vector tt = ptr->mapFromRef(t,sys,sky);
    xc[i] = tt[0];
    yc[i] = tt[1];

    // check for nan
    double v = ptr->getValueDouble(t*mm);
    if (!isnand(v)) {
      y[i] = v;
      cnt[i] = 1;
    }
    else
      y[i] = getnand();

    Vector ps = (p2-p1).normalize();
    Vector ss = Vector(-ps[1],ps[0]);

    for (int j=1; j<width; j++) {
      Vector tt = p1 + s*i + ss*j;
      double value = ptr->getValueDouble(tt*mm);
      // check for nan
      if (!isnand(value) && !isnand(y[i])) {
	y[i] += value;
	cnt[i]++;
      }
      else
	y[i] = getnand();
    }
  }

  // average if needed
  if (avg)
    for (int i=0; i<num; i++)
      if (!isnand(y[i]))
	y[i] /= cnt[i];

  // convert nand's to low
  for (int j=0; j<num; j++)
    if (isnand(y[j]))
      y[j] = ptr->getLowDouble();

  delete [] cnt;

#endif
}

void FrameBase::bltProjMosaic(double* x, double* y, 
			      double* xc, double* yc, int num,
			      CoordSystem sys, SkyFrame sky,
			      const Vector& p1, const Vector& p2, 
			      double width, int avg)
{
#if BLT == 1

  int* cnt = new int[num];
  Vector s = (p2-p1)/(num-1);

  for (int i=0; i<num; i++) {
    Vector t = p1 + s*i;

    x[i] = i+1;

    FitsImage* sptr = currentFits;
    while (sptr) {
      Vector tt = sptr->mapFromRef(t,sys,sky);
      xc[i] = tt[0];
      yc[i] = tt[1];

      int* params = sptr->getDataParams(currentScale->scanMode());
      int& srcw = params[0];
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      Vector z = t * sptr->getRefToData();

      if (z[0]>=xmin && z[0]<xmax && z[1]>=ymin && z[1]<ymax) {

	// check for nan
	double v = sptr->getValueDouble(z);
	if (!isnand(v)) {
	  y[i] = v;
	  cnt[i] = 1;
	}
	else
	  y[i] = getnand();

	Vector ps = (p2-p1).normalize();
	Vector ss = Vector(-ps[1],ps[0]);

	for (int j=1; j<width; j++) {
	  Vector tt = p1 + s*i + ss*j;

	  FitsImage* spptr = *channelFits;
	  while (spptr) {
	    int* pparams = spptr->getDataParams(currentScale->scanMode());
	    int& ssrcw = pparams[0];
	    int& xxmin = pparams[1];
	    int& xxmax = pparams[2];
	    int& yymin = pparams[3];
	    int& yymax = pparams[4];

	    Vector zz = tt * spptr->getRefToData();
		
	    if (zz[0]>=xxmin && zz[0]<xxmax && 
		zz[1]>=yymin && zz[1]<yymax) {

	      double vvalue = spptr->getValueDouble(zz);
	      // check for nan
	      if (!isnand(vvalue) && !isnand(y[i])) {
		y[i] += vvalue;
		cnt[i]++;
	      }
	      else
		y[i] = getnand();

	      break;
	    }
	    else
	      spptr = spptr->nextMosaic();
	  }

	  if (!spptr)
	    y[i] = getnand();
	}

	break;
      }
      else
	sptr = sptr->nextMosaic();
    }

    if (!sptr)
      y[i] = getnand();
  }

  // average if needed
  if (avg)
    for (int i=0; i<num; i++)
      if (!isnand(y[i]))
	y[i] /= cnt[i];

      // convert nand's to low
  for (int j=0; j<num; j++)
    if (isnand(y[j]))
      y[j] = currentScale->low();

  delete [] cnt;

#endif
}

void FrameBase::bltCut(char* xname, char* yname, 
		       Orientation axis, const Vector& rr)
{
#if BLT == 1

  if (!currentFits)
    return;

  // Vector r is in Widget coords
  Vector r = rr;

  int size;
  if (axis == XX)
    size = options->width;
  else
    size = options->height;

  long length = (size+1) * 2;
  double* x = (double*)malloc(sizeof(double)*length);
  double* y = (double*)malloc(sizeof(double)*length);

  if (*currentMosaicCount == 0 && *currentSliceCount == 0) {
    for (int i=0; i<=size; i++) {
      x[i*2] = i;
      x[i*2+1] = i;
      y[i*2] = 0;
      y[i*2+1] = 0;
    }
  }
  else {
    if (!isMosaic()) 
      bltCutSingle(x, y, size, r, axis);
    else
      bltCutMosaic(x, y, size, r, axis);
  }

  Blt_Vector* xx;
  if (Blt_GetVector(interp, xname, &xx) != TCL_OK)
    goto error;

  if (Blt_ResetVector(xx, x, length, length*sizeof(double), TCL_DYNAMIC) != 
      TCL_OK)
    goto error;

  Blt_Vector* yy;
  if (Blt_GetVector(interp, yname, &yy) != TCL_OK)
    goto error;

  if (Blt_ResetVector(yy, y, length, length*sizeof(double), TCL_DYNAMIC) != 
      TCL_OK)
    goto error;

  return;

 error:
    result = TCL_ERROR;
    return;

#endif
}

void FrameBase::bltCutSingle(double* x, double* y, int size, Vector& r, 
			 Orientation axis)
{
  Matrix m = currentFits->getWidgetToData();
  int* params = currentFits->getDataParams(currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  double prev = currentScale->low();

  for (int i=0; i<=size; i++) {
    double v = currentScale->low();

    Vector img;
    if (axis == XX)
      img = Vector(1+i,r[1]) * m;
    else
      img = Vector(r[0],1+i) * m;

    if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax) {
      double value = currentFits->getValueDouble(img);

      if (!isnand(value))
	v = value;
    }

    x[2*i] = i;
    x[2*i +1] = i;
    
    y[2*i] = prev;
    y[2*i +1] = v;
    prev = v;
  }
}

void FrameBase::bltCutMosaic(double* x, double* y, int size, Vector& r, 
			 Orientation axis)
{
  double prev = currentScale->low();

  for (int i=0; i<=size; i++) {
    double v = currentScale->low();

    // find y first
    FitsImage* sptr = currentFits;
    while (sptr) {
      Matrix& mx = sptr->getWidgetToData();
      int* params = sptr->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      Vector img;
      if (axis == XX)
	img = Vector(1+i,r[1]) * mx;
      else
	img = Vector(r[0],1+i) * mx;

      if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax) {
	double value = sptr->getValueDouble(img);

	if (!isnand(value))
	  v = value;

	break;
      }
      else
	sptr = sptr->nextMosaic();
    }

    x[2*i] = i;
    x[2*i +1] = i;
    
    y[2*i] = prev;
    y[2*i +1] = v;
    prev = v;
  }
}

