// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "bpanda.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Bpanda::Bpanda(const Bpanda& a) : BasePanda(a), BaseBox(a) {}

Bpanda::Bpanda(FrameBase* p, const Vector& ctr, 
	       double a1, double a2, int an,
	       const Vector& r1, const Vector& r2, int rn,
	       double ang,
	       const char* clr, int wth, const char* fnt, const char* txt, 
	       unsigned short prop, const char* cmt, 
	       const List<Tag>& tg, const List<CallBack>& cb)
  : BasePanda(a1, a2, an),
    BaseBox(p, ctr, ang, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn+1;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = ((r2-r1)/rn)*i+r1;

  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  strcpy(type, "bpanda");
  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
}

Bpanda::Bpanda(FrameBase* p, const Vector& ctr, 
	       int an, double* a,
	       int rn, Vector* r,
	       double ang,
	       const char* clr, int wth, const char* fnt, const char* txt, 
	       unsigned short prop, const char* cmt, 
	       const List<Tag>& tg, const List<CallBack>& cb)
  : BasePanda(an, a),
    BaseBox(p, ctr, ang, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = r[i];
  sortAnnuli();

  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  strcpy(type, "bpanda");
  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
}

void Bpanda::updateHandles()
{
  // handles are in canvas coords
  // we can't garantee that the radii have been sorted yet
  // also, annuli[i] could be negative

  if (handle)
    delete [] handle;
  handle = new Vector[numHandle];

  Vector max;
  for(int i=0; i<numAnnuli_; i++) {
    Vector foo = annuli_[i].abs();
    if (max[0]<foo[0])
      max = foo;
  }
  Vector s = max;

  Matrix mm = fwdCanvasMatrix();

  handle[0] = Vector(-s[0],-s[1])/2 * mm;
  handle[1] = Vector( s[0],-s[1])/2 * mm;
  handle[2] = Vector( s[0], s[1])/2 * mm;
  handle[3] = Vector(-s[0], s[1])/2 * mm;

  // annuli_
  for (int i=0; i<numAnnuli_; i++)
    handle[i+4] = Vector((annuli_[i][0])/2.,0) * mm;

  // angles_
  Vector rr = annuli_[numAnnuli_-1]/2;
  double phi = rr.angle();

  for (int i=0; i<numAngles_; i++) {
    double ang = angles_[i];

    if (ang >= 0 && ang < phi) {
      double a1 = 0;
      double a2 = phi;
      double r = rr[1]*(ang-a1)/(a2-a1);
      handle[4+numAnnuli_+i] = Vector(rr[0],-r) * mm;
    }
    else if (ang >= phi && ang < M_PI-phi) {
      double a1 = phi;
      double a2 = M_PI-phi;
      double r = rr[0] - (2*rr[0]*(ang-a1)/(a2-a1));
      handle[4+numAnnuli_+i] = Vector(r,-rr[1]) * mm;
    }
    else if (ang >= M_PI-phi && ang < M_PI+phi) {
      double a1 = M_PI-phi;
      double a2 = M_PI+phi;
      double r = rr[1] - (2*rr[1]*(ang-a1)/(a2-a1));
      handle[4+numAnnuli_+i] = Vector(-rr[0],-r) * mm;
    }
    else if (ang >= M_PI+phi && ang < 2*M_PI-phi) {
      double a1 = M_PI+phi;
      double a2 = 2*M_PI-phi;
      double r = rr[0] - (2*rr[0]*(ang-a1)/(a2-a1));
      handle[4+numAnnuli_+i] = Vector(-r, rr[1]) * mm;
    }
    else {
      double a1 = 2*M_PI-phi;
      double a2 = 2*M_PI;
      double r = rr[1] - (rr[1]*(ang-a1)/(a2-a1));
      handle[4+numAnnuli_+i] = Vector(rr[0],r) * mm;
    }
  }
}

void Bpanda::editBegin(int h)
{
  if (h<5) {
    switch (h) {
    case 1:
      return;
    case 2:
      annuli_[numAnnuli_-1] = Vector(-annuli_[numAnnuli_-1][0],annuli_[numAnnuli_-1][1]);
      return;
    case 3:
      annuli_[numAnnuli_-1] = -annuli_[numAnnuli_-1];
      return;
    case 4:
      annuli_[numAnnuli_-1] = Vector(annuli_[numAnnuli_-1][0],-annuli_[numAnnuli_-1][1]);
      return;
    }
  }

  doCallBack(CallBack::EDITBEGINCB);
}

void Bpanda::edit(const Vector& v, int h)
{
  Matrix mm = bckCanvasMatrix();
  Matrix nn = mm.invert();
  Vector n = v * mm;

  // This sizes about the opposite node
  if (h<5) {
    Vector o = annuli_[numAnnuli_-1];
    Vector n = (annuli_[numAnnuli_-1]/2) - (v*mm);

    // don't go thru opposite node
    if (n[0]!=0 && n[1]!=0) {
      Vector ov = annuli_[numAnnuli_-1]/2 * nn;
      annuli_[numAnnuli_-1] = n;
      Vector nv = annuli_[numAnnuli_-1]/2 * nn;
      center -= nv-ov;

      for (int i=0; i<numAnnuli_-1; i++) {
	annuli_[i][0] *= fabs(n[0]/o[0]);
	annuli_[i][1] *= fabs(n[1]/o[1]);
      }
    }
  }
  else if (h<(5+numAnnuli_)) {
    // we must have some length
  double l = (v * mm * 2).length();
  annuli_[h-5] = annuli_[numAnnuli_-1] * l/annuli_[numAnnuli_-1][0];
  }
  else {
    angles_[h-5-numAnnuli_] = -((v * mm).angle());
    sortAngles();
    startAng_ = radToDeg(angles_[0]);
    stopAng_ = radToDeg(angles_[numAngles_-1]);
  }

  updateBBox();
  doCallBack(CallBack::EDITCB);
  doCallBack(CallBack::MOVECB);
}

void Bpanda::editEnd()
{
  for (int i=1; i<numAnnuli_; i++)
    annuli_[i] = annuli_[i].abs();
  sortAnnuli();
  sortAngles();
  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  updateBBox();
  doCallBack(CallBack::EDITENDCB);
}

int Bpanda::addAnnuli(const Vector& v)
{
  Matrix mm = bckCanvasMatrix();
  double l = (v * mm * 2).length();
  Vector rr = annuli_[numAnnuli_-1] * l/annuli_[numAnnuli_-1][0];

  // we need to insert into the next to the last location
  // new size array
  Vector* old = annuli_;
  annuli_ = new Vector[numAnnuli_+1];

  // copy old values
  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = old[i];

  // save last
  annuli_[numAnnuli_] = old[numAnnuli_-1];

  // delete old
  if (old)
    delete [] old;

  // new size on end
  annuli_[numAnnuli_-1] = rr;

  numAnnuli_++;
  numHandle++;

  updateBBox();

  // return handle number
  return 4+numAnnuli_-1;
}

int Bpanda::addAngles(const Vector& v)
{
  Matrix mm = bckCanvasMatrix();
  addAngle(-((v * mm).angle()));
  numHandle++;

  updateBBox();

  // return handle number
  return 4+numAnnuli_+numAngles_-1;
}

void Bpanda::setAnglesAnnuli(double a1, double a2, int an, 
			    Vector r1, Vector r2, int rn)
{
  numAnnuli_ = rn+1;
  if (annuli_)
    delete [] annuli_;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = ((r2-r1)/rn)*i+r1;
  sortAnnuli();

  setAngles(a1,a2,an);
  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Bpanda::setAnglesAnnuli(const double* a, int an, const Vector* r, int rn)
{
  numAnnuli_ = rn;
  if (annuli_)
    delete [] annuli_;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = r[i];
  sortAnnuli();

  setAngles(an,a);
  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Bpanda::deleteAnglesAnnuli(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (numAnnuli_>2 && hh<numAnnuli_) {
      // new annuli_ array
      Vector* old = annuli_;
      annuli_ = new Vector[numAnnuli_-1];

      // copy up to annuli_ in question
      for (int i=0; i<hh; i++)
	annuli_[i] = old[i];

      // copy remainder
      for (int i=hh; i<numAnnuli_-1; i++)
	annuli_[i] = old[i+1];

      if (old)
	delete [] old;
      numAnnuli_--;
    }
    else if (numAngles_>2 && hh<(numAnnuli_+numAngles_)) {
      hh -= numAnnuli_;
      deleteAngle(hh);
    }

    numHandle = 4 + numAnnuli_ + numAngles_;

    startAng_ = radToDeg(angles_[0]);
    stopAng_ = radToDeg(angles_[numAngles_-1]);

    updateBBox();
    doCallBack(CallBack::EDITCB);
  }
}

// private

void Bpanda::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  BaseMarker::render(drawable, mx, mode);

  // angles
  Matrix mm = fwdRefMatrix() * mx;

  Vector rr0 = (annuli_[0]/2).abs();
  Vector rr1 = (annuli_[numAnnuli_-1]/2).abs();
  double phi = rr1.angle();

  for (int i=0; i<numAngles_; i++) {
    double ang = angles_[i];
    Vector r0,r1;

    if (ang >= 0 && ang < phi) {
      double a1 = 0;
      double a2 = phi;
      double b0 = rr0[1]*(ang-a1)/(a2-a1);
      double b1 = rr1[1]*(ang-a1)/(a2-a1);
      r0 = Vector(rr0[0],-b0) * mm;
      r1 = Vector(rr1[0],-b1) * mm;
    }
    else if (ang >= phi && ang < M_PI-phi) {
      double a1 = phi;
      double a2 = M_PI-phi;
      double b0 = rr0[0] - (2*rr0[0]*(ang-a1)/(a2-a1));
      double b1 = rr1[0] - (2*rr1[0]*(ang-a1)/(a2-a1));
      r0 = Vector(b0,-rr0[1]) * mm;
      r1 = Vector(b1,-rr1[1]) * mm;
    }
    else if (ang >= M_PI-phi && ang < M_PI+phi) {
      double a1 = M_PI-phi;
      double a2 = M_PI+phi;
      double b0 = rr0[1] - (2*rr0[1]*(ang-a1)/(a2-a1));
      double b1 = rr1[1] - (2*rr1[1]*(ang-a1)/(a2-a1));
      r0 = Vector(-rr0[0],-b0) * mm;
      r1 = Vector(-rr1[0],-b1) * mm;
    }
    else if (ang >= M_PI+phi && ang < 2*M_PI-phi) {
      double a1 = M_PI+phi;
      double a2 = 2*M_PI-phi;
      double b0 = rr0[0] - (2*rr0[0]*(ang-a1)/(a2-a1));
      double b1 = rr1[0] - (2*rr1[0]*(ang-a1)/(a2-a1));
      r0 = Vector(-b0, rr0[1]) * mm;
      r1 = Vector(-b1, rr1[1]) * mm;
    }
    else {
      double a1 = 2*M_PI-phi;
      double a2 = 2*M_PI;
      double b0 = rr0[1] - (rr0[1]*(ang-a1)/(a2-a1));
      double b1 = rr1[1] - (rr1[1]*(ang-a1)/(a2-a1));
      r0 = Vector(rr0[0],b0) * mm;
      r1 = Vector(rr1[0],b1) * mm;
    }

    switch (mode) {
    case SRC:
      if (selected) {
	if (i == 0)
	  XSetForeground(display, gc, parent->getRedColor());
	else if (i == numAngles_-1)
	  XSetForeground(display, gc, parent->getBlueColor());
	else
	  XSetForeground(display, gc, color);
      }
      else
	XSetForeground(display, gc, color);
      break;
    case XOR:
      break;
    }

    XDRAWLINE(display, drawable, gc, (int)r0[0], (int)r0[1], 
	      (int)r1[0], (int)r1[1]);    
  }
}

void Bpanda::ps(int mode)
{
  BaseMarker::ps(mode);

  // annuli_
  Marker::ps(mode);

  Matrix mm = fwdCanvasMatrix();
  Vector rr0 = annuli_[0]/2;
  Vector rr1 = annuli_[numAnnuli_-1]/2;
  double phi = rr1.angle();

  for (int i=0; i<numAngles_; i++) {
    double ang = angles_[i];
    Vector r0,r1;

    if (ang >= 0 && ang < phi) {
      double a1 = 0;
      double a2 = phi;
      double b0 = rr0[1]*(ang-a1)/(a2-a1);
      double b1 = rr1[1]*(ang-a1)/(a2-a1);
      r0 = Vector(rr0[0],-b0) * mm;
      r1 = Vector(rr1[0],-b1) * mm;
    }
    else if (ang >= phi && ang < M_PI-phi) {
      double a1 = phi;
      double a2 = M_PI-phi;
      double b0 = rr0[0] - (2*rr0[0]*(ang-a1)/(a2-a1));
      double b1 = rr1[0] - (2*rr1[0]*(ang-a1)/(a2-a1));
      r0 = Vector(b0,-rr0[1]) * mm;
      r1 = Vector(b1,-rr1[1]) * mm;
    }
    else if (ang >= M_PI-phi && ang < M_PI+phi) {
      double a1 = M_PI-phi;
      double a2 = M_PI+phi;
      double b0 = rr0[1] - (2*rr0[1]*(ang-a1)/(a2-a1));
      double b1 = rr1[1] - (2*rr1[1]*(ang-a1)/(a2-a1));
      r0 = Vector(-rr0[0],-b0) * mm;
      r1 = Vector(-rr1[0],-b1) * mm;
    }
    else if (ang >= M_PI+phi && ang < 2*M_PI-phi) {
      double a1 = M_PI+phi;
      double a2 = 2*M_PI-phi;
      double b0 = rr0[0] - (2*rr0[0]*(ang-a1)/(a2-a1));
      double b1 = rr1[0] - (2*rr1[0]*(ang-a1)/(a2-a1));
      r0 = Vector(-b0, rr0[1]) * mm;
      r1 = Vector(-b1, rr1[1]) * mm;
    }
    else {
      double a1 = 2*M_PI-phi;
      double a2 = 2*M_PI;
      double b0 = rr0[1] - (rr0[1]*(ang-a1)/(a2-a1));
      double b1 = rr1[1] - (rr1[1]*(ang-a1)/(a2-a1));
      r0 = Vector(rr0[0],b0) * mm;
      r1 = Vector(rr1[0],b1) * mm;
    }

    ostringstream str;
    str << "newpath " 
    	<< r0.TkCanvasPs(parent->canvas) << "moveto"
    	<< r1.TkCanvasPs(parent->canvas) << "lineto"
    	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

// list

void Bpanda::list(ostream& str, CoordSystem sys, SkyFrame sky,
		 SkyFormat format, int conj, int strip)
{
  int regular = 1;
  if (numAngles_>2) {
    double delta;
    if (angles_[1] > angles_[0])
      delta = angles_[1]-angles_[0];
    else
      delta = angles_[1]+2*M_PI-angles_[0];

    for (int i=2; i<numAngles_; i++) {
      double diff;
      if (angles_[i] > angles_[i-1])
	diff = angles_[i]-angles_[i-1];
      else
	diff = angles_[i]+2*M_PI-angles_[i-1];

      if (diff < delta-FLT_EPSILON || diff > delta+FLT_EPSILON) {
	regular = 0;
	break;
      }
    }
  }

  if (numAnnuli_>2) {
    double delta = annuli_[1][0]-annuli_[0][0];
    for (int i=2; i<numAnnuli_; i++) {
      double diff = annuli_[i][0]-annuli_[i-1][0];
      if (diff < delta-FLT_EPSILON || diff > delta+FLT_EPSILON) {
	regular = 0;
	break;
      }
    }
  }

  if (regular)
    listA(str, sys, sky, format, conj, strip);
  else
    listB(str, sys, sky, format, conj, strip);
}

void Bpanda::listA(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      Vector r1 = ptr->mapLenFromRef(annuli_[0],sys);
      Vector r2 =  ptr->mapLenFromRef(annuli_[numAnnuli_-1],sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	  << radToDeg(parent->mapAngleFromReff(angles_[0],sys)) << ',' 
	  << radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],sys)) 
	  << ','
	  << numAngles_-1 << ','
	  << r1[0] << ',' << r1[1] << ','
	  << r2[0] << ',' << r2[1] << ','
	  << numAnnuli_-1 << ','
	  << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    Vector r1 = ptr->mapLenFromRef(annuli_[0],sys,ARCSEC);
	    Vector r2 =  ptr->mapLenFromRef(annuli_[numAnnuli_-1],sys,ARCSEC);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
		<< radToDeg(parent->mapAngleFromReff(angles_[0],sys,sky))
		<< ',' 
		<< radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],
						     sys,sky)) <<','
		<< numAngles_-1 << ','
		<< r1[0] << '"' << ',' << r1[1] << '"' << ','
		<< r2[0] << '"' << ',' << r2[1] << '"' << ','
		<< numAnnuli_-1 << ','
		<< radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;
	    Vector r1 = ptr->mapLenFromRef(annuli_[0],sys,ARCSEC);
	    Vector r2 =  ptr->mapLenFromRef(annuli_[numAnnuli_-1],sys,ARCSEC);
	    str << type << '(' << ra << ',' << dec << ','
		<< radToDeg(parent->mapAngleFromReff(angles_[0],sys,sky))
		<< ',' 
		<< radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],
						     sys,sky)) <<','
		<< numAngles_-1 << ','
		<< r1[0] << '"' << ',' << r1[1] << '"' << ','
		<< r2[0] << '"' << ',' << r2[1] << '"' << ','
		<< numAnnuli_-1 << ','
		<< radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	Vector r1 = ptr->mapLenFromRef(annuli_[0],sys);
	Vector r2 =  ptr->mapLenFromRef(annuli_[numAnnuli_-1],sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << radToDeg(parent->mapAngleFromReff(angles_[0],sys)) << ',' 
	    << radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],sys))
	    <<','
	    << numAngles_-1 << ','
	    << r1[0] << ',' << r1[1] << ','
	    << r2[0] << ',' << r2[1] << ','
	    << numAnnuli_-1 << ','
	    << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Bpanda::listB(ostream& str, CoordSystem sys, SkyFrame sky,
		  SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      for (int j=1; j<numAngles_; j++)
	for (int i=1; i<numAnnuli_; i++) {

	  listPre(str, sys, sky, ptr, strip, 0);

	  Vector r1 = ptr->mapLenFromRef(annuli_[i-1],sys);
	  Vector r2 =  ptr->mapLenFromRef(annuli_[i],sys);	
	  str << type << '(' << setprecision(8) 
	      << v[0] << ',' << v[1] << ','
	      << radToDeg(parent->mapAngleFromReff(angles_[j-1],sys)) << ',' 
	      << radToDeg(parent->mapAngleFromReff(angles_[j],sys)) << ',' 
	      << "1,"
	      << r1[0] << ',' << r1[1] << ','
	      << r2[0] << ',' << r2[1] << ','
	      << "1,"
	      << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';

	  if (!strip) {
	    if (conj || (!(j==numAngles_-1 && i==numAnnuli_-1)))
	      str << " ||";

	    str << " # bpanda=";
	    if (i==1 && j==1 && !strip) {
	      str << '(';
	      for (int k=0; k<numAngles_; k++)
		str << radToDeg(parent->mapAngleFromReff(angles_[k],sys))
		    << ((k<numAngles_-1) ? ' ' : ')');
	      str << '(';
	      for (int k=0; k<numAnnuli_; k++) {
		Vector r = ptr->mapLenFromRef(annuli_[k],sys);
		str << r[0] << ' ' << r[1] 
		    << ((k<numAnnuli_-1) ? ' ' : ')');
	      }
	      str << '(' 
		  << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	      
	      listProps(str);
	    }
	    else
	      str << "ignore";

	    str << (strip ? ';' : '\n');
	  }
	  else {
	    if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
	      str << "||";
	    else
	      str << ";";
	  }
	}
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    for (int j=1; j<numAngles_; j++)
	      for (int i=1; i<numAnnuli_; i++) {

		listPre(str, sys, sky, ptr, strip, 0);

		Vector r1 = ptr->mapLenFromRef(annuli_[i-1],sys,ARCSEC);
		Vector r2 =  ptr->mapLenFromRef(annuli_[i],sys,ARCSEC);	
		str << type << '(' << setprecision(8) 
		    << v[0] << ',' << v[1] << ','
		    << radToDeg(parent->mapAngleFromReff(angles_[j-1],sys,sky))
		    <<',' 
		    << radToDeg(parent->mapAngleFromReff(angles_[j],sys,sky))
		    << ",1,"
		    << r1[0] << '"' << ',' << r1[1] << '"' <<',' 
		    << r2[0] << '"' << ',' << r2[1] << '"' <<',' 
		    << "1," 
		    << radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
		
		if (!strip) {
		  if (conj || (!(j==numAngles_-1 && i==numAnnuli_-1)))
		    str << " ||";

		  str << " # bpanda=";
		  if (i==1 && j==1 && !strip) {
		    str << '(';
		    for (int k=0; k<numAngles_; k++)
		      str << 
			radToDeg(parent->mapAngleFromReff(angles_[k],sys,sky))
			  << ((k<numAngles_-1) ? ' ' : ')');
		    str << '(';
		    for (int k=0; k<numAnnuli_; k++) {
		      Vector r = ptr->mapLenFromRef(annuli_[k],sys,ARCSEC);
		      str << r[0] << '"' << ' ' << r[1] << '"' 
			  << ((k<numAnnuli_-1) ? ' ' : ')');
		    }
		    str << '(' 
			<< radToDeg(parent->mapAngleFromRef(angle,sys,sky)) 
			<< ')';
		    listProps(str);
		  }
		  else
		    str << "ignore";

		  str << (strip ? ';' : '\n');
		}
		else {
		  if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		    str << "||";
		  else
		    str << ";";
		}
	      }
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;

	    for (int j=1; j<numAngles_; j++)
	      for (int i=1; i<numAnnuli_; i++) {

		listPre(str, sys, sky, ptr, strip, 0);

		Vector r1 = ptr->mapLenFromRef(annuli_[i-1],sys,ARCSEC);
		Vector r2 =  ptr->mapLenFromRef(annuli_[i],sys,ARCSEC);	
		str << type << '(' << ra << ',' << dec << ','
		    << radToDeg(parent->mapAngleFromReff(angles_[j-1],sys,sky))
		    <<',' 
		    << radToDeg(parent->mapAngleFromReff(angles_[j],sys,sky))
		    << ",1,"
		    << r1[0] << '"' << ',' << r1[1] << '"' << ',' 
		    << r2[0] << '"' << ',' << r2[1] << '"' << ',' 
		    << "1,"
		    << radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';

		if (!strip) {
		  if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		    str << " ||";

		  str << " # bpanda=";
		  if (i==1 && j==1 && !strip) {
		    str << '(';
		    for (int k=0; k<numAngles_; k++)
		      str << 
			radToDeg(parent->mapAngleFromReff(angles_[k],sys,sky))
			  << ((k<numAngles_-1) ? ' ' : ')');
		    str << '(';
		    for (int k=0; k<numAnnuli_; k++) {
		      Vector r = ptr->mapLenFromRef(annuli_[k],sys,ARCSEC);
		      str << r[0] << '"' << ' ' << r[1] << '"' 
			  << ((k<numAnnuli_-1) ? ' ' : ')');
		    }
		    str << '(' 
			<< radToDeg(parent->mapAngleFromRef(angle,sys,sky)) 
			<< ')';
		    listProps(str);
		  }
		  else
		    str << "ignore";

		  str << (strip ? ';' : '\n');
		}
		else {
		  if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		    str << "||";
		  else
		    str << ";";
		}
	      }
	  }
	  break;
	}
	break;
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	for (int j=1; j<numAngles_; j++)
	  for (int i=1; i<numAnnuli_; i++) {

	    listPre(str, sys, sky, ptr, strip, 0);

	    Vector r1 = ptr->mapLenFromRef(annuli_[i-1],sys);
	    Vector r2 =  ptr->mapLenFromRef(annuli_[i],sys);	
	    str << type << '(' << setprecision(8) 
		<< v[0] << ',' << v[1] << ','
		<< radToDeg(parent->mapAngleFromReff(angles_[j-1],sys)) << ',' 
		<< radToDeg(parent->mapAngleFromReff(angles_[j],sys)) << ',' 
		<< "1,"
		<< r1[0] << ',' << r1[1] << ','
		<< r2[0] << ',' << r2[1] << ','
		<< "1,"
		<< radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';

	    if (!strip) {
	      if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		str << " ||";

	      str << " # bpanda=";
	      if (i==1 && j==1 && !strip) {
		str << '(';
		for (int k=0; k<numAngles_; k++)
		  str << radToDeg(parent->mapAngleFromReff(angles_[k],sys))
		      << ((k<numAngles_-1) ? ' ' : ')');
		str << '(';
		for (int k=0; k<numAnnuli_; k++) {
		  Vector r = ptr->mapLenFromRef(annuli_[k],sys);
		  str << r[0] << ' ' << r[1] 
		      << ((k<numAnnuli_-1) ? ' ' : ')');
		}
		str << '(' 
		    << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
		listProps(str);
	      }
	      else
		str << "ignore";

	      str << (strip ? ';' : '\n');
	    }
	    else {
	      if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		str << "||";
	      else
		str << ";";
	    }
	  }
      }
    }
  }
}

