// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "annulus.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Annulus::Annulus(const Annulus& a) : BaseEllipse(a) {}

Annulus::Annulus(FrameBase* p, const Vector& ctr, 
		 double r1, double r2, int rn,
		 const char* clr, int wth, const char* fnt, const char* txt, 
		 unsigned short prop, const char* cmt, 
		 const List<Tag>& tg, const List<CallBack>& cb)
  : BaseEllipse(p, ctr, 0, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn+1;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++) {
    double r = i*(r2-r1)/rn+r1;
    annuli_[i] = Vector(r,r);
  }

  strcpy(type, "annulus");
  numHandle = 4 + numAnnuli_;

  updateBBox();
}

Annulus::Annulus(FrameBase* p, const Vector& ctr, 
		 int rn, double* r,
		 const char* clr, int wth, const char* fnt, const char* txt, 
		 unsigned short prop, const char* cmt, 
		 const List<Tag>& tg, const List<CallBack>& cb)
  : BaseEllipse(p, ctr, 0, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = Vector(r[i],r[i]);
  sortAnnuli();

  strcpy(type, "annulus");
  numHandle = 4 + numAnnuli_;

  updateBBox();
}

void Annulus::edit(const Vector& v, int h)
{
  Matrix mm = bckCanvasMatrix();

  if (h<5) {
    // calc dist between edge of circle and handle
    double d = annuli_[numAnnuli_-1].length() - annuli_[numAnnuli_-1][0];

    for (int i=0; i<numAnnuli_; i++) {
      double r = ((v * mm).length() - d)/annuli_[numAnnuli_-1][0];
      annuli_[i] *= Vector(r,r);
    }
  }
  else {
    double d = (v * mm).length();
    annuli_[h-5] = Vector(d,d);
  }

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Annulus::editEnd()
{
  sortAnnuli();

  updateBBox();
  doCallBack(CallBack::EDITENDCB);
}

int Annulus::addAnnuli(const Vector& v)
{
  Matrix mm = bckCanvasMatrix();
  double l = (v * mm).length();

  return insertAnnuli(Vector(l,l));
}

// list

void Annulus::list(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';
      for (int i=0; i<numAnnuli_; i++) {
	double r = ptr->mapLenFromRef(annuli_[i][0],sys);
	str << r;
	if (i!=numAnnuli_-1)
	  str << ',';
      }
      str << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<',';
	    for (int i=0; i<numAnnuli_; i++) {
	      double r = ptr->mapLenFromRef(annuli_[i][0],sys,ARCSEC);
	      str << r << "\"";
	      if (i!=numAnnuli_-1)
		str << ',';
	    }
	    str << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;

	    str << type << '(' << ra << ',' << dec << ',' ;
	    for (int i=0; i<numAnnuli_; i++) {
	      double r = ptr->mapLenFromRef(annuli_[i][0],sys,ARCSEC);
	      str << r << "\"";
	      if (i!=numAnnuli_-1)
		str << ',';
	    }
	    str << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';
	for (int i=0; i<numAnnuli_; i++) {
	  double r = ptr->mapLenFromRef(annuli_[i][0],sys);
	  str << r;
	  if (i!=numAnnuli_-1)
	    str << ',';
	}
	str << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Annulus::listCiao(ostream& str, CoordSystem sys, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      for (int i=0; i<numAnnuli_-1; i++) {
	listCiaoPre(str);

	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << ptr->mapLenFromRef(annuli_[i][0],PHYSICAL) << ','
	    << ptr->mapLenFromRef(annuli_[i+1][0],PHYSICAL) << ')';

	listCiaoPost(str, conj, strip);
      }
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {

      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
      string x(buf);
      istringstream wcs(x);
      wcs >> ra >> dec;

      for (int i=0; i<numAnnuli_-1; i++) {
	listCiaoPre(str);

	str << type << '(' << ra << ',' << dec << ','
	    << ptr->mapLenFromRef(annuli_[i][0],sys,ARCMIN) << '\'' << ','
	    << ptr->mapLenFromRef(annuli_[i+1][0],sys,ARCMIN) << '\'' << ')';

	listCiaoPost(str, conj, strip);
      }
    }
  }
}

void Annulus::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      listProsCoordSystem(str,sys,sky);
      str << "; " << type << ' ' << setprecision(8) << v;
      for (int i=0; i<numAnnuli_; i++) {
	double r = ptr->mapLenFromRef(annuli_[i][0],IMAGE);
	str << r << ' ';
      }
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  listProsCoordSystem(str,sys,sky);
	  str << "; " << type << ' ' << setprecision(8) 
	      << v[0] << "d " << v[1] <<"d";
	  for (int i=0; i<numAnnuli_; i++) {
	    double r = ptr->mapLenFromRef(annuli_[i][0],sys,ARCSEC);
	    str << ' ' << r << '"';
	  }
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char decc[16];
	  char *dec = decc;
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    dec++;

	  listProsCoordSystem(str,sys,sky);
	  str << "; " << type << ' ' << ra << ' ' << dec;
	  for (int i=0; i<numAnnuli_; i++) {
	    double r = ptr->mapLenFromRef(annuli_[i][0],sys,ARCSEC);
	    str << ' ' << r << '"';
	  }
	}
	break;
      }
    }
  }

  listProsPost(str, strip);
}

void Annulus::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOimagePre(str);

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';

  for (int i=0; i<numAnnuli_; i++) {
    double r = ptr->mapLenFromRef(annuli_[i][0],IMAGE);
    str << r;
    if (i!=numAnnuli_-1)
      str << ',';
  }

  str << ')';
  listSAOimagePost(str, strip);
}

// special composite funtionallity

void Annulus::setComposite(const Matrix& mx, double aa)
{
  center *= mx;
  updateBBox();
}
