// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <limits.h>
#include <string.h>

#include "colorbartrue24.h"
#include "util.h"

// Tk Canvas Widget Function Declarations

int ColorbarTrueColor24CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int, 
				  Tcl_Obj *const []);

// Colorbar Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

struct ColorbarTrueColor24Options {
  Tk_Item item;              // required by tk
  int x, y;                  // Coordinates of positioning point on canvas
  int width;                 // widget width
  int height;                // widget height
  Tk_Anchor anchor;          // Where to anchor widget relative to x,y
  char* cmdName;             // Suggested Tcl command name

  Widget* widget;            // pointer to widget class

  int colors;
};

static Tk_ConfigSpec colorbarTrueColor24Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "colorbar",
   Tk_Offset(ColorbarTrueColor24Options, cmdName), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(ColorbarTrueColor24Options, x), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(ColorbarTrueColor24Options, y), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(ColorbarTrueColor24Options, width), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "22",
   Tk_Offset(ColorbarTrueColor24Options, height), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(ColorbarTrueColor24Options, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_INT, "-colors", NULL, NULL, "1024",
   Tk_Offset(ColorbarTrueColor24Options, colors), 0, NULL},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType colorbarTrueColor24Type = {
  "colorbartruecolor24",        // name
  sizeof(ColorbarTrueColor24Options), // size
  ColorbarTrueColor24CreateProc, // configProc
  colorbarTrueColor24Specs,     // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  (Tk_ItemCursorProc*)NULL,     // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int ColorbarTrueColor24_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&colorbarTrueColor24Type);
  return TCL_OK;
}

int ColorbarTrueColor24CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
				  Tk_Item* item, int argc, 
				  Tcl_Obj *const argv[])
{
  ColorbarTrueColor24* colorbar = new ColorbarTrueColor24(interp,canvas,item);

  // and set default configuration

  if (colorbar->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete colorbar;
    Tcl_AppendResult(interp, " error occured while creating colorbar.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// ColorbarTrueColor24

ColorbarTrueColor24::ColorbarTrueColor24(Tcl_Interp* i, Tk_Canvas c, 
					 Tk_Item* item) : 
  ColorbarTrueColor(i, c, item)
{
  configSpecs = colorbarTrueColor24Specs;  // colorbar configure options

  loadDefaultCMaps();
}

int ColorbarTrueColor24::initColormap()
{
  colorCount = (((ColorbarTrueColor24Options*)options)->colors);
  return initColormapForReal();
}

void ColorbarTrueColor24::updateColors()
{
  updateColorCells();

  // fill in xmap
  // make sure we have a pixmap

  if (!pixmap || !xmap)
    return;

  int& width = options->width;
  int& height = options->height;
  char* data = xmap->data;
    
  // Fill in colorbar data
  // Calculate first row
  // check for bits/pixel

  switch (xmap->bits_per_pixel) {
  case 32:
    updateColors32();
    break;
  case 24:
    updateColors24();
    break;
  default:
    cerr << "ColorbarTrueColor24 Internal Error: bad bits/pixel" << endl;
    exit(1);
  }

  // --and duplicate for remaining rows

  for (int j=1; j<height; j++)
    memcpy(data+(j*xmap->bytes_per_line), data, xmap->bytes_per_line);

  // update the pixmap from map image and clean up

  XPutImage(display, pixmap, gc, xmap, 0, 0, 0, 0, width, height);

  // border
  renderBorder();

  redraw();
}

void ColorbarTrueColor24::updateColors24()
{
  int& width = options->width;
  int& height = options->height;
  char* data = xmap->data;

#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif

  // if we have cross platforms, we need to byte swap

  if ((!xmap->byte_order && lsb()) || (xmap->byte_order && !lsb())) {
    for (int i=0; i<width; i++) {
      unsigned int r = colorCells[(int)(double(i)/width*colorCount)*3+2];
      unsigned int g = colorCells[(int)(double(i)/width*colorCount)*3+1];
      unsigned int b = colorCells[(int)(double(i)/width*colorCount)*3];
      unsigned int a = 0;
      a |= r << rs;
      a |= g << gs;
      a |= b << bs;

      memcpy(data+i*3, &a, 3);
    }
  }
  else {
    for (int i=0; i<width; i++) {
      unsigned int r = colorCells[(int)(double(i)/width*colorCount)*3+2];
      unsigned int g = colorCells[(int)(double(i)/width*colorCount)*3+1];
      unsigned int b = colorCells[(int)(double(i)/width*colorCount)*3];
      unsigned int a = 0;
      a |= r << rs;
      a |= g << gs;
      a |= b << bs;

      unsigned char* rr = (unsigned char*)(&a);
      *(data+i*3) = *(rr+3);
      *(data+i*3+1) = *(rr+2);
      *(data+i*3+2) = *(rr+1);
    }
  }
}

void ColorbarTrueColor24::updateColors32()
{
  int& width = options->width;
  int& height = options->height;
  char* data = xmap->data;

#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif

  // if we have cross platforms, we need to byte swap

  if ((!xmap->byte_order && lsb()) || (xmap->byte_order && !lsb())) {
    for (int i=0; i<width; i++) {
      unsigned int r = colorCells[(int)(double(i)/width*colorCount)*3+2];
      unsigned int g = colorCells[(int)(double(i)/width*colorCount)*3+1];
      unsigned int b = colorCells[(int)(double(i)/width*colorCount)*3];
      unsigned int a = 0;
      a |= r << rs;
      a |= g << gs;
      a |= b << bs;

      memcpy(data+i*4, &a, 4);
    }
  }
  else {
    for (int i=0; i<width; i++) {
      unsigned int r = colorCells[(int)(double(i)/width*colorCount)*3+2];
      unsigned int g = colorCells[(int)(double(i)/width*colorCount)*3+1];
      unsigned int b = colorCells[(int)(double(i)/width*colorCount)*3];
      unsigned int a = 0;
      a |= r << rs;
      a |= g << gs;
      a |= b << bs;

      unsigned char* rr = (unsigned char*)(&a);
      *(data+i*4) = *(rr+3);
      *(data+i*4+1) = *(rr+2);
      *(data+i*4+2) = *(rr+1);
      *(data+i*4+3) = *(rr);
    }
  }
}
