#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# Print procs

proc Print {} {
    global tcl_platform
    global message

    if {[PrintDialog ps]} {
	if {$tcl_platform(platform)!= "windows"} {
	    set w .status
	    DisplayStatusMsg $w "$message(info,print,wait)"
	    if [catch {PostScript} printError] {
		Error "$message(error,print,error) $printError"
	    }
	    DestroyStatusMsg $w
	} else {
	    if [catch {PostScript} printError] {
		Error "$message(error,print,error) $printError"
	    }
	}
    }
}

proc PostScript {} {
    global ds9
    global ps
    global message

    RealizeDS9

    set options " "

    # Orientation

    switch -- $ps(orient) {
	portrait {append options " -rotate false"}
	landscape {append options " -rotate true"}
    }

    # Page size

    if {$ps(scale) == "scaled"} {
	switch -- $ps(size) {
	    letter {PSPageSize 4.25i 5.5i 7.5i 10.i options}
	    legal {PSPageSize 4.25i 7.i 7.5i 13.i options}
	    tabloid {PSPageSize 5.5i 8.5i 10.i 16.i options}
	    poster {PSPageSize 18.i 24.i 35.i 47.i options}
	    a4 {PSPageSize 105m 148.5m 185m 272m options}
	    other {
		if {$ps(width) != "" && $ps(height) != ""} {
		    set x [expr double($ps(width))/2.]
		    set y [expr double($ps(height))/2.]
		    set w [expr $ps(width)-1.]
		    set h [expr $ps(height)-1.]
		    PSPageSize [append x i] [append y i] \
			[append w i] [append h i] options
		}
	    }
	    othermm {
		if {$ps(width) != "" && $ps(height) != ""} {
		    set x [expr double($ps(width))/2.]
		    set y [expr double($ps(height))/2.]
		    set w [expr $ps(width)-1.]
		    set h [expr $ps(height)-1.]
		    PSPageSize [append x m] [append y m] \
			[append w m] [append h m] options
		}
	    }
	}
    }

    # Printer vs File
    
    set channel ""
    if {$ps(dest) == "file" && $ps(filename) != {}} {
	append options " -file \{$ps(filename)\}"
    } else {
	set channel [open "| $ps(cmd)" w]
	append options " -channel $channel"
    }

    # set color specific postscript options

    switch -- $ds9(visual) {
	pseudocolor {
	    colorbar postscript level $ps(level)
	    colorbar postscript colorspace $ps(color)
	    colorbar postscript resolution $ps(resolution)
	}
	truecolor {
	    colorbar postscript level $ps(level)
	    colorbar postscript colorspace $ps(color)
	    colorbar postscript resolution $ps(resolution)

	    colorbarrgb postscript level $ps(level)
	    colorbarrgb postscript colorspace $ps(color)
	    colorbarrgb postscript resolution $ps(resolution)
	}
    }

    # set frame specific postscript options

    foreach f $ds9(frames) {
	$f postscript level $ps(level)
	$f postscript colorspace $ps(color)
	$f postscript resolution $ps(resolution)
    }

    # now invoke canvas postscript command

    if [catch {eval $ds9(canvas) postscript $options} psError] {
	Error "$message(error,print,ps) $psError"
    }

    if {$channel != ""} {
	close $channel
    }
}

proc PSPageSize {x y w h optname} {
    upvar $optname options

    global ds9
    global ps

    set width [winfo width $ds9(canvas)]
    set height [winfo height $ds9(canvas)]
    set wo [string trim $w "icmp"]
    set ho [string trim $h "icmp"]

    switch -- $ps(orient) {
	portrait {
	    if {[expr double($wo)/$width] < [expr double($ho)/$height]} {
		append options " -pagewidth $w"
	    } else {
		append options " -pageheight $h"
	    }
	}
	landscape {
	    if {[expr double($wo)/$width] > [expr double($ho)/$height]} {
		append options " -pageheight $w"
	    } else {
		append options " -pagewidth $h"
	    }
	}
    }

    append options " -pagex $x -pagey $y"
}

# Print Dialog procs

proc PrintDialog {which} {
    global ps

    PrintTemplate $which ps(dest) ps(cmd) ps(filename) \
	ps(color) ps(color2) ps(level) ps(resolution)

}

proc PrefsPrintDialog {which} {
    global prefs

    PrintTemplate $which prefs(ps,dest) prefs(ps,cmd) prefs(ps,filename) \
	prefs(ps,color) prefs(ps,color2) prefs(ps,level) prefs(ps,resolution)
}

proc PrintTemplate {which destvar cmdvar filenamevar \
			colorvar color2var levelvar resolutionvar} {
    global ed
    global menu

    upvar $destvar dest
    upvar $cmdvar cmd
    upvar $filenamevar filename
    upvar $colorvar color
    upvar $color2var color2
    upvar $levelvar level
    upvar $resolutionvar resolution

    set ed(ok) 0
    set ed(dest) $dest
    set ed(cmd) $cmd
    set ed(filename) $filename
    set ed(color) $color
    set ed(color2) $color2
    set ed(level) $level
    set ed(resolution) $resolution

    set w ".print"

    DialogCreate $w "Print" -borderwidth 2
    frame $w.print -relief groove -borderwidth 2
    frame $w.options -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2

    switch -- $which {
	ap -
	ps {pack $w.print $w.options $w.buttons -fill x -ipadx 4 -ipady 4}
	txt {pack $w.print $w.buttons -fill x -ipadx 4 -ipady 4}
    }

    # Print

    label $w.print.title1 -text "Print To:"
    radiobutton $w.print.printer -text "Printer" -variable ed(dest) \
	-value printer -selectcolor $menu(selectcolor)
    radiobutton $w.print.file -text "File" -variable ed(dest) \
	-value file -selectcolor $menu(selectcolor)

    label $w.print.title2 -text "Print Command:"
    entry $w.print.cmd -textvariable ed(cmd) -width 30

    label $w.print.title3 -text "File Name:"
    entry $w.print.name -textvariable ed(filename) -width 30

    button $w.print.browse -text "Browse" \
	-command "PrintDialogBrowse ed(filename)"

    grid rowconfigure $w.print 0 -pad 4
    grid rowconfigure $w.print 1 -pad 4
    grid rowconfigure $w.print 2 -pad 4

    global tcl_platform
    if {$tcl_platform(platform)!= "windows"} {
	grid $w.print.title1 -row 0 -column 0 -padx 4 -sticky w
	grid $w.print.printer -row 0 -column 1 -padx 4 -sticky w
	grid $w.print.file -row 0 -column 2 -padx 4 -sticky w

	grid $w.print.title2 -row 1 -column 0 -padx 4 -sticky w
	grid $w.print.cmd -row 1 -column 1 -columnspan 2 -padx 4 -sticky w
    }
    grid $w.print.title3 -row 2 -column 0 -padx 4 -sticky w
    grid $w.print.name -row 2 -column 1 -columnspan 2 -padx 4 -sticky w

    grid $w.print.browse -row 2 -column 3 -padx 4

    # Options

    switch -- $which {
	txt {}
	ps {
	    label $w.options.title2 -text "Print:"
	    radiobutton $w.options.rgb -text "RGB Color" \
		-variable ed(color) -value rgb \
		-selectcolor $menu(selectcolor)
	    radiobutton $w.options.cmyk -text "CMYK Color" \
		-variable ed(color) -value cmyk \
		-selectcolor $menu(selectcolor)
	    radiobutton $w.options.gray -text "Grayscale" \
		-variable ed(color) -value gray \
		-selectcolor $menu(selectcolor)

	    label $w.options.title4 -text "Postscript:"
	    radiobutton $w.options.level1 -text "Level 1" \
		-variable ed(level) -value 1 \
		-selectcolor $menu(selectcolor) -command "PSLevel $w"
	    radiobutton $w.options.level2 -text "Level 2" \
		-variable ed(level) -value 2 \
		-selectcolor $menu(selectcolor) -command "PSLevel $w"

	    label $w.options.title5 -text "Resolution:"

	    frame $w.options.resolution
	    label $w.options.resolution.title -text "dpi"
	    menubutton $w.options.resolution.button -relief raised \
		-menu $w.options.resolution.button.menu \
		-textvariable ed(resolution)
	    menu $w.options.resolution.button.menu \
		-tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
	    $w.options.resolution.button.menu add radiobutton -label "53" \
		-variable ed(resolution) -value 53
	    $w.options.resolution.button.menu add radiobutton -label "72" \
		-variable ed(resolution) -value 72
	    $w.options.resolution.button.menu add radiobutton -label "75" \
		-variable ed(resolution) -value 75
	    $w.options.resolution.button.menu add radiobutton -label "150" \
		-variable ed(resolution) -value 150
	    $w.options.resolution.button.menu add radiobutton -label "300" \
		-variable ed(resolution) -value 300
	    $w.options.resolution.button.menu add radiobutton -label "600" \
		-variable ed(resolution) -value 600
	    pack $w.options.resolution.button $w.options.resolution.title \
		-side left

	    grid rowconfigure $w.options 0 -pad 4
	    grid rowconfigure $w.options 1 -pad 4
	    grid rowconfigure $w.options 2 -pad 4

	    grid $w.options.title2 $w.options.rgb $w.options.cmyk \
		$w.options.gray -padx 4 -sticky w
	    grid $w.options.title4 $w.options.level2 $w.options.level1 \
		-padx 4 -sticky w
	    grid $w.options.title5 $w.options.resolution -padx 4 -sticky w

	    PSLevel $w
	}
	ap {
	    label $w.options.title2 -text "Print:"
	    radiobutton $w.options.color -text "Color" \
		-variable ed(color2) -value color \
		-selectcolor $menu(selectcolor)
	    radiobutton $w.options.gray -text "Grayscale" \
		-variable ed(color2) -value gray \
		-selectcolor $menu(selectcolor)
	    radiobutton $w.options.mono -text "Black & White" \
		-variable ed(color2) -value mono \
		-selectcolor $menu(selectcolor)

	    grid rowconfigure $w.options 0 -pad 4

	    grid $w.options.title2 -row 0 -column 0 -padx 4 -sticky w
	    grid $w.options.color -row 0 -column 1 -padx 4 -sticky w
	    grid $w.options.gray -row 0 -column 2 -padx 4 -sticky w
	    grid $w.options.mono -row 0 -column 3 -padx 4 -sticky w
	}
    }


    # Buttons

    button $w.buttons.ok -text "OK" -default active \
	-command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w
    DialogWait $w ed(ok) $w.buttons.ok
    DialogDismiss $w

    if {$ed(ok)} {
	set dest $ed(dest)
	set cmd $ed(cmd)
	set filename $ed(filename)
	set color $ed(color)
	set color2 $ed(color2)
	set level $ed(level)
	set resolution $ed(resolution)
    }

    set r $ed(ok)
    unset ed
    return $r
}

proc PSLevel {w} {
    global ed
    switch -- $ed(level) {
	1 {
	    $w.options.cmyk configure -state disabled
	    if {$ed(color)=="cmyk"} {
		set ed(color) rgb
	    }
	}
	2 {
	    $w.options.cmyk configure -state normal
	}
    }
}

proc PrintDialogBrowse {varname} {
    upvar $varname var

    FileLast pssavfbox $var
    set var [SaveFileDialog pssavfbox]
}

proc ProcessPrintCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ps
    global current

    switch -- [string tolower [lindex $var $i]] {
	destination {incr i; set ps(dest) [lindex $var $i]}
	command {incr i; set ps(cmd) [lindex $var $i]}
	filename {incr i; set ps(filename) [file normalize [lindex $var $i]] }
	palette {incr i; set ps(color) [lindex $var $i] }
	level {incr i; set ps(level) [lindex $var $i] }
	interpolate {incr i}
	resolution {incr i; set ps(resolution) [lindex $var $i] }

	{} {PostScript}
	default {incr i -1; PostScript}
    }
}

# Page Setup Dialog procs

proc PageSetupDialog {} {
    global ps

    PageSetupTemplate ps(orient) ps(scale) ps(size) ps(width) ps(height)
}

proc PrefsPageSetupDialog {} {
    global prefs

    PageSetupTemplate prefs(ps,orient) prefs(ps,scale) prefs(ps,size) \
	prefs(ps,width) prefs(ps,height)
}

proc PageSetupTemplate {orientvar scalevar sizevar widthvar heightvar} {
    global ed
    global menu

    upvar $orientvar orient
    upvar $scalevar scale
    upvar $sizevar size
    upvar $widthvar width
    upvar $heightvar height

    set ed(ok) 0
    set ed(orient) $orient
    set ed(scale) $scale
    set ed(size) $size
    set ed(width) $width
    set ed(height) $height

    set w ".page"

    DialogCreate $w "Page Setup" -borderwidth 2
    frame $w.options -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.options $w.buttons -fill x -ipadx 4 -ipady 4

    # Options

    label $w.options.title1 -text "Orientation:"
    radiobutton $w.options.portrait -text "Portrait" \
	-variable ed(orient) \
	-value portrait -selectcolor $menu(selectcolor)
    radiobutton $w.options.landscape -text "Landscape" \
	-variable ed(orient) \
	-value landscape -selectcolor $menu(selectcolor)

    label $w.options.title2 -text "Page Size:"
    radiobutton $w.options.fixed -text "Fixed in Size" \
	-variable ed(scale) \
	-value fixed -selectcolor $menu(selectcolor) \
	-command "PageSetupDialogSize $w"
    radiobutton $w.options.scaled -text "Scale to fit Page" \
	-variable ed(scale) \
	-value scaled -selectcolor $menu(selectcolor) \
	-command "PageSetupDialogSize $w"

    frame $w.options.s -relief groove -borderwidth 2
    radiobutton $w.options.s.letter -text "Letter(8.5 x 11 in)" \
	-variable ed(size) -value letter -selectcolor $menu(selectcolor)
    radiobutton $w.options.s.legal -text "Legal(8.5 x 14 in)" \
	-variable ed(size) \
	-value legal -selectcolor $menu(selectcolor)
    radiobutton $w.options.s.tabloid -text "Tabloid(11 x 17 in)" \
	-variable ed(size) -value tabloid -selectcolor $menu(selectcolor)
    radiobutton $w.options.s.poster -text "Poster(36 x 48 in)" \
	-variable ed(size) -value poster -selectcolor $menu(selectcolor)
    radiobutton $w.options.s.a4 -text "A4(210 x 297 mm)" \
	-variable ed(size) \
	-value a4 -selectcolor $menu(selectcolor)
    radiobutton $w.options.s.other -text "Other (inches)" \
	-variable ed(size) \
	-value other -selectcolor $menu(selectcolor)
    radiobutton $w.options.s.othermm -text "Other (mm)" \
	-variable ed(size) \
	-value othermm -selectcolor $menu(selectcolor)

    frame $w.options.s.wh
    label $w.options.s.wh.title3 -text "Width"
    entry $w.options.s.wh.width -textvariable ed(width) -width 10

    label $w.options.s.wh.title4 -text "Height"
    entry $w.options.s.wh.height -textvariable ed(height) -width 10

    grid rowconfigure $w.options 0 -pad 4
    grid rowconfigure $w.options 1 -pad 4
    grid rowconfigure $w.options 2 -pad 4

    grid $w.options.title1 -row 0 -column 0 -padx 4 -sticky w
    grid $w.options.portrait -row 0 -column 1 -padx 4 -sticky w
    grid $w.options.landscape -row 0 -column 2 -padx 4 -sticky w

    grid $w.options.title2 -row 1 -column 0 -padx 4 -sticky w
    grid $w.options.scaled -row 1 -column 1 -padx 4 -sticky w
    grid $w.options.fixed -row 1 -column 2 -padx 4 -sticky w

    grid $w.options.s -row 2 -column 1 -padx 4 -sticky w -columnspan 2

    grid rowconfigure $w.options.s 0 -pad 4
    grid rowconfigure $w.options.s 1 -pad 4
    grid rowconfigure $w.options.s 2 -pad 4
    grid rowconfigure $w.options.s 3 -pad 4
    grid rowconfigure $w.options.s 4 -pad 4

    grid $w.options.s.letter -row 0 -column 1 -padx 4 -sticky w
    grid $w.options.s.legal -row 1 -column 1 -padx 4 -sticky w
    grid $w.options.s.tabloid -row 2 -column 1 -padx 4 -sticky w
    grid $w.options.s.poster -row 3 -column 1 -padx 4 -sticky w
    grid $w.options.s.a4 -row 4 -column 1 -padx 4 -sticky w
    grid $w.options.s.other -row 5 -column 1 -padx 4 -sticky w
    grid $w.options.s.othermm -row 6 -column 1 -padx 4 -sticky w

    grid $w.options.s.wh -row 5 -column 2 -padx 4 -sticky w -rowspan 2

    grid rowconfigure $w.options.s.wh 0 -pad 4
    grid rowconfigure $w.options.s.wh 1 -pad 4

    grid $w.options.s.wh.title3 -row 0 -column 0 -padx 4 -sticky w
    grid $w.options.s.wh.width -row 0 -column 1 -padx 4 -sticky w
    grid $w.options.s.wh.title4 -row 1 -column 0 -padx 4 -sticky w
    grid $w.options.s.wh.height -row 1 -column 1 -padx 4 -sticky w

    # Buttons

    button $w.buttons.ok -text "OK" -default active \
	-command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    PageSetupDialogSize $w
    DialogCenter $w
    DialogWait $w ed(ok) $w.buttons.ok 
    DialogDismiss $w

    if {$ed(ok)} {
	set orient $ed(orient)
	set scale $ed(scale)
	set size $ed(size)
	set width $ed(width)
	set height $ed(height)
    }

    set r $ed(ok)
    unset ed
    return $r
}

proc PageSetupDialogSize {w} {
    global ed

    if {$ed(scale) == "scaled"} {
	$w.options.s.letter configure -state normal
	$w.options.s.legal configure -state normal
	$w.options.s.tabloid configure -state normal
	$w.options.s.poster configure -state normal
	$w.options.s.a4 configure -state normal
	$w.options.s.other configure -state normal
	$w.options.s.othermm configure -state normal
    } else {
	$w.options.s.letter configure -state disabled
	$w.options.s.legal configure -state disabled
	$w.options.s.tabloid configure -state disabled
	$w.options.s.poster configure -state disabled
	$w.options.s.a4 configure -state disabled
	$w.options.s.other configure -state disabled
	$w.options.s.othermm configure -state disabled
    }
}

proc ProcessPageSetupCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ps

    switch -- [string tolower [lindex $var $i]] {
	orientation {incr i; set ps(orient) [string tolower [lindex $var $i]]}
	pagescale {incr i; set ps(scale) [string tolower [lindex $var $i]]}
	pagesize {incr i; set ps(size) [string tolower [lindex $var $i]] }
    }
}
