#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc AnalysisPlot {tt wtt title xaxis yaxis dim data} {
    global ap

    # make the window name unique
    set i [lsearch $ap(xpa) $tt]
    if {$i>=0} {
	append tt $ap(unique,id)
	incr ap(unique,id)
    }

    # set the window title if none
    if {$wtt == {}} {
	set wtt $tt
    }

    # doit
    APDialog $tt $wtt $title $xaxis $yaxis
    APDataSet $tt $dim $data

    return $tt
}

proc APDialog {tt wtt title xaxis yaxis} {
    global ds9
    global ap
    global menu
    global prefs

    if {[APPing $tt]} {
	APRaise $tt
	return
    }

    set ap($tt,top) ".${tt}"
    set ap($tt,mb) ".${tt}mb"

    set ap($tt,data,total) 0
    set ap($tt,data,current) 0

    set ap($tt,xdata) {}
    set ap($tt,ydata) {}
    set ap($tt,xedata) {}
    set ap($tt,yedata) {}

    set ap($tt,graph,title) "$title"
    set ap($tt,graph,xaxis) "$xaxis"
    set ap($tt,graph,yaxis) "$yaxis"

    set ap($tt,graph,x,auto) 1
    set ap($tt,graph,x,min) {}
    set ap($tt,graph,x,max) {}
    set ap($tt,graph,y,auto) 1
    set ap($tt,graph,y,min) {}
    set ap($tt,graph,y,max) {}

    set ap($tt,grid) $prefs(ap,grid)
    set ap($tt,grid,log) $prefs(ap,grid,log) 

    set ap($tt,discrete) $prefs(ap,discrete) 
    set ap($tt,discrete,symbol) $prefs(ap,discrete,symbol) 
    set ap($tt,discrete,color) $prefs(ap,discrete,color) 

    set ap($tt,linear) $prefs(ap,linear) 
    set ap($tt,linear,width) $prefs(ap,linear,width) 
    set ap($tt,linear,color) $prefs(ap,linear,color) 
    set ap($tt,linear,dash) $prefs(ap,linear,dash) 

    set ap($tt,step) $prefs(ap,step) 
    set ap($tt,step,width) $prefs(ap,step,width) 
    set ap($tt,step,color) $prefs(ap,step,color) 
    set ap($tt,step,dash) $prefs(ap,step,dash) 

    set ap($tt,quadratic) $prefs(ap,quadratic) 
    set ap($tt,quadratic,width) $prefs(ap,quadratic,width) 
    set ap($tt,quadratic,color) $prefs(ap,quadratic,color) 
    set ap($tt,quadratic,dash) $prefs(ap,quadratic,dash) 

    set ap($tt,error) $prefs(ap,error) 
    set ap($tt,error,width) $prefs(ap,error,width) 
    set ap($tt,error,color) $prefs(ap,error,color) 
    set ap($tt,error,style) $prefs(ap,error,style) 

    set ap($tt,titleFont) $prefs(ap,titleFont) 
    set ap($tt,titleSize) $prefs(ap,titleSize) 
    set ap($tt,titleStyle) $prefs(ap,titleStyle) 

    set ap($tt,textlabFont) $prefs(ap,textlabFont) 
    set ap($tt,textlabSize) $prefs(ap,textlabSize) 
    set ap($tt,textlabStyle) $prefs(ap,textlabStyle) 

    set ap($tt,numlabFont) $prefs(ap,numlabFont) 
    set ap($tt,numlabSize) $prefs(ap,numlabSize) 
    set ap($tt,numlabStyle) $prefs(ap,numlabStyle)

    toplevel $ap($tt,top) -colormap $ds9(main)
    wm title $ap($tt,top) $wtt
    wm iconname $ap($tt,top) $wtt
    wm group $ap($tt,top) $ds9(top)
    wm protocol $ap($tt,top) WM_DELETE_WINDOW "APDestroy $tt"

    $ap($tt,top) configure -menu $ap($tt,mb)

    menu $ap($tt,mb) -selectcolor $menu(selectcolor)
    $ap($tt,mb) add cascade -label File -menu $ap($tt,mb).file
    $ap($tt,mb) add cascade -label Edit -menu $ap($tt,mb).edit
    $ap($tt,mb) add cascade -label Graph -menu $ap($tt,mb).graph
    $ap($tt,mb) add cascade -label Font -menu $ap($tt,mb).font
    $ap($tt,mb) add cascade -label DataSet -menu $ap($tt,mb).dataset
    $ap($tt,mb) add cascade -label View -menu $ap($tt,mb).view
    $ap($tt,mb) add cascade -label Color -menu $ap($tt,mb).color
    $ap($tt,mb) add cascade -label Line -menu $ap($tt,mb).line

    menu $ap($tt,mb).file -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).file add command -label "Load Data..." \
	-command "APLoadData $tt"
    $ap($tt,mb).file add command -label "Save Data..." \
	-command "APSaveData $tt"
    $ap($tt,mb).file add command -label "Clear Data" \
	-command "APClearData $tt"
    $ap($tt,mb).file add separator
    $ap($tt,mb).file add command -label "Load Configuration..." \
	-command "APLoadConfig $tt"
    $ap($tt,mb).file add command -label "Save Configuration..." \
	-command "APSaveConfig $tt"
    $ap($tt,mb).file add separator
    $ap($tt,mb).file add command -label "Print..." -command "APPrint $tt"
    $ap($tt,mb).file add command -label "Page Setup..." \
	-command PageSetupDialog
    $ap($tt,mb).file add separator
    $ap($tt,mb).file add command -label "Close" \
	-command "APDestroy $tt"

    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	$ap($tt,mb).file entryconfig "Print..." -state disabled
	$ap($tt,mb).file entryconfig "Page Setup..." -state disabled
    }

    menu $ap($tt,mb).edit -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).edit add command -label "Cut" -state disabled
    $ap($tt,mb).edit add command -label "Copy" -state disabled
    $ap($tt,mb).edit add command -label "Paste" -state disabled
    $ap($tt,mb).edit add command -label "Clear" -state disabled

    menu $ap($tt,mb).graph -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).graph add checkbutton -label "Grid" \
	-variable ap($tt,grid) -command "APUpdateGraph $tt"
    $ap($tt,mb).graph add separator
    $ap($tt,mb).graph add radiobutton -label "Linear-Linear" \
	-variable ap($tt,grid,log) -value linearlinear \
	-command "APUpdateGraph $tt"
    $ap($tt,mb).graph add radiobutton -label "Linear-Log" \
	-variable ap($tt,grid,log) -value linearlog \
	-command "APUpdateGraph $tt"
    $ap($tt,mb).graph add radiobutton -label "Log-Linear" \
	-variable ap($tt,grid,log) -value loglinear \
	-command "APUpdateGraph $tt"
    $ap($tt,mb).graph add radiobutton -label "Log-Log" \
	-variable ap($tt,grid,log) -value loglog \
	-command "APUpdateGraph $tt"
    $ap($tt,mb).graph add separator
    $ap($tt,mb).graph add command -label "Axis Range..." \
	-command "APRange $tt"
    $ap($tt,mb).graph add command -label "Labels..." -command "APTitle $tt"

    menu $ap($tt,mb).font -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).font add cascade -label "Axis Numbers" \
	-menu $ap($tt,mb).font.numlab
    $ap($tt,mb).font add cascade -label "Axis Label" \
	-menu $ap($tt,mb).font.textlab
    $ap($tt,mb).font add cascade -label "Title" \
	-menu $ap($tt,mb).font.title

    APCreateFontMenu $ap($tt,mb).font.numlab \
	ap($tt,numlabFont) ap($tt,numlabSize) ap($tt,numlabStyle) \
	"APUpdateGraph $tt"
    APCreateFontMenu $ap($tt,mb).font.textlab \
	ap($tt,textlabFont) ap($tt,textlabSize) ap($tt,textlabStyle) \
	"APUpdateGraph $tt"
    APCreateFontMenu $ap($tt,mb).font.title \
	ap($tt,titleFont) ap($tt,titleSize) ap($tt,titleStyle) \
	"APUpdateGraph $tt"

    menu $ap($tt,mb).dataset -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)

    menu $ap($tt,mb).view -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).view add checkbutton -label Discrete \
	-variable ap($tt,discrete) -command "APUpdateState $tt"
    $ap($tt,mb).view add checkbutton -label Line \
	-variable ap($tt,linear) -command "APUpdateState $tt"
    $ap($tt,mb).view add checkbutton -label Step \
	-variable ap($tt,step) -command "APUpdateState $tt"
    $ap($tt,mb).view add checkbutton -label Quadratic \
	-variable ap($tt,quadratic) -command "APUpdateState $tt"
    $ap($tt,mb).view add checkbutton -label Errorbar \
	-variable ap($tt,error) -command "APUpdateState $tt"

    menu $ap($tt,mb).color -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).color add cascade -label Discrete \
	-menu $ap($tt,mb).color.discrete
    $ap($tt,mb).color add cascade -label Line \
	-menu $ap($tt,mb).color.linear
    $ap($tt,mb).color add cascade -label Step \
	-menu $ap($tt,mb).color.step
    $ap($tt,mb).color add cascade -label Quadratic \
	-menu $ap($tt,mb).color.quadratic
    $ap($tt,mb).color add cascade -label Errorbar \
	-menu $ap($tt,mb).color.error

    APCreateColorMenu $ap($tt,mb).color.discrete \
	ap($tt,discrete,color) "APUpdateState $tt"
    APCreateColorMenu $ap($tt,mb).color.linear \
	ap($tt,linear,color) "APUpdateState $tt"
    APCreateColorMenu $ap($tt,mb).color.step \
	ap($tt,step,color) "APUpdateState $tt"
    APCreateColorMenu $ap($tt,mb).color.quadratic \
	ap($tt,quadratic,color) "APUpdateState $tt"
    APCreateColorMenu $ap($tt,mb).color.error \
	ap($tt,error,color) "APUpdateState $tt"

    menu $ap($tt,mb).line -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).line add cascade -label Discrete \
	-menu $ap($tt,mb).line.discrete
    $ap($tt,mb).line add cascade -label Line -menu $ap($tt,mb).line.linear
    $ap($tt,mb).line add cascade -label Step -menu $ap($tt,mb).line.step
    $ap($tt,mb).line add cascade -label Quadratic \
	-menu $ap($tt,mb).line.quadratic
    $ap($tt,mb).line add cascade -label Errorbar \
	-menu $ap($tt,mb).line.error

    menu $ap($tt,mb).line.discrete -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).line.discrete add radiobutton -label "Circle" \
	-variable ap($tt,discrete,symbol) -value circle \
	-command "APUpdateState $tt"
    $ap($tt,mb).line.discrete add radiobutton -label "Diamond" \
	-variable ap($tt,discrete,symbol) -value diamond \
	-command "APUpdateState $tt"
    $ap($tt,mb).line.discrete add radiobutton -label "Plus" \
	-variable ap($tt,discrete,symbol) -value plus \
	-command "APUpdateState $tt"
    $ap($tt,mb).line.discrete add radiobutton -label "Cross" \
	-variable ap($tt,discrete,symbol) -value cross \
	-command "APUpdateState $tt"

    APCreateLineMenu $ap($tt,mb).line.linear \
	ap($tt,linear,width) ap($tt,linear,dash) "APUpdateState $tt"
    APCreateLineMenu $ap($tt,mb).line.step \
	ap($tt,step,width) ap($tt,step,dash) "APUpdateState $tt"
    APCreateLineMenu $ap($tt,mb).line.quadratic \
	ap($tt,quadratic,width) ap($tt,quadratic,dash) "APUpdateState $tt"

    menu $ap($tt,mb).line.error -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $ap($tt,mb).line.error add radiobutton -label 1 \
	-variable ap($tt,error,width) -value 1 -command "APUpdateState $tt"
    $ap($tt,mb).line.error add radiobutton -label 2 \
	-variable ap($tt,error,width) -value 2 -command "APUpdateState $tt"
    $ap($tt,mb).line.error add radiobutton -label 3 \
	-variable ap($tt,error,width) -value 3 -command "APUpdateState $tt"
    $ap($tt,mb).line.error add radiobutton -label 4 \
	-variable ap($tt,error,width) -value 4 -command "APUpdateState $tt"
    $ap($tt,mb).line.error add separator
    $ap($tt,mb).line.error add radiobutton -label "Style 1" \
	-variable ap($tt,error,style) -value 0 -command "APUpdateState $tt"
    $ap($tt,mb).line.error add radiobutton -label "Style 2" \
	-variable ap($tt,error,style) -value 1 -command "APUpdateState $tt"

    set ap($tt,graph) [blt::graph $ap($tt,top).g -plotrelief groove \
			   -plotbackground [$ds9(main) cget -background]]
    $ap($tt,graph) legend configure -hide yes

    Blt_ZoomStack $ap($tt,graph)

    pack $ap($tt,graph) -expand yes -fill both

    APUpdateGraph $tt
}

proc APDestroy {tt} {
    global ap

    # see if it still is around
    if {![APPing $tt]} {
	return
    }

    for {set nn 1} {$nn<=$ap($tt,data,total)} {incr nn} {
	if {$ap($tt,$nn,manage)} {
	    blt::vector destroy \
		$ap($tt,$nn,xdata) $ap($tt,$nn,ydata) \
		$ap($tt,$nn,xedata) $ap($tt,$nn,yedata)
	}
    }

    destroy $ap($tt,top)
    destroy $ap($tt,mb)

    # unset all ap($tt,*)
    foreach x [array names ap] {
	set f [split $x ,]
	if {[lindex $f 0] == $tt} {
	    unset ap($x)
	}
    }

    # if an xpa plot, delete it from the list
    set i [lsearch $ap(xpa) $tt]
    if {$i>=0} {
	set ap(xpa) [lreplace $ap(xpa) $i $i]
    }
}

proc APCurrentData {tt} {
    global ap

    if {$ap($tt,data,total) > 0} {
	set nn $ap($tt,data,current)

	set ap($tt,xdata) $ap($tt,$nn,xdata)
	set ap($tt,ydata) $ap($tt,$nn,ydata)
	set ap($tt,xedata) $ap($tt,$nn,xedata)
	set ap($tt,yedata) $ap($tt,$nn,yedata)

	set ap($tt,discrete) $ap($tt,$nn,discrete) 
	set ap($tt,discrete,symbol) $ap($tt,$nn,discrete,symbol) 
	set ap($tt,discrete,color) $ap($tt,$nn,discrete,color) 

	set ap($tt,linear) $ap($tt,$nn,linear) 
	set ap($tt,linear,width) $ap($tt,$nn,linear,width) 
	set ap($tt,linear,color) $ap($tt,$nn,linear,color) 
	set ap($tt,linear,dash) $ap($tt,$nn,linear,dash) 

	set ap($tt,step) $ap($tt,$nn,step) 
	set ap($tt,step,width) $ap($tt,$nn,step,width) 
	set ap($tt,step,color) $ap($tt,$nn,step,color) 
	set ap($tt,step,dash) $ap($tt,$nn,step,dash) 

	set ap($tt,quadratic) $ap($tt,$nn,quadratic) 
	set ap($tt,quadratic,width) $ap($tt,$nn,quadratic,width) 
	set ap($tt,quadratic,color) $ap($tt,$nn,quadratic,color) 
	set ap($tt,quadratic,dash) $ap($tt,$nn,quadratic,dash) 

	set ap($tt,error) $ap($tt,$nn,error) 
	set ap($tt,error,width) $ap($tt,$nn,error,width) 
	set ap($tt,error,color) $ap($tt,$nn,error,color) 
	set ap($tt,error,style) $ap($tt,$nn,error,style) 
    }
}

proc APDataFormatDialog {} {
    global ed

    set w ".apdata"

    set ed(ok) 0
    set ed(dim) xy

    DialogCreate $w "Data Format" -borderwidth 2
    frame $w.dim  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.dim $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.dim.title -text "Data Format:"
    radiobutton $w.dim.xy -text "X Y" -variable ed(dim) -value xy
    radiobutton $w.dim.xyex -text "X Y XErr" -variable ed(dim) -value xyex
    radiobutton $w.dim.xyey -text "X Y YErr" -variable ed(dim) -value xyey
    radiobutton $w.dim.xyexey -text "X Y XErr YErr" -variable ed(dim) \
	-value xyexey

    grid $w.dim.title -padx 4 -sticky w
    grid $w.dim.xy -padx 4 -sticky w
    grid $w.dim.xyex -padx 4 -sticky w
    grid $w.dim.xyey -padx 4 -sticky w
    grid $w.dim.xyexey -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok) $w.dim.xy
    DialogDismiss $w
}

proc APLoadData {tt} {
    global ds9
    global ap
    global ed

    set filename [OpenFileDialog apdatafbox]
    if {$filename != {}} {
	APDataFormatDialog

	if {$ed(ok)} {
	    APPLoadData $tt $filename $ed(dim)
	}
	unset ed
    }
}

proc APPLoadData {tt filename dim} {
    set ch [open $filename]
    set data [read $ch]
    close $ch

    APRaise $tt
    APDataSet $tt $dim $data
}

proc APSaveData {tt} {
    APPSaveData $tt [SaveFileDialog apdatafbox]
}

proc APPSaveData {tt filename} {
    global ap
    global ds9

    if {$filename != {}} {
	set ch [open $filename w]

	global $ap($tt,xdata) $ap($tt,ydata) $ap($tt,xedata) $ap($tt,yedata)
	set l [$ap($tt,xdata) length]

	if {([$ap($tt,xedata) length] == 0) && 
	    ([$ap($tt,yedata) length] == 0)} {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$ap($tt,xdata)\($i\)"]
		set y [expr "$$ap($tt,ydata)\($i\)"]
		puts $ch "$x $y"
	    }
	} elseif {([$ap($tt,xedata) length] == 0) &&
		  ([$ap($tt,yedata) length] != 0)} {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$ap($tt,xdata)\($i\)"]
		set y [expr "$$ap($tt,ydata)\($i\)"]
		set ye [expr "$$ap($tt,yedata)\($i\)"]
		puts $ch "$x $y $ye"
	    }
	} elseif {([$ap($tt,xedata) length] != 0) &&
		  ([$ap($tt,yedata) length] == 0)} {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$ap($tt,xdata)\($i\)"]
		set y [expr "$$ap($tt,ydata)\($i\)"]
		set xe [expr "$$ap($tt,xedata)\($i\)"]
		puts $ch "$x $y $xe"
	    }
	} else {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$ap($tt,xdata)\($i\)"]
		set y [expr "$$ap($tt,ydata)\($i\)"]
		set xe [expr "$$ap($tt,xedata)\($i\)"]
		set ye [expr "$$ap($tt,yedata)\($i\)"]
		puts $ch "$x $y $xe $ye"
	    }
	}

	close $ch
    }
    APRaise $tt
}

proc APClearData {tt} {
    global ap

    # this is hard coded to ignore an external first dataset

    if {$ap($tt,1,manage)} {
	set clear 1
	set nn 1
    } else {
	set clear 0
	set nn 2
    }

    for {} {$nn<=$ap($tt,data,total)} {incr nn} {
	# delete elements
	foreach el [$ap($tt,graph) element names] {
	    set f [split $el -]
	    if {[lindex $f 1] == $nn} {
		$ap($tt,graph) element delete $el
	    }
	}

	blt::vector destroy $ap($tt,$nn,xdata) $ap($tt,$nn,ydata) \
	    $ap($tt,$nn,xedata) $ap($tt,$nn,yedata)

	# unset all ap($tt,$nn,*)
	foreach x [array names ap] {
	    set f [split $x ,]
	    if {([lindex $f 0] == $tt) && ([lindex $f 1] == $nn)} {
		unset ap($x)
	    }
	}
    }

    if {$clear} {
	set ap($tt,data,total) 0
	set ap($tt,data,current) 0

	# reset other variables
	set ap($tt,xdata) {}
	set ap($tt,ydata) {}
	set ap($tt,xedata) {}
	set ap($tt,yedata) {}

	set ap($tt,graph,title) ""
	set ap($tt,graph,xaxis) ""
	set ap($tt,graph,yaxis) ""

	set ap($tt,graph,x,auto) 1
	set ap($tt,graph,x,min) {}
	set ap($tt,graph,x,max) {}
	set ap($tt,graph,y,auto) 1
	set ap($tt,graph,y,min) {}
	set ap($tt,graph,y,max) {}
	
	$ap($tt,mb).dataset delete 1 end
    } else {
	if {$ap($tt,data,total) > 1} {
	    $ap($tt,mb).dataset delete 2 end
	}

	set ap($tt,data,total) 1
	set ap($tt,data,current) 1
    }

    APUpdateGraph $tt
}

proc APLoadConfig {tt} {
    APPLoadConfig $tt [OpenFileDialog apconfigfbox]
}

proc APPLoadConfig {tt filename} {
    global ds9
    global ap

    if {$filename != {}} {
	source $filename

        set ap($tt,discrete) $analysisplot(discrete)
        set ap($tt,discrete,symbol) $analysisplot(discrete,symbol)
        set ap($tt,discrete,color) $analysisplot(discrete,color)

        set ap($tt,linear) $analysisplot(linear)
        set ap($tt,linear,width) $analysisplot(linear,width)
        set ap($tt,linear,color) $analysisplot(linear,color)
        set ap($tt,linear,dash) $analysisplot(linear,dash)

        set ap($tt,step) $analysisplot(step)
        set ap($tt,step,width) $analysisplot(step,width)
        set ap($tt,step,color) $analysisplot(step,color)
        set ap($tt,step,dash) $analysisplot(step,dash)

        set ap($tt,quadratic) $analysisplot(quadratic)
        set ap($tt,quadratic,width) $analysisplot(quadratic,width)
        set ap($tt,quadratic,color) $analysisplot(quadratic,color)
        set ap($tt,quadratic,dash) $analysisplot(quadratic,dash)

        set ap($tt,error) $analysisplot(error)
        set ap($tt,error,width) $analysisplot(error,width)
        set ap($tt,error,color) $analysisplot(error,color)
        set ap($tt,error,style) $analysisplot(error,style)

	APUpdateState $tt
    }
}

proc APSaveConfig {tt} {
    APPSaveConfig $tt [SaveFileDialog apconfigfbox]
}

proc APPSaveConfig {tt filename} {
    global ds9
    global ap

    if {$filename != {}} {
	set file [open $filename w]

        puts $file "set analysisplot(discrete) $ap($tt,discrete)"
        puts $file "set analysisplot(discrete,symbol) $ap($tt,discrete,symbol)"
        puts $file "set analysisplot(discrete,color) $ap($tt,discrete,color)"

        puts $file "set analysisplot(linear) $ap($tt,linear)"
        puts $file "set analysisplot(linear,width) $ap($tt,linear,width)"
        puts $file "set analysisplot(linear,color) $ap($tt,linear,color)"
        puts $file "set analysisplot(linear,dash) \"$ap($tt,linear,dash)\""

        puts $file "set analysisplot(step) $ap($tt,step)"
        puts $file "set analysisplot(step,width) $ap($tt,step,width)"
        puts $file "set analysisplot(step,color) $ap($tt,step,color)"
        puts $file "set analysisplot(step,dash) \"$ap($tt,step,dash)\""

        puts $file "set analysisplot(quadratic) $ap($tt,quadratic)"
        puts $file "set analysisplot(quadratic,width) $ap($tt,quadratic,width)"
        puts $file "set analysisplot(quadratic,color) $ap($tt,quadratic,color)"
        puts $file \
	    "set analysisplot(quadratic,dash) \"$ap($tt,quadratic,dash)\""

        puts $file "set analysisplot(error) $ap($tt,error)"
        puts $file "set analysisplot(error,width) $ap($tt,error,width)"
        puts $file "set analysisplot(error,color) $ap($tt,error,color)"
        puts $file "set analysisplot(error,style) \"$ap($tt,error,style)\""

	close $file
    }
}

proc APRange {tt} {
    global ap
    global ed

    set w ".aptitle"

    set ed(ok) 0
    set ed(xauto) $ap($tt,graph,x,auto)
    set ed(xmin) $ap($tt,graph,x,min)
    set ed(xmax) $ap($tt,graph,x,max)
    set ed(yauto) $ap($tt,graph,y,auto)
    set ed(ymin) $ap($tt,graph,y,min)
    set ed(ymax) $ap($tt,graph,y,max)

    DialogCreate $w Range -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.param.xlabel -text "X Axis"
    entry $w.param.xmin -textvariable ed(xmin) -width 12
    label $w.param.xlabel2 -text "to"
    entry $w.param.xmax -textvariable ed(xmax) -width 12
    checkbutton $w.param.xauto -text "Auto" -variable ed(xauto)

    label $w.param.ylabel -text "Y Axis"
    entry $w.param.ymin -textvariable ed(ymin) -width 12
    label $w.param.ylabel2 -text "to"
    entry $w.param.ymax -textvariable ed(ymax) -width 12
    checkbutton $w.param.yauto -text "Auto" -variable ed(yauto)

    grid $w.param.xlabel $w.param.xmin $w.param.xlabel2 $w.param.xmax \
	$w.param.xauto  -padx 4 -pady 1 -sticky w
    grid $w.param.ylabel $w.param.ymin $w.param.ylabel2 $w.param.ymax \
	$w.param.yauto  -padx 4 -pady 1 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok) $w.param.xmin
    DialogDismiss $w

    if {$ed(ok)} {
	set ap($tt,graph,x,auto) $ed(xauto)
	set ap($tt,graph,x,min) $ed(xmin) 
	set ap($tt,graph,x,max) $ed(xmax) 
	set ap($tt,graph,y,auto) $ed(yauto) 
	set ap($tt,graph,y,min) $ed(ymin) 
	set ap($tt,graph,y,max) $ed(ymax) 

	APUpdateGraph $tt
    }
    
    unset ed
}

proc APTitle {tt} {
    global ap
    global ed

    set w ".aptitle"

    set ed(ok) 0
    set ed(title) $ap($tt,graph,title)
    set ed(xaxis) $ap($tt,graph,xaxis)
    set ed(yaxis) $ap($tt,graph,yaxis)

    DialogCreate $w Labels -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.param.label -text "Plot Title"
    entry $w.param.title -textvariable ed(title) -width 23
    label $w.param.xlabel -text "X Axis Label"
    entry $w.param.xtitle -textvariable ed(xaxis) -width 23
    label $w.param.ylabel -text "Y Axis Label"
    entry $w.param.ytitle -textvariable ed(yaxis) -width 23

    grid $w.param.label $w.param.title -padx 4 -pady 1 -sticky w
    grid $w.param.xlabel $w.param.xtitle -padx 4 -pady 1 -sticky w
    grid $w.param.ylabel $w.param.ytitle -padx 4 -pady 1 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok) $w.param.title
    DialogDismiss $w

    if {$ed(ok)} {
	set ap($tt,graph,title) $ed(title)
	set ap($tt,graph,xaxis) $ed(xaxis)
	set ap($tt,graph,yaxis) $ed(yaxis)
	APUpdateGraph $tt
    }
    
    unset ed
}

proc APCreateLineMenu {which var1 var2 cmd} {
    global ap
    global menu

    menu $which -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $which add radiobutton -label 1 -variable $var1 -value 1 -command $cmd
    $which add radiobutton -label 2 -variable $var1 -value 2 -command $cmd
    $which add radiobutton -label 3 -variable $var1 -value 3 -command $cmd
    $which add radiobutton -label 4 -variable $var1 -value 4 -command $cmd
    $which add separator
    $which add radiobutton -label Solid -variable $var2 -value " " \
	-command $cmd
    $which add radiobutton -label Dash  -variable $var2 -value "8 3" \
	-command $cmd
}

proc APCreateColorMenu {which var cmd} {
    global ap
    global menu

    menu $which -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $which add radiobutton -label "Black" -variable $var -value black \
	-command $cmd
    $which add radiobutton -label "White" -variable $var -value white \
	-command $cmd
    $which add radiobutton -label "Red" -variable $var -value red \
	-command $cmd
    $which add radiobutton -label "Green" -variable $var -value green \
	-command $cmd
    $which add radiobutton -label "Blue" -variable $var -value blue \
	-command $cmd
    $which add radiobutton -label "Cyan" -variable $var -value cyan \
	-command $cmd
    $which add radiobutton -label "Magenta" -variable $var -value magenta \
	-command $cmd
    $which add radiobutton -label "Yellow" -variable $var -value yellow \
	-command $cmd
}

proc APCreateFontMenu {which var1 var2 var3 cmd} {
    global ap
    global menu

    menu $which -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $which add radiobutton -label Times -variable $var1 -value times \
	-command $cmd
    $which add radiobutton -label Helvetica -variable $var1 -value helvetica \
	-command $cmd
    $which add radiobutton -label Symbol -variable $var1 -value symbol \
	-command $cmd
    $which add radiobutton -label Courier -variable $var1 -value courier \
	-command $cmd
    $which add separator
    $which add radiobutton -label 9 -variable $var2 -value 9 -command $cmd
    $which add radiobutton -label 10 -variable $var2 -value 10 -command $cmd
    $which add radiobutton -label 12 -variable $var2 -value 12 -command $cmd
    $which add radiobutton -label 14 -variable $var2 -value 14 -command $cmd
    $which add radiobutton -label 18 -variable $var2 -value 18 -command $cmd
    $which add radiobutton -label 24 -variable $var2 -value 24 -command $cmd
    $which add radiobutton -label 30 -variable $var2 -value 30 -command $cmd
    $which add radiobutton -label 36 -variable $var2 -value 36 -command $cmd
    $which add separator
    $which add radiobutton -label Plain -variable $var3 -value normal \
	-command $cmd
    $which add radiobutton -label Bold -variable $var3 -value bold \
	-command $cmd
    $which add radiobutton -label Italic -variable $var3 -value italic \
	-command $cmd
}

proc APPrint {tt} {
    global tcl_platform
    global message

    if {[PrintDialog ap]} {
	if {$tcl_platform(platform)!= "windows"} {
	    set w .status
	    DisplayStatusMsg $w "$message(info,print,wait)"
	    if [catch {APPostScript $tt} printError] {
		Error "$message(error,plot,print) $printError"
	    }
	    DestroyStatusMsg $w
	} else {
	    if [catch {APPostScript $tt} printError] {
		Error "$message(error,plot,print) $printError"
	    }
	}
    }
}

proc APPostScript {tt} {
    global ps
    global ap

    global debug
    if {$debug(tcl,idletasks)} {
	puts "APPostScript"
    }
    update idletasks

    set options "-decorations false -colormode $ps(color2)"

    # Orientation

    switch -- $ps(orient) {
	portrait {set options "$options -landscape false"}
	landscape {set options "$options -landscape true"}
    }

    # Page size

    switch -- $ps(size) {
	letter {APPageSize 8.5i 11.i options $tt}
	legal {APPageSize 8.5i 14.i options $tt}
	tabloid {APPageSize 11.i 17.i options $tt}
	poster {APPageSize 36.i 48.i options $tt}
	a4 {APPageSize 195m 282m options $tt}
	other {
	    if {$ps(width) != "" && $ps(height) != ""} {
		set w [expr $ps(width)]
		set h [expr $ps(height)]
		APPageSize [append w i] [append h i] options $tt
	    }
	}
	othermm {
	    if {$ps(width) != "" && $ps(height) != ""} {
		set w [expr $ps(width)]
		set h [expr $ps(height)]
		APPageSize [append w m] [append h m] options $tt
	    }
	}
    }

    if {$ps(dest) == "file" && $ps(filename) != ""} {
	eval $ap($tt,graph) postscript output $ps(filename) $options
    } else {
	set ch [open "| $ps(cmd)" w]
	puts $ch [eval $ap($tt,graph) postscript output $options]
	close $ch
    }
}

proc APPageSize {w h optname tt} {
    upvar $optname options
    global ps
    global ap

    set options "$options -paperwidth $w -paperheight $h"

    set width [winfo width $ap($tt,graph)]
    set height [winfo height $ap($tt,graph)]
    set aspect [expr double($height)/$width]

    if {$ps(scale) == "scaled"} {
	regexp {([0-9.]*)([icmp])?} $w foo a b
	set options "$options -width $w -height [expr $a*$aspect]${b}"
    } else {
	set options "$options -width 0 -height 0"
    }
}

proc APPrefsMenu {} {
    global ds9
    global menu
    global prefs

    set mb $ds9(mb).prefs.analysis.plot

    menu $mb -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $mb add cascade -label "Graph" -menu $mb.graph
    $mb add cascade -label "Font" -menu $mb.font
    $mb add cascade -label "View" -menu $mb.view
    $mb add cascade -label "Line" -menu $mb.line
    $mb add cascade -label "Color" -menu $mb.color

    # Analysis Plot Graph Prefs

    menu $mb.graph -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $mb.graph add checkbutton -label "Grid" -variable prefs(ap,grid)
    $mb.graph add separator
    $mb.graph add radiobutton -label "Linear-Linear" \
	-variable prefs(ap,grid,log) -value linearlinear
    $mb.graph add radiobutton -label "Linear-Log" \
	-variable prefs(ap,grid,log) -value linearlog
    $mb.graph add radiobutton -label "Log-Linear" \
	-variable prefs(ap,grid,log) -value loglinear
    $mb.graph add radiobutton -label "Log-Log" \
	-variable prefs(ap,grid,log) -value loglog

    # Analysis Plot Font Prefs

    menu $mb.font -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $mb.font add cascade -label "Axis Numbers" -menu $mb.font.numlab
    $mb.font add cascade -label "Axis Label" -menu $mb.font.textlab
    $mb.font add cascade -label "Title" -menu $mb.font.title

    APCreateFontPrefsMenu $mb.font.numlab \
	prefs(ap,numlabFont) prefs(ap,numlabSize) prefs(ap,numlabStyle)
    APCreateFontPrefsMenu $mb.font.textlab \
	prefs(ap,textlabFont) prefs(ap,textlabSize) prefs(ap,textlabStyle)
    APCreateFontPrefsMenu $mb.font.title \
	prefs(ap,titleFont) prefs(ap,titleSize) prefs(ap,titleStyle)

    # Analysis Plot View Prefs

    menu $mb.view -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $mb.view add checkbutton -label Discrete -variable prefs(ap,discrete) 
    $mb.view add checkbutton -label Line -variable prefs(ap,linear) 
    $mb.view add checkbutton -label Step -variable prefs(ap,step) 
    $mb.view add checkbutton -label Quadratic -variable prefs(ap,quadratic)
    $mb.view add checkbutton -label Error -variable prefs(ap,error)

    # Analysis Plot Color Prefs

    menu $mb.color -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $mb.color add cascade -label Discrete -menu $mb.color.discrete
    $mb.color add cascade -label Line -menu $mb.color.linear
    $mb.color add cascade -label Step -menu $mb.color.step
    $mb.color add cascade -label Quadratic -menu $mb.color.quadratic
    $mb.color add cascade -label Errorbar -menu $mb.color.error

    APCreateColorPrefsMenu $mb.color.discrete prefs(ap,discrete,color)
    APCreateColorPrefsMenu $mb.color.linear prefs(ap,linear,color)
    APCreateColorPrefsMenu $mb.color.step prefs(ap,step,color)
    APCreateColorPrefsMenu $mb.color.quadratic prefs(ap,quadratic,color)
    APCreateColorPrefsMenu $mb.color.error prefs(ap,error,color)

    # Analysis Plot Line Prefs

    menu $mb.line -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $mb.line add cascade -label Discrete -menu $mb.line.discrete
    $mb.line add cascade -label Line -menu $mb.line.linear
    $mb.line add cascade -label Step -menu $mb.line.step
    $mb.line add cascade -label Quadratic -menu $mb.line.quadratic
    $mb.line add cascade -label Errorbar -menu $mb.line.error

    menu $mb.line.discrete -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    $mb.line.discrete add radiobutton -label "Circle" \
	-variable prefs(ap,discrete,symbol) -value circle 	
    $mb.line.discrete add radiobutton -label "Diamond" \
	-variable prefs(ap,discrete,symbol) -value diamond 	
    $mb.line.discrete add radiobutton -label "Plus" \
	-variable prefs(ap,discrete,symbol) -value plus 	
    $mb.line.discrete add radiobutton -label "Cross" \
	-variable prefs(ap,discrete,symbol) -value cross 	

    APCreateLinePrefsMenu $mb.line.linear prefs(ap,linear,width) \
	prefs(ap,linear,dash)
    APCreateLinePrefsMenu $mb.line.step prefs(ap,step,width) \
	prefs(ap,step,dash)
    APCreateLinePrefsMenu $mb.line.quadratic prefs(ap,quadratic,width) \
	prefs(ap,quadratic,dash)

    menu $mb.line.error -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $mb.line.error add radiobutton -label 0 \
	-variable prefs(ap,error,width) -value 0 
    $mb.line.error add radiobutton -label 1 \
	-variable prefs(ap,error,width) -value 1 
    $mb.line.error add radiobutton -label 2 \
	-variable prefs(ap,error,width) -value 2 
    $mb.line.error add radiobutton -label 3 \
	-variable prefs(ap,error,width) -value 3 
    $mb.line.error add separator
    $mb.line.error add radiobutton -label "Style 1" \
	-variable prefs(ap,error,style) -value 0 
    $mb.line.error add radiobutton -label "Style 2" \
	-variable prefs(ap,error,style) -value 1 
}

proc APCreateLinePrefsMenu {which var1 var2} {
    global menu
    global prefs

    menu $which -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $which add radiobutton -label 0 -variable $var1 -value 0
    $which add radiobutton -label 1 -variable $var1 -value 1
    $which add radiobutton -label 2 -variable $var1 -value 2
    $which add radiobutton -label 3 -variable $var1 -value 3
    $which add separator
    $which add radiobutton -label Solid -variable $var2 -value " "
    $which add radiobutton -label Dash  -variable $var2 -value "8 3"
}

proc APCreateColorPrefsMenu {which var} {
    global menu
    global prefs

    menu $which -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $which add radiobutton -label "Black" -variable $var -value black
    $which add radiobutton -label "White" -variable $var -value white
    $which add radiobutton -label "Red" -variable $var -value red
    $which add radiobutton -label "Green" -variable $var -value green
    $which add radiobutton -label "Blue" -variable $var -value blue
    $which add radiobutton -label "Cyan" -variable $var -value cyan
    $which add radiobutton -label "Magenta" -variable $var -value magenta
    $which add radiobutton -label "Yellow" -variable $var -value yellow
}

proc APCreateFontPrefsMenu {which var1 var2 var3} {
    global menu
    global prefs

    menu $which -tearoff $menu(tearoff) -selectcolor $menu(selectcolor)
    $which add radiobutton -label Times -variable $var1 -value times
    $which add radiobutton -label Helvetica -variable $var1 -value helvetica
    $which add radiobutton -label Symbol -variable $var1 -value symbol
    $which add radiobutton -label Courier -variable $var1 -value courier
    $which add separator
    $which add radiobutton -label 9 -variable $var2 -value 9
    $which add radiobutton -label 10 -variable $var2 -value 10
    $which add radiobutton -label 12 -variable $var2 -value 12
    $which add radiobutton -label 14 -variable $var2 -value 14
    $which add radiobutton -label 18 -variable $var2 -value 18
    $which add radiobutton -label 24 -variable $var2 -value 24
    $which add radiobutton -label 30 -variable $var2 -value 30
    $which add radiobutton -label 36 -variable $var2 -value 36
    $which add separator
    $which add radiobutton -label Plain -variable $var3 -value normal
    $which add radiobutton -label Bold -variable $var3 -value bold
    $which add radiobutton -label Italic -variable $var3 -value italic
}


proc ProcessPlotCmd {varname iname bufname} {
    upvar $varname var
    upvar $iname i
    upvar $bufname buf

    global ap
    global message

    set tt $ap(tt)
    set itt 0

    # determine which plot
    switch -- [string tolower [lindex $var $i]] {
	{} -
	new {}

	data -
	load -
	save -
	clear -
	loadconfig -
	saveconfig -
	print -
	page -
	close -

	graph -
	font -
	dataset -
	view -
	color -
	line {
	    set tt [lindex $ap(xpa) end]
	    set itt [lsearch $ap(xpa) $tt]
	}

	default {
	    set tt [lindex $var $i]
	    set itt [lsearch $ap(xpa) $tt]
	    incr i
	}
    }

    # we better have a tt by now
    if {$itt == -1} {
	Error "$message(error,plot,window) $tt"
	return
    }

    # now, process plot command
    switch -- [string tolower [lindex $var $i]] {
	{} {APProcessNew $tt $varname $iname $bufname}
	new {
	    incr i
	    switch -- [lindex $var $i] {
		name {
		    set tt [lindex $var [expr $i+1]]
		    incr i 2
		}
	    }
	    APProcessNew $tt $varname $iname $bufname
	}
	data {incr i; APProcessData $tt $varname $iname $bufname}
	load {
	    set fn [file normalize [lindex $var [expr $i+1]]]
	    set dim [lindex $var [expr $i+2]]
	    APPLoadData $tt $fn $dim
	    FileLastFull apdatafbox $fn
	    incr i 2
	}
	save {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    APPSaveData $tt $fn
	    FileLastFull apdatafbox $fn
	}
	clear {APClearData $tt}
	loadconfig {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    APPLoadConfig $tt $fn
	    FileLastFull apconfigfbox $fn
	}
	saveconfig {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    APPSaveConfig $tt $fn
	    FileLastFull apconfigfbox $fn
	}
	print {incr i; APProcessPrint $tt $varname $iname}
	page {incr i; APProcessPage $tt $varname $iname}
	close {APDestroy $tt}

	graph {incr i; APProcessGraph $tt $varname $iname}
	font {incr i; APProcessFont $tt $varname $iname}
	dataset {incr i; APProcessDataset $tt $varname $iname}
	view {incr i; APProcessView $tt $varname $iname}
	color {incr i; APProcessColor $tt $varname $iname}
	line {incr i; APProcessLine $tt $varname $iname}
    }
}

proc APProcessNew {tt varname iname bufname} {
    upvar 2 $varname var
    upvar 2 $iname i
    upvar 2 $bufname buf

    global ap

    switch -- [string tolower [lindex $var $i]] {
	stdin {lappend ap(xpa) [AnalysisPlotStdin $tt {} buf]}
	{} {
	    lappend ap(xpa) [AnalysisPlot $tt {} {} {} {} xy $buf]
	    incr i -1
	}
	default {
	    lappend ap(xpa) \
		[AnalysisPlot $tt {} \
		     [lindex $var $i] \
		     [lindex $var [expr $i+1]] \
		     [lindex $var [expr $i+2]] \
		     [lindex $var [expr $i+3]] \
		     $buf]
	    incr i 3
	}
    }
}

proc APProcessData {tt varname iname bufname} {
    upvar 2 $varname var
    upvar 2 $iname i
    upvar 2 $bufname buf

    global ap

    APRaise $tt
    APDataSet $tt [lindex $var $i] $buf
}

proc APProcessPrint {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ps
    global current

    switch -- [string tolower [lindex $var $i]] {
	destination {incr i; set ps(dest) [lindex $var $i]}
	command {incr i; set ps(cmd) [lindex $var $i]}
	filename {incr i; set ps(filename) [lindex $var $i] }
	palette {incr i; set ps(color2) [lindex $var $i] }

	{} {APPostScript $tt}
	default {incr i -1; APPostScript $tt}
    }
}

proc APProcessPage {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ps

    switch -- [string tolower [lindex $var $i]] {
	orientation {incr i; set ps(orient) [lindex $var $i]}
	pagescale {incr i; set ps(scale) [lindex $var $i]}
	pagesize {incr i; set ps(size) [lindex $var $i] }
    }
}

proc APProcessGraph  {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ap

    switch -- [string tolower [lindex $var $i]] {
	grid {incr i; set ap($tt,grid) [FromYesNo [lindex $var $i]]}
	scale {incr i; set ap($tt,grid,log) [lindex $var $i]}
	range {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		x {
		    incr i
		    switch -- [string tolower [lindex $var $i]] {
			auto {incr i; set ap($tt,graph,x,auto) \
				  [FromYesNo [lindex $var $i]]}
			min {incr i; set ap($tt,graph,x,min) [lindex $var $i]}
			max {incr i; set ap($tt,graph,x,max) [lindex $var $i]}
		    }
		}
		y {
		    incr i
		    switch -- [string tolower [lindex $var $i]] {
			auto {incr i; set ap($tt,graph,y,auto) \
				  [FromYesNo [lindex $var $i]]}
			min {incr i; set ap($tt,graph,y,min) [lindex $var $i]}
			max {incr i; set ap($tt,graph,y,max) [lindex $var $i]}
		    }
		}
	    }
	}
	labels {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		title {incr i; set ap($tt,graph,title) [lindex $var $i]}
		xaxis {incr i; set ap($tt,graph,xaxis) [lindex $var $i]}
		yaxis {incr i; set ap($tt,graph,yaxis) [lindex $var $i]}
	    }
	}
    }
    APUpdateGraph $tt
}

proc APProcessFont  {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ap

    switch -- [string tolower [lindex $var $i]] {
	numbers {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		font {incr i; set ap($tt,numlabFont) [lindex $var $i]}
		size {incr i; set ap($tt,numlabSize) [lindex $var $i]}
		style {incr i; set ap($tt,numlabStyle) [lindex $var $i]}
	    }
	}
	labels {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		font {incr i; set ap($tt,textlabFont) [lindex $var $i]}
		size {incr i; set ap($tt,textlabSize) [lindex $var $i]}
		style {incr i; set ap($tt,textlabStyle) [lindex $var $i]}
	    }
	}
	title {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		font {incr i; set ap($tt,titleFont) [lindex $var $i]}
		size {incr i; set ap($tt,titleSize) [lindex $var $i]}
		style {incr i; set ap($tt,titleStyle) [lindex $var $i]}
	    }
	}
    }
    APUpdateGraph $tt
}

proc APProcessDataset  {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ap

    set  ap($tt,data,current) [lindex $var $i]
    APCurrentData $tt
}

proc APProcessView  {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ap

    switch -- [string tolower [lindex $var $i]] {
	discrete {incr i; set ap($tt,discrete) [FromYesNo [lindex $var $i]]}
	line {incr i; set ap($tt,linear) [FromYesNo [lindex $var $i]]}
	step {incr i; set ap($tt,step) [FromYesNo [lindex $var $i]]}
	quadratic {incr i; set ap($tt,quadratic) [FromYesNo [lindex $var $i]]}
	errorbar {incr i; set ap($tt,error) [FromYesNo [lindex $var $i]]}
    }

    APUpdateState $tt
}

proc APProcessColor  {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ap

    switch -- [string tolower [lindex $var $i]] {
	discrete {incr i; set ap($tt,discrete,color) [lindex $var $i]}
	line {incr i; set ap($tt,linear,color) [lindex $var $i]}
	step {incr i; set ap($tt,step,color) [lindex $var $i]}
	quadratic {incr i; set ap($tt,quadratic,color) [lindex $var $i]}
	errorbar {incr i; set ap($tt,error,color) [lindex $var $i]}
    }

    APUpdateState $tt
}

proc APProcessLine  {tt varname iname} {
    upvar 2 $varname var
    upvar 2 $iname i

    global ap

    switch -- [string tolower [lindex $var $i]] {
	discrete {incr i; set ap($tt,discrete,symbol) [lindex $var $i]}
	line {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		width {incr i; set ap($tt,linear,width) [lindex $var $i]}
		dash {
		    incr i
		    set ap($tt,linear,dash) [FromYesNo [lindex $var $i]]}
	    }
	}
	step {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		width {incr i; set ap($tt,step,width) [lindex $var $i]}
		dash {
		    incr i
		    set ap($tt,step,dash) [FromYesNo [lindex $var $i]]}
	    }
	}
	quadratic {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		width {incr i; set ap($tt,quadratic,width) [lindex $var $i]}
		dash {
		    incr i
		    set ap($tt,quadratic,dash) [FromYesNo [lindex $var $i]]}
	    }
	}
	errorbar {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		width {incr i; set ap($tt,error,width) [lindex $var $i]}
		style {
		    incr i;
		    set ap($tt,error,style) [expr [lindex $var $i]-1]}
	    }
	}
    }

    APUpdateState $tt
}
