#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc CATSearch {varname} {
    upvar #0 $varname var
    global $varname

    global cat
    global ed

    set w ".catcds"

    set lbw 15
    set lbh 10

    set ed(ok) 0

    set ed(source) $cat(source)
    set ed(words) $cat(words)
    set ed(wave) $cat(wave)
    set ed(mission) $cat(mission)
    set ed(astro) $cat(astro)

    set ed(list,wave) [list none Radio IR optical UV EUV X-ray Gamma-ray]
    set ed(list,mission) [list none ANS ASCA BeppoSAX CGRO COBE Chandra Copernicus EUVE EXOSAT Einstein FAUST FUSE GINGA GOODS GRANAT HEAO HST HUT Hipparcos INTEGRAL IRAS ISO IUE MSX OAO-2 ORFEUS ROSAT RXTE SAS-2 SIRTF SMM SOHO Spitzer TD1 UIT ULYSSES WUPPE XMM]
    set ed(list,astro) [list none AGN Abundances Ages Associations Atomic_Data BL_Lac_objects Binaries:cataclysmic Binaries:eclipsing Binaries:spectroscopic Blue_objects Clusters_of_galaxies Constellations Diameters Earth Ephemerides Equivalent_widths Extinction Galaxies Galaxies:Markarian Galaxies:spectra Globular_Clusters Gravitational_lensing HII_regions Interstellar_Medium Magnetic_fields Masers Masses Models Multiple_Stars Nebulae Nonstellar Novae Obs_Log Open_Clusters Orbits Parallaxes Photometry Photometry:intermediate-band Photometry:narrow-band Photometry:surface Photometry:wide-band Planetary_Nebulae Planets+Asteroids Polarization Positional_Data Proper_Motions Pulsars QSOs Redshifts Rotational_Velocities Seyfert_Galaxies Spectral_Classification Spectrophotometry Spectroscopy Stars Stars:Emission Stars:WR Stars:early-type Stars:late-type Stars:peculiar Stars:variable Stars:white_dwarf Sun SuperNovae SuperNovae_Remnants Velocities YSOs _META_]

    DialogCreate $w "Search for Catalogs" -borderwidth 2
    frame $w.ed -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2

    pack $w.buttons -side bottom -fill x -ipadx 4 -ipady 4
    pack $w.ed -side top -fill both -expand true -ipadx 4 -ipady 4

    frame $w.ed.name 
    label $w.ed.or -text "Or"
    frame $w.ed.words
    frame $w.ed.srch 
    pack $w.ed.name $w.ed.or $w.ed.words -side top -anchor w -pady 4
    pack $w.ed.srch -side bottom -fill both -expand true -anchor w -pady 4

    # name
    label $w.ed.name.title -text "Name or Designation"
    entry $w.ed.name.source -textvariable ed(source) -width 45
    pack $w.ed.name.title $w.ed.name.source -side top -anchor w

    # keywords
    label $w.ed.words.title -text "Words matching title, description"
    entry $w.ed.words.key -textvariable ed(words) -width 45
    pack $w.ed.words.title $w.ed.words.key -side top -anchor w

    # search
    frame $w.ed.srch.wave
    frame $w.ed.srch.mission
    frame $w.ed.srch.astro
    pack $w.ed.srch.wave $w.ed.srch.mission $w.ed.srch.astro \
	-side left -fill both -expand true -padx 4

    # wave
    frame $w.ed.srch.wave.f
    label $w.ed.srch.wave.title -text "Wavelength"
    pack $w.ed.srch.wave.title -side top -anchor w
    pack $w.ed.srch.wave.f -side bottom -fill both -expand true -anchor w

    scrollbar $w.ed.srch.wave.f.scroll \
	-command "$w.ed.srch.wave.f.list yview"
    set ed(listbox,wave) [listbox $w.ed.srch.wave.f.list \
			      -yscroll "$w.ed.srch.wave.f.scroll set" \
			      -setgrid 1 \
			      -width $lbw \
			      -height $lbh \
			      -selectmode browse \
			      -exportselection 0 \
			      -listvariable ed(list,wave)]
    grid $w.ed.srch.wave.f.list $w.ed.srch.wave.f.scroll -sticky news
    grid rowconfigure $w.ed.srch.wave.f 0 -weight 1
    grid columnconfigure $w.ed.srch.wave.f 0 -weight 1

    # mission
    frame $w.ed.srch.mission.f
    label $w.ed.srch.mission.title -text "Mission"
    pack $w.ed.srch.mission.title -side top -anchor w
    pack $w.ed.srch.mission.f -side bottom -fill both -expand true  -anchor w

    scrollbar $w.ed.srch.mission.f.scroll \
	-command "$w.ed.srch.mission.f.list yview"
    set ed(listbox,mission) [listbox $w.ed.srch.mission.f.list \
				 -yscroll "$w.ed.srch.mission.f.scroll set"\
				 -setgrid 1 \
				 -width $lbw \
				 -height $lbh \
				 -selectmode browse \
				 -exportselection 0 \
				 -listvariable ed(list,mission)]
    grid $w.ed.srch.mission.f.list $w.ed.srch.mission.f.scroll -sticky news
    grid rowconfigure $w.ed.srch.mission.f 0 -weight 1
    grid columnconfigure $w.ed.srch.mission.f 0 -weight 1

    # astro
    frame $w.ed.srch.astro.f
    label $w.ed.srch.astro.title -text "Astronomical"
    pack $w.ed.srch.astro.title -side top -anchor w
    pack $w.ed.srch.astro.f -side bottom -fill both -expand true -anchor w

    scrollbar $w.ed.srch.astro.f.scroll \
	-command "$w.ed.srch.astro.f.list yview"
    set ed(listbox,astro) [listbox $w.ed.srch.astro.f.list \
			       -yscroll "$w.ed.srch.astro.f.scroll set" \
			       -setgrid 1 \
			       -width $lbw \
			       -height $lbh \
			       -selectmode browse \
			       -exportselection 0 \
			       -listvariable ed(list,astro)]
    grid $w.ed.srch.astro.f.list $w.ed.srch.astro.f.scroll -sticky news
    grid rowconfigure $w.ed.srch.astro.f 0 -weight 1
    grid columnconfigure $w.ed.srch.astro.f 0 -weight 1

    # buttons
    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok $w.buttons.cancel -side left -expand true -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    if {$ed(wave) != {}} {
	$ed(listbox,wave) selection set [lsearch $ed(list,wave) $ed(wave)]
    } else {
	$ed(listbox,wave) selection set 0
    }
    if {$ed(mission) != {}} {
	$ed(listbox,mission) selection set \
	    [lsearch $ed(list,mission) $ed(mission)]
    } else {
	$ed(listbox,mission) selection set 0
    }
    if {$ed(astro) != {}} {
	$ed(listbox,astro) selection set [lsearch $ed(list,astro) $ed(astro)]
    } else {
	$ed(listbox,astro) selection set 0
    }

    $w.ed.name.source select range 0 end
    DialogWait $w ed(ok) $w.ed.name.source

    set id [$ed(listbox,wave) curselection]
    if {$id > 0} {
	set ed(wave) [lindex $ed(list,wave) $id]
    } else {
	set ed(wave) {}
    }
    set id [$ed(listbox,mission) curselection]
    if {$id > 0} {
	set ed(mission) [lindex $ed(list,mission) $id]
    } else {
	set ed(mission) {}
    }
    set id [$ed(listbox,astro) curselection]
    if {$id > 0} {
	set ed(astro) [lindex $ed(list,astro) $id]
    } else {
	set ed(astro) {}
    }

    DialogDismiss $w

    if {$ed(ok)} {
	set cat(source) $ed(source)
	set cat(words) $ed(words)
	set cat(wave) $ed(wave)
	set cat(mission) $ed(mission)
	set cat(astro) $ed(astro)

	$var(mb).file entryconfig Retrieve -state disabled
	$var(mb).file entryconfig Cancel -state normal

	$var(apply) configure -state disabled
	$var(cancel) configure -state normal

	ARStatus $varname "Searching for catalogs..."

	CATCDSSearch $varname "$ed(source)" "$ed(words)" "$ed(wave)" \
	    "$ed(mission)" "$ed(astro)"
    }

    unset ed
}

proc CATCDSSearch {varname source words wave mission astro} {
    upvar #0 $varname var
    global $varname

    global http

    switch -- $var(server) {
	cds {set site "vizier.u-strasbg.fr"}
	sao {set site "vizier.cfa.harvard.edu"}
	cadc {set site "vizier.hia.nrc.ca"}
	adac {set site "vizier.nao.ac.jp"}
	iucaa {set site "urania.iucaa.ernet.in"}
	moscow {set site "www.inasan.rssi.ru"}
	bejing {set site "data.bao.ac.cn"}
	cambridge {set site "archive.ast.cam.ac.uk"}
	ukirt {set site "www.ukirt.jach.hawaii.edu"}
    }

    set cgidir "viz-bin"
    set script "asu-tsv"

    set url "http://$site/$cgidir/$script"
    
    # defaults
    set query {}
    append query "-meta&"
    append query "-out.max=1000&"
    if {$source != {}} {
	append query "-source=$source&"
    }
    if {$words !={}} {
	append query "-words=$words&"
    }
    if {$wave !={}} {
	append query "-kw.Wavelength=$wave&"
    }
    if {$mission !={}} {
	append query "-kw.Mission=$mission&"
    }
    if {$astro !={}} {
	append query "-kw.Astronomical=$astro&"
    }

    global $var(dir)
    catch {unset $var(dir)}
    # geturl
    if {$var(sync)} {
	set token [http::geturl $url?$query \
		       -handler [list CATCDSSearchReader $var(dir)] \
		       -headers "[ProxyHTTP]"]
	set var(state) 1
	set var(token) $token

	CATCDSSearchFinish $varname $token
    } else {
	set token [http::geturl $url?$query \
		       -handler [list CATCDSSearchReader $var(dir)] \
		       -command [list CATCDSSearchFinish $varname] \
		       -headers "[ProxyHTTP]"]

	set var(state) 1
	set var(token) $token
    }
}

proc CATCDSSearchReader {t sock token} {
    upvar #0 $t T
    global $t

    set result 0

    if { ![info exists ${t}(state)]  } {
	set T(state) 0
    }

    switch -- $T(state) {
	0 {
	    # init db
	    fconfigure $sock -blocking 1
	    set T(Nrows) 0
	    set T(Ncols) 0
	    set T(Header) {}
	    set T(HLines) 0

	    set T(state) 1
	}

	1 {
	    # process header
	    incr ${t}(HLines)
	    set n $T(HLines)
	    
	    if { [gets $sock line] == -1 } {
		set T(state) -1
		set T(HLines) [expr $T(HLines) - 1]
		set T(Nrows) 0
		set T(Ncols) 0
		return
	    }
	    set result [string length "$line"]

	    set T(H_$n) $line
	    if {[string range $line 0 4] == "\#RESO"} {
		# remove units line
		unset T(H_$n)
		incr ${t}(HLines) -1
		
		# create header
		set T(H_$n) "Resource\tDescription"
		set T(Header) [split $T(H_$n) "\t"]
		incr ${t}(HLines)
		set T(H_[expr $n+1]) "--------\t-----------"
		set T(Dashes) [split $T(H_$n) "\t"]
		set T(Ndshs) [llength $T(Dashes)]
		
		starbase_colmap $t

		# process $line
		incr ${t}(Nrows)
		set r $T(Nrows)
		set T($r,1) "[lindex [split $line {=}] 1]"
		set T($r,2) {}

		set T(state) 2
	    }
	}

	2 {
	    # process table
	    if {[gets $sock line] == -1} {
		set T(state) 0
		return
	    } else {
		set result [string length "$line"]
		set line [string trim $line]
		if {$line != {}} {
		    switch -- [string range $line 0 4] {
			"#RESO" {
			    incr ${t}(Nrows)
			    set r $T(Nrows)
			    set T($r,1) [lindex [split $line {=}] 1]
			    set T($r,2) {}
			}
			"#INFO" {
			    set r $T(Nrows)
			    if {$r>0} {
				set val [lindex [split $line {=}] 1]
				set id [string range "$val" 0 0]
				if {$id == "1" || $id == "0"} {
				    set l [string length "$val"]
				    set val [string range "$val" 4 $l]
				    set l [string last {(link)} "$val"]
				    if {$l > 0} {
					set val [string replace "$val" $l end]
				    }
				    set val [string trim "$val"]
				    # I don't know why this generates an erro
				    catch {set T($r,2) "$val"}
				}
			    }
			}
		    }
		}
	    }
	}
    }

    return $result
}

proc CATCDSSearchFinish {varname token} {
    upvar #0 $varname var
    global $varname

    HTTPLog $token
    if {$var(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    ARError $varname $code
	    return
	}

	ARStatus $varname {Done}
	ARReset $varname

	CATCDSSearchSLB $varname
    } else {
	ARStatus $varname {Cancelled}
	ARReset $varname
    }
}

proc CATCDSSearchSLB {varname} {
    upvar #0 $varname var
    global $varname
    global $var(dir)

    CATClear $varname

    set r [starbase_nrows $var(dir)]
    if {$r == 0} {
	CATSet $varname {} {}
	Message "No Catalogs Found"
	return
    }

    if {$r == 1} {
	set title [starbase_get $var(dir) 1 2]
	if {$title == {}} {
	    set title [starbase_get $var(dir) 1 1]
	}
	CATSet $varname [starbase_get $var(dir) 1 1] $title
	CATApply $varname 0
	return
    }

    if {$r > 1} {
	global slb

	set slb(count) $r
	for {set ii 1} {$ii <= $r} {incr ii} {
	    set slb($ii,value) [starbase_get $var(dir) $ii 1]
	    set slb($ii,item) [starbase_get $var(dir) $ii 2]
	}
	SLBDialog slb {Choose Catalog} 60
	    
	if {$slb(value) != {}} {
	    CATSet $varname $slb(value) $slb(item)
	    CATApply $varname 0
	}
	unset slb
    }
}

