/* initdialog.c - handle sanduhr's initial dialog
 *
 * Copyright (C) 1999, 2000  Jochen Voss.  */

static const  char  rcsid[] = "$Id: initdialog.c 5727 2004-06-01 22:11:03Z voss $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>

#include <gtk/gtk.h>

#include "interface.h"
#include "support.h"

#include "sandcommon.h"


static void
check_time_spec (struct timer *timer, const char *time_spec)
/* Check TIME_SPEC for validity and enter the time into TIMER.
 * If the specification is valid set 'timer->time_valid'.  */
{
  if (time_spec) {
    begin_arg (time_spec);
    if (yyparse (timer) == 0)  timer->time_valid = TRUE;
    end_arg ();
  }
  
  if (! timer->time_valid)  timer->target_time_rel = +100;

  if (timer->time_valid) {
    timer->start_time = vclock ();
    initialize_time (timer);
    if (timer->target_time_rel < 1)  timer->time_valid = FALSE;
  }
}

static void
check_time_error (struct timer *timer, GtkWidget *parent, const gchar *spec)
{
  if (! spec || ! *spec)  return;
  
  if (! timer->time_valid) {
    gchar *msg;

    if (timer->target_time_rel < 1) {
      msg = g_strdup (_("Too short alarm period\n"
			"(minimum is 1 second)"));
    } else {
      msg = g_strdup_printf (_("invalid time specification\n"
			       "\"%s\""), spec);
    }
    display_error_message (msg, GTK_WINDOW(parent));
    g_free (msg);
  }
}

static GtkWidget *
setup_dialog (const char *message, const char *time_spec)
/* Create the initial dialog.
 * MESSAGE and TIME_SPEC are used as default values for the
 * corresponding entries.  */
{
  GtkWidget *initdialog = create_initdialog ();
  
  gtk_dialog_set_default_response (GTK_DIALOG(initdialog), GTK_RESPONSE_OK);
  if (message && *message) {
    GtkWidget *msg_entry = lookup_widget (initdialog, "message");
    gtk_entry_set_text (GTK_ENTRY (msg_entry), message);
  }
  if (time_spec && *time_spec) {
    GtkWidget *time_entry = lookup_widget (initdialog, "custom_entry");
    gtk_entry_set_text (GTK_ENTRY (time_entry), time_spec);
  }

  return  initdialog;
}

static void
extract_input (struct timer *timer, GtkWidget *initdialog)
/* Extract the data from INITDIALOG and enter it into TIMER. */
{
  GtkWidget *time_entry, *msg_entry;
  const char *time_spec = NULL;
  const gchar *msg;

  time_entry = lookup_widget (initdialog, "custom_entry");
  msg_entry = lookup_widget (initdialog, "message");

  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (lookup_widget (initdialog, "b90")))) {
    timer->is_absolute = FALSE;
    timer->target_time_rel = 90;
    timer->time_valid = TRUE;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (initdialog, "b180")))) {
    timer->is_absolute = FALSE;
    timer->target_time_rel = 180;
    timer->time_valid = TRUE;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (initdialog, "b300")))) {
    timer->is_absolute = FALSE;
    timer->target_time_rel = 300;
    timer->time_valid = TRUE;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (initdialog, "b1800")))) {
    timer->is_absolute = FALSE;
    timer->target_time_rel = 1800;
    timer->time_valid = TRUE;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (initdialog, "bxxx")))) {
    time_spec = gtk_entry_get_text (GTK_ENTRY (time_entry));
  } else {
    g_assert_not_reached ();
  }

  check_time_spec (timer, time_spec);
  check_time_error (timer, initdialog, time_spec);
  
  g_free (timer->message);
  msg = gtk_entry_get_text (GTK_ENTRY (msg_entry));
  if (msg && *msg) {
    timer->message = g_strdup (msg);
    gnome_config_set_string ("/SandUhr/preferences/message", msg);
    gnome_config_sync ();
  } else {
    timer->message = NULL;
  }
}

struct cb_data {
  struct timer *timer;
  void (*success) (struct timer *);
  void (*failure) (struct timer *);
};

static void
dialog_clicked_cb (GtkObject *object, gint response, gpointer data)
/* Callback to handle the result of the initdialog.
 * This is called when the dialog is finished by either clicking
 * one of the buttons or by closing it.  */
{
  GtkWidget *initdialog = (GtkWidget *)object;
  struct cb_data *cbd = data;
  struct timer *timer = cbd->timer;
  gboolean  is_success = FALSE;

  switch (response) {
  case GTK_RESPONSE_OK:
    extract_input (timer, initdialog);
    if (timer->time_valid) {
      is_success = TRUE;
    }
    break;
  case GTK_RESPONSE_NONE:
  case GTK_RESPONSE_CANCEL:
  case GTK_RESPONSE_DELETE_EVENT:
    break;
  case GTK_RESPONSE_HELP:
    puts("help!");
    gnome_help_display ("sanduhr", "timespec", NULL);
    return;
  default:
    g_assert_not_reached ();
  }

  if (is_success) {
    cbd->success (timer);
  } else {
    cbd->failure (timer);
  }
  gtk_widget_destroy (initdialog);
  g_free (cbd);
}

void
ask_for_time (struct timer *timer, const char *time_spec,
	      void (*success)(struct timer *), void (*failure)(struct timer *),
	      CORBA_Environment *ev)
/* Asynchronously determine the alarm time, based on TIME_SPEC.
 * If the TIME_SPEC contains no valid alarm time and if EV is not set,
 * then pop up a dialog to ask the user.  After the outcome becames clear,
 * call one of the callback functions SUCCESS or FAILURE.  */
{
  GtkWidget *initdialog;
  struct cb_data *cbd;
  
  check_time_spec (timer, time_spec);
  if (timer->time_valid) {
    success (timer);
    return;
  } else if (ev && time_spec && *time_spec) {
    CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			 ex_SandUhr_TimerFactory_InvalidTime,
			 NULL);
    failure (timer);
    return;
  }
  
  cbd = g_new (struct cb_data, 1);
  cbd->timer = timer;
  cbd->success = success;
  cbd->failure = failure;
  
  initdialog = setup_dialog (timer->message, time_spec);
  gtk_signal_connect (GTK_OBJECT (initdialog),
		      "response",
		      GTK_SIGNAL_FUNC (dialog_clicked_cb),
		      cbd);
  gtk_widget_show (initdialog);
}
