/***************************************************************************
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 *   Licensed under the GNU GPL                                            *
 *   Original code supplied under license from RaLink Inc, 2003.           *
 ***************************************************************************/

 /***************************************************************************
 *      Module Name: mlme.c
 *
 *      Abstract:
 *
 *      Revision History:
 *      Who             When            What
 *      --------        -----------     -----------------------------
 *      MarkW           9th  Feb 04     Baseline of code
 *      Ivo		26th Jun 04	Fixed Uninitialised timer (964429)
 *	MarkW		7th  Dec 04	Fixed auto-reconnect (1034910)
 *      RobinC          21st Dec 04     RFMON Support
 *      MarkW           21st Dec 04     iwconfig mode monitor support
 ***************************************************************************/


#include "rt_config.h"

UCHAR BROADCAST_ADDR[ETH_ALEN] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
UCHAR ZERO_MAC_ADDR[ETH_ALEN] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

/*
    ==========================================================================
    Description:
        initialize the MLME task and its data structure (queue, spinlock,
        timer, state machines).
    Return:
        always return NDIS_STATUS_SUCCESS
    ==========================================================================
*/
NDIS_STATUS MlmeInit(
    IN PRTMP_ADAPTER pAd)
{
    NDIS_STATUS Status = NDIS_STATUS_SUCCESS;

    DBGPRINT(RT_DEBUG_TRACE, "--> MLME Initialize\n");

    do
    {
        pAd->Mlme.Running = FALSE;

        // initialize the two tables
        // MacTableInit(pAd);
        BssTableInit(&pAd->PortCfg.BssTab);

        // init state machines
        ASSERT(ASSOC_FUNC_SIZE == MAX_ASSOC_MSG * MAX_ASSOC_STATE);
        AssocStateMachineInit(pAd, &pAd->Mlme.AssocMachine, pAd->Mlme.AssocFunc);

        ASSERT(AUTH_FUNC_SIZE == MAX_AUTH_MSG * MAX_AUTH_STATE);
        AuthStateMachineInit(pAd, &pAd->Mlme.AuthMachine, pAd->Mlme.AuthFunc);

        ASSERT(AUTH_RSP_FUNC_SIZE == MAX_AUTH_RSP_MSG * MAX_AUTH_RSP_STATE);
        AuthRspStateMachineInit(pAd, &pAd->Mlme.AuthRspMachine, pAd->Mlme.AuthRspFunc);

        ASSERT(SYNC_FUNC_SIZE == MAX_SYNC_MSG * MAX_SYNC_STATE);
        SyncStateMachineInit(pAd, &pAd->Mlme.SyncMachine, pAd->Mlme.SyncFunc);

        // Since we are using switch/case to implement it, the init is different from the above
        // state machine init
        MlmeCntlInit(pAd, &pAd->Mlme.CntlMachine, NULL);

        // Regularly check the timer
        init_timer(&pAd->Mlme.PeriodicTimer);
		pAd->Mlme.PeriodicTimer.expires = jiffies + MLME_TASK_EXEC_INTV;	// the time, in jiffies
		pAd->Mlme.PeriodicTimer.data = (unsigned long)pAd;
		pAd->Mlme.PeriodicTimer.function = &MlmePeriodicExec;				/* timer handler */
		add_timer(&pAd->Mlme.PeriodicTimer);

	init_timer(&pAd->PortCfg.LedCntl.BlinkTimer);
	if (pAd->PortCfg.LedMode == LED_MODE_TXRX_ACTIVITY)
        {
			pAd->PortCfg.LedCntl.BlinkTimer.expires = jiffies + (70 * HZ)/1000;	// the time, in jiffies
			pAd->PortCfg.LedCntl.BlinkTimer.data = (unsigned long)pAd;
			pAd->PortCfg.LedCntl.BlinkTimer.function = &AsicLedPeriodicExec;				/* timer handler */
			add_timer(&pAd->PortCfg.LedCntl.BlinkTimer);
        }

        // for dynamic "R13: RX AGC VGC INIT" control
        {
            UCHAR FalseCcaHigh;
            pAd->PortCfg.RxVgc.Duration = 1;
            pAd->PortCfg.RxVgc.FlaseCcaThreshold = 8;
            pAd->PortCfg.RxVgc.RxAgcVgcDelta = 2;
            pAd->PortCfg.RxVgc.MaxRxAgcVgc = 0x20;
            RTMP_BBP_IO_READ32_BY_REG_ID(pAd, 39, &FalseCcaHigh);
            pAd->PortCfg.BbpFalseCca = FalseCcaHigh; //(FalseCcaHigh << 8) + FalseCcaLow;
        }
    } while (FALSE);

    DBGPRINT(RT_DEBUG_TRACE, "<-- MLME Initialize\n");

    return Status;
}

/*
    ==========================================================================
    Description:
        main loop of the MLME
    Pre:
        Mlme has to be initialized, and there are something inside the queue
    Note:
        This function is invoked from MPSetInformation and MPReceive;
        This task guarantee only one MlmeHandler will run.
    ==========================================================================
 */
VOID MlmeHandler(
    IN PRTMP_ADAPTER pAd)
{
    MLME_QUEUE_ELEM        *Elem = NULL;
    unsigned long flags;

    // Only accept MLME and Frame from peer side, no other (control/data)
	// frame should get into this state machine

	// We fix the multiple context service drop problem identified by
	// Ben Hutchings in an SMP- safe way by combining TaskLock and Queue.Lock
	// per his suggestion.
    spin_lock_irqsave(&pAd->Mlme.Queue.Lock, flags);
    if(pAd->Mlme.Running)
    {
        spin_unlock_irqrestore(&pAd->Mlme.Queue.Lock, flags);
        return;
    }
	pAd->Mlme.Running = TRUE;

	// If there's a bubble, wait for it to collapse before proceeding.
    while (MlmeGetHead(&pAd->Mlme.Queue, &Elem)) {
		smp_read_barrier_depends();
		if (!Elem->Occupied) break;

		spin_unlock_irqrestore(&pAd->Mlme.Queue.Lock, flags);

        //From message type, determine which state machine I should drive
        if (pAd->PortCfg.BssType != BSS_MONITOR) switch (Elem->Machine)
        {
            case ASSOC_STATE_MACHINE:
                StateMachinePerformAction(pAd, &pAd->Mlme.AssocMachine, Elem);
                break;
            case AUTH_STATE_MACHINE:
                StateMachinePerformAction(pAd, &pAd->Mlme.AuthMachine, Elem);
                break;
            case AUTH_RSP_STATE_MACHINE:
                StateMachinePerformAction(pAd, &pAd->Mlme.AuthRspMachine, Elem);
                break;
            case SYNC_STATE_MACHINE:
                StateMachinePerformAction(pAd, &pAd->Mlme.SyncMachine, Elem);
                break;
            case MLME_CNTL_STATE_MACHINE:
                MlmeCntlMachinePerformAction(pAd, &pAd->Mlme.CntlMachine, Elem);
                break;
            default:
                DBGPRINT(RT_DEBUG_ERROR,
						"ERROR: Illegal machine in MlmeHandler()\n");
                break;
        } // end of switch

        // free MLME element
        smp_mb();
        Elem->Occupied = FALSE;	// sic - bb
		spin_lock_irqsave(&pAd->Mlme.Queue.Lock, flags);
		MlmeDequeue(&pAd->Mlme.Queue);
    }
    pAd->Mlme.Running = FALSE;
    spin_unlock_irqrestore(&pAd->Mlme.Queue.Lock,flags);
}

/*
    ==========================================================================
    Description:
        Destructor of MLME (Destroy queue, state machine, spin lock and timer)
    Parameters:
        Adapter - NIC Adapter pointer
    Post:
        The MLME task will no longer work properly
    ==========================================================================
 */
VOID MlmeHalt(
    IN PRTMP_ADAPTER pAd)
{
    MLME_DISASSOC_REQ_STRUCT DisReq;
    MLME_QUEUE_ELEM          MsgElem;

    DBGPRINT(RT_DEBUG_TRACE, "==> MlmeHalt\n");

    if (INFRA_ON(pAd))
    {
        COPY_MAC_ADDR(&DisReq.Addr, &pAd->PortCfg.Bssid);
        DisReq.Reason =  REASON_DISASSOC_STA_LEAVING;

        MsgElem.Machine = ASSOC_STATE_MACHINE;
        MsgElem.MsgType = MT2_MLME_DISASSOC_REQ;
        MsgElem.MsgLen = sizeof(MLME_DISASSOC_REQ_STRUCT);
        memcpy(MsgElem.Msg, &DisReq, sizeof(MLME_DISASSOC_REQ_STRUCT));

        MlmeDisassocReqAction(pAd, &MsgElem);

        udelay(1000);		// just for delay little microsecond, recommend Max is 1000 us
    }

    // disable BEACON generation and other BEACON related hardware timers
    AsicDisableSync(pAd);

    MlmeQueueDestroy(&pAd->Mlme.Queue);
    StateMachineDestroy(&pAd->Mlme.AssocMachine);
    StateMachineDestroy(&pAd->Mlme.AuthMachine);
    StateMachineDestroy(&pAd->Mlme.AuthRspMachine);
    StateMachineDestroy(&pAd->Mlme.SyncMachine);
    //    StateMachineDestroy(&pAd->Mlme.CntlMachine);
//    spin_unlock(&pAd->Mlme.TaskLock);
    // NdisFreeSpinLock(&pAd->PortCfg.MacTab.Lock);

    // Cancel pending timers
    del_timer_sync(&pAd->Mlme.AssocAux.AssocTimer);
    del_timer_sync(&pAd->Mlme.AssocAux.ReassocTimer);
    del_timer_sync(&pAd->Mlme.AssocAux.DisassocTimer);
    del_timer_sync(&pAd->Mlme.AuthAux.AuthTimer);
    del_timer_sync(&pAd->Mlme.AuthRspAux.AuthRspTimer);
    del_timer_sync(&pAd->Mlme.SyncAux.BeaconTimer);
    del_timer_sync(&pAd->Mlme.SyncAux.ScanTimer);
    del_timer_sync(&pAd->Mlme.PeriodicTimer);
    // del_timer_sync(&pAd->PortCfg.MacTab.AgedOutTimer);
    del_timer_sync(&pAd->PortCfg.LedCntl.BlinkTimer);
    ASIC_LED_ACT_OFF(pAd);

	DBGPRINT(RT_DEBUG_TRACE, "<== MlmeHalt\n");
}


/*
    ==========================================================================
    Description:
        Conforirm function for out MLME packets
    Parameters:
        Adapter - NIC Adapter pointer
        Msg     - Message returned from ASIC
        MsgLen  - message length, not used in this implementation
    Pre:
        The Msg has to be allocated by NdisAllocateMemoryWithTag
    ==========================================================================
 */
VOID MlmeConfirm(
    IN PRTMP_ADAPTER pAd,
    IN VOID *Msg,
    IN INT MsgLen)
{
    kfree(Msg);
}


/*
    ==========================================================================
    Description:
        This routine is executed periodically to -
        1. Decide if it's a right time to turn on PwrMgmt bit of all
           outgoiing frames
        2. Calculate FoundedRate, and RoamCqi based on statistics of the last
           period. FoundedRate is used to constain actual TX rate, so that
           TX rate won't toggling very frequently between a successful TX
           and a failed TX (rate too high).
        3. If the calculated RoamCqi indicated current connection not healthy,
           Then a ROAMing attempt is tried here.
    ==========================================================================
 */
#define ADHOC_BEACON_LOST_TIME      4096  // 4 sec
VOID MlmePeriodicExec(
    IN	unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;
    ULONG Now32;

    if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF))
		return;

    if (pAd->PortCfg.BssType == BSS_MONITOR)
    {
	mod_timer(&pAd->Mlme.PeriodicTimer, jiffies + MLME_TASK_EXEC_INTV);
        return;
    }

    // dynamic tune BBP R13: RX_AGC_VGC_INIT based on false CCA
    if (pAd->PortCfg.EnableRxAgcVgcTuning &&
        (pAd->Mlme.PeriodicRound < 60) &&       // turn off R13 tuning after 1 minute
        pAd->PortCfg.RxVgc.FlaseCcaThreshold &&
        pAd->PortCfg.RxVgc.RxAgcVgcDelta &&
        ((pAd->Mlme.PeriodicRound % pAd->PortCfg.RxVgc.Duration)==0))
    {
        UCHAR FalseCcaHigh;
        USHORT FalseCca;
        LONG FalseCcaDelta;

        RTMP_BBP_IO_READ32_BY_REG_ID(pAd, 39, &FalseCcaHigh);
        FalseCca = FalseCcaHigh;
        FalseCcaDelta = FalseCca - pAd->PortCfg.BbpFalseCca;
        pAd->PortCfg.BbpFalseCca = FalseCca;
        DBGPRINT(RT_DEBUG_TRACE,"FalseCca(R39)=%d, delta=%d\n",FalseCca,FalseCcaDelta);
        if (FalseCcaDelta >= pAd->PortCfg.RxVgc.FlaseCcaThreshold)
        {
            UCHAR R13;
            RTMP_BBP_IO_READ32_BY_REG_ID(pAd, 13, &R13);
            R13 += pAd->PortCfg.RxVgc.RxAgcVgcDelta;
            if (R13 <= pAd->PortCfg.RxVgc.MaxRxAgcVgc)
            {
                RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, 13, R13);
                DBGPRINT(RT_DEBUG_TRACE,"FalseCcaDelta=%d, >=%d, set R13=%d\n",
                    FalseCcaDelta, pAd->PortCfg.RxVgc.FlaseCcaThreshold, R13);
            }
        }
    }

    Now32 = jiffies;

    if (pAd->RalinkCounters.MgmtRingFullCount >= 2)
    {
	RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_HARDWARE_ERROR);
    }
    else
    {
	pAd->RalinkCounters.MgmtRingFullCount = 0;
    }

    if (INFRA_ON(pAd))
    {
    	// Is PSM bit consistent with user power management policy?
    	// This is the only place that will set PSM bit ON.
      	MlmeCheckForPsmChange(pAd, Now32);

        // send out a NULL frame every 10 sec. for what??? inform "PwrMgmt" bit?

        if ((pAd->Mlme.PeriodicRound % 10) == 8)
        {
//            EnqueueNullFrame(pAd);
        }

        // update channel quality (for Roaming and DRS) and TxRate
        if ((pAd->Mlme.PeriodicRound % 2) == 1)
        {
        MlmeCheckChannelQuality(pAd, Now32);
		if (pAd->bRaConfig == TRUE)
		{
        	if (CQI_IS_BAD(pAd->Mlme.RoamCqi))
        	{
            	pAd->RalinkCounters.BadCQIAutoRecoveryCount ++;
            	DBGPRINT(RT_DEBUG_TRACE, "MMCHK - Bad CQI. Auto Recovery attempt #%d\n", pAd->RalinkCounters.BadCQIAutoRecoveryCount);
            	MlmeAutoRecoverNetwork(pAd);
        	}

        	else if (CQI_IS_FAIR(pAd->Mlme.RoamCqi) || CQI_IS_POOR(pAd->Mlme.RoamCqi))
        	{
            	//pAd->RalinkCounters.PoorCQIRoamingCount ++;
            	MlmeCheckForRoaming(pAd, Now32);
            	//DBGPRINT(RT_DEBUG_TRACE, ("MMCHK - Poor CQI. Roaming attempt #%d\n", pAd->RalinkCounters.PoorCQIRoamingCount));
        	}
		}
    }
    }
    else if (ADHOC_ON(pAd))
    {
        // update channel quality, decide TxRate
        if ((pAd->Mlme.PeriodicRound % 2) == 1)
        {
			MlmeCheckChannelQuality(pAd, Now32);
        }

#ifndef	SINGLE_ADHOC_LINKUP
        // If all peers leave, and this STA becomes the last one in this IBSS, then change MediaState
        // to DISCONNECTED. But still holding this IBSS (i.e. sending BEACON) so that other STAs can
        // join later.
        if ((pAd->PortCfg.LastBeaconRxTime + ADHOC_BEACON_LOST_TIME < Now32) &&
            (pAd->MediaState == NdisMediaStateConnected))
        {
            DBGPRINT(RT_DEBUG_TRACE, "MMCHK - excessive BEACON lost, last STA in this IBSS, MediaState=Disconnected\n");
            pAd->MediaState = NdisMediaStateDisconnected;
//        	NdisMIndicateStatus(pAd->AdapterHandle, NDIS_STATUS_MEDIA_DISCONNECT, (PVOID)NULL, 0);
//			NdisMIndicateStatusComplete(pAd->AdapterHandle);
			// clean up previous SCAN result, add current BSS back to table if any
			BssTableDeleteEntry(&pAd->PortCfg.BssTab, &(pAd->PortCfg.Bssid));
			pAd->PortCfg.LastScanTime = Now32;
        }
#endif
    }
    else
    {
		DBGPRINT(RT_DEBUG_INFO, "MLME periodic exec, no association so far\n");
		if (pAd->PortCfg.AutoReconnect == TRUE)
		{
			if ((pAd->PortCfg.BssTab.BssNr==0) && (pAd->Mlme.CntlMachine.CurrState == CNTL_IDLE))
			{
				MLME_SCAN_REQ_STRUCT	   ScanReq;
				CHAR					   BroadSsid[MAX_LEN_OF_SSID];

				if ((pAd->PortCfg.LastScanTime + 10 * 1000) < Now32)
				{
					DBGPRINT(RT_DEBUG_TRACE, "CNTL - No matching BSS, start a new scan\n");
					BroadSsid[0] = '\0';
					ScanParmFill(pAd, &ScanReq, BroadSsid, 0, BSS_ANY, SCAN_ACTIVE);
					MlmeEnqueue(&pAd->Mlme.Queue, SYNC_STATE_MACHINE, MT2_MLME_SCAN_REQ, sizeof(MLME_SCAN_REQ_STRUCT), &ScanReq);
					pAd->Mlme.CntlMachine.CurrState = CNTL_WAIT_OID_LIST_SCAN;
					// Reset Missed scan number
					pAd->PortCfg.IgnoredScanNumber = 0;
					pAd->PortCfg.LastScanTime = Now32;
				}
			}
			else
			{
				DBGPRINT(RT_DEBUG_INFO, "pAd->PortCfg.AutoReconnect is TRUE\n");
				MlmeAutoReconnectLastSSID(pAd);
			}
		}
        // no connection so far, try to recover latest conencted network every 10 sec, if any
        else
        {
        	if ((pAd->Mlme.PeriodicRound % 10) == 2)
        		MlmeAutoScan(pAd);
			else if (pAd->PortCfg.SsidLen <= MAX_LEN_OF_SSID)
        	{
        		if ((pAd->Mlme.PeriodicRound % 10) == 6)   // 6 sec later, retry latest network
        			MlmeAutoReconnectLastSSID(pAd);
			}
        }
	}
    pAd->Mlme.PeriodicRound ++;

	mod_timer(&pAd->Mlme.PeriodicTimer, jiffies + MLME_TASK_EXEC_INTV);
}

VOID MlmeAutoScan(
    IN PRTMP_ADAPTER pAd)
{
    // check CntlMachine.CurrState to avoid collision with NDIS SetOID request
    if (pAd->Mlme.CntlMachine.CurrState == CNTL_IDLE)
    {
        DBGPRINT(RT_DEBUG_TRACE, "MMCHK - Driver auto scan\n");

        // tell CNTL state machine NOT to call NdisMSetInformationComplete() after completing
        // this request, because this request is initiated by driver itself.
        pAd->Mlme.CntlAux.CurrReqIsFromNdis = FALSE;

        MlmeEnqueue(&pAd->Mlme.Queue,
                    MLME_CNTL_STATE_MACHINE,
                    OID_802_11_BSSID_LIST_SCAN,
                    0,
                    NULL);
        MlmeHandler(pAd);
    }
}

VOID MlmeAutoRecoverNetwork(
    IN PRTMP_ADAPTER pAd)
{
    // check CntlMachine.CurrState to avoid collision with NDIS SetOID request
    if (pAd->Mlme.CntlMachine.CurrState == CNTL_IDLE)
    {
        NDIS_802_11_SSID OidSsid;
        OidSsid.SsidLength = pAd->PortCfg.SsidLen;
        memcpy(OidSsid.Ssid, pAd->PortCfg.Ssid, pAd->PortCfg.SsidLen);

        // 2002-11-26 comments off the following code, because LinkDown() will be performed
        // inside CNTL state machine. No need to repeat it here
#if 0
        if (INFRA_ON(pAd))
        {
            // TODO: something more in addition to LinkDown??? How about a Disassociate frame
            LinkDown(pAd);
        }
#endif

        DBGPRINT(RT_DEBUG_TRACE, "MMCHK - Driver auto recovering network - %s\n", pAd->PortCfg.Ssid);

        // tell CNTL state machine NOT to call NdisMSetInformationComplete() after completing
        // this request, because this request is initiated by driver itself.
        pAd->Mlme.CntlAux.CurrReqIsFromNdis = FALSE;

        MlmeEnqueue(&pAd->Mlme.Queue,
                    MLME_CNTL_STATE_MACHINE,
                    OID_802_11_SSID,
                    sizeof(NDIS_802_11_SSID),
                    &OidSsid);
        MlmeHandler(pAd);
    }

}

VOID MlmeAutoReconnectLastSSID(
    IN PRTMP_ADAPTER pAd)
{
    // check CntlMachine.CurrState to avoid collision with NDIS SetOID request
    if (pAd->Mlme.CntlMachine.CurrState == CNTL_IDLE)
    {
        NDIS_802_11_SSID OidSsid;
        OidSsid.SsidLength = pAd->Mlme.CntlAux.SsidLen;
        memcpy(OidSsid.Ssid, pAd->Mlme.CntlAux.Ssid, pAd->Mlme.CntlAux.SsidLen);

        DBGPRINT(RT_DEBUG_TRACE, "Driver auto reconnect to last OID_802_11_SSID setting - %s\n", pAd->Mlme.CntlAux.Ssid);

		// We will only try this attemp once, therefore change the AutoReconnect flag afterwards.
        pAd->Mlme.CntlAux.CurrReqIsFromNdis = FALSE;
        pAd->PortCfg.AutoReconnect = FALSE;

        MlmeEnqueue(&pAd->Mlme.Queue,
                    MLME_CNTL_STATE_MACHINE,
                    OID_802_11_SSID,
                    sizeof(NDIS_802_11_SSID),
                    &OidSsid);
        MlmeHandler(pAd);
    }
}

/*
    ==========================================================================
    Description:
        This routine checks if there're other APs out there capable for
        roaming. Caller should call this routine only when Massoc=TRUE and
        channel quality is below CQI_GOOD_THRESHOLD.
    Output:
    ==========================================================================
 */
VOID MlmeCheckForRoaming(
    IN PRTMP_ADAPTER pAd,
    IN ULONG    Now32)
{
    USHORT     i;
    BSS_TABLE  *pBssTab = &pAd->Mlme.CntlAux.SsidBssTab;
    BSS_TABLE  *pRoamTab = &pAd->Mlme.CntlAux.RoamTab;
    BSS_ENTRY  *pBss;

    // put all roaming candidates into RoamTab, and sort in RSSI order
    BssTableInit(pRoamTab);
    for (i = 0; i < pBssTab->BssNr; i++)
    {
        pBss = &pBssTab->BssEntry[i];

        if ((pBssTab->BssEntry[i].LastBeaconRxTime + BEACON_LOST_TIME) < Now32)
            continue;    // AP disappear
        if (pBss->Rssi <= RSSI_THRESHOLD_FOR_ROAMING)
            continue;    // RSSI too weak. forget it.
        if (MAC_ADDR_EQUAL(&pBssTab->BssEntry[i].Bssid, &pAd->PortCfg.Bssid))
            continue;    // skip current AP
        if (CQI_IS_FAIR(pAd->Mlme.RoamCqi) && (pAd->PortCfg.LastRssi + RSSI_DELTA > pBss->Rssi))
            continue;    // we're still okay, only AP with stronger RSSI is eligible for roaming

        // AP passing all above rules is put into roaming candidate table
        memcpy(&pRoamTab->BssEntry[pRoamTab->BssNr], pBss, sizeof(BSS_ENTRY));
        pRoamTab->BssNr += 1;
    }

    if (pRoamTab->BssNr > 0)
    {
        // check CntlMachine.CurrState to avoid collision with NDIS SetOID request
        if (pAd->Mlme.CntlMachine.CurrState == CNTL_IDLE)
        {
            // tell CNTL state machine NOT to call NdisMSetInformationComplete() after completing
            // this request, because this request is initiated by driver itself, not from NDIS.
            pAd->Mlme.CntlAux.CurrReqIsFromNdis = FALSE;

          	pAd->RalinkCounters.PoorCQIRoamingCount ++;
            DBGPRINT(RT_DEBUG_TRACE, "MMCHK - Roaming attempt #%d\n", pAd->RalinkCounters.PoorCQIRoamingCount);
            MlmeEnqueue(&pAd->Mlme.Queue, MLME_CNTL_STATE_MACHINE, MT2_MLME_ROAMING_REQ, 0, NULL);
            MlmeHandler(pAd);
        }
    }

}

/*
    ==========================================================================
    Description:
        This routine calculates TxPER, RxPER of the past N-sec period. And
        according to the calculation result, change PortCfg.FoundedRate which
        is the stable TX Rate we expect the Radio situation could sustained.
        RoamCqi is also calculated here to decide if current AP is still doing
        the job. If RoamCqi is not good, a ROAMing attempt may be tried later.

        PortCfg.TxRate will change dynamically within {RATE_1, FoundedRate}
        whenever a TxResult (SUCC, RETRY, or FAIL} is obtained. In comparasion
        to that, FoundedRate changes not that fast within {RATE_1..RATE_11}
        based on TxPER and RxPER for the past N-sec period.
    Output:
        PortCfg.RoamCqi - 0..100
        PortCfg.DrsCqi  - 0..100
        PortCfg.FoundedRate - RATE_1, RATE_2, RATE_5_5, RATE_11
    ==========================================================================
 */
UCHAR DynamicRateDownThreshold[]={0, 16, 20, 25};  // RATE_1 << -84 dB << RATE_2 << -80dB << RATE_5_5 << -75 dB << RATE_11
UCHAR DynamicRateUpThreshold[]=  {16, 16, 20, 255}; // RATE_1 >> -84 dB >> RATE_2 >> -84dB >> RATE_5_5 >> -80dB >> RATE_11
VOID MlmeCheckChannelQuality(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Now32)
{
    ULONG RtsFailCnt, TxFailCnt, TxOkCnt, TxOneRetryCnt, TxMRetryCnt, TxCnt, TxPER;
    ULONG RxFailCnt, RxOkCnt, RxCnt, RxPER, TxPRR, beacon_lost;

    //
    // monitor TX counters change for the past period
    //
    TxFailCnt     = pAd->WlanCounters.FailedCount -
                    pAd->Mlme.PrevWlanCounters.FailedCount;
    RtsFailCnt    = pAd->WlanCounters.RTSFailureCount -
                    pAd->Mlme.PrevWlanCounters.RTSFailureCount;
    TxOneRetryCnt = pAd->WlanCounters.RetryCount -
                    pAd->Mlme.PrevWlanCounters.RetryCount;
    TxMRetryCnt   = pAd->WlanCounters.MultipleRetryCount -
                    pAd->Mlme.PrevWlanCounters.MultipleRetryCount;
    TxOkCnt       = pAd->WlanCounters.TransmittedFragmentCount -
                    pAd->Mlme.PrevWlanCounters.TransmittedFragmentCount;
    TxCnt = TxOkCnt + TxFailCnt;

    if (TxCnt)
        pAd->Mlme.TxPER = (TxFailCnt * 100) / TxCnt;
    else
        pAd->Mlme.TxPER = 0;

    if (TxCnt < 10) // if less than 10 TX sample, skip TX related statistics
    {
        TxPER = 0;  // don't take TxPER into CQI consideration if too few sample
        TxPRR = 0;
    }
    else
    {
        TxPER = pAd->Mlme.TxPER;
        TxPRR = ((TxOneRetryCnt + TxMRetryCnt + TxFailCnt) * 100) / TxCnt;
    }

    //
    // calculate RX PER
    //
    RxOkCnt   = pAd->WlanCounters.ReceivedFragmentCount -
                pAd->Mlme.PrevWlanCounters.ReceivedFragmentCount;
    RxFailCnt = pAd->WlanCounters.FCSErrorCount -
                pAd->Mlme.PrevWlanCounters.FCSErrorCount;
    RxCnt = RxOkCnt + RxFailCnt;

    if (RxCnt)
        pAd->Mlme.RxPER = (RxFailCnt * 100) / RxCnt;
    else
        pAd->Mlme.RxPER = 0;

    if (RxCnt < 10)
        RxPER = 0;  // don't take RxPER into CQI consideration if too few sample
    else
        RxPER = pAd->Mlme.RxPER;

#if 1
    //
    // dynamic change TX rate; available for both ADHOC mode and INFRA mode
    //
    if (pAd->MediaState == NdisMediaStateConnected)
    {
        // downgrade TA rate only when -
        // 1. there's TX error (otherwise who care); AND
        // 2. current rate > RATE_1, cause can't go lower; AND
        // 3. RSSI below a threshold which implies far away from AP. downgrage rate may help
	// 4. TX rate not fixed
        if (((TxFailCnt + RtsFailCnt) >= 1)  &&
            (pAd->PortCfg.TxRate > RATE_1) &&
	    !(pAd->PortCfg.TxRateFixed) &&
            (pAd->PortCfg.LastRssi < DynamicRateDownThreshold[pAd->PortCfg.TxRate]))
        {
            pAd->PortCfg.TxRate --;
            DBGPRINT(RT_DEBUG_TRACE, "MMCHK - downgrade TX rate to %d <<<\n", pAd->PortCfg.TxRate);
        }
        // upgrade TX rate only when -
        // 1. can't go higher than MAX TX rate; AND
        // 2. RSSI restore to better value which means approaching AP; AND
        // 3. no TX error
        // 4. TX rate not fixed.
        else if ((pAd->PortCfg.TxRate < pAd->PortCfg.MaxTxRate) &&
                 (pAd->PortCfg.LastRssi > DynamicRateUpThreshold[pAd->PortCfg.TxRate]) &&
                 ((TxFailCnt + RtsFailCnt)==0) && !(pAd->PortCfg.TxRateFixed))
        {
            pAd->PortCfg.TxRate ++;
            DBGPRINT(RT_DEBUG_TRACE, "MMCHK - upgrade TX rate to %d <<<\n", pAd->PortCfg.TxRate);
        }
    }
#endif

    //
    // decide CQI based on: 1)last BEACON received time, 2)last RSSI, 3)TxPER, and 4)RxPER
    //
    // This value also decides when all roaming fails (or no roaming candidates at
    // all), should this STA stay with original AP, or a LinkDown signal
    // is indicated to NDIS
    //
    if (INFRA_ON(pAd) &&
        (pAd->PortCfg.LastBeaconRxTime + BEACON_LOST_TIME < Now32))  // BEACON starving?
    {
    	// Ignore lost beacon when NIC in reset state
    	// Ignore lost beacon when if traffic still goes well
    	if (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS) && (TxOkCnt < 20))
    	{
		beacon_lost = BEACON_LOST_TIME/HZ;
        	DBGPRINT(RT_DEBUG_TRACE, "MMCHK - BEACON lost for more than %d sec, let CQI = 0\n", beacon_lost);
        	pAd->Mlme.RoamCqi = 0;
        	// Lost AP, send disconnect & link down event
			LinkDown(pAd);
    	}
    }
    else
    {
        // RoamCQI = W1*RSSI + W2*TxPRR + W3*RxPER    (RSSI 0..100), (TxPER 100..0), (RxPER 100..0)
        // DrsCQI  = W1*RSSI + W2*TxPRR + W3*RxPER
        pAd->Mlme.RoamCqi = (RSSI_WEIGHTING * pAd->PortCfg.LastRssi +
                             TX_WEIGHTING * (100 - TxPRR) +
                             RX_WEIGHTING* (100 - RxPER)) / 100;
        if (pAd->Mlme.RoamCqi >= 100)
            pAd->Mlme.RoamCqi = 100;
        pAd->Mlme.DrsCqi  = pAd->Mlme.RoamCqi;
    }

    DBGPRINT(RT_DEBUG_INFO, "MMCHK - CQI= %d, (Tx=%d/%d/%d, Rx=%d/%d, RSSI=%d)\n",
        pAd->Mlme.RoamCqi, TxFailCnt, TxOneRetryCnt+TxMRetryCnt, TxCnt, RxFailCnt, RxCnt, pAd->PortCfg.LastRssi);

    // latch current WLAN counters for next check-for-roaming usage
    memcpy(&pAd->Mlme.PrevWlanCounters, &pAd->WlanCounters, sizeof(COUNTER_802_11));

}

/*
    ==========================================================================
    Description:
        This routine is executed periodically inside MlmePeriodicExec() after
        association with an AP.
        It checks if PortCfg.Psm is consistent with user policy (recorded in
        PortCfg.WindowsPowerMode). If not, enforce user policy. However,
        there're some conditions to consider:
        1. we don't support power-saving in ADHOC mode, so Psm=PWR_ACTIVE all
           the time when Mibss==TRUE
        2. When Massoc==TRUE (INFRA mode), Psm should not be switch to PWR_SAVE
           if outgoing traffic available in TxRing or PrioRing.
    Output:
        1. change pAd->PortCfg.Psm to PWR_SAVE or leave it untouched
    ==========================================================================
 */
VOID MlmeCheckForPsmChange(
    IN PRTMP_ADAPTER pAd,
    IN ULONG    Now32)
{
	ULONG	PowerMode;
    // condition -
    // 1. Psm maybe ON only happen in INFRASTRUCTURE mode
    // 2. user wants either MAX_PSP or FAST_PSP
    // 3. but current psm is not in PWR_SAVE
    // 4. CNTL state machine is not doing SCANning
    // 5. no TX SUCCESS event for the past period
#ifdef NDIS51_MINIPORT
	if (pAd->PortCfg.WindowsPowerProfile == NdisPowerProfileBattery)
    	PowerMode = pAd->PortCfg.WindowsBatteryPowerMode;
    else
#endif
    	PowerMode = pAd->PortCfg.WindowsPowerMode;

    if (INFRA_ON(pAd) &&
        (PowerMode != Ndis802_11PowerModeCAM) &&
        (pAd->PortCfg.Psm == PWR_ACTIVE) &&
        (pAd->Mlme.CntlMachine.CurrState == CNTL_IDLE) &&
        (pAd->WlanCounters.TransmittedFragmentCount == pAd->Mlme.PrevTxCnt))
    {
        MlmeSetPsmBit(pAd, PWR_SAVE);
//        EnqueueNullFrame(pAd);
    }

    // latch current count for next-time comparison
    pAd->Mlme.PrevTxCnt = pAd->WlanCounters.TransmittedFragmentCount;
}

VOID MlmeSetPsmBit(
    IN PRTMP_ADAPTER pAd,
    IN USHORT psm)
{
    TXCSR7_STRUC txcsr7;

    txcsr7.word = 0;
    pAd->PortCfg.Psm = psm;

    DBGPRINT(RT_DEBUG_TRACE, "MMCHK - change PSM bit to %d <<<\n", psm);
    if (psm == PWR_SAVE)
    {
        txcsr7.field.ARPowerManage = 1;
        RTMP_IO_WRITE32(pAd, TXCSR7, txcsr7.word);
    }
    else
    {
        txcsr7.field.ARPowerManage = 0;
        RTMP_IO_WRITE32(pAd, TXCSR7, txcsr7.word);
    }
}

VOID MlmeSetTxPreamble(
    IN PRTMP_ADAPTER pAd,
    IN USHORT TxPreamble)
{
    ULONG Arcsr2 = 0x00700400;     // 0x13c, ACK/CTS PLCP at 1 Mbps
    ULONG Arcsr3 = 0x00380401;     // 0x140, ACK/CTS PLCP at 2 Mbps
    ULONG Arcsr4 = 0x00150402;     // 0x144, ACK/CTS PLCP at 5.5 Mbps
    ULONG Arcsr5 = 0x000b8403;     // 0x148, ACK/CTS PLCP at 11 Mbps

    if (TxPreamble == Rt802_11PreambleShort)
    {
        DBGPRINT(RT_DEBUG_TRACE, "---> set to SHORT PREAMBLE\n");
        Arcsr2 |= 0x00000008;
        Arcsr3 |= 0x00000008;
        Arcsr4 |= 0x00000008;
        Arcsr5 |= 0x00000008;
        pAd->PortCfg.TxPreamble = Rt802_11PreambleShort;
        pAd->PortCfg.Dpreamble = 72;
        pAd->PortCfg.Dplcp = 24;
    }
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, "---> set to LONG PREAMBLE\n");
        pAd->PortCfg.TxPreamble = Rt802_11PreambleLong;
        pAd->PortCfg.Dpreamble = 144;
        pAd->PortCfg.Dplcp = 48;
    }

    RTMP_IO_WRITE32(pAd, ARCSR2, Arcsr2);
    RTMP_IO_WRITE32(pAd, ARCSR3, Arcsr3);
    RTMP_IO_WRITE32(pAd, ARCSR4, Arcsr4);
    RTMP_IO_WRITE32(pAd, ARCSR5, Arcsr5);
}

VOID MlmeUpdateTxRates(
    IN PRTMP_ADAPTER pAd)
{
    int i;
    UCHAR Rate, Desire = RATE_1, Support = RATE_1, MaxBasicRate = RATE_1;

    // find max desired rate
    for (i=0; i<MAX_LEN_OF_SUPPORTED_RATES; i++)
    {
        switch (pAd->PortCfg.DesiredRates[i] & 0x7f)
        {
            case 2:  Rate = RATE_1;   break;
            case 4:  Rate = RATE_2;   break;
            case 11: Rate = RATE_5_5; break;
            case 22: Rate = RATE_11;  break;
            default: Rate = RATE_1;   break;
        }
        if (Desire < Rate)
            Desire = Rate;
    }

    // find max supported rate
    for (i=0; i<pAd->PortCfg.SupportedRatesLen; i++)
    {
        switch (pAd->PortCfg.SupportedRates[i] & 0x7f)
        {
            case 2:  Rate = RATE_1;   break;
            case 4:  Rate = RATE_2;   break;
            case 11: Rate = RATE_5_5; break;
            case 22: Rate = RATE_11;  break;
            default: Rate = RATE_1;   break;
        }
        if (Support < Rate)
            Support = Rate;
        if ((pAd->PortCfg.SupportedRates[i] & 0x80) && (MaxBasicRate < Rate))
            MaxBasicRate = Rate;
    }

    // max tx rate = min {max desire rate, max supported rate}
    if (Support < Desire)
        Rate = Support;
    else
        Rate = Desire;

	if(pAd->PortCfg.TxRateFixed){
		pAd->PortCfg.TxRate = pAd->PortCfg.MaxTxRate;
		pAd->PortCfg.MaxBasicRate = pAd->PortCfg.MaxTxRate;
	} else {
		pAd->PortCfg.MaxTxRate = Rate;
		pAd->PortCfg.TxRate = Rate;
		pAd->PortCfg.MaxBasicRate = MaxBasicRate;
	}

    switch (pAd->PortCfg.MaxBasicRate)
    {
    	case RATE_1:
    		RTMP_IO_WRITE32(pAd, TXCSR1, 0x016275ff);	// ACK as 1Mb time
    		RTMP_IO_WRITE32(pAd, ARCSR1, 0x00000001);   // basic rate set {1} Mbps
			break;

    	case RATE_2:
    		RTMP_IO_WRITE32(pAd, TXCSR1, 0x016205ff);	// ACK as 2Mb time
    		RTMP_IO_WRITE32(pAd, ARCSR1, 0x00000003);   // basic rate set {1, 2} Mbps
			break;

    	case RATE_5_5:
    		RTMP_IO_WRITE32(pAd, TXCSR1, 0x0161bfff);	// ACK as 5.5Mb time
    		RTMP_IO_WRITE32(pAd, ARCSR1, 0x00000007);   // basic rate set {1, 2, 5.5} Mbps
			break;

    	case RATE_11:
    		RTMP_IO_WRITE32(pAd, TXCSR1, 0x0161abff);	// ACK as 11Mb time
    		RTMP_IO_WRITE32(pAd, ARCSR1, 0x0000000f);   // basic rate set {1, 2, 5.5, 11} Mbps
			break;
    }

    DBGPRINT(RT_DEBUG_TRACE, "MMCHK - update (MaxTxRate=%d, CurrTxRate=%d, MaxBasicRate=%d)\n",
        Rate, Rate, MaxBasicRate);
}

VOID MlmeRadioOff(
    IN PRTMP_ADAPTER pAd)
{
	// Set Radio off flag
	RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF);

	// Link down first if any association exists
	if (INFRA_ON(pAd) || ADHOC_ON(pAd))
		LinkDown(pAd);

	// Abort Tx
	RTMP_IO_WRITE32(pAd, TXCSR0, 0x08);
	// Disable Rx
	RTMP_IO_WRITE32(pAd, RXCSR0, 0x01);
	// Turn off radio
	RTMP_IO_WRITE32(pAd, PWRCSR0, 0x00000000);

	if (pAd->PortCfg.LedMode == LED_MODE_ASUS)
	{
		ASIC_LED_ACT_OFF(pAd);
	}
}

VOID MlmeRadioOn(
    IN PRTMP_ADAPTER pAd)
{
	// Turn on radio
	RTMP_IO_WRITE32(pAd, PWRCSR0, 0x3f3b3100);

	// Abort Tx
	RTMP_IO_WRITE32(pAd, TXCSR0, 0x08);
	// Disable Rx
	RTMP_IO_WRITE32(pAd, RXCSR0, 0x01);

	RTMPRingCleanUp(pAd, TX_RING);
	RTMPRingCleanUp(pAd, PRIO_RING);
	RTMPRingCleanUp(pAd, RX_RING);

	NICResetFromError(pAd);
	// Clear Radio off flag
	RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF);

	if (pAd->PortCfg.LedMode == LED_MODE_ASUS)
	{
		RTMP_IO_WRITE32(pAd, LEDCSR, 0x0002461E);
	}
}

// ===========================================================================================
// bss_table.c
// ===========================================================================================


/*! \brief initialize BSS table
 *  \param p_tab pointer to the table
 *  \return none
 *  \pre
 *  \post
 */
VOID BssTableInit(
    IN BSS_TABLE *Tab)
{
    int i;

    Tab->BssNr = 0;
    for (i = 0; i < MAX_LEN_OF_BSS_TABLE; i++)
    {
        memset(&Tab->BssEntry[i], 0, sizeof(BSS_ENTRY));
    }
}

/*! \brief search the BSS table by SSID
 *  \param p_tab pointer to the bss table
 *  \param ssid SSID string
 *  \return index of the table, BSS_NOT_FOUND if not in the table
 *  \pre
 *  \post
 *  \note search by sequential search
 */
ULONG BssTableSearch(
    IN BSS_TABLE *Tab,
    IN PMACADDR Bssid)
{
    UCHAR i;

    for (i = 0; i < Tab->BssNr; i++)
    {
        //printf("comparing %s and %s\n", p_tab->bss[i].ssid, ssid);
        if (MAC_ADDR_EQUAL(&(Tab->BssEntry[i].Bssid), Bssid))
        {
            return i;
        }
    }
    return (ULONG)BSS_NOT_FOUND;
}

VOID BssTableDeleteEntry(
    IN OUT	BSS_TABLE *Tab,
    IN		PMACADDR Bssid)
{
    UCHAR i, j;

    for (i = 0; i < Tab->BssNr; i++)
    {
        //printf("comparing %s and %s\n", p_tab->bss[i].ssid, ssid);
        if (MAC_ADDR_EQUAL(&(Tab->BssEntry[i].Bssid), Bssid))
        {
        	for (j = i; j < Tab->BssNr - 1; j++)
        	{
        		memcpy(&(Tab->BssEntry[j]), &(Tab->BssEntry[j + 1]), sizeof(BSS_ENTRY));
        	}
	        Tab->BssNr -= 1;
            return;
        }
    }
}


UCHAR	ZeroSsid[32] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

/*! \brief
 *  \param
 *  \return
 *  \pre
 *  \post
 */
VOID BssEntrySet(
    IN	PRTMP_ADAPTER	pAd,
    OUT BSS_ENTRY *pBss,
    IN MACADDR *pBssid,
    IN CHAR Ssid[],
    IN UCHAR SsidLen,
    IN UCHAR BssType,
    IN USHORT BeaconPeriod,
    IN BOOLEAN CfExist,
    IN CF_PARM *pCfParm,
    IN USHORT AtimWin,
    IN USHORT CapabilityInfo,
    IN UCHAR Rates[],
    IN UCHAR RatesLen,
    IN UCHAR Channel,
    IN UCHAR Rssi)
{
    COPY_MAC_ADDR(&pBss->Bssid, pBssid);
	if (SsidLen > 0)
	{
		// For hidden SSID AP, it might send beacon with SSID len equal to 0
		// Or send beacon /probe response with SSID len matching real SSID length,
		// but SSID is all zero. such as "00-00-00-00" with length 4.
		// We have to prevent this case overwrite correct table
		if (RTMPEqualMemory(Ssid, ZeroSsid, SsidLen) == 0)
		{
    memcpy(pBss->Ssid, Ssid, SsidLen);
    pBss->SsidLen = SsidLen;
		}
	}
    pBss->BssType = BssType;
    pBss->BeaconPeriod = BeaconPeriod;
    if (BssType == BSS_INFRA)
    {
        if (CfExist)
        {
            pBss->CfpCount = pCfParm->CfpCount;
            pBss->CfpPeriod = pCfParm->CfpPeriod;
            pBss->CfpMaxDuration = pCfParm->CfpMaxDuration;
            pBss->CfpDurRemaining = pCfParm->CfpDurRemaining;
        }
    }
    else
    {
        pBss->AtimWin = AtimWin;
    }

    pBss->CapabilityInfo = CapabilityInfo;
    pBss->Privacy = CAP_IS_PRIVACY_ON(pBss->CapabilityInfo);
    memcpy(pBss->Rates, Rates, RatesLen);
    pBss->RatesLen = RatesLen;
    pBss->Channel = Channel;
    pBss->Rssi = Rssi;
}

/*!
 *  \brief insert an entry into the bss table
 *  \param p_tab The BSS table
 *  \param Bssid BSSID
 *  \param ssid SSID
 *  \param ssid_len Length of SSID
 *  \param bss_type
 *  \param beacon_period
 *  \param timestamp
 *  \param p_cf
 *  \param atim_win
 *  \param cap
 *  \param rates
 *  \param rates_len
 *  \param channel_idx
 *  \return none
 *  \pre
 *  \post
 *  \note If SSID is identical, the old entry will be replaced by the new one
 */
ULONG BssTableSetEntry(
    IN	PRTMP_ADAPTER	pAd,
    OUT BSS_TABLE *Tab,
    IN MACADDR *Bssid,
    IN CHAR Ssid[],
    IN UCHAR SsidLen,
    IN UCHAR BssType,
    IN USHORT BeaconPeriod,
    IN BOOLEAN CfExist,
    IN CF_PARM *CfParm,
    IN USHORT AtimWin,
    IN USHORT CapabilityInfo,
    IN UCHAR Rates[],
    IN UCHAR RatesLen,
    IN UCHAR ChannelNo,
    IN UCHAR Rssi)
{
    ULONG   Idx;

    Idx = BssTableSearch(Tab, Bssid);
    if (Idx == BSS_NOT_FOUND)
    {
        if (Tab->BssNr >= MAX_LEN_OF_BSS_TABLE)
            return BSS_NOT_FOUND;
        Idx = Tab->BssNr;
        BssEntrySet(pAd, &Tab->BssEntry[Idx], Bssid, Ssid, SsidLen, BssType, BeaconPeriod,
                    CfExist, CfParm, AtimWin, CapabilityInfo, Rates, RatesLen, ChannelNo, Rssi);
        Tab->BssNr++;
    }
    else
    {
        BssEntrySet(pAd, &Tab->BssEntry[Idx], Bssid, Ssid, SsidLen, BssType, BeaconPeriod,
                    CfExist, CfParm, AtimWin, CapabilityInfo, Rates, RatesLen, ChannelNo, Rssi);
    }

    return Idx;
}

VOID BssTableSsidSort(
    IN  BSS_TABLE *InTab,
    OUT BSS_TABLE *OutTab,
    IN  CHAR Ssid[],
    IN  UCHAR SsidLen,
    IN  UCHAR BssType,
    IN  UCHAR Privacy)
{
    INT i;
    BssTableInit(OutTab);

    for (i = 0; i < InTab->BssNr; i++)
    {
        BSS_ENTRY *pInBss = &InTab->BssEntry[i];

        if ((pInBss->BssType == BssType) && ((pInBss->SsidLen==SsidLen) && RTMPEqualMemory(pInBss->Ssid, Ssid, (ULONG) SsidLen)))
        {
            BSS_ENTRY *pOutBss = &OutTab->BssEntry[OutTab->BssNr];

            // copy matching BSS from InTab to OutTab
            memcpy(pOutBss, pInBss, sizeof(BSS_ENTRY));

            OutTab->BssNr++;
        }
        else if ((pInBss->BssType == BssType) && (SsidLen == 0) && (pInBss->Privacy == Privacy))
        {
            BSS_ENTRY *pOutBss = &OutTab->BssEntry[OutTab->BssNr];

            // copy matching BSS from InTab to OutTab
            memcpy(pOutBss, pInBss, sizeof(BSS_ENTRY));

            OutTab->BssNr++;
        }
		else if ((pInBss->BssType == BssType) && (pInBss->SsidLen == 0))
		{
			// Add for hidden SSID
			BSS_ENTRY *pOutBss = &OutTab->BssEntry[OutTab->BssNr];

			// copy matching BSS from InTab to OutTab
			memcpy(pOutBss, pInBss, sizeof(BSS_ENTRY));

			OutTab->BssNr++;
		}

		if (OutTab->BssNr >= MAX_LEN_OF_BSS_TABLE)
			break;

    }

    BssTableSortByRssi(OutTab);
}


VOID BssTableSortByRssi(
    IN OUT BSS_TABLE *OutTab)
{
    INT       i, j;
    BSS_ENTRY TmpBss;

    for (i = 0; i < OutTab->BssNr - 1; i++)
    {
        for (j = i+1; j < OutTab->BssNr; j++)
        {
            if (OutTab->BssEntry[j].Rssi > OutTab->BssEntry[i].Rssi)
            {
                memcpy(&TmpBss, &OutTab->BssEntry[j], sizeof(BSS_ENTRY));
                memcpy(&OutTab->BssEntry[j], &OutTab->BssEntry[i], sizeof(BSS_ENTRY));
                memcpy(&OutTab->BssEntry[i], &TmpBss, sizeof(BSS_ENTRY));
            }
        }
    }
}

// ===========================================================================================
// mac_table.c
// ===========================================================================================

/*! \brief generates a random mac address value for IBSS BSSID
 *  \param Addr the bssid location
 *  \return none
 *  \pre
 *  \post
 */
VOID MacAddrRandomBssid(
    IN PRTMP_ADAPTER pAd,
    OUT MACADDR *Addr)
{
    INT i;

    for (i = 0; i < MAC_ADDR_LEN; i++)
    {
        Addr->Octet[i] = RandomByte(pAd);
    }

    Addr->Octet[0] = (Addr->Octet[0] & 0xfe) | 0x02;  // the first 2 bits must be 01xxxxxxxx
}

/*! \brief init the management mac frame header
 *  \param p_hdr mac header
 *  \param subtype subtype of the frame
 *  \param p_ds destination address, don't care if it is a broadcast address
 *  \return none
 *  \pre the station has the following information in the pAd->PortCfg
 *   - bssid
 *   - station address
 *  \post
 *  \note this function initializes the following field
 */
VOID MgtMacHeaderInit(
    IN	PRTMP_ADAPTER	pAd,
    IN OUT PMACHDR Hdr,
    IN UCHAR Subtype,
    IN UCHAR ToDs,
    IN PMACADDR Ds,
    IN PMACADDR Bssid)
{
    memset(Hdr, 0, sizeof(MACHDR));
    Hdr->Type = BTYPE_MGMT;
    Hdr->SubType = Subtype;
    Hdr->Tods = ToDs;
    COPY_MAC_ADDR(&Hdr->Addr1, Ds);
    COPY_MAC_ADDR(&Hdr->Addr2, &pAd->CurrentAddress);
    COPY_MAC_ADDR(&Hdr->Addr3, Bssid);
}

// ===========================================================================================
// mem_mgmt.c
// ===========================================================================================

/*!***************************************************************************
 * This routine build an outgoing frame, and fill all information specified
 * in argument list to the frame body. The actual frame size is the summation
 * of all arguments.
 * input params:
 *      Buffer - pointer to a pre-allocated memory segment
 *      args - a list of <int arg_size, arg> pairs.
 *      NOTE NOTE NOTE!!!! the last argument must be NULL, otherwise this
 *                         function will FAIL!!!
 * return:
 *      Size of the buffer
 * usage:
 *      MakeOutgoingFrame(Buffer, 2, &fc, 2, &dur, 6, p_addr1, 6,p_addr2, NULL);
 ****************************************************************************/
ULONG MakeOutgoingFrame(
    OUT CHAR *Buffer,
    OUT ULONG *FrameLen, ...)
{
    CHAR   *p;
    int     leng;
    ULONG   TotLeng;
    va_list Args;

    // calculates the total length
    TotLeng = 0;
    va_start(Args, FrameLen);
    do
    {
        leng = va_arg(Args, int);
        if (leng == 0)
        {
            break;
        }
        p = va_arg(Args, PVOID);
        memcpy(&Buffer[TotLeng], p, leng);
        TotLeng = TotLeng + leng;
    } while(TRUE);

    va_end(Args); /* clean up */
    *FrameLen = TotLeng;
    return TotLeng;
}

// ===========================================================================================
// mlme_queue.c
// ===========================================================================================

/*! \brief  Initialize The MLME Queue, used by MLME Functions
 *  \param  *Queue     The MLME Queue
 *  \return Always     Return NDIS_STATE_SUCCESS in this implementation
 *  \pre
 *  \post
 *  \note   Because this is done only once (at the init stage), no need to be locked
 */
NDIS_STATUS MlmeQueueInit(
    IN MLME_QUEUE *Queue)
{
    INT i;

    spin_lock_init(&Queue->Lock);

    Queue->Num  = 0;
    Queue->Head = 0;
    Queue->Tail = 0;

    for (i = 0; i < MAX_LEN_OF_MLME_QUEUE; i++)
    {
        Queue->Entry[i].Occupied = FALSE;
        Queue->Entry[i].MsgLen = 0;
        memset(Queue->Entry[i].Msg, 0, MAX_LEN_OF_MLME_QUEUE_MSG);
    }

    return NDIS_STATUS_SUCCESS;
}


/*! \brief   Enqueue a message for other threads, if they want to send messages to MLME thread
 *  \param  *Queue    The MLME Queue
 *  \param   Machine  The State Machine Id
 *  \param   MsgType  The Message Type
 *  \param   MsgLen   The Message length
 *  \param  *Msg      The message pointer
 *  \return  TRUE if enqueue is successful, FALSE if the queue is full
 *  \pre
 *  \post
 *  \note    The message has to be initialized
 */
BOOLEAN MlmeEnqueue(
    OUT MLME_QUEUE *Queue,
    IN ULONG Machine,
    IN ULONG MsgType,
    IN ULONG MsgLen,
    IN VOID *Msg)
{
    INT Tail;
    unsigned long flags;

    spin_lock_irqsave(&Queue->Lock, flags);
    if (Queue->Num == MAX_LEN_OF_MLME_QUEUE) {
		spin_unlock_irqrestore(&Queue->Lock, flags);
        KPRINT(KERN_ERR, "MlmeEnqueue full, msg dropped and may corrupt MLME\n");
        return FALSE;
    }
	// If another context preempts us, it uses the next element - sic. bb
    Tail = Queue->Tail++;
    Queue->Tail %= MAX_LEN_OF_MLME_QUEUE;
    Queue->Num++;

	// We guard against Ben Hutchings' incomplete queue element problem by not
	// setting the Occupied flag until the memcpy is done. The ocurrence of a
	// refresh cycle during a copy can stretch the time by up to 100 usec
	// (well, quite a few usec, anyway); not good when interrupts are disabled.
	// Note that this can leave a bubble in the queue, but it will have
	// disappeared by the time this thread gets around to calling MlmeHandler.
	// All items will be handled in their proper order, but possibly not in the
	// context in which they were added. - bb
    spin_unlock_irqrestore(&Queue->Lock, flags);
    DBGPRINT(RT_DEBUG_INFO, "MlmeEnqueue, num=%d\n",Queue->Num);

    Queue->Entry[Tail].Machine = Machine;
    Queue->Entry[Tail].MsgType = MsgType;
    Queue->Entry[Tail].MsgLen  = MsgLen;
	memcpy(Queue->Entry[Tail].Msg, Msg, MsgLen);

	//MlmeHandler will stop when it finds this false.
    smp_wmb();
    Queue->Entry[Tail].Occupied = TRUE;

    return TRUE;
}

/*! \brief   This function is used when Recv gets a MLME message
 *  \param  *Queue           The MLME Queue
 *  \param   TimeStampHigh   The upper 32 bit of timestamp
 *  \param   TimeStampLow    The lower 32 bit of timestamp
 *  \param   Rssi            The receiving RSSI strength
 *  \param   MsgLen          The length of the message
 *  \param  *Msg             The message pointer
 *  \return  TRUE if everything ok, FALSE otherwise (like Queue Full)
 *  \pre
 *  \post
 */
BOOLEAN MlmeEnqueueForRecv(
    IN	PRTMP_ADAPTER	pAd,
    OUT MLME_QUEUE *Queue,
    IN ULONG TimeStampHigh,
    IN ULONG TimeStampLow,
    IN UCHAR Rssi,
    IN ULONG MsgLen,
    IN VOID *Msg)
{
    INT          Tail, Machine;
    MACFRAME    *Fr = (MACFRAME *)Msg;
    ULONG        MsgType;
    unsigned long flags;

    if ((Fr->Hdr.Type != BTYPE_MGMT) || (!MsgTypeSubst(Fr, &Machine, &MsgType)))
    {
        KPRINT(KERN_ERR, "MlmeEnqueueForRecv (MsgType error)\n");
        return FALSE;
    }

    spin_lock_irqsave(&Queue->Lock, flags);
    if (Queue->Num == MAX_LEN_OF_MLME_QUEUE) {
		spin_unlock_irqrestore(&Queue->Lock, flags);
        KPRINT(KERN_ERR, "MlmeEnqueueForRecv (queue full error)\n");
        return FALSE;
    }
    Tail = Queue->Tail++;
    Queue->Tail %= MAX_LEN_OF_MLME_QUEUE;
    Queue->Num++;
    spin_unlock_irqrestore(&Queue->Lock, flags);
    DBGPRINT(RT_DEBUG_INFO, "MlmeEnqueueForRecv, num=%d\n",Queue->Num);

    // OK, we got all the informations, it is time to put things into queue
	// See MlmeEnqueue note for use of Occupied flag.
    Queue->Entry[Tail].Machine = Machine;
    Queue->Entry[Tail].MsgType = MsgType;
    Queue->Entry[Tail].MsgLen  = MsgLen;
//    Queue->Entry[Tail].TimeStamp.LowPart = TimeStampLow;
//    Queue->Entry[Tail].TimeStamp.HighPart = TimeStampHigh;
    Queue->Entry[Tail].TimeStamp = jiffies;
    Queue->Entry[Tail].Rssi = Rssi;
	memcpy(Queue->Entry[Tail].Msg, Msg, MsgLen);
    smp_wmb();
    Queue->Entry[Tail].Occupied = TRUE;

    MlmeHandler(pAd);

    return TRUE;
}

/*! \brief   Get the first message from the MLME Queue
 * 			WARNING: Must be call with Mlme.Queue.Lock held
 *  \param  *Queue    The MLME Queue
 *  \param  *Elem     The message dequeued from MLME Queue
 *  \return  TRUE if the Elem contains something, FALSE otherwise
 *  \pre
 *  \post
 */
BOOLEAN MlmeGetHead(
    IN MLME_QUEUE *Queue,
    OUT MLME_QUEUE_ELEM **Elem)
{
    if (Queue->Num == 0)
	    return FALSE;
    *Elem = &Queue->Entry[Queue->Head];
    return TRUE;
}

/*! \brief   Remove the first message from the MLME Queue
 * 			WARNING: Must be call with Mlme.Queue.Lock held
 *  \param  *Queue    The MLME Queue
 *  \return  TRUE if a message was removed, FALSE if the queue was empty
 *  \pre
 *  \post
 */
BOOLEAN MlmeDequeue(
    IN MLME_QUEUE *Queue)
{
    if (Queue->Num == 0)
	    return FALSE;
    Queue->Head = (Queue->Head + 1) % MAX_LEN_OF_MLME_QUEUE;
    Queue->Num--;
    DBGPRINT(RT_DEBUG_INFO, "MlmeDequeue, num=%d\n",Queue->Num);

    return TRUE;
}

/*! \brief   The destructor of MLME Queue
 *  \param
 *  \return
 *  \pre
 *  \post
 *  \note   Clear Mlme Queue, Set Queue->Num to Zero.
 */
VOID MlmeQueueDestroy(
    IN MLME_QUEUE *Queue)
{
    unsigned long flags;
    spin_lock_irqsave(&(Queue->Lock), flags);
    Queue->Num  = 0;
    Queue->Head = 0;
    Queue->Tail = 0;
    spin_unlock_irqrestore(&(Queue->Lock), flags);
}

/*! \brief   To substitute the message type if the message is coming from external
 *  \param  *Fr            The frame received
 *  \param  *Machine       The state machine
 *  \param  *MsgType       the message type for the state machine
 *  \return TRUE if the substitution is successful, FALSE otherwise
 *  \pre
 *  \post
 */
BOOLEAN MsgTypeSubst(
    IN MACFRAME *Fr,
    OUT INT *Machine,
    OUT INT *MsgType)
{
    USHORT Seq;

    switch (Fr->Hdr.SubType)
    {
        case SUBTYPE_ASSOC_REQ:
            *Machine = ASSOC_STATE_MACHINE;
            *MsgType = MT2_PEER_ASSOC_REQ;
            break;
        case SUBTYPE_ASSOC_RSP:
            *Machine = ASSOC_STATE_MACHINE;
            *MsgType = MT2_PEER_ASSOC_RSP;
            break;
        case SUBTYPE_REASSOC_REQ:
            *Machine = ASSOC_STATE_MACHINE;
            *MsgType = MT2_PEER_REASSOC_REQ;
            break;
        case SUBTYPE_REASSOC_RSP:
            *Machine = ASSOC_STATE_MACHINE;
            *MsgType = MT2_PEER_REASSOC_RSP;
            break;
        case SUBTYPE_PROBE_REQ:
            *Machine = SYNC_STATE_MACHINE;
            *MsgType = MT2_PEER_PROBE_REQ;
            break;
        case SUBTYPE_PROBE_RSP:
            *Machine = SYNC_STATE_MACHINE;
            *MsgType = MT2_PEER_PROBE_RSP;
            break;
        case SUBTYPE_BEACON:
            *Machine = SYNC_STATE_MACHINE;
            *MsgType = MT2_PEER_BEACON;
            break;
        case SUBTYPE_ATIM:
            *Machine = SYNC_STATE_MACHINE;
            *MsgType = MT2_PEER_ATIM;
            break;
        case SUBTYPE_DISASSOC:
            *Machine = ASSOC_STATE_MACHINE;
            *MsgType = MT2_PEER_DISASSOC_REQ;
            break;
        case SUBTYPE_AUTH:
            // get the sequence number from payload 24 Mac Header + 2 bytes algorithm
            memcpy(&Seq, &Fr->Octet[2], sizeof(USHORT));
            if (Seq == 1 || Seq == 3)
            {
                *Machine = AUTH_RSP_STATE_MACHINE;
                *MsgType = MT2_PEER_AUTH_ODD;
            }
            else if (Seq == 2 || Seq == 4)
            {
                *Machine = AUTH_STATE_MACHINE;
                *MsgType = MT2_PEER_AUTH_EVEN;
            }
            else
            {
                return FALSE;
            }
            break;
        case SUBTYPE_DEAUTH:
            *Machine = AUTH_RSP_STATE_MACHINE;
            *MsgType = MT2_PEER_DEAUTH;
            break;
        default:
            return FALSE;
            break;
    }

    return TRUE;
}

// ===========================================================================================
// state_machine.c
// ===========================================================================================

/*! \brief Initialize the state machine.
 *  \param *S           pointer to the state machine
 *  \param  Trans       State machine transition function
 *  \param  StNr        number of states
 *  \param  MsgNr       number of messages
 *  \param  DefFunc     default function, when there is invalid state/message combination
 *  \param  InitState   initial state of the state machine
 *  \param  Base        StateMachine base, internal use only
 *  \pre p_sm should be a legal pointer
 *  \post
 */

VOID StateMachineInit(
    IN STATE_MACHINE *S,
    IN STATE_MACHINE_FUNC Trans[],
    IN ULONG StNr,
    IN ULONG MsgNr,
    IN STATE_MACHINE_FUNC DefFunc,
    IN ULONG InitState,
    IN ULONG Base)
{
    ULONG i, j;

    // set number of states and messages
    S->NrState = StNr;
    S->NrMsg   = MsgNr;
    S->Base    = Base;

    S->TransFunc  = Trans;

    // init all state transition to default function
    for (i = 0; i < StNr; i++)
    {
        for (j = 0; j < MsgNr; j++)
        {
            S->TransFunc[i * MsgNr + j] = DefFunc;
        }
    }

    // set the starting state
    S->CurrState = InitState;

}

/*! \brief This function fills in the function pointer into the cell in the state machine
 *  \param *S   pointer to the state machine
 *  \param St   state
 *  \param Msg  incoming message
 *  \param f    the function to be executed when (state, message) combination occurs at the state machine
 *  \pre *S should be a legal pointer to the state machine, st, msg, should be all within the range, Base should be set in the initial state
 *  \post
 */
VOID StateMachineSetAction(
    IN STATE_MACHINE *S,
    IN ULONG St,
    IN ULONG Msg,
    IN STATE_MACHINE_FUNC Func)
{
    ULONG MsgIdx;

    MsgIdx = Msg - S->Base;

    if (St < S->NrState && MsgIdx < S->NrMsg)
    {
        // boundary checking before setting the action
        S->TransFunc[St * S->NrMsg + MsgIdx] = Func;
    }
}

/*! \brief   The destructor of the state machine
 *  \param  *S the statemachine
 *  \note    doing nothing at this moment, may need to do something if the implementation changed
 */
VOID
StateMachineDestroy(IN STATE_MACHINE *S)
{
}

/*! \brief   This function does the state transition
 *  \param   *Adapter the NIC adapter pointer
 *  \param   *S       the state machine
 *  \param   *Elem    the message to be executed
 *  \return   None
 */
VOID StateMachinePerformAction(
    IN	PRTMP_ADAPTER	pAd,
    IN STATE_MACHINE *S,
    IN MLME_QUEUE_ELEM *Elem)
{
    (*(S->TransFunc[S->CurrState * S->NrMsg + Elem->MsgType - S->Base]))(pAd, Elem);
}

/*
    ==========================================================================
    Description:
        The drop function, when machine executes this, the message is simply
        ignored. This function does nothing, the message is freed in
        StateMachinePerformAction()
    ==========================================================================
 */
VOID Drop(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
#if 0
    if ((Elem->MsgType == MT2_PEER_BEACON) ||
        (Elem->MsgType == MT2_PEER_PROBE_REQ) ||
        (Elem->MsgType == MT2_PEER_PROBE_RSP))
        ;
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, ("Warn:>>Drop Msg=%d<<\n",Elem->MsgType));
    }
#endif
}

// ===========================================================================================
// lfsr.c
// ===========================================================================================

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID LfsrInit(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Seed)
{
    if (Seed == 0)
        pAd->Mlme.ShiftReg = 1;
    else
        pAd->Mlme.ShiftReg = Seed;
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
UCHAR RandomByte(
    IN PRTMP_ADAPTER pAd)
{
    ULONG i;
    UCHAR R, Result;

    R = 0;

    for (i = 0; i < 8; i++)
    {
        if (pAd->Mlme.ShiftReg & 0x00000001)
        {
            pAd->Mlme.ShiftReg = ((pAd->Mlme.ShiftReg ^ LFSR_MASK) >> 1) | 0x80000000;
            Result = 1;
        }
        else
        {
            pAd->Mlme.ShiftReg = pAd->Mlme.ShiftReg >> 1;
            Result = 0;
        }
        R = (R << 1) | Result;
    }

    return R;
}

// ===========================================================================================
// asic.c
// ===========================================================================================

//
// RF register initialization set, support RT2420/RT2421
//
ULONG	RF2420RegTable[] = {
		0x94022058, 0x940c203e, 0x94000111, // chanel 6
		};

ULONG	RF2421RegTable[] = {
		0x94022058, 0x940c203e, 0x94000101, // chanel 6
		};

ULONG   RF2421RegTableAt500KRefClk[] = {
		0x940220b0, 0x940c407a, 0x94000101, // chanel 6
        };

ULONG   RF2421RegTableAtInvalidRefClk[] = {
        0x94022058, 0x940c2a32, 0x94000101
        };

// Ralink RF channel value, support RT2420/RT2421
ULONG	RFChannelValue[] = {
	0x940c1fda, 0x940c1fee, 0x940c2002, 0x940c2016, 0x940c202a, 0x940c203e, 0x940c2052,
	0x940c2066, 0x940c207a, 0x940c208e, 0x940c20a2, 0x940c20b6, 0x940c20ca, 0x940c20fa,
};

// RF channel value for RT2421B using 500Khz reference clock
ULONG   RFChannelValueAt500KRefClk[] = {
	0x940c3fb2, 0x940c3fda, 0x940c4002, 0x940c402a, 0x940c4052, 0x940c407a, 0x940c40a2,
	0x940c40ca, 0x940c40f2, 0x940c411a, 0x940c4142, 0x940c416a, 0x940c4192, 0x940c41f2,
};

VOID SetTxPower(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Channel)
{
	int BBPR3Val;

	BBPR3Val = (int)(pAd->PortCfg.ChannelTxPower[Channel - 1]);

	switch(pAd->PortCfg.CurrentTxPowerLevelIndex) {
		case 1://MinPower
			BBPR3Val += 23;
			break;
		case 2://1*MaxPower/4
			BBPR3Val += 18;
			break;
		case 3://2*MaxPower/4
			BBPR3Val += 9;
			break;
		case 4://3*MaxPower/4
			BBPR3Val += 4;
			break;
		default://MaxPower
			break;
	}

	// select corresponding TX power for this channel
	RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Tx_Power, BBPR3Val);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicSwitchChannel(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Channel)
{
    DBGPRINT(RT_DEBUG_INFO, "AsicSwitchChannel to #%d\n", Channel);

	SetTxPower(pAd, Channel);
	// Initialize RF register to default value
	if (pAd->PortCfg.RfType == RFIC_2421)
	{
	    RTMP_RF_IO_WRITE32(pAd, RF2421RegTable[0]); // program RF R1
        RTMP_RF_IO_WRITE32(pAd, RFChannelValue[Channel - 1]);   // program RF R2
	    RTMP_RF_IO_WRITE32(pAd, RF2421RegTable[2]); // program RF R3
    }
    else
    {
	    RTMP_RF_IO_WRITE32(pAd, RF2420RegTable[0]); // program RF R1
        RTMP_RF_IO_WRITE32(pAd, RFChannelValue[Channel - 1]);   // program RF R2
	    RTMP_RF_IO_WRITE32(pAd, RF2420RegTable[2]); // program RF R3
    }
	}

/*
    ==========================================================================
    Description:
        This function is required for 2421 only, and should not be used during
        site survey. It's only required after NIC decided to stay at a channel
        for a longer period.
        When this function is called, it's always after AsicSwitchChannel().
    ==========================================================================
 */
VOID AsicLockChannel(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Channel)
{
	if (pAd->PortCfg.RfType == RFIC_2421)
    {
        // program RF at some invalid reference clock rate to activate the auto-tuning function
	    RTMP_RF_IO_WRITE32(pAd, RF2421RegTableAtInvalidRefClk[0]); // program RF R1
        RTMP_RF_IO_WRITE32(pAd, RF2421RegTableAtInvalidRefClk[1]); // program RF R2
	    RTMP_RF_IO_WRITE32(pAd, RF2421RegTableAtInvalidRefClk[2]); // program RF R3
	    pAd->PortCfg.RfTuningStep = 1;
	    pAd->PortCfg.RfTuningChannel = (UCHAR)Channel;
        DBGPRINT(RT_DEBUG_TRACE, "1. R1=0x%x, R2=0x%0x, R3=0x%x\n",
            RF2421RegTableAtInvalidRefClk[0],
            RF2421RegTableAtInvalidRefClk[1],
            RF2421RegTableAtInvalidRefClk[2]);

        // perform next step 1-ms later
	mod_timer(&pAd->PortCfg.RfTuningTimer, jiffies + 1);
    }
}

VOID AsicRfTuningExec(
    IN unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;

    if (pAd->PortCfg.RfTuningStep == 1)
    {
	    // re-program RF using 1Mhz reference clock
	    RTMP_RF_IO_WRITE32(pAd, RF2421RegTable[0]); // program RF R1
        RTMP_RF_IO_WRITE32(pAd, RFChannelValue[pAd->PortCfg.RfTuningChannel - 1]);   // program RF R2
	    RTMP_RF_IO_WRITE32(pAd, RF2421RegTable[2]); // program RF R3
	    pAd->PortCfg.RfTuningStep = 2;
        DBGPRINT(RT_DEBUG_TRACE, "2. R1=0x%x, R2=0x%0x, R3=0x%x\n",
            RF2421RegTable[0],
            RFChannelValue[pAd->PortCfg.RfTuningChannel - 1],
            RF2421RegTable[2]);

        // set 1 ms one-shot timer to turn off RF auto tuning
	mod_timer(&pAd->PortCfg.RfTuningTimer, jiffies + 1);
    }
    else if (pAd->PortCfg.RfTuningStep == 2)
    {
	    ULONG R1 = RF2421RegTable[0] & 0xfffdffff; // RF R1.bit17 "tune_en1" OFF
		ULONG R3 = RF2421RegTable[2] & 0xfffffeff; // RF R3.bit8 "tune_en2" OFF

        // disable auto-tuning to local on this channel
		RTMP_RF_IO_WRITE32(pAd, R1);
		RTMP_RF_IO_WRITE32(pAd, R3);
	    pAd->PortCfg.RfTuningStep = 0;

        DBGPRINT(RT_DEBUG_TRACE, "3. R1=0x%x, R3=0x%x\n", R1, R3);
    }
}

/*
    ==========================================================================
    Description:
        put PHY to sleep here, and set next wakeup timer
    ==========================================================================
 */
VOID AsicSleepThenAutoWakeup(
    IN PRTMP_ADAPTER pAd,
    IN USHORT TbttNumToNextWakeUp)
{
    CSR20_STRUC Csr20;
    PWRCSR1_STRUC Pwrcsr1;

    // we have decided to SLEEP, so at least do it for a BEACON period.
    if (TbttNumToNextWakeUp==0)
        TbttNumToNextWakeUp=1;

    // PWRCSR0 remains untouched

    // set CSR20 for next wakeup
    Csr20.word = 0;
    Csr20.field.NumBcnBeforeWakeup = TbttNumToNextWakeUp - 1;
    Csr20.field.DelayAfterBcn = (pAd->PortCfg.BeaconPeriod - 20) << 4; // 20 TU ahead of desired TBTT
    Csr20.field.AutoWake = 1;
    RTMP_IO_WRITE32(pAd, CSR20, Csr20.word);

    // set PWRCSR1 to put PHY into SLEEP state
    Pwrcsr1.word = 0;
    Pwrcsr1.field.PutToSleep = 1;
    Pwrcsr1.field.BbpDesireState = 1; // 01:SLEEP
    Pwrcsr1.field.RfDesireState = 1;  // 01:SLEEP
    RTMP_IO_WRITE32(pAd, PWRCSR1, Pwrcsr1.word);
    pAd->PortCfg.Pss = PWR_SAVE;
}

/*
    ==========================================================================
    Description:
        AsicForceWakeup() is used whenever manual wakeup is required
        AsicForceSleep() should only be used when Massoc==FALSE. When
        Massoc==TRUE, we should use AsicSleepThenAutoWakeup() instead.
    ==========================================================================
 */
VOID AsicForceSleep(
    IN PRTMP_ADAPTER pAd)
{
    PWRCSR1_STRUC Pwrcsr1;

    if (pAd->PortCfg.Pss == PWR_ACTIVE)
    {
        DBGPRINT(RT_DEBUG_TRACE, ">>>AsicForceSleep<<<\n");
        Pwrcsr1.word = 0;
        Pwrcsr1.field.RfDesireState = 1; // 01:SLEEP state
        Pwrcsr1.field.BbpDesireState = 1; // 01:SLEEP state
        Pwrcsr1.field.SetState = 1;
        RTMP_IO_WRITE32(pAd, PWRCSR1, Pwrcsr1.word);
        pAd->PortCfg.Pss = PWR_SAVE;
    }
}

VOID AsicForceWakeup(
    IN PRTMP_ADAPTER pAd)
{
    PWRCSR1_STRUC Pwrcsr1;

    if (pAd->PortCfg.Pss == PWR_SAVE)
    {
        DBGPRINT(RT_DEBUG_TRACE, ">>>AsicForceWakeup<<<\n");
        Pwrcsr1.word = 0;
        Pwrcsr1.field.RfDesireState = 3; // 11:AWAKE state
        Pwrcsr1.field.BbpDesireState = 3; // 11:AWAKE state
        Pwrcsr1.field.SetState = 1;
        RTMP_IO_WRITE32(pAd, PWRCSR1, Pwrcsr1.word);
        pAd->PortCfg.Pss = PWR_ACTIVE;
    }
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicSetBssid(
    IN PRTMP_ADAPTER pAd,
    IN MACADDR *Bssid)
{
    ULONG         Addr4;

    Addr4 = (ULONG)(Bssid->Octet[0]) |
            (ULONG)(Bssid->Octet[1] << 8) |
            (ULONG)(Bssid->Octet[2] << 16) |
            (ULONG)(Bssid->Octet[3] << 24);
    RTMP_IO_WRITE32(pAd, CSR5, Addr4);

    Addr4 = (ULONG)(Bssid->Octet[4]) | (ULONG)(Bssid->Octet[5] << 8);
    RTMP_IO_WRITE32(pAd, CSR6, Addr4);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicDisableSync(
    IN PRTMP_ADAPTER pAd)
{
    DBGPRINT(RT_DEBUG_TRACE, "--->Disable TSF synchronization\n");
    RTMP_IO_WRITE32(pAd, CSR14, 0x00000000);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicEnableBssSync(
    IN PRTMP_ADAPTER pAd)
{
    CSR12_STRUC Csr12;
    CSR13_STRUC Csr13;
    CSR14_STRUC Csr14;
    BCNCSR1_STRUC Bcncsr1;
    BOOLEAN IsApPc;

    DBGPRINT(RT_DEBUG_TRACE, "--->AsicEnableBssSync(INFRA mode)\n");

    RTMP_IO_WRITE32(pAd, CSR14, 0x00000000);

    Csr12.word = 0;
    Csr12.field.BeaconInterval = pAd->PortCfg.BeaconPeriod << 4; // ASIC register in units of 1/16 TU
    Csr12.field.CfpMaxDuration = pAd->PortCfg.CfpMaxDuration << 4; // ASIC register in units of 1/16 TU
    RTMP_IO_WRITE32(pAd, CSR12, Csr12.word);

    Csr13.word = 0;
    Csr13.field.CfpPeriod = pAd->PortCfg.CfpDurRemain << 4; // ASIC register in units of 1/16 TU
    RTMP_IO_WRITE32(pAd, CSR13, Csr13.word);

    Bcncsr1.word = 0;
    Bcncsr1.field.Preload = TBTT_PRELOAD_TIME; // we guess TBTT is 2 TU ahead of BEACON-RxEnd time
    RTMP_IO_WRITE32(pAd, BCNCSR1, Bcncsr1.word);

    IsApPc = (CAP_IS_CF_POLLABLE_ON(pAd->PortCfg.CapabilityInfo) &&
              CAP_IS_CF_POLL_REQ_ON(pAd->PortCfg.CapabilityInfo));
    IsApPc = FALSE; // TODO: not support so far

    Csr14.word = 0;
    Csr14.field.TsfCount = 1;
    Csr14.field.TsfSync = 1; // sync TSF in INFRASTRUCTURE mode
    if (IsApPc)
    {
        Csr14.field.CfpCntPreload = pAd->PortCfg.CfpCount;
        Csr14.field.Tcfp = 1;
    }
    Csr14.field.BeaconGen = 0;
//  Csr14.field.TbcnPreload = (pAd->PortCfg.BeaconPeriod - 30) << 4; // TODO: ???? 1 TU ???
    Csr14.field.Tbcn = 1;
    RTMP_IO_WRITE32(pAd, CSR14, Csr14.word);

}

/*
    ==========================================================================
    Description:
    Note:
        BEACON frame in shared memory should be built ok before this routine
        can be called. Otherwise, a garbage frame maybe transmitted out every
        Beacon period.
    ==========================================================================
 */
const USHORT RateIdTo500Kbps[] = {2, 4, 11, 22};

VOID AsicEnableIbssSync(
    IN PRTMP_ADAPTER pAd)
{
	CSR12_STRUC Csr12;
    CSR13_STRUC Csr13;
    CSR14_STRUC Csr14;
    BCNCSR1_STRUC Bcncsr1;

    DBGPRINT(RT_DEBUG_TRACE, "--->AsicEnableIbssSync(ADHOC mode)\n");

    RTMP_IO_WRITE32(pAd, CSR14, 0x00000000);

    Csr12.word = 0;
    Csr12.field.BeaconInterval = pAd->PortCfg.BeaconPeriod << 4; // ASIC register in units of 1/16 TU
    RTMP_IO_WRITE32(pAd, CSR12, Csr12.word);

    Csr13.word = 0;
    Csr13.field.AtimwDuration = pAd->PortCfg.AtimWin << 4; // ASIC register in units of 1/16 TU
    RTMP_IO_WRITE32(pAd, CSR13, Csr13.word);

    Bcncsr1.word = 0;
    Bcncsr1.field.Preload = (pAd->PortCfg.BeaconPeriod * 30) +   // we've delay 1us clock to be 33*33.3Mhz, so preload this value to increase BEACON send ratio
                            pAd->PortCfg.Dpreamble +
                            pAd->PortCfg.Dplcp +
                            ((24 << 4) / RateIdTo500Kbps[RATE_2]);
    RTMP_IO_WRITE32(pAd, BCNCSR1, Bcncsr1.word);

    Csr14.word = 0;
    Csr14.field.TsfCount = 1;
    Csr14.field.TsfSync = 2; // sync TSF in IBSS mode
    Csr14.field.Tbcn = 1;
//  if (pAd->PortCfg.AtimWin)
//      Csr14.field.Tatimw = 1; // enable Tatimw only when AtimWindow is NOT ZERO
//  Csr14.field.TbcnPreload = (pAd->PortCfg.BeaconPeriod - 1) << 4;
    Csr14.field.BeaconGen = 1;
    RTMP_IO_WRITE32(pAd, CSR14, Csr14.word);
}

VOID AsicLedPeriodicExec(
    IN unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;
    ULONG LedCsr = 0x0000461E; // 0x0000461E;

    pAd->PortCfg.LedCntl.fOdd = ! pAd->PortCfg.LedCntl.fOdd;

    if (INFRA_ON(pAd) || ADHOC_ON(pAd))
        LedCsr |= 0x00010000; // enable hardwired TX activity LED
    if (pAd->PortCfg.LedCntl.fOdd && pAd->PortCfg.LedCntl.fRxActivity)
        LedCsr |= 0x00020000; // turn on software-based RX activity LED
    pAd->PortCfg.LedCntl.fRxActivity = FALSE;

    if (LedCsr != pAd->PortCfg.LedCntl.LastLedCsr)
    {
//        DBGPRINT(RT_DEBUG_TRACE, ("AsicLedPeriodicExec(%8x)\n",LedCsr));
        pAd->PortCfg.LedCntl.LastLedCsr = LedCsr;
        RTMP_IO_WRITE32(pAd, LEDCSR, LedCsr);
    }
}


