/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2005: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 01/Sep/2005                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <termios.h>
#include <sys/mman.h>
#include <string.h>
#include <rsbac/types.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_SHADOW_H
#include <shadow.h>
#endif

#define MAX_TRIES 3

  char * progname;
  char password[RSBAC_MAXNAMELEN] = "";
  char * int_pass = NULL;
  char * crypt_pass = NULL;
  rsbac_time_t ttl = 0;
  char * moregroups = NULL;
  char * skeldir = "/etc/skel/";
  int verbose = 0;
  int err;
  int useold = 0;
  int addallold = 0;
  int homedirgiven = 0;
  int createhome = 0;
  int askpass = 0;
  rsbac_list_ta_number_t ta_number = 0;

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [flags] username\n"), progname);
      printf(gettext(" -c comment = fullname or comment,\n"));
      printf(gettext(" -d dir = homedir of user,\n"));
      printf(gettext(" -g group = main / initial Linux group,\n"));
      printf(gettext(" -G group1[,group2,...] = add more Linux groups,\n"));
      printf(gettext(" -p password = password in plaintext,\n"));
      printf(gettext(" -P = ask for password,\n"));
      printf(gettext(" -Q password = encrypted password (from backup),\n"));
      printf(gettext(" -s shell = user's shell,\n"));
      printf(gettext(" -u uid = uid to use,\n"));
      printf(gettext(" -m = create user home dir from skeleton,\n"));
      printf(gettext(" -k dir = use this skeleton dir instead of /etc/skel/,\n"));
      printf(gettext(" -n minchange-days = minimum days between password changes,\n"));
      printf(gettext(" -x maxchange-days = maximum days between password changes,\n"));
      printf(gettext(" -w warnchange-days = warning days before password must be changed,\n"));
      printf(gettext(" -f inactive-days = period between password expiry and account disabling,\n"));
      printf(gettext(" -e expire-days = days since 1/Jan/1970 when account gets disabled,\n"));
      printf(gettext(" -t = set relative time-to-live in secs (role/type comp, admin, assign only)\n"));
      printf(gettext(" -T = set absolute time-to-live in secs (role/type comp, admin, assign only)\n"));
      printf(gettext(" -D = set relative time-to-live in days (role/type comp, admin, assign only)\n"));
      printf(gettext(" -o = use values from old passwd/shadow entry,\n"));
      printf(gettext(" -O = add all existing users (implies -o)\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    }

int password_read(char * to, char * from)
  {
    char * f = from;
    char * t = to;
    char   tmp[3];
    int i;
    int res = 0;

    if(strlen(from) != RSBAC_UM_PASS_LEN * 2)
      {
        fprintf(stderr, gettext("Wrong encrypted password length!\n"));
        return -RSBAC_EINVALIDVALUE;
      }
    res = mlock(&tmp, 3);
    if (res) {
	    fprintf(stderr, gettext("Unable to lock password into physical memory!\n"));
    }
    
    tmp[2] = 0;
    while(f[0] && f[1])
      {
        tmp[0] = f[0];
        tmp[1] = f[1];
        i = strtoul(tmp, 0, 16);
        if(i < 0 || i > 255) {
		memset(&tmp, 0, 3);
		munlock(&tmp, 3);
		return -RSBAC_EINVALIDVALUE;
	}
        *t = i;
        t++;
        f += 2;
      }
    memset(&tmp, 0, 3);
    munlock(&tmp, 3);
    return 0;
  }

int get_pass(char * username, char ** my_int_pass_p)
  {
    char * pass1 = malloc(RSBAC_MAXNAMELEN);
    char * pass2 = malloc(RSBAC_MAXNAMELEN);
    struct termios old_term;
    struct termios tmp_term;
    int res;
    int i;

    if(!pass1)
      return -ENOMEM;
    if(!pass2)
      {
        free(pass1);
        return -ENOMEM;
      }
    res = mlock(pass1, RSBAC_MAXNAMELEN);
    if (res) {
	    fprintf(stderr, gettext("Unable to lock password into physical memory!\n"));
    }
    res = mlock(pass2, RSBAC_MAXNAMELEN);
    if (res) {
	    fprintf(stderr, gettext("Unable to lock password into physical memory!\n"));
    }
    for(i = 0; i < MAX_TRIES; i++)
      {
        printf("Password for user %s (empty password not allowed): ", username);
        if(isatty(STDIN_FILENO))
          {
            res = tcgetattr(STDIN_FILENO, &old_term);
            if(res)
              {
		memset(pass1, 0, RSBAC_MAXNAMELEN);
                munlock(pass1, RSBAC_MAXNAMELEN);
		free(pass1);
		memset(pass2, 0, RSBAC_MAXNAMELEN);
		munlock(pass2, RSBAC_MAXNAMELEN);
                free(pass2);
                return res;
              }
            memcpy(&tmp_term, &old_term, sizeof(old_term));
            tmp_term.c_lflag &= ~(ECHO);
            res = tcsetattr(STDIN_FILENO, TCSAFLUSH, &tmp_term);
            if(res)
              {
		memset(pass1, 0, RSBAC_MAXNAMELEN);
		munlock(pass1, RSBAC_MAXNAMELEN);
                free(pass1);
		memset(pass2, 0, RSBAC_MAXNAMELEN);
		munlock(pass2, RSBAC_MAXNAMELEN);
                free(pass2);
                return res;
              }
          }
        res = scanf("%254s", pass1);
        if(isatty(STDIN_FILENO))
          tcsetattr(STDIN_FILENO, TCSAFLUSH, &old_term);
        printf("\n");
        if(res < 0)
          {
            fprintf(stderr, gettext("%s: invalid password!\n"), progname);
	    memset(pass1, 0, RSBAC_MAXNAMELEN);
	    munlock(pass1, RSBAC_MAXNAMELEN);
	    free(pass1);
	    memset(pass2, 0, RSBAC_MAXNAMELEN);
	    munlock(pass2, RSBAC_MAXNAMELEN);
	    free(pass2);
            return -RSBAC_EINVALIDVALUE;
          }
        if(!res)
          {
            memset(pass1, 0, RSBAC_MAXNAMELEN);
	    munlock(pass1, RSBAC_MAXNAMELEN);
	    free(pass1);
	    memset(pass2, 0, RSBAC_MAXNAMELEN);
	    munlock(pass2, RSBAC_MAXNAMELEN);
            free(pass2);
            return 0;
          }
        printf("Repeat password for user %s: ", username);
        if(isatty(STDIN_FILENO))
          {
            res = tcgetattr(STDIN_FILENO, &old_term);
            if(res)
              {
		memset(pass1, 0, RSBAC_MAXNAMELEN);
		munlock(pass1, RSBAC_MAXNAMELEN);
                free(pass1);
		memset(pass2, 0, RSBAC_MAXNAMELEN);
		munlock(pass2, RSBAC_MAXNAMELEN);
                free(pass2);
                return res;
              }
            memcpy(&tmp_term, &old_term, sizeof(old_term));
            tmp_term.c_lflag &= ~(ECHO);
            res = tcsetattr(STDIN_FILENO, TCSAFLUSH, &tmp_term);
            if(res)
              {
                memset(pass1, 0, RSBAC_MAXNAMELEN);
		munlock(pass1, RSBAC_MAXNAMELEN);
		free(pass1);
                memset(pass2, 0, RSBAC_MAXNAMELEN);
		munlock(pass2, RSBAC_MAXNAMELEN);
		free(pass2);
                return res;
              }
          }
        res = scanf("%254s", pass2);
        if(isatty(STDIN_FILENO))
          tcsetattr(STDIN_FILENO, TCSAFLUSH, &old_term);
        printf("\n");
        if(res <= 0)
          {
            fprintf(stderr, gettext("%s: invalid password!\n"), progname);
            return -RSBAC_EINVALIDVALUE;
          }
        if(!strcmp(pass1,pass2))
          {
            *my_int_pass_p = pass1;
	    memset(pass2, 0, RSBAC_MAXNAMELEN);
	    munlock(pass2, RSBAC_MAXNAMELEN);
            free(pass2);
            return 0;
          }
        else
          fprintf(stderr, gettext("%s: password mismatch!\n"), progname);
      }
    fprintf(stderr, gettext("%s: Too many tries, using default password!\n"), progname);
    memset(pass1, 0, RSBAC_MAXNAMELEN);
    munlock(pass1, RSBAC_MAXNAMELEN);
    free(pass1);
    memset(pass2, 0, RSBAC_MAXNAMELEN);
    munlock(pass2, RSBAC_MAXNAMELEN);
    free(pass2);
    return 0;
  }

int process(char * name, rsbac_uid_t user, struct rsbac_um_user_entry_t entry)
    {
      int res = 0;
      char * my_int_pass = int_pass;

      if(useold)
        {
          struct passwd * pwentry;
          #ifdef HAVE_SHADOW_H
          struct spwd * spentry = NULL;
          #endif

          pwentry = getpwnam(name);
          if(!pwentry)
            {
              fprintf(stderr, "%s: old entry not found!\n", name);
              return -RSBAC_ENOTFOUND;
            }
          user = pwentry->pw_uid;
          entry.group = pwentry->pw_gid;
          strncpy(entry.fullname, pwentry->pw_gecos, RSBAC_UM_FULLNAME_LEN);
          entry.fullname[RSBAC_UM_FULLNAME_LEN - 1] = 0;
          strncpy(entry.homedir, pwentry->pw_dir, RSBAC_UM_HOMEDIR_LEN);
          entry.homedir[RSBAC_UM_HOMEDIR_LEN - 1] = 0;
          strncpy(entry.shell, pwentry->pw_shell, RSBAC_UM_SHELL_LEN);
          entry.shell[RSBAC_UM_SHELL_LEN - 1] = 0;
          entry.lastchange = 0;

          #ifdef HAVE_SHADOW_H
          spentry = getspnam(name);
          if(!spentry)
            {
              fprintf(stderr, "%s: old shadow entry not found, adding with default values!\n", name);
            }
          else
            {
              entry.minchange = spentry->sp_min;
              entry.maxchange = spentry->sp_max;
              entry.warnchange = spentry->sp_warn;
              entry.inactive = spentry->sp_inact;
              entry.expire = spentry->sp_expire;
              if(strlen(spentry->sp_pwdp) == 1)
                {
                  my_int_pass = NULL;
                  if(verbose)
                    printf("Account %s seems to be disabled, disabling password\n", name);
                }
              else
                if(askpass)
                  get_pass(name, &my_int_pass);
            }
          #else
          if(!strcmp(pwentry->pw_passwd, "*"))
            {
              if(askpass)
                get_pass(name, &my_int_pass);
              else
                fprintf(stderr, "%s: shadow not supported, adding with default values!\n", name);
            }
          else
            {
              if(strlen(pwentry->pw_passwd) == 1)
                {
                  my_int_pass = NULL;
                  if(verbose)
                    printf("Account %s seems to be disabled, disabling password\n", name);
                }
              else
                if(askpass)
                  get_pass(name, &my_int_pass);
            }
          #endif

          if(verbose)
            printf("Adding old user %s\n", name);
        }
      else
        {
          if(verbose)
            printf("Adding user %s\n", name);
          if(askpass)
            get_pass(name, &my_int_pass);
        }
      strncpy(entry.name, name, RSBAC_UM_NAME_LEN);
      entry.name[RSBAC_UM_NAME_LEN - 1] = 0;
      if(!homedirgiven && !useold)
        {
          snprintf(entry.homedir, RSBAC_UM_HOMEDIR_LEN, "/home/%s", name);
          entry.homedir[RSBAC_UM_HOMEDIR_LEN - 1] = 0;
        }
      res = rsbac_um_add_user(ta_number, user, &entry, my_int_pass, ttl);
      if(my_int_pass && (my_int_pass != int_pass))
        free(my_int_pass);
      if(res)
        {
          fprintf(stderr, "%s: ", name);
          show_error(res);
          return res;
        }
      /* copy user home dir from skel */
      if(createhome)
        {
          struct stat statbuf;

          if(!stat(entry.homedir, &statbuf))
            {
              fprintf(stderr, "User %s homedir path %s already exists\n", name, entry.homedir);
            }
          else
            {
              char command[RSBAC_MAXNAMELEN];
              FILE * pfile;

              snprintf(command, RSBAC_MAXNAMELEN, "/bin/cp -a \"%s\" \"%s\"",
                       skeldir, entry.homedir);
              pfile = popen(command, "w");
              if(!pfile)
                {
                  fprintf(stderr, "Copying user %s homedir %s failed with error", name, entry.homedir);
                  show_error(res);
                  fprintf(stderr, "\n");
                }
              else
                {
                  pclose(pfile);
                  snprintf(command, RSBAC_MAXNAMELEN, "/bin/chown -R \"%s:\" \"%s\"",
                           name, entry.homedir);
                  pfile = popen(command, "w");
                  if(!pfile)
                    {
                      fprintf(stderr, "Chown of homedir %s to %s failed with error", entry.homedir, name);
                      show_error(res);
                      fprintf(stderr, "\n");
                    }
                  else
                    pclose(pfile);
                }
            }
        }

      if(moregroups)
        {
          char * p;
          char * m;
          rsbac_gid_t group;

          res = rsbac_um_get_uid(ta_number, entry.name, &user);
          error_exit(res);
          m = moregroups;
          p = moregroups;
          while(*m)
            {
              while(*p && (*p != ','))
                p++;
              if(*p)
                {
                  *p = 0;
                  if(rsbac_um_get_gid(ta_number, m, &group))
                    {
                      group = strtoul(m,0,0);
                      if(!group && strcmp(m,"0"))
                        {
                          fprintf(stderr, gettext("%s: Unknown group %s\n"), progname, m);
                          p++;
                          m = p;
                          continue;
                        }
                    }
                  printf("String %s, value %u\n", m, group);
                  p++;
                  m = p;
                }
              else
                {
                  if(rsbac_um_get_gid(ta_number, m, &group))
                    {
                      group = strtoul(m,0,0);
                      if(!group && strcmp(m,"0"))
                        {
                          fprintf(stderr, gettext("%s: Unknown group %s\n"), progname, m);
                          m = p;
                          continue;
                        }
                    }
                  printf("String %s, value %u\n", m, group);
                  m = p;
                }
              if(verbose)
                printf("Adding user %s group membership %u\n", name, group);
              res = rsbac_um_add_gm(ta_number, user, group, 0);
              if(res)
                {
                  fprintf(stderr, "%s group membership %u: ", name, group);
                  show_error(res);
                }
            }
        }
      if(crypt_pass)
        {
          union rsbac_um_mod_data_t data;
	  int data_s;

          data_s = sizeof(&data);
	  res = mlock(&data, data_s);
	  if (res) {
		  fprintf(stderr, gettext("Unable to lock password into physical memory!\n"));
	  }
	  memcpy(data.string, crypt_pass, RSBAC_UM_PASS_LEN);
          res = rsbac_um_mod_user(ta_number, user, UM_cryptpass, &data);
	  memset(&data, 0, data_s);
	  munlock(&data, data_s);
          show_error(res);
        }
      return 0;
    }


int main(int argc, char ** argv)
{
  int res = 0;
  struct rsbac_um_user_entry_t entry = DEFAULT_UM_U_ENTRY;
  rsbac_uid_t user = RSBAC_NO_USER;

  locale_init();

  progname = argv[0];
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }

  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'h':
                use();
                return 0;
              case 'v':
                verbose++;
                break;
              case 'o':
                useold = 1;
                break;
              case 'O':
                addallold = 1;
                useold = 1;
                break;
              case 'c':
                if(argc > 2)
                  {
                    strncpy(entry.fullname, argv[2], RSBAC_UM_FULLNAME_LEN);
                    entry.fullname[RSBAC_UM_FULLNAME_LEN - 1] = 0;
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'd':
                if(argc > 2)
                  {
                    strncpy(entry.homedir, argv[2], RSBAC_UM_HOMEDIR_LEN);
                    entry.homedir[RSBAC_UM_HOMEDIR_LEN - 1] = 0;
                    homedirgiven = 1;
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'p':
                if(argc > 2)
                  {
                    err = mlock(&password, RSBAC_MAXNAMELEN);
		    if (err) {
			    fprintf(stderr, gettext("Unable to lock password into physical memory!\n"));
		    }
		    strncpy(password, argv[2], RSBAC_MAXNAMELEN);
                    password[RSBAC_MAXNAMELEN - 1] = 0;
                    int_pass = password;
                    crypt_pass = NULL;
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'P':
                askpass = 1;
                break;
              case 'Q':
                if(argc > 2)
                  {
                    err = mlock(&password, RSBAC_MAXNAMELEN);
		    if (err) {
			    fprintf(stderr, gettext("Unable to lock password into physical memory!\n"));
		    }
		    err = password_read(password, argv[2]);
                    error_exit(err);
                    crypt_pass = password;
                    int_pass = NULL;
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 's':
                if(argc > 2)
                  {
                    strncpy(entry.shell, argv[2], RSBAC_UM_SHELL_LEN);
                    entry.shell[RSBAC_UM_SHELL_LEN - 1] = 0;
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'g':
                if(argc > 2)
                  {
                    if(rsbac_um_get_gid(ta_number, argv[2], &entry.group))
                      {
                        entry.group = strtoul(argv[2],0,0);
                        if(!entry.group && strcmp(argv[2],"0"))
                          {
                            fprintf(stderr, gettext("%s: Unknown group %s\n"), progname, argv[2]);
                            exit(1);
                          }
                      }
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'G':
                if(argc > 2)
                  {
                    moregroups = argv[2];
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'u':
                if(argc > 2)
                  {
                    user = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'l':
                if(argc > 2)
                  {
                    entry.lastchange = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'n':
                if(argc > 2)
                  {
                    entry.minchange = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'x':
                if(argc > 2)
                  {
                    entry.maxchange = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'w':
                if(argc > 2)
                  {
                    entry.warnchange = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'i':
              case 'f':
                if(argc > 2)
                  {
                    entry.inactive = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'e':
                if(argc > 2)
                  {
                    entry.expire = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'm':
                createhome = 1;
                break;
              case 'k':
                if(argc > 2)
                  {
                    struct stat statbuf;

                    if(stat(argv[2], &statbuf))
                      {
                        fprintf(stderr, gettext("%s: cannot lookup skel dir %s\n"), progname, argv[2]);
                        exit(1);
                      }
                    else
                      if(!S_ISDIR(statbuf.st_mode))
                        {
                          fprintf(stderr, gettext("%s: skel dir %s is no dir\n"), progname, argv[2]);
                          exit(1);
                        }
                    skeldir = argv[2];
                    if(strlen(skeldir) > RSBAC_MAXNAMELEN - 50)
                      {
                        fprintf(stderr, gettext("%s: skel dir name %s is too long\n"), progname, argv[2]);
                        exit(1);
                      }
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 't':
                if(argc > 2)
                  {
                    ttl = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing ttl value for parameter %c\n"), progname, *pos);
                break;
              case 'D':
                if(argc > 2)
                  {
                    ttl = 86400 * strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing ttl value for parameter %c\n"), progname, *pos);
                break;
              case 'T':
                if(argc > 2)
                  {
                    rsbac_time_t now = time(NULL);
                    ttl = strtoul(argv[2], 0, 10);
                    if(ttl > now)
                      {
                        ttl -= now;
                        argc--;
                        argv++;
                      }
                    else
                      {
                        fprintf(stderr,
                                gettext("%s: ttl value for parameter %c is in the past, exiting\n"), progname, *pos);
                        exit(1);
                      }
                  }
                else
                  fprintf(stderr, gettext("%s: missing ttl value for parameter %c\n"), progname, *pos);
                break;
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if(addallold)
    {
      struct passwd * user_info_p;
      rsbac_uid_t tmp_uid; 

      setpwent();
      while((user_info_p = getpwent()))
        {
          if(   !rsbac_um_user_exists(ta_number, user_info_p->pw_uid)
             && rsbac_um_get_uid(ta_number, user_info_p->pw_name, &tmp_uid)
            )
            process(user_info_p->pw_name, user, entry);
          else
          if(verbose)
            printf("Skipping existing user %s / uid %u\n",
                  user_info_p->pw_name,
                  user_info_p->pw_uid);
        }
      endpwent();
      exit(0);
    }
  else
  if (argc > 1)
    {
      int i;

      for(i=1; i< argc; i++)
        {
          process(argv[i], user, entry);
        }
      memset(password, 0, RSBAC_MAXNAMELEN);
      munlock(&password, RSBAC_MAXNAMELEN);
      exit(0);
    }
  else
    {
      use();
      return 1;
    }
  exit(0);
}
