/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2005: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 31/May/2005                        */
/*************************************************** */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <grp.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rsbac/types.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

  char * progname;
  char password[RSBAC_MAXNAMELEN] = "";
  char * crypt_pass = NULL;
  rsbac_time_t ttl = 0;
  int verbose = 0;
  int useold = 0;
  int addallold = 0;
  rsbac_list_ta_number_t ta_number = 0;

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [flags] groupname\n"), progname);
      printf(gettext(" -p password = password in plaintext,\n"));
      printf(gettext(" -g gid = gid to use,\n"));
      printf(gettext(" -t = set relative time-to-live in secs (role/type comp, admin, assign only)\n"));
      printf(gettext(" -T = set absolute time-to-live in secs (role/type comp, admin, assign only)\n"));
      printf(gettext(" -D = set relative time-to-live in days (role/type comp, admin, assign only)\n"));
      printf(gettext(" -o = use values from old group entry,\n"));
      printf(gettext(" -O = add all existing groups (implies -o)\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    }

int password_read(char * to, char * from)
  {
    char * f = from;
    char * t = to;
    char   tmp[3];
    int i;

    if(strlen(from) != RSBAC_UM_PASS_LEN * 2)
      {
        fprintf(stderr, "Wrong encrypted password length!\n");
        return -RSBAC_EINVALIDVALUE;
      }
    tmp[2] = 0;
    while(f[0] && f[1])
      {
        tmp[0] = f[0];
        tmp[1] = f[1];
        i = strtoul(tmp, 0, 16);
        if(i < 0 || i > 255)
          return -RSBAC_EINVALIDVALUE;
        *t = i;
        t++;
        f += 2;
      }
    return 0;
  }

int process(char * name, rsbac_gid_t group,
            struct rsbac_um_group_entry_t entry,
            char ** gr_mem)
  {
      int res;

      if(useold)
        {
          if(verbose)
            printf("Adding old group %s with gid %u\n", name, group);
        }
      else
        if(verbose)
          printf("Adding group %s\n", name);
      strncpy(entry.name, name, RSBAC_UM_NAME_LEN);
      entry.name[RSBAC_UM_NAME_LEN - 1] = 0;
      res = rsbac_um_add_group(ta_number, group, &entry, password, ttl);
      if(res)
        {
          fprintf(stderr, "%s: ", name);
          show_error(res);
        }
      if(gr_mem)
        {
          rsbac_uid_t tmp_uid;

          while(*gr_mem)
            {
              if(verbose)
                printf("Adding group %s member %s\n", name, *gr_mem);
              res = rsbac_um_get_uid(ta_number, *gr_mem, &tmp_uid);
              if(res)
                {
                  fprintf(stderr, "Lookup group %s member %s: ", name, *gr_mem);
                  show_error(res);
                }
              else
                {
                  res = rsbac_um_add_gm(ta_number, tmp_uid, group, ttl);
                  if(res)
                    {
                      fprintf(stderr, "Adding group %s member %s (uid %u): ", name, *gr_mem, tmp_uid);
                      show_error(res);
                    }
                }
              gr_mem++;
            }
        }
      if(crypt_pass)
        {
          union rsbac_um_mod_data_t data;

          memcpy(data.string, crypt_pass, RSBAC_UM_PASS_LEN);
          res = rsbac_um_mod_group(ta_number, group, UM_cryptpass, &data);
          show_error(res);
        }
     return res;
  }

int main(int argc, char ** argv)
{
  int res = 0;
  struct rsbac_um_group_entry_t entry = DEFAULT_UM_G_ENTRY;
  rsbac_gid_t group = RSBAC_NO_GROUP;

  locale_init();

  progname = argv[0];
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'h':
                use();
                return 0;
              case 'v':
                verbose++;
                break;
              case 'o':
                useold = 1;
                break;
              case 'O':
                addallold = 1;
                useold = 1;
                break;
              case 'p':
                if(argc > 2)
                  {
                    strncpy(password, argv[2], RSBAC_MAXNAMELEN);
                    password[RSBAC_MAXNAMELEN - 1] = 0;
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'Q':
                if(argc > 2)
                  {
                    crypt_pass = malloc(RSBAC_MAXNAMELEN);
                    if(!crypt_pass)
                      error_exit(-ENOMEM);
                    res = password_read(crypt_pass, argv[2]);
                    error_exit(res);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 'g':
                if(argc > 2)
                  {
                    group = strtoul(argv[2],0,0);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing argument for parameter %c\n"), progname, *pos);
                break;
              case 't':
                if(argc > 2)
                  {
                    ttl = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing ttl value for parameter %c\n"), progname, *pos);
                break;
              case 'D':
                if(argc > 2)
                  {
                    ttl = 86400 * strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  fprintf(stderr, gettext("%s: missing ttl value for parameter %c\n"), progname, *pos);
                break;
              case 'T':
                if(argc > 2)
                  {
                    rsbac_time_t now = time(NULL);
                    ttl = strtoul(argv[2], 0, 10);
                    if(ttl > now)
                      {
                        ttl -= now;
                        argc--;
                        argv++;
                      }
                    else
                      {
                        fprintf(stderr,
                                gettext("%s: ttl value for parameter %c is in the past, exiting\n"), progname, *pos);
                        exit(1);
                      }
                  }
                else
                  fprintf(stderr, gettext("%s: missing ttl value for parameter %c\n"), progname, *pos);
                break;
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if(addallold)
    {
      struct group * group_info_p;
      rsbac_gid_t tmp_gid; 

      setgrent();
      while((group_info_p = getgrent()))
        process(group_info_p->gr_name, group_info_p->gr_gid,
                entry, group_info_p->gr_mem);
      endgrent();
      memset(password, 0, RSBAC_MAXNAMELEN);
      exit(0);
    }
  else
  if (argc > 1)
    {
      int i;
      struct group * group_info_p;

      for(i=1; i< argc; i++)
        {
          if(useold)
            {
              group_info_p = getgrnam(argv[i]);
              if(!group_info_p)
                fprintf(stderr, "%s: old entry not found!\n", argv[i]);
              else
                process(group_info_p->gr_name, group_info_p->gr_gid,
                        entry, group_info_p->gr_mem);
            }
          else
            process(argv[i], group, entry, NULL);
        }
      memset(password, 0, RSBAC_MAXNAMELEN);
      exit(0);
    }
  else
    {
      use();
      return 1;
    }
  return (res);
}
