#include "file.h"
#include <stdlib.h>
#include <string.h>
#include <assert.h>


#ifdef DMALLOC
#include <dmalloc.h>
#endif

/****h* ROBODoc/Filename
 * NAME
 *   Functions to deal with keeping track
 *   of filenames and directory names.
 *****
 */

/****f* Filename/RB_Get_RB_Filename
 * NAME
 *   RB_Get_RB_Filename
 * INPUTS
 *    arg_rb_filename --
 *    arg_rb_path --
 * FUNCTION
 *   Create a new RB_Filename structure based on arg_filename and
 *   arg_rb_path.
 * SOURCE
 */

struct RB_Filename *
RB_Get_RB_Filename( char *arg_filename, struct RB_Path *arg_rb_path )
{
    struct RB_Filename *rb_filename =
        ( struct RB_Filename * ) malloc( sizeof( struct RB_Filename ) );
    rb_filename->name = ( char * ) malloc( strlen( arg_filename ) + 1 );
    rb_filename->docname = 0;
    rb_filename->fullname = 0;
    rb_filename->fulldocname = 0;
    strcpy( rb_filename->name, arg_filename );
    rb_filename->path = arg_rb_path;
    return rb_filename;
}

/*****/

void
RB_Filename_Dump( struct RB_Filename *arg_rb_filename )
{
    printf( "[%s %s %s]  ", RB_Get_Path( arg_rb_filename ),
            RB_Get_Filename( arg_rb_filename ),
            RB_Get_Extension( arg_rb_filename ) );
    printf( "%s\n", RB_Get_Fullname( arg_rb_filename ) );
}

/*x**f* Filename/RB_Free_RB_Filename 
 * NAME
 *   RB_Free_RB_Filename -- free a RB_Filename structure.
 *
 *****
 * TODO Documentation
 */

void
RB_Free_RB_Filename( struct RB_Filename *arg_rb_filename )
{
    free( arg_rb_filename->name );
    if ( arg_rb_filename->docname )
    {
        free( arg_rb_filename->docname );
    }
    if ( arg_rb_filename->fullname )
    {
        free( arg_rb_filename->fullname );
    }
    if ( arg_rb_filename->fulldocname )
    {
        free( arg_rb_filename->fulldocname );
    }
    free( arg_rb_filename );
}

/* TODO Documentation RB_Get_FullDocname */
char               *
RB_Get_FullDocname( struct RB_Filename *arg_rb_filename )
{
    char               *result = arg_rb_filename->fulldocname;

    if ( result == NULL )
    {
        unsigned int  size = strlen( arg_rb_filename->docname ) +
            strlen( arg_rb_filename->path->docname ) + 1;
        result = ( char * ) malloc( size * sizeof( char ) );
        assert( result );
        *result = '\0';
        strcat( result, arg_rb_filename->path->docname );
        strcat( result, arg_rb_filename->docname );
        /* Save the result so it can be reused later on, and we can properly deallocate it. */
        arg_rb_filename->fulldocname = result;
    }
    return result;
}


/****f* Filename/RB_Get_Fullname
 * NAME
 *   RB_Get_Fullname --
 * FUNCTION
 *   Give the full name of the file, that is the name of
 *   the file including the extension and the path.
 *   The path can be relative or absolute.
 * NOTE
 *   The string returned is owned by this function
 *   so don't change it.
 ******
 */

char               *
RB_Get_Fullname( struct RB_Filename *arg_rb_filename )
{
    char               *result = arg_rb_filename->fullname;

    if ( result == NULL )
    {
        unsigned int        size = strlen( arg_rb_filename->name ) +
            strlen( arg_rb_filename->path->name ) + 1;
        result = ( char * ) malloc( size * sizeof( char ) );
        assert( result );
        *result = '\0';
        strcat( result, arg_rb_filename->path->name );
        strcat( result, arg_rb_filename->name );
        /* Save the result so it can be reused later on, and we can properly deallocate it. */
        arg_rb_filename->fullname = result;
    }
    return result;
}

/****f* Filename/RB_Get_Path
 * NAME
 *   RB_Get_Path --
 * FUNCTION
 *   Give the path for this file.
 * NOTE
 *   The string returned is owned by this function
 *   so don't change it.
 ******
 */


char               *
RB_Get_Path( struct RB_Filename *arg_rb_filename )
{
    return arg_rb_filename->path->name;
}


/****f* Filename/RB_Get_Extension
 * NAME
 *   RB_Get_Extension --
 * FUNCTION
 *   Give the extension of this file. That is the part after
 *   the last '.' if there is any.
 * RESULT
 *   pointer to the extension
 *   pointer to a '\0' if no extension was found.
 * NOTE
 *   The string returned is owned by this function
 *   so don't change it.
 ******
 */

char               *
RB_Get_Extension( struct RB_Filename *arg_rb_filename )
{
    char               *c = arg_rb_filename->name;
    int                 i = strlen( c );

    for ( c += i; c != arg_rb_filename->name && ( *c != '.' ); --c )
    {
        /* Empty */
    }
    if ( *c == '.' )
    {
        ++c;
    }
    else
    {
        c = arg_rb_filename->name;
        c += i;
    }
    return c;
}

/****f* Filename/RB_Get_Filename
 * NAME
 *   RB_Get_Filename --
 * FUNCTION
 *   Give the name of this file. That is the name
 *   of the file without its path but with the
 *   extension.
 * RESULT
 *   pointer to the extension
 *   pointer to a '\0' if no extension was found.
 * NOTE
 *   The string returned is owned by this function
 *   so don't change it.
 ******
 */

char               *
RB_Get_Filename( struct RB_Filename *arg_rb_filename )
{
    return arg_rb_filename->name;
}
