#ifndef _RHEOLEF_FORM_ELEMENT_H
#define _RHEOLEF_FORM_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/scalar_traits.h"
#include "rheolef/numbering.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis_on_pointset.h"
#include "rheolef/tensor.h"
#include "rheolef/field.h"
#include "rheolef/band.h"

#include <boost/numeric/ublas/matrix.hpp>

namespace rheolef {
namespace ublas = boost::numeric::ublas;

template<class T, class M>
class form_element_rep {
public:
// typedefs:
    typedef geo_element::size_type                          size_type;
    typedef typename scalar_traits<T>::type                 float_type;
    typedef point_basic<float_type>                         vertex_type;
    typedef space_basic<float_type,M>                       space_type;
    typedef geo_basic<float_type,M>                         geo_type;
    typedef space_constant::coordinate_type                 coordinate_type;
    typedef typename basis_on_pointset<T>::const_iterator   quad_const_iterator;
    typedef typename basis_on_pointset<T>::const_iterator_grad quad_const_iterator_grad;

// allocators:

    form_element_rep();

    form_element_rep(
	const space_type&       X,
	const space_type&       Y,
	const geo_type&         omega,
	const quadrature_option_type& qopt);

    virtual ~form_element_rep();
    void     initialize_all() const;

// the main functionality, virtual, defined in concrete derived classes:

    virtual void operator() (const geo_element& K, ublas::matrix<T>& m) const = 0;
    virtual void initialize () const = 0;
    virtual bool is_symmetric () const = 0;

// accessors:

    const space_type& get_first_space() const;
    const space_type& get_second_space() const;

    const basis&  get_first_basis() const;
    const basis&  get_second_basis() const;
    const basis&  get_piola_basis () const;
    size_type     coordinate_dimension() const;
    coordinate_type coordinate_system() const;
    size_type     n_derivative() const;
    void          set_n_derivative (size_type n) const;

    const vertex_type& dis_node(size_type dis_inod) const;

// predefined forms:

    void build_scalar_mass      (const geo_element& K, ublas::matrix<T>& m) const;
    void build_general_mass     (const geo_element& K, ublas::matrix<T>& m) const;
    void build_scalar_grad_grad (const geo_element& K, ublas::matrix<T>& a) const;
    void build_gradt_grad       (const geo_element& K, ublas::matrix<T>& a) const;
    void build_div_div          (const geo_element& K, ublas::matrix<T>& a) const;
    void build_d_dx             (const geo_element& K, ublas::matrix<T>& a, size_type i_comp) const;

// utilities:

    static void cumul_otimes (
    	ublas::matrix<T>& m, 
    	const T& w,
    	quad_const_iterator phi,
    	quad_const_iterator last_phi,
    	quad_const_iterator first_psi,
    	quad_const_iterator last_psi);

    static void cumul_otimes (
        ublas::matrix<T>& m,
	const tensor_basic<T>& Dw,
	quad_const_iterator_grad grad_phi,
	quad_const_iterator_grad last_grad_phi,
	quad_const_iterator_grad first_grad_psi,
	quad_const_iterator_grad last_grad_psi,
	size_type map_d);

// for scalar-weighted forms:

    void set_weight (const field_basic<T,M>& wh) const;
    bool is_weighted() const;
    const field_basic<T,M>& get_weight () const;
    T weight (const geo_element& K, size_type q) const;

// for axisymmetric systems:

    T weight_coordinate_system (
    	const geo_element&            K,
    	const std::vector<size_type>& dis_inod,
    	size_type                     q) const;

    void set_use_coordinate_system_weight (bool use) const
		{ _use_coordinate_system_weight = use; }
    bool use_coordinate_system_weight () const
		{ return _use_coordinate_system_weight; }
    void set_use_coordinate_system_dual_weight (bool use) const
		{ _use_coordinate_system_dual_weight = use; }
    bool use_coordinate_system_dual_weight () const
		{ return _use_coordinate_system_dual_weight; }

// for banded level set method:

    bool is_on_band() const { return _is_on_band; }
    void set_band (const band_basic<T,M>& bh) const { _band = bh; _is_on_band = true; }
    const band_basic<T,M>& get_band() const { return _band; }

// data:
protected:
    space_type                     _X;	// also origin      dof numbering
    space_type                     _Y;  // also destination dof numbering
    geo_type                       _omega; // for the access to distributed vertices

    mutable bool                   _initialized;
    mutable size_type              _n_derivative; // e.g. =2 for grad_grad and =0 for mass
    mutable quadrature<T>          _quad;
    mutable quadrature_option_type _qopt;
    mutable basis_on_pointset<T>   _bx_table;
    mutable basis_on_pointset<T>   _by_table;
    mutable basis_on_pointset<T>   _piola_table;

    mutable bool                   _is_weighted;
    mutable field_basic<T,M>       _wh;
    mutable basis_on_pointset<T>   _bw_table;
    mutable bool                   _use_coordinate_system_weight;
    mutable bool                   _use_coordinate_system_dual_weight;

    mutable bool                   _is_on_band;
    mutable band_basic<T,M>        _band;

// internal:
public:
    // automatically generated implementation in form_element_make.cc :
    static form_element_rep<T,M>* make_ptr (
	std::string           name,
	const space_type&     X,
	const space_type&     Y,
	const geo_type&       omega,
	const quadrature_option_type& qopt);
};
// -----------------------------------------------------------------------
// inlined
// -----------------------------------------------------------------------

template<class T, class M>
inline
form_element_rep<T,M>::form_element_rep()
 : _X(),
   _Y(),
   _omega(),
   _initialized(false),
   _n_derivative(std::numeric_limits<size_type>::max()),
   _quad(),
   _qopt(),
   _bx_table(),
   _by_table(),
   _piola_table(),
   _is_weighted(false),
   _wh(),
   _bw_table(),
   _use_coordinate_system_weight(true),
   _use_coordinate_system_dual_weight(false),
   _is_on_band(false),
   _band()
{
}
template<class T, class M>
inline
form_element_rep<T,M>::form_element_rep (
    const space_type&       X,
    const space_type&       Y,
    const geo_type&         omega,
    const quadrature_option_type& qopt)
 : _X (X),
   _Y (Y),
   _omega (omega),
   _initialized(false),
   _n_derivative(std::numeric_limits<size_type>::max()),
   _quad(),
   _qopt(qopt),
   _bx_table(),
   _by_table(),
   _piola_table(),
   _is_weighted(false),
   _wh(),
   _bw_table(),
   _use_coordinate_system_weight(true),
   _use_coordinate_system_dual_weight(false),
   _is_on_band(false),
   _band()
{
}
template<class T, class M>
inline
form_element_rep<T,M>::~form_element_rep() 
{
}
template<class T, class M>
inline
const typename form_element_rep<T,M>::vertex_type&
form_element_rep<T,M>::dis_node(size_type dis_inod) const
{
  return _omega.dis_node (dis_inod);
}
template<class T, class M>
inline
const typename form_element_rep<T,M>::space_type&
form_element_rep<T,M>::get_first_space() const
{
  return _X;
}
template<class T, class M>
inline
const typename form_element_rep<T,M>::space_type&
form_element_rep<T,M>::get_second_space() const
{
  return _Y;
}
template<class T, class M>
inline
const basis& 
form_element_rep<T,M>::get_first_basis() const
{
  return get_first_space().get_numbering().get_basis();
}
template<class T, class M>
inline
const basis& 
form_element_rep<T,M>::get_second_basis() const
{
  return get_second_space().get_numbering().get_basis();
}
template<class T, class M>
inline
const basis&
form_element_rep<T,M>::get_piola_basis () const
{
  return _omega.get_piola_basis();
}
template<class T, class M>
inline
typename form_element_rep<T,M>::size_type
form_element_rep<T,M>::coordinate_dimension() const
{
  return _omega.dimension();
}
template<class T, class M>
inline
typename form_element_rep<T,M>::coordinate_type
form_element_rep<T,M>::coordinate_system() const
{
  return _omega.coordinate_system();
}
template<class T, class M>
inline
typename form_element_rep<T,M>::size_type
form_element_rep<T,M>::n_derivative() const
{
  return _n_derivative;
}
template<class T, class M>
inline
void
form_element_rep<T,M>::set_n_derivative (size_type n) const
{
  _n_derivative = n;
}
template<class T, class M>
inline
bool
form_element_rep<T,M>::is_weighted() const
{
  return _is_weighted;
}
template<class T, class M>
inline
const field_basic<T,M>&
form_element_rep<T,M>::get_weight () const
{
  return _wh;
}
// ----------------------------------------------------------
// wrapper class
// ----------------------------------------------------------

/*Class:form_element
NAME: @code{form_element} - bilinear form on a single element
@cindex  bilinear form
@cindex  geometrical element
@clindex basis
@clindex numbering
@clindex geo_element
SYNOPSYS:
  @noindent
  The @code{form_element} class defines functions that compute
  a bilinear form defined between two polynomial basis on a single
  geometrical element. This bilinear form is represented
  by a matrix.

  The bilinear form is designated by a string, e.g. "mass", "grad_grad", ...
  indicating the form. The form depends also of the geometrical element:
  triangle, square, tetrahedron (@pxref{geo_element iclass}).

IMPLEMENTATION NOTE:
  @noindent
  The @code{form_element} class 
  is managed by (@pxref{smart_pointer iclass}).
  This class uses a pointer on a pure virtual class @code{form_element_rep}
  while the effective code refers to the specific concrete derived classes:
  mass, grad_grad, etc.

AUTHORS: Pierre.Saramito@imag.fr
DATE:   7 january 2004
End:
*/
//<form_element:
template <class T, class M>
class form_element : public smart_pointer<form_element_rep<T,M> > {
public:

// typedefs:

    typedef form_element_rep<T,M>         rep;
    typedef smart_pointer<rep>            base;
    typedef typename rep::size_type       size_type;
    typedef typename rep::vertex_type     vertex_type;
    typedef typename rep::space_type      space_type;
    typedef typename rep::geo_type        geo_type;
    typedef typename rep::coordinate_type coordinate_type;

// constructors:

    form_element ();
    form_element (
	std::string	      name,
	const space_type&     X,
	const space_type&     Y,
	const geo_type&       omega,
	const quadrature_option_type& qopt);

// accessors & modifier:

    void operator() (const geo_element& K, ublas::matrix<T>& m) const;
    virtual bool is_symmetric () const;

// for scalar-weighted forms:

    void set_weight (const field_basic<T,M>& wh) const;
    bool is_weighted() const;
    const field_basic<T,M>& get_weight () const;

// for banded level set method:

    bool is_on_band() const;
    const band_basic<T,M>& get_band() const;
    void set_band (const band_basic<T,M>& bh) const;

};
//>form_element:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template <class T, class M>
inline
form_element<T,M>::form_element ()
 : base (rep::make_ptr())
{
}
template <class T, class M>
inline
form_element<T,M>::form_element (
	std::string	      name,
	const space_type&     X,
	const space_type&     Y,
	const geo_type&       omega,
	const quadrature_option_type& qopt)
 : base (rep::make_ptr(name,X,Y,omega,qopt))
{
}
template <class T, class M>
inline
void
form_element<T,M>::operator() (const geo_element& K, ublas::matrix<T>& m) const
{
    base::data().operator() (K, m);
}
template <class T, class M>
inline
bool
form_element<T,M>::is_symmetric () const
{
    return base::data().is_symmetric();
}
template <class T, class M>
inline
void
form_element<T,M>::set_weight (const field_basic<T,M>& wh) const
{
    return base::data().set_weight(wh);
}
template <class T, class M>
inline
bool
form_element<T,M>::is_weighted() const
{
    return base::data().is_weighted();
}
template <class T, class M>
inline
const field_basic<T,M>&
form_element<T,M>::get_weight () const
{
    return base::data().get_weight();
}
template <class T, class M>
inline
bool
form_element<T,M>::is_on_band() const
{
    return base::data().is_on_band();
}
template <class T, class M>
inline
void
form_element<T,M>::set_band(const band_basic<T,M>& bh) const
{
    return base::data().set_band(bh);
}
template <class T, class M>
inline
const band_basic<T,M>&
form_element<T,M>::get_band() const
{
    return base::data().get_band();
}

}// namespace rheolef
#endif // _RHEOLEF_FORM_ELEMENT_H
