#ifndef _RHEO_FIELD_ELEMENT_REP_H
#define _RHEO_FIELD_ELEMENT_REP_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

// used for assembly process of right-hand-sides

#include "rheolef/point.h"
#include "rheolef/field.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis_on_pointset.h"
#include "rheolef/piola.h"
#include "rheolef/band.h"
#include <boost/functional.hpp>

namespace rheolef { 

template<class T, class M>
class field_element {
public:

// typedef:

    typedef size_t size_type;

// allocators:

    field_element();

// accessors

    template <class Function>
    void operator() (const geo_element& K, Function f, std::vector<T>& lk) const;

    // specializations for field as special Function classes (avoid localization octree search)
    void operator() (const geo_element& K, const field_basic<T,M>& fh, std::vector<T>& lk) const;

    void eval (const geo_element& K, const std::vector<T>&               fq, std::vector<T>& lk) const;
    void eval (const geo_element& K, const std::vector<point_basic<T> >& fq, std::vector<T>& lk) const;

// modifiers (const, since data are mutable: avoid copies):

    void initialize (const space_basic<T,M>& X,
	const quadrature_option_type& qopt
	 = quadrature_option_type(quadrature_option_type::gauss)) const;

    void set_band (const band_basic<T,M>& gh) const { _band = gh; _is_on_band = true; }

// accessors (for building elementary fields):

    const space_basic<T,M>&   get_space() const { return _X; }
    const numbering<T,M>&     get_numbering() const { return _X.get_numbering(); }
    const basis_basic<T>&     get_basis() const { return get_numbering().get_basis(); }
    bool                      is_on_band() const { return _is_on_band; }
    const band_basic<T,M>&    get_band () const { return _band; }
    const geo_basic<T,M>&     get_geo() const { return _is_on_band ? _band.level_set() : get_space().get_geo(); }
    const basis_basic<T>&     get_piola_basis () const { return get_geo().get_piola_basis(); }
    size_type                 dimension() const { return get_geo().dimension(); }

// data:
protected:
    mutable space_basic<T,M>       _X;	// origin dof numbering
    mutable quadrature<T>          _quad;
    mutable basis_on_pointset<T>   _basis_on_quad;
    mutable basis_on_pointset<T>   _piola_on_quad;
    mutable bool                   _initialized;
    mutable bool                   _is_on_band;
    mutable band_basic<T,M>        _band;
};
// -----------------------------------------------------------------------
// inlined
// -----------------------------------------------------------------------
template<class T, class M>
inline
field_element<T,M>::field_element()
 : _X(),
   _quad(),
   _basis_on_quad(),
   _initialized(false),
   _is_on_band(false),
   _band()
{
}
// evaluate a template class-function f at quadrature points
// then call a compiled function for the integrations over element K
template<class T, class M>
template<class Function>
void 
field_element<T,M>::operator() (
    const geo_element& K,
    Function           f,
    std::vector<T>&    lk) const
{
  typedef typename boost::unary_traits<Function>::result_type result_type;
  const geo_basic<T,M>& omega = get_geo();
  reference_element hat_K (K.variant());
  std::vector<result_type> fq (_quad.size(hat_K));
  std::vector<size_type> dis_inod;
  omega.dis_inod (K, dis_inod);
  for (size_type q = 0, nq = _piola_on_quad.size(hat_K); q < nq; q++) {
    point_basic<T> xq = piola_transformation (omega, _piola_on_quad, K, dis_inod, q);
    fq[q] = f(xq);
  }
  eval (K, fq, lk);
}

}// namespace rheolef
#endif // _RHEO_FIELD_ELEMENT_REP_H
