///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// check:
//  \int_\Gamma weight(x) dx = mes(\Gamma) with weight
//
// where Gamma is a boundary of the [0,1]^d cube
// 
// example:
//   form_mass_bdr_tst -app P1 -weight 0 -I../data carre-v2 left
//
// Important note (23/03/2011, Pierre.Saramito@imag.fr)
//	3d bdry domain names not those of:  mkgeo_grid -T 5 > cube.geo
//	there are swaped, and the test uses only "cube-grummp.geo" 
//	where there are also swapped. E.g. left is {x=0} in cube-grumm.geo while
//	left is {y=0} in cube.geo. 
//	=> do not use cube.geo for this test !
//	See ../ptst/form_mass_bdr_tst.cc for a new rewritting of this test.
//
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

typedef Float (*function_type)(const point&);

Float weight_1    (const point& x) { return 1; }
Float weight_x   (const point& x) { return x[0]; }
Float weight_y   (const point& x) { return x[1]; }
Float weight_z   (const point& x) { return x[2]; }
Float weight_x2 (const point& x) { return x[0]*x[0]; }
Float weight_xy (const point& x) { return x[0]*x[1]; }
Float weight_y2 (const point& x) { return x[1]*x[1]; }
Float weight_xz (const point& x) { return x[0]*x[2]; }
Float weight_yz (const point& x) { return x[1]*x[2]; }
Float weight_z2 (const point& x) { return x[2]*x[2]; }

const char * domain_name [6] = {
	"bottom",		// y=0, xz varies
	"left",			// x=0, yz varies
	"back",			// z=0, xy varies (3d)

	"top",			// y=1, xz varies
	"right",		// x=1, yz varies
	"front",		// z=1, xy varies (3d)
};
unsigned int
domain_index (const string& name) {

    for  (unsigned int i = 0; i < 6; i++)
	if (domain_name[i] == name) return i;
    error_macro ("unexpected domain name `" << name << "'");
    return 6;
}

struct list_type {
        string name;
        function_type function;
        Float  expect [6];
};
typedef Float T;

//
// cartesian case
//
list_type weight_list[]={

		// bottom   left     back       top       right     front
		// y=0      x=0      z=0        y=1       x=1       z=1
                // dx.dz    dy.dz    dx.dy	dx.dz     dy.dz     dx.dy

 {"1", weight_1,  {1,	     1,	     1,   	1, 	  1, 	    1     } },
 {"x", weight_x,  {0.5,	     0,      0.5,	0.5,      1,        0.5   } },
 {"x2",weight_x2, {T(1)/3,   0,      T(1)/3,    T(1)/3,   1,        T(1)/3} },
 {"y", weight_y,  {0,        0.5,    0.5,       1,        0.5,      0.5   } },
 {"xy",weight_xy, {0,        0,      0.25,      0.5,      0.5,      0.25  } },
 {"y2",weight_y2, {0,        T(1)/3, T(1)/3,    1,        T(1)/3,   T(1)/3} },
 {"z", weight_z,  {0.5,      0.5,    0,         0.5,      0.5,      1     } },
 {"xz",weight_xz, {0.25,     0,      0,         0.25,     0.5,      0.5   } },
 {"yz",weight_yz, {0,        0.25,   0,         0.5,      0.25,     0.5   } },  
 {"z2",weight_z2, {T(1)/3,   T(1)/3, 0,         T(1)/3,   T(1)/3,   1     } }
};
const unsigned int max_weight = sizeof(weight_list)/sizeof(list_type);
//
// axisymmetric case
//
list_type weight_list_rz []={

		// bottom   left     back       top       right     front
		// y=0      x=0      z=0        y=1       x=1       z=1
                // x.dx     0        -      	x.dx      dy        -

 {"1", weight_1,  {0.5,	     0,	     -1,   	0.5, 	  1, 	    -1     } },
 {"x", weight_x,  {T(1)/3,   0,      -1,	T(1)/3,   1,        -1     } },
 {"x2",weight_x2, {0.25,     0,      -1, 	0.25,     1,        -1     } },
 {"y", weight_y,  {0,        0,      -1,        0.5,      0.5,      -1     } },
 {"xy",weight_xy, {0,        0,      -1,        T(1)/3,   0.5,      -1     } },
 {"y2",weight_y2, {0,        0,      -1,        0.5,      T(1)/3,   -1     } }
};
const unsigned int max_weight_rz = sizeof(weight_list_rz)/sizeof(list_type);

unsigned int
weight_index (const string& name, const string& coord_sys)
{
  if (coord_sys == "cartesian") {

    for  (unsigned int i = 0; i < max_weight; i++)
	if (weight_list[i].name == name) return i;
    error_macro ("unexpected weight `" << name << "'");
    return max_weight;

  } else { // axisymmetric rz

    for  (unsigned int i = 0; i < max_weight_rz; i++)
	if (weight_list_rz[i].name == name) return i;
    error_macro ("unexpected weight `" << name << "' in axisymmetric system");
    return max_weight_rz;
  }
}
function_type
get_function (string weight, string coord_sys)
{
  int weight_idx = weight_index (weight, coord_sys);
  if (coord_sys == "cartesian") {
    return weight_list    [weight_idx].function;
  } else {
    return weight_list_rz [weight_idx].function;
  }
}
Float
get_expected_value (string weight, string domain_name, string coord_sys)
{
  int weight_idx = weight_index (weight, coord_sys);
  int domain_idx = domain_index (domain_name);
  if (coord_sys == "cartesian") {
    return weight_list    [weight_idx].expect [domain_idx];
  } else {
    return weight_list_rz [weight_idx].expect [domain_idx];
  }
}
bool
is_on_axis (const space& W)
{
    const point& xmin = W.get_geo().xmin();
    const point& xmax = W.get_geo().xmax();
    return (xmin[0] == Float(0) && xmax[0] == Float(0));
}

void usage()
{
      cerr << "form_mass_bdr_tst: usage: form_mass_bdr_tst"
	   << " -app string"
	   << " {-Igeodir}*"
	   << " -|mesh[.geo]"
	   << " {domain}+"
	   << " [-weight string]"
	   << " [-rz]"
	   << endl;
      cerr << "example:\n";
      cerr << "  form_mass_bdr_tst -app P1 -I../data carre top -weight x\n";
      exit (1);
}
int main(int argc, char**argv)
{
    typedef geo::size_type size_type;
    Float epsilon = 1e-15;
    //
    // load geometry and options
    //
    geo omega;
    string approx = "P1";
    string weight = "1";
    bool mesh_done = false;
    geo::size_type n_dom = 0;
    string dom_name;
    string coord_sys = "cartesian";
    int status = 0;

    if (argc <= 1) usage() ;
    cerr << noverbose;

    for (int i=1 ; i < argc ; i++ ) {

      if (argv [i][0] == '-' && argv [i][1] == 'I')
	append_dir_to_rheo_path (argv[i]+2) ;
      else if (strcmp(argv[i], "-app") == 0)    approx = argv[++i];
      else if (strcmp(argv[i], "-weight") == 0) weight = argv[++i];
      else if (strcmp(argv[i], "-rz") == 0)     coord_sys = "rz";
      else if (strcmp(argv[i], "-") == 0) {

	  // input geo on standard input
	  if (mesh_done) usage() ;
	  cerr << " ! mass: read geo on stdin" << endl ;
	  cin >> omega ;
	  mesh_done = true ;

      } else if (!mesh_done) {

	  // input geo on file
	  omega = geo(argv[i]);
	  mesh_done = true ;
 
      } else {

          // a domain
	  dom_name = argv[i];
      }
    }
warning_macro ("main [1]");
    if (!mesh_done) {
      warning_macro("no mesh specified.");
      usage();
    }
    if (weight == "") {
      warning_macro("no weight specified");
      usage();
    }
    if (dom_name == "") {
      warning_macro("no boundary domain specified");
      usage();
    }
warning_macro ("main [2]");
    omega.set_coordinate_system(coord_sys);
    //
    // build boundary domain
    //
warning_macro ("main [3]");
    domain gamma(omega[dom_name]) ;
    //
    // build space
    //
warning_macro ("main [4]");
    space V(omega, approx)        ; V.freeze();
    space W(omega, gamma, approx) ; W.freeze();
    //
    // build field on V
    //
warning_macro ("main [5]");
    field one(V,1);
    function_type weight_fct = get_function(weight, omega.coordinate_system());
    field weight_h = interpolate(V,weight_fct);
    //
    // build forms
    //
warning_macro ("main [6]");
    form  m_h (V, V, "mass", gamma) ;
    // ---------------------------------------------------
    // first check : compute boundary area with i-th weight
    // ---------------------------------------------------
warning_macro ("main [7]");
    Float mes_gamma_1 = dot(one, m_h*weight_h);
    cout << "mes1(gamma) = " << double(mes_gamma_1) << endl;
    // ---------------------------------------------------
    // 2nd check : use trace
    // ---------------------------------------------------
    if (coord_sys != "rz" || !is_on_axis(W)) {
        //
        // build field on W
        // 
warning_macro ("main [8]");
        field weight_bdr_bug = interpolate(W,weight_fct);
        //
        // build form on W
        //
warning_macro ("main [9]");
        form  m_gamma_h (W, W, "mass") ;
warning_macro ("main [9a]");
        form  m_trace_h (V, W, "mass") ;
        //
        // deduce field on W by solving
        //   m_gamma_h*weight_bdr = m_trace_h*weight_h
        //
warning_macro ("main [10]");
        field one_bdr (W, 1.);
        field weight_bdr(W);
        ssk<Float> fact = ldlt(m_gamma_h.uu);
        weight_bdr.b = 0;
        weight_bdr.u = fact.solve(m_trace_h.uu*weight_h.u 
	    + m_trace_h.ub*weight_h.b - m_gamma_h.ub*weight_bdr.b);
        //
        // compute boundary area with i-th weight
        //
warning_macro ("main [11]");
        Float mes_gamma_2 = dot(one_bdr, m_gamma_h*weight_bdr);
        cout << "mes2(gamma) = " << double(mes_gamma_2) << endl;
        // ---------------------------------------------------
        // 3nd check : compare two results
        // ---------------------------------------------------
warning_macro ("main [12]");
        if (fabs(mes_gamma_1-mes_gamma_2) > sqrt(epsilon)) {
          cerr << "boundary mass problem detected." << endl;
          status = 1;
        }
    }
    // ---------------------------------------------------
    // 4th check : compare with exact data
    // ---------------------------------------------------
warning_macro ("main [13]");
    Float expect = get_expected_value (weight, dom_name, coord_sys);
    cout << "exact       = " << expect << endl;
    if (fabs(mes_gamma_1-expect) <= sqrt(epsilon)) {
        cerr << "ok" << endl;
    } else {
        cerr << "error = " << mes_gamma_1-expect << endl;
        status = 1;
    }
warning_macro ("main [14]");

#ifdef TO_CLEAN
    cout << ml
         << "weight   = " << weight_h.u << ";" << endl
         << "one_bdr  = " << one_bdr.u << ";" << endl
         << "m_bdr    = " << m_gamma_h.uu << ";" << endl
         << "m_trace  = " << m_trace_h.uu << ";" << endl
         << "weight_bdr  = " << weight_bdr.u << ";" << endl
        ;
#endif // TO_CLEAN

    return status;
}
