///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// USAGE :
//  mkgeo_grid -t 100 10 -a 0 -b 10 -c 0 -d 1 > square-10.geo
//  ./form_convect_dg_nh_tst square-10

/*
Date: Thu, 07 May 2009 17:07:03 +0200
From: Ibrahim Cheddadi <ibrahim.cheddadi@imag.fr>
Subject: transport + rheolef
To: Pierre Saramito <Pierre.Saramito@imag.fr>

Salut Pierre,

J'ai test le terme de bord sur un cas de transport simple (partial 
u/partial t + (v.grad)u = 0) : sur un canal  [0,10]x[0,1] avec v = 
(1,0), je mets en terme de  bord  sin(t)*exp(-10*(y-0.5)*(y-0.5)), 
c'est- dire un train de sinus moduls par une gaussienne, qui doit 
atteindre l'extrmit du canal  t = 10. Comme on le craignait, pour 
crer le terme de bord en second membre, l'interpolation d'une fonction 
de bord en P1d ne marche pas, mais j'ai contourn le problme en 
interpolant d'abord en P1 puis en projetant en P1d. Et a marche aprs 
une correction mineure dans form.cc. Les modifs sont remontes sous cvs.
Pour faire fonctionner le fichier d'exemple :
maillage : mkgeo_grid -t 100 -t 10 -a 0 -b 10 -c 0 -d 1 > omega-10.geo
puis : ./transport_pur omega-10.geo [N] > tp-10.branch, qui fait le 
calcul jusqu' t=10, en N pas de temps

a affiche l'erreur par rapport  la solution exacte tous les 10 pas de 
temps et le .branch contient 10 solutions en plus de la condition 
initiale (prise  zro). Ca converge bien en Dx et en Dt.

Par ailleurs, pour le bug des produits de formes : voil un exemple

space Vh(omega,"P2",vector);
space Qh(omega,"P1");
form b(Vh,Qh,"div");
form mq(Qh,Qh,"mass");
form A = trans(b)*mq*b;
field uh(Vh,0);
cout << A*uh;

a donne une seg fault alors que a devrait pas ! Et si on affiche le nb 
de lignes et de colonnes des formes, on voit que ce n'est pas bon.

Bon we,
Ibrahim
*/

#include "rheolef.h"
using namespace rheolef;
using namespace std;

double pi = 3.14159265;
point v (const point & x) { return point(1, 0); }
struct bdr : unary_function<point,Float> {  
  Float operator() (const point& x) {
    return exp(-30*(x[1]-0.5)*(x[1]-0.5))*sin(pi*t);
  }
  bdr (Float _t) : t(_t) {}
  protected: Float t;
};
struct u_ex : unary_function<point,Float> {  
  Float operator() (const point& x) {
    if (x[0] < t)
      return exp(-30*(x[1]-0.5)*(x[1]-0.5))*sin(pi*(t-x[0]));
    return 0;
  }
  u_ex (Float _t) : t(_t) {}
  protected: Float t;
};
int main (int argc, char **argv) {
#ifdef HAVE_UMFPACK
  geo omega (argv[1]);
  size_t n_max = (argc > 2) ? atoi(argv[2]) : 100 ;
  double delta_t = 10.0/n_max ;
  space Vh (omega, "P2", "vector");
  space Xh (omega, "P1d");
  space XhP1 (omega, "P1");
  field vh = interpolate (Vh, v);
  field uh(Xh, 0);
  form a0 (Xh, Xh, "d_dx0", vh[0]); 
  form a1 (Xh, Xh, "d_dx1", vh[1]); 
  form m  (Xh, Xh, "mass");
  form cv (Xh, Xh, "convect", vh);
  form cv_bdr(Xh, Xh, "convect_bdr", vh);
  form a = (a0 + a1 - cv) + 1.0/delta_t*m;
  ssk<Float> a_fact = lu(a.uu);
  field ubdrP1(XhP1,0);
  field ubdr(Xh,0);
  ubdrP1["left"] = interpolate(XhP1,bdr(0));
  ubdr = interpolate(Xh,ubdrP1);
  Float meas_omega = m(field(Xh,1),field(Xh,1));
  branch event ("t","u");
  cout << event(0, uh);
  size_t i = 1;
  size_t n = 1 ;
  cerr << "# n err_l2" << endl;
  Float err_linf_l2 = 0;
  Float err_l2_l2 = 0;
  for (; n <= n_max ; n++) {
    ubdrP1 = field(XhP1,0);
    ubdrP1["left"] = interpolate(XhP1,bdr(n*delta_t));
    ubdr = interpolate(Xh,ubdrP1);
    field f = 1.0/delta_t*m*uh + cv_bdr*ubdr;
    uh.u = a_fact.solve(f.u);
    field e = uh - interpolate (Xh, u_ex(n*delta_t));
    // 10 sorties en plus de la condition initiale
    if (10*n/n_max >= i) {
      cout << event (n*delta_t,uh);
      i++;
    }
    Float err_l2 = sqrt(m(e,e)/meas_omega);
    err_l2_l2 += sqr(err_l2);
    err_linf_l2 = max(err_linf_l2, err_l2);
    cerr << n*delta_t << "\t" << err_l2 << endl;
  }
  Float T = n_max*delta_t;
  err_l2_l2 = sqrt(err_l2_l2*delta_t/T);
  cerr << "# err_linf_l2 = " << err_linf_l2 << endl;
  cerr << "# err_l2_l2   = " << err_l2_l2 << endl;
  return (err_l2_l2 < 0.5) ? 0 : 1;
#else // ! HAVE_UMFPACK
  return 0;
#endif // ! HAVE_UMFPACK
}
