///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/config.h"

#ifdef _RHEOLEF_HAVE_MPI
#include "rheolef/geo_domain.h"

namespace rheolef {

/** Implementation notes:

   There are four vertices numbering:

    a) bgd_iv : background vertex index
             i.e. index of vertex as numbered in the backgroud geo omega

    b) bgd_ios_iv : background vertex index for i/o (nproc independent)

    c) dom_iv : index of a vertex as numbered in the current geo_domain
             this is a reduced set of vertices: vertices that not appears
             in the list of elements of the domain are skipped.
	     This numbering is constrained to follow the background
	     vertex distribution.

    d) dom_ios_iv : index of a vertex on the domain for i/o (nproc independent)
	     This numbering is constrained to follow the background
	     ios vertex distribution.

    The numbering bgd_iv and bgd_ios_iv are already defined.
    The numbering dom_iv and dom_ios_iv are defined here.
    The correspondances dom_iv <--> dom_ios_iv are also defined for i/o.

  Algorithm: for vertices renumbering

    1) scan the domain and mark vertex in the bgd numbering:
    bool bgd_iv_is_on_domain     [bgd_iv_ownership] = false
    bool bgd_ios_iv_is_on_domain [bgd_ios_iv_ownership] = false
    for K in omega.domain
      for iloc=0..K.size
        bgd_dis_iv = K[iloc]
        bgd_ios_dis_ie = omega.dis_ie2ios_dis_ie (0,bgd_iv)
        bgd_iv_is_on_domain     [[bgd_dis_iv]]     = true
        bgd_ios_iv_is_on_domain [[bgd_ios_dis_iv]] = true
    bgd_iv_is_on_domain.assembly
    bgd_ios_iv_is_on_domain.assembly
 
    2) counting & distribution for dom_iv 
    dom_nv = 0
    for bdg_iv = ...
      if bgd_iv_is_on_domain [bgd_iv] then
	dom_nv++
    dom_iv_ownership = distributor (-,comm,dom_nv)

    3) numbering dom_iv 
     & permutation: bgd_iv --> dom_iv
    array bgd_iv2dom_iv [bgd_iv_ownership]
    dom_iv = 0
    for bdg_iv = ...
      if bgd_iv_is_on_domain [bgd_iv] then
        bgd_iv2dom_iv [bgd_iv] = dom_iv
        dom_iv++

    4) counting & distribution for dom_ios_iv 
    dom_ios_nv = 0
    for bdg_ios_iv = ...
      if bgd_ios_iv_is_on_domain [bgd_ios_iv] then
	dom_ios_nv++
    dom_ios_iv_ownership = distributor (-,comm,dom_ios_nv)

    5) numbering dom_ios_iv 
     & permutation: bgd_ios_iv --> dom_ios_iv
    array bgd_ios_iv2dom_ios_iv [bgd_ios_iv_ownership]
    dom_ios_iv = 0
    for bdg_ios_iv = ...
      if bgd_ios_iv_is_on_domain [bgd_ios_iv] then
        bgd_ios_iv2dom_ios_iv [bgd_ios_iv] = dom_ios_iv
        dom_ios_iv++

    6) permutation: bgd_iv --> dom_ios_iv
    array bgd_iv2dom_ios_dis_iv   [bgd_iv_ownership]
    for bdg_ios_iv = ...
      if bgd_ios_iv_is_on_domain [bgd_ios_iv] then
        dom_ios_iv = bgd_ios_iv2dom_ios_iv [bgd_ios_iv]
	bgd_dis_iv = omega.ios_iv2dis_iv (bgd_ios_iv)
        bgd_iv2dom_ios_dis_iv [[bgd_dis_iv]] = dom_ios_dis_iv
    bgd_iv2dom_ios_dis_iv.assembly

    7) permutations: dom_ios_iv <--> dom_iv
    array dom_ios_iv2dom_dis_iv   [dom_iv_ownership]
    array dom_iv2dom_ios_dis_iv   [dom_ios_iv_ownership]
    for bdg_iv = ...
      if bgd_iv_is_on_domain [bgd_iv] then
        dom_iv = bgd_iv2dom_iv [bgd_iv]
        dom_ios_dis_iv = bgd_iv2dom_ios_dis_iv [bgd_iv]
        dom_iv2dom_ios_dis_iv [dom_iv] = dom_ios_dis_iv
        dom_dis_iv = first_dom_dis_iv + dom_iv
        dom_ios_iv2dom_dis_iv [[dom_ios_dis_iv]] = dom_dis_iv
    dom_ios_iv2dom_dis_iv.assembly

    8) loop on elements and copy from domain with new vertex numbering

  */ 

template <class T>
void
geo_rep<T,distributed>::build_from_domain (
    const domain_indirect_rep<distributed>&    indirect, 
    const geo_abstract_rep<T,distributed>& bgd_omega)
{
  warning_macro ("build_from_domain...");
  base::_name = bgd_omega.name() + "." + indirect.name();
  base::_version = 1; // not yet connectivity
  base::_dimension     = bgd_omega.dimension();
  base::_map_dimension = indirect.map_dimension();
  size_type map_dim = base::_map_dimension;
  // ------------------------------------------------------------------------
  // 1) vertex compact re-numbering
  // ------------------------------------------------------------------------
  // 1.1 loop on elements and mark used vertices
  size_type first_dis_ige = bgd_omega.geo_element_ownership (map_dim).first_index();
  size_type first_bgd_dis_iv = bgd_omega.geo_element_ownership(0).first_index();
  size_type  last_bgd_dis_iv = bgd_omega.geo_element_ownership(0).last_index();
  std::set<size_type> ext_bgd_dis_iv_set;
  array<size_type> bgd_iv_is_on_domain     (bgd_omega.geo_element_ownership(0),     0); // logical, init to "false"
  array<size_type> bgd_ios_iv_is_on_domain (bgd_omega.geo_element_ios_ownership(0), 0);
  for (size_type ioige = 0, noige = indirect.size(); ioige < noige; ioige++) {
    size_type ige = indirect.oige (ioige).index();
    const geo_element& S = bgd_omega.get_geo_element (map_dim, ige);
    for (size_type iloc = 0; iloc < S.size(); iloc++) {
      size_type bgd_dis_iv = S[iloc];
      bgd_iv_is_on_domain.dis_entry (bgd_dis_iv) += 1;
      if (bgd_dis_iv < first_bgd_dis_iv || bgd_dis_iv >= last_bgd_dis_iv) {
	ext_bgd_dis_iv_set.insert (bgd_dis_iv);
      }
      size_type bgd_ios_dis_iv = bgd_omega.dis_ige2ios_dis_ige (0, bgd_dis_iv);
      bgd_ios_iv_is_on_domain.dis_entry (bgd_ios_dis_iv) += 1;
    }
  }
  bgd_iv_is_on_domain.dis_entry_assembly     (set_add_op<size_type,size_type>()); // logical "or"
  bgd_ios_iv_is_on_domain.dis_entry_assembly (set_add_op<size_type,size_type>());

  // 1.2 counting & distribution for dom_iv 
  size_type dom_nv = 0;
  for (size_type bgd_iv = 0, bgd_nv = bgd_omega.geo_element_ownership(0).size(); bgd_iv < bgd_nv; bgd_iv++) {
    if (bgd_iv_is_on_domain[bgd_iv] != 0) dom_nv++ ;
  }
  // 1.3 numbering dom_iv & permutation: bgd_iv <--> dom_iv
  communicator comm = bgd_omega.geo_element_ownership(0).comm();
  distributor dom_iv_ownership (distributor::decide, comm, dom_nv);
  size_type first_dom_dis_iv = dom_iv_ownership.first_index();
  array<size_type> bgd_iv2dom_dis_iv (bgd_omega.geo_element_ownership(0), std::numeric_limits<size_type>::max());
  array<size_type> dom_iv2bgd_iv (dom_iv_ownership, std::numeric_limits<size_type>::max());
  for (size_type dom_iv = 0, bgd_iv = 0, bgd_nv = bgd_omega.geo_element_ownership(0).size(); bgd_iv < bgd_nv; bgd_iv++) {
    if (bgd_iv_is_on_domain[bgd_iv] == 0) continue;
    size_type dom_dis_iv = first_dom_dis_iv + dom_iv;
    bgd_iv2dom_dis_iv [bgd_iv] = dom_dis_iv;
    dom_iv2bgd_iv [dom_iv] = bgd_iv;
    dom_iv++;
  }
  // 1.4 counting & distribution for dom_ios_iv 
  size_type dom_ios_nv = 0;
  for (size_type bgd_ios_iv = 0, bgd_ios_nv = bgd_omega.geo_element_ios_ownership(0).size(); bgd_ios_iv < bgd_ios_nv; bgd_ios_iv++) {
    if (bgd_ios_iv_is_on_domain[bgd_ios_iv] != 0) dom_ios_nv++ ;
  }
  // 1.5 numbering dom_ios_iv & permutation: bgd_ios_iv <--> dom_ios_iv
  distributor dom_ios_iv_ownership (distributor::decide, comm, dom_ios_nv);
  size_type first_dom_ios_dis_iv = dom_ios_iv_ownership.first_index();
  array<size_type> bgd_ios_iv2dom_ios_dis_iv (bgd_omega.geo_element_ios_ownership(0), std::numeric_limits<size_type>::max());
  array<size_type> dom_ios_iv2bgd_ios_iv (dom_ios_iv_ownership, std::numeric_limits<size_type>::max());
  for (size_type dom_ios_iv = 0, bgd_ios_iv = 0, bgd_ios_nv = bgd_omega.geo_element_ios_ownership(0).size(); bgd_ios_iv < bgd_ios_nv; bgd_ios_iv++) {
    if (bgd_ios_iv_is_on_domain[bgd_ios_iv] == 0) continue;
    size_type dom_ios_dis_iv = first_dom_ios_dis_iv + dom_ios_iv;
    bgd_ios_iv2dom_ios_dis_iv [bgd_ios_iv] = dom_ios_dis_iv;
    dom_ios_iv2bgd_ios_iv [dom_ios_iv] = bgd_ios_iv;
    dom_ios_iv++;
  }
  // 1.6 permutation: bgd_iv --> dom_ios_iv
  array<size_type> bgd_iv2dom_ios_dis_iv (bgd_omega.geo_element_ownership(0), std::numeric_limits<size_type>::max());
  for (size_type dom_ios_iv = 0; dom_ios_iv < dom_ios_nv; dom_ios_iv++) {
    size_type bgd_ios_iv = dom_ios_iv2bgd_ios_iv [dom_ios_iv];
    size_type dom_ios_dis_iv = dom_ios_iv + first_dom_ios_dis_iv;
    size_type bgd_dis_iv = bgd_omega.ios_ige2dis_ige (0, bgd_ios_iv);
    bgd_iv2dom_ios_dis_iv.dis_entry (bgd_dis_iv) = dom_ios_dis_iv;
  }
  bgd_iv2dom_ios_dis_iv.dis_entry_assembly();

  // 1.7 permutations: dom_ios_iv <--> dom_iv
  array<size_type> dom_iv2dom_ios_dis_iv (dom_iv_ownership,     std::numeric_limits<size_type>::max());
  array<size_type> dom_ios_iv2dom_dis_iv (dom_ios_iv_ownership, std::numeric_limits<size_type>::max());
  for (size_type dom_iv = 0; dom_iv < dom_nv; dom_iv++) {
    size_type bgd_iv = dom_iv2bgd_iv [dom_iv];
    size_type dom_dis_iv = dom_iv + first_dom_dis_iv;
    size_type dom_ios_dis_iv = bgd_iv2dom_ios_dis_iv [bgd_iv];
    dom_iv2dom_ios_dis_iv [dom_iv] = dom_ios_dis_iv;
    dom_ios_iv2dom_dis_iv.dis_entry (dom_ios_dis_iv) = dom_dis_iv;
  }
  dom_ios_iv2dom_dis_iv.dis_entry_assembly();
  // ------------------------------------------------------------------------
  // 2) defines _geo_element[0] and P.set_ios_dis_ie
  // ------------------------------------------------------------------------
  base::_geo_element[0].resize (dom_iv_ownership);
  for (size_type dom_iv = 0; dom_iv < dom_nv; dom_iv++) {
    size_type bgd_iv = dom_iv2bgd_iv [dom_iv];
    size_type dom_dis_iv = first_dom_dis_iv + dom_iv;
    base::_geo_element[0][dom_iv] = geo_element_p (dom_dis_iv);
    geo_element& P = base::_geo_element[0][dom_iv];
    size_type dom_ios_dis_iv = dom_iv2dom_ios_dis_iv [dom_iv];
    P.set_ios_dis_ie (dom_ios_dis_iv);
  }
  // ------------------------------------------------------------------------
  // 3) raw copy _vertex
  // TODO: use shallow copy & indirection vertex(iv) access via geo_elment[0] renumbering
  // ------------------------------------------------------------------------
  base::_vertex.resize (dom_iv_ownership);
  for (size_type dom_iv = 0; dom_iv < dom_nv; dom_iv++) {
    size_type bgd_iv = dom_iv2bgd_iv [dom_iv];
    base::_vertex [dom_iv] = bgd_omega.vertex (bgd_iv);
  }
  // ------------------------------------------------------------------------
  // 4) copy _geo_element[map_dim] with compact vertex numbering
  // ------------------------------------------------------------------------
  size_type first_dis_ioige  = indirect.ownership().first_index();
  base::_geo_element[map_dim].resize (indirect.ownership());
  std::map<size_type,size_type> ext_bgd_dis_iv_map;
  bgd_iv2dom_dis_iv.get_dis_entry (ext_bgd_dis_iv_set, ext_bgd_dis_iv_map);
  for (size_type ioige = 0, noige = indirect.size(); ioige < noige; ioige++) {
    size_type dis_ioige = first_dis_ioige + ioige;
    size_type ige = indirect.oige (ioige).index();
    geo_element::intrusive_set (base::_geo_element[map_dim][ioige], bgd_omega.get_geo_element (map_dim, ige));
    geo_element& S = base::_geo_element[map_dim][ioige];
    S.set_dis_ie (dis_ioige);
    size_type ini_dis_ioige = indirect.ioige2ini_dis_ioige (ioige);
    S.set_ios_dis_ie (ini_dis_ioige);
    // TODO: apply also orientation to S
    if (S.dimension() > 0) {
      for (size_type iloc = 0, nloc = S.size(); iloc < nloc; iloc++) {
        size_type bgd_dis_iv = S[iloc];
        size_type dom_dis_iv;
        // TODO: ce shema revient tres souvent: a encapsuler, style: map_val = dis_x[set_dis_i]
        if (bgd_dis_iv >= first_bgd_dis_iv && bgd_dis_iv < last_bgd_dis_iv) {
          size_type bgd_iv = bgd_dis_iv - first_bgd_dis_iv;
          dom_dis_iv = bgd_iv2dom_dis_iv [bgd_iv];
        } else {
          typename std::map<size_type,size_type>::const_iterator iter = ext_bgd_dis_iv_map.find (bgd_dis_iv);
	  check_macro (iter != ext_bgd_dis_iv_map.end(), "unexpected index " << bgd_dis_iv);
          dom_dis_iv = (*iter).second;
        }
        S[iloc] = dom_dis_iv;
      }
    }
  }
  // ------------------------------------------------------------------------
  // 5) set reverse permutations for i/o
  // ------------------------------------------------------------------------
  _ios_ige2dis_ige [0]       = dom_ios_iv2dom_dis_iv;
  _ios_ige2dis_ige [map_dim] = indirect._ini_ioige2dis_ioige;
  // ------------------------------------------------------------------------
  // 6) external entities
  // ------------------------------------------------------------------------
  build_external_entities ();
  // ------------------------------------------------------------------------
  // 7) size_by
  // ------------------------------------------------------------------------
  reset_size_by ();

  /* There is still a problem with ios_size_by_variant[] :
      - first, its formaly "ini_size_by_variant" for domain_indirect
        since i/o are associated to "ini" numbering for domain_indirect.
      - physically, geo_elements of the background geo are never distributed
        with the "ini" ownership so, the associated number of variants are unavailable
     Notice that this "ini" domain_indirect numbering becomes the "ios" for the geo_domain.
     Also, the new numbering is denoted by "dom" for geo_domain.
     Thus, there are at least two solutions for geo_domain :
     1) via the new "dom_ige" numbering, access physically to elements and store variants in:
		array<size_t> dom_variant (dom_ige_ownership)
	then, permut it using ios_ige2dom_dis_ige and get
		array<size_t> ios_variant (ios_ige_ownership)
	finaly, count variants and store it in
  		_ios_size_by_variant[]
     2) on non-mixed domains, the variant array is not necessary, and assembly communications also
        can be avoided. Just identify the used variant related to map_dimension and store its
        number from dom_ige_domain.size in _ios_size_by_variant[].
   */

  std::fill (_ios_size_by_variant, _ios_size_by_variant+reference_element::max_size, 0);
  bool is_mixed = false;
  switch (map_dim) {
    case 2:
      is_mixed   = _dis_size_by_variant[reference_element::t] != 0 &&
                   _dis_size_by_variant[reference_element::q] != 0;
      break;
    case 3:
      is_mixed = ( _dis_size_by_variant[reference_element::T] != 0 &&
	           _dis_size_by_variant[reference_element::P] != 0) ||
	         ( _dis_size_by_variant[reference_element::P] != 0 &&
	           _dis_size_by_variant[reference_element::H] != 0) ||
	         ( _dis_size_by_variant[reference_element::H] != 0 &&
	           _dis_size_by_variant[reference_element::T] != 0);
      break;
  }
  if (is_mixed) {
    error_macro ("domain \""<<base::_name<<"\" with mixed element variants: not yet");
  } else {
    size_type ios_ne = _ios_ige2dis_ige[map_dim].ownership().size();
    warning_macro ("ios_ne = " << ios_ne);
    switch (map_dim) {
      case 0:
        _ios_size_by_variant[reference_element::p] = ios_ne;
	break;
      case 1:
        _ios_size_by_variant[reference_element::e] = ios_ne;
	break;
      case 2:
        if (_dis_size_by_variant[reference_element::t] != 0) {
            _ios_size_by_variant[reference_element::t] = ios_ne;
        } else {
            _ios_size_by_variant[reference_element::q] = ios_ne;
        }
	break;
      case 3:
        if (_dis_size_by_variant[reference_element::T] != 0) {
            _ios_size_by_variant[reference_element::T] = ios_ne;
        } else 
        if (_dis_size_by_variant[reference_element::P] != 0) {
            _ios_size_by_variant[reference_element::P] = ios_ne;
        } else {
            _ios_size_by_variant[reference_element::H] = ios_ne;
        }
	break;
    }
  }
  /** TODO: autres champs a initialiser :

   	 _geo_element [dim]     : pour dim=1,..,map_dim-1
    	_domains		: intersection avec le dom courant

   et distributed :
    	_ios_ige2dis_ige[dim] : pour dim=1,..,map_dim
   */
  warning_macro ("build_from_domain done");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class geo_rep<Float,distributed>;

} // namespace rheolef
#endif // _RHEOLEF_HAVE_MPI
