///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/element.h"
#include "rheolef/basis_on_lattice.h"

namespace rheolef {

template<class T>
element_rep<T>::element_rep (std::string name, std::string numb) 
 : _name((numb != "") ? numb : name),
   _b(name),
   _numb( ((numb != "") ? numb : name) + "_numbering"),
   _tr_p1("P1"),
   _tr_p1_value()
{
   _tr_p1_value.set (_b, _tr_p1);
}
// ----------------------------------------------------------------------------
// node associated to a dof: local (K,loc_idof) argument
// ----------------------------------------------------------------------------
template<class T>
template<class M>
basic_point<T>
element_rep<T>::x_dof (const geo_element& K, size_type loc_idof, const geo_basic<T,M>& omega) const
{
    // TODO: only P1 transformation supported : isoparametric not yet !
    warning_macro ("x_dof("<<name()<<") on K"<< K.dis_ie());
    basis_on_nodal_basis::const_iterator first = _tr_p1_value.begin(K,loc_idof);
    basis_on_nodal_basis::const_iterator last  = _tr_p1_value.end  (K,loc_idof);
    point x;
    size_type dim = omega.dimension();
    for (size_type k = 0; first != last; first++, k++) {
        const point& ak = omega.vertex (K[k]);
	Float ck = (*first);
	warning_macro ("coef["<<k<<"]="<<ck);
	for (size_type i = 0; i < dim; i++) {
	  x[i] += ck * ak[i];
        }
    }
    return x;
}
// ----------------------------------------------------------------------------
// node associated to a dof: "idof" argument
// ----------------------------------------------------------------------------
template<class T>
template<class M>
basic_point<T>
element_rep<T>::x_dof (size_type idof, const geo_basic<T,M>& omega) const
{
    if (name() == "P1") {
	return omega.vertex (idof);
    } else if (name() == "P0") {
	const geo_element& K = omega[idof];
        basic_point<T> x;
	for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
	  x += omega.dis_vertex (K[iloc]);
        }
	x /= K.size();
	warning_macro ("x_dof = " << x);
	return x;
    } else if (name() == "P1d") {
        size_type nvert_per_elt;
        switch (omega.map_dimension()) {
  	  case 0: {
            nvert_per_elt = 1;
  	    break;
          }
  	  case 1: {
            nvert_per_elt = 2;
  	    break;
          }
  	  case 2: {
  	    if (omega.dis_size_by_variant()[reference_element::t] != 0) {
              nvert_per_elt = 3;
  	    } else {
              nvert_per_elt = 4;
            }
  	    break;
          }
  	  case 3:
  	  default: {
  	    if (omega.dis_size_by_variant()[reference_element::T] != 0) {
              nvert_per_elt = 4;
  	    } else if (omega.dis_size_by_variant()[reference_element::P] != 0) {
              nvert_per_elt = 6;
  	    } else {
              nvert_per_elt = 8;
  	    }
          }
        }
        // TODO: extend P1d mixed (t,q) or (T,P,H) meshes: nvert_per_elt is no more constant
	size_type ie   = idof/nvert_per_elt;
	size_type iloc = idof - ie*nvert_per_elt;
	const geo_element& K = omega[ie];
	size_type dis_iv = K[iloc];
        const basic_point<T>& x = omega.dis_vertex(dis_iv);
	return x;
    } else {
        // TODO: extend P2
	error_macro ("unsupported x_dof for \""<<name()<<"\" element");
	return basic_point<T>();
    }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class element<Float>;
template class element_rep<Float>;
template class lagrange<Float>;

template
basic_point<Float> element_rep<Float>::x_dof (const geo_element& K, size_type loc_idof, const geo_basic<Float,sequential>& omeqa) const;
template
basic_point<Float> element_rep<Float>::x_dof (size_type idof, const geo_basic<Float,sequential>& omeqa) const;

#ifdef _RHEOLEF_HAVE_MPI
template
basic_point<Float> element_rep<Float>::x_dof (const geo_element& K, size_type loc_idof, const geo_basic<Float,distributed>& omeqa) const;
template
basic_point<Float> element_rep<Float>::x_dof (size_type idof, const geo_basic<Float,distributed>& omeqa) const;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
